/* Giram: a simple 3D modeller
 * Copyright (C) 2001 DindinX <David@dindinx.org>
 *
 * LIBGIMP - The GIMP Library
 * Copyright (C) 1995-1997 Peter Mattis and Spencer Kimball
 *
 * giramwidgets.c
 * Copyright (C) 2000 Michael Natterer <mitch@gimp.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "girampixmap.h"

/**
 * giram_spin_button_new:
 * @adjustment:     Returns the spinbutton's #GtkAdjustment.
 * @value:          The initial value of the spinbutton.
 * @lower:          The lower boundary.
 * @upper:          The uppper boundary.
 * @step_increment: The spinbutton's step increment.
 * @page_increment: The spinbutton's page increment (mouse button 2).
 * @page_size:      The spinbutton's page size.
 * @climb_rate:     The spinbutton's climb rate.
 * @digits:         The spinbutton's number of decimal digits.
 *
 * This function is a shortcut for gtk_adjustment_new() and a subsequent
 * gtk_spin_button_new() and does some more initialisation stuff like
 * setting a standard minimun horizontal size.
 *
 * Returns: A #GtkSpinbutton and it's #GtkAdjustment.
 **/
GtkWidget *giram_spin_button_new(GtkObject **adjustment,  /* return value */
                                 gfloat      value,
                                 gfloat      lower,
                                 gfloat      upper,
                                 gfloat      step_increment,
                                 gfloat      page_increment,
                                 gfloat      page_size,
                                 gfloat      climb_rate,
                                 guint       digits)
{
  GtkWidget *spinbutton;

  *adjustment = gtk_adjustment_new(value, lower, upper,
                                   step_increment, page_increment, page_size);

  spinbutton = gtk_spin_button_new(GTK_ADJUSTMENT(*adjustment),
                                   climb_rate, digits);
  //gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spinbutton), GTK_SHADOW_NONE);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spinbutton), TRUE);
  gtk_widget_set_usize(spinbutton, 75, -1);

  return spinbutton;
}

/**
 * giram_pixmap_button_new:
 * @xpm_data: The XPM data which will be passed to giram_pixmap_new().
 * @text:     An optional text which will appear right of the pixmap.
 *
 * Returns: A #GtkButton with a #GimpPixmap and an optional #GtkLabel.
 **/
GtkWidget *giram_pixmap_button_new(gchar       **xpm_data,
                                   const gchar  *text)
{
  GtkWidget *button;
  GtkWidget *pixmap;

  button = gtk_button_new();
  pixmap = giram_pixmap_new(xpm_data);

  if (text)
  {
    GtkWidget *abox;
    GtkWidget *hbox;
    GtkWidget *label;

    abox = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
    gtk_container_add(GTK_CONTAINER(button), abox);
    gtk_widget_show(abox);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(abox), hbox);
    gtk_widget_show(hbox);

    gtk_box_pack_start(GTK_BOX(hbox), pixmap, FALSE, FALSE, 4);
    gtk_widget_show(pixmap);

    label = gtk_label_new(text);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 4);
    gtk_widget_show(label);
  } else
  {
    gtk_container_add(GTK_CONTAINER(button), pixmap);
    gtk_widget_show(pixmap);
  }

  return button;
}

/*
 *  Helper Functions
 */

/**
 * giram_table_attach_aligned:
 * @table:      The #GtkTable the widgets will be attached to.
 * @column:     The column to start with.
 * @row:        The row to attach the eidgets.
 * @label_text: The text for the #GtkLabel which will be attached left of the
 *              widget.
 * @xalign:     The horizontal alignment of the #GtkLabel.
 * @yalign:     The vertival alignment of the #GtkLabel.
 * @widget:     The #GtkWidget to attach right of the label.
 * @colspan:    The number of columns the widget will use.
 * @left_align: #TRUE if the widget should be left-aligned.
 *
 * Note that the @label_text can be #NULL and that the widget will be attached
 * starting at (@column + 1) in this case, too.
 **/
void giram_table_attach_aligned(GtkTable    *table,
                                gint         column,
                                gint         row,
                                const gchar *label_text,
                                gfloat       xalign,
                                gfloat       yalign,
                                GtkWidget   *widget,
                                gint         colspan,
                                gboolean     left_align)
{
  if (label_text)
  {
    GtkWidget *label;

    label = gtk_label_new(label_text);
    gtk_misc_set_alignment(GTK_MISC(label), xalign, yalign);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
    gtk_table_attach(table, label,
                     column, column + 1,
                     row, row + 1,
                     GTK_FILL, GTK_FILL, 0, 0);
    gtk_widget_show(label);
  }

  if (left_align)
  {
    GtkWidget *alignment;

    alignment = gtk_alignment_new(0.0, 0.5, 0.0, 0.0);
    gtk_container_add(GTK_CONTAINER(alignment), widget);
    gtk_widget_show(widget);

    widget = alignment;
  }

  gtk_table_attach(table, widget,
                   column + 1, column + 1 + colspan,
                   row, row + 1,
                   GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
  gtk_widget_show (widget);
}

