/* tool_torus.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "csgtree.h"
#include "snap.h"
#include "utils.h"
#include "view.h"
#include "giramcursor.h"
#include "primitives/torus.h"
#include "widgets/giramviewshell.h"
#include "pixmaps/torus.xpm"

#include "tool_torus.h"

#include "giramintl.h"

static double Xorg, Yorg, Radius, MinorRadius;
static guint id;
static Vector Center;

static GdkPixmap *backpixmap;

#define NONE           1
#define MOVING         2
#define BETWEEN_RADIUS 3
#define MINOR_RADIUS   4

static int TORUS_STATE = NONE;

/*****************************************************************************
*  tool_torus_button_press
******************************************************************************/
static void tool_torus_button_press(GtkWidget *canvas, GdkEventButton *bevent)
{
  ViewStruct  *view_data;
  GtkWidget   *status_bar;
  double       Zoom, Xoff, Yoff;

  if (TORUS_STATE == BETWEEN_RADIUS)
  {
    TORUS_STATE = MINOR_RADIUS;
    return;
  }
  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "new torus");
  MouseToReal(bevent->x, bevent->y, canvas, &Xorg, &Yorg);

  if (backpixmap)
    g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(canvas->window,
                              canvas->allocation.width,
                              canvas->allocation.height,
                              -1);

  draw_all(canvas, backpixmap, view_data->camera_style,
           canvas->allocation.width,
           canvas->allocation.height);
  gdk_window_set_back_pixmap(canvas->window, backpixmap, FALSE);
  gdk_window_clear(canvas->window);

  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(Center, Xorg, Yorg, 0.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(Center, Xorg, 0.0, Yorg);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(Center, 0.0, Yorg, Xorg);
      break;
  }
  Radius = 0.0;

  TORUS_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, _("Torus: "));
  gdk_pointer_grab(canvas->window, FALSE,
                   GDK_BUTTON_PRESS_MASK |
                   GDK_POINTER_MOTION_MASK |
                   GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  tool_torus_motion
******************************************************************************/
static void tool_torus_motion(GtkWidget *canvas, GdkEventMotion *Mev)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;
  double      Zoom, Xoff, Yoff, Xold, Yold;
  gchar      *Message;
  double      NewRadius;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  if (TORUS_STATE == MOVING)
  {
    gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

    gdk_window_clear(canvas->window);

    MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
    Radius = sqrt((Xold-Xorg)*(Xold-Xorg)+(Yold-Yorg)*(Yold-Yorg));
    gdk_draw_arc(canvas->window, giram_purple_gc, FALSE,
                 canvas->allocation.width/2+(Xorg-Radius-Xoff)*Zoom,
                 canvas->allocation.height/2-(Yorg+Radius-Yoff)*Zoom,
                 Radius*2.0*Zoom, Radius*2.0*Zoom,  0, 64*360);
    Message = g_strdup_printf(_("Torus: Center <%g, %g, %g>, Major Radius: %g"),
                              Center[0], Center[1], Center[2], Radius);
    gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, Message);
    g_free(Message);
  } else if ((TORUS_STATE == MINOR_RADIUS) || (TORUS_STATE == BETWEEN_RADIUS))
  {
    gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

    gdk_window_clear(canvas->window);

    MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
    NewRadius = sqrt((Xold-Xorg)*(Xold-Xorg)+(Yold-Yorg)*(Yold-Yorg));
    MinorRadius = fabs(NewRadius-Radius);
    gdk_draw_arc(canvas->window, giram_purple_gc, FALSE,
                 canvas->allocation.width/2+(Xorg-(Radius-MinorRadius)-Xoff)*Zoom,
                 canvas->allocation.height/2-(Yorg+(Radius-MinorRadius)-Yoff)*Zoom,
                 (Radius-MinorRadius)*2.0*Zoom,
                 (Radius-MinorRadius)*2.0*Zoom, 0,64*360);
    gdk_draw_arc(canvas->window, giram_purple_gc, FALSE,
                 canvas->allocation.width/2+(Xorg-(Radius+MinorRadius)-Xoff)*Zoom,
                 canvas->allocation.height/2-(Yorg+(Radius+MinorRadius)-Yoff)*Zoom,
                 (Radius+MinorRadius)*2.0*Zoom,
                 (Radius+MinorRadius)*2.0*Zoom, 0,64*360);
    Message = g_strdup_printf(_("Torus: Center <%g, %g, %g>, Major Radius: %g, Minor Radius: %g"),
            Center[0], Center[1], Center[2], Radius, MinorRadius);
    gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, Message);
    g_free(Message);
  }
}

/*****************************************************************************
*  tool_torus_button_release
******************************************************************************/
static void tool_torus_button_release(GtkWidget *canvas, GdkEventButton *Bev)
{
  ViewStruct   *view_data;
  GtkWidget    *status_bar;
  double        Zoom, Xoff, Yoff;
  GSList       *tmp_list;
  ViewStruct   *TmpView;
  ObjectStruct *Torus;
  Vector        Vect;

  if (TORUS_STATE == MOVING)
  {
    TORUS_STATE = BETWEEN_RADIUS;
    MinorRadius = 0.0;
    return;
  }

  if (TORUS_STATE != MINOR_RADIUS)
    return;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

  Torus = giram_torus_new(Radius, MinorRadius);
  Torus->name = create_uniq_object_name(view_data->frame, _("torus"));

  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(Vect, 90.0, 0.0, 0.0);
      giram_object_rotate(Torus, Vect);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(Vect, 0.0, 0.0, 90.0);
      giram_object_rotate(Torus, Vect);
      break;
  }
  giram_object_translate(Torus, Center);
  view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects,
                                                 Torus);
  Torus->frame = view_data->frame;
  giram_object_build_triangle_mesh(Torus);
  TORUS_STATE = NONE;

  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(view_data->frame);
  gdk_pointer_ungrab(Bev->time);
  g_object_unref(backpixmap);
  backpixmap = NULL;
}

/****************************************************************************
*  tool_torus_cursor_update
*****************************************************************************/
static void tool_torus_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;
          
  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_torus_register
*****************************************************************************/
GiramTool *giram_tool_torus_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New Torus");
  tool->Icon           = torus_icon;
  tool->Path           = "<ToolBar/Shape3D>";
  tool->ID             = MT_NEW_TORUS;
  tool->OptionsFunc    = NULL;
  tool->button_press   = tool_torus_button_press;
  tool->motion         = tool_torus_motion;
  tool->button_release = tool_torus_button_release;
  tool->cursor_update  = tool_torus_cursor_update;

  return tool;
}

