/* tool_sphere.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "csgtree.h"
#include "snap.h"
#include "utils.h"
#include "view.h"
#include "giramcursor.h"
#include "primitives/sphere.h"
#include "widgets/giramviewshell.h"
#include "pixmaps/sphere.xpm"

#include "tool_sphere.h"

#include "giramintl.h"

static double Xorg, Yorg, Radius;
static guint id;

static GdkPixmap *backpixmap;

#define NONE   1
#define MOVING 2
static int SPHERE_STATE = NONE;

/*****************************************************************************
*  tool_sphere_button_press
******************************************************************************/
static void tool_sphere_button_press(GtkWidget *canvas, GdkEventButton *bevent)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "new sphere");
  MouseToReal(bevent->x, bevent->y, canvas, &Xorg, &Yorg);

  if (backpixmap)
    g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(canvas->window,
                              canvas->allocation.width,
                              canvas->allocation.height,
                              -1);

  draw_all(canvas, backpixmap, view_data->camera_style,
           canvas->allocation.width,
           canvas->allocation.height);
  gdk_window_set_back_pixmap(canvas->window, backpixmap, FALSE);
  gdk_window_clear(canvas->window);

  Radius = 0.0;

  SPHERE_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, _("Sphere, center: "));
  gdk_pointer_grab(canvas->window, FALSE,
                   GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  tool_sphere_motion
******************************************************************************/
static void tool_sphere_motion(GtkWidget *canvas,
                               GdkEventMotion *Mev)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;
  double      Zoom, Xoff, Yoff, Xold, Yold;
  gchar      *Message;
  Vector      Center;

  if (SPHERE_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

  gdk_window_clear(canvas->window);

  MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
  Radius = sqrt((Xold-Xorg)*(Xold-Xorg)+(Yold-Yorg)*(Yold-Yorg));
  gdk_draw_arc(canvas->window, giram_purple_gc, FALSE,
               canvas->allocation.width/2+(Xorg-Radius-Xoff)*Zoom,
               canvas->allocation.height/2-(Yorg+Radius-Yoff)*Zoom,
               Radius*2.0*Zoom, Radius*2.0*Zoom,  0, 64*360);
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(Center, Xorg, Yorg, 0.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(Center, Xorg, 0.0, Yorg);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(Center, 0.0, Yorg, Xorg);
      break;
  }
  Message = g_strdup_printf(_("Sphere, center: <%g, %g, %g>, Radius: %g"),
                            Center[0], Center[1], Center[2], Radius);
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, Message);
  g_free(Message);
}

/*****************************************************************************
*  tool_sphere_button_release
******************************************************************************/
static void tool_sphere_button_release(GtkWidget *canvas, GdkEventButton *Bev)
{
  ViewStruct   *view_data;
  GtkWidget    *status_bar;
  double        Zoom, Xoff, Yoff, Xold, Yold;
  GSList       *tmp_list;
  ViewStruct   *TmpView;
  ObjectStruct *sphere;
  Vector        center;

  if (SPHERE_STATE != MOVING)
    return;
  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);
  MouseToReal(Bev->x, Bev->y, canvas, &Xold, &Yold);
  Radius = sqrt((Xold-Xorg)*(Xold-Xorg)+(Yold-Yorg)*(Yold-Yorg));
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(center, Xorg, Yorg, 0.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(center, Xorg, 0.0, Yorg);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(center, 0.0, Yorg, Xorg);
      break;
  }
  sphere = giram_sphere_new(center, Radius);
  sphere->name = create_uniq_object_name(view_data->frame, _("sphere"));
  view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects,
                                                 sphere);
  sphere->frame = view_data->frame;
  giram_object_build_triangle_mesh(sphere);
  SPHERE_STATE = NONE;

  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(view_data->frame);
  gdk_pointer_ungrab(Bev->time);
  g_object_unref(backpixmap);
  backpixmap = NULL;
}

/****************************************************************************
*  tool_sphere_cursor_update
*****************************************************************************/
static void tool_sphere_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;
          
  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_sphere_register
*****************************************************************************/
GiramTool *giram_tool_sphere_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New Sphere");
  tool->Icon           = sphere_icon;
  tool->Path           = "<ToolBar/Shape3D>";
  tool->ID             = MT_NEW_SPHERE;
  tool->OptionsFunc    = NULL;
  tool->button_press   = tool_sphere_button_press;
  tool->motion         = tool_sphere_motion;
  tool->button_release = tool_sphere_button_release;
  tool->cursor_update  = tool_sphere_cursor_update;

  return tool;
}

