/* tool_select.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <math.h>
#include "giram.h"
#include "object.h"
#include "utils.h"
#include "csgtree.h"
#include "tools.h"
#include "giramcursor.h"

#include "pixmaps/select.xpm"

#include "giramintl.h"

GList *LocalSelection = NULL;
int SelectToggleFlag=0;	/* 0 if select always selects, -1 if selection is toggled */

static void ToggleSelectToggle(GtkWidget *Check);

/*****************************************************************************
*  DestroySelection
******************************************************************************/
void DestroySelection(GList **pselection)
{
  GList        *tmp_sel;
  ObjectStruct *object;

  if (*pselection)
  {
    for (tmp_sel = *pselection ;
         tmp_sel ;
         tmp_sel = g_list_next(tmp_sel) )
    {
      object = tmp_sel->data;
      object->selected = 0;
      CSGTreeUnselect(object);
    }
    g_list_free(*pselection);
    *pselection = NULL;
  }
}

/*****************************************************************************
*  ToggleSelect
******************************************************************************/
static void ToggleSelect(GtkWidget *DrawingArea, GdkEventButton *bev)
{
  ViewStruct      *view_data;
  FrameStruct     *LocalFrame;
  ObjectStruct    *object;
  gdouble          x, y;
  GSList          *tmp_list;
  ViewStruct      *TmpView;
  GList           *Prev;
  GList           *TmpSelection;
  static GList    *PreviousSelection;            /* FIXME: should be one */
  static gdouble   old_x = -10.0, old_y = -10.0; /* per scene/frame */
  gboolean         RebuildLocal;

  view_data = get_current_view_data();
  LocalFrame = view_data->frame;

  RebuildLocal = TRUE;
  if (bev->state & 8)
  { /* Alt is pressed */
    if (LocalSelection && (fabs(bev->x-old_x) < 4.0 /* XXX Should be settable */)
                       && (fabs(bev->y-old_y) < 4.0))
    { /* Just rotate the selection FIXME!!! */
      if (LocalSelection->next)
      {
        LocalSelection = LocalSelection->next;
      } else
      {
        while (LocalSelection->prev)
          LocalSelection = LocalSelection->prev;
      }
      RebuildLocal = FALSE;
    } else
    {
      DestroySelection(&PreviousSelection);
      if (LocalFrame->selection)
      {
        for (Prev = LocalFrame->selection ;
             Prev ;
             Prev = g_list_next(Prev) )
        {
          object = Prev->data;
          PreviousSelection = g_list_append(PreviousSelection, object);
          object->selected = TRUE;
          CSGTreeSelect(object);
        }
      }
    }
  }
  old_x = bev->x;
  old_y = bev->y;
  if (RebuildLocal)
  {
    if (LocalSelection)
    {
      g_list_free(LocalSelection);
      LocalSelection = NULL;
    }
    x = view_data->x_off+(bev->x-DrawingArea->allocation.width/2) / view_data->zoom;
    y = view_data->y_off-(bev->y-DrawingArea->allocation.height/2) / view_data->zoom;
    switch (view_data->camera_style)
    {
      case ORTHO_XY_CAMERA:
        for (tmp_list = LocalFrame->all_objects ;
             tmp_list ;
             tmp_list = g_slist_next(tmp_list))
        {
          object = tmp_list->data;
          if (giram_object_click_in_XY(object, x, y))
            LocalSelection = g_list_append(LocalSelection, object);
        }
        break;

      case ORTHO_XZ_CAMERA:
        for (tmp_list = LocalFrame->all_objects ;
             tmp_list ;
             tmp_list = g_slist_next(tmp_list))
        {
          object = tmp_list->data;
          if (giram_object_click_in_XZ(object, x, y))
            LocalSelection = g_list_append(LocalSelection, object);
        }
        break;

      case ORTHO_ZY_CAMERA:
        for (tmp_list = LocalFrame->all_objects ;
             tmp_list ;
             tmp_list = g_slist_next(tmp_list))
        {
          object = tmp_list->data;
          if (giram_object_click_in_ZY(object, x, y))
            LocalSelection = g_list_append(LocalSelection, object);
        }
        break;
    }
  }
  /* Update the scene selection */
  if (bev->state & 8)
  { /* Alt */
    if (bev->state & 4)
    { /* Alt & Ctrl */
      if (bev->state & 1)
      { /* Alt & Ctrl & Shift */
        /* The Frame Selection is the first object of the Local Selection
         * but only if it was already present in the Previous Selection */
        DestroySelection(&(LocalFrame->selection));
        if (LocalSelection)
        {
          object = LocalSelection->data;
          if (g_list_find(PreviousSelection, object))
          {
            LocalFrame->selection = g_list_append(LocalFrame->selection, object);
            object->selected = TRUE;
            CSGTreeSelect(object);
          }
        }
      } else
      { /* Alt & Ctrl & No Shift */
        /* The Frame Selection is the Previous Selection LESS the first object
         * of the Local Selection */
        DestroySelection(&(LocalFrame->selection));
        if (PreviousSelection)
        {
          for (Prev = PreviousSelection ;
               Prev ;
               Prev = g_list_next(Prev) )
          {
            object = Prev->data;
            LocalFrame->selection = g_list_append(LocalFrame->selection, object);
            object->selected = TRUE;
            CSGTreeSelect(object);
          }
        }
        if (LocalSelection)
        {
          object = LocalSelection->data;
          if (g_list_find(LocalFrame->selection, object))
          {
            LocalFrame->selection = g_list_remove(LocalFrame->selection, object);
            object->selected = FALSE;
            CSGTreeUnselect(object);
          }
        }
      }
    } else
    { /* Alt & No Ctrl */
      if (bev->state & 1)
      { /* Alt & No Ctrl & Shift */
        /* The Frame Selection is the Previous Selection PLUS the first object
         * of the Local Selection */
        DestroySelection(&(LocalFrame->selection));
        if (PreviousSelection)
        {
          for (Prev = PreviousSelection ;
               Prev ;
               Prev = g_list_next(Prev) )
          {
            object = Prev->data;
            LocalFrame->selection = g_list_append(LocalFrame->selection, object);
            object->selected = TRUE;
            CSGTreeSelect(object);
          }
        }
        if (LocalSelection)
        {
          object = LocalSelection->data;
          if (!g_list_find(LocalFrame->selection, object))
          {
            LocalFrame->selection = g_list_append(LocalFrame->selection, object);
            object->selected = TRUE;
            CSGTreeSelect(object);
          }
        }
      } else
      { /* Alt & No Ctrl & No Shift */
        /* The Frame Selection is now the First object of the Local selection */
        DestroySelection(&(LocalFrame->selection));
        if (LocalSelection)
        {
          object = LocalSelection->data;
          LocalFrame->selection = g_list_append(LocalFrame->selection, object);
          object->selected = TRUE;
          CSGTreeSelect(object);
        }
      }
    }
  } else
  { /* No Alt */
    if (bev->state & 4)
    { /* No Alt & Ctrl */
      if (bev->state & 1)
      { /* No Alt & Ctrl & Shift */
        /* The Frame selection is the object that were in the frame
         * selection AND are in the new selection */
        TmpSelection = NULL;
        if (LocalSelection)
        {
          for (Prev = LocalSelection ;
               Prev ;
               Prev = g_list_next(Prev) )
          {
            object = Prev->data;
            if (g_list_find(LocalFrame->selection, object))
            {
              TmpSelection = g_list_append(TmpSelection, object);
              object->selected = TRUE;
              CSGTreeSelect(object);
            }
          }
        }
        DestroySelection(&(LocalFrame->selection));
        LocalFrame->selection = TmpSelection;
      } else
      { /* No Alt & Ctrl & No Shift */
        /* Take the object of the new Selection Out of the Frame Selection */
        if (LocalSelection)
        {
          for (Prev = LocalSelection ;
               Prev ;
               Prev = g_list_next(Prev) )
          {
            object = Prev->data;
            if (g_list_find(LocalFrame->selection, object))
            {
              LocalFrame->selection = g_list_remove(LocalFrame->selection, object);
              object->selected = FALSE;
              CSGTreeUnselect(object);
            }
          }
        }
      }
    } else
    { /* No Alt & No Ctrl */
      if (bev->state & 1)
      { /* No Alt & No Ctrl & Shift */
        /* Add the new selection to the Frame selection */
        if (LocalSelection)
        {
          for (Prev = LocalSelection ;
               Prev ;
               Prev = g_list_next(Prev) )
          {
            object = Prev->data;
            if (!g_list_find(LocalFrame->selection, object))
            {
              LocalFrame->selection = g_list_append(LocalFrame->selection, object);
              object->selected = TRUE;
              CSGTreeSelect(object);
            }
          }
        }
      } else
      { /* No Alt & No Ctrl & No Shift */
        /* If the Select Toggle flag is set to zero, then
	 * the Frame's selection is replaced by the new selection.
	 * Otherwise, the selected object's selection state is toggled.*/
       if (SelectToggleFlag==0) {
          DestroySelection(&(LocalFrame->selection));
          if (LocalSelection)
          {
            for (Prev = LocalSelection ;
                 Prev ;
                 Prev = g_list_next(Prev) )
            {
              object = Prev->data;
              LocalFrame->selection = g_list_append(LocalFrame->selection, object);
              object->selected = TRUE;
              CSGTreeSelect(object);
            }
          }
	} else
        {
	  /* Toggle the select state of the current object */
	  if (LocalSelection)
          {
            object = LocalSelection->data;
	    if (!(object->selected))
            { /* Object not selected. Select it. */
              for (Prev = LocalSelection ;
                   Prev ;
                   Prev = g_list_next(Prev) )
              {
                object = Prev->data;
                if (!g_list_find(LocalFrame->selection, object))
                {
                  LocalFrame->selection = g_list_append(LocalFrame->selection, object);
                  object->selected = TRUE;
                  CSGTreeSelect(object);
                }
              }
	    } else
            { /* Object is selected. Deselect it. */
              for (Prev = LocalSelection ;
                   Prev ;
                   Prev = g_list_next(Prev) )
              {
                object = Prev->data;
                if (g_list_find(LocalFrame->selection, object))
                {
                  LocalFrame->selection = g_list_remove(LocalFrame->selection, object);
                  object->selected = FALSE;
                  CSGTreeUnselect(object);
                }
              }
	    }
	  }
	}
      }
    }
  }

  /* We redraw all the Views */
  for (tmp_list = LocalFrame->all_views ; tmp_list ; tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}
/*****************************************************************************
*  ToggleSelectToggle
* Callback function when toggle select is toggled.
******************************************************************************/
static void ToggleSelectToggle(GtkWidget *Check)
{
  SelectToggleFlag = !SelectToggleFlag;
}

/*****************************************************************************
*  BuildSelectOptions
* Creates the options box for the select tool.
******************************************************************************/
static GtkWidget *BuildSelectOptions(GtkWidget *VBox)
{
  GtkWidget *vbox;
  GtkWidget *Check;
  GtkWidget *label;

  /*  the main vbox  */
  vbox = gtk_vbox_new (FALSE, 1);
  /*  the main label  */
  label = gtk_label_new(_("Select"));
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);
  /* The Toggle Select Check Button */
  Check = gtk_check_button_new_with_label(_("Toggle Mode"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(Check), SelectToggleFlag);
  g_signal_connect(G_OBJECT(Check), "toggled",
                   G_CALLBACK(ToggleSelectToggle), NULL);
  gtk_box_pack_start(GTK_BOX(vbox), Check, FALSE, FALSE, 0);
  gtk_widget_show(Check);

  gtk_box_pack_start(GTK_BOX(VBox), vbox, TRUE, TRUE, 0);
  gtk_widget_show(vbox);
  return vbox;
}

/****************************************************************************
*  tool_select_cursor_update
*****************************************************************************/
static void tool_select_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_select_register
*****************************************************************************/
GiramTool *giram_tool_select_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("Selection");
  tool->Icon           = selection_icon;
  tool->Path           = "<ToolBar>";
  tool->ID             = MT_SELECT;
  tool->OptionsFunc    = BuildSelectOptions;
  tool->button_press   = ToggleSelect;
  tool->motion         = NULL;
  tool->button_release = NULL;
  tool->cursor_update  = tool_select_cursor_update;

  return tool;
}

