/* tool_cylinder.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "primitives/cylinder.h"
#include "csgtree.h"
#include "snap.h"
#include "utils.h"
#include "view.h"
#include "giramcursor.h"
#include "widgets/giramviewshell.h"

#include "pixmaps/cylinder.xpm"

#include "tool_cylinder.h"

#include "giramintl.h"

gboolean CylinderOpenFlag = 0;

double Xorg, Yorg, Xold, Yold;
guint id;

static GdkPixmap *backpixmap = NULL;

#define NONE   1
#define MOVING 2
static int CYLINDER_STATE = NONE;

/*****************************************************************************
*  tool_cylinder_button_press
******************************************************************************/
static void tool_cylinder_button_press(GtkWidget *canvas,
                                       GdkEventButton *bevent)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "new cylinder");
  MouseToReal(bevent->x, bevent->y, canvas, &Xorg, &Yorg);

  if (backpixmap)
    g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(canvas->window,
                              canvas->allocation.width,
                              canvas->allocation.height,
                              -1);

  draw_all(canvas, backpixmap, view_data->camera_style,
           canvas->allocation.width,
           canvas->allocation.height);
  gdk_window_set_back_pixmap(canvas->window, backpixmap, FALSE);
  gdk_window_clear(canvas->window);

  Xold = Xorg; Yold = Yorg;

  CYLINDER_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, _("Cylinder: "));
  gdk_pointer_grab(canvas->window, FALSE,
                   GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  tool_cylinder_motion
******************************************************************************/
static void tool_cylinder_motion(GtkWidget *canvas, GdkEventMotion *Mev)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;
  double      Zoom, Xoff, Yoff;
  gchar      *Message;
  double      x1, y1, x2, y2;
  double      gx1, gy1;
  double      Length, Radius;
  Vector      From;

  if (CYLINDER_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

  gdk_window_clear(canvas->window);

  MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
  if (Xorg < Xold)
  {
    x1 = Xorg-(Xold-Xorg); x2 = Xold;
  } else
  {
    x1 = Xold; x2 = Xorg+Xorg-Xold;
  }
  y1 = MIN(Yorg, Yold); y2 = MAX(Yorg, Yold);
  gdk_draw_rectangle(canvas->window, giram_purple_gc, FALSE,
                     canvas->allocation.width/2+x1*Zoom,
                     canvas->allocation.height/2-y2*Zoom,
                     (x2-x1)*Zoom, (y2-y1)*Zoom);

  gx1 = Xoff+(Xorg-canvas->allocation.width /2)/Zoom;
  Radius = fabs(Xorg-Mev->x)/Zoom;

  gy1 = MIN(Yorg, Yold);
  Length = fabs(Yorg-Yold);
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(From, Xorg, gy1, 0.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(From, Xorg, 0.0, gy1);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(From, 0.0, gy1, Xorg);
      break;
  }
  Message = g_strdup_printf(_("Cylinder: From <%g, %g, %g>, Length: %g, Radius: %g"),
                            From[0], From[1], From[2], Length, Radius);
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, Message);
  g_free(Message);
}

/*****************************************************************************
*  tool_cylinder_button_release
******************************************************************************/
static void tool_cylinder_button_release(GtkWidget *canvas,
                                         GdkEventButton *Bev)
{
  ViewStruct     *view_data;
  GtkWidget      *status_bar;
  double          Zoom, Xoff, Yoff;
  GSList         *tmp_list;
  ViewStruct     *TmpView;
  ObjectStruct   *cylinder;
  CylinderStruct *ccylinder;
  double          gx1, gy1, gy2;
  double          Radius;
  Vector          Base, Apex;

  if (CYLINDER_STATE != MOVING)
    return;
  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(status_bar), id);

  MouseToReal(Bev->x, Bev->y, canvas, &Xold, &Yold);

  gx1 = Xorg;
  Radius = fabs(Xorg-Xold);

  gy1 = MIN(Yorg, Yold);
  gy2 = MAX(Yorg, Yold);

  if ((Radius > 10e-6) && (gy2-gy1 > 10e-6))
  {
    switch (view_data->camera_style)
    {
      case ORTHO_XY_CAMERA:
        Base[0] = gx1; Base[1] = gy1; Base[2] = 0.0;
        Apex[0] = gx1; Apex[1] = gy2; Apex[2] = 0.0;
        break;

      case ORTHO_XZ_CAMERA:
        Base[0] = gx1; Base[1] = 0.0; Base[2] = gy1;
        Apex[0] = gx1; Apex[1] = 0.0; Apex[2] = gy2;
        break;

      case ORTHO_ZY_CAMERA:
        Base[0] = 0.0; Base[1] = gy1; Base[2] = gx1;
        Apex[0] = 0.0; Apex[1] = gy2; Apex[2] = gx1;
        break;
    }
    cylinder = giram_cylinder_new(Base, Apex, Radius);
    cylinder->name = create_uniq_object_name(view_data->frame, _("cylinder"));
    ccylinder = (CylinderStruct *)cylinder;
    ccylinder->Open = CylinderOpenFlag;
    view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects,
                                                   cylinder);
    cylinder->frame = view_data->frame;
    giram_object_build_triangle_mesh(cylinder);
  }
  CYLINDER_STATE = NONE;

  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(view_data->frame);
  gdk_pointer_ungrab(Bev->time);
  g_object_unref(backpixmap);
  backpixmap = NULL;
}

/*****************************************************************************
*  ToggleCylinderOpen
******************************************************************************/
static void ToggleCylinderOpen(GtkWidget *Check)
{
  CylinderOpenFlag = !CylinderOpenFlag;
}

/*****************************************************************************
*  BuildCylinderOptions
******************************************************************************/
static GtkWidget *BuildCylinderOptions(GtkWidget *VBox)
{
  GtkWidget *vbox, *CheckOpen, *label;

  /*  the main vbox  */
  vbox = gtk_vbox_new (FALSE, 1);
  /*  the main label  */
  label = gtk_label_new(_("Cylinder"));
  gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

  CheckOpen = gtk_check_button_new_with_label(_("Open-ended"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CheckOpen), CylinderOpenFlag);
  g_signal_connect(G_OBJECT(CheckOpen), "toggled",
                   G_CALLBACK(ToggleCylinderOpen), NULL);

  gtk_box_pack_start(GTK_BOX(vbox), CheckOpen, FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(VBox), vbox, TRUE, TRUE, 0);
  gtk_widget_show_all(vbox);

  return vbox;
}

/****************************************************************************
*  tool_cylinder_cursor_update
*****************************************************************************/
static void tool_cylinder_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_cylinder_register
*****************************************************************************/
GiramTool *giram_tool_cylinder_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New Cylinder");
  tool->Icon           = cylinder_icon;
  tool->Path           = "<ToolBar/Shape3D>";
  tool->ID             = MT_NEW_CYLINDER;
  tool->OptionsFunc    = BuildCylinderOptions;
  tool->button_press   = tool_cylinder_button_press;
  tool->motion         = tool_cylinder_motion;
  tool->button_release = tool_cylinder_button_release;
  tool->cursor_update  = tool_cylinder_cursor_update;

  return tool;
}

