/* tool_box.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2000 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>

#include "giram.h"
#include "primitives/box.h"
#include "csgtree.h"
#include "snap.h"
#include "utils.h"
#include "view.h"
#include "giramcursor.h"
#include "widgets/giramviewshell.h"

#include "pixmaps/box.xpm"

#include "tool_box.h"

#include "giramintl.h"

double Xorg, Yorg, Xold, Yold;
guint id;

static GdkPixmap *backpixmap = NULL;

#define NONE   1
#define MOVING 2
int BOX_STATE = NONE;

/*****************************************************************************
*  tool_box_button_press
******************************************************************************/
static void tool_box_button_press(GtkWidget *canvas, GdkEventButton *bevent)
{
  ViewStruct *view_data;
  GtkWidget  *status_bar;

  view_data = get_current_view_data();
  status_bar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  id = gtk_statusbar_get_context_id(GTK_STATUSBAR(status_bar), "new box");
  MouseToReal(bevent->x, bevent->y, canvas, &Xorg, &Yorg);

  if (backpixmap)
    g_object_unref(backpixmap);

  backpixmap = gdk_pixmap_new(canvas->window,
                              canvas->allocation.width,
                              canvas->allocation.height,
                              -1);

  draw_all(canvas, backpixmap, view_data->camera_style,
           canvas->allocation.width,
           canvas->allocation.height);
  gdk_window_set_back_pixmap(canvas->window, backpixmap, FALSE);
  gdk_window_clear(canvas->window);

  Xold = Xorg; Yold = Yorg;

  BOX_STATE = MOVING;
  gtk_statusbar_push(GTK_STATUSBAR(status_bar), id, _("Box: "));
  gdk_pointer_grab(canvas->window, FALSE,
                   GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
                   NULL, NULL, bevent->time);
}

/*****************************************************************************
*  tool_box_motion
******************************************************************************/
static void tool_box_motion(GtkWidget *canvas, GdkEventMotion *Mev)
{
  ViewStruct *view_data;
  GtkWidget  *StatusBar;
  double      Zoom, Xoff, Yoff;
  gchar      *Message;
  double      x1, y1, x2, y2;
  Vector      MinCorner, MaxCorner;

  if (BOX_STATE != MOVING)
    return;

  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;
  gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), id);

  gdk_window_clear(canvas->window);

  MouseToReal(Mev->x, Mev->y, canvas, &Xold, &Yold);
  x1 = MIN(Xorg, Xold); x2 = MAX(Xorg, Xold);
  y1 = MIN(Yorg, Yold); y2 = MAX(Yorg, Yold);
  gdk_draw_rectangle(canvas->window, giram_purple_gc, FALSE,
                     canvas->allocation.width/2+(x1-Xoff)*Zoom,
                     canvas->allocation.height/2-(y2-Yoff)*Zoom,
                     (x2-x1)*Zoom, (y2-y1)*Zoom);
  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(MinCorner, x1, y1, 0.0);
      V3Deq(MaxCorner, x2, y2, 1.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(MinCorner, x1, 0.0, y1);
      V3Deq(MaxCorner, x2, 1.0, y2);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(MinCorner, 0.0, y1, x1);
      V3Deq(MaxCorner, 1.0, y2, x2);
      break;
  }
  Message = g_strdup_printf(_("Box: <%g, %g, %g>, <%g, %g, %g>"),
                             MinCorner[0], MinCorner[1], MinCorner[2],
                             MaxCorner[0], MaxCorner[1], MaxCorner[2]);
  gtk_statusbar_push(GTK_STATUSBAR(StatusBar), id, Message);
  g_free(Message);
}

/*****************************************************************************
*  tool_box_button_release
******************************************************************************/
static void tool_box_button_release(GtkWidget *canvas, GdkEventButton *Bev)
{
  ViewStruct   *view_data;
  GtkWidget    *StatusBar;
  double        Zoom, Xoff, Yoff;
  GSList       *tmp_list;
  ViewStruct   *TmpView;
  ObjectStruct *Box;
  Vector        MinCorner, MaxCorner;
  double        gx1, gy1, gx2, gy2;

  if (BOX_STATE != MOVING)
    return;
  view_data = get_current_view_data();
  StatusBar = GIRAM_VIEW_SHELL(view_data->shell)->statusbar;
  Zoom = view_data->zoom;
  Xoff = view_data->x_off;
  Yoff = view_data->y_off;

  gtk_statusbar_pop(GTK_STATUSBAR(StatusBar), id);

  MouseToReal(Bev->x, Bev->y, canvas, &Xold, &Yold);
  gx1 = MIN(Xorg, Xold);
  gx2 = MAX(Xorg, Xold);
  gy1 = MIN(Yorg, Yold);
  gy2 = MAX(Yorg, Yold);

  if ((fabs(gx1-gx2)<1E-6) || (fabs(gy1-gy2)<1E-6))
  {
    BOX_STATE = NONE;
    gdk_pointer_ungrab(Bev->time);
    g_object_unref(backpixmap);
    backpixmap = NULL;
    for (tmp_list = view_data->frame->all_views ;
         tmp_list ;
         tmp_list = tmp_list->next)
    {
      TmpView = tmp_list->data;
      gtk_widget_queue_draw(TmpView->canvas);
    }
    return;
  }

  switch (view_data->camera_style)
  {
    case ORTHO_XY_CAMERA:
      V3Deq(MinCorner, gx1, gy1, 0.0);
      V3Deq(MaxCorner, gx2, gy2, 1.0);
      break;

    case ORTHO_XZ_CAMERA:
      V3Deq(MinCorner, gx1, 0.0, gy1);
      V3Deq(MaxCorner, gx2, 1.0, gy2);
      break;

    case ORTHO_ZY_CAMERA:
      V3Deq(MinCorner, 0.0, gy1, gx1);
      V3Deq(MaxCorner, 1.0, gy2, gx2);
      break;
  }
  Box = giram_box_new(MinCorner, MaxCorner);
  Box->name = create_uniq_object_name(view_data->frame, _("box"));
  view_data->frame->all_objects = g_slist_append(view_data->frame->all_objects, Box);
  Box->frame = view_data->frame;
  giram_object_build_triangle_mesh(Box);

  BOX_STATE = NONE;

  for (tmp_list = view_data->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(view_data->frame);
  gdk_pointer_ungrab(Bev->time);
  g_object_unref(backpixmap);
  backpixmap = NULL;
}

/****************************************************************************
*  tool_box_cursor_update
*****************************************************************************/
static void tool_box_cursor_update(GtkWidget *canvas, guint state)
{
  GdkCursor *cursor;

  cursor = giram_cursor_new(GIRAM_MOUSE_CURSOR,
                            GIRAM_TOOL_CURSOR_NONE,
                            GIRAM_CURSOR_MODIFIER_NONE);
  gdk_window_set_cursor(canvas->window, cursor);
  gdk_cursor_unref(cursor);
}

/****************************************************************************
*  giram_tool_box_register
*****************************************************************************/
GiramTool *giram_tool_box_register(void)
{
  GiramTool *tool;

  tool = g_new(GiramTool, 1);
  tool->ToolTip        = _("New Box");
  tool->Icon           = box_icon;
  tool->Path           = "<ToolBar/Shape3D>";
  tool->ID             = MT_NEW_BOX;
  tool->OptionsFunc    = NULL;
  tool->button_press   = tool_box_button_press;
  tool->motion         = tool_box_motion;
  tool->button_release = tool_box_button_release;
  tool->cursor_update  = tool_box_cursor_update;

  return tool;
}

