/* torus.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "torus.h"
#include "trimesh.h"
#include "csgtree.h"
#include "root_solver.h"


#include "giramintl.h"

static void giram_torus_build_triangle_mesh(ObjectStruct *torus);
static gboolean giram_torus_inside(ObjectStruct *torus, double x, double y, double z);
static gboolean giram_torus_is_intersection(ObjectStruct *torus, Vector org, Vector dir);
static gboolean giram_torus_find_intersection_segment(ObjectStruct *torus,
                                                      Vector in_point, Vector out_point,
                                                      Vector inter_point, Vector inter_norm);

/*****************************************************************************
*  giram_torus_build_triangle_mesh
******************************************************************************/
static void giram_torus_build_triangle_mesh(ObjectStruct *torus)
{ /* XXX: clipped_by */
  int          i, j;
  int          DetailLevel;
  Vector       P1, P2, P3, P4;
  Vector       N1, N2, N3, N4;
  TorusStruct *ttorus = (TorusStruct *)torus;

  DetailLevel = 16; /* XXX should be an internal parameter */

  if (torus->FirstTriangle)
    DestroyObjectTriangleMesh(torus);

  for (i=0 ; i<DetailLevel ; i++)
    for (j=0 ; j<DetailLevel ; j++)
    {
      P1[0] = (ttorus->Major+ttorus->Minor*cos(j*2.0*M_PI/DetailLevel))*cos(i*2.0*M_PI/DetailLevel);
      P1[1] = ttorus->Minor*sin(j*2.0*M_PI/DetailLevel);
      P1[2] = (ttorus->Major+ttorus->Minor*cos(j*2.0*M_PI/DetailLevel))*sin(i*2.0*M_PI/DetailLevel);
      P2[0] = (ttorus->Major+ttorus->Minor*cos((j+1)*2.0*M_PI/DetailLevel))*cos(i*2.0*M_PI/DetailLevel);
      P2[1] = ttorus->Minor*sin((j+1)*2.0*M_PI/DetailLevel);
      P2[2] = (ttorus->Major+ttorus->Minor*cos((j+1)*2.0*M_PI/DetailLevel))*sin(i*2.0*M_PI/DetailLevel);
      P3[0] = (ttorus->Major+ttorus->Minor*cos(j*2.0*M_PI/DetailLevel))*cos((i+1)*2.0*M_PI/DetailLevel);
      P3[1] = ttorus->Minor*sin(j*2.0*M_PI/DetailLevel);
      P3[2] = (ttorus->Major+ttorus->Minor*cos(j*2.0*M_PI/DetailLevel))*sin((i+1)*2.0*M_PI/DetailLevel);
      P4[0] = (ttorus->Major+ttorus->Minor*cos((j+1)*2.0*M_PI/DetailLevel))*cos((i+1)*2.0*M_PI/DetailLevel);
      P4[1] = ttorus->Minor*sin((j+1)*2.0*M_PI/DetailLevel);
      P4[2] = (ttorus->Major+ttorus->Minor*cos((j+1)*2.0*M_PI/DetailLevel))*sin((i+1)*2.0*M_PI/DetailLevel);

      N1[0] = cos(i*2.0*M_PI/DetailLevel)*cos(j*2.0*M_PI/DetailLevel);
      N1[1] = sin(j*2.0*M_PI/DetailLevel);
      N1[2] = sin(i*2.0*M_PI/DetailLevel)*cos(j*2.0*M_PI/DetailLevel);
      N2[0] = cos(i*2.0*M_PI/DetailLevel)*cos((j+1)*2.0*M_PI/DetailLevel);
      N2[1] = sin((j+1)*2.0*M_PI/DetailLevel);
      N2[2] = sin(i*2.0*M_PI/DetailLevel)*cos((j+1)*2.0*M_PI/DetailLevel);
      N3[0] = cos((i+1)*2.0*M_PI/DetailLevel)*cos(j*2.0*M_PI/DetailLevel);
      N3[1] = sin(j*2.0*M_PI/DetailLevel);
      N3[2] = sin((i+1)*2.0*M_PI/DetailLevel)*cos(j*2.0*M_PI/DetailLevel);
      N4[0] = cos((i+1)*2.0*M_PI/DetailLevel)*cos((j+1)*2.0*M_PI/DetailLevel);
      N4[1] = sin((j+1)*2.0*M_PI/DetailLevel);
      N4[2] = sin((i+1)*2.0*M_PI/DetailLevel)*cos((j+1)*2.0*M_PI/DetailLevel);
      if (torus->Trans)
      {
        MEvaluatePoint(P1, torus->Trans, P1);
        MEvaluatePoint(P2, torus->Trans, P2);
        MEvaluatePoint(P3, torus->Trans, P3);
        MEvaluatePoint(P4, torus->Trans, P4);
        MEvaluateVector(N1, torus->Trans, N1);
        MEvaluateVector(N2, torus->Trans, N2);
        MEvaluateVector(N3, torus->Trans, N3);
        MEvaluateVector(N4, torus->Trans, N4);
      }
      AddTriangleToObjectMesh(torus, P1,P2,P3, N1,N2,N3);
      AddTriangleToObjectMesh(torus, P4,P2,P3, N4,N2,N3);
    }
}

/*****************************************************************************
*  giram_torus_inside
******************************************************************************/
static gboolean giram_torus_inside(ObjectStruct *torus, double x, double y, double z)
{ /* XXX: clipped_by */
  TorusStruct *ttorus = (TorusStruct *)torus;
  Vector       vect;
  double       p, pp;

  V3Deq(vect, x, y, z);
  if (torus->Trans)
  {
    MInverseEvaluatePoint(vect, torus->Trans, vect);
  }
  p = sqrt(vect[0]*vect[0] + vect[2]*vect[2]);
  pp = vect[1]*vect[1] + (p-ttorus->Major)*(p-ttorus->Major);
  if (pp <= ttorus->Minor*ttorus->Minor)
    return !(torus->Inverse);
  else
    return torus->Inverse;
}

/*****************************************************************************
*  giram_torus_is_intersection
******************************************************************************/
static gboolean giram_torus_is_intersection(ObjectStruct *torus, Vector origin, Vector direction)
{ /* XXX: clipped_by */
  Vector       org, dir;
  int          i;
  double       R2, r2;
  double       Py2, Dy2, PDy2, k1, k2;
  double       c[5], s[4];
  TorusStruct *ttorus = (TorusStruct *)torus;
  gdouble      length_dir;
  int          num;

  if (torus->Trans)
  {
    MInverseEvaluatePoint(org, torus->Trans, origin);
    MInverseEvaluateVector(dir, torus->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      org[i] = origin[i];
      dir[i] = direction[i];
    }
  }

  length_dir = V3DLength(dir);
  dir[0] /=length_dir;
  dir[1] /=length_dir;
  dir[2] /=length_dir;

  R2   = ttorus->Major * ttorus->Major;
  r2   = ttorus->Minor * ttorus->Minor;

  Py2  = org[1] * org[1];
  Dy2  = dir[1] * dir[1];
  PDy2 = org[1] * dir[1];

  k1   = org[0] * org[0] + org[2] * org[2] + Py2 - R2 - r2;
  k2   = org[0] * dir[0] + org[2] * dir[2] + PDy2;

  c[4] = 1.0;
  c[3] = 4.0 * k2;
  c[2] = 2.0 * (k1 + 2.0 * (k2 * k2 + R2 * Dy2));
  c[1] = 4.0 * (k2 * k1 + 2.0 * R2 * PDy2);
  c[0] = k1 * k1 + 4.0 * R2 * (Py2 - r2);

  num = SolveQuartic(c, s);
  //g_print("%s: %d iter\n", torus->name, num);
  for (i=0; i<num ; i++)
    if (s[i]>0.0001)
      return TRUE;
  return FALSE;
}

/*****************************************************************************
*  giram_torus_find_intersection
******************************************************************************/
static gboolean giram_torus_find_intersection(ObjectStruct *torus,
                                              Vector        origin,
                                              Vector        direction,
                                              Vector        intersection,
                                              Vector        normal)
{
  Vector       org, dir;
  int          i;
  double       R2, r2;
  double       Py2, Dy2, PDy2, k1, k2;
  double       c[5], s[4];
  TorusStruct *ttorus = (TorusStruct *)torus;
  gdouble      Length, length_dir;
  int          num;
  

  if (torus->Trans)
  {
    MInverseEvaluatePoint(org, torus->Trans, origin);
    MInverseEvaluateVector(dir, torus->Trans, direction);
  } else
  {
    V3Dcopy(org, origin);
    V3Dcopy(dir, direction);
  }

  length_dir = V3DLength(dir);
  dir[0] /=length_dir;
  dir[1] /=length_dir;
  dir[2] /=length_dir;
  
  R2   = ttorus->Major * ttorus->Major;
  r2   = ttorus->Minor * ttorus->Minor;

  Py2  = org[1] * org[1];
  Dy2  = dir[1] * dir[1];
  PDy2 = org[1] * dir[1];

  k1   = org[0] * org[0] + org[2] * org[2] + Py2 - R2 - r2;
  k2   = org[0] * dir[0] + org[2] * dir[2] + PDy2;

  c[4] = 1.0;
  c[3] = 4.0 * k2;
  c[2] = 2.0 * (k1 + 2.0 * (k2 * k2 + R2 * Dy2));
  c[1] = 4.0 * (k2 * k1 + 2.0 * R2 * PDy2);
  c[0] = k1 * k1 + 4.0 * R2 * (Py2 - r2);

  num  = SolveQuartic(c, s);

  if (num)
  {
    gdouble nearest=0.0, dist;
    gboolean have_nearest = FALSE;
    Vector M;

    for (i=0 ; i<num ; i++)
    {
      if (s[i] > 0.0)
      {
        if (!have_nearest)
        {
          nearest = s[i];
          have_nearest = TRUE;
        } else
        {
          if (s[i]<nearest)
            nearest = s[i];
        }
      }
    }
    if (!have_nearest)
      return FALSE;
    intersection[0] = org[0] + nearest * dir[0];
    intersection[1] = org[1] + nearest * dir[1];
    intersection[2] = org[2] + nearest * dir[2];

    dist = sqrt(intersection[0]*intersection[0]+intersection[2]*intersection[2]);

    if (dist > 0.00001)
    {
      M[0] = ttorus->Major*intersection[0] / dist;
      M[1] = 0.0;
      M[2] = ttorus->Major*intersection[2] / dist;
    } else
    {
      V3Deq(M, 0.0, 0.0, 0.0);
    }
    
    normal[0]  = intersection[0]-M[0];
    normal[1]  = intersection[1]-M[1];
    normal[2]  = intersection[2]-M[2];

    /* transform back the intersection point and the normal to the scene space */
    if (torus->Trans)
    {
      MEvaluatePoint(intersection, torus->Trans, intersection);
      MEvaluateVector(normal, torus->Trans, normal);
    }

    Length = V3DLength(normal);
    normal[0] /= Length;
    normal[1] /= Length;
    normal[2] /= Length;
    return TRUE;
  }
  return FALSE;
}

/*****************************************************************************
*  giram_torus_find_intersection_segment FIXME
******************************************************************************/
static gboolean giram_torus_find_intersection_segment(ObjectStruct *torus,
                                                      Vector in_point, Vector out_point,
                                                      Vector inter_point, Vector inter_norm)
{
  return FALSE;
}

/*****************************************************************************
*  giram_torus_new
******************************************************************************/
ObjectStruct *giram_torus_new(double major_radius, double minor_radius)
{
  ObjectStruct *torus;
  TorusStruct  *ttorus;
  static GiramObjectClass *torus_klass = NULL;

  if (torus_klass == NULL)
  {
    torus_klass = giram_object_class_new();

    torus_klass->name                      = _("Torus");
    torus_klass->build_triangle_mesh       = giram_torus_build_triangle_mesh;
    torus_klass->inside                    = giram_torus_inside;
    torus_klass->is_intersection           = giram_torus_is_intersection;
    torus_klass->find_intersection_segment = giram_torus_find_intersection_segment;
    torus_klass->find_intersection         = giram_torus_find_intersection;
  }
  ttorus = g_new(TorusStruct, 1);
  torus = (ObjectStruct *)ttorus;
  InitObject(torus);
  torus->Type = TORUS_OBJECT;
  torus->klass = torus_klass;
  ttorus->Major = major_radius;
  ttorus->Minor = minor_radius;
  return torus;
}

/*****************************************************************************
*  PDBCreateTorus
******************************************************************************/
/*int PDBCreateTorus(int *Id, double *MajorRadius, double *MinorRadius)
{
  ObjectStruct *Torus;
  FrameStruct  *LocalFrame;
  GSList       *tmp_list;
  ViewStruct   *TmpView;

  Torus = giram_torus_new(*MajorRadius, *MinorRadius);

  LocalFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    LocalFrame = tmp_list->data;
    if (LocalFrame->Id == *Id)
      break;
  }
  if (LocalFrame == NULL)
  {
    g_message(_("PDBCreateTorus() called with an unknown Id (%d)\n"), *Id);
    return 0;
  }
  LocalFrame->all_objects = g_slist_append(LocalFrame->all_objects, Torus);
  giram_object_build_triangle_mesh(Torus);
  for (tmp_list = LocalFrame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->drawing_area);
  }
  RebuildCSGTree(NULL, LocalFrame);
  return Torus->Id;
}*/

