/* sphere.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "sphere.h"
#include "trimesh.h"
#include "csgtree.h"


#include "giramintl.h"

static void giram_sphere_build_triangle_mesh(ObjectStruct *sphere);
static gboolean giram_sphere_inside(ObjectStruct *sphere, double x, double y, double z);
static gboolean giram_sphere_is_intersection(ObjectStruct *sphere, Vector org, Vector dir);
static gboolean giram_sphere_click_in_XY(ObjectStruct *sphere, double x, double y);
static gboolean giram_sphere_click_in_XZ(ObjectStruct *sphere, double x, double z);
static gboolean giram_sphere_click_in_ZY(ObjectStruct *sphere, double z, double y);
static gboolean giram_sphere_find_intersection_segment(ObjectStruct *sphere,
                                                       Vector in_point, Vector out_point,
                                                       Vector inter_point, Vector inter_norm);
static gboolean giram_sphere_find_intersection(ObjectStruct *sphere,
                                               Vector        origin,
                                               Vector        direction,
                                               Vector        intersection,
                                               Vector        normal);

/*****************************************************************************
*  giram_sphere_mesh_normal
******************************************************************************/
static void giram_sphere_mesh_normal(Vector result, ObjectStruct *sphere, Vector inter)
{
  double norme;

  result[0] = inter[0] - ((SphereStruct *)sphere)->Center[0];
  result[1] = inter[1] - ((SphereStruct *)sphere)->Center[1];
  result[2] = inter[2] - ((SphereStruct *)sphere)->Center[2];
  norme = V3DLength(result);
  result[0] /= norme;
  result[1] /= norme;
  result[2] /= norme;
  if (sphere->Inverse)
  {
    result[0] = -result[0];
    result[1] = -result[1];
    result[2] = -result[2];
  }
}

/*****************************************************************************
*  giram_sphere_build_triangle_mesh
******************************************************************************/
static void giram_sphere_build_triangle_mesh(ObjectStruct *sphere)
{ /* XXX: clipped_by */
  Vector p1, p2, p3, p4;
  Vector n1, n2, n3, n4;
  int i, j;
  int DetailLevel;
  SphereStruct *ssphere = (SphereStruct *)sphere;

  DetailLevel = 16; /* FIXME: should be a private field */

  if (sphere->FirstTriangle)
    DestroyObjectTriangleMesh(sphere);
  /* First the North Pole */
  p1[0] = ssphere->Center[0];
  p1[1] = ssphere->Center[1]+ssphere->Radius;
  p1[2] = ssphere->Center[2];
  giram_sphere_mesh_normal(n1, sphere, p1);
  if (sphere->Trans)
  {
    MEvaluatePoint(p1, sphere->Trans, p1);
    MEvaluateVector(n1, sphere->Trans, n1);
  }
  for (i=0; i<DetailLevel; i++)
  { /* The "DetailLevel" triangles adjacents to the pole */
    p2[0] = ssphere->Center[0]+ssphere->Radius*sin(M_PI/DetailLevel)
                                              *cos(i*2.0*M_PI/DetailLevel);
    p2[1] = ssphere->Center[1]+ssphere->Radius*cos(M_PI/DetailLevel);
    p2[2] = ssphere->Center[2]+ssphere->Radius*sin(M_PI/DetailLevel)
                                              *sin(i*2.0*M_PI/DetailLevel);
    giram_sphere_mesh_normal(n2, sphere, p2);
    if (sphere->Trans)
    {
      MEvaluatePoint(p2, sphere->Trans, p2);
      MEvaluateVector(n2, sphere->Trans, n2);
    }
    p3[0] = ssphere->Center[0]+ssphere->Radius*sin(M_PI/DetailLevel)
                                              *cos((i+1)*2.0*M_PI/DetailLevel);
    p3[1] = ssphere->Center[1]+ssphere->Radius*cos(M_PI/DetailLevel);
    p3[2] = ssphere->Center[2]+ssphere->Radius*sin(M_PI/DetailLevel)
                                              *sin((i+1)*2.0*M_PI/DetailLevel);
    giram_sphere_mesh_normal(n3, sphere, p3);
    if (sphere->Trans)
    {
      MEvaluatePoint(p3, sphere->Trans, p3);
      MEvaluateVector(n3, sphere->Trans, n3);
    }
    AddTriangleToObjectMesh(sphere, p1,p2,p3, n1,n2,n3);
  }
  /* Then the Mid Zone */
  for (i=1 ; i<DetailLevel-1 ; i++)
  {
    for (j=0 ; j<DetailLevel ; j++)
    {
      p1[0] = ssphere->Center[0]+ssphere->Radius*sin(i*M_PI/DetailLevel)
                                                *cos(j*2.0*M_PI/DetailLevel);
      p1[1] = ssphere->Center[1]+ssphere->Radius*cos(i*M_PI/DetailLevel);
      p1[2] = ssphere->Center[2]+ssphere->Radius*sin(i*M_PI/DetailLevel)
                                                *sin(j*2.0*M_PI/DetailLevel);
      giram_sphere_mesh_normal(n1, sphere, p1);
      if (sphere->Trans)
      {
        MEvaluatePoint(p1, sphere->Trans, p1);
        MEvaluateVector(n1, sphere->Trans, n1);
      }
      p2[0] = ssphere->Center[0]+ssphere->Radius*sin((i+1)*M_PI/DetailLevel)
                                                *cos(j*2.0*M_PI/DetailLevel);
      p2[1] = ssphere->Center[1]+ssphere->Radius*cos((i+1)*M_PI/DetailLevel);
      p2[2] = ssphere->Center[2]+ssphere->Radius*sin((i+1)*M_PI/DetailLevel)
                                                *sin(j*2.0*M_PI/DetailLevel);
      giram_sphere_mesh_normal(n2, sphere, p2);
      if (sphere->Trans)
      {
        MEvaluatePoint(p2, sphere->Trans, p2);
        MEvaluateVector(n2, sphere->Trans, n2);
      }
      p3[0] = ssphere->Center[0]+ssphere->Radius*sin(i*M_PI/DetailLevel)
                                              *cos((j+1)*2.0*M_PI/DetailLevel);
      p3[1] = ssphere->Center[1]+ssphere->Radius*cos(i*M_PI/DetailLevel);
      p3[2] = ssphere->Center[2]+ssphere->Radius*sin(i*M_PI/DetailLevel)
                                              *sin((j+1)*2.0*M_PI/DetailLevel);
      giram_sphere_mesh_normal(n3, sphere, p3);
      if (sphere->Trans)
      {
        MEvaluatePoint(p3, sphere->Trans, p3);
        MEvaluateVector(n3, sphere->Trans, n3);
      }
      p4[0] = ssphere->Center[0]+ssphere->Radius*sin((i+1)*M_PI/DetailLevel)
                                              *cos((j+1)*2.0*M_PI/DetailLevel);
      p4[1] = ssphere->Center[1]+ssphere->Radius*cos((i+1)*M_PI/DetailLevel);
      p4[2] = ssphere->Center[2]+ssphere->Radius*sin((i+1)*M_PI/DetailLevel)
                                              *sin((j+1)*2.0*M_PI/DetailLevel);
      giram_sphere_mesh_normal(n4, sphere, p4);
      if (sphere->Trans)
      {
        MEvaluatePoint(p4, sphere->Trans, p4);
        MEvaluateVector(n4, sphere->Trans, n4);
      }
      AddTriangleToObjectMesh(sphere, p1,p2,p4, n1,n2,n4);
      AddTriangleToObjectMesh(sphere, p1,p3,p4, n1,n3,n4);
    }
  }
  /* And the South Pole */
  p1[0] = ssphere->Center[0];
  p1[1] = ssphere->Center[1]-ssphere->Radius;
  p1[2] = ssphere->Center[2];
  giram_sphere_mesh_normal(n1, sphere, p1);
  if (sphere->Trans)
  {
    MEvaluatePoint(p1, sphere->Trans, p1);
    MEvaluateVector(n1, sphere->Trans, n1);
  }
  for (i=0 ; i<DetailLevel ; i++)
  { /* The "DetailLevel" triangles adjacents to the south pole */
    p2[0] = ssphere->Center[0]+ssphere->Radius*sin(M_PI/DetailLevel)
                                              *cos(i*2.0*M_PI/DetailLevel);
    p2[1] = ssphere->Center[1]-ssphere->Radius*cos(M_PI/DetailLevel);
    p2[2] = ssphere->Center[2]+ssphere->Radius*sin(M_PI/DetailLevel)
                                              *sin(i*2.0*M_PI/DetailLevel);
    giram_sphere_mesh_normal(n2, sphere, p2);
    if (sphere->Trans)
    {
      MEvaluatePoint(p2, sphere->Trans, p2);
      MEvaluateVector(n2, sphere->Trans, n2);
    }
    p3[0] = ssphere->Center[0]+ssphere->Radius*sin(M_PI/DetailLevel)
                                              *cos((i+1)*2.0*M_PI/DetailLevel);
    p3[1] = ssphere->Center[1]-ssphere->Radius*cos(M_PI/DetailLevel);
    p3[2] = ssphere->Center[2]+ssphere->Radius*sin(M_PI/DetailLevel)
                                              *sin((i+1)*2.0*M_PI/DetailLevel);
    giram_sphere_mesh_normal(n3, sphere, p3);
    if (sphere->Trans)
    {
      MEvaluatePoint(p3, sphere->Trans, p3);
      MEvaluateVector(n3, sphere->Trans, n3);
    }
    AddTriangleToObjectMesh(sphere, p1,p2,p3, n1,n2,n3);
  }
}

/*****************************************************************************
*  giram_sphere_inside
******************************************************************************/
static gboolean giram_sphere_inside(ObjectStruct *sphere, double x, double y, double z)
{ /* XXX: clipped_by */
  Vector vect;

  V3Deq(vect, x, y, z);
  if (sphere->Trans)
  {
    MInverseEvaluatePoint(vect, sphere->Trans, vect);
  }
  vect[0] -= ((SphereStruct *)sphere)->Center[0];
  vect[1] -= ((SphereStruct *)sphere)->Center[1];
  vect[2] -= ((SphereStruct *)sphere)->Center[2];
  if (vect[0]*vect[0]+vect[1]*vect[1]+vect[2]*vect[2] >
           (((SphereStruct *)sphere)->Radius *
            ((SphereStruct *)sphere)->Radius))
    return sphere->Inverse;
  else
    return !(sphere->Inverse);
}

/*****************************************************************************
*  giram_sphere_is_intersection
******************************************************************************/
static gboolean giram_sphere_is_intersection(ObjectStruct *sphere, Vector origin, Vector direction)
{ /* XXX: clipped_by */
  Vector        org, dir;
  Vector        OriginToCenter;
  int           i;
  double        Length;
  double        Tca, thc, D2;
  SphereStruct *ssphere = (SphereStruct *)sphere;

  if (sphere->Trans)
  {
    MInverseEvaluatePoint(org, sphere->Trans, origin);
    MInverseEvaluateVector(dir, sphere->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      org[i] = origin[i];
      dir[i] = direction[i];
    }
  }
  OriginToCenter[0] = ssphere->Center[0] - org[0];
  OriginToCenter[1] = ssphere->Center[1] - org[1];
  OriginToCenter[2] = ssphere->Center[2] - org[2];
  Length = V3DLength(dir);
  dir[0] /= Length; dir[1] /= Length; dir[2] /= Length;
  Tca = V3DDot(OriginToCenter, dir);
  D2 = V3DDot(OriginToCenter, OriginToCenter) - Tca*Tca;
  thc =    ssphere->Radius * ssphere->Radius - D2;
  
  if (V3DLength(OriginToCenter) < ssphere->Radius)
  { 
    /* we're inside the sphere */
    /* Dealing wiht precision pb*/
    if(Tca+sqrt(thc)<(0.00001*ssphere->Radius))
      return FALSE;
  }
  else
  { /* The origin is outside the sphere */
    if (Tca<0)
    { /* The Ray is pointing away from the sphere */
      return FALSE;
    }
   else
     {   
       if (thc < 0)
	 return FALSE;
       else
	 {
	   if(Tca-sqrt(thc)<(0.00001*ssphere->Radius))
	     return FALSE;
	 }
    }
  }
 
  return TRUE;
}

/*****************************************************************************
*  giram_sphere_click_in_XY
******************************************************************************/
static gboolean giram_sphere_click_in_XY(ObjectStruct *sphere, double x, double y)
{ /* XXX: clipped_by */
  Vector        origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector        direction = {0.0, 0.0, 1.0, 0.0, 0.0};
  SphereStruct *ssphere = (SphereStruct *)sphere;

  if (sphere->Trans)
  {
    origin[0] = x;
    origin[1] = y;
    return giram_sphere_is_intersection(sphere, origin, direction);
  } else if ( (x-ssphere->Center[0])*(x-ssphere->Center[0])+
              (y-ssphere->Center[1])*(y-ssphere->Center[1]) <
              ssphere->Radius * ssphere->Radius )
    return TRUE;
  else
    return FALSE;
}

/*****************************************************************************
*  giram_sphere_click_in_XZ
******************************************************************************/
static gboolean giram_sphere_click_in_XZ(ObjectStruct *sphere, double x, double z)
{ /* XXX: clipped_by */
  Vector        origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector        direction = {0.0, 1.0, 0.0, 0.0, 0.0};
  SphereStruct *ssphere = (SphereStruct *)sphere;

  if (sphere->Trans)
  {
    origin[0] = x;
    origin[2] = z;
    return giram_sphere_is_intersection(sphere, origin, direction);
  } else if ( (x-ssphere->Center[0])*(x-ssphere->Center[0])+
              (z-ssphere->Center[2])*(z-ssphere->Center[2]) <
              ssphere->Radius * ssphere->Radius )
    return TRUE;
  else
    return FALSE;
}

/*****************************************************************************
*  giram_sphere_click_in_ZY
******************************************************************************/
static gboolean giram_sphere_click_in_ZY(ObjectStruct *sphere, double z, double y)
{ /* XXX: clipped_by */
  Vector        origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector        direction = {1.0, 0.0, 0.0, 0.0, 0.0};
  SphereStruct *ssphere = (SphereStruct *)sphere;

  if (sphere->Trans)
  {
    origin[2] = z;
    origin[1] = y;
    return giram_sphere_is_intersection(sphere, origin, direction);
  } else if ( (z-ssphere->Center[2])*(z-ssphere->Center[2])+
              (y-ssphere->Center[1])*(y-ssphere->Center[1]) <
              ssphere->Radius * ssphere->Radius )
    return TRUE;
  else
    return FALSE;
}

/*****************************************************************************
*  giram_sphere_find_intersection_segment
******************************************************************************/
static gboolean giram_sphere_find_intersection_segment(ObjectStruct *sphere,
                                                       Vector in_point, Vector out_point,
                                                       Vector inter_point, Vector inter_norm)
{
  Vector        InP, OutP, realin, realout;
  Vector        Vd;
  double        Norme;
  double        B, C, t;
  SphereStruct *ssphere = (SphereStruct *)sphere;
  int           i;

  if (giram_object_inside(sphere, out_point[0], out_point[1], out_point[2]))
  {
    realin[0] = out_point[0]; realin[1] = out_point[1]; realin[2] = out_point[2];
    realout[0] = in_point[0]; realout[1] = in_point[1]; realout[2] = in_point[2];    
  } else
  {
    realout[0] = out_point[0]; realout[1] = out_point[1]; realout[2] = out_point[2];
    realin[0] = in_point[0];   realin[1] = in_point[1];   realin[2] = in_point[2];    
  }

  if (sphere->Trans)
  {
    MInverseEvaluatePoint(InP, sphere->Trans, realin);
    MInverseEvaluatePoint(OutP, sphere->Trans, realout);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      InP[i] = realin[i];
      OutP[i] = realout[i];
    }
  }
  Vd[0] = InP[0] - OutP[0]; Vd[1] = InP[1] - OutP[1]; Vd[2] = InP[2] - OutP[2];
  Norme = V3DLength(Vd);
  Vd[0] /= Norme; Vd[1] /= Norme; Vd[2] /= Norme;
  B = 2 * ( Vd[0]*(OutP[0]-ssphere->Center[0])+
            Vd[1]*(OutP[1]-ssphere->Center[1])+
            Vd[2]*(OutP[2]-ssphere->Center[2]));
  C = (OutP[0]-ssphere->Center[0])*(OutP[0]-ssphere->Center[0]) +
      (OutP[1]-ssphere->Center[1])*(OutP[1]-ssphere->Center[1]) +
      (OutP[2]-ssphere->Center[2])*(OutP[2]-ssphere->Center[2])
      - ssphere->Radius * ssphere->Radius;
  t = (- B - sqrt(B*B - 4.0*C)) / 2.0;
  if (t >= 0.0 && t < Norme)
  {
    inter_point[0] = OutP[0]+t*Vd[0];
    inter_point[1] = OutP[1]+t*Vd[1];
    inter_point[2] = OutP[2]+t*Vd[2];
    giram_sphere_mesh_normal(inter_norm, sphere, inter_point);
    if (sphere->Trans)
    {
      MEvaluatePoint(inter_point, sphere->Trans, inter_point);
      MEvaluateVector(inter_norm, sphere->Trans, inter_norm);
    }
    return TRUE;
  }
  t = (- B + sqrt(B*B - 4.0*C)) / 2.0;
  if (t >= 0.0 && t < Norme)
  {
    inter_point[0] = OutP[0]+t*Vd[0];
    inter_point[1] = OutP[1]+t*Vd[1];
    inter_point[2] = OutP[2]+t*Vd[2];
    giram_sphere_mesh_normal(inter_norm, sphere, inter_point);
    if (sphere->Trans)
    {
      MEvaluatePoint(inter_point, sphere->Trans, inter_point);
      MEvaluateVector(inter_norm, sphere->Trans, inter_norm);
    }
    return TRUE;
  }

  return FALSE;
}

/*****************************************************************************
*  giram_sphere_find_intersection
******************************************************************************/
static gboolean giram_sphere_find_intersection(ObjectStruct *sphere,
                                               Vector        origin,
                                               Vector        direction,
                                               Vector        intersection,
                                               Vector        normal)
{
  Vector        org, dir;
  Vector        OriginToCenter;
  int           i;
  double        Length;
  double        Tca;
  SphereStruct *ssphere = (SphereStruct *)sphere;
  gdouble       D2,thc;

  if (sphere->Trans)
  {
    MInverseEvaluatePoint(org, sphere->Trans, origin);
    MInverseEvaluateVector(dir, sphere->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      org[i] = origin[i];
      dir[i] = direction[i];
    }
  }
  OriginToCenter[0] = ssphere->Center[0] - org[0];
  OriginToCenter[1] = ssphere->Center[1] - org[1];
  OriginToCenter[2] = ssphere->Center[2] - org[2];

  Length = V3DLength(dir);
  dir[0] /= Length; dir[1] /= Length; dir[2] /= Length;
  Tca = V3DDot(OriginToCenter, dir);

  D2 = V3DDot(OriginToCenter, OriginToCenter) - Tca*Tca;
  thc =    ssphere->Radius * ssphere->Radius - D2;
  
  if (V3DLength(OriginToCenter) < ssphere->Radius)
  { /* we're inside the sphere */
    /* Dealing wiht precision pb*/
    if(Tca+sqrt(thc)>(0.00001*ssphere->Radius))
     {
       intersection[0] = org[0] + dir[0] * (Tca+sqrt(thc));
       intersection[1] = org[1] + dir[1] * (Tca+sqrt(thc));
       intersection[2] = org[2] + dir[2] * (Tca+sqrt(thc));

       normal[0] = ssphere->Center[0] - intersection[0];
       normal[1] = ssphere->Center[1] - intersection[1];
       normal[2] = ssphere->Center[2] - intersection[2];
     }
    else
      return FALSE;
  } 
  else
  { /* The origin is outside the sphere */
    if (Tca<0)
    { /* The Ray is pointing away from the sphere */
      return FALSE;
    }
   else
     {   
       if (thc < 0)
	 return FALSE;
       else
	 {
	   if(Tca-sqrt(thc)>0.0001*ssphere->Radius)
	     {
	       intersection[0] = org[0] + dir[0] * (Tca-sqrt(thc));
	       intersection[1] = org[1] + dir[1] * (Tca-sqrt(thc));
	       intersection[2] = org[2] + dir[2] * (Tca-sqrt(thc));
	 
	       normal[0] = intersection[0] - ssphere->Center[0];
	       normal[1] = intersection[1] - ssphere->Center[1];
	       normal[2] = intersection[2] - ssphere->Center[2];
	     }
	   else
	     return FALSE;
	 }
    }
  }

  /* transform back the intersection point and the normal to the scene space */
  if (sphere->Trans)
  {
    MEvaluatePoint(intersection, sphere->Trans, intersection);
    MEvaluateVector(normal, sphere->Trans, normal);
  }

  Length = V3DLength(normal);
  normal[0] /= Length;
  normal[1] /= Length;
  normal[2] /= Length;      

  return TRUE;
}

/*****************************************************************************
*  giram_sphere_new
******************************************************************************/
ObjectStruct *giram_sphere_new(Vector center, double radius)
{
  ObjectStruct *sphere;
  SphereStruct *ssphere;
  static GiramObjectClass *sphere_klass = NULL;

  if (sphere_klass == NULL)
  {
    sphere_klass = giram_object_class_new();

    sphere_klass->name                      = _("Sphere");
    sphere_klass->build_triangle_mesh       = giram_sphere_build_triangle_mesh;
    sphere_klass->inside                    = giram_sphere_inside;
    sphere_klass->is_intersection           = giram_sphere_is_intersection;
    sphere_klass->click_in_XY               = giram_sphere_click_in_XY;
    sphere_klass->click_in_XZ               = giram_sphere_click_in_XZ;
    sphere_klass->click_in_ZY               = giram_sphere_click_in_ZY;
    sphere_klass->find_intersection_segment = giram_sphere_find_intersection_segment;
    sphere_klass->find_intersection         = giram_sphere_find_intersection;
  }
  ssphere = g_new(SphereStruct, 1);
  sphere = (ObjectStruct *)ssphere;
  InitObject(sphere);
  sphere->Type = SPHERE_OBJECT;
  sphere->klass = sphere_klass;
  V3Dcopy(ssphere->Center, center);
  ssphere->Radius = radius;
  return sphere;
}

/*****************************************************************************
*  PDBCreateSphere
******************************************************************************/
/*int PDBCreateSphere(int *Id, Vector *Center, double *Radius)
{
  ObjectStruct *sphere;
  FrameStruct  *LocalFrame;
  GSList       *tmp_list;
  ViewStruct   *TmpView;

  sphere = giram_sphere_new(*Center, *Radius);

  LocalFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    LocalFrame = tmp_list->data;
    if (LocalFrame->Id == *Id)
      break;
  }
  if (LocalFrame == NULL)
  {
    g_message(_("PDBCreateSphere() called with an unknown Id (%d)\n"), *Id);
    return 0;
  }
  LocalFrame->all_objects = g_slist_append(LocalFrame->all_objects, sphere);
  giram_object_build_triangle_mesh(sphere);
  for (tmp_list = LocalFrame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->drawing_area);
  }
  RebuildCSGTree(NULL, LocalFrame);
  return sphere->Id;
}*/

