/* plane.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "plane.h"
#include "trimesh.h"
#include "csgtree.h"

#include "giramintl.h"


static void giram_plane_build_triangle_mesh(ObjectStruct *plane);
static gboolean giram_plane_inside(ObjectStruct *plane, double x, double y, double z);
static gboolean giram_plane_find_intersection_segment(ObjectStruct *plane,
                                                      Vector in_point, Vector out_point,
                                                      Vector inter_point, Vector inter_norm);

/*****************************************************************************
*  giram_plane_build_triangle_mesh
******************************************************************************/
static void giram_plane_build_triangle_mesh(ObjectStruct *plane)
{ /* XXX: clipped_by */
  Vector       CentralPoint;
  Vector       U1, U2;
  double       ULength;
  Vector       P1, P2, P3, P4;
  Vector       N;
  int          i,j;
  PlaneStruct *pplane = (PlaneStruct *)plane;

  if (plane->FirstTriangle)
    DestroyObjectTriangleMesh(plane);
  V3Dcopy(N, pplane->Normal);
  if (plane->Inverse)
  {
    N[0] = -N[0];
    N[1] = -N[1];
    N[2] = -N[2];
  }
  CentralPoint[0] = pplane->Normal[0]*pplane->Distance;
  CentralPoint[1] = pplane->Normal[1]*pplane->Distance;
  CentralPoint[2] = pplane->Normal[2]*pplane->Distance;

  if ( (fabs(N[0])>fabs(N[1])) && (fabs(N[0])>fabs(N[2])) )
  { /* The plane normal is upon the X axis */
    U1[0] = -N[1]/N[0];
    U1[1] = 1;
    U1[2] = 0;
  } else if ( (fabs(N[1])>fabs(N[0])) && (fabs(N[1])>fabs(N[2])) )
  { /* The plane normal is upon the Y axis */
    U1[0] = 1;
    U1[1] = -N[0]/N[1];
    U1[2] = 0;
  } else
  { /* The plane normal is upon the Z axis */
    U1[0] = 0;
    U1[1] = 1;
    U1[2] = -N[1]/N[2];
  }
  ULength = V3DLength(U1);
  U1[0]/=ULength; U1[1]/=ULength; U1[2]/=ULength;
  VCross(U2, N, U1);
  ULength = V3DLength(U2);
  U2[0]/=ULength; U2[1]/=ULength; U2[2]/=ULength;

  if (plane->Trans)
  {
    MEvaluateVector(N, plane->Trans, N);
  }

  for (i=-pplane->NbSquare; i<pplane->NbSquare ;i++)
    for (j=-pplane->NbSquare; j<pplane->NbSquare ;j++)
    {
      P1[0] = CentralPoint[0]+pplane->SquareSize*(i*U1[0]+j*U2[0]);
      P1[1] = CentralPoint[1]+pplane->SquareSize*(i*U1[1]+j*U2[1]);
      P1[2] = CentralPoint[2]+pplane->SquareSize*(i*U1[2]+j*U2[2]);

      P2[0] = CentralPoint[0]+pplane->SquareSize*((i+1)*U1[0]+j*U2[0]);
      P2[1] = CentralPoint[1]+pplane->SquareSize*((i+1)*U1[1]+j*U2[1]);
      P2[2] = CentralPoint[2]+pplane->SquareSize*((i+1)*U1[2]+j*U2[2]);

      P3[0] = CentralPoint[0]+pplane->SquareSize*(i*U1[0]+(j+1)*U2[0]);
      P3[1] = CentralPoint[1]+pplane->SquareSize*(i*U1[1]+(j+1)*U2[1]);
      P3[2] = CentralPoint[2]+pplane->SquareSize*(i*U1[2]+(j+1)*U2[2]);

      P4[0] = CentralPoint[0]+pplane->SquareSize*((i+1)*U1[0]+(j+1)*U2[0]);
      P4[1] = CentralPoint[1]+pplane->SquareSize*((i+1)*U1[1]+(j+1)*U2[1]);
      P4[2] = CentralPoint[2]+pplane->SquareSize*((i+1)*U1[2]+(j+1)*U2[2]);

      if (plane->Trans)
      {
        MEvaluatePoint(P1, plane->Trans, P1);
        MEvaluatePoint(P2, plane->Trans, P2);
        MEvaluatePoint(P3, plane->Trans, P3);
        MEvaluatePoint(P4, plane->Trans, P4);
      }
      AddTriangleToObjectMesh(plane, P1, P2, P3, N,N,N);
      AddTriangleToObjectMesh(plane, P2, P3, P4, N,N,N);
    }
}

/*****************************************************************************
*  giram_plane_inside
******************************************************************************/
static gboolean giram_plane_inside(ObjectStruct *plane, double x, double y, double z)
{ /* XXX: clipped_by */
  double       a;
  Vector       Vect;
  PlaneStruct *pplane = (PlaneStruct *)plane;

  V3Deq(Vect, x, y, z);
  if (plane->Trans)
  {
    MInverseEvaluatePoint(Vect, plane->Trans, Vect);
  }
  a  = V3DDot(Vect, pplane->Normal);
  if (a > pplane->Distance)
    return plane->Inverse;
  else
    return !(plane->Inverse);
}

/*****************************************************************************
*  giram_plane_find_intersection_segment
******************************************************************************/
static gboolean giram_plane_find_intersection_segment(ObjectStruct *plane,
                                                      Vector in_point, Vector out_point,
                                                      Vector inter_point, Vector inter_norm)
{
  return FALSE; /*  FIXME */
}

/*****************************************************************************
*  giram_plane_is_intersection
******************************************************************************/
static gboolean giram_plane_is_intersection(ObjectStruct *plane,
                                            Vector        origin,
                                            Vector        direction)
{
  Vector       org, dir, normal;
  double       Length;
  PlaneStruct *pplane = (PlaneStruct *)plane;
  gdouble      vd, v0, t;

  /* go to the plane space */
  if (plane->Trans)
  {
    MInverseEvaluatePoint(org, plane->Trans, origin);
    MInverseEvaluateVector(dir, plane->Trans, direction);
  } else
  {
    V3Dcopy(org, origin);
    V3Dcopy(dir, direction);
  }

  /* algorithm taken in Glassner,  p51  */
  V3Dcopy(normal, pplane->Normal); /* FIXME should be normal */
  Length = V3DLength(normal);
  normal[0] /= Length;
  normal[1] /= Length;
  normal[2] /= Length;

  vd = V3DDot(normal, dir);
  if (fabs(vd)<0.000001)
    return FALSE;

  v0 = -(V3DDot(normal,org)-pplane->Distance);

  t = v0/vd;

  if (t<0.000001)
    return FALSE; /* the intersection point is behind the camera  */

  return TRUE;
}
/*****************************************************************************
*  giram_plane_find_intersection
******************************************************************************/
static gboolean giram_plane_find_intersection(ObjectStruct *plane,
                                              Vector        origin,
                                              Vector        direction,
                                              Vector        intersection,
                                              Vector        normal)
{
  Vector       org, dir;
  double       Length;
  PlaneStruct *pplane = (PlaneStruct *)plane;
  gdouble      vd, v0, t;

  /* go to the plane space */
  if (plane->Trans)
  {
    MInverseEvaluatePoint(org, plane->Trans, origin);
    MInverseEvaluateVector(dir, plane->Trans, direction);
  } else
  {
    V3Dcopy(org, origin);
    V3Dcopy(dir, direction);
  }

  /* algorithm taken in Glassner,  p51  */
  V3Dcopy(normal, pplane->Normal); /* FIXME should be normal */
  Length = V3DLength(normal);
  normal[0] /= Length;
  normal[1] /= Length;
  normal[2] /= Length;

  vd = V3DDot(normal, dir);
  if (fabs(vd)<0.000001)
    return FALSE;

  v0 = -(V3DDot(normal,org)-pplane->Distance);

  t = v0/vd;

  if (t<0.000001)
    return FALSE; /* the intersection point is behind the camera  */

  intersection[0]  = org[0] + t * dir[0];
  intersection[1]  = org[1] + t * dir[1];
  intersection[2]  = org[2] + t * dir[2];
  
  if (vd  >  0)
  {
    normal[0] = -normal[0];
    normal[1] = -normal[1];
    normal[2] = -normal[2];
  }
 
  /* transform back the intersection point and the normal to the scene space */
  if (plane->Trans)
  {
    MEvaluatePoint(intersection, plane->Trans, intersection);
    MEvaluateVector(normal, plane->Trans, normal);
  }

  Length = V3DLength(normal);
  normal[0] /= Length;
  normal[1] /= Length;
  normal[2] /= Length;

  return TRUE;
}

/*****************************************************************************
*  giram_plane_new
******************************************************************************/
ObjectStruct *giram_plane_new(Vector Normal, double Distance)
{
  ObjectStruct *Plane;
  PlaneStruct  *PPlane;
  static GiramObjectClass *plane_klass = NULL;

  if (plane_klass == NULL)
  {
    plane_klass = giram_object_class_new();

    plane_klass->name                      = _("Plane");
    plane_klass->build_triangle_mesh       = giram_plane_build_triangle_mesh;
    plane_klass->inside                    = giram_plane_inside;
    plane_klass->find_intersection_segment = giram_plane_find_intersection_segment;
    plane_klass->find_intersection         = giram_plane_find_intersection;
    plane_klass->is_intersection           = giram_plane_is_intersection;
  }
  PPlane = g_new(PlaneStruct, 1);
  Plane = (ObjectStruct *)PPlane;
  InitObject(Plane);
  Plane->Type = PLANE_OBJECT;
  Plane->klass = plane_klass;
  V3Dcopy(PPlane->Normal, Normal);
  PPlane->Distance = Distance;
  PPlane->NbSquare = 4;
  PPlane->SquareSize = 2.5;
  return Plane;
}

/*****************************************************************************
*  PDBCreatePlane
******************************************************************************/
/*int PDBCreatePlane(int *Id, Vector *Normal, double *Distance)
{
  ObjectStruct *Plane;
  FrameStruct  *LocalFrame;
  GSList       *tmp_list;
  ViewStruct   *TmpView;

  Plane = giram_plane_new(*Normal, *Distance);

  LocalFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    LocalFrame = tmp_list->data;
    if (LocalFrame->Id == *Id)
      break;
  }
  if (LocalFrame == NULL)
  {
    g_message("PDBCreatePlane() called with an unknown Id (%d)\n", *Id);
    return 0;
  }
  LocalFrame->all_objects = g_slist_append(LocalFrame->all_objects, Plane);
  giram_object_build_triangle_mesh(Plane);
  for (tmp_list = LocalFrame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->drawing_area);
  }
  RebuildCSGTree(NULL, LocalFrame);
  return Plane->Id;
}*/

