/* cylinder.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "cylinder.h"
#include "trimesh.h"
#include "csgtree.h"
#include "tools/tool_cylinder.h"


#include "giramintl.h"

static void giram_cylinder_build_triangle_mesh(ObjectStruct *cylinder);
static gboolean giram_cylinder_inside(ObjectStruct *cylinder, double x, double y, double z);
static gboolean giram_cylinder_is_intersection(ObjectStruct *cylinder, Vector org, Vector dir);
static gboolean giram_cylinder_click_in_XY(ObjectStruct *cylinder, double x, double y);
static gboolean giram_cylinder_click_in_XZ(ObjectStruct *cylinder, double x, double z);
static gboolean giram_cylinder_click_in_ZY(ObjectStruct *cylinder, double z, double y);
static gboolean giram_cylinder_find_intersection_segment(ObjectStruct *cylinder,
                                                         Vector in_point, Vector out_point,
                                                         Vector inter_point, Vector inter_norm);

/*****************************************************************************
*  giram_cylinder_build_triangle_mesh
******************************************************************************/
static void giram_cylinder_build_triangle_mesh(ObjectStruct *cylinder)
{ /* XXX: clipped_by */
  Vector          P1, P2, P3, P4;
  Vector          N1, N2, N3, N4;
  Vector          NO1, NO2;
  Vector          PO1, PO2;
  int             DetailLevel;
  int             i;
  CylinderStruct *ccylinder = (CylinderStruct *)cylinder;

  DetailLevel = 16; /* XXX: Should be a private field */

  if (cylinder->FirstTriangle)
    DestroyObjectTriangleMesh(cylinder);

  V3Deq(NO1, 0.0, -1.0, 0.0);
  V3Deq(NO2, 0.0,  1.0, 0.0);
  V3Deq(PO1, 0.0,  0.0, 0.0);
  V3Deq(PO2, 0.0,  1.0, 0.0);
  if (cylinder->Trans)
  {
    MEvaluatePoint(PO1, cylinder->Trans, PO1);
    MEvaluateVector(NO1, cylinder->Trans, NO1);
    MEvaluatePoint(PO2, cylinder->Trans, PO2);
    MEvaluateVector(NO2, cylinder->Trans, NO2);
  }
  for (i=0 ; i<DetailLevel ; i++)
  {
    N1[0] = P1[0] = cos((i*2.0)*M_PI/DetailLevel);
    N1[1] = P1[1] = 0.0;
    N1[2] = P1[2] = sin((i*2.0)*M_PI/DetailLevel);
    N2[0] = P2[0] = cos((i*2.0+1.0)*M_PI/DetailLevel);
    N2[1] = 0.0; P2[1] = 1.0;
    N2[2] = P2[2] = sin((i*2.0+1.0)*M_PI/DetailLevel);
    N3[0] = P3[0] = cos((i+1)*2.0*M_PI/DetailLevel);
    N3[1] = P3[1] = 0.0;
    N3[2] = P3[2] = sin((i+1)*2.0*M_PI/DetailLevel);
    N4[0] = P4[0] = cos(((i+1)*2.0+1.0)*M_PI/DetailLevel);
    N4[1] = 0.0; P4[1] = 1.0;
    N4[2] = P4[2] = sin(((i+1)*2.0+1.0)*M_PI/DetailLevel);
    if (cylinder->Trans)
    {
      MEvaluatePoint(P1, cylinder->Trans, P1);
      MEvaluateVector(N1, cylinder->Trans, N1);
      MEvaluatePoint(P2, cylinder->Trans, P2);
      MEvaluateVector(N2, cylinder->Trans, N2);
      MEvaluatePoint(P3, cylinder->Trans, P3);
      MEvaluateVector(N3, cylinder->Trans, N3);
      MEvaluatePoint(P4, cylinder->Trans, P4);
      MEvaluateVector(N4, cylinder->Trans, N4);
    }
    AddTriangleToObjectMesh(cylinder, P1,P2,P3, N1,N2,N3);
    AddTriangleToObjectMesh(cylinder, P2,P3,P4, N2,N3,N4);
    if (!(ccylinder->Open))
    {
      AddTriangleToObjectMesh(cylinder, PO1,P1,P3, NO1, NO1, NO1);
      AddTriangleToObjectMesh(cylinder, PO2,P2,P4, NO2, NO2, NO2);
    }
  }
}

/*****************************************************************************
*  giram_cylinder_inside
******************************************************************************/
static gboolean giram_cylinder_inside(ObjectStruct *cylinder, double x, double y, double z)
{ /* XXX: clipped_by */
  Vector Vect;

  V3Deq(Vect, x, y, z);
  if (cylinder->Trans)
  {
    MInverseEvaluatePoint(Vect, cylinder->Trans, Vect);
  }
  if (Vect[1] < 0.0 || Vect[1] > 1.0)
    return cylinder->Inverse;
  if (Vect[0]*Vect[0]+Vect[2]*Vect[2] > 1.0)
    return cylinder->Inverse;
  else
    return !(cylinder->Inverse);
}

/*****************************************************************************
*  giram_cylinder_is_intersection
******************************************************************************/
static gboolean giram_cylinder_is_intersection(ObjectStruct *cylinder, Vector origin, Vector direction)
{ /* XXX: clipped_by */
  Vector          Org, Dir;
  int             i;
  double          A, B, C, D;
  double          t1, t2, z1, z2;
  CylinderStruct *ccylinder = (CylinderStruct *)cylinder;

  if (cylinder->Trans)
  {
    MInverseEvaluatePoint(Org, cylinder->Trans, origin);
    MInverseEvaluateVector(Dir, cylinder->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      Org[i] = origin[i];
      Dir[i] = direction[i];
    }
  }
  if (Dir[0] == 0.0 && Dir[2] == 0.0)
  {
    if (ccylinder->Open)
      return FALSE;
    else
    {
      if (Org[0]*Org[0]+Org[2]*Org[2] < 1.0)
        return TRUE;
      else
        return FALSE;
    }
  }
  A = Dir[0]*Dir[0] + Dir[2]*Dir[2];
  B = 2.0*(Org[0]*Dir[0] + Org[2]*Dir[2]);
  C = Org[0]*Org[0] + Org[2]*Org[2] - 1.0;
  D = B*B - 4.0*A*C;
  if (D < 0.0)
    return FALSE;
  t1 = (-B-sqrt(D)) / 2.0 / A;
  z1 = Org[1] + t1*Dir[1];
  t2 = (-B+sqrt(D)) / 2.0 / A;
  z2 = Org[1] + t2*Dir[1];
  if ((z1>=0.0 && z1<=1.0) || (z2>=0.0 && z2<=1.0))
    return TRUE;
  if ((z1>1.0) && (z2>1.0))
    return FALSE;
  if ((z1<0.0) && (z2<0.0))
    return FALSE;
  return !(ccylinder->Open);
}

/*****************************************************************************
*  giram_cylinder_click_in_XY
******************************************************************************/
static gboolean giram_cylinder_click_in_XY(ObjectStruct *cylinder, double x, double y)
{
  Vector Origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector Direction = {0.0, 0.0, 1.0, 0.0, 0.0};

  if (cylinder->Trans)
  {
    Origin[0] = x;
    Origin[1] = y;
    return giram_object_is_intersection(cylinder, Origin, Direction);
  } else if ( (x>=-1.0) && (x<=1.0) && (y>=0.0) && (y<=1.0) )
    return TRUE;
  else
    return FALSE;
}

/****************************************************************************
*  giram_cylinder_click_in_XZ
*****************************************************************************/
static gboolean giram_cylinder_click_in_XZ(ObjectStruct *cylinder, double x, double z)
{
  Vector Origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector Direction = {0.0, 1.0, 0.0, 0.0, 0.0};

  if (cylinder->Trans)
  {
    Origin[0] = x;
    Origin[2] = z;
    return giram_object_is_intersection(cylinder, Origin, Direction);
  } else if ( x*x+z*z <= 1.0 )
    return TRUE;
  else
    return FALSE;
}

/****************************************************************************
*  giram_cylinder_click_in_ZY
*****************************************************************************/
static gboolean giram_cylinder_click_in_ZY(ObjectStruct *cylinder, double z, double y)
{
  Vector Origin = {0.0, 0.0, 0.0, 0.0, 0.0};
  Vector Direction = {1.0, 0.0, 0.0, 0.0, 0.0};

  if (cylinder->Trans)
  {
    Origin[2] = z;
    Origin[1] = y;
    return giram_object_is_intersection(cylinder, Origin, Direction);
  } else if ( (z>=-1.0) && (z<=1.0) && (y>=0.0) && (y<=1.0) )
    return TRUE;
  else
    return FALSE;
}

/*****************************************************************************
*  giram_cylinder_find_intersection_segment
******************************************************************************/
static gboolean giram_cylinder_find_intersection_segment(ObjectStruct *cylinder,
                                                         Vector in_point, Vector out_point,
                                                         Vector inter_point, Vector inter_norm)
{
  int    i;
  Vector InP, OutP;
  double A, B, C, Delta, t;
  Vector Tmp;

  if (cylinder->Trans)
  {
    MInverseEvaluatePoint(InP, cylinder->Trans, in_point);
    MInverseEvaluatePoint(OutP, cylinder->Trans, out_point);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      InP[i] = in_point[i];
      OutP[i] = out_point[i];
    }
  }
  A = (OutP[0]-InP[0])*(OutP[0]-InP[0]) + (OutP[2]-InP[2])*(OutP[2]-InP[2]);
  B = 2.0*(InP[0]*(OutP[0]-InP[0]) + InP[2]*(OutP[2]-InP[2]));
  C = InP[0]*InP[0] + InP[2]*InP[2] - 1;
  Delta = B*B-4*A*C;
  if (Delta < 0.0)
  { /* The Ray miss the cylinder or is parallel to the cylinder axis */
    if (fabs(A) < 10e-6)
    { /* The Ray is parallel to the Cylinder axis */
      if (C < 0.0)
      {
        if ( ( (InP[1] >= 0.0) && (InP[1] <= 1.0) && (OutP[1] >= 1.0) ) ||
             ( (OutP[1] >= 0.0) && (OutP[1] <= 1.0) && (InP[1] >= 1.0) ) )
        { /* The Ray Intersect the Top */
          inter_point[0] = InP[0]; inter_point[1] = 1.0; inter_point[2] = InP[2];
          inter_norm[0] = 0.0; inter_norm[1] = 1.0; inter_norm[2] = 0.0;
          if (cylinder->Trans)
          {
            MEvaluatePoint(inter_point, cylinder->Trans, inter_point);
            MEvaluateVector(inter_norm, cylinder->Trans, inter_norm);
          }
          return TRUE;
        }
        if ( ( (InP[1] >= 0.0) && (InP[1] <= 1.0) && (OutP[1] <= 0.0) ) ||
             ( (OutP[1] >= 0.0) && (OutP[1] <= 1.0) && (InP[1] <= 0.0) ) )
        { /* The Ray Intersect the Bottom */
          inter_point[0] = InP[0]; inter_point[1] = 0.0; inter_point[2] = InP[2];
          inter_norm[0] = 0.0; inter_norm[1] = -1.0; inter_norm[2] = 0.0;
          if (cylinder->Trans)
          {
            MEvaluatePoint(inter_point, cylinder->Trans, inter_point);
            MEvaluateVector(inter_norm, cylinder->Trans, inter_norm);
          }
          return TRUE;
        }
      }
    }
    return FALSE;
  }
  t = (-B - sqrt(Delta)) / (2.0 * A);
  if ((t >= 0.0) && (t <= 1.0))
  {
    Tmp[0] = InP[0] + (OutP[0] - InP[0]) * t;
    Tmp[1] = InP[1] + (OutP[1] - InP[1]) * t;
    Tmp[2] = InP[2] + (OutP[2] - InP[2]) * t;
  } else
  {
    t = (-B + sqrt(Delta)) / (2.0 * A);
    if ((t >= 0.0) && (t <= 1.0))
    {
      Tmp[0] = InP[0] + (OutP[0] - InP[0]) * t;
      Tmp[1] = InP[1] + (OutP[1] - InP[1]) * t;
      Tmp[2] = InP[2] + (OutP[2] - InP[2]) * t;
    } else
      return FALSE;
  }
  if ( (Tmp[1] >= 0.0) && (Tmp[1] <= 1.0) )
  {
    inter_point[0] = Tmp[0]; inter_point[1] = Tmp[1]; inter_point[2] = Tmp[2];
    inter_norm[0] = Tmp[0]; inter_norm[1] = 0.0; inter_norm[2] = Tmp[2];
    if (cylinder->Trans)
    {
      MEvaluatePoint(inter_point, cylinder->Trans, inter_point);
      MEvaluateVector(inter_norm, cylinder->Trans, inter_norm);
    }
    return TRUE;
  }
  return FALSE;
}

/*****************************************************************************
*  giram_cylinder_new
******************************************************************************/
ObjectStruct *giram_cylinder_new(Vector Base, Vector Apex, double Radius)
{
  ObjectStruct    *cylinder;
  CylinderStruct  *ccylinder;
  Vector           LocalVector1, LocalVector2, Translate, Scale, N, U1, U2;
  double Length,   ULength;
  TransformStruct  Transform;
  gint             i, j;

  static GiramObjectClass *cylinder_klass = NULL;

  if (cylinder_klass == NULL)
  {
    cylinder_klass = giram_object_class_new();

    cylinder_klass->name                      = _("Cylinder");
    cylinder_klass->build_triangle_mesh       = giram_cylinder_build_triangle_mesh;
    cylinder_klass->inside                    = giram_cylinder_inside;
    cylinder_klass->is_intersection           = giram_cylinder_is_intersection;
    cylinder_klass->click_in_XY               = giram_cylinder_click_in_XY;
    cylinder_klass->click_in_XZ               = giram_cylinder_click_in_XZ;
    cylinder_klass->click_in_ZY               = giram_cylinder_click_in_ZY;
    cylinder_klass->find_intersection_segment = giram_cylinder_find_intersection_segment;
  }
  ccylinder = g_new(CylinderStruct, 1);
  cylinder = (ObjectStruct *)ccylinder;
  InitObject(cylinder);
  cylinder->Type = CYLINDER_OBJECT;
  cylinder->klass = cylinder_klass;
  V3Dcopy(LocalVector1, Base);
  V3Dcopy(LocalVector2, Apex);
  V3Dcopy(Translate, LocalVector1);
  LocalVector2[0] -= Translate[0];
  LocalVector2[1] -= Translate[1];
  LocalVector2[2] -= Translate[2];

  Length = V3DLength(LocalVector2);
  V3Deq(Scale, Radius, Length, Radius);
  giram_object_scale(cylinder, Scale);

  N[0] = LocalVector2[0] / Length;
  N[1] = LocalVector2[1] / Length;
  N[2] = LocalVector2[2] / Length;
  if ( (fabs(N[0])>fabs(N[1])) && (fabs(N[0])>fabs(N[2])) )
  { /* The cylinder axis is mainly upon the X axis */
    V3Deq(U1, -N[1]/N[0], 1, 0);
  } else if ( (fabs(N[1])>fabs(N[0])) && (fabs(N[1])>fabs(N[2])) )
  { /* The cylinder axis is mainly upon the Y axis */
    V3Deq(U1, 1, -N[0]/N[1], 0);
  } else
  { /* The cylinder axis is mainly upon the Z axis */
    V3Deq(U1, 0, 1, -N[1]/N[2]);
  }
  ULength = V3DLength(U1);
  U1[0]/=ULength; U1[1]/=ULength; U1[2]/=ULength;
  VCross(U2, N, U1);
  ULength = V3DLength(U2);
  U2[0]/=ULength; U2[1]/=ULength; U2[2]/=ULength;
  Transform.Direct[0][0] = U1[0];
  Transform.Direct[0][1] = U1[1];
  Transform.Direct[0][2] = U1[2];
  Transform.Direct[0][3] = 0.0;
  Transform.Direct[1][0] = N[0];
  Transform.Direct[1][1] = N[1];
  Transform.Direct[1][2] = N[2];
  Transform.Direct[1][3] = 0.0;
  Transform.Direct[2][0] = U2[0];
  Transform.Direct[2][1] = U2[1];
  Transform.Direct[2][2] = U2[2];
  Transform.Direct[2][3] = 0.0;
  Transform.Direct[3][0] = 0.0;
  Transform.Direct[3][1] = 0.0;
  Transform.Direct[3][2] = 0.0;
  Transform.Direct[3][3] = 1.0;
  MInverse(Transform.Inverse, Transform.Direct);
  if (cylinder->Trans)
    ComposeTrans(cylinder->Trans, &Transform);
  else
  {
    cylinder->Trans = g_new(TransformStruct, 1);
    for (i=0 ; i<4 ; i++)
      for (j=0 ; j<4 ; j++)
      {
        cylinder->Trans->Direct[i][j] = Transform.Direct[i][j];
        cylinder->Trans->Inverse[i][j] = Transform.Inverse[i][j];
      }
  }
  giram_object_translate(cylinder, Translate);
  ccylinder->Open = CylinderOpenFlag;
  return cylinder;
}

/*****************************************************************************
*  PDBCreateCylinder
******************************************************************************/
/*int PDBCreateCylinder(int *Id, Vector *Base, Vector *Apex, double *Radius)
{
  ObjectStruct *Cylinder;
  FrameStruct  *LocalFrame;
  GSList       *tmp_list;
  ViewStruct   *TmpView;

  Cylinder = giram_cylinder_new(*Base, *Apex, *Radius);

  LocalFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    LocalFrame = tmp_list->data;
    if (LocalFrame->Id == *Id)
      break;
  }
  if (LocalFrame == NULL)
  {
    g_message("PDBCreateCylinder() called with an unknown Id (%d)\n", *Id);
    return 0;
  }
  LocalFrame->all_objects = g_slist_append(LocalFrame->all_objects, Cylinder);
  giram_object_build_triangle_mesh(Cylinder);
  for (tmp_list = LocalFrame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->drawing_area);
  }
  RebuildCSGTree(NULL, LocalFrame);
  return Cylinder->Id;
}*/

