/* cone.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2000 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "giram.h"
#include "cone.h"
#include "trimesh.h"
#include "csgtree.h"
#include "tools/tool_cone.h"

#include "giramintl.h"

static void giram_cone_build_triangle_mesh(ObjectStruct *cone);
static gboolean giram_cone_inside(ObjectStruct *cone, double x, double y, double z);
static gboolean giram_cone_is_intersection(ObjectStruct *cone, Vector origin, Vector direction);
static gboolean giram_cone_find_intersection_segment(ObjectStruct *cone,
                                                     Vector in_point, Vector out_point,
                                                     Vector inter_point, Vector inter_norm);

/*****************************************************************************
*  giram_cone_build_triangle_mesh
******************************************************************************/
static void giram_cone_build_triangle_mesh(ObjectStruct *cone)
{ /* XXX: clipped_by */
  Vector      P1, P2, P3, P4;
  Vector      N1, N2, N3, N4;
  Vector      NO1, NO2;
  Vector      PO1, PO2;
  int         DetailLevel;
  int         i;
  ConeStruct *ccone = (ConeStruct *)cone;

  DetailLevel = 16; /* XXX: Should be a private field */

  if (cone->FirstTriangle)
    DestroyObjectTriangleMesh(cone);

  V3Deq(NO1, 0.0, -1.0, 0.0);
  V3Deq(NO2, 0.0,  1.0, 0.0);
  V3Deq(PO1, 0.0,  ccone->Radius1, 0.0);
  V3Deq(PO2, 0.0,  ccone->Radius2, 0.0);
  if (cone->Trans)
  {
    MEvaluatePoint(PO1, cone->Trans, PO1);
    MEvaluateVector(NO1, cone->Trans, NO1);
    MEvaluatePoint(PO2, cone->Trans, PO2);
    MEvaluateVector(NO2, cone->Trans, NO2);
  }
  for (i=0 ; i<DetailLevel ; i++)
  {
    P1[0] = ccone->Radius1*cos((i*2.0)*M_PI/DetailLevel);
    P1[1] = ccone->Radius1;
    P1[2] = ccone->Radius1*sin((i*2.0)*M_PI/DetailLevel);
    N1[0] = sqrt(1.0/2.0)*cos((i*2.0)*M_PI/DetailLevel);
    if (ccone->Radius1 > 0.0)
      N1[1] = -sqrt(1.0/2.0);
    else
      N1[1] = sqrt(1.0/2.0);
    N1[2] = sqrt(1.0/2.0)*sin((i*2.0)*M_PI/DetailLevel);

    P2[0] = ccone->Radius2*cos((i*2.0+1.0)*M_PI/DetailLevel);
    P2[1] = ccone->Radius2;
    P2[2] = ccone->Radius2*sin((i*2.0+1.0)*M_PI/DetailLevel);
    N2[0] = sqrt(1.0/2.0)*cos((i*2.0+1.0)*M_PI/DetailLevel);
    if (ccone->Radius1 > 0.0)
      N2[1] = -sqrt(1.0/2.0);
    else
      N2[1] = sqrt(1.0/2.0);
    N2[2] = sqrt(1.0/2.0)*sin((i*2.0+1.0)*M_PI/DetailLevel);

    P3[0] = ccone->Radius1*cos((i+1)*2.0*M_PI/DetailLevel);
    P3[1] = ccone->Radius1;
    P3[2] = ccone->Radius1*sin((i+1)*2.0*M_PI/DetailLevel);
    N3[0] = sqrt(1.0/2.0)*cos(((i+1)*2.0)*M_PI/DetailLevel);
    if (ccone->Radius1 > 0.0)
      N3[1] = -sqrt(1.0/2.0);
    else
      N3[1] = sqrt(1.0/2.0);
    N3[2] = sqrt(1.0/2.0)*sin(((i+1)*2.0)*M_PI/DetailLevel);

    P4[0] = ccone->Radius2*cos(((i+1)*2.0+1.0)*M_PI/DetailLevel);
    P4[1] = ccone->Radius2;
    P4[2] = ccone->Radius2*sin(((i+1)*2.0+1.0)*M_PI/DetailLevel);
    N4[0] = sqrt(1.0/2.0)*cos(((i+1)*2.0+1.0)*M_PI/DetailLevel);
    if (ccone->Radius1 > 0.0)
      N4[1] = -sqrt(1.0/2.0);
    else
      N4[1] = sqrt(1.0/2.0);
    N4[2] = sqrt(1.0/2.0)*sin(((i+1)*2.0+1.0)*M_PI/DetailLevel);
    if (cone->Trans)
    {
      MEvaluatePoint(P1, cone->Trans, P1);
      MEvaluateVector(N1, cone->Trans, N1);
      MEvaluatePoint(P2, cone->Trans, P2);
      MEvaluateVector(N2, cone->Trans, N2);
      MEvaluatePoint(P3, cone->Trans, P3);
      MEvaluateVector(N3, cone->Trans, N3);
      MEvaluatePoint(P4, cone->Trans, P4);
      MEvaluateVector(N4, cone->Trans, N4);
    }
    AddTriangleToObjectMesh(cone, P1,P2,P3, N1,N2,N3);
    AddTriangleToObjectMesh(cone, P2,P3,P4, N2,N3,N4);
    if (!(ccone->Open))
    {
      if (ccone->Radius1)
        AddTriangleToObjectMesh(cone, PO1,P1,P3, NO1, NO1, NO1);
      if (ccone->Radius2)
        AddTriangleToObjectMesh(cone, PO2,P2,P4, NO2, NO2, NO2);
    }
  }
}

/*****************************************************************************
*  giram_cone_inside
******************************************************************************/
static gboolean giram_cone_inside(ObjectStruct *cone, double x, double y, double z)
{ /* XXX: clipped_by */
  Vector      Vect;
  ConeStruct *ccone = (ConeStruct *)cone;

  V3Deq(Vect, x, y, z);
  if (cone->Trans)
  {
    MInverseEvaluatePoint(Vect, cone->Trans, Vect);
  }
  if ( (ccone->Radius1 > Vect[1]) || (ccone->Radius2 < Vect[1]) )
    return cone->Inverse;
  if (Vect[0]*Vect[0]+Vect[2]*Vect[2] < Vect[1]*Vect[1])
    return !(cone->Inverse);
  else
    return cone->Inverse;
}

/*****************************************************************************
*  giram_cone_find_intersection_segment
******************************************************************************/
static gboolean giram_cone_find_intersection_segment(ObjectStruct *cone,
                                                     Vector in_point, Vector out_point,
                                                     Vector inter_point, Vector inter_norm)
{
  int         i;
  Vector      InP, OutP;
  double      A, B, C, Delta, t;
  Vector      Tmp;
  ConeStruct *ccone = (ConeStruct *)cone;

  if (cone->Trans)
  {
    MInverseEvaluatePoint(InP, cone->Trans, in_point);
    MInverseEvaluatePoint(OutP, cone->Trans, out_point);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      InP[i] = in_point[i];
      OutP[i] = out_point[i];
    }
  }
  A =   (OutP[0]-InP[0])*(OutP[0]-InP[0])
      - (OutP[1]-InP[1])*(OutP[1]-InP[1])
      + (OutP[2]-InP[2])*(OutP[2]-InP[2]);
  B = 2.0* ( InP[0]*(OutP[0]-InP[0]) - InP[1]*(OutP[1]-InP[1]) + InP[2]*(OutP[2]-InP[2]) );
  C = InP[0]*InP[0] - InP[1]*InP[1] + InP[2]*InP[2];
  Delta = B*B - 4*A*C;
  if (fabs(A) < 10e-6)
  {
    /* FIXME */
    return 0;
  }
  if (Delta < 0.0)
    return FALSE;
  t = (-B-sqrt(Delta)) / (2.0*A);
  if ((t >= 0.0) && (t <= 1.0))
  {
    Tmp[0] = InP[0] + (OutP[0] - InP[0]) * t;
    Tmp[1] = InP[1] + (OutP[1] - InP[1]) * t;
    Tmp[2] = InP[2] + (OutP[2] - InP[2]) * t;
    if ( (Tmp[1] > ccone->Radius1) && (Tmp[1] < ccone->Radius2) )
    {
      if (Tmp[0]*Tmp[0]+Tmp[2]*Tmp[2] > 10e-6)
      {
        inter_point[0] = Tmp[0]; inter_point[1] = Tmp[1]; inter_point[2] = Tmp[2];
        inter_norm[0] = sqrt(0.5) * Tmp[0] / sqrt(Tmp[0]*Tmp[0]+Tmp[2]*Tmp[2]);
        if (Tmp[1] > 0.0)
          inter_norm[1] = -sqrt(1.0 / 2.0);
        else
          inter_norm[1] = sqrt(1.0 / 2.0);
        inter_norm[2] = sqrt(0.5) * Tmp[2] / sqrt(Tmp[0]*Tmp[0]+Tmp[2]*Tmp[2]);
      } else
      {
        inter_norm[0] = 0.0; inter_norm[1] = 1.0; inter_norm[2] = 0.0;
      }
      if (cone->Trans)
      {
        MEvaluatePoint(inter_point, cone->Trans, inter_point);
        MEvaluateVector(inter_norm, cone->Trans, inter_norm);
      }
      return TRUE;
    }
  }
  t = (-B + sqrt(Delta)) / (2.0 * A);
  if ((t >= 0.0) && (t <= 1.0))
  {
    Tmp[0] = InP[0] + (OutP[0] - InP[0]) * t;
    Tmp[1] = InP[1] + (OutP[1] - InP[1]) * t;
    Tmp[2] = InP[2] + (OutP[2] - InP[2]) * t;
    if ( (Tmp[1] > ccone->Radius1) && (Tmp[1] < ccone->Radius2) )
    {
      inter_point[0] = Tmp[0]; inter_point[1] = Tmp[1]; inter_point[2] = Tmp[2];
      if (Tmp[0]*Tmp[0]+Tmp[2]*Tmp[2] > 10e-6)
      {
        inter_point[0] = Tmp[0]; inter_point[1] = Tmp[1]; inter_point[2] = Tmp[2];
        inter_norm[0] = sqrt(0.5) * Tmp[0] / sqrt(Tmp[0]*Tmp[0]+Tmp[2]*Tmp[2]);
        if (Tmp[1] > 0.0)
          inter_norm[1] = -sqrt(1.0 / 2.0);
        else
          inter_norm[1] = sqrt(1.0 / 2.0);
        inter_norm[2] = sqrt(0.5) * Tmp[2] / sqrt(Tmp[0]*Tmp[0]+Tmp[2]*Tmp[2]);
      } else
      {
        inter_norm[0] = 0.0; inter_norm[1] = 1.0; inter_norm[2] = 0.0;
      }
      if (cone->Trans)
      {
        MEvaluatePoint(inter_point, cone->Trans, inter_point);
        MEvaluateVector(inter_norm, cone->Trans, inter_norm);
      }
      return TRUE;
    }
  }
  if (fabs(OutP[1]-InP[1]) < 10e-6)
    return 0;
  /* Check for the bottom */
  t = (ccone->Radius1-InP[1]) / (OutP[1]-InP[1]);
  Tmp[0] = InP[0] + (OutP[0] - InP[0]) * t;
  Tmp[1] = ccone->Radius1;
  Tmp[2] = InP[2] + (OutP[2] - InP[2]) * t;
  if (Tmp[0]*Tmp[0]+Tmp[2]*Tmp[2] < Tmp[1]*Tmp[1])
  {
    inter_point[0] = Tmp[0]; inter_point[1] = Tmp[1]; inter_point[2] = Tmp[2];
    inter_norm[0] = 0.0; inter_norm[1] = -1.0; inter_norm[0] = 0.0;
    if (cone->Trans)
    {
      MEvaluatePoint(inter_point, cone->Trans, inter_point);
      MEvaluateVector(inter_norm, cone->Trans, inter_norm);
    }
    return TRUE;
  }
  /* Check for the top */
  t = (ccone->Radius2-InP[1]) / (OutP[1]-InP[1]);
  Tmp[0] = InP[0] + (OutP[0] - InP[0]) * t;
  Tmp[1] = ccone->Radius2;
  Tmp[2] = InP[2] + (OutP[2] - InP[2]) * t;
  if (Tmp[0]*Tmp[0]+Tmp[2]*Tmp[2] < Tmp[1]*Tmp[1])
  {
    inter_point[0] = Tmp[0]; inter_point[1] = Tmp[1]; inter_point[2] = Tmp[2];
    inter_norm[0] = 0.0; inter_norm[1] = 1.0; inter_norm[0] = 0.0;
    if (cone->Trans)
    {
      MEvaluatePoint(inter_point, cone->Trans, inter_point);
      MEvaluateVector(inter_norm, cone->Trans, inter_norm);
    }
    return TRUE;
  }
  return FALSE;
}

/*****************************************************************************
*  IsConeIntersection
******************************************************************************/
static gboolean giram_cone_is_intersection(ObjectStruct *cone, Vector origin, Vector direction)
{ /* XXX: clipped_by */
  Vector      Org, Dir;
  int         i;
  double      A, B, C, Delta;
  double      t;
  double      Alt;
  ConeStruct *ccone = (ConeStruct *)cone;

  if (cone->Trans)
  {
    MInverseEvaluatePoint(Org, cone->Trans, origin);
    MInverseEvaluateVector(Dir, cone->Trans, direction);
  } else
  {
    for (i=0 ; i<3 ; i++)
    {
      Org[i] = origin[i];
      Dir[i] = direction[i];
    }
  }
  A = Dir[0]*Dir[0]-Dir[1]*Dir[1]+Dir[2]*Dir[2];
  B = 2.0 * (Org[0]*Dir[0]-Org[1]*Dir[1]+Org[2]*Dir[2]);
  C = Org[0]*Org[0]-Org[1]*Org[1]+Org[2]*Org[2];
  if (fabs(A) < 10e-6)
  { /* The Ray is Parallel to the generatory line */
    if (fabs(B) < 10e-6)
      return FALSE;
    t = -C / B;
    Alt = Org[1] + t*Dir[1];
    if ( (Alt > ccone->Radius1) && (Alt < ccone->Radius2) )
      return TRUE;
    else return FALSE;
  }
  Delta = B*B - 4*A*C;
  if (Delta < 0.0)
    return FALSE;
  t = (-B-sqrt(Delta)) / (2.0 * A);
  Alt = Org[1] + t*Dir[1];
  if ( (Alt > ccone->Radius1) && (Alt < ccone->Radius2) )
    return TRUE;
  t = (-B+sqrt(Delta)) / (2.0 * A);
  Alt = Org[1] + t*Dir[1];
  if ( (Alt > ccone->Radius1) && (Alt < ccone->Radius2) )
    return TRUE;
  return FALSE;
}

/*****************************************************************************
*  giram_cone_new
******************************************************************************/
ObjectStruct *giram_cone_new(Vector Base, double BaseRadius,
                             Vector Apex, double ApexRadius)
{
  ObjectStruct            *Cone;
  ConeStruct              *CCone;
  static GiramObjectClass *cone_klass = NULL;
  Vector                   LocalVector1, LocalVector2, N, TmpVect, Scale, Translate, U1, U2;
  double                   TmpRad, Length, ULength;
  TransformStruct          Transform;
  guint                    i, j;

  if (cone_klass == NULL)
  {
    cone_klass = giram_object_class_new();

    cone_klass->name                      = _("Cone");
    cone_klass->build_triangle_mesh       = giram_cone_build_triangle_mesh;
    cone_klass->inside                    = giram_cone_inside;
    cone_klass->is_intersection           = giram_cone_is_intersection;
    cone_klass->find_intersection_segment = giram_cone_find_intersection_segment;
  }
  CCone = g_new(ConeStruct, 1);
  Cone = (ObjectStruct *)CCone;
  InitObject(Cone);
  Cone->Type = CONE_OBJECT;
  CCone->Open = ConeOpenFlag;
  Cone->klass = cone_klass;
  CCone->Radius1 = BaseRadius;
  CCone->Radius2 = ApexRadius;
  V3Dcopy(LocalVector1, Base);
  V3Dcopy(LocalVector2, Apex);

  if (CCone->Radius1 > CCone->Radius2)
  { /* We want Radius1 < Radius2, no matter what sign */
    TmpRad = CCone->Radius1;
    CCone->Radius1 = CCone->Radius2;
    CCone->Radius2 = TmpRad;
    for (i=0 ; i<3 ; i++)
    {
      TmpVect[i] = LocalVector1[i];
      LocalVector1[i] = LocalVector2[i];
      LocalVector2[i] = TmpVect[i];
    }
  }
  N[0] = LocalVector2[0] - LocalVector1[0];
  N[1] = LocalVector2[1] - LocalVector1[1];
  N[2] = LocalVector2[2] - LocalVector1[2];
  Length = V3DLength(N);
  if (Length == 0.0)
  { /* FIXME */
  }
  if (CCone->Radius2 - CCone->Radius1 < 10e-6)
  { /* FIXME */
  }
  Scale[0] = 1.0;
  Scale[1] = Length / (CCone->Radius2-CCone->Radius1);
  Scale[2] = 1.0;
  giram_object_scale(Cone, Scale);
  N[0] /= Length;
  N[1] /= Length;
  N[2] /= Length;
  if ( (fabs(N[0])>fabs(N[1])) && (fabs(N[0])>fabs(N[2])) )
  { /* The cone axis is mainly upon the X axis */
    V3Deq(U1, -N[1]/N[0], 1.0, 0.0);
  } else if ( (fabs(N[1])>fabs(N[0])) && (fabs(N[1])>fabs(N[2])) )
  { /* The cone axis is mainly upon the Y axis */
    V3Deq(U1, 1.0, -N[0]/N[1], 0.0);
  } else
  { /* The cone axis is mainly upon the Z axis */
    V3Deq(U1, 0.0, 1.0, -N[1]/N[2]);
  }
  ULength = V3DLength(U1);
  U1[0]/=ULength; U1[1]/=ULength; U1[2]/=ULength;
  VCross(U2, N, U1);
  ULength = V3DLength(U2);
  U2[0]/=ULength; U2[1]/=ULength; U2[2]/=ULength;
  Transform.Direct[0][0] = U1[0];
  Transform.Direct[0][1] = U1[1];
  Transform.Direct[0][2] = U1[2];
  Transform.Direct[0][3] = 0.0;
  Transform.Direct[1][0] = N[0];
  Transform.Direct[1][1] = N[1];
  Transform.Direct[1][2] = N[2];
  Transform.Direct[1][3] = 0.0;
  Transform.Direct[2][0] = U2[0];
  Transform.Direct[2][1] = U2[1];
  Transform.Direct[2][2] = U2[2];
  Transform.Direct[2][3] = 0.0;
  Transform.Direct[3][0] = 0.0;
  Transform.Direct[3][1] = 0.0;
  Transform.Direct[3][2] = 0.0;
  Transform.Direct[3][3] = 1.0;
  MInverse(Transform.Inverse, Transform.Direct);

  if (Cone->Trans)
    ComposeTrans(Cone->Trans, &Transform);
  else
  {
    Cone->Trans = g_new(TransformStruct, 1);
    for (i=0 ; i<4 ; i++)
      for (j=0 ; j<4 ; j++)
      {
        Cone->Trans->Direct[i][j] = Transform.Direct[i][j];
        Cone->Trans->Inverse[i][j] = Transform.Inverse[i][j];
      }
  }
  Translate[0] = LocalVector1[0]+(LocalVector2[0]-LocalVector1[0])*
                                 CCone->Radius1/(CCone->Radius1-CCone->Radius2);
  Translate[1] = LocalVector1[1]+(LocalVector2[1]-LocalVector1[1])*
                                 CCone->Radius1/(CCone->Radius1-CCone->Radius2);
  Translate[2] = LocalVector1[2]+(LocalVector2[2]-LocalVector1[2])*
                                 CCone->Radius1/(CCone->Radius1-CCone->Radius2);
  giram_object_translate(Cone, Translate);

  return Cone;
}

/*****************************************************************************
*  PDBCreateCone
******************************************************************************/
/*int PDBCreateCone(int *Id, Vector *Base, double *BaseRadius,
                           Vector *Apex, double *ApexRadius)
{
  ObjectStruct *Cone;
  FrameStruct  *LocalFrame;
  GSList       *tmp_list;
  ViewStruct   *TmpView;

  Cone = giram_cone_new(*Base, *BaseRadius, *Apex, *ApexRadius);

  LocalFrame = NULL;
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list) )
  {
    LocalFrame = tmp_list->data;
    if (LocalFrame->Id == *Id)
      break;
  }
  if (LocalFrame == NULL)
  {
    g_message("PDBCreateCone() called with an unknown Id (%d)\n", *Id);
    return 0;
  }

  LocalFrame->all_objects = g_slist_append(LocalFrame->all_objects, Cone);
  giram_object_build_triangle_mesh(Cone);
  for (tmp_list = LocalFrame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->drawing_area);
  }
  RebuildCSGTree(NULL, LocalFrame);
  return Cone->Id;
}*/

