/* giramobjecteditor.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include "giram.h"

#include "giramintl.h"
#include "widgets/gtkcolorbutton.h"
#include "widgets/giramfileselection.h"

#include "widgets/giramvectorframe.h"
#include "csgtree.h"
#include "texture.h"
#include "view.h"

#include "primitives/bicubic_patch.h"
#include "primitives/box.h"
#include "primitives/heightfield.h"
#include "primitives/plane.h"
#include "primitives/quadric.h"
#include "primitives/sphere.h"
#include "primitives/torus.h"

#include "giramobjecteditor.h"

static GtkWidgetClass *parent_class = NULL;
static void giram_object_editor_texture_button(GiramObjectEditor *goe);

#define GOE_TYPE_OBJECT     (1)
#define GOE_TYPE_BACKGROUND (2)
#define GOE_TYPE_LIGHT      (3)
#define GOE_TYPE_TRANSFORM  (4)

#define DUMMY_PAGE          (0)
#define BACKGROUND_PAGE     (1)
#define LIGHT_SOURCE_PAGE   (2)
#define BOX_PAGE            (3)
#define CONE_PAGE           (4)
#define CYLINDER_PAGE       (5)
#define DISC_PAGE           (6)
#define PLANE_PAGE          (7)
#define SOR_PAGE            (8)
#define SPHERE_PAGE         (9)
#define SUPERELLIPSOID_PAGE (10)
#define TORUS_PAGE          (11)
#define CSG_PAGE            (12)
#define TRANSFORMATION_PAGE (13)
#define HEIGHTFIELD_PAGE    (14)
#define QUADRIC_PAGE        (15)
#define BICUBIC_PATCH_PAGE  (16)

/*************************************************************************
*  giram_object_editor_class_init
**************************************************************************/
static void giram_object_editor_class_init(GiramObjectEditorClass *class)
{
  parent_class = gtk_type_class(gtk_notebook_get_type());
}

/*************************************************************************
*  giram_object_editor_init
**************************************************************************/
static void giram_object_editor_init(GiramObjectEditor *goe)
{

}

/*************************************************************************
*  giram_object_editor_get_type
**************************************************************************/
GType giram_object_editor_get_type(void)
{
  static GType goe_type = 0;

  if (!goe_type)
  {
    static const GTypeInfo goe_info =
    {
      sizeof(GiramObjectEditorClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc)giram_object_editor_class_init,
      NULL, /* class_finalize */
      NULL, /* class_data */
      sizeof (GiramObjectEditor),
      0, /* n_preallocs */
      (GInstanceInitFunc)giram_object_editor_init
    };
    goe_type = g_type_register_static(GTK_TYPE_NOTEBOOK, "GiramObjectEditor", &goe_info, 0);
  }
  return goe_type;
}

/*************************************************************************
*  giram_object_editor_update_sphere
**************************************************************************/
static void giram_object_editor_update_sphere(GtkEntry *entry, GiramObjectEditor *goe)
{
  SphereStruct *sphere = (SphereStruct *)goe->object;
  GSList       *tmp_list;
  ViewStruct   *TmpView;

  giram_vector_frame_get_value(GIRAM_VECTOR_FRAME(goe->sphere_center), sphere->Center);
  sphere->Radius = strtod(gtk_entry_get_text(GTK_ENTRY(goe->sphere_radius)), NULL);

  giram_object_build_triangle_mesh(goe->object);

  for (tmp_list = goe->object->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*************************************************************************
*  giram_object_editor_create_sphere_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_sphere_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *hbox, *vbox, *label, *button, *main_vbox;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox =  gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Sphere"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(main_vbox), hbox);
  gtk_widget_show(hbox);

  goe->sphere_center = giram_vector_frame_new("Center:");
  gtk_box_pack_start(GTK_BOX(hbox), goe->sphere_center, FALSE, FALSE, 2);
  gtk_widget_show(goe->sphere_center);

  /* the radius */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Radius:");
  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);
  gtk_widget_show(label);

  goe->sphere_radius = gtk_entry_new();
  gtk_box_pack_start_defaults(GTK_BOX(vbox), goe->sphere_radius);
  gtk_widget_show(goe->sphere_radius);
  g_signal_connect(G_OBJECT(goe->sphere_radius), "activate",
                   G_CALLBACK(giram_object_editor_update_sphere), goe);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);
  return page;
}

/*************************************************************************
*  giram_object_editor_update_box
**************************************************************************/
static void giram_object_editor_update_box(GtkEntry *entry, GiramObjectEditor *goe)
{
  BoxStruct  *box = (BoxStruct *)goe->object;
  GSList     *tmp_list;
  ViewStruct *TmpView;

  box->MinCorner[0] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->box_min_x)), NULL);
  box->MinCorner[1] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->box_min_y)), NULL);
  box->MinCorner[2] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->box_min_z)), NULL);
  box->MaxCorner[0] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->box_max_x)), NULL);
  box->MaxCorner[1] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->box_max_y)), NULL);
  box->MaxCorner[2] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->box_max_z)), NULL);

  giram_object_build_triangle_mesh(goe->object);

  for (tmp_list = goe->object->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*************************************************************************
*  giram_object_editor_create_box_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_box_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *hbox, *table, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Box"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(main_vbox), hbox);
  gtk_widget_show(hbox);

  /* the min vertex */
  table = gtk_table_new(4, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), table);
  gtk_widget_show(table);

  label = gtk_label_new("Min vertex:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,2, 0,1);
  gtk_widget_show(label);

  label = gtk_label_new("x: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  gtk_widget_show(label);

  goe->box_min_x = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->box_min_x, 1,2, 1,2);
  gtk_widget_show(goe->box_min_x);
  g_signal_connect(G_OBJECT(goe->box_min_x), "activate",
                   G_CALLBACK(giram_object_editor_update_box), goe);

  label = gtk_label_new("y: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2,3);
  gtk_widget_show(label);

  goe->box_min_y = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->box_min_y, 1,2, 2,3);
  gtk_widget_show(goe->box_min_y);
  g_signal_connect(G_OBJECT(goe->box_min_y), "activate",
                   G_CALLBACK(giram_object_editor_update_box), goe);

  label = gtk_label_new("z: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 3,4);
  gtk_widget_show(label);

  goe->box_min_z = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->box_min_z, 1,2, 3,4);
  gtk_widget_show(goe->box_min_z);
  g_signal_connect(G_OBJECT(goe->box_min_z), "activate",
                   G_CALLBACK(giram_object_editor_update_box), goe);

  /* the max vertex */
  table = gtk_table_new(4, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), table);
  gtk_widget_show(table);

  label = gtk_label_new("Max vertex:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,2, 0,1);
  gtk_widget_show(label);

  label = gtk_label_new("x: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  gtk_widget_show(label);

  goe->box_max_x = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->box_max_x, 1,2, 1,2);
  gtk_widget_show(goe->box_max_x);
  g_signal_connect(G_OBJECT(goe->box_max_x), "activate",
                   G_CALLBACK(giram_object_editor_update_box), goe);

  label = gtk_label_new("y: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2,3);
  gtk_widget_show(label);

  goe->box_max_y = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->box_max_y, 1,2, 2,3);
  gtk_widget_show(goe->box_max_y);
  g_signal_connect(G_OBJECT(goe->box_max_y), "activate",
                   G_CALLBACK(giram_object_editor_update_box), goe);

  label = gtk_label_new("z: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 3,4);
  gtk_widget_show(label);

  goe->box_max_z = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->box_max_z, 1,2, 3,4);
  gtk_widget_show(goe->box_max_z);
  g_signal_connect(G_OBJECT(goe->box_max_z), "activate",
                   G_CALLBACK(giram_object_editor_update_box), goe);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_update_torus
**************************************************************************/
static void giram_object_editor_update_torus(GtkEntry *entry, GiramObjectEditor *goe)
{
  TorusStruct *torus = (TorusStruct *)goe->object;
  GSList      *tmp_list;
  ViewStruct  *TmpView;

  torus->Major = strtod(gtk_entry_get_text(GTK_ENTRY(goe->torus_major)), NULL);
  torus->Minor = strtod(gtk_entry_get_text(GTK_ENTRY(goe->torus_minor)), NULL);

  giram_object_build_triangle_mesh(goe->object);

  for (tmp_list = goe->object->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*************************************************************************
*  giram_object_editor_create_torus_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_torus_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *hbox, *vbox, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);

  label  = gtk_label_new(_("Torus"));

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(main_vbox), hbox);
  gtk_widget_show(hbox);

  /* the major radius */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Major radius:");
  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);
  gtk_widget_show(label);

  goe->torus_major = gtk_entry_new();
  gtk_box_pack_start_defaults(GTK_BOX(vbox), goe->torus_major);
  gtk_widget_show(goe->torus_major);
  g_signal_connect(G_OBJECT(goe->torus_major), "activate",
                   G_CALLBACK(giram_object_editor_update_torus), goe);

  /* the minor radius */
  vbox = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Minor radius:");
  gtk_box_pack_start_defaults(GTK_BOX(vbox), label);
  gtk_widget_show(label);

  goe->torus_minor = gtk_entry_new();
  gtk_box_pack_start_defaults(GTK_BOX(vbox), goe->torus_minor);
  gtk_widget_show(goe->torus_minor);
  g_signal_connect(G_OBJECT(goe->torus_minor), "activate",
                   G_CALLBACK(giram_object_editor_update_torus), goe);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_update_plane
**************************************************************************/
static void giram_object_editor_update_plane(GtkEntry *entry, GiramObjectEditor *goe)
{
  PlaneStruct *plane = (PlaneStruct *)goe->object;
  GSList      *tmp_list;
  ViewStruct  *TmpView;

  plane->Normal[0] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->plane_normal_x)), NULL);
  plane->Normal[1] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->plane_normal_y)), NULL);
  plane->Normal[2] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->plane_normal_z)), NULL);
  plane->Distance = strtod(gtk_entry_get_text(GTK_ENTRY(goe->plane_distance)), NULL);

  giram_object_build_triangle_mesh(goe->object);

  for (tmp_list = goe->object->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*************************************************************************
*  giram_object_editor_create_plane_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_plane_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *hbox, *table, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Plane"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(main_vbox), hbox);
  gtk_widget_show(hbox);

  /* the normal */
  table = gtk_table_new(4, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), table);
  gtk_widget_show(table);

  label = gtk_label_new("Normal:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,2, 0,1);
  gtk_widget_show(label);

  label = gtk_label_new("x: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  gtk_widget_show(label);

  goe->plane_normal_x = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->plane_normal_x, 1,2, 1,2);
  gtk_widget_show(goe->plane_normal_x);
  g_signal_connect(G_OBJECT(goe->plane_normal_x), "activate",
                   G_CALLBACK(giram_object_editor_update_plane), goe);

  label = gtk_label_new("y: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2,3);
  gtk_widget_show(label);

  goe->plane_normal_y = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->plane_normal_y, 1,2, 2,3);
  gtk_widget_show(goe->plane_normal_y);
  g_signal_connect(G_OBJECT(goe->plane_normal_y), "activate",
                   G_CALLBACK(giram_object_editor_update_plane), goe);

  label = gtk_label_new("z: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 3,4);
  gtk_widget_show(label);

  goe->plane_normal_z = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->plane_normal_z, 1,2, 3,4);
  gtk_widget_show(goe->plane_normal_z);
  g_signal_connect(G_OBJECT(goe->plane_normal_z), "activate",
                   G_CALLBACK(giram_object_editor_update_plane), goe);

  /* the distance value */
  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), table);
  gtk_widget_show(table);

  label = gtk_label_new("Distance:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,2, 0,1);
  gtk_widget_show(label);

  label = gtk_label_new("value: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  gtk_widget_show(label);

  goe->plane_distance = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->plane_distance, 1,2, 1,2);
  gtk_widget_show(goe->plane_distance);
  g_signal_connect(G_OBJECT(goe->plane_distance), "activate",
                   G_CALLBACK(giram_object_editor_update_plane), goe);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_create_cone_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_cone_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Cone"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_create_cylinder_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_cylinder_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Cylinder"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_create_disc_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_disc_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Disc"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_create_sor_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_sor_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Sor"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_create_superellipsoid_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_superellipsoid_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Superellipsoid"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_update_heightfield
**************************************************************************/
static void giram_object_editor_update_heightfield(GtkEntry *entry, GiramObjectEditor *goe)
{
  HeightFieldStruct *heightfield = (HeightFieldStruct *)goe->object;

  g_free(heightfield->filename);
  heightfield->filename = g_strdup(giram_file_selection_get_filename(GIRAM_FILE_SELECTION(goe->heightfield_filename)));

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(goe->heightfield_gif)))
    heightfield->type = IMAGE_TYPE_GIF;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(goe->heightfield_tga)))
    heightfield->type = IMAGE_TYPE_TGA;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(goe->heightfield_pot)))
    heightfield->type = IMAGE_TYPE_POT;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(goe->heightfield_png)))
    heightfield->type = IMAGE_TYPE_PNG;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(goe->heightfield_pgm)))
    heightfield->type = IMAGE_TYPE_PGM;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(goe->heightfield_ppm)))
    heightfield->type = IMAGE_TYPE_PPM;
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(goe->heightfield_sys)))
    heightfield->type = IMAGE_TYPE_SYS;

  heightfield->waterlevel = gtk_spin_button_get_value(GTK_SPIN_BUTTON(goe->heightfield_waterlevel));
  heightfield->smooth = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(goe->heightfield_smooth));
/*
  GSList      *tmp_list;
  ViewStruct  *TmpView;

  giram_object_build_triangle_mesh(goe->object);

  for (tmp_list = goe->object->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }*/
}

/*************************************************************************
*  giram_object_editor_create_heightfield_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_heightfield_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *hbox, *label, *main_vbox, *button, *frame, *table;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);

  label  = gtk_label_new(_("HeightField"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* filename */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  label = gtk_label_new(_("filename: "));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  goe->heightfield_filename = giram_file_selection_new(_("Height Field image file"),
                                                       "", FALSE, TRUE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), goe->heightfield_filename);
  gtk_widget_show(goe->heightfield_filename);

  /* type */
  frame = gtk_frame_new(_("type:"));
  gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 0);
  gtk_widget_show(frame);
  table = gtk_table_new(4, 2, TRUE);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_widget_show(table);
  goe->heightfield_gif = gtk_radio_button_new_with_label(NULL, "gif");
  gtk_table_attach_defaults(GTK_TABLE(table), goe->heightfield_gif, 0,1, 0,1);
  gtk_widget_show(goe->heightfield_gif);
  goe->heightfield_tga = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(goe->heightfield_gif), "tga");
  gtk_table_attach_defaults(GTK_TABLE(table), goe->heightfield_tga, 1,2, 0,1);
  gtk_widget_show(goe->heightfield_tga);
  goe->heightfield_pot = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(goe->heightfield_tga), "pot");
  gtk_table_attach_defaults(GTK_TABLE(table), goe->heightfield_pot, 0,1, 1,2);
  gtk_widget_show(goe->heightfield_pot);
  goe->heightfield_png = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(goe->heightfield_pot), "png");
  gtk_table_attach_defaults(GTK_TABLE(table), goe->heightfield_png, 1,2, 1,2);
  gtk_widget_show(goe->heightfield_png);
  goe->heightfield_pgm = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(goe->heightfield_png), "pgm");
  gtk_table_attach_defaults(GTK_TABLE(table), goe->heightfield_pgm, 0,1, 2,3);
  gtk_widget_show(goe->heightfield_pgm);
  goe->heightfield_ppm = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(goe->heightfield_pgm), "ppm");
  gtk_table_attach_defaults(GTK_TABLE(table), goe->heightfield_ppm, 1,2, 2,3);
  gtk_widget_show(goe->heightfield_ppm);
  goe->heightfield_sys = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(goe->heightfield_ppm), "sys");
  gtk_table_attach_defaults(GTK_TABLE(table), goe->heightfield_sys, 0,1, 3,4);
  gtk_widget_show(goe->heightfield_sys);

  /* water_level */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);
  label = gtk_label_new(_("Water level: "));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);
  goe->heightfield_waterlevel = gtk_spin_button_new_with_range(0.0, 1.0, 0.001);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), goe->heightfield_waterlevel);
  gtk_widget_show(goe->heightfield_waterlevel);

  /* smooth */
  goe->heightfield_smooth = gtk_check_button_new_with_label(_("smooth"));
  gtk_box_pack_start(GTK_BOX(main_vbox), goe->heightfield_smooth, FALSE, FALSE, 0);
  gtk_widget_show(goe->heightfield_smooth);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_update_quadric
**************************************************************************/
static void giram_object_editor_update_quadric(GtkEntry *entry, GiramObjectEditor *goe)
{
  QuadricStruct *quadric = (QuadricStruct *)goe->object;

  quadric->A = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_a)), NULL);
  quadric->B = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_b)), NULL);
  quadric->C = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_c)), NULL);
  quadric->D = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_d)), NULL);
  quadric->E = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_e)), NULL);
  quadric->F = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_f)), NULL);
  quadric->G = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_g)), NULL);
  quadric->H = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_h)), NULL);
  quadric->I = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_i)), NULL);
  quadric->J = strtod(gtk_entry_get_text(GTK_ENTRY(goe->quadric_j)), NULL);
/*
  GSList      *tmp_list;
  ViewStruct  *TmpView;

  giram_object_build_triangle_mesh(goe->object);

  for (tmp_list = goe->object->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }*/
}

/*************************************************************************
*  giram_object_editor_create_quadric_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_quadric_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *label, *main_vbox, *button, *table;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);

  label  = gtk_label_new(_("Quadric"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  table = gtk_table_new(4, 6, FALSE);
  gtk_box_pack_start(GTK_BOX(main_vbox), table, FALSE, FALSE, 0);
  gtk_widget_show(table);

  /* <A, B, C> */
  label = gtk_label_new(_("A:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 0,1);
  gtk_widget_show(label);
  goe->quadric_a = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_a, 1,2, 0,1);
  gtk_widget_show(goe->quadric_a);
  g_signal_connect(G_OBJECT(goe->quadric_a), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);
  label = gtk_label_new(_("B:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2,3, 0,1);
  gtk_widget_show(label);
  goe->quadric_b = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_b, 3,4, 0,1);
  gtk_widget_show(goe->quadric_b);
  g_signal_connect(G_OBJECT(goe->quadric_b), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);
  label = gtk_label_new(_("C:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 4,5, 0,1);
  gtk_widget_show(label);
  goe->quadric_c = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_c, 5,6, 0,1);
  gtk_widget_show(goe->quadric_c);
  g_signal_connect(G_OBJECT(goe->quadric_c), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);

  /* <D, E, F> */
  label = gtk_label_new(_("D:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  gtk_widget_show(label);
  goe->quadric_d = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_d, 1,2, 1,2);
  gtk_widget_show(goe->quadric_d);
  g_signal_connect(G_OBJECT(goe->quadric_d), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);
  label = gtk_label_new(_("E:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2,3, 1,2);
  gtk_widget_show(label);
  goe->quadric_e = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_e, 3,4, 1,2);
  gtk_widget_show(goe->quadric_e);
  g_signal_connect(G_OBJECT(goe->quadric_e), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);
  label = gtk_label_new(_("F:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 4,5, 1,2);
  gtk_widget_show(label);
  goe->quadric_f = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_f, 5,6, 1,2);
  gtk_widget_show(goe->quadric_f);
  g_signal_connect(G_OBJECT(goe->quadric_f), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);

  /* <G, H, I> */
  label = gtk_label_new(_("G:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2,3);
  gtk_widget_show(label);
  goe->quadric_g = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_g, 1,2, 2,3);
  gtk_widget_show(goe->quadric_g);
  g_signal_connect(G_OBJECT(goe->quadric_g), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);
  label = gtk_label_new(_("H:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2,3, 2,3);
  gtk_widget_show(label);
  goe->quadric_h = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_h, 3,4, 2,3);
  gtk_widget_show(goe->quadric_h);
  g_signal_connect(G_OBJECT(goe->quadric_h), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);
  label = gtk_label_new(_("I:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 4,5, 2,3);
  gtk_widget_show(label);
  goe->quadric_i = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_i, 5,6, 2,3);
  gtk_widget_show(goe->quadric_i);
  g_signal_connect(G_OBJECT(goe->quadric_i), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);

  /* J */
  label = gtk_label_new(_("J:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 3,4);
  gtk_widget_show(label);
  goe->quadric_j = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->quadric_j, 1,2, 3,4);
  gtk_widget_show(goe->quadric_j);
  g_signal_connect(G_OBJECT(goe->quadric_j), "activate",
                   G_CALLBACK(giram_object_editor_update_quadric), goe);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_update_bicubic_patch
**************************************************************************/
static void giram_object_editor_update_bicubic_patch(GtkEntry *entry, GiramObjectEditor *goe)
{
  GSList             *tmp_list;
  ViewStruct         *TmpView;
  BicubicPatchStruct *bicubic_patch = (BicubicPatchStruct *)goe->object;
  gint                i, j;

  bicubic_patch->u_steps = atoi(gtk_entry_get_text(GTK_ENTRY(goe->bicubic_patch_u_steps)));
  bicubic_patch->v_steps = atoi(gtk_entry_get_text(GTK_ENTRY(goe->bicubic_patch_v_steps)));
  for (i=0 ; i<16 ; i++)
    for (j=0 ; j<3 ; j++)
      bicubic_patch->control_points[i][j] =
        strtod(gtk_entry_get_text(GTK_ENTRY(goe->bicubic_patch_control_points[i][j])), NULL);

  giram_object_build_triangle_mesh(goe->object);

  for (tmp_list = goe->object->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*************************************************************************
*  giram_object_editor_create_bicubic_patch_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_bicubic_patch_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *label, *main_vbox, *button, *table;
  gint       i, j;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);

  label  = gtk_label_new(_("Bicubic Patch"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  table = gtk_table_new(18, 4, FALSE);
  gtk_box_pack_start(GTK_BOX(main_vbox), table, FALSE, FALSE, 0);
  gtk_widget_show(table);

  /* u and v parameters */
  label = gtk_label_new(_("u_steps:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 0,1);
  gtk_widget_show(label);
  goe->bicubic_patch_u_steps = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->bicubic_patch_u_steps, 1,2, 0,1);
  gtk_widget_show(goe->bicubic_patch_u_steps);
  g_signal_connect(G_OBJECT(goe->bicubic_patch_u_steps), "activate",
                   G_CALLBACK(giram_object_editor_update_bicubic_patch), goe);
  label = gtk_label_new(_("v_steps:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2,3, 0,1);
  gtk_widget_show(label);
  goe->bicubic_patch_v_steps = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->bicubic_patch_v_steps, 3,4, 0,1);
  gtk_widget_show(goe->bicubic_patch_v_steps);
  g_signal_connect(G_OBJECT(goe->bicubic_patch_v_steps), "activate",
                   G_CALLBACK(giram_object_editor_update_bicubic_patch), goe);

  /* x, y and z labels */
  label = gtk_label_new(_("X:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 1,2, 1,2);
  gtk_widget_show(label);
  label = gtk_label_new(_("Y:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2,3, 1,2);
  gtk_widget_show(label);
  label = gtk_label_new(_("Z:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 3,4, 1,2);
  gtk_widget_show(label);

  /* the sixteen controls points */
  for (i=0 ; i<16 ; i++)
  {
    gchar *tmp;
    tmp = g_strdup_printf(_("point %d"), i+1);
    label = gtk_label_new(tmp);
    g_free(tmp);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2+i,3+i);
    gtk_widget_show(label);

    for (j=0 ; j<3 ; j++)
    {
      goe->bicubic_patch_control_points[i][j] = gtk_entry_new();
      gtk_table_attach_defaults(GTK_TABLE(table),
                                goe->bicubic_patch_control_points[i][j],
                                j+1, j+2, 2+i, 3+i);
      gtk_widget_show(goe->bicubic_patch_control_points[i][j]);
      g_signal_connect(G_OBJECT(goe->bicubic_patch_control_points[i][j]), "activate",
                       G_CALLBACK(giram_object_editor_update_bicubic_patch), goe);
    }
  }

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_create_csg_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_csg_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *label, *main_vbox, *button;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("CSG object"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* let's add a button at the bottom for the texture editor */
  button = gtk_button_new_with_label(_("Edit Texture"));
  gtk_box_pack_end(GTK_BOX(main_vbox), button, FALSE, FALSE, 0);
  g_signal_connect_swapped(G_OBJECT(button), "clicked",
                           G_CALLBACK(giram_object_editor_texture_button), goe);
  gtk_widget_show(button);

  return page;
}

/*************************************************************************
*  giram_object_editor_update_background
**************************************************************************/
static void giram_object_editor_update_background(GtkColorButton *gcb,
                                                  GiramObjectEditor *goe)
{
  V3Dcopy(goe->background_color, gcb->color);
}

/*************************************************************************
*  giram_object_editor_create_background_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_background_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *hbox, *label, *main_vbox;

  gdouble color[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Background color"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  label = gtk_label_new(_("color: "));
  gtk_box_pack_start_defaults(GTK_BOX(hbox), label);
  gtk_widget_show(label);

  goe->background_color_button = gtk_color_button_new(_("background color"),
                                                      30, 20, color);
  gtk_box_pack_start(GTK_BOX(hbox), goe->background_color_button, FALSE, FALSE, 0);
  gtk_widget_show(goe->background_color_button);
  g_signal_connect(G_OBJECT(goe->background_color_button), "color_changed",
                   G_CALLBACK(giram_object_editor_update_background), goe);

  return page;
}

/*************************************************************************
*  giram_object_editor_update_lightsource
**************************************************************************/
static void giram_object_editor_update_lightsource(GtkWidget *wid,
                                                   GiramObjectEditor *goe)
{
  GSList      *tmp_list;
  ViewStruct  *TmpView;

  goe->light->Location[0] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->lightsource_pos_x)), NULL);
  goe->light->Location[1] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->lightsource_pos_y)), NULL);
  goe->light->Location[2] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->lightsource_pos_z)), NULL);
  V3Dcopy(goe->light->Color, GTK_COLOR_BUTTON(goe->lightsource_color_button)->color);

  for (tmp_list = goe->light->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*************************************************************************
*  giram_object_editor_create_lightsource_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_lightsource_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *hbox, *table, *label, *main_vbox;
  gdouble color[] = { 0.0, 0.0, 0.0, 0.0, 0.0 };

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  label = gtk_label_new(_("Lightsource"));
  gtk_box_pack_start(GTK_BOX(main_vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start_defaults(GTK_BOX(main_vbox), hbox);
  gtk_widget_show(hbox);

  /* the position */
  table = gtk_table_new(4, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), table);
  gtk_widget_show(table);

  label = gtk_label_new("position:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,2, 0,1);
  gtk_widget_show(label);

  label = gtk_label_new("x: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  gtk_widget_show(label);

  goe->lightsource_pos_x = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->lightsource_pos_x, 1,2, 1,2);
  gtk_widget_show(goe->lightsource_pos_x);
  g_signal_connect(G_OBJECT(goe->lightsource_pos_x), "activate",
                   G_CALLBACK(giram_object_editor_update_lightsource), goe);
  
  label = gtk_label_new("y: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2,3);
  gtk_widget_show(label);

  goe->lightsource_pos_y = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->lightsource_pos_y, 1,2, 2,3);
  gtk_widget_show(goe->lightsource_pos_y);
  g_signal_connect(G_OBJECT(goe->lightsource_pos_y), "activate",
                   G_CALLBACK(giram_object_editor_update_lightsource), goe);
  
  label = gtk_label_new("z: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 3,4);
  gtk_widget_show(label);

  goe->lightsource_pos_z = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->lightsource_pos_z, 1,2, 3,4);
  gtk_widget_show(goe->lightsource_pos_z);
  g_signal_connect(G_OBJECT(goe->lightsource_pos_z), "activate",
                   G_CALLBACK(giram_object_editor_update_lightsource), goe);
  
  /* the color */
  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(hbox), table);
  gtk_widget_show(table);

  label = gtk_label_new("color:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,2, 0,1);
  gtk_widget_show(label);

  label = gtk_label_new("color: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  gtk_widget_show(label);

  goe->lightsource_color_button = gtk_color_button_new(_("light source color"),
                                                       30, 20, color);
  gtk_table_attach_defaults(GTK_TABLE(table), goe->lightsource_color_button, 1,2, 1,2);
  gtk_widget_show(goe->lightsource_color_button);
  g_signal_connect(G_OBJECT(goe->lightsource_color_button), "color_changed",
                   G_CALLBACK(giram_object_editor_update_lightsource), goe);

  return page;
}

/*************************************************************************
*  giram_object_editor_update_transformation
**************************************************************************/
static void giram_object_editor_update_transformation(GtkWidget *wid,
                                                      GiramObjectEditor *goe)
{
  GSList       *tmp_list;
  ViewStruct   *TmpView;
  ObjectStruct *object;
  goe->transformation->vect[0] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->transformation_x)), NULL);
  goe->transformation->vect[1] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->transformation_y)), NULL);
  goe->transformation->vect[2] = strtod(gtk_entry_get_text(GTK_ENTRY(goe->transformation_z)), NULL);

  switch (goe->transformation->type)
  {
    case TRANSLATION:
      ComputeTranslateTrans(&(goe->transformation->transform), goe->transformation->vect);
      break;
    case ROTATION:
      ComputeRotateTrans(&(goe->transformation->transform), goe->transformation->vect);
      break;
    case SCALE:
      ComputeScaleTrans(&(goe->transformation->transform), goe->transformation->vect);
      break;
    default:
      g_print("error in giram_object_editor_update_transformation\n");
  }
 
  object = goe->transformation->object;
  
  MIdentity(object->Trans->Direct);
  MIdentity(object->Trans->Inverse);

  for (tmp_list = object->all_transforms ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TransformationStruct *transformation = tmp_list->data;
    ComposeTrans(object->Trans, &(transformation->transform));
  }
    
  giram_object_build_triangle_mesh(object);
  for (tmp_list = object->frame->all_views ;
       tmp_list ;
       tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
}

/*************************************************************************
*  giram_object_editor_create_transformation_page
**************************************************************************/
static GtkWidget *giram_object_editor_create_transformation_page(GiramObjectEditor *goe)
{
  GtkWidget *page, *table, *main_vbox, *label;

  page = gtk_frame_new(NULL);
  gtk_widget_show(page);

  main_vbox  = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(page), main_vbox);
  gtk_widget_show(main_vbox);

  goe->transformation_label = gtk_label_new("---");
  gtk_box_pack_start(GTK_BOX(main_vbox), goe->transformation_label, FALSE, FALSE, 0);
  gtk_widget_show(goe->transformation_label);

  /* the vector */
  table = gtk_table_new(4, 2, FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(main_vbox), table);
  gtk_widget_show(table);

  label = gtk_label_new(_("vector:"));
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,2, 0,1);
  gtk_widget_show(label);

  label = gtk_label_new("x: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 1,2);
  gtk_widget_show(label);

  goe->transformation_x = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->transformation_x, 1,2, 1,2);
  gtk_widget_show(goe->transformation_x);
  g_signal_connect(G_OBJECT(goe->transformation_x), "activate",
                   G_CALLBACK(giram_object_editor_update_transformation), goe);
  
  label = gtk_label_new("y: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 2,3);
  gtk_widget_show(label);

  goe->transformation_y = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->transformation_y, 1,2, 2,3);
  gtk_widget_show(goe->transformation_y);
  g_signal_connect(G_OBJECT(goe->transformation_y), "activate",
                   G_CALLBACK(giram_object_editor_update_transformation), goe);
  
  label = gtk_label_new("z: ");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1, 3,4);
  gtk_widget_show(label);

  goe->transformation_z = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), goe->transformation_z, 1,2, 3,4);
  gtk_widget_show(goe->transformation_z);
  g_signal_connect(G_OBJECT(goe->transformation_z), "activate",
                   G_CALLBACK(giram_object_editor_update_transformation), goe);
 
  return page;
}
/*************************************************************************
*  giram_object_editor_texture_button
**************************************************************************/
static void giram_object_editor_texture_button(GiramObjectEditor *goe)
{
  if (goe->texture_editor)
  {
    if (!GTK_WIDGET_VISIBLE(goe->texture_editor))
      gtk_widget_show(goe->texture_editor);
    else
      gdk_window_raise(goe->texture_editor->window);
  } else
  {
    if (goe->object->Texture)
    {
      goe->texture_editor = texture_editor(goe->object->Texture);
      g_signal_connect(G_OBJECT(goe->texture_editor), "destroy",
                       G_CALLBACK(gtk_widget_destroyed),
                       &(goe->texture_editor));
    } else
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new(NULL, 0, GTK_MESSAGE_WARNING, GTK_BUTTONS_OK,
                                      _("This object has no texture attached."));
      gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
    }
  }
}

/*************************************************************************
*  giram_object_editor_new
**************************************************************************/
GtkWidget *giram_object_editor_new(ObjectStruct *object)
{
  GiramObjectEditor *goe;
  GtkWidget         *page;

  goe = g_object_new(GIRAM_TYPE_OBJECT_EDITOR, NULL);

  goe->object = object;

  gtk_notebook_set_show_border(GTK_NOTEBOOK(goe), FALSE);
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(goe), FALSE);
  gtk_notebook_popup_disable(GTK_NOTEBOOK(goe));

  /* Create an empty page, for default DUMMY_PAGE */
  gtk_notebook_append_page(GTK_NOTEBOOK(goe),
                           gtk_event_box_new(),
                           gtk_label_new("dummy"));
  
  /* Create the page for the background BACKGROUND_PAGE */
  page = giram_object_editor_create_background_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("background"));
  
  /* Create the page for lightsources LIGHT_SOURCE_PAGE */
  page = giram_object_editor_create_lightsource_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                            gtk_label_new("lights"));

  /* Create the page for box objects BOX_PAGE */
  page = giram_object_editor_create_box_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("box"));

  /* Create the page for cone objects CONE_PAGE */
  page = giram_object_editor_create_cone_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("cone"));

  /* Create the page for cylinder objects CYLINDER_PAGE */
  page = giram_object_editor_create_cylinder_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("cylinder"));

  /* Create the page for disc objects DISC_PAGE */
  page = giram_object_editor_create_disc_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("disc"));

  /* Create the page for plane objects PLANE_PAGE */
  page = giram_object_editor_create_plane_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("plane"));

  /* Create the page for the sor SOR_PAGE */
  page = giram_object_editor_create_sor_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("sor"));

  /* Create the page for the sphere SPHERE_PAGE */
  page = giram_object_editor_create_sphere_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("sphere"));

  /* Create the page for superellipsoid objects SUPERELLIPSOID_PAGE */
  page = giram_object_editor_create_superellipsoid_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("superellipsoid"));

  /* Create the page for torus objects TORUS_PAGE */
  page = giram_object_editor_create_torus_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("torus"));

  /* Create the page for CSG objects CSG_PAGE */
  page = giram_object_editor_create_csg_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("csg"));

  /* Create the page for the transformations TRANSFORMATION_PAGE */
  page = giram_object_editor_create_transformation_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("transformation"));

  /* Create the page for the heightfield objects HEIGHTFIELD_PAGE */
  page = giram_object_editor_create_heightfield_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("heightfield"));

  /* Create the page for the quadric objects QUADRIC_PAGE */
  page = giram_object_editor_create_quadric_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("quadric"));

  /* Create the page for the bicubic_patch objects BICUBIC_PATCH_PAGE */
  page = giram_object_editor_create_bicubic_patch_page(goe);
  gtk_notebook_append_page(GTK_NOTEBOOK(goe), page,
                           gtk_label_new("bicubic_patch"));

  goe->texture_editor = NULL;
  return (GTK_WIDGET(goe));
}

/*************************************************************************
*  giram_object_editor_update_object
**************************************************************************/
static void giram_object_editor_object_update(GiramObjectEditor *goe)
{
  if (!goe)
    return;

  if (goe->object)
  {
    gchar *tmp;
    if (goe->texture_editor)
    {
      if (goe->object->Texture)
        texture_editor_set_texture(G_OBJECT(goe->texture_editor),
                                   goe->object->Texture);
      else
        gtk_widget_destroy(goe->texture_editor);
    }
    switch (goe->object->Type)
    {
      case SPHERE_OBJECT:
        {
          SphereStruct *sphere = (SphereStruct *)goe->object;
 
          giram_vector_frame_set_value(GIRAM_VECTOR_FRAME(goe->sphere_center),
                                                          sphere->Center);
          
          tmp = g_strdup_printf("%g", sphere->Radius);
          gtk_entry_set_text(GTK_ENTRY(goe->sphere_radius), tmp);
          g_free(tmp);
          gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), SPHERE_PAGE);
        }
        break;
      case BICUBIC_PATCH_OBJECT:
        {
          BicubicPatchStruct *bicubic_patch = (BicubicPatchStruct *)goe->object;
          gint                i, j;

          tmp = g_strdup_printf("%d", bicubic_patch->u_steps);
          gtk_entry_set_text(GTK_ENTRY(goe->bicubic_patch_u_steps), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%d", bicubic_patch->v_steps);
          gtk_entry_set_text(GTK_ENTRY(goe->bicubic_patch_v_steps), tmp);
          g_free(tmp);
          
          for (i=0 ; i<16 ; i++) for (j=0 ; j<3 ; j++)
          {
            tmp = g_strdup_printf("%g", bicubic_patch->control_points[i][j]);
            gtk_entry_set_text(GTK_ENTRY(goe->bicubic_patch_control_points[i][j]), tmp);
            g_free(tmp);
          }

          gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), BICUBIC_PATCH_PAGE);
        }
        break;
      case BOX_OBJECT:
        {
          BoxStruct *box = (BoxStruct *)goe->object;

          tmp = g_strdup_printf("%g", box->MinCorner[0]);
          gtk_entry_set_text(GTK_ENTRY(goe->box_min_x), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", box->MinCorner[1]);
          gtk_entry_set_text(GTK_ENTRY(goe->box_min_y), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", box->MinCorner[2]);
          gtk_entry_set_text(GTK_ENTRY(goe->box_min_z), tmp);
          g_free(tmp);

          tmp = g_strdup_printf("%g", box->MaxCorner[0]);
          gtk_entry_set_text(GTK_ENTRY(goe->box_max_x), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", box->MaxCorner[1]);
          gtk_entry_set_text(GTK_ENTRY(goe->box_max_y), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", box->MaxCorner[2]);
          gtk_entry_set_text(GTK_ENTRY(goe->box_max_z), tmp);
          g_free(tmp);
          gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), BOX_PAGE);
        }
        break;
      case QUADRIC_OBJECT:
        {
          QuadricStruct *quadric = (QuadricStruct *)goe->object;

          tmp = g_strdup_printf("%g", quadric->A);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_a), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", quadric->B);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_b), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", quadric->C);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_c), tmp);
          g_free(tmp);

          tmp = g_strdup_printf("%g", quadric->D);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_d), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", quadric->E);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_e), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", quadric->F);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_f), tmp);
          g_free(tmp);

          tmp = g_strdup_printf("%g", quadric->G);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_g), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", quadric->H);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_h), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", quadric->I);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_i), tmp);
          g_free(tmp);

          tmp = g_strdup_printf("%g", quadric->J);
          gtk_entry_set_text(GTK_ENTRY(goe->quadric_j), tmp);
          g_free(tmp);
          gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), QUADRIC_PAGE);
        }
        break;
      case TORUS_OBJECT:
        {
          TorusStruct *torus = (TorusStruct *)goe->object;

          tmp = g_strdup_printf("%g", torus->Major);
          gtk_entry_set_text(GTK_ENTRY(goe->torus_major), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", torus->Minor);
          gtk_entry_set_text(GTK_ENTRY(goe->torus_minor), tmp);
          g_free(tmp);
          gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), TORUS_PAGE);
        }
        break;
      case PLANE_OBJECT:
        {
          PlaneStruct *plane = (PlaneStruct *)goe->object;

          tmp = g_strdup_printf("%g", plane->Normal[0]);
          gtk_entry_set_text(GTK_ENTRY(goe->plane_normal_x), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", plane->Normal[1]);
          gtk_entry_set_text(GTK_ENTRY(goe->plane_normal_y), tmp);
          g_free(tmp);
          tmp = g_strdup_printf("%g", plane->Normal[2]);
          gtk_entry_set_text(GTK_ENTRY(goe->plane_normal_z), tmp);
          g_free(tmp);

          tmp = g_strdup_printf("%g", plane->Distance);
          gtk_entry_set_text(GTK_ENTRY(goe->plane_distance), tmp);
          g_free(tmp);
          gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), PLANE_PAGE);
        }
        break;
      case CONE_OBJECT:
        gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), CONE_PAGE);
        break;
      case CYLINDER_OBJECT:
        gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), CYLINDER_PAGE);
        break;
      case  DISC_OBJECT:
        gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), DISC_PAGE);
        break;
      case  SOR_OBJECT:
        gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), SOR_PAGE);
        break;
      case SUPERELLIPSOID_OBJECT:
        gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), SUPERELLIPSOID_PAGE);
        break;
      case CSG_OBJECT:
        gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), CSG_PAGE);
        break;
      case HEIGHT_FIELD_OBJECT:
        {
          HeightFieldStruct *heightfield = (HeightFieldStruct *)goe->object;

          giram_file_selection_set_filename(GIRAM_FILE_SELECTION(goe->heightfield_filename),
                                            heightfield->filename);
          switch (heightfield->type)
          {
            case IMAGE_TYPE_GIF:
              gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(goe->heightfield_gif), TRUE);
              break;
            case IMAGE_TYPE_TGA:
              gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(goe->heightfield_tga), TRUE);
              break;
            case IMAGE_TYPE_POT:
              gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(goe->heightfield_pot), TRUE);
              break;
            case IMAGE_TYPE_PNG:
              gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(goe->heightfield_png), TRUE);
              break;
            case IMAGE_TYPE_PGM:
              gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(goe->heightfield_pgm), TRUE);
              break;
            case IMAGE_TYPE_PPM:
              gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(goe->heightfield_ppm), TRUE);
              break;
            case IMAGE_TYPE_SYS:
              gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(goe->heightfield_sys), TRUE);
              break;                            
          }
          gtk_spin_button_set_value(GTK_SPIN_BUTTON(goe->heightfield_waterlevel),
                                    heightfield->waterlevel);
          gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(goe->heightfield_smooth),
                                       heightfield->smooth);
        }
        gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), HEIGHTFIELD_PAGE);
        break;
      default:
        gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), DUMMY_PAGE);
        break;
    }
  } else
  {
    gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), DUMMY_PAGE);
  }
}

/*************************************************************************
*  giram_object_editor_background_update
**************************************************************************/
static void giram_object_editor_background_update(GiramObjectEditor *goe)
{
  gtk_color_button_set_color(GTK_COLOR_BUTTON(goe->background_color_button),
                             goe->background_color);
  gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), BACKGROUND_PAGE);
}

/*************************************************************************
*  giram_object_editor_lightsource_update
**************************************************************************/
static void giram_object_editor_lightsource_update(GiramObjectEditor *goe)
{
  gchar *tmp;

  /* color */
  gtk_color_button_set_color(GTK_COLOR_BUTTON(goe->lightsource_color_button),
                             goe->light->Color);
  /* position */
  tmp = g_strdup_printf("%g", goe->light->Location[0]);
  gtk_entry_set_text(GTK_ENTRY(goe->lightsource_pos_x), tmp);
  g_free(tmp);
  tmp = g_strdup_printf("%g", goe->light->Location[1]);
  gtk_entry_set_text(GTK_ENTRY(goe->lightsource_pos_y), tmp);
  g_free(tmp);
  tmp = g_strdup_printf("%g", goe->light->Location[2]);
  gtk_entry_set_text(GTK_ENTRY(goe->lightsource_pos_z), tmp);
  g_free(tmp);
  gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), LIGHT_SOURCE_PAGE);
}

/*************************************************************************
* giram_object_editor_transform_update
**************************************************************************/
static void giram_object_editor_transform_update(GiramObjectEditor *goe)
{
  gchar *tmp;

  gtk_widget_set_sensitive(goe->transformation_x, TRUE);
  gtk_widget_set_sensitive(goe->transformation_y, TRUE);
  gtk_widget_set_sensitive(goe->transformation_z, TRUE);
  /* vector */
  tmp = g_strdup_printf("%g", goe->transformation->vect[0]);
  gtk_entry_set_text(GTK_ENTRY(goe->transformation_x), tmp);
  g_free(tmp);
  tmp = g_strdup_printf("%g", goe->transformation->vect[1]);
  gtk_entry_set_text(GTK_ENTRY(goe->transformation_y), tmp);
  g_free(tmp);
  tmp = g_strdup_printf("%g", goe->transformation->vect[2]);
  gtk_entry_set_text(GTK_ENTRY(goe->transformation_z), tmp);
  g_free(tmp);
  switch (goe->transformation->type)
  {
    case TRANSLATION:
      gtk_label_set_text(GTK_LABEL(goe->transformation_label),
                         _("Translation"));
      break;
    case ROTATION:
      gtk_label_set_text(GTK_LABEL(goe->transformation_label),
                         _("Rotation"));
      break;
    case SCALE:
      gtk_label_set_text(GTK_LABEL(goe->transformation_label),
                         _("Scale"));
      break;
    case MATRIX:
      gtk_label_set_text(GTK_LABEL(goe->transformation_label),
                         _("Matrix (no edition for now)"));
      gtk_widget_set_sensitive(goe->transformation_x, FALSE);
      gtk_widget_set_sensitive(goe->transformation_y, FALSE);
      gtk_widget_set_sensitive(goe->transformation_z, FALSE);
      break;
  }
  gtk_notebook_set_current_page(GTK_NOTEBOOK(goe), TRANSFORMATION_PAGE);
}

/*************************************************************************
*  giram_object_editor_update
**************************************************************************/
void giram_object_editor_update(GiramObjectEditor *goe)
{
  if (!goe)
    return;

  switch (goe->type)
  {
    case GOE_TYPE_OBJECT:
      giram_object_editor_object_update(goe);
      break;
    case GOE_TYPE_BACKGROUND:
      giram_object_editor_background_update(goe);
      break;
    case GOE_TYPE_LIGHT:
      giram_object_editor_lightsource_update(goe);
      break;
    case GOE_TYPE_TRANSFORM:
      giram_object_editor_transform_update(goe);
      break;
  }
}

/*************************************************************************
*  giram_object_editor_set_object
**************************************************************************/
void giram_object_editor_set_object(GiramObjectEditor *goe,
                                    ObjectStruct      *object)
{
  if (!goe)
    return;
  goe->object = object;
  goe->type = GOE_TYPE_OBJECT;
  giram_object_editor_update(goe);
}

/*************************************************************************
*  giram_object_editor_set_background
**************************************************************************/
void giram_object_editor_set_background(GiramObjectEditor *goe,
                                        gdouble           *color)
{
  if (!goe || !color)
    return;
  goe->background_color = color;
  goe->type = GOE_TYPE_BACKGROUND;
  giram_object_editor_update(goe);
}

/*************************************************************************
*  giram_object_editor_set_lightsource
**************************************************************************/
void giram_object_editor_set_lightsource(GiramObjectEditor *goe,
                                         LightSourceStruct *light)
{
   if (!goe || !light)
     return;
   goe->light = light;
   goe->type = GOE_TYPE_LIGHT;
   giram_object_editor_update(goe);
}

/*************************************************************************
*  giram_object_editor_set_transformation
**************************************************************************/
void giram_object_editor_set_transformation(GiramObjectEditor *goe,
                                            TransformationStruct *transformation)
{
  if (!goe || !transformation)
    return;
  goe->transformation = transformation;
  goe->type = GOE_TYPE_TRANSFORM;
  giram_object_editor_update(goe);
}

