/* giram.h
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __GIRAM_H__
#define __GIRAM_H__

/* This file should always be the first included. */
#include "config.h"
#include "giramlib/vectors.h"

#include <stdio.h>
#include <gtk/gtk.h>

gchar *AccelKeyFile;

#include "texture.h"

typedef GdkPixmap *(*RenderingFunc)(GtkWidget *area, gint width, gint height);

typedef struct _slow_module_fd slow_module_fd;

typedef void (*UpdateFunc)(slow_module_fd *smfd);

typedef struct module_struct
{
  gchar          *name;
  gchar         **xpm;
  RenderingFunc   render_func;
  gint            type;
  UpdateFunc      update_func;
} module_struct;

struct _slow_module_fd
{
  FILE          *pipe;
  gint           fd;
  GtkWidget     *area;
  gint           width, height;
  guchar        *big_buf;
  guint          buf_size;
  guint          read_total;
  GdkPixmap     *pixmap;
  module_struct *module;
  gint           input_id;
};

GList *render_modules;

/* hash tables */
//GHashTable *colors_hash;

typedef struct SkySphereStruct
{
  PigmentStruct **pPigmentList; /* Skysphere, is merely a list of layered
                                   pigments */
  int NbPigment;
  TransformStruct *Trans;
} SkySphereStruct;

/* A list of triangles is attached to each object to draw them on the screen */
typedef struct TriangleListStruct
{
  Vector P1, P2, P3;
  Vector N1, N2, N3;
  struct TriangleListStruct *Next;
} TriangleListStruct;

/* Here begins the differents objects definitions */
#define BICUBIC_PATCH_OBJECT 1
#define BOX_OBJECT 3
#define CONE_OBJECT 4
#define CSG_OBJECT 5
#define CYLINDER_OBJECT 7
#define DISC_OBJECT 8
#define HEIGHT_FIELD_OBJECT 9
#define MESH_OBJECT 12
#define PLANE_OBJECT 13
#define QUADRIC_OBJECT 14
#define SOR_OBJECT 19
#define SPHERE_OBJECT 20
#define SUPERELLIPSOID_OBJECT 21
#define TORUS_OBJECT 23
#define TRIANGLE_OBJECT 24


#define POINT_LIGHTSOURCE       0
#define SPOTLIGHT_LIGHTSOURCE   1
#define CYLINDER_LIGHTSOURCE    2
#define SHADOWLESS_LIGHTSOURCE 16
typedef struct LightSourceStruct
{
  gchar              *name;
  Vector              Location;
  Vector              Color;
  int                 Type;
  Vector              PointAt;
  double              Radius;
  double              FallOff;
  double              Tightness;
  int                 AreaLight;
  Vector              Area1, Area2;
  double              AreaSize1, AreaSize2;
  double              Adaptive;
  double              Jitter;
  //ObjectStruct       *LooksLike;
  double              FadeDistance;
  double              FadePower;
  int                 AtmosphericAttenuation;
  int                 Atmosphere;
  struct FrameStruct *frame;  
} LightSourceStruct;

typedef struct SymbolTableEntry
{
  int   type;
  void *data;
} SymbolTableEntry;

typedef struct FrameStruct
{
  gchar        *file_name; /* Its name */
  int Id;     /* This is used by the plugins interface */
  GSList       *all_objects;
  GList        *all_cameras;
  GSList       *all_views;
  GSList       *all_light_sources;
  GList        *selection;

  SkySphereStruct *SkySphere;

  GHashTable *all_constants[257];

  Vector RotationCenter;
  Vector ScaleCenter;
  /* The things for the snap... */
  gboolean SnapToGrid, ShowGrid;
  Vector   SnapGrid;
  Vector   background_color;
} FrameStruct;

GSList *all_frames;

#define FRAME_CAMERA 1
#define ORTHO_XY_CAMERA 2
#define ORTHO_XZ_CAMERA 3
#define ORTHO_ZY_CAMERA 4


typedef struct Triangle2D
{
  Vector   P1,P2,P3;
  double   distance;
  gboolean selected;
} Triangle2D;

/* This is used for the Gouraud rendering funcs */
typedef struct Triangle2DNormCol
{
  int    P[3][2];
  double Color[3][3];
  double Distance[3];
} Triangle2DNormCol;

//#include "tools/tools.h"
/* The tools of the main tool box. */
typedef enum
{
  MT_SELECT                 =  1,
  MT_MOVE                   =  2,
  MT_ROTATE                 =  3,
  MT_SCALE                  =  4,
  MT_ZOOM                   =  5,
  MT_NEW_BOX                =  6,
  MT_NEW_CONE               =  7,
  MT_NEW_CYLINDER           =  8,
  MT_NEW_DISC               =  9,
  MT_NEW_PLANE              = 10,
  MT_NEW_SOR                = 11,
  MT_NEW_SPHERE             = 12,
  MT_NEW_SUPERELLIPSOID     = 13,
  MT_NEW_TORUS              = 14,
  MT_NEW_TRIANGLE           = 15,
  MT_NEW_LIGHTSOURCE        = 16,
  MT_PAN                    = 17,
  MT_NEW_CSG                = 18,
  MT_NEW_HEIGHTFIELD        = 19,
  MT_NEW_QUADRIC            = 20,
  MT_NEW_BICUBIC_PATCH      = 21,
  MT_LAST
} GiramModeToolType;

/* The common GCs */
extern GdkGC *giram_black_gc;
extern GdkGC *giram_white_gc;
extern GdkGC *giram_purple_gc;
extern GdkGC *giram_green_gc;
extern GdkGC *giram_red_gc;
extern GdkGC *giram_yellow_gc;

/* The prototypes used in giram.c */
void CreateView(void);
void LoadPovFunc(void);
void LoadDXFFunc(void);
#ifdef USE_LIB3DS
void Load3DSFunc(void);
void Insert3DSFunc(void);
#endif
void LoadGTSFunc(void);
void InsertGTSFunc(void);
void InsertDXFFunc(void);
void SavePovFunc(void);
void SaveDXFFunc(void);
void SaveSelectedDXFFunc(void);
#ifdef WITH_S3D
void SaveS3DFunc(void);
void SaveSelectedS3DFunc(void);
#endif
void ExitFunc(void);
int comparTri(const void *Tri1, const void *Tri2);
extern void DrawGouraudTriangleZBuffer(GtkWidget *View, double *Zbuffer,
                                       Triangle2DNormCol *Tri, guchar *rgb_buf);

gchar *help_path, *plugins_path, *temp_path;
gint toolbox_style;
gint dynamic_toolbar_shape;

#endif /* __GIRAM_H__ */
