/* giram.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/wait.h>

#ifndef  WAIT_ANY
#define  WAIT_ANY (-1)
#endif   /*  WAIT_ANY  */

#include "giram.h"
#include "about_dialog.h"
#include "help_dialog.h"
#include "frame.h"
#include "csgtree.h"
#include "filetypes/povray/LoadPov.h"
#include "filetypes/dxf/dxf.h"
#ifdef WITH_S3D
#include "filetypes/s3d/s3d.h"
#endif
#include "filetypes/gts/gts.h"
#include "filetypes/povray/SavePov.h"
#ifdef USE_LIB3DS
#include "filetypes/3ds/3ds.h"
#endif

#include "tools/tool_cone.h"

#include "tools/tool_rotate.h"
#include "tools/tool_scale.h"
#include "tools/tool_options.h"
#include "tools/tools.h"
#include "view.h"
#include "camera_view.h"

#include "plugins.h"
#include "preferences.h"
#include "utils.h"
#include "copy.h"
#include "giramrc.h"
#include "giramenv.h"
#include "giramintl.h"
#include "appenv.h"
#include "errors.h"
#include "giramsignal.h"
#include "user_install.h"
#include "app_procs.h"
#include "widgets/giramcolorlist.h"
#include "widgets/giramviewshell.h"

//#include "loaders/pov35.h"

//#include "giramwidgets.h"
#include "color_map.h"

#include "file-utils.h"

static void giram_sigfatal_handler(gint sig_num);
static void giram_sigchld_handler(gint sig_num);

static void init(void);
static void giram_error_handler(const gchar    *domain,
                                GLogLevelFlags  flags,
                                const gchar    *msg,
                                gpointer        user_data);

/* GLOBAL data */
gboolean be_verbose        = FALSE;
gboolean use_debug_handler = FALSE;
gboolean console_messages  = FALSE;

MessageHandlerType message_handler = CONSOLE;

gchar *prog_name                = NULL; /* The path name we are invoked with */
gchar *alternate_giramrc        = NULL;
gchar *alternate_system_giramrc = NULL;

GdkGC *giram_black_gc;
GdkGC *giram_white_gc;
GdkGC *giram_purple_gc;
GdkGC *giram_green_gc;
GdkGC *giram_red_gc;
GdkGC *giram_yellow_gc;

/*****************************************************************************
*  CreateDefaultTexture -> should go in texture.c
******************************************************************************/
void CreateDefaultTexture(void)
{
  /* pigment */
  DefaultPigment = g_new(PigmentStruct, 1);
  DefaultPigment->refcount = 1;
  DefaultPigment->Type = PAT_SOLID_COLOR;
  DefaultPigment->Trans = NULL;
  V5Deq(DefaultPigment->Color, 1.0, 1.0, 1.0, 0.0, 0.0);
  V5Deq(DefaultPigment->Color2, 1.0, 1.0, 1.0, 0.0, 0.0);
  V5Deq(DefaultPigment->Color3, 1.0, 1.0, 1.0, 0.0, 0.0);
  {
    gdouble color[5];
    DefaultPigment->color_map = NULL;
    V5Deq(color, 0.0, 0.0, 0.0, 0.0, 0.0);
    DefaultPigment->color_map = color_map_add_point(DefaultPigment->color_map, 0.0, color);
    V5Deq(color, 1.0, 0.0, 0.0, 0.0, 0.0);
    DefaultPigment->color_map = color_map_add_point(DefaultPigment->color_map, 0.3, color);
    V5Deq(color, 1.0, 1.0, 0.0, 0.0, 0.0);
    DefaultPigment->color_map = color_map_add_point(DefaultPigment->color_map, 0.6, color);
    V5Deq(color, 1.0, 1.0, 1.0, 0.0, 0.0);
    DefaultPigment->color_map = color_map_add_point(DefaultPigment->color_map, 1.0, color);
  }

  /* finish */
  DefaultFinish = g_new(FinishStruct, 1);
  V3Deq(DefaultFinish->ambient, 0.1, 0.1, 0.1);
  DefaultFinish->diffuse = 0.6;
  DefaultFinish->brilliance = 1.0;
  V3Deq(DefaultFinish->reflection, 0.0, 0.0, 0.0);
  DefaultFinish->crand = 0.0;
  DefaultFinish->phong = 0.0;
  DefaultFinish->phong_size = 40.0;
  DefaultFinish->specular = 0.0;
  DefaultFinish->roughness = 0.05;
  
  /* texture */
  DefaultTexture = g_new(TextureStruct, 1);
  DefaultTexture->Pigment = CopyPigment(DefaultPigment);
  DefaultTexture->finish = CopyFinish(DefaultFinish);
  DefaultTexture->Trans = NULL;
  DefaultTexture->refcount = 1;
}

/*****************************************************************************
*  giramInit
******************************************************************************/
void giramInit(void)
{
  all_frames = NULL;
  EvenlyScaleFlag = TRUE;
  ConeFunnelFlag = TRUE;
  SnapToAngleFlag = FALSE;
  SnapAngle = 15.0;

/*  InitPlugins();*/ /* XXX */

  CreateDefaultTexture();

  tool_init();
  CreateToolOptionsDialog();
  current_tool = all_tools->data;
  ShowOptionsForTool(current_tool);
}

/*****************************************************************************
*  comparTri FIXME: this had nothing to do here. -> view.c
******************************************************************************/
int comparTri(const void *Tri1, const void *Tri2)
{
  return (((Triangle2D *)Tri2)->distance - ((Triangle2D *)Tri1)->distance);
}

typedef struct RasterLine
{
  int minX, maxX;
  double MinColor[3];
  double MaxColor[3];
  double MinDistance, MaxDistance;
} RasterLine;

/*****************************************************************************
*  DrawGouraudTriangleZBuffer FIXME: this had nothing to do here. -> view.c
******************************************************************************/
void DrawGouraudTriangleZBuffer(GtkWidget *View, double *Zbuffer,
                                Triangle2DNormCol *Tri, guchar *rgb_buf)
{
  RasterLine *ListLine;
  gint        minY, maxY;
  gint        deltaX, deltaY;
  gint        offx, offy;
  gint        i, j;
  gint        x, y;
  gint        bx;
  gdouble     distance;

  maxY = minY = Tri->P[0][1];
  maxY = MAX(Tri->P[1][1], maxY);
  minY = MIN(Tri->P[1][1], minY);
  maxY = MAX(Tri->P[2][1], maxY);
  minY = MIN(Tri->P[2][1], minY);
  if (minY < 0) return;
  if (maxY >= View->allocation.height) return;
  ListLine = g_new(RasterLine, maxY-minY+1);
  for(i=minY ; i<=maxY ; i++)
  {
    ListLine[i-minY].minX = 2<<29;
    ListLine[i-minY].maxX = -(2<<29);
  }
  /* First, Bresenham algorithm */
  /* Line P[0] -> P[1] */
  deltaX = abs(Tri->P[1][0]-Tri->P[0][0]);
  deltaY = abs(Tri->P[1][1]-Tri->P[0][1]);
  if (Tri->P[1][0] > Tri->P[0][0]) offx=1; else offx=-1;
  if (Tri->P[1][1] > Tri->P[0][1]) offy=1; else offy=-1;
  /* "Trace" du premier point */
  ListLine[Tri->P[0][1]-minY].minX = Tri->P[0][0];
  ListLine[Tri->P[0][1]-minY].maxX = Tri->P[0][0];
  ListLine[Tri->P[0][1]-minY].MinColor[0] = Tri->Color[0][0];
  ListLine[Tri->P[0][1]-minY].MinColor[1] = Tri->Color[0][1];
  ListLine[Tri->P[0][1]-minY].MinColor[2] = Tri->Color[0][2];
  ListLine[Tri->P[0][1]-minY].MaxColor[0] = Tri->Color[0][0];
  ListLine[Tri->P[0][1]-minY].MaxColor[1] = Tri->Color[0][1];
  ListLine[Tri->P[0][1]-minY].MaxColor[2] = Tri->Color[0][2];
  ListLine[Tri->P[0][1]-minY].MinDistance =
  ListLine[Tri->P[0][1]-minY].MaxDistance = Tri->Distance[0];
  if (deltaX > deltaY)
  { /* Plutot horizontal */
    x = Tri->P[0][0];
    y = Tri->P[0][1];
    bx = 2*deltaY-deltaX;
    while(x != Tri->P[1][0])
    {
      x+=offx;
      if (bx<0)
        bx+=2*deltaY;
      else
      {
        bx+=2*(deltaY-deltaX);
        y+=offy;
      }
      /* "Trace" du point */
      if (ListLine[y-minY].minX > x)
      {
        ListLine[y-minY].minX = x;
        ListLine[y-minY].MinColor[0]=Tri->Color[0][0]+((double)x-Tri->P[0][0])/(Tri->P[1][0]-Tri->P[0][0])*(Tri->Color[1][0]-Tri->Color[0][0]);
        ListLine[y-minY].MinColor[1]=Tri->Color[0][1]+((double)x-Tri->P[0][0])/(Tri->P[1][0]-Tri->P[0][0])*(Tri->Color[1][1]-Tri->Color[0][1]);
        ListLine[y-minY].MinColor[2]=Tri->Color[0][2]+((double)x-Tri->P[0][0])/(Tri->P[1][0]-Tri->P[0][0])*(Tri->Color[1][2]-Tri->Color[0][2]);
        ListLine[y-minY].MinDistance=Tri->Distance[0]+((double)x-Tri->P[0][0])/(Tri->P[1][0]-Tri->P[0][0])*(Tri->Distance[1]-Tri->Distance[0]);
      }
      if (ListLine[y-minY].maxX < x)
      {
        ListLine[y-minY].maxX = x;
        ListLine[y-minY].MaxColor[0]=Tri->Color[0][0]+((double)x-Tri->P[0][0])/(Tri->P[1][0]-Tri->P[0][0])*(Tri->Color[1][0]-Tri->Color[0][0]);
        ListLine[y-minY].MaxColor[1]=Tri->Color[0][1]+((double)x-Tri->P[0][0])/(Tri->P[1][0]-Tri->P[0][0])*(Tri->Color[1][1]-Tri->Color[0][1]);
        ListLine[y-minY].MaxColor[2]=Tri->Color[0][2]+((double)x-Tri->P[0][0])/(Tri->P[1][0]-Tri->P[0][0])*(Tri->Color[1][2]-Tri->Color[0][2]);
        ListLine[y-minY].MaxDistance=Tri->Distance[0]+((double)x-Tri->P[0][0])/(Tri->P[1][0]-Tri->P[0][0])*(Tri->Distance[1]-Tri->Distance[0]);
      }
    }
  } else
  { /* Plutot vertical */
    x = Tri->P[0][0];
    y = Tri->P[0][1];
    bx = 2*deltaX-deltaY;
    while (y != Tri->P[1][1])
    {
      y+=offy;
      if (bx<0) bx+=2*deltaX;
      else
      {
        bx+=2*(deltaX-deltaY);
        x+=offx;
      }
      /* "Trace" du point */
      if (ListLine[y-minY].minX > x)
      {
        ListLine[y-minY].minX = x;
        ListLine[y-minY].MinColor[0]=Tri->Color[0][0]+((double)y-Tri->P[0][1])/(Tri->P[1][1]-Tri->P[0][1])*(Tri->Color[1][0]-Tri->Color[0][0]);
        ListLine[y-minY].MinColor[1]=Tri->Color[0][1]+((double)y-Tri->P[0][1])/(Tri->P[1][1]-Tri->P[0][1])*(Tri->Color[1][1]-Tri->Color[0][1]);
        ListLine[y-minY].MinColor[2]=Tri->Color[0][2]+((double)y-Tri->P[0][1])/(Tri->P[1][1]-Tri->P[0][1])*(Tri->Color[1][2]-Tri->Color[0][2]);
        ListLine[y-minY].MinDistance=Tri->Distance[0]+((double)y-Tri->P[0][1])/(Tri->P[1][1]-Tri->P[0][1])*(Tri->Distance[1]-Tri->Distance[0]);
      }
      if (ListLine[y-minY].maxX < x)
      {
        ListLine[y-minY].maxX = x;
        ListLine[y-minY].MaxColor[0]=Tri->Color[0][0]+((double)y-Tri->P[0][1])/(Tri->P[1][1]-Tri->P[0][1])*(Tri->Color[1][0]-Tri->Color[0][0]);
        ListLine[y-minY].MaxColor[1]=Tri->Color[0][1]+((double)y-Tri->P[0][1])/(Tri->P[1][1]-Tri->P[0][1])*(Tri->Color[1][1]-Tri->Color[0][1]);
        ListLine[y-minY].MaxColor[2]=Tri->Color[0][2]+((double)y-Tri->P[0][1])/(Tri->P[1][1]-Tri->P[0][1])*(Tri->Color[1][2]-Tri->Color[0][2]);
        ListLine[y-minY].MaxDistance=Tri->Distance[0]+((double)y-Tri->P[0][1])/(Tri->P[1][1]-Tri->P[0][1])*(Tri->Distance[1]-Tri->Distance[0]);
      }
    }
  }
  /* Line P[1] -> P[2] */
  deltaX = abs(Tri->P[2][0]-Tri->P[1][0]);
  deltaY = abs(Tri->P[2][1]-Tri->P[1][1]);
  if (Tri->P[2][0] > Tri->P[1][0]) offx=1; else offx=-1;
  if (Tri->P[2][1] > Tri->P[1][1]) offy=1; else offy=-1;
  /* "Trace" du premier point */
  if (ListLine[Tri->P[1][1]-minY].minX > Tri->P[1][0])
  {
    ListLine[Tri->P[1][1]-minY].minX=Tri->P[1][0];
    ListLine[Tri->P[1][1]-minY].MinColor[0]=Tri->Color[1][0];
    ListLine[Tri->P[1][1]-minY].MinColor[1]=Tri->Color[1][1];
    ListLine[Tri->P[1][1]-minY].MinColor[2]=Tri->Color[1][2];
    ListLine[Tri->P[1][1]-minY].MinDistance=Tri->Distance[1];
  }
  if (ListLine[Tri->P[1][1]-minY].maxX < Tri->P[1][0])
  {
    ListLine[Tri->P[1][1]-minY].maxX=Tri->P[1][0];
    ListLine[Tri->P[1][1]-minY].MaxColor[0]=Tri->Color[1][0];
    ListLine[Tri->P[1][1]-minY].MaxColor[1]=Tri->Color[1][1];
    ListLine[Tri->P[1][1]-minY].MaxColor[2]=Tri->Color[1][2];
    ListLine[Tri->P[1][1]-minY].MaxDistance=Tri->Distance[1];
  }
  if (deltaX > deltaY)
  { /* Plutot horizontal */
    x = Tri->P[1][0];
    y = Tri->P[1][1];
    bx = 2*deltaY-deltaX;
    while(x != Tri->P[2][0])
    {
      x+=offx;
      if (bx<0) bx+=2*deltaY;
      else
      {
        bx+=2*(deltaY-deltaX);
        y+=offy;
      }
      /* "Trace" du point */
      if (ListLine[y-minY].minX > x)
      {
        ListLine[y-minY].minX = x;
        ListLine[y-minY].MinColor[0]=Tri->Color[1][0]+((double)x-Tri->P[1][0])/(Tri->P[2][0]-Tri->P[1][0])*(Tri->Color[2][0]-Tri->Color[1][0]);
        ListLine[y-minY].MinColor[1]=Tri->Color[1][1]+((double)x-Tri->P[1][0])/(Tri->P[2][0]-Tri->P[1][0])*(Tri->Color[2][1]-Tri->Color[1][1]);
        ListLine[y-minY].MinColor[2]=Tri->Color[1][2]+((double)x-Tri->P[1][0])/(Tri->P[2][0]-Tri->P[1][0])*(Tri->Color[2][2]-Tri->Color[1][2]);
        ListLine[y-minY].MinDistance=Tri->Distance[1]+((double)x-Tri->P[1][0])/(Tri->P[2][0]-Tri->P[1][0])*(Tri->Distance[2]-Tri->Distance[1]);
      }
      if (ListLine[y-minY].maxX < x)
      {
        ListLine[y-minY].maxX = x;
        ListLine[y-minY].MaxColor[0]=Tri->Color[1][0]+((double)x-Tri->P[1][0])/(Tri->P[2][0]-Tri->P[1][0])*(Tri->Color[2][0]-Tri->Color[1][0]);
        ListLine[y-minY].MaxColor[1]=Tri->Color[1][1]+((double)x-Tri->P[1][0])/(Tri->P[2][0]-Tri->P[1][0])*(Tri->Color[2][1]-Tri->Color[1][1]);
        ListLine[y-minY].MaxColor[2]=Tri->Color[1][2]+((double)x-Tri->P[1][0])/(Tri->P[2][0]-Tri->P[1][0])*(Tri->Color[2][2]-Tri->Color[1][2]);
        ListLine[y-minY].MaxDistance=Tri->Distance[1]+((double)x-Tri->P[1][0])/(Tri->P[2][0]-Tri->P[1][0])*(Tri->Distance[2]-Tri->Distance[1]);
      }
    }
  } else
  { /* Plutot vertical */
    x = Tri->P[1][0];
    y = Tri->P[1][1];
    bx = 2*deltaX-deltaY;
    while(y != Tri->P[2][1])
    {
      y+=offy;
      if (bx<0) bx+=2*deltaX;
      else
      {
        bx+=2*(deltaX-deltaY);
        x+=offx;
      }
      /* "Trace" du point */
      if (ListLine[y-minY].minX > x)
      {
        ListLine[y-minY].minX = x;
        ListLine[y-minY].MinColor[0]=Tri->Color[1][0]+((double)y-Tri->P[1][1])/(Tri->P[2][1]-Tri->P[1][1])*(Tri->Color[2][0]-Tri->Color[1][0]);
        ListLine[y-minY].MinColor[1]=Tri->Color[1][1]+((double)y-Tri->P[1][1])/(Tri->P[2][1]-Tri->P[1][1])*(Tri->Color[2][1]-Tri->Color[1][1]);
        ListLine[y-minY].MinColor[2]=Tri->Color[1][2]+((double)y-Tri->P[1][1])/(Tri->P[2][1]-Tri->P[1][1])*(Tri->Color[2][2]-Tri->Color[1][2]);
        ListLine[y-minY].MinDistance=Tri->Distance[1]+((double)y-Tri->P[1][1])/(Tri->P[2][1]-Tri->P[1][1])*(Tri->Distance[2]-Tri->Distance[1]);
      }
      if (ListLine[y-minY].maxX < x)
      {
        ListLine[y-minY].maxX = x;
        ListLine[y-minY].MaxColor[0]=Tri->Color[1][0]+((double)y-Tri->P[1][1])/(Tri->P[2][1]-Tri->P[1][1])*(Tri->Color[2][0]-Tri->Color[1][0]);
        ListLine[y-minY].MaxColor[1]=Tri->Color[1][1]+((double)y-Tri->P[1][1])/(Tri->P[2][1]-Tri->P[1][1])*(Tri->Color[2][1]-Tri->Color[1][1]);
        ListLine[y-minY].MaxColor[2]=Tri->Color[1][2]+((double)y-Tri->P[1][1])/(Tri->P[2][1]-Tri->P[1][1])*(Tri->Color[2][2]-Tri->Color[1][2]);
        ListLine[y-minY].MaxDistance=Tri->Distance[1]+((double)y-Tri->P[1][1])/(Tri->P[2][1]-Tri->P[1][1])*(Tri->Distance[2]-Tri->Distance[1]);
      }
    }
  }
  /* Line P[2] -> P[0] */
  deltaX = abs(Tri->P[0][0]-Tri->P[2][0]);
  deltaY = abs(Tri->P[0][1]-Tri->P[2][1]);
  if (Tri->P[0][0] > Tri->P[2][0]) offx=1; else offx=-1;
  if (Tri->P[0][1] > Tri->P[2][1]) offy=1; else offy=-1;
  /* "Trace" du premier point */
  if (ListLine[Tri->P[2][1]-minY].minX > Tri->P[2][0])
  {
    ListLine[Tri->P[2][1]-minY].minX=Tri->P[2][0];
    ListLine[Tri->P[2][1]-minY].MinColor[0]=Tri->Color[2][0];
    ListLine[Tri->P[2][1]-minY].MinColor[1]=Tri->Color[2][1];
    ListLine[Tri->P[2][1]-minY].MinColor[2]=Tri->Color[2][2];
    ListLine[Tri->P[2][1]-minY].MinDistance=Tri->Distance[2];
  }
  if (ListLine[Tri->P[2][1]-minY].maxX < Tri->P[2][0])
  {
    ListLine[Tri->P[2][1]-minY].maxX=Tri->P[2][0];
    ListLine[Tri->P[2][1]-minY].MaxColor[0]=Tri->Color[2][0];
    ListLine[Tri->P[2][1]-minY].MaxColor[1]=Tri->Color[2][1];
    ListLine[Tri->P[2][1]-minY].MaxColor[2]=Tri->Color[2][2];
    ListLine[Tri->P[2][1]-minY].MaxDistance=Tri->Distance[2];
  }
  if (deltaX > deltaY)
  { /* Plutot horizontal */
    x = Tri->P[2][0];
    y = Tri->P[2][1];
    bx = 2*deltaY-deltaX;
    while(x != Tri->P[0][0])
    {
      x+=offx;
      if (bx<0) bx+=2*deltaY;
      else
      {
        bx+=2*(deltaY-deltaX);
        y+=offy;
      }
      /* "Trace" du point */
      if (ListLine[y-minY].minX > x)
      {
        ListLine[y-minY].minX = x;
        ListLine[y-minY].MinColor[0]=Tri->Color[2][0]+((double)x-Tri->P[2][0])/(Tri->P[0][0]-Tri->P[2][0])*(Tri->Color[0][0]-Tri->Color[2][0]);
        ListLine[y-minY].MinColor[1]=Tri->Color[2][1]+((double)x-Tri->P[2][0])/(Tri->P[0][0]-Tri->P[2][0])*(Tri->Color[0][1]-Tri->Color[2][1]);
        ListLine[y-minY].MinColor[2]=Tri->Color[2][2]+((double)x-Tri->P[2][0])/(Tri->P[0][0]-Tri->P[2][0])*(Tri->Color[0][2]-Tri->Color[2][2]);
        ListLine[y-minY].MinDistance=Tri->Distance[2]+((double)x-Tri->P[2][0])/(Tri->P[0][0]-Tri->P[2][0])*(Tri->Distance[0]-Tri->Distance[2]);
      }
      if (ListLine[y-minY].maxX < x)
      {
        ListLine[y-minY].maxX = x;
        ListLine[y-minY].MaxColor[0]=Tri->Color[2][0]+((double)x-Tri->P[2][0])/(Tri->P[0][0]-Tri->P[2][0])*(Tri->Color[0][0]-Tri->Color[2][0]);
        ListLine[y-minY].MaxColor[1]=Tri->Color[2][1]+((double)x-Tri->P[2][0])/(Tri->P[0][0]-Tri->P[2][0])*(Tri->Color[0][1]-Tri->Color[2][1]);
        ListLine[y-minY].MaxColor[2]=Tri->Color[2][2]+((double)x-Tri->P[2][0])/(Tri->P[0][0]-Tri->P[2][0])*(Tri->Color[0][2]-Tri->Color[2][2]);
        ListLine[y-minY].MaxDistance=Tri->Distance[2]+((double)x-Tri->P[2][0])/(Tri->P[0][0]-Tri->P[2][0])*(Tri->Distance[0]-Tri->Distance[2]);
      }
    }
  } else
  { /* Plutot vertical */
    x = Tri->P[2][0];
    y = Tri->P[2][1];
    bx = 2*deltaX-deltaY;
    while(y != Tri->P[0][1])
    {
      y+=offy;
      if (bx<0) bx+=2*deltaX;
      else
      {
        bx+=2*(deltaX-deltaY);
        x+=offx;
      }
      /* "Trace" du point */
      if (ListLine[y-minY].minX > x)
      {
        ListLine[y-minY].minX = x;
        ListLine[y-minY].MinColor[0]=Tri->Color[2][0]+((double)y-Tri->P[2][1])/(Tri->P[0][1]-Tri->P[2][1])*(Tri->Color[0][0]-Tri->Color[2][0]);
        ListLine[y-minY].MinColor[1]=Tri->Color[2][1]+((double)y-Tri->P[2][1])/(Tri->P[0][1]-Tri->P[2][1])*(Tri->Color[0][1]-Tri->Color[2][1]);
        ListLine[y-minY].MinColor[2]=Tri->Color[2][2]+((double)y-Tri->P[2][1])/(Tri->P[0][1]-Tri->P[2][1])*(Tri->Color[0][2]-Tri->Color[2][2]);
        ListLine[y-minY].MinDistance=Tri->Distance[2]+((double)y-Tri->P[2][1])/(Tri->P[0][1]-Tri->P[2][1])*(Tri->Distance[0]-Tri->Distance[2]);
      }
      if (ListLine[y-minY].maxX < x)
      {
        ListLine[y-minY].maxX = x;
        ListLine[y-minY].MaxColor[0]=Tri->Color[2][0]+((double)y-Tri->P[2][1])/(Tri->P[0][1]-Tri->P[2][1])*(Tri->Color[0][0]-Tri->Color[2][0]);
        ListLine[y-minY].MaxColor[1]=Tri->Color[2][1]+((double)y-Tri->P[2][1])/(Tri->P[0][1]-Tri->P[2][1])*(Tri->Color[0][1]-Tri->Color[2][1]);
        ListLine[y-minY].MaxColor[2]=Tri->Color[2][2]+((double)y-Tri->P[2][1])/(Tri->P[0][1]-Tri->P[2][1])*(Tri->Color[0][2]-Tri->Color[2][2]);
        ListLine[y-minY].MaxDistance=Tri->Distance[2]+((double)y-Tri->P[2][1])/(Tri->P[0][1]-Tri->P[2][1])*(Tri->Distance[0]-Tri->Distance[2]);
      }
    }
  }
  /* Then, Zbuffer drawing */
  for (i=minY;i<=maxY;i++)
  {
    if ((ListLine[i-minY].minX < View->allocation.width) && (ListLine[i-minY].maxX >= 0))
      for (j=ListLine[i-minY].minX ; j<=ListLine[i-minY].maxX ; j++)
      {
        if ((i>=0) && (i<View->allocation.height) &&
            (j>=0) && (j<View->allocation.width))
        {
          if (ListLine[i-minY].maxX != ListLine[i-minY].minX)
            distance = ListLine[i-minY].MinDistance + (j-ListLine[i-minY].minX)/(ListLine[i-minY].maxX-ListLine[i-minY].minX)*(ListLine[i-minY].MaxDistance-ListLine[i-minY].MinDistance);
          else
            distance = ListLine[i-minY].MinDistance;
          if (Zbuffer[i*View->allocation.width+j] > distance)
          {
            Zbuffer[i*View->allocation.width+j] = distance;
            rgb_buf[(i*View->allocation.width+j)*3+0] =
                (guchar)(255.0*(ListLine[i-minY].MinColor[0]+((double)j-(double)(ListLine[i-minY].minX))/((double)(ListLine[i-minY].maxX)-(double)(ListLine[i-minY].minX))*(ListLine[i-minY].MaxColor[0]-ListLine[i-minY].MinColor[0])));
            rgb_buf[(i*View->allocation.width+j)*3+1] =
                (guchar)(255.0*(ListLine[i-minY].MinColor[1]+((double)j-(double)(ListLine[i-minY].minX))/((double)(ListLine[i-minY].maxX)-(double)(ListLine[i-minY].minX))*(ListLine[i-minY].MaxColor[1]-ListLine[i-minY].MinColor[1])));
            rgb_buf[(i*View->allocation.width+j)*3+2] =
                (guchar)(255.0*(ListLine[i-minY].MinColor[2]+((double)j-(double)(ListLine[i-minY].minX))/((double)(ListLine[i-minY].maxX)-(double)(ListLine[i-minY].minX))*(ListLine[i-minY].MaxColor[2]-ListLine[i-minY].MinColor[2])));
          }
        }
      }
  }
  g_free(ListLine);
}

/*****************************************************************************
*  NewViewOK
******************************************************************************/
FrameStruct *NewViewFrame;
void NewViewOK(GObject *dialog)
{
  GtkToggleButton *button;
  GtkWidget       *shell;
  ViewStruct      *view_data;

  if (NewViewFrame == NULL)
    NewViewFrame = NewFrame(NULL);

  button = g_object_get_data(dialog, "x-y");
  if (gtk_toggle_button_get_active(button))
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(NewViewFrame, ORTHO_XY_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
  }

  button = g_object_get_data(dialog, "x-z");
  if (gtk_toggle_button_get_active(button))
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(NewViewFrame, ORTHO_XZ_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
  }

  button = g_object_get_data(dialog, "z-y");
  if (gtk_toggle_button_get_active(button))
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(NewViewFrame, ORTHO_ZY_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
  }

  button = g_object_get_data(dialog, "camera");
  if (gtk_toggle_button_get_active(button))
  {
    /* FIXME: shell */
    camera_view_new(NewViewFrame);
  }

  button = g_object_get_data(dialog, "both");
  if (gtk_toggle_button_get_active(button))
  {
    GtkWidget *vpaned, *hpaned, *event_box;

    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 1024, 768);
    gtk_widget_show(shell);
    
    hpaned = gtk_hpaned_new();
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox), hpaned);
    gtk_widget_show(hpaned);
    
    vpaned = gtk_vpaned_new();
    gtk_paned_add1(GTK_PANED(hpaned), vpaned);
    gtk_widget_show(vpaned);

    view_data = giram_view_new(NewViewFrame,  ORTHO_ZY_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_paned_add1(GTK_PANED(vpaned), view_data->disp_vbox);

    event_box = gtk_event_box_new();
    gtk_paned_add2(GTK_PANED(vpaned), event_box);
    gtk_widget_show(event_box);
    
    gtk_paned_set_position(GTK_PANED(vpaned), 384);
    
    vpaned = gtk_vpaned_new();
    gtk_paned_add2(GTK_PANED(hpaned), vpaned);
    gtk_widget_show(vpaned);
    
    view_data = giram_view_new(NewViewFrame, ORTHO_XY_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_paned_add1(GTK_PANED(vpaned), view_data->disp_vbox);
    
    view_data = giram_view_new(NewViewFrame, ORTHO_XZ_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_paned_add2(GTK_PANED(vpaned), view_data->disp_vbox);
    gtk_paned_set_position(GTK_PANED(vpaned), 384);
    gtk_paned_set_position(GTK_PANED(hpaned), 512);
  }

  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/*****************************************************************************
*  NewViewSetFrame
******************************************************************************/
static void NewViewSetFrame(GtkWidget *dummy, FrameStruct *LocalFrame)
{
  NewViewFrame = LocalFrame;
}

/*****************************************************************************
*  new_view_response
******************************************************************************/
static void new_view_response(GtkWidget *dialog, gint response)
{
  if (response == GTK_RESPONSE_ACCEPT)
    NewViewOK(G_OBJECT(dialog));
  else
    gtk_widget_destroy(dialog);
}

/*****************************************************************************
*  new_view_callback
******************************************************************************/
static void new_view_callback(GtkWidget *dummy)
{
  GSList      *tmp_list;
  FrameStruct *TmpFrame;
  GtkWidget   *dialog, *option_menu, *scene_menu, *menu_item;
  GtkWidget   *separator, *check;

  dialog = gtk_dialog_new_with_buttons(_("New view"), NULL, 0,
                                       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
                                       NULL);
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_MOUSE);
  gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
  g_signal_connect(G_OBJECT(dialog), "response",
                   G_CALLBACK(new_view_response), NULL);

  /* The Option Menu for the frame (scene) */
  option_menu = gtk_option_menu_new();
  scene_menu = gtk_menu_new();
  NewViewFrame = NULL;
  menu_item = gtk_menu_item_new_with_label(_("New Frame"));
  gtk_menu_shell_append(GTK_MENU_SHELL(scene_menu), menu_item);
  gtk_widget_show(menu_item);
  g_signal_connect(G_OBJECT(menu_item), "activate",
                   G_CALLBACK(NewViewSetFrame), NULL);
  for (tmp_list = all_frames ; tmp_list ; tmp_list = g_slist_next(tmp_list))
  {
    TmpFrame = tmp_list->data;
    if (TmpFrame->file_name)
      menu_item = gtk_menu_item_new_with_label(TmpFrame->file_name);
    else
      menu_item = gtk_menu_item_new_with_label(_("Untitled"));
    g_signal_connect(G_OBJECT(menu_item), "activate",
                     G_CALLBACK(NewViewSetFrame), TmpFrame);
    gtk_menu_shell_append(GTK_MENU_SHELL(scene_menu), menu_item);
    gtk_widget_show(menu_item);
  }

  gtk_option_menu_set_menu(GTK_OPTION_MENU(option_menu), scene_menu);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), option_menu,
                     FALSE, FALSE, 0);
  /* Separator */
  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), separator,
                     FALSE, FALSE, 5);
  /* View Options */
  check = gtk_check_button_new_with_label(_("View X-Y"));
  g_object_set_data(G_OBJECT(dialog), "x-y", check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), TRUE);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), check);

  check = gtk_check_button_new_with_label(_("View X-Z"));
  g_object_set_data(G_OBJECT(dialog), "x-z", check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), TRUE);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), check);

  check = gtk_check_button_new_with_label(_("View Z-Y"));
  g_object_set_data(G_OBJECT(dialog), "z-y", check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), TRUE);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), check);

  check = gtk_check_button_new_with_label(_("Camera View"));
  g_object_set_data(G_OBJECT(dialog), "camera", check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), TRUE);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), check);

  check = gtk_check_button_new_with_label(_("Three at once"));
  g_object_set_data(G_OBJECT(dialog), "both", check);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), TRUE);
  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), check);

  gtk_widget_show_all(dialog);
}

/*****************************************************************************
*  LoadPovFunc
******************************************************************************/
void LoadPovFunc(void)
{
  GtkWidget *FileSelector;

  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Load Pov File"));
  /* Connect the ok_button to the LoadPov function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(LoadPov), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy),
                           FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy),
                           FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  LoadPov35Func
******************************************************************************/
#if 0
void LoadPov35Func(void)
{
  GtkWidget *FileSelector;

  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Load Povray-3.5 File"));
  /* Connect the ok_button to the LoadPov function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(load_pov_35), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy),
                           FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy),
                           FileSelector);
  gtk_widget_show(FileSelector);
}
#endif

#ifdef USE_LIB3DS
/*****************************************************************************
*  Load3DSFunc
******************************************************************************/
void Load3DSFunc(void)
{
  GtkWidget *FileSelector;

  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Load 3DStudio 3DS File"));
  /* Connect the ok_button to the Load3DS function */
/*  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                     "clicked", (GtkSignalFunc)giram_load_3ds, FileSelector);FIXME*/
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy),
                           FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  Insert3DSFunc
******************************************************************************/
void Insert3DSFunc(void)
{
  GtkWidget  *FileSelector;
  ViewStruct *view_data;

  view_data = get_current_view_data();
  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Insert 3DS objects"));
  g_object_set_data(G_OBJECT(FileSelector), "frame", view_data->frame);
  /* Connect the ok_button to file_ok_sel function */
/*  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                     "clicked", (GtkSignalFunc)giram_insert_3ds, FileSelector);FIXME*/

  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}
#endif

/*****************************************************************************
*  LoadGTSFunc
******************************************************************************/
void LoadGTSFunc(void)
{
  GtkWidget *FileSelector;

  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Load GTS File"));
  /* Connect the ok_button to the giram_load_gts function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(giram_load_gts), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  InsertGTSFunc
******************************************************************************/
void InsertGTSFunc(void)
{
  GtkWidget  *FileSelector;
  ViewStruct *view_data;

  view_data = get_current_view_data();
  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Insert GTS objects"));
  g_object_set_data(G_OBJECT(FileSelector), "frame", view_data->frame);
  /* Connect the ok_button to file_ok_sel function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(giram_insert_gts), FileSelector);

  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  LoadDXFFunc
******************************************************************************/
void LoadDXFFunc(void)
{
  GtkWidget *FileSelector;

  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Load AutoCAD DXF File"));
  /* Connect the ok_button to the LoadDXF function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(LoadDXF), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  InsertDXFFunc
******************************************************************************/
void InsertDXFFunc(void)
{
  GtkWidget  *FileSelector;
  ViewStruct *view_data;

  view_data = get_current_view_data();
  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Insert DXF objects"));
  g_object_set_data(G_OBJECT(FileSelector), "frame", view_data->frame);
  /* Connect the ok_button to file_ok_sel function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(InsertDXF), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  SavePovFunc
******************************************************************************/
void SavePovFunc(void)
{
  GtkWidget  *FileSelector;
  ViewStruct *view_data;

  view_data = get_current_view_data();
  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Save Pov File"));
  g_object_set_data(G_OBJECT(FileSelector), "frame", view_data->frame);
  /* Connect the ok_button to file_ok_sel function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(SavePov), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  SaveDXFFunc
******************************************************************************/
void SaveDXFFunc(void)
{
  GtkWidget  *FileSelector;
  ViewStruct *view_data;

  view_data = get_current_view_data();
  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Save DXF File"));
  g_object_set_data(G_OBJECT(FileSelector), "frame", view_data->frame);
  /* Connect the ok_button to file_ok_sel function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(SaveDXF), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  SaveSelectedDXFFunc
******************************************************************************/
void SaveSelectedDXFFunc(void)
{
  GtkWidget  *FileSelector;
  ViewStruct *view_data;

  view_data = get_current_view_data();
  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Save DXF File"));
  g_object_set_data(G_OBJECT(FileSelector), "frame", view_data->frame);
  /* Connect the ok_button to file_ok_sel function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(SaveSelectedDXF), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}

#ifdef WITH_S3D
/*****************************************************************************
*  SaveS3DFunc
******************************************************************************/
void SaveS3DFunc(void)
{
  GtkWidget  *FileSelector;
  ViewStruct *view_data;

  view_data = get_current_view_data();
  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Save S3D File"));
  g_object_set_data(G_OBJECT(FileSelector), "frame", view_data->frame);
  /* Connect the ok_button to file_ok_sel function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(SaveS3D), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}

/*****************************************************************************
*  SaveSelectedS3DFunc
******************************************************************************/
void SaveSelectedS3DFunc(void)
{
  GtkWidget  *FileSelector;
  ViewStruct *view_data;

  view_data = get_current_view_data();
  /* Create a new file selection widget */
  FileSelector = gtk_file_selection_new(_("Save S3D File"));
  g_object_set_data(G_OBJECT(FileSelector), "frame", view_data->frame);
  /* Connect the ok_button to file_ok_sel function */
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                   "clicked", G_CALLBACK(SaveSelectedS3D), FileSelector);
  /* Connect the cancel_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  /* Connect the ok_button to destroy the widget */
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(FileSelector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), FileSelector);
  gtk_widget_show(FileSelector);
}
#endif

/*****************************************************************************
*  ExitFunc
******************************************************************************/
void ExitFunc(void)
{
  gtk_main_quit();
}

/*****************************************************************************
*  giram_gc_create
******************************************************************************/
void  giram_gc_create(GtkWidget *widget)
{
  GdkColormap *colormap;
  GdkColor     purple, red, yellow, green;

  giram_black_gc = widget->style->black_gc;
  giram_white_gc = widget->style->white_gc;

  colormap = gdk_drawable_get_colormap(widget->window);

  giram_purple_gc = gdk_gc_new(widget->window);
  purple.red = 65535; purple.green = 0; purple.blue = 65535;
  gdk_color_alloc(colormap, &purple);
  gdk_gc_set_foreground(giram_purple_gc, &purple);

  giram_green_gc = gdk_gc_new(widget->window);
  green.red = 0; green.green = 65535; green.blue = 0;
  gdk_color_alloc(colormap, &green);
  gdk_gc_set_foreground(giram_green_gc, &green);

  giram_red_gc = gdk_gc_new(widget->window);
  red.red = 65535; red.green = 0; red.blue = 0;
  gdk_color_alloc(colormap, &red);
  gdk_gc_set_foreground(giram_red_gc, &red);

  giram_yellow_gc = gdk_gc_new(widget->window);
  yellow.red = 65535; yellow.green = 65535; yellow.blue = 0;
  gdk_color_alloc(colormap, &yellow);
  gdk_gc_set_foreground(giram_yellow_gc, &yellow);
}

/***************************************************************************/

static void dump_one_float_constant(gpointer key, SymbolTableEntry *entry, gpointer data)
{
  gchar   *mykey = key;
  gdouble myvalue = *((gdouble *)(entry->data));

  if (entry->type == TFloatID)
    g_print("%s = %g\n", mykey, myvalue);
}

static void dump_float_constants(void)
{
  FrameStruct *frame;
  
  if (!all_frames)
    return;
  
  frame = all_frames->data;
  if (frame->all_constants[0])
  {
    g_hash_table_foreach(frame->all_constants[0], dump_one_float_constant, NULL);
  } else
  {
    g_print("no float constants in this frame...\n");
  }
}

static void dump_one_color_constant(gpointer key, SymbolTableEntry *entry, gpointer data)
{
  gchar   *mykey = key;
  Vector5D *myvalue = (Vector5D *)(entry->data);

  if (entry->type == TColorID)
    g_print("%s = <%g, %g, %g,  %g, %g>\n", mykey, (*myvalue)[0], (*myvalue)[1], (*myvalue)[2], (*myvalue)[3], (*myvalue)[4]);
}

static void dump_color_constants(void)
{
  FrameStruct *frame;
  
  if (!all_frames)
    return;
  
  frame = all_frames->data;
  if (frame->all_constants[0])
  {
    g_hash_table_foreach(frame->all_constants[0], dump_one_color_constant, NULL);
  } else
  {
    g_print("no float constants in this frame...\n");
  }
}

/* LOCAL data */
static gint    giram_argc = 0;
static gchar **giram_argv = NULL;

/*****************************************************************************
*  main
******************************************************************************/
int main(int argc, char *argv[])
{
  GtkWidget *ToolsWindow;
  GtkWidget *Box;
  GtkWidget *menu_bar;
  GtkWidget *HandleBox;
  GtkAccelGroup *AccelGroup;
  GtkItemFactory *Factory;

static GtkItemFactoryEntry MainMenu[] =
{
  {"/_File",                   NULL, NULL,                      0, "<Branch>"},
  {"/File/---",                NULL, NULL,                      0, "<Tearoff>"},
  {"/File/New _View",          NULL, new_view_callback,         0, "<StockItem>", GTK_STOCK_NEW },
  {"/File/Load _Pov",          NULL, LoadPovFunc,               0, "<StockItem>", GTK_STOCK_OPEN },
//  {"/File/Load povray-3.5",    NULL, LoadPov35Func,             0, "<StockItem>", GTK_STOCK_OPEN },
  {"/File/Load AutoCAD _DXF",  NULL, LoadDXFFunc,               0, "<StockItem>", GTK_STOCK_OPEN },
#ifdef USE_LIB3DS
  {"/File/Load 3DStudio _3DS", NULL, Load3DSFunc,               0, "<StockItem>", GTK_STOCK_OPEN },
#endif
  {"/File/Load GTS",           NULL, LoadGTSFunc,               0, "<StockItem>", GTK_STOCK_OPEN },
  {"/File/Load File",          NULL, giram_load_file,           0, "<StockItem>", GTK_STOCK_OPEN },
  {"/File/_About...",          NULL, about_dialog_create,       0, "<Item>"},
  {"/File/_Help...",           NULL, ShowHelpDialog,            0, "<StockItem>", GTK_STOCK_HELP },
  {"/File/_Preferences...",    NULL, preferences_dialog_create, 0, "<StockItem>", GTK_STOCK_PREFERENCES },
  {"/File/Dum",                NULL, NULL,                      0, "<Separator>"},
  {"/File/_Quit",              NULL, ExitFunc,                  0, "<StockItem>",  GTK_STOCK_QUIT },
  {"/_Dialogs",                NULL, NULL,                      0, "<Branch>"},
  {"/Dialogs/---",             NULL, NULL,                      0, "<Tearoff>" },
  {"/Dialogs/_CSG Tree",       NULL, ShowCSGTree,               0, "<Item>"},
  {"/Dialogs/_Tool Options",   NULL, ShowToolOptionsDialog,     0, "<Item>"},
  {"/Dialogs/_Debug",          NULL, NULL,                      0, "<Branch>"},
  {"/Dialogs/Debug/---",       NULL, NULL,                      0, "<Tearoff>"},
  {"/Dialogs/Debug/floats",    NULL, dump_float_constants,      0, "<Item>"},
  {"/Dialogs/Debug/colors",    NULL, dump_color_constants,      0, "<Item>"}
};

static guint NbEntries = G_N_ELEMENTS(MainMenu);
  GtkWidget *ToolsTable;
  gboolean show_version = FALSE;
  gboolean show_help = FALSE;
  gint i;

#ifdef HAVE_PUTENV
  gchar *display_env;
#endif

  g_atexit(g_mem_profile);

  /* Initialize variables */

  render_modules = NULL;

  prog_name = argv[0];

  /* Initialize i18n support */
  INIT_LOCALE("giram");

#ifdef ENABLE_NLS
  bindtextdomain ("giram-libgiram", LOCALEDIR);
#endif

  gtk_init(&argc, &argv);
  gdk_rgb_init();

  setlocale(LC_NUMERIC, "C");  /* gtk seems to zap this during init.. */

#ifdef HAVE_PUTENV
  display_env = g_strconcat("DISPLAY=", gdk_get_display(), NULL);
  putenv(display_env);
#endif

  for (i=1 ; i<argc ; i++)
  {
    if (strcmp(argv[i], "--system-giramrc") == 0)
    {
      argv[i] = NULL;
      if (argc <= ++i)
      {
        show_help = TRUE;
      } else
      {
        alternate_system_giramrc = argv[i];
        argv[i] = NULL;
      }
    } else if ((strcmp(argv[i], "--giramrc") == 0) ||
               (strcmp(argv[i], "-g") == 0))
    {
      argv[i] = NULL;
      if (argc <= ++i)
      {
        show_help = TRUE;
      } else
      {
        alternate_giramrc = argv[i];
        argv[i] = NULL;
      }
    } else if ((strcmp(argv[i], "--help") == 0) ||
               (strcmp(argv[i], "-h") == 0))
    {
      show_help = TRUE;
      argv[i] = NULL;
    } else if ((strcmp(argv[i], "--version") == 0) ||
               (strcmp (argv[i], "-v") == 0))
    {
      show_version = TRUE;
      argv[i] = NULL;
    } else if (strcmp(argv[i], "--verbose") == 0)
    {
      be_verbose = TRUE;
      argv[i] = NULL;
    } else if (strcmp(argv[i], "--debug-handlers") == 0)
    {
      use_debug_handler = TRUE;
      argv[i] = NULL;
    } else if ((strcmp(argv[i], "--console-messages") == 0) ||
               (strcmp(argv[i], "-c") == 0))
    {
      console_messages = TRUE;
      argv[i] = NULL;
    } else if (strcmp(argv[i], "--enable-stack-trace") == 0)
    {
      argv[i] = NULL;
      if (argc <= ++i)
      {
        show_help = TRUE;
      } else
      {
        if (!strcmp(argv[i], "never"))
          stack_trace_mode = STACK_TRACE_NEVER;
        else if (!strcmp(argv[i], "query"))
          stack_trace_mode = STACK_TRACE_QUERY;
        else if (!strcmp(argv[i], "always"))
          stack_trace_mode = STACK_TRACE_ALWAYS;
        else
          show_help = TRUE;

        argv[i] = NULL;
      }
    }
    /*
     * ANYTHING ELSE starting with a '-' is an error.
     */
    else if (argv[i][0] == '-')
    {
      g_print(_("\nInvalid option.\n"));
      show_help = TRUE;
    }
  }

/*  if (show_version)
    g_print("%s %s\n", _("Giram version"), VERSION);*/

  if (show_help)
  {
    g_print(_("\nUsage: %s [option ... ] [file ... ]\n\n"), argv[0]);
    g_print(_("Options:\n"));
    g_print(_("  -c, --console-messages    Display warnings to console instead of a dialog box.\n"));
    g_print(_("  -g, --giramrc <giramrc>   Use an alternate giramrc file.\n"));
    g_print(_("  -h, --help                Output this help.\n"));
    g_print(_("  -v, --version             Output version information.\n"));
    g_print(_("  --verbose                 Show startup messages.\n"));
    g_print(_("  --debug-handlers          Enable non-fatal debugging signal handlers.\n"));
    g_print(_("  --display <display>       Use the designated X display.\n"));
    g_print(_("  --system-giramrc <gimprc> Use an alternate system giramrc file.\n"));
    g_print("  --enable-stack-trace <never | query | always>\n");
    g_print(_("                            Debugging mode for fatal signals.\n\n"));
  }

  if (show_version || show_help)
  {
    exit(0);
  }

  g_log_set_handler("Giram", G_LOG_LEVEL_MESSAGE,
                    giram_message_func, NULL);

  /* Handle fatal signals */

  giram_signal_private(SIGHUP,  giram_sigfatal_handler, 0);
  giram_signal_private(SIGINT,  giram_sigfatal_handler, 0);
  giram_signal_private(SIGQUIT, giram_sigfatal_handler, 0);
  giram_signal_private(SIGABRT, giram_sigfatal_handler, 0);
  giram_signal_private(SIGBUS,  giram_sigfatal_handler, 0);
  giram_signal_private(SIGSEGV, giram_sigfatal_handler, 0);
  giram_signal_private(SIGTERM, giram_sigfatal_handler, 0);
  giram_signal_private(SIGFPE,  giram_sigfatal_handler, 0);

  /* Ignore SIGPIPE because plug_in.c handles broken pipes (well, it will) */

  giram_signal_private(SIGPIPE, SIG_IGN, 0);

  /* Collect dead children */

  giram_signal_private(SIGCHLD, giram_sigchld_handler, SA_RESTART);

  g_log_set_handler(NULL,
                    G_LOG_LEVEL_ERROR | G_LOG_FLAG_FATAL,
                    giram_error_handler, NULL);

  /* Keep the command line arguments--for use in giram_init */
  giram_argc = argc - 1;
  giram_argv = argv + 1;

  giramInit();
  /* Check the user_installation */
  user_install_verify(init);


  g_print("povray include path: '%s'\n'", pov_include_path);
  g_print("renderer modules path: '%s'\n'", renderer_modules_path);

  ToolsWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(ToolsWindow), _("Giram Tools"));
  /* Here we just set a handler for delete_event that immediately
   * exits GTK. */
  g_signal_connect(G_OBJECT(ToolsWindow), "delete_event",
                   G_CALLBACK(ExitFunc), NULL);
  gtk_container_set_border_width(GTK_CONTAINER(ToolsWindow), 0);
  gtk_window_set_resizable(GTK_WINDOW(ToolsWindow), FALSE);

  Box = gtk_vbox_new(FALSE, 0);
  /* Put the box into the main window. */
  gtk_container_add(GTK_CONTAINER(ToolsWindow), Box);


  HandleBox = gtk_handle_box_new();
  gtk_box_pack_start(GTK_BOX(Box), HandleBox, FALSE, FALSE, 2);
  gtk_widget_show(HandleBox);

  /* New Keyboard accel group */
  AccelGroup = gtk_accel_group_new();

  /* Creation of the item factory */
  Factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<Giram>", AccelGroup);

  gtk_item_factory_create_items(Factory, NbEntries, MainMenu, NULL);
//  gtk_accel_group_attach(AccelGroup, GTK_OBJECT(ToolsWindow));

  /* We get the menubar from the factory */
  menu_bar = gtk_item_factory_get_widget(Factory,"<Giram>");
  gtk_widget_show(menu_bar);

  gtk_container_add(GTK_CONTAINER(HandleBox), menu_bar);


  /* The Tool Table */
  ToolsTable = CreateTools(Box);

  CreateViewMenu();

  gtk_widget_show(ToolsTable);
  gtk_widget_show(Box);
  gtk_widget_show(ToolsWindow);

  giram_gc_create(ToolsWindow);

  gtk_main();
  return 0;
}

static void init(void)
{
  /*  Continue initializing  */
  app_giram_init(giram_argc, giram_argv);
}

static void giram_error_handler(const gchar    *domain,
                                GLogLevelFlags  flags,
                                const gchar    *msg,
                                gpointer        user_data)
{
  giram_fatal_error("%s", msg);
}

/* giram core signal handler for fatal signals */
static void giram_sigfatal_handler(gint sig_num)
{
  switch (sig_num)
  {
    case SIGHUP:
    case SIGINT:
    case SIGQUIT:
    case SIGABRT:
    case SIGTERM:
      giram_terminate((gchar *)g_strsignal(sig_num));
      break;
    case SIGBUS:
    case SIGSEGV:
    case SIGFPE:
    default:
      giram_fatal_error((gchar *)g_strsignal(sig_num));
      break;
  }
}

/* giram core signal handler for death-of-child signals */

static void giram_sigchld_handler(gint sig_num)
{
  gint pid;
  gint status;

  while (TRUE)
  {
    pid = waitpid(WAIT_ANY, &status, WNOHANG);

    if (pid <= 0)
      break;
  }
}

