/* LoadPov.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#include "giram.h"
#include "utils.h"
#include "frame.h"
#include "primitives/bicubic_patch.h"
#include "primitives/box.h"
#include "primitives/cone.h"
#include "primitives/cylinder.h"
#include "primitives/disc.h"
#include "primitives/heightfield.h"
#include "primitives/mesh.h"
#include "primitives/plane.h"
#include "primitives/quadric.h"
#include "primitives/sor.h"
#include "primitives/sphere.h"
#include "primitives/superellipsoid.h"
#include "primitives/torus.h"
#include "primitives/triangle.h"
#include "csg.h"
#include "object.h"
#include "camera.h"
#include "csgtree.h"
#include "view.h"
#include "camera_view.h"
#include "copy.h"
#include "widgets/giramviewshell.h"
#include "LoadPov.h"

#include "color.h"

#include "giramrc.h"
#include "giramintl.h"

#define SMALL_VALUE    10e-10
#define EPSILON    10e-10

TOKEN_IDS Conversion_Util_Table[LAST_TOKEN];

struct { TOKEN_IDS Token; char *String; } ReservedWordTable[LAST_TOKEN] =
{
  { Taa_threshold, "aa_threshold"},
  { Taa_level, "aa_level"},
  { Tabsorption, "absorption"},
  { Tabs, "abs"},
  { Taccuracy, "accuracy"},
  { Tacosh, "acosh"},
  { Tacos, "acos"},
  { Tadaptive, "adaptive"},
  { Tadc_bailout, "adc_bailout"},
  { Tagate, "agate"},
  { Tagate_turb, "agate_turb"},
  { Tall_intersections, "all_intersections"},
  { Tall, "all"},
  { Talpha, "alpha"},
  { Talways_sample, "always_sample"},
  { Taltitude, "altitude"},
  { Tambient_light, "ambient_light"},
  { Tambient, "ambient"},
  { Tampersand, "&"},
  { Tangle, "angle"},
  { Taperture, "aperture"},
  { Tappend, "append"},
  { Tarc_angle, "arc_angle"},
  { Tarea_light, "area_light"},
  { TArrayID, "array identifier"},
  { Tarray, "array"},
  { Tascii, "ascii"},
  { Tasc, "asc"},
  { Tasinh, "asinh"},
  { Tasin, "asin"},
  { Tassumed_gamma, "assumed_gamma"},
  { Tatan2, "atan2"},
  { Tatanh, "atanh"},
  { Tatan, "atan"},
  { Tat, "@"},
  { Taverage, "average"},
  { Tbackground, "background"},
  { Tback_quote, "`"},
  { Tback_slash, "\\"},
  { Tbar, "|"},
  { Tbezier_spline, "bezier_spline"},
  { Tbicubic_patch, "bicubic_patch"},
  { Tblack_hole, "black_hole"},
  { Tblob, "blob"},
  { Tblue, "blue"},
  { Tblur_samples, "blur_samples"},
  { Tbounded_by, "bounded_by"},
  { Tboxed, "boxed"},
  { Tbox, "box"},
  { Tbozo, "bozo"},
  { Tbreak, "break"},
  { Tbrick_size, "brick_size"},
  { Tbrick, "brick"},
  { Tbrightness, "brightness" },
  { Tbrilliance, "brilliance"},
  { Tbumps, "bumps"},
  { Tbump_map, "bump_map"},
  { Tbump_size, "bump_size"},
  { Tb_spline, "b_spline"},
  { TCameraID, "camera identifier"},
  { Tcamera, "camera"},
  { Tcase, "case"},
  { Tcaustics, "caustics"},
  { Tceil, "ceil"},
  { Tcells, "cells"},
  { Tcharset, "charset"},
  { Tchecker, "checker"},
  { Tchr, "chr"},
  { Tcircular,"circular"},
  { Tclipped_by, "clipped_by"},
  { Tclock_delta, "clock_delta"},
  { Tclock_on, "clock_on"},
  { Tclock, "clock"},
  { Tcolon, ":"},
  { TColorID, "colour identifier"},
  { Tcolour_key, "color keyword"},
  { TColor_mapID, "colour map identifier"},
  { Tcolour_map, "color_map"},
  { Tcolour_map, "colour_map"},
  { Tcolor, "color"},
  { Tcolour, "colour"},
  { Tcomma, ", "},
  { Tcomponent, "component"},
  { Tcomposite, "composite"},
  { Tconcat, "concat"},
  { Tcone, "cone"},
  { Tconfidence, "confidence"},
  { Tconic_sweep, "conic_sweep"},
  { Tcontained_by,"contained_by"},
  { Tcontrol0, "control0"},
  { Tcontrol1, "control1"},
  { Tcoords, "coords" },
  { Tcosh, "cosh"},
  { Tcos, "cos"},
  { Tcount, "count" },
  { Tcrackle, "crackle"},
  { Tcrand, "crand"},
  { Tcube, "cube"},
  { Tcubic_spline, "cubic_spline"},
  { Tcubic, "cubic"},
  { Tcubic_wave, "cubic_wave"},
  { Tcylinder, "cylinder"},
  { Tcylindrical, "cylindrical"},
  { Tdash, "-"},
  { Tdebug, "debug"},
  { Tdeclare, "declare"},
  { Tdefault, "default"},
  { Tdefined, "defined"},
  { Tdegrees, "degrees"},
  { Tdensity_file, "density_file"},
  { TDensityID, "density identifier"},
  { TDensity_mapID, "density_map identifier"},
  { Tdensity_map, "density_map"},
  { Tdensity, "density"},
  { Tdents, "dents"},
  { Tdf3, "df3"},
  { Tdifference, "difference"},
  { Tdiffuse, "diffuse"},
  { Tdimensions, "dimensions"},
  { Tdimension_size, "dimension_size"},
  { Tdirection, "direction"},
  { Tdisc, "disc"},
  { Tdispersion, "dispersion"},
  { Tdispersion_samples, "dispersion_samples"},
  { Tdistance_maximum, "distance_maximum" },
  { Tdistance, "distance"},
  { Tdist_exp, "dist_exp"},
  { Tdiv, "div"},
  { Tdollar, "$"},
  { Tdouble_illuminate, "double_illuminate"},
  { Teccentricity, "eccentricity"},
  { Telse, "else"},
  { Temission, "emission"},
  { TEmptyArray, "empty array"},
  { TEOF, "End of File"},
  { Tend, "end"},
  { Tequals, "="},
  { Terror_bound, "error_bound" },
  { Terror, "error"},
  { Tevaluate, "evaluate"},
  { Texclamation, "!"},
  { Texp, "exp"},
  { Texponent, "exponent"},
  { Texterior, "exterior"},
  { Textinction, "extinction"},
  { Tfacets, "facets"},
  { Tface_indices, "face_indices"},
  { Tfade_colour, "fade_colour"},
  { Tfade_colour, "fade_color"},
  { Tfade_distance, "fade_distance"},
  { Tfade_power, "fade_power"},
  { Tfalloff_angle, "falloff_angle"},
  { Tfalloff, "falloff"},
  { Tfalse, "false"},
  { Tfclose, "fclose"},
  { Tfile_exists, "file_exists"},
  { TFileID, "file identifier"},
  { Tfill_light, "shadowless"},
  { Tfilter, "filter"},
  { Tfinalclock, "final_clock"},
  { Tfinalframe, "final_frame"},
  { TFinishID, "finish identifier"},
  { Tfinish, "finish"},
  { Tfisheye, "fisheye"},
  { Tflatness, "flatness"},
  { Tflip, "flip"},
  { Tfloat_funct, "float function"},
  { TFloatID, "float identifier"},
  { TFloatLiteral, "float constant"},
  { Tfloor, "floor"},
  { Tfocal_point, "focal_point"},
  { Tfog_alt, "fog_alt"},
  { TFogID, "fog identifier"},
  { Tfog_offset, "fog_offset"},
  { Tfog, "fog"},
  { Tfog_type, "fog_type"},
  { Tfopen, "fopen"},
  { Tform, "form"},
  { Tframenumber, "frame_number"},
  { Tfrequency, "frequency"},
  { Tfresnel, "fresnel"},
  { Tfunction, "function"},
  { TFunctID, "function identifier"},
  { TVectFunctID, "vector function identifier"},
  { Tgif, "gif"},
  { Tglobal_settings, "global_settings" },
  { Tgradient, "gradient"},
  { Tgranite, "granite"},
  { Tgray, "gray"},
  { Tgray_threshold, "gray_threshold" },
  { Tgreen, "green"},
  { Thash, "#"},
  { That, "^"},
  { Theight_field, "height_field"},
  { Thexagon, "hexagon"},
  { Thf_gray_16, "hf_gray_16" },
  { Thierarchy, "hierarchy"},
  { Thollow, "hollow"},
  { Thypercomplex, "hypercomplex"},
  { TIdentifier, "undeclared identifier"},
  { Timage_width,"image_width"},
  { Timage_height,"image_height"},
  { Tinitialclock, "initial_clock"},
  { Tinitialframe, "initial_frame"},
  { Tinterior_texture, "interior_texture"},
  { Tifdef, "ifdef"},
  { Tiff, "iff"},
  { Tifndef, "ifndef"},
  { Tif, "if"},
  { Timage_map, "image_map"},
  { Timage_pattern, "image_pattern"},
  { Tinclude, "include"},
  { Tinside, "inside"},
  { Tinside_vector, "inside_vector"},
  { TInteriorID, "interior identifier"},
  { Tinterior, "interior"},
  { Tinterpolate, "interpolate"},
  { Tintersection, "intersection"},
  { Tintervals, "intervals"},
  { Tint, "int"},
  { Tinverse, "inverse"},
  { Tior, "ior"},
  { Tirid, "irid"},
  { Tirid_wavelength, "irid_wavelength"},
  { Tisosurface, "isosurface"},
  { Tjitter, "jitter"},
  { Tjulia, "julia"},
  { Tjulia_fractal, "julia_fractal"},
  { Tjpeg, "jpeg"},
  { Tlambda, "lambda"},
  { Tlathe, "lathe"},
  { Tleft_angle, "<"},
  { Tleft_curly, "{"},
  { Tleft_paren, "("},
  { Tleft_square, "["},
  { Tleopard, "leopard"},
  { Tlight_group, "light_group"},
  { Tlight_source, "light_source"},
  { Tlinear_spline, "linear_spline"},
  { Tlinear_sweep, "linear_sweep"},
  { Tln, "ln"},
  { Tload_file, "load_file"},
  { Tlocal, "local"},
  { Tlocation, "location"},
  { Tlog, "log"},
  { Tlooks_like, "looks_like"},
  { Tlook_at, "look_at"},
  { Tlow_error_factor, "low_error_factor" },
  { TMacroID, "macro identifier"},
  { Tmacro, "macro"},
  { Tmagnet, "magnet"},
  { Tmajor_radius, "major_radius"},
  { Tmandel, "mandel"},
  { Tmap_type, "map_type"},
  { Tmarble, "marble"},
  { TMaterialID, "material identifier"},
  { Tmaterial_map, "material_map"},
  { Tmaterial, "material"},
  { Tmatrix, "matrix"},
  { Tmax_extent, "max_extent"},
  { Tmax_gradient, "max_gradient"},
  { Tmax_intersections, "max_intersections"},
  { Tmax_iteration, "max_iteration"},
  { Tmax_sample, "max_sample"},
  { Tmax, "max"},
  { Tmax_trace_level, "max_trace_level"},
  { Tmax_trace, "max_trace"},
  { Tmedia_attenuation, "media_attenuation"},
  { TMediaID, "media identifier"},
  { Tmedia_interaction, "media_interaction"},
  { Tmedia, "media"},
  { Tmerge, "merge"},
  { Tmesh2, "mesh2"},
  { Tmesh, "mesh"},
  { Tmetallic, "metallic"},
  { Tmethod, "method"},
  { Tmetric, "metric" },
  { Tminimum_reuse, "minimum_reuse" },
  { Tmin_extent, "min_extent" },
  { Tmin, "min"},
  { Tmod, "mod"},
  { Tmortar, "mortar"},
  { Tnatural_spline, "natural_spline"},
  { Tnearest_count, "nearest_count" },
  { Tnormal_indices, "normal_indices"},
  { TNormalMapID, "normal_map identifier"},
  { Tnormal_map, "normal_map"},
  { Tnormal_vectors, "normal_vectors"},
  { Tno_image, "no_image"},
  { Tno_reflection, "no_reflection"},
  { Tno_shadow, "no_shadow"},
  { Tno, "no"},
  { Tnumber_of_waves, "number_of_waves"},
  { TObjectID, "object identifier"},
  { Tobject, "object"},
  { Toctaves, "octaves"},
  { Toffset, "offset"},
  { Toff, "off"},
  { Tomega, "omega"},
  { Tomnimax, "omnimax"},
  { Tonce, "once"},
  { Tonion, "onion"},
  { Ton, "on"},
  { Topen, "open"},
  { Torientation, "orientation"},
  { Torient,"orient"},
  { Torthographic, "orthographic"},
  { Tpanoramic, "panoramic"},
  { Tparallel, "parallel"},
  { TParameterID, "parameter identifier"},
  { Tparametric, "parametric"},
  { Tpattern, "pattern"},
  { Tpercent, "%"},
  { Tperiod, ". (period)"},
  { Tperspective, "perspective"},
  { Tpgm, "pgm"},
  { Tphase, "phase"},
  { Tphong_size, "phong_size"},
  { Tphong, "phong"},
  { TPigmentID, "pigment identifier"},
  { TPigmentMapID, "pigment_map identifier"},
  { Tpigment_map, "pigment_map"},
  { Tpigment, "pigment"},
  { Tpi, "pi"},
  { Tplanar, "planar"},
  { Tplane, "plane"},
  { Tplus, "+"},
  { Tpng, "png"},
  { Tpoint_at, "point_at"},
  { Tpolygon, "polygon"},
  { Tpoly, "poly"},
  { Tpoly_wave, "poly_wave"},
  { Tpot, "pot"},
  { Tpow, "pow"},
  { Tppm, "ppm"},
  { Tprecision, "precision"},
  { Tprecompute,"precompute"},
  { Tpretrace_start, "pretrace_start"},
  { Tpretrace_end, "pretrace_end"},
  { Tprism, "prism"},
  { Tprod, "prod"},
  { Tprojected_through, "projected_through"},
  { Tpwr, "pwr"},
  { Tquadratic_spline, "quadratic_spline"},
  { Tquadric, "quadric"},
  { Tquartic, "quartic"},
  { Tquaternion, "quaternion"},
  { Tquestion, "?"},
  { Tquick_colour, "quick_color"},
  { Tquick_colour, "quick_colour"},
  { Tquilted, "quilted"},
  { Tradial, "radial"},
  { Tradians, "radians"},
  { Tradiosity, "radiosity" },
  { Tradius, "radius"},
  { TRainbowID, "rainbow identifier"},
  { Trainbow, "rainbow"},
  { Tramp_wave, "ramp_wave"},
  { Trand, "rand"},
  { Trange, "range"},
  { Tratio, "ratio"},
  { Tread, "read"},
  { Treciprocal, "reciprocal" },
  { Trecursion_limit, "recursion_limit" },
  { Tred, "red"},
  { Treflection_exponent, "reflection_exponent"},
  { Treflection, "reflection"},
  { Trefraction, "refraction"},
  { Trel_ge, ">="},
  { Trel_le, "<="},
  { Trel_ne, "!="},
  { Trender, "render"},
  { Trepeat, "repeat"},
  { Trgbft, "rgbft"},
  { Trgbf, "rgbf"},
  { Trgbt, "rgbt"},
  { Trgb, "rgb"},
  { Tright_angle, ">"},
  { Tright_curly, "}"},
  { Tright_paren, ")"},
  { Tright_square, "]"},
  { Tright, "right"},
  { Tripples, "ripples"},
  { Trotate, "rotate"},
  { Troughness, "roughness"},
  { Tsamples, "samples"},
  { Tsave_file, "save_file"},
  { Tscale, "scale"},
  { Tscallop_wave, "scallop_wave"},
  { Tscattering, "scattering"},
  { Tseed, "seed"},
  { Tselect, "select"},
  { Tsemi_colon, ";"},
  { Tsine_wave, "sine_wave"},
  { Tsingle_quote, "'"},
  { Tsinh, "sinh"},
  { Tsin, "sin"},
  { Tsize, "size" },
  { TSkysphereID, "sky_sphere identifier"},
  { Tskysphere, "sky_sphere"},
  { Tsky, "sky"},
  { Tslash, "/"},
  { Tslice, "slice"},
  { TSlopeMapID, "slope_map identifier"},
  { Tslope_map, "slope_map"},
  { Tslope,"slope"},
  { Tsmooth, "smooth"},
  { Tsmooth_triangle, "smooth_triangle"},
  { Tsolid, "solid" },
  { Tsor, "sor"},
  { Tspecular, "specular"},
  { Tsphere_sweep, "sphere_sweep"},
  { Tsphere, "sphere"},
  { Tspherical, "spherical"},
  { Tspiral1, "spiral1"},
  { Tspiral2, "spiral2"},
  { TSplineID, "spline identifier"},
  { Tspline, "spline"},
  { Tspotlight, "spotlight"},
  { Tspotted, "spotted"},
  { Tsqrt, "sqrt"},
  { Tsqr, "sqr"},
  { Tstar, "*"},
  { Tstatistics, "statistics"},
  { Tstrcmp, "strcmp"},
  { Tstrength, "strength"},
  { TStringID, "string identifier"},
  { TStringLiteral, "string literal"},
  { Tstrlen, "strlen"},
  { Tstrlwr, "strlwr"},
  { Tstrupr, "strupr"},
  { Tstr, "str"},
  { Tsturm, "sturm"},
  { Tsubstr, "substr"},
  { Tsum, "sum"},
  { Tsuperellipsoid, "superellipsoid"},
  { Tswitch, "switch"},
  { Tsys, "sys"},
  { Ttanh, "tanh"},
  { Ttan, "tan"},
  { TTemporaryMacroID, "unfinished macro declaration"},
  { TTextureID, "texture identifier"},
  { Ttexture_list, "texture_list"},
  { TTextureMapID, "texture_map identifier"},
  { Ttexture_map, "texture_map"},
  { Ttexture, "texture"},
  { Ttext, "text"},
  { Ttga, "tga"},
  { Tthickness, "thickness"},
  { Tthreshold, "threshold"},
  { Ttiff, "tiff"},
  { Ttightness, "tightness"},
  { Ttilde, "~"},
  { Ttile2, "tile2"},
  { Ttiles, "tiles"},
  { TNormalID, "normal identifier"},
  { Tnormal, "normal"},
  { Ttolerance, "tolerance"},
  { Ttoroidal, "toroidal"},
  { Ttorus, "torus"},
  { Ttrace, "trace"},
  { TTransformID, "transform identifier"},
  { Ttransform, "transform"},
  { Ttranslate, "translate"},
  { Ttransmit, "transmit"},
  { Ttriangle, "triangle"},
  { Ttriangle_wave, "triangle_wave"},
  { Ttrue, "true"},
  { Tttf, "ttf"},
  { Tturbulence, "turbulence"},
  { Tturb_depth, "turb_depth"},
  { Ttype, "type"},
  { Tt, "t"},
  { Tultra_wide_angle, "ultra_wide_angle"},
  { Tundef, "undef"},
  { Tunion, "union"},
  { Tup, "up"},
  { Tuse_alpha, "use_alpha"},
  { Tuse_colour, "use_color"},
  { Tuse_colour, "use_colour"},
  { Tuse_index, "use_index"},
  { Tutf8, "utf8"},
  { TuvID, "uv vector identifier"},
  { Tuv_indices, "uv_indices"},
  { Tuv_mapping, "uv_mapping"},
  { Tuv_vectors, "uv_vectors"},
  { Tu_steps, "u_steps"},
  { Tu, "u"},
  { Tval, "val"},
  { Tvariance, "variance"},
  { Tvaxis_rotate, "vaxis_rotate"},
  { Tvcross, "vcross"},
  { Tvdot, "vdot"},
  { TVector4DID, "4d-vector identifier"},
  { Tvector_funct, "vector function"},
  { TVectorID, "vector identifier"},
  { Tversion, "version"},
  { Tvertex_vectors, "vertex_vectors"},
  { Tvlength, "vlength"},
  { Tvnormalize, "vnormalize"},
  { Tvrotate, "vrotate"},
  { Tvstr, "vstr"},
  { Tvturbulence, "vturbulence"},
  { Tv_steps, "v_steps"},
  { Tv, "v"},
  { Twarning, "warning"},
  { Twarp, "warp"},
  { Twater_level, "water_level"},
  { Twaves, "waves"},
  { Twhile, "while"},
  { Twidth, "width"},
  { Twood, "wood"},
  { Twrinkles, "wrinkles"},
  { Twrite, "write"},
  { Tx, "x"},
  { Tyes, "yes"},
  { Ty, "y"},
  { Tz, "z"},
  /* NK phmap */
  { Tphotons, "photons"},
  { Tsteps,"steps"},
  { Tpass_through, "pass_through"},
  { Tcollect, "collect"},
  { Tautostop, "autostop"},
  { Tgather, "gather"},
  { Tsplit_union, "split_union"},
  { Texpand_thresholds, "expand_thresholds"},
  { Tspacing, "spacing"},
#ifdef GLOBAL_PHOTONS
  { Tglobal, "global"},
#endif
  { Ttarget, "target"},
  { Tconserve_energy, "conserve_energy"},
  { Tcutaway_textures, "cutaway_textures"},
  { Tpigment_pattern,"pigment_pattern"},
  { Tno_bump_scale, "no_bump_scale"},
  { Tglobal_lights, "global_lights"},
  { Tinternal, "internal"},
  { Tnoise_generator, "noise_generator"}
};

ListFile *PovFile;
int       OldChar, IsOldChar;
int       OldToken, IsOldToken;
double    PovVersionLangage;
double    LastFloat;
char     *LastString = NULL;

char             *last_constant_name;
SymbolTableEntry *last_constant_entry;

double         PovClock = 0.0;
static guint   NbPovRandomTable;
static gulong *PovRandomTable;
int            ParsingDirective;
int            DontParse;
int            local_table_index;

typedef struct ParsedVector
{
  Vector Val;
  int    nbTerms;
} ParsedVector;

#define IF_TRUE_IF_ELSE           1
#define IF_TRUE_ELSE_END          2
#define IF_FALSE_IF_ELSE          3
#define IF_FALSE_ELSE_END         4
#define WHILE                     5
#define SWITCH_SWITCH_CASE        6
#define SWITCH_CASE_BREAK_TRUE    7
#define SWITCH_CASE_BREAK_FALSE   8
#define SWITCH_BREAK_END          9
#define UNPARSED                 10

#define PARSE_COMMA if (parse_comma()) return 1;
#define PARSE_OPEN_CURLY get_token();         \
                         if (Token.token_id != Tleft_curly) \
                         {                           \
                           g_message(_("%s, line %d\nA '{' was expected here...\n"), PovFile->Name, PovFile->Line); \
                           return 1; \
                         }
#define PARSE_VECTOR(x) if (parse_vector(&(x))) return 1;
#define PARSE_FLOAT(x) if (parse_float(&(x))) return 1;

#define PARSE_LEFT_PAREN(x) if (paren_error((x), Tleft_paren)) return 1;
#define PARSE_RIGHT_PAREN(x) if (paren_error((x), Tright_paren)) return 1;

typedef struct CondDirective
{
  struct CondDirective *Next;
  int                   Style;
  double                SwitchValue;
  int                   WhileLineNo;
  ListFile             *WhileFile;
  long                  WhilePos;
} CondDirective;

CondDirective *ConditionalStack;

#define COND_STACK_SIZE (200)

typedef enum cond_type
{
  ROOT_COND=0,
  WHILE_COND,
  IF_TRUE_COND,
  IF_FALSE_COND,
  ELSE_COND,
  SWITCH_COND,
  CASE_TRUE_COND,
  CASE_FALSE_COND,
  SKIP_TIL_END_COND,
  INVOKING_MACRO_COND,
  DECLARING_MACRO_COND
} COND_TYPE;

typedef struct TokenStruct
{
  TOKEN_IDS  token_id;
  TOKEN_IDS  function_id;
  int        token_line_no;
  int        local_table_index;
  char      *token_string;
  double     token_float;
  gboolean   unget_token;
  gboolean   end_of_file;
  char      *filename;
  void      *data;
  int       *number_ptr;
  void     **data_ptr;
  gboolean   is_array_elem;
} TokenStruct;

#define MAX_PARAMETER_LIST (56)

typedef struct PovMacroStruct
{
  char *Macro_Name;
  char *Macro_Filename;
  long  Macro_Pos,Macro_Line_No,Macro_End;
  int   Num_Of_Pars;
  char *Par_Name[MAX_PARAMETER_LIST];
} PovMacroStruct;

typedef struct Cond_Stack_Entry CS_ENTRY;

struct Cond_Stack_Entry
{
  COND_TYPE       Cond_Type;
  double          Switch_Value;
  ListFile       *While_File;
  ListFile       *Macro_File;
  char           *Macro_Return_Name;
  int             Macro_Same_Flag;
  PovMacroStruct *PMac;
  long            Pos,Line_No;
};

static CS_ENTRY *Cond_Stack;
static int CS_Index;
static gboolean Skipping;
static int Got_EOF;
static gboolean Inside_Ifdef;
static gboolean Inside_MacroDef;
static gboolean Ok_To_Declare;

GString       *string;
TokenStruct    Token;
FrameStruct   *CurrentFrame;

static void write_token(TOKEN_IDS id);
static void get_token(void);
static int parse_float(double *pLocalFloat);
static int parse_color(Vector TmpVector);
static int parse_pigment(PigmentStruct **pLocalPigment);
static int parse_finish(FinishStruct **pLocalFinish);
static int parse_rel_factor(ParsedVector *TmpParsedVector);
static int parse_object(ObjectStruct **pLocalObject);
static int parse_light_source(LightSourceStruct **pLocalLightSource);
static int parse_texture(TextureStruct **pLocalTexture);
static int parse_camera(CameraStruct **pLocalCamera);
static void skip_tokens(COND_TYPE cond);
int Parse_RValue(int Previous, int *NumberPtr, void **DataPtr, SymbolTableEntry *entry,
                 gboolean ParFlag, gboolean SemiFlag, gboolean is_local, gboolean allow_redefine, int old_table_index);

/*****************************************************************************
*  next_char
******************************************************************************/
static int next_char(void)
{
  if (IsOldChar)
  {
    IsOldChar = 0;
    return OldChar;
  }
  return getc(PovFile->handle);
}

/*****************************************************************************
*  ignore_space
******************************************************************************/
static int ignore_space(void)
{
  int c;

  do
  {
    c = next_char();
    if (c=='\n')
    {
      while (c=='\n')
      {
        PovFile->Line++;
        c = next_char();
        if (c=='\r') c = next_char();
      }
    } else if (c=='\r')
    {
      while (c=='\r')
      {
        PovFile->Line++;
        c = next_char();
        if (c=='\n') c = next_char();
      }
    }
  } while (c==' ' || c=='\t' || c=='' || c=='\r' || c=='\n');
  OldChar = c; IsOldChar = 1;
  return 0;
}

/*****************************************************************************
*  parse_comma
******************************************************************************/
static int parse_comma(void)
{
  get_token();
  if (Token.token_id != Tcomma)
    Token.unget_token = TRUE;
  
  return 0; /* FIXME */
}

/*****************************************************************************
*  parse_float_literal
******************************************************************************/
static int parse_float_literal(void)
{
  int c;

  g_string_assign(string, "");

  c = next_char();
  if (c == '.')
  {
    g_string_assign(string, "0.");
    c = next_char();
    if (!isdigit(c))
    {
      g_message(_("%s, line %d\nMisformed float\n"), PovFile->Name, PovFile->Line);
      return 1;
    }
    while (isdigit(c))
    {
      g_string_append_c(string, c);
      c = next_char();
    }
    OldChar = c; IsOldChar = 1;
  } else
  { /* c = [0-9] */
    while (isdigit(c))
    {
      g_string_append_c(string, c);
      c = next_char();
    }
    if (c=='.')
    {
      g_string_append_c(string, c);
      c = next_char();
      if (!isdigit(c))
        g_string_append_c(string, '0');
      while (isdigit(c))
      {
        g_string_append_c(string, c);
        c = next_char();
      }
    }
    OldChar = c; IsOldChar = 1;
  }
  /* At this point, FloatText is either
     0.[0-9]+, [0-9]+, [0-9]+.0 or [0-9]+.[0-9]+ */
  c = next_char();
  if (c=='e' || c=='E')
  {
    g_string_append_c(string, 'e');
    c = next_char();
    if (c=='+' || c=='-')
    {
      g_string_append_c(string, c);
      c = next_char();
    }
    if (!isdigit(c))
    {
      g_message(_("%s, line %d\nMisformed float\n"),
                PovFile->Name, PovFile->Line);
      return 1;
    }
    while (isdigit(c))
    {
      g_string_append_c(string, c);
      c = next_char();
    }
  }
  OldChar = c; IsOldChar = 1;
  write_token(TFloatLiteral);
  Token.token_float = atof(string->str);
  return 0;
}

/*****************************************************************************
*  paren_error
******************************************************************************/
static int paren_error(char *Function, TOKEN_IDS id)
{
  get_token();
  if (Token.token_id != id)
  {
    g_message(_("%s, line %d\nThe argument of '%s' must be enclosed between parentheses...\n"),
              PovFile->Name, PovFile->Line, Function);
    return 1;
  }
  return 0;
}

/*****************************************************************************
*  write_token
******************************************************************************/
static void write_token(TOKEN_IDS id)
{
  Token.token_line_no = PovFile->Line;
  Token.filename      = PovFile->Name;
  Token.token_string  = NULL;
  Token.data          = NULL;
  Token.token_id      = Conversion_Util_Table[id];
  Token.function_id   = id;
}

/*****************************************************************************
*  parse_string_literal
******************************************************************************/
static void parse_string_literal(void)
{
  int c;

  g_string_assign(string, "");
  c = next_char();
  while(c != '"')
  {
    if (c == EOF)
      g_message(_("%s, line %d\nUnfinished string\n"),
                        PovFile->Name, PovFile->Line);
    if (c == '\\')
    {
      switch (c = next_char())
      {
        case '\n':
        case '\r':
        case EOF:
          g_message(_("%s, line %d\nUnfinished string\n"),
                    PovFile->Name, PovFile->Line);
          break;
        case '\"':
          c = '\"';
          break;
        default:
          g_string_append_c(string, '\\');
      }
      g_string_append_c(string, c);
    } else
    {
      g_string_append_c(string, c);
    }
    c = next_char();
  }
  write_token(TStringLiteral);
  Token.token_string = g_strdup(string->str);
}

/*****************************************************************************
*  parse_string
******************************************************************************/
static int parse_string(char **pLocalString)
{
  char    TmpString1[100], TmpString2[100];
  char   *AppendString;
  int     i;
  double  Val1, Val2, Val3;

  get_token();

  switch (Token.token_id)
  {
    case TStringLiteral:
      *pLocalString = Token.token_string;
      break;

    case Tstr:
      PARSE_LEFT_PAREN("str")
      PARSE_FLOAT(Val1)
      PARSE_COMMA
      PARSE_FLOAT(Val2)
      PARSE_COMMA
      PARSE_FLOAT(Val3)
      PARSE_RIGHT_PAREN("str")
      if (Val3>=0)
      {
        if (Val2>0)
          sprintf(TmpString1, "%%%d.%df", (int)Val2, (int)Val3);
        else if (Val2 < 0)
          sprintf(TmpString1, "%%0%d.%df", (int)-Val2, (int)Val3);
        else
          sprintf(TmpString1, "%%.%df", (int)Val3);
      } else
      {
        if (Val2>0)
          sprintf(TmpString1, "%%%df",(int)Val2);
        else if (Val2 < 0)
          sprintf(TmpString1, "%%0%df",(int)-Val2);
        else
          strcpy(TmpString1, "%f");
      }
      sprintf(TmpString2, TmpString1, Val1);
      *pLocalString = g_malloc(strlen(TmpString2)+1);
      strcpy(*pLocalString, TmpString2);
      break;

    case Tconcat:
      PARSE_LEFT_PAREN("concat")
      if (parse_string(pLocalString)) return 1;
      get_token();
      while(Token.token_id != Tright_paren)
      {
        PARSE_COMMA
        parse_string(&AppendString);
        *pLocalString = g_realloc(*pLocalString, strlen(*pLocalString)+strlen(AppendString)+1);
        strcat(*pLocalString, AppendString);
        g_free(AppendString);
        get_token();
      }
      break;

    case Tchr:
      PARSE_LEFT_PAREN("chr")
      PARSE_FLOAT(Val1)
      PARSE_RIGHT_PAREN("chr")
      if (Val1 < 0.0 || Val1 >= 256.0)
      {
        g_message(_("%s, line %d\nBad value for chr() : %g...\n"),
                  PovFile->Name, PovFile->Line, Val1);
        return 1;
      }
      *pLocalString = g_malloc(2);
      *pLocalString[0] = (char)((int)Val1);
      *pLocalString[1] = 0;
      break;

    case Tsubstr:
      PARSE_LEFT_PAREN("substr")
      if (parse_string(&AppendString))
        return 1;
      PARSE_COMMA
      PARSE_FLOAT(Val1)
      PARSE_COMMA
      PARSE_FLOAT(Val2)
      PARSE_RIGHT_PAREN("substr")
      if (((Val1+Val2-1.0) > strlen(AppendString)) ||
         (Val1<0.0) ||
         (Val2<1.0))
      {
        g_message(_("%s, line %d\nBad params in substr(%s,%g,%g)...\n"),
                  PovFile->Name, PovFile->Line, AppendString, Val1, Val2);
        return 1;
      }
      *pLocalString = g_malloc((guint)Val2+1);
      strncpy(*pLocalString,&(AppendString[(int)Val1-1]),(guint)Val2);
      *pLocalString[(int)Val2]='\0';
      g_free(AppendString);
      break;

    case Tstrupr:
      PARSE_LEFT_PAREN("strupr")
      if (parse_string(pLocalString))
        return 1;
      PARSE_RIGHT_PAREN("strupr")
      for (i=0 ; i<(gint)strlen(*pLocalString) ; i++)
        if (*pLocalString[i]>='a' && *pLocalString[i] <='z')
          *pLocalString[i]-=32;
      break;

    case Tstrlwr:
      PARSE_LEFT_PAREN("strlwr")
      if (parse_string(pLocalString))
        return 1;
      PARSE_RIGHT_PAREN("strlwr")
      for (i=0 ; i<(gint)strlen(*pLocalString) ; i++)
        if (*pLocalString[i]>='A' && *pLocalString[i] <='Z')
          *pLocalString[i]+=32;
      break;

    case TStringID:
      *pLocalString = g_strdup(last_constant_entry->data);
      break;

    default:
      g_message(_("%s, line %d\nA String was expected here...\n"),
                PovFile->Name, PovFile->Line);
      return 1;
  }
  return 0;
}

/*****************************************************************************
*  parse_vector
******************************************************************************/
static int parse_vector(ParsedVector *TmpParsedVector)
{
  if (parse_rel_factor(TmpParsedVector))
    return 1;
  if (TmpParsedVector->nbTerms>3)
  {
    g_message(_("%s, line %d\nExpecting a vector but got a color expression...\n"),
              PovFile->Name, PovFile->Line);
    return 1;
  }
  if (TmpParsedVector->nbTerms==1)
  {
    TmpParsedVector->Val[1] = TmpParsedVector->Val[2] = TmpParsedVector->Val[0];
  } else if (TmpParsedVector->nbTerms==2)
  {
    TmpParsedVector->Val[2] = 0.0;
  }
  TmpParsedVector->nbTerms = 3;
  return 0;
}

/*****************************************************************************
*  parse_rel_term
******************************************************************************/
static int parse_rel_term(ParsedVector *TmpParsedVector)
{
  ParsedVector SecondFactor;

  if (parse_rel_factor(TmpParsedVector))
    return 1;
  get_token();
  while ((Token.token_id==Tleft_angle) || (Token.token_id==Trel_le) ||
         (Token.token_id==Tequals)     || (Token.token_id==Trel_ne) ||
         (Token.token_id==Trel_ge)     || (Token.token_id==Tright_angle))
  {
    if (parse_rel_factor(&SecondFactor))
      return 1;
    TmpParsedVector->nbTerms = 1;
    switch(Token.token_id)
    {
      case Tleft_angle:
        TmpParsedVector->Val[0] =
               (TmpParsedVector->Val[0] < SecondFactor.Val[0]) ? 1.0 : 0.0;
        break;
      case Trel_le:
        TmpParsedVector->Val[0] =
               (TmpParsedVector->Val[0] <= SecondFactor.Val[0]) ? 1.0 : 0.0;
        break;
      case Tequals:
        TmpParsedVector->Val[0] =
               (TmpParsedVector->Val[0] == SecondFactor.Val[0]) ? 1.0 : 0.0;
        break;
      case Trel_ne:
        TmpParsedVector->Val[0] =
               (TmpParsedVector->Val[0] != SecondFactor.Val[0]) ? 1.0 : 0.0;
        break;
      case Trel_ge:
        TmpParsedVector->Val[0] =
               (TmpParsedVector->Val[0] >= SecondFactor.Val[0]) ? 1.0 : 0.0;
        break;
      case Tright_angle:
        TmpParsedVector->Val[0] =
               (TmpParsedVector->Val[0] > SecondFactor.Val[0]) ? 1.0 : 0.0;
        break;
      default:
       /* panic!!! */
    }
    get_token();
  }
  Token.unget_token = TRUE;
  return 0;
}

/*****************************************************************************
*  parse_logical
******************************************************************************/
static int parse_logical(ParsedVector *TmpParsedVector)
{
  ParsedVector SecondTerm;

  if (parse_rel_term(TmpParsedVector))
    return 1;
  get_token();
  while ((Token.token_id==Tampersand) || (Token.token_id==Tbar))
  {
    if (parse_rel_term(&SecondTerm))
      return 1;
    TmpParsedVector->nbTerms = 1;
    if (Token.token_id==Tampersand)
    {
      TmpParsedVector->Val[0] = TmpParsedVector->Val[0] && SecondTerm.Val[0];
    } else
    {
      TmpParsedVector->Val[0] = TmpParsedVector->Val[0] || SecondTerm.Val[0];
    }
    get_token();
  }
  Token.unget_token = TRUE;
  return 0;
}

/*****************************************************************************
*  parse_express
******************************************************************************/
static int parse_express(ParsedVector *TmpParsedVector)
{
  ParsedVector LocalVector;
  ParsedVector LocalVector1;
  ParsedVector LocalVector2;
  int          i;

  if (parse_logical(&LocalVector))
    return 1;
  get_token();
  if (Token.token_id == Tquestion)
  {
    if (LocalVector.nbTerms != 1)
    {
      g_message(_("%s, line %d\nConditional must evaluate to a float...\n"),
                PovFile->Name, PovFile->Line);
      return 1;
    }
    if (parse_express(&LocalVector1))
      return 1;
    get_token();
    if (Token.token_id != Tcolon)
    {
      g_message(_("%s, line %d\nConditional must be followed by a colon...\n"),
                PovFile->Name, PovFile->Line);
      return 1;
    }
    if (parse_express(&LocalVector2))
      return 1;
    if (LocalVector.Val[0])
    {
      TmpParsedVector->nbTerms = LocalVector1.nbTerms;
      for (i=0 ; i<TmpParsedVector->nbTerms ; i++)
        TmpParsedVector->Val[i] = LocalVector1.Val[i];
    } else
    {
      TmpParsedVector->nbTerms = LocalVector2.nbTerms;
      for (i=0 ; i<TmpParsedVector->nbTerms ; i++)
        TmpParsedVector->Val[i] = LocalVector2.Val[i];
    }
  } else
  {
    Token.unget_token = TRUE;
    TmpParsedVector->nbTerms = LocalVector.nbTerms;
    for (i=0 ; i<TmpParsedVector->nbTerms ; i++)
      TmpParsedVector->Val[i] = LocalVector.Val[i];
  }
  return 0;
}

/*****************************************************************************
*  parse_num_factor
******************************************************************************/
static int parse_num_factor(ParsedVector *TmpParsedVector)
{
  int c, i;
  ParsedVector LocalVector;
  ParsedVector LocalVector2;
  double Val1, Val2;
  FILE *f;
  char *LocalString, *LocalString2;

  get_token();
  while (Token.token_id == Tplus)
  {
    get_token();
  }
  switch (Token.function_id)
  {
    case Tabs:
      PARSE_LEFT_PAREN("abs")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'abs' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = fabs(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("abs")
      break;

    case Tacos:
      PARSE_LEFT_PAREN("acos")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'acos' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      if (fabs(LocalVector.Val[0]) > 1.0)
      {
        g_message(_("%s, line %d\nThe argument of 'acos' must be between -1 and 1...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = acos(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("acos")
      break;

    case Tval:
      PARSE_LEFT_PAREN("val")
      if (parse_string(&LocalString))
        return 1;
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = atof(LocalString);
      g_free(LocalString);
      PARSE_RIGHT_PAREN("val")
      break;

    case Tasc:
      PARSE_LEFT_PAREN("asc")
      if (parse_string(&LocalString))
        return 1;
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = LocalString[0];
      g_free(LocalString);
      PARSE_RIGHT_PAREN("asc")
      break;

    case Tasin:
      PARSE_LEFT_PAREN("asin")
        return 1;
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'asin' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      if (fabs(LocalVector.Val[0]) > 1.0)
      {
        g_message(_("%s, line %d\nThe argument of 'asin' must be between -1 and 1...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = asin(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("asin")
      break;

    case Tatan2:
      PARSE_LEFT_PAREN("atan2")
      PARSE_FLOAT(Val1)
      PARSE_COMMA
      PARSE_FLOAT(Val2)
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = atan2(Val1,Val2);
      PARSE_RIGHT_PAREN("atan2")
      break;

    case Tceil:
      PARSE_LEFT_PAREN("ceil")
        return 1;
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'ceil' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = ceil(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("ceil")
      break;

    case Tclock: /* FIXME: should be special... */
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = PovClock;
      break;

    case Tcos:
      PARSE_LEFT_PAREN("cos")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'cos' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = cos(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("cos")
      break;

    case Tdegrees:
      PARSE_LEFT_PAREN("degrees")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'degrees' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = LocalVector.Val[0]/M_PI*180.0;
      PARSE_RIGHT_PAREN("degrees")
      break;

    case Tdiv:
      PARSE_LEFT_PAREN("div")
      PARSE_FLOAT(Val1)
      PARSE_COMMA
      PARSE_FLOAT(Val2)
      if (fabs(Val2) <= 1.0e-6)
      {
        g_message(_("%s, line %d\nThe second argument of 'div' mustn't be nul...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = (double)((int)(Val1 / Val2));
      PARSE_RIGHT_PAREN("div")
      break;

    case Texp:
      PARSE_LEFT_PAREN("exp")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'exp' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = exp(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("exp")
      break;

    case Tfile_exists:
      PARSE_LEFT_PAREN("file_exists")
      if (parse_string(&LocalString))
        return 1;
      TmpParsedVector->nbTerms = 1;
      f = LibOpen(LocalString);
      if (f)
      {
        TmpParsedVector->Val[0] = 1.0;
        fclose(f);
      }
      else
        TmpParsedVector->Val[0] = 0.0;
      g_free(LocalString);
      PARSE_RIGHT_PAREN("file_exists")
      break;

    case TFloatID:
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = *((double *)( last_constant_entry->data ));
      break;

    case TFloatLiteral:
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = Token.token_float;
      break;

    case Tfloor:
      PARSE_LEFT_PAREN("floor")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'floor' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = floor(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("floor")
      break;

    case Tint:
      PARSE_LEFT_PAREN("int")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'int' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = (double)((int)LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("int")
      break;

    case Tlog:
      PARSE_LEFT_PAREN("log")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'log' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      if (TmpParsedVector->Val[0] <= 0.0)
      {
        g_message(_("%s, line %d\nThe argument of 'log' must be strictly positive...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = log(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("log")
      break;

    case Tmin:
      PARSE_LEFT_PAREN("min")
      PARSE_FLOAT(Val1)
      PARSE_COMMA
      PARSE_FLOAT(Val2)
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = Val1>Val2?Val2:Val1;
      PARSE_RIGHT_PAREN("min")
      break;

    case Tmax:
      PARSE_LEFT_PAREN("max")
      PARSE_FLOAT(Val1)
      PARSE_COMMA
      PARSE_FLOAT(Val2)
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = Val1<Val2?Val2:Val1;
      PARSE_RIGHT_PAREN("max")
      break;

    case Tmod:
      PARSE_LEFT_PAREN("mod")
      PARSE_FLOAT(Val1)
      PARSE_COMMA
      PARSE_FLOAT(Val2)
      if (fabs(Val2) <= 1.0e-6)
      {
        g_message(_("%s, line %d\nThe second argument of 'mod' mustn't be nul...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = fmod(Val1,Val2);
      PARSE_RIGHT_PAREN("mod")
      break;

    case Tpi:
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = M_PI;
      break;

    case Tpow:
      PARSE_LEFT_PAREN("pow")
      PARSE_FLOAT(Val1)
      PARSE_COMMA
      PARSE_FLOAT(Val2)
      if (Val1 <= 0.0 )
        TmpParsedVector->Val[0] = pow(Val1, Val2);
      else /* FIXME ??? */
        TmpParsedVector->Val[0] = pow(Val1, Val2);
      TmpParsedVector->nbTerms = 1;
      PARSE_RIGHT_PAREN("pow")
      break;

    case Tradians:
      PARSE_LEFT_PAREN("radians")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'radians' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = LocalVector.Val[0] * M_PI / 180.0;
      PARSE_RIGHT_PAREN("radians")
      break;

    case Tsin:
      PARSE_LEFT_PAREN("sin")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'sin' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = sin(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("sin")
      break;

    case Tsqrt:
      PARSE_LEFT_PAREN("sqrt")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'sqrt' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      if (LocalVector.Val[0] < 0.0)
      {
        g_message(_("%s, line %d\nThe argument of 'sqrt' must be postive...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = sqrt(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("sqrt")
      break;

    case Tstrcmp:
      PARSE_LEFT_PAREN("strcmp")
      if (parse_string(&LocalString))
        return 1;
      PARSE_COMMA
      if (parse_string(&LocalString2))
        return 1;
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = strcmp(LocalString, LocalString2);
      g_free(LocalString);
      g_free(LocalString2);
      PARSE_RIGHT_PAREN("strcmp")
      break;

    case Tstrlen:
      PARSE_LEFT_PAREN("strlen")
      if (parse_string(&LocalString))
        return 1;
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = strlen(LocalString);
      g_free(LocalString);
      PARSE_RIGHT_PAREN("strlen")
      break;

    case Ttan:
      PARSE_LEFT_PAREN("tan")
      if (parse_express(&LocalVector))
        return 1;
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'tan' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = tan(LocalVector.Val[0]);
      PARSE_RIGHT_PAREN("tan")
      break;

    case Tvdot:
      PARSE_LEFT_PAREN("vdot")
      PARSE_VECTOR(LocalVector)
      PARSE_COMMA
      PARSE_VECTOR(LocalVector2)
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = V3DDot(LocalVector.Val, LocalVector2.Val);
      PARSE_RIGHT_PAREN("vdot")
      break;

    case Tvlength:
      PARSE_LEFT_PAREN("vlength")
      PARSE_VECTOR(LocalVector)
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = V3DLength(LocalVector.Val);
      PARSE_RIGHT_PAREN("vlength")
      break;

    case Tversion:
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = PovVersionLangage;
      break;

    case Ttrue:
    case Tyes:
    case Ton:
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = 1.0;
      break;

    case Tfalse:
    case Tno:
    case Toff:
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = 0.0;
      break;

    case Tseed:
      PARSE_LEFT_PAREN("seed")
      if (parse_express(&LocalVector))
        return 1;
      PARSE_RIGHT_PAREN("seed")
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'seed' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      PovRandomTable = g_realloc(PovRandomTable,
                                 (NbPovRandomTable+1)*sizeof(unsigned long));
      PovRandomTable[NbPovRandomTable] = (unsigned long)LocalVector.Val[0];
      NbPovRandomTable++;
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] = NbPovRandomTable-1;
      break;

    case Trand:
      PARSE_LEFT_PAREN("rand")
      if (parse_express(&LocalVector))
        return 1;
      PARSE_RIGHT_PAREN("rand")
      if (LocalVector.nbTerms != 1)
      {
        g_message(_("%s, line %d\nThe argument of 'rand' must be a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      PovRandomTable[(int)LocalVector.Val[0]] =
             PovRandomTable[(int)LocalVector.Val[0]] * 1812433253L + 12345L;
      TmpParsedVector->nbTerms = 1;
      TmpParsedVector->Val[0] =
             ((double)(PovRandomTable[(int)LocalVector.Val[0]] & 0xFFFFFFFFUL)
              / 0xFFFFFFFFUL);
      break;

    case Tvaxis_rotate:
      PARSE_LEFT_PAREN("vaxis_rotate")
      PARSE_VECTOR(LocalVector)
      PARSE_COMMA
      PARSE_VECTOR(LocalVector2)
      PARSE_COMMA
      PARSE_FLOAT(Val1)
      PARSE_RIGHT_PAREN("vaxis_rotate")
      /* FIXME */
      g_message(_("%s, line %d\nThe function 'vaxis_rotate' not yet implemented\n"),
                PovFile->Name, PovFile->Line);
      return 1;
      break;

    case Tvcross:
      PARSE_LEFT_PAREN("vcross")
      PARSE_VECTOR(LocalVector)
      PARSE_COMMA
      PARSE_VECTOR(LocalVector2)
      PARSE_RIGHT_PAREN("vcross")
      TmpParsedVector->nbTerms = 3;
      VCross(TmpParsedVector->Val, LocalVector.Val, LocalVector2.Val);
      break;

    case TVectorID:
      TmpParsedVector->nbTerms = 3;
      V3Dcopy(TmpParsedVector->Val, (double *)( last_constant_entry->data ));
      break;

    case Tvnormalize:
      PARSE_LEFT_PAREN("vnormalize")
      if (parse_vector(TmpParsedVector))
        return 1;
      PARSE_RIGHT_PAREN("vnormalize")
      Val1 = V3DLength(TmpParsedVector->Val);
      if (Val1 != 0.0)
      {
        TmpParsedVector->Val[0] /= Val1;
        TmpParsedVector->Val[1] /= Val1;
        TmpParsedVector->Val[2] /= Val1;
      }
      break;

    case Tvrotate:
      g_message(_("%s, line %d\nThe function 'vrotate' not yet implemented\n"),
                PovFile->Name, PovFile->Line);
      return 1;
      /* FIXME */
      break;

    case Tx:
      TmpParsedVector->nbTerms = 3;
      V3Deq(TmpParsedVector->Val, 1.0, 0.0, 0.0);
      break;

    case Ty:
      TmpParsedVector->nbTerms = 3;
      V3Deq(TmpParsedVector->Val, 0.0, 1.0, 0.0);
      break;

    case Tz:
      TmpParsedVector->nbTerms = 3;
      V3Deq(TmpParsedVector->Val, 0.0, 0.0, 1.0);
      break;

    case TColorID:
      TmpParsedVector->nbTerms = 5;
      V5Dcopy(TmpParsedVector->Val, (double *)( last_constant_entry->data ));
      break;

    case Tt:
      TmpParsedVector->nbTerms = 4;
      TmpParsedVector->Val[0] = 0.0;
      TmpParsedVector->Val[1] = 0.0;
      TmpParsedVector->Val[2] = 0.0;
      TmpParsedVector->Val[3] = 1.0;
      break;

    case Tu:
      TmpParsedVector->nbTerms = 2;
      TmpParsedVector->Val[0] = 1.0;
      TmpParsedVector->Val[1] = 0.0;
      break;

    case Tv:
      TmpParsedVector->nbTerms = 2;
      TmpParsedVector->Val[0] = 0.0;
      TmpParsedVector->Val[1] = 1.0;
      break;

    case Tdash:
      if (parse_num_factor(TmpParsedVector))
        return 1;
      for (i=0 ; i<5 ; i++)
        TmpParsedVector->Val[i] = -TmpParsedVector->Val[i];
      break;

    case Texclamation:
      if (parse_num_factor(TmpParsedVector))
        return 1;
      for (i=0 ; i<TmpParsedVector->nbTerms ; i++)
        TmpParsedVector->Val[i] = fabs(TmpParsedVector->Val[i])>10e-6?0.0:1.0;
      break;

    case Tleft_paren:
      if (parse_express(TmpParsedVector))
        return 1;
      if (paren_error(_("Left Parenthese"), Tright_paren))
        return 1;
      break;

    case Tleft_angle:
      if (parse_float(&(TmpParsedVector->Val[0])))
        return 1;
      PARSE_COMMA
      get_token();
      if (Token.token_id == Tright_angle)
      {
        TmpParsedVector->nbTerms = 1;
      } else
      {
        Token.unget_token = TRUE;
        if (parse_float(&(TmpParsedVector->Val[1])))
          return 1;
        PARSE_COMMA
        get_token();
        if (Token.token_id == Tright_angle)
        {
          TmpParsedVector->nbTerms = 2;
        } else
        {
          Token.unget_token = TRUE;
          if (parse_float(&(TmpParsedVector->Val[2])))
            return 1;
          PARSE_COMMA
          get_token();
          if (Token.token_id == Tright_angle)
          {
            TmpParsedVector->nbTerms = 3;
          } else
          {
            Token.unget_token = TRUE;
            if (parse_float(&(TmpParsedVector->Val[3])))
              return 1;
            PARSE_COMMA
            get_token();
            if (Token.token_id == Tright_angle)
            {
              TmpParsedVector->nbTerms = 4;
            } else
            {
              Token.unget_token = TRUE;
              if (parse_float(&(TmpParsedVector->Val[4])))
                return 1;
              PARSE_COMMA
              get_token();
              if (Token.token_id == Tright_angle)
              {
                TmpParsedVector->nbTerms = 5;
              } else
              {
                g_message(_("%s, line %d\nVectors can only have 5 terms or less...\n"),
                          PovFile->Name, PovFile->Line);
                return 1;
              }
            }
          }
        }
      }
      break;

    default:
      g_message(_("%s, line %d\nNumeric Expression expected...\n"),
                PovFile->Name, PovFile->Line);
      return 1;
  }
  if (ignore_space())
    return 1;
  c = next_char();
  if (c != '.')
  {
    IsOldChar = 1; OldChar = c;
    return 0;
  }
  get_token();
  switch (Token.token_id)
  {
    case Tx:
      if (TmpParsedVector->nbTerms != 3)
      {
        g_message(_("%s, line %d\nThe 'x' suffix can only be used with vectors...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
      }
      break;

    case Ty:
      if (TmpParsedVector->nbTerms != 3)
      {
        g_message(_("%s, line %d\nThe 'y' suffix can only be used with vectors...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
        TmpParsedVector->Val[0] = TmpParsedVector->Val[1];
      }
      break;

    case Tz:
      if (TmpParsedVector->nbTerms != 3)
      {
        g_message(_("%s, line %d\nThe 'z' suffix can only be used with vectors...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
        TmpParsedVector->Val[0] = TmpParsedVector->Val[2];
      }
      break;

    case Tred:
      if (TmpParsedVector->nbTerms != 5)
      {
        g_message(_("%s, line %d\nThe 'red' suffix can only be used with colors...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
      }
      break;

    case Tgreen:
      if (TmpParsedVector->nbTerms != 5)
      {
        g_message(_("%s, line %d\nThe 'green' suffix can only be used with colors...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
        TmpParsedVector->Val[0] = TmpParsedVector->Val[1];
      }
      break;

    case Tblue:
      if (TmpParsedVector->nbTerms != 5)
      {
        g_message(_("%s, line %d\nThe 'blue' suffix can only be used with colors...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
        TmpParsedVector->Val[0] = TmpParsedVector->Val[2];
      }
      break;

    case Tfilter:
      if (TmpParsedVector->nbTerms != 5)
      {
        g_message(_("%s, line %d\nThe 'filter' suffix can only be used with colors...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
        TmpParsedVector->Val[0] = TmpParsedVector->Val[3];
      }
      break;

    case Ttransmit:
      if (TmpParsedVector->nbTerms != 5)
      {
        g_message(_("%s, line %d\nThe 'transmit' suffix can only be used with colors...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
        TmpParsedVector->Val[0] = TmpParsedVector->Val[4];
      }
      break;

    case Tu:
      TmpParsedVector->nbTerms = 1;
      break;

    case Tv:
      if (TmpParsedVector->nbTerms == 1)
      {
        g_message(_("%s, line %d\nThe 'v' suffix can't be used with float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
        TmpParsedVector->Val[0] = TmpParsedVector->Val[1];
      }
      break;

    case Tt:
      if (TmpParsedVector->nbTerms < 4)
      {
        g_message(_("%s, line %d\nThe 't' suffix can't be used there...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      } else
      {
        TmpParsedVector->nbTerms = 1;
        TmpParsedVector->Val[0] = TmpParsedVector->Val[3];
      }
      break;

    default:
      g_message(_("%s, line %d\nA dot suffix was expected here...\n"),
                PovFile->Name, PovFile->Line);
  }
  return 0;
}

/*****************************************************************************
*  parse_num_term
******************************************************************************/
static int parse_num_term(ParsedVector *TmpParsedVector)
{
  ParsedVector SecondFactor;
  int          i;

  if (parse_num_factor(TmpParsedVector))
    return 1;
  get_token();
  while ((Token.token_id==Tslash) || (Token.token_id==Tstar))
  {
    if (parse_num_factor(&SecondFactor))
      return 1;
    if (SecondFactor.nbTerms != TmpParsedVector->nbTerms)
    {
      if (SecondFactor.nbTerms > TmpParsedVector->nbTerms)
      {
        if (TmpParsedVector->nbTerms == 1)
          for (i=1 ; i<SecondFactor.nbTerms ; i++)
            TmpParsedVector->Val[i] = TmpParsedVector->Val[0];
        else
          for (i=TmpParsedVector->nbTerms ; i<SecondFactor.nbTerms ; i++)
            TmpParsedVector->Val[i] = 0.0;
        TmpParsedVector->nbTerms = SecondFactor.nbTerms;
      } else
      {
        if (SecondFactor.nbTerms == 1)
          for (i=1 ; i<TmpParsedVector->nbTerms ; i++)
            SecondFactor.Val[i] = SecondFactor.Val[0];
        else
          for (i=SecondFactor.nbTerms ; i<TmpParsedVector->nbTerms ; i++)
            SecondFactor.Val[i] = 0.0;
      }
    }
    if (Token.token_id==Tstar)
    {
      for (i=0 ; i<TmpParsedVector->nbTerms ; i++)
        TmpParsedVector->Val[i] *= SecondFactor.Val[i];
    } else
    {
      for (i=0 ; i<TmpParsedVector->nbTerms ; i++)
        TmpParsedVector->Val[i] /= SecondFactor.Val[i];
    }
    get_token();
  }
  Token.unget_token = TRUE;
  return 0;
}

/*****************************************************************************
*  parse_rel_factor
******************************************************************************/
static int parse_rel_factor(ParsedVector *TmpParsedVector)
{
  ParsedVector SecondTerm;
  int          i;

  if (parse_num_term(TmpParsedVector))
    return 1;
  get_token();
  while ((Token.token_id==Tplus) || (Token.token_id==Tdash))
  {
    if (parse_num_term(&SecondTerm))
      return 1;
    if (SecondTerm.nbTerms != TmpParsedVector->nbTerms)
    {
      if (SecondTerm.nbTerms > TmpParsedVector->nbTerms)
      {
        if (TmpParsedVector->nbTerms == 1)
          for (i=1 ; i<SecondTerm.nbTerms ; i++)
            TmpParsedVector->Val[i] = TmpParsedVector->Val[0];
        else
          for (i=TmpParsedVector->nbTerms ; i<SecondTerm.nbTerms ; i++)
            TmpParsedVector->Val[i] = 0.0;
        TmpParsedVector->nbTerms = SecondTerm.nbTerms;
      } else
      {
        if (SecondTerm.nbTerms == 1)
          for (i=1 ; i<TmpParsedVector->nbTerms ; i++)
            SecondTerm.Val[i] = SecondTerm.Val[0];
        else
          for (i=SecondTerm.nbTerms ; i<TmpParsedVector->nbTerms ; i++)
            SecondTerm.Val[i] = 0.0;
      }
    }
    if (Token.token_id==Tplus)
    {
      for (i=0 ; i<TmpParsedVector->nbTerms ; i++)
        TmpParsedVector->Val[i] += SecondTerm.Val[i];
    } else
    {
      for (i=0 ; i<TmpParsedVector->nbTerms ; i++)
        TmpParsedVector->Val[i] -= SecondTerm.Val[i];
    }
    get_token();
  }
  Token.unget_token = TRUE;
  return 0;
}

/*****************************************************************************
*  parse_float
******************************************************************************/
static int parse_float(double *pLocalFloat)
{
  ParsedVector TmpParsedVector;

  parse_rel_factor(&TmpParsedVector);
  if (TmpParsedVector.nbTerms != 1)
  {
    g_message(_("%s, line %d\nExpecting a Float but got a vector...\n"),
              PovFile->Name, PovFile->Line);
    return 1;
  }
  *pLocalFloat = TmpParsedVector.Val[0];
  return 0;
}

/*****************************************************************************
*  parse_word
******************************************************************************/
static void parse_word(void)
{
  int c, i;

  g_string_assign(string, "");
  c = next_char();
  while((c>='a' && c<='z') || (c>='A' && c<='Z') || isdigit(c) || c=='_')
  {
    g_string_append_c(string, c);
    c = next_char();
  }
  OldChar = c; IsOldChar = 1;

  /* if we're inside the parsing of an #ifdef directive, we can stop here, since
     we'll analyse the string afterwards */
  if (Inside_Ifdef)
  {
    Token.token_id      = TIdentifier;
    Token.is_array_elem = FALSE;

    return;
  }
  
  for (i=0 ; i<G_N_ELEMENTS(ReservedWordTable) ; i++)
  { /* XXX: not very optimized so far... */
    if (!strcmp(ReservedWordTable[i].String, string->str))
    {
      write_token(ReservedWordTable[i].Token);
      Token.token_string = string->str;
      return;
    }
  }


  if (!Skipping)
  {
    last_constant_name = g_strdup(string->str);
    for (i = local_table_index ; i >=0 ; i--)
    {
      last_constant_entry = g_hash_table_lookup(CurrentFrame->all_constants[i], string->str);
      if (last_constant_entry)
      {
        write_token(last_constant_entry->type);
        Token.token_string = string->str;
        return;
      }
    }
  }
  /*
  if (*pToken == -1)
  {
    g_message(_("%s, line %d\nUnknown keyword : '%s'...\n"),
              PovFile->Name, PovFile->Line, Word);
    return 1;
  }*/
  write_token(TIdentifier);
  Token.token_string = string->str;
  return;
}

/*****************************************************************************
*  get_type_id
******************************************************************************/
static int get_type_id(char *Name)
{
  SymbolTableEntry  *entry;
  gint               i;

  for (i=local_table_index ; i>=0 ; i--)
  {
    entry = g_hash_table_lookup(CurrentFrame->all_constants[i], Name);
    if (entry)
      return entry->type;
  }

  return 0;
}

/*****************************************************************************
*  parse_declare
*  Parsing of "#declare Name = blah" sort of things
******************************************************************************/
static void parse_declare(gboolean local, gboolean after_hash)
{
  int                PreviousTypeId;
  int                declare_table_index;
  gboolean           local_flag;
  gboolean           done;
  gboolean           allow_redefine = TRUE;
  SymbolTableEntry  *entry = NULL;
  int                previous = -1;

  Ok_To_Declare = FALSE;

  if (!after_hash)
  {
    g_print("declare and local must be preceded by an hash\n");
  }
   
  /* support for local variables */
  local_flag = (Token.token_id == Tlocal);
  if (local_flag)
    declare_table_index = local_table_index;
  else
    declare_table_index = 0;
  
  done = FALSE;
  while (!done)
  {
    get_token();
    switch (Token.token_id)
    {
      case TIdentifier:
        allow_redefine  = !Token.is_array_elem;
        entry = g_new(SymbolTableEntry, 1);
        g_hash_table_insert(CurrentFrame->all_constants[declare_table_index],
                            g_strdup(Token.token_string),
                            entry);
        entry->type = TIdentifier;
        Token.number_ptr = &(entry->type);
        Token.data_ptr = &(entry->data);
        previous        = Token.token_id;
        done = TRUE;
        break;

      case TFunctID: case TVectFunctID:
        if((!Token.is_array_elem) || (*(Token.data_ptr) != NULL))
        {
          /* Error! */
          g_print("Redeclaring functions is not allowed - #undef the function first!");
        }
        // fall through
        // These are also used in Parse_Directive UNDEF_TOKEN section and Parse_Macro,
        // and all three functions should accept exactly the same identifiers! [trf]
      case TNormalID: case TFinishID: case TTextureID: case TObjectID:
      case TColor_mapID: case TTransformID: case TCameraID: case TPigmentID:
      case TSlopeMapID: case TNormalMapID: case TTextureMapID: case TColorID:
      case TPigmentMapID: case TMediaID: case TStringID: case TInteriorID:
      case TDensity_mapID: case TArrayID: case TDensityID: case TuvID:
      case TVector4DID: case TRainbowID: case TFogID: case TSkysphereID:
      case TMaterialID: case TSplineID:
        allow_redefine  = !Token.is_array_elem;
        if (local_flag && (Token.local_table_index != local_table_index))
        {
          entry = g_new(SymbolTableEntry, 1);
          g_hash_table_insert(CurrentFrame->all_constants[declare_table_index],
                              g_strdup(Token.token_string),
                              entry);
          entry->type = TIdentifier;
          Token.number_ptr = &(entry->type);
          Token.data_ptr   = &(entry->data);
          previous        = TIdentifier;
        } else
        {
          previous        = Token.token_id;
        }
        done = TRUE;
        break;

      case TEmptyArray:
        allow_redefine  = !Token.is_array_elem;
        previous = Token.token_id;
        done = TRUE;
        break;

      case Tvector_funct: case Tfloat_funct:
        allow_redefine  = !Token.is_array_elem;
        switch(Token.function_id)
        {
          case TVectorID:
          case TFloatID:
            if (local_flag && (Token.local_table_index != local_table_index))
            {
              entry = g_new(SymbolTableEntry, 1);
              g_hash_table_insert(CurrentFrame->all_constants[declare_table_index],
                                  g_strdup(Token.token_string),
                                  entry);
              entry->type      = TIdentifier;
              Token.number_ptr = &(entry->type);
              Token.data_ptr   = &(entry->data);
            }
            previous           = Token.function_id;
            break;

          default:
            /* Parse_Error(TIdentifier);*/
            break;
        }
        done = TRUE;
        break;

      default:
        allow_redefine  = !Token.is_array_elem;
        /*Parse_Error(TIdentifier);*/
        break;
    }
  }

  /*LValue_Ok = false;*/
  get_token();
  if (Token.token_id != Tequals)
  {
    /* Error */
  }
  Ok_To_Declare = TRUE;
  if (!Parse_RValue(previous, Token.number_ptr, Token.data_ptr, entry, FALSE, TRUE, local, allow_redefine, 256))
  {
    /* Expectation_Error("RValue to declare");FIXME!*/
  }
  if ( after_hash )
  {
    Ok_To_Declare = FALSE;
    get_token();
    if (Token.token_id != Tsemi_colon)
      Token.unget_token = TRUE;
    Ok_To_Declare = TRUE;
  }
}

/*************************************************************************
*  Parse_RValue
**************************************************************************/
int Parse_RValue(int Previous, int *NumberPtr, void **DataPtr, SymbolTableEntry *entry,
                 gboolean ParFlag, gboolean SemiFlag, gboolean is_local, gboolean allow_redefine, int old_table_index)
{
/*  EXPRESS Local_Express;*/
  Vector5D *Local_Colour;
/*  PIGMENT *Local_Pigment;
  TNORMAL *Local_Tnormal;
  FINISH *Local_Finish;
  TEXTURE *Local_Texture, *Temp_Texture;
  TRANSFORM *Local_Trans;
  OBJECT *Local_Object;
  CAMERA *Local_Camera;
  IMEDIA *Local_Media;
  PIGMENT *Local_Density;
  INTERIOR *Local_Interior;
  MATERIAL *Local_Material;
  void *Temp_Data;
  POV_PARAM *New_Par;*/
  int Found=TRUE;
  int Temp_Count=30000;
  int Old_Ok=Ok_To_Declare;
  int Terms;
  gboolean function_identifier ;
  gboolean callable_identifier ;
  gboolean had_callable_identifier ;
  gboolean done, done2;

  done = FALSE;
  while (!done)
  {
    get_token();
    switch (Token.token_id)
    {
#if 0
      case TNormalID: case TFinishID: case TTextureID: case TObjectID:
      case TColor_mapID: case TTransformID: case TCameraID: case TPigmentID:
      case TSlopeMapID: case TNormalMapID: case TTextureMapID: case TArrayID:
      case TPigmentMapID: case TMediaID: case TInteriorID: case TDensityID:
      case TDensity_mapID: case TRainbowID: case TFogID: case TSkysphereID:
      case TMaterialID: case TStringID:
        if ((ParFlag) && (Token.Table_Index <= old_table_index))
        { // pass by reference
          New_Par            = (POV_PARAM *)POV_MALLOC(sizeof(POV_PARAM),"parameter");
          New_Par->NumberPtr = Token.NumberPtr;
          New_Par->DataPtr   = Token.DataPtr;
          New_Par->Table_Index = Token.Table_Index;
          *NumberPtr = PARAMETER_ID_TOKEN;
          *DataPtr   = (void *)New_Par;
        } else
        { // pass by value
          Temp_Data  = (void *) Copy_Identifier((void *)*Token.DataPtr,*Token.NumberPtr);
          *NumberPtr = *Token.NumberPtr;
          Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
          *DataPtr   = Temp_Data;
        }
        done = TRUE;
        break;

      case TIdentifier:
        if (ParFlag)
        {
          Error("Cannot pass uninitialized identifier as macro parameter.\nInitialize identifier first.");
        } else
        {
          Error("Cannot assign uninitialized identifier.");
        }
        done = TRUE;
        break;
#endif

#if 0 /* DDX decembre 2002 */
      case Tcolor: case Tcolour_key: case TColorID:
        Token.unget_token = TRUE;
        if (Token.token_id != TColorID)
        {
          Local_Colour  = color_new();
          Ok_To_Declare = FALSE;
          parse_color(*Local_Colour);
          if (SemiFlag)
          {
            get_token();
            if (Token.token_id != Tsemi_colon)
            {
              /* Error */
              g_print("A semicolon was expected here.\n");
            }
          }
          Ok_To_Declare = TRUE;
          *NumberPtr    = TColorID;
          //Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
          *DataPtr      = (void *) Local_Colour;
          g_print("got a color constant!: %s = <%g, %g, %g, %g, %g>\n",
                  "nom", (*Local_Colour)[0], (*Local_Colour)[1], (*Local_Colour)[2],
                  (*Local_Colour)[3], (*Local_Colour)[4]);
          done = TRUE;
          break;
        } // intentional to allow color dot expressions as macro parameters if #version is 3.5 or higher [trf]

      case TVectFunctID: case Tvector_funct: case Tleft_angle:
      case Tu: case Tv: case TuvID: case TVector4DID: case TSplineID:
      case Tleft_paren: case Tfloat_funct:
      case Tplus: case Tdash: case TFunctID: case Texclamation:
        Token.unget_token = TRUE;           
        // It seems very few people understand what is going on here, so let me try to
        // explain it. All comments below are mine and they are based on how I think it
        // works and understand it. As I didn't write most of the code I cannot really
        // tell for sure, so if anybody finds incorrect comments please let me know!
        // BTW, when saying #declare it always implies "or #local" :-)

        // determine the type of the first identifier
        function_identifier = (Token.token_id==TFunctID) || (Token.token_id==TVectFunctID);
        callable_identifier = (Token.token_id==TFunctID) || (Token.token_id==TVectFunctID) || (Token.token_id==TSplineID);
        // don't allow #declares from here
        Ok_To_Declare = FALSE;

        // if what follows could be a function/spline call or
        // is a macro parameter taking a float, vector or ids
        // of a float, vector or color then count the tokens
        // found between now and the time when the function
        // Parse_Unknown_Vector returns
        if (callable_identifier || (ParFlag &&
            (((Token.token_id==Tfloat_funct) && (Token.function_id==TFloatID)) ||
             ((Token.token_id==Tvector_funct) &&  (Token.function_id==TVectorID)) ||
              (Token.token_id==TVector4DID) || (Token.token_id==TuvID) || (Token.token_id==TColorID))))
        {
          /*Temp_Count=token_count;FIXME*/
        }
        // assume no callable identifier (that is a function or spline identifier) has been found
        had_callable_identifier = FALSE;

        // parse the expression and determine if it was a callable identifier
        Terms = Parse_Unknown_Vector (Local_Express, TRUE, &had_callable_identifier);

        // if in a #declare force a semicolon at the end
        if (SemiFlag)
          Parse_Semi_Colon(TRUE);

        // get the number of tokens found
        /*Temp_Count -= token_count; FIXME: implement token_count */
        Temp_Count = -1;

        // no tokens have been found or a fucntion call had no parameters in parenthesis
        if (!((Temp_Count==-1) || (Temp_Count==1000)) && had_callable_identifier)
          Error("Identifier expected, incomplete function call or spline call found instead.");

        // only one identifier token has been found so pass it by reference
        if (((Temp_Count==-1) || (Temp_Count==1000)) && (Token.Table_Index <= old_table_index))
        {
          // It is important that functions are passed by value and not by reference! [trf]
          if (!(ParFlag) || (ParFlag && function_identifier))
          { // pass by value
            Temp_Data  = (void *) Copy_Identifier((void *)*Token.DataPtr,*Token.NumberPtr);
            *NumberPtr = *Token.NumberPtr;
            Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
            *DataPtr   = Temp_Data;
          } else
          { // pass by reference
            New_Par            = (POV_PARAM *)POV_MALLOC(sizeof(POV_PARAM),"parameter");
            New_Par->NumberPtr = Token.NumberPtr;
            New_Par->DataPtr   = Token.DataPtr;
            New_Par->Table_Index = Token.Table_Index;

            *NumberPtr = TParameterID;
            *DataPtr   = (void *)New_Par;
          }
        } else // an expression has been found, so create a new identifier
        {
          switch (Terms)
          {
            case 1:
              *NumberPtr = TFloatID;
              Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
              *DataPtr   = (void *) Create_Float();
              *((DBL *)*DataPtr)  = Local_Express[X];
              break;

            case 2:
              *NumberPtr = TuvID;
              Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
              *DataPtr   = (void *) Create_UV_Vect();
              Assign_UV_Vect(*DataPtr, Local_Express);
              break;

            case 3:
              *NumberPtr = TVectorID;
              Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
              *DataPtr   = (void *) Create_Vector();
              Assign_Vector(*DataPtr, Local_Express);
              break;

            case 4:
              *NumberPtr = TVector4DID;
              Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
              *DataPtr   = (void *) Create_Vector_4D();
              Assign_Vector_4D(*DataPtr, Local_Express);
              break;

            case 5:
              *NumberPtr    = TColorID;
              Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
              *DataPtr      = (void *) Create_Colour();
              Assign_Colour_Express((COLC*)(*DataPtr), Local_Express);
              break;
          }
        }

        // allow #declares again
        Ok_To_Declare = TRUE;
        done = TRUE;
        break;
#endif
#if 0
      case Tpigment:
        Local_Pigment = Copy_Pigment((Default_Texture->Pigment));
        Parse_Begin ();
        Parse_Pigment (&Local_Pigment);
        Parse_End ();
        *NumberPtr = PIGMENT_ID_TOKEN;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = (void *)Local_Pigment;
        done = TRUE;
        break;

      case Tnormal:
        Local_Tnormal = Copy_Tnormal((Default_Texture->Tnormal));
        Parse_Begin ();
        Parse_Tnormal (&Local_Tnormal);
        Parse_End ();
        *NumberPtr = TNORMAL_ID_TOKEN;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = (void *) Local_Tnormal;
        done = TRUE;
        break;

      case Tfinish:
        Local_Finish = Copy_Finish((Default_Texture->Finish));
        Parse_Finish (&Local_Finish);
        *NumberPtr = FINISH_ID_TOKEN;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = (void *) Local_Finish;
        done = TRUE;
        break;

      case Tcamera:
        Local_Camera = Copy_Camera(Default_Camera);
        Parse_Camera (&Local_Camera);
        *NumberPtr = CAMERA_ID_TOKEN;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = (void *) Local_Camera;
        done = TRUE;
        break;

      case Ttexture:
        Parse_Begin ();
        Local_Texture = Parse_Texture ();
        Parse_End ();
        Temp_Texture=NULL;
        Link_Textures(&Temp_Texture, Local_Texture);
        Ok_To_Declare = FALSE;
        done2 = FALSE;
        while (!done2)
        {
          get_token();
          switch (Token.token_id)
          {
            case Ttexture:
              Parse_Begin ();
              Local_Texture = Parse_Texture ();
              Parse_End ();
              Link_Textures(&Temp_Texture, Local_Texture);
              break;

            default:
              Token.unget_token = TRUE;
              done2 = TRUE;
              break;
          }
        }
 
        *NumberPtr    = TTextureID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr      = (void *)Temp_Texture;
        Ok_To_Declare = TRUE;
        done = TRUE;
        break;

      case Tcolour_map:
        Temp_Data=(void *) Parse_Colour_Map ();
        *NumberPtr = TColor_mapID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tpigment_map:
        Temp_Data  = (void *) Parse_Blend_Map (PIGMENT_TYPE,NO_PATTERN);
        *NumberPtr = TPigmentMapID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tspline:
        Experimental_Flag |= EF_SPLINE;
        Parse_Begin();
        Temp_Data=(char *) Parse_Spline();
        Parse_End();
        *NumberPtr = TSplineID;
        Test_Redefine( Previous, NumberPtr, *DataPtr , allow_redefine);
        *DataPtr = (void *)Temp_Data;
        done = TRUE;
        break;

      case Tdensity_map:
        Temp_Data  = (void *) Parse_Blend_Map (DENSITY_TYPE,NO_PATTERN);
        *NumberPtr = TDensity_mapID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tslope_map:
        Temp_Data  = (void *) Parse_Blend_Map (SLOPE_TYPE,NO_PATTERN);
        *NumberPtr = TSlopeMapID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Ttexture_map:
        Temp_Data  = (void *) Parse_Blend_Map (TEXTURE_TYPE,NO_PATTERN);
        *NumberPtr = TTextureMapID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tnormal_map:
        Temp_Data  = (void *) Parse_Blend_Map (NORMAL_TYPE,NO_PATTERN);
        *NumberPtr = TNormalMapID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Trainbow:
        Temp_Data  = (void *) Parse_Rainbow();
        *NumberPtr = TRainbowID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tfog:
        Tmp_Data  = (void *) Parse_Fog();
        *NumberPtr = TFogID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tmedia:
        Local_Media = NULL;
        Parse_Media(&Local_Media);
        Temp_Data  = (void *)Local_Media;
        *NumberPtr = TMediaID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tdensity:
        Local_Density = NULL;
        Parse_Begin ();
        Parse_Media_Density_Pattern (&Local_Density);
        Parse_End ();
        *NumberPtr = TDensityID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = (void *)Local_Density;
        done  = TRUE;
        break;

      case Tinterior:
        Local_Interior = NULL;
        Parse_Interior(&Local_Interior);
        Temp_Data  = (void *)Local_Interior;
        *NumberPtr = TInteriorID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tmaterial:
        Local_Material = Create_Material();
        Parse_Material(Local_Material);
        Temp_Data  = (void *)Local_Material;
        *NumberPtr = TMaterialID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tskysphere:
        Temp_Data  = (void *) Parse_Skysphere();
        *NumberPtr = TSkysphereID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tfunction:
      // Do NOT allow to redefine functions! [trf]
      //   #declare foo = function(x) { x }
      //   #declare foo = function(x) { foo(x) } // Error!
      // Reason: Code like this would be unreadable but possible. Is it
      // a recursive function or not? - It is not recursive because the
      // foo in the second line refers to the first function, which is
      // not logical. Further, recursion is not supported in POV-Ray 3.5
      // anyway. However, allowing such code now would cause problems
      // implementing recursive functions after POV-Ray 3.5!
        if (entry != NULL)
          Temp_Data  = (void *)Parse_DeclareFunction(NumberPtr, entry->Token_Name, is_local);
        else
          Temp_Data  = (void *)Parse_DeclareFunction(NumberPtr, NULL, is_local);
        Test_Redefine(Previous, NumberPtr, *DataPtr, false);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Ttransform:
        Local_Trans = Parse_Transform ();
        *NumberPtr  = TTransformID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr    = (void *) Local_Trans;
        done = TRUE;
        break;

      case TStringLiteral: case Tchr: case Tsubstr: case Tstr: case Tvstr:
      case Tconcat: case Tstrupr: case Tstrlwr:
        Token_Name.unget_token = TRUE;
        Temp_Data  = Parse_String();
        *NumberPtr = TStringID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;

      case Tarray:
        Temp_Data  = (void *) Parse_Array_Declare();
        *NumberPtr = TArrayID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr   = Temp_Data;
        done = TRUE;
        break;
#endif
      default:
        Token.unget_token = TRUE;
/*        Local_Object = Parse_Object ();
        Found=(Local_Object!=NULL);
        *NumberPtr   = TObjectID;
        Test_Redefine(Previous,NumberPtr,*DataPtr, allow_redefine);
        *DataPtr     = (void *) Local_Object;*/
        done = TRUE;
        break;
    }
  }

  Ok_To_Declare=Old_Ok;
  return(Found);
}

/*************************************************************************
*  inc_cond_stack_index
**************************************************************************/
static void inc_cond_stack_index(void)
{
  if (++CS_Index >= COND_STACK_SIZE)
  {
    /* FIXME: ERROR */
  }
}

/*************************************************************************
*  parse_ifdef_param
***************************************************************************/
static gboolean parse_ifdef_param(void)
{
  int               index;
  SymbolTableEntry *entry;
  gboolean          retval = FALSE;
  gchar            *string2;
/*  int        i,j,k;
  int        c;
  gdouble    val;
  POV_ARRAY *a;*/

  get_token();
  if (Token.token_id!=Tleft_paren)
  {
    g_print("a '(' was epected here.\n");
    /*Error!*/
  }
  Inside_Ifdef = TRUE;
  get_token();
  string2 = g_strdup(string->str);
  g_print("ifdef: search for string: '%s'\n", string2);
  Inside_Ifdef = FALSE;

  /* Search tables from newest to oldest */
  for (index = local_table_index ; index >= 0; index--)
  {
    entry = g_hash_table_lookup(CurrentFrame->all_constants[local_table_index],
                                string2);
    
    if (entry != NULL)
    {
      Token.token_id  =   entry->type;
      Token.is_array_elem = FALSE;
      Token.number_ptr = &(entry->type);
      Token.data_ptr   = &(entry->data);

      /*if ( Token.Token_Id == PARAMETER_ID_TOKEN )
      {
        Token.NumberPtr = ((POV_PARAM *)(Entry->Data))->NumberPtr;
        Token.DataPtr   = ((POV_PARAM *)(Entry->Data))->DataPtr;
      }*/
      /*if (Token.NumberPtr && *(Token.NumberPtr)==ARRAY_ID_TOKEN)
      {
        Skip_Spaces();
        c = Echo_getc();
        Echo_ungetc(c);
        if (c!='[')
        {
          retval = true;
          break;
        }
        a = (POV_ARRAY *)(*(Token.DataPtr));
        j = 0;
        for (i=0; i <= a->Dims; i++)
        {
          GET(LEFT_SQUARE_TOKEN)
          val=Parse_Float();
          k=(int)(1.0e-08+val);
          if (k<0.0)
          {
            Error("Negative subscript");
          }
          if (k>=a->Sizes[i])
          {
            Error("Array subscript out of range");
          }
          j += k * a->Mags[i];
          GET(RIGHT_SQUARE_TOKEN)
        }
        Token.DataPtr   = &(a->DataPtrs[j]);
        Token.NumberPtr = &(a->Type);
        Token.Token_Id = a->Type;
        Token.is_array_elem = true;
        retval = (*Token.DataPtr != NULL);
        break;
      } else*/
      {
        retval = TRUE;
        break;
      }
    }
  }
  get_token();
  if (Token.token_id!=Tright_paren)
  {
    g_print("a '(' was epected here.\n");
    /*Error!*/
  }
  g_free(string2);
  return retval;
}

/*************************************************************************
*  parse_directive
**************************************************************************/
static void parse_directive(gboolean after_hash)
{
  gdouble         Value, Value2;
  int             Flag;
  gchar          *ts;
  PovMacroStruct *PMac=NULL;
  COND_TYPE       Curr_Type = Cond_Stack[CS_Index].Cond_Type;
  long            Hash_Loc/* = Data_File->In_File->tellg()*/;
  gboolean        done, done2;

  if (Curr_Type == INVOKING_MACRO_COND)
  {
    if (Cond_Stack[CS_Index].PMac->Macro_End==Hash_Loc)
    {
/*      Return_From_Macro();*/
      if (--CS_Index < 0)
      {
        /*Error("Mis-matched '#end'.");*/
      }
      Token.token_id = TEOF;
      Token.is_array_elem = FALSE;

      return;
    }
  }

  if (!Ok_To_Declare)
  {
    if (after_hash)
    {
      Token.token_id=Thash;
      Token.is_array_elem = FALSE;
    }
    Token.unget_token = FALSE;
    return;
  }

  done = FALSE;
  while (!done)
  {
    get_token();
    switch (Token.token_id)
    {
      case Tifdef:
        inc_cond_stack_index();

        if (Skipping)
        {
          Cond_Stack[CS_Index].Cond_Type = SKIP_TIL_END_COND;
          skip_tokens(SKIP_TIL_END_COND);
        } else
        {
          if (parse_ifdef_param())
          {
            Cond_Stack[CS_Index].Cond_Type=IF_TRUE_COND;
          } else
          {
            Cond_Stack[CS_Index].Cond_Type=IF_FALSE_COND;
            skip_tokens(IF_FALSE_COND);
          }
         }
        done = TRUE;
        break;

      case Tifndef:
        inc_cond_stack_index();

        if (Skipping)
        {
          Cond_Stack[CS_Index].Cond_Type = SKIP_TIL_END_COND;
          skip_tokens(SKIP_TIL_END_COND);
        } else
        {
          if (parse_ifdef_param())
          {
            Cond_Stack[CS_Index].Cond_Type=IF_FALSE_COND;
            skip_tokens(IF_FALSE_COND);
          } else
          {
            Cond_Stack[CS_Index].Cond_Type=IF_TRUE_COND;
          }
        }
        done = TRUE;
        break;

      case Tif:
        inc_cond_stack_index();

        if (Skipping)
        {
          Cond_Stack[CS_Index].Cond_Type = SKIP_TIL_END_COND;
          skip_tokens(SKIP_TIL_END_COND);
        } else
        {
/*          Value=Parse_Cond_Param();*/

          if (fabs(Value)>EPSILON)
          {
            Cond_Stack[CS_Index].Cond_Type=IF_TRUE_COND;
          } else
          {
            Cond_Stack[CS_Index].Cond_Type=IF_FALSE_COND;
            skip_tokens(IF_FALSE_COND);
          }
        }
        done = TRUE;
        break;

      case Twhile:
        inc_cond_stack_index();

        if (Skipping)
        {
          Cond_Stack[CS_Index].Cond_Type = SKIP_TIL_END_COND;
          skip_tokens(SKIP_TIL_END_COND);
        } else
        {
/*          Cond_Stack[CS_Index].While_File = Data_File->In_File;
          Cond_Stack[CS_Index].Pos        = Data_File->In_File->tellg();
          Cond_Stack[CS_Index].Line_No    = Data_File->Line_Number;
          Value=Parse_Cond_Param();*/
          if (fabs(Value)>EPSILON)
          {
            Cond_Stack[CS_Index].Cond_Type = WHILE_COND;
          } else
          {
            Cond_Stack[CS_Index].Cond_Type = SKIP_TIL_END_COND;
            skip_tokens(SKIP_TIL_END_COND);
          }
        }
        done = TRUE;
        break;

      case Telse:
         switch (Curr_Type)
        {
          case IF_TRUE_COND:
            Cond_Stack[CS_Index].Cond_Type = SKIP_TIL_END_COND;
            skip_tokens(SKIP_TIL_END_COND);
            break;

          case IF_FALSE_COND:
            Cond_Stack[CS_Index].Cond_Type = ELSE_COND;
            Token.token_id=Thash; /*insures Skip_Token takes notice*/
            Token.is_array_elem = FALSE;
            Token.unget_token = TRUE;
            break;

          case CASE_TRUE_COND:
          case SKIP_TIL_END_COND:
             break;

          case CASE_FALSE_COND:
            Cond_Stack[CS_Index].Cond_Type = CASE_TRUE_COND;
            if (Skipping)
            {
              Token.token_id=Thash; /*insures Skip_Token takes notice*/
              Token.is_array_elem = FALSE;
              Token.unget_token = TRUE;
            }
            break;

          default:
/*            Error("Mis-matched '#else'.");*/
            g_print("Mis-matched '#else'.\n");
        }
        done = TRUE;
        break;

      case Tswitch:
        inc_cond_stack_index();

        if (Skipping)
        {
          Cond_Stack[CS_Index].Cond_Type = SKIP_TIL_END_COND;
          skip_tokens(SKIP_TIL_END_COND);
        } else
        {
/*          Cond_Stack[CS_Index].Switch_Value=Parse_Cond_Param();*/
          Cond_Stack[CS_Index].Cond_Type=SWITCH_COND;
          done2 = FALSE;
          while (!done2)
          {
            get_token();
            switch (Token.token_id)
            {
              case Tcase: case Trange:
                if (Token.token_id==Tcase)
                {
/*                  Value=Parse_Cond_Param();*/
                  Flag = (fabs(Value-Cond_Stack[CS_Index].Switch_Value)<EPSILON);
                } else
                {
/*                  Parse_Cond_Param2(&Value,&Value2);*/
                  Flag = ((Cond_Stack[CS_Index].Switch_Value >= Value) &&
                          (Cond_Stack[CS_Index].Switch_Value <= Value2));
                }
                if (Flag)
                {
                  Cond_Stack[CS_Index].Cond_Type=CASE_TRUE_COND;
                } else
                {
                  Cond_Stack[CS_Index].Cond_Type=CASE_FALSE_COND;
                  skip_tokens(CASE_FALSE_COND);
                }
                done2 = TRUE;
                break;

              default:
                /*Error("#switch not followed by #case or #range.");*/
                break;
            }
          }
        }
        done = TRUE;
        break;

      case Tbreak:
        if (Curr_Type==CASE_TRUE_COND)
        {
          Cond_Stack[CS_Index].Cond_Type=SKIP_TIL_END_COND;
          skip_tokens(SKIP_TIL_END_COND);
        }
        done = TRUE;
        break;

      case Tcase: case Trange:
        switch(Curr_Type)
        {
          case CASE_TRUE_COND:
          case CASE_FALSE_COND:
            if (Token.token_id==Tcase)
            {
/*              Value=Parse_Cond_Param();*/
              Flag = (fabs(Value-Cond_Stack[CS_Index].Switch_Value)<EPSILON);
            } else
            {
/*              Parse_Cond_Param2(&Value,&Value2);*/
              Flag = ((Cond_Stack[CS_Index].Switch_Value >= Value) &&
                      (Cond_Stack[CS_Index].Switch_Value <= Value2));
            }
 
            if(Flag && (Curr_Type==CASE_FALSE_COND))
            {
              Cond_Stack[CS_Index].Cond_Type=CASE_TRUE_COND;
              if (Skipping)
              {
                Token.token_id=Thash; /*insures Skip_Token takes notice*/
                Token.is_array_elem = FALSE;
                Token.unget_token = TRUE;
              }
             }
            break;

          case SWITCH_COND:
            Token.unget_token = TRUE;
          case SKIP_TIL_END_COND:
            break;

          default:
            /*Error("Mis-matched '#case' or '#range'.");*/
        }
        done = TRUE;
        break;

      case Tend:
        switch (Curr_Type)
        {
          case INVOKING_MACRO_COND:
/*            Return_From_Macro();*/
            if (--CS_Index < 0)
            {
              /*Error("Mis-matched '#end'.");*/
            }
            break;

          case IF_FALSE_COND:
            Token.token_id=Thash; /*insures Skip_Token takes notice*/
            Token.is_array_elem = FALSE;
            Token.unget_token = TRUE;
          case IF_TRUE_COND:
          case ELSE_COND:
          case CASE_TRUE_COND:
          case CASE_FALSE_COND:
          case DECLARING_MACRO_COND:
          case SKIP_TIL_END_COND:
            if (Curr_Type==DECLARING_MACRO_COND)
            {
              if ((PMac=Cond_Stack[CS_Index].PMac)!=NULL)
              {
                PMac->Macro_End=Hash_Loc;
              }
            }
            if (--CS_Index < 0)
            {
              /*Error("Mis-matched '#end'.");*/
            }
            if (Skipping)
            {
              Token.token_id=Thash; /*insures Skip_Token takes notice*/
              Token.is_array_elem = FALSE;
              Token.unget_token = TRUE;
            }
            break;

          case WHILE_COND:
/*            if (Cond_Stack[CS_Index].While_File != Data_File->In_File)
            {
              Error("#while loop did not end in file where it started.");
            }*/
            Got_EOF=FALSE;
/*            if (!Data_File->In_File->seekg(Cond_Stack[CS_Index].Pos))
            {
              Error("Unable to seek in input file for #while directive.");
            }
            Data_File->Line_Number = Cond_Stack[CS_Index].Line_No;
            Value=Parse_Cond_Param();*/
            if (fabs(Value)<EPSILON)
            {
              Cond_Stack[CS_Index].Cond_Type = SKIP_TIL_END_COND;
              skip_tokens(SKIP_TIL_END_COND);
            }
            break;

          default:
/*            Error("Mis-matched '#end'.");*/
        }
        done = TRUE;
        break;

      case Tdeclare: case Tlocal:
        if (Skipping)
        {
          Token.unget_token = TRUE;
          done = TRUE;
        } else
        {
          parse_declare(Token.token_id == Tlocal, after_hash);
          Curr_Type = Cond_Stack[CS_Index].Cond_Type;
          if (Token.unget_token)
          {
            switch (Token.token_id)
            {
              case Thash:
                Token.unget_token=FALSE;
                break;

              case TMacroID:
                break;

              default:
                done = TRUE;
            }
          } else
          {
            done = TRUE;
          }
        }
        break;

      case Tdefault:
        if ( Skipping )
        {
          Token.unget_token = TRUE;
        } else
        {
/*          Parse_Default();*/
        }
        done = TRUE;
        break;

      case Tinclude:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
/*          Open_Include();*/
        }
        done = TRUE;
        break;

      case Tfloat_funct:
        if (Skipping)
        {
          Token.unget_token = TRUE;
          done = TRUE;
        } else
        {
          switch (Token.function_id)
          {
            case Tversion:
              Ok_To_Declare = FALSE;
/*              opts.Language_Version = (int)(Parse_Float() * 100 + 0.5);*/
              get_token(); if (Token.token_id != Tsemi_colon) Token.unget_token = TRUE;
/*              if (opts.Language_Version > OFFICIAL_VERSION_NUMBER)
              {
                Error("Your scene file requires POV-Ray version %g or later!\n", (DBL)(opts.Language_Version / 100.0));
              }*/
              Ok_To_Declare = TRUE;
              Curr_Type = Cond_Stack[CS_Index].Cond_Type;
              if (Token.unget_token && (Token.token_id==Thash))
              {
                Token.unget_token=FALSE;
              } else
              {
                done = TRUE;
              }
              break;

            default:
              Token.unget_token = TRUE;
/*              Expectation_Error ("object or directive.");*/
              break;
          }
        }
        break;

      case Twarning:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
/*          ts=Parse_C_String();*/
          if (strlen(ts) > 160) // intentional 160, not 128 [trf]
          {
            ts[124] = ts[125] = ts[126] = '.';
            ts[127] = 0;
          }
/*          Warning(0, "%s", ts);
          POV_FREE(ts);*/
        }
        done = TRUE;
        break;

      case Terror:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        }
        else
        {
/*          ts=Parse_C_String();*/
          if(strlen(ts) > 160) // intentional 160, not 128 [trf]
          {
            ts[124] = ts[125] = ts[126] = '.';
            ts[127] = 0;
          }
/*          Error("User message '%s'\nParsing stopped at user's request!", ts);
          POV_FREE(ts);*/
        }
        done = TRUE;
        break;

      /* Note: The new message driven output system does not support
       * generic user output to the render and statistics streams.
       * Both streams are now directed into the debug stream. */
      case Trender:
      case Tstatistics:
/*        Warning(0, "#render and #statistics streams are no longer available.\nRedirecting output to #debug stream.");*/
        // Intentional, redirect output to debug stream.
      case Tdebug:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
/*          ts=Parse_C_String();*/
          if(strlen(ts) > 160) // intentional 160, not 128 [trf]
          {
            ts[124] = ts[125] = ts[126] = '.';
            ts[127] = 0;
          }
/*          Debug_Info("%s", ts);
          POV_FREE(ts);*/
        }
        done = TRUE;
        break;

      case Tfopen:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
/*          Parse_Fopen();*/
        }
        done = TRUE;
        break;

      case Tfclose:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
/*          Parse_Fclose();*/
        }
        done = TRUE;
        break;

      case Tread:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
/*          Parse_Read();*/
        }
        done  = TRUE;
        break;

      case Twrite:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
/*          Parse_Write();*/
        }
        done = TRUE;
        break;

      case Tundef:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
          Ok_To_Declare = FALSE;
          done2 = FALSE;
          while (!done2)
          {
            get_token();
            switch(Token.token_id)
            {
              case TIdentifier:
/*                Warning(0,"Attempt to undef unknown identifier");*/
                done2 = TRUE;
                break;

              case TMacroID: case TParameterID:
              case TFileID: case TFunctID: case TVectFunctID:
              // These have to match Parse_Declare in parse.cpp! [trf]
              case TNormalID: case TFinishID: case TTextureID: case TObjectID:
              case TColor_mapID: case TTransformID: case TCameraID: case TPigmentID:
              case TSlopeMapID: case TNormalMapID: case TTextureMapID: case TColorID:
              case TPigmentMapID: case TMediaID: case TStringID: case TInteriorID:
              case TDensityID: case TArrayID: case TDensity_mapID: case TuvID:
              case TVector4DID: case TRainbowID: case TFogID: case TSkysphereID:
              case TMaterialID: case TSplineID:
/*                Remove_Symbol (Token.Table_Index, Token.Token_String, Token.is_array_elem, Token.DataPtr, Token.Token_Id);*/
                done2 = TRUE;
                break;
        
              case Tvector_funct: case Tfloat_funct:
                switch(Token.function_id)
                {
                  case TVectorID:case TFloatID:
/*                    Remove_Symbol (Token.Table_Index, Token.Token_String, Token.is_array_elem, Token.DataPtr, Token.Token_Id);*/
                    break;
              
                  default:
/*                    Parse_Error(TIdentifier);*/
                    break;
                }
                done2 = TRUE;
                break;

              default:
/*                Parse_Error(TIdentifier);*/
                break;
            }
          }
          Ok_To_Declare = TRUE;
        }
        done = TRUE;
        break;

      case TMacroID:
        if (Skipping)
        {
          Token.unget_token = TRUE;
        } else
        {
/*          Invoke_Macro();*/
        }
        done = TRUE;
        break;

      case Tmacro:
        if (!Skipping)
        {
          if (Inside_MacroDef)
          {
/*            Error("Cannot nest macro definitions");*/
          }
          Inside_MacroDef=TRUE;
          /*PMac=Parse_Macro();*/
          Inside_MacroDef=FALSE;
        }
        inc_cond_stack_index();
        Cond_Stack[CS_Index].Cond_Type = DECLARING_MACRO_COND;
        Cond_Stack[CS_Index].PMac      = PMac;
        skip_tokens(DECLARING_MACRO_COND);
        done = TRUE;
        break;

      default:
        Token.unget_token = TRUE;
        done = TRUE;
        break;
    }
  }

  if (Token.unget_token)
  {
    Token.unget_token = FALSE;
  } else
  {
    Token.token_id = TEOF;
    Token.is_array_elem = FALSE;
  }

#if 0
  char          *Directive;
  guint          DirectiveLength, DirectiveMaxLength;
  int            c;
  char          *FileName;
  ParsedVector   LocalExpress;
  CondDirective *TmpCond;
  int            LocalToken;
  char          *LocalString;

  if (ignore_space()) /* Are Spaces really allowed between Hash and Directive ?!? */
    return 1;
  ParsingDirective = 1;
  DirectiveMaxLength = 20;
  Directive = g_malloc(21);
  DirectiveLength = 0;
  c = next_char();
  while(c>='a' && c<='z')
  {
    Directive[DirectiveLength++]=(char)c;
    if (DirectiveLength==DirectiveMaxLength)
    {
      DirectiveMaxLength *= 2;
      Directive = g_realloc(Directive, DirectiveMaxLength);
    }
    c = next_char();
  }
  OldChar = c; IsOldChar = 1;
  Directive[DirectiveLength] = 0;

  if (!strcmp(Directive, "version"))
  {
    PARSE_FLOAT(PovVersionLangage)
  } else if (!strcmp(Directive, "include"))
  {
    if (parse_string(&FileName))
      return 1;
    PovFile->Next = g_new(ListFile, 1);
    PovFile->Next->Prev = PovFile;
    PovFile = PovFile->Next;
    PovFile->Next = NULL;
    PovFile->handle = LibOpen(FileName);
    if (PovFile->handle == NULL)
    {
      g_message(_("%s, line %d\nCan't open file \"%s\"...\n"),
                PovFile->Prev->Name, PovFile->Prev->Line, FileName);
      return 1;
    }
    PovFile->Name = FileName;
    PovFile->Line = 1;
    local_table_index++;
    CurrentFrame->all_constants[local_table_index] = g_hash_table_new(g_str_hash, g_str_equal);
  } else if (!strcmp(Directive, "ifdef"))
  {
    TmpCond = g_new(CondDirective, 1);
    TmpCond->Next = ConditionalStack;
    ConditionalStack = TmpCond;
    if (DontParse)
    {
      ConditionalStack->Style = UNPARSED;
    } else
    {
      PARSE_LEFT_PAREN("#ifdef")
      LocalToken = 0;
/*      if (parse_word(&LocalToken))
        return 1;*/
      PARSE_RIGHT_PAREN("#ifdef")
      if (LocalToken)
      {
        ConditionalStack->Style = IF_TRUE_IF_ELSE;
      } else
      {
        ConditionalStack->Style = IF_FALSE_IF_ELSE;
        DontParse = TRUE;
      }
    }
  } else if (!strcmp(Directive, "ifndef"))
  {
    TmpCond = g_new(CondDirective, 1);
    TmpCond->Next = ConditionalStack;
    ConditionalStack = TmpCond;
    if (DontParse)
    {
      ConditionalStack->Style = UNPARSED;
    } else
    {
      PARSE_LEFT_PAREN("#ifndef")
      LocalToken = 0;
/*      if (parse_word(&LocalToken))
        return 1;*/
      PARSE_RIGHT_PAREN("#ifndef")
      if (!LocalToken)
      {
        ConditionalStack->Style = IF_TRUE_IF_ELSE;
      } else
      {
        ConditionalStack->Style = IF_FALSE_IF_ELSE;
        DontParse = TRUE;
      }
    }
  } else if (!strcmp(Directive, "if"))
  {
    TmpCond = g_new(CondDirective, 1);
    TmpCond->Next = ConditionalStack;
    ConditionalStack = TmpCond;
    if (DontParse)
    {
      ConditionalStack->Style = UNPARSED;
    } else
    {
      PARSE_LEFT_PAREN("#if")
      if (parse_express(&LocalExpress))
        return 1;
      PARSE_RIGHT_PAREN("#if")
      if (LocalExpress.nbTerms != 1)
      {
        g_message(_("%s, line %d\nConditions must evaluate to a float...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
      }
      if (fabs(LocalExpress.Val[0]) > SMALL_VALUE)
      {
        ConditionalStack->Style = IF_TRUE_IF_ELSE;
      } else
      {
        ConditionalStack->Style = IF_FALSE_IF_ELSE;
        DontParse = TRUE;
      }
    }
  } else if (!strcmp(Directive, "else"))
  {
    if (!ConditionalStack)
    {
      g_message(_("%s, line %d\nMisplaced #else...\n"), PovFile->Name, PovFile->Line);
      return 1;
    }
    switch (ConditionalStack->Style)
    {
      case IF_TRUE_IF_ELSE :
        DontParse = TRUE;
        ConditionalStack->Style = IF_TRUE_ELSE_END;
        break;

      case IF_TRUE_ELSE_END :
      case IF_FALSE_ELSE_END :
      case WHILE :
      case SWITCH_SWITCH_CASE :
        g_message(_("%s, line %d\nMisplaced #else...\n"), PovFile->Name, PovFile->Line);
        return 1;

      case IF_FALSE_IF_ELSE :
      case SWITCH_CASE_BREAK_FALSE :
        DontParse = FALSE;
        ConditionalStack->Style = IF_FALSE_ELSE_END;
        break;

      case SWITCH_CASE_BREAK_TRUE :
        DontParse = TRUE;
        ConditionalStack->Style = SWITCH_BREAK_END;
        break;

      case SWITCH_BREAK_END:
      case UNPARSED:
        /* Nothing */
        break;
    }
  } else if (!strcmp(Directive, "end"))
  {
    if (!ConditionalStack)
    {
      g_message(_("%s, line %d\nMisplaced #end...\n"), PovFile->Name, PovFile->Line);
      return 1;
    }
    switch (ConditionalStack->Style)
    {
      case IF_TRUE_IF_ELSE:
      case IF_TRUE_ELSE_END:
      case IF_FALSE_IF_ELSE:
      case IF_FALSE_ELSE_END:
      case SWITCH_BREAK_END:
      case SWITCH_CASE_BREAK_TRUE:
      case SWITCH_CASE_BREAK_FALSE:
      case SWITCH_SWITCH_CASE:
        TmpCond = ConditionalStack->Next;
        g_free(ConditionalStack);
        ConditionalStack = TmpCond;
        DontParse = FALSE;
        break;

      case WHILE:
        if (ConditionalStack->WhileFile != PovFile)
        {
          g_message(_("%s, line %d\n#while loop didn't end in file where it started...\n"),
                    PovFile->Name, PovFile->Line);
          return 1;
        }
        fseek(PovFile->handle, ConditionalStack->WhilePos, SEEK_SET);
        PovFile->Line = ConditionalStack->WhileLineNo;
        PARSE_LEFT_PAREN("#while")
        if (parse_express(&LocalExpress))
          return 1;
        PARSE_RIGHT_PAREN("#while")
        if (fabs(LocalExpress.Val[0]) > SMALL_VALUE)
        {
          /* Nothing ! ? */
        } else
        {
          DontParse = TRUE;
          ConditionalStack->Style = IF_TRUE_ELSE_END;
        }
        break;

      case UNPARSED:
        TmpCond = ConditionalStack->Next;
        g_free(ConditionalStack);
        ConditionalStack = TmpCond;
        break;
    }
  } else if (!strcmp(Directive, "declare"))
  {
    if (parse_declare(FALSE))
      return 1;
  } else if (!strcmp(Directive, "local"))
  {
    if (parse_declare(TRUE))
      return 1;
  } else if (!strcmp(Directive, "debug"))
  {
    if (!DontParse)
    {
      if (parse_string(&LocalString))
        return 1;
      g_print("%s\n", LocalString); /* FIXME Should be redirected on the debug stream */
      g_free(LocalString);
    }
  } else if (!strcmp(Directive, "error"))
  {
    if (parse_string(&LocalString))
      return 1;
    if (!DontParse)
    {
      g_message(_("%s, line %d\nError: %s"),
                PovFile->Name, PovFile->Line, LocalString);
      return 1;
    }
    g_free(LocalString);
  } else if (!strcmp(Directive, "render"))
  {
    if (parse_string(&LocalString))
      return 1;
    if (!DontParse)
    {
      printf(LocalString); /* FIXME */
    }
    g_free(LocalString);
  } else if (!strcmp(Directive, "statistics"))
  {
    if (parse_string(&LocalString))
      return 1;
    if (!DontParse)
    {
      printf(LocalString); /* FIXME */
    }
    g_free(LocalString);
  } else if (!strcmp(Directive, "warning"))
  {
    if (parse_string(&LocalString))
      return 1;
    if (!DontParse)
    {
      printf(LocalString); /* FIXME */
    }
    g_free(LocalString);
  } else
  {
    g_message(_("%s, line %d\nUnknown directive : %s\n"),
              PovFile->Name, PovFile->Line, Directive);
    return 1;
  }
  g_free(Directive);
  ParsingDirective = 0;
  return 0;
#endif
}

/*************************************************************************
*  skip_tokens
**************************************************************************/
static void skip_tokens(COND_TYPE cond)
{
  int Temp      = CS_Index;
  int Prev_Skip = Skipping;

  Skipping = TRUE;

  while ((CS_Index > Temp) || 
         ((CS_Index == Temp) && (Cond_Stack[CS_Index].Cond_Type == cond)))
  {
    get_token();
  }

  Skipping=Prev_Skip;

  if (Token.token_id==Thash)
  { /* ??? I don't understand the rational behind this */
    Token.token_id=TEOF;
    Token.is_array_elem = FALSE;
    Token.unget_token   = FALSE;
  }
  else
  {
    Token.unget_token = TRUE;
  }
}

/*****************************************************************************
*  next_token
******************************************************************************/
static void get_token(void)
{
  int c, c1, c2;

  if (Token.unget_token)
  {
    Token.unget_token = FALSE;
    return;
  }

  if (Token.end_of_file)
    return;

  Token.token_id = TEOF;
  Token.is_array_elem = FALSE;

  while (Token.token_id == TEOF)
  {
    ignore_space();
    c = next_char();

    if (c == EOF)
    {
#if 0
      if (Data_File->R_Flag)
      {
        Token.Token_Id = END_OF_FILE_TOKEN;
        Token.is_array_elem = false;
        Token.End_Of_File = true;
        return;
      }
#endif
#if 0
      if (Include_File_Index == 0)
      {
        if (CS_Index !=0)
          Error("End of file reached but #end expected.");
#endif
        Token.token_id = TEOF;
        Token.is_array_elem = FALSE;

        Token.end_of_file = TRUE;
        return;
//      }
#if 0
      POV_DELETE(Data_File->In_File, POV_ISTREAM); /* added to fix open file buildup JLN 12/91 */
      Got_EOF=false;

      Destroy_Table(Table_Index--);

      POV_FREE (Data_File->Filename);
      Data_File = &Include_Files[--Include_File_Index];
      continue;
#endif
    }
#if 0
    Begin_String_Fast();
    String[0] = c; /* This isn't necessary but helps debugging */
    String[1] = '\0';
#endif
     
    switch (c)
    {
      case '{':
        write_token(Tleft_curly);
        break;

      case '}':
        write_token(Tright_curly);
        break;

      case '@' :
        write_token(Tat);
        break;

      case '&':
        write_token(Tampersand);
        break;

      case '`':
        write_token(Tback_quote);
        break;

      case '\\':
        write_token(Tback_slash);
        break;

      case '|' :
        write_token(Tbar);
        break;

      case ':':
        write_token(Tcolon);
        break;

      case ',':
        write_token(Tcomma);
        break;

      case '-':
        write_token(Tdash);
        break;

      case '$':
        write_token(Tdollar);
        break;

      case '=':
        write_token(Tequals);
        break;

      case '!':
        c2 = next_char();
        if (c2 == '=')
        {
          write_token(Trel_ne);
        }
        else
        {
          write_token(Texclamation);
          IsOldChar = 1 ; OldChar = c2;
        }
        break;

      case '#':
        parse_directive(TRUE);
        /* Write_Token (HASH_TOKEN);*/
        break;

      case '^' :
        write_token(That);
        break;

      case '<':
        c2 = next_char();
        if (c2 == '=')
        {
          write_token(Trel_le);
        }
        else
        {
          write_token(Tleft_angle);
          IsOldChar = TRUE; OldChar = c2;
        }
        break;

      case '(':
        write_token(Tleft_paren);
        break;

      case '[':
        write_token(Tleft_square);
        break;

      case '%':
        write_token(Tpercent);
        break;

      case '+':
        write_token(Tplus);
        break;

      case '?':
        write_token(Tquestion);
        break;

      case '>':
        c2 = next_char();
        if (c2 == '=')
        {
          write_token (Trel_ge);
        }
        else
        {
          write_token(Tright_angle);
          IsOldChar = TRUE; OldChar = c2;
        }
        break;

      case ')':
        write_token(Tright_paren);
        break;

      case ']':
        write_token(Tright_square);
        break;

      case ';':
        write_token(Tsemi_colon);
        break;

      case '\'':
        write_token(Tsingle_quote);
        break;

      case '/' :
        c = next_char();
        if (c == '/')
        { /* comments ala C++ */
          while (c!='\r' && c!='\n')
          {
            c = next_char();
          }
          if (c=='\r')
          {
            c = next_char();
            if (c=='\n') c = next_char();
          }
          if (c=='\n')
          {
            c = next_char();
            if (c=='\r') c = next_char();
          }
          PovFile->Line++;
          OldChar = c; IsOldChar = 1;
          break;
        }
        if (c == '*')
        { /* comments ala C */
          c1 = c2 = next_char();
          do
          {
            if (c2=='\n')
            {
              PovFile->Line++;
              c2 = next_char();
              if (c2=='\r') c2 = next_char();
            } else if (c2=='\r')
            {
              PovFile->Line++;
              c2 = next_char();
              if (c2=='\n') c2 = next_char();
            } else
            {
              c1 = c2;
              c2 = next_char();
            }
          } while (c1!='*' || c2!='/');
          break;
        }
        OldChar = c; IsOldChar = 1;
        write_token(Tslash);
        break;

      case '*':
        write_token(Tstar);
        break;

      case '~':
        write_token(Ttilde);
        break;

      case '"':
        parse_string_literal();
        break;

      case '0'...'9':case '.':
        IsOldChar = TRUE; OldChar = c;
        parse_float_literal();
        break;

      case 'a'...'z':
      case 'A'...'Z':
      case '_':
        IsOldChar = TRUE; OldChar = c;
        parse_word();
        break;

      case '\t':
      case '\r':
      case '\032':   /* Control Z - EOF on many systems */
      case '\0':
        break;

      default:
        g_message(_("%s, line %d\nUnexpected character : '%c'\n"),
                  PovFile->Name, PovFile->Line, c);
        break;
    }
  }
   
#if 0
  while (*pToken == 0)
  {
    c = next_char();
    switch (c)
    {
      case EOF :
        if (PovFile->Prev)
        {
          fclose(PovFile->handle);
          g_free(PovFile->Name);
          PovFile = PovFile->Prev;
          g_free(PovFile->Next);
          g_hash_table_destroy(CurrentFrame->all_constants[local_table_index]);
          local_table_index--;
        } else *pToken = TEOF;
        break;

    }
    if (DontParse)
      *pToken = 0;
  }
  return 0;
#endif
}

/*****************************************************************************
*  parse_color
******************************************************************************/
static int parse_color(Vector LocalColor)
{
  int          i, Done;
  ParsedVector TmpColor;

  get_token();
  if (Token.token_id != Tcolor && Token.token_id != Tcolour)
    Token.unget_token = TRUE;
  Done = FALSE;
  TmpColor.nbTerms = 5;
  for (i=0 ; i<5 ; i++)
    LocalColor[i] = TmpColor.Val[i]=0.0;
  while (!Done)
  {
    get_token();
    switch (Token.function_id)
    {
      case Tred:
        PARSE_FLOAT(TmpColor.Val[0])
        LocalColor[0] = TmpColor.Val[0];
        break;

      case Tgreen:
        PARSE_FLOAT(TmpColor.Val[1])
        LocalColor[1] = TmpColor.Val[1];
        break;

      case Tblue:
        PARSE_FLOAT(TmpColor.Val[2])
        LocalColor[2] = TmpColor.Val[2];
        break;

      case Talpha: case Tfilter:
        PARSE_FLOAT(TmpColor.Val[3])
        LocalColor[3] = TmpColor.Val[3];
        break;

      case Ttransmit:
        PARSE_FLOAT(TmpColor.Val[4])
        LocalColor[4] = TmpColor.Val[4];
        break;

      case Trgb:
        if (parse_express(&TmpColor))
          return 1;
        if (TmpColor.nbTerms == 1)
        {
          TmpColor.Val[2] = TmpColor.Val[1] = TmpColor.Val[0];
        } else if (TmpColor.nbTerms < 3)
        {
          for (i=TmpColor.nbTerms ; i<3 ; i++)
            TmpColor.Val[i] = 0.0;
        }
        for (i=0 ; i<3 ; i++)
          LocalColor[i] = TmpColor.Val[i];
        LocalColor[3] = LocalColor[4] = 0.0;
        break;

      case Trgbf:
        if (parse_express(&TmpColor))
          return 1;
        if (TmpColor.nbTerms == 1)
        {
          TmpColor.Val[3] = TmpColor.Val[2] =
                            TmpColor.Val[1] =
                            TmpColor.Val[0];
        } else if (TmpColor.nbTerms < 4)
        {
          for (i=TmpColor.nbTerms ; i<4 ; i++)
            TmpColor.Val[i] = 0.0;
        }
        for (i=0 ; i<4 ; i++)
          LocalColor[i] = TmpColor.Val[i];
        LocalColor[4] = 0.0;
        break;

      case Trgbft:
        if (parse_express(&TmpColor))
          return 1;
        if (TmpColor.nbTerms == 1)
        {
          TmpColor.Val[4] = TmpColor.Val[3] = TmpColor.Val[2] =
                            TmpColor.Val[1] = TmpColor.Val[0];
        } else if (TmpColor.nbTerms < 5)
        {
          for (i=TmpColor.nbTerms ; i<5 ; i++)
            TmpColor.Val[i] = 0.0;
        }
        for (i=0 ; i<5 ; i++)
          LocalColor[i] = TmpColor.Val[i];
        break;

      case Trgbt:
        if (parse_express(&TmpColor))
          return 1;
        if (TmpColor.nbTerms == 1)
        {
          TmpColor.Val[3] = TmpColor.Val[2] =
                            TmpColor.Val[1] =
                            TmpColor.Val[0];
        } else if (TmpColor.nbTerms < 4)
        {
          for (i=TmpColor.nbTerms ; i<4 ; i++)
            TmpColor.Val[i] = 0.0;
        }
        V3Dcopy(LocalColor, TmpColor.Val);
        LocalColor[3] = 0.0;
        LocalColor[4] = TmpColor.Val[3];
        break;

      case Tplus: case Tdash:
      case Tabs: case Tacos: case Tval: case Tasc: case Tasin: case Tatan2:
      case Tceil: case Tclock: case Tcos: case Tdegrees: case Tdiv:
      case Texp: case Tfile_exists: case TFloatID: case TFloatLiteral:
      case Tfloor: case Tint: case Tlog: case Tmin: case Tmax: case Tmod:
      case Tpi: case Tpow: case Tradians: case Tsin: case Tsqrt:
      case Tstrcmp: case Tstrlen: case Ttan: case Tvdot: case Tvlength:
      case Tversion: case Ttrue: case Tyes: case Ton: case Tfalse:
      case Tno: case Toff: case Tseed: case Trand: case Tvaxis_rotate:
      case Tvcross: case TVectorID: case Tvnormalize: case Tvrotate:
      case Tx: case Ty: case Tz: case TColorID: case Tt: case Tu: case Tv:
      case Texclamation: case Tleft_paren: case Tleft_angle:
        Token.unget_token = TRUE;
        if (parse_express(&TmpColor))
          return 1;
        if (TmpColor.nbTerms == 1)
        {
          TmpColor.Val[4] = TmpColor.Val[3] = TmpColor.Val[2] =
                            TmpColor.Val[1] = TmpColor.Val[0];
        } else if (TmpColor.nbTerms < 5)
        {
          for (i=TmpColor.nbTerms ; i<5 ; i++)
            TmpColor.Val[i] = 0.0;
        }
        for (i=0 ; i<5 ; i++)
          LocalColor[i] = TmpColor.Val[i];
        break;

      default:
        Token.unget_token = TRUE;
        Done = TRUE;
    }
  }
  return 0;
}

/*****************************************************************************
*  parse_matrix
******************************************************************************/
static int parse_matrix(TransformStruct *pTrans)
{
  int i, j;

  pTrans->Direct[0][3] = 0.0;
  pTrans->Direct[1][3] = 0.0;
  pTrans->Direct[2][3] = 0.0;
  pTrans->Direct[3][3] = 1.0;
  get_token();
  if (Token.token_id != Tleft_angle)
  {
    g_message(_("%s, line %d\nA '<' was expected here...\n"), PovFile->Name, PovFile->Line);
    return 1;
  }
  
  for (j=0 ; j<4 ; j++)
    for (i=0 ; i<3 ; i++)
    {
      PARSE_FLOAT(pTrans->Direct[j][i])
      if ((i!=2) || (j!=3))
      {
        PARSE_COMMA
      }
    }
  get_token();
  if (Token.token_id != Tright_angle)
  {
    g_message(_("%s, line %d\nA '>' was expected here...\n"), PovFile->Name, PovFile->Line);
    return 1;
  }
  MInverse(pTrans->Inverse, pTrans->Direct);
  return 0;
}

/*****************************************************************************
*  parse_pigment
******************************************************************************/
static int parse_pigment(PigmentStruct **pLocalPigment)
{
  ParsedVector    LocalVector;
  TransformStruct Trans;

  PARSE_OPEN_CURLY

  get_token();
  if (Token.token_id == TPigmentID)
  {
    *pLocalPigment = CopyPigment(last_constant_entry->data);
  } else
  {
    if (*pLocalPigment == NULL)
    {
      (*pLocalPigment) = g_new(PigmentStruct, 1);
      (*pLocalPigment)->Type = PAT_SOLID_COLOR;
      (*pLocalPigment)->Trans = NULL;
      V3Deq((*pLocalPigment)->Color, 0.0, 0.0, 0.0);
      V3Deq((*pLocalPigment)->Color2, 0.0, 0.0, 0.0);
      V3Deq((*pLocalPigment)->Color3, 0.0, 0.0, 0.0);
      (*pLocalPigment)->color_map = NULL;
    }
    Token.unget_token = TRUE;
  }
  get_token();
  while (Token.token_id != Tright_curly)
  {
    switch(Token.token_id)
    {
      case TColorID: case Tred: case Tgreen: case Tblue: case Tfilter:
      case Ttransmit: case Tcolor: case Talpha: case Trgb: case Trgbt:
      case Trgbf: case Trgbft:
        Token.unget_token = TRUE;
        if (parse_color((*pLocalPigment)->Color))
          return 1;
        (*pLocalPigment)->Type = PAT_SOLID_COLOR;
        break;

      case Tbrick:
        if (parse_color((*pLocalPigment)->Color))
          return 1;
        PARSE_COMMA
        if (parse_color((*pLocalPigment)->Color2))
          return 1;
        (*pLocalPigment)->Type = PAT_BRICK;
        break;

      case Tchecker:
        if (parse_color((*pLocalPigment)->Color))
          return 1;
        PARSE_COMMA
        if (parse_color((*pLocalPigment)->Color2))
          return 1;
        (*pLocalPigment)->Type = PAT_CHECKER;
        break;

      case Thexagon:
        if (parse_color((*pLocalPigment)->Color))
          return 1;
        PARSE_COMMA
        if (parse_color((*pLocalPigment)->Color2))
          return 1;
        PARSE_COMMA
        if (parse_color((*pLocalPigment)->Color3))
          return 1;
        (*pLocalPigment)->Type = PAT_HEXAGON;
        break;

      case Tonion:
        (*pLocalPigment)->Type = PAT_ONION;
        break;

      case Tleopard:
        (*pLocalPigment)->Type = PAT_LEOPARD;
        break;

      case Ttranslate:
        PARSE_VECTOR(LocalVector)
        if ((*pLocalPigment)->Trans)
        {
          ComputeTranslateTrans(&Trans, LocalVector.Val);
          ComposeTrans((*pLocalPigment)->Trans, &Trans);
        } else
        {
          (*pLocalPigment)->Trans = g_new(TransformStruct, 1);
          ComputeTranslateTrans((*pLocalPigment)->Trans, LocalVector.Val);
        }
        break;

      case Trotate:
        PARSE_VECTOR(LocalVector)
        if ((*pLocalPigment)->Trans)
        {
          ComputeRotateTrans(&Trans, LocalVector.Val);
          ComposeTrans((*pLocalPigment)->Trans, &Trans);
        } else
        {
          (*pLocalPigment)->Trans = g_new(TransformStruct, 1);
          ComputeRotateTrans((*pLocalPigment)->Trans, LocalVector.Val);
        }
        break;

      case Tscale:
        PARSE_VECTOR(LocalVector)
        if ((*pLocalPigment)->Trans)
        {
          ComputeScaleTrans(&Trans, LocalVector.Val);
          ComposeTrans((*pLocalPigment)->Trans, &Trans);
        } else
        {
          (*pLocalPigment)->Trans = g_new(TransformStruct, 1);
          ComputeScaleTrans((*pLocalPigment)->Trans, LocalVector.Val);
        }
        break;

      case Tmatrix:
        if ((*pLocalPigment)->Trans)
        {
          if (parse_matrix(&Trans))
            return 1;
          ComposeTrans((*pLocalPigment)->Trans, &Trans);
        } else
        {
          (*pLocalPigment)->Trans = g_new(TransformStruct, 1);
          if (parse_matrix((*pLocalPigment)->Trans))
            return 1;
        }
        break;

      default:
        g_message(_("%s, line %d\nA pigment modifier was expected here...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  parse_finish
******************************************************************************/
static int parse_finish(FinishStruct **pLocalFinish)
{
  PARSE_OPEN_CURLY

  get_token();
  if (Token.token_id == TFinishID)
  {
    *pLocalFinish = CopyFinish(last_constant_entry->data);
  } else
  {
    if (*pLocalFinish == NULL)
    {
      (*pLocalFinish) = g_new(FinishStruct, 1);
      (*pLocalFinish)->diffuse = 0.6;
      (*pLocalFinish)->ambient[0] = 0.1;
      (*pLocalFinish)->ambient[1] = 0.1;
      (*pLocalFinish)->ambient[2] = 0.1;
      /* FIXME! should initialize every field! */
    }
    Token.unget_token = TRUE;
  }
  get_token();
  while (Token.token_id != Tright_curly)
  {
    switch(Token.token_id)
    {
      case Tambient:
        if (parse_color((*pLocalFinish)->ambient))
          return 1;
        break;

      case Treflection:
        if (parse_color((*pLocalFinish)->reflection))
          return 1;
        break;

      case Tbrilliance:
        PARSE_FLOAT((*pLocalFinish)->brilliance)
        break;

      case Tcrand:
        PARSE_FLOAT((*pLocalFinish)->crand)
        break;

      case Tphong:
        PARSE_FLOAT((*pLocalFinish)->phong)
        break;

      case Tphong_size:
        PARSE_FLOAT((*pLocalFinish)->phong_size)
        break;

      case Tdiffuse:
        PARSE_FLOAT((*pLocalFinish)->diffuse)
        break;

      case Tspecular:
        PARSE_FLOAT((*pLocalFinish)->specular)
        break;

      case Troughness:
        PARSE_FLOAT((*pLocalFinish)->roughness)
        break;

      default:
        g_message(_("%s, line %d\nA finish modifier was expected here...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  parse_texture
******************************************************************************/
static int parse_texture(TextureStruct **pLocalTexture)
{
  ParsedVector LocalVector;
  TransformStruct Trans;

  PARSE_OPEN_CURLY

  get_token();
  if (Token.token_id == TTextureID)
  {
    *pLocalTexture = CopyTexture(last_constant_entry->data);
  } else
  {
    if (*pLocalTexture == NULL)
    {
      (*pLocalTexture) = g_new(TextureStruct, 1);
      (*pLocalTexture)->Pigment = NULL;
      (*pLocalTexture)->finish = NULL;
      (*pLocalTexture)->Trans = NULL;
    }
    Token.unget_token = TRUE;
  }
  get_token();
  while (Token.token_id != Tright_curly)
  {
    switch(Token.token_id)
    {
      case Tpigment:
        if (parse_pigment(&((*pLocalTexture)->Pigment)) )
          return 1;
        break;

      case Tfinish:
        if (parse_finish(&((*pLocalTexture)->finish)) )
          return 1;
        break;

      case Ttranslate:
        PARSE_VECTOR(LocalVector)
        if ((*pLocalTexture)->Trans)
        {
          ComputeTranslateTrans(&Trans, LocalVector.Val);
          ComposeTrans((*pLocalTexture)->Trans, &Trans);
        } else
        {
          (*pLocalTexture)->Trans = g_new(TransformStruct, 1);
          ComputeTranslateTrans((*pLocalTexture)->Trans, LocalVector.Val);
        }
        break;

      case Trotate:
        PARSE_VECTOR(LocalVector)
        if ((*pLocalTexture)->Trans)
        {
          ComputeRotateTrans(&Trans, LocalVector.Val);
          ComposeTrans((*pLocalTexture)->Trans, &Trans);
        } else
        {
          (*pLocalTexture)->Trans = g_new(TransformStruct, 1);
          ComputeRotateTrans((*pLocalTexture)->Trans, LocalVector.Val);
        }
        break;

      case Tscale:
        PARSE_VECTOR(LocalVector)
        if ((*pLocalTexture)->Trans)
        {
          ComputeScaleTrans(&Trans, LocalVector.Val);
          ComposeTrans((*pLocalTexture)->Trans, &Trans);
        } else
        {
          (*pLocalTexture)->Trans = g_new(TransformStruct, 1);
          ComputeScaleTrans((*pLocalTexture)->Trans, LocalVector.Val);
        }
        break;

      case Tmatrix:
        if ((*pLocalTexture)->Trans)
        {
          if (parse_matrix(&Trans))
            return 1;
          ComposeTrans((*pLocalTexture)->Trans, &Trans);
        } else
        {
          (*pLocalTexture)->Trans = g_new(TransformStruct, 1);
          if (parse_matrix((*pLocalTexture)->Trans))
            return 1;
        }
        break;

      default:
        g_message(_("%s, line %d\nA texture modifier was expected here...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  parse_object_modifier
******************************************************************************/
static int parse_object_modifier(ObjectStruct *LocalObject)
{
  ParsedVector     LocalVector;
  TransformStruct  Trans;

  get_token();
  while (Token.token_id != Tright_curly)
  {
    switch (Token.token_id)
    {
      case Ttranslate:
        PARSE_VECTOR(LocalVector)
        giram_object_translate(LocalObject, LocalVector.Val);
        break;

      case Trotate:
        PARSE_VECTOR(LocalVector)
        giram_object_rotate(LocalObject, LocalVector.Val);
        break;

      case Tscale:
        PARSE_VECTOR(LocalVector)
        if (LocalVector.Val[0] == 0.0)
        {
          g_message(_("%s, line %d\nYou can't scale by 0.0\nchanged to 1.0"),
                    PovFile->Name, PovFile->Line);
          LocalVector.Val[0] = 1.0;
        }
        if (LocalVector.Val[1] == 0.0)
        {
          g_message(_("%s, line %d\nYou can't scale by 0.0\nchanged to 1.0"),
                    PovFile->Name, PovFile->Line);
          LocalVector.Val[1] = 1.0;
        }
        if (LocalVector.Val[2] == 0.0)
        {
          g_message(_("%s, line %d\nYou can't scale by 0.0\nchanged to 1.0"),
                    PovFile->Name, PovFile->Line);
          LocalVector.Val[2] = 1.0;
        }
        giram_object_scale(LocalObject, LocalVector.Val);
        break;

      case Ttransform:
        /* FIXME */
        break;

      case Tmatrix:
        if (LocalObject->Trans)
        {
          TransformationStruct *transform;
          int  i,j;

          if (parse_matrix(&Trans))
            return 1;
          ComposeTrans(LocalObject->Trans, &Trans);
          transform = g_new(TransformationStruct, 1);
          transform->type = MATRIX;
          transform->name = g_strdup("matrix");
          V3Deq(transform->vect, 0,0,0);

          for (i=0 ; i<4 ; i++)
            for (j=0 ; j<4 ; j++)
          {
            transform->transform.Direct[i][j] = Trans.Direct[i][j];
            transform->transform.Inverse[i][j] = Trans.Inverse[i][j];
          }
          transform->active = TRUE;
          transform->object = LocalObject;
          LocalObject->all_transforms = g_slist_append(LocalObject->all_transforms, transform);
        } else
        {
          TransformationStruct *transform;
          int  i,j;

          LocalObject->Trans = g_new(TransformStruct, 1);
          if (parse_matrix(LocalObject->Trans))
            return 1;
          transform = g_new(TransformationStruct, 1);
          transform->type = MATRIX;
          transform->name = g_strdup("matrix");
          V3Deq(transform->vect, 0,0,0);

          for (i=0 ; i<4 ; i++)
            for (j=0 ; j<4 ; j++)
          {
            transform->transform.Direct[i][j] = LocalObject->Trans->Direct[i][j];
            transform->transform.Inverse[i][j] = LocalObject->Trans->Inverse[i][j];
          }
          transform->active = TRUE;
          transform->object = LocalObject;
          LocalObject->all_transforms = g_slist_append(LocalObject->all_transforms, transform);
        }
        break;

      case Ttexture:
        if (parse_texture(&(LocalObject->Texture)))
          return 1;
        break;

      case Tpigment:
        if (LocalObject->Texture == NULL)
        {
          LocalObject->Texture = g_new(TextureStruct, 1);
          LocalObject->Texture->Pigment = NULL;
          LocalObject->Texture->finish = NULL;
          LocalObject->Texture->Trans = NULL;
        }
        if (parse_pigment(&(LocalObject->Texture->Pigment)))
          return 1;
        break;

      case Tfinish:
        if (LocalObject->Texture == NULL)
        {
          LocalObject->Texture = g_new(TextureStruct, 1);
          LocalObject->Texture->Pigment = NULL;
          LocalObject->Texture->finish = NULL;
          LocalObject->Texture->Trans = NULL;
        }
        if (parse_finish(&(LocalObject->Texture->finish)))
          return 1;
        break;

      case Tinverse: /* A test for Flat Object could be welcomed here */
        LocalObject->Inverse = ! (LocalObject->Inverse);
        break;

      case Tno_shadow:
        LocalObject->NoShadow = 1;
        break;

      case Thollow:
        LocalObject->Hollow = 1;
        break;

      case Topen:
        switch (LocalObject->Type)
        {
          case CYLINDER_OBJECT:
            ((CylinderStruct *)LocalObject)->Open = TRUE;
            break;
          case SOR_OBJECT:
            ((SorStruct *)LocalObject)->Open = TRUE;
            break;
          case CONE_OBJECT:
            ((ConeStruct *)LocalObject)->Open = TRUE;
            break;
          default:
            g_message(_("%s, line %d\nThe 'open' keyword can only be used with some objects\n"),
                      PovFile->Name, PovFile->Line);
            return 1;
        }
        break;

      case Tsmooth:
        if (LocalObject->Type == HEIGHT_FIELD_OBJECT)
          ((HeightFieldStruct *)LocalObject)->smooth = TRUE;
        else
        {
          g_message(_("%s, line %d\nThe 'smooth' keyword can only be used with height fields\n"),
                    PovFile->Name, PovFile->Line);
          return 1;
        }
        break;

      case Twater_level:
        if (LocalObject->Type == HEIGHT_FIELD_OBJECT)
        {
          PARSE_FLOAT( ((HeightFieldStruct*)LocalObject)->waterlevel  )
        } else
        {
          g_message(_("%s, line %d\nThe 'water_level' keyword can only be used with height fields\n"),
                    PovFile->Name, PovFile->Line);
          return 1;
        }
        break;

      default:
        g_message(_("%s, line %d\nAn object modifier was expected here...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  parse_bicubic_patch
*  Parsing of a bicubic patch object
******************************************************************************/
static int parse_bicubic_patch(ObjectStruct **pBicubicPatch)
{
  Vector       controls_points[16];
  int          i;
  gdouble      type, flatness, u_steps=4, v_steps=4;
  ParsedVector LocalVector;

  PARSE_OPEN_CURLY

  get_token();
  while ((Token.token_id == Ttype) ||
         (Token.token_id == Tu_steps) ||
         (Token.token_id == Tv_steps) ||
         (Token.token_id == Tflatness))
  {
    switch (Token.token_id)
    {
      case Ttype:
        PARSE_FLOAT(type)
        break;

      case Tu_steps:
        PARSE_FLOAT(u_steps)
        break;

      case Tv_steps:
        PARSE_FLOAT(v_steps)
        break;

      case Tflatness:
        PARSE_FLOAT(flatness)
        break;
      default:
        /* panic!!! */
    }
    get_token();
  }
  Token.unget_token = TRUE;

  for (i=0 ; i<16 ; i++)
  {
    PARSE_VECTOR(LocalVector)
    V3Dcopy(controls_points[i], LocalVector.Val);
    if (i!=15)
    {
      PARSE_COMMA
    }
  }

  *pBicubicPatch = giram_bicubic_patch_new(u_steps, v_steps, controls_points);

  if (parse_object_modifier(*pBicubicPatch))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_box
*  Parsing of box object
******************************************************************************/
static int parse_box(ObjectStruct **pBox)
{
  ParsedVector LocalVector1, LocalVector2;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalVector1)
  PARSE_COMMA
  PARSE_VECTOR(LocalVector2)

  *pBox = giram_box_new(LocalVector1.Val, LocalVector2.Val);

  if (parse_object_modifier(*pBox))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_cone
*  Parsing of cone object
******************************************************************************/
static int parse_cone(ObjectStruct **pCone)
{
  ParsedVector    LocalVector1, LocalVector2;
  double          Radius1, Radius2;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalVector1)
  PARSE_COMMA
  PARSE_FLOAT(Radius1)
  PARSE_COMMA
  PARSE_VECTOR(LocalVector2)
  PARSE_COMMA
  PARSE_FLOAT(Radius2)

  *pCone = giram_cone_new(LocalVector1.Val, Radius1, LocalVector2.Val, Radius2);

  if (parse_object_modifier(*pCone))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_cylinder
*  Parsing of cylinder object
******************************************************************************/
static int parse_cylinder(ObjectStruct **pCylinder)
{
  ParsedVector LocalVector1, LocalVector2;
  double       Radius;
  Vector       TmpVect;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalVector1)
  PARSE_COMMA
  PARSE_VECTOR(LocalVector2)
  PARSE_COMMA
  PARSE_FLOAT(Radius)

  if (Radius <= 0.0)
  {
    g_message(_("%s, line %d\nThe Radius of cylinder must be strictly positive.\n"),
              PovFile->Name, PovFile->Line);
    return 1;
  }
  TmpVect[0] = LocalVector2.Val[0] - LocalVector1.Val[0];
  TmpVect[1] = LocalVector2.Val[1] - LocalVector1.Val[1];
  TmpVect[2] = LocalVector2.Val[2] - LocalVector1.Val[2];
  if (fabs(V3DLength(TmpVect)) < 10e-6)
  {
    g_message(_("%s, line %d\nThe Length of cylinder mustn't be nul.\n"),
              PovFile->Name, PovFile->Line);
    return 1;
  }
  *pCylinder = giram_cylinder_new(LocalVector1.Val, LocalVector2.Val, Radius);
  if (parse_object_modifier(*pCylinder))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_disc
*  Parsing of disc object
******************************************************************************/
static int parse_disc(ObjectStruct **pDisc)
{
  ParsedVector LocalVector;
  Vector       Center, Normal;
  double       Radius, HoleRadius = 0.0;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalVector)
  V3Dcopy(Center, LocalVector.Val);
  PARSE_COMMA
  PARSE_VECTOR(LocalVector)
  V3Dcopy(Normal, LocalVector.Val);
  PARSE_COMMA
  PARSE_FLOAT(Radius)
  PARSE_COMMA
  get_token();
  Token.unget_token = TRUE;
  switch (Token.token_id)
  {
    case TFloatID: case TVectorID: case Tplus: case Tdash: case Tabs:
    case Tacos: case Tval: case Tasc: case Tasin: case Tatan2: case Tceil:
    case Tclock: case Tcos: case Tdegrees: case Tdiv: case Texp:
    case Tfile_exists: case TFloatLiteral: case Tfloor: case Tint: case Tlog:
    case Tmin: case Tmax: case Tmod: case Tpi: case Tpow: case Tradians:
    case Tsin: case Tsqrt: case Tstrcmp: case Tstrlen: case Ttan: case Tvdot:
    case Tvlength: case Tversion: case Ttrue: case Tyes: case Ton: case Tfalse:
    case Tno: case Toff: case Tseed: case Trand: case Tvaxis_rotate:
    case Tvcross: case Tvnormalize: case Tvrotate: case Tx: case Ty: case Tz:
    case Tt: case Tu: case Tv: case Texclamation: case Tleft_paren:
    case Tleft_angle:
      PARSE_FLOAT(HoleRadius)
      break;
    default: /* nothing */
  }

  *pDisc = giram_disc_new(Center, Normal, Radius, HoleRadius);

  if (parse_object_modifier(*pDisc))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_heightfield
*  Parsing of heightfield object
******************************************************************************/
static int parse_heightfield(ObjectStruct **pHeightField)
{
  HFImageType  type;
  gchar       *filename = NULL;

  PARSE_OPEN_CURLY

  get_token();
  switch (Token.token_id)
  {
    case Tgif:
      type = IMAGE_TYPE_GIF;
      break;
    case Ttga:
      type = IMAGE_TYPE_TGA;
      break;
    case Tpot:
      type = IMAGE_TYPE_POT;
      break;
    case Tpng:
      type = IMAGE_TYPE_PNG;
      break;
    case Tpgm:
      type = IMAGE_TYPE_PGM;
      break;
    case Tppm:
      type = IMAGE_TYPE_PPM;
      break;
    case Tsys:
      type = IMAGE_TYPE_SYS;
      break;
    default:
      g_message(_("%s, line %d\nAn image type was expected here...\n"), PovFile->Name, PovFile->Line);
      return 1;
  }
  
  if (parse_string(&filename)) return 1;

  *pHeightField = giram_heightfield_new(filename, type, 0.0, FALSE);

  if (parse_object_modifier(*pHeightField))
    return 1;
  return 0;
}
/*****************************************************************************
*  parse_mesh
*  Parsing of mesh object
******************************************************************************/
static int parse_mesh(ObjectStruct **pMesh)
{
  MeshStruct         *MMesh;
  ParsedVector        LocalVector;
  TriangleListStruct *TmpTri;
  Vector              V1, V2;

  PARSE_OPEN_CURLY

  *pMesh = giram_mesh_new();
  MMesh = (MeshStruct *)(*pMesh);
  get_token();
  while (Token.token_id != Tright_curly)
  {
    switch(Token.token_id)
    {
      case Tsmooth_triangle:

        PARSE_OPEN_CURLY

        TmpTri = g_new(TriangleListStruct, 1);
        TmpTri->Next = MMesh->FirstTriangle;
        MMesh->FirstTriangle = TmpTri;
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->P1, LocalVector.Val);
        PARSE_COMMA
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->N1, LocalVector.Val);
        PARSE_COMMA
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->P2, LocalVector.Val);
        PARSE_COMMA
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->N2, LocalVector.Val);
        PARSE_COMMA
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->P3, LocalVector.Val);
        PARSE_COMMA
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->N3, LocalVector.Val);
        get_token();
        if (Token.token_id != Tright_curly)
        {
          g_message(_("%s, line %d\nA '}' was expected here...\n"), PovFile->Name, PovFile->Line);
          return 1;
        }
        break;

      case Ttriangle:
        PARSE_OPEN_CURLY

        TmpTri = g_new(TriangleListStruct, 1);
        TmpTri->Next = MMesh->FirstTriangle;
        MMesh->FirstTriangle = TmpTri;
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->P1, LocalVector.Val);
        PARSE_COMMA
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->P2, LocalVector.Val);
        PARSE_COMMA
        PARSE_VECTOR(LocalVector)
        V3Dcopy(TmpTri->P3, LocalVector.Val);
        get_token();
        if (Token.token_id != Tright_curly)
        {
          g_message(_("%s, line %d\nA '}' was expected here...\n"), PovFile->Name, PovFile->Line);
          return 1;
        }
        V1[0] = TmpTri->P2[0] - TmpTri->P1[0];
        V1[1] = TmpTri->P2[1] - TmpTri->P1[1];
        V1[2] = TmpTri->P2[2] - TmpTri->P1[2];
        V2[0] = TmpTri->P3[0] - TmpTri->P1[0];
        V2[1] = TmpTri->P3[1] - TmpTri->P1[1];
        V2[2] = TmpTri->P3[2] - TmpTri->P1[2];
        VCross(TmpTri->N1, V1, V2);
        VCross(TmpTri->N2, V1, V2);
        VCross(TmpTri->N3, V1, V2);
        break;

      default:
        Token.unget_token = TRUE;
        if (parse_object_modifier(*pMesh))
          return 1;
/*        IsOldToken = 1;
        OldToken = Tright_curly; FIXME what w/ get_token() ? */
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  parse_plane
*  Parsing of plane object
******************************************************************************/
static int parse_plane(ObjectStruct **pPlane)
{
  ParsedVector LocalVector;
  double       Distance;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalVector)
  PARSE_COMMA
  PARSE_FLOAT(Distance)

  (*pPlane) = giram_plane_new(LocalVector.Val, Distance);
  if (parse_object_modifier(*pPlane))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_quadric
*  Parsing of quadric object
******************************************************************************/
static int parse_quadric(ObjectStruct **pQuadric)
{
  ParsedVector LocalVector1, LocalVector2, LocalVector3;
  double       J;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalVector1)
  PARSE_COMMA
  PARSE_VECTOR(LocalVector2)
  PARSE_COMMA
  PARSE_VECTOR(LocalVector3)
  PARSE_COMMA
  PARSE_FLOAT(J)

  (*pQuadric) = giram_quadric_new(LocalVector1.Val[0], LocalVector1.Val[1], LocalVector1.Val[2],
                                  LocalVector2.Val[0], LocalVector2.Val[1], LocalVector2.Val[2],
                                  LocalVector3.Val[0], LocalVector3.Val[1], LocalVector3.Val[2], J);
  if (parse_object_modifier(*pQuadric))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_smooth_triangle
*  Parsing of smooth triangles
******************************************************************************/
static int parse_smooth_triangle(ObjectStruct **pTriangle)
{
  TriangleStruct *TTriangle;
  ParsedVector    LocalP1, LocalN1, LocalP2, LocalN2, LocalP3, LocalN3;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalP1)
  PARSE_COMMA
  PARSE_VECTOR(LocalN1)
  PARSE_COMMA
  PARSE_VECTOR(LocalP2)
  PARSE_COMMA
  PARSE_VECTOR(LocalN2)
  PARSE_COMMA
  PARSE_VECTOR(LocalP3)
  PARSE_COMMA
  PARSE_VECTOR(LocalN3)

  *pTriangle = giram_triangle_new(LocalP1.Val, LocalP2.Val, LocalP3.Val);
  TTriangle = (TriangleStruct *)(*pTriangle);
  V3Dcopy(TTriangle->N1, LocalN1.Val);
  V3Dcopy(TTriangle->N2, LocalN2.Val);
  V3Dcopy(TTriangle->N3, LocalN3.Val);
  if (parse_object_modifier(*pTriangle))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_sor
*  Parsing of sor object
******************************************************************************/
static int parse_sor(ObjectStruct **pSor)
{
  SorStruct    *SSor;
  double        LocalFloat;
  guint         i;
  ParsedVector  LocalVector;

  PARSE_OPEN_CURLY

  *pSor = giram_sor_new(0, NULL);
  SSor = (SorStruct *)(*pSor);
  PARSE_FLOAT(LocalFloat)
  SSor->NumberOfPoints = (int)LocalFloat;
  if (SSor->NumberOfPoints < 4)
  {
    g_message(_("%s, line %d\nIllegal number of point in a SOR object...\n"), PovFile->Name, PovFile->Line);
    return 1;
  }
  SSor->Point = g_new(Vector, SSor->NumberOfPoints);
  for (i=0 ; i<SSor->NumberOfPoints ; i++)
  {
    PARSE_COMMA
    PARSE_VECTOR(LocalVector)
    SSor->Point[i][0] = LocalVector.Val[0];
    SSor->Point[i][1] = LocalVector.Val[1];
  }
  if (parse_object_modifier(*pSor))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_sphere
*  Parsing of sphere object
******************************************************************************/
static int parse_sphere(ObjectStruct **pSphere)
{
  double       Radius;
  ParsedVector LocalVector;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalVector)
  PARSE_COMMA
  PARSE_FLOAT(Radius)

  *pSphere = giram_sphere_new(LocalVector.Val, Radius);
  if (parse_object_modifier(*pSphere))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_super_ellipsoid
*  Parsing of superellipsoid object
******************************************************************************/
static int parse_super_ellipsoid(ObjectStruct **pSuperEllipsoid)
{
  double N, E;

  PARSE_OPEN_CURLY

  get_token();
  if (Token.token_id != Tleft_angle)
  {
    g_message(_("%s, line %d\nA '<' was expected here...\n"), PovFile->Name, PovFile->Line);
    return 1;
  }

  PARSE_FLOAT(E)
  PARSE_COMMA
  PARSE_FLOAT(N)

  get_token();
  if (Token.token_id != Tright_angle)
  {
    g_message(_("%s, line %d\nA '>' was expected here...\n"), PovFile->Name, PovFile->Line);
    return 1;
  }
  *pSuperEllipsoid = giram_superellipsoid_new(N, E);
  if (parse_object_modifier(*pSuperEllipsoid))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_torus
*  Parsing of torus object
******************************************************************************/
static int parse_torus(ObjectStruct **pTorus)
{
  double Major, Minor;

  PARSE_OPEN_CURLY

  PARSE_FLOAT(Major)
  PARSE_COMMA
  PARSE_FLOAT(Minor)

  *pTorus = giram_torus_new(Major, Minor);
  if (parse_object_modifier(*pTorus))
    return 1;
  return 0;
}

/*****************************************************************************
*  parse_triangle
*  Parsing of triangle object
******************************************************************************/
static int parse_triangle(ObjectStruct **pTriangle)
{
  ParsedVector    LocalP1, LocalP2, LocalP3;

  PARSE_OPEN_CURLY

  PARSE_VECTOR(LocalP1)
  PARSE_COMMA
  PARSE_VECTOR(LocalP2)
  PARSE_COMMA
  PARSE_VECTOR(LocalP3)

  *pTriangle = giram_triangle_new(LocalP1.Val, LocalP2.Val, LocalP3.Val);
  if (parse_object_modifier(*pTriangle)) return 1;
  return 0;
}

/*****************************************************************************
*  add_object_to_csg
******************************************************************************/
static void add_object_to_csg(ObjectStruct *CSG, ObjectStruct *LocalObject)
{
  CSGStruct *CCSG = (CSGStruct *)CSG;

  CCSG->all_objects = g_slist_append(CCSG->all_objects, LocalObject);
  LocalObject->parent = CSG;
}

/*****************************************************************************
*  parse_csg
*  Parsing of CSG object
******************************************************************************/
static int parse_csg(int Type, ObjectStruct **pCSG)
{
  ObjectStruct *TmpObject;

  PARSE_OPEN_CURLY

  *pCSG = giram_csg_new(Type);
  get_token();
  while (Token.token_id != Tright_curly)
  {
    switch(Token.token_id)
    {
      case Tbicubic_patch:
        if (parse_bicubic_patch(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tbox:
        if (parse_box(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tcone:
        if (parse_cone(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tcylinder:
        if (parse_cylinder(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tdifference:
        if (parse_csg(CSG_DIFFERENCE, &TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tdisc:
        if (parse_disc(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Theight_field:
        if (parse_heightfield(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tintersection:
        if (parse_csg(CSG_INTERSECTION, &TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tmerge:
        if (parse_csg(CSG_MERGE, &TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tmesh:
        if (parse_mesh(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tobject:
        Token.unget_token = TRUE;
        TmpObject = NULL;
        if (parse_object(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tplane:
        if (parse_plane(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tquadric:
        if (parse_quadric(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tsmooth_triangle:
        if (parse_smooth_triangle(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tsor:
        if (parse_sor(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tsphere:
        if (parse_sphere(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tsuperellipsoid:
        if (parse_super_ellipsoid(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Ttorus:
        if (parse_torus(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Ttriangle:
        if (parse_triangle(&TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      case Tunion:
        if (parse_csg(CSG_UNION, &TmpObject))
          return 1;
        add_object_to_csg(*pCSG, TmpObject);
        break;

      default:
        Token.unget_token = TRUE;
        if (parse_object_modifier(*pCSG))
          return 1;
/*        IsOldToken = 1;
        OldToken = Tright_curly;FIXME*/
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  parse_object
******************************************************************************/
static int parse_object(ObjectStruct **pLocalObject)
{
  get_token();
  switch (Token.token_id)
  {
    case TObjectID:
      *pLocalObject = CopyObject(last_constant_entry->data);
      break;

    case Tobject:
      PARSE_OPEN_CURLY

      if (parse_object(pLocalObject))
        return 1;
      if (parse_object_modifier(*pLocalObject))
        return 1;
      break;

    case Tbicubic_patch:
      if (parse_bicubic_patch(pLocalObject))
        return 1;
      break;

    case Tbox:
      if (parse_box(pLocalObject))
        return 1;
      break;

    case Tcone:
      if (parse_cone(pLocalObject))
        return 1;
      break;

    case Tcylinder:
      if (parse_cylinder(pLocalObject))
        return 1;
      break;

    case Tdifference:
      if (parse_csg(CSG_DIFFERENCE, pLocalObject))
        return 1;
      break;

    case Tdisc:
      if (parse_disc(pLocalObject))
        return 1;
      break;

    case Theight_field:
      if (parse_heightfield(pLocalObject))
        return 1;
      break;

    case Tintersection:
      if (parse_csg(CSG_INTERSECTION, pLocalObject))
        return 1;
      break;

    case Tmerge:
      if (parse_csg(CSG_MERGE, pLocalObject))
        return 1;
      break;

    case Tmesh:
      if (parse_mesh(pLocalObject))
        return 1;
      break;

    case Tplane:
      if (parse_plane(pLocalObject))
        return 1;
      break;

    case Tquadric:
      if (parse_quadric(pLocalObject))
        return 1;
      break;

    case Tsmooth_triangle:
      if (parse_smooth_triangle(pLocalObject))
        return 1;
      break;

    case Tsor:
      if (parse_sor(pLocalObject))
        return 1;
      break;

    case Tsphere:
      if (parse_sphere(pLocalObject))
        return 1;
      break;

    case Tsuperellipsoid:
      if (parse_super_ellipsoid(pLocalObject))
        return 1;
      break;

    case Ttorus:
      if (parse_torus(pLocalObject))
        return 1;
      break;

    case Ttriangle:
      if (parse_triangle(pLocalObject))
        return 1;
      break;

    case Tunion:
      if (parse_csg(CSG_UNION, pLocalObject))
        return 1;
      break;

    default:
      return 1;
  }
  return 0;
}

/*****************************************************************************
*  parse_light_source
******************************************************************************/
static int parse_light_source(LightSourceStruct **pLocalLightSource)
{
  ParsedVector LocalVector;

  PARSE_OPEN_CURLY

  /* Creation & initialization */
  get_token();
#if 0 /* there's no such thing in povray 3.5 ??? */
  if (t == TLight_sourceID)
  {
    *pLocalLightSource = CopyLightSource(last_constant_entry->data);
    return 0; /* XXX: Is it the 'Right Thing To Do' ? */
  } else
#endif
  {
    (*pLocalLightSource) = g_new(LightSourceStruct, 1);
    V3Deq((*pLocalLightSource)->Location, 0.0, 0.0, 0.0);
    V5Deq((*pLocalLightSource)->Color, 1.0, 1.0, 1.0, 0.0, 0.0);
    (*pLocalLightSource)->Type = POINT_LIGHTSOURCE;
    V3Deq((*pLocalLightSource)->PointAt, 0.0, 0.0, 1.0);
    (*pLocalLightSource)->Radius = 0.35;
    (*pLocalLightSource)->FallOff = 0.35;
    (*pLocalLightSource)->Tightness = 10.0;
    (*pLocalLightSource)->AreaLight = FALSE;
    V3Deq((*pLocalLightSource)->Area1, 0.0, 0.0, 1.0);
    V3Deq((*pLocalLightSource)->Area2, 0.0, 1.0, 0.0);
    (*pLocalLightSource)->AreaSize1 = 0.0;
    (*pLocalLightSource)->AreaSize2 = 0.0;
    (*pLocalLightSource)->Adaptive = 100.0;
    (*pLocalLightSource)->Jitter = FALSE;
    //(*pLocalLightSource)->LooksLike = NULL;
    (*pLocalLightSource)->FadeDistance = 0.0;
    (*pLocalLightSource)->FadePower = 0.0;
    (*pLocalLightSource)->AtmosphericAttenuation = FALSE;
    (*pLocalLightSource)->Atmosphere = FALSE;
    (*pLocalLightSource)->name = NULL;
    (*pLocalLightSource)->frame = CurrentFrame;
    Token.unget_token = TRUE;
  }
  /* Parsing the mandatory parameters */
  PARSE_VECTOR(LocalVector)
  V3Dcopy((*pLocalLightSource)->Location, LocalVector.Val);
  PARSE_COMMA
  if (parse_color((*pLocalLightSource)->Color))
    return 1;
  /* Parsing the optionnal parameters */
  get_token();
  while (Token.token_id != Tright_curly)
  {
    switch (Token.token_id)
    {
      case Tspotlight:
        (*pLocalLightSource)->Type = SPOTLIGHT_LIGHTSOURCE;
        break;

      case Tcylinder:
        (*pLocalLightSource)->Type = CYLINDER_LIGHTSOURCE;
        break;

      case Tpoint_at:
        PARSE_VECTOR(LocalVector)
        V3Dcopy((*pLocalLightSource)->PointAt, LocalVector.Val);
        break;

      case Tradius:
        PARSE_FLOAT((*pLocalLightSource)->Radius)
        break;

      case Tfalloff:
        PARSE_FLOAT((*pLocalLightSource)->FallOff)
        break;

      case Ttightness:
        PARSE_FLOAT((*pLocalLightSource)->Tightness)
        break;

      case Tarea_light:
        /* FIXME */
        break;

      case Tadaptive:
        PARSE_FLOAT((*pLocalLightSource)->Adaptive)
        break;

      case Tjitter:
        PARSE_FLOAT((*pLocalLightSource)->Jitter)
        break;

      case Tlooks_like:
        /* FIXME */
        break;

      case Tfade_distance:
        PARSE_FLOAT((*pLocalLightSource)->FadeDistance)
        break;

      case Tfade_power:
        PARSE_FLOAT((*pLocalLightSource)->FadePower)
        break;

      case Ttranslate:
        PARSE_VECTOR(LocalVector)
        /* FIXME */
        break;

      case Trotate:
        PARSE_VECTOR(LocalVector)
        /* FIXME */
        break;

      case Tscale:
        PARSE_VECTOR(LocalVector)
          return 1;
        /* FIXME */
        break;

      case Ttransform:
        /* FIXME */
        break;

      case Tmatrix:
        /* FIXME */
        break;

      default:
        g_message(_("%s, line %d\nA Light Source modifier was expected here...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  parse_camera
******************************************************************************/
static int parse_camera(CameraStruct **pCamera)
{
  double       LocalFloat;
  ParsedVector LocalVector;
  Vector       Vect;
  double       k1, k2;
  double       Norme;
  double       DirectionLength, UpLength, RightLength;
  double       Handedness;

  PARSE_OPEN_CURLY

  get_token();
  if (Token.token_id == TCameraID)
  {
    *pCamera = CopyCamera(last_constant_entry->data);
  } else
  {
    if (*pCamera == NULL)
    {
      (*pCamera) = g_new(CameraStruct, 1);
      (*pCamera)->Type = CAMERA_PERSPECTIVE;
      V3Deq((*pCamera)->Location,  0.0, 0.0, 0.0);
      V3Deq((*pCamera)->LookAt,    0.0, 0.0, 1.0);
      V3Deq((*pCamera)->Right,     1.33,0.0, 0.0);
      V3Deq((*pCamera)->Up,        0.0, 1.0, 0.0);
      V3Deq((*pCamera)->Direction, 0.0, 0.0, 1.0);
      V3Deq((*pCamera)->Sky,       0.0, 1.0, 0.0);
      (*pCamera)->Angle = 90.0;
      (*pCamera)->BlurSamples = 0;
      (*pCamera)->Aperture = 0.0;
      V3Deq((*pCamera)->FocalPoint, 0.0, 0.0, 1.0);
      (*pCamera)->Confidence = 0.9;
      (*pCamera)->Variance = 1.0/128.0;
      (*pCamera)->name = g_strdup(_("Camera 1"));
    }
    Token.unget_token = TRUE;
  }
  get_token();
  while (Token.token_id != Tright_curly)
  {
    switch(Token.token_id)
    {
      case Tperspective:
        (*pCamera)->Type = CAMERA_PERSPECTIVE;
        break;

      case Torthographic:
        (*pCamera)->Type = CAMERA_ORTHOGRAPHIC;
        Vect[0] = (*pCamera)->LookAt[0] - (*pCamera)->Location[0];
        Vect[1] = (*pCamera)->LookAt[1] - (*pCamera)->Location[1];
        Vect[2] = (*pCamera)->LookAt[2] - (*pCamera)->Location[2];
        k1 = V3DLength(Vect);
        k2 = V3DLength((*pCamera)->Direction);
        if ((k1 > 10e-6) && (k2 > 10e-6))
        {
          (*pCamera)->Right[0] *= k1/k2;
          (*pCamera)->Right[1] *= k1/k2;
          (*pCamera)->Right[2] *= k1/k2;
          (*pCamera)->Up[0] *= k1/k2;
          (*pCamera)->Up[1] *= k1/k2;
          (*pCamera)->Up[2] *= k1/k2;
        }
        break;

      case Tfisheye:
        (*pCamera)->Type = CAMERA_FISHEYE;
        break;

      case Tultra_wide_angle:
        (*pCamera)->Type = CAMERA_ULTRA_WIDE_ANGLE;
        break;

      case Tomnimax:
        (*pCamera)->Type = CAMERA_OMNIMAX;
        break;

      case Tpanoramic:
        (*pCamera)->Type = CAMERA_PANORAMIC;
        break;

      case Tcylinder:
        PARSE_FLOAT(LocalFloat)
        switch ((int)LocalFloat)
        {
          case 1: (*pCamera)->Type = CAMERA_CYLINDER_1; break;
          case 2: (*pCamera)->Type = CAMERA_CYLINDER_2; break;
          case 3: (*pCamera)->Type = CAMERA_CYLINDER_3; break;
          case 4: (*pCamera)->Type = CAMERA_CYLINDER_4; break;
          default:
            g_message(_("%s, line %d\nThere's only 4 cylinder type of camera...\n"),
                      PovFile->Name, PovFile->Line);
            return 1;
        }
        break;

      case Tlocation:
        PARSE_VECTOR(LocalVector)
        V3Dcopy((*pCamera)->Location, LocalVector.Val);
        break;

      case Tlook_at:
        DirectionLength = V3DLength((*pCamera)->Direction);
        UpLength = V3DLength((*pCamera)->Up);
        RightLength = V3DLength((*pCamera)->Right);
        VCross(Vect, (*pCamera)->Up, (*pCamera)->Direction);
        Handedness = V3DDot(Vect, (*pCamera)->Right);
        PARSE_VECTOR(LocalVector)
        (*pCamera)->LookAt[0] = (*pCamera)->Direction[0] = LocalVector.Val[0];
        (*pCamera)->LookAt[1] = (*pCamera)->Direction[1] = LocalVector.Val[1];
        (*pCamera)->LookAt[2] = (*pCamera)->Direction[2] = LocalVector.Val[2];
        (*pCamera)->Direction[0] -= (*pCamera)->Location[0];
        (*pCamera)->Direction[1] -= (*pCamera)->Location[1];
        (*pCamera)->Direction[2] -= (*pCamera)->Location[2];
        /* Check for zero length direction vector. */
        if (V3DLength((*pCamera)->Direction) < 10e-6)
        {
          g_message(_("%s, line %d\nCamera location and look_at point must be different.\n"),
                    PovFile->Name, PovFile->Line);
          return 1;
        }
        Norme = V3DLength((*pCamera)->Direction);
        (*pCamera)->Direction[0] /= Norme;
        (*pCamera)->Direction[1] /= Norme;
        (*pCamera)->Direction[2] /= Norme;
        VCross((*pCamera)->Right, (*pCamera)->Sky, (*pCamera)->Direction);
        Norme = V3DLength((*pCamera)->Right);
        (*pCamera)->Right[0] /= Norme;
        (*pCamera)->Right[1] /= Norme;
        (*pCamera)->Right[2] /= Norme;
        VCross((*pCamera)->Up, (*pCamera)->Direction, (*pCamera)->Right);
        (*pCamera)->Direction[0] *= DirectionLength;
        (*pCamera)->Direction[1] *= DirectionLength;
        (*pCamera)->Direction[2] *= DirectionLength;
        if (Handedness > 0.0)
        {
          (*pCamera)->Right[0] *= RightLength;
          (*pCamera)->Right[1] *= RightLength;
          (*pCamera)->Right[2] *= RightLength;
        }
        else
        {
          (*pCamera)->Right[0] *= -RightLength;
          (*pCamera)->Right[1] *= -RightLength;
          (*pCamera)->Right[2] *= -RightLength;
        }
        (*pCamera)->Up[0] *= UpLength;
        (*pCamera)->Up[1] *= UpLength;
        (*pCamera)->Up[2] *= UpLength;
        break;

      case Tright:
        PARSE_VECTOR(LocalVector)
        V3Dcopy((*pCamera)->Right, LocalVector.Val);
        break;

      case Tup:
        PARSE_VECTOR(LocalVector)
        V3Dcopy((*pCamera)->Up, LocalVector.Val);
        break;

      case Tdirection:
        PARSE_VECTOR(LocalVector)
        V3Dcopy((*pCamera)->Direction, LocalVector.Val);
        break;

      case Tsky:
        PARSE_VECTOR(LocalVector)
        V3Dcopy((*pCamera)->Sky, LocalVector.Val);
        break;

      case Tangle:
        PARSE_FLOAT((*pCamera)->Angle)
        if ((*pCamera)->Angle < 0.0)
        {
          g_message(_("%s, line %d\nNegative viewing angle."),
                    PovFile->Name, PovFile->Line);
          return 1;
        }
        if ((*pCamera)->Type == CAMERA_PERSPECTIVE)
        {
          if ((*pCamera)->Angle >= 180.0)
          {
            g_message(_("%s, line %d\nViewing angle has to be smaller than 180 degrees."),
                      PovFile->Name, PovFile->Line);
          }
          Norme = V3DLength((*pCamera)->Direction);
          (*pCamera)->Direction[0] /= Norme;
          (*pCamera)->Direction[1] /= Norme;
          (*pCamera)->Direction[2] /= Norme;
          Norme = V3DLength((*pCamera)->Right) / tan((*pCamera)->Angle/2.0 * M_PI/180.0)/2.0;
          (*pCamera)->Direction[0] *= Norme;
          (*pCamera)->Direction[1] *= Norme;
          (*pCamera)->Direction[2] *= Norme;
        }
        break;

      case Tblur_samples:
        PARSE_FLOAT(LocalFloat)
        (*pCamera)->BlurSamples = (int)LocalFloat;
        if ((*pCamera)->BlurSamples <=0 )
        {
          g_message(_("%s, line %d\nThe number of focal blur samples must be strictly positive...\n"),
                    PovFile->Name, PovFile->Line);
          return 1;
        }
        break;

      case Taperture:
        PARSE_FLOAT((*pCamera)->Aperture)
        break;

      case Tfocal_point:
        PARSE_VECTOR(LocalVector)
        V3Dcopy( (*pCamera)->FocalPoint, LocalVector.Val);
        break;

      case Tconfidence:
        PARSE_FLOAT(LocalFloat)
        if ((LocalFloat > 0.0) && (LocalFloat < 1.0))
          (*pCamera)->Confidence = LocalFloat;
        else
          g_message(_("%s, line %d\nIllegal value of confidence last one will be use...\n"),
                    PovFile->Name, PovFile->Line);
        break;

      case Tvariance:
        PARSE_FLOAT(LocalFloat)
        if ((LocalFloat >= 0.0) && (LocalFloat <= 1.0))
          (*pCamera)->Variance = LocalFloat;
        else
          g_message(_("%s, line %d\nIllegal value of variance last one will be use...\n"),
                    PovFile->Name, PovFile->Line);
        break;

      case Tscale:
        PARSE_VECTOR(LocalVector)
        /* FIXME */
        break;

      case Trotate:
        PARSE_VECTOR(LocalVector)
        /* FIXME */
        break;

      case Ttranslate:
        PARSE_VECTOR(LocalVector)
        (*pCamera)->Location[0] += LocalVector.Val[0];
        (*pCamera)->Location[1] += LocalVector.Val[1];
        (*pCamera)->Location[2] += LocalVector.Val[2];
        (*pCamera)->LookAt[0] += LocalVector.Val[0];
        (*pCamera)->LookAt[1] += LocalVector.Val[1];
        (*pCamera)->LookAt[2] += LocalVector.Val[2];
        break;

      case Tmatrix:
        /* FIXME */
        break;

      case Ttransform:
        /* FIXME */
        break;

      default:
        g_message(_("%s, line %d\nA Camera modifier was expected here...\n"),
                  PovFile->Name, PovFile->Line);
        return 1;
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  init_load_pov
******************************************************************************/
static void init_load_pov(char *Name)
{
  int i;

  IsOldChar         = FALSE;
  IsOldToken        = FALSE;
  PovVersionLangage = 3.5;
  LastString        = NULL;
  PovClock          = 0.0;
  NbPovRandomTable  = 0;
  DontParse         = 0;
  ParsingDirective  = 0;
  PovRandomTable    = NULL;
  CurrentFrame      = NewFrame(Name);
  ConditionalStack  = NULL;
  local_table_index = 0;

  /* token initialization */
  Token.token_line_no = 0;
  Token.token_string  = NULL;
  Token.unget_token   = FALSE;
  Token.end_of_file   = FALSE;
  Token.filename      = NULL;
  Token.data          = NULL;

  string              = g_string_new("");

  CS_Index            = 0;
  Skipping            = FALSE;
  Inside_Ifdef        = FALSE;
  Inside_MacroDef     = FALSE;
  Cond_Stack          = NULL;

  Ok_To_Declare       = TRUE;
  
  Cond_Stack = g_new(CS_ENTRY, COND_STACK_SIZE);
  Cond_Stack[0].Cond_Type = ROOT_COND;
  Cond_Stack[0].Switch_Value = 0.0;

  for (i = 0; i < LAST_TOKEN; i++)
  {
    if (i < Tfloat_funct)
      Conversion_Util_Table[i] = Tfloat_funct;
    else if (i < Tvector_funct)
      Conversion_Util_Table[i] = Tvector_funct;
    else if(i < Tcolour_key)
      Conversion_Util_Table[i] = Tcolour_key;
    else
      Conversion_Util_Table[i] = i;
  }
}

/*****************************************************************************
*  parse_background
******************************************************************************/
static int parse_background(void)
{
  PARSE_OPEN_CURLY

  if (parse_color(CurrentFrame->background_color))
    return 1;
  
  get_token();
  if ( Token.token_id != Tright_curly)
  {
    g_message(_("%s, line %d\nA '}' was expected here...\n"), PovFile->Name, PovFile->Line);
    return 1;
  }
  return 0;
}

/*****************************************************************************
*  parse_pov
******************************************************************************/
static int parse_pov(void)
{
  LightSourceStruct *TmpLightSource;
  ObjectStruct      *TmpObject;
  CameraStruct      *TmpCamera = NULL;

  get_token();
  while (Token.token_id != TEOF)
  {
    switch (Token.token_id)
    {
      case Tbackground:
        if (parse_background())
          return 1;
        break;

      case Tcamera:
        if (parse_camera(&TmpCamera))
          return 1;
        CurrentFrame->all_cameras = g_list_prepend(CurrentFrame->all_cameras,
                                                   TmpCamera);
        break;

      case Tlight_source:
        if (parse_light_source(&TmpLightSource))
          return 1;
        CurrentFrame->all_light_sources = g_slist_append(CurrentFrame->all_light_sources,
                                                         TmpLightSource);
        TmpLightSource->name = NULL;
        TmpLightSource->name = create_uniq_light_source_name(CurrentFrame, _("light source"));
        break;

      case Tbicubic_patch:
        if (parse_bicubic_patch(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("bicubic patch"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tbox:
        if (parse_box(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("box"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tcone:
        if (parse_cone(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("cone"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tcylinder:
        if (parse_cylinder(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("cylinder"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tdifference:
        if (parse_csg(CSG_DIFFERENCE, &TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("difference"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tdisc:
        if (parse_disc(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("disc"));
        TmpObject->frame = CurrentFrame;
        break;

      case Theight_field:
        if (parse_heightfield(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("heightfielf"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tintersection:
        if (parse_csg(CSG_INTERSECTION, &TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("intersection"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tmerge:
        if (parse_csg(CSG_MERGE, &TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("merge"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tmesh:
        if (parse_mesh(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("mesh"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tplane:
        if (parse_plane(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("plane"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tquadric:
        if (parse_quadric(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("quadric"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tsmooth_triangle:
        if (parse_smooth_triangle(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("triangle"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tsor:
        if (parse_sor(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("sor"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tsphere:
        if (parse_sphere(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("sphere"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tsuperellipsoid:
        if (parse_super_ellipsoid(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("superellipsoid"));
        TmpObject->frame = CurrentFrame;
        break;

      case Ttorus:
        if (parse_torus(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("torus"));
        TmpObject->frame = CurrentFrame;
        break;

      case Ttriangle:
        if (parse_triangle(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("triangle"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tunion:
        if (parse_csg(CSG_UNION, &TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("union"));
        TmpObject->frame = CurrentFrame;
        break;

      case Tobject:
        Token.unget_token = TRUE;
        if (parse_object(&TmpObject))
          return 1;
        CurrentFrame->all_objects = g_slist_append(CurrentFrame->all_objects,
                                                   TmpObject);
        TmpObject->name = create_uniq_object_name(CurrentFrame, _("object"));
        TmpObject->frame = CurrentFrame;
        break;

      default:
        g_message("parse error... unknown token (%s)\n", ReservedWordTable[Token.function_id].String);
        return 1;
    }
    get_token();
  }
  return 0;
}

/*****************************************************************************
*  load_pov_by_name
******************************************************************************/
void load_pov_by_name(gchar *name)
{
  GtkWidget  *shell;
  ViewStruct *view_data;
  init_load_pov(name);
  PovFile = g_new(ListFile, 1);
  PovFile->Prev = PovFile->Next = NULL;
  PovFile->Line = 1;
  PovFile->handle = LibOpen(name);
  PovFile->Name = g_strdup(name);

  /* test for #ifdef/#ifndef directive */
  {
    gdouble *new_float;
    SymbolTableEntry *new_entry = g_new(SymbolTableEntry, 1);
    new_entry->type = TFloatID;
    new_entry->data = new_float = g_new(double, 1);
    *new_float = 3.14;
    g_hash_table_insert(CurrentFrame->all_constants[0],
                        "dindinx", new_entry);
  }
  if (parse_pov())
  { /* There's an error in the parsing -> we must free the memory. */
    /* FIXME: DestroyFrame(CurrentFrame); */
    return;
  }

  g_slist_foreach(CurrentFrame->all_objects,
                  (GFunc)giram_object_build_triangle_mesh,
                  NULL);
  ComputeCamera(CurrentFrame->all_cameras->data);

  /* FIXME: update the csg_tree_model! */
  if (show_xy_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(CurrentFrame, ORTHO_XY_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
    FitToSceneFunc();
  }
  if (show_xz_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(CurrentFrame, ORTHO_XZ_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
    FitToSceneFunc();
  }
  if (show_zy_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(CurrentFrame, ORTHO_ZY_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
    FitToSceneFunc();
  }
  if (show_camera_view)
  {
    camera_view_new(CurrentFrame);
    /* FIXME */
  }
}

/*****************************************************************************
*  LoadPov
******************************************************************************/
void LoadPov(GtkWidget *w, GtkFileSelection *FS)
{
  gchar *Name;

  Name = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(FS)));

  load_pov_by_name(Name);
}

