/* dxf.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2001 DindinX <David@dindinx.org>
 * Copyright (C) 1999-2001 Noah Davis of VIEWS Net, Inc. <noah@viewsnet.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * This file is from Noah Davis.
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#ifndef DXF_STANDALONE
#include "giram.h"
#include "giramrc.h"
#include "object.h"
#include "csgtree.h"
#include "view.h"
#include "camera.h"
#include "camera_view.h"
#include "frame.h"
#include "primitives/cylinder.h"
#include "primitives/sphere.h"
#include "primitives/triangle.h"
#include "primitives/mesh.h"
#include "widgets/giramviewshell.h"
#endif

#include "dxf.h"

#ifdef DXF_STANDALONE
int main (int argc, char**argv) { return (dxf_parse(argv[1])); }
#endif

DXF_Data *dxf_parse(char *fname)
{
  FILE *fp;
  char buf[256];
  int code;
  char value[DXF_LINE_MAX];
  int err;
  DXF_Data *dat;

  dat = malloc(sizeof(DXF_Data));
  /* Clear out our struct */

  bzero(dat, sizeof(DXF_Data));

  fp = dxf_open(fname);

  if (fp==NULL)
  {
    sprintf (buf, "Couldn't open file '%s'", fname);
    perror(buf);
    return NULL;
  }

  dxf_determine_type(fp, dat);

#ifdef DXF_DEBUG
  if (dat->type == DXF_ASCII)
    fprintf(stderr, "ASCII DXF file.\n");
  else if (dat->type == DXF_BINARY)
    fprintf(stderr, "Binary DXF File.\n");
  else
#endif
  if (dat->type == DXF_ERROR)
  {
    perror("Couldn't determine DXF file type");
    dxf_close(fp);
    return NULL;
  }
  if (dat->type == DXF_BINARY)
  {
    dxf_err("Binary DXF files not implemented yet.\n", dat);
    dxf_close(fp);
    return NULL;
  }
  /* ASCII Only for now... */
  while ((err = dxf_getpair(fp, &code, value, dat)))
  {
    dat->lineno++; /* First line of the pair */
    /* We're expecting section headers here, and only section headers! */
    if (code != DXF_CODE_ENTITY)
    {
      /* Damn.. not a section header.. not even an entity type! */
      dxf_err("Didn't find expected entity type.", dat);
      dxf_close(fp);
      return NULL;
    }
    /* Up the line count */
    dat->lineno++;

    /* Now to check the entity type.... */
    if (strstr(value, "EOF"))
    {
      /* Found the proper End Of File marker... bail out */
      break;
    }
    /* Not EOF, so it'd better be a SECTION or we're DOOMED! :) */
    if (!strstr(value, "SECTION"))
    {
      /* Doomed, I tell ya.... */
      dxf_err("Didn't find expected SECTION entity.", dat);
      dxf_close(fp);
      return NULL;
    }
    if (!dxf_parse_section(fp, dat))
    {
      dxf_err("Error parsing section", dat);
      dxf_close(fp);
      return NULL;
	  }
  }
  if (err == FALSE)
  {
    dxf_err("Some kind of error occured.", dat);
  }

#ifdef DXF_DEBUG
  else
    fprintf(stderr, "DXF File parsed successfully.\n");

  fprintf(stderr, "BBox:  x1:% 9.5f y1:% 9.5f z1:% 9.5f\n",
          dat->bbox_xmin, dat->bbox_ymin, dat->bbox_zmin);
  fprintf(stderr, "       x2:% 9.5f y2:% 9.5f z2:% 9.5f\n",
          dat->bbox_xmax, dat->bbox_ymax, dat->bbox_zmax);
  fprintf(stderr, "\nLayers: %d\n", dat->layer_count);

  fprintf(stderr, "\nTotal Entities: %d\n\n", dat->entities);
  fprintf(stderr, "SOLIDS: %d\n3DFACES: %d\nPOLYLINES: %d\n",
          dat->solid_count, dat->threedface_count, dat->pline_count);
  fprintf(stderr, "VERTICES: %d\nLINES: %d\nARCS: %d\n",
          dat->vert_count, dat->line_count, dat->arc_count);
  fprintf(stderr,"CIRCLES: %d\nPOINTS: %d\n",
          dat->circle_count, dat->point_count);
#endif
  dxf_close(fp);

#ifdef DXF_DEBUG
  dxf_free_all(dat);
   free(dat);
#endif

  return dat;
}

/* Simple "print error and line number" routine */
void dxf_err(char *string, DXF_Data *dat)
{
  fprintf (stderr, "%d: %s\n", dat->lineno, string);
}

/* Nuke trailing CRs and LFs from the end of a string. Not prefixed with
 * "dxf_" because this routine might be useful elsewhere. Let me know if
 * there's a clash
 */
void cr_strip(char *string)
{
  int len=strlen(string)-1;

  while ((string[len] == '\n') || (string[len] == '\r'))
  {
    string[len] = 0;
    len = strlen(string-1);
  }
}

/* Resize the bounding-box of this object/scene */
void dxf_dobox(double x, double y, double z, DXF_Data *dat)
{
   if (x > dat->bbox_xmax) dat->bbox_xmax = x;
   if (y > dat->bbox_ymax) dat->bbox_ymax = y;
   if (z > dat->bbox_zmax) dat->bbox_zmax = z;

   if (x < dat->bbox_xmin) dat->bbox_xmin = x;
   if (y < dat->bbox_ymin) dat->bbox_ymin = y;
   if (z < dat->bbox_zmin) dat->bbox_zmin = z;
}

/* Add a polyline to the linked list of polylines in DXF_Data */
int dxf_add_polyline(DXF_Polyline *pline, DXF_Data *dat)
{
  DXF_Polyline *iter;

  /* Up the counts */
  dat->pline_count++;
  dat->entities++;

  /* NULL out the next pointer on new polyline */
  pline->next=NULL;

  /* First case: no plines added yet */

  if (dat->plines==NULL)
  {
    dat->plines = pline;
    return TRUE;
  }

  /* Okay, now to find the end of the chain */
  for (iter=dat->plines ; iter->next ; iter=iter->next);

  /* Now to add the new one */
  iter->next = pline;

  return TRUE;
}

/* Add a 3d-face to the list */
int dxf_add_3dface(DXF_3DFace *face, DXF_Data *dat)
{
  DXF_3DFace *iter;

  /* Up the counts */
  dat->threedface_count++;
  dat->entities++;

  /* NULL out the next pointer on new face */
  face->next = NULL;

  /* First case: no faces added yet */

  if (dat->threedfaces == NULL)
  {
    dat->threedfaces = face;
    return (TRUE);
  }

  /* Okay, now to find the end of the chain */
  for (iter = dat->threedfaces ; iter->next ; iter = iter->next);

  /* Now to add the new one */
  iter->next = face;

  return (TRUE);
}

/* Add a CIRCLE */
int dxf_add_circle(DXF_Circle *circle, DXF_Data *dat)
{
  DXF_Circle *iter;

  /* Up the counts */
  dat->circle_count++;
  dat->entities++;

  /* NULL out the next pointer on new face */
  circle->next = NULL;

  /* First case: no faces added yet */
  if (dat->circles == NULL)
  {
    dat->circles=circle;
    return (TRUE);
  }

  /* Okay, now to find the end of the chain */
  for (iter=dat->circles ; iter->next ; iter=iter->next);

  /* Now to add the new one */
  iter->next = circle;

  return (TRUE);
}

/* Add an ARC */
int dxf_add_arc(DXF_Arc *arc, DXF_Data *dat)
{
  DXF_Arc *iter;

  /* Up the counts */
  dat->arc_count++;
  dat->entities++;

  /* NULL out the next pointer on new face */
  arc->next = NULL;

  /* First case: no faces added yet */

  if (dat->arcs == NULL)
  {
    dat->arcs = arc;
    return (TRUE);
  }

  /* Okay, now to find the end of the chain */
  for (iter=dat->arcs ; iter->next ; iter=iter->next);

  /* Now to add the new one */
  iter->next = arc;

  return (TRUE);
}

/* Add SOLID */
int dxf_add_solid(DXF_Solid *solid, DXF_Data *dat)
{
  DXF_Solid *iter;

  /* Up the counts */
  dat->solid_count++;
  dat->entities++;

  /* NULL out the next pointer on new face */
  solid->next = NULL;

  /* First case: no faces added yet */

  if (dat->solids == NULL)
  {
    dat->solids = solid;
    return (TRUE);
  }

  /* Okay, now to find the end of the chain */
  for (iter=dat->solids ; iter->next ; iter=iter->next);

  /* Now to add the new one */
  iter->next = solid;

  return (TRUE);
}

/* Add a solo vertex */
int dxf_add_vertex(DXF_Vertex *vert, DXF_Data *dat)
{
  DXF_Vertex *iter;

  vert->next = NULL;

  /* Up the counts */
  dat->vert_count++;
  dat->entities++;

  if (dat->verts == NULL)
  {
    dat->verts = vert;
    return (TRUE);
  }

  /* Okay, now to find the end of the chain */
  for (iter=dat->verts ; iter->next ; iter=iter->next);

  /* Now to add the new one */
  iter->next = vert;

  return (TRUE);
}

/* Insert a LINE into the list */
int dxf_add_line(DXF_Line *line, DXF_Data *dat)
{
  DXF_Line *iter;

  line->next = NULL;

  /* Up the counts */
  dat->line_count++;
  dat->entities++;

  if (dat->lines == NULL)
  {
    dat->lines = line;
    return (TRUE);
  }

  /* Okay, now to find the end of the chain */
  for (iter=dat->lines ; iter->next ; iter=iter->next);

  /* Now to add the new one */
  iter->next = line;

  return (TRUE);
}

/* Insert a new POINT */
int dxf_add_point(DXF_Point *point, DXF_Data *dat)
{
  DXF_Point *iter;

  point->next = NULL;

  /* Up the counts */
  dat->point_count++;
  dat->entities++;

  if (dat->points == NULL)
  {
    dat->points = point;
    return (TRUE);
  }

  /* Okay, now to find the end of the chain */
  for (iter=dat->points ; iter->next ; iter=iter->next);

  /* Now to add the new one */
  iter->next = point;

  return (TRUE);
}

/* Add a VERTEX to the linked list of vertices in a POLYLINE */
int dxf_add_vertex_to_polyline(DXF_Vertex *vertex, DXF_Polyline *pline)
{
  DXF_Vertex *iter;

  /* NULL out new next */
  vertex->next = NULL;

  /* First case */
  if (pline->verts == NULL)
  {
    pline->verts = vertex;
    return (TRUE);
  }

  /* Okay, traverse the chain */
  for (iter=pline->verts ; iter->next ; iter=iter->next);

  /* Plug it in */
  iter->next = vertex;

  return (TRUE);
}

/* Add a layer to the list and return a pointer, or return a pointer
 * to an existing layer string if a match already exists
 */
char *dxf_add_layer(char *name, DXF_Data *dat)
{
  DXF_Layer *iter;
  DXF_Layer *layer;

  layer = malloc(sizeof(DXF_Layer));
  if (layer == NULL)
    return (NULL);

  strcpy(layer->name, name);
  layer->next = NULL;

  /* First case: is the layers pointer empty? If so, life is easy */
  if (dat->layers == NULL)
  {
    dat->layers = layer;
    dat->layer_count++;
    return (layer->name);
  }

  /* Determine if we know about this layer already */
  for (iter=dat->layers ; iter ; iter=iter->next)
  {
    if (!strcmp(iter->name, name))
    {
      free(layer);
      return (iter->name);
    }
  }

  /* Okay, need to insert a new layer object */
  for (iter=dat->layers ; iter->next ; iter=iter->next);
  iter->next = layer;
  dat->layer_count++;
  return (layer->name);
}

/* Free up the entire tree... this should de-allocate ANY allocated mem,
 * since all allocated mem is attatched to DXF_Data somehow.
 */
void dxf_free_all(DXF_Data *dat)
{
  DXF_Vertex *verts, *vold;
  DXF_Line *lines, *lold;
  DXF_3DFace *faces, *fold;
  DXF_Polyline *plines, *pold;
  DXF_Layer *layers, *lsold;
  DXF_Arc *arcs, *aold;
  DXF_Circle *circs, *cold;
  DXF_Solid *sols, *sold;
  DXF_Point *points, *pnold;

  /* Free the solo verts */
  verts = dat->verts;
  while (verts)
  {
    vold = verts;
    verts = verts->next;
    free(vold);
  }

  /* Free the lines */
  lines = dat->lines;
  while (lines)
  {
    lold = lines;
    lines = lines->next;
    free(lold);
  }

  /* Free the faces */
  faces = dat->threedfaces;
  while(faces)
  {
    fold = faces;
    faces = faces->next;
    free(fold);
  }

  /* Free up the polylines */
  plines = dat->plines;
  while (plines)
  {
    verts = plines->verts;
    while (verts)
    {
      vold = verts;
      verts = verts->next;
      free(vold);
    }
    pold = plines;
    plines = plines->next;
    free(pold);
  }

  /* Free the ARCS */
  arcs = dat->arcs;
  while (arcs)
  {
    aold = arcs;
    arcs = arcs->next;
    free(aold);
  }

  /* Free the CIRCLES */
  circs = dat->circles;
  while (circs)
  {
    cold = circs;
    circs = circs->next;
    free(cold);
  }

  /* Free the SOLIDS */
  sols = dat->solids;
  while (sols)
  {
    sold = sols;
    sols = sols->next;
    free(sold);
  }

  /* Free the POINTS */
  points = dat->points;
  while (points)
  {
    pnold = points;
    points = points->next;
    free(pnold);
  }

  /* Free up the layers */
  layers = dat->layers;
  while (layers)
  {
    lsold = layers;
    layers = layers->next;
    free(lsold);
  }
}

#ifndef DXF_STANDALONE
/*****************************************************************************
*  LoadDXF
******************************************************************************/
void LoadDXF(GtkWidget *w, GtkFileSelection *FS)
{
  char        *Name;
  FrameStruct *TmpFrame;
  DXF_Data    *dat;
  DXF_3DFace  *Tmp3dFace;
  DXF_Line    *TmpLine;
  DXF_Vertex  *TmpVertex;
  GtkWidget   *shell;
  ViewStruct  *view_data;

  Name = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(FS)));

  dat = dxf_parse(Name);

/***************************** INSERT HERE ********************************/
  if (dat == NULL)
    return;
  TmpFrame = NewFrame(Name);

  /* 3DFaces*/
  {
    GSList       *tmp_list;
    ObjectStruct *TmpMesh;

    for (Tmp3dFace = dat->threedfaces ; Tmp3dFace ; Tmp3dFace=Tmp3dFace->next)
    {
      Vector P1, P2, P3;

      TmpMesh = NULL;
      for (tmp_list = TmpFrame->all_objects ; tmp_list ; tmp_list = tmp_list->next)
      {
        ObjectStruct *tmp_object = tmp_list->data;

        if ( (tmp_object->Type == MESH_OBJECT) &&
             (tmp_object->name)                &&
             (strcmp(tmp_object->name, Tmp3dFace->layer) == 0) )
        {
          TmpMesh = tmp_object;
          break;
        }
      }
      if (TmpMesh == NULL)
      {
        TmpMesh = giram_mesh_new();
        TmpMesh->name = g_strdup(Tmp3dFace->layer);
        TmpFrame->all_objects = g_slist_append(TmpFrame->all_objects, TmpMesh);
      }

      P1[0] = Tmp3dFace->x1; P1[1] = Tmp3dFace->y1; P1[2] = Tmp3dFace->z1;
      P2[0] = Tmp3dFace->x2; P2[1] = Tmp3dFace->y2; P2[2] = Tmp3dFace->z2;
      P3[0] = Tmp3dFace->x3; P3[1] = Tmp3dFace->y3; P3[2] = Tmp3dFace->z3;
      giram_mesh_add_triangle(TmpMesh, P1, P2, P3);

      if ((Tmp3dFace->x3 != Tmp3dFace->x4) ||
          (Tmp3dFace->y3 != Tmp3dFace->y4) ||
          (Tmp3dFace->z3 != Tmp3dFace->z4))
      {

        P1[0] = Tmp3dFace->x1; P1[1] = Tmp3dFace->y1; P1[2] = Tmp3dFace->z1;
        P2[0] = Tmp3dFace->x4; P2[1] = Tmp3dFace->y4; P2[2] = Tmp3dFace->z4;
        P3[0] = Tmp3dFace->x3; P3[1] = Tmp3dFace->y3; P3[2] = Tmp3dFace->z3;
        giram_mesh_add_triangle(TmpMesh, P1, P2, P3);
      }
    }
  }
  /* Lines*/
  for (TmpLine = dat->lines ; TmpLine ; TmpLine=TmpLine->next)
  {
    ObjectStruct *Cylinder;
    Vector Base, Apex;

    Base[0] = TmpLine->x1; Base[1] = TmpLine->y1; Base[2] = TmpLine->z1;
    Apex[0] = TmpLine->x2; Apex[1] = TmpLine->y2; Apex[2] = TmpLine->z2;
    Cylinder = giram_cylinder_new(Base, Apex, 10e-4);
    TmpFrame->all_objects = g_slist_append(TmpFrame->all_objects, Cylinder);
  }

  /* Vertices */
  for (TmpVertex = dat->verts ; TmpVertex ; TmpVertex=TmpVertex->next)
  {
    ObjectStruct *Sphere;
    Vector Center;

    Center[0]=TmpVertex->x; Center[1]=TmpVertex->y; Center[2]=TmpVertex->z;
    Sphere = giram_sphere_new(Center, 10e-4);
    TmpFrame->all_objects = g_slist_append(TmpFrame->all_objects, Sphere);
  }
/**************************************************************************/
  dxf_free_all(dat);
  free(dat);
  g_slist_foreach(TmpFrame->all_objects,
                  (GFunc)giram_object_build_triangle_mesh,
                  NULL);
  ComputeCamera(TmpFrame->all_cameras->data);
  giram_create_tree_model(TmpFrame);
    
  /* the created views should be chosen from the 'Preferences' */
  if (show_xy_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(TmpFrame, ORTHO_XY_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
    FitToSceneFunc();
  }
  if (show_xz_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(TmpFrame, ORTHO_XZ_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
    FitToSceneFunc();
  }
  if (show_zy_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    view_data = giram_view_new(TmpFrame, ORTHO_ZY_CAMERA);
    giram_view_shell_add_view(GIRAM_VIEW_SHELL(shell), view_data);
    gtk_box_pack_start_defaults(GTK_BOX(GIRAM_VIEW_SHELL(shell)->vbox),
                                view_data->disp_vbox);
    gtk_widget_show(shell);
    FitToSceneFunc();
  }
  if (show_camera_view)
  {
    /* FIXME */
    camera_view_new(TmpFrame);
  }
}

/*****************************************************************************
*  InsertDXF
******************************************************************************/
void InsertDXF(GtkWidget *w, GtkFileSelection *FS)
{
  char        *Name;
  FrameStruct *TmpFrame;
  GSList      *tmp_list;
  ViewStruct  *TmpView;
  DXF_Data    *dat;
  DXF_3DFace  *Tmp3dFace;
  DXF_Line    *TmpLine;
  DXF_Vertex  *TmpVertex;

  Name = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(FS)));

  dat = dxf_parse(Name);

/***************************** INSERT HERE ********************************/
  if (dat == NULL)
    return;
  TmpFrame = g_object_get_data(G_OBJECT(FS), "frame");

  /* 3DFaces*/
  {
    ObjectStruct *TmpMesh;
    GSList       *tmp_list;

    for (Tmp3dFace = dat->threedfaces ; Tmp3dFace ; Tmp3dFace=Tmp3dFace->next)
    {
      Vector P1, P2, P3;

      TmpMesh = NULL;
      for (tmp_list = TmpFrame->all_objects ; tmp_list ; tmp_list = tmp_list->next)
      {
        ObjectStruct *tmp_object = tmp_list->data;

        if ( (tmp_object->Type == MESH_OBJECT) &&
             (tmp_object->name)                &&
             (strcmp(tmp_object->name, Tmp3dFace->layer) == 0) )
        {
          TmpMesh = tmp_object;
          break;
        }
      }
      if (TmpMesh == NULL)
      {
        TmpMesh = giram_mesh_new();
        TmpMesh->name = g_strdup(Tmp3dFace->layer);
        TmpFrame->all_objects = g_slist_append(TmpFrame->all_objects, TmpMesh);
      }

      P1[0] = Tmp3dFace->x1; P1[1] = Tmp3dFace->y1; P1[2] = Tmp3dFace->z1;
      P2[0] = Tmp3dFace->x2; P2[1] = Tmp3dFace->y2; P2[2] = Tmp3dFace->z2;
      P3[0] = Tmp3dFace->x3; P3[1] = Tmp3dFace->y3; P3[2] = Tmp3dFace->z3;
      giram_mesh_add_triangle(TmpMesh, P1, P2, P3);

      if ((Tmp3dFace->x3 != Tmp3dFace->x4) ||
          (Tmp3dFace->y3 != Tmp3dFace->y4) ||
          (Tmp3dFace->z3 != Tmp3dFace->z4))
      {

        P1[0] = Tmp3dFace->x1; P1[1] = Tmp3dFace->y1; P1[2] = Tmp3dFace->z1;
        P2[0] = Tmp3dFace->x4; P2[1] = Tmp3dFace->y4; P2[2] = Tmp3dFace->z4;
        P3[0] = Tmp3dFace->x3; P3[1] = Tmp3dFace->y3; P3[2] = Tmp3dFace->z3;
        giram_mesh_add_triangle(TmpMesh, P1, P2, P3);
      }
    }
  }

  /* Lines*/
  for (TmpLine = dat->lines ; TmpLine ; TmpLine=TmpLine->next)
  {
    ObjectStruct *Cylinder;
    Vector Base, Apex;

    Base[0] = TmpLine->x1; Base[1] = TmpLine->y1; Base[2] = TmpLine->z1;
    Apex[0] = TmpLine->x2; Apex[1] = TmpLine->y2; Apex[2] = TmpLine->z2;
    Cylinder = giram_cylinder_new(Base, Apex, 10e-4);
    TmpFrame->all_objects = g_slist_append(TmpFrame->all_objects, Cylinder);
  }

  /* Vertices */
  for (TmpVertex = dat->verts ; TmpVertex ; TmpVertex=TmpVertex->next)
  {
    ObjectStruct *Sphere;
    Vector Center;

    Center[0]=TmpVertex->x; Center[1]=TmpVertex->y; Center[2]=TmpVertex->z;
    Sphere = giram_sphere_new(Center, 10e-4);
    TmpFrame->all_objects = g_slist_append(TmpFrame->all_objects, Sphere);
  }
/**************************************************************************/
  dxf_free_all(dat);
  free(dat);
  for (tmp_list = TmpFrame->all_views ; tmp_list ; tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(TmpFrame);
}
#endif
