/* 3ds.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 1999-2000 DindinX <David@dindinx.org>
 * Copyright (C) 1999-2000 Noah Davis of VIEWS Net, Inc. <noah@viewsnet.com>
 * Copyright (C) 2001 Fabrice Jouhaud <fjouhaud@easynet.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Based on dxf.c
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "giram.h"
#include "giramrc.h"
#include "object.h"
#include "csgtree.h"
#include "view.h"
#include "camera.h"
#include "camera_view.h"
#include "frame.h"
#include "primitives/cylinder.h"
#include "primitives/sphere.h"
#include "primitives/triangle.h"
#include "primitives/mesh.h"
#include "texture.h"

#include "giramintl.h"
#include "widgets/giramviewshell.h"

#ifdef USE_LIB3DS
#include <lib3ds/file.h>
#include <lib3ds/chunk.h>
#include <lib3ds/mesh.h>
#include <lib3ds/camera.h>
#include <lib3ds/light.h>
#include <lib3ds/material.h>

static int giram_parse_3ds(char *file_name, FrameStruct *frame);

/*
 * Compute the new vectors of the camera
 */
static void compute_camera_vector(CameraStruct *pCamera)
{
  Vector Vect;
  double Norme;
  double DirectionLength, UpLength, RightLength;
  double Handedness;

  /* Angle */
  if (pCamera->Angle < 0.0)
  {
    g_message(_("3DS: Negative viewing angle."));
    return;
  }
  if (pCamera->Type == CAMERA_PERSPECTIVE)
  {
    if (pCamera->Angle >= 180.0)
    {
      g_message(_("3DS: Viewing angle has to be smaller than 180 degrees."));
    }
    /* Normalize vector */
    Norme = V3DLength(pCamera->Direction);
    pCamera->Direction[0] /= Norme;
    pCamera->Direction[1] /= Norme;
    pCamera->Direction[2] /= Norme;
    /* Scale vector */
    Norme = V3DLength(pCamera->Right) / tan(pCamera->Angle/2.0 * M_PI/180.0)/2.0;
    pCamera->Direction[0] *= Norme;
    pCamera->Direction[1] *= Norme;
    pCamera->Direction[2] *= Norme;
  }

  /* Look At vector */
  DirectionLength = V3DLength(pCamera->Direction);
  UpLength = V3DLength(pCamera->Up);
  RightLength = V3DLength(pCamera->Right);
  VCross(Vect, pCamera->Up, pCamera->Direction);
  Handedness = V3DDot(Vect, pCamera->Right);

  pCamera->Direction[0] = pCamera->LookAt[0] - pCamera->Location[0];
  pCamera->Direction[1] = pCamera->LookAt[1] - pCamera->Location[1];
  pCamera->Direction[2] = pCamera->LookAt[2] - pCamera->Location[2];
  /* Check for zero length direction vector. */
  if (V3DLength(pCamera->Direction) < 10e-6)
  {
    g_message(_("3DS: Camera location and look_at point must be different.\n"));
    return;
  }
  /* Normalize vector */
  Norme = V3DLength(pCamera->Direction);
  pCamera->Direction[0] /= Norme;
  pCamera->Direction[1] /= Norme;
  pCamera->Direction[2] /= Norme;

  VCross(pCamera->Right, pCamera->Sky, pCamera->Direction);
  /* Normalize vector */
  Norme = V3DLength(pCamera->Right);
  pCamera->Right[0] /= Norme;
  pCamera->Right[1] /= Norme;
  pCamera->Right[2] /= Norme;

  VCross(pCamera->Up, pCamera->Direction, pCamera->Right);

  /* Scale vector */
  pCamera->Direction[0] *= DirectionLength;
  pCamera->Direction[1] *= DirectionLength;
  pCamera->Direction[2] *= DirectionLength;
  if (Handedness > 0.0)
  {
    pCamera->Right[0] *= RightLength;
    pCamera->Right[1] *= RightLength;
    pCamera->Right[2] *= RightLength;
  } else {
    pCamera->Right[0] *= -RightLength;
    pCamera->Right[1] *= -RightLength;
    pCamera->Right[2] *= -RightLength;
  }
  pCamera->Up[0] *= UpLength;
  pCamera->Up[1] *= UpLength;
  pCamera->Up[2] *= UpLength;
}

/*
 * Parse the 3DS file
 */
static int giram_parse_3ds(char *file_name, FrameStruct *frame)
{
  Lib3dsFile   *f = 0;
  Lib3dsMesh   *m;
  Lib3dsLight  *light;
  unsigned      p;
  ObjectStruct *TmpMesh;

  f = lib3ds_file_load(file_name);
  if (!f)
  {
    fprintf (stderr, "***ERROR***\nLoading file %s failed\n", file_name);
    return -1;
  }

  /* mesh */
  for (m = f->meshes; m; m = m->next)
  {
    Lib3dsMaterial *mat;

    if (m->faces == 0)
      continue;

    TmpMesh = giram_mesh_new();
    TmpMesh->name = g_strdup(m->name);

    printf("Mesh name=%s\n", TmpMesh->name);
    /*
       printf("Mesh name=%s\n",TmpMesh->Name);
       printf("\tnb faces=%ld\n",m->faces);
       printf("\tcolor=%d\n",m->color);
       printf("\tmaterial=%s\n",m->faceL->material);
     */
    if(m->faceL && m->faceL->material[0])
    {
      mat = lib3ds_file_material_by_name(f, m->faceL->material);
      printf("\tmaterial=%s\n", m->faceL->material);
      if (mat->texture1_map.name[0])
        printf("\t\tmaterial texture1=%s\n", mat->texture1_map.name);
      if (mat->texture1_mask.name[0])
        printf("\t\tmaterial texture1=%s\n", mat->texture1_mask.name);
      if (mat->texture2_map.name[0])
        printf("\t\tmaterial texture1=%s\n", mat->texture2_map.name);
      if (mat->texture2_mask.name[0])
        printf("\t\tmaterial texture1=%s\n", mat->texture2_mask.name);
    /*
       printf("\tambient=%f %f %f %f\n", mat->ambient[0], mat->ambient[1], mat->ambient[2], mat->ambient[3]);
       printf("\tdiffuse=%f %f %f %f\n", mat->diffuse[0], mat->diffuse[1], mat->diffuse[2],mat->diffuse[3]);
       printf("\tspecular=%f %f %f %f\n", mat->specular[0], mat->specular[1], mat->specular[2],mat->specular[3]);
     */
      printf("\ttransparence=%f\n", mat->transparency);
      TmpMesh->Texture->Pigment->Type = PAT_SOLID_COLOR;
      TmpMesh->Texture->Pigment->Color[0] = mat->ambient[0];
      TmpMesh->Texture->Pigment->Color[1] = mat->ambient[1];
      TmpMesh->Texture->Pigment->Color[2] = mat->ambient[2];
      TmpMesh->Texture->Pigment->Color[3] = mat->transparency;
      TmpMesh->Texture->Pigment->Color[4] = 0.0;
    }

    for (p = 0; p < m->faces; ++p)
    {
      Vector P1, P2, P3;
      Lib3dsFace *face = &m->faceL[p];

      P1[0] = m->pointL[face->points[0]].pos[1];
      P1[1] = m->pointL[face->points[0]].pos[2];
      P1[2] = m->pointL[face->points[0]].pos[0];
      P2[0] = m->pointL[face->points[1]].pos[1];
      P2[1] = m->pointL[face->points[1]].pos[2];
      P2[2] = m->pointL[face->points[1]].pos[0];
      P3[0] = m->pointL[face->points[2]].pos[1];
      P3[1] = m->pointL[face->points[2]].pos[2];
      P3[2] = m->pointL[face->points[2]].pos[0];
      giram_mesh_add_triangle(TmpMesh, P1, P2, P3);
    }
    frame->all_objects = g_slist_append(frame->all_objects, TmpMesh);
  }

  /* Camera */
  if (f->cameras)
  {
    Vector P1;
    CameraStruct *Camera;
/*
    printf("Camera Name=%s\n", f->cameras->name);
    printf("\tpos=%f %f %f\n",
           f->cameras->position[1],
           f->cameras->position[2],
           f->cameras->position[0]);
    printf("\ttarget=%f %f %f\n",
           f->cameras->target[1],
           f->cameras->target[2],
           f->cameras->target[0]);
    printf("\tfov=%f\n",
           f->cameras->fov);
    printf("\troll=%f\n",
           f->cameras->roll);
*/
    Camera = frame->all_cameras->data;
    P1[0] = f->cameras->position[1];
    P1[1] = f->cameras->position[2];
    P1[2] = f->cameras->position[0];
    V3Deq(Camera->Location, P1[0], P1[1], P1[2]);
    P1[0] = f->cameras->target[1];
    P1[1] = f->cameras->target[2];
    P1[2] = f->cameras->target[0];
    V3Deq(Camera->LookAt, P1[0], P1[1], P1[2]);
    Camera->Angle = f->cameras->fov;

    compute_camera_vector(Camera);
  }

  /* Lights */
  for (light = f->lights; light; light = light->next)
  {
    LightSourceStruct *TmpLightSource;

    TmpLightSource = g_new(LightSourceStruct, 1);

    V3Dcopy(TmpLightSource->Location, light->position);

    TmpLightSource->Color[0] = light->color[0];
    TmpLightSource->Color[1] = light->color[1];
    TmpLightSource->Color[2] = light->color[2];
    TmpLightSource->Color[3] = 0.0;
    TmpLightSource->Color[4] = 0.0;
    TmpLightSource->FallOff = light->fall_off;
    if (light->spot_light)
      TmpLightSource->Type = SPOTLIGHT_LIGHTSOURCE;
    else
      TmpLightSource->Type = POINT_LIGHTSOURCE;

    frame->all_light_sources = g_slist_append(frame->all_light_sources, TmpLightSource);
  }

  lib3ds_file_free(f);

  g_slist_foreach(frame->all_objects,
                  (GFunc)giram_object_build_triangle_mesh,
                  NULL);

  return 0;
}

/*****************************************************************************
*  giram_load_3ds
******************************************************************************/
void giram_load_3ds(GtkWidget *w, GtkFileSelection *FS)
{
  char         *Name;
  FrameStruct  *TmpFrame;
  GtkWidget    *shell;

  Name = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(FS)));
  if(!strlen(Name) || Name == NULL)
    return;

  if(Name == NULL)
    return;

  TmpFrame = NewFrame(Name);

  if(giram_parse_3ds(Name, TmpFrame) != 0)
  {
    /* Error the frame must be destroyed */
    /* FIXME: DestroyFrame(TmpFrame); */
    return;
  }

  ComputeCamera(TmpFrame->all_cameras->data);
  giram_create_tree_model(TmpFrame);
    
  /* the created views should be chosen from the 'Preferences' */
  if (show_xy_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_wmclass(GTK_WINDOW(shell), "view_shell", "Giram");
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    NewView(TmpFrame, ORTHO_XY_CAMERA, shell);
    FitToSceneFunc();
  }
  if (show_xz_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_wmclass(GTK_WINDOW(shell), "view_shell", "Giram");
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    NewView(TmpFrame, ORTHO_XZ_CAMERA, shell);
    FitToSceneFunc();
  }
  if (show_zy_view)
  {
    shell = giram_view_shell_new();
    gtk_window_set_wmclass(GTK_WINDOW(shell), "view_shell", "Giram");
    gtk_window_set_default_size(GTK_WINDOW(shell), 512, 384);
    NewView(TmpFrame, ORTHO_ZY_CAMERA, shell);
    FitToSceneFunc();
  }
  if (show_camera_view)
  {
    camera_view_new(TmpFrame);
  }
}

/*****************************************************************************
*  giram_import_3ds
******************************************************************************/
void giram_insert_3ds(GtkWidget *w, GtkFileSelection *FS)
{
  char            *filename;
  FrameStruct     *TmpFrame;
  GSList          *tmp_list;
  ViewStruct      *TmpView;
  CameraStruct    *Camera;

  filename = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(FS)));
  if (!filename || strlen(filename)==0)
    return;

  TmpFrame = g_object_get_data(G_OBJECT(FS), "frame");

  /* Save frame camera */
  Camera = TmpFrame->all_cameras->data;

  if (giram_parse_3ds(filename, TmpFrame) != 0)
    return;

  /* destroy the camera load from the file */
  g_free(TmpFrame->all_cameras->data);

  /* Restore the frame camera */
  TmpFrame->all_cameras->data = Camera;

  for (tmp_list = TmpFrame->all_views ; tmp_list ; tmp_list = tmp_list->next)
  {
    TmpView = tmp_list->data;
    gtk_widget_queue_draw(TmpView->canvas);
  }
  giram_create_tree_model(TmpFrame);
}
#endif
