/* camera_view.c
 * Giram - A GPLed Modelling Program.
 * Copyright (C) 2001-2002 DindinX <David@dindinx.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "widgets/girampixmap.h"

#include "giram.h"
#include "camera.h"

#include "giramintl.h"

/******************************************************************************
*  camera_view_cancel_rendering
*******************************************************************************/
static void camera_view_cancel_rendering(GtkWidget *button)
{
  slow_module_fd *smfd;
  GtkWidget      *status;
  GtkAdjustment  *adj;

  g_print("Cancel rendering\n");
  smfd = g_object_get_data(G_OBJECT(button), "smfd");
  if (smfd)
  {
    g_print("Cancelling");
    pclose(smfd->pipe);
    g_free(smfd->big_buf);
    smfd->big_buf = NULL;
    gdk_input_remove(smfd->input_id);

    status = g_object_get_data(G_OBJECT(smfd->area), "status");
    gtk_label_set_text(GTK_LABEL(status), _("Done"));
    gtk_widget_set_sensitive(button, FALSE);

    adj = g_object_get_data(G_OBJECT(smfd->area), "progress_adj");
    gtk_adjustment_set_value(adj, 0.0);
    g_free(smfd);
    g_object_set_data(G_OBJECT(smfd->area), "smfd", NULL);
  }
}

/******************************************************************************
*  camera_slow_module_update
*******************************************************************************/
static void camera_slow_module_update(slow_module_fd *smfd/*, gint fd,
                                      GdkInputCondition condition*/)
{
  GtkAdjustment *adj;
  GtkWidget     *status, *cancel_button;

  smfd->module->update_func(smfd);
  adj = g_object_get_data(G_OBJECT(smfd->area), "progress_adj");
  if (smfd->big_buf)
  {
    gdk_window_set_back_pixmap(smfd->area->window, smfd->pixmap, FALSE);
    gdk_window_clear(smfd->area->window);

    gtk_adjustment_set_value(adj, 100.0*smfd->read_total/(gdouble)smfd->buf_size);
  } else
  {
    gdk_input_remove(smfd->input_id);
    g_object_set_data(G_OBJECT(smfd->area), "smfd", NULL);
    
    status = g_object_get_data(G_OBJECT(smfd->area), "status");
    
    gtk_label_set_text(GTK_LABEL(status), _("Done"));
    cancel_button = g_object_get_data(G_OBJECT(smfd->area), "cancel_button");
    g_free(smfd);
    gtk_widget_set_sensitive(cancel_button, FALSE);
    gtk_adjustment_set_value(adj, 0.0);
  }
}

/******************************************************************************
*  camera_canvas_configure_callback
*******************************************************************************/
static gboolean camera_canvas_configure_callback(GtkWidget *canvas,
                                                 GdkEventConfigure *ev)
{
  GtkWidget      *window;
  module_struct  *current_module;
  GdkPixmap      *back_pixmap = NULL;
  slow_module_fd *smfd;
  GtkWidget      *status;
  GtkWidget      *cancel_button;
  GtkAdjustment  *adj;

  window = gtk_widget_get_toplevel(canvas);
  current_module = g_object_get_data(G_OBJECT(canvas), "active_module");
  status = g_object_get_data(G_OBJECT(canvas), "status");

  switch (current_module->type)
  {
    case 1: /* fast module */
      back_pixmap = current_module->render_func(canvas, ev->width, ev->height);
      if (back_pixmap)
      {
        gdk_window_set_back_pixmap(canvas->window, back_pixmap, FALSE);
        gdk_drawable_unref(back_pixmap);
        gdk_window_clear(canvas->window);
      }
      gtk_label_set_text(GTK_LABEL(status), _("Done"));
      break;
    case 2: /* slow module */
      //g_print("slow module!\n");
      smfd = g_object_get_data(G_OBJECT(canvas), "smfd");
      if (smfd)
      {
        g_print("Cancelling");
        pclose(smfd->pipe);
        g_free(smfd->big_buf);
        smfd->big_buf = NULL;
        gdk_input_remove(smfd->input_id);

        status = g_object_get_data(G_OBJECT(smfd->area), "status");
        gtk_label_set_text(GTK_LABEL(status), _("Done"));
        cancel_button = g_object_get_data(G_OBJECT(smfd->area), "cancel_button");
        gtk_widget_set_sensitive(cancel_button, FALSE);

        adj = g_object_get_data(G_OBJECT(smfd->area), "progress_adj");
        gtk_adjustment_set_value(adj, 0.0);
        g_free(smfd);
        g_object_set_data(G_OBJECT(canvas), "smfd", NULL);
      }
      
      smfd = (slow_module_fd*)current_module->render_func(canvas, ev->width, ev->height);
      if (smfd)
      {
        if (smfd->pixmap)
        {
          gdk_window_set_back_pixmap(canvas->window, smfd->pixmap, FALSE);
          gdk_window_clear(canvas->window);
        }
        smfd->module = current_module;
        //g_print("adding fd: %d\n", smfd->fd);
        smfd->input_id = gdk_input_add(smfd->fd, GDK_INPUT_READ,
                                       (GdkInputFunction)camera_slow_module_update,
                                       smfd);
        gtk_label_set_text(GTK_LABEL(status), _("Rendering"));
        cancel_button = g_object_get_data(G_OBJECT(canvas), "cancel_button");
        g_object_set_data(G_OBJECT(cancel_button), "smfd", smfd);
        g_object_set_data(G_OBJECT(canvas), "smfd", smfd);
        gtk_widget_set_sensitive(cancel_button, TRUE);
      }
      break;
  }

  return FALSE;
}

/*****************************************************************************
*  camera_module_changed
******************************************************************************/
static void camera_module_changed(GtkWidget *button, module_struct *module)
{
  GtkWidget      *canvas, *status, *cancel_button;
  GtkAdjustment  *adj;
  module_struct  *old_module;
  slow_module_fd *smfd;

  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)))
  {
    canvas = g_object_get_data(G_OBJECT(button), "canvas");
    old_module = g_object_get_data(G_OBJECT(canvas), "active_module");
    if (old_module->type == 2)
    {
      g_print("should cancel the slow module!\n");
      smfd = g_object_get_data(G_OBJECT(canvas), "smfd");
      if (smfd)
      {
        g_print("Cancelling");
        pclose(smfd->pipe);
        g_free(smfd->big_buf);
        smfd->big_buf = NULL;
        gdk_input_remove(smfd->input_id);

        status = g_object_get_data(G_OBJECT(smfd->area), "status");
        gtk_label_set_text(GTK_LABEL(status), _("Done"));
        cancel_button = g_object_get_data(G_OBJECT(smfd->area), "cancel_button");
        gtk_widget_set_sensitive(cancel_button, FALSE);

        adj = g_object_get_data(G_OBJECT(smfd->area), "progress_adj");
        gtk_adjustment_set_value(adj, 0.0);
        g_free(smfd);
        g_object_set_data(G_OBJECT(canvas), "smfd", NULL);
      }
    }
    g_object_set_data(G_OBJECT(canvas), "active_module", module);
    gtk_widget_queue_resize(canvas);
  }
}

void camera_view_save_image_callback(GtkWidget *button, GtkWidget *selector)
{
  GdkPixbuf *gdk_pixbuf;
  GError    *error;
  GtkWidget *canvas;
  gchar     *filename;

  canvas = g_object_get_data(G_OBJECT(selector), "canvas");
  filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(selector));
  gdk_pixbuf = gdk_pixbuf_get_from_drawable(NULL, canvas->window,
                                            gtk_widget_get_colormap(canvas),
                                            0, 0, 0, 0,
                                            canvas->allocation.width,
                                            canvas->allocation.height);
  gdk_pixbuf_save(gdk_pixbuf, filename,  "jpeg", &error, "quality", "100",  NULL);
}

void camera_view_save_image(GtkWidget *button, GtkWidget *canvas)
{
  GtkWidget *selector;

  selector = gtk_file_selection_new(_("Save image (only jpeg)"));
  g_object_set_data(G_OBJECT(selector), "canvas", canvas);
  g_signal_connect(G_OBJECT(GTK_FILE_SELECTION(selector)->ok_button), "clicked",
                   G_CALLBACK(camera_view_save_image_callback), selector);
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(selector)->ok_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy),
                           selector);
  g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(selector)->cancel_button),
                           "clicked", G_CALLBACK(gtk_widget_destroy), selector);
  gtk_widget_show(selector);
}

/*static GtkItemFactoryEntry camera_view_menu[] =
{
  {"/Save this image", NULL, camera_view_save_image, 0, "<Item>"},
  {"/Refresh",         NULL, camera_view_refresh,    0, "<Item>"}
};*/

/*****************************************************************************
*  new_camera_view_new
******************************************************************************/
void camera_view_new(FrameStruct *frame)
{
  GtkWidget     *window, *vbox, *notebook, *canvas;
  GtkWidget     *hbox, *pixmap, *button = NULL;
  GList         *tmp_list;
  GtkWidget     *status_label, *progress, *cancel_button;
  GtkWidget     *menu_button;
  GtkAdjustment *adj;
  CameraStruct  *Camera;
  GtkTooltips   *tooltip;

  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  tooltip = gtk_tooltips_new();

  notebook = gtk_notebook_new();
  gtk_container_add(GTK_CONTAINER(window), notebook);

  {
    vbox = gtk_vbox_new(FALSE, 2);
    Camera = frame->all_cameras->data;
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook),
                             vbox,
                             gtk_label_new(Camera->name));

    canvas = gtk_drawing_area_new();
    gtk_widget_set_size_request(canvas, 400, 300);
    gtk_box_pack_start_defaults(GTK_BOX(vbox), canvas);
    g_signal_connect(G_OBJECT(canvas), "configure_event",
                     G_CALLBACK(camera_canvas_configure_callback), NULL);
    g_object_set_data(G_OBJECT(canvas), "frame", frame);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    for (tmp_list = render_modules ; tmp_list ; tmp_list = g_list_next(tmp_list) )
    {
      module_struct *module = tmp_list->data;

      pixmap = giram_pixmap_new(module->xpm);
      button = gtk_radio_button_new_from_widget((GtkRadioButton *)button);
      gtk_toggle_button_set_mode(GTK_TOGGLE_BUTTON(button), FALSE);
      gtk_container_add(GTK_CONTAINER(button), pixmap);
      gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
      g_signal_connect(G_OBJECT(button), "toggled",
                       G_CALLBACK(camera_module_changed), module);
      g_object_set_data(G_OBJECT(button), "canvas", canvas);
      gtk_tooltips_set_tip(tooltip, button, module->name, module->name);      
    }
    /* select first module by default FIXME: what if there's no modules? */
    /* well, there's at least our modules... */
    g_object_set_data(G_OBJECT(canvas), "active_module", render_modules->data);

    status_label = gtk_label_new("Initializing");
    gtk_misc_set_alignment(GTK_MISC(status_label), 0.0, 0.5);
    gtk_misc_set_padding(GTK_MISC(status_label), 6, 0);
    gtk_box_pack_start_defaults(GTK_BOX(hbox), status_label);
    g_object_set_data(G_OBJECT(canvas), "status", status_label);

    adj = (GtkAdjustment *)gtk_adjustment_new(0.0, 0.0, 100.0, 1.0, 1.0, 1.0);
    progress = gtk_progress_bar_new_with_adjustment(adj);
    gtk_box_pack_start(GTK_BOX(hbox), progress, FALSE, FALSE, 0);
    g_object_set_data(G_OBJECT(canvas), "progress_adj", adj);

    cancel_button = gtk_button_new_with_label(_("Cancel"));
    gtk_widget_set_sensitive(cancel_button, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), cancel_button, FALSE, FALSE, 0);
    g_object_set_data(G_OBJECT(canvas), "cancel_button", cancel_button);
    g_signal_connect(G_OBJECT(cancel_button), "clicked",
                     G_CALLBACK(camera_view_cancel_rendering), NULL);

    menu_button = gtk_button_new_with_label(_("Save"));
    gtk_box_pack_start(GTK_BOX(hbox), menu_button, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(menu_button), "clicked",
                     G_CALLBACK(camera_view_save_image), canvas);
  }

  gtk_widget_show_all(window);
}
