/* The GIMP -- an image manipulation program
 * Copyright (C) 1995-1997 Spencer Kimball and Peter Mattis
 *
 * GIMP Color Manager Plug-In
 * Copyright (C) 2000, 2001  Karl Heinz Kremer <khk@khk.net>
 * Gimp 2.X port 2004 Jordi Cantón <jordinitrox@virtual-sub.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "config.h"

#include <sys/stat.h>
#include <errno.h>

#include <gtk/gtk.h>

#include <libgimp/gimp.h>

#include <lcms.h>

#include "utils.h"
#include "render.h"

#include "plugin-intl.h"

/* local functions */

static void cm_error (const char *, ...);

/*  Public functions  */


GimpPDBStatusType    render (gint32              image_ID,
	             GimpDrawable       *drawable,
                     PlugInVals         *vals,
		     PlugInImageVals    *image_vals,
	             PlugInUIVals * ui)
{
  //GimpDrawable *drawable;
  gint32 drawable_ID=drawable->drawable_id;
  GimpPixelRgn src_rgn, dest_rgn;
  guchar *src, *dest;
  gpointer pr;
  gint y, x1, x2, y1, y2;
  gint total, processed = 0;
  cmsHTRANSFORM xform;
  cmsHPROFILE *in = NULL, *out = NULL;
  guchar in_profile_path[MAXPATHLEN];
  guchar out_profile_path[MAXPATHLEN];
  guchar ws_profile_path[MAXPATHLEN];
  guchar *in_path = NULL, *out_path = NULL;
  GimpParasite *parasite;
  gint profile_size = 0;
  guchar *icc_profile = NULL;
  
  cmsErrorAction(LCMS_ERROR_IGNORE);
  
  drawable = gimp_drawable_get (drawable_ID);

  sprintf (in_profile_path, "%s%c%s", vals->cm_directory,G_DIR_SEPARATOR,
	   vals->cm_input_profile);
  sprintf (out_profile_path, "%s%c%s", vals->cm_directory,G_DIR_SEPARATOR,
	   vals->cm_output_profile);
  sprintf (ws_profile_path, "%s%c%s", vals->cm_directory,G_DIR_SEPARATOR,
	   vals->cm_work_space);

  /* load profiles */

  /* 
   * If the image already comes with a profile we are using this instead
   * of the configured color workspace profile.
   */
  parasite = gimp_image_parasite_find (image_ID, "icc-profile");

  if (parasite)
    {
      profile_size = gimp_parasite_data_size (parasite);
      icc_profile =(guchar *) gimp_parasite_data (parasite);
    }

  if (vals->cm_direction == INPUT_PROFILE || vals->cm_direction == WS_PROFILE )
    {
      if (vals->cm_use_embedded && ui->cm_have_embedded)
	{
	  out = cmsOpenProfileFromMem (icc_profile, profile_size);
	  gimp_parasite_free (parasite);
	  out_path = _("from memory");
	}
      else
	{
	     out = cmsOpenProfileFromFile (ws_profile_path, "r");
	     out_path = ws_profile_path;
	}
         in = cmsOpenProfileFromFile (in_profile_path, "r");
    }
  else if (vals->cm_direction == WS_PROFILE )
    {
      if (ui->cm_have_embedded)
	{
	  in = cmsOpenProfileFromMem (icc_profile, profile_size);
	  gimp_parasite_free (parasite);
	  in_path = _("from memory");
	}
      else
	{
	  in = NULL;
	  in_path = _("not found");
	}
      out = cmsOpenProfileFromFile (ws_profile_path, "r");
    }
  else if (vals->cm_direction == OUTPUT_PROFILE || vals->cm_preview == TRUE)
    {
      if (ui->cm_have_embedded && vals->cm_use_embedded)
	{
	  in = cmsOpenProfileFromMem (icc_profile, profile_size);
	  gimp_parasite_free (parasite);
	  in_path = _("from memory");
	}
      else
	{
	  in = cmsOpenProfileFromFile (ws_profile_path, "r");
	  in_path = ws_profile_path;
	}
      out = cmsOpenProfileFromFile (out_profile_path, "r");
    }
  if (in == NULL)
    {
      cm_error (_("Can not open profile %s\n"), in_path);
      return GIMP_PDB_EXECUTION_ERROR;
    }

  if (out == NULL)
    {
      cm_error (_("Can not open profile %s\n"), out_path);
      return GIMP_PDB_EXECUTION_ERROR;
    }

  gimp_drawable_mask_bounds (drawable_ID, &x1, &y1, &x2, &y2);
  total = (x2 - x1) * (y2 - y1);


  gimp_tile_cache_ntiles (2 * (drawable->width / gimp_tile_width () + 1));
  gimp_pixel_rgn_init (&src_rgn, drawable,
		       x1, y1, (x2 - x1), (y2 - y1), FALSE, FALSE);
  gimp_pixel_rgn_init (&dest_rgn, drawable,
		       x1, y1, (x2 - x1), (y2 - y1), TRUE, TRUE);


  if (vals->cm_preview == TRUE && vals->cm_direction == OUTPUT_PROFILE)
    {
      /* create proofing transform - need a third profile for this:
       *
       * Workspace profile
       * Display profile
       * Output (printer) profile
       *
       * So far we have not dealt with display profiles (mainly 
       * because it is not possible to do this without modifying
       * any of the display_modules), so for this purpose I'm just
       * using the workspace profile twice so that only the proof
       * is applied.
       */

      cmsSetAlarmCodes (0, 255, 127);
      xform = cmsCreateProofingTransform (in, TYPE_RGB_8, in, TYPE_RGB_8,
					  out, vals->cm_rendering_intent,
					  INTENT_ABSOLUTE_COLORIMETRIC,
					  vals->
					  cm_gamut_check ? cmsFLAGS_GAMUTCHECK
					  : 0);

      if (xform == NULL)
	cm_error (_("Can not create xform\n"));
    }
  else
    {
      if (gimp_drawable_has_alpha (drawable_ID)
	  && !gimp_drawable_is_indexed (drawable_ID))
	{
	  xform = cmsCreateTransform (in, TYPE_RGBA_8,
				      out, TYPE_RGBA_8,
				      vals->cm_rendering_intent, 0);
	}
      else
	{
	  xform = cmsCreateTransform (in, TYPE_RGB_8,
				      out, TYPE_RGB_8,
				      vals->cm_rendering_intent, 0);
	}
    }

  gimp_progress_init (_("Color Manager: Processing ..."));

  if (gimp_drawable_is_indexed (drawable_ID))
    {
      int colors;
      guchar *cmap;

      gimp_progress_update (0.25);

      cmap = gimp_image_get_cmap (image_ID, &colors);

      gimp_progress_update (0.5);

      src = cmap;
      dest = cmap;

      cmsDoTransform (xform, src, dest, colors * 3);

      gimp_progress_update (0.75);

      gimp_image_set_cmap (image_ID, cmap, colors);

      gimp_progress_update (1.0);

    }
  else if (gimp_drawable_is_rgb (drawable_ID))
    {

      for (pr = gimp_pixel_rgns_register (2, &src_rgn, &dest_rgn);
	   pr != NULL; pr = gimp_pixel_rgns_process (pr))
	{
	  for (y = 0; y < src_rgn.h; y++)
	    {
	      src = src_rgn.data + y * src_rgn.rowstride;
	      dest = dest_rgn.data + y * dest_rgn.rowstride;

	      cmsDoTransform (xform, src, dest, src_rgn.w);

	      processed += src_rgn.w * 3;

	      if ((processed % (total / PROGRESS_UPDATE_NUM)) == 0)
		gimp_progress_update ((gdouble) processed / (gdouble) total);
	    }
	}
    }

  cmsDeleteTransform (xform);
  cmsCloseProfile (in);
  cmsCloseProfile (out);

  /* create a new parasite icc-profile containing the actual profile data */

  if (vals->cm_direction == INPUT_PROFILE && !ui->cm_have_embedded)
    {
      GimpPDBStatusType result;
      
      result = embed_profile (image_ID, ws_profile_path);
      if (result != GIMP_PDB_SUCCESS)
	{
	  return result; 
	}
    }
  else if (vals->cm_direction == WS_PROFILE )
    {
      GimpPDBStatusType result;
      
      result = embed_profile (image_ID, ws_profile_path);
      if (result != GIMP_PDB_SUCCESS)
	{
	  return result; 
	}
    }

  gimp_progress_update (1.0);
  gimp_drawable_flush (drawable);
  gimp_drawable_merge_shadow (drawable->drawable_id, TRUE);
  gimp_drawable_update (drawable->drawable_id, x1, y1, (x2 - x1), (y2 - y1));

  gimp_drawable_detach (drawable);

  return GIMP_PDB_SUCCESS;
}

static void
cm_error (const char *fmt, ...)
{
  va_list ap;
  guchar str[MAXPATHLEN];

  va_start (ap, fmt);
  vsnprintf (str, MAXPATHLEN, fmt, ap);
  va_end (ap);
  
  g_message (str);
}

