/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * GIMP Color Manager Plug-In
 * Copyright (C) 2000, 2001  Karl Heinz Kremer <khk@khk.net>
 * Conversion to Gimp 2.0 plugin  2004  Jordi Canton Padilla <jordinitrox@virtual-sub.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "config.h"

#include <assert.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <dirent.h>

#include <stdlib.h>		/*  For srand()  */
#include <time.h>		/*  For time()   */

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include <lcms.h>

#include "utils.h"
#include "main.h"

#include "plugin-intl.h"

#include "interface.h"


/*  Constants  */

#define INTENT_STR_LENGTH	(32)

#define PERCEPTUAL_NAME         _("Perceptual")
#define REL_COLORIMETRIC_NAME   _("Relative Colorimetric")
#define SATURATION_NAME         _("Saturation")
#define ABS_COLORIMETRIC_NAME   _("Absolute Colorimetric")
#define INPUT_NAME  _("Input to work space")
#define OUTPUT_NAME _("Work Space to Output")
#define WORKSPACE_NAME _("Embedded profile to default work space")
#define ENBED_BUT_NAME _("Assign WS Profile")
#define ENBED_BUT_NAME2 _("Detach Embedded Profile")
/*  Local function prototypes  */

static void profile_combo_cb (GtkWidget *, gpointer);
static void rendering_intent_combo_cb (GtkWidget *, gpointer);

static void fselect_callback (GtkWidget * widget, gpointer data);

static void create_file_selection (void);
static void load_file_selection (void);
static void file_ok_sel (GtkWidget *, GtkFileSelection *);
static void file_cancel_sel (GtkWidget *, GtkFileSelection *);
static void refresh_callback (GtkWidget *, gpointer);
static void embed_profile_callback (GtkWidget *, gpointer);
static void file_ok_sel (GtkWidget *, GtkFileSelection *);
static void set_conversion_type ();
static void rescan_dir (guchar *);
static void cm_error (const char *, ...);
static int get_rendering_intent (void);
static gboolean check_for_gamut_tag(guchar * theProfileName);

/*  Local variables  */

static PlugInUIVals *ui_state = NULL;
static PlugInVals *state = NULL;

static GtkWidget *rendering_intent_combo = NULL;	/* rendeing intent selection */
static GtkWidget *in_profile_combo = NULL;		/* input profile selection */
static GtkWidget *out_profile_combo = NULL;		/* output profile selection */
static GtkWidget *work_space_combo = NULL;	/* work space profile selection */
static GtkWidget *transformation_combo = NULL;	/* transformation selection */
static GtkWidget *file_sel = NULL;				/* directory selection */
static GtkWidget *embedded_button = NULL;		/* use embedded profile check box */
static GtkWidget *preview_button = NULL;		/* preview button */
static GtkWidget *gamut_check_button = NULL;	/* use gamut alert color */
static GtkWidget *refresh_button = NULL;		/* refresh list of profiles */
static GtkWidget *dir_button = NULL;			/* select directory */
static GtkWidget *embed_profile_button = NULL;	/* embed current workspace profile */
static GtkWidget *info_tbl = NULL;			/*  */
static GtkWidget *top_tbl = NULL;			/*Main table conyaining most widgets  */
static GtkWidget *dir_lbl = NULL;			/*Current profile directory label  */
static GtkWidget *emb_lbl = NULL;			/*Current embebed profile label  */

static gboolean run = FALSE;


/*  Public functions  */

gboolean
dialog (gint32              image_ID,
	GimpDrawable       *drawable,
	PlugInVals         *vals,
	PlugInImageVals    *image_vals,
	PlugInDrawableVals *drawable_vals,
	PlugInUIVals       *ui)
{
  GtkWidget *dlg = NULL;
  GtkWidget *main_vbox = NULL;
  GtkWidget *info_box = NULL;
  GtkWidget *check_box = NULL;
  GtkWidget *label = NULL;
  GList *glist = NULL;
  cmsHPROFILE *out=NULL;
  GimpParasite *parasite;

  ui_state = ui;
  state = vals;


  parasite = gimp_image_parasite_find (image_ID, "icc-profile");

  if (parasite)
    {
      gint profile_size = 0;
      guchar *icc_profile;
      ui->cm_have_embedded = TRUE;
      profile_size = gimp_parasite_data_size (parasite);
      icc_profile =(guchar *) gimp_parasite_data (parasite);
      out = cmsOpenProfileFromMem (icc_profile, profile_size);
      gimp_parasite_free (parasite);
    }
  else
    {
      ui->cm_have_embedded = FALSE;
    }

  gimp_ui_init (PLUGIN_NAME, TRUE);


  dlg = gimp_dialog_new (_("GIMP Color Manager"), PLUGIN_NAME,
                         NULL, 0,
			 gimp_standard_help_func, "Apply Color Transformation",
			 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			 _("Apply Color Transformation"),     GTK_RESPONSE_OK,
			 NULL);
			 
			 
  /*  Initialize tooltips  */
  gimp_help_enable_tooltips();

  /* create top level container */

  main_vbox = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (main_vbox), 6);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dlg)->vbox), main_vbox);

  /*Create the top table to hold buttons and combos*/
  top_tbl = gtk_table_new (2, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (main_vbox), top_tbl, FALSE, FALSE, 0);
  gtk_widget_show (top_tbl);

  /* create  box containing check options */

  check_box = gtk_hbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (check_box), 6);
  gtk_box_pack_start (GTK_BOX (main_vbox), check_box, FALSE, FALSE, 0);
  gtk_widget_show (check_box);
  
  /* create info box containing the info labels */

  info_box = gtk_vbox_new (FALSE, 4);
  gtk_container_set_border_width (GTK_CONTAINER (info_box), 6);
  gtk_box_pack_start (GTK_BOX (main_vbox), info_box, FALSE, FALSE, 0);
  gtk_widget_show (info_box);
  
  /* add transformation selection of input/output profile */


  label = gtk_label_new (_("Transformation:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), label, 0, 1, 6, 7);
  gtk_widget_show (label);
  
  /* Create the transformation combo*/

  transformation_combo = gtk_combo_new ();

  glist = g_list_append (glist, INPUT_NAME);
  glist = g_list_append (glist, OUTPUT_NAME);
  glist = g_list_append (glist, WORKSPACE_NAME);
  
  gtk_combo_set_popdown_strings (GTK_COMBO (transformation_combo), glist);

  /* free the glist */
  g_list_free (glist);
  glist = NULL;

  /* make the combo read only */
  
  gtk_editable_set_editable(GTK_EDITABLE (GTK_COMBO (transformation_combo)->entry), FALSE);
  
  if (state->cm_direction == INPUT_PROFILE)
    {
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (transformation_combo)->entry),
			      INPUT_NAME);
    }
  else if (state->cm_direction == OUTPUT_PROFILE)
    {
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (transformation_combo)->entry),
			      OUTPUT_NAME);
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (transformation_combo)->entry),
			     WORKSPACE_NAME);
    }

  gtk_table_attach_defaults (GTK_TABLE (top_tbl), transformation_combo,
			     1, 3, 6, 7);
  gtk_widget_show (transformation_combo);
  

  /* add the elements to top_tlb */

  label = gtk_label_new (_("Rendering Intent:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), label, 0, 1, 1, 2);
  gtk_widget_show (label);

  rendering_intent_combo = gtk_combo_new ();

  glist = g_list_append (glist, PERCEPTUAL_NAME);
  glist = g_list_append (glist, REL_COLORIMETRIC_NAME);
  glist = g_list_append (glist, SATURATION_NAME);
  glist = g_list_append (glist, ABS_COLORIMETRIC_NAME);

  gtk_combo_set_popdown_strings (GTK_COMBO (rendering_intent_combo), glist);

  /* free the glist */
  g_list_free (glist);

  /* make the combo read only */
  
  gtk_editable_set_editable(GTK_EDITABLE (GTK_COMBO (rendering_intent_combo)->entry), FALSE);

  gtk_table_attach_defaults (GTK_TABLE (top_tbl), rendering_intent_combo,
			     1, 3, 1, 2);
  gtk_widget_show (rendering_intent_combo);


  label = gtk_label_new (_("Input Profile:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), label, 0, 1, 2, 3);
  gtk_widget_show (label);

  in_profile_combo = gtk_combo_new ();

  /* make the combo read only */
  
  gtk_editable_set_editable(GTK_EDITABLE (GTK_COMBO (in_profile_combo)->entry), FALSE);

  gtk_table_attach_defaults (GTK_TABLE (top_tbl), in_profile_combo, 1, 3,
			     2, 3);
  gtk_widget_show (in_profile_combo);

  label = gtk_label_new (_("Output Profile:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), label, 0, 1, 3, 4);
  gtk_widget_show (label);

  out_profile_combo = gtk_combo_new ();

  glist = NULL;

  /* make the combo read only */
  
  gtk_editable_set_editable(GTK_EDITABLE (GTK_COMBO (out_profile_combo)->entry), FALSE);
  
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), out_profile_combo, 1, 3,
			     3, 4);
  gtk_widget_show (out_profile_combo);

  /* --- */

  work_space_combo = gtk_combo_new ();

  /* make the combo read only */
  
  gtk_editable_set_editable(GTK_EDITABLE (GTK_COMBO (work_space_combo)->entry), FALSE);
  
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), work_space_combo,
			     1, 3, 5, 6);
  gtk_widget_show (work_space_combo);

  label = gtk_label_new (_("Default Work Space Profile:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), label, 0, 1, 5, 6);
  gtk_widget_show (label);

  /* --- */

  preview_button = gtk_check_button_new_with_label (_("Preview"));
  gtk_box_pack_start (GTK_BOX (check_box), preview_button, FALSE, FALSE, 0);


  state->cm_preview = (state->cm_direction == OUTPUT_PROFILE);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (preview_button),
				state->cm_preview);
  gtk_widget_show (preview_button);

  gamut_check_button = gtk_check_button_new_with_label (_("Gamut Check"));
  gtk_box_pack_start (GTK_BOX (check_box), gamut_check_button, FALSE, FALSE, 0);
  
  state->cm_gamut_check = (state->cm_direction == OUTPUT_PROFILE);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gamut_check_button),
				state->cm_gamut_check);
  gtk_widget_show (gamut_check_button);

  embedded_button =
    gtk_check_button_new_with_label (_("Use Embedded Profile"));
  gtk_box_pack_start (GTK_BOX (check_box), embedded_button, FALSE, FALSE, 0);
  gtk_widget_show (embedded_button);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (embedded_button),
				state->cm_use_embedded);
  gtk_widget_set_sensitive (embedded_button, ui_state->cm_have_embedded);

  
  /* Creates a table to pack together some buttons and labels*/
  
  info_tbl = gtk_table_new (2, 2, FALSE);
    
  label = gtk_label_new (_("ICC directory:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), label, 0, 1, 0, 1);
  gtk_widget_show (label);
  
  dir_lbl = gtk_label_new (_("<undefined>"));
  gtk_misc_set_alignment (GTK_MISC (dir_lbl), 0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), dir_lbl, 1, 2, 0, 1);
  gtk_widget_show (dir_lbl);
  
  label = gtk_label_new (_("Embedded profile:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), label, 0, 1, 4, 5);
  gtk_widget_show (label);
  
  if(out)
  {
     char msgstring[MAXPATHLEN];
     sprintf (msgstring," <span foreground=\"blue\">%s</span>", cmsTakeProductName(out));
     emb_lbl = gtk_label_new (msgstring);
     gtk_label_set_markup (GTK_LABEL (emb_lbl),msgstring);
     cmsCloseProfile(out);  
  }
  else
     {
     char msgstring[MAXPATHLEN];
     emb_lbl = gtk_label_new (_("<none>"));
     sprintf (msgstring," <span foreground=\"red\">no profile attached</span>"); 
     gtk_label_set_markup (GTK_LABEL (emb_lbl),msgstring);
     }
  gtk_misc_set_alignment (GTK_MISC (emb_lbl), 0, 0);
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), emb_lbl, 1, 3, 4, 5);
  gtk_widget_show (emb_lbl);
  
  gtk_box_pack_start (GTK_BOX (info_box), info_tbl, FALSE, FALSE, 0);
  gtk_widget_show(info_tbl);
  /* create the buttons in the button box */

  dir_button = gtk_button_new_with_label (_("ICC Directory"));
  gtk_table_attach_defaults (GTK_TABLE (top_tbl), dir_button, 2, 3, 0, 1);
  gtk_widget_show (dir_button);

  refresh_button = gtk_button_new_with_label (_("Refresh"));
  gtk_table_attach_defaults (GTK_TABLE (info_tbl), refresh_button, 0, 1, 0, 1);
  gtk_widget_show (refresh_button);

  if(ui->cm_have_embedded)
     embed_profile_button = gtk_button_new_with_label (ENBED_BUT_NAME2);
  else
     embed_profile_button = gtk_button_new_with_label (ENBED_BUT_NAME);
  gtk_table_attach_defaults (GTK_TABLE (info_tbl), embed_profile_button, 1, 2, 0, 1);
  gtk_widget_show (embed_profile_button);


  gtk_widget_show (GTK_DIALOG (dlg)->vbox);

  create_file_selection ();

  rescan_dir (state->cm_directory);


  /*  Show the main containers  */
  gtk_widget_show (main_vbox);
  gtk_widget_show (dlg);


  /* hide the preview checkmark if the input direction is selected */

  if (state->cm_direction == INPUT_PROFILE)
    {
      gtk_widget_set_sensitive (preview_button, FALSE);
      gtk_widget_set_sensitive (gamut_check_button, FALSE);
    }

  /* -------------------------------------------------------------------------- *
   * setup the callbacks
   * -------------------------------------------------------------------------- */


  g_signal_connect (GTK_OBJECT (GTK_COMBO (transformation_combo)->entry),
		      "changed", GTK_SIGNAL_FUNC (set_conversion_type),
		      NULL);
  
  g_signal_connect (GTK_OBJECT (GTK_COMBO (rendering_intent_combo)->entry),
		      "changed", GTK_SIGNAL_FUNC (rendering_intent_combo_cb),
		      NULL);

  g_signal_connect (GTK_OBJECT (GTK_COMBO (in_profile_combo)->entry),
		      "changed", GTK_SIGNAL_FUNC (profile_combo_cb),
		      state->cm_input_profile);

  g_signal_connect (GTK_OBJECT (GTK_COMBO (out_profile_combo)->entry),
		      "changed", GTK_SIGNAL_FUNC (profile_combo_cb),
		      state->cm_output_profile);

  g_signal_connect (GTK_OBJECT (GTK_COMBO (work_space_combo)->entry),
		      "changed", GTK_SIGNAL_FUNC (profile_combo_cb),
		      state->cm_work_space);
  
  g_signal_connect (GTK_OBJECT (preview_button), "toggled",
		      GTK_SIGNAL_FUNC (gimp_toggle_button_update),
		      &state->cm_preview);

  g_signal_connect (GTK_OBJECT (gamut_check_button), "toggled",
		      GTK_SIGNAL_FUNC (gimp_toggle_button_update),
		      &state->cm_gamut_check);

  g_signal_connect (GTK_OBJECT (dir_button), "clicked",
		      GTK_SIGNAL_FUNC (fselect_callback),
		      GTK_DIALOG (dlg)->vbox);

  g_signal_connect (GTK_OBJECT (refresh_button), "clicked",
		      GTK_SIGNAL_FUNC (refresh_callback), NULL);

  g_signal_connect (GTK_OBJECT (embed_profile_button), "clicked",
		      GTK_SIGNAL_FUNC (embed_profile_callback),
		      (void *) image_ID);

  g_signal_connect (GTK_OBJECT (embedded_button), "toggled",
		      GTK_SIGNAL_FUNC (gimp_toggle_button_update),
		      &state->cm_use_embedded);

  
  run = (gimp_dialog_run (GIMP_DIALOG (dlg)) == GTK_RESPONSE_OK);
  gtk_widget_destroy (dlg);
 
  return run;
}


/*  Private functions  */

static void
set_conversion_type()
{

  char conversion[INTENT_STR_LENGTH];
  
  strncpy (conversion,
	   gtk_entry_get_text (GTK_ENTRY
			       (GTK_COMBO (transformation_combo)->entry)),
	   INTENT_STR_LENGTH);
 
  if (strncmp (conversion,INPUT_NAME, INTENT_STR_LENGTH) == 0)
    state->cm_direction = INPUT_PROFILE;
  else if (strncmp (conversion, OUTPUT_NAME, INTENT_STR_LENGTH) == 0)
    state->cm_direction = OUTPUT_PROFILE;
  else if (strncmp (conversion,  WORKSPACE_NAME, INTENT_STR_LENGTH) == 0)
    state->cm_direction = WS_PROFILE;
  
  if (state->cm_direction == OUTPUT_PROFILE)
    {
      gtk_widget_set_sensitive (preview_button, TRUE);
      gtk_widget_set_sensitive (gamut_check_button, TRUE);

    }
  else 
    {
      /* disable the preview checkbox */
      gtk_widget_set_sensitive (preview_button, FALSE);
      gtk_widget_set_sensitive (gamut_check_button, FALSE);
    }

  rescan_dir (state->cm_directory);
}


static void
refresh_callback (GtkWidget * widget, gpointer data)
{

  /*
   * get the directory from the file selection widget and rescan the directory
   */

  rescan_dir (state->cm_directory);
}


static void
embed_profile_callback (GtkWidget * widget, gpointer data)
{
  guchar ws_profile_path[MAXPATHLEN];
  
  GimpParasite *parasite;
  
  if(ui_state->cm_have_embedded)
  {
    delete_embed_profile((gint32) data);
    gtk_button_set_label(GTK_BUTTON(embed_profile_button),ENBED_BUT_NAME);
    ui_state->cm_have_embedded = FALSE;
    char msgstring[MAXPATHLEN];
    sprintf (msgstring," <span foreground=\"red\">%s</span>", "no profile attached");
    gtk_label_set_markup (GTK_LABEL (emb_lbl),msgstring);
    return;
  }
  sprintf (ws_profile_path, "%s%c%s", state->cm_directory,G_DIR_SEPARATOR,
	   state->cm_work_space);
    
  embed_profile ((gint32) data, ws_profile_path);
  
  parasite = gimp_image_parasite_find ((gint32) data, "icc-profile");
  if (parasite)
    {
      gtk_button_set_label(GTK_BUTTON(embed_profile_button),ENBED_BUT_NAME2);
      gint profile_size = 0;
      guchar *icc_profile;
      profile_size = gimp_parasite_data_size (parasite);
      icc_profile =(guchar *) gimp_parasite_data (parasite);
      cmsHPROFILE *out = cmsOpenProfileFromMem (icc_profile, profile_size);
      gimp_parasite_free (parasite);
      ui_state->cm_have_embedded = TRUE;
      if(out)
      {
        char msgstring[MAXPATHLEN];
        sprintf (msgstring," <span foreground=\"blue\">%s</span>", cmsTakeProductName(out));
        gtk_label_set_markup (GTK_LABEL (emb_lbl),msgstring);
        cmsCloseProfile(out);  
      }
    }
  // activate the "embed profile" check box
  gtk_widget_set_sensitive (embedded_button, TRUE);
      
}



static void
create_file_selection (void)
{
  if (!file_sel)
    {
      file_sel = gtk_file_selection_new (_("Select ICC Profile Directory"));

      gtk_file_selection_set_filename (GTK_FILE_SELECTION (file_sel),
				       state->cm_directory);

      /* hide the file selection widgets, leave just the directory stuff */

      gtk_widget_hide (GTK_FILE_SELECTION (file_sel)->fileop_del_file);
      gtk_widget_hide (GTK_FILE_SELECTION (file_sel)->fileop_ren_file);
      gtk_widget_hide (GTK_FILE_SELECTION (file_sel)->fileop_c_dir);
      gtk_widget_hide (GTK_FILE_SELECTION (file_sel)->file_list->parent);

      gtk_window_set_position (GTK_WINDOW (file_sel), GTK_WIN_POS_NONE);

      g_signal_connect (GTK_OBJECT (file_sel), "destroy",
			  GTK_SIGNAL_FUNC (gtk_widget_destroyed), &file_sel);

      g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (file_sel)->ok_button),
			  "clicked", GTK_SIGNAL_FUNC (file_ok_sel),
			  GTK_OBJECT (file_sel));

      g_signal_connect (GTK_OBJECT
			  (GTK_FILE_SELECTION (file_sel)->cancel_button),
			  "clicked", GTK_SIGNAL_FUNC (file_cancel_sel),
			  GTK_OBJECT (file_sel));

      gtk_widget_hide (file_sel);

    }
}


static void
fselect_callback (GtkWidget * widget, gpointer data)
{
  load_file_selection ();
}


static void
rendering_intent_combo_cb (GtkWidget * widget, gpointer data)
{
  /*
   * get the rendering intent value from the combo box and
   * store the value to rendering_intent / the string is
   * converted in get_rendering_intent () to an int.
   */

  state->cm_rendering_intent = get_rendering_intent ();

}

static void
profile_combo_cb (GtkWidget * widget, gpointer data)
{
  strncpy ((guchar *) data,
	   gtk_entry_get_text (GTK_ENTRY (widget)), MAXPATHLEN);

  if ((data == state->cm_output_profile)
      && state->cm_direction == OUTPUT_PROFILE)
    {
      gtk_widget_set_sensitive (gamut_check_button,
				check_for_gamut_tag ((guchar *) data));
    }
}

static void
load_file_selection (void)
{
  /* Load a single object */
  if (!GTK_WIDGET_VISIBLE (file_sel))
    gtk_widget_show (file_sel);
}


static void
file_ok_sel (GtkWidget * w, GtkFileSelection * fs)
{
  /* get the selected file name */
  strncpy (state->cm_directory,
	   gtk_file_selection_get_filename (GTK_FILE_SELECTION (fs)),
	   MAXPATHLEN);

  rescan_dir (state->cm_directory);

  gtk_widget_hide (GTK_WIDGET (fs));

}


static void
file_cancel_sel (GtkWidget * w, GtkFileSelection * fs)
{
  gtk_widget_hide (GTK_WIDGET (fs));
}


static void
rescan_dir (guchar * directory)
{
  struct stat statbuf;
  DIR *dir;
  struct dirent *entry;
  GList *output_list = NULL;
  GList *input_list = NULL;
  GList *display_list = NULL;
  guchar *tmp_input;
  guchar *tmp_output;
  guchar *tmp_workspace;
  char msgstring[MAXPATHLEN];
  
  cmsErrorAction(LCMS_ERROR_IGNORE);
  
  sprintf (msgstring, "  %s  ", state->cm_directory);
  gtk_label_set_text((GtkLabel *)dir_lbl,msgstring);  
     
  if ((lstat (directory, &statbuf)) != 0)
    {
      cm_error ("%s %s\n", _("Can not stat directory "), directory);
      strcpy (state->cm_directory, "/");
      if ((lstat (state->cm_directory, &statbuf)) != 0)
	return;
    }


  /* store the current selections in temp variables so that we 
     can restore them after updating the combo boxes */
  tmp_input = strdup (state->cm_input_profile);
  tmp_output = strdup (state->cm_output_profile);
  tmp_workspace =  strdup (state->cm_work_space);
  
  if (S_ISDIR (statbuf.st_mode))
    {
      dir = opendir (directory);

      if (!dir)
	{
	  cm_error ("%s %s\n", _("Can not open directory"), directory);
	  free (tmp_input);
	  free (tmp_output);
	  return;
	}

      while ((entry = readdir (dir)))
	{
	  char newfile[MAXPATHLEN];
	  char *name = entry->d_name;
	  cmsHPROFILE profile;

	  if ((strcmp (name, "..") == 0) || (strcmp (name, ".") == 0))
	    {
	      continue;
	    }

	  sprintf (newfile, "%s/%s", directory, name);

	  //check if file exist
	  
	  /*
	   * Open the file and try to read it using the lcms library
	   * - if this fails it was not a valid profile.
	   */
	  profile = cmsOpenProfileFromFile (newfile, "r");

	  if (profile != NULL)
	    {
	      char *strPtr;

	      if ((cmsGetDeviceClass (profile) == icSigOutputClass) &&
		  (cmsGetColorSpace (profile) == icSigRgbData))
		{
		  strPtr = strdup (name);
		  output_list = g_list_append (output_list, strPtr);
		}
	      else if ((cmsGetDeviceClass (profile) == icSigInputClass) &&
		       (cmsGetColorSpace (profile) == icSigRgbData))
		{
		  strPtr = strdup (name);
		  input_list = g_list_append (input_list, strPtr);
		}
	      else if ((cmsGetDeviceClass (profile) == icSigDisplayClass) &&
		       (cmsGetColorSpace (profile) == icSigRgbData))
		{
		  strPtr = strdup (name);
		  display_list = g_list_append (display_list, strPtr);
		}
	    }
	  else
	    {
	      /* not able to open profile - just ignore it for now */
	    }
	  cmsCloseProfile (profile);

	}

      // make sure that we have at least one element in the list(s)
      if (input_list == NULL)
	input_list = g_list_append (input_list, strdup (""));
      if (output_list == NULL)
	output_list = g_list_append (output_list, strdup (""));
      if (display_list == NULL)
	display_list = g_list_append (display_list, strdup (""));

      gtk_combo_set_popdown_strings (GTK_COMBO (in_profile_combo),
				     input_list);
      gtk_combo_set_popdown_strings (GTK_COMBO (out_profile_combo),
				     output_list);
      gtk_combo_set_popdown_strings (GTK_COMBO (work_space_combo),
				     display_list);

      /* set the current selection in the lists */

      if (g_list_find_custom (input_list, tmp_input, (GCompareFunc) strcmp) !=
	  NULL)
	{
	  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (in_profile_combo)->entry),
			      tmp_input);
	}
      else
	{
	  // copy the current selection to state->cm_input_profile
	  strncpy (state->cm_input_profile,
		   gtk_entry_get_text (GTK_ENTRY
				       (GTK_COMBO (in_profile_combo)->entry)),
		   MAXPATHLEN);
	}
      if (g_list_find_custom (output_list, tmp_output, (GCompareFunc) strcmp)
	  != NULL)
	{
	  gtk_entry_set_text (GTK_ENTRY
			      (GTK_COMBO (out_profile_combo)->entry),
			      tmp_output);
	}
      else
	{
	  // copy the current selection to state->cm_output_profile
	  strncpy (state->cm_output_profile,
		   gtk_entry_get_text (GTK_ENTRY
				       (GTK_COMBO (out_profile_combo)->
					entry)), MAXPATHLEN);
	}
      if (g_list_find_custom (display_list, tmp_workspace, (GCompareFunc) strcmp)
	  != NULL)
	{
	  gtk_entry_set_text (GTK_ENTRY
			      (GTK_COMBO (work_space_combo)->entry),
			      tmp_workspace);
	}
      else
	{
	  // copy the current selection to state->cm_output_profile
	  strncpy (state->cm_work_space,
		   gtk_entry_get_text (GTK_ENTRY
				       (GTK_COMBO (work_space_combo)->
					entry)), MAXPATHLEN);
	}

      gtk_widget_show (in_profile_combo);
      gtk_widget_show (out_profile_combo);
      gtk_widget_show (work_space_combo);
      closedir (dir);

      /* free the input_list elements and the list */
      g_list_foreach (input_list, (GFunc) g_free, NULL);
      g_list_free (input_list);

      /* free the output_list elements and the list */
      g_list_foreach (output_list, (GFunc) g_free, NULL);
      g_list_free (output_list);
      
      /* free the display_list elements and the list */
      g_list_foreach (display_list, (GFunc) g_free, NULL);
      g_list_free (display_list);
    }
  else
    {
      cm_error ("%s %s\n", _("Not a directory: "), directory);
    }
  free (tmp_input);
  free (tmp_output);
}


static void
cm_error (const char *fmt, ...)
{
  va_list ap;
  guchar str[MAXPATHLEN];

  va_start (ap, fmt);
  vsnprintf (str, MAXPATHLEN, fmt, ap);
  va_end (ap);

  gimp_message (str);
}


static int
get_rendering_intent (void)
{
  char intent[INTENT_STR_LENGTH];
  int result;

  strncpy (intent,
	   gtk_entry_get_text (GTK_ENTRY
			       (GTK_COMBO (rendering_intent_combo)->entry)),
	   INTENT_STR_LENGTH);

  if (strncmp (intent, PERCEPTUAL_NAME, INTENT_STR_LENGTH) == 0)
    result = INTENT_PERCEPTUAL;
  else if (strncmp (intent, REL_COLORIMETRIC_NAME, INTENT_STR_LENGTH) == 0)
    result = INTENT_RELATIVE_COLORIMETRIC;
  else if (strncmp (intent, SATURATION_NAME, INTENT_STR_LENGTH) == 0)
    result = INTENT_SATURATION;
  else if (strncmp (intent, ABS_COLORIMETRIC_NAME, INTENT_STR_LENGTH) == 0)
    result = INTENT_ABSOLUTE_COLORIMETRIC;
  else
    result = INTENT_PERCEPTUAL;

  return result;
}

static gboolean
check_for_gamut_tag (guchar * theProfileName)
{
  cmsHPROFILE profile;
  int retVal = FALSE;
  guchar fileName[MAXPATHLEN];
  sprintf (fileName, "%s%c%s", state->cm_directory,G_DIR_SEPARATOR, theProfileName);

  profile = cmsOpenProfileFromFile (fileName, "r");
  if (profile != NULL)
    {
      retVal = cmsIsTag (profile, icSigGamutTag);
    }
  cmsCloseProfile (profile);
  return retVal;
}
