#ifndef __ARGP_LOCAL_H
#define __ARGP_LOCAL_H

//
// This is fast and dirty argp_parse() replacement for non-glibc systems.
// It implements only features used by fix-mime-charset and cannot be used as
// all-purpose argp replacement.
//
// Copyright (c) 2003 by Dmitry E. Melamud <balu@migdal.ru>
//

#include <getopt.h>
#include <string.h>
#include <iostream>

#define __need_error_t
#include <errno.h>

#ifndef __error_t_defined
typedef int error_t;
#define __error_t_defined
#endif

#define ARGP_DISPLAY_WIDTH 78
#define ARGP_OPTION_WIDTH 29
#define ARGP_USAGE_SHIFT 12

struct argp_state
{
int argc;
char **argv;
};

#define ARGP_KEY_ARG 0

struct argp_option
{
const char *name;
int key;
const char *arg;
int flags;
const char *doc;
};

typedef error_t (*argp_parser_t)(int key,char *arg,argp_state *state);

struct argp
{
const argp_option *options;
argp_parser_t parser;
const char *args_doc;
const char *doc;
};

extern const char *argp_program_version;
extern const char *argp_program_bug_address;

const argp_option _argp_default_options[]=
     {
      {"help",'h',0,0,"Give this help list"},
      {"usage",256,0,0,"Give a short usage message"},
      {"version",'V',0,0,"Print program version"},
      {0}
     };

string _argp_progname(const char *arg)
{
char *p=strrchr(arg,'/');
return p==NULL ? arg : p+1;
}

string _argp_cut(string &text)
{
string part;
if(text.length()<=ARGP_DISPLAY_WIDTH)
  {
  part=text;
  text="";
  }
else
  {
  int pos=text.find_last_of("\t ",ARGP_DISPLAY_WIDTH);
  if(pos==string::npos)
    pos=ARGP_DISPLAY_WIDTH;
  part=text.substr(0,pos+1);
  text=text.substr(part.length());
  }
return part;
}

void _argp_help(const string &name,const argp *_argp)
{
cout<<"Usage: "<<name<<" [OPTION...] "<<_argp->args_doc<<endl;
string doc=_argp->doc;
while(doc.length()!=0)
     cout<<_argp_cut(doc)<<endl;
cout<<endl;
for(const argp_option *p=_argp->options;p!=_argp_default_options+3;p++)
   {
   if(!p->name && !p->key)
     p=_argp_default_options;
   string info="  ";
   if(p->key>' ' && p->key<='z')
     info+=string()+'-'+(char)p->key+", ";
   else
     info+="    ";
   info+=string()+"--"+p->name;
   if(p->arg)
     info+=string()+'='+p->arg;
   if(info.length()>ARGP_OPTION_WIDTH-1)
     {
     cout<<info<<endl;
     info=string(ARGP_OPTION_WIDTH,' ');
     }
   else
     info+=string(ARGP_OPTION_WIDTH-info.length(),' ');
   info+=p->doc;
   while(info.length()!=ARGP_OPTION_WIDTH)
        {
        cout<<_argp_cut(info)<<endl;
        info=string(ARGP_OPTION_WIDTH,' ')+info;
        }
   }
cout<<endl;
cout<<"Mandatory or optional arguments to long options are also mandatory"
      " or optional"<<endl;
cout<<"for any corresponding short options."<<endl;
cout<<endl;
cout<<"Report bugs to "<<argp_program_bug_address<<'.'<<endl;
}

void _argp_version(const string &name,const argp *_argp)
{
cout<<argp_program_version<<endl;
}

void _argp_usage(const string &name,const argp *_argp)
{
vector<string> parts;
parts.push_back("Usage: "+name);
for(int n=0;n<3;n++)
   {
   string shorts;
   for(const argp_option *p=_argp->options;p!=_argp_default_options+3;p++)
      {
      if(!p->name && !p->key)
        p=_argp_default_options;
      switch(n)
            {
            case 0:
                 if(p->key>' ' && p->key<='z' && p->arg==NULL)
                   shorts+=p->key;
                 break;
            case 1:
                 if(p->key>' ' && p->key<='z' && p->arg!=NULL)
                   parts.push_back(string()+"[-"+(char)p->key+' '+p->arg+']');
                 break;
            case 2:
                 parts.push_back(string()+"[--"+p->name+
                                 (p->arg ? string()+'='+p->arg : "")+']');
                 break;
            }
      }
   if(n==0)
     parts.push_back("[-"+shorts+']');
   }
parts.push_back(_argp->args_doc);
string info;
for(vector<string>::iterator i=parts.begin();i!=parts.end();i++)
   {
   if(info.length()+i->length()+1<=ARGP_DISPLAY_WIDTH)
     {
     if(info.length()!=0)
       info+=' ';
     info+=*i;
     }
   else
     {
     cout<<info<<endl;
     info=string(ARGP_USAGE_SHIFT,' ')+*i;
     }
   }
cout<<info<<endl;
}

void argp_parse(const argp *_argp,int _argc,char **_argv,unsigned,int *,void *)
{
int argpc=0;
const argp_option *p;
for(p=_argp->options;p->name && p->key;p++,argpc++);
option longopts[argpc+4];
string shortopts;
option *q=longopts;
for(p=_argp->options;p!=_argp_default_options+3;p++)
   {
   if(!p->name && !p->key)
     p=_argp_default_options;
   memset(q,0,sizeof(option));
   q->name=p->name;
   q->has_arg=p->arg==NULL ? no_argument : required_argument;
   if(p->key>' ' && p->key<='z')
     {
     shortopts+=p->key;
     if(p->arg!=NULL)
       shortopts+=':';
     }
   q->val=p->key;
   q++;
   }
memset(q,0,sizeof(option));
shortopts+="hV";

string name=_argp_progname(_argv[0]);
argp_state state={_argc,_argv};
int val;
string msg;
while((val=getopt_long(_argc,_argv,shortopts.c_str(),longopts,NULL))!=-1)
     switch(val)
           {
           case '?':
                msg="Try `"+name+" --help' or `"+name+
                    " --usage' for more information.";
                while(msg.length()!=0)
                     cerr<<_argp_cut(msg)<<endl;
                exit(1);
           case 'h':
                _argp_help(name,_argp);
                exit(0);
           case 'V':
                _argp_version(name,_argp);
                exit(0);
           case 256:
                _argp_usage(name,_argp);
                exit(0);
           default:
                (*_argp->parser)(val,optarg,&state);
           }
}

#endif /* __ARGP_LOCAL_H */
