#include "prjlibs-c/standards.h"
#include <unistd.h>
#include <fcntl.h>
#include <limits.h>
#include <errno.h>

#include "prjlibs-c/types.h"
#include "fdtools.h"

type_status fd_shuffle(size_t nfds, type_fd* current, type_fd const* wanted) {
  size_t i, j, movable_squatter, squatted;
  type_fd cur, moved;
  if (nfds==0) return 0;
  for (;;) { start:
    /* squatted is the index of a descriptor that is not yet where it wants
       to be, but can't be moved immediately because some other file is
       already open on that descriptor. */
    squatted=-1;
    /* movable_squatter is the index of a file which is allowed to end up on
       any descriptor (wanted[moveable_squatter]<0) and which is occupying
       the descriptor that some other file wants. */
    movable_squatter=-1;
    for (i=0; i!=nfds; ++i) {
      if (wanted[i]<0) continue; /* We don't care where this one ends up. */
      if (wanted[i]==current[i]) continue; /* This one is where we want it. */
      /* current[i] needs to be dup()ed to wanted[i] */
      for (j=0;; ++j) {
        if (j==nfds) {
          /* We checked every current[j] and didn't find any that were
             squatting on this wanted[i]. */
          cur=current[i];
          if (dup2(cur, wanted[i])<0) return -1;
          current[i]=wanted[i];
          while (close(cur)!=0 && errno!=EINTR);
          goto start;
        }
        if (current[j]==wanted[i]) {
          /* j is currently occupying the descriptor that i wants.  j would
             have to be dup()ed away before i could be dup()ed here. */
          if (movable_squatter==(size_t)-1) {
            /* We haven't yet found a squatter that's easily movable (i.e., a
               don't-care. */
            if (wanted[j]<0) movable_squatter=j; /* Now we have. */
            squatted=i;
          }
          break;
        }
      }
    }
    if (squatted==(size_t)-1) return 0; /* Everything is in its place. */
    /* There is more dup()ing to do, but moving anything directly to its
       destination would replace another descriptor.  We have to move
       something out of the way before we can proceed.  If there is a
       don't-care in the way, we prefer to move that, since it won't have to
       be moved again.  Otherwise, all that's left are cycles: 1 wants to be
       2, 2 wants to be 3, 3 wants to be 1.  Every squatted is also a
       squatter.  Move one of those so the others can be put into place, and
       eventually the one we move now will have its destination become
       available. */
    if (movable_squatter!=(size_t)-1) i=movable_squatter;
    else i=squatted;
    cur=current[i];
    moved=dup(cur);
    if (moved<0) return -1;
    current[i]=moved;
    while (close(cur)!=0 && errno==EINTR);
  }
}
