/*****************************************************************
 * fbgamma.c: FBM Release 1.2 07-Apr-93 Michael Mauldin
 *
 * Copyright (C) 1989-1993 by Michael Mauldin.  Permission is granted
 * to use this file in whole or in part for any purpose, educational,
 * recreational or commercial, provided that this copyright notice
 * is retained unchanged.  This software is available to all free of
 * charge by anonymous FTP and in the UUNET archives.
 *
 * fbgamma.c: Apply a gamma transform to a colormapped image
 *
 * USAGE
 *     % fbgamma [ -<type> ] [ amount ] < image1 > image2
 *
 * EDITLOG
 *     LastEditDate = Mon Jun 25 00:02:30 1990 - Michael Mauldin
 *     LastFileName = /usr2/mlm/src/misc/fbm/fbgamma.c
 *
 * HISTORY
 * 07-Apr-93  Michael Mauldin (mlm) at Carnegie-Mellon University
 *	Added -J switch
 *
 * 25-Jun-90  Michael Mauldin (mlm@cs.cmu.edu) Carnegie Mellon
 *	Package for Release 1.0
 *
 * 05-Apr-89  Michael Mauldin (mlm) at Carnegie-Mellon University
 *     Created.
 *****************************************************************/

# include <stdio.h>
# include <math.h>
# include "fbm.h"

# define USAGE \
  "Usage: fbgamma [ -<type> ] [ amount ] < image > image"

#ifndef lint
static char *fbmid =
"$FBM fbgamma.c <1.2> 07-Apr-93 (C) 1989-1993 by Michael Mauldin, source \
code available free from MLM@CS.CMU.EDU and from UUNET archives$";
#endif

int debug = 0;
double expon = 1.0;

main (argc, argv)
char *argv[];
{ FBM image;
  int outtype = DEF_8BIT;
  double amount = 2.2, atof ();
  char *fname = NULL;

  /* Get the options */
  while (--argc > 0 && (*++argv)[0] == '-')
  { while (*++(*argv))
    { switch (**argv)
      {
	case 'd':	debug++; break;
	case 'A':	outtype = FMT_ATK; break;
	case 'B':	outtype = FMT_FACE; break;
	case 'F':	outtype = FMT_FBM; break;
	case 'G':	outtype = FMT_GIF; break;
	case 'I':	outtype = FMT_IFF; break;
	case 'J':	outtype = FMT_JPEG; break;
	case 'L':	outtype = FMT_LEAF; break;
	case 'M':	outtype = FMT_MCP; break;
	case 'P':	outtype = FMT_PBM; break;
	case 'R':	outtype = FMT_RLE; break;
	case 'S':	outtype = FMT_SUN; break;
	case 'T':	outtype = FMT_TIFF; break;
	case 'X':	outtype = FMT_X11; break;
	case 'Z':	outtype = FMT_PCX; break;
	default:        fprintf (stderr, "%s\n", USAGE);
                        exit (1);
      }
    }
  }

  if (argc > 0)
  { amount = atof (argv[0]); }

  if (argc > 1)
  { fname = argv[1]; }

  /* Clear the memory pointers so alloc_fbm won't be confused */
  image.cm  = image.bm  = (unsigned char *) NULL;

  /* Read the image */
  if (read_bitmap (&image, fname))
  { int h, w, k;

    /* Determine output height & width (oh*ow <= size) */
    h = image.hdr.rows;
    w = image.hdr.cols;
    k = image.hdr.planes;

    fprintf (stderr,
	     "Gamma correct \"%s\", gamma %1.3lf [%dx%dx%d]\n",
	     image.hdr.title[0] ? image.hdr.title : "(untitled)",
	     amount, w, h, k);

    if (gamma_fbm (&image, amount) &&
        write_bitmap (&image, stdout, outtype))
    { exit (0); }
  }

  exit (1);
}

gamma_fbm (image, amount)
FBM *image;
double amount;
{ double clr;
  register int i, w, h, k;
  register unsigned char *bmp, *btail;
  int cmap[256];

  /* Determine output height & width (oh*ow <= size) */
  h = image->hdr.rows;
  w = image->hdr.cols;
  k = image->hdr.planes;

  expon = 1.0 / amount;

  if (image->hdr.clrlen > 0)
  {
    for (i=0; i<image->hdr.clrlen; i++)
    {
      if (debug) fprintf (stderr, "value %3d => ", image->cm[i]);
      image->cm[i] = gamma_clr (image->cm[i]);
      if (debug) fprintf (stderr, "%3d\n", image->cm[i]);
    }
  }
  else
  {
    for (i=0; i<256; i++)
    { cmap[i] = gamma_clr (i);

      if (debug) fprintf (stderr, "value %3d => %3d\n", i, cmap[i]);
    }
    
    bmp = image->bm;
    btail = bmp + image->hdr.plnlen * image->hdr.planes;
    
    while (bmp < btail)
    { *bmp = cmap[*bmp]; bmp++; }
  }

  
  return (1);
}

int gamma_clr (value)
int value;
{ double clr, floor ();
  int newclr;

  /* First compute new color */
  clr = 255.0 * pow ((value / 255.0), expon);

  /* Round to nearest integer */
  newclr = (int) floor (clr + 0.5);
  
  if (newclr < 0) return (0);
  if (newclr > 255) return (255);
  
  return (newclr);
}
