/*****************************************************************************
 *  ENTROPY - emerging network to reduce orwellian potency yield
 *
 *  Copyright (C) 2002 Juergen Buchmueller <pullmoll@stop1984.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *	$Id: crypt8.c,v 1.4 2005/07/12 23:12:29 pullmoll Exp $
 *****************************************************************************/
#include "crypt.h"
#if	CRYPT8_TEST
#define	xmalloc(size) malloc(size)
#define	xcalloc(num,size) calloc(num,size)
#define	xfree(ptr) if (NULL != ptr) { free(ptr); ptr = NULL; }
#else
#include "memalloc.h"
#endif
#include "logger.h"

#define	OTPSIZE		32768
#define	HASHSIZE	(OTPSIZE*2)

/*****************************************************************************
 *	Incremental crypto - The idea
 *
 *	Each plain text message of X bytes is accompanied by another X bytes of
 *	pseudo random data. The 2 * X bytes are then encrypted using up 2 * X
 *	bytes from the key buffer. Alice sends this 2*X ciphertext to Bob.
 *	Bob decrypts the X bytes plain text as well as the X bytes pseudo random
 *	data from alice.
 *	Both, Alice and Bob, expand the X bytes of pseudo random data to twice
 *	the number of bytes again, using a LFSR that is initialized with the
 *	previous key bytes and the transmitted pseudo random bytes.
 *	The generated 2*X from 1*X bytes are then used to fill up the key buffer.
 *
 *	This implementation interleaves a plain text of X characters with X bytes
 *	from a pseudo random source: rnd_get(). Each two byte packet is then
 *	XORed with the next two bytes from the key buffer. A 32 bit accumulator
 *	is initialized from the previous key and new PRNG bytes as follows:
 *	bits  0- 7 = old key[0]
 *	bits  8-15 = new PRNG value
 *	The accumulator is then shifted 16 times through a LFSR (linear feedback
 *	shift register) that implements the maximum primitve polynomial for 16 bit:
 *	x^16 + x^5 + x^3 + x^2 + 1
 *	The resulting 16 bits are then used to replace the previous key[0]
 *	and key[1] bytes.
 *****************************************************************************/

#define	CRYPT8_MAGIC	(('I'<<0)|('N'<<8)|('C'<<16)|('R'<<24))
#define	KEYSIZE			65536

typedef struct crypt8_s {
	int magic;
	size_t size;			/* allocated size of the key */
	size_t offs;			/* current offset */
	uint8_t key[KEYSIZE];	/* the key */
}	crypt8_t;

/*
 * maximum primitive polynomial for 16 bits: x^16 + x^5 + x^3 + x^2 + 1
 * count the number of bits in the 16 bit word accu & 0x80017
 * shift in a 1 bit for even bit count, 0 bit for odd bit count
 */
static inline uint32_t crypt8_poly(uint32_t accu) {
	uint32_t n = accu & ((1<<15)|(1<<4)|(1<<2)|(1<<1)|1);
	accu <<= 1;
	n = (n & 0x55555555) + ((n >> 1) & 0x55555555);
	n = (n & 0x33333333) + ((n >> 2) & 0x33333333);
	n = (n + (n >> 4)) & 0x0f0f0f0f;
	n = (n + (n >> 8));
	n = (n + (n >> 16));
	accu ^= n & 1;
	return accu;
}

int crypt8_crypto(void)
{
	FUN("crypt8_crypto");

	/* no initialization required */

	return 0;
}

/***********************************************************************
 *  encrypt_msg()
 *  Encrypt a 'plaintext' of length 'insize' into a 'ciphertext'
 *	The buffer at ciphertext can hold at most '*outsize' bytes
 ***********************************************************************/
int crypt8_encrypt_msg(void *ptr, void *ciphertext, void *plaintext,
	size_t *outsize, size_t insize)
{
	crypt8_t *c = (crypt8_t *)ptr;
	uint8_t *ct = (uint8_t *)ciphertext;
	uint8_t *pt = (uint8_t *)plaintext;
	uint8_t *pad;
	size_t i, j, offs, esize;
	uint32_t accu;
	FUN("crypt8_encrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT8_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %#x (expected %#x)",
			c->magic, CRYPT8_MAGIC));
		errno = EINVAL;
		return -1;
	}

	if (insize > KEYSIZE/2) {
		LOGS(L_CRYPTO,L_ERROR,("insize is too big (have %#x, max %#x)",
			(unsigned)insize, (unsigned)(KEYSIZE/2)));
		errno = ENOMEM;
		return -1;
	}
	/* encrypted size is twice the insize */
	esize = 2 * insize;
	if (esize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %#x, want %#x)",
			(unsigned)*outsize, (unsigned)esize));
		errno = ENOMEM;
		return -1;
	}

	pad = xmalloc(insize);
	if (0 != rnd_get(pad, insize)) {
		LOGS(L_CRYPTO,L_ERROR,("rnd_get(%p,%#x) failed (%s)",
			pad, (unsigned)insize, strerror(errno)));
		xfree(pad);
		errno = EINVAL;
		return -1;
	}

	for (i = 0, offs = c->offs; i < insize; i++) {
		register uint8_t l, h;

		l = c->key[offs + 0];
		h = c->key[offs + 1];
		*ct++ = *pt++ ^ l;
		*ct++ = pad[i] ^ h;
		/* rehash the l and pad[i] values into 16 bits */
		accu = l | (pad[i] << 8);
		for (j = 0; j < 16; j++)
			accu = crypt8_poly(accu);
		/* replace key bytes */
		c->key[offs + 0] = accu % 256;
		c->key[offs + 1] = accu / 256;
		offs = (offs + 2) % KEYSIZE;
	}
	c->offs = offs;

	*outsize = esize;
	xfree(pad);
	return 0;
}

/***********************************************************************
 *  decrypt_msg()
 *  Decrypt a 'ciphertext' of the length 'insize' into a 'plaintext'
 *	The buffer at ciphertext can hold at most '*outsize' bytes
 ***********************************************************************/
int crypt8_decrypt_msg(void *ptr, void *plaintext, void *ciphertext,
	size_t *outsize, size_t insize)
{
	crypt8_t *c = (crypt8_t *)ptr;
	uint8_t *pt = (uint8_t *)plaintext;
	uint8_t *ct = (uint8_t *)ciphertext;
	uint8_t pad;
	size_t i, j, offs, dsize;
	uint32_t accu;
	FUN("crypt8_decrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT8_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %#x (expected %#x)",
			c->magic, CRYPT8_MAGIC));
		errno = EINVAL;
		return -1;
	}
	if (insize & 1) {
		LOGS(L_CRYPTO,L_ERROR,("insize is odd (%#x)",
			(unsigned)insize));
		errno = EINVAL;
		return -1;
	}
	dsize = insize / 2;
	if (dsize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %#x, want %#x)",
			(unsigned)*outsize, (unsigned)dsize));
		errno = ENOMEM;
		return -1;
	}

	for (i = 0, offs = c->offs; i < insize; i += 2) {
		register uint8_t l, h;

		l = c->key[offs + 0];
		h = c->key[offs + 1];
		*pt++ = *ct++ ^ l;
		pad  = *ct++ ^ h;
		/* rehash the l, h and pad values into 16 bits */
		accu = l | (pad << 8);
		for (j = 0; j < 16; j++)
			accu = crypt8_poly(accu);
		/* replace key bytes */
		c->key[offs + 0] = accu % 256;
		c->key[offs + 1] = accu / 256;
		offs = (offs + 2) % KEYSIZE;
	}
	c->offs = offs;

	/* FIXME: try with dsize derived from pointer */
	dsize = (size_t)(pt - (uint8_t *)plaintext);

	/* FIXME: zero out the remaining buffer */
	memset(pt, 0, *outsize - dsize);

	*outsize = dsize;
	return 0;
}

int crypt8_init(void **pptr, int init, void *data, size_t size)
{
	crypt8_t *c;
	size_t offs;
	FUN("crypt8_init");

	if (NULL == pptr) {
		LOGS(L_CRYPTO,L_ERROR,("pptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (0 != init) {
		LOGS(L_CRYPTO,L_ERROR,("init is not 0\n"));
		errno = EINVAL;
		return -1;
	}
	if (size < 32) {
		LOGS(L_CRYPTO,L_ERROR,("key size too small (%#x)\n",
			(unsigned)size));
		errno = EINVAL;
		return -1;
	}

	c = (crypt8_t *)xcalloc(sizeof(crypt8_t), 1);
	*pptr = c;

	if (size > KEYSIZE) {
		size = KEYSIZE;
	}

	memcpy(c->key, data, size);

	/* fill up the key space if size is less than KEYSIZE */
	if (size < KEYSIZE) {
		uint32_t accu;
		for (offs = 0, accu = 0; offs < size; offs++)
			accu += (uint32_t)(c->key[size] >> 4) << (c->key[size] & 0x0f);
		for (/* */; size < KEYSIZE; size++) {
			accu = crypt8_poly(accu);
			accu = crypt8_poly(accu);
			accu = crypt8_poly(accu);
			accu = crypt8_poly(accu);
			accu = crypt8_poly(accu);
			accu = crypt8_poly(accu);
			accu = crypt8_poly(accu);
			accu = crypt8_poly(accu);
			c->key[size] = (uint8_t)accu;
		}
	}
	c->size = size;
	c->magic = CRYPT8_MAGIC;

	return 0;
}

int crypt8_exit(void *ptr)
{
	crypt8_t *c = (crypt8_t *)ptr;
	FUN("crypt8_exit");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT8_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %#x (expected %#x)",
			c->magic, CRYPT8_MAGIC));
		errno = EINVAL;
		return -1;
	}
	xfree(c);

	return 0;
}

#if	CRYPT8_TEST
configuration_t *g_conf = NULL;
#define	SIZE	1024
FILE *rnd;

int pm_vsnprintf(char *dst, size_t size, const char *fmt, va_list ap)
{
	return vsnprintf(dst, size, fmt, ap);
}

int pm_snprintf(char *dst, size_t size, const char *fmt, ...)
{
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = vsnprintf(dst, size, fmt, ap);
	va_end(ap);
	return len;
}

int rnd_get(void *buff, size_t size)
{
	size_t done;
	if (NULL == rnd) {
		errno = EIO;
		return -1;
	}
	done = fread(buff, 1, size, rnd);
	if (done != size)
		return -1;
	return 0;
}

int main(int argc, char **argv)
{
	char rndname[MAXPATHLEN] = "/dev/urandom";
	char outname[MAXPATHLEN] = "crypt8.bin";
	FILE *out;
	void *ctx;
	int mode = 0;
	size_t outsize = SIZE;
	size_t insize = SIZE;
	size_t size;
	size_t count = 1;
	size_t n;
	size_t initial_size;
	unsigned char *initial;
	unsigned char *plaintext;
	unsigned char *ciphertext;
	unsigned char *deciphered;
	int i;
	int rc;

	for (i = 1; i < argc; i++) {
		if (0 == strcmp(argv[i], "--diehard")) {
			insize = 0x1000;
			outsize = 0x2000;
			count = 0x1000;
		} else if (0 == strncmp(argv[i], "-r", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-r")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-o", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-o")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-m", 2) && strlen(argv[i]) > 2) {
			mode = strtol(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-m")) {
			mode = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-s", 2) && strlen(argv[i]) > 2) {
			insize = strtoul(&argv[i][2], NULL, 0);
			outsize = 2 * insize;
		} else if (0 == strcmp(argv[i], "-s")) {
			insize = strtoul(argv[++i], NULL, 0);
			outsize = 2 * insize;
		} else if (0 == strncmp(argv[i], "-c", 2) && strlen(argv[i]) > 2) {
			count = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-c")) {
			count = strtoul(argv[++i], NULL, 0);
		}
	}

	initial_size = 256;

	printf("open random source: %s\n", rndname);
	rnd = fopen(rndname, "rb");
	if (NULL == rnd) {
		perror(rndname);
		exit(1);
	}

	initial = xcalloc(initial_size, sizeof(uint8_t));
	if (initial_size != fread(initial, 1, initial_size, rnd)) {
		perror("PRNG data");
		exit(2);
	}

	printf("create output file: %s\n", outname);
	out = fopen(outname, "wb");
	if (NULL == out) {
		perror(outname);
		exit(3);
	}

	plaintext = xcalloc(insize, sizeof(uint8_t));
	ciphertext = xcalloc(outsize, sizeof(uint8_t));

	if (0 != (rc = crypt8_init(&ctx, mode, initial, initial_size))) {
		fprintf(stderr, "crypt8_init failed (%d)\n", rc);
		exit(4);
	}

	size = fread(plaintext, 1, insize, rnd);
	printf("%d bytes plaintext\n", size);
	outsize = 2 * size;
	if (0 != crypt8_encrypt_msg(ctx, ciphertext, plaintext,
		&outsize, size)) {
		fprintf(stderr, "crypt8_encrypt_msg failed (%d)\n", rc);
		exit(5);
	}
	printf("encrypted %d bytes ciphertext\n", outsize);

	if (0 != (rc = crypt8_exit(ctx))) {
		fprintf(stderr, "crypt8_exit failed (%d)\n", rc);
		exit(6);
	}

	/* zap the plaintext */
	if (0 != (rc = crypt8_init(&ctx, mode, initial, initial_size))) {
		fprintf(stderr, "crypt8_init failed (%d)\n", rc);
		exit(7);
	}

	size = outsize;
	outsize = size * 2;
	deciphered = xcalloc(outsize, sizeof(uint8_t));

	if (0 != crypt8_decrypt_msg(ctx, deciphered, ciphertext,
		&outsize, size)) {
		fprintf(stderr, "crypt8_decrypt_msg failed (%d)\n", rc);
		exit(8);
	}
	printf("decrypted %d bytes plaintext\n", outsize);

	if (0 != memcmp(plaintext, deciphered, outsize)) {
		fprintf(stderr, "plaintext mismatch\n");
		exit(9);
	}
	printf("plaintext match\n");

	for (n = 0; n < count; n++) {
		outsize = 2 * insize;
		if (0 != crypt8_encrypt_msg(ctx, ciphertext, plaintext,
			&outsize, insize)) {
			fprintf(stderr, "crypt8_encrypt_msg failed (%d)\n", rc);
			exit(10);
		}
		if (outsize != fwrite(ciphertext, 1, outsize, out)) {
			perror("fwrite");
			exit(11);
		}
	}

	fclose(out);
	if (0 != (rc = crypt8_exit(ctx))) {
		fprintf(stderr, "crypt8_exit failed (%d)\n", rc);
		exit(12);
	}

	fclose(rnd);
	return 0;
}

#endif
