/*****************************************************************************
 *  ENTROPY - emerging network to reduce orwellian potency yield
 *
 *  Copyright (C) 2002 Juergen Buchmueller <pullmoll@stop1984.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *	$Id: crypt1.c,v 1.3 2005/07/12 23:12:28 pullmoll Exp $
 *****************************************************************************/
#include "crypt.h"
#if	CRYPT1_TEST
#define	xmalloc(size) malloc(size)
#define	xcalloc(num,size) calloc(num,size)
#define	xfree(ptr) if (NULL != ptr) { free(ptr); ptr = NULL; }
#else
#include "memalloc.h"
#endif
#include "logger.h"

#define	CRYPT1_MAGIC	0x47110815
#define	KEYSIZE			4096
#define	SKIPLENG		8088

typedef struct crypt1_s {
	int magic;
	size_t size;			/* allocated size of the key */
	size_t curr;			/* current offset */
	uint8_t key[KEYSIZE];	/* the key */
	uint8_t tmp[KEYSIZE];	/* temporary buffer to rotate key */
}	crypt1_t;

int crypt1_crypto(void)
{
	FUN("crypt1_crypto");

	/* no initialization required */

	return 0;
}

static inline uint8_t crypt1_next(crypt1_t *c)
{
	uint8_t this, last;
	size_t i, j;

	if (c->curr >= c->size) {
		/* make a copy of the key and sum all bytes */
		for (i = 0, j = 0; i < c->size; i++) {
			j += c->key[i];
			c->tmp[i] = c->key[i];
		}
		/* sum modulo size */
		j = j % c->size;
		/* last byte */
		last = c->tmp[j];
		for (i = 0; i < c->size; i++) {
			/* new index = old index + (key[i] ^ last) modulo size */
			j = (j + (c->key[i] ^ last)) % c->size;
			/* this byte */
			this = c->tmp[j];
			/* xor last - this with key[i] */
			c->key[i] ^= last - this;
			last = this;
		}
		c->curr = 0;
	}
	this = c->key[c->curr];
	c->curr++;
	return this;
}

/***********************************************************************
 *  encrypt_msg()
 *  Encrypt a 'plaintext' of length 'insize' into a 'ciphertext'
 *	The buffer at ciphertext can hold at most '*outsize' bytes
 ***********************************************************************/
int crypt1_encrypt_msg(void *ptr, void *ciphertext, void *plaintext,
	size_t *outsize, size_t insize)
{
	crypt1_t *c = (crypt1_t *)ptr;
	uint8_t *ct = (uint8_t *)ciphertext;
	uint8_t *pt = (uint8_t *)plaintext;
	size_t i;
	FUN("crypt1_encrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (CRYPT1_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %x (expected %x)",
			c->magic, CRYPT1_MAGIC));
		errno = EINVAL;
		return -1;
	}

	for (i = 0; i < insize; i++)
		*ct++ = *pt++ ^ crypt1_next(c);

	*outsize = insize;
	return 0;
}

/***********************************************************************
 *  decrypt_msg()
 *  Decrypt a 'ciphertext' of the length 'insize' into a 'plaintext'
 *	The buffer at ciphertext can hold at most '*outsize' bytes
 ***********************************************************************/
int crypt1_decrypt_msg(void *ptr, void *plaintext, void *ciphertext,
	size_t *outsize, size_t insize)
{
	crypt1_t *c = (crypt1_t *)ptr;
	uint8_t *pt = (uint8_t *)plaintext;
	uint8_t *ct = (uint8_t *)ciphertext;
	size_t i;
	FUN("crypt1_decrypt_msg");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == plaintext) {
		LOGS(L_CRYPTO,L_ERROR,("plaintext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == ciphertext) {
		LOGS(L_CRYPTO,L_ERROR,("ciphertext is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (NULL == outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (insize > *outsize) {
		LOGS(L_CRYPTO,L_ERROR,("outsize is too small (have %x, want %x)",
			(unsigned)*outsize, (unsigned)insize));
		errno = ENOMEM;
		return -1;
	}
	if (CRYPT1_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %x (expected %x)",
			c->magic, CRYPT1_MAGIC));
		errno = EINVAL;
		return -1;
	}

	for (i = 0; i < insize; i++)
		*pt++ = *ct++ ^ crypt1_next(c);

	*outsize = insize;
	return 0;
}

int crypt1_init(void **pptr, int init, void *data, size_t size)
{
	crypt1_t *c;
	uint8_t *bytes = (uint8_t *)data;
	size_t offs;
	FUN("crypt1_init");

	if (NULL == pptr) {
		LOGS(L_CRYPTO,L_ERROR,("pptr is NULL\n"));
		errno = EINVAL;
		return -1;
	}
	if (0 != init) {
		LOGS(L_CRYPTO,L_ERROR,("init is not 0\n"));
		errno = EINVAL;
		return -1;
	}
	if (size < 32) {
		LOGS(L_CRYPTO,L_ERROR,("key size too small (0x%x)\n", (unsigned)size));
		errno = EINVAL;
		return -1;
	}

	c = (crypt1_t *)xcalloc(1, sizeof(crypt1_t));
	*pptr = c;

	for (offs = 0; offs < size; offs++)
		c->key[offs % KEYSIZE] += bytes[offs];
	if (size > KEYSIZE)
		size = KEYSIZE;
	c->size = size;
	c->magic = CRYPT1_MAGIC;

	for (offs = 0; offs < SKIPLENG; offs++)
		(void)crypt1_next(c);

	/* Extend key to KEYSIZE if size is too small */
	while (size < KEYSIZE) {
		c->key[size++] = crypt1_next(c);
		c->size = size;
	}

	return 0;
}

int crypt1_exit(void *ptr)
{
	crypt1_t *c = (crypt1_t *)ptr;
	FUN("crypt1_exit");

	if (NULL == ptr) {
		LOGS(L_CRYPTO,L_ERROR,("ptr is NULL"));
		errno = EINVAL;
		return -1;
	}
	if (CRYPT1_MAGIC != c->magic) {
		LOGS(L_CRYPTO,L_ERROR,("invalid magic %x (expected %x)",
			c->magic, CRYPT1_MAGIC));
		errno = EINVAL;
		return -1;
	}

	xfree(c);

	return 0;
}

#if	CRYPT1_TEST
configuration_t *g_conf = NULL;
#define	SIZE	1024

int pm_vsnprintf(char *dst, size_t size, const char *fmt, va_list ap)
{
	return vsnprintf(dst, size, fmt, ap);
}

int pm_snprintf(char *dst, size_t size, const char *fmt, ...)
{
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = vsnprintf(dst, size, fmt, ap);
	va_end(ap);
	return len;
}

int main(int argc, char **argv)
{
	char rndname[MAXPATHLEN] = "/dev/urandom";
	char outname[MAXPATHLEN] = "crypt1.bin";
	FILE *rnd, *out;
	void *ctx;
	int mode = 0;
	size_t outsize = SIZE;
	size_t insize = SIZE;
	size_t count = 1;
	size_t n;
	size_t initial_size;
	unsigned char *initial;
	unsigned char *plaintext;
	unsigned char *ciphertext;
	int i;
	int rc;

	for (i = 1; i < argc; i++) {
		if (0 == strcmp(argv[i], "--diehard")) {
			insize = outsize = 0x1000;
			count = 0x1000;
		} else if (0 == strncmp(argv[i], "-r", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-r")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-o", 2) && strlen(argv[i]) > 2) {
			strcpy(rndname, &argv[i][2]);
		} else if (0 == strcmp(argv[i], "-o")) {
			strcpy(rndname, argv[++i]);
		} else if (0 == strncmp(argv[i], "-m", 2) && strlen(argv[i]) > 2) {
			mode = strtol(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-m")) {
			mode = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-s", 2) && strlen(argv[i]) > 2) {
			insize = outsize = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-s")) {
			insize = outsize = strtoul(argv[++i], NULL, 0);
		} else if (0 == strncmp(argv[i], "-c", 2) && strlen(argv[i]) > 2) {
			count = strtoul(&argv[i][2], NULL, 0);
		} else if (0 == strcmp(argv[i], "-c")) {
			count = strtoul(argv[++i], NULL, 0);
		}
	}

	initial_size = 256;

	printf("open random source: %s\n", rndname);
	rnd = fopen(rndname, "rb");
	if (NULL == rnd) {
		perror(rndname);
		exit(1);
	}

	initial = xcalloc(initial_size, sizeof(uint8_t));
	if (initial_size != fread(initial, 1, initial_size, rnd)) {
		perror("PRNG data");
		exit(2);
	}
	fclose(rnd);

	printf("create output file: %s\n", outname);
	out = fopen(outname, "wb");
	if (NULL == out) {
		perror(outname);
		exit(3);
	}

	plaintext = xcalloc(insize, sizeof(uint8_t));
	ciphertext = xcalloc(outsize, sizeof(uint8_t));

	if (0 != (rc = crypt1_init(&ctx, mode, initial, initial_size))) {
		fprintf(stderr, "crypt1_init failed (%d)", rc);
		exit(4);
	}

	for (n = 0; n < count; n++) {
		outsize = insize;
		if (0 != crypt1_encrypt_msg(ctx, ciphertext, plaintext,
			&outsize, insize)) {
			fprintf(stderr, "crypt1_encrypt_msg failed (%d)", rc);
			exit(5);
		}
		if (outsize != fwrite(ciphertext, 1, outsize, out)) {
			perror("fwrite");
			exit(6);
		}
	}

	fclose(out);

	return 0;
}

#endif
