static char rcsid[] = "@(#)$Id: mime_encode.c,v 1.22 2001/06/09 13:37:24 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.22 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include <string.h>

#include "headers.h"
#include "s_elm.h"
#include "me.h"

#include <sys/time.h>

DEBUG_VAR(Debug,__FILE__,"mime");

extern short mime_count;


static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}


int update_encoding(top_encoding,encoding)
     int *top_encoding;
     int encoding;
{
  if (encoding == ENCODING_8BIT &&
      (*top_encoding) != ENCODING_BINARY)
    (*top_encoding) = ENCODING_8BIT;
  if (encoding == ENCODING_BINARY)
    (*top_encoding) = ENCODING_BINARY;
  
  return (*top_encoding);
}

static void write_failed P_((void)) {
  MoveCursor(elm_LINES, 0);
  Raw(OFF);
  lib_error(CATGETS(elm_msg_cat, ElmSet, ElmWriteFMimeEncode,
		    "Write failed in mime_encode\n"));
  emergency_exit(0);
}

#define ERROR_EOF(x) do { \
   int error_eof = (x); if (EOF == error_eof) write_failed(); } while(0)


void print_EOLN(fp,top_encoding) 
     FILE *fp;
     int top_encoding;
{  
    /* Write canoniocal end of line of we are sending binary mail */
    if (top_encoding == ENCODING_BINARY)
	fputc('\r',fp);
    fputc('\n',fp);
}

char * mime_generate_boundary (str, size)
     char *str;
{
  time_t t = time (NULL);
	
  elm_sfprintf (str, size,
		FRM("ELM%d-%d-%d_"), t, getpid(), mime_count++);
  return str;
}

void add_parameter_t (t,name,value,quoted) 
     mime_t *t;
     char *name, *value;
     int quoted;
{
    char buffer[1025];
    
    if (t->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"add_paramater_t",
		   "Bad magic number");
    
    buffer[0] = '\0';
    
    if (t->type_opts)
    strfcpy(buffer,t->type_opts,sizeof(buffer));
    
    add_parameter(buffer,name,value,sizeof buffer,quoted);
    
    t->type_opts = strmcpy(t->type_opts,buffer);
}

void add_parameter(opts,name,value,size,quoted) 
     char *opts, *name, *value;
     int size, quoted;
{
    int len = strlen(opts);
    int ln = strlen(name);
    char * ptr = opts + len;
    int need_quotation = 0;
    
    /* Following characters require quotation: ( ) < > @ , ; : \ " / [ ] ? =
     * Also whitespace requires quotation. See Mime Draft Standard
     */
    if (!quoted && (NULL != strpbrk(value,"()<>@,;:\\\"/[]?= \t")
		    || value[0] == '\0'))
	need_quotation = 1;
    
    if (len + strlen(value) + ln + 4 + 2 * need_quotation > size)
	return; /* Don't fit anyway */
    
    if (ptr != opts) {
	*ptr++ = ';'; *ptr++ = ' ';
    }
    if (need_quotation) 
	elm_sfprintf(ptr,size - (ptr-opts),FRM("%s=%Q"),
		 name,value);
    else
	elm_sfprintf(ptr,size - (ptr-opts),FRM("%s=%s"),
		     name,value);
    return;
}

/* Prototype */
void write_encoded P_((FILE *srcfp, FILE *fpout, 
		       int encoding, int is_text, 
		       mime_send_t *mime_info));

/* For some ANSI C compilers 'char *' and 'unsigned char *' are incompatible
 * types -- we don't use casting macro here because it disables type checking
 * altogether:
 */
static char * std_str P_((unsigned char * str));
static char * std_str(str) 
     unsigned char *str;
{ 
  return (char *)str; 

}

int attach_message(part,dest,mime_info,X)
     mime_t *part; 
     FILE *dest; 
     mime_send_t *mime_info;
     struct mime_send_part * X;
{
    int err;
    FILE *srcfp;
    int is_text;


    if (part->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_message",
		   "Bad magic number");

    if ((err = can_open(part->pathname,"r")) != 0) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailErrno,
			  "Failed: %.30s: %.40s"),
		  part->pathname,error_description(err));
	return 0;
    }

    srcfp = fopen (part->pathname, "r");
    if (!srcfp) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorOpeningName,
			  "Error opening %s!"), 
		  part->pathname);
    }
    
    /* 1 if is text type (true)
     * 0 if not text type
     * -1 if can't be encoded (ie structured) Message/ or Multpart/
     */
    is_text = is_text_type (mime_types[part->type], part->subtype, 
			    part->encoding);

    if (is_text < 0 && (part->encoding == ENCODING_QUOTED ||
			part->encoding == ENCODING_BASE64)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet,ElmDontEncoding,
			  "Content-Type don't allow encoding -- ignoring this part."));
	return 0;
    }

    (void) update_encoding(&(mime_info->encoding_top),part->encoding);
	

    X -> type_part    = part ->type;
    X -> subtype_part = safe_strdup(part->subtype);
    if (part->type_opts)
	X -> type_opts_part = safe_strdup(part->type_opts);
    X ->disposition   = part -> disposition;
    if (part->disposition_opts)
	X -> disposition_opts = safe_strdup(part->disposition_opts);
    if (part->description)
	X -> description      = dup_string(part->description);
    X -> encoding_part = part->encoding;

    X->start_loc = ftell(dest);
    (void) write_encoded (srcfp, dest, part->encoding, is_text,
			  mime_info);
    X->end_loc = ftell(dest);

    fclose (srcfp);
    if (ferror(dest)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmWriteFailedCopy,
			  "Write failed to temp file in copy"));
	return 0;   /* Just indicate failure */
    }

    DPRINT(Debug,4, 
	   (&Debug, 
	    "Preparing mail for sending: part %d (attach) is %d bytes (%s)\n",
	    X-mime_info->top_parts,
	    X->end_loc-X->start_loc,
	    ENCODING(X->encoding_part)));
    
    return 1; /* ok */
}

void
base64_encode (srcfp, fpout, istext, mime_info)
     FILE *srcfp, *fpout;
     int istext;
     mime_send_t * mime_info;
{
  int c1, c2, c3 = 0;
  int ch1, ch2, ch3, ch4;
  int chars = 0;
  int last_c = 0;

  for (;;) {
      c1 = fgetc (srcfp);
      if (c1 == -1)
	  break;
      if (istext && last_c != '\r' && c1 == '\n') {
	  /* In text end of line must be coded as CR LF */
	  c1 = '\r';
	  c2 = '\n';
      }
      else
	  c2 = fgetc (srcfp);

      if (istext && c1 != '\r' && c2 == '\n') {
	  /* In text end of line must be coded as CR LF */
	  c2 = '\r';
	  c3 = '\n';
      }
      else if (c2 != -1)
	  c3 = fgetc (srcfp);
 
      if (istext && c2 != '\r' && c3 == '\n') {
	  /* In text end of line must be coded as CR LF */
	  ungetc(c3,srcfp);
	  c3 = '\r';
      }
      
      last_c = c3;
 
      ch1 = c1 >> 2;
      ch1 = to64(ch1);
      
      if (c2 != -1) {
	  ch2 = ((c1 & 0x3) << 4) | (c2 >> 4);
      ch2 = to64(ch2);
      
      if (c3 != -1) {
	  ch3 = ((c2 & 0xf) << 2) | (c3 >> 6);
	  ch3 = to64(ch3);
	  ch4 = c3 & 0x3f;
	  ch4 = to64(ch4);
      }
      else {
	  ch3 = (c2 & 0xf) << 2;
	  ch3 = to64(ch3);
	  ch4 = '=';
      }
      }
      else {
	  ch2 = (c1 & 0x3) << 4;
	  ch2 = to64(ch2);
	  ch3 = '=';
	  ch4 = '=';
      }
      
      ERROR_EOF(fputc (ch1, fpout));
      ERROR_EOF(fputc (ch2, fpout));
      ERROR_EOF(fputc (ch3, fpout));
      ERROR_EOF(fputc (ch4, fpout));
      chars += 4;
      
      if (chars >= 76) {
	  print_EOLN(fpout,mime_info->encoding_top);
	  chars = 0;
      }	
  }
  print_EOLN(fpout,mime_info->encoding_top);
  return;
}

void line_quoted_printable_encode (input,fpout,len,istext,mime_info) 
     char *input;
     FILE *fpout;
     int len; /* length of data -- be binary clean */
     int istext; /* if binary (not text) also CRLF need to be encoded */
     mime_send_t *mime_info;
{
    int chars = 0;
    char buffer[STRING];
    unsigned char c1, c2, c3;
    unsigned char lastchar = 0;
    unsigned char lastchar2 = 0;
  
    DPRINT(Debug,8, (&Debug,  
		     "line_quoted_printable_encode: len=%d, istext=%d\n",
		     len,istext));

    if (istext) {  
	if (len > 0 && input[len-1] == '\n') {
	    lastchar = input[len-1];
	    input[len-1] = '\0';
	    len--;
	    if (len > 0 && input[len-1] == '\r')  {   /* Was CR LF */
		lastchar2 = input[len-1];
		input[len-1] = '\0';
		len--;
	    } else if (mime_info->encoding_top == ENCODING_BINARY)
		lastchar2 = '\r';     /* Simulate it */
	}
    }
    
    /* I don't test agaist macros bacause these encodings are recommended
     * according MIME Draft Standard anyway and MIME encodings are
     * reversible.
     *
     * DONT_ESCAPE_MESSAGES refers '>' escaping -- not reversible
     * MIME encoding -- '>' escaping was not reversible -- this is.
     *
     * We also want do these encodings when sending (copy == 0)
     * not only when copying to another folder 
     *            -- K E H <hurtta@dionysos.FMI.FI>                     */

    while (len > 0)  {
	/* Assume that buffer don't have newlines (or they have binary data) ...
	   this routine encodes one line */
	c1 = (unsigned char) *input++;
	len--;

	if (c1 == '=') {
	    if (chars > 72) {	    
		buffer[chars++] = '=';
		buffer[chars++] = '\0';
		chars = 0;
		
		fputs (buffer, fpout);
		print_EOLN(fpout,mime_info->encoding_top);
	    }
	    buffer[chars++] = '=';
	    buffer[chars++] = '3';
	    buffer[chars++] = 'D';
	}    
	/*  printable characters -- EXCEPT:   Encode "From " in beginning */
	else if (((c1 > 31 && c1 < 127) && 
		  ((c1 != ' ' || chars != 4 || 
		    strncmp(buffer,"From",4) != 0)
		   /* Encode "." if only in line alone */
		   && (c1 != '.' || chars != 0 || len > 0)
		   /* Last space must encode also */
		   && (c1 != ' ' || len > 0))) 
		 ||
		 (c1 == 9 && len > 0 && istext)) { 
	    /* Don't make sense wrap before last character, when last character
	     * is not encoded -- wrapping and character use equal number of columns.
	     * But left space for '=\n' if we will print it in end of function
	     * instead of '\n'. */
	    if (chars > 74 && (len > 0 || !lastchar)) {
		buffer[chars++] = '=';
		buffer[chars++] = '\0';
		chars = 0;
		
		fputs (buffer, fpout);
		print_EOLN(fpout,mime_info->encoding_top);
	    }
	    buffer[chars++] = c1;
	}
	else {
	    if (chars > 72) {
		buffer[chars++] = '=';
		buffer[chars++] = '\0';
		chars = 0;
		
		fputs (buffer, fpout);
		print_EOLN(fpout,mime_info->encoding_top);
	    }
	    c2 = (c1 >> 4) & 0xf;
	    c3 = c1 & 0xf;
	    buffer[chars++] = '=';
	    buffer[chars++] = hexchars[c2];
	    buffer[chars++] = hexchars[c3];
	}
    }
    
    /* Make sure to flush the buffer.  */
    if (chars > 0) {
	buffer[chars] = '\0';
	fputs (buffer, fpout);
    }
    if (lastchar2)
	fputc (lastchar2, fpout);
    if (lastchar) {
	fputc(lastchar,fpout);
    } else { /* If input line don't terminate NL then print shoft wrap to end
	      * instead of hard NL */
	fputs ("=", fpout);
	print_EOLN(fpout,mime_info->encoding_top);
    }
}

void quoted_printable_encode (srcfp, fpout, istext, mime_info)
     FILE *srcfp, *fpout;
     int istext;  /* if binary (not text) also CRLF need to be encoded */
     mime_send_t *mime_info;
{
  char buffer[VERY_LONG_STRING];
  int len;

  DPRINT(Debug,10,(&Debug, 
		   "quoted_printable_encode: istext=%d\n",
		   istext));

  if (istext) {
      /* mail_gets is in ../lib -- it handles also NUL characters */
      while ((len = mail_gets (buffer, VERY_LONG_STRING, srcfp)) > 0)
	  line_quoted_printable_encode (buffer, fpout, len, istext,
					mime_info);
  } else {
      /* mail_gets may add LF to end of file if file don't end with LF
       * So it is not good for binary data */
      while ((len = fread(buffer, 1, sizeof(buffer), srcfp)) > 0)
	  line_quoted_printable_encode (buffer, fpout, len, istext,
					mime_info);
  }

  return;
}

void write_encoded (srcfp, fpout, encoding, is_text, mime_info)
     FILE *srcfp, *fpout;
     int encoding, is_text;
     mime_send_t *mime_info;
{
    char buffer[VERY_LONG_STRING];
    int line_len;
    
    DPRINT(Debug,12,(&Debug, 
		     "write_encoded: encoding=%d, is_text=%d\n",
		     encoding,is_text));

    if (encoding == ENCODING_BASE64)
	base64_encode (srcfp, fpout, is_text, mime_info);
    else if (encoding == ENCODING_QUOTED)
	quoted_printable_encode (srcfp, fpout, is_text, mime_info);
    else if (mime_info-> encoding_top == ENCODING_BINARY && is_text > 0) {
	/* It is better perhaps use canonical eol (CRLF) when mail have
	 * content transfer encoding BINARY somewhere (see notes about 
	 * BINARYMIME)
	 */
	while ((line_len = mail_gets(buffer, sizeof(buffer)-1, srcfp)) > 0) {
	    if (buffer[line_len-1] == '\n') {
		int add = 1;
		if (line_len >1 && buffer[line_len-2] == '\r')
		    add = 0;
		if (add) {
		    buffer[line_len-1] = '\r';
		    buffer[line_len] = '\n';
		    line_len++;
		}
	    }
	    fwrite(buffer, 1, line_len, fpout);
	}
    } else {
	while (1) {
	    if ((line_len = fread(buffer, 1,sizeof(buffer)-1, srcfp)) <= 0)
		break;     
	    fwrite(buffer, 1, line_len, fpout);
	}
    }
    return;
}

void mime_write_part_headers(fp,ptr,part)
     FILE *fp;
     mime_send_t *ptr;
     struct mime_send_part *part;
{
    /* 1) Content-Transfer-Encoding */
    if (part->encoding_part < ENCODING_EXPERIMENTAL &&
	part->encoding_part > ENCODING_NONE) {
	fprintf(fp,"Content-Transfer-Encoding: %s",
	       ENCODING(part->encoding_part));
	print_EOLN(fp,ptr->encoding_top);
    } else if (part->encoding_part == ENCODING_EXPERIMENTAL &&
	       part->encoding_part_text) {
	fprintf(fp,"Content-Transfer-Encoding: %s",
	       part->encoding_part_text);
	print_EOLN(fp,ptr->encoding_top);
    }

    /* 2) Content-Type */
    if (MIME_TYPE_UNKNOWN == part -> type_part &&
	part -> type_part_text)
	fprintf(fp,"Content-Type: %s/%s",
		part -> type_part_text,part->subtype_part);
    else
	fprintf(fp,"Content-Type: %s/%s",
		TYPE(part->type_part),part->subtype_part);
    if (part->type_opts_part) {
	fputc(';',fp);
	if (strlen(TYPE(part->type_part)) + strlen(part->subtype_part) +
	    strlen(part->type_opts_part) > 70) {
	    print_EOLN(fp,ptr->encoding_top);
	}
	fputc(' ',fp);
	fputs(part->type_opts_part,fp);
    }
    print_EOLN(fp,ptr->encoding_top);

    /* 3) Content-Disposition */
    if (part->disposition != DISP_INLINE ||
	part->disposition_opts) {
	fprintf(fp,"Content-Disposition: %s",
		DISPOSITION(part->disposition));
	if (part->disposition_opts) {
	    fputc(';',fp);
	    if (strlen(DISPOSITION(part->disposition)) +
		strlen(part->disposition_opts) > 70) {
		print_EOLN(fp,ptr->encoding_top);
	    }
	    fputc(' ',fp);
	    fputs(part->disposition_opts,fp);
	}
	print_EOLN(fp,ptr->encoding_top);
    }

    /* 4) Content-Description */
    if (part->description) {
	write_string_header(fp,"Content-Description",
			    part->description,ptr->encoding_top,
			    ptr->encode_hdr,ptr->hdr_charset);
    }
}

void mime_write_top_headers(fp, ptr)
     FILE *fp;
     mime_send_t *ptr;
{
    fputs(MIME_HEADER, fp);
    print_EOLN(fp, ptr->encoding_top);  

    if (ptr->msg_is_multipart) {
	fprintf(fp,"Content-Type: multipart/mixed; %s",
		ptr->type_opts_top);
	print_EOLN(fp, ptr->encoding_top);  
	fprintf(fp,"Content-Transfer-Encoding: %s",
	       ENCODING(ptr->encoding_top));
	print_EOLN(fp, ptr->encoding_top);  
    } else if (1 == ptr->top_parts_count) {
	if (MIME_TYPE_TEXT == ptr->top_parts[0].type_part &&
	    0 == istrcmp("Plain",ptr->top_parts[0].subtype_part) &&
	    ptr->top_parts[0].result_charset &&
	    ptr->top_parts[0].result_charset->MIME_name &&
	    0 == istrcmp("US-ASCII",
			 ptr->top_parts[0].result_charset->MIME_name) &&
	    (ENCODING_NONE == ptr->top_parts[0].encoding_part ||
	     ENCODING_7BIT == ptr->top_parts[0].encoding_part) &&
	    !send_mime_plain) {
	    DPRINT(Debug,2,(&Debug, 
			    "mime_write_header: send_mime_plain==%d: Omitting mime headers.\n",
			    send_mime_plain));
	    return;
	} else {
	    mime_write_part_headers(fp,ptr,&(ptr->top_parts[0]));
	} 
    } else {
	DPRINT(Debug,2,(&Debug, 
			"mime_write_header: msg_is_multipart=%d, top_parts_count=%d: Odd values?\n",
		   ptr->msg_is_multipart,ptr->top_parts_count));
    }
} 

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
