static char rcsid[] = "@(#)$Id: state.c,v 1.21 2001/06/09 10:23:56 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.21 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "headers.h"
#include "melib.h"

DEBUG_VAR(Debug,__FILE__,"misc");

static void state_panic P_((char *,int,char *, char *)); /* Prototype */
static void state_panic(f,ln,pr,ms) 
     char * f;
     int ln;
     char *pr;
     char *ms;

{
    panic("STATE PANIC",f,ln,pr,ms,0);
}

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}


static void init_si_file P_((in_state_t *)); /* Prototype */
static void init_si_file(s)
     in_state_t *s;
{
    s->magic  = STATE_in_file;
    s->u.file.fpin = NULL;
}

static void init_si_string P_((in_state_t *)); /* Prototype */
static void init_si_string(s)
     in_state_t *s;
{
  s->magic  = STATE_in_string;
  s->u.string.inbuf = NULL;
  s->u.string.inreadp = NULL;
}

void in_state_clear (s,m)
     in_state_t *s;
     int m;
{
  s->magic = 0;  /* Not initilized yet */

  switch(m) {
  case STATE_in_file:   init_si_file(s);   break;
  case STATE_in_string: init_si_string(s); break;
  default:
    state_panic(__FILE__,__LINE__,"in_state_clear","Bad magic number");
  }
}

static void dest_si_file P_((in_state_t *)); /* Prototype */
static void dest_si_file(s)
     in_state_t *s;
{
  if (s->magic  != STATE_in_file)
      state_panic(__FILE__,__LINE__,"dest_si_file","Bad magic number");
  s->u.file.fpin = NULL;
}

static void dest_si_string P_((in_state_t *)); /* Prototype */
static void dest_si_string(s)
     in_state_t *s;
{
  if (s->magic  != STATE_in_string)
    state_panic(__FILE__,__LINE__,"dest_si_string","Bad magic number");

  if (s->u.string.inbuf)
    free(s->u.string.inbuf);
  
  s->u.string.inbuf = NULL;
  s->u.string.inreadp = NULL;
}

void in_state_destroy (s)
     in_state_t *s;
{
  switch(s->magic) {
  case STATE_in_file:   dest_si_file(s);   break;
  case STATE_in_string: dest_si_string(s); break;
  default:
    state_panic(__FILE__,__LINE__,"in_state_destroy","Bad magic number");
  }
  s->magic = 0; /* State destroyed */
}

void set_in_state_buffer (c, s)
     char *c;
     in_state_t *s;
{
  if (s->magic  != STATE_in_string)
    state_panic(__FILE__,__LINE__,"set_in_state_buffer","Bad magic number");
  
  if (s->u.string.inbuf)
    state_panic(__FILE__,__LINE__,"set_in_state_buffer","Already called");
  
  s->u.string.inreadp = s->u.string.inbuf = strmcpy (s->u.string.inbuf, c);
}

void set_in_state_file (F, s)
     FILE *F;
     in_state_t *s;
{
  if (s->magic  != STATE_in_file)
    state_panic(__FILE__,__LINE__,"set_in_state_file","Bad magic number");

  if (s->u.file.fpin)
    state_panic(__FILE__,__LINE__,"set_in_state_file","Already called");

  s->u.file.fpin = F;
}

int in_state_seekable (s)
     in_state_t *s;
{
  switch(s->magic) {
  case STATE_in_file:   return 1;   
  case STATE_in_string: return 0; 
  default:
    state_panic(__FILE__,__LINE__,"in_state_seekable","Bad magic number");
  }

  return 0;
}

int in_state_fseek (s,pos)
     in_state_t *s;
     long pos;
{
    if (s->magic  != STATE_in_file)
	state_panic(__FILE__,__LINE__,"in_state_fseek","Bad magic number");

  if (s->u.file.fpin == NULL)
    state_panic(__FILE__,__LINE__,"in_state_fseek","NULL file pointer");

  return fseek(s->u.file.fpin,pos,SEEK_SET);
}

long in_state_ftell (s)
     in_state_t *s;
{
  if (s->magic  != STATE_in_file)
    state_panic(__FILE__,__LINE__,"in_state_ftell","Bad magic number");
  
  if (s->u.file.fpin == NULL)
    state_panic(__FILE__,__LINE__,"in_state_ftell","NULL file pointer");

  return ftell(s->u.file.fpin);
}

FILE * in_state_FILE (s) 
     in_state_t *s;
{
  if (s->magic  != STATE_in_file)
    state_panic(__FILE__,__LINE__,"in_state_FILE","Bad magic number");

  if (s->u.file.fpin == NULL)
    state_panic(__FILE__,__LINE__,"in_state_FILE","NULL file pointer");
  
  return s->u.file.fpin;
}

static int getc_si_file P_((in_state_t *)); /* Prototype */
static int getc_si_file(s)
     in_state_t *s;
{
  if (s->magic  != STATE_in_file)
    state_panic(__FILE__,__LINE__,"getc_si_file","Bad magic number");
  if (s->u.file.fpin == NULL)
    state_panic(__FILE__,__LINE__,"getc_si_file","NULL filepointer");

  return fgetc (s->u.file.fpin);
}

static int getc_si_string P_((in_state_t *)); /* Prototype */
static int getc_si_string(s)
     in_state_t *s;
{
  if (s->magic  != STATE_in_string)
    state_panic(__FILE__,__LINE__,"getc_si_string","Bad magic number");

  if (s->u.string.inbuf) {
    unsigned char ret;

    if (! *s->u.string.inreadp) {
      /* No more data left in the buffer, so clean up... */
      s->u.string.inreadp = NULL;
      free (s->u.string.inbuf);
      s->u.string.inbuf = NULL;
      return EOF;
    }
    
    if (*s->u.string.inreadp)
      ret = (unsigned char) *s->u.string.inreadp++;
    return ret;
  }
  
  DPRINT(Debug,5,(&Debug,
		  "getc_si_string: inbuf==NULL\n"));
  return EOF;
}

int state_getc (s)
     in_state_t *s;
{
  switch(s->magic) {
  case STATE_in_file:   return getc_si_file(s);   
  case STATE_in_string: return getc_si_string(s); 
  default:
    state_panic(__FILE__,__LINE__,"state_getc","Bad magic number");
  }

  return EOF;
}

static int ungetc_si_file P_((int,in_state_t *)); /* Prototype */
static int ungetc_si_file (c,s)
     int c;
     in_state_t *s;
{
  if (s->magic  != STATE_in_file)
    state_panic(__FILE__,__LINE__,"ungetc_si_file","Bad magic number");
  if (s->u.file.fpin == NULL)
    state_panic(__FILE__,__LINE__,"umgetc_si_file","NULL filepointer");

  return ungetc(c,s->u.file.fpin);
}

static int ungetc_si_string P_((int,in_state_t *)); /* Prototype */
static int ungetc_si_string (c,s)
     int c;
     in_state_t *s;
{
  if (s->magic  != STATE_in_string)
    state_panic(__FILE__,__LINE__,"ungetc_si_string","Bad magic number");

  if (s->u.string.inbuf) {
    if (s->u.string.inreadp > s->u.string.inbuf) {
      return *(--(s->u.string.inreadp));
    } else {
	DPRINT(Debug,5,(&Debug,
			"ungetc_si_string: In beginning of buffer.\n"));
      return EOF;
    }
  } else {
      DPRINT(Debug,5,(&Debug,
	       "ungetc_si_string: inbuf==NULL\n"));
    return EOF;
  }
}

int state_ungetc (c,s)
     int c;
     in_state_t *s;
{
  switch(s->magic) {
  case STATE_in_file:   return ungetc_si_file(c,s);   
  case STATE_in_string: return ungetc_si_string(c,s); 
  default:
    state_panic(__FILE__,__LINE__,"state_ungetc","Bad magic number");
  }
  return EOF;
}


static char *gets_si_file P_((char *, int, in_state_t *));
static char *gets_si_file (dest, length, s)
     char *dest;
     int length;
     in_state_t *s;
{
  if (s->magic  != STATE_in_file)
    state_panic(__FILE__,__LINE__,"gets_si_file","Bad magic number");
  if (s->u.file.fpin == NULL)
    state_panic(__FILE__,__LINE__,"gets_si_file","NULL filepointer");
  
  return (fgets (dest, length, s->u.file.fpin));
}

static int getl_si_file P_((char *, int, in_state_t *));
static int getl_si_file (dest, length, s)
     char *dest;
     int length;
     in_state_t *s;
{
  if (s->magic  != STATE_in_file)
    state_panic(__FILE__,__LINE__,"getl_si_file","Bad magic number");
  if (s->u.file.fpin == NULL)
    state_panic(__FILE__,__LINE__,"getl_si_file","NULL filepointer");
  
  return (mail_gets (dest, length, s->u.file.fpin));
}


static int getl_si_string P_((char *, int, in_state_t *));
static int getl_si_string (dest, length, s)
     char *dest;
     int length;
     in_state_t *s;
{
  int i = 0, ch;
  
  if (s->magic  != STATE_in_string)
    state_panic(__FILE__,__LINE__,"getl_si_string","Bad magic number");

  while (length > 1) {
    ch = getc_si_string (s);
    if (ch == EOF)
      break;
    else if (ch == '\n') {
      dest[i++] = '\n';
      break;
    }
    else
      dest[i++] = ch;
    length--;
  }
  dest[i] = '\0';
  return i;
}
  
static char *gets_si_string P_((char *, int, in_state_t *));
static char *gets_si_string (dest, length, s)
     char *dest;
     int length;
     in_state_t *s;
{
  int len;

  if (s->magic  != STATE_in_string)
    state_panic(__FILE__,__LINE__,"getl_si_string","Bad magic number");

  len = getl_si_string (dest, length, s);
  if (len > 0)
    return dest;
  else
    return NULL;
}

char *state_gets (dest, length, s)
     char *dest;
     int length;
     in_state_t *s;
{
  switch(s->magic) {
  case STATE_in_file:   return gets_si_file(dest,length,s);   
  case STATE_in_string: return gets_si_string(dest,length,s); 
  default:
    state_panic(__FILE__,__LINE__,"state_gets","Bad magic number");
  }
  return NULL;
}

int state_getl (dest, length, s)
     char *dest;
     int length;
     in_state_t *s;
{
  switch(s->magic) {
  case STATE_in_file:   return getl_si_file(dest,length,s);   
  case STATE_in_string: return getl_si_string(dest,length,s); 
  default:
    state_panic(__FILE__,__LINE__,"state_getl","Bad magic number");
  }
  return 0;
}


static void init_so_file P_((out_state_t *)); /* Prototype */
static void init_so_file(s)
     out_state_t *s;
{
    s->magic   = STATE_out_file;
    s->u.file.fpout = NULL;
}

static void init_so_string P_((out_state_t *)); /* Prototype */
static void init_so_string(s)
     out_state_t *s;
{
    s->magic        = STATE_out_string;
    s->u.string.outbuf     = NULL;
    s->u.string.outwritep  = NULL;
    s->u.string.outbufsize = 0;
}

static void init_so_buffer P_((out_state_t *)); /* Prototype */
static void init_so_buffer(s)
     out_state_t *s;
{
    s->magic                = STATE_out_buffer;
    s->u.buffer             = NULL;
}

static void init_so_dir P_((out_state_t *)); /* Prototype */
static void init_so_dir(s)
     out_state_t *s;
{
    s->magic                 = STATE_out_dir;
    s->u.dir.dir             = NULL;
    s->u.dir.write_state_ptr = NULL;
}

void out_state_clear (s,m)
     out_state_t *s;
     int m;
{
    static charset_t default_vector[2];

    default_vector[0] = display_charset;
    default_vector[1] = NULL;

    s->magic = 0;  /* Not initilized yet */

    s->displaying      = 0;
    s->EOLN_is_CRLF    = 0;         /* EOLN is \n */
    s->prefix          = NULL;
    s->filter          = NULL;
    s->display_charset = default_vector;
    s->filter_line     = NULL;
    
    switch(m) {
    case STATE_out_file:     init_so_file(s);   break;
    case STATE_out_string:   init_so_string(s); break;
    case STATE_out_buffer:   init_so_buffer(s); break;
    case STATE_out_dir:      init_so_dir(s);    break;
    default:
	state_panic(__FILE__,__LINE__,"out_state_clear","Bad magic number");
    }
}

static void dest_so_file P_((out_state_t *)); /* Prototype */
static void dest_so_file(s)
     out_state_t *s;
{
    if (s->magic   != STATE_out_file)
	state_panic(__FILE__,__LINE__,"dest_so_file","Bad magic number");
    s->u.file.fpout = NULL;
}

static void dest_so_string P_((out_state_t *)); /* Prototype */
static void dest_so_string(s)
     out_state_t *s;
{
    if (s->magic        != STATE_out_string)
	state_panic(__FILE__,__LINE__,"dest_so_string","Bad magic number");
    
    s->u.string.outbuf     = NULL;
    s->u.string.outwritep  = NULL;
    s->u.string.outbufsize = 0;
}

static void dest_so_buffer P_((out_state_t *)); /* Prototype */
static void dest_so_buffer(s)
     out_state_t *s;
{
    if (s->magic        != STATE_out_buffer)
	state_panic(__FILE__,__LINE__,"dest_so_buffer","Bad magic number");
    
    s->u.buffer            = NULL;
}

static void dest_so_dir P_((out_state_t *)); /* Prototype */
static void dest_so_dir(s)
     out_state_t *s;
{
    if (s->magic        != STATE_out_dir)
	state_panic(__FILE__,__LINE__,"dest_so_dir","Bad magic number");
    
    s->u.dir.dir             = NULL;
    s->u.dir.write_state_ptr = NULL;
}

static int seekable_so_dir P_((out_state_t *)); /* Prototype */
static int seekable_so_dir(s)
     out_state_t *s;
{
    if (s->magic        != STATE_out_dir)
	state_panic(__FILE__,__LINE__,"seekable_so_dir","Bad magic number");

    
    if (tell_dir_write_state(s->u.dir.dir,s->u.dir.write_state_ptr) > 0)
	return 1;
    return 0;
}

int  out_state_seekable(s)
     out_state_t *s;
{
    switch(s->magic) {
    case STATE_out_file:   return 1;   
    case STATE_out_string: return 0; 
    case STATE_out_buffer: return 0; 
    case STATE_out_dir:    return seekable_so_dir(s);
    default:
	state_panic(__FILE__,__LINE__,
		    "out_state_seekable","Bad magic number");
    }

    return 0;
}

static void flush_filter P_((out_state_t *s, int munge_eoln));

/* Returns NULL if no available ... */
FILE *out_state_FILE(s)
     out_state_t *s;
{
    if (s->filter_line)
	flush_filter(s,0);

    switch(s->magic) {
    case STATE_out_file:   return s->u.file.fpout;   
    case STATE_out_string: return NULL; 
    case STATE_out_buffer: return NULL; 
    case STATE_out_dir:    return NULL; 
    default:
	state_panic(__FILE__,__LINE__,
		    "out_state_FILE","Bad magic number");
    }

    return NULL;
}

int  out_state_fseek(s,pos)
     out_state_t *s;
     long pos;
{ 
    if (s->filter_line)
	flush_filter(s,0);

    switch (s->magic) {

    case STATE_out_file:

	if (!s->u.file.fpout)
	    state_panic(__FILE__,__LINE__,"out_state_fseek",
			"NULL file pointer");

	return fseek(s->u.file.fpout,pos,SEEK_SET);
	
    case STATE_out_dir:
	/* Note reversed return value on functions! */

	if (seek_dir_write_state(s->u.dir.dir,
				 s->u.dir.write_state_ptr,
				 pos))
	    return 0; /* SUCCEES */
	return -1; /* FAILUE */

    }
    state_panic(__FILE__,__LINE__,"out_state_fseek","Bad magic number");
    return -1;
}

long out_state_ftell (s)
     out_state_t *s;
{ 
    if (s->filter_line)
	flush_filter(s,0);

    switch(s->magic) {
    case STATE_out_file:

	if (!s->u.file.fpout)
	    state_panic(__FILE__,__LINE__,"out_state_ftell",
			"NULL file pointer");

	return ftell(s->u.file.fpout); 
    case STATE_out_dir:
	return tell_dir_write_state(s->u.dir.dir,s->u.dir.write_state_ptr);
    }
    state_panic(__FILE__,__LINE__,"out_state_ftell","Bad magic number");
    return -1;
}

void out_state_destroy (s)
     out_state_t *s;
{

    if (s->filter_line)
	flush_filter(s,0);

    switch(s->magic) {
    case STATE_out_file:     dest_so_file(s);   break;
    case STATE_out_string:   dest_so_string(s); break;
    case STATE_out_buffer:   dest_so_buffer(s); break;
    case STATE_out_dir:      dest_so_dir(s); break;
    default:
	state_panic(__FILE__,__LINE__,"out_state_destroy","Bad magic number");
    }

    s->displaying      = 0;
    s->prefix          = NULL;
    s->filter          = NULL;
    s->display_charset = NULL;
    if (s->filter_line)
	free_string(&(s->filter_line));
    s->magic      = 0; /* No longer initialized */
}

void set_out_state_buffer (buffer,size,s)
     char * buffer;
     int size;
     out_state_t *s;
{
    if (s->magic        != STATE_out_string)
	state_panic(__FILE__,__LINE__,"set_out_state_buffer",
		    "Bad magic number");
    
    if (s->u.string.outbuf     != NULL)
	state_panic(__FILE__,__LINE__,"set_out_state_buffer","Already called");

    s->u.string.outbuf     = buffer;
    s->u.string.outwritep  = buffer;
    s->u.string.outbufsize = size;
}

extern void set_out_state_dir(dir,write_state_ptr,s)
     struct folder_browser *dir;
     WRITE_STATE  write_state_ptr;
     out_state_t *s;  
{
    if (s->magic != STATE_out_dir )
	state_panic(__FILE__,__LINE__,"set_out_state_dir",
		    "Bad magic number");

    if (s->u.dir.dir || s->u.dir.write_state_ptr) 
	state_panic(__FILE__,__LINE__,"set_out_state_dir","Already called");
	
    if (!dir || !write_state_ptr)
	state_panic(__FILE__,__LINE__,"set_out_state_dir","Bad argument");

    if (!s->display_charset) 
	state_panic(__FILE__,__LINE__,"set_out_state_dir",
		    "Display_charset -vector must have been set (possible to RAW_BUFFER)");

    
    s->u.dir.dir             = dir;
    s->u.dir.write_state_ptr = write_state_ptr;
}

void out_state_ref_buffer (s,p,len)
     out_state_t *s;
     char **p;
     int *len;
{
    if (s->magic        != STATE_out_string)
	state_panic(__FILE__,__LINE__,"out_state_ref_buffer",
		    "Bad magic number");

    if (s->u.string.outbuf     == NULL)
	state_panic(__FILE__,__LINE__,
		    "out_state_ref_buffer","NULL buffer pointer");

    *p   = s->u.string.outbuf;
    *len = s->u.string.outwritep - s->u.string.outbuf;
}

/* STATE_out_buffer */
extern void set_out_state_sbuffer(buffer,s)
     struct stringbuffer *buffer;
     out_state_t *s;  
{
    if (s->magic        != STATE_out_buffer)
	state_panic(__FILE__,__LINE__,"set_out_state_sbuffer",
		    "Bad magic number");
    
    if (s->u.buffer       != NULL)
	state_panic(__FILE__,__LINE__,"set_out_state_sbuffer",
		    "Already called");

    s->u.buffer         = buffer;
}

void set_out_state_file (file,s)
     FILE * file;
     out_state_t *s;
{
    if (s->magic   != STATE_out_file)
	state_panic(__FILE__,__LINE__,"set_out_state_file","Bad magic number");
    if (s->u.file.fpout != NULL)
	state_panic(__FILE__,__LINE__,"set_out_state_file","Already called");
    s->u.file.fpout = file;
}

static int putc_so_file P_((out_state_t *, int)); /* Prototype */
static int putc_so_file(s,ch)
     out_state_t *s;
     int ch;
{
    if (s->magic   != STATE_out_file)
	state_panic(__FILE__,__LINE__,"putc_so_file","Bad magic number");
    if (s->u.file.fpout == NULL)
	state_panic(__FILE__,__LINE__,"putc_so_file","NULL file pointer");
    
    return (fputc (ch, s->u.file.fpout));
}

static int putc_so_string P_((out_state_t *, int)); /* Prototype */
static int putc_so_string(s,ch)
     out_state_t *s;
     int ch;
{
    if (s->magic   != STATE_out_string)
	state_panic(__FILE__,__LINE__,"putc_so_string","Bad magic number");
    if (s->u.string.outbuf == NULL)
	state_panic(__FILE__,__LINE__,"putc_so_string","NULL buffer pointer");
    
    if (s->u.string.outwritep - s->u.string.outbuf < 
	s->u.string.outbufsize-1) {
	*s->u.string.outwritep++ = ch;
	*s->u.string.outwritep = '\0';
	return 1;
    }
    else
	return EOF;
}

int state_putc (ch, s)
     int ch;                        /* unsigned char assumed */
     out_state_t *s;
{
    if (s->filter_line || s->filter ||
	s->magic   == STATE_out_buffer ||
	s->magic   == STATE_out_dir) {
	int r;
	if (!s->filter_line) {
	    if (!s->filter) {
		/* STATE_out_dir also requires s->display_charset[0]
		   -- value RAW_BUFFER indicates raw copying ... 
		*/
		s->filter_line = new_string(s->display_charset[0]);
	    } else
		s->filter_line = new_string(s->filter);
	}
	r = add_streambyte_to_string(s->filter_line,ch);
	if ('\n' == ch) {
	    flush_filter(s,0); 
	}
	return r ? 1 : EOF;
    }

    switch(s->magic) {
    case STATE_out_file:      return putc_so_file(s,ch);   
    case STATE_out_string:    return putc_so_string(s,ch); 
    default: state_panic(__FILE__,__LINE__,"state_putc","Bad magic number");
    }
    return EOF;
}

static int put_so_file P_((out_state_t *, char *,int)); /* Prototype */
static int put_so_file(s,string,len)
     out_state_t *s;
     char *string;
     int len;
{
    if (s->magic   != STATE_out_file)
	state_panic(__FILE__,__LINE__,"put_so_file","Bad magic number");
    if (s->u.file.fpout == NULL)
	state_panic(__FILE__,__LINE__,"put_so_file","NULL file pointer");
    
    return fwrite (string, 1, len, s->u.file.fpout);
}

static int put_so_string P_((out_state_t *, char *,int)); /* Prototype */
static int put_so_string(s,string,len)
     out_state_t *s;
     char *string;
     int len;
{
    int count;
    
    if (s->magic   != STATE_out_string)
	state_panic(__FILE__,__LINE__,"put_so_string","Bad magic number");
    if (s->u.string.outbuf == NULL)
	state_panic(__FILE__,__LINE__,"put_so_string","NULL buffer pointer");
    
    for (count = 0; count < len; count++) {
	if (putc_so_string (s,(unsigned char) string[count]) == EOF)
	    return (count > 0 ? count : EOF);
    }
    return count;
}

int state_put (string, len, s)
     char *string;
     int len;
     out_state_t *s;
{
    if (s->filter_line || s->filter ||
	s->magic   == STATE_out_buffer ||
	s->magic   == STATE_out_dir) {
	int i, first = 0;
	
	for (i = 0; i < len; i++) {
	    if ('\n' == string[i]) {
		int r;
		if (!s->filter_line) {
		    if (!s->filter) {
			if (s->magic   == STATE_out_dir)
			    s->filter_line = new_string(RAW_BUFFER);
			else
			    s->filter_line = new_string(s->display_charset[0]);
		    } else
			s->filter_line = new_string(s->filter);
		}
		r = add_streambytes_to_string(s->filter_line,
					  i-first+1,s2us(string+first));
		flush_filter(s, 0); 
		if (r < i-first+1)
		    return first + r;  /* FAILURE */
		first = i+1;
	    }
	}
	if (first < len) {
	    int r;
	    if (!s->filter_line) {
		if (!s->filter) {
		    if (s->magic   == STATE_out_dir)
			s->filter_line = new_string(RAW_BUFFER);
		    else
			s->filter_line = new_string(s->display_charset[0]);
		} else
		    s->filter_line = new_string(s->filter);
	    }

	    r = add_streambytes_to_string(s->filter_line,
					  len-first,s2us(string+first));
	    if (r < len-first)
		return first + r;  /* FAILURE */
	}
	return len;
    }

    switch(s->magic) {
    case STATE_out_file:      return put_so_file(s,string,len);   
    case STATE_out_string:    return put_so_string(s,string,len);
    default:  state_panic(__FILE__,__LINE__,"state_put","Bad magic number");
    }
    return EOF;
}

/* convert one line -- *len is assumed to point end of line */
void state_convert_EOLN(buffer,len,buffer_size,st)
     char *buffer; 
     int *len; 
     int buffer_size;
     out_state_t *st;
{
    int buf_len = *len;

    if (st->EOLN_is_CRLF) {

	if (buf_len > 0 && buffer[buf_len - 1] == '\n' &&
	    (buf_len < 1 || buffer[buf_len - 2] != '\r')) {

	    if (buf_len < buffer_size -2) {
		buffer[buf_len - 1] = '\r';
		buffer[buf_len] = '\n';
		buffer[buf_len+1] = '\0';
		buf_len++;

		DPRINT(Debug,25,(&Debug,
				 "state_convert_EOLN: '%.10s'... LF -> CRLF\n",
				 buffer));

	    } else {
		DPRINT(Debug,2,(&Debug,
				"state_convert_EOLN: '%.10s'... No space (size=%d, len=%d) for LF -> CRLF conversion\n",
				buffer,buffer_size,buf_len));
	    }
	}

    } else {

	/* Take care of CRLF => LF conversion */
	if (buf_len > 1 &&
	    buffer[buf_len - 1] == '\n' &&
	    buffer[buf_len - 2] == '\r') {
	    buffer[buf_len - 2] = '\n';
	    buffer[buf_len - 1] = '\0';
	    buf_len--;

	    DPRINT(Debug,25,(&Debug,
			     "state_convert_EOLN: '%.10s'... CRLF -> LF\n",
			     buffer));

	}

    }

    if (*len != buf_len) {
	DPRINT(Debug,25,(&Debug,
			 "state_convert_EOLN- len=%d -> %d\n",
			 *len,buf_len));
    }

    *len = buf_len;
}


int state_printf P_((out_state_t *s,
		     const char * format, const char *msg, ...));

int state_printf (
#if ANSI_C
		  out_state_t *s,
		  CONST char * format, 
		  CONST char *msg, ...
#else
		  s, format, msg, va_alist
#endif
		  )
#if !ANSI_C
     out_state_t *s;
     CONST char * format;
     CONST char *msg;
     va_dcl
#endif
{
    int ret = 0;
    va_list vl;
    int  t     = strlen(format);
    char * t1  = strchr(format,'\n');

    if (t1 && t1 != &(format[t-1])) 
	state_panic(__FILE__,__LINE__,"state_printf",
		    "Embedded newlines are not supported");

    Va_start(vl, msg);           /* defined in defs.h */


    if (s->filter_line || s->filter ||
	s->magic   == STATE_out_buffer ||
	s->magic   == STATE_out_dir) {
	struct string * temp = elm_smessage(-1,format,msg,vl);

	ret = string_len(temp);
	if (!s->filter_line) {	   
	    if (!s->filter)
		s->filter_line = temp;
	    else {
		s->filter_line = convert_string(s->filter,temp,0);
		free_string(&temp);
	    }
	} else {
	    struct string * temp2 = cat_strings(s->filter_line,
						temp,0);
	    free_string(&(s->filter_line));
	    if (!s->filter)
		s->filter_line = temp2;
	    else {
		s->filter_line = convert_string(s->filter,temp2,0);
		free_string(&temp2);
	    }
	    free_string(&temp);
	}
	if (t1)
	    flush_filter(s, 1 /* do LF <=> CRLF conversion */);

    } else {
	char * str = elm_vmessage(-1,format,msg,vl);
	int len = strlen(str);

	if (t1) {
	    int size = len + 4;
	    
	    str = safe_realloc(str,size);

	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(str,&len,size,s);
	}

	switch(s->magic) {
	case STATE_out_file:   ret = put_so_file(s,str,len);   break;   
	case STATE_out_string: ret = put_so_string(s,str,len); break;
	default:  state_panic(__FILE__,__LINE__,"state_printf",
			      "Bad magic number");
	}

	free(str);
    }

    va_end(vl);

    return ret;
}

static void flush_filter(s, munge_eoln)
     out_state_t *s;
     int munge_eoln;
{
    if (s->magic   == STATE_out_buffer) {
	int len = string_len(s->filter_line);
	if (len > 0) {   /* Remove LF from end */
	    uint16 c = give_unicode_from_string(s->filter_line,len-1);
	    struct string * t;
	    int X = 0;

	    if (0x000A == c) {
		len--;
		if (len > 0) {     /* And CR (ie. CRLF) from end */
		    uint16 c = give_unicode_from_string(s->filter_line,len-1);
		    if (0x000D == c)
			len--;
		}
	    }
	    t = clip_from_string(s->filter_line,&X,len);
	    add_line_to_stringbuffer(s->u.buffer,t);
	    free_string(&t);
	} else
	    add_line_to_stringbuffer(s->u.buffer,s->filter_line);
    } else if (s->magic   == STATE_out_dir) {
	char * s1 = NULL;
	int l = 0;
	if (RAW_BUFFER == s->display_charset[0]) {
	    DPRINT(Debug,10,(&Debug,
			     "flush_filter: STATE_out_dir: Raw copy%s\n",
			     munge_eoln ? " -- convert EOLN" : "")); 
	    bytestream_from_string(s->filter_line,&s1,&l); 
	} else {
	    struct string * tmp = convert_string(s->display_charset[0],
						 s->filter_line,0);
	    /* Can't ask just printable characters because we need
	       NLs also ...
	    */
	    DPRINT(Debug,10,(&Debug,
			     "flush_filter: STATE_out_dir: Conversion%s\n",
			     munge_eoln ? " -- convert EOLN" : "")); 
	    bytestream_from_string(tmp,&s1,&l); 
	    free_string(&tmp);
	}

	if (munge_eoln) {
	    int size = l + 4;

	    s1 = safe_realloc(s1,size);
		    
	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(s1,&l,size,s);	    
	}

	if (!write_dir_write_state(s->u.dir.dir,s->u.dir.write_state_ptr,
				   l,s1)) {
	    DPRINT(Debug,1,(&Debug,
			    "state: start_fd_write_state failed\n"));
	}
	
	free(s1);
    } else {
	char *s1;
	int l;
	/* s->display_charset[0] is either display_charset (internal pager)
	 * or system_charset (external pager)
	 */
	struct string * tmp = convert_string(s->display_charset[0],
					     s->filter_line,0);
	/* Can't ask just printable characters because we need
	   NLs also ...
	*/
	s1 = us2s(stream_from_string(tmp,0,NULL));
	l = strlen(s1);

	if (munge_eoln) {
	    int size = l + 4;

	    s1 = safe_realloc(s1,size);
		    
	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(s1,&l,size,s);	    
	}
	
	switch(s->magic) {
	case STATE_out_file:      put_so_file(s,s1,l);   break;
	case STATE_out_string:    put_so_string(s,s1,l); break;
	default:  state_panic(__FILE__,__LINE__,"flush_filter",
			      "Bad magic number");
	}
	free(s1);
	free_string(&tmp);
    }
    free_string(&(s->filter_line));	
}


int state_puts (string, state)
     char *string;
     out_state_t *state;
{
    return state_put(string,strlen(string),state);
}

void state_nlputs(string,state)
     char *string;
     out_state_t *state;
{
    int i;
    int last_start = 0;

    for (i = 0; string[i]; i++) {
	switch (string[i]) {
	case '\n':
	    if (last_start < i) {
		int x = i;
		
		if ('\r' == string[x-1])
		    x--;
		if (last_start < x) {
		    state_put(string + last_start, x - last_start, state);
		}
	    }
	    if (state->EOLN_is_CRLF)
		state_puts("\r\n",state);
	    else
		state_puts("\n",state);
	    last_start = i+1;
	    break;
	}
    }

    /* Print rest of dtata in case there is no \n on end ... */
    if (last_start < i) {
	state_put(string + last_start, i - last_start, state);
    }
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
