static char rcsid[] = "@(#)$Id: mime_decode.c,v 1.39 2001/06/06 18:09:02 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.39 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "headers.h"
#include "melib.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mime");

/* Prototype */
static void Xbit_decode	P_((in_state_t *, out_state_t *, int, int)); 

static void Xbit_decode (s_in, s_out, len, is_text)
     in_state_t *s_in;
     out_state_t *s_out;
     int len;
     int is_text;
{
    char buf[VERY_LONG_STRING];
    int offset = 0, length;
    
    if (s_out->prefix) {
	strfcpy (buf, s_out->prefix, sizeof buf);
	offset = strlen (buf);
    }
    while (len > 0) {
	if (0 ==
	    (length = state_getl (buf + offset, sizeof buf - offset, s_in)))
	    break;
	len -= length;
		
	if (is_text > 0) {

	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(buf + offset, &length,
			       sizeof buf - offset,
			       s_out);
	}
    
	state_put(buf,offset+length,s_out);
    }
}

void base64_decode (s_in, s_out, length, astext)
     in_state_t  *s_in;
     out_state_t *s_out;
     int length;
     int astext;
{
  /* Notes:  this routine assumes that strlen(line length) % 4 == 0.
     Most intelligent software will do this, so I haven't worried
     about being *completely* compliant with the standard.  I've
     not run into any encoded data where this was not true.  */
  
  char buf[STRING], *p;
  int
    bytes = 0, /* the total number of bytes read so far */
    terminate = FALSE,
    c1, c2, c3, c4;
  unsigned char
    ch,
    store_ch = 0; /* for astext */
  int corrupted = FALSE;

  DPRINT(Debug,10,(&Debug, 
		   "base64_decode: length=%d, prefix=%s, astext=%d\n",
		   length,
		   s_out->prefix ? s_out->prefix : "<NULL>",
		   astext));

  /* Make sure to quote the first line! */
  state_add_prefix(s_out);

  while (! terminate) {
    int len;

    if (bytes >= length)
      break;

    if ((len = state_getl (buf, sizeof buf, s_in)) <= 0)
	break;

    bytes += len;

    p = buf;
    while (*p) {
      c1 = *p++;

      if (base64(c1) == -1) {
        p--;
        /* partial fix -- K E H   <hurtta@dionysos.FMi.FI>  */

        while (*p == ' ' || *p == '\t' || *p == '\r')
          p++;

        if (*p != '\n')
          corrupted = TRUE;  /* Perhaps corrupted */

        if (*p == 0)
          break;

        c1 = *p++;
      }

      /* End of the line. */
      if (c1 == '\n')
	continue;

      c2 = *p++;
      
      c1 = base64(c1);
      if (c1 == -1)
        corrupted = TRUE;
      c2 = base64(c2);
      if (c2 == -1)
        corrupted = TRUE;
      ch = (c1 << 2) | (c2 >> 4);
      
      if (store_ch && ch != '\n') /* for astext */
	  state_putc(store_ch,s_out);

      store_ch = 0;
      if (astext && ch == '\r')
	  store_ch = ch;
      else {
	  if (ch == '\n' && astext && s_out->EOLN_is_CRLF)
	      state_putc('\r',s_out);

	  state_putc(ch,s_out);
	  if (ch == '\n')
	      state_add_prefix(s_out);
      }      
    
      c3 = *p++;
      if (c3 == '=') {
	terminate = TRUE;
	break;
      }
      c3 = base64(c3);
      if (c3 == -1)
	corrupted = TRUE;
      ch = ((c2 & 0xf) << 4) | (c3 >> 2);

      if (store_ch && ch != '\n')
	  state_putc(store_ch,s_out);

      store_ch = 0;

      if (astext && ch == '\r')
	  store_ch = ch;
      else {
	  if (ch == '\n' && astext && s_out->EOLN_is_CRLF)
	      state_putc('\r',s_out);

	  state_putc(ch,s_out);
	  if (ch == '\n')
	      state_add_prefix(s_out);
      }
  
      c4 = *p++;
      if (c4 == '=') {
	terminate = TRUE;
	break;
      }
      
      c4 = base64(c4);
      if (c4 == -1)
	corrupted = TRUE;
      ch = ((c3 & 0x3) << 6) | c4;
      
      if (store_ch && ch != '\n')
	  state_putc(store_ch,s_out);

      store_ch = 0;

      if (astext && ch == '\r') {
	  store_ch = ch;
      }
      else {
	  if (ch == '\n' && astext && s_out->EOLN_is_CRLF)
	      state_putc('\r',s_out);

	  state_putc(ch,s_out);
	  if (ch=='\n')
	      state_add_prefix(s_out);
      }
    }
  }

  /* Make sure to flush anything left in the internal buffer. */
  if (store_ch)  /* for astext */
    state_putc(store_ch,s_out);

  if (corrupted) {
      if (s_out->displaying) {
	  /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	     conversions */
	  state_nlputs("\n[ ",s_out);
	  state_printf(s_out,CATGETS(elm_msg_cat, MeSet, MeDecodeBASE64Corrupt,
				     "BASE64 data was corrupt!"));
	  state_nlputs(" ]\n",s_out);
      }
      lib_error (CATGETS(elm_msg_cat, MeSet, MeDecodeBASE64Corrupt,
			 "BASE64 data was corrupt!"));
  }

  DPRINT(Debug,10,(&Debug, 
		   "base64_decode: readed=%d bytes, corrupted=%d.\n",
		   bytes,corrupted));   

  return;
}
  
void quoted_printable_decode (s_in, s_out, length, astext)
     in_state_t  *s_in;
     out_state_t *s_out;
     int length;
     int astext;
{
    int bytes = 0; /* number of bytes read */
    int nl = TRUE; /* flag to indicate hitting a new line */
    char *p;
    int c1, c2;
    unsigned char ch, store_ch = 0;
    char buf[VERY_LONG_STRING];
    int corrupted = 0;

    DPRINT(Debug,10,(&Debug, 			
		     "quoted_printable_decode: length=%d, prefix=%s, astext=%d\n",
		     length,
		     s_out->prefix ? s_out->prefix : "<NULL>",
		     astext));
    
    for (;;) {
	int len;
	
	if (bytes >= length)
	    break;

	if ((len=state_getl (buf, sizeof buf, s_in)) <= 0)
	    break;
	bytes += len;

	p = buf;
	while (*p) {

	    /* If there is a prefix and this is the beginning of a new 
	     *  line... 
	     */
	    if (nl) {
		state_add_prefix(s_out);
		nl = FALSE;
	    }

	    if (store_ch && *p != '\n') /* for astext */
		state_putc(store_ch,s_out);
	    store_ch = 0;
      
	    if (*p == '=') {
		p++;
		/* Ignore spaces in end of line   -- see MIME */
		if (*p == '\r' || *p == ' ' || *p == '\t') {
		    char *t = p;
		    while (*t && (*t == '\r' || *t == ' ' || *t == '\t'))
			t++;
		    if (*t && *t == '\n')
			p = t;
		}

		if (*p == '\n') { /* soft linebreak */
		    if (p >= buf +len)
			break;
		    p++;
		} else {
		    c1 = hex(*p);
		    if (c1 == -1)
			corrupted = TRUE;
		    p++;
		    c2 = hex(*p);
		    if (c2 == -1)
			corrupted = TRUE;
		    p++;
		    ch = (c1 << 4) | c2;

		    /* We not need here CR LF -> LF removing, because
		     * CRLF's which presents end of line should NOT be 
		     * encoded.
		     */

		    state_putc(ch,s_out);
		}
	    }
	    else {
		if (astext && *p == '\r')
		    store_ch = *p;
		else {
		    if (*p == '\n' && astext && s_out->EOLN_is_CRLF)
			state_putc('\r',s_out);

		    state_putc(*p,s_out);
		}
		if (*p == '\n') {
		    nl = TRUE;
		}
		if (p >= buf +len)
		    break;
		p++;
	    }  
	}
    }

    /* Flush anything left in the buffer */
    if (store_ch) { /* for astext */
	state_putc(store_ch,s_out);
	store_ch = 0;
    }

    if (corrupted) {
	if (s_out -> displaying) {
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */
	    state_nlputs("\n[ ",s_out);
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeQPCorrupt,
				 "Seems that QUOTED-PRINTABLE data was corrupted."));
	    state_nlputs(" ]\n",s_out);
	}
	lib_error (CATGETS(elm_msg_cat, MeSet, MeDecodeQPCorrupt,
			   "Seems that QUOTED-PRINTABLE data was corrupted."));
    }
    
    DPRINT(Debug,10,(&Debug, 
		     "quoted_printable_decode: readed=%d bytes, corrupted=%d.\n",
		     bytes,corrupted));
    
    return;
}

/* Prototype */
static void multipart_decode  P_((mime_t *, 
				  in_state_t *, out_state_t *,
				  charset_t)); 

static void multipart_decode (att, state_in, state_out, defcharset)
     mime_t *att; /* The list of attachments for this MULTIPART data */
     in_state_t   *state_in;
     out_state_t  *state_out;
     charset_t defcharset;
{
    int nattach = 0;

    DPRINT(Debug,11,(&Debug, 
		     "multipart_decode -> START\n"));

    if (att->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"multipart_decode",
		   "Bad magic number");

    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"multipart_decode",
		   "multipart_decode: unsupported input state");
    }

    while (att) {
	if (att->magic != MIME_magic)
	    mime_panic(__FILE__,__LINE__,"multipart_decode",
		       "Bad magic number (next -chain)");

	nattach++;

	DPRINT(Debug,11,(&Debug, 
			 "multipart_decode: [%d]: Type: %.15s/%.30s, Encoding: %d, Size: %ld\n",
			 nattach, 
			 mime_types[att->type], att->subtype,
			 att->encoding, (long) att->length));
	
	if (in_state_fseek(state_in,att->offset) != 0) {
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */
	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartSeekFail,
				 "multipart_decode: seek failed"));
	    state_nlputs(" ]\n",state_out);
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartSeekFail,
			      "multipart_decode: seek failed"));
	    return;
	}

	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */
	if (nattach > 1)
	    state_nlputs("\n", state_out);

	if (state_out->displaying) {
	    char tmp[40];
	    char *Encoding = "???";
	    
	    if (att->disposition != DISP_INLINE)
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeAttachA, 
					       "[ Attach #%d: "),
			     nattach);
	    else
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodePartA,
					       "[ Part #%d: "),
			     nattach);

	    tmp[0] = '\0';
	    if (att->description) {
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeDescA,
					       "\"%.60S\" ]\n"),
			     att->description);
		state_puts("[ ",state_out);
	    } else if (att->disposition_opts &&
		       mime_get_param ("filename", tmp, att->disposition_opts, 
				       sizeof(tmp))) {
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeDescFilenameA,
					       "Filename: %.60s ]\n"),
			     tmp);
		state_puts("[ ",state_out);
	    } 
	    
	    Encoding = ENCODING(att->encoding);
	    
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */

	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, MeDecodeDescC,
					   "Type: %.15s/%.30s, Encoding: %s, Size: %d ]\n"),
			 mime_types[att->type], att->subtype,
			 Encoding, att->length);
	    state_nlputs("\n", state_out);

	} else {	    
	    if (att->description) {
		/* state_nlputs or state_printf is needed for EOLN_is_CRLF
		   conversions */
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeContentDescA,
					       "Content-Description: %.300S\n"),
			     att->description);
		state_nlputs("\n", state_out);
	    } 
	}
      
	mime_decode (att, state_in, state_out, defcharset);
	att = att->next;
    }

    DPRINT(Debug,11,(&Debug, 
		     "multipart_decode <- END\n"));
}

static void null_SG_decoder(body,sign,state_in,state_out,micalg, defcharset)
     mime_t *body; 
     mime_t *sign;
     in_state_t   *state_in;
     out_state_t  *state_out;
     char *micalg;
     charset_t defcharset;
{
    if (state_out->displaying) {	
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */
     	
	state_nlputs("\n[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, 
			     MeDecodeUnsupportedSig,
			     "Signature %s/%.30s is unsupported, not checked"), 
		     TYPE(sign->type), sign->subtype);
	state_nlputs(" ]\n",state_out);       
    }
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedSig,
		      "Signature %s/%.30s is unsupported, not checked"),
	      TYPE(sign->type), sign->subtype);

    mime_decode(body,state_in,state_out, defcharset);
}

static void null_EC_decoder(init,data,state_in,state_out,defcharset)
     mime_t *init;
     mime_t *data;
     in_state_t   *state_in;
     out_state_t  *state_out;
     charset_t defcharset;
{
    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
       conversions */
  	
    state_nlputs("\n[ ",state_out);
    state_printf(state_out,
		 CATGETS(elm_msg_cat, MeSet, 
			 MeDecodeUnsupportedEnc,
			 "Encryption %s/%.30s is unsupported"), 
		 TYPE(init->type), init->subtype);
    state_nlputs(" ]\n",state_out);
    
    lib_error(CATGETS(elm_msg_cat, MeSet, 
		      MeDecodeUnsupportedEnc,
		      "Encryption %s/%.30s is unsupported"), 
	      TYPE(init->type), init->subtype);
}

typedef void signed_decoder P_((mime_t *body, mime_t *sign,
				in_state_t   *state_in,
				out_state_t  *state_out,
				char *micalg, charset_t defcharset));
typedef signed_decoder * SG_decoder_t;

static SG_decoder_t select_SG_decoder P_((char *protocol));
static SG_decoder_t select_SG_decoder(protocol)
     char *protocol;
{
#ifdef USE_PGP
    if (0 == istrcmp(protocol,"application/pgp-signature")) {
	DPRINT(Debug,11,(&Debug, 
			 "select_SG_decoder [%s] = pgp_SG_decoder\n",
			 protocol));
	return pgp_SG_decoder;
    }
#endif

    DPRINT(Debug,11,(&Debug, 
		     "select_SG_decoder [%s] = null_SG_decoder\n",
		     protocol));
    return null_SG_decoder;
}

/* Prototype */
static void signed_decode P_((mime_t *, in_state_t *, out_state_t *, 
			      charset_t)); 

static void signed_decode (ptr, state_in, state_out, defcharset)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{
    char protocol[100];
    char micalg[100];
    SG_decoder_t decode;

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"signed_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"signed_decode",
		   "signed_decode: unsupported input state");
    }
    
    if (! ptr->parts || !ptr->parts->next) {

	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */
	
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeSignedNosubtypes,
			     "Content-Type: multipart/signed, no subtypes (parts=null)"));
	state_nlputs(" ]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeSignedNosubtypes,
			  "Content-Type: multipart/signed, no subtypes (parts=null)"));
	return;
    }
    
    if (!mime_get_param ("protocol", protocol, ptr->type_opts, 
			 sizeof protocol)) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocol,
			     "'protocol' paramater is missing from Multipart/Signed -type!"));
	state_nlputs(" ]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocol,
			  "'protocol' paramater is missing from Multipart/Signed -type!"));
	return;
    }

    if (!mime_get_param ("micalg", micalg, ptr->type_opts, 
			 sizeof micalg)) {
	micalg[0] = '\0';
    }


    decode = select_SG_decoder(protocol);

    decode(ptr->parts,ptr->parts->next,state_in,state_out,micalg, defcharset);
}


typedef void encrypted_decoder P_((mime_t *init, mime_t *data,
				   in_state_t   *state_in,
				   out_state_t  *state_out,
				   charset_t    defcharset));

typedef encrypted_decoder * EC_decoder_t;


static EC_decoder_t select_EC_decoder P_((char *protocol));
static EC_decoder_t select_EC_decoder(protocol)
     char *protocol;
{
#ifdef USE_PGP
    if (0 == istrcmp(protocol,"application/pgp-encrypted")) {
	DPRINT(Debug,11,(&Debug, 
			 "select_EC_decoder [%s] = pgp_EC_decoder\n",
			 protocol));
	return pgp_EC_decoder;
    }
#endif

    DPRINT(Debug,11,(&Debug, 
		     "select_EC_decoder [%s] = null_EC_decoder\n",
		     protocol));
    return null_EC_decoder;
}

/* Prototype */
static void encrypted_decode P_((mime_t *, in_state_t *, out_state_t *,
				 charset_t)); 

static void encrypted_decode (ptr, state_in, state_out, defcharset)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{
    char protocol[100];
    EC_decoder_t decode;

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"encrypted_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"encrypted_decode",
		   "encrypted_decode: unsupported input state");
    }
    
    if (! ptr->parts || !ptr->parts->next) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeEncryptedNosubtypes,
			     "Content-Type: multipart/encrypted, no subtypes (parts=null)"));
	state_nlputs(" ]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeEncryptedNosubtypes,
			  "Content-Type: multipart/encrypted, no subtypes (parts=null)"));
	return;
    }
    
    if (!mime_get_param ("protocol", protocol, ptr->type_opts, 
			 sizeof protocol)) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocolEnc,
			     "'protocol' paramater is missing from multipart/encrypted -type!"));
	state_nlputs(" ]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocolEnc,
			  "'protocol' paramater is missing from multipart/encrypted -type!"));
	return;
    }

    decode = select_EC_decoder(protocol);

    decode(ptr->parts,ptr->parts->next,state_in,state_out,defcharset);
}


static mime_t * best_alternative P_((mime_t *att));
/* Prototype */
static void alternative_decode P_((mime_t *, 
				   in_state_t *, out_state_t *, charset_t)); 

static void alternative_decode (ptr, state_in, state_out, defcharset)
     mime_t     *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t  defcharset;
{
    mime_t *best;
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"alternative_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"alternative_decode",
		   "alternative_decode: unsupported input state");
    }
    
    if (! ptr->parts) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeNosubtypes,
			     "Content-Type: multipart/alternative, no subtypes (parts=null)"));
	state_nlputs(" ]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeNosubtypes,
			  "Content-Type: multipart/alternative, no subtypes (parts=null)"));
	return;
    }
    
    best = best_alternative(ptr->parts);
    
    if (best) {
	DPRINT(Debug,11,(&Debug, 
			 "multipart_alternative: Type: %.15s/%.30s, Encoding: %d, Size: %ld\n",
			 mime_types[best->type], best->subtype,
			 best->encoding, (long) best->length));
	
	if (in_state_fseek(state_in,best->offset) != 0) {
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */

	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeSeekFailed,
				 "multipart_alternative: seek failed"));
	    state_nlputs(" ]\n",state_out);
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeSeekFailed,
			      "multipart_alternative: seek failed"));
	    return;
	}


	if (state_out->displaying) {
	    char tmp[40];
	    char *Encoding = "???";
	    
	    if (best->disposition != DISP_INLINE)
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeSelAttachA, 
					       "[ Selecting attach: "));
	    else
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeSelPartA,
					       "[ Selecting part: "));

	    tmp[0] = '\0';
	    if (best->description) {
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeDescA,
					       "\"%.60S\" ]\n"),
			     best->description);
		state_puts("[ ",state_out);
	    } else if (best->disposition_opts &&
		       mime_get_param ("filename", tmp, 
				       best->disposition_opts, 
				       sizeof(tmp))) {
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeDescFilenameA,
					       "Filename: %.60s ]\n"),
			     tmp);
		state_puts("[ ",state_out);
	    } 
	    
	    Encoding = ENCODING(best->encoding);
	    
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */

	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, MeDecodeDescC,
					   "Type: %.15s/%.30s, Encoding: %s, Size: %d ]\n"),
			 mime_types[best->type], best->subtype,
			 Encoding, best->length);
	    state_nlputs("\n", state_out);

	} else {	    
	    if (best->description) {
		/* state_nlputs or state_printf is needed for EOLN_is_CRLF
		   conversions */

		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeContentDescA,
					       "Content-Description: %.300S\n"),
			     best->description);
		state_nlputs("\n", state_out);
	    } 
	}

	mime_decode (best, state_in, state_out, defcharset);
	
    } else {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	state_puts("[ ", state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeCtNoAlternative,
			     "Content-Type: multipart/alternative, alternative not found"));
    
	state_nlputs(" ]\n", state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeCtNoAlternative,
			  "Content-Type: multipart/alternative, alternative not found"));
    }
}

/* Prototype */
static void multipart_0_decode P_((mime_t *, 
				   in_state_t *, out_state_t *,
				   charset_t)); 

static void multipart_0_decode (ptr, state_in, state_out, defcharset)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"multipart_0_decode",
		   "Bad magic number");
    
    if (ptr->parts)
	multipart_decode (ptr->parts, state_in, state_out, defcharset);
    else {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartNoparts,
			     "Content-Type: multipart/*, no subtypes (parts = NULL)"));
	state_nlputs(" ]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartNoparts,
			  "Content-Type: multipart/*, no subtypes (parts = NULL)"));
    }
}

/* Prototype */
static void rfc822_decode P_((mime_t *, 
			      in_state_t *, out_state_t *, charset_t)); 



int rfc822_header_filter(hdr,flag)
     header_list_ptr hdr;
     int flag;                  /* elm_filter */
{
  char buf[80];

  strfcpy(buf,give_header_name(hdr->header_name),78);
  strfcat(buf,": ", sizeof buf);
  
  if (flag) {
    if (matches_weedlist (buf)) 
      return 0;
    else
      return 1;
  } else
    return 1;
}

static void nomime_decode P_((mime_t *ptr,
			      in_state_t *state_in,
			      out_state_t *state_out));
static void nomime_decode (ptr, state_in, state_out)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
    int j;

    DPRINT(Debug,11,(&Debug, 
		     "nomime_decode -> state: offset=%ld, length=%ld -- ",
		     (long) ptr -> offset, (long) ptr -> length));
    for (j = 0; state_out->display_charset[j]; j++) {
	DPRINT(Debug,11,(&Debug, 
			 "'display charset[%d]'=%s ",j,
			 state_out->display_charset[j]->MIME_name ? 
			 state_out->display_charset[j]->MIME_name :
			 "<no MIME name>"));
    }
    DPRINT(Debug,11,(&Debug, 
		     "\n"));
    
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"nomime_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"nomime_decode",
		   "nomime_decode: unsupported input state");
    }
    
    
    if (in_state_fseek(state_in,ptr->offset) != 0) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeNoMimeSeekFailed,
			     "nomime_decode: seek failed"));
	state_nlputs(" ]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoMimeSeekFailed,
			  "nomime_decode: seek failed"));
	return;
    }
    
    /* default-nomime-charset gives assumed charset */
    state_out -> filter = default_nomime_charset;

    /* No MIME is just text ... */
    Xbit_decode (state_in, state_out, ptr->length, 1);
    
    DPRINT(Debug,11,(&Debug, 
		     "nomime_decode <- END\n"));
}

static void rfc822_decode (mt, state_in, state_out, defcharset)
     mime_t *mt;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{
    /* This looks a lot like multipart_decode() except that we want to print
     * the headers for the message.  NOTE: "mt" should be a pointer to the
     * RFC822 attachment, NOT its subpart, since we need the offset in order
     * to print out the headers.
     */

    header_list_ptr headers = NULL;
    header_list_ptr From_header, mime_version, osv;
    int decode_headers = 1;

    if (mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"rfc822_decode",
		   "Bad magic number");
  
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"rfc822_decode",
		   "rfc822_decode: unsupported input state");
    }

  if (in_state_fseek(state_in,mt->offset) != 0) {
      /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	 conversions */
      
      state_puts("[ ",state_out);
      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeRFC822SeekFailed,
			   "rfc822_decode: seek failed"));
      state_nlputs(" ]\n",state_out);
      lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeRFC822SeekFailed,
			"rfc822_decode: seek failed"));
      return;
  }

  headers = state_read_headers(state_in, RHL_MARK_FOLDING);

  mime_version = locate_header_by_name(headers,"MIME-Version");
  osv          = locate_header_by_name(headers,"X-ELM-OSV");

  if (!mime_version && !req_mime_hdrencoding && !req_mime_hdrencoding)
      decode_headers = 0;
  if (osv && osv->body) {
      char value[20];
      if (mime_get_param("no-hdr-encoding",value,osv->body,
			 sizeof value) && atoi(value) > 0)
	  decode_headers = 0;

      /* FIXME: Update defcharset */
  }
  
  From_header  = locate_header_by_name(headers,"From");

  if (From_header) {
      struct string * buffer = give_decoded_header(From_header,
						   decode_headers,
						   defcharset);

      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeStartMail,
			   "-- Start of included mail From: %S\n"),
		   buffer);

      free_string(&buffer);
  }
  else
      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeStartMail1,
			   "-- Start of included mail.\n"));

  /* state_nlputs or state_printf is needed for EOLN_is_CRLF
     conversions */

  state_nlputs("\n",state_out);

  state_write_headers(state_out,headers,
		      rfc822_header_filter,
		      elm_filter,
		      decode_headers,
		      defcharset
		      );


  delete_headers(&headers); 

  /* state_nlputs or state_printf is needed for EOLN_is_CRLF
     conversions */

  state_nlputs("\n",state_out);

  if (mt->parts == NULL) {
      /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	 conversions */

      state_puts("[ ",state_out);
      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeNoRFC822,
			   "rfc822_decode: no body of RFC 822 mail (parts = NULL)"));
      state_nlputs(" ]\n",state_out);
      lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoRFC822,
			"rfc822_decode: no body of RFC 822 mail (parts = NULL)"));
      return;
  }

  if (mt->parts->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"rfc822_decode",
	       "Bad magic number (parts)");

  /* TODO: Handle Pre-MIME Content-types ... */
  if (mime_version || !req_mime_bodyencoding)
      mime_decode (mt->parts, state_in, state_out, defcharset);
  else
      nomime_decode (mt->parts, state_in, state_out);

  state_printf(state_out,
	       CATGETS(elm_msg_cat, MeSet, MeDecodeEndMail,
		       "-- End of included mail.\n"));
  return;
}

/* Prototype */
static int run_decoder P_((mime_t *, in_state_t *, out_state_t *)); 

static int run_decoder (ptr, state_in, state_out) 
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
{
    int is_text;

    DPRINT(Debug,11,(&Debug, 
		     "run_decoder -> state: offset=%ld, length=%ld\n",
		     (long) ptr -> offset, (long) ptr -> length));

  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"run_decoder",
	       "Bad magic number");


  if (in_state_seekable(state_in)) {
      DPRINT(Debug,11,(&Debug, 
		       "              (file): ftell=%ld\n",
		       in_state_ftell(state_in))); 
  } else {
    	DPRINT(Debug,11,(&Debug, 			
			 "              (\?\?\?\?): magic=%d\n",
			 state_in->magic));
  }

  is_text = is_text_type (mime_types[ptr->type], ptr->subtype, 
			  ptr->encoding);

  DPRINT(Debug,11,(&Debug, 
		   "run_decoder: is_text=%d; type=%s (%d)/%s; encoding =%s (%d)\n",
		   is_text, mime_types[ptr->type], ptr->type, ptr->subtype,
		   ENCODING(ptr->encoding),
		   ptr->encoding));

  if (ptr->encoding == ENCODING_BASE64)
      base64_decode (state_in, state_out, ptr->length, is_text);
  else if (ptr->encoding == ENCODING_QUOTED)
      quoted_printable_decode (state_in, state_out, ptr->length, is_text);
  else if (ptr->encoding != ENCODING_NONE && 
	   ptr->encoding != ENCODING_7BIT &&
	   ptr->encoding != ENCODING_8BIT &&
	   ptr->encoding != ENCODING_BINARY) {

      DPRINT(Debug,11,(&Debug, 			
		       "run_decoder=0 <- END; \n"));
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "              (file); ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }
    return 0;
  }
  else
    Xbit_decode (state_in, state_out, ptr->length, is_text);

  DPRINT(Debug,11,(&Debug, 		   
		   "run_decoder=1 <- END; \n"));
  if (in_state_seekable(state_in)) {
      DPRINT(Debug,11,(&Debug, 
		       "              (file); ftell=%ld\n",
		       in_state_ftell(state_in))); 
  }
  
  return 1;
}

int set_filter (ptr, state) 
     mime_t *ptr;
     out_state_t *state;
{
    int code;
    charset_t tmp;

    DPRINT(Debug,40,(&Debug, 
		     "set_filter -> state: offset=%d, length=%ld\n",
		     (long) ptr -> offset, (long) ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"set_filter",
		   "Bad magic number");
    
    state -> filter = NULL; /* So that messages can be printed .. */
           
    if (0 == (code = mime_get_charset (&tmp, ptr->type_opts,
				       state->display_charset))) {

    	/* Don't show this part (or copy to reply buffer) ... */	  
	
	state_printf(state,
		     CATGETS(elm_msg_cat, MeSet, 
			     MeDecodeCharsetSkipping,
			     "[ Charset %.15s unsupported, skipping... ]\n"), 
		     tmp->MIME_name ? tmp->MIME_name : "<none>");
	
	if (state->displaying)
	    state_printf(state,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeUseVtoView,
				 "[ Use 'v' to view or save this part. ]\n"));
	
	state -> filter = tmp;

       	DPRINT(Debug,40,(&Debug, 
			 "set_filter=0 filter=%p '%s' (type=%p)\n",
			 state -> filter,
			 state -> filter->MIME_name ?
			 state -> filter->MIME_name : "<none>",
			 state -> filter->charset_type));
	
	return 0;
    } else {
	if (code < 0) {
	    state_printf(state,
			 CATGETS(elm_msg_cat, MeSet, 
				 MeDecodeCharsetConvert,
				 "[ Charset %.15s unsupported, converting... ]\n"),
			 tmp->MIME_name ? tmp->MIME_name : 
			 "<none>");
	    
	    if (state->displaying) {	
		/* state_nlputs or state_printf is needed for EOLN_is_CRLF
		   conversions */

		state_printf(state,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeUseAlsoV,
				     "[ You can also use 'v' to view or save this part. ]\n"));	    
		state_nlputs("\n",state);
	    }
	}
	state -> filter = tmp;

	DPRINT(Debug,40,(&Debug, 
			 "set_filter=1 filter=%p '%s' (type=%p)\n",
			 state -> filter,
			 state -> filter->MIME_name ?
			 state -> filter->MIME_name : "<none>",
			 state -> filter->charset_type));	
	return 1;
    }
}

/* Prototype */
static void text_decode	P_((mime_t *, in_state_t *, out_state_t *, 
			    charset_t)); 

static void text_decode (ptr, state_in, state_out, defcharset) 
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{
    DPRINT(Debug,11,(&Debug, 			
		     "text_decode -> state: offset=%ld, length=%ld \n",
		     (long) ptr -> offset, (long) ptr -> length));

  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"text_decode",
	       "Bad magic number");

  if (in_state_seekable(state_in)) {
      DPRINT(Debug,11,(&Debug, 
		       "              (file): ftell=%ld\n",
		       in_state_ftell(state_in))); 
  }

  if (set_filter(ptr,state_out)) {
    if (!run_decoder(ptr,state_in, state_out)) {
      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeEncodingSkipping,
			   "[ Unsupported encoding, skipping... ]\n"));
      if (state_out->displaying)	
	  state_printf(state_out,
		       CATGETS(elm_msg_cat, MeSet, MeDecodeUseVEncodedSave,
			       "[ Use 'v' to save this part in encoded form. ]\n"));
    }
  }

  state_out -> filter = NULL;

  DPRINT(Debug,11,(&Debug, 
		   "text_decode <- END; \n"));

  if (in_state_seekable(state_in)) {
      DPRINT(Debug,11,(&Debug, 
		       "          (file); ftell=%ld\n",
		       in_state_ftell(state_in))); 
  }
}

FILE * arrange_binary(ptr,state_in,state_out,newstate2, name)
     mime_t *ptr;
     in_state_t  *state_in;
     out_state_t *state_out;
     in_state_t *newstate2;
     char **name;
{
    FILE * tmpfp = NULL;
    char *fname = NULL;
    
    DPRINT(Debug,11,(&Debug, 
		     "arrange_binary -> state: offset=%ld, length=%ld \n",
		     (long) ptr -> offset, (long) ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"arrange_binary",
		   "Bad magic number");
    
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "                 (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }
  
    fname = elm_message(FRM("%selmdecode-%d"), 
			temp_dir, getpid ());
    if (NULL == (tmpfp = safeopen_rdwr(fname))) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			  "Failed to create file for decoding."));
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			     "Failed to create file for decoding."));
	state_nlputs(" ]\n",state_out);
    } else { /* Tempfile opened */
	int c = EOF, prev = EOF;
	int is_binary = ptr->encoding == ENCODING_BINARY;
	long pos;

	if (name)
	    *name = fname;
	else
	    unlink(fname);  /* We can unlink it now ... */

	DPRINT(Debug,11,(&Debug, 
			 "                 : is_binary = %d\n",is_binary));

	if (in_state_fseek(state_in,ptr->begin_offset) != 0) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeArrangeSeekFail,
			      "arrange_binary: seek failed"));
	    return NULL;
	}

	while ((pos = in_state_ftell(state_in)) < 
	       ptr -> offset + ptr ->length &&
	       EOF != (c = state_getc(state_in))) {

	    if (pos < ptr -> offset) {   /*  --- HEADER PART -- */
		if ('\n' == c && '\r' != prev) {
		    if (is_binary) {
			DPRINT(Debug,11,(&Debug, 
					 "arrange_binary -- (header) not really binary input!\n")); 
			is_binary = 0;
		    }
		    fputc('\r',tmpfp);
		}		
	    } else if (!is_binary && '\n' == c && '\r' != prev) {
		fputc('\r',tmpfp);
	    }
	    fputc(c,tmpfp);
	    prev = c;
	}
	rewind(tmpfp); /* Rewind it for reading */	    
	set_in_state_file(tmpfp,newstate2);
    }

    if (!name)
	free(fname);
    return tmpfp;
}

FILE * arrange_decoded(ptr,state_in,state_out,newstate2,name)
     mime_t *ptr;
     in_state_t  *state_in;
     out_state_t *state_out;
     in_state_t *newstate2;
     char **name;
{
    FILE * tmpfp = NULL;
    int result = 1;
    char *fname = NULL;
    out_state_t newstate;
    
    out_state_clear(&newstate,STATE_out_file);
    
    DPRINT(Debug,11,(&Debug, 
		     "arrange_decoded -> state: offset=%ld, length=%ld \n",
		     (long) ptr -> offset, (long) ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"arrange_decoded",
		   "Bad magic number");
    
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "                 (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }
  
    fname = elm_message(FRM("%selmdecode.%d"), 
			temp_dir, getpid ());
    if (NULL == (tmpfp = safeopen_rdwr(fname))) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */
	
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			  "Failed to create file for decoding."));
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			     "Failed to create file for decoding."));
	state_nlputs(" ]\n",state_out);
	result = 0;
    } else { /* Tempfile opened */

	if (name)
	    *name = fname;
	else
	    unlink(fname);  /* We can unlink it now ... */
    
	newstate.displaying = state_out->displaying;
	
	set_out_state_file(tmpfp, &newstate);
	newstate.prefix          = NULL;             /* No prefix in that pass */
	newstate.filter          = state_out->filter;
	newstate.display_charset = state_out->display_charset;

	if (run_decoder(ptr,state_in,&newstate)) { 
	    
	    if (EOF == fflush(tmpfp)) {
		/* state_nlputs or state_printf is needed for EOLN_is_CRLF
		   conversions */

		lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeErrorFlush,
				  "Error when flushing temporary file."));
		state_puts("[ ",state_out);
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeErrorFlush,
				     "Error when flushing temporary file."));
		state_nlputs(" ]\n",state_out);
		result = 0;
	    }
	    rewind(tmpfp); /* Rewind it for reading */
	    
	    set_in_state_file(tmpfp,newstate2);
	} else { /* Run decoder failed */
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeEncodingSkipping,
				 "[ Unsupported encoding, skipping... ]\n"));
	    if (state_out->displaying)	
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeUseVEncodedSave,
				     "[ Use 'v' to save this part in encoded form. ]\n"));
	    result = 0;
	}
    }
    out_state_destroy(&newstate);

    DPRINT(Debug,11,(&Debug, 
		     "arrange_decoded; result=%d\n",result));
    
    if (!result && tmpfp) {
	fclose(tmpfp);
	tmpfp = NULL;
	if (name)
	    *name = NULL;
	free(fname);
    }

    if (!name)
	free(fname);
    return tmpfp;
}

/* Prototype */
static void elm_decode	P_((mime_t *, in_state_t *, out_state_t *, 
			    charset_t)); 

static void elm_decode (ptr, state_in, state_out, defcharset) 
     mime_t *ptr;
     in_state_t  *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{  
  in_state_t newstate2;

  FILE *tmpfp;
  char buffer[LONG_STRING];
  
  in_state_clear(&newstate2,STATE_in_file);

  DPRINT(Debug,11,(&Debug, 
		   "elm_decode -> state: offset=%ld, length=%ld; \n",
		   (long) ptr -> offset, (long) ptr -> length));

  if (ptr->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"elm_decode",
	       "Bad magic number");

  if (in_state_seekable(state_in)) {
      DPRINT(Debug,11,(&Debug, 
		       "             (file): ftell=%ld\n",
		       in_state_ftell(state_in))); 
  }

  if (NULL != (tmpfp = 
	       arrange_decoded(ptr,state_in,state_out,&newstate2,NULL))) {
      if (set_filter(ptr,state_out)) {
	  int len, crypted = 0;
	  int count = 0;
	  getkey(OFF);
      
	  while((len = state_getl(buffer,sizeof(buffer),&newstate2)) > 0) {
	      count += len;
	      
	if (!strncmp(buffer, START_ENCODE, strlen(START_ENCODE))) {
	  state_printf(state_out,
		       CATGETS(elm_msg_cat, MeSet, MeDecodeStartElmEncoded,
			       "-- Start of (Elm) encoded section.\n"));
	  crypted = ON;
	  continue;
	} else if (!strncmp(buffer, END_ENCODE, strlen(END_ENCODE))) {
	  crypted = OFF;
	  state_printf(state_out,
		       CATGETS(elm_msg_cat, MeSet, MeDecodeEndElmEncoded,
			       "-- End of (Elm) encoded section.\n"));
	  continue;
	} else if (crypted) {
	    no_ret(buffer);
	    encode(buffer);      
	    if (state_out->EOLN_is_CRLF)
		strfcat(buffer, "\r\n", sizeof buffer);
	    else
		strfcat(buffer, "\n", sizeof buffer);	    
	}
	state_add_prefix(state_out);
	state_puts(buffer,state_out);
      }

	  DPRINT(Debug,11,(&Debug, 
			   "elm_decode: Readed %d bytes from temp file\n",
			   count));
      }
      fclose(tmpfp);
  }

  DPRINT(Debug,11,(&Debug, 
		   "elm_decode <- END; \n"));
  if (in_state_seekable(state_in)) {
      DPRINT(Debug,11,(&Debug, 			
		       "           (file); ftell=%ld\n",
		       in_state_ftell(state_in))); 
  }

  in_state_destroy(&newstate2);
}
 /* Prototype */
static void text_unsupported_decode P_((mime_t *, 
					in_state_t *, out_state_t *,
					charset_t));

static void text_unsupported_decode (ptr, state_in, state_out, defcharset) 
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{
    DPRINT(Debug,11,(&Debug, 
		     "text_unsupported_decode -> state: offset=%ld, length=%ld; \n",
		     (long) ptr -> offset, (long) ptr -> length));

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"text_unsupported_decode",
		   "Bad magic number");

    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "                          (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }

    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
       conversions */

    state_printf(state_out,
		 CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedText,
			 "[ %s/%.30s is unsupported, treating like TEXT/PLAIN ]\n"), 
		 TYPE(ptr->type), ptr->subtype);
    state_nlputs("\n",state_out);
    text_decode (ptr, state_in, state_out, defcharset);

    DPRINT(Debug,11,(&Debug, 
		     "text_unsupported_decode <- END; \n"));
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "                        (file); ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }
}

void null_decode (ptr, state_in, state_out, defcharset) 
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{
    char *buf = NULL;

    DPRINT(Debug,11,(&Debug, 
		     "null_decode <-> state: offset=%ld, length=%ld; \n",
		     (long) ptr -> offset, (long) ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"null_decode",
		   "Bad magic number");
    
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "               (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }

    state_printf (state_out,
		  CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedSkip,
			  "[ %.15s/%.30s is not supported, skipping... ]\n"),
		  TYPE(ptr->type),ptr->subtype);
    if (state_out->displaying)
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeUseVtosave,
			     "[ Use 'v' to view or save this part. ]\n"));
}

CT_decoder_t select_CT_decoder (ptr) 
     mime_t *ptr;
{
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"select_CT_decoder",
		   "Bad magic number");

    if (ptr->type == MIME_TYPE_MULTIPART) {
	if (ptr->flags & MIME_ALTERNATIVE) {
	    DPRINT(Debug,11,(&Debug, 
			     "select_CT_decoder [%s/%s] = alternative_decode\n",
			     mime_types[ptr->type], ptr->subtype
			     ));
	    return alternative_decode;
	} else if (ptr->flags & MIME_SIGNED) {
	    DPRINT(Debug,11,(&Debug, 				
			     "select_CT_decoder [%s/%s] = signed_decode\n",
			     mime_types[ptr->type], ptr->subtype
			     ));
	    return signed_decode;
	} else if (ptr->flags & MIME_ENCRYPTED) {
	    DPRINT(Debug,11,(&Debug, 
			    "select_CT_decoder [%s/%s] = encrypted_decode\n",
			    mime_types[ptr->type], ptr->subtype
			    ));
	    return encrypted_decode;
	} else {
	    DPRINT(Debug,11,(&Debug, 
			     "select_CT_decoder [%s/%s] = multipart_0_decode\n",
			     mime_types[ptr->type], ptr->subtype));
	    return multipart_0_decode;
	}
    } else if (ptr->flags & MIME_RFC822) {
	DPRINT(Debug,11,(&Debug, 				
			 "select_CT_decoder [%s/%s] = rfc822_decode\n",
			 mime_types[ptr->type], ptr->subtype));
	return rfc822_decode;
    }
    else if (ptr->type == MIME_TYPE_MESSAGE &&
	     (0 == istrcmp(ptr->subtype, "delivery-status") ||
	      0 == strincmp(ptr->subtype,"X-delivery-status-",18))) {
	DPRINT(Debug,11,(&Debug, 
			 "select_CT_decoder [%s/%s] = text_decode\n",
			 mime_types[ptr->type], ptr->subtype));
	return text_decode;
    }
    else if (ptr->type == MIME_TYPE_TEXT) {
	if (0 == istrcmp(ptr->subtype, "rfc822-headers") ||
	    0 == istrcmp(ptr->subtype, "plain")) {
	    DPRINT(Debug,11,(&Debug, 
			     "select_CT_decoder [%s/%s] = text_decode\n",
			     mime_types[ptr->type], ptr->subtype));
	    return text_decode;
	}
#ifdef USE_PGP
	if (0 == istrcmp(ptr->subtype, "x-pgp")) {
	    DPRINT(Debug,11,(&Debug, 
			     "select_CT_decoder [%s/%s] = pgp_decode\n",
			     mime_types[ptr->type], ptr->subtype));
	    return pgp_decode;
	}
#endif
	DPRINT(Debug,11,(&Debug, 
			 "select_CT_decoder [%s/%s] = text_unsupported_decode\n",
			 mime_types[ptr->type], ptr->subtype));
	return text_unsupported_decode; /* For that type we try use
					   metamail if available! */
    }
#ifdef USE_PGP
    else if (ptr->type == MIME_TYPE_APPLICATION &&
	     (istrcmp(ptr->subtype, "pgp") == 0)) {
	DPRINT(Debug,11,(&Debug, 
			 "select_CT_decoder [%s/%s] = pgp_decode\n",
			 mime_types[ptr->type], ptr->subtype));
	return pgp_decode;
    }
    else if (ptr->type == MIME_TYPE_APPLICATION &&
	     (strincmp(ptr->subtype,"x-pgp",5) == 0)) {
       	DPRINT(Debug,11,(&Debug, 
			 "select_CT_decoder [%s/%s] = pgp_decode\n",
			 mime_types[ptr->type], ptr->subtype));
	return pgp_decode;
    }
#endif
    else if (ptr->type == MIME_TYPE_APPLICATION &&
	     (istrcmp(ptr->subtype, "X-ELM-encode") == 0)) {
	DPRINT(Debug,11,(&Debug, 
			 "select_CT_decoder [%s/%s] = elm_decode\n",
			 mime_types[ptr->type], ptr->subtype));
	return elm_decode;
    }
  
    DPRINT(Debug,11,(&Debug, 
		     "select_CT_decoder [%s/%s] = null_decode\n",
		     mime_types[ptr->type], ptr->subtype));
    return null_decode;
}


void mime_decode (ptr, state_in, state_out, defcharset)
     mime_t *ptr;
     in_state_t *state_in;
     out_state_t *state_out;
     charset_t defcharset;
{
    int j;
    /* This routine calls the appropriate routine to decode the data
     * described in "ptr".
     */
    
    DPRINT(Debug,11,(&Debug, 
		     "mime_decode -> state: offset=%ld, length=%ld -- ",
		     (long) ptr -> offset, (long) ptr -> length));
    for (j = 0; state_out->display_charset[j]; j++) {
	DPRINT(Debug,11,(&Debug, 
			 "'display charset[%d]'=%s ",j,
			 state_out->display_charset[j]->MIME_name ? 
			 state_out->display_charset[j]->MIME_name :
			 "<no MIME name>"));
    }
    DPRINT(Debug,11,(&Debug, 
		     ", header charset='%s'\n",
		     defcharset->MIME_name ?
		     defcharset->MIME_name :
		     "<no MIME name>"));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"mime_decode",
		   "mime_decode: unsupported input state");
    }
    
    if (ptr->disposition == DISP_INLINE) {
	CT_decoder_t decoder = select_CT_decoder(ptr);
    
	DPRINT(Debug,11,(&Debug, 
			 "mime_decode: decoder=%p\n",decoder));
	
	if (in_state_fseek(state_in,ptr->offset) != 0) {
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */

	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeMimeSeekFailed,
				 "mime_decode: seek failed"));
	    state_nlputs(" ]\n",state_out);
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMimeSeekFailed,
			      "mime_decode: seek failed"));
	    return;
	}

	decoder (ptr, state_in, state_out, defcharset);    

    } else { 
	if (state_out->displaying)      
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeAttachUseV,
				 "[ Attachment, skipping...  Use 'v' to view this part. ]\n"));
	else
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeAttachSkipping,
				 "[ Attachment, skipping... ]\n"));
    }
    
    DPRINT(Debug,11,(&Debug, 
		     "mime_decode <- END\n"));
}

int is_rfc1522 (s)
     char *s;
{
  /* Returns TRUE if the string contains RFC 1522 format encoded data */

  while ((s = strchr (s, '=')) != NULL) {
    s++;
    if (*s == '?')
      return TRUE;
  }
  return FALSE;
}

static mime_t * best_alternative(att)     
     mime_t *att;
{
  mime_t *ret = att;
  if (att->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"best_alternative",
	       "Bad magic number");

  while (att) {
    if (!mime_notplain(att)) {
	DPRINT(Debug,9,(&Debug, 
			"-- best_alternative, found: %s/%s\n",
			mime_types[att->type], att->subtype));
      ret = att;
    }
    att = att->next;
  }

  if (ret) {
      DPRINT(Debug,9,(&Debug, 
		      "best_alternative - returns: %s/%s\n",
		      mime_types[ret->type], ret->subtype));
  } else {
      DPRINT(Debug,9,(&Debug, 
		      "best_alternative=NULL"));
  }

  return ret;
}

static int can_handle P_((mime_t *att));
static int can_handle(att)     
     mime_t *att;
{
  if (att->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"can_handle",
	       "Bad magic number");

  while (att) {
    if (att->magic != MIME_magic)
      mime_panic(__FILE__,__LINE__,"can_handle",
		 "Bad magic number (next -chain)");

    if (att->disposition == DISP_INLINE) {      
      if (mime_notplain(att)) {
	  DPRINT(Debug,9,(&Debug, 
			  "can_handle=NO -- Failed: %s/%s\n",
			  mime_types[att->type], att->subtype));
	return FALSE;
      }
      DPRINT(Debug,9,(&Debug, 
		      "can_handle:     Succeed: %s/%s\n",
		      mime_types[att->type], att->subtype));
    } else {
      	DPRINT(Debug,9,(&Debug, 
			"can_handle:  Attachment: %s/%s\n",
			mime_types[att->type], att->subtype));
    }
    att = att->next;
  }

  DPRINT(Debug,9,(&Debug, 
		  "can_handle=TRUE\n"));
  return TRUE;
}

int mime_notplain (p)
     mime_t *p;
{
    CT_decoder_t decoder = select_CT_decoder(p);
    mime_t *z;
    char buf[STRING];
    
    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_not_plain",
		   "Bad magic number");
    
    if (p->encoding < ENCODING_NONE ||
	p->encoding >= ENCODING_EXPERIMENTAL) {
	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=TRUE -- type: %s/%s; encoding=%s (%d)\n",
			mime_types[p->type], p->subtype,
			ENCODING(p->encoding),p->encoding));
    return TRUE;      
  }

    if (decoder == rfc822_decode && p->parts && can_handle(p->parts)) {
	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=FALSE -- type: %s/%s\n",
			mime_types[p->type], p->subtype));
	return FALSE;      
    }
    if (decoder == rfc822_decode) {
	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=TRUE -- type: %s/%s\n",
			mime_types[p->type], p->subtype));
	return TRUE;
    }
 
    /* Notice: if (decoder == text_unsupported_decode)
     *         we return TRUE, because we want call metamail
     *         if it is available
     */
 
    if (decoder == text_decode || decoder == elm_decode) {
	/* If mime_get_charsets < 0, we may want call metamail
	 * (instead of filtering to US-ASCII)
	 */
	int r;
	charset_t buf1;
	charset_t disp_vector[2];

	disp_vector[0] = display_charset;
	disp_vector[1] = NULL;
	
	if ((r=mime_get_charset(&buf1,p->type_opts,disp_vector)) > 0) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_notplain=FALSE -- type: %s/%s; charset=%s\n",
			    mime_types[p->type], p->subtype,
			    buf1->MIME_name ? buf1->MIME_name : 
			    "<no MIME name>"));
	    return FALSE;
	} 
	if (r < 0 && buf1 == system_charset) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_notplain=FALSE -- type: %s/%s; charset=%s -- is system charset ...\n",
			    mime_types[p->type], p->subtype,
			    buf1->MIME_name ? buf1->MIME_name : "<no MIME name>"));
	    return FALSE;
	}
	if (r < 0 && page_known_charsets) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_notplain=FALSE -- type: %s/%s; charset=%s -- page_known_charsets=TRUE\n",
			    mime_types[p->type], p->subtype,
			    buf1->MIME_name ? buf1->MIME_name : "<no MIME name>"));
	    return FALSE;
	}
    }
    else if (decoder == multipart_0_decode &&
	     ((p->flags & MIME_MIXED) || (p->flags & MIME_DIGEST)) &&
	     p->parts && can_handle(p->parts)) {
	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=FALSE -- type: %s/%s, flags=%d\n",
			mime_types[p->type], p->subtype, p->flags));
	return FALSE;	   
    }
    else if (decoder == alternative_decode &&
	     p->parts && ( z= best_alternative(p->parts)) && 
	     (pagealternative ? !mime_notplain(z) : can_handle(p->parts))) {
	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=FALSE -- type: %s/%s, "
			"best_alternative: %s/%s\n",
			mime_types[p->type], p->subtype, 
			mime_types[z->type], z->subtype));
	return FALSE;
    } else if (decoder == signed_decode && 
	       p->parts && !mime_notplain(p->parts)) {
	SG_decoder_t sg_decode;
	if (!mime_get_param ("protocol", buf, p->type_opts, 
			     sizeof buf)) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_notplain=TRUE -- type: %s/%s, flags=%d -- no 'protocol'\n",
			    mime_types[p->type], p->subtype, p->flags));
	    return TRUE;	   	    
	}
	sg_decode = select_SG_decoder(buf);

	if (sg_decode != null_SG_decoder) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_notplain=FALSE -- type: %s/%s, flags=%d, protocol=%s\n",
			    mime_types[p->type], p->subtype, p->flags, buf));
	    return FALSE;
	} else if (pagesigned) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_notplain=FALSE -- type: %s/%s, flags=%d, protocol=%s, pagesigned=TRUE\n",
			    mime_types[p->type], p->subtype, p->flags, buf));
	    return FALSE;
	}
	
	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=TRUE -- type: %s/%s, flags=%d, protocol=%s\n",
			mime_types[p->type], p->subtype, p->flags, buf));
	return TRUE;	
    } else if (decoder == encrypted_decode && p->parts) {
	EC_decoder_t ec_decode;
	if (!mime_get_param ("protocol", buf, p->type_opts, 
			     sizeof buf)) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_notplain=TRUE -- type: %s/%s, flags=%d -- no 'protocol'\n",
			    mime_types[p->type], p->subtype, p->flags));
	    return TRUE;	   	    
	}
	ec_decode = select_EC_decoder(buf);

	if (ec_decode != null_EC_decoder) {
	    DPRINT(Debug,9,(&Debug, 
			    "mime_notplain=FALSE -- type: %s/%s, flags=%d, protocol=%s\n",
			    mime_types[p->type], p->subtype, p->flags, buf));
	    return FALSE;
	} 

	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=TRUE -- type: %s/%s, flags=%d, protocol=%s\n",
			mime_types[p->type], p->subtype, p->flags, buf));
	return TRUE;	
    } else if (p->type == MIME_TYPE_MULTIPART && pagemultipart) {
      	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=FALSE -- type: %s/%s "
			"(pagemultipart=TRUE)\n",
			mime_types[p->type], p->subtype));
	return FALSE;
    }
#ifdef USE_PGP
    else if (decoder == pgp_decode) {
	DPRINT(Debug,9,(&Debug, 
			"mime_notplain=FALSE -- type: %s/%s\n",
			mime_types[p->type], p->subtype));
	return FALSE;
    }
#endif
    DPRINT(Debug,9,(&Debug, 
		    "mime_notplain=TRUE -- type: %s/%s\n",
		    mime_types[p->type], p->subtype));
    return TRUE;
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
