static char rcsid[] = "@(#)$Id: iso2022.c,v 1.5 2001/06/06 18:09:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.5 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *****************************************************************************/

#include "headers.h"
#include "s_me.h"
#include "cs_imp.h"

DEBUG_VAR(Debug,__FILE__,"charset");

struct iso2022_setid return_to_iso2022 = 
{ bank_unspecified, iso2022_other , { 0x40 , 0x00, 0x00, 0x00 } };

static struct iso2022_setid set0_utf8 =
{ bank_unspecified, iso2022_other , { 0x47 , 0x00, 0x00, 0x00 } };

static struct iso2022_setid set1_latin8 = /* ISO-8859-14 */
{ bank_unspecified, iso2022_96, { 0x5F, 0x00, 0x00, 0x00 } };   

static struct iso2022_setid latin_banks[] = {
    { bank_G0, iso2022_94, { 0x42, 0x00, 0x00, 0x00 } },   /* ASCII */
    { bank_unspecified, iso2022_96, { 0x41, 0x00, 0x00, 0x00 } },   /* 1 */ 
    { bank_unspecified, iso2022_96, { 0x42, 0x00, 0x00, 0x00 } },   /* 2 */ 
    { bank_unspecified, iso2022_96, { 0x43, 0x00, 0x00, 0x00 } },   /* 3 */ 
    { bank_unspecified, iso2022_96, { 0x44, 0x00, 0x00, 0x00 } },   /* 4 */ 
    { bank_unspecified, iso2022_96, { 0x4C, 0x00, 0x00, 0x00 } },   /* 5 */
    { bank_unspecified, iso2022_96, { 0x47, 0x00, 0x00, 0x00 } },   /* 6 */
    { bank_unspecified, iso2022_96, { 0x46, 0x00, 0x00, 0x00 } },   /* 7 */
    { bank_unspecified, iso2022_96, { 0x48, 0x00, 0x00, 0x00 } },   /* 8 */
    { bank_unspecified, iso2022_96, { 0x4D, 0x00, 0x00, 0x00 } },   /* 9 */
};

struct iso2022_setid * CONST ASCII_BANK = &latin_banks[0];

static int same_setid P_((const struct iso2022_setid A,
			 const struct iso2022_setid B));
static int same_setid(A,B)
     CONST struct iso2022_setid A;
     CONST struct iso2022_setid B;
{
    int i;
    if (A.bank != B.bank)
	return 0;
    if (A.type != B.type)
	return 0;

    for (i = 0; i < sizeof A.bytes; i++)
	if (A.bytes[i] != B.bytes[i])
	    return 0;
    return 1;
}

static struct iso2022_setid  * loc_setid P_((const struct iso2022_setid c));
static struct iso2022_setid  * loc_setid(c)
     CONST struct iso2022_setid c;
{
    static struct iso2022_setid  **setid_list = NULL;
    static int setid_count = 0;

    struct iso2022_setid * ret;

    int i;

    if (same_setid(return_to_iso2022,c)) {
	ret = &return_to_iso2022;
	goto found;
    }

    for (i = 0; i < sizeof latin_banks / sizeof (latin_banks[0]); i++)
	if (same_setid(latin_banks[i],c)) {
	    ret = &(latin_banks[i]);
	    goto found;
	}

    for (i = 0; i < setid_count; i++)
	if (same_setid(*(setid_list[i]),c)) {
	    ret = setid_list[i];
	    goto found;
	}

    /* Create new setid */

    ret = safe_malloc(sizeof (struct iso2022_setid));
    *ret = c;

    setid_list = safe_realloc(setid_list, 
			      (setid_count+1) * 
			      sizeof (struct iso2022_setid *));
    setid_list[setid_count++] = ret;

    DPRINT(Debug,25,(&Debug, 
		     "loc_setid: Adding new id to list, count %d",
		     setid_count++));

 found:
    
    DPRINT(Debug,25,(&Debug, 
		 "loc_setid=%p\n",ret));
    
    return ret;
}

struct setlist set_utf8 = { &set0_utf8, NULL };
struct setlist set_latin8 = { &(latin_banks[0]), &set1_latin8 };

struct setlist sets_iso_8859_X [] = { 
    { &(latin_banks[0]), NULL },
    { &(latin_banks[0]), &(latin_banks[1]), NULL },
    { &(latin_banks[0]), &(latin_banks[2]), NULL },
    { &(latin_banks[0]), &(latin_banks[3]), NULL },
    { &(latin_banks[0]), &(latin_banks[4]), NULL },
    { &(latin_banks[0]), &(latin_banks[5]), NULL },
    { &(latin_banks[0]), &(latin_banks[6]), NULL },
    { &(latin_banks[0]), &(latin_banks[7]), NULL },
    { &(latin_banks[0]), &(latin_banks[8]), NULL },
    { &(latin_banks[0]), &(latin_banks[9]), NULL },
    { &(latin_banks[0]), &(latin_banks[10]), NULL },
};

static int same_setlist P_((const struct setlist A,
			 const struct setlist B));
static int same_setlist(A,B)
     CONST struct setlist A;
     CONST struct setlist B;
{
    int i;

    for (i = 0; 
	 i < sizeof (A.sets) / sizeof (A.sets[0]);
	 i++)
	if (A.sets[i] != B.sets[i])
	    return 0;

    return 1;
}

struct setlist *loc_setlist(l)
     CONST struct setlist l;
{
    static struct setlist ** setlists = NULL;
    static int setlist_count          = 0;
    struct setlist *ret;

    int i;

    for (i = 0; 
	 i < sizeof sets_iso_8859_X / sizeof (sets_iso_8859_X[0]); 
	 i++) {
	if (same_setlist(sets_iso_8859_X[i],l)) {
	    ret = &(sets_iso_8859_X[i]);
	    goto found;
	}
    }

    for (i = 0; i < setlist_count; i++) 
	if (same_setlist(*(setlists[i]),l)) {
	    ret = setlists[i];
	    goto found;
	}
	
    ret = safe_malloc(sizeof (struct setlist));
    *ret = l;
    
    setlists = safe_realloc(setlists,
			    (setlist_count+1) * sizeof (struct setlist *));
    setlists[setlist_count++] = ret;

    DPRINT(Debug,25,(&Debug, 
		     "loc_setlist: Addind new setlist to list, count %d\n",
		     setlist_count));

 found:
    DPRINT(Debug,25,(&Debug, 
		     "loc_setlist=%p\n",ret));
    return ret;
}

char * iso2022_setid_stream(c)
     CONST struct iso2022_setid  c;
{
    int len,j;
    char * ret;
    int idx = 0;

    for (len = 0; len < sizeof c.bytes; len++)
	if (!c.bytes[len])
	    break;
    len += 4;       /* ESC {twobyte} bank {bytes} */

    ret = safe_malloc(len+1);

#define ADD(x) if (idx >= len) \
 panic("ISO2022 PANIC",__FILE__,__LINE__,"iso2022_setid_stream","Overlow",0);\
 else { ret[idx++]=(x); }

    ADD(0x1B);                /* ESC */
    
    switch(c.type) {
    case iso2022_other:
	ADD(0x25);
	break;
    case iso2022_94x94: 
	ADD(0x24);
	/* FALLTHRU */
    case iso2022_94:
	switch(c.bank) {	    
	case bank_G0: ADD(0x28); break;
	case bank_G1: ADD(0x29); break;
	case bank_G2: ADD(0x2A); break;
	case bank_G3: ADD(0x2B); break;
	}
	break;
    case iso2022_96x96:
	ADD(0x24);
	/* FALLTHRU */
    case iso2022_96:
	switch(c.bank) {
	case bank_G1: ADD(0x2D); break;
	case bank_G2: ADD(0x2E); break;
	case bank_G3: ADD(0x2F); break;
	}
	break;	
    }
 
    for (j = 0; j < sizeof c.bytes && c.bytes[j]; j++) {
	ADD(c.bytes[j]);
    }        
#undef ADD

    ret[idx] = '\0';
    return ret;
}

char * lock_shift(r,bank)
     int r; 
     int bank;
{
    int len = 3;
    int idx = 0;
    char * ret = safe_malloc(len+1);

#define ADD(x) if (idx >= len) \
  panic("ISO2022 PANIC",__FILE__,__LINE__,"lock_shift","Overlow",0);\
  else { ret[idx++]=(x); }

    switch(r) {
    case 0:
	switch (bank) {
	case bank_G0:     ADD(0x0F);   /* SI */ break;
	case bank_G1:     ADD(0x0E);   /* SO */ break;
	case bank_G2:     ADD(0x1B);   ADD(0x6E);  break;
	case bank_G3:     ADD(0x1B);   ADD(0x6F);  break;
	}
	break;
    default:
	switch (bank) {
	case bank_G1:     ADD(0x1B);   ADD(0x7E);   break;
	case bank_G2:     ADD(0x1B);   ADD(0x7D);   break;
	case bank_G3:     ADD(0x1B);   ADD(0x7C);   break;
	}
    }
#undef ADD

    ret[idx] = '\0';
    return ret;
}

static struct iso2022_keyword {
    CONST char *          keyword;
    enum iso2022_bank     bank; 
    enum iso2022_settype  type;
} iso2022_keywords[] = {
    { "other-set",   bank_unspecified,  iso2022_other },

    /* Sets without bank specification */
    { "bank-94",    bank_unspecified,  iso2022_94 },
    { "bank-96",    bank_unspecified,  iso2022_96 },
    { "bank-94x94", bank_unspecified,  iso2022_94x94 },
    { "bank-96x96", bank_unspecified,  iso2022_96x96 },

    /* Bank G0 */
    { "bank-G0-94",    bank_G0,  iso2022_94 },
    { "bank-G0-94x94", bank_G0,  iso2022_94x94 },

    /* Bank G1 */
    { "bank-G1-94",    bank_G1,  iso2022_94 },
    { "bank-G1-96",    bank_G1,  iso2022_96 },
    { "bank-G1-94x94", bank_G1,  iso2022_94x94 },
    { "bank-G1-96x96", bank_G1,  iso2022_96x96 },

    /* Bank G2 */
    { "bank-G2-94",    bank_G2,  iso2022_94 },
    { "bank-G2-96",    bank_G2,  iso2022_96 },
    { "bank-G2-94x94", bank_G2,  iso2022_94x94 },
    { "bank-G2-96x96", bank_G2,  iso2022_96x96 },

    /* Bank G3 */
    { "bank-G3-94",    bank_G3,  iso2022_94 },
    { "bank-G3-96",    bank_G3,  iso2022_96 },
    { "bank-G3-94x94", bank_G3,  iso2022_94x94 },
    { "bank-G3-96x96", bank_G3,  iso2022_96x96 },

    { NULL, bank_unspecified,  iso2022_other }
};

/* Special return values:    0    Not a ISO2022 ident
 *                          -1    Incomplete ident
 *                          > 0   len of ident
 */

static int eat_ISO2022_ident P_((CONST unsigned char *buffer, int len)); 
static int eat_ISO2022_ident(buffer,len)			      
     CONST unsigned char *buffer;
     int len;
{
    int i = 0;

    while (i < len && buffer[i] >= 0x20 && buffer[i] <= 0x3F) {
	DPRINT(Debug,60,(&Debug, 
			 "eat_ISO2022_ident: accept [%d]=%02X\n",
			 i,buffer[i]));
	i++;
    }
    if (i < len && buffer[i] >= 0x40 && buffer[i] <= 0x7D) {
	DPRINT(Debug,60,(&Debug, 
			 "eat_ISO2022_ident=%d: final [%d]=%02X\n",
			 i+1,i,buffer[i]));
	i++;
	return i;
    }
    if (i == len) {
	DPRINT(Debug,60,(&Debug,  
			 "eat_ISO2022_ident=-1: Too short %d\n",
		     i));
	return -1;
    }
    DPRINT(Debug,60,(&Debug, 
		     "eat_ISO2022_ident=0: reject [%d]=%02X\n",
		     i,buffer[i]));
    return 0;
}

int parse_gen_spec(val,bytes,size)
     CONST char *val; 
     unsigned char *bytes;
     int size;
{
    int ret = 0;
    int ptr = 0;
    char * temp = safe_strdup(val);
    char *wrk;

    for (wrk = strtok(temp," \t,"); wrk; wrk = strtok(NULL," \t,")) {
	char *end;
	long l = strtol(wrk,&end,10); 
	
	if (ptr == size) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeTooLongident,
			      "Too long ident: %.20s..."),
		      temp);
	    ret = 0;
	    goto fail;
	}

	if ((*end == '\0')  &&
	    l >= 0x01 && l <= 0x7F) {
	    bytes[ptr] = l;
	    DPRINT(Debug,11,(&Debug, 
			     "parse_gen_ident_spec: [%d]=%02X\n",
			     ptr,bytes[ptr]));
	    ptr++;
	} else if (*end == '/' && l >= 0 && l <= 15 &&
		   *(end+1) != '\0') {	    
	    long l1; 

	    wrk = end+1;
	    l1 = strtol(wrk,&end,10);
		
	    if ((*end == '\0') &&
		l1 >= 0 && l1 <= 15 &&
		(l != 0 || l1 != 0)) {	    
		bytes[ptr] = l*16 + l1;
		DPRINT(Debug,11,(&Debug, 
				 "parse_gen_ident_spec: [%d]=%02X\n",
				 ptr,bytes[ptr]));
		ptr++;
	    } else {
		DPRINT(Debug,1,(&Debug, 
				"parse_gen_ident_spec: wrk=%s, *end=%c,l1=%ld\n",
				wrk,*end,l1));
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeBadident,
				  "Bad ident: %.20s..."),
			  wrk);
		ret = 0;
		goto fail;
	    }
	} else {
	    DPRINT(Debug,1,(&Debug, 
			"parse_gen_ident_spec: wrk=%s, *end=%c,l=%ld\n",
			wrk,*end,l));

	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeBadident,
			      "Bad ident: %.20s..."),
		      wrk);
	    ret = 0;
	    goto fail;
	}
    }
    ret = ptr;

 fail:
    free(temp);
    while (ptr < size)
	bytes[ptr++] = '\0';    
    return ret;
}
			      

static int parse_ISO2022_ident_spec P_((const char *val, 
					unsigned char *bytes,
					int size));

static int parse_ISO2022_ident_spec(val,bytes,size)
     CONST char *val;
     unsigned char * bytes;
     int size;
{
    int ret = 0, ptr,l;

    DPRINT(Debug,11,(&Debug, 
		     "parse_ISO2022_ident_spec: val=%s\n",val));

    ptr = parse_gen_spec(val,bytes,size);
    
    if (ptr) {
	ret = 1;
	l = eat_ISO2022_ident(bytes,ptr);
	if (l != ptr) {
	    DPRINT(Debug,1,(&Debug, 
			    "parse_ISO2022_ident_spec: eat_ISO2022_ident=%d, ptr=%d\n",
			    l,ptr));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeBadISO2022ident,
			      "Bad ISO2022 ident: %.20s..."),
		      val);
	    ret = 0;
	}
    }

    return ret;
}

int parse_iso2022_specification(param,value,count,list)
     CONST char     *param;
     CONST char     *value;
     int            *count;
     struct setlist *list; 
{
    int i;
    struct iso2022_setid  TMP;

    for (i = 0; iso2022_keywords[i].keyword; i++)
	if (0 == strcmp(iso2022_keywords[i].keyword,param))
	    break;

    if (!iso2022_keywords[i].keyword)
	return 0;

    if ((*count) >= sizeof (list->sets) / sizeof (list->sets[0])) {
	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeTooManyBank,
			  "Too many banks on charset: %.20s"),
		  param);
	return 0;
    }
    
    TMP.bank = iso2022_keywords[i].bank;
    TMP.type = iso2022_keywords[i].type;
    
    if (!parse_ISO2022_ident_spec(value,TMP.bytes,sizeof (TMP.bytes)))
	return 0;

    if (*count > 0 &&
	list->sets[0]->type == iso2022_other) {

	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeOtherSetIncompatible,
			  "Param other-set is incompatible with param %s"),
		  param);
	return 0;
    }

    list->sets[(*count)++] = loc_setid(TMP);
    
    return 1;
}

char * iso2022_codestr(bytes,size)
     unsigned char * bytes;
     int size;
{
    char *ret = NULL;
    int i;

    for (i = 0; i < size && bytes[i]; i++) {
	char temp[10];
	int val = (unsigned char) bytes[i];
	sprintf(temp,"%d/%d",val/16,val%16);
	if (ret)
	    ret = strmcat(ret," ");
	ret = strmcat(ret,temp);
    }
    return ret;
}


void print_setlist(f,l)
     FILE *f;
     struct setlist *l;
{
    int i;

    for (i = 0; l->sets[i]; i++) {
	int j;

	for (j = 0; iso2022_keywords[j].keyword; j++) {
	    if (iso2022_keywords[j].bank == l->sets[i]->bank &&
		iso2022_keywords[j].type == l->sets[i]->type) {
		char * t = iso2022_codestr(l->sets[i]->bytes,
					   sizeof l->sets[i]->bytes);
		if (t) {
		    if (0 == strchr(t,' '))
			elm_fprintf(f,FRM(";%s=%s"),
				    iso2022_keywords[j].keyword,
				    t);
		    else
			elm_fprintf(f,FRM(";%s=%Q"),
				    iso2022_keywords[j].keyword,
				    t);
		    free(t);
		}
	    }
	}

    }
}
	
/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */


