/***********************************************************
        Copyright 1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[]="$Id: Lock.c,v 1.7 1995/03/20 20:46:00 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */


/*
 * routines for manipulating the depot.lock file 
 * 
 * most routines:
 *   returns NULL on success
 *   returns NULL and PROGRAM_ErrorNo is set on system failure
 *   returns userid if failed to change the lock
 * Lock_QueryLock returns
 *  NULL if the file is empty
 *  the userid in the file, otherwise
 *
 * This code is based on the Lock stuff written by Sohan. I just simplified things a bit
 * 
 * WARNING: These routines are not thread safe.
 */



/*
 * Author: Walter Wong
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "Lock.h"
#include "File.h"

static char *
QueryUser() {
  static char user[LOCKSMITHNAMESIZE];
  struct passwd *pw;
  
  if ((pw = getpwuid(getuid())) == NULL) {
      FatalError(E_BADUSER, "Attempt to unset lock by unknown user\n");
  } 

  (void)strcpy(user, pw->pw_name); 
  return user;
}

char *
Lock_QueryLock(LockFile)
  char *LockFile;
{
  int lockfd;
  static char locker[LOCKSMITHNAMESIZE];

  if ((lockfd = open(LockFile, O_RDONLY)) < 0) {
    if (errno == ENOENT) {
      return NULL;
    }
    FatalError(E_OPENFAILED, "Unable to open lock file %s: %s\n",
	       LockFile, strerror(errno));
    return NULL;
  }
  if (read(lockfd, locker, LOCKSMITHNAMESIZE) < 0) {
    FatalError(E_READFAILED, "Unable to read lockfile: %s: %s\n",
	       LockFile, strerror(errno));
    (void)close(lockfd);
    return NULL;
  }
  if (close(lockfd) < 0) {
    FatalError(E_CLOSEFAILED, "Unable to close lockfile: %s: %s\n",
	       LockFile, strerror(errno));
  }
  return locker;
}
  

char *
Lock_SetLock(LockFile)
  char *LockFile;
{
  char *Current_Locker, *User;

  if ((User = QueryUser()) == NULL) {
    return NULL;
  }
 
  Current_Locker = Lock_QueryLock(LockFile);
  if (PROGRAM_ErrorNo != E_NULL) {
    return NULL;
  }

  if (Current_Locker == NULL) {
    int lockfd;

    if ((lockfd = open(LockFile, O_EXCL | O_CREAT | O_WRONLY, 0777)) < 0) {
      FatalError(E_CANTLOCK, "Unable to set write lock file: %s: %s\n", 
		 LockFile, strerror(errno));
      return NULL;
    }
    if (write(lockfd, User, strlen(User)) < 0) {
      FatalError(E_WRITEFAILED, "Unable to write to lock file: %s: %s\n",
		 LockFile, strerror(errno));
    }
    if (close(lockfd) < 0) {
      FatalError(E_CLOSEFAILED, "Unable to close lockfile: %s: %s\n",
		 LockFile, strerror(errno));
    }
  } else if (strcmp(Current_Locker, User) == 0) {
    Current_Locker = NULL; 
  }
  
  return Current_Locker;
}


void
Lock_RemoveLock(LockFile) 
  char *LockFile;
{
  if (unlink(LockFile) < 0) {
    FatalError(E_FOOLPROOFLOCK, "Unable to reset lockfile: %s: %s\n",
	       LockFile, strerror(errno));
  }
}

  
char *
Lock_UnsetLock(LockFile)
  char *LockFile;
{
  char *Locker, *User;

  Locker = Lock_QueryLock(LockFile);

  if ((PROGRAM_ErrorNo != E_NULL) || (Locker == NULL)) {
    return NULL;
  }
  
  User = QueryUser();
  if (strcmp(User, Locker) == 0) {
    Lock_RemoveLock(LockFile);
    if (PROGRAM_ErrorNo != E_NULL) {
      return Locker;
    }
    return NULL;
  }
  
  return Locker;
}

/* This routine checks the time of the lock file and if it is older than
 * EXPIRE_LOCKTIME then it returns TRUE, else it returns FALSE
 */
Boolean
Lock_TestLock(LockFile) 
  char *LockFile;
{
  FILESTAT *fs;
  time_t now = time(0);

  if (File_GetStatus(LockFile, &fs, 0) < 0) {
    FatalError(E_GETSTATUSFAILED, "Unable to check lockfile: %s: %s\n", 
	       LockFile, strerror(errno));
    return FALSE;
  }

  if ((now - DEPOT_EXPIRE_LOCKFILE) > FILESTAT_MTime(fs)) 
    return TRUE; 
  else
    return FALSE;
}
  

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/022/src/lib/Lock/RCS/Lock.c,v $ */
