/***********************************************************
        Copyright 1991,1994 by Carnegie Mellon University

                      All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of CMU not be
used in advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

CMU DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
CMU BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
SOFTWARE.
******************************************************************/


#ifndef SABER
#ifndef LINT
static char rcs_id[] = "$Id: DepotConfUtil.c,v 1.7 1994/09/06 15:11:45 ww0r Exp $";
#endif /* LINT */
#endif /* SABER */

/*
 * Author: Sohan C. Ramakrishna Pillai
 */

#include "depotlib.h"

#include "util.h"
#include "DepotErrorCodes.h"
#include "DepotConf.h"

static struct DepotConfPair {
  char *specstring;
  unsigned spec;
} DepotConfList[] =

{
  {
    "map", DC_MAP
  },
  {
    "delete", DC_DELETE
  },
  {
    "command", DC_COMMAND
  },
  {
    "deletecommand", DC_DELETECOMMAND
  },
};

static unsigned NDepotConfTypes = sizeof(DepotConfList) / sizeof(DepotConfList[0]);

static DEPOTCONF *DepotConf();
static Boolean DepotConf_Comparator();


static struct DepotConfPair *
DepotConfPair_LocateByName(specname)
     char *specname;
{
  register struct DepotConfPair *dp;
  register unsigned int i;
  register Boolean FoundSpecName;

  i = 0;
  dp = DepotConfList;
  FoundSpecName = FALSE;
  while ((i < NDepotConfTypes) && !FoundSpecName) {
    if (strcmp(specname, dp->specstring) == 0)
      FoundSpecName = TRUE;
    else {
      i++, dp++;
    }
  }

  return (FoundSpecName == TRUE) ? dp : NULL;
}


static struct DepotConfPair *
DepotConfPair_LocateByType(spec)
     unsigned short spec;
{
  register struct DepotConfPair *dp;
  register unsigned int i;
  register Boolean FoundSpec;

  i = 0;
  dp = DepotConfList;
  FoundSpec = FALSE;
  while ((i < NDepotConfTypes) && !FoundSpec) {
    if (spec == dp->spec)
      FoundSpec = TRUE;
    else {
      i++, dp++;
    }
  }

  return (FoundSpec == TRUE) ? dp : NULL;
}


unsigned short 
DepotConf_SpecType(specname)
     char *specname;
{
  register struct DepotConfPair *dp;

  dp = DepotConfPair_LocateByName(specname);
  if (dp == NULL) {
    FatalError(E_BADDEPOTCONFSPECNAME,
	       "Unknown depot.conf specification %s\n", specname);
  }
  return (PROGRAM_ErrorNo == E_NULL) ? (dp->spec) : DC_UNKNOWN;
}


char *
DepotConf_SpecString(spec)
     unsigned short spec;
{
  register struct DepotConfPair *dp;

  dp = DepotConfPair_LocateByType(spec);
  if (dp == NULL) {
    FatalError(E_BADDEPOTCONFSPEC,
	       "Unknown depot.conf specification type %d\n", spec);
  }
  return (PROGRAM_ErrorNo == E_NULL) ? (dp->specstring) : NULL;
}


int 
DepotConfDB_Size(confdbp)
     DEPOTCONFDB *confdbp;
{
  register DEPOTCONF **dp;
  int size;

  if ((confdbp == NULL) || (*confdbp == NULL))
    size = 0;
  else {
    dp = *confdbp;
    while (*dp != NULL)
      dp++;
    size = dp - *confdbp;
  }

  return size;
}


static DEPOTCONF *
DepotConf(confitemp)
     DEPOTCONF *confitemp;
{
  DEPOTCONF *newconfitemp;

  if (confitemp == NULL)
    newconfitemp = NULL;
  else {
    newconfitemp = (DEPOTCONF *) emalloc(sizeof(DEPOTCONF));
    if (PROGRAM_ErrorNo == E_NULL) {
      DEPOTCONF_CollectionName(newconfitemp) =
	String(DEPOTCONF_CollectionName(confitemp),
	       strlen(DEPOTCONF_CollectionName(confitemp)));
      DEPOTCONF_Spec(newconfitemp) = DEPOTCONF_Spec(confitemp);
      DEPOTCONF_ConfValue(newconfitemp) =
	StringArray(DEPOTCONF_ConfValue(confitemp),
		    StringArray_Size(DEPOTCONF_ConfValue(confitemp)));
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? newconfitemp : NULL;
}


DEPOTCONFDB *
DepotConfDB(confdbp)
     DEPOTCONFDB *confdbp;
{
  register int i;
  register DEPOTCONF **fp, **tp;

  DEPOTCONFDB *newconfdbp;
  int confdbsize;

  confdbsize = DepotConfDB_Size(confdbp) + 1;
  newconfdbp = (DEPOTCONFDB *) emalloc(sizeof(DEPOTCONFDB));

  if (PROGRAM_ErrorNo == E_NULL) {
    *newconfdbp = (DEPOTCONFDB) emalloc((unsigned int) confdbsize
					* sizeof(DEPOTCONF *));
  }
  for (i = 0, fp = *confdbp, tp = *newconfdbp;
       (PROGRAM_ErrorNo == E_NULL) && (i < (confdbsize - 1));
       i++, fp++, tp++) {
    *tp = DepotConf(*fp);
  }
  if (PROGRAM_ErrorNo == E_NULL) {
    tp = *newconfdbp + confdbsize - 1;
    *tp = NULL;
  }
  return (PROGRAM_ErrorNo == E_NULL) ? newconfdbp : NULL;
}


static Boolean 
DepotConf_Comparator(dc1, dc2)
     DEPOTCONF *dc1, *dc2;
{
  Boolean Comparison;

  if ((dc1 == NULL) || (dc2 == NULL)) {
    if ((dc1 == NULL) && (dc2 == NULL))
      Comparison = TRUE;
    else
      Comparison = FALSE;
  } else {
    if (String_Comparator(DEPOTCONF_CollectionName(dc1),
			  DEPOTCONF_CollectionName(dc2)) != 0)
      Comparison = FALSE;
    else if (DEPOTCONF_Spec(dc1) != DEPOTCONF_Spec(dc2))
      Comparison = FALSE;
    else if (StringArray_Comparator(DEPOTCONF_ConfValue(dc1),
				    DEPOTCONF_ConfValue(dc2)) != 0)
      Comparison = FALSE;
    else
      Comparison = TRUE;
  }

  return (PROGRAM_ErrorNo == E_NULL) ? Comparison : FALSE;
}


Boolean 
DepotConfDB_Comparator(confdb1, confdb2)
     DEPOTCONFDB *confdb1, *confdb2;
{
  register DEPOTCONF **dp1, **dp2;
  Boolean eoconfdb;
  Boolean Comparison;
  Boolean nulldb1 = FALSE;
  Boolean nulldb2 = FALSE;

  if ((confdb1 == NULL) || (*confdb1 == NULL))
    nulldb1 = TRUE;

  if ((confdb2 == NULL) || (*confdb2 == NULL))
    nulldb2 = TRUE;

  if (nulldb1 || nulldb2) {
    if (nulldb1 && nulldb2)
      Comparison = TRUE;
    else 
      Comparison = FALSE;
  } else {
    dp1 = *confdb1;
    dp2 = *confdb2;
    Comparison = TRUE;
    eoconfdb = FALSE;
    while ((PROGRAM_ErrorNo == E_NULL)
	   && (Comparison == TRUE) && !eoconfdb) {
      if ((*dp1 == NULL) || (*dp2 == NULL)) {
	if ((*dp1 == NULL) && (*dp2 == NULL))
	  Comparison = TRUE;
	else
	  Comparison = FALSE;
	eoconfdb = TRUE;
      } else {
	Comparison = DepotConf_Comparator(*dp1, *dp2);
	dp1++;
	dp2++;
      }
    }
  }

  return (PROGRAM_ErrorNo == E_NULL) ? Comparison : FALSE;
}

unsigned int
DepotConfTypes_Size()
{
  return NDepotConfTypes;
}

STRINGSET *
DepotConfDB_GetCommandList(confdbp)
     DEPOTCONFDB *confdbp;
{
  register DEPOTCONF **dp;

  STRINGSET *commandset;

  if (confdbp == NULL) {
    FatalError(E_BADDEPOTCONFDB,
	     "Attempt to get command list from NULL depot.conf database\n");
  }
  commandset = NULL;
  if ((PROGRAM_ErrorNo == E_NULL) && (*confdbp != NULL)) {
    for (dp = *confdbp;
	 (PROGRAM_ErrorNo == E_NULL) && (*dp != NULL);
	 dp++) {
      if ((DEPOTCONF_Spec(*dp) & DC_COMMAND)
	  && (DEPOTCONF_ConfValue(*dp) != NULL)
	  && (STRINGARRAY_Size(DEPOTCONF_ConfValue(*dp)) > 0)
	  && (STRINGARRAY_String(DEPOTCONF_ConfValue(*dp), 0) != NULL)) {
	commandset =
	  StringSet_AddString(commandset,
			      STRINGARRAY_String(DEPOTCONF_ConfValue(*dp),
						 0));
      }
    }
  }
  return (PROGRAM_ErrorNo == E_NULL) ? commandset : NULL;
}

/* $Source: /afs/andrew.cmu.edu/system/src/local/depot2/012/src/lib/DepotConf/RCS/DepotConfUtil.c,v $ */
