package ffmpreg_test

import (
	"context"
	"crypto/md5"
	"fmt"
	"io"
	"os"
	"path"
	"testing"

	"codeberg.org/gruf/go-ffmpreg/ffmpreg"
	"codeberg.org/gruf/go-ffmpreg/wasm"
	"github.com/tetratelabs/wazero"
)

var dir string

func init() {
	var err error

	// Get current test dir.
	dir, err = os.Getwd()
	if err != nil {
		panic(err)
	}

	// Determine base repo dir.
	dir = path.Join(dir, "../")

	fmt.Println("initializing ...")

	// Precompile so tests don't appear to hang.
	if err := ffmpreg.Initialize(); err != nil {
		panic(err)
	}

	fmt.Println("success.")
}

func TestClearMetadataJPEG(t *testing.T) {
	input := path.Join(dir, "testdata/this-is-fine.jpg")
	output := path.Join(dir, "testdata/this-is-fine-clean.jpg")
	check := path.Join(dir, "testdata/this-is-fine-check.jpg")
	testCleanMetadata(t, input, output, check)
}

func TestClearMetadataPNG(t *testing.T) {
	input := path.Join(dir, "testdata/waldorf.png")
	output := path.Join(dir, "testdata/waldorf_clean.png")
	check := path.Join(dir, "testdata/waldorf_check.png")
	testCleanMetadata(t, input, output, check)
}

func TestClearMetadataMP4(t *testing.T) {
	input := path.Join(dir, "testdata/this-is-fine-dog.mp4")
	output := path.Join(dir, "testdata/this-is-fine-dog-clean.mp4")
	check := path.Join(dir, "testdata/this-is-fine-dog-check.mp4")
	testCleanMetadata(t, input, output, check)
}

func TestClearMetadataOPUS(t *testing.T) {
	input := path.Join(dir, "testdata/sample1.opus")
	output := path.Join(dir, "testdata/sample1-clean.opus")
	check := path.Join(dir, "testdata/sample1-check.opus")
	testCleanMetadata(t, input, output, check)
}

func TestExtractThumbMP4(t *testing.T) {
	input := path.Join(dir, "testdata/this-is-fine-dog.mp4")
	output := path.Join(dir, "testdata/this-is-fine-dog-thumb.jpeg")
	check := path.Join(dir, "testdata/this-is-fine-dog-thumb-check.jpeg")
	test(t, []string{
		"-loglevel", "error",
		"-i", input,
		"-vf", "thumbnail=n=10",
		"-frames:v", "1",
		"-c:v", "mjpeg",
		"-fflags", "+bitexact",
		"-flags:v", "+bitexact",
		"-flags:a", "+bitexact",
		output,
	}, output, check)
}

func testCleanMetadata(t *testing.T, input, output, check string) {
	test(t, []string{
		"-loglevel", "error",
		"-i", input,
		"-map_metadata", "-1",
		"-c:v", "copy",
		"-c:a", "copy",
		"-fflags", "+bitexact",
		"-flags:v", "+bitexact",
		"-flags:a", "+bitexact",
		output,
	}, output, check)
}

func BenchmarkFfmpegCleanMetadata(b *testing.B) {
	b.Run("jpeg", func(b *testing.B) { benchmarkFfmpegCleanMetadata(b, []string{"testdata/this-is-fine.jpg"}) })
	b.Run("png", func(b *testing.B) { benchmarkFfmpegCleanMetadata(b, []string{"testdata/waldorf.png"}) })
	b.Run("mp4", func(b *testing.B) { benchmarkFfmpegCleanMetadata(b, []string{"testdata/this-is-fine-dog.mp4"}) })
	b.Run("opus", func(b *testing.B) { benchmarkFfmpegCleanMetadata(b, []string{"testdata/sample1.opus"}) })
}

func BenchmarkFfmpegTranscodeTo(b *testing.B) {
	for _, vcodec := range []string{
		"libwebp",
		"gif",
	} {
		b.Run("jpeg to "+vcodec, func(b *testing.B) { benchmarkFfmpegTranscodeTo(b, []string{"testdata/this-is-fine.jpg"}, vcodec) })
		b.Run("png to "+vcodec, func(b *testing.B) { benchmarkFfmpegTranscodeTo(b, []string{"testdata/waldorf.png"}, vcodec) })
		b.Run("mp4 to "+vcodec, func(b *testing.B) { benchmarkFfmpegTranscodeTo(b, []string{"testdata/this-is-fine-dog.mp4"}, vcodec) })
		b.Run("opus to "+vcodec, func(b *testing.B) { benchmarkFfmpegTranscodeTo(b, []string{"testdata/sample1.opus"}, vcodec) })
	}
}

func benchmarkFfmpegCleanMetadata(b *testing.B, inputs []string) {
	config := func(cfg wazero.ModuleConfig) wazero.ModuleConfig {
		fscfg := wazero.NewFSConfig()
		fscfg = fscfg.WithDirMount("/", "/")
		return cfg.WithFSConfig(fscfg)
	}

	for i, input := range inputs {
		inputs[i] = path.Join(dir, input)
	}

	b.ResetTimer()

	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			for _, input := range inputs {
				_, _ = ffmpreg.Ffmpeg(context.Background(), wasm.Args{
					Config: config,

					Stdin:  nil,
					Stdout: io.Discard,
					Stderr: io.Discard,

					Args: []string{
						"-loglevel", "error",
						"-i", input,
						"-map_metadata", "-1",
						"-c:v", "copy",
						"-c:a", "copy",
						"-fflags", "+bitexact",
						"-flags:v", "+bitexact",
						"-flags:a", "+bitexact",
						"-f", "null",
						"-",
					},
				})
			}
		}
	})
}

func benchmarkFfmpegTranscodeTo(b *testing.B, inputs []string, vcodec string) {
	config := func(cfg wazero.ModuleConfig) wazero.ModuleConfig {
		fscfg := wazero.NewFSConfig()
		fscfg = fscfg.WithDirMount("/", "/")
		return cfg.WithFSConfig(fscfg)
	}

	for i, input := range inputs {
		inputs[i] = path.Join(dir, input)
	}

	b.ResetTimer()

	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {
			for _, input := range inputs {
				_, _ = ffmpreg.Ffmpeg(context.Background(), wasm.Args{
					Config: config,

					Stdin:  nil,
					Stdout: io.Discard,
					Stderr: os.Stderr,

					Args: []string{
						"-loglevel", "error",
						"-i", input,
						"-frames:v", "1",
						"-codec:v", vcodec,
						"-f", "null",
						"-",
					},
				})
			}
		}
	})
}

func test(t *testing.T, args []string, output, check string) {
	defer os.Remove(output)

	// Create new test context.
	ctx := context.Background()
	ctx, cncl := context.WithCancel(ctx)
	defer cncl()

	// Run ffmpeg with given arguments.
	rc, err := ffmpreg.Ffmpeg(ctx, wasm.Args{
		Stdout: os.Stdout,
		Stderr: os.Stderr,
		Args:   args,
		Config: func(cfg wazero.ModuleConfig) wazero.ModuleConfig {
			fscfg := wazero.NewFSConfig()
			fscfg = fscfg.WithDirMount(dir, dir)
			return cfg.WithFSConfig(fscfg)
		},
	})
	if err != nil {
		t.Fatal(err)
	} else if rc != 0 {
		t.Fatal("non-zero exit code:", rc)
	}

	// Read output file into memory.
	ob, err := os.ReadFile(output)
	if err != nil {
		t.Fatal(err)
	}

	// Read 'check' into memory.
	cb, err := os.ReadFile(check)
	if err != nil {
		t.Fatal(err)
	}

	// Compare file md5 hash sums.
	if md5.Sum(ob) != md5.Sum(cb) {
		t.Fatal("unexpected output")
	}
}
