use forgejo_api::structs::Branch;
use miette::IntoDiagnostic;

use crate::actions::GlobalArgs;
use crate::render::json::JsonToStdout;
use crate::render::option::option_display;
use crate::render::spinner::spin_until_ready;
use crate::types::context::BergContext;
use crate::types::git::OwnerRepo;
use crate::types::output::OutputMode;

use clap::Parser;

/// List repository branches
#[derive(Parser, Debug)]
pub struct RepoBranchesArgs {}

impl RepoBranchesArgs {
    pub async fn run(self, global_args: GlobalArgs) -> miette::Result<()> {
        let ctx = BergContext::new(self, global_args).await?;

        let OwnerRepo { repo, owner } = ctx.owner_repo()?;

        let (_, branches) = spin_until_ready(
            ctx.client
                .repo_list_branches(owner.as_str(), repo.as_str())
                .send(),
        )
        .await
        .into_diagnostic()?;

        match ctx.global_args.output_mode {
            OutputMode::Pretty => {
                present_branches(&ctx, branches);
            }
            OutputMode::Json => branches.print_json()?,
        }

        Ok(())
    }
}

fn present_branches(ctx: &BergContext<RepoBranchesArgs>, branches: Vec<Branch>) {
    let table = ctx
        .make_table()
        .set_header(vec!["Branch", "Commit"])
        .add_rows(branches.into_iter().map(|branch| {
            vec![
                option_display(&branch.name),
                option_display(&branch.commit.as_ref().and_then(|c| c.id.as_ref())),
            ]
        }));

    println!("{table}", table = table.show());
}
