/*

	$Id: pcap2ipstat.c,v 1.3 2002/08/25 13:17:44 tron Exp $

*/

#include "checksum.h"

#include <net/if.h>
#include <net/if_ether.h>
#include <net/ethertypes.h>
#include <net/ppp_defs.h>
#include <netinet/tcp.h>
#include <netinet/udp.h>

#include <ctype.h>
#include <netdb.h>
#include <pcap.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "iprecord.h"

#define TRUE  1
#define FALSE 0

char *Progname;

static void Usage(void)

{
 (void)fprintf(stderr,
               "Usage: %s logfilefile [logfilefile ...]\n",
               Progname);
 exit(EXIT_FAILURE);
}

const void *GetPktData(const struct pcap_pkthdr *PktHdr,
                       const u_char *PktData,
                       bpf_u_int32 Offset,
                       bpf_u_int32 Length)

{
 if ((Offset+Length)>PktHdr->caplen) return NULL;

 return &PktData[Offset];
}

u_char *StripPkt(const struct pcap_pkthdr *PktHdr,
                 const u_char *PktData,
                 bpf_u_int32 Offset,
                 struct pcap_pkthdr *NewPktHdr)

{
 u_char *NewPktData;
 bpf_u_int32 Length;

 if (Offset>=PktHdr->caplen) return NULL;

 Length=PktHdr->caplen-Offset;
 if ((NewPktData=malloc(Length))==NULL) return NULL;

 NewPktHdr->ts=PktHdr->ts;
 NewPktHdr->caplen=Length;
 NewPktHdr->len=PktHdr->len-Offset;

 return memcpy(NewPktData,&PktData[Offset],Length);
}

void FilterIP(u_char *Ptr,
              const struct pcap_pkthdr *PktHdr,
              const u_char *PktData)

{
 const struct ip *IP;

 if ((IP=GetPktData(PktHdr,PktData,0,sizeof(*IP)))!=NULL)
  if ((IP->ip_v==4)&&CheckIPHdr(IP))
   {
    IPRecord IR;

    IR.ir_SrcAddr=IP->ip_src;
    IR.ir_DstAddr=IP->ip_dst;
    IR.ir_Proto=IP->ip_p;
    IR.ir_SrcPort=0;
    IR.ir_DstPort=0;
    switch (IR.ir_Proto)
     {
      case IPPROTO_TCP:
       {
        const struct tcphdr *TH;

        if ((TH=GetPktData(PktHdr,PktData,sizeof(*IP),sizeof(*TH)))!=NULL)
         {
          IR.ir_SrcPort=ntohs(TH->th_sport);
          IR.ir_DstPort=ntohs(TH->th_dport);
         }
        break;
       }
      case IPPROTO_UDP:
       {
        const struct udphdr *UH;

        if ((UH=GetPktData(PktHdr,PktData,sizeof(*IP),sizeof(*UH)))!=NULL)
         {
          IR.ir_SrcPort=ntohs(UH->uh_sport);
          IR.ir_DstPort=ntohs(UH->uh_dport);
         }
       }
     }
    IR.ir_Packets=1;
    IR.ir_Bytes=PktHdr->len;

    (void)fwrite(&IR,sizeof(IPRecord),1,stdout);
   }
}

void FilterNULL(u_char *Ptr,
                const struct pcap_pkthdr *PktHdr,
                const u_char *PktData)

{
 const bpf_u_int32 *Type;
 struct pcap_pkthdr NewPktHdr;
 u_char *NewPktData;

 if (((Type=GetPktData(PktHdr,PktData,0,sizeof(*Type)))==NULL)||
     (htonl(*Type)!=ETHERTYPE_IP)) return;

 if ((NewPktData=StripPkt(PktHdr,
                          PktData,
                          sizeof(*Type),
                          &NewPktHdr))!=NULL)
  {
   FilterIP(Ptr,&NewPktHdr,NewPktData);
   free(NewPktData);
  }
}

#define PPP_HEADERLEN	2

void FilterPPP(u_char *Ptr,
               const struct pcap_pkthdr *PktHdr,
               const u_char *PktData)

{
 const u_char *PH;
 bpf_u_int32 Proto,Offset;
 struct pcap_pkthdr NewPktHdr;
 u_char *NewPktData;

 if ((PH=GetPktData(PktHdr,PktData,0,sizeof(PPP_HEADERLEN)))==NULL) return;

 if ((PH[0]&1)!=0)
  {
   Proto=PH[0];
   Offset=1;
  }
 else
  {
   Proto=(PH[0]<<8)+PH[1];
   Offset=2;
  }

 if ((Proto==PPP_IP)&&
     ((NewPktData=StripPkt(PktHdr,
                           PktData,
                           Offset,
                           &NewPktHdr))!=NULL))
  {
   FilterIP(Ptr,&NewPktHdr,NewPktData);
   free(NewPktData);
  }
}

#define PPPOE_HEADERLEN	6
#define	PPPOE_VERTYPE	0x11

void FilterPPPoE(u_char *Ptr,
                 const struct pcap_pkthdr *PktHdr,
                 const u_char *PktData)

{
 const u_char *PH;
 bpf_u_int32 Length;
 struct pcap_pkthdr NewPktHdr;
 u_char *NewPktData;

 if ((PH=GetPktData(PktHdr,PktData,0,sizeof(PPPOE_HEADERLEN)))==NULL) return;

 /* Unsupported version of type */
 if (PH[0]!=PPPOE_VERTYPE) return;

 /* No data packet */
 if (PH[1]!=0) return;

 Length=(PH[4]<<8)+PH[5];
 if ((NewPktData=StripPkt(PktHdr,
                          PktData,
                          PPPOE_HEADERLEN,
                          &NewPktHdr))!=NULL)
  {
   FilterPPP(Ptr,&NewPktHdr,NewPktData);
   free(NewPktData);
  }
}

void FilterEther(u_char *Ptr,
                 const struct pcap_pkthdr *PktHdr,
                 const u_char *PktData)

{
 const struct ether_header *EH;
 struct pcap_pkthdr NewPktHdr;
 u_char *NewPktData;
 pcap_handler FilterProc;

 if ((EH=GetPktData(PktHdr,PktData,0,sizeof(*EH)))==NULL) return;

 switch (htons(EH->ether_type))
  {
   case ETHERTYPE_IP:
    FilterProc=FilterIP;
    break;
   case ETHERTYPE_PPPOE:
    FilterProc=FilterPPPoE;
    break;
   default:
    FilterProc=NULL;
  }

 if ((FilterProc!=NULL)&&
     ((NewPktData=StripPkt(PktHdr,
                           PktData,
                           sizeof(*EH),
                           &NewPktHdr))!=NULL))
  {
   FilterProc(Ptr,&NewPktHdr,NewPktData);
   free(NewPktData);
  }
}

void ReadPCAPFile(char *Name)

{
 char ErrorBuffer[PCAP_ERRBUF_SIZE];
 pcap_t *PCap;
 pcap_handler FilterProc;

 if ((PCap=pcap_open_offline(Name,ErrorBuffer))==NULL)
  {
   (void)fprintf(stderr,"%s: %s\n",Name,ErrorBuffer);
   return;
  }

 switch (pcap_datalink(PCap))
  {
   case DLT_NULL:
    FilterProc=FilterNULL;
    break;
   case DLT_EN10MB:
    FilterProc=FilterEther;
    break;
   case DLT_PPP_SERIAL:
    FilterProc=FilterPPP;
    break;
#ifdef DLT_PPP_ETHER
   case DLT_PPP_ETHER:
    FilterProc=FilterPPPoE;
    break;
#endif
   default:
    (void)fprintf(stderr,"%s: unsupported network hardware\n",Name);
    (void)pcap_close(PCap);
    return;
  }

 if (pcap_dispatch(PCap,-1,FilterProc,NULL)<0)
  (void)fprintf(stderr,"%s: %s\n",Name,ErrorBuffer);

 (void)pcap_close(PCap);
}

int main(int argc,char **argv)

{
 int Index;

 Progname=argv[0];
 if (argc<2) Usage();

 for (Index=1; Index<argc; Index++)
  if (strcmp(argv[Index],"-")==0) ReadPCAPFile("/dev/stdin");
  else ReadPCAPFile(argv[Index]);

 return EXIT_SUCCESS;
}
