/* $Id$ */

/*
 * Copyright (c) 2015,2017,2025 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <errno.h>
#include <string.h>
#include <limits.h>
#include <time.h>

#include <mysql/mysql.h>

#ifndef STATE_OK
#define STATE_OK        0
#define STATE_WARNING   1
#define STATE_CRITICAL  2
#define STATE_UNKNOWN   3 
#endif

#define DEFAULT_TIMEOUT	5
#define DEFAULT_WARN	300
#define DEFAULT_CRIT	600

#define DEFAULT_CONF	"/etc/my.cnf"
#define DEFAULT_HOST	"localhost"
#define DEFAULT_PORT	MYSQL_PORT
#define DEFAULT_USER	"root"
#define DEFAULT_PASS	""
#define DEFAULT_SOCK	MYSQL_UNIX_ADDR

void
usage(void)
{
	fprintf(stderr, "usage: check_mysql_slave [options]\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "  -w warn Warning lag, default %ds\n", DEFAULT_WARN);
	fprintf(stderr, "  -c crit Critical lag, default %ds\n", DEFAULT_CRIT);
	fprintf(stderr, "  -f file Configuration file, default %s\n",
			DEFAULT_CONF);
	fprintf(stderr, "  -h host MySQL host, default %s\n", DEFAULT_HOST);
	fprintf(stderr, "  -P port MySQL port, default %d\n", DEFAULT_PORT);
	fprintf(stderr, "  -S sock MySQL socket, default %s\n", DEFAULT_SOCK);
	fprintf(stderr, "  -u user MySQL user, default %s\n", DEFAULT_USER);
	fprintf(stderr, "  -p pass MySQL password, default \"%s\"\n",
			DEFAULT_PASS);

	return;
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	int fl;
	time_t timeout = DEFAULT_TIMEOUT;
	int warn = DEFAULT_WARN;
	int crit = DEFAULT_CRIT;
	const char *conf = DEFAULT_CONF;
	const char *host = DEFAULT_HOST;
	const char *user = DEFAULT_USER;
	const char *pass = DEFAULT_PASS;
	unsigned int port = DEFAULT_PORT;
	const char *sock = DEFAULT_SOCK;
	int sync_delay;
	char status_str[512];
	MYSQL *m;
	MYSQL_RES *mr;
	char **ml;
	MYSQL_FIELD *mf;
	int i;
	int sync_delay_field = -1;
	int slave_io_field = -1;
	int master_field = -1;
	int io_error_field = -1;
	int sql_error_field = -1;

	while ((fl = getopt(argc, argv, "t:w:c:f:h:P:u:p:S:")) != -1) {
		switch(fl) {
		case 't':
			timeout = (time_t)atoi(optarg);
			break;
		case 'w':
			warn = atoi(optarg);
			break;
		case 'c':
			crit = atoi(optarg);
			break;
		case 'f':
			conf = optarg;
			break;
		case 'h':
			host = optarg;
			break;
		case 'P':
			port = atoi(optarg);
			break;
		case 'u':
			user = optarg;
			break;
		case 'p':
			pass = optarg;
			break;
		case 'S':
			sock = optarg;
			break;
		default:
			usage();
			exit(STATE_UNKNOWN);
			break;
		}
	}

	timeout += time(NULL);

	if ((m = mysql_init(NULL)) == NULL) {
		printf("UNKNOWN: init failed: %s\n", strerror(errno));
		exit(STATE_UNKNOWN);
	}

	if (mysql_options(m, MYSQL_READ_DEFAULT_FILE, conf) != 0) {
		printf("UNKNOWN: options failed: %s\n", mysql_error(m));
		exit(STATE_UNKNOWN);
	}

	mysql_ssl_set(m, NULL, NULL, NULL, NULL, NULL);

	if (mysql_real_connect(m, host, user, pass, NULL,
			       port, sock, 0) == NULL) {
		printf("UNKNOWN: connect failed: %s\n", mysql_error(m));
		exit(STATE_UNKNOWN);
	}

retry:
	if (mysql_query(m, "SHOW SLAVE STATUS;") != 0) {
		printf("UNKNOWN: query failed: %s\n", mysql_error(m));
		exit(STATE_UNKNOWN);
	}

	if ((mr = mysql_store_result(m)) == NULL) {
		printf("UNKNOWN: store_result failed: %s\n", mysql_error(m));
		exit(STATE_UNKNOWN);
	}


	for (i = 0; i < mysql_field_count(m); i++) {
		if ((mf = mysql_fetch_field(mr)) == NULL) {
			printf("UNKNOWN: fetch_field failed: %s\n",
			       mysql_error(m));
			exit(STATE_UNKNOWN);
		}

		if (strcmp(mf->name, "Seconds_Behind_Master") == 0) {
			sync_delay_field = i;
			continue;
		}

		if (strcmp(mf->name, "Slave_IO_Running") == 0) {
			slave_io_field = i;
			continue;
		}

		if (strcmp(mf->name, "Master_Host") == 0) {
			master_field = i;
			continue;
		}

		if (strcmp(mf->name, "Last_IO_Error") == 0) {
			io_error_field = i;
			continue;
		}

		if (strcmp(mf->name, "Last_SQL_Error") == 0) {
			sql_error_field = i;
			continue;
		}
	}

	if (sync_delay_field < 0 || sync_delay_field >= mysql_field_count(m)) {
		printf("UNKNOWN: Seconds_Behind_Master field not found\n");
		exit(STATE_UNKNOWN);
	}

	if (slave_io_field < 0 || slave_io_field >= mysql_field_count(m)) {
		printf("UNKNOWN: Slave_IO_Running field not found\n");
		exit(STATE_UNKNOWN);
	}

	if (master_field < 0 || master_field >= mysql_field_count(m)) {
		printf("UNKNOWN: Master_Host field not found\n");
		exit(STATE_UNKNOWN);
	}

	if (io_error_field < 0 || io_error_field >= mysql_field_count(m)) {
		printf("UNKNOWN: Last_IO_Error field not found\n");
		exit(STATE_UNKNOWN);
	}

	if (sql_error_field < 0 || sql_error_field >= mysql_field_count(m)) {
		printf("UNKNOWN: Last_SQL_Error field not found\n");
		exit(STATE_UNKNOWN);
	}

	if ((ml = mysql_fetch_row(mr)) == NULL) {	
		printf("UNKNOWN: fetch_row failed: %s\n", mysql_error(m));
		exit(STATE_UNKNOWN);
	}

	if (ml[sync_delay_field] == NULL) {
		if (strcmp(ml[slave_io_field], "Preparing") == 0) {
			if (time(NULL) > timeout) { 
				sync_delay = INT_MAX;
				(void)snprintf(status_str, sizeof(status_str), 
				       "Starting sync timeout with %s master.",
				       ml[master_field]);
			} else {
				sleep(1);
				goto retry;
			}
		} else {
			sync_delay = INT_MAX;
			(void)snprintf(status_str, sizeof(status_str), 
				       "Out of sync with %s master.",
				       ml[master_field]);
		}
	} else {
		sync_delay = atoi(ml[sync_delay_field]);
		(void)snprintf(status_str, sizeof(status_str), 
			       "%ds behind %s master.",
			       sync_delay, ml[master_field]);
	}

	if (sync_delay > crit) {
		printf("CRITIAL: %s %s %s\n", status_str,
		       ml[io_error_field], ml[sql_error_field]);
		exit(STATE_CRITICAL);
	} 

	if (sync_delay > warn) {
		printf("WARNING: %s %s %s\n", status_str,
		       ml[io_error_field], ml[sql_error_field]);
		exit(STATE_WARNING);
	} 

	if (strlen(ml[sql_error_field]) != 0 ||
	    strlen(ml[sql_error_field]) != 0) {
		printf("WARNING: %s %s %s\n", status_str,
		       ml[io_error_field], ml[sql_error_field]);
		exit(STATE_WARNING);
	}

	printf("OK: %s\n", status_str);

	return STATE_OK;
}
