/*
 * Copyright (c) 2005 Matthieu Castet <castet.matthieu@free.fr>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <fuse.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>

#include <stdlib.h>
#include <unistd.h>

#include <cdda_interface.h>
#include <cdda_paranoia.h>

#define dprintf(...)

#define RETRY 5

static char wavheader[] = {
	'R', 'I', 'F', 'F',
	0, 0, 0, 0,					//file size
	'W', 'A', 'V', 'E',
	'f', 'm', 't', ' ',
	16, 0, 0, 0,				//size of tag
	1, 0,						//Format
	2, 0,						//Channels
	0x44, 0xac, 0, 0,			// Samplerate 44100
	0x10, 0xb1, 0x2, 0,			// avg byte/sec 44100*2*2
	4, 0,						//Block align
	16, 0,						//bits/sample
	'd', 'a', 't', 'a',
	0, 0, 0, 0					//size of tag
};

static cdrom_drive *cdd = NULL;
static int mode = PARANOIA_MODE_DISABLE;
struct cddfs {
	cdrom_paranoia *cdp;
	int last_size;
	int last_remaining;
	//XXX endianness
	//int16_t *buffer;
	char *buffer;
	off_t last_pos;
	int track;
};

static int track(const char *path)
{
	int i;
	if (strlen(path) == 13 && !memcmp(path, "/track-", 7)
		&& !strcmp(path + 9, ".wav")
		&& sscanf(path + 7, "%d", &i) == 1 && i > 0
		&& i <= cdda_tracks(cdd))
		return i;
	return 0;
}

static int cddfs_getattr(const char *path, struct stat *stbuf)
{
	int res = 0;
	int i;

	memset(stbuf, 0, sizeof(struct stat));
	if (strcmp(path, "/") == 0) {
		stbuf->st_mode = S_IFDIR | 0555;
		stbuf->st_nlink = 2;
	} else if ((i = track(path))) {
		stbuf->st_mode = S_IFREG | 0644;
		stbuf->st_nlink = 1;
		stbuf->st_size =
			(cdda_track_lastsector(cdd, i) -
			 cdda_track_firstsector(cdd, i) + 1) * CD_FRAMESIZE_RAW
			+ sizeof(wavheader);
		stbuf->st_blksize = CD_FRAMESIZE_RAW;	/* ignored... */
		stbuf->st_blocks = stbuf->st_size / 512;	/* for du */
	} else
		res = -ENOENT;

	return res;
}

static int cddfs_getdir(const char *path, fuse_dirh_t h,
						fuse_dirfil_t filler)
{
	int i;
	if (strcmp(path, "/") != 0)
		return -ENOENT;

	filler(h, ".", 0, 0);
	filler(h, "..", 0, 0);
	for (i = 1; i <= cdda_tracks(cdd); i++) {
		char track_name[80];
		sprintf(track_name, "track-%.2d.wav", i);
		filler(h, track_name, 0, 0);
	}

	return 0;
}

static int cddfs_open(const char *path, struct fuse_file_info *fi)
{
	int i;
	struct cddfs *priv;
	if (!(i = track(path)))
		return -ENOENT;

	if ((fi->flags & 3) != O_RDONLY)
		return -EACCES;
	priv = malloc(sizeof(struct cddfs));

	if (!priv)
		return -ENOMEM;

	priv->last_size =
		(cdda_track_lastsector(cdd, i) - cdda_track_firstsector(cdd, i) +
		 1)
		* CD_FRAMESIZE_RAW + sizeof(wavheader);
	priv->last_remaining = 0;
	priv->cdp = paranoia_init(cdd);
	paranoia_modeset(priv->cdp, mode);
	priv->last_pos = -1;
	priv->track = i;
	fi->fh = (typeof(fi->fh)) priv;
	return 0;
}

static int cddfs_release(const char *path, struct fuse_file_info *fi)
{
	struct cddfs *priv;

	priv = (struct cddfs *) fi->fh;
	paranoia_free(priv->cdp);
	free(priv);
	fi->fh = (typeof(fi->fh)) NULL;
	return 0;
}

#define CPU2LE(w,v) ((w)[0] = (u_int8_t)(v), \
                     (w)[1] = (u_int8_t)((v) >> 8), \
                     (w)[2] = (u_int8_t)((v) >> 16), \
                     (w)[3] = (u_int8_t)((v) >> 24))

static inline void *copy(void *dest, const void *src, size_t n)
{
	dprintf("**cpy %p %p %d\n", dest, src, n);
	dprintf("**cpy+ %p %p\n", dest + n, src + n);
	if (src)
		return memcpy(dest, src, n);
}

static int cddfs_read(const char *path, char *buf, size_t size,
					  off_t offset, struct fuse_file_info *fi)
{
	size_t len;
	(void) fi;
	int first_sec, lastsize, headersize = 0;
	struct cddfs *priv;

	priv = (struct cddfs *) fi->fh;
	if (!priv)
		return -ENOENT;

	if (offset > priv->last_size)
		return 0;

	if (offset + size > priv->last_size)
		size = priv->last_size - offset;

#if 1
	if (offset < sizeof(wavheader)) {
		CPU2LE(wavheader + 4, priv->last_size - 8);
		CPU2LE(wavheader + 40, priv->last_size - sizeof(wavheader));
		len = sizeof(wavheader) - offset;
		if (len > size)
			len = size;
		memcpy(buf, wavheader + offset, len);
		if (len == size)
			return len;
		size -= len;
		buf += len;
		headersize = len;
		offset = 0;

	} else
		offset -= sizeof(wavheader);
#endif

	if (size == 0)
		return headersize;

	dprintf("****%d %d %d \n", (int) priv->last_pos, (int) offset,
			priv->last_remaining);


	if (priv->last_pos != offset) {
		//seek
		first_sec =
			offset / CD_FRAMESIZE_RAW + cdda_track_firstsector(cdd,
															   priv->
															   track);
		if (first_sec > cdda_track_lastsector(cdd, priv->track))
			return 0;
		if (paranoia_seek(priv->cdp, first_sec, SEEK_SET) == -1)
			return -ENOENT;
		dprintf("****paranoia_seek %d\n", first_sec);

		/* we could want only the end of the first sector */
		priv->buffer =
			(char *) paranoia_read_limited(priv->cdp, NULL, RETRY);
		dprintf("****paranoia_read_limited %p\n", priv->buffer);
		if (priv->buffer)
			priv->buffer += offset % CD_FRAMESIZE_RAW;
		len = CD_FRAMESIZE_RAW - offset % CD_FRAMESIZE_RAW;
	} else {
		len = priv->last_remaining;
	}

	dprintf("****%d %d\n", len, priv->last_remaining);
	priv->last_pos = offset + size;

	if (len > size) {
		priv->last_remaining = len - size;
		len = size;
	} else
		priv->last_remaining = 0;

	copy(buf, priv->buffer, len);
	priv->buffer += len;
	/* we don't want all the last sector */
	lastsize = (size - len) % CD_FRAMESIZE_RAW;
	size -= lastsize;

	while (len < size) {
		priv->buffer =
			(char *) paranoia_read_limited(priv->cdp, NULL, RETRY);
		dprintf("****paranoia_read_limited %p\n", priv->buffer);
		copy(buf + len, priv->buffer, CD_FRAMESIZE_RAW);
		len += CD_FRAMESIZE_RAW;
	}

	if (lastsize) {
		dprintf("****lastsize%d\n", lastsize);
		priv->buffer =
			(char *) paranoia_read_limited(priv->cdp, NULL, RETRY);
		dprintf("****paranoia_read_limited %p\n", priv->buffer);
		copy(buf + len, priv->buffer, lastsize);
		len += lastsize;
		if (priv->buffer)
			priv->buffer += lastsize;
		priv->last_remaining = CD_FRAMESIZE_RAW - lastsize;
	}

	return len + headersize;
}

static struct fuse_operations cddfs_oper = {
	.getattr = cddfs_getattr,
	.getdir = cddfs_getdir,
	.open = cddfs_open,
	.read = cddfs_read,
	.release = cddfs_release,
};

static void usage()
{
	fprintf(stderr,
			"-D device (default /dev/cdrom)\n"
			"-m mode (0,1,2) (default 0)\n");
}

int main(int argc, char *argv[])
{
	char *device = "/dev/cdrom";
	char c;
	const char *optstring = "+D:m:h::";
	opterr = 0;

	while ((c = getopt(argc, argv, optstring)) != EOF) {
		switch (c) {
		case 'D':
			device = strdup(optarg);
		case 'm':
			switch (atoi(optarg)) {
			case 1:
				mode = PARANOIA_MODE_OVERLAP;
				break;
			case 2:
				mode = PARANOIA_MODE_SCRATCH;
				break;
			case 3:
				mode = PARANOIA_MODE_FULL;
				break;
			default:
				mode = PARANOIA_MODE_DISABLE;
			}
			break;
		case 'h':
			usage();
			break;
		}
		if (c == '?' || c == 'h') {
			optind--;
			break;
		}
	}

	if (c != 'h') {
		cdd = cdda_identify(device, 0, NULL);
		if (!cdd) {
			printf("Can't open cdda device\n");
			return 1;
		}

		if (cdda_open(cdd) != 0) {
			printf("Can't open disc\n");
			cdda_close(cdd);
			return 1;
		}
		printf("Found Audio CD with %lu tracks\n", cdda_tracks(cdd));
		/* lock the door */
		open(device, O_RDONLY);
	}
	optind--;
	argv[optind] = argv[0];

	return fuse_main(argc - optind, argv + optind, &cddfs_oper);
}
