/* blib - Library of useful things to hack the Blinkenlights
 *
 * Copyright (c) 2001-2002  The Blinkenlights Crew
 *                          Sven Neumann <sven@gimp.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include <gtk/gtk.h>

#include "blib/blib.h"

#include "bview-gtk.h"


static void      b_view_gtk_class_init   (BViewGtkClass   *class);
static void      b_view_gtk_init         (BViewGtk        *view);
static void      b_view_gtk_finalize     (GObject         *object);
static void      b_view_gtk_realize      (GtkWidget       *widget);
static void      b_view_gtk_size_request (GtkWidget       *widget,
                                          GtkRequisition  *requisition);
static gboolean  b_view_gtk_expose_event (GtkWidget       *widget,
                                          GdkEventExpose  *event);
static GdkPixbuf * load_image            (BViewGtk        *view,
                                          const gchar     *filename);
static void      gdk_pixbuf_fill_rect    (GdkPixbuf       *pixbuf,
                                          GdkRectangle    *rect,
                                          BColor          *color);
static void      gdk_pixbuf_blend_rect   (GdkPixbuf       *dest,
                                          GdkPixbuf       *src,
                                          GdkRectangle    *dest_rect,
                                          gint             src_x,
                                          gint             src_y,
                                          guchar           opacity);


static GtkDrawingAreaClass *parent_class = NULL;


GType
b_view_gtk_get_type (void)
{
  static GType view_type = 0;

  if (!view_type)
    {
      static const GTypeInfo view_info =
      {
        sizeof (BViewGtkClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) b_view_gtk_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (BViewGtk),
        0,              /* n_preallocs */
        (GInstanceInitFunc) b_view_gtk_init,
      };

      view_type = g_type_register_static (GTK_TYPE_DRAWING_AREA,
                                          "BViewGtk", &view_info, 0);
    }
  
  return view_type;
}

static void
b_view_gtk_class_init (BViewGtkClass *class)
{
  GObjectClass   *object_class;
  GtkWidgetClass *widget_class;

  parent_class = g_type_class_peek_parent (class);
  object_class = G_OBJECT_CLASS (class);
  widget_class = GTK_WIDGET_CLASS (class);

  object_class->finalize = b_view_gtk_finalize;

  widget_class->realize      = b_view_gtk_realize;
  widget_class->size_request = b_view_gtk_size_request;
  widget_class->expose_event = b_view_gtk_expose_event;
}

static void
b_view_gtk_init (BViewGtk *view)
{
  view->theme = NULL;
  view->images = g_hash_table_new_full (g_str_hash, g_str_equal,
                                        (GDestroyNotify) NULL,
                                        (GDestroyNotify) g_object_unref);
}

static void
b_view_gtk_finalize (GObject *object)
{
  BViewGtk *view;

  view = B_VIEW_GTK (object);

  if (view->overlay)
    g_object_unref (view->overlay);

  g_hash_table_destroy (view->images);
  
  g_object_unref (view->theme);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
b_view_gtk_realize (GtkWidget *widget)
{
  BViewGtk *view  = B_VIEW_GTK (widget);
  GdkColor  color;

  GTK_WIDGET_CLASS (parent_class)->realize (widget);

  color.red   = view->theme->bg_color.r;
  color.green = view->theme->bg_color.g;
  color.blue  = view->theme->bg_color.b;

  if (gdk_colormap_alloc_color
      (gdk_drawable_get_colormap (GDK_DRAWABLE (widget->window)),
       &color, FALSE, TRUE))
    gdk_window_set_background (widget->window, &color);
}

static void
b_view_gtk_size_request (GtkWidget      *widget,
                         GtkRequisition *requisition)
{
  BViewGtk *view = B_VIEW_GTK (widget);

  requisition->width  = view->theme->width;
  requisition->height = view->theme->height;
}

static gboolean
b_view_gtk_expose_event  (GtkWidget      *widget,
                          GdkEventExpose *event)
{
  BViewGtk     *view;
  GdkRectangle  rect;
  gint          x, y;

  view = B_VIEW_GTK (widget);

  rect.width  = view->theme->width;
  rect.height = view->theme->height;
  rect.x = x  = (widget->allocation.width  - rect.width)  / 2;
  rect.y = y  = (widget->allocation.height - rect.height) / 2;

  if (!gdk_rectangle_intersect (&rect, &event->area, &rect))
    return TRUE;

  if (view->background)
    gdk_pixbuf_render_to_drawable (view->background,
                                   widget->window,
                                   widget->style->fg_gc[GTK_STATE_NORMAL],
                                   rect.x - x, rect.y - y,
                                   rect.x, rect.y, rect.width, rect.height,
                                   GDK_RGB_DITHER_NORMAL,
                                   - event->area.x, - event->area.y);

  rect = view->rect;
  rect.x += x;
  rect.y += y;

  if (!gdk_rectangle_intersect (&rect, &event->area, &rect))
    return TRUE;

  gdk_pixbuf_render_to_drawable_alpha (view->overlay,
                                       widget->window,
                                       rect.x - x, rect.y - y,
                                       rect.x, rect.y, rect.width, rect.height,
                                       GDK_PIXBUF_ALPHA_FULL, /* ignored */
                                       0x80,                  /* ignored */
                                       GDK_RGB_DITHER_NORMAL,
                                       - event->area.x, - event->area.y);

  return TRUE;
}

/**
 * b_view_gtk_new:
 * @theme: a #BTheme object
 * @error: location to store the error occuring, or %NULL to ignore errors
 * 
 * Creates a new widget suitable to display Blinkenlights movies that
 * fit the @theme.
 * 
 * Return value: a new widget or %NULL in case of an error
 **/
GtkWidget *
b_view_gtk_new (BTheme  *theme,
                GError **error)
{
  BViewGtk *view;
  GList    *list;

  g_return_val_if_fail (B_IS_THEME (theme), NULL);
  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

  if (theme->channels != 1)
    {
      g_set_error (error, 0, 0, "Channels != 1 is not (yet) supported");
      return NULL;
    }

  view = B_VIEW_GTK (g_object_new (B_TYPE_VIEW_GTK, NULL));

  view->theme = g_object_ref (theme);

  view->overlay = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8,
                                  theme->width, theme->height);

  if (theme->bg_image)
    view->background = load_image (view, theme->bg_image);

  for (list = theme->overlays; list; list = list->next)
    {
      BOverlay *overlay = list->data;

      if (!overlay->image)
        continue;

      load_image (view, overlay->image);
    }

  view->rect.x      = 0;
  view->rect.y      = 0;
  view->rect.width  = theme->width;
  view->rect.height = theme->height;

  b_view_gtk_update (view, NULL);

  return GTK_WIDGET (view);
}

/**
 * b_view_gtk_update:
 * @view: a #BViewGtk widget
 * @frame_data: the frame data to display
 * 
 * Displays a new frame on the @view. The @view expects @frame_data
 * in the range of 0 to 255.
 **/
void
b_view_gtk_update (BViewGtk     *view,
                   const guchar *frame_data)
{
  BTheme *theme;
  GList  *list;

  g_return_if_fail (B_IS_VIEW_GTK (view));
  g_return_if_fail (B_IS_THEME (view->theme));
  
  theme = view->theme;

  gdk_pixbuf_fill_rect (view->overlay, &view->rect, NULL);
      
  view->rect.width  = 0;
  view->rect.height = 0;

  if (!frame_data)
    {
      gtk_widget_queue_draw (GTK_WIDGET (view));
      return;
    }

  for (list = theme->overlays; list; list = list->next)
    {
      BOverlay  *overlay = list->data;
      GdkPixbuf *pixbuf  = NULL;
      GList     *windows;
      
      if (overlay->image)
        pixbuf = g_hash_table_lookup (view->images, overlay->image);
      
      if (pixbuf)
        {
          for (windows = overlay->windows; windows; windows = windows->next) 
            {
              BWindow *window = windows->data;
              guchar   value;

              value = frame_data[(window->column +
                                  window->row * theme->columns)];
              if (!value)
                continue;
              
              window += (value * theme->maxval) / 256;

              gdk_pixbuf_blend_rect (view->overlay, pixbuf,
                                     (GdkRectangle *) &window->rect,
                                     window->src_x, window->src_y,
                                     (window->value == B_WINDOW_VALUE_ALL) ?
                                     value : 0xFF);
              gdk_rectangle_union (&view->rect, (GdkRectangle *) &window->rect,
                                   &view->rect);
            }
        }
      else
        {
          BColor color = overlay->color;

          for (windows = overlay->windows; windows; windows = windows->next) 
            {
              BWindow *window = windows->data;
              guchar   value;

              value = frame_data[(window->column +
                                  window->row * theme->columns)];
              if (!value)
                continue;

              window += (value * theme->maxval) / 256;

              color.a = overlay->color.a;
              if (window->value == B_WINDOW_VALUE_ALL)
                color.a = ((color.a + 1) * value) >> 8;

              gdk_pixbuf_fill_rect (view->overlay,
                                    (GdkRectangle *) &window->rect, &color);
              gdk_rectangle_union (&view->rect,
                                   (GdkRectangle *) &window->rect, 
                                   &view->rect);
            }
        }
    }

  gtk_widget_queue_draw (GTK_WIDGET (view));
}

static GdkPixbuf *
load_image (BViewGtk    *view,
            const gchar *filename)
{
  GdkPixbuf *pixbuf;
  GError    *error = NULL;

  pixbuf = g_hash_table_lookup (view->images, filename);
  
  if (pixbuf)
    return pixbuf;

  pixbuf = gdk_pixbuf_new_from_file (filename, &error);

  if (!pixbuf)
    {
      g_printerr ("Error loading image from file '%s': %s\n",
                  filename, error->message);
    }

  g_hash_table_insert (view->images, (gpointer) filename, pixbuf);

  return pixbuf;
}

/* this function assumes that the pixbuf has an alpha channel */
static void
gdk_pixbuf_fill_rect (GdkPixbuf    *pixbuf,
                      GdkRectangle *rect,
                      BColor       *color)
{
  guchar       *data;
  gint          pitch;
  BColor        trans = { 0, 0, 0, 0 };
  GdkRectangle  fill  = { 0, 0,
                          gdk_pixbuf_get_width  (pixbuf),
                          gdk_pixbuf_get_height (pixbuf) };

  if (!gdk_rectangle_intersect (rect, &fill, &fill))
    return;

  data  = gdk_pixbuf_get_pixels (pixbuf);
  pitch = gdk_pixbuf_get_rowstride (pixbuf);

  if (!color)
    color = &trans;

  data += fill.y * pitch + fill.x * 4;

  do
    {
      guchar *d = data;
      guint   w = fill.width;

      do
        {
          d[0] = color->r;
          d[1] = color->g;
          d[2] = color->b;
          d[3] = color->a;

          d += 4;
        }
      while (--w);

      data += pitch;
    }
  while (--fill.height);
}

/* this function assumes that the dest pixbuf has an alpha channel */
static void
gdk_pixbuf_blend_rect (GdkPixbuf    *dest,
                       GdkPixbuf    *src,
                       GdkRectangle *dest_rect,
                       gint          src_x,
                       gint          src_y,
                       guchar        opacity)
{
  guchar       *src_data;
  guchar       *dest_data;
  gint          src_pitch;
  gint          dest_pitch;
  gint          i;
  GdkRectangle  rect = { 0, 0, 
                         gdk_pixbuf_get_width  (dest),
                         gdk_pixbuf_get_height (dest) };

  if (!opacity)
    return;

  if (!gdk_rectangle_intersect (dest_rect, &rect, &rect))
    return;

  src_x += rect.x - dest_rect->x;
  src_y += rect.y - dest_rect->y;

  if (src_x < 0)
    rect.width += src_x, src_x = 0;
  
  if (src_y < 0)
    rect.height += src_y, src_y = 0;

  i = gdk_pixbuf_get_width (src);
  rect.width = MIN (i - src_x, rect.width);
  if (rect.width < 1)
    return;
  
  i = gdk_pixbuf_get_height (src);
  rect.height = MIN (i - src_y, rect.height);
  if (rect.height < 1)
    return;

  src_data   = gdk_pixbuf_get_pixels (src);
  src_pitch  = gdk_pixbuf_get_rowstride (src);

  dest_data  = gdk_pixbuf_get_pixels (dest);
  dest_pitch = gdk_pixbuf_get_rowstride (dest);

  dest_data += rect.y * dest_pitch + rect.x * 4;

  switch (gdk_pixbuf_get_n_channels (src))
    {
    case 3:
      src_data += src_y * src_pitch + src_x * 3;

      switch (opacity)
        {
        case 0xff:
          do
            {
              guchar *s = src_data;
              guchar *d = dest_data;
              gint    w = rect.width;
              
              do
                {
                  d[0] = s[0];
                  d[1] = s[1];
                  d[2] = s[2];
                  d[3] = 0xff;
                  
                  d += 4;
                  s += 3;
                }
              while (--w);
              
              src_data  += src_pitch;
              dest_data += dest_pitch;
            }
          while (--rect.height);
          break;

        default:
          do
            {
              guchar *s = src_data;
              guchar *d = dest_data;
              gint    w = rect.width;
              
              do
                {
                  guint a0 = 0xff * opacity;
                  guint a1 = (0xff - opacity) * d[3];
                  guint a  = a0 + a1;

		  d[0] = (a0 * s[0] + a1 * d[0]) / a;
		  d[1] = (a0 * s[1] + a1 * d[1]) / a;
		  d[2] = (a0 * s[2] + a1 * d[2]) / a;
		  d[3] = a / 0xff;

                  d += 4;
                  s += 3;
                }
              while (--w);
          
              src_data  += src_pitch;
              dest_data += dest_pitch;
            }
          while (--rect.height);
          break;
        }
      break;

    case 4:
      src_data += src_y * src_pitch + src_x * 4;

      switch (opacity)
        {
        case 0xff:
          do
            {
              guchar *s = src_data;
              guchar *d = dest_data;
              gint    w = rect.width;

              do
                {
                  switch (s[3])
                    {
                    case 0:
                      break;
                      
                    case 0xff:
                      d[0] = s[0];
                      d[1] = s[1];
                      d[2] = s[2];
                      d[3] = 0xff;
                      break;
                      
                    default:
                      {
                        guint a0 = s[3] * 0xff;
                        guint a1 = (0xff - s[3]) * d[3];
                        guint a  = a0 + a1;
                        
                        d[0] = (a0 * s[0] + a1 * d[0]) / a;
                        d[1] = (a0 * s[1] + a1 * d[1]) / a;
                        d[2] = (a0 * s[2] + a1 * d[2]) / a;
                        d[3] = a / 0xff;
                      }
                      break;
                    }

                  s += 4;
                  d += 4;
                }
              while (--w);
          
              src_data  += src_pitch;
              dest_data += dest_pitch;
            }
          while (--rect.height);
          break;

        default:
          do
            {
              guchar *s = src_data;
              guchar *d = dest_data;
              gint    w = rect.width;

              do
                {
                  if (s[3])
                    {
                      guint a0 = s[3] * opacity;
                      guint a1 = (0xff - (a0 / 0xff)) * d[3];
                      guint a  = a0 + a1;

                      d[0] = (a0 * s[0] + a1 * d[0]) / a;
                      d[1] = (a0 * s[1] + a1 * d[1]) / a;
                      d[2] = (a0 * s[2] + a1 * d[2]) / a;
                      d[3] = a / 0xff;
                    }

                  s += 4;
                  d += 4;
                }
              while (--w);
          
              src_data  += src_pitch;
              dest_data += dest_pitch;
            }
          while (--rect.height);
          break;
        }
    }
}
