/*
 * Copyright (c) 2005 Jacob Meuser <jakemsr@jakemsr.com>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

/*
 *  $Id: bktrplay.c,v 1.21 2006/03/30 09:21:26 jakemsr Exp $
 */

#include "includes.h"

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/mman.h>
#include <sys/shm.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/XShm.h>
#include <X11/extensions/Xvlib.h>
#include <X11/Xatom.h>

#include "bsdav.h"


struct bp_display {
	Display		*dpy;
	Window		 window;
	Window		 rootwindow;
	XEvent		 event;
	GC		 gc;
	Atom		 wmdelwin;
	XWindowAttributes *norm_winatt;
	XvPortID	 port;
	XvImage		*xv_image;
	XShmSegmentInfo	 shminfo;
	char		*pixels;
	unsigned int	 adaptor;
	int		 width;
	int		 height;
	int		 screen_id;
};

struct bp_bktr {
	uint8_t		*buffer;
	size_t		 buffer_size;
	char		 dev[FILENAME_MAX];
	int		 norm;
	int		 source;
	int		 fps;
	int		 fd;
	int		 width;
	int		 height;
	int		 fields;
};

struct bktrplay {
	struct bp_display *display;
	struct bp_bktr	*bktr;
	struct bsdav_ratio aspect;
	struct bsdav_crop crop;
	int		 format;
	int		 verbose;
	int		 pause;
	int		 shutdown;
	int		 full_screen;
	int		 aspect_lock;
};

volatile sig_atomic_t bktr_frame_waiting;

void sig_handler(int);
int xv_init(struct bktrplay *);
int create_win(struct bktrplay *);
int stream(struct bktrplay *);
int set_format(struct bktrplay *);
int display_event(struct bktrplay *);
int resize_window(struct bktrplay *, int);
int switch_source(struct bktrplay *, int);
int switch_pause(struct bktrplay *);
void cleanup(struct bktrplay *);
void usage(void);

extern char *__progname;

void
usage(void)
{
	fprintf(stderr,
		"Usage: %s [-v] [-a aspect] [-c crop] [-e fourcc] [-f file]\n"
		"          [-i field ] [-n norm] [-r rate] [-s source]\n",
		__progname);
	return;
}

void
sig_handler(int signal)
{
	if (signal == SIGUSR1)
		bktr_frame_waiting++;

	return;
}


int
xv_init(struct bktrplay *bdt)
{
XvImageFormatValues *xvformats;
XvAdaptorInfo *ainfo;
int i, j, k;
int num_formats;
int num_adaptors;


	bdt->display->rootwindow = DefaultRootWindow(bdt->display->dpy);

	if (Success != XvQueryAdaptors(bdt->display->dpy,
	    bdt->display->rootwindow, &num_adaptors, &ainfo)) {
		warnx("no Xv adaptors present");
		return (1);
	}

	for (i = 0; i < num_adaptors && (bdt->display->port == 0); i++) {
		if ((ainfo[i].type & XvInputMask) &&
		    (ainfo[i].type & XvImageMask)) {
			if (bdt->verbose > 1)
				warnx("adaptor %d can PutImage", i);
			xvformats = XvListImageFormats(bdt->display->dpy,
			    ainfo[i].base_id, &num_formats);
			for (j = 0; j < num_formats; j++) {
				if (xvformats[j].id ==
				    bsdav_vid_fmts[bdt->format].xv_id) {
					bdt->display->adaptor = j;
					break;
				}
			}
			if (xvformats != NULL)
				XFree(xvformats);
		}
		for (k = 0; k < ainfo[i].num_ports; ++k) {
			if (Success == XvGrabPort(bdt->display->dpy,
			    ainfo[i].base_id, CurrentTime)) {
				bdt->display->port = ainfo[i].base_id;
				if (bdt->verbose > 1)
					warnx("grabbed Xv port %ld",
					    bdt->display->port);
				break;
			}
		}
	}

	XvFreeAdaptorInfo(ainfo);

	if (bdt->display->port == 0) {
		warnx("could not find valid Xv port");
		return (1);
	}

	return (0);
}


int
create_win(struct bktrplay * bdt)
{
XGCValues values;
XTextProperty WinName;
XSizeHints szhints;
XWMHints wmhints;
char *name;

	name = __progname;
	XStringListToTextProperty(&name, 1, &WinName);

	szhints.flags = PSize | PMaxSize | PMinSize;
	szhints.width = bdt->display->width;
	szhints.height = bdt->display->height;
	szhints.max_width = 2048;  /* get this from xv_init + DisplayWidth */
	szhints.max_height = 2048;  /* get this from xv_init + DisplayHeight*/
	szhints.min_width = 160;
	szhints.min_height = 120;

	wmhints.flags = InputHint | StateHint;
	wmhints.input = True;
	wmhints.initial_state = NormalState;

	bdt->display->window = XCreateSimpleWindow(bdt->display->dpy,
	    bdt->display->rootwindow, 0, 0, bdt->display->width,
	    bdt->display->height, 0,
	    XWhitePixel(bdt->display->dpy, bdt->display->screen_id),
	    XBlackPixel(bdt->display->dpy, bdt->display->screen_id));

	XSetWMProperties(bdt->display->dpy, bdt->display->window, &WinName,
	    &WinName, NULL, 0, &szhints, &wmhints, NULL);

	XSelectInput(bdt->display->dpy, bdt->display->window,
	    KeyPressMask | ButtonPressMask | StructureNotifyMask);

	bdt->display->wmdelwin = XInternAtom(bdt->display->dpy,
	    "WM_DELETE_WINDOW", False);
	XSetWMProtocols(bdt->display->dpy, bdt->display->window,
	    &bdt->display->wmdelwin, 1);

	XMapRaised(bdt->display->dpy, bdt->display->window);

	bdt->display->gc = XCreateGC(bdt->display->dpy, bdt->display->window,
	    0, &values);

	bdt->display->xv_image = XvShmCreateImage(bdt->display->dpy,
	    bdt->display->port, bsdav_vid_fmts[bdt->format].xv_id,
	    bdt->display->pixels, bdt->bktr->width, bdt->bktr->height,
	    &bdt->display->shminfo);

	bdt->display->shminfo.shmid = shmget(IPC_PRIVATE,
	    bdt->bktr->buffer_size, IPC_CREAT | 0777);

	if (bdt->display->shminfo.shmid < 0) {
		warn("shmget");
		return (1);
	}

	if ((bdt->display->shminfo.shmaddr =
	    shmat(bdt->display->shminfo.shmid, 0, 0)) == (void *)-1) {
		warn("shmat");
		return (1);
	}

	bdt->display->xv_image->data = bdt->display->pixels =
	    bdt->display->shminfo.shmaddr;

	XShmAttach(bdt->display->dpy, &bdt->display->shminfo);

	XSync(bdt->display->dpy, False);

	return (0);
}


int
stream(struct bktrplay *bdt)
{
struct sigaction act;
sigset_t sa_mask;
struct itimerval tim;
struct timeval tp_start;
struct timeval tp_end;
struct timeval tp_run;
double run_time;
long frames_played = 0;
long sequence = 100;
uint c;
int ret = 0;


	bdt->pause = 0;

	/* for sigsuspend() */
	sigfillset(&sa_mask);
	sigdelset(&sa_mask, SIGALRM);
	sigdelset(&sa_mask, SIGUSR1);

	/* for setitimer() */
	memset(&tim, 0, sizeof(tim));
	tim.it_value.tv_sec = 1;

	memset(&act, 0, sizeof(act));
	sigemptyset(&act.sa_mask);
	act.sa_handler = sig_handler;
	sigaction(SIGUSR1, &act, NULL);
	sigaction(SIGALRM, &act, NULL);

	c = METEOR_CAP_CONTINOUS;
	if (ioctl(bdt->bktr->fd, METEORCAPTUR, &c) < 0) {
		warn("METEORCAPTUR");
		return (1);
	}

	/* start the "buffer ready" signal */
	c = SIGUSR1;
	if (ioctl(bdt->bktr->fd, METEORSSIGNAL, &c) < 0) {
		warn("METEORSSIGNAL");
		return (1);
	}

	/* timeout alarm */
	setitimer(ITIMER_REAL, &tim, NULL);

	/* wait for the first signal, or bail if timeout */
	sigsuspend(&sa_mask);
	if (bktr_frame_waiting == 0) {
		warnx("frame grab timeout");
		return (1);
	}
	bktr_frame_waiting = 0;

	gettimeofday(&tp_start, NULL);

	while (bdt->shutdown == 0) {

		display_event(bdt);

		while (bdt->pause == 1) {
			if (tim.it_value.tv_sec != 0) {
				tim.it_value.tv_sec = 0;
				setitimer(ITIMER_REAL, &tim, NULL);
			}
			usleep(50000);
			display_event(bdt);
		}

		if (bktr_frame_waiting == 0)
			sigsuspend(&sa_mask);

		if (bktr_frame_waiting > 0) {
			memcpy(bdt->display->pixels, bdt->bktr->buffer,
			    bdt->bktr->buffer_size);
			bktr_frame_waiting--;
			if (bktr_frame_waiting > 0) {
				if (bdt->verbose > 0)
					warnx("missed %d frames",
					    (int)bktr_frame_waiting);
				bktr_frame_waiting = 0;
			}
			XvShmPutImage(bdt->display->dpy,
			    bdt->display->port, bdt->display->window,
			    bdt->display->gc, bdt->display->xv_image,
			    0, 0, bdt->bktr->width, bdt->bktr->height,
			    0, 0, bdt->display->width,
			    bdt->display->height, False);
			frames_played++;
			/* reset the timeout timer */
			tim.it_value.tv_sec = 1;
			setitimer(ITIMER_REAL, &tim, NULL);
		} else {
			if (bdt->shutdown == 0)
				warnx("frame capture timeout");
			break;
		}

		if ((frames_played > 0) && (frames_played % sequence == 0)) {
			gettimeofday(&tp_end, NULL);
			timersub(&tp_end, &tp_start, &tp_run);
			run_time = tp_run.tv_sec +
			    (double)tp_run.tv_usec / 1000000;
			if (bdt->verbose > 1) {
				fprintf(stderr, "frames: %08ld, "
				    "seconds: %09.2f, fps: %08.5f\r",
				    frames_played, run_time,
				    (double)frames_played / run_time);
				fflush(stderr);
			}
		}
	}
	gettimeofday(&tp_end, NULL);

	if (bdt->verbose > 1)
		fprintf(stderr, "\n");

	/* close down the signals */

	tim.it_value.tv_sec = 0;
	setitimer(ITIMER_REAL, &tim, NULL);

	c = METEOR_SIG_MODE_MASK;
	if (ioctl(bdt->bktr->fd, METEORSSIGNAL, &c) < 0) {
		warn("METEORSSIGNAL");
		ret = 1;
	}

	c = METEOR_CAP_STOP_CONT;
	if (ioctl(bdt->bktr->fd, METEORCAPTUR, &c) < 0) {
		warn("METEORCAPTUR");
		ret = 1;
	}

	act.sa_handler = SIG_DFL;
	sigaction(SIGUSR1, &act, NULL);
	sigaction(SIGALRM, &act, NULL);

	if (bdt->verbose > 0) {
		timersub(&tp_end, &tp_start, &tp_run);
		run_time = tp_run.tv_sec + (double)tp_run.tv_usec / 1000000;
		warnx("playing time: %f seconds", run_time);
		warnx("frames played: %ld", frames_played);
		warnx("fps: %f", (double)frames_played / run_time);
	}

	return (ret);
}


int
set_format(struct bktrplay *bdt)
{
	if (bsdav_vid_fmts[bdt->format].bktr_idx < 0) {
		warnx("%s can't do %s", bdt->bktr->dev,
		    bsdav_vid_fmts[bdt->format].name);
		return (1);
	} else {
		if (xv_init(bdt) > 0) {
			warnx("could not init Xv for %s",
			    bsdav_vid_fmts[bdt->format].name);
			return (1);
		}
	}

	if (ioctl(bdt->bktr->fd, METEORSACTPIXFMT,
	    &bsdav_vid_fmts[bdt->format].bktr_idx) < 0) {
		warn("METEORSACTPIXFMT");
		return (1);
	}

	return (0);
}


int
display_event(struct bktrplay *bdt)
{
char str;

	if (XPending(bdt->display->dpy)) {
		XNextEvent(bdt->display->dpy, &bdt->display->event);
		switch (bdt->display->event.type) {
		case KeyPress:
			if (bdt->verbose > 1)
				warnx("got keypress event");
			XLookupString(&bdt->display->event.xkey, &str, 1,
			    NULL, NULL);
			switch (str) {
			case 'a':
				if (bdt->aspect_lock == 1) {
					bdt->aspect_lock = 0;
				} else {
					bdt->aspect_lock = 1;
					resize_window(bdt, 0);
				}
				break;
			case 'c':
				switch_source(bdt, BSDAV_VIDSOURCE_RCA);
				break;
			case 'f':
				resize_window(bdt, 1);
				break;
			case 'p':
				switch_pause(bdt);
				break;
			case 'q':
				bdt->shutdown = 1;
				break;
			case 's':
				switch_source(bdt, BSDAV_VIDSOURCE_SVID);
				break;
			case 't':
				switch_source(bdt, BSDAV_VIDSOURCE_TUNER);
				break;
			default:
				break;
			}
			break;
		case ClientMessage:
			if (bdt->verbose > 1)
				warnx("got ClientMessage event");
			if (bdt->display->event.xclient.data.l[0] ==
			    bdt->display->wmdelwin) {
				bdt->shutdown = 1;
				break;
			}
			break;
		case ConfigureNotify:
			if (bdt->verbose > 1)
				warnx("got ConigureNotify event");
			resize_window(bdt, 0);
			break;
		default:
			break;
		}
	}
	XSync(bdt->display->dpy, False);

	return (0);
}


int
switch_source(struct bktrplay *bdt, int source)
{
int c;

	c = bsdav_vid_srcs[source].bktr_id;
	if  (ioctl(bdt->bktr->fd, METEORSINPUT, &c) < 0) {
		warn("METEORSINPUT");
		return (1);
	}

	return (0);
}


int
switch_pause(struct bktrplay *bdt)
{
uint c;

	if (bdt->pause == 1) {
		c = SIGUSR1;
		if (ioctl(bdt->bktr->fd, METEORSSIGNAL, &c) < 0) {
			warn("METEORSSIGNAL");
			return (1);
		}
		c = METEOR_CAP_CONTINOUS;
		if (ioctl(bdt->bktr->fd, METEORCAPTUR, &c) < 0) {
			warn("METEORCAPTUR");
			return (1);
		}
		bdt->pause = 0;
	} else {
		bdt->pause = 1;
		c = METEOR_CAP_STOP_CONT;
		if (ioctl(bdt->bktr->fd, METEORCAPTUR, &c) < 0) {
			warn("METEORCAPTUR");
			return (1);
		}
		c = METEOR_SIG_MODE_MASK;
		if (ioctl(bdt->bktr->fd, METEORSSIGNAL, &c) < 0) {
			warn("METEORSSIGNAL");
			return (1);
		}
	}
	return (0);
}


int
resize_window(struct bktrplay *bdt, int fullscreen)
{
XWindowAttributes *winatt;
double aspect;
int new_width;
int new_height;
int new_x;
int new_y;

	aspect = (double)bdt->aspect.n / bdt->aspect.d;

	if (fullscreen == 1) {
		if (bdt->full_screen == 1) {
			new_width = bdt->display->norm_winatt->width;
			new_height = bdt->display->norm_winatt->height;
			new_x = bdt->display->norm_winatt->x;
			new_y = bdt->display->norm_winatt->y;
			bdt->full_screen = 0;
		} else {
			XGetWindowAttributes(bdt->display->dpy,
			    bdt->display->window, bdt->display->norm_winatt);
			new_width = DisplayWidth(bdt->display->dpy,
			    bdt->display->screen_id);
			new_height = DisplayHeight(bdt->display->dpy,
			    bdt->display->screen_id);
			new_x = 0;
			new_y = 0;
			if (bdt->aspect_lock == 1) {
				if (new_width <= new_height * aspect) {
					new_width -= new_width % 16;
					new_height = new_width / aspect;
				} else {
					new_height -= new_height % 16;
					new_width = new_height * aspect;
				}
			}
			bdt->full_screen = 1;
		}
		XMoveResizeWindow(bdt->display->dpy, bdt->display->window,
		    new_x, new_y, new_width, new_height);
	} else {
		if ((winatt = malloc(sizeof(XWindowAttributes))) == NULL) {
			warn("winatt");
			return (1);
		}
		XGetWindowAttributes(bdt->display->dpy, bdt->display->window,
		    winatt);
		new_x = winatt->x;
		new_y = winatt->y;
		new_width = winatt->width;
		new_height = winatt->height;
		free(winatt);
		if (bdt->aspect_lock == 1) {
			new_width = (new_width + (new_height * aspect)) / 2;
			new_width -= new_width % 16;
			new_height = new_width / aspect;
		}
		XResizeWindow(bdt->display->dpy, bdt->display->window,
		    new_width, new_height);
	}
	XSync(bdt->display->dpy, False);
	bdt->display->width = new_width;
	bdt->display->height = new_height;
	XNextEvent(bdt->display->dpy, &bdt->display->event);
	XSync(bdt->display->dpy, True);

	return (0);
}


void
cleanup(struct bktrplay *bdt)
{
	if (bdt->display->shminfo.shmaddr != NULL)
		shmdt(bdt->display->shminfo.shmaddr);
	bdt->display->shminfo.shmaddr = NULL;

	if (bdt->display->shminfo.shmid > 0)
		shmctl(bdt->display->shminfo.shmid, IPC_RMID, 0);
	bdt->display->shminfo.shmid = 0;

	if (bdt->display->xv_image != NULL)
		XFree(bdt->display->xv_image);
	bdt->display->xv_image = NULL;

	if (bdt->display->gc != NULL)
		XFreeGC(bdt->display->dpy, bdt->display->gc);
	bdt->display->gc = NULL;

	if (bdt->display->window != 0)
		XDestroyWindow(bdt->display->dpy, bdt->display->window);
	bdt->display->window = 0;

	if (bdt->display->port != 0)
		XvUngrabPort(bdt->display->dpy, bdt->display->port,
		    CurrentTime);
	bdt->display->port = 0;

	if (bdt->display->dpy != NULL)
		XCloseDisplay(bdt->display->dpy);
	bdt->display->dpy = NULL;

	if (bdt->bktr->buffer != MAP_FAILED)
		bsdav_unmap_vid_buffer(&bdt->bktr->buffer, bdt->bktr->buffer_size);
	bdt->bktr->buffer = MAP_FAILED;

	if (bdt->bktr->fd >= 0)
		close(bdt->bktr->fd);
	bdt->bktr->fd = -1;
}


int
main(int argc, char *argv[])
{
struct bktrplay *bdt;
const char *errstr;
unsigned int p_version;
unsigned int p_release;
unsigned int p_request_base;
unsigned int p_event_base;
unsigned int p_error_base;
char	*disname = NULL;
int	 ch;
int	 err = 0;

	if ((bdt = malloc(sizeof(struct bktrplay))) == NULL) {
		warn("bdt");
		exit(1);
	}
	memset(bdt, 0, sizeof(struct bktrplay));

	if ((bdt->display = malloc(sizeof(struct bp_display))) == NULL) {
		warn("display");
		cleanup(bdt);
		exit(1);
	}
	memset(bdt->display, 0, sizeof(struct bp_display));

	if ((bdt->bktr = malloc(sizeof(struct bp_bktr))) == NULL) {
		warn("bktr");
		cleanup(bdt);
		exit(1);
	}
	memset(bdt->bktr, 0, sizeof(struct bp_bktr));

	/* defaults */
	snprintf(bdt->bktr->dev, FILENAME_MAX, DEFAULT_VIDEO_DEVICE);
	bdt->bktr->norm = BSDAV_VIDNORM_NTSC;
	bdt->bktr->source = BSDAV_VIDSOURCE_TUNER;
	bdt->aspect.n = 4;
	bdt->aspect.d = 3;
	bdt->aspect_lock = 1;
	bdt->verbose = 0;
	bdt->format = BSDAV_VIDFMT_UYVY;

	while ((ch = getopt(argc, argv, "a:c:e:f:i:n:r:s:v")) != -1) {
		switch (ch) {
		case 'a':
			if (bsdav_parse_ratio(optarg, &bdt->aspect) != 0) {
				warnx("invalid aspect: %s", optarg);
				err++;
			} else {
				if (bdt->aspect.n == 0)
					bdt->aspect_lock = 0;
			}
			break;
		case 'c':
			if (bsdav_parse_crop(optarg, &bdt->crop) != 0) {
				warnx("invalid crop: %s", optarg);
				err++;
			}
			break;
		case 'e':
			bdt->format = bsdav_find_vid_fmt(optarg);
			if (bdt->format < 0) {
				warnx("invalid fourcc: %s", optarg);
				err++;
			}
			break;
		case 'f':
			snprintf(bdt->bktr->dev, FILENAME_MAX, optarg);
			break;
		case 'i':
			if (strncmp(optarg, "odd", 3) == 0) {
				bdt->bktr->fields = METEOR_GEO_ODD_ONLY;
			} else if (strncmp(optarg, "even", 3) == 0) {
				bdt->bktr->fields = METEOR_GEO_EVEN_ONLY;
			} else {
				warnx("invalid fields: %s", optarg);
				err++;
			}
			break;
		case 'n':
			bdt->bktr->norm = bsdav_find_vid_norm(optarg);
			if (bdt->bktr->norm < 0) {
				warnx("invalid norm: %s", optarg);
				err++;
			}
			break;
		case 'r':
			bdt->bktr->fps = (int)strtonum(optarg, 0, 30, &errstr);
			if (errstr != NULL) {
				warnx("rate is %s: %s:", errstr, optarg);
				err++;
			}
			break;
		case 's':
			bdt->bktr->source = bsdav_find_vid_source(optarg);
			if (bdt->bktr->source < 0) {
				warnx("invalid source: %s", optarg);
				err++;
			}
			break;
		case 'v':
			bdt->verbose++;
			break;
		default:
			usage();
			cleanup(bdt);
			exit(1);
		}
		if (err > 0)
			break;
	}
	if (err > 0) {
		usage();
		cleanup(bdt);
		exit(1);
	}

	argc -= optind;
	argv += optind;


	if ((bdt->display->norm_winatt =
	    malloc(sizeof(XWindowAttributes))) == NULL) {
		warn("norm_winatt");
		cleanup(bdt);
		exit(1);
	}

	if ((bdt->display->dpy = XOpenDisplay(disname)) == NULL) {
		warnx("cannot open display %s", XDisplayName(NULL));
		cleanup(bdt);
		exit(1);
	}

	if (Success != XvQueryExtension(bdt->display->dpy, &p_version,
	    &p_release, &p_request_base, &p_event_base, &p_error_base)) {
		warnx("Xv not available");
		cleanup(bdt);
		exit(1);
	}

	bdt->display->screen_id = DefaultScreen(bdt->display->dpy);

	if ((bdt->bktr->fd = open(bdt->bktr->dev, O_RDONLY)) < 0) {
		warn("%s", bdt->bktr->dev);
		cleanup(bdt);
		exit(1);
	}

	if (bsdav_bktr_init(bdt->bktr->fd, bdt->bktr->norm, &bdt->bktr->width,
	    &bdt->bktr->height, bdt->bktr->fields, bdt->format,
	    bdt->bktr->source, &bdt->bktr->fps, &bdt->crop) != 0) {
		warnx("failed to init %s", bdt->bktr->dev);
		cleanup(bdt);
		exit(1);
	}

	if (bdt->verbose > 1) {
		warnx("cropping: t = %d, l = %d, b = %d, r = %d",
		    bdt->crop.t, bdt->crop.l, bdt->crop.b, bdt->crop.r);
		warnx("frame size: %dx%d", bdt->bktr->width, bdt->bktr->height);
		if (bdt->bktr->fields > 0) {
			if (bdt->bktr->fields == METEOR_GEO_EVEN_ONLY)
				warnx("grabbing only even fields");
			else if (bdt->bktr->fields == METEOR_GEO_ODD_ONLY)
				warnx("grabbing only odd fields");
		}
	}

	if (set_format(bdt) != 0) {
		warnx("failed to set video format");
		cleanup(bdt);
		exit(1);
	}

	if (bdt->full_screen == 1) {
		bdt->display->width = DisplayWidth(bdt->display->dpy,
		    bdt->display->screen_id);
		bdt->display->height = DisplayHeight(bdt->display->dpy,
		    bdt->display->screen_id);
	} else {
		bdt->display->width = bdt->bktr->width;
		if (bdt->bktr->fields != 0)
			bdt->display->height = bdt->bktr->height * 2;
		else
			bdt->display->height = bdt->bktr->height;
	}

	bdt->bktr->buffer_size = bsdav_map_vid_buffer(&bdt->bktr->buffer,
	    bdt->bktr->fd, bdt->bktr->width, bdt->bktr->height, bdt->format);
	if (bdt->bktr->buffer_size == 0) {
		warnx("failed to mmap hardware buffer");
		cleanup(bdt);
		exit(1);
	}

	if (bdt->verbose > 1)
		warnx("buffer size = %llu", (unsigned long long)bdt->bktr->buffer_size);

	if (create_win(bdt) != 0) {
		warnx("failed to create window");
		cleanup(bdt);
		exit(1);
	}

	if (stream(bdt) != 0) {
		warnx("failed to stream from %s", bdt->bktr->dev);
		cleanup(bdt);
		exit(1);
	}

	cleanup(bdt);

	return (0);
}
