package daemon

import (
	"context"
	"fmt"
	"log/slog"
	"slices"

	"git.sr.ht/~whynothugo/ImapGoose/internal/notify"
	"git.sr.ht/~whynothugo/ImapGoose/internal/watcher"
)

// WorkerTask represents a sync task with completion notification.
// Workers send the mailbox name to Done channel when task completes.
type WorkerTask struct {
	Done       chan<- string
	Mailbox    string
	AbsPaths   []string // Sync these paths specifically.
	ScanLocal  bool     // Scan local filesystem
	ScanRemote bool     // Scan IMAP server
}

// PendingWork represents pending synchronization work for a mailbox.
// Tracks which scans are needed and any file-level syncs.
type PendingWork struct {
	AbsPaths   []string // File-level syncs (empty if ScanLocal=true)
	ScanLocal  bool     // Scan local filesystem
	ScanRemote bool     // Scan IMAP server
}

// Dispatcher coordinates task distribution with mailbox-level serialisation.
// Prevents workers concurrently operating on the same mailbox and deduplicates
// pending tasks.
type Dispatcher struct {
	activeMailboxes map[string]bool
	pendingTasks    map[string]*PendingWork // Pending work per mailbox
	remoteQueue     <-chan notify.RemoteEvent
	localQueue      <-chan watcher.LocalEvent
	workerQueue     chan WorkerTask
	completionQueue chan string
	logger          *slog.Logger
}

// NewDispatcher creates a new task dispatcher.
func NewDispatcher(localQueue <-chan watcher.LocalEvent, remoteQueue <-chan notify.RemoteEvent, logger *slog.Logger) *Dispatcher {
	return &Dispatcher{
		activeMailboxes: make(map[string]bool),
		pendingTasks:    make(map[string]*PendingWork),
		remoteQueue:     remoteQueue,
		localQueue:      localQueue,
		workerQueue:     make(chan WorkerTask), // Unbuffered: dispatcher controls flow.
		completionQueue: make(chan string, 4),  // Buffered to prevent workers blocking.
		logger:          logger,
	}
}

// WorkerQueue returns the channel from which workers read.
func (d *Dispatcher) WorkerQueue() <-chan WorkerTask {
	return d.workerQueue
}

// Run starts the dispatcher's event loop.
// Unified select loop to:
// - Read from remoteQueue and localQueue, handle events
// - Dispatch tasks to workerQueue when mailboxes are idle
// - Handle completions from workers
func (d *Dispatcher) Run(ctx context.Context) error {
	d.logger.Info("Dispatcher starting")
	defer d.logger.Info("Dispatcher stopped")
	defer close(d.workerQueue) // Ensure workerQueue is always closed.

	var nextTask *WorkerTask
	for {
		if nextTask != nil {
			select {
			case <-ctx.Done():
				return ctx.Err()
			case event, ok := <-d.remoteQueue:
				if !ok {
					return fmt.Errorf("remote queue closed")
				}
				d.handleRemoteEvent(event)
			case event, ok := <-d.localQueue:
				if !ok {
					return fmt.Errorf("local queue closed")
				}
				d.handleLocalEvent(event)
			case mailbox := <-d.completionQueue:
				d.markIdle(mailbox)
			case d.workerQueue <- *nextTask:
				d.logger.Debug(
					"Dispatched task",
					"mailbox", nextTask.Mailbox,
					"absPaths", nextTask.AbsPaths,
					"scanLocal", nextTask.ScanLocal,
					"scanRemote", nextTask.ScanRemote,
				)
				d.activeMailboxes[nextTask.Mailbox] = true
				nextTask = nil
			}
		} else {
			select {
			case <-ctx.Done():
				return ctx.Err()
			case event, ok := <-d.remoteQueue:
				if !ok {
					return nil
				}
				d.handleRemoteEvent(event)
			case event, ok := <-d.localQueue:
				if !ok {
					return nil
				}
				d.handleLocalEvent(event)
			case mailbox := <-d.completionQueue:
				d.markIdle(mailbox)
			}
		}

		if nextTask == nil {
			nextTask = d.getNextDispatchableTask()
		}
	}
}

// handleRemoteEvent processes a remote IMAP change notification.
func (d *Dispatcher) handleRemoteEvent(event notify.RemoteEvent) {
	work := d.getOrCreateWork(event.Mailbox)

	if work.ScanRemote {
		d.logger.Debug("Remote scan already queued", "mailbox", event.Mailbox)
		return
	} // If only avoids superfluous log line.

	work.ScanRemote = true
	d.logger.Info("Queued remote scan", "mailbox", event.Mailbox)
}

// handleLocalEvent processes a local filesystem change notification.
func (d *Dispatcher) handleLocalEvent(event watcher.LocalEvent) {
	work := d.getOrCreateWork(event.Mailbox)

	if event.AbsPath == "" {
		// No path, want ScanLocal.
		if work.ScanLocal {
			d.logger.Debug("Local scan already pending", "mailbox", event.Mailbox)
			return
		}
		d.logger.Debug(
			"ScanLocal supersedes file syncs",
			"mailbox", event.Mailbox,
			"superseded", len(work.AbsPaths),
		)
		work.ScanLocal = true
		work.AbsPaths = []string{}
		d.logger.Info("Queued local scan", "mailbox", event.Mailbox)
		return
	}
	// else: AbsPath not nil.

	if work.ScanLocal {
		d.logger.Debug(
			"File sync discarded (local scan pending)",
			"mailbox", event.Mailbox,
			"absPath", event.AbsPath,
		)
		return
	}

	if work.AbsPaths == nil {
		work.AbsPaths = []string{}
	}

	if slices.Contains(work.AbsPaths, event.AbsPath) {
		d.logger.Debug("Deduplicating file sync", "mailbox", event.Mailbox, "absPath", event.AbsPath)
		return
	}

	work.AbsPaths = append(work.AbsPaths, event.AbsPath)
	d.logger.Debug("Queued file sync", "mailbox", event.Mailbox, "absPath", event.AbsPath, "pendingCount", len(work.AbsPaths))

	// Squash many file syncs into full local scan.
	if len(work.AbsPaths) > 5 {
		d.logger.Debug("Squashing file syncs to local scan", "mailbox", event.Mailbox, "count", len(work.AbsPaths))
		work.ScanLocal = true
		work.AbsPaths = []string{}
		d.logger.Info("Queued local scan (squashed)", "mailbox", event.Mailbox)
	}
}

// getOrCreateWork retrieves or creates PendingWork for a mailbox.
func (d *Dispatcher) getOrCreateWork(mailbox string) *PendingWork {
	work := d.pendingTasks[mailbox]
	if work == nil {
		work = &PendingWork{}
		d.pendingTasks[mailbox] = work
	}
	return work
}

// getNextDispatchableTask finds a mailbox with pending work that is currently idle.
// Returns a WorkerTask constructed from PendingWork, with all pending AbsPaths.
func (d *Dispatcher) getNextDispatchableTask() *WorkerTask {
	for mailbox, work := range d.pendingTasks {
		if !d.activeMailboxes[mailbox] {
			task := WorkerTask{
				Mailbox:    mailbox,
				AbsPaths:   work.AbsPaths,
				ScanLocal:  work.ScanLocal,
				ScanRemote: work.ScanRemote,
				Done:       d.completionQueue,
			}

			// Clear pending work and remove from pending tasks.
			delete(d.pendingTasks, mailbox)

			return &task
		}
	}
	return nil
}

// markIdle marks a mailbox as idle (no longer active).
// Next iteration of dispatch loop will dispatch pending tasks for this mailbox if any.
func (d *Dispatcher) markIdle(mailbox string) {
	delete(d.activeMailboxes, mailbox)
	d.logger.Debug("Mailbox idle", "mailbox", mailbox)
}
