package config

import (
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"strconv"
	"strings"

	"codeberg.org/emersion/go-scfg"
)

// Config represents the ImapGoose configuration.
type Config struct {
	Accounts []Account
}

// Account represents a single IMAP account configuration.
type Account struct {
	Name           string
	Server         string
	Username       string
	PasswordCmd    []string // Command and arguments to retrieve password
	Password       string   // Resolved password (populated by ResolvePassword)
	LocalPath      string
	PostSyncCmd    []string // Command to run after each sync.
	Plaintext      bool     // Use plaintext connection (for testing only)
	StateDir       string   // Optional custom state directory (for testing)
	MaxConnections int
}

// Load reads and parses the configuration file.
func Load(path string) (*Config, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, fmt.Errorf("failed to open config file: %w", err)
	}
	defer func() {
		_ = f.Close() // Best effort close
	}()

	cfg, err := scfg.Read(f)
	if err != nil {
		return nil, fmt.Errorf("failed to parse config: %w", err)
	}

	config := &Config{}

	for _, directive := range cfg {
		switch directive.Name {
		case "account":
			if len(directive.Params) != 1 {
				return nil, fmt.Errorf("account directive requires exactly one parameter (name)")
			}

			account := Account{
				Name:           directive.Params[0],
				MaxConnections: 3, // Default: 3 connections (1 NOTIFY + 2 workers)
			}

			// TODO: discovery via SRV records with just email.
			for _, child := range directive.Children {
				switch child.Name {
				case "server":
					if len(child.Params) != 1 {
						return nil, fmt.Errorf("server requires exactly one parameter")
					}
					account.Server = child.Params[0]
				case "username":
					if len(child.Params) != 1 {
						return nil, fmt.Errorf("username requires exactly one parameter")
					}
					account.Username = child.Params[0]
				case "password-cmd":
					if len(child.Params) < 1 {
						return nil, fmt.Errorf("password-cmd requires at least one parameter (command)")
					}
					account.PasswordCmd = child.Params
				case "local-path":
					if len(child.Params) != 1 {
						return nil, fmt.Errorf("local-path requires exactly one parameter")
					}
					account.LocalPath = child.Params[0]
				case "post-sync-cmd":
					if len(child.Params) < 1 {
						return nil, fmt.Errorf("post-sync-cmd requires at least one parameter (command)")
					}
					account.PostSyncCmd = child.Params
				case "max-connections":
					if len(child.Params) != 1 {
						return nil, fmt.Errorf("max-connections requires exactly one parameter")
					}
					var err error
					account.MaxConnections, err = strconv.Atoi(child.Params[0])
					if err != nil {
						return nil, fmt.Errorf("max-connections must be a number: %w", err)
					}
					if account.MaxConnections < 2 {
						return nil, fmt.Errorf("max-connections must be at least 2 (1 for NOTIFY + 1 worker)")
					}
				default:
					return nil, fmt.Errorf("unknown account directive: %s", child.Name)
				}
			}

			// Validate required fields.
			if account.Server == "" {
				return nil, fmt.Errorf("account %s: server is required", account.Name)
			}
			if account.Username != "" && len(account.PasswordCmd) == 0 {
				return nil, fmt.Errorf("account %s: password-cmd is required when username is set", account.Name)
			}
			if account.LocalPath == "" {
				return nil, fmt.Errorf("account %s: local-path is required", account.Name)
			}

			// Expand and validate paths immediately.
			localPath, err := expandPath(account.LocalPath)
			if err != nil {
				return nil, fmt.Errorf("expanding local-path %s: %w", account.LocalPath, err)
			}
			if !filepath.IsAbs(localPath) {
				return nil, fmt.Errorf("local-path must be absolute after expansion (got: %s)", path)
			}
			account.LocalPath = localPath

			config.Accounts = append(config.Accounts, account)
		default:
			return nil, fmt.Errorf("unknown top-level directive: %s", directive.Name)
		}
	}

	if len(config.Accounts) == 0 {
		return nil, fmt.Errorf("no accounts configured")
	}

	return config, nil
}

// ResolvePassword executes the password-cmd and stores the result in Password field.
// The password is read from stdout and trimmed of whitespace/newlines.
func (a *Account) ResolvePassword() error {
	if a.Username == "" {
		a.Password = ""
		return nil
	}

	if len(a.PasswordCmd) == 0 {
		return fmt.Errorf("password-cmd not configured")
	}

	cmd := exec.Command(a.PasswordCmd[0], a.PasswordCmd[1:]...)
	output, err := cmd.Output()
	if err != nil {
		if exitErr, ok := err.(*exec.ExitError); ok {
			return fmt.Errorf("password-cmd failed with exit code %d: %s", exitErr.ExitCode(), string(exitErr.Stderr))
		}
		return fmt.Errorf("password-cmd failed: %w", err)
	}

	// Trim whitespace and newlines from password.
	a.Password = strings.TrimSpace(string(output))

	if a.Password == "" {
		return fmt.Errorf("password-cmd returned empty password")
	}

	return nil
}

// expandPath expands environment variables in the input path.
func expandPath(path string) (string, error) {
	path = os.ExpandEnv(path)

	if strings.HasPrefix(path, "~") {
		home, err := os.UserHomeDir()
		if err != nil {
			return "", fmt.Errorf("failed to get home directory: %w", err)
		}
		if path == "~" {
			path = home
		} else if strings.HasPrefix(path, "~/") {
			path = filepath.Join(home, path[2:])
		} else {
			// ~user is not supported
			return "", fmt.Errorf("local-path with ~user format not supported (got: %s)", path)
		}
	}

	return path, nil
}
