// Check whether an IMAP server has our require CAPABILITIES.
//
// Usage: go run ./cmd/capcheck -server imap.example.com:993 -username user@example.com
package main

import (
	"crypto/tls"
	"flag"
	"fmt"
	"log"
	"os"
	"strings"

	"github.com/emersion/go-imap/v2"
	"github.com/emersion/go-imap/v2/imapclient"
	"golang.org/x/term"
)

func main() {
	server := flag.String("server", "", "IMAP server address (e.g., imap.example.com:993)")
	username := flag.String("username", "", "IMAP username")
	insecure := flag.Bool("insecure", false, "Skip TLS certificate verification")
	flag.Parse()

	if *server == "" || *username == "" {
		fmt.Fprintf(os.Stderr, "Usage: %s -server <server:port> -username <user>\n", os.Args[0])
		fmt.Fprintf(os.Stderr, "Password will be read from stdin (one line)\n")
		flag.PrintDefaults()
		os.Exit(1)
	}

	// Read password from stdin without echoing.
	fmt.Fprintf(os.Stderr, "Password: ")
	passwordBytes, err := term.ReadPassword(int(os.Stdin.Fd()))
	fmt.Fprintf(os.Stderr, "\n")
	if err != nil {
		log.Fatalf("Failed to read password from stdin: %v", err)
	}
	password := strings.TrimSpace(string(passwordBytes))
	if password == "" {
		log.Fatalf("Password cannot be empty")
	}

	// Configure TLS.
	tlsConfig := &tls.Config{
		InsecureSkipVerify: *insecure,
	}

	// Connect to the IMAP server.
	log.Printf("Connecting to %s...", *server)
	client, err := imapclient.DialTLS(*server, &imapclient.Options{
		TLSConfig: tlsConfig,
	})
	if err != nil {
		log.Fatalf("Failed to connect: %v", err)
	}
	defer func() {
		_ = client.Close() // Best effort close
	}()

	log.Println("Connected successfully")

	// Login.
	log.Printf("Logging in as %s...", *username)
	if err := client.Login(*username, password).Wait(); err != nil {
		log.Fatalf("Failed to login: %v", err)
	}
	log.Println("Logged in successfully")

	// Get server capabilities.
	caps := client.Caps()
	log.Println("\nServer capabilities:")
	for cap := range caps {
		log.Printf("  - %s", cap)
	}

	// Check for required capabilities.
	fmt.Println("\nChecking for required capabilities:")

	requiredCaps := []imap.Cap{"UIDPLUS", "NOTIFY", "CONDSTORE", "QRESYNC"}
	allSupported := true

	for _, reqCap := range requiredCaps {
		if caps.Has(reqCap) {
			fmt.Printf("✓ %s: SUPPORTED\n", reqCap)
		} else {
			fmt.Printf("✗ %s: NOT SUPPORTED\n", reqCap)
			allSupported = false
		}
	}

	// Logout (not really necessary I guess).
	if err := client.Logout().Wait(); err != nil {
		log.Printf("Logout error: %v", err)
	}

	if allSupported {
		fmt.Println("\n✓ All required capabilities are supported")
		os.Exit(0)
	} else {
		fmt.Println("\n✗ Some required capabilities are missing")
		os.Exit(1)
	}
}
