// FBNEO NES/Famicom FDS emulator
// (c)2019-2020 dink, kev, iq_132

#include "tiles_generic.h"
#include "bitswap.h"
#include "m6502_intf.h"
#include "nes_apu.h"
#include "ay8910.h"			// mapper 69 (sunsoft / gimmick!)
#include "burn_ym2413.h"	// mapper 85 (vrc7 / lagrange point)
#include "burn_gun.h"		// zapper games
#include "burn_led.h"		// fds disk status
#if !defined (_MSC_VER)
#include <unistd.h>         // for unlink()/rename() in ips_patch()/ips_make()
#endif
#include <errno.h>          // .. unlink()/rename()


static UINT8 DrvReset = 0;
static UINT32 DrvPalette[(0x40 * 8) /*nes*/ + 0x100 /*icons*/];
static UINT8 NESRecalc;

static UINT8 NESJoy1[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 NESJoy2[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 NESJoy3[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 NESJoy4[8]     = { 0, 0, 0, 0, 0, 0, 0, 0 };
static UINT8 DrvInputs[4]   = { 0, 0, 0, 0 };
static UINT8 DrvDips[3]     = { 0, 0, 0 };

static UINT32 JoyShifter[2] = { 0, 0 };
static UINT8 JoyStrobe      = 0;

// Zapper emulation
static INT16 ZapperX;
static INT16 ZapperY;
static UINT8 ZapperFire;
static UINT8 ZapperReload;

// FDS-Buttons
static UINT8 FDSEject;
static UINT8 FDSSwap;

// FDS-Timer (for insert/swap robot)
static UINT32 FDSFrameCounter;
static UINT32 FDSFrameAction;
static UINT32 FDSFrameDiskIcon;

// game-specific options not found in header.  see bottom of cartridge_load()
static UINT32 NESMode = 0;
#define NO_WORKRAM		0x0001 // 6000-7fff reads data openbus
#define BUS_CONFLICTS	0x0002 // rom conflicts with writes, needs special handling
#define ALT_MMC3        0x0004 // alternate mmc3 scanline timing
#define USE_4SCORE      0x0100 // 4-Player device (NES)
#define USE_HORI4P      0x0200 // 4-Player device Mode 2 (Famicom)
#define USE_ZAPPER      0x0400 // Zapper Gun device
#define MAPPER_NOCLEAR  0x0800 // preserve mapper regs on soft-reset
#define IS_PAL          0x1000 // PAL-mode (not fully supported)
#define IS_FDS          0x2000 // Famicom Disk System
#define SHOW_OVERSCAN   0x4000 // - for debugging -
#define ALT_TIMING      0x8000 // for games that use "BIT PPUSTATUS; BIT PPUSTATUS; BPL -"
							   // Assimilate, Star Wars, full_palette.nes, etc.
#define ALT_TIMING2     0x0080 // Don Doko Don 2 doesn't like the nmi delay that gunnac, b-wings, etc needs.

// Usually for Multi-Cart mappers
static UINT32 RESETMode = 0;
#define RESET_POWER     0x0001
#define RESET_BUTTON    0x0002

static struct BurnInputInfo NESInputList[] =
{
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	{"P2 Select",	  BIT_DIGITAL,   NESJoy2 + 2, "p2 select" },
	{"P2 Start",	  BIT_DIGITAL,   NESJoy2 + 3, "p2 start"  },
	{"P2 Up",	  	  BIT_DIGITAL,   NESJoy2 + 4, "p2 up"     },
	{"P2 Down",	  	  BIT_DIGITAL,   NESJoy2 + 5, "p2 down"   },
	{"P2 Left",		  BIT_DIGITAL,   NESJoy2 + 6, "p2 left"   },
	{"P2 Right",	  BIT_DIGITAL,   NESJoy2 + 7, "p2 right"  },
	{"P2 Button B",	  BIT_DIGITAL,   NESJoy2 + 1, "p2 fire 1" },
	{"P2 Button A",	  BIT_DIGITAL,   NESJoy2 + 0, "p2 fire 2" },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
};

STDINPUTINFO(NES)

#define A(a, b, c, d) {a, b, (UINT8*)(c), d}
static struct BurnInputInfo NESZapperInputList[] =
{
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	A("P2 Gun X",     BIT_ANALOG_REL, &ZapperX,   "p2 x-axis" ),
	A("P2 Gun Y",     BIT_ANALOG_REL, &ZapperY,   "p2 y-axis" ),
	{"P2 Fire",       BIT_DIGITAL,  &ZapperFire,  "p2 fire 1" },
	{"P2 Reload",     BIT_DIGITAL,  &ZapperReload,"p2 fire 2" },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
};
#undef A
STDINPUTINFO(NESZapper)

static struct BurnInputInfo NES4ScoreInputList[] =
{
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	{"P2 Select",	  BIT_DIGITAL,   NESJoy2 + 2, "p2 select" },
	{"P2 Start",	  BIT_DIGITAL,   NESJoy2 + 3, "p2 start"  },
	{"P2 Up",	  	  BIT_DIGITAL,   NESJoy2 + 4, "p2 up"     },
	{"P2 Down",	  	  BIT_DIGITAL,   NESJoy2 + 5, "p2 down"   },
	{"P2 Left",		  BIT_DIGITAL,   NESJoy2 + 6, "p2 left"   },
	{"P2 Right",	  BIT_DIGITAL,   NESJoy2 + 7, "p2 right"  },
	{"P2 Button B",	  BIT_DIGITAL,   NESJoy2 + 1, "p2 fire 1" },
	{"P2 Button A",	  BIT_DIGITAL,   NESJoy2 + 0, "p2 fire 2" },

	{"P3 Select",	  BIT_DIGITAL,   NESJoy3 + 2, "p3 select" },
	{"P3 Start",	  BIT_DIGITAL,   NESJoy3 + 3, "p3 start"  },
	{"P3 Up",	  	  BIT_DIGITAL,   NESJoy3 + 4, "p3 up"     },
	{"P3 Down",	  	  BIT_DIGITAL,   NESJoy3 + 5, "p3 down"   },
	{"P3 Left",		  BIT_DIGITAL,   NESJoy3 + 6, "p3 left"   },
	{"P3 Right",	  BIT_DIGITAL,   NESJoy3 + 7, "p3 right"  },
	{"P3 Button B",	  BIT_DIGITAL,   NESJoy3 + 1, "p3 fire 1" },
	{"P3 Button A",	  BIT_DIGITAL,   NESJoy3 + 0, "p3 fire 2" },

	{"P4 Select",	  BIT_DIGITAL,   NESJoy4 + 2, "p4 select" },
	{"P4 Start",	  BIT_DIGITAL,   NESJoy4 + 3, "p4 start"  },
	{"P4 Up",	  	  BIT_DIGITAL,   NESJoy4 + 4, "p4 up"     },
	{"P4 Down",	  	  BIT_DIGITAL,   NESJoy4 + 5, "p4 down"   },
	{"P4 Left",		  BIT_DIGITAL,   NESJoy4 + 6, "p4 left"   },
	{"P4 Right",	  BIT_DIGITAL,   NESJoy4 + 7, "p4 right"  },
	{"P4 Button B",	  BIT_DIGITAL,   NESJoy4 + 1, "p4 fire 1" },
	{"P4 Button A",	  BIT_DIGITAL,   NESJoy4 + 0, "p4 fire 2" },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
};

STDINPUTINFO(NES4Score)

static struct BurnInputInfo NESFDSInputList[] =
{
	{"P1 Select",	  BIT_DIGITAL,   NESJoy1 + 2, "p1 select" },
	{"P1 Start",	  BIT_DIGITAL,   NESJoy1 + 3, "p1 start"  },
	{"P1 Up",	  	  BIT_DIGITAL,   NESJoy1 + 4, "p1 up"     },
	{"P1 Down",	  	  BIT_DIGITAL,   NESJoy1 + 5, "p1 down"   },
	{"P1 Left",		  BIT_DIGITAL,   NESJoy1 + 6, "p1 left"   },
	{"P1 Right",	  BIT_DIGITAL,   NESJoy1 + 7, "p1 right"  },
	{"P1 Button B",	  BIT_DIGITAL,   NESJoy1 + 1, "p1 fire 1" },
	{"P1 Button A",	  BIT_DIGITAL,   NESJoy1 + 0, "p1 fire 2" },

	{"P2 Microphone", BIT_DIGITAL,   NESJoy2 + 2, "p2 select" },
	{"P2 Up",	  	  BIT_DIGITAL,   NESJoy2 + 4, "p2 up"     },
	{"P2 Down",	  	  BIT_DIGITAL,   NESJoy2 + 5, "p2 down"   },
	{"P2 Left",		  BIT_DIGITAL,   NESJoy2 + 6, "p2 left"   },
	{"P2 Right",	  BIT_DIGITAL,   NESJoy2 + 7, "p2 right"  },
	{"P2 Button B",	  BIT_DIGITAL,   NESJoy2 + 1, "p2 fire 1" },
	{"P2 Button A",	  BIT_DIGITAL,   NESJoy2 + 0, "p2 fire 2" },

	{"Swap Disk Side",	  BIT_DIGITAL, &FDSSwap,  "swap"      },
	{"Eject/Insert Disk", BIT_DIGITAL, &FDSEject, "eject"     },

	{"Reset", 		  BIT_DIGITAL,   &DrvReset  , "reset"     },
	{"Dip A",		  BIT_DIPSWITCH, DrvDips + 0, "dip"       },
	{"Dip B",		  BIT_DIPSWITCH, DrvDips + 1, "dip"       },
	{"Dip C",		  BIT_DIPSWITCH, DrvDips + 2, "dip"       },
};

STDINPUTINFO(NESFDS)

static struct BurnDIPInfo NESDIPList[] =
{
	{0x11, 0xf0, 0xff, 0xff, "dip_offset"	},
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "Palette"		},
	{0x02, 0x01, 0x01, 0x00, "Famicom"		},
	{0x02, 0x01, 0x01, 0x01, "RGB"			},
};

STDDIPINFO(NES)

static struct BurnDIPInfo NESZapperDIPList[] =
{
	{0x0d, 0xf0, 0xff, 0xff, "dip_offset"	},
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "Palette"		},
	{0x02, 0x01, 0x01, 0x00, "Famicom"		},
	{0x02, 0x01, 0x01, 0x01, "RGB"			},
};

STDDIPINFO(NESZapper)

static struct BurnDIPInfo NES4ScoreDIPList[] =
{
	{0x21, 0xf0, 0xff, 0xff, "dip_offset"	},
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "Palette"		},
	{0x02, 0x01, 0x01, 0x00, "Famicom"		},
	{0x02, 0x01, 0x01, 0x01, "RGB"			},
};

STDDIPINFO(NES4Score)

static struct BurnDIPInfo NESFDSDIPList[] =
{
	{0x12, 0xf0, 0xff, 0xff, "dip_offset"	},
	{0x00, 0xff, 0xff, 0x00, NULL			},
	{0x01, 0xff, 0xff, 0x00, NULL			},
	{0x02, 0xff, 0xff, 0x00, NULL			},

	{0   , 0xfe, 0   ,    2, "Sprite Limit"	},
	{0x00, 0x01, 0x01, 0x01, "Off"			},
	{0x00, 0x01, 0x01, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Auto Insert/Eject" },
	{0x00, 0x01, 0x02, 0x02, "Off"			},
	{0x00, 0x01, 0x02, 0x00, "On"			},

	{0   , 0xfe, 0   ,    2, "Aspect Ratio"	},
	{0x01, 0x01, 0x01, 0x00, "Off"			},
	{0x01, 0x01, 0x01, 0x01, "4:3"			},

	{0   , 0xfe, 0   ,    2, "Stereoizer"	},
	{0x01, 0x01, 0x02, 0x00, "Off"			},
	{0x01, 0x01, 0x02, 0x02, "On"			},

	{0   , 0xfe, 0   ,    2, "Palette"		},
	{0x02, 0x01, 0x01, 0x00, "Famicom"		},
	{0x02, 0x01, 0x01, 0x01, "RGB"			},
};

STDDIPINFO(NESFDS)

#define SCREEN_WIDTH 256
#define SCREEN_HEIGHT (240-16) // top and bottom 8px is overscan
#define SCREEN_HEIGHT_PAL (SCREEN_HEIGHT + 8)

static UINT8* NES_CPU_RAM;

static UINT8* rom;
static INT32 cyc_counter = 0; // frame-based cycle counter
static UINT64 mega_cyc_counter = 0; // "from reset"-based cycle counter

static UINT8 cpu_open_bus;
static INT32 scanline, pixel;
static UINT8 (*read_nt)(UINT16 address);
static void (*write_nt)(UINT16 address, UINT8 data);

enum Scanline  { VISIBLE = 0, POSTRENDER, VBLANK, PRERENDER };
enum Mirroring { VERTICAL = 0, HORIZONTAL, SINGLE_LOW, SINGLE_HIGH, FOUR_SCREEN, SACHEN };

// Reference: https://wiki.nesdev.com/w/index.php?title=PPU_registers

union PPUCTRL
{
#ifdef LSB_FIRST
	struct {
		UINT8 nttab   : 2;  // Base nametable address (0 = $2000; 1 = $2400; 2 = $2800; 3 = $2C00)
		UINT8 incr    : 1;  // VRAM address increment (0 = 1, 1 = 32)
		UINT8 sprtab  : 1;  // Sprite Pattern table address for 8x8 sprites
		UINT8 bgtab   : 1;  // Background pattern table address
		UINT8 sprsize : 1;  // Sprite size, 8x8 / 8x16
		UINT8 slave   : 1;  // PPU master/slave select
		UINT8 nmi     : 1;  // Generate NMI @ VBL
	} bit;
	UINT8 reg;
#else
	UINT8 reg;
	struct {
		UINT8 nmi     : 1;  // Generate NMI @ VBL
		UINT8 slave   : 1;  // PPU master/slave select
		UINT8 sprsize : 1;  // Sprite size, 8x8 / 8x16
		UINT8 bgtab   : 1;  // Background pattern table address
		UINT8 sprtab  : 1;  // Sprite Pattern table address for 8x8 sprites
		UINT8 incr    : 1;  // VRAM address increment (0 = 1, 1 = 32)
		UINT8 nttab   : 2;  // Base nametable address (0 = $2000; 1 = $2400; 2 = $2800; 3 = $2C00)
	} bit;
#endif
};

union PPUMASK
{
#ifdef LSB_FIRST
	struct {
		UINT8 gray    : 1;  // Grayscale
		UINT8 bgLeft  : 1;  // Show background in left 8 px of screen
		UINT8 sprLeft : 1;  // Show sprite in left 8 px
		UINT8 bg      : 1;  // Show background
		UINT8 spr     : 1;  // Show sprites
		UINT8 red     : 1;  // Emphasize red
		UINT8 green   : 1;  // Emphasize green
		UINT8 blue    : 1;  // Emphasize blue
	} bit;
	UINT8 reg;
#else
	UINT8 reg;
	struct {
		UINT8 blue    : 1;  // Emphasize blue
		UINT8 green   : 1;  // Emphasize green
		UINT8 red     : 1;  // Emphasize red
		UINT8 spr     : 1;  // Show sprites
		UINT8 bg      : 1;  // Show background
		UINT8 sprLeft : 1;  // Show sprite in left 8 px
		UINT8 bgLeft  : 1;  // Show background in left 8 px of screen
		UINT8 gray    : 1;  // Grayscale
	} bit;
#endif
};

union PPUSTATUS
{
#ifdef LSB_FIRST
	struct {
		UINT8 bus     : 5;  // Open bus
		UINT8 spovrf  : 1;  // Sprite overflow (8+ sprites/scanline)
		UINT8 sp0hit  : 1;  // Sprite 0 Hit
		UINT8 VBlank  : 1;  // Vertical blanking
	} bit;
	UINT8 reg;
#else
	UINT8 reg;
	struct {
		UINT8 VBlank  : 1;  // Vertical blanking
		UINT8 sp0hit  : 1;  // Sprite 0 Hit
		UINT8 spovrf  : 1;  // Sprite overflow (8+ sprites/scanline)
		UINT8 bus     : 5;  // Open bus
	} bit;
#endif
};

struct OAMBUF
{
	UINT8 idx;   // # in OAM
	UINT8 x;     // following 4 bytes (x, y, tile, attr) mirror oam_ram
	UINT8 y;
	UINT8 tile;
	UINT8 attr;
	UINT8 tileL; // tile data from CHR
	UINT8 tileH;
};

struct cartridge {
	UINT8	*PRGRom;
	INT32	 PRGRomSize;
	INT32	 PRGRomMask;

	UINT8	*WorkRAM;
	INT32	 WorkRAMSize;
	INT32	 WorkRAMMask;
	INT32	 BatteryBackedSRAM;

	UINT8	*CHRRom;
	INT32	 CHRRomSize;
	UINT8	*CHRRam;
	INT32	 CHRRamSize;

	INT32	 Mapper;
	INT32	 Mirroring;
	INT32	 Trainer;
	UINT32	 Crc;

	INT32	 FDSMode;
	INT32	 FDSDiskSides;
	UINT8	*FDSDiskRawOrig;	// FDS Disk loaded (unmodified)
	INT32    FDSDiskRawOrigSize;

	UINT8	*FDSDiskRaw;		// FDS Disk loaded (possibly modified)
	INT32    FDSDiskRawSize;
	UINT8	*FDSDisk;			// Same as above, but ptr adjusted to skip the header (if avail.)
	INT32    FDSDiskSize;
};

static cartridge Cart;

// forward decl's (used in mappers)
static void set_mirroring(INT32 mode);
static void nametable_map(INT32 nt, INT32 ntbank);
static void nametable_mapraw(INT32 nt, UINT8 *ntraw, UINT8 type);
static void nametable_mapall(INT32 ntbank0, INT32 ntbank1, INT32 ntbank2, INT32 ntbank3);

enum { MEM_RAM = 0, MEM_RAM_RO = 1, MEM_ROM = 2 };
static void mapper_set_chrtype(INT32 type);

static INT32 mapper_init(INT32 mappernum);
static UINT8 (*mapper_prg_read)(UINT16 addr);
static UINT8 mapper_prg_read_int(UINT16 address); // mapper_prg_read points here by default
static void (*mapper_write)(UINT16 address, UINT8 data) = NULL;
static void (*mapper_map)() = NULL;
static void (*mapper_scanline)() = NULL;                // once per scanline
static void (*mapper_cycle)() = NULL;                   // once per cpu-cycle
static void (*mapper_ppu_clock)(UINT16 busaddr) = NULL; // called during visible & prerender
static void (*mapper_ppu_clockall)(UINT16 busaddr) = NULL; // called during every cycle
static void (*mapper_scan_cb)() = NULL;                 // state scanning
static void (*mapper_scan_cb_nvram)() = NULL;           // state scanning (nvram)

static INT32 cartridge_load(UINT8* ROMData, UINT32 ROMSize, UINT32 ROMCRC)
{
	if (ROMData == NULL || ROMSize < 16384 ) {
		bprintf(0, _T("Invalid ROM data!\n"));
		return 1;
	}

	if (memcmp("NES\x1a", &ROMData[0], 4)) {
		bprintf(0, _T("Invalid ROM header!\n"));
		return 1;
	}

	INT32 nes20 = (ROMData[7] & 0xc) == 0x8;

	memset(&Cart, 0, sizeof(Cart));

	Cart.Crc = ROMCRC;
	Cart.PRGRomSize = ROMData[4] * 0x4000;
	Cart.CHRRomSize = ROMData[5] * 0x2000;

	if (Cart.Crc == 0x2a798367) {
		// JY 45-in-1 can't be represented by regular nes header.
		Cart.CHRRomSize = 128 * 0x4000;
	}

	bprintf(0, _T("PRG Size: %d\n"), Cart.PRGRomSize);
	bprintf(0, _T("CHR Size: %d\n"), Cart.CHRRomSize);

	if (ROMData[6] & 0x8)
		Cart.Mirroring = 4;
	else
		Cart.Mirroring = ROMData[6] & 1;

	switch (Cart.Mirroring) {
		case 0: set_mirroring(HORIZONTAL); break;
		case 1: set_mirroring(VERTICAL); break;
		case 4: set_mirroring(FOUR_SCREEN); break;
	}

	// Parse MAPPER iNES + NES 2.0
	Cart.Mapper = (ROMData[6] >> 4) | (ROMData[7] & 0xf0);

	if (!memcmp("DiskDude!", &ROMData[7], 9)) {
		bprintf(PRINT_ERROR, _T("``DiskDude!'' spam, ignoring upper bits of mapper.\n"));

		Cart.Mapper &= 0x0f; // remove spam from upper bits of mapper
	}

	if (nes20) {
		Cart.Mapper |= (ROMData[8] & 0x0f) << 8;
		if (Cart.Mapper & 0xf00)
			bprintf(0, _T("NES 2.0 Extended Mapper: %d\n"), Cart.Mapper);
	}

	// Mapper EXT-hardware inits
	// Initted here, because mapper_init() is called on reset
	if (Cart.Mapper == 69) { // SunSoft fme-7 (5b) audio expansion - ay8910
		AY8910Init(0, 1789773 / 2, 1);
		AY8910SetAllRoutes(0, 0.70, BURN_SND_ROUTE_BOTH);
		AY8910SetBuffered(M6502TotalCycles, 1789773);
	}

	if (Cart.Mapper == 85) { // VRC7 audio expansion - YM2413
		BurnYM2413Init(3579545, 1);
		BurnYM2413SetAllRoutes(2.00, BURN_SND_ROUTE_BOTH);
	}

	Cart.Trainer = (ROMData[6] & 0x4) >> 2;
	Cart.BatteryBackedSRAM = (ROMData[6] & 0x2) >> 1;

	if (Cart.Trainer) bprintf(PRINT_ERROR, _T("Has Trainer data!\n"));

	Cart.PRGRom = ROMData + 0x10 + (Cart.Trainer ? 0x200 : 0);

	// Default CHR-Ram size (8k), always set-up (for advanced mappers, etc)
	Cart.CHRRamSize = 0x2000;

	if (nes20) {
		// NES2.0 header specifies CHR-Ram size (Nalle Land, Haunted Halloween '86)
		Cart.CHRRamSize = 64 << (ROMData[0xb] & 0xf);
	}

	if (Cart.Mapper == 30 || Cart.Mapper == 111) { // UNROM-512, GTROM(Cheapocabra) defaults to 32k chr-ram
		Cart.CHRRamSize = 0x8000;
	}

	if (Cart.Crc == 0xf0847322) {
		Cart.CHRRamSize = 0x4000; // 16k chr-ram for Videomation
	}

	if (Cart.Crc == 0xdd65a6cc) { // Street Heroes 262
		Cart.Mapper = 262;
	}

	if (Cart.Crc == 0xd2f19ba1) { // Haradius Zero (flash-version)
		Cart.Mapper = 303; // fake mapper#
	}

	Cart.CHRRam = (UINT8*)BurnMalloc(Cart.CHRRamSize);

	if (Cart.CHRRomSize) {
		Cart.CHRRom = Cart.PRGRom + Cart.PRGRomSize;
		mapper_set_chrtype(MEM_ROM);
		bprintf(0, _T("Using ROM-supplied CHR data\n"));
	} else {
		mapper_set_chrtype(MEM_RAM);
		bprintf(0, _T("Using RAM CHR data (%dk bytes)\n"), Cart.CHRRamSize / 1024);
	}

	// set-up Cart PRG-RAM/WORK-RAM (6000-7fff)

	Cart.WorkRAMSize = (Cart.Mapper != 5) ? 0x2000 : (8 * 0x2000);

	if (nes20) {
		// NES2.0 header specifies NV-Ram size (ex. Nova the Squirrel)
		Cart.WorkRAMSize = 64 << ((ROMData[0xa] & 0xf0) >> 4);
	}

	switch (ROMCRC) {
		case 0x478a04d8:
			Cart.WorkRAMSize = 0x4000; // Genghis Khan 16k SRAM/WRAM
			break;

		case 0x6f5f3bd2:
		case 0x6f8906ad:
			Cart.WorkRAMSize = 0x8000; // Final Fantasy 1 & 2 (Japan/T-eng) 32k SRAM/WRAM
			break;
	}

	bprintf(0, _T("Cartridge RAM: %d\n"), Cart.WorkRAMSize);
	Cart.WorkRAM = (UINT8*)BurnMalloc(Cart.WorkRAMSize);

	// set-up MAPPER
	bprintf(0, _T("Cartridge Mapper: %d   Mirroring: "), Cart.Mapper);
	switch (Cart.Mirroring) {
		case 0: bprintf(0, _T("Horizontal")); break;
		case 1: bprintf(0, _T("Vertical")); break;
		case 4: bprintf(0, _T("Four-Screen")); break;
		default: bprintf(0, _T("???")); break;
	}
	bprintf(0, _T(" (%d)\n"), Cart.Mirroring);

	if (mapper_init(Cart.Mapper) == -1) {
		bprintf(0, _T("Unsupported mapper\n"));
		return 1;
	}

	if (Cart.BatteryBackedSRAM) {
		bprintf(0, _T("Game has battery-backed sRAM\n"));
	}

	Cart.PRGRomMask = Cart.PRGRomSize - 1;
	Cart.WorkRAMMask = Cart.WorkRAMSize - 1;

	// Game-specific stuff:
	NESMode = 0;

	if (Cart.Mapper == 7 || Cart.Mirroring == 4) // Mapper 7 or 4-way mirroring usually gets no workram (6000-7fff)
		NESMode |= NO_WORKRAM;

	NESMode |= (ROMCRC == 0xab29ab28) ? BUS_CONFLICTS : 0; // Dropzone
	NESMode |= (ROMCRC == 0xe3a6d7f6) ? BUS_CONFLICTS : 0; // Cybernoid
	NESMode |= (ROMCRC == 0x552a903a) ? BUS_CONFLICTS : 0; // Huge Insect
	NESMode |= (ROMCRC == 0xb90a1ca1) ? NO_WORKRAM : 0; // Low G Man
	NESMode |= (ROMCRC == 0xa905cc12) ? NO_WORKRAM : 0; // Bill & Ted
	NESMode |= (ROMCRC == 0xd2f19ba1) ? NO_WORKRAM : 0; // Haradius Zero
	NESMode |= (ROMCRC == 0x585f3500) ? ALT_MMC3 : 0; // Darkwing Duck (T-Chi)
	NESMode |= (ROMCRC == 0x560142bc) ? ALT_TIMING2 : 0; // don doko don 2
	NESMode |= (ROMCRC == 0x3616c7dd) ? ALT_TIMING : 0; // days of thunder
	NESMode |= (ROMCRC == 0xeb506bf9) ? ALT_TIMING : 0; // star wars
	NESMode |= (ROMCRC == 0xa2d504a8) ? ALT_TIMING : 0; // assimilate
	NESMode |= (ROMCRC == 0x4df9d7c8) ? ALT_TIMING : 0; // overlord
	NESMode |= (ROMCRC == 0x85784e11) ? ALT_TIMING : 0; // blargg full palette
	NESMode |= (ROMCRC == 0x5da28b4f) ? ALT_TIMING : 0; // cmc! wall demo
	NESMode |= (ROMCRC == 0xab862073) ? ALT_TIMING : 0; // ultimate frogger champ.
	NESMode |= (ROMCRC == 0x2a798367) ? ALT_TIMING : 0; // jy 45-in-1
	NESMode |= (ROMCRC == 0x149e367f) ? IS_PAL : 0; // Lion King, The
	NESMode |= (ROMCRC == 0xbf80b241) ? IS_PAL : 0; // Mr. Gimmick
	NESMode |= (ROMCRC == 0x4cf12d39) ? IS_PAL : 0; // Elite
	NESMode |= (ROMCRC == 0x3a0b6dd2) ? IS_PAL : 0; // Hero Quest
	NESMode |= (ROMCRC == 0x6d1e30a7) ? IS_PAL : 0; // TMHT
	NESMode |= (ROMCRC == 0xa5bbb96b) ? IS_PAL : 0; // TMHTII
	NESMode |= (ROMCRC == 0x6453f65e) ? IS_PAL : 0; // Uforia
	NESMode |= (ROMCRC == 0x55cbc495) ? IS_PAL : 0; // Super Turrican
	NESMode |= (ROMCRC == 0x11a245a0) ? IS_PAL : 0; // Rod Land
	NESMode |= (ROMCRC == 0xa535e1be) ? IS_PAL : 0; // Rackets and Rivals
	NESMode |= (ROMCRC == 0x8236d3e0) ? IS_PAL : 0; // Probotector
	NESMode |= (ROMCRC == 0x1d2fb2b7) ? IS_PAL : 0; // Shadow Warriors
	NESMode |= (ROMCRC == 0x96b6a919) ? IS_PAL : 0; // Shadow Warriors II
	NESMode |= (ROMCRC == 0x22ad4753) ? IS_PAL : 0; // Parodius
	NESMode |= (ROMCRC == 0xdcd55bec) ? IS_PAL : 0; // Aladdin
	NESMode |= (ROMCRC == 0xe08a5881) ? IS_PAL : 0; // Beauty and the Beast
	NESMode |= (ROMCRC == 0xcbde707e) ? IS_PAL : 0; // International Cricket
	NESMode |= (ROMCRC == 0xe2685bbf) ? IS_PAL : 0; // Kick Off
	NESMode |= (ROMCRC == 0xab21ab5f) ? IS_PAL : 0; // Noah's Ark
	NESMode |= (ROMCRC == 0xab29ab28) ? IS_PAL : 0; // Dropzone
	NESMode |= (ROMCRC == 0xde7e6767) ? IS_PAL : 0; // Asterix
	NESMode |= (ROMCRC == 0xdc7a16e6) ? IS_PAL : 0; // Parasol Stars
	NESMode |= (ROMCRC == 0xfac97247) ? IS_PAL : 0; // Rainbow Islands (Ocean)

	if (NESMode) {
		bprintf(0, _T("Game-specific configuration:\n"));

		for (UINT16 i = 1; i != 0x0000; i <<= 1) {
			switch (NESMode & i) {
				case NO_WORKRAM:
					bprintf(0, _T("*  Disabling cart. work-ram (6000-7fff)\n"));
					break;

				case BUS_CONFLICTS:
					bprintf(0, _T("*  Enabling BUS-CONFLICTS.\n"));
					break;

				case ALT_TIMING:
					bprintf(0, _T("*  Enabling ALT-TIMING.\n"));
					break;

				case ALT_TIMING2:
					bprintf(0, _T("*  Enabling ALT-TIMING2.\n"));
					break;

				case IS_PAL:
					bprintf(0, _T("*  Enabling PAL weirdness.\n"));
					break;
			}
		}
	}

	return 0;
}

static INT32 filediff(TCHAR *file1, TCHAR *file2)
{
	FILE *fp1, *fp2;
	INT32 len1, len2;
	UINT8 c1, c2;

	fp1 = _tfopen(file1, _T("rb"));
	if (!fp1) return -1;
	fp2 = _tfopen(file2, _T("rb"));
	if (!fp2) {
		fclose(fp1);
		return -2;
	}

	fseek(fp1, 0, SEEK_END);
	len1 = ftell(fp1);
	fseek(fp1, 0, SEEK_SET);

	fseek(fp2, 0, SEEK_END);
	len2 = ftell(fp2);
	fseek(fp2, 0, SEEK_SET);

	if (!len1 || !len2 || len1 != len2) {
		fclose(fp1);
		fclose(fp2);
		bprintf(0, _T("filediff(): length mismatch\n"));
		return -3;
	}

	for (INT32 i = 0; i < len1; i++) {
		fread(&c1, 1, 1, fp1);
		fread(&c2, 1, 1, fp2);
		if (c1 != c2) {
			fclose(fp1);
			fclose(fp2);
			bprintf(0, _T("filediff(): difference at offset $%x\n"));
			return -3;
		}
	}

	fclose(fp1);
	fclose(fp2);

	return 0; // file1 == file2
}

static INT32 ips_make(UINT8 *orig_data, UINT8 *new_data, INT32 size_data, TCHAR *ips_dir, TCHAR *ips_fn)
{
	#ifndef BUILD_WIN32
	#define _wunlink unlink
	#define _wrename rename
	#endif
	char iPATCH[6] = "PATCH";
	char iEOF[4] = "EOF";
	TCHAR ips_path_fn[MAX_PATH];
	TCHAR ips_path_fn_temp[MAX_PATH];
	INT32 ips_address;
	UINT8 ips_address_c[3];

	INT32 ips_length;
	UINT8 ips_length_c[2];

	INT32 rle_good;
	INT32 rle_length;
	UINT8 rle_byte;

	INT32 is_difference = 0;
	for (INT32 i = 0; i < size_data; i++)
	{
		if (orig_data[i] != new_data[i]) {
			is_difference = 1;
			break;
		}
	}

	if (is_difference == 0) return -2; // no change!

	_stprintf(ips_path_fn, _T("%s%s"), ips_dir, ips_fn);
	_stprintf(ips_path_fn_temp, _T("%s%s.temp"), ips_dir, ips_fn);

	bprintf(0, _T("ips_make() writing to [%s]\n"), ips_path_fn_temp);
	FILE *f = _tfopen(ips_path_fn_temp, _T("wb"));
	if (!f) return -1; // uhoh! (can't open file for writing)

	fwrite(&iPATCH, 1, 5, f);
	for (INT32 i = 0; i < size_data; i++)
	{
		if (orig_data[i] == new_data[i]) continue;

		ips_address = i;

		rle_good = 0;
		rle_length = 0;
		rle_byte = new_data[i];
		ips_length = 0;

		if (ips_address == 0x454f46) { // 'EOF'
			ips_length = 1; // Patch @ BAD ADDRESS (0x454f46), let's go back 1 byte and start here.
			ips_address--;
			if (rle_byte == new_data[ips_address]) {
				//printf("BAD ADDRESS-1 + RLE possible.\n");
				rle_length++;
			}
		}

		while ( (i < size_data) && (ips_length < 0xfce2)
			   && ((orig_data[i] != new_data[i])
				   || (rle_good && rle_byte == new_data[i])) ) // rle-opti: so we don't create multiple rle-chunks if a couple bytes are unchanged
		{
			if (rle_byte == new_data[i]) {
				if (rle_length == ips_length && rle_length > 3) {
					rle_good = 1;
				}
				rle_length++;
				if (rle_length > 5 && rle_good == 0) {
					// dump patch from before rle so the next contiguous block
					// is the rle block.
					if (i - rle_length == 0x454f46-1) {
						// this is not a good spot to rewind to.. abort rle for now.
						rle_length = 0;
					} else {
						i -= rle_length;
						ips_length -= rle_length;
						i--;
						break;
					}
				}
			} else if (rle_good) {
				i--;
				break;
			} else {
				rle_length = 0;
				rle_byte = new_data[i];
			}

			ips_length++;
			i++;
		}
		//printf("%spatchy @ %x   len %x  rle_len %x\n", ((rle_good)?"RLE-":""),ips_address, ips_length, rle_length);
		ips_address_c[0] = (ips_address >> 16) & 0xff;
		ips_address_c[1] = (ips_address >>  8) & 0xff;
		ips_address_c[2] = (ips_address >>  0) & 0xff;

		if (rle_good) ips_length = 0;

		ips_length_c[0] = (ips_length >> 8) & 0xff;
		ips_length_c[1] = (ips_length >> 0) & 0xff;

		fwrite(&ips_address_c, 1, 3, f);
		fwrite(&ips_length_c, 1, 2, f);
		if (rle_good) {
			ips_length_c[0] = (rle_length >> 8) & 0xff;
			ips_length_c[1] = (rle_length >> 0) & 0xff;
			fwrite(&ips_length_c, 1, 2, f);
			fwrite(&rle_byte, 1, 1, f);
		} else {
			fwrite(&new_data[ips_address], 1, ips_length, f);
		}
	}

	fwrite(&iEOF, 1, 3, f);

	fclose(f);

	// Check if the newly written patch is the same as previously written..
	if (filediff(ips_path_fn_temp, ips_path_fn)) {
		// Patch differs!
		// re-order backups
		const INT32 MAX_BACKUPS = 10;
		for (INT32 i = MAX_BACKUPS; i >= 0; i--) {
			TCHAR szBackupNameTo[MAX_PATH];
			TCHAR szBackupNameFrom[MAX_PATH];

			_stprintf(szBackupNameTo, _T("%s%s.backup%d"), ips_dir, ips_fn, i + 1);
			_stprintf(szBackupNameFrom, _T("%s%s.backup%d"), ips_dir, ips_fn, i);
			if (i == MAX_BACKUPS) {
				_wunlink(szBackupNameFrom); // make sure there is only MAX_BACKUPS :)
			} else {
				_wrename(szBackupNameFrom, szBackupNameTo); //derp.ips.backup0 -> derp.ips.backup1
				if (i == 0) {
					_wrename(ips_path_fn, szBackupNameFrom); //derp.ips -> derp.ips.backup0
				}
			}
		}
		// Rename temp patch filename to real patch filename
		int rc = _wrename(ips_path_fn_temp, ips_path_fn);
		bprintf(0, _T("wrename rc = %d   errno = %d\n"), rc, errno);
		bprintf(0, _T("-- lastly temp to non-temp -- rename %s %s\n"), ips_path_fn_temp, ips_path_fn);
		bprintf(0, _T("--- Fresh new IPS patch written: %s\n"), ips_path_fn);
	} else {
		bprintf(0, _T("--- IPS Patch from this session same as last - aborting write!\n"));
		return -2; // no change!
	}

	return 0;
}

static INT32 ips_patch(UINT8 *data, INT32 size_data, TCHAR *ips_fn)
{
	UINT8 buf[7] = "\0\0\0\0\0\0";

	UINT32 ips_address;
	UINT32 ips_length;

	INT32 rle_repeat;
	UINT8 rle_byte;

	FILE *f = _tfopen(ips_fn, _T("rb"));
	if (!f) return -1;

	fread(&buf, 1, 5, f);
	if (memcmp(&buf, "PATCH", 5) != 0) {
		fclose(f);
		return -2;
	}

	while (!feof(f))
	{
		memset(&buf, 0, 3);
		fread(buf, 1, 3, f);
		if (memcmp(&buf, "EOF", 3) == 0) {
			break;
		}
		ips_address = ((buf[0] << 16) & 0xff0000) | ((buf[1] << 8) & 0xff00) | (buf[2] & 0xff);

		memset(&buf, 0, 3);
		fread(buf, 1, 2, f);
		ips_length = ((buf[0] << 8) & 0xff00) | (buf[1] & 0xff);

		if (ips_length == 0) { // RLE chunk
			fread(buf, 1, 3, f);
			rle_repeat = ((buf[0] << 8) & 0xff00) | (buf[1] & 0xff);
			rle_byte = buf[2];

			if (ips_address + rle_repeat <= size_data) { // ok to patch
				memset(&data[ips_address], rle_byte, rle_repeat);
			} else {
				//printf("patch grow disabled, aborting.\n");
				break;
			}
		} else { // normal chunk
			if (ips_address + ips_length <= size_data) { // ok to patch
				fread(&data[ips_address], 1, ips_length, f);
			} else {
				//printf("patch grow disabled, aborting.\n");
				break;
			}
		}
	}

	fclose(f);

	return 0;
}

static INT32 fds_load(UINT8* ROMData, UINT32 ROMSize, UINT32 ROMCRC)
{
	bprintf(0, _T("FDS Loader\n"));
	if (NULL == ROMData || ROMSize < 8192 ) {
		bprintf(0, _T("Invalid ROM data!\n"));
		return 1;
	}

	// FDS mem map:
	// 6000 - dfff RAM
	// e000 - ffff ROM (FDS Bios)

	memset(&Cart, 0, sizeof(Cart));

	Cart.FDSMode = 1;
	Cart.FDSDiskRaw = (UINT8*)BurnMalloc(0x100000);
	Cart.FDSDiskRawOrig = (UINT8*)BurnMalloc(0x100000);
	Cart.FDSDisk = Cart.FDSDiskRaw;
	Cart.FDSDiskRawSize = ROMSize;
	Cart.FDSDiskRawOrigSize = ROMSize;
	Cart.FDSDiskSize = ROMSize;
	Cart.FDSDiskSides = ROMSize / 65500;
	if (BurnLoadRom(Cart.FDSDiskRaw, 0, 1)) return 1; // load FDS Disk Image
	if (BurnLoadRom(Cart.FDSDiskRawOrig, 0, 1)) return 1; // load FDS Disk Image

	// Load IPS patch (aka: disk-saves @ exit)
	TCHAR szFilename[MAX_PATH];
	_stprintf(szFilename, _T("%s%s.ips"), szAppEEPROMPath, BurnDrvGetText(DRV_NAME));
	INT32 ips = ips_patch(Cart.FDSDiskRaw, Cart.FDSDiskRawSize, szFilename);
	bprintf(0, _T("* FDS DISK patch: %s\n"), (ips == 0) ? _T("Loaded") : _T("Can't Load/Not Found."));

	if (!memcmp("FDS\x1a", &Cart.FDSDiskRaw[0], 4) && ROMSize > 0x10) {
		Cart.FDSDisk += 0x10;
		Cart.FDSDiskSize -= 0x10;
		bprintf(0, _T("* Skipping useless fds header..\n"));
	}

	Cart.Crc = ROMCRC;
	Cart.PRGRomSize = 0x8000;
	Cart.CHRRomSize = 0; // ram only
	Cart.Mirroring = 1; // for bios, but fds-mapper-controlled
	Cart.Mapper = 0x808; // fake mapper# for fds

	bprintf(0, _T("Disk Size: %d (%d sides)\n"), ROMSize, Cart.FDSDiskSides);
	bprintf(0, _T("PRG Size: %d\n"), Cart.PRGRomSize);
	bprintf(0, _T("CHR Size: %d\n"), Cart.CHRRomSize);

	switch (Cart.Mirroring) {
		case 0: set_mirroring(HORIZONTAL); break;
		case 1: set_mirroring(VERTICAL); break;
		case 4: set_mirroring(FOUR_SCREEN); break;
	}

//	Cart.BatteryBackedSRAM = (ROMData[6] & 0x2) >> 1;
	Cart.PRGRom = ROMData;

	// Default CHR-Ram size (8k), always set-up (for advanced mappers, etc)
	Cart.CHRRamSize = 0x2000;
	Cart.CHRRam = (UINT8*)BurnMalloc(Cart.CHRRamSize);
	mapper_set_chrtype(MEM_RAM);
	bprintf(0, _T("Using RAM CHR data (%dk bytes)\n"), Cart.CHRRamSize / 1024);

	// set-up Cart PRG-RAM/WORK-RAM (6000-7fff)
	Cart.WorkRAMSize = 0x2000;
	bprintf(0, _T("Cartridge RAM: %d\n"), Cart.WorkRAMSize);
	Cart.WorkRAM = (UINT8*)BurnMalloc(Cart.WorkRAMSize);

	// set-up MAPPER
	bprintf(0, _T("Cartridge Mapper: %d   Mirroring: "), Cart.Mapper);
	switch (Cart.Mirroring) {
		case 0: bprintf(0, _T("Horizontal")); break;
		case 1: bprintf(0, _T("Vertical")); break;
		case 4: bprintf(0, _T("Four-Screen")); break;
		default: bprintf(0, _T("???")); break;
	}
	bprintf(0, _T(" (%d)\n"), Cart.Mirroring);

	if (mapper_init(Cart.Mapper) == -1) {
		bprintf(0, _T("Unsupported mapper\n"));
		return 1;
	}

	if (Cart.BatteryBackedSRAM) {
		bprintf(0, _T("Game has battery-backed sRAM\n"));
	}

	Cart.PRGRomMask = Cart.PRGRomSize - 1;
	Cart.WorkRAMMask = Cart.WorkRAMSize - 1;

	// Game-specific stuff:
	NESMode = IS_FDS;

	if (NESMode) {
		bprintf(0, _T("Game-specific configuration:\n"));

		switch (NESMode) {
			case NO_WORKRAM:
				bprintf(0, _T("*  Disabling cart. work-ram (6000-7fff)\n"));
				break;

			case BUS_CONFLICTS:
				bprintf(0, _T("*  Enabling BUS-CONFLICTS.\n"));
				break;

			case ALT_TIMING:
				bprintf(0, _T("*  Enabling ALT-TIMING.\n"));
				break;

			case IS_PAL:
				bprintf(0, _T("*  Enabling PAL weirdness.\n"));
				break;
		}
	}

	return 0;
}

// ---- mapper system begins! ----
static UINT32 PRGMap[4];
static UINT8  PRGType[4];
static UINT32 CHRMap[8];
static UINT8  CHRType[8]; // enum { MEM_RAM = 0, MEM_RAM_RO = 1, MEM_ROM = 2};
static UINT8  mapper_regs[0x20]; // General-purpose mapper registers (8bit)
static UINT16 mapper_regs16[0x20]; // General-purpose mapper registers (16bit)
static INT32 mapper_irq_exec; // cycle-delayed irq for mapper_irq();
static void mapper_irq(INT32 cyc); // forward

// mapping expansion ram/rom (6000 - 7fff)  refer to mapper69 for hookup info
static INT32 PRGExpMap;
static void  (*cart_exp_write)(UINT16 address, UINT8 data) = NULL;
static UINT8 (*cart_exp_read)(UINT16 address) = NULL;
static INT32 cart_exp_write_abort; // abort fallthrough - see mapper68
// mapping 4020 - 5fff, use these callbacks. refer to Sachen 8259
static void  (*psg_area_write)(UINT16 address, UINT8 data) = NULL;
static UINT8 (*psg_area_read)(UINT16 address) = NULL;
static UINT8 read_nt_int(UINT16 address);

static void mapper_map_prg(INT32 pagesz, INT32 slot, INT32 bank, INT32 type = MEM_ROM) // 8000 - ffff
{
	INT32 ramromsize = (type == MEM_ROM) ? Cart.PRGRomSize : Cart.WorkRAMSize;

	if (bank < 0) { // negative bank == map page from end of rom
		bank = (ramromsize / (pagesz * 1024)) + bank;
	}

	for (INT32 i = 0; i < (pagesz / 8); i++) {
		PRGMap[(pagesz / 8) * slot + i] = (pagesz * 1024 * bank + 0x2000 * i) % ramromsize;
		PRGType[(pagesz / 8) * slot + i] = type;
	}
}

static void mapper_map_exp_prg(INT32 bank, INT32 type = MEM_ROM) // 6000 - 7fff area (cartridge expansion ram/sram or battery backed ram)
{
	INT32 ramromsize = (type == MEM_ROM) ? Cart.PRGRomSize : Cart.WorkRAMSize;

	if (bank < 0) { // negative bank == map page from end of rom
		bank = (ramromsize / 0x2000) + bank;
	}

	PRGExpMap = (0x2000 * bank) % ramromsize;
}

static void mapper_set_chrtype(INT32 type)
{
	for (INT32 i = 0; i < 8; i++)
		CHRType[i] = type;
}

static void mapper_map_chr(INT32 pagesz, INT32 slot, INT32 bank)
{
	for (INT32 i = 0; i < pagesz; i++) {
		switch (CHRType[pagesz * slot + i]) {
			case MEM_ROM:
				CHRMap[pagesz * slot + i] = (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRomSize;
				break;

			case MEM_RAM:
			case MEM_RAM_RO:
				CHRMap[pagesz * slot + i] = (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRamSize;
				break;
		}
	}
}

static void mapper_map_chr_ramrom(INT32 pagesz, INT32 slot, INT32 bank, INT32 type)
{
	for (INT32 i = 0; i < pagesz; i++) {
		switch (type) {
			case MEM_ROM:
				CHRMap[pagesz * slot + i] = (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRomSize;
				CHRType[pagesz * slot + i] = MEM_ROM;
				break;

			case MEM_RAM:
			case MEM_RAM_RO:
				CHRMap[pagesz * slot + i] = (pagesz * 1024 * bank + 1024 * i) % Cart.CHRRamSize;
				CHRType[pagesz * slot + i] = MEM_RAM;
				break;
		}
	}
}

static UINT8 mapper_prg_read_int(UINT16 address) // mapper_prg_read points here
{
	switch (PRGType[(address & ~0x8000) / 0x2000]) {
		case MEM_ROM:
			return Cart.PRGRom[PRGMap[(address & ~0x8000) / 0x2000] + (address & 0x1fff)];
		case MEM_RAM:
			return Cart.WorkRAM[PRGMap[(address & ~0x8000) / 0x2000] + (address & 0x1fff)];
	}

	bprintf(0, _T("PRGType[] corruption?\n"));
	return 0x00;
}

static void mapper_prg_write(UINT16 address, UINT8 data)
{
	switch (PRGType[(address & ~0x8000) / 0x2000]) {
		case MEM_RAM:
			Cart.WorkRAM[PRGMap[(address & ~0x8000) / 0x2000] + (address & 0x1fff)] = data;
	}
}

static UINT8 mapper_chr_read(UINT16 address)
{
	switch (CHRType[address / 1024]) {
		case MEM_ROM:
			return Cart.CHRRom[CHRMap[address / 1024] + (address & (1024 - 1))];

		case MEM_RAM:
		case MEM_RAM_RO:
			return Cart.CHRRam[CHRMap[address / 1024] + (address & (1024 - 1))];
	}

	return 0x00;
}

static void mapper_chr_write(UINT16 address, UINT8 data)
{
	if (CHRType[address / 1024] == MEM_RAM) {
		Cart.CHRRam[CHRMap[address / 1024] + (address & (1024 - 1))] = data;
	}
}

// Flippy Disk ICON (thanks iq_132! :)
static UINT8 disk_ab[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0f, 0x01, 0x0e, 0x01, 0x03, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0c, 0x0d, 0x00, 0x0d, 0x0e, 0x0a, 0x00, 0x02, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0d, 0x00, 0x00, 0x0c, 0x09, 0x0f, 0x00, 0x00, 0x06, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x08, 0x00, 0x00, 0x0d, 0x09, 0x09, 0x09, 0x0a, 0x00, 0x00, 0x03, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x06, 0x09, 0x09, 0x09, 0x0d, 0x00, 0x00, 0x06, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x00, 0x09, 0x09, 0x00, 0x07, 0x03, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x00, 0x00, 0x00, 0x0e, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x0a, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x00, 0x09, 0x09, 0x00, 0x07, 0x0f, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x03, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x0c, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x00, 0x00, 0x07, 0x03, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x0b, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x01, 0x00, 0x00, 0x00, 0x0d, 0x0e, 0x01, 0x0e, 0x05, 0x00, 0x00, 0x00, 0x08, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x03, 0x03, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x0d, 0x00, 0x00, 0x00, 0x0b, 0x09,
	0x00, 0x03, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x03, 0x00, 0x09, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x01, 0x09,
	0x00, 0x03, 0x07, 0x07, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x07, 0x07, 0x03, 0x00, 0x09, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x0d, 0x00, 0x00, 0x00, 0x01, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09, 0x02, 0x00, 0x00, 0x00, 0x01, 0x09,
	0x00, 0x07, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0d, 0x0e, 0x0c, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x0b, 0x03, 0x06, 0x00, 0x00, 0x00, 0x01, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x0f, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x09, 0x03, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x03, 0x09, 0x03, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x00, 0x00, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x03, 0x09, 0x0b, 0x00, 0x00, 0x00, 0x0a, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x03, 0x07, 0x00, 0x09, 0x09, 0x00, 0x07, 0x03, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x0a, 0x0d, 0x0a, 0x00, 0x0a, 0x02, 0x03, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x0f, 0x07, 0x00, 0x09, 0x09, 0x00, 0x07, 0x0f, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x02, 0x0c, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x00, 0x00, 0x07, 0x03, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x0f, 0x09, 0x0b, 0x00, 0x00, 0x00, 0x05, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x07, 0x07, 0x03, 0x07, 0x07, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x09, 0x00, 0x00, 0x00, 0x00, 0x03, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x07, 0x07, 0x03, 0x03, 0x03, 0x03, 0x07, 0x07, 0x07, 0x07, 0x07, 0x00, 0x09, 0x04, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x09, 0x00, 0x00, 0x00, 0x00, 0x03, 0x09, 0x09,
	0x00, 0x03, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x03, 0x00, 0x09, 0x04, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x06, 0x00, 0x00, 0x00, 0x0c, 0x09, 0x09, 0x00, 0x00, 0x00, 0x0a, 0x09, 0x09, 0x09,
	0x00, 0x03, 0x07, 0x07, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x07, 0x07, 0x03, 0x00, 0x09, 0x04, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x01, 0x03, 0x06, 0x00, 0x00, 0x0a, 0x0f, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x07, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x07, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x0e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x02, 0x01, 0x0c, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x07, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x07, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09
};

static UINT8 disk_ab_pal[] = {
	0x01, 0x01, 0x01,
	0x87, 0x87, 0x87,
	0x4c, 0x4c, 0x4c,
	0xc4, 0xc4, 0xc4,
	0x25, 0x1d, 0xed,
	0x27, 0x27, 0x27,
	0x66, 0x66, 0x66,
	0xb1, 0xe5, 0xef,
	0xa6, 0xa6, 0xa6,
	0xff, 0xff, 0xff,
	0x19, 0x19, 0x19,
	0x98, 0x98, 0x98,
	0xe1, 0xe1, 0xe1,
	0x3b, 0x3b, 0x3b,
	0x75, 0x75, 0x75,
	0xba, 0xba, 0xba
};

// ---[ mapper FDS (Famicom Disk System) #0x808 (2056)
#define mapperFDS_ioenable		(mapper_regs[0x1f - 0]) // 4023
#define mapperFDS_control		(mapper_regs[0x1f - 1]) // 4025

#define mapperFDS_eighth_bit 	(mapper_regs[0x1f - 2])
#define mapperFDS_in_irq		(mapper_regs[0x1f - 3])

#define mapperFDS_irqenable		(mapper_regs[0x1f - 4])
#define mapperFDS_irqrepeat		(mapper_regs[0x1f - 5])
#define mapperFDS_irqcount		(mapper_regs16[0x1f - 0])
#define mapperFDS_irqlatch		(mapper_regs16[0x1f - 1])

#define mapperFDS_diskside		(mapper_regs[0x1f - 6])
#define mapperFDS_diskinsert	(mapper_regs[0x1f - 7])
#define mapperFDS_diskaccessed	(mapper_regs[0x1f - 8])
#define mapperFDS_diskaddr		(mapper_regs16[0x1f - 2])
#define mapperFDS_diskirqcount	(mapper_regs16[0x1f - 3])
#define mapperFDS_arm_timer() 	do { mapperFDS_diskirqcount = 150; } while (0)
#define mapperFDS_filesize      (mapper_regs16[0x1f - 4])
#define mapperFDS_blockid       (mapper_regs16[0x1f - 5])
#define mapperFDS_blockstart    (mapper_regs16[0x1f - 6])
#define mapperFDS_blocklen      (mapper_regs16[0x1f - 7])
#define fds_disk() 				(Cart.FDSDisk[(mapperFDS_diskside * 65500) + mapperFDS_blockstart + mapperFDS_diskaddr])

static char fdsfile[10]; // for debugging.. remove later!

enum FDS_FrameActions { FA_IDLE = 0, INSERT_DISK = 1, SWAP_DISK = 2, FA_FADEOUT};
enum FDS_DiskBlockIDs { DSK_INIT = 0, DSK_VOLUME, DSK_FILECNT, DSK_FILEHDR, DSK_FILEDATA };

static void FDS_FrameAction(INT32 todo);

static void FDS_Insert(INT32 status)
{
	static INT32 debounce = 0;
	if (status && debounce == 0) {
		mapperFDS_diskinsert ^= 1;
		bprintf(0, _T("FDS_Insert: %d\n"), mapperFDS_diskinsert);
	}
	debounce = status;
}

static void FDS_SwapSides(INT32 status, INT32 no_debounce = 0)
{
	static INT32 debounce = 0;
	if (status && (debounce == 0 || no_debounce) && mapperFDS_diskinsert == 0) {
		mapperFDS_diskside = (mapperFDS_diskside + 1) % Cart.FDSDiskSides;
		bprintf(0, _T("FDS_SwapSides: %d\n"), mapperFDS_diskside);
	}
	debounce = status;
}

static void FDS_SwapSidesAuto(INT32 status)
{
	static INT32 debounce = 0;
	if (status && debounce == 0) {
		FDS_FrameAction(SWAP_DISK);
	}
	debounce = status;
}

static void FDS_FrameAction(INT32 todo)
{
	FDSFrameCounter = 0;
	FDSFrameAction = todo;
}

static void FDS_FrameTicker()
{
	switch (FDSFrameAction) {
		case FA_IDLE:
			break;

		case INSERT_DISK:
			if (FDSFrameCounter == 5) {
				mapperFDS_diskinsert = 1;
				FDSFrameAction = FA_IDLE;
			}
			break;

		case SWAP_DISK:
			if (FDSFrameCounter == 0) {
				FDSFrameDiskIcon = ((mapperFDS_diskside + 1) % Cart.FDSDiskSides) & 1;
			}
			if (FDSFrameCounter == 5) {
				mapperFDS_diskinsert = 0;
			}
			if (FDSFrameCounter == 75) {
				bprintf(0, _T("(auto) "));
				FDS_SwapSides(1, 1);
			}
			if (FDSFrameCounter == 155) {
				mapperFDS_diskinsert = 1;
				FDSFrameAction = FA_FADEOUT;
			}
			break;
		case FA_FADEOUT:
			if (FDSFrameCounter == 155+20) {
				FDSFrameAction = FA_IDLE;
			}
			break;
	}
	FDSFrameCounter++;
}

struct fds_sound {
	UINT8 volmaster;
	UINT8 wavwren_hold;
	UINT8 volgain; // r:4090 w:4080
	UINT8 modgain; // r:4092 w:4084
	UINT8 waveram[0x40];
	UINT8 modwaveram[0x40];

	UINT8 env_master;

	UINT32 wavefreq;
	UINT32 wavepos;
	UINT32 modphase;
	UINT32 modfreq;
	INT32 mod_accu;

	UINT8 mod_counter;
	UINT8 env_all_stop;
	UINT8 wav_stop;
	UINT8 mod_cnt_stop;

	INT32 env_vol_accu;
	UINT8 env_vol_stop;
	UINT8 env_vol_direction;
	UINT8 env_vol_period;

	INT32 env_mod_accu;
	UINT8 env_mod_stop;
	UINT8 env_mod_direction;
	UINT8 env_mod_period;

	INT32 filt_prev;
	INT32 filt_sa;
	INT32 filt_sb;
};

static fds_sound fds;

static void mapperFDS_reset()
{
	BurnLEDReset();

	memset(&fds, 0, sizeof(fds));

	// Auto-"Insertion/Swap" robot
	if (~DrvDips[0] & 2)
		FDS_FrameAction(INSERT_DISK);

	// Init 2khz lp filter coefficient @ 1.786mhz
	double omeg = exp(-2.0 * 3.1415926 * 2000 / (29781 * 60));
    fds.filt_sa = (INT32)(omeg * (double)(1 << 12));
    fds.filt_sb = (1 << 12) - fds.filt_sa;
}

static void mapperFDS_scan()
{
	SCAN_VAR(fds);
	SCAN_VAR(FDSFrameCounter);
	SCAN_VAR(FDSFrameAction);

	// 6000-7fff WRAM(cart_ext) is scanned by system
	ScanVar(Cart.PRGRom, 0x6000, "PRGRAM 8000-dfff");
}

static UINT8 fds_sound_read(UINT16 address)
{
	if (address >= 0x4040 && address <= 0x407f) {
		return fds.waveram[address & 0x3f] | (cpu_open_bus & 0xc0);
	}

	switch (address) {
		case 0x4090: return (fds.volgain & 0x3f) | (cpu_open_bus & 0xc0);
		case 0x4092: return (fds.modgain & 0x3f) | (cpu_open_bus & 0xc0);
	}

	return cpu_open_bus;
}

static void fds_sound_write(UINT16 address, UINT8 data)
{
	if (address >= 0x4040 && address <= 0x407f) {
		if (fds.wavwren_hold) {
			fds.waveram[address & 0x3f] = data & 0x3f;
		}
		return;
	}

	switch (address) {
		case 0x4080:
			fds.env_vol_stop = (data & 0x80) >> 7;
			fds.env_vol_direction = (data & 0x40) >> 6;
			fds.env_vol_period = data & 0x3f;
			if (data & 0x80) {
				fds.volgain = data & 0x3f;
			}
			break;
		case 0x4082:
			fds.wavefreq = (fds.wavefreq & 0x0f00) | data;
			break;
		case 0x4083:
			fds.wavefreq = (fds.wavefreq & 0x00ff) | ((data & 0xf) << 8);
			fds.env_all_stop = (data & 0x40) >> 6;
			fds.wav_stop = (data & 0x80) >> 7;

			if (data & 0x80) {
				// phase reset
				fds.wavepos = 0;
			}
			break;
		case 0x4084:
			fds.env_mod_stop = (data & 0x80) >> 7;
			fds.env_mod_direction = (data & 0x40) >> 6;
			fds.env_mod_period = data & 0x3f;
			if (data & 0x80) {
				fds.modgain = data & 0x3f;
			}
			break;
		case 0x4085:
			fds.mod_counter = data & 0x7f;
			fds.modphase &= 0x3f0000;
			break;
		case 0x4086:
			fds.modfreq = (fds.modfreq & 0x0f00) | data;
			break;
		case 0x4087:
			fds.modfreq = (fds.modfreq & 0x00ff) | ((data & 0xf) << 8);
			fds.mod_cnt_stop = (data & 0x80) >> 7;
			if (data & 0x80) {
				fds.mod_accu = 0;
			}
			break;
		case 0x4088:
			if (fds.mod_cnt_stop) {
				fds.modwaveram[(fds.modphase >> 16) & 0x3f] = data & 7;
				fds.modphase = (fds.modphase + 0x10000) & 0x3fffff;
				fds.modwaveram[(fds.modphase >> 16) & 0x3f] = data & 7;
				fds.modphase = (fds.modphase + 0x10000) & 0x3fffff;
			}
			break;
		case 0x4089:
			fds.volmaster = data & 3;
			fds.wavwren_hold = (data & 0x80) >> 7;
			break;
		case 0x408a:
			fds.env_master = data;
			break;
	}
}

static void env_vol()
{
	if (fds.env_vol_stop == 0) {
		fds.env_vol_accu--;
		if (fds.env_vol_accu <= 0) {
			fds.env_vol_accu += ((fds.env_vol_period + 1) * fds.env_master) << 3;
			if (fds.env_vol_direction) {
				if (fds.volgain < 0x3f) {
					fds.volgain++;
				}
			} else {
				if (fds.volgain > 0) {
					fds.volgain--;
				}
			}
		}
	}
}

static void env_mod()
{
	if (fds.env_mod_stop == 0) {
		fds.env_mod_accu--;
		if (fds.env_mod_accu <= 0) {
			fds.env_mod_accu += ((fds.env_mod_period + 1) * fds.env_master) << 3;
			if (fds.env_mod_direction) {
				if (fds.modgain < 0x3f) {
					fds.modgain++;
				}
			} else {
				if (fds.modgain > 0) {
					fds.modgain--;
				}
			}
		}
	}
}

static void mod_sweep()
{
	INT32 mod = 0;
	if (fds.mod_cnt_stop == 0) {
		fds.mod_accu += fds.modfreq;
		if(fds.mod_accu >= 0x10000) {
			fds.mod_accu -= 0x10000;
			const INT32 BIASTBL[8] = { 0, 1, 2, 4, 0, -4, -2, -1 };
			UINT8 w = fds.modwaveram[(fds.modphase >> 16) & 0x3f];
			fds.mod_counter = (w == 4) ? 0 : (fds.mod_counter + BIASTBL[w]) & 0x7f;
		}
		fds.modphase = (fds.modphase + fds.modfreq) & 0x3fffff;
	}

	if (fds.wav_stop == 0) {
		if (fds.modgain != 0) {
			INT32 pos = (fds.mod_counter < 64) ? fds.mod_counter : (fds.mod_counter-128);

			// nesdev to the rescue! :) (yet again!)
			INT32 temp = pos * ((fds.modgain < 0x20) ? fds.modgain : 0x20);
			INT32 remainder = temp & 0x0f;
			temp >>= 4;
			if (remainder > 0 && ~temp & 0x80) {
				temp += (pos < 0) ? -1 : 2;
			}

			if (temp >= 192) temp -= 256;
			else if (temp < -64) temp += 256;

			temp = fds.wavefreq * temp;
			remainder = temp & 0x3f;
			temp >>= 6;
			if (remainder >= 32) temp += 1;
			mod = temp;
		}

		fds.wavepos = (fds.wavepos + fds.wavefreq + mod) & 0x3fffff;
	}
}

static INT16 mapperFDS_mixer()
{
	UINT32 sample = 0;

	if (fds.env_all_stop == 0 && fds.wav_stop == 0 && fds.env_master != 0) {
		env_vol();
		env_mod();
	}

	mod_sweep();

	if (fds.wavwren_hold == 0) {
		// render sample
		INT32 voltab[4] = { 0x173, 0xf7, 0xb9, 0x95 };
		UINT8 envvol = (fds.volgain > 0x20) ? 0x20 : fds.volgain;
		sample = (fds.waveram[(fds.wavepos >> 16) & 0x3f] * envvol) * voltab[fds.volmaster];
		sample >>= 7;

		// 2khz lp filter
		sample = ((fds.filt_prev * fds.filt_sa) + (sample * fds.filt_sb)) >> 12;
		fds.filt_prev = sample;
	} else {
		sample = fds.filt_prev;
	}

	return sample;
}

static void mapperFDS_cycle()
{
	if (mapperFDS_irqenable) {
		mapperFDS_irqcount--;
		// m6502 cpu core isn't great.. compensate by running irq for 5 cycles longer.
		// Testcases: Marchen Veil HUD, KaetteKita Mario Bros (white bricks cutscene)
		if (mapperFDS_irqcount == 0xffff-5) {
			mapperFDS_irqcount = mapperFDS_irqlatch;
			if (mapperFDS_irqrepeat == 0) {
				mapperFDS_irqenable = 0;
			}
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapperFDS_in_irq = 1;
		}
	}

	if (mapperFDS_diskirqcount > 0) {
		mapperFDS_diskirqcount--;
		if (mapperFDS_diskirqcount == 0) {
			if (mapperFDS_control & 0x80) {
				M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
				mapperFDS_eighth_bit = 1;
			}
		}
	}
}

static void mapperFDS_map()
{
	set_mirroring((mapperFDS_control & 8) ? HORIZONTAL : VERTICAL);
}

static UINT8 mapperFDS_read(UINT16 address)
{
	UINT8 ret = 0;

	if (mapperFDS_ioenable & 2 && address >= 0x4040 && address <= 0x4097)
		return fds_sound_read(address);

	if (~mapperFDS_ioenable & 1) return cpu_open_bus; // io disabled.

	switch (address) {
		case 0x4030:
			if (mapperFDS_in_irq) ret |= 1;
			if (mapperFDS_eighth_bit) ret |= 2;

			mapperFDS_in_irq = mapperFDS_eighth_bit = 0;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;

		case 0x4031:
			ret = 0xff;
			if (mapperFDS_diskinsert && mapperFDS_control & 0x04) {
				mapperFDS_diskaccessed = 1;

				ret = 0;

				switch (mapperFDS_blockid) {
					case DSK_FILEHDR:
						if (mapperFDS_diskaddr < mapperFDS_blocklen) {
							ret = fds_disk();
							switch (mapperFDS_diskaddr) {
								case 13: mapperFDS_filesize = ret; break;
								case 14:
									mapperFDS_filesize |= ret << 8;
									strncpy(fdsfile, (char*)&Cart.FDSDisk[(mapperFDS_diskside * 65500) + mapperFDS_blockstart + 3], 8);
									bprintf(0, _T("Read file: %S (size: %d)\n"), fdsfile, mapperFDS_filesize);
									break;
							}
							mapperFDS_diskaddr++;
						}
						break;
					default:
						if (mapperFDS_diskaddr < mapperFDS_blocklen) {
							ret = fds_disk();
							mapperFDS_diskaddr++;
						}
						break;
				}

				mapperFDS_arm_timer();
				mapperFDS_eighth_bit = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			}
			break;

		case 0x4032:
			ret = cpu_open_bus & 0xf8;
			if (mapperFDS_diskinsert == 0) {
				ret |= 4 /*write protect or eject*/ | 1 /*disk not inserted*/;
			}
			if ((mapperFDS_diskinsert == 0) || (~mapperFDS_control & 1/*motor off*/) || (mapperFDS_control & 2/*transfer reset*/)) {
				ret |= 2 /*disk not ready*/;
			}
			break;

		case 0x4033: // battery status of drive unit
			ret |= 0x80;
			break;
	}

	return ret;
}

static void mapperFDS_prg_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000 && address <= 0xdfff) {
		Cart.PRGRom[address & 0x7fff] = data;
	}
}

static void mapperFDS_write(UINT16 address, UINT8 data)
{
	if (mapperFDS_ioenable & 2 && address >= 0x4040 && address <= 0x4097)
		return fds_sound_write(address, data);

	if ((~mapperFDS_ioenable & 1) && address >= 0x4024) return; // io disabled.

	switch (address) {
		case 0x4020:
			mapperFDS_irqlatch = (mapperFDS_irqlatch & 0xff00) | data;
			break;
		case 0x4021:
			mapperFDS_irqlatch = (mapperFDS_irqlatch & 0x00ff) | (data << 8);
			break;

		case 0x4022:
			mapperFDS_irqenable = (data & 2) && (mapperFDS_ioenable & 1);
			mapperFDS_irqrepeat = data & 1;
			if (mapperFDS_irqenable) {
				mapperFDS_irqcount = mapperFDS_irqlatch;
			} else {
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				mapperFDS_in_irq = 0;
			}
			break;

		case 0x4023:
			mapperFDS_ioenable = data;
			if (~data & 1) {
				mapperFDS_irqenable = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				mapperFDS_in_irq = mapperFDS_eighth_bit = 0;
			}
			break;

		case 0x4024:
			if (mapperFDS_diskinsert && ~mapperFDS_control & 0x04) {

				if (mapperFDS_diskaccessed == 0) {
					mapperFDS_diskaccessed = 1;
					break;
				}

				switch (mapperFDS_blockid) {
					case DSK_FILEHDR:
						if (mapperFDS_diskaddr < mapperFDS_blocklen) {
							fds_disk() = data;
							switch (mapperFDS_diskaddr) {
								case 13: mapperFDS_filesize = data; break;
								case 14:
									mapperFDS_filesize |= data << 8;
									fdsfile[0] = 0;
									strncpy(fdsfile, (char*)&Cart.FDSDisk[(mapperFDS_diskside * 65500) + mapperFDS_blockstart + 3], 8);
									bprintf(0, _T("Write file: %S (size: %d)\n"), fdsfile, mapperFDS_filesize);
									break;
							}
							mapperFDS_diskaddr++;
						}
						break;
					default:
						if (mapperFDS_diskaddr < mapperFDS_blocklen) {
							fds_disk() = data;
							mapperFDS_diskaddr++;
						}
						break;
				}

			}
			break;

		case 0x4025:
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);

			//bprintf(0, _T("0x4025: nCurrentFrame %d  param %x   ppc %x\n"), nCurrentFrame, data, M6502GetPrevPC(-1));
			if (mapperFDS_diskinsert) {
				if (data & 0x40 && ~mapperFDS_control & 0x40) {
					mapperFDS_diskaccessed = 0;

					mapperFDS_arm_timer();

					// blockstart  - address of block on disk
					// diskaddr    - address relative to blockstart
					mapperFDS_blockstart += mapperFDS_diskaddr;
					mapperFDS_diskaddr = 0;

					mapperFDS_blockid++;
					if (mapperFDS_blockid > DSK_FILEDATA)
						mapperFDS_blockid = DSK_FILEHDR;

					switch (mapperFDS_blockid) {
						case DSK_VOLUME:
							mapperFDS_blocklen = 0x38;
							break;
						case DSK_FILECNT:
							mapperFDS_blocklen = 0x02;
							break;
						case DSK_FILEHDR:
							mapperFDS_blocklen = 0x10;
							break;
						case DSK_FILEDATA:		 // [blockid]<filedata>
							mapperFDS_blocklen = 0x01 + mapperFDS_filesize;
							break;
					}
				}

				if (data & 0x02) { // transfer reset
					mapperFDS_blockid = DSK_INIT;
					mapperFDS_blockstart = 0;
					mapperFDS_blocklen = 0;
					mapperFDS_diskaddr = 0;
					mapperFDS_arm_timer();
				}
				if (data & 0x40 && ~data & 2) { // turn on motor
					mapperFDS_arm_timer();
					BurnLEDSetStatus(0, 1);
					BurnLEDSetColor((mapperFDS_control & 0x04) ? LED_COLOR_GREEN : LED_COLOR_RED);
				} else {
					BurnLEDSetStatus(0, 0);
				}
			}
			mapperFDS_control = data;

			mapper_map(); // set mirror

			break;
	}
}

#undef mapperFDS_ioenable
#undef mapperFDS_control
#undef mapperFDS_eighth_bit
#undef mapperFDS_in_irq
#undef mapperFDS_irqenable
#undef mapperFDS_irqrepeat
#undef mapperFDS_irqcount
#undef mapperFDS_irqlatch
#undef mapperFDS_diskside
#undef mapperFDS_diskinsert
#undef mapperFDS_diskaccessed
#undef mapperFDS_diskaddr
#undef mapperFDS_diskirqcount
#undef mapperFDS_arm_timer
#undef mapperFDS_filesize
#undef mapperFDS_blockid
#undef mapperFDS_blockstart

// ---[ mapper 01 (MMC1)
#define mapper01_serialbyte	(mapper_regs[0x1f - 0])
#define mapper01_bitcount	(mapper_regs[0x1f - 1])
#define mapper01_lastwrite  (mapper_regs[0x1f - 2])
#define mapper01_exbits		(mapper_regs[0x1f - 3])
#define mapper01_prg2x		(mapper_regs[0x1f - 4])

static void mapper01_write(UINT16 address, UINT8 data)
{
	static INT32 last_write = 0; // check only happens @ boot w/bill & ted, no need to s.state this. -dink
	if (address & 0x8000) {
		if (mega_cyc_counter - last_write < 2) {
			// https://wiki.nesdev.com/w/index.php/MMC1
			// Bill & Ted's Excellent Adventure resets the MMC1 by doing INC
			// on a ROM location containing $FF; the MMC1 sees the $FF
			// written back and ignores the $00 written on the next cycle.
			last_write = mega_cyc_counter;
			return;
		}
        if (data & 0x80) { // bit 7, mapper reset
			mapper01_bitcount = 0;
			mapper01_serialbyte = 0;
			mapper_regs[0] |= 0x0c;
			if (mapper_map) mapper_map();
		} else {
			mapper01_serialbyte |= (data & 1) << mapper01_bitcount;
			mapper01_bitcount++;
			if (mapper01_bitcount == 5) {
				UINT8 reg = (address >> 13) & 0x3;
				mapper_regs[reg] = mapper01_serialbyte;
				switch (reg) {
					case 1: mapper01_lastwrite = 0; break;
					case 2: mapper01_lastwrite = 1; break;
				}
                mapper01_bitcount = 0;
                mapper01_serialbyte = 0;
                if (mapper_map) mapper_map();
			}
		}
		last_write = mega_cyc_counter;
    }
}

static void mapper01_exp_write(UINT16 address, UINT8 data) // 6000 - 7fff
{
	if (~mapper_regs[3] & 0x10) {
		Cart.WorkRAM[PRGExpMap + (address & 0x1fff)] = data;
	}
	cart_exp_write_abort = 1; // don't fall-through after callback!
}

static UINT8 mapper01_exp_read(UINT16 address)             // 6000 - 7fff
{
	return (~mapper_regs[3] & 0x10) ? Cart.WorkRAM[PRGExpMap + (address & 0x1fff)] : cpu_open_bus;
}

static void mapper01_map()
{
	mapper01_exbits = ((mapper01_lastwrite == 1) && (mapper_regs[0] & 0x10)) ? mapper_regs[2] : mapper_regs[1];

	if (Cart.WorkRAMSize > 0x2000) {
		// MMC1 Bankable WRAM/SRAM
		UINT8 page = 0;
		switch (Cart.WorkRAMSize) {
			case 0x8000: // 32k (Final Fantasy 1 & 2)
				page = (mapper_regs[1] >> 2) & 3;
				break;
			default:
				page = (mapper_regs[1] >> 3) & 1;
				break;
		}
		mapper_map_exp_prg(page);
	}

	INT32 bigcart = 0;

	if (Cart.PRGRomSize >= 0x80000) {
		bigcart = 1;
		// SUROM (512K/1M rom), use extra mapping bit
		mapper01_prg2x = mapper01_exbits & 0x10;
	}

	if (mapper_regs[0] & 0x8) {
		if (mapper_regs[0] & 0x4) {
            mapper_map_prg(16, 0, (mapper_regs[3] & 0xf) | mapper01_prg2x);
			if (bigcart) {
				mapper_map_prg(16, 1, 0xf | mapper01_prg2x);
			} else {
				mapper_map_prg(16, 1, -1);
			}
        } else {
            mapper_map_prg(16, 0, 0 | mapper01_prg2x);
            mapper_map_prg(16, 1, (mapper_regs[3] & 0xf) | mapper01_prg2x);
        }
	} else {
		mapper_map_prg(32, 0, ((mapper_regs[3] & 0xf) | mapper01_prg2x) >> 1);
	}

	if (mapper_regs[0] & 0x10) {
        mapper_map_chr( 4, 0, mapper_regs[1]);
        mapper_map_chr( 4, 1, mapper_regs[2]);
	} else {
		mapper_map_chr( 8, 0, mapper_regs[1] >> 1);
	}

	switch (mapper_regs[0] & 3) {
        case 0: set_mirroring(SINGLE_LOW); break;
        case 1: set_mirroring(SINGLE_HIGH); break;
        case 2: set_mirroring(VERTICAL); break;
        case 3: set_mirroring(HORIZONTAL); break;
	}
}
#undef mapper01_serialbyte
#undef mapper01_bitcount

// ---[ mapper 02 (UxROM)
static void mapper02_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper02_map()
{
    mapper_map_prg(16, 0, mapper_regs[0] & 0xff);
    mapper_map_prg(16, 1, -1);
    mapper_map_chr( 8, 0, 0);
}

// ---[ mapper 41 Caltron 6-in-1
#define mapper41_prg	(mapper_regs[0])
#define mapper41_chr	(mapper_regs[1])
#define mapper41_mirror	(mapper_regs[2])

static void mapper41_write(UINT16 address, UINT8 data)
{
	if (address >= 0x6000 && address <= 0x67ff) {
		mapper41_prg = address & 0x07;
		mapper41_chr = (mapper41_chr & 0x03) | ((address & 0x18) >> 1);
		mapper41_mirror = (address & 0x20) >> 5;
	}
	if (address >= 0x8000 && address <= 0xffff) {
		if (mapper41_prg & 4) {
			mapper41_chr = (mapper41_chr & 0x0c) | (data & 0x03);
		}
	}

	mapper_map();
}

static void mapper41_map()
{
	mapper_map_prg(32, 0, mapper41_prg);
	mapper_map_chr( 8, 0, mapper41_chr);

	set_mirroring((mapper41_mirror & 0x01) ? HORIZONTAL : VERTICAL);
}
#undef mapper41_prg
#undef mapper41_chr
#undef mapper41_mirror

// ---[ mapper 53 SuperVision 16-in-1
#define mapper53_reg(x)	(mapper_regs[0 + (x)])

static void mapper53_write(UINT16 address, UINT8 data)
{
	mapper53_reg((address & 0x8000) >> 15) = data;

	mapper_map();
}

static UINT8 mapper53_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

static void mapper53_map()
{
	UINT8 bank = (mapper53_reg(0) & 0xf) << 3;
	mapper_map_exp_prg((0xf | (bank << 1)) + 4);
	if (mapper53_reg(0) & 0x10) {
		mapper_map_prg(16, 0, ((bank | (mapper53_reg(1) & 7)) + 2));
		mapper_map_prg(16, 1, ((bank | 7) + 2));
	} else {
		mapper_map_prg(32, 0, 0);
	}

	mapper_map_chr( 8, 0, 0);

	set_mirroring((mapper53_reg(0) & 0x20) ? HORIZONTAL : VERTICAL);
}
#undef mapper53_reg

// ---[ mapper 104 Golden Five, Pegasus 5-in-1
#define mapper104_prg(x)	(mapper_regs[0 + (x)])

static void mapper104_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000 && address <= 0xbfff) {
		if (data & 0x08) {
			mapper104_prg(0) = (mapper104_prg(0) & 0x0f) | (data & 0x07) << 4;
			mapper104_prg(1) = (data & 0x07) << 4;
		}
	}
	if (address >= 0xc000 && address <= 0xffff) {
		mapper104_prg(0) = (mapper104_prg(0) & 0x70) | (data & 0x0f);
	}

	mapper_map();
}

static void mapper104_map()
{
	mapper_map_prg(16, 0, mapper104_prg(0));
	mapper_map_prg(16, 1, mapper104_prg(1) | 0x0f);
	mapper_map_chr( 8, 0, 0);
}
#undef mapper104_prg

// ---[ mapper 190 Magic Kid Googoo
#define mapper190_prg		(mapper_regs[0])
#define mapper190_chr(x)	(mapper_regs[1 + (x)])
static void mapper190_write(UINT16 address, UINT8 data)
{
	switch (address & ~0x1fff) {
		case 0x8000:
			mapper190_prg = data & 0x07;
			break;
		case 0xa000:
			mapper190_chr(address & 0x03) = data & 0x3f;
			break;
		case 0xc000:
			mapper190_prg = (data & 0x07) | 0x08;
			break;
	}

	mapper_map();
}

static void mapper190_map()
{
    mapper_map_prg(16, 0, mapper190_prg);
    mapper_map_prg(16, 1, 0); // second 16k bank mapped to first prg bank

	mapper_map_chr( 2, 0, mapper190_chr(0));
	mapper_map_chr( 2, 1, mapper190_chr(1));
	mapper_map_chr( 2, 2, mapper190_chr(2));
	mapper_map_chr( 2, 3, mapper190_chr(3));

	set_mirroring(VERTICAL);
}
#undef mapper190_prg
#undef mapper190_chr

// ---[ mapper 193 NTDEC TC-112 (War in the Gulf, Fighting Hero, ...)
#define mapper193_prg		(mapper_regs[0])
#define mapper193_chr(x)	(mapper_regs[1 + (x)])
#define mapper193_mirror	(mapper_regs[4])

static void mapper193_write(UINT16 address, UINT8 data)
{
	switch (address & 0xe007) {
		case 0x6000:
		case 0x6001:
		case 0x6002:
			mapper193_chr(address & 3) = data;
			break;
		case 0x6003:
			mapper193_prg = data;
			break;
		case 0x6004:
			mapper193_mirror = data;
			break;
	}

	mapper_map();
}

static void mapper193_map()
{
	mapper_map_prg( 8, 0, mapper193_prg);
	mapper_map_prg( 8, 1, -3);
	mapper_map_prg( 8, 2, -2);
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 4, 0, mapper193_chr(0) >> 2);

	mapper_map_chr( 2, 2, mapper193_chr(1) >> 1);
	mapper_map_chr( 2, 3, mapper193_chr(2) >> 1);

	set_mirroring((mapper193_mirror & 0x01) ? HORIZONTAL : VERTICAL);
}
#undef mapper193_prg
#undef mapper193_chr
#undef mapper193_mirror

// ---[ mapper 15 Contra 168-in-1 Multicart
#define mapper15_prg		(mapper_regs[0])
#define mapper15_prgbit		(mapper_regs[1])
#define mapper15_prgmode	(mapper_regs[2])
#define mapper15_mirror		(mapper_regs[3])

static void mapper15_write(UINT16 address, UINT8 data)
{
	mapper15_mirror = data & 0x40;
	mapper15_prg = (data & 0x7f) << 1;
	mapper15_prgbit = (data & 0x80) >> 7;
	mapper15_prgmode = address & 0xff; // must ignore weird writes.

	mapper_map();
}

static void mapper15_map()
{
	switch (mapper15_prgmode) {
		case 0x00:
			mapper_map_prg( 8, 0, (mapper15_prg + 0) ^ mapper15_prgbit);
			mapper_map_prg( 8, 1, (mapper15_prg + 1) ^ mapper15_prgbit);
			mapper_map_prg( 8, 2, (mapper15_prg + 2) ^ mapper15_prgbit);
			mapper_map_prg( 8, 3, (mapper15_prg + 3) ^ mapper15_prgbit);
			break;
		case 0x01:
			mapper_map_prg( 8, 0, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 1, (mapper15_prg + 1) | mapper15_prgbit);
			mapper_map_prg( 8, 2, (mapper15_prg + 0) | 0x0e | mapper15_prgbit);
			mapper_map_prg( 8, 3, (mapper15_prg + 1) | 0x0e | mapper15_prgbit);
			break;
		case 0x02:
			mapper_map_prg( 8, 0, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 1, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 2, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 3, (mapper15_prg + 0) | mapper15_prgbit);
			break;
		case 0x03:
			mapper_map_prg( 8, 0, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 1, (mapper15_prg + 1) | mapper15_prgbit);
			mapper_map_prg( 8, 2, (mapper15_prg + 0) | mapper15_prgbit);
			mapper_map_prg( 8, 3, (mapper15_prg + 1) | mapper15_prgbit);
			break;
	}

	mapper_map_chr_ramrom( 8, 0, 0, (mapper15_prgmode == 3) ? MEM_RAM_RO : MEM_RAM);

	set_mirroring((mapper15_mirror & 0x40) ? HORIZONTAL : VERTICAL);
}

#undef mapper15_prg
#undef mapper15_prgbit
#undef mapper15_prgmode
#undef mapper15_mirror

// ---[ mapper 389 Caltron 9-in-1
#define mapper389_reg8  (mapper_regs[0])
#define mapper389_reg9  (mapper_regs[1])
#define mapper389_rega  (mapper_regs[2])

static void mapper389_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper389_reg8 = address & 0xff; break;
		case 0x9000: mapper389_reg9 = address & 0xff; break;
		default: mapper389_rega = address & 0xff; break;
	}

	mapper_map();
}

static void mapper389_map()
{
	if (mapper389_reg9 & 0x02) { // UNROM-064 mode
		mapper_map_prg(16, 0, (mapper389_reg8 >> 2) | ((mapper389_rega >> 2) & 0x03));
		mapper_map_prg(16, 1, (mapper389_reg8 >> 2) | 0x03);
	} else { // NROM-256 mode
		mapper_map_prg(32, 0, mapper389_reg8 >> 3);
	}
	mapper_map_chr( 8, 0, ((mapper389_reg9 & 0x38) >> 1) | (mapper389_rega & 0x03));

	set_mirroring((mapper389_reg8 & 0x01) ? HORIZONTAL : VERTICAL);
}

#undef mapper389_reg8
#undef mapper389_reg9
#undef mapper389_rega

// ---[ mapper 216 (??) Magic Jewelry 2
#define mapper216_prg   (mapper_regs[0])
#define mapper216_chr   (mapper_regs[1])
static void mapper216_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000) {
		mapper216_prg = address & 1;
		mapper216_chr = (address & 0x000e) >> 1;
		mapper_map();
	}
}

static UINT8 mapper216_read(UINT16 address)
{
	return 0; // must be 0 @ 0x5000
}

static void mapper216_map()
{
    mapper_map_prg(32, 0, mapper216_prg);
	mapper_map_chr( 8, 0, mapper216_chr);
}
#undef mapper216_prg
#undef mapper216_chr

// ---[ mapper 132 (TXC Micro Genius) Qi Wang
#define mapper132_reg(x)   (mapper_regs[0 + (x)])
#define mapper132_reghi    (mapper_regs[8])
static void mapper132_write(UINT16 address, UINT8 data)
{
	if (address >= 0x4100 && address <= 0x4103) {
		mapper132_reg(address & 3) = data;
		//mapper_map(); // only 8000+!
	}

	if (address >= 0x8000) {
		mapper132_reghi = data;
		mapper_map();
	}
}

static UINT8 mapper132_read(UINT16 address)
{
	if (address == 0x4100) {
		return (mapper132_reg(1) ^ mapper132_reg(2)) | 0x40;
	}
	return cpu_open_bus;
}

static void mapper132_map()
{
    mapper_map_prg(32, 0, (mapper132_reg(2) >> 2) & 1);
	mapper_map_chr( 8, 0, mapper132_reg(2) & 3);
}
#undef mapper132_reg
#undef mapper132_reghi

// ---[ mapper 30 (UNROM-512)
#define mapper30_mirroring_en   (mapper_regs[1])
static void mapper30_write(UINT16 address, UINT8 data)
{
	if (address >= 0xc000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper30_map()
{
    mapper_map_prg(16, 0, mapper_regs[0] & 0x1f);
    mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, (mapper_regs[0] >> 5) & 0x03);
	if (mapper30_mirroring_en) {
		set_mirroring((mapper_regs[0] & 0x80) ? SINGLE_HIGH : SINGLE_LOW);
	}
}

// ---[ mapper 03 (CNROM)
#define mapper03_need_update   (mapper_regs[1])
static void mapper03_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		if (NESMode & BUS_CONFLICTS) {
			data &= mapper_prg_read(address);
		}
		mapper_regs[0] = data;
		mapper03_need_update = 1;

		if (Cart.Crc != 0xab29ab28) // dropzone gets delayed a bit.
			mapper_map();
	}
}

static void mapper03_map()
{
    mapper_map_chr( 8, 0, mapper_regs[0]);
}

// dropzone likes to change the chr bank too soon sometimes, this causes a line
// through the player when he is near the bottom of the screen.
// Solution: delay chr bank switches until after sprites load
static void mapper03_cycle()
{
	if (pixel > 321 && mapper03_need_update) {
		mapper03_need_update = 0;
		mapper_map();
	}

}
#undef mapper03_need_update

// ---[ mapper 04 (mmc3) & mmc3-based: 12, 76, 95, 108, 115, 118, 119, 189, 262
#define mapper4_banksel         (mapper_regs[0x1f - 0])
#define mapper4_mirror			(mapper_regs[0x1f - 1])
#define mapper4_irqlatch 		(mapper_regs[0x1f - 2])
#define mapper4_irqcount		(mapper_regs[0x1f - 3])
#define mapper4_irqenable		(mapper_regs[0x1f - 4])
#define mapper4_irqreload		(mapper_regs[0x1f - 5])
#define mapper12_lowchr			(mapper_regs16[0x1f - 0])
#define mapper12_highchr		(mapper_regs16[0x1f - 1])
#define mapper115_prg           (mapper_regs[0x1f - 6])
#define mapper115_chr           (mapper_regs[0x1f - 7])
#define mapper115_prot          (mapper_regs[0x1f - 8])
#define mapper262_reg           (mapper_regs[0x1f - 9])
#define mapper189_reg           (mapper_regs[0x1f - 9]) // same as 262

static void mapper04_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
        switch (address & 0xE001) {
            case 0x8000: mapper4_banksel = data; break;
            case 0x8001: mapper_regs[(mapper4_banksel & 0x7)] = data; break;
			case 0xA000: mapper4_mirror = ~data & 1; break;
            case 0xC000: mapper4_irqlatch = data; break;
            case 0xC001: mapper4_irqreload = 1; break;
            case 0xE000: mapper4_irqenable = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
            case 0xE001: mapper4_irqenable = 1; break;
        }
		mapper_map();
    }
}

static void mapper04_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, (mapper_regs[0] + mapper12_lowchr) >> 1);
        mapper_map_chr(2, 1, (mapper_regs[1] + mapper12_lowchr) >> 1);

		mapper_map_chr(1, 4, mapper_regs[2] + mapper12_highchr);
		mapper_map_chr(1, 5, mapper_regs[3] + mapper12_highchr);
		mapper_map_chr(1, 6, mapper_regs[4] + mapper12_highchr);
		mapper_map_chr(1, 7, mapper_regs[5] + mapper12_highchr);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2] + mapper12_lowchr);
		mapper_map_chr(1, 1, mapper_regs[3] + mapper12_lowchr);
		mapper_map_chr(1, 2, mapper_regs[4] + mapper12_lowchr);
		mapper_map_chr(1, 3, mapper_regs[5] + mapper12_lowchr);

		mapper_map_chr(2, 2, (mapper_regs[0] + mapper12_highchr) >> 1);
		mapper_map_chr(2, 3, (mapper_regs[1] + mapper12_highchr) >> 1);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper76_map()
{
	mapper_map_prg(8, 0, mapper_regs[6]);
    mapper_map_prg(8, 1, mapper_regs[7]);
	mapper_map_prg(8, 2, -2);

	mapper_map_chr(2, 0, mapper_regs[2]);
	mapper_map_chr(2, 1, mapper_regs[3]);
	mapper_map_chr(2, 2, mapper_regs[4]);
	mapper_map_chr(2, 3, mapper_regs[5]);

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper95_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

	mapper_map_prg(8, 0, mapper_regs[6]);
	mapper_map_prg(8, 2, -2);

	mapper_map_chr(2, 0, mapper_regs[0] >> 1);
	mapper_map_chr(2, 1, mapper_regs[1] >> 1);

	nametable_mapall((mapper_regs[0] >> 5) & 1, (mapper_regs[0] >> 5) & 1, (mapper_regs[1] >> 5) & 1, (mapper_regs[1] >> 5) & 1);

	mapper_map_chr(1, 4, mapper_regs[2]);
	mapper_map_chr(1, 5, mapper_regs[3]);
	mapper_map_chr(1, 6, mapper_regs[4]);
	mapper_map_chr(1, 7, mapper_regs[5]);
}

static void mapper12_write(UINT16 address, UINT8 data)
{
	if (address & 0x4000) {
		mapper12_lowchr  = (data & 0x01) << 8;
		mapper12_highchr = (data & 0x10) << 4;

		mapper_map();
	}
}

static UINT8 mapper12_read(UINT16 address)
{
	UINT8 ret = cpu_open_bus;

	if (address & 0x4000) {
		ret = mapper12_lowchr >> 8;
		ret |= mapper12_highchr >> 4;
	}
	return ret;
}

static void mapper95_write(UINT16 address, UINT8 data)
{
	if ((address & 0x8001) == 0x8000)
		data &= 0x3f;

	mapper04_write(address & 0x8001, data);

	mapper_map();
}

static void mapper115_map()
{
	if (mapper115_prg & 0x80) {
		// mmc3 prg-override
		if (mapper115_prg & 0x20) {
			mapper_map_prg(32, 0, (mapper115_prg & 0xf) >> 1);
		} else {
			mapper_map_prg(16, 0, mapper115_prg & 0xf);
			mapper_map_prg(16, 1, mapper115_prg & 0xf);
		}
	} else {
		mapper_map_prg(8, 1, mapper_regs[7]);

		if (~mapper4_banksel & 0x40) {
			mapper_map_prg(8, 0, mapper_regs[6]);
			mapper_map_prg(8, 2, -2);
		} else {
			mapper_map_prg(8, 0, -2);
			mapper_map_prg(8, 2, mapper_regs[6]);
		}
		mapper_map_prg( 8, 3, -1);
	}

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, (mapper_regs[0] + mapper12_lowchr) >> 1);
        mapper_map_chr(2, 1, (mapper_regs[1] + mapper12_lowchr) >> 1);

		mapper_map_chr(1, 4, mapper_regs[2] + mapper12_highchr);
		mapper_map_chr(1, 5, mapper_regs[3] + mapper12_highchr);
		mapper_map_chr(1, 6, mapper_regs[4] + mapper12_highchr);
		mapper_map_chr(1, 7, mapper_regs[5] + mapper12_highchr);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2] + mapper12_lowchr);
		mapper_map_chr(1, 1, mapper_regs[3] + mapper12_lowchr);
		mapper_map_chr(1, 2, mapper_regs[4] + mapper12_lowchr);
		mapper_map_chr(1, 3, mapper_regs[5] + mapper12_lowchr);

		mapper_map_chr(2, 2, (mapper_regs[0] + mapper12_highchr) >> 1);
		mapper_map_chr(2, 3, (mapper_regs[1] + mapper12_highchr) >> 1);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static UINT8 mapper115_read(UINT16 address)
{
	UINT8 ret = cpu_open_bus;

	switch (address) {
		case 0x5080:
			ret = mapper115_prot;
			break;
		case 0x6000:
			ret = mapper115_prg;
			break;
		case 0x6001:
			ret = mapper115_chr;
			break;
	}
	return ret;
}

static void mapper115_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0x5080:
			mapper115_prot = data;
			break;
		case 0x6000:
			mapper115_prg = data;
			break;
		case 0x6001:
			mapper115_chr = data;
			mapper12_highchr = (data & 1) << 8;
			mapper12_lowchr = (data & 1) << 8;
			break;
	}

	mapper_map();
}

static void mapper118_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, mapper_regs[0] >> 1);
        mapper_map_chr(2, 1, mapper_regs[1] >> 1);
		nametable_map(0, (mapper_regs[0] >> 7) & 1);
		nametable_map(1, (mapper_regs[0] >> 7) & 1);
		nametable_map(2, (mapper_regs[1] >> 7) & 1);
		nametable_map(3, (mapper_regs[1] >> 7) & 1);

		mapper_map_chr(1, 4, mapper_regs[2]);
		mapper_map_chr(1, 5, mapper_regs[3]);
		mapper_map_chr(1, 6, mapper_regs[4]);
		mapper_map_chr(1, 7, mapper_regs[5]);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2]);
		mapper_map_chr(1, 1, mapper_regs[3]);
		mapper_map_chr(1, 2, mapper_regs[4]);
		mapper_map_chr(1, 3, mapper_regs[5]);
		nametable_map(0, (mapper_regs[2] >> 7) & 1);
		nametable_map(1, (mapper_regs[3] >> 7) & 1);
		nametable_map(2, (mapper_regs[4] >> 7) & 1);
		nametable_map(3, (mapper_regs[5] >> 7) & 1);

		mapper_map_chr(2, 2, mapper_regs[0] >> 1);
		mapper_map_chr(2, 3, mapper_regs[1] >> 1);
	}
}

static void mapper189_write(UINT16 address, UINT8 data)
{
	mapper189_reg = data | (data >> 4);
	mapper_map();
}

static void mapper189_map()
{
	mapper_map_prg(32, 0, mapper189_reg & 0x07);

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr(2, 0, mapper_regs[0] >> 1);
        mapper_map_chr(2, 1, mapper_regs[1] >> 1);

		mapper_map_chr(1, 4, mapper_regs[2]);
		mapper_map_chr(1, 5, mapper_regs[3]);
		mapper_map_chr(1, 6, mapper_regs[4]);
		mapper_map_chr(1, 7, mapper_regs[5]);
	} else {
		mapper_map_chr(1, 0, mapper_regs[2]);
		mapper_map_chr(1, 1, mapper_regs[3]);
		mapper_map_chr(1, 2, mapper_regs[4]);
		mapper_map_chr(1, 3, mapper_regs[5]);

		mapper_map_chr(2, 2, mapper_regs[0] >> 1);
		mapper_map_chr(2, 3, mapper_regs[1] >> 1);
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper119_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

    if (~mapper4_banksel & 0x80) {
		mapper_map_chr_ramrom(2, 0, (mapper_regs[0] >> 1) & 0x1f, ((mapper_regs[0]) & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(2, 1, (mapper_regs[1] >> 1) & 0x1f, ((mapper_regs[1]) & 0x40) ? MEM_RAM : MEM_ROM);

		mapper_map_chr_ramrom(1, 4, mapper_regs[2] & 0x3f, (mapper_regs[2] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 5, mapper_regs[3] & 0x3f, (mapper_regs[3] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 6, mapper_regs[4] & 0x3f, (mapper_regs[4] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 7, mapper_regs[5] & 0x3f, (mapper_regs[5] & 0x40) ? MEM_RAM : MEM_ROM);
	} else {
		mapper_map_chr_ramrom(1, 0, mapper_regs[2] & 0x3f, (mapper_regs[2] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 1, mapper_regs[3] & 0x3f, (mapper_regs[3] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 2, mapper_regs[4] & 0x3f, (mapper_regs[4] & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(1, 3, mapper_regs[5] & 0x3f, (mapper_regs[5] & 0x40) ? MEM_RAM : MEM_ROM);

		mapper_map_chr_ramrom(2, 2, (mapper_regs[0] >> 1) & 0x1f, ((mapper_regs[0]) & 0x40) ? MEM_RAM : MEM_ROM);
		mapper_map_chr_ramrom(2, 3, (mapper_regs[1] >> 1) & 0x1f, ((mapper_regs[1]) & 0x40) ? MEM_RAM : MEM_ROM);
	}

	if (Cart.Mirroring != 4)
		set_mirroring((mapper4_mirror) ? VERTICAL : HORIZONTAL);
}

static void mapper262_map()
{
    mapper_map_prg(8, 1, mapper_regs[7]);

    if (~mapper4_banksel & 0x40) {
        mapper_map_prg(8, 0, mapper_regs[6]);
        mapper_map_prg(8, 2, -2);
    } else {
        mapper_map_prg(8, 0, -2);
        mapper_map_prg(8, 2, mapper_regs[6]);
    }

	if (mapper262_reg & 0x40) {
	   mapper_map_chr_ramrom(8, 0, 0, MEM_RAM);
	} else {
		mapper_set_chrtype(MEM_ROM);

		if (~mapper4_banksel & 0x80) {
			mapper_map_chr(1, 0, (mapper_regs[0] & 0xfe) + ((mapper262_reg & 8) << 5));
			mapper_map_chr(1, 1, (mapper_regs[0] | 0x01) + ((mapper262_reg & 8) << 5));

			mapper_map_chr(1, 2, (mapper_regs[1] & 0xfe) + ((mapper262_reg & 4) << 6));
			mapper_map_chr(1, 3, (mapper_regs[1] | 0x01) + ((mapper262_reg & 4) << 6));

			mapper_map_chr(1, 4, mapper_regs[2] + ((mapper262_reg & 1) << 8));
			mapper_map_chr(1, 5, mapper_regs[3] + ((mapper262_reg & 1) << 8));
			mapper_map_chr(1, 6, mapper_regs[4] + ((mapper262_reg & 2) << 7));
			mapper_map_chr(1, 7, mapper_regs[5] + ((mapper262_reg & 2) << 7));
		} else {
			mapper_map_chr(1, 0, mapper_regs[2] + ((mapper262_reg & 8) << 5));
			mapper_map_chr(1, 1, mapper_regs[3] + ((mapper262_reg & 8) << 5));
			mapper_map_chr(1, 2, mapper_regs[4] + ((mapper262_reg & 4) << 6));
			mapper_map_chr(1, 3, mapper_regs[5] + ((mapper262_reg & 4) << 6));

			mapper_map_chr(1, 4, (mapper_regs[0] & 0xfe) + ((mapper262_reg & 1) << 8));
			mapper_map_chr(1, 5, (mapper_regs[0] | 0x01) + ((mapper262_reg & 1) << 8));

			mapper_map_chr(1, 6, (mapper_regs[1] & 0xfe) + ((mapper262_reg & 2) << 7));
			mapper_map_chr(1, 7, (mapper_regs[1] | 0x01) + ((mapper262_reg & 2) << 7));
		}
	}

	if (Cart.Mirroring != 4)
		set_mirroring(mapper4_mirror ? VERTICAL : HORIZONTAL);
}

static UINT8 mapper262_read(UINT16 address)
{
	if (address == 0x4100) {
		// 0 = Street heroes, 0xff = Shihun
		return 0;
	}

	return 0;
}

static void mapper262_write(UINT16 address, UINT8 data)
{
	if (address == 0x4100) {
		mapper262_reg = data;
		mapper_map();
	}
}

// mapper 303 - haradius zero (mmc3 + flash chip(prg,rw) + flash chip(chr,r)
// dev notes: https://pastebin.com/9UH8yGg6
#define mapper303_cmd           (mapper_regs[0x1f - 0x9])
#define mapper303_busy          (mapper_regs16[0x1f - 0x0])

static UINT8 mapper303_read(UINT16 address)
{
	if (mapper303_cmd == 0x90) { // flash chip "silicon-id" mode
		return (address & 0x01) ? 0xa4 : 0xc2;
	}
	if (mapper303_busy != 0) { // flash chip "erasing sector or chip" mode (it takes time..)
		mapper303_busy--;
		if (mapper303_busy < 2) return 0x80;
		return (mapper303_busy & 0x01) ? 0x40 : 0x00;
	}

	return mapper_prg_read_int(address);
}

static void mapper303_prg_write(UINT16 address, UINT8 data)
{
	Cart.PRGRom[PRGMap[(address & ~0x8000) / 0x2000] + (address & 0x1fff)] = data;
}

static void flash303_write(UINT16 address, UINT8 data)
{
	if (data == 0xf0) {
		mapper303_cmd = 0;
		return;
	}

	switch (mapper303_cmd) {
		case 0x00:
		case 0x80:
			if (address == 0xd555 && data == 0xaa)
				mapper303_cmd++;
			break;
		case 0x01:
		case 0x81:
			if (address == 0xaaaa && data == 0x55)
				mapper303_cmd++;
			break;
		case 0x02:
			if (address == 0xd555)
				mapper303_cmd = data;
			break;
		case 0x82: {
			switch (data) {
				case 0x10:
					bprintf(0, _T("mapper 303: full flash erase not impl.\n"));
					mapper303_busy = Cart.PRGRomSize / 0x100; // fake it
					break;
				case 0x30:
					bprintf(0, _T("mapper 303: sector erase.  addr %x\n"), address);
					for (INT32 i = 0; i < 0x10000; i++) {
						Cart.PRGRom[(PRGMap[(address & ~0x8000) / 0x2000] & 0x7f0000) + i] = 0xff;
					}
					mapper303_busy = 0x10000 / 0x100;
					break;
			}
			break;
		}
		case 0xa0:
			mapper303_prg_write(address, data);
			mapper303_cmd = 0;
			break;
	}
}

static void mapper303_write(UINT16 address, UINT8 data)
{
	flash303_write(address, data);
	mapper04_write((address & 0xfffe) | ((address >> 1) & 1), data);
}

static void mapper303_scan()
{
	ScanVar(&Cart.PRGRom[0x50000], 0x10000, "Mapper303 HighScore Sector");
}

static UINT8 *mmc5_mask; // mmc3/mmc5 ppumask-sniffer // 0x18 = rendering

static void mapper04_scanline()
{
	if (NESMode & ALT_MMC3 && (mmc5_mask[0] & 0x18) == 0x00) {
		return;
	}

	INT32 cnt = mapper4_irqcount;
	if (mapper4_irqcount == 0 || mapper4_irqreload) {
		mapper4_irqreload = 0;
		mapper4_irqcount = mapper4_irqlatch;
	} else {
		mapper4_irqcount--;
	}

	if (cnt && mapper4_irqenable && mapper4_irqcount == 0) {
		if ((~NESMode & ALT_MMC3 && mmc5_mask[0] & 0x18) || NESMode & ALT_MMC3) { // aka if (RENDERING)
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
		}
	}
}

//#undef mapper4_mirror // used in mapper_init()
#undef mapper4_irqlatch
#undef mapper4_irqcount
#undef mapper4_irqenable
#undef mapper4_irqreload
#undef mapper4_banksel

// ---[ mapper 05 (MMC5) Castlevania III, Uchuu Keibitai SDF
// PPU Hooks
static UINT8 *mmc5_nt_ram; // pointer to our ppu's nt ram
static UINT8 *mmc5_ctrl; // mmc5 ppuctrl-sniffer // 0x20 = 8x16 spr
#define MMC5RENDERING (mmc5_mask[0] & 0x18)
#define MMC58x16 (mmc5_ctrl[0] & 0x20)

// Mapper registers & ram
static UINT8 mmc5_expram[1024];

#define mmc5_prgmode		(mapper_regs[0x0])
#define mmc5_chrmode		(mapper_regs[0x1])
#define mmc5_prgprot1		(mapper_regs[0x2])
#define mmc5_prgprot2		(mapper_regs[0x3])
#define mmc5_expram_mode	(mapper_regs[0x4])
#define mmc5_mirror(x)		(mapper_regs[0x1b + (x)])
#define mmc5_filltile		(mapper_regs[0x5])
#define mmc5_fillcolor		(mapper_regs[0x6])
#define mmc5_prgexp			(mapper_regs[0x7])

#define mmc5_prg(x)			(mapper_regs16[0x4 + (x)])
#define mmc5_chr(x)			(mapper_regs16[0x10 + (x)])
#define mmc5_upperchr		(mapper_regs[0x8])

#define mmc5_split			(mapper_regs[0x9])
#define mmc5_splitside		(mapper_regs[0xa])
#define mmc5_splittile		(mapper_regs[0xb])
#define mmc5_splitscroll	(mapper_regs[0xc])
#define mmc5_splitscrollmod (mapper_regs[0xd])
#define mmc5_splitchr		(mapper_regs[0xe])

#define mmc5_irqenable		(mapper_regs[0xf])
#define mmc5_irqcompare		(mapper_regs[0x10])
#define mmc5_irqpend		(mapper_regs[0x11])

#define mmc5_mult0			(mapper_regs[0x12])
#define mmc5_mult1			(mapper_regs[0x13])

#define mmc5_irqcount		(mapper_regs[0x14])
#define mmc5_inframe		(mapper_regs[0x15])

#define mmc5_lastchr		(mapper_regs[0x16])
#define mmc5_expramattr		(mapper_regs[0x17])

#define mmc5_pcmwrmode		(mapper_regs[0x18])
#define mmc5_pcmirq			(mapper_regs[0x19])
#define mmc5_pcmdata		(mapper_regs[0x1a])

enum { CHR_GUESS = 0, CHR_TILE, CHR_SPRITE, CHR_LASTREG };

static void mapper5_reset()
{
	memset(&mmc5_expram, 0xff, sizeof(mmc5_expram));

	mmc5_prgmode = 3;
	mmc5_chrmode = 3;
	mmc5_prgexp = 7;

	mmc5_prg(0) = 0x7f;
	mmc5_prg(1) = 0x7f;
	mmc5_prg(2) = 0x7f;
	mmc5_prg(3) = 0x7f;

	mmc5_filltile = 0xff;
	mmc5_fillcolor = 0xff;
	mmc5_mult0 = 0xff;
	mmc5_mult1 = 0xff; // default

	mmc5_pcmwrmode = 0x00;
	mmc5_pcmirq = 0x00;
	mmc5_pcmdata = 0x00;
}

static void mapper5_scan()
{
	SCAN_VAR(mmc5_expram);
}

static INT16 mapper5_mixer()
{
	return (INT16)(mmc5_pcmdata << 4);
}

static void mmc5_mapchr(UINT8 type)
{
	// https://wiki.nesdev.com/w/index.php/MMC5#CHR_mode_.28.245101.29
	// When using 8x8 sprites, only registers $5120-$5127 are used. Registers $5128-$512B are completely ignored.
	// When using 8x16 sprites, registers $5120-$5127 specify banks for sprites, registers $5128-$512B apply to background tiles, and the last set of registers written to (either $5120-$5127 or $5128-$512B) will be used for I/O via PPUDATA ($2007).

	if (~MMC58x16) mmc5_lastchr = 0;

	UINT8 bank = (MMC58x16 == 0) || (MMC58x16 && type == CHR_SPRITE);

	if (MMC58x16 && type == CHR_LASTREG)
		bank = mmc5_lastchr < 8;

	UINT8 banks[0x2][0xf] = {
/*bg*/  { 0xb, 0xb, 0xb, 0x9, 0xb, 0x9, 0xb, 0x8, 0x9, 0xa, 0xb, 0x8, 0x9, 0xa, 0xb },
/*spr*/ { 0x7, 0x3, 0x7, 0x1, 0x3, 0x5, 0x7, 0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7 } };

	switch (mmc5_chrmode) {
		case 0:
			mapper_map_chr(8, 0, mmc5_chr(banks[bank][0x0]));
			break;

		case 1:
			mapper_map_chr(4, 0, mmc5_chr(banks[bank][0x1]));
			mapper_map_chr(4, 1, mmc5_chr(banks[bank][0x2]));
			break;

		case 2:
			mapper_map_chr(2, 0, mmc5_chr(banks[bank][0x3]));
			mapper_map_chr(2, 1, mmc5_chr(banks[bank][0x4]));
			mapper_map_chr(2, 2, mmc5_chr(banks[bank][0x5]));
			mapper_map_chr(2, 3, mmc5_chr(banks[bank][0x6]));
			break;

		case 3:
			mapper_map_chr(1, 0, mmc5_chr(banks[bank][0x7]));
			mapper_map_chr(1, 1, mmc5_chr(banks[bank][0x8]));
			mapper_map_chr(1, 2, mmc5_chr(banks[bank][0x9]));
			mapper_map_chr(1, 3, mmc5_chr(banks[bank][0xa]));
			mapper_map_chr(1, 4, mmc5_chr(banks[bank][0xb]));
			mapper_map_chr(1, 5, mmc5_chr(banks[bank][0xc]));
			mapper_map_chr(1, 6, mmc5_chr(banks[bank][0xd]));
			mapper_map_chr(1, 7, mmc5_chr(banks[bank][0xe]));
			break;
	}
}

static void mapper5_map()
{
	switch (mmc5_prgmode) {
		case 0:
			mapper_map_prg(32, 0, (mmc5_prg(3) & 0x7c) >> 2);
			break;

		case 1:
			mapper_map_prg(16, 0, (mmc5_prg(1) & 0x7e) >> 1, (mmc5_prg(1) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(16, 1, (mmc5_prg(3) & 0x7e) >> 1);
			break;

		case 2:
			mapper_map_prg(16, 0, (mmc5_prg(1) & 0x7e) >> 1, (mmc5_prg(1) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 2, mmc5_prg(2) & 0x7f, (mmc5_prg(2) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 3, mmc5_prg(3) & 0x7f);
			break;

		case 3:
			mapper_map_prg(8, 0, mmc5_prg(0) & 0x7f, (mmc5_prg(0) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 1, mmc5_prg(1) & 0x7f, (mmc5_prg(1) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 2, mmc5_prg(2) & 0x7f, (mmc5_prg(2) & 0x80) ? MEM_ROM : MEM_RAM);
			mapper_map_prg(8, 3, mmc5_prg(3) & 0x7f);
			break;
	}

	mapper_map_exp_prg(mmc5_prgexp);

	// Note: mapper5_ppu_clk() takes care of chr banking
}

static UINT8 mapper5_read(UINT16 address)
{
	if (address >= 0x5000 && address <= 0x5015) {
		switch (address) {
			case 0x5010: {
				bprintf(0, _T("mmc5 irq ack\n"));
				UINT8 ret = ((mmc5_pcmirq & 1) << 7) | (~mmc5_pcmwrmode & 1);
				mmc5_pcmirq &= ~1; // clear flag
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				return ret;
			}
			default:
				return nesapuRead(0, (address & 0x1f) | 0x80);
		}
	}

	if (address >= 0x5c00 && address <= 0x5fff) {
		return (mmc5_expram_mode & 2) ? mmc5_expram[address & 0x3ff] : cpu_open_bus;
	}

	switch (address) {
		case 0x5204: {
			UINT8 ret =	(mmc5_irqpend << 7) | (mmc5_inframe << 6) | (cpu_open_bus & 0x3f);
			mmc5_irqpend = 0;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			return ret;
		}

		case 0x5205:
		case 0x5206: return ((mmc5_mult0 * mmc5_mult1) >> ((address & 0x2) << 2)) & 0xff;
	}

	return cpu_open_bus;
}


static void mapper5_exp_write(UINT16 address, UINT8 data) // 6000 - 7fff
{
	Cart.WorkRAM[PRGExpMap + (address & 0x1fff)] = data;
	cart_exp_write_abort = 1;
}

static UINT8 mapper5_exp_read(UINT16 address)             // 6000 - 7fff
{
	return Cart.WorkRAM[PRGExpMap + (address & 0x1fff)];
}

static void mapper5_prg_write(UINT16 address, UINT8 data)
{
	if (mmc5_prgprot1 == 0x02 && mmc5_prgprot2 == 0x01) {
		mapper_prg_write(address, data);
	}
}

static void mapper5_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5000 && address <= 0x5015) {
		// Audio section
		switch (address) {
			case 0x5010:
				mmc5_pcmwrmode = ~data & 1;
				mmc5_pcmirq = data & 0x80;
				break;
			case 0x5011:
				if (mmc5_pcmwrmode) {
					if (data == 0x00 && mmc5_pcmirq) {
						mapper_irq(0);
						mmc5_pcmirq |= 0x01;
					}
					mmc5_pcmdata = data;
				}
				break;
			default:
				nesapuWrite(0, (address & 0x1f) | 0x80, data);
				break;
		}
		return;
	}

	if (address >= 0x5c00 && address <= 0x5fff) {
		switch (mmc5_expram_mode) {
			case 0:
			case 1:
				mmc5_expram[address & 0x3ff] = (MMC5RENDERING) ? data : 0x00;
				break;
			case 2:
				mmc5_expram[address & 0x3ff] = data;
				break;
        }
	}

	switch (address) {
		case 0x5100: mmc5_prgmode = data & 3; break;
		case 0x5101: mmc5_chrmode = data & 3; break;
		case 0x5102: mmc5_prgprot1 = data; break;
		case 0x5103: mmc5_prgprot2 = data; break;
		case 0x5104: mmc5_expram_mode = data & 3; break;
		case 0x5105:
			mmc5_mirror(0) = (data >> 0) & 3;
			mmc5_mirror(1) = (data >> 2) & 3;
			mmc5_mirror(2) = (data >> 4) & 3;
			mmc5_mirror(3) = (data >> 6) & 3;
			break;

		case 0x5106: mmc5_filltile = data; break;
		case 0x5107: mmc5_fillcolor = ((data & 3) << 6) | ((data & 3) << 4) | ((data & 3) << 2) | (data & 3); break;

		case 0x5113: mmc5_prgexp = data & 7; break;
		case 0x5114:
		case 0x5115:
		case 0x5116:
		case 0x5117: mmc5_prg(address & 3) = data; break;

		case 0x5120:
		case 0x5121:
		case 0x5122:
		case 0x5123:
		case 0x5124:
		case 0x5125:
		case 0x5126:
		case 0x5127:
		case 0x5128:
		case 0x5129:
		case 0x512a:
		case 0x512b:
			mmc5_chr(address & 0xf) = (mmc5_upperchr << 8) | data;
			mmc5_lastchr = (MMC58x16) ? (address & 0xf) : 0;
			break;

		case 0x5130: mmc5_upperchr = data & 3; break;

		case 0x5200:
			mmc5_splittile = data & 0x1f;
			mmc5_splitside = data & 0x40;
			mmc5_split = (data & 0x80) >> 7;
			break;
		case 0x5201:
			mmc5_splitscroll = data >> 3;
			mmc5_splitscrollmod = (mmc5_splitscroll < 30 ? 30 : 32);
			break;
		case 0x5202: mmc5_splitchr = data; break;

		case 0x5203: mmc5_irqcompare = data; break;
		case 0x5204:
			mmc5_irqenable = (data & 0x80) >> 7;
			M6502SetIRQLine(0, (mmc5_irqenable && mmc5_irqpend) ? CPU_IRQSTATUS_ACK : CPU_IRQSTATUS_NONE);
			break;

		case 0x5205: mmc5_mult0 = data; break;
		case 0x5206: mmc5_mult1 = data; break;
	}

    mapper_map();
}

static UINT8 mapper5_ntread(UINT16 address)
{
	if (mmc5_expram_mode == 1) {
		if ((address & 0x3c0) < 0x3c0) {
			UINT8 temp = mmc5_expram[(32 * ((address >> 5) & 0x1f)) + (address & 0x1f)];
			mmc5_expramattr = (temp & 0xc0) >> 6;
            mapper_map_chr(4, 0, (mmc5_upperchr << 6) | (temp & 0x3f));
            mapper_map_chr(4, 1, (mmc5_upperchr << 6) | (temp & 0x3f));
        } else {
			return (mmc5_expramattr << 6) | (mmc5_expramattr << 4) | (mmc5_expramattr << 2) | mmc5_expramattr;
        }
    }

	if (mmc5_split && mmc5_expram_mode <= 1) {
        UINT8 cur_tile = ((pixel >> 3) + 2) % 32;

		if ((mmc5_splitside && cur_tile >= mmc5_splittile) || (!mmc5_splitside && cur_tile < mmc5_splittile)) {
                mapper_map_chr(4, 0, mmc5_splitchr);
                mapper_map_chr(4, 1, mmc5_splitchr);
                UINT8 y = ((scanline >> 3) + mmc5_splitscroll) % mmc5_splitscrollmod;
				address = (pixel & 2) ? ((y * 32) & 0x3e0) | cur_tile : 0x3c0 | ((y << 1) & 0x38) | (cur_tile >> 2);
                return mmc5_expram[address & 0x3ff];
		} else {
			mmc5_mapchr(CHR_GUESS);
		}
    }

	switch (mmc5_mirror((address & 0x1fff) >> 10)) {
		case 0: return mmc5_nt_ram[(address & 0x3ff)];
		case 1: return mmc5_nt_ram[(address & 0x3ff) + 0x400];
		case 2: return (mmc5_expram_mode <= 1) ? mmc5_expram[address & 0x3ff] : 0;
		case 3: return ((address & 0x3c0) == 0x3c0) ? mmc5_fillcolor : mmc5_filltile;
	}

	return 0x00;
}

static void mapper5_ntwrite(UINT16 address, UINT8 data)
{
	switch (mmc5_mirror((address & 0x1fff) >> 10)) {
		case 0: mmc5_nt_ram[(address & 0x3ff)] = data; break;
		case 1: mmc5_nt_ram[(address & 0x3ff) + 0x400] = data; break;
		case 2: if (mmc5_expram_mode <= 1) mmc5_expram[address & 0x3ff] = data; break;
		case 3: /* invalid */ break;
	}
}

static void mmc5_lineadvance()
{
	if (scanline <= 240) {
		if (mmc5_inframe == 0 && MMC5RENDERING) {
			mmc5_inframe = 1;
			mmc5_irqcount = 0;
			mmc5_irqpend = 0;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
		} else if (mmc5_inframe) {
			mmc5_irqcount++;
			if (mmc5_irqcount == mmc5_irqcompare) {
				mmc5_irqpend = 1;
				if (mmc5_irqenable) {
					M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
				}
			}
		}
	}
}

static void mapper5_ppu_clk(UINT16 address)
{
	if (MMC5RENDERING == 0 || scanline >= 240) {
		mmc5_inframe = 0;
		mmc5_mapchr(CHR_LASTREG);
		return;
	}

	switch (pixel) {
		case  16: mmc5_lineadvance(); break;
		case 257: mmc5_mapchr(CHR_SPRITE); break; // sprite bank switch
		case 321: mmc5_mapchr(CHR_TILE); break; // bg bank switch
	}
}
#undef mmc5_prgmode
#undef mmc5_chrmode
#undef mmc5_prgprot1
#undef mmc5_prgprot2
#undef mmc5_expram_mode
#undef mmc5_mirror
#undef mmc5_filltile
#undef mmc5_fillcolor
#undef mmc5_prgexp
#undef mmc5_prg
#undef mmc5_chr
#undef mmc5_upperchr
#undef mmc5_split
#undef mmc5_splitside
#undef mmc5_splittile
#undef mmc5_splitscroll
#undef mmc5_splitscrollmod
#undef mmc5_splitchr
#undef mmc5_irqenable
#undef mmc5_irqcompare
#undef mmc5_irqpend
#undef mmc5_mult0
#undef mmc5_mult1
#undef mmc5_irqcount
#undef mmc5_inframe
#undef mmc5_lastchr
#undef mmc5_expramattr

// ---[ mapper 07 (AxROM) Battle Toads, Marble Madness, RC Pro-Am
static void mapper07_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper07_map()
{
	set_mirroring((mapper_regs[0] & 0x10) ? SINGLE_HIGH : SINGLE_LOW);
    mapper_map_prg(32, 0, mapper_regs[0] & 0x7);
}

// ---[ mapper 09 (mmc2) / 10 (mmc4) Mike Tyson's Punch-Out!!, Fire Emblem
#define mapper9_prg				(mapper_regs[0xf - 0])
#define mapper9_chr_lo(x)		(mapper_regs[0xf - 1 - x])
#define mapper9_chr_hi(x)		(mapper_regs[0xf - 3 - x])
#define mapper9_chr_lo_C000		(mapper_regs[0xf - 5])
#define mapper9_chr_hi_E000		(mapper_regs[0xf - 6])
#define mapper9_mirror			(mapper_regs[0xf - 7])

static void mapper09_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		switch (address & 0xf000) {
			case 0xa000: mapper9_prg       = data & 0xf; break;
			case 0xb000: mapper9_chr_lo(0) = data & 0x1f; break;
			case 0xc000: mapper9_chr_lo(1) = data & 0x1f; break;
			case 0xd000: mapper9_chr_hi(0) = data & 0x1f; break;
			case 0xe000: mapper9_chr_hi(1) = data & 0x1f; break;
			case 0xf000: mapper9_mirror    = data & 0x1; break;
		}
		mapper_map();
	}
}

static void mapper09_map()
{
	set_mirroring((mapper9_mirror) ? HORIZONTAL : VERTICAL);
	mapper_map_prg( 8, 0, mapper9_prg);
	mapper_map_chr( 4, 0, mapper9_chr_lo(mapper9_chr_lo_C000));
	mapper_map_chr( 4, 1, mapper9_chr_hi(mapper9_chr_hi_E000));
}

static void mapper10_map()
{
	set_mirroring((mapper9_mirror) ? HORIZONTAL : VERTICAL);
	mapper_map_prg(16, 0, mapper9_prg);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 4, 0, mapper9_chr_lo(mapper9_chr_lo_C000));
	mapper_map_chr( 4, 1, mapper9_chr_hi(mapper9_chr_hi_E000));
}

static void mapper09_ppu_clk(UINT16 busaddr)
{
	switch (busaddr & 0x3ff8) {
		case 0x0fd8:
			mapper9_chr_lo_C000 = 0;
			mapper_map();
			break;
		case 0x0fe8:
			mapper9_chr_lo_C000 = 1;
			mapper_map();
			break;
		case 0x1fd8:
			mapper9_chr_hi_E000 = 0;
			mapper_map();
			break;
		case 0x1fe8:
			mapper9_chr_hi_E000 = 1;
			mapper_map();
			break;
	}
}

#undef mapper9_prg
#undef mapper9_chr_lo
#undef mapper9_chr_hi
#undef mapper9_chr_lo_C000
#undef mapper9_chr_hi_E000
#undef mapper9_mirror

// ---[ mapper 13 (CPROM) Videomation
static void mapper13_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
	}

	mapper_map();
}

static void mapper13_map()
{
	mapper_map_chr(4, 0, 0);
	mapper_map_chr(4, 1, mapper_regs[0] & 0x03);
}

// ---[ mapper 133 (Sachen ??) Jovial Race
static void mapper133_write(UINT16 address, UINT8 data)
{
	if ((address & 0x4100) == 0x4100) {
		mapper_regs[0] = data;
	}

	mapper_map();
}

static void mapper133_map()
{
	mapper_map_chr(8, 0, mapper_regs[0] & 0x03);
	mapper_map_prg(32, 0, (mapper_regs[0] >> 2) & 0x01);
}

// --[ mapper 16, 153, 159 - Bandai FCG
#define mapper16_mirror		(mapper_regs[0x1f - 0])
#define mapper16_irqenable	(mapper_regs[0x1f - 1])
#define mapper16_irqcount	(mapper_regs16[0x1f - 0])
#define mapper16_irqlatch	(mapper_regs16[0x1f - 1])

static void mapper16_write(UINT16 address, UINT8 data)
{
	switch (address & 0x000f) {
		case 0x00: case 0x01: case 0x02:
		case 0x03: case 0x04: case 0x05:
		case 0x06: case 0x07: case 0x08:
			mapper_regs[address & 0xf] = data;
			break; // mixed prg, chr

		case 0x09:
			mapper16_mirror = data & 0x3;
			break;

		case 0x0a:
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			mapper16_irqenable = data & 1;
			mapper16_irqcount = mapper16_irqlatch;
			break;

		case 0x0b:
			mapper16_irqlatch = (mapper16_irqlatch & 0xff00) | data;
			break;

		case 0x0c:
			mapper16_irqlatch = (mapper16_irqlatch & 0x00ff) | (data << 8);
			break;

		case 0x0d: // x24 i2c write
			return; // don't bother mapper_map(); below..
			break;
	}

	mapper_map();
}

static void mapper16_map()
{
	mapper_map_prg(16, 0, mapper_regs[8]);
	mapper_map_prg(16, 1, -1);

	mapper_map_chr( 1, 0, mapper_regs[0]);
	mapper_map_chr( 1, 1, mapper_regs[1]);
	mapper_map_chr( 1, 2, mapper_regs[2]);
	mapper_map_chr( 1, 3, mapper_regs[3]);
	mapper_map_chr( 1, 4, mapper_regs[4]);
	mapper_map_chr( 1, 5, mapper_regs[5]);
	mapper_map_chr( 1, 6, mapper_regs[6]);
	mapper_map_chr( 1, 7, mapper_regs[7]);

	switch (mapper16_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper153_map()
{
	mapper_map_prg(16, 0, (mapper_regs[8] & 0xf) | ((mapper_regs[0] & 0x1) << 4));
	mapper_map_prg(16, 1, 0xf | ((mapper_regs[0] & 0x1) << 4));

	mapper_map_chr( 8, 0, 0);

	switch (mapper16_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper16_cycle()
{
	if (mapper16_irqenable) {
		mapper16_irqcount--;

		if (mapper16_irqcount == 0xffff) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper16_irqenable = 0;
		}
	}
}
#undef mapper16_mirror
#undef mapper16_irqenable
#undef mapper16_irqcount

// --[ mapper 18 - Jaleco (Magical John, Pizza Pop etc)
#define mapper18_irqenable	(mapper_regs[0x1f - 0])
#define mapper18_mirror     (mapper_regs[0x1f - 1])
#define mapper18_prg(x)     (mapper_regs[0 + (x)])
#define mapper18_chr(x)     (mapper_regs[4 + (x)])
#define mapper18_irqcount	(mapper_regs16[0x1f - 0])
#define mapper18_irqlatch	(mapper_regs16[0x1f - 1])
#define mapper18_irqmask	(mapper_regs16[0x1f - 2])

static void nib2byte(UINT8 &byte, UINT8 nib, INT32 high)
{
	if (high == 0) {
		byte = (byte & 0xf0) | (nib & 0xf);
	} else {
		byte = (byte & 0x0f) | (nib & 0xf) << 4;
	}
}

static void mapper18_write(UINT16 address, UINT8 data)
{
	INT32 highbits = address & 1;

	switch (address & 0xf003) {
		case 0x8000: case 0x8001:
			nib2byte(mapper18_prg(0), data, highbits);
			break;
		case 0x8002: case 0x8003:
			nib2byte(mapper18_prg(1), data, highbits);
			break;
		case 0x9000: case 0x9001:
			nib2byte(mapper18_prg(2), data, highbits);
			break;

		case 0xa000: case 0xa001:
			nib2byte(mapper18_chr(0), data, highbits);
			break;
		case 0xa002: case 0xa003:
			nib2byte(mapper18_chr(1), data, highbits);
			break;

		case 0xb000: case 0xb001:
			nib2byte(mapper18_chr(2), data, highbits);
			break;
		case 0xb002: case 0xb003:
			nib2byte(mapper18_chr(3), data, highbits);
			break;

		case 0xc000: case 0xc001:
			nib2byte(mapper18_chr(4), data, highbits);
			break;
		case 0xc002: case 0xc003:
			nib2byte(mapper18_chr(5), data, highbits);
			break;

		case 0xd000: case 0xd001:
			nib2byte(mapper18_chr(6), data, highbits);
			break;
		case 0xd002: case 0xd003:
			nib2byte(mapper18_chr(7), data, highbits);
			break;

		case 0xe000: mapper18_irqlatch = (mapper18_irqlatch & 0xfff0) | (data & 0xf); break;
		case 0xe001: mapper18_irqlatch = (mapper18_irqlatch & 0xff0f) | ((data & 0xf) << 4); break;
		case 0xe002: mapper18_irqlatch = (mapper18_irqlatch & 0xf0ff) | ((data & 0xf) << 8); break;
		case 0xe003: mapper18_irqlatch = (mapper18_irqlatch & 0x0fff) | ((data & 0xf) << 12); break;
		case 0xf000:
			mapper18_irqcount = mapper18_irqlatch;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf001: {
			const UINT16 masks[4] = { (1<<16) - 1, (1<<12) - 1, (1<<8) - 1, (1<<4) - 1 };
			UINT8 maskpos = 0;
			mapper18_irqenable = data & 1;
			switch (data & (2|4|8)) {
				case 8: maskpos = 3; break;
				case 4: maskpos = 2; break;
				case 2: maskpos = 1; break;
				default: maskpos = 0; break;
			}
			mapper18_irqmask = masks[maskpos];

			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		}
		case 0xf002: mapper18_mirror = data & 3; break;
	}

	mapper_map();
}

static void mapper18_map()
{
	mapper_map_prg( 8, 0, mapper18_prg(0));
	mapper_map_prg( 8, 1, mapper18_prg(1));
	mapper_map_prg( 8, 2, mapper18_prg(2));
	mapper_map_prg( 8, 3, -1);

	for (INT32 i = 0; i < 8; i++) {
		mapper_map_chr( 1, i, mapper18_chr(i));
	}

	switch (mapper18_mirror) {
		case 0: set_mirroring(HORIZONTAL); break;
		case 1: set_mirroring(VERTICAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper18_cycle()
{
	if (mapper18_irqenable) {
		UINT16 count = mapper18_irqcount & mapper18_irqmask;
		count--;
		if (count == 0) {
			mapper_irq(2); // 2 cyc delay fixes jeebies in pizza-pop (mid-hud) and magic john (intro)
		}
		mapper18_irqcount = (mapper18_irqcount & ~mapper18_irqmask) | (count & mapper18_irqmask);
	}
}
#undef mapper18_irqenable
#undef mapper18_mirror
#undef mapper18_prg
#undef mapper18_chr
#undef mapper18_irqcount
#undef mapper18_irqlatch
#undef mapper18_irqmask

// ---[ mapper 19 Namco 163 + WSG (wave sound gen)
#define mapper19_irqcount		(mapper_regs16[0x1f - 0])
#define mapper19_irqenable  	(mapper_regs[0x1f - 0])
#define mapper19_soundaddr		(mapper_regs[0x1f - 1])
#define mapper19_soundaddrinc	(mapper_regs[0x1f - 2])
#define mapper19_soundenable	(mapper_regs[0x1f - 3])
#define mapper19_chrram_lo		(mapper_regs[0x1f - 4])
#define mapper19_chrram_mid		(mapper_regs[0x1f - 5])
#define mapper19_prg(x)     	(mapper_regs[0 + (x)]) // 0 - 2
#define mapper19_mapper210      (mapper_regs[0x1f - 6])
#define namco340_mirror			(mapper_regs[0x1f - 7])

// since chr mapping is a bit "advanced", we use a little struct
// instead of the usual mapper registers. mapper19_scan() takes care
// of the state-stuff (as well as the WSG channel regs, etc)

struct namco163_chrstuff {
	UINT8 ram;
	UINT8 data;
};

static namco163_chrstuff n163_chr[12]; // chr mapping: 8 chr banks, 4 nt banks

static UINT8 mapper19_soundram[0x80];

struct namco163_channel {
	UINT32 enabled;
	UINT32 freq;
	UINT32 phase;
	UINT32 vol;
	UINT32 len;
	UINT32 address;
	UINT32 accum;
};

static namco163_channel n163_ch[8];
static INT32 n163_channels = 0;

static void namco163_process_channel(INT16 address, UINT8 data) {
	namco163_channel *c = &n163_ch[(address >> 3) & 7];

	switch (address & 7) {
		case 0x1:
			c->phase = (c->phase & ~0x0000ff) | data;
			break;
		case 0x3:
			c->phase = (c->phase & ~0x00ff00) | (data << 8);
			break;
		case 0x5:
			c->phase = (c->phase & ~0xff0000) | (data << 16);
			break;
		case 0x0:
			c->freq = (c->freq & ~0x0000ff) | data;
			break;
		case 0x2:
			c->freq = (c->freq & ~0x00ff00) | (data << 8);
			break;
		case 0x4:
			c->freq = (c->freq & ~0xff0000) | ((data & 3) << 16);
			c->len = 0x100 - (data & 0xfc);
			c->enabled = data >> 5;
#if 0
			bprintf(0, _T("ch %d enabled%X?\n"), (address / 8) & 7, c->enabled);
			bprintf(0, _T("len: %X\n"), c->len);
			bprintf(0, _T("phase: %X\n"), c->phase);
#endif
			break;
		case 0x6:
			c->address = data;
			break;
		case 0x7:
			c->vol = (data & 0xf) * 8;
			if (address == 0x7f) {
				n163_channels = (data >> 4) & 7;
			}
#if 0
			bprintf(0, _T("ch %d vol %X data %x.     channels %d    (cycle: %d)\n"), (address / 8) & 7, c->vol, data, n163_channels, M6502TotalCycles());
#endif
			break;
	}
}

static void namco163_update_phase(namco163_channel *c, INT32 ch)
{
	ch = 0x40 + ch * 8;
	mapper19_soundram[ch + 5] = (c->phase >> 16) & 0xff;
	mapper19_soundram[ch + 3] = (c->phase >>  8) & 0xff;
	mapper19_soundram[ch + 1] = (c->phase >>  0) & 0xff;
}

static UINT32 namco163_fetch_sample(namco163_channel *c)
{
	UINT32 wave_addr = c->address + (c->phase >> 16);
	UINT32 wave_sam = mapper19_soundram[(wave_addr & 0xff) >> 1];
	wave_sam >>= (wave_addr & 1) << 2; // LE-ordered nibble

	return ((wave_sam & 0xf) - 8) * c->vol;
}

static INT16 namco163_wavesynth()
{
	if (!mapper19_soundenable) return 0;

	INT32 sample = 0;

	for (INT32 ch = 7; ch >= (7 - n163_channels); ch--) {
		namco163_channel *c = &n163_ch[ch];

		if (c->enabled && c->vol) {
			sample += namco163_fetch_sample(c);

			if (c->accum == 0) {
				c->accum = (n163_channels + 1) * 0xf;
				c->phase = (c->phase + c->freq) % (c->len << 16);
				namco163_update_phase(c, ch);
			}

			c->accum--;
		}
	}

	return sample;
}

static void mapper19_reset()
{
	memset(&n163_chr, 0, sizeof(n163_chr));
	memset(&n163_ch, 0, sizeof(n163_ch));
	memset(&mapper19_soundram, 0, sizeof(mapper19_soundram));
}

static void mapper19_scan()
{
	SCAN_VAR(n163_chr); // chr/nt mapping regs

	SCAN_VAR(mapper19_soundram); // WSG soundram

	for (INT32 i = 0x40; i < 0x80; i++) { // rebuild n163_ch from soundram
		namco163_process_channel(i, mapper19_soundram[i]);
	}
}

static INT16 mapper19_mixer()
{
	INT32 sum = namco163_wavesynth();
	return (INT16)(sum * 1.75);
}

static void mapper19_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf800) {
		case 0x4800:
			if (mapper19_soundaddr >= 0x40) {
				namco163_process_channel(mapper19_soundaddr, data);
			}
			mapper19_soundram[mapper19_soundaddr] = data;
			mapper19_soundaddr = (mapper19_soundaddr + mapper19_soundaddrinc) & 0x7f;
			return; // [sound core] avoid calling mapper_map() below
		case 0x5000:
			mapper19_irqcount = (mapper19_irqcount & 0xff00) | data;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0x5800:
			mapper19_irqcount = (mapper19_irqcount & 0x00ff) | ((data & 0x7f) << 8);
			mapper19_irqenable = data & 0x80;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xe000:
			mapper19_prg(0) = data & 0x3f;
			mapper19_soundenable = !(data & 0x40);
			namco340_mirror = (data & 0xc0) >> 6;
			break;
		case 0xe800:
			mapper19_prg(1) = data & 0x3f;
			mapper19_chrram_lo = !(data & 0x40);
			mapper19_chrram_mid = !(data & 0x80);
			break;
		case 0xf000:
			mapper19_prg(2) = data & 0x3f;
			break;
		case 0xf800:
			mapper19_soundaddr = data & 0x7f;
			mapper19_soundaddrinc = (data >> 7) & 1;
			return; // [sound core] avoid calling mapper_map() below

		case 0x8000: case 0x8800:
		case 0x9000: case 0x9800: {
			INT32 bank = (address - 0x8000) >> 11;
			n163_chr[bank].ram = mapper19_chrram_lo;
			n163_chr[bank].data = data;
			break;
		}
		case 0xa000: case 0xa800:
		case 0xb000: case 0xb800: {
			INT32 bank = (address - 0x8000) >> 11;
			n163_chr[bank].ram = mapper19_chrram_mid;
			n163_chr[bank].data = data;
			break;
		}
		case 0xc000: case 0xc800:
		case 0xd000: case 0xd800: {
			INT32 nt = (address - 0xc000) >> 11;
			n163_chr[8 + nt].data = data;
			break;
		}
	}

	mapper_map();
}

static UINT8 mapper19_read(UINT16 address)
{
	UINT8 ret = 0;

	switch (address & 0xf800) {
		case 0x4800:
			ret = mapper19_soundram[mapper19_soundaddr];
			mapper19_soundaddr = (mapper19_soundaddr + mapper19_soundaddrinc) & 0x7f;
			break;

		case 0x5000:
			ret = mapper19_irqcount & 0xff;
			break;

		case 0x5800:
			ret = (mapper19_irqcount >> 8) & 0xff;
			break;
	}

	return ret;
}

static void mapper19_map()
{
	mapper_map_prg(8, 0, mapper19_prg(0));
	mapper_map_prg(8, 1, mapper19_prg(1));
	mapper_map_prg(8, 2, mapper19_prg(2));
	mapper_map_prg(8, 3, -1);

	for (INT32 i = 0; i < 8; i++) {
		if (mapper19_mapper210 == 0 && n163_chr[i].ram && n163_chr[i].data >= 0xe0) {
			mapper_set_chrtype(MEM_RAM);
			mapper_map_chr(1, i, n163_chr[i].data & 0x1f);
		} else {
			mapper_set_chrtype(MEM_ROM);
			mapper_map_chr(1, i, n163_chr[i].data);
		}
	}

	switch (mapper19_mapper210) {
		case 0:	// mapper 19 (namco163) only!
			for (INT32 i = 8; i < 12; i++) {
				if (n163_chr[i].data >= 0xe0) {
					nametable_map(i & 3, n163_chr[i].data & 1);
				} else {
					nametable_mapraw(i & 3, Cart.CHRRom + (n163_chr[i].data << 10), MEM_ROM);
				}
			}
			break;
		case 1: // mapper 210 submapper 1 (namco 175): hardwired h/v mirroring
			break;
		case 2: // mapper 210 submapper 2 (namco 340): selectable
			switch (namco340_mirror) {
				case 0: set_mirroring(SINGLE_LOW); break;
				case 1: set_mirroring(VERTICAL); break;
				case 2: set_mirroring(HORIZONTAL); break;
				case 3: set_mirroring(SINGLE_HIGH); break;
			}
			break;
	}
}

static void mapper19_cycle()
{
	if (mapper19_irqenable) {
		mapper19_irqcount++;

		if (mapper19_irqcount == 0x7fff) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper19_irqenable = 0;
		}
	}
}
#undef mapper19_irqcount
#undef mapper19_irqenable
#undef mapper19_soundaddr
#undef mapper19_soundaddrinc
#undef mapper19_soundenable
#undef mapper19_chrram_lo
#undef mapper19_chrram_mid
#undef mapper19_prg
//#undef mapper19_mapper210 -- used in mapper_init!
#undef namco340_mirror

// --[ mapper 21, 22, 23, 25 - Konami VRC2/VRC4
#define mapper23_prg(x)		(mapper_regs[0x00 + x])
#define mapper23_chr(x)		(mapper_regs[0x02 + x])
#define mapper23_chrhigh(x) (mapper_regs[0x0a + x])
#define mapper23_prgswap	(mapper_regs[0x12])
#define mapper23_irqrepeat	(mapper_regs[0x13])
#define mapper23_mirror		(mapper_regs[0x14])
#define vrc2and4_ines22		(mapper_regs[0x17])

#define mapper23_irqenable	(mapper_regs[0x18])
#define mapper23_irqlatch	(mapper_regs[0x19])
#define mapper23_irqmode	(mapper_regs[0x1a])
#define mapper23_irqcycle	(mapper_regs16[0x1f - 0])
#define mapper23_irqcount	(mapper_regs16[0x1f - 1])

static void vrc2vrc4_map()
{
	if (mapper23_prgswap & 2) {
		mapper_map_prg( 8, 0, -2);
		mapper_map_prg( 8, 1, mapper23_prg(1));
		mapper_map_prg( 8, 2, mapper23_prg(0));
		mapper_map_prg( 8, 3, -1);
	} else {
		mapper_map_prg( 8, 0, mapper23_prg(0));
		mapper_map_prg( 8, 1, mapper23_prg(1));
		mapper_map_prg( 8, 2, -2);
		mapper_map_prg( 8, 3, -1);
	}

	for (INT32 i = 0; i < 8; i++) {
		mapper_map_chr( 1, i, ((mapper23_chrhigh(i) << 4) | mapper23_chr(i)) >> vrc2and4_ines22);
	}

	switch (mapper23_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void vrc2vrc4_write(UINT16 address, UINT8 data)
{
	address &= 0xf003;
	if (address >= 0xb000 && address <= 0xe003) {
		UINT8 reg = ((address >> 1) & 1) | ((address - 0xb000) >> 11);
		if (address & 1) { // high 5-bits
			mapper23_chrhigh(reg) = data & 0x1f;
		} else { // low 4-bits
			mapper23_chr(reg) = data & 0xf;
		}
		mapper_map();
	} else {
		switch (address & 0xf003) {
			case 0x8000:
			case 0x8001:
			case 0x8002:
			case 0x8003:
				mapper23_prg(0) = data & 0x1f;
				mapper_map();
				break;
			case 0xA000:
			case 0xA001:
			case 0xA002:
			case 0xA003:
				mapper23_prg(1) = data & 0x1f;
				mapper_map();
				break;
			case 0x9000:
			case 0x9001:
				if (data != 0xff) { // Wai Wai World, what are you doing?
					mapper23_mirror = data & 3;
				}
				mapper_map();
				break;
			case 0x9002:
			case 0x9003:
				mapper23_prgswap = data;
				mapper_map();
				break;
			case 0xf000:
				mapper23_irqlatch = (mapper23_irqlatch & 0xf0) | (data & 0xf);
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 0xf001:
				mapper23_irqlatch = (mapper23_irqlatch & 0x0f) | ((data & 0xf) << 4);
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 0xf002:
				mapper23_irqrepeat = data & 1;
				mapper23_irqenable = data & 2;
				mapper23_irqmode = data & 4; // 1 cycle, 0 scanline
				if (data & 2) {
					mapper23_irqcycle = 0;
					mapper23_irqcount = mapper23_irqlatch;
				}
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 0xf003:
				mapper23_irqenable = mapper23_irqrepeat;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
		}
	}
}

static void vrc2vrc4_cycle()
{
	if (mapper23_irqenable) {
		if (mapper23_irqmode) { // cycle mode
			mapper23_irqcount++;
			if (mapper23_irqcount >= 0x100) {
				mapper_irq(5);
				mapper23_irqcount = mapper23_irqlatch;
			}
		} else {
			mapper23_irqcycle += 3; // scanline mode
			if (mapper23_irqcycle >= 341) {
				mapper23_irqcycle -= 341;
				mapper23_irqcount++;
				if (mapper23_irqcount == 0x100) {
					mapper_irq(5);
					mapper23_irqcount = mapper23_irqlatch;
				}
			}
		}
	}
}

static void mapper21_write(UINT16 address, UINT8 data)
{
	address = (address & 0xf000) | ((address >> 1) & 0x3);

	vrc2vrc4_write(address, data);
}

static void mapper22_write(UINT16 address, UINT8 data)
{
	address |= ((address >> 2) & 0x3);

	vrc2vrc4_write((address & 0xf000) | ((address >> 1) & 1) | ((address << 1) & 2), data);
}

static void mapper23_write(UINT16 address, UINT8 data)
{
	address |= ((address >> 2) & 0x3) | ((address >> 4) & 0x3) | ((address >> 6) & 0x3);

	vrc2vrc4_write(address, data);
}

#define mapper25_write		mapper22_write   // same address line hookup/swapping

// --[ mapper 24, 26 (Konami VRC6)
#define mapper24_prg(x)		(mapper_regs[0x00 + x])
#define mapper24_chr(x)		(mapper_regs[0x02 + x])
#define mapper24_mirror		(mapper_regs[0x10])
#define mapper24_irqenable	(mapper_regs[0x11])
#define mapper24_irqrepeat	(mapper_regs[0x12])
#define mapper24_irqlatch	(mapper_regs[0x13])
#define mapper24_irqmode	(mapper_regs[0x14])
#define mapper24_irqcount	(mapper_regs16[0x1f - 0])
#define mapper24_irqcycle	(mapper_regs16[0x1f - 1])
#define mapper26			(mapper_regs[0x15])

struct vrc6_channel {
	INT32 dutyacc;
	INT32 phaseacc;
	INT32 volacc;
	INT16 sample;
	UINT8 regs[4];
};

struct vrc6_master {
	UINT8 reg;
	INT32 disable;
	INT32 octave_shift;
};

static vrc6_channel vrc6_ch[3];
static vrc6_master vrc6_cntrl;

static void vrc6_reset()
{
	memset(&vrc6_ch, 0, sizeof(vrc6_ch));
	memset(&vrc6_cntrl, 0, sizeof(vrc6_cntrl));
}

static void vrc6_scan()
{
	SCAN_VAR(vrc6_ch);
	SCAN_VAR(vrc6_cntrl);
}

static INT16 vrc6_pulse(INT32 num)
{
	struct vrc6_channel *ch = &vrc6_ch[num];

	if (ch->regs[2] & 0x80 && vrc6_cntrl.disable == 0) { // channel enabled?
		ch->sample = 0;
		if (ch->regs[0] & 0x80) { // ignore duty
			ch->sample = (ch->regs[0] & 0xf) << 8;
		} else {
			if (ch->dutyacc > ((ch->regs[0] >> 4) & 7))
				ch->sample = (ch->regs[0] & 0xf) << 8;
			ch->phaseacc--;
			if (ch->phaseacc <= 0) {
				ch->phaseacc = ((((ch->regs[2] & 0xf) << 8) | ch->regs[1]) + 1) >> vrc6_cntrl.octave_shift;
				ch->dutyacc = (ch->dutyacc + 1) & 0xf;
			}
		}
	}
	return ch->sample;
}

static INT16 vrc6_saw()
{
	struct vrc6_channel *ch = &vrc6_ch[2];

	if (ch->regs[2] & 0x80 && vrc6_cntrl.disable == 0) { // channel enabled?
		ch->sample = ((ch->volacc >> 3) & 0x1f) << 8;
		ch->phaseacc--;
		if (ch->phaseacc <= 0) {
			ch->phaseacc = ( ((((ch->regs[2] & 0xf) << 8) | ch->regs[1]) + 1)
							 >> vrc6_cntrl.octave_shift ) * 2; // each phase takes 2 cycles (* 2)
			ch->volacc += ch->regs[0] & 0x3f;
			ch->dutyacc++;
			if (ch->dutyacc == 7) {
				ch->dutyacc = 0;
				ch->volacc = 0;
			}
		}
	}
	return ch->sample;
}

static INT16 vrc6_mixer()
{
	INT32 sum = vrc6_saw();
    sum += vrc6_pulse(0);
    sum += vrc6_pulse(1);
	return (INT16)(sum * 0.75);
}

static void vrc6_sound_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0x9003:                           // master control reg
			vrc6_cntrl.reg = data;
			vrc6_cntrl.disable = data & 1;
			vrc6_cntrl.octave_shift = (data & 4) ? 8 : 0;
			if (vrc6_cntrl.octave_shift == 0)
				vrc6_cntrl.octave_shift = (data & 2) ? 4 : 0;
			break;
		case 0x9000: case 0x9001: case 0x9002: // pulse
		case 0xa000: case 0xa001: case 0xa002: // pulse 2
		case 0xb000: case 0xb001: case 0xb002: // saw
			vrc6_ch[((address >> 12) - 9) & 3].regs[address & 3] = data;
			break;
	}
}

static void vrc6_map()
{
	mapper_map_prg(16, 0, mapper24_prg(0));
	mapper_map_prg( 8, 2, mapper24_prg(1));
	mapper_map_prg( 8, 3, -1);

	for (INT32 i = 0; i < 8; i++)
		mapper_map_chr( 1, i, mapper24_chr(i));

	switch (mapper24_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void vrc6_write(UINT16 address, UINT8 data)
{
	if (mapper26) {
		// VRC6b has bits 0 & 1 swapped
		address = (address & 0xfffc) | ((address >> 1) & 1) | ((address << 1) & 2);
	}

	if (address >= 0x9000 && address <= 0xb002) {
		vrc6_sound_write(address & 0xf003, data);
		return;
	}

	switch (address & 0xf003) {
		case 0x8000: mapper24_prg(0) = data; break;
		case 0xb003: mapper24_mirror = (data & 0xc) >> 2; break;
		case 0xc000: mapper24_prg(1) = data; break;
		case 0xd000: mapper24_chr(0) = data; break;
		case 0xd001: mapper24_chr(1) = data; break;
		case 0xd002: mapper24_chr(2) = data; break;
		case 0xd003: mapper24_chr(3) = data; break;
		case 0xe000: mapper24_chr(4) = data; break;
		case 0xe001: mapper24_chr(5) = data; break;
		case 0xe002: mapper24_chr(6) = data; break;
		case 0xe003: mapper24_chr(7) = data; break;
		case 0xf000:
			mapper24_irqlatch = data;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf001:
			mapper24_irqrepeat = data & 1;
			mapper24_irqenable = data & 2;
			mapper24_irqmode = data & 4;
			if (mapper24_irqenable) {
				mapper24_irqcycle = 0;
				mapper24_irqcount = mapper24_irqlatch;
			}
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf002:
			mapper24_irqenable = mapper24_irqrepeat;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
	}

	mapper_map();
}

static void vrc6_cycle()
{
	if (mapper24_irqenable) {
		if (mapper24_irqmode) { // cycle mode
			mapper24_irqcount++;
			if (mapper24_irqcount >= 0x100) {
				mapper_irq(0);
				mapper24_irqcount = mapper24_irqlatch;
			}
		} else {
			mapper24_irqcycle += 3; // scanline mode
			if (mapper24_irqcycle >= 341) {
				mapper24_irqcycle -= 341;
				mapper24_irqcount++;
				if (mapper24_irqcount == 0x100) {
					mapper_irq(0);
					mapper24_irqcount = mapper24_irqlatch;
				}
			}
		}
	}
}
#undef mapper24_prg
#undef mapper24_chr
#undef mapper24_mirror
#undef mapper24_irqenable
#undef mapper24_irqrepeat
#undef mapper24_irqlatch
#undef mapper24_irqmode
#undef mapper24_irqcount
#undef mapper24_irqcycle

// --[ mapper 228: Action52
#define mapper228_mirror	(mapper_regs[0x1f - 0])
#define mapper228_prg		(mapper_regs[0x1f - 1])
#define mapper228_prgtype	(mapper_regs[0x1f - 2])
#define mapper228_chr		(mapper_regs[0x1f - 3])
#define mapper228_weird(x)	(mapper_regs[0 + (x)])

static void mapper228_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		UINT8 csel = (address >> 11) & 0x03;
		if (csel == 3) csel = 2;

		mapper228_prg = ((address >> 6) & 0x1f) | (csel << 5);
		mapper228_prgtype = address & 0x20;
		mapper228_chr = ((address & 0x0f) << 2) | (data & 0x03);
		mapper228_mirror = (~address & 0x2000) >> 13;

		mapper_map();
	}
}

static void mapper228_psg_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5ff0) {
		mapper228_weird(address & 3) = data;
	}
}

static UINT8 mapper228_psg_read(UINT16 address)
{
	if (address >= 0x5ff0) {
		return mapper228_weird(address & 3);
	}

	return cpu_open_bus;
}

static void mapper228_map()
{
	if (mapper228_prgtype) {
		mapper_map_prg(16, 0, mapper228_prg);
		mapper_map_prg(16, 1, mapper228_prg);
	} else {
		mapper_map_prg(16, 0, (mapper228_prg & ~1));
		mapper_map_prg(16, 1, mapper228_prg | 1);
	}

	mapper_map_chr( 8, 0, mapper228_chr);
	set_mirroring((mapper228_mirror) ? VERTICAL : HORIZONTAL);
}
#undef mapper228_mirror
#undef mapper228_prg
#undef mapper228_prgtype
#undef mapper228_chr

// --[ mapper 90: jy company
#define mapper90_209                (mapper_regs[0x1f - 0x0])
#define mapper90_211                (mapper_regs[0x1f - 0x1])

// 5800&3: multiplier / accumulator
#define mapper90_mul0   			(mapper_regs[0x1f - 0x2])
#define mapper90_mul1   			(mapper_regs[0x1f - 0x3])
#define mapper90_accu   			(mapper_regs[0x1f - 0x4])
#define mapper90_testreg			(mapper_regs[0x1f - 0x5])

// c000 - cfff&7: IRQ
#define mapper90_irqenable			(mapper_regs[0x1f - 0x6])
#define mapper90_irqmode			(mapper_regs[0x1f - 0x7])
#define mapper90_irqcounter  		(mapper_regs[0x1f - 0x8])
#define mapper90_irqprescaler	  	(mapper_regs[0x1f - 0x9])
#define mapper90_irqxor  			(mapper_regs[0x1f - 0xa])
#define mapper90_irqprescalermask	(mapper_regs[0x1f - 0xb])
#define mapper90_irqunknown  		(mapper_regs[0x1f - 0xc])

// d000 - d7ff&3: mode
#define mapper90_mode				(mapper_regs[0x1f - 0xd])
#define mapper90_mirror				(mapper_regs[0x1f - 0xe])
#define mapper90_ppu				(mapper_regs[0x1f - 0xf])
#define mapper90_obank				(mapper_regs[0x1f - 0x10])

// 8000 - 87ff&3: PRG
#define mapper90_prg(x)				(mapper_regs[0x00 + (x)])

// 9000 - 97ff&7: CHRlo
#define mapper90_chrlo(x)			(mapper_regs[0x04 + (x)])

// a000 - a7ff&7: CHRhi (actually 8bit, ran out of 8bit regs)
#define mapper90_chrhi(x)			(mapper_regs16[0x00 + (x)])

// mmc4-like 4k chr banking latch
#define mapper90_chrlatch(x)        (mapper_regs16[0x08 + (x)])

// b000 - b7ff&3: nametable config (&4 = MSB)
#define mapper90_nt(x)              (mapper_regs16[0x0a + (x)])

// forward --
static void mapper90_ppu_clockmmc4(UINT16 address);

static void mapper90_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000 && address <= 0x87ff) {
		mapper90_prg(address & 3) = data & 0x3f;
	}

	if (address >= 0x9000 && address <= 0x97ff) {
		mapper90_chrlo(address & 7) = data;
	}

	if (address >= 0xa000 && address <= 0xa7ff) {
		mapper90_chrhi(address & 7) = data;
	}

	if (address >= 0xb000 && address <= 0xb7ff) {
		if (~address & 4) { // LSB
			mapper90_nt(address & 3) = (mapper90_nt(address & 3) & 0xff00) | (data << 0);
		} else { // MSB
			mapper90_nt(address & 3) = (mapper90_nt(address & 3) & 0x00ff) | (data << 8);
		}
	}

	if (address >= 0xc000 && address <= 0xcfff) {
		switch (address & 0x7) {
			case 0:
				mapper90_irqenable = data & 1;
				if (!mapper90_irqenable) {
					M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				}
				break;
			case 1:
				mapper90_irqmode = data;
				mapper90_irqprescalermask = (data & 4) ? 0x7 : 0xff;
				break;
			case 2:
				mapper90_irqenable = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 3:
				mapper90_irqenable = 1;
				break;
			case 4:
				mapper90_irqprescaler = data ^ mapper90_irqxor;
				break;
			case 5:
				mapper90_irqcounter = data ^ mapper90_irqxor;
				break;
			case 6:
				mapper90_irqxor = data;
				break;
			case 7:
				mapper90_irqunknown = data;
				break;
		}
		return;
	}

	if (address >= 0xd000 && address <= 0xd7ff) {
		switch (address & 0x0003) {
			case 0:
				mapper90_mode = data | ((mapper90_211) ? 0x20 : 0x00);
				break;
			case 1: mapper90_mirror = data; break;
			case 2: mapper90_ppu = data; break;
			case 3:
				mapper90_obank = data;

				if (mapper90_209 && mapper90_obank & 0x80) {
					mapper_ppu_clock = mapper90_ppu_clockmmc4; // chr latching. enabled dynamically
				} else {
					mapper_ppu_clock = NULL;
				}
				break;
		}
	}

	mapper_map();
}

static void mapper90_psg_write(UINT16 address, UINT8 data)
{
	switch (address & 0xfc03) {
		case 0x5800: mapper90_mul0 = data; break;
		case 0x5801: mapper90_mul1 = data; break;
		case 0x5802: mapper90_accu += data; break;
		case 0x5803: mapper90_testreg = data; mapper90_accu = 0; break;
	}
}

static UINT8 mapper90_psg_read(UINT16 address)
{
	switch (address & 0xfc03) {
		case 0x5800: return ((mapper90_mul0 * mapper90_mul1) >> 0) & 0xff;
		case 0x5801: return ((mapper90_mul0 * mapper90_mul1) >> 8) & 0xff;
		case 0x5802: return mapper90_accu;
		case 0x5803: return mapper90_testreg;
	}

	switch (address & 0xffff) {
		case 0x5000: // jumper/DIP register
		case 0x5400:
		case 0x5c00:
			return 0x00;
	}

	return cpu_open_bus;
}

static void mapper90_clockirq()
{
	switch (mapper90_irqmode & 0xc0) {
		case 0x40:
			mapper90_irqcounter++;
			if ((mapper90_irqcounter == 0) && mapper90_irqenable) {
				//bprintf(0, _T("irq+ (mode %x) @ SL %d\n"), mapper90_irqmode, scanline);
				mapper_irq(2);
			}
			break;
		case 0x80:
			mapper90_irqcounter--;
			if ((mapper90_irqcounter == 0xff) && mapper90_irqenable) {
				//bprintf(0, _T("irq- (mode %x) @ SL %d\n"), mapper90_irqmode, scanline);
				mapper_irq(2); // 2 - "super mario world (unl)" HUD shaking
			}
			break;
	}
}

static void mapper90_clockpre()
{
	switch (mapper90_irqmode & 0xc0) {
		case 0x40:
			mapper90_irqprescaler++;
			if ((mapper90_irqprescaler & mapper90_irqprescalermask) == 0) {
				mapper90_clockirq();
			}
			break;
		case 0x80:
			mapper90_irqprescaler--;
			if ((mapper90_irqprescaler & mapper90_irqprescalermask) == mapper90_irqprescalermask) {
				mapper90_clockirq();
			}
			break;
	}
}

static void mapper90_ppu_clock(UINT16 address)
{
	// ppu clock mode
	if ((mapper90_irqmode & 3) == 2) {
		mapper90_clockpre();
		mapper90_clockpre();
	}
}

static void mapper90_ppu_clockmmc4(UINT16 address)
{
	switch (address & 0x3ff8) {
		case 0x0fd8:
			mapper90_chrlatch(0) = 0;
			mapper_map();
			break;
		case 0x0fe8:
			mapper90_chrlatch(0) = 2;
			mapper_map();
			break;
		case 0x1fd8:
			mapper90_chrlatch(1) = 4;
			mapper_map();
			break;
		case 0x1fe8:
			mapper90_chrlatch(1) = 6;
			mapper_map();
			break;
	}
}

static void mapper90_scanline()
{
	if ((mapper90_irqmode & 3) == 1 && (mmc5_mask[0] & 0x18) /* rendering? */) {
		for (INT32 i = 0; i < 8; i++)
			mapper90_clockpre();
	}
}

static void mapper90_cycle()
{
	if ((mapper90_irqmode & 3) == 0)
		mapper90_clockpre();
}

static UINT8 mapper90_exp_read(UINT16 address)
{
	return (mapper90_mode & 0x80) ? Cart.PRGRom[PRGExpMap + (address & 0x1fff)] : Cart.WorkRAM[address & 0x1fff];
}

static void mapper90_exp_write(UINT16 address, UINT8 data) // 6000 - 7fff
{
	if (mapper90_mode & 0x80) {
		// prg-rom mode, abort write to wram
		cart_exp_write_abort = 1; // don't fall-through after callback!
	}
}

static UINT8 mapper90_ntread(UINT16 address) // this only gets mapped for 209, 211!
{
	if (mapper90_mode & 0x20) {
		INT32 nt = (address & 0xfff) / 0x400;
		if (mapper90_mode & 0x40 || ((mapper90_ppu & 0x80) ^ (mapper90_nt(nt) & 0x80))) {
			return Cart.CHRRom[(mapper90_nt(nt)) * 0x400 + (address & 0x3ff)];
		}
	}

	return read_nt_int(address); // fall back to internal
}

static UINT16 mapper90_getchr(INT32 num)
{
	UINT16 bank = 0;
	UINT16 mask = 0xffff;

	if (~mapper90_obank & 0x20) { // outer-bank mode
		bank = (mapper90_obank & 1) | ((mapper90_obank >> 2) & 6);

		switch (mapper90_mode & 0x18) {
			case 0x00: bank <<= 5; mask = 0x1f; break;
			case 0x08: bank <<= 6; mask = 0x3f; break;
			case 0x10: bank <<= 7; mask = 0x7f; break;
			case 0x18: bank <<= 8; mask = 0xff; break;
		}
	}

	return ((mapper90_chrlo(num) | (mapper90_chrhi(num) << 8)) & mask) | bank;
}

static UINT8 mapper90_bitswap06(UINT8 data)
{
	return (data & 0x40) >> 6 | (data & 0x20) >> 4 | (data & 0x10) >> 2 | (data & 0x08) | (data & 0x04) << 2 | (data & 0x02) << 4 | (data & 0x01) << 6;
}

static void mapper90_map()
{
	// prg
	INT32 prg8_obank  = (mapper90_obank & 6) << 5;
	INT32 prg16_obank = (mapper90_obank & 6) << 4;
	INT32 prg32_obank = (mapper90_obank & 6) << 3;
	switch (mapper90_mode & 3) {
		case 0x00:
			mapper_map_prg(32, 0, prg32_obank | ((mapper90_mode & 0x04) ? (mapper90_prg(3) & 0xf) : 0xf));
			if (mapper90_mode & 0x80) {
				mapper_map_exp_prg(prg8_obank | (((mapper90_prg(3) << 2) + 3) & 0x3f));
			}
			break;
		case 0x01:
			//bprintf(0, _T("Mapper: JyCompany - 16k prg mode. *untested*\n"));
			mapper_map_prg(16, 0, prg16_obank | (mapper90_prg(1) & 0x1f));
			mapper_map_prg(16, 1, prg16_obank | ((mapper90_mode & 0x04) ? (mapper90_prg(3) & 0x1f) : 0x1f));
			if (mapper90_mode & 0x80) {
				mapper_map_exp_prg(prg8_obank | (((mapper90_prg(3) << 1) + 1) & 0x3f));
			}
			break;
		case 0x02:
			mapper_map_prg(8, 0, prg8_obank | (mapper90_prg(0) & 0x3f));
			mapper_map_prg(8, 1, prg8_obank | (mapper90_prg(1) & 0x3f));
			mapper_map_prg(8, 2, prg8_obank | (mapper90_prg(2) & 0x3f));
			mapper_map_prg(8, 3, prg8_obank | ((mapper90_mode & 0x04) ? (mapper90_prg(3) & 0x3f) : 0x3f));
			if (mapper90_mode & 0x80) {
				mapper_map_exp_prg(prg8_obank | (mapper90_prg(3) & 0x3f));
			}
			break;
		case 0x03: // same as 0x02, but with inverted bits
			//bprintf(0, _T("Mapper: JyCompany - inverted bits. *untested*\n"));
			mapper_map_prg(8, 0, prg8_obank | (mapper90_bitswap06(mapper90_prg(0)) & 0x3f));
			mapper_map_prg(8, 1, prg8_obank | (mapper90_bitswap06(mapper90_prg(1)) & 0x3f));
			mapper_map_prg(8, 2, prg8_obank | (mapper90_bitswap06(mapper90_prg(2)) & 0x3f));
			mapper_map_prg(8, 3, prg8_obank | ((mapper90_mode & 0x04) ? (mapper90_bitswap06(mapper90_prg(3)) & 0x3f) : 0x3f));
			if (mapper90_mode & 0x80) {
				mapper_map_exp_prg(prg8_obank | (mapper90_bitswap06(mapper90_prg(3)) & 0x3f));
			}
			break;
	}

	// chr
	switch (mapper90_mode & 0x18) {
		case 0x00:
			mapper_map_chr( 8, 0, mapper90_getchr(0));
			break;
		case 0x08:
			if (~mapper90_obank & 0x80) { // normal 4k banking
				mapper_map_chr( 4, 0, mapper90_getchr(0));
				mapper_map_chr( 4, 1, mapper90_getchr(4));
			} else {                      // mmc4-like latch 4k banking
				mapper_map_chr( 4, 0, mapper90_getchr(mapper90_chrlatch(0)));
				mapper_map_chr( 4, 1, mapper90_getchr(mapper90_chrlatch(1)));
			}
			break;
		case 0x10:
			mapper_map_chr( 2, 0, mapper90_getchr(0));
			mapper_map_chr( 2, 1, mapper90_getchr(2));
			mapper_map_chr( 2, 2, mapper90_getchr(4));
			mapper_map_chr( 2, 3, mapper90_getchr(6));
			break;
		case 0x18:
			mapper_map_chr( 1, 0, mapper90_getchr(0));
			mapper_map_chr( 1, 1, mapper90_getchr(1));
			mapper_map_chr( 1, 2, mapper90_getchr(2));
			mapper_map_chr( 1, 3, mapper90_getchr(3));
			mapper_map_chr( 1, 4, mapper90_getchr(4));
			mapper_map_chr( 1, 5, mapper90_getchr(5));
			mapper_map_chr( 1, 6, mapper90_getchr(6));
			mapper_map_chr( 1, 7, mapper90_getchr(7));
			break;
	}

	// nametable config - if rom nt's are selected, they will be fed via mapper90_ntread()
	// a RAM nt is always selected for writing, though. (re: Tiny Toon Adv. 6 intro)
	if (mapper90_209 && mapper90_mode & 0x20) {
		nametable_map(0, mapper90_nt(0) & 1);
		nametable_map(1, mapper90_nt(1) & 1);
		nametable_map(2, mapper90_nt(2) & 1);
		nametable_map(3, mapper90_nt(3) & 1);
	} else {
		//standard nt config
		switch (mapper90_mirror & 0x3) {
			case 0: set_mirroring(VERTICAL); break;
			case 1: set_mirroring(HORIZONTAL); break;
			case 2: set_mirroring(SINGLE_LOW); break;
			case 3: set_mirroring(SINGLE_HIGH); break;
		}
	}
}
#undef mapper90_irqenable
#undef mapper90_irqmode
#undef mapper90_irqcounter
#undef mapper90_irqprescaler
#undef mapper90_irqxor
#undef mapper90_irqprescalermask
#undef mapper90_irqunknown
#undef mapper90_mode
#undef mapper90_mirror
#undef mapper90_ppu
#undef mapper90_obank
#undef mapper90_prg
#undef mapper90_chrlo
#undef mapper90_chrhi
#undef mapper90_nt

// --[ mapper 91: older JyCompany/Hummer
#define mapper91_prg(x)		(mapper_regs[0x00 + (x)])
#define mapper91_chr(x)		(mapper_regs[0x04 + (x)])
#define mapper91_irqcount   (mapper_regs[0x1f - 0x00])
#define mapper91_irqenable	(mapper_regs[0x1f - 0x01])

static void mapper91_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x6000:
			mapper91_chr(address & 3) = data;
			break;
		case 0x7000:
			switch (address & 3) {
				case 0:
				case 1:
					mapper91_prg(address & 1) = data;
					break;
				case 2:
					mapper91_irqenable = 0;
					mapper91_irqcount = 0;
					M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
					break;
				case 3:
					mapper91_irqenable = 1;
					break;
			}
			break;
	}

	mapper_map();
}

static void mapper91_scanline()
{
	if (mapper91_irqenable && (mmc5_mask[0] & 0x18) /* rendering? */) {
		mapper91_irqcount++;
		if (mapper91_irqcount == 8) {
			mapper_irq(0x14); // 0x14 - gets rid of artefacts in "dragon ball z - super butouden 2"
			mapper91_irqenable = 0;
		}
	}
}

static void mapper91_map()
{
	mapper_map_prg( 8, 0, mapper91_prg(0));
	mapper_map_prg( 8, 1, mapper91_prg(1));
	mapper_map_prg( 8, 2, -2);
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 2, 0, mapper91_chr(0));
	mapper_map_chr( 2, 1, mapper91_chr(1));
	mapper_map_chr( 2, 2, mapper91_chr(2));
	mapper_map_chr( 2, 3, mapper91_chr(3));
}
#undef mapper91_prg
#undef mapper91_chr
#undef mapper91_irqcount
#undef mapper91_irqenable

// --[ mapper 17: FFE / Front Far East SMC (type 17)
#define mapper17_prg(x)		(mapper_regs[0x00 + (x)])
#define mapper17_chr(x)		(mapper_regs[0x04 + (x)])
#define mapper17_irqcount   (mapper_regs16[0x00])
#define mapper17_irqenable	(mapper_regs[0x1f - 0x00])
#define mapper17_mirror		(mapper_regs[0x1f - 0x01])

static void mapper17_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0x42fe:
			switch (data & 0x10) {
				case 0x00: mapper17_mirror = 2; break;
				case 0x10: mapper17_mirror = 3; break;
			}
			break;
		case 0x42ff:
			switch (data & 0x10) {
				case 0x00: mapper17_mirror = 0; break;
				case 0x10: mapper17_mirror = 1; break;
			}
			break;

		case 0x4501:
			mapper17_irqenable = 0;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0x4502:
			mapper17_irqcount = (mapper17_irqcount & 0xff00) | data;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0x4503:
			mapper17_irqcount = (mapper17_irqcount & 0x00ff) | (data << 8);
			mapper17_irqenable = 1;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;

		case 0x4504:
		case 0x4505:
		case 0x4506:
		case 0x4507:
			mapper17_prg(address & 3) = data;
			break;

		case 0x4510:
		case 0x4511:
		case 0x4512:
		case 0x4513:
		case 0x4514:
		case 0x4515:
		case 0x4516:
		case 0x4517:
			mapper17_chr(address & 7) = data;
			break;
	}

	mapper_map();
}

static void mapper17_cycle()
{
	if (mapper17_irqenable) {
		mapper17_irqcount++;
		if (mapper17_irqcount == 0x0000) {
			mapper_irq(0);
			mapper17_irqenable = 0;
		}
	}
}

static void mapper17_map()
{
	mapper_map_prg( 8, 0, mapper17_prg(0));
	mapper_map_prg( 8, 1, mapper17_prg(1));
	mapper_map_prg( 8, 2, mapper17_prg(2));
	mapper_map_prg( 8, 3, mapper17_prg(3));

	mapper_map_chr( 1, 0, mapper17_chr(0));
	mapper_map_chr( 1, 1, mapper17_chr(1));
	mapper_map_chr( 1, 2, mapper17_chr(2));
	mapper_map_chr( 1, 3, mapper17_chr(3));
	mapper_map_chr( 1, 4, mapper17_chr(4));
	mapper_map_chr( 1, 5, mapper17_chr(5));
	mapper_map_chr( 1, 6, mapper17_chr(6));
	mapper_map_chr( 1, 7, mapper17_chr(7));

	switch (mapper17_mirror & 0x3) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}
#undef mapper17_prg
#undef mapper17_chr
#undef mapper17_irqcount
#undef mapper17_irqenable
#undef mapper17_mirror

// --[ mapper 28: Action53 Home-brew multicart
#define mapper28_mirror		(mapper_regs[0x1f - 0])
#define mapper28_mirrorbit  (mapper_regs[0x1f - 1])
#define mapper28_cmd		(mapper_regs[0x1f - 2])
#define mapper28_regs(x)	(mapper_regs[(x)])

static void mapper28_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5000 && address <= 0x5fff) {
		mapper28_cmd = ((data >> 6) & 2) | (data & 1);
	} else if (address >= 0x8000) {
		switch (mapper28_cmd) {
			case 0:
			case 1:
				mapper28_mirrorbit = (data >> 4) & 1;
				break;
			case 2:
				mapper28_mirrorbit = data & 1;
				break;
		}
		mapper28_regs(mapper28_cmd & 3) = data;

		mapper_map();
	}
}

static void mapper28_map()
{
	UINT8 slot = (mapper28_regs(2) & 0x04) >> 2;
	UINT8 outerbanksize = (mapper28_regs(2) & 0x30) >> 4;
	UINT8 outerbank = mapper28_regs(3) << 1;
	UINT8 prgbank = (mapper28_regs(1) & 0xf) << ((~mapper28_regs(2) & 0x08) >> 3);
	UINT8 page = (outerbank & ~((1 << (outerbanksize+1))-1)) | (prgbank & ((1 << (outerbanksize+1))-1));

	if (mapper28_regs(2) & 0x08) {
		mapper_map_prg(16, slot ^ 1, page);
		mapper_map_prg(16, slot    , (outerbank & 0x1fe) | slot);
	} else {
		mapper_map_prg(16, 0, page);
		mapper_map_prg(16, 1, page | 1);
	}

	mapper_map_chr( 8, 0, mapper28_regs(0) & 0x03);

	UINT8 mirror = mapper28_regs(2) & 0x03;
	if (~mirror & 0x02) {
		mirror = mapper28_mirrorbit;
	}

	switch (mirror) {
		case 0: set_mirroring(SINGLE_LOW); break;
		case 1: set_mirroring(SINGLE_HIGH); break;
		case 2: set_mirroring(VERTICAL); break;
		case 3: set_mirroring(HORIZONTAL); break;
	}
}
#undef mapper28_mirror
#undef mapper28_mirrorbit
#undef mapper28_cmd
#undef mapper28_regs

// --[ mapper 33, 48: taito
#define mapper33_prg(x)		(mapper_regs[0 + x])
#define mapper33_chr(x)		(mapper_regs[2 + x])
#define mapper33_mirror		(mapper_regs[0x1f - 0])
#define mapper33_irqenable	(mapper_regs[0x1f - 1])
#define mapper48            (mapper_regs[0x1f - 2])
#define mapper33_irqcount	(mapper_regs16[0x1f - 0])
#define mapper33_irqlatch	(mapper_regs16[0x1f - 1])
#define mapper33_irqreload  (mapper_regs16[0x1f - 2])
#define mapper48_flintstones (mapper_regs16[0x1f - 3])

static void mapper33_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf003) {
		case 0x8000:
			mapper33_prg(0) = data & 0x3f;
			if (mapper48 == 0) mapper33_mirror = data & 0x40;
			break;
		case 0x8001: mapper33_prg(1) = data & 0x3f; break;

		case 0x8002: mapper33_chr(0) = data; break;
		case 0x8003: mapper33_chr(1) = data; break;
		case 0xa000: mapper33_chr(2) = data; break;
		case 0xa001: mapper33_chr(3) = data; break;
		case 0xa002: mapper33_chr(4) = data; break;
		case 0xa003: mapper33_chr(5) = data; break;
	}

	if (mapper48) {
		switch (address & 0xf003) {
			case 0xc000: mapper33_irqlatch = (data ^ 0xff) + ((mapper48_flintstones) ? 0 : 1); M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
			case 0xc001: mapper33_irqreload = 1; mapper33_irqcount = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
			case 0xc002: mapper33_irqenable = 1; break;
			case 0xc003: mapper33_irqenable = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
			case 0xe000: if (mapper48) mapper33_mirror = data & 0x40; break;
		}
	}

	mapper_map();
}

static void mapper33_map()
{
	mapper_map_prg( 8, 0, mapper33_prg(0));
	mapper_map_prg( 8, 1, mapper33_prg(1));
	mapper_map_prg( 8, 2, -2);
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 2, 0, mapper33_chr(0));
	mapper_map_chr( 2, 1, mapper33_chr(1));

	mapper_map_chr( 1, 4, mapper33_chr(2));
	mapper_map_chr( 1, 5, mapper33_chr(3));
	mapper_map_chr( 1, 6, mapper33_chr(4));
	mapper_map_chr( 1, 7, mapper33_chr(5));

	set_mirroring((mapper33_mirror) ? HORIZONTAL : VERTICAL);
}

static void mapper33_scanline()
{
	INT32 cnt = mapper33_irqenable;
	if (mapper33_irqcount == 0 || mapper33_irqreload) {
		mapper33_irqreload = 0;
		mapper33_irqcount = mapper33_irqlatch;
	} else {
		mapper33_irqcount--;
	}

	if (cnt && mapper33_irqenable && mapper33_irqcount == 0) {
		mapper_irq((mapper48_flintstones) ? 0x13 : 0x06);
	}
}
#undef mapper33_mirror
#undef mapper33_irqenable
#undef mapper33_prg
#undef mapper33_chr
#undef mapper33_irqcount
#undef mapper33_irqlatch
#undef mapper33_irqreload

// --[ mapper 36 - TXC (Policeman, Strike Wolf)
#define mapper36_prg		(mapper_regs[0x1f - 0])
#define mapper36_chr		(mapper_regs[0x1f - 1])
#define mapper36_RR			(mapper_regs[0x1f - 2])
#define mapper36_PP			(mapper_regs[0x1f - 3])
#define mapper36_invert		(mapper_regs[0x1f - 4])
#define mapper36_mode		(mapper_regs[0x1f - 5])

static UINT8 mapper36_read(UINT16 address)
{
	if ((address & 0xe100) == 0x4100)
		return (cpu_open_bus & 0xcf) | (mapper36_RR << 4);
	else
		return cpu_open_bus;
}

static void mapper36_write(UINT16 address, UINT8 data)
{
	if (address > 0x7fff) {
		mapper36_prg = mapper36_RR;
	} else {
		switch (address & 0xe103) {
			case 0x4100:
				if (mapper36_mode) {
					mapper36_RR++;
				} else {
					mapper36_RR = mapper36_PP;
				}
				break;
			case 0x4101:
				mapper36_invert = data & 0x10;
				break;
			case 0x4102:
				mapper36_PP = (data >> 4) & 0x3;
				break;
			case 0x4103:
				mapper36_mode = data & 0x10;
				break;
		}
		switch (address & 0xe200) {
			case 0x4200:
				mapper36_chr = data & 0xf;
				break;
		}
	}

	mapper_map();
}

static void mapper36_map()
{
	mapper_map_prg(32, 0, mapper36_prg);

	mapper_map_chr(8, 0, mapper36_chr);
}

#undef mapper36_prg
#undef mapper36_chr
#undef mapper36_RR
#undef mapper36_mode

// --[ mapper 42 Ai Senshi Nicol FDS conversion
#define mapper42_mirror		(mapper_regs[0x1f - 0])
#define mapper42_prg		(mapper_regs[0x1f - 1])
#define mapper42_chr		(mapper_regs[0x1f - 2])
#define mapper42_irqenable	(mapper_regs[0x1f - 3])
#define mapper42_irqcount   (mapper_regs16[0])

static void mapper42_write(UINT16 address, UINT8 data)
{
	switch (address & 0xe003) {
		case 0x8000: mapper42_chr = data; break;
		case 0xe000: mapper42_prg = data & 0xf; break;
		case 0xe001: mapper42_mirror = data; break;
		case 0xe002:
			mapper42_irqenable = data & 2;
			if (!mapper42_irqenable) {
				mapper42_irqcount = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			}
			break;
	}

	mapper_map();
}

static void mapper42_map()
{
	// set prg-rom @ 6000 - 7fff, fed by mapper42_exp_read()
	mapper_map_exp_prg(mapper42_prg);
	// normal prg-rom @ 8000 - ffff
	mapper_map_prg(32, 0, -1);

	mapper_map_chr(8, 0, mapper42_chr);

	set_mirroring((mapper42_mirror & 0x8) ? HORIZONTAL : VERTICAL);
}

static UINT8 mapper42_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

static void mapper42_cycle()
{
	if (mapper42_irqenable) {
		mapper42_irqcount++;
		if (mapper42_irqcount >= 0x8000) {
			mapper42_irqcount -= 0x8000;
		}
		if (mapper42_irqcount >= 0x6000) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
		} else {
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
		}
	}
}

#undef mapper42_mirror
#undef mapper42_prg
#undef mapper42_chr
#undef mapper42_irqenable
#undef mapper42_irqcount

// --[ mapper 108 Meikyuu Jiin Dababa (Japan) FDS conversion
#define mapper108_prg		(mapper_regs[0x1f - 0])

static void mapper108_write(UINT16 address, UINT8 data)
{
	if ((address >= 0x8000 && address <= 0x8fff) || (address >= 0xf000 && address <= 0xffff)) {
		mapper108_prg = data;

		mapper_map();
	}
}

static void mapper108_map()
{
	// set prg-rom @ 6000 - 7fff, fed by mapper108_exp_read()
	mapper_map_exp_prg(mapper108_prg);
	// normal prg-rom @ 8000 - ffff
	mapper_map_prg(32, 0, -1);

	mapper_map_chr(8, 0, 0);
}

static UINT8 mapper108_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

#undef mapper108_prg

// --[ mapper 111 Cheapocabra (GTROM)
#define mapper111_reg		(mapper_regs[0x1f - 0])

static void mapper111_write(UINT16 address, UINT8 data)
{
	if ((address >= 0x5000 && address <= 0x5fff) || (address >= 0x7000 && address <= 0x7fff)) {
		mapper111_reg = data;
		mapper_map();
	}
}

static void mapper111_map()
{
	mapper_map_prg(32, 0, mapper111_reg & 0xf);
	mapper_map_chr(8, 0, (mapper111_reg >> 4) & 0x1);

	INT32 nt = (mapper111_reg & 0x20) ? (0x4000 + 0x2000) : (0x4000 + 0x0000);

	for (INT32 i = 0; i < 4; i++) {
		nametable_mapraw(i & 3, Cart.CHRRam + nt + (i * 0x400), MEM_RAM);
	}
}

#undef mapper111_reg

// --[ mapper 120 Tobidase Daisakusen (Japan) FDS Conv. *game is buggy*
#define mapper120_prg		(mapper_regs[0x1f - 0])

static void mapper120_write(UINT16 address, UINT8 data)
{
	if (address == 0x41ff) {
		mapper120_prg = data & 0x7;
	}

	mapper_map();
}

static void mapper120_map()
{
	// set prg-rom @ 6000 - 7fff, fed by mapper120_exp_read()
	mapper_map_exp_prg(mapper120_prg);
	// normal prg-rom @ 8000 - ffff
	mapper_map_prg(32, 0, 2);

	mapper_map_chr(8, 0, 0);
}

static UINT8 mapper120_exp_read(UINT16 address)
{
	return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
}

#undef mapper120_prg

// --[ mapper 64 - Tengen (Atari)
#define mapper64_mirror			(mapper_regs[0x1f - 0])
#define mapper64_regnum			(mapper_regs[0x1f - 1])
#define mapper64_reload			(mapper_regs[0x1f - 2])
#define mapper64_irqmode		(mapper_regs[0x1f - 3])
#define mapper64_irqenable		(mapper_regs[0x1f - 4])
#define mapper64_irqlatch		(mapper_regs[0x1f - 5])
#define mapper64_irqcount		(mapper_regs[0x1f - 6])
#define mapper64_irqprescale	(mapper_regs[0x1f - 7])
#define mapper64_cycles         (mapper_regs16[0])

static void mapper64_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
        switch (address & 0xe001) {
			case 0xA000: mapper64_mirror = data & 1; mapper_map(); break;
            case 0x8000: mapper64_regnum = data; break;
			case 0x8001:
				mapper_regs[(mapper64_regnum & 0xf)] = data;
				mapper_map();
				break;

			case 0xC000:
				mapper64_irqlatch = data;
				break;
			case 0xC001:
				mapper64_reload = 1;
				mapper64_irqprescale = 0;
				mapper64_irqmode = data & 1;
				break;
			case 0xE000:
				mapper64_irqenable = 0;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
			case 0xE001:
				mapper64_irqenable = 1;
				M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
				break;
        }
	}
}

static void mapper64_map()
{
	if (mapper64_regnum & 0x20) {
		mapper_map_chr(1, 0, mapper_regs[0]);
		mapper_map_chr(1, 1, mapper_regs[8]);
		mapper_map_chr(1, 2, mapper_regs[1]);
		mapper_map_chr(1, 3, mapper_regs[9]);
	} else {
		mapper_map_chr(2, 0, mapper_regs[0] >> 1);
		mapper_map_chr(2, 1, mapper_regs[1] >> 1);
	}
	mapper_map_chr(1, 4, mapper_regs[2]);
	mapper_map_chr(1, 5, mapper_regs[3]);
	mapper_map_chr(1, 6, mapper_regs[4]);
	mapper_map_chr(1, 7, mapper_regs[5]);

	mapper_map_prg(8, 0, mapper_regs[6]);
	mapper_map_prg(8, 1, mapper_regs[7]);
	mapper_map_prg(8, 2, mapper_regs[0xf]);
	mapper_map_prg(8, 3, -1);
	set_mirroring(mapper64_mirror ? HORIZONTAL : VERTICAL);
}

static void mapper64_irq_reload_logic()
{
	if (mapper64_reload) {
		mapper64_irqcount = (mapper64_irqlatch) ? mapper64_irqlatch | 1 : 0;
		if (mapper64_irqcount == 0 && mapper64_cycles > 0x10)
			mapper64_irqcount = 1;
		mapper64_reload = 0;
		mapper64_cycles = 0;
	} else if (mapper64_irqcount == 0) {
		mapper64_irqcount = mapper64_irqlatch;
		if (mapper64_cycles > 0x10)
			mapper64_cycles = 0;
	} else mapper64_irqcount --;
}

static void mapper64_scanline()
{
	if (mmc5_mask[0] & 0x18) { // aka if (RENDERING) ...
		if (mapper64_irqmode == 0) {

			mapper64_irq_reload_logic();

			if (mapper64_irqcount == 0 && mapper64_irqenable) {
				mapper_irq(1); // assert irq in 1 m2 cycle
			}
		}
	}
}

static void mapper64_cycle()
{
	if (mapper64_cycles == 0xffff) mapper64_cycles = 0x10;
	mapper64_cycles++;
	if (mapper64_irqmode == 1) {
		mapper64_irqprescale++;
		while (mapper64_irqprescale == 4) {
			mapper64_irqprescale = 0;

			mapper64_irq_reload_logic();

			if (mapper64_irqcount == 0 && mapper64_irqenable) {
				mapper_irq(4); // assert irq in 4 m2 cycles
			}
		}
	}
}

#undef mapper64_mirror
#undef mapper64_regnum
#undef mapper64_reload
#undef mapper64_irqmode
#undef mapper64_irqenable
#undef mapper64_irqlatch
#undef mapper64_irqcount
#undef mapper64_irqprescale

// --[ mapper 65 - irem h3001(?): Spartan X2, Kaiketsu Yanchamaru 3: Taiketsu! Zouringen,
#define mapper65_mirror		(mapper_regs[0x1f - 0])
#define mapper65_irqenable	(mapper_regs[0x1f - 1])
#define mapper65_prg(x)		(mapper_regs[0 + x])
#define mapper65_chr(x)		(mapper_regs[3 + x])
#define mapper65_irqcount	(mapper_regs16[0x1f - 0])
#define mapper65_irqlatch	(mapper_regs16[0x1f - 1])


static void mapper65_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0x8000: mapper65_prg(0) = data; break;
		case 0xa000: mapper65_prg(1) = data; break;
		case 0xc000: mapper65_prg(2) = data; break;
		case 0x9001: mapper65_mirror = (~data >> 7) & 1; break;
		case 0x9003: mapper65_irqenable = data & 0x80; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
		case 0x9004: mapper65_irqcount = mapper65_irqlatch; break;
		case 0x9005: mapper65_irqlatch = (mapper65_irqlatch & 0x00ff) | (data << 8); break;
		case 0x9006: mapper65_irqlatch = (mapper65_irqlatch & 0xff00) | data; break;
		case 0xb000: mapper65_chr(0) = data; break;
		case 0xb001: mapper65_chr(1) = data; break;
		case 0xb002: mapper65_chr(2) = data; break;
		case 0xb003: mapper65_chr(3) = data; break;
		case 0xb004: mapper65_chr(4) = data; break;
		case 0xb005: mapper65_chr(5) = data; break;
		case 0xb006: mapper65_chr(6) = data; break;
		case 0xb007: mapper65_chr(7) = data; break;
	}
	mapper_map();
}

static void mapper65_map()
{
	mapper_map_prg( 8, 0, mapper65_prg(0));
	mapper_map_prg( 8, 1, mapper65_prg(1));
	mapper_map_prg( 8, 2, mapper65_prg(2));
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 1, 0, mapper65_chr(0));
	mapper_map_chr( 1, 1, mapper65_chr(1));
	mapper_map_chr( 1, 2, mapper65_chr(2));
	mapper_map_chr( 1, 3, mapper65_chr(3));
	mapper_map_chr( 1, 4, mapper65_chr(4));
	mapper_map_chr( 1, 5, mapper65_chr(5));
	mapper_map_chr( 1, 6, mapper65_chr(6));
	mapper_map_chr( 1, 7, mapper65_chr(7));

	set_mirroring(mapper65_mirror ? VERTICAL : HORIZONTAL);
}

static void mapper65_cycle()
{
	if (mapper65_irqenable) {
		mapper65_irqcount--;

		if (mapper65_irqcount == 0) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper65_irqenable = 0;
		}
	}
}
#undef mapper65_mirror
#undef mapper65_irqenable
#undef mapper65_prg
#undef mapper65_chr
#undef mapper65_irqcount
#undef mapper65_irqlatch


// --[ mapper 67 - Sunsoft-3 (Fantasy Zone II, Mito Koumon II: Sekai Manyuu Ki)
#define mapper67_mirror		(mapper_regs[0x1f - 0])
#define mapper67_irqenable	(mapper_regs[0x1f - 1])
#define mapper67_irqtoggle  (mapper_regs[0x1f - 2])
#define mapper67_prg		(mapper_regs[0])
#define mapper67_chr(x)		(mapper_regs[1 + x])
#define mapper67_irqcount	(mapper_regs16[0x1f - 0])

static void mapper67_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf800) {
		case 0x8800: mapper67_chr(0) = data; break;
		case 0x9800: mapper67_chr(1) = data; break;
		case 0xa800: mapper67_chr(2) = data; break;
		case 0xb800: mapper67_chr(3) = data; break;
		case 0xc000:
		case 0xc800:
			if (mapper67_irqtoggle == 0) {
				mapper67_irqcount = (mapper67_irqcount & 0x00ff) | (data << 8);
			} else {
				mapper67_irqcount = (mapper67_irqcount & 0xff00) | (data << 0);
			}
			mapper67_irqtoggle ^= 1;
			break;
		case 0xd800:
			mapper67_irqtoggle = 0;
			mapper67_irqenable = data & 0x10;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xe800:
			mapper67_mirror = data & 3;
			break;
		case 0xf800:
			mapper67_prg = data;
			break;
	}
	mapper_map();
}

static void mapper67_map()
{
	mapper_map_prg(16, 0, mapper67_prg);
	mapper_map_prg(16, 1, -1);

	mapper_map_chr( 2, 0, mapper67_chr(0));
	mapper_map_chr( 2, 1, mapper67_chr(1));
	mapper_map_chr( 2, 2, mapper67_chr(2));
	mapper_map_chr( 2, 3, mapper67_chr(3));

	switch (mapper67_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper67_cycle()
{
	if (mapper67_irqenable) {
		mapper67_irqcount--;

		if (mapper67_irqcount == 0xffff) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper67_irqenable = 0;
		}
	}
}
#undef mapper67_mirror
#undef mapper67_irqenable
#undef mapper67_irqtoggle
#undef mapper67_prg
#undef mapper67_chr
#undef mapper67_irqcount

// --[ mapper 68 - Sunsoft-4 (After Burner 1 & 2, Nantettatte!! Baseball, Maharaja)
// Notes: Nantettatte!! Baseball ext.roms not supported yet.
#define mapper68_mirror		(mapper_regs[0x1f - 0])
#define mapper68_prg		(mapper_regs[0x1f - 1])
#define mapper68_wram_en    (mapper_regs[0x1f - 2])
#define mapper68_nt0		(mapper_regs[0x1f - 3])
#define mapper68_nt1		(mapper_regs[0x1f - 4])
#define mapper68_chr(x)		(mapper_regs[0 + (x)])

static INT32 *mapper68_timer = (INT32*)&mapper_regs16[0];

static UINT8 mapper68_exp_read(UINT16 address) // 6000-7fff
{
	if (mapper68_wram_en) {
		return Cart.WorkRAM[address & 0x1fff];
	} else {
		return cpu_open_bus;
	}
}

static void mapper68_exp_write(UINT16 address, UINT8 data) // 6000-7fff
{
	if (mapper68_wram_en) {
		mapper68_timer[0] = 107520;
	} else {
		cart_exp_write_abort = 1; // wram/sram area disabled (disable writing on return from this callback)
	}
}

static void mapper68_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper68_chr(0) = data; break;
		case 0x9000: mapper68_chr(1) = data; break;
		case 0xa000: mapper68_chr(2) = data; break;
		case 0xb000: mapper68_chr(3) = data; break;
		case 0xc000: mapper68_nt0 = data | 0x80; break;
		case 0xd000: mapper68_nt1 = data | 0x80; break;
		case 0xe000: mapper68_mirror = data; break;
		case 0xf000:
			mapper68_prg = data & 7;
			mapper68_wram_en = data & 0x10;
			break;
	}
	mapper_map();
}

static void mapper68_map()
{
	mapper_map_prg(16, 0, mapper68_prg & 7);
	mapper_map_prg(16, 1, -1);

	mapper_map_chr( 2, 0, mapper68_chr(0));
	mapper_map_chr( 2, 1, mapper68_chr(1));
	mapper_map_chr( 2, 2, mapper68_chr(2));
	mapper_map_chr( 2, 3, mapper68_chr(3));

	if (mapper68_mirror & 0x10) {
		switch (mapper68_mirror & 3) {
			case 0:
				nametable_mapraw(0, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(2, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(1, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(3, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				break;
			case 1:
				nametable_mapraw(0, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(1, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(2, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(3, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				break;
			case 2:
				nametable_mapraw(0, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(1, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(2, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				nametable_mapraw(3, Cart.CHRRom + (mapper68_nt0 << 10), MEM_ROM);
				break;
			case 3:
				nametable_mapraw(0, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(1, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(2, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				nametable_mapraw(3, Cart.CHRRom + (mapper68_nt1 << 10), MEM_ROM);
				break;
		}

	} else {
		switch (mapper68_mirror & 3) {
			case 0: set_mirroring(VERTICAL); break;
			case 1: set_mirroring(HORIZONTAL); break;
			case 2: set_mirroring(SINGLE_LOW); break;
			case 3: set_mirroring(SINGLE_HIGH); break;
		}
	}
}

static void mapper68_cycle()
{
	if (mapper68_timer[0] > 0) {
		mapper68_timer[0]--;

		if (mapper68_timer[0] == 0) {
		}
	}
}

#undef mapper68_mirror
#undef mapper68_prg
#undef mapper68_chr
#undef mapper68_wram_en
#undef mapper68_nt0
#undef mapper68_nt1

// --[ mapper 69 (sunsoft fme-7: Gimmick!, Hebereke, Pyokotan)
#define mapper69_mirror		(mapper_regs[0x1f - 0])
#define mapper69_regnum		(mapper_regs[0x1f - 1])
#define mapper69_irqenable	(mapper_regs[0x1f - 2])
#define mapper69_irqcount   (mapper_regs16[0])

static void mapper69_write(UINT16 address, UINT8 data)
{
	switch (address & 0xe000) {
		case 0x8000: mapper69_regnum = data & 0xf; break;
		case 0xa000:
			switch (mapper69_regnum) {
				case 0x0:
				case 0x1:
				case 0x2:
				case 0x3:
				case 0x4:
				case 0x5:
				case 0x6:
				case 0x7: mapper_regs[(mapper69_regnum & 0xf)] = data; break;
				case 0x8: mapper_regs[0x8 + 3] = data; break;
				case 0x9: mapper_regs[0x8 + 0] = data; break;
				case 0xa: mapper_regs[0x8 + 1] = data; break;
				case 0xb: mapper_regs[0x8 + 2] = data; break;
				case 0xc: mapper69_mirror = data & 3; break;
				case 0xd: mapper69_irqenable = data; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
				case 0xe: mapper69_irqcount = (mapper69_irqcount & 0xff00) | data; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
				case 0xf: mapper69_irqcount = (mapper69_irqcount & 0x00ff) | (data << 8); M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
			}
			if (mapper69_regnum < 0xd)
				mapper_map();
			break;

		case 0xc000:
			AY8910Write(0, 0, data);
			break;
		case 0xe000:
			AY8910Write(0, 1, data);
			break;
	}
}

static void mapper69_map()
{
	if ((mapper_regs[8 + 3] & 0xc0) == 0xc0) {
		// set prg-ram @ 6000 - 7fff
		// handled in mapper69_exp_write/read
	} else {
		// set prg-rom @ 6000 - 7fff
		mapper_map_exp_prg(mapper_regs[8 + 3] & 0x3f);
	}

	mapper_map_prg( 8, 0, mapper_regs[8 + 0]);
	mapper_map_prg( 8, 1, mapper_regs[8 + 1]);
	mapper_map_prg( 8, 2, mapper_regs[8 + 2]);
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr(1, 0, mapper_regs[0]);
	mapper_map_chr(1, 1, mapper_regs[1]);
	mapper_map_chr(1, 2, mapper_regs[2]);
	mapper_map_chr(1, 3, mapper_regs[3]);
	mapper_map_chr(1, 4, mapper_regs[4]);
	mapper_map_chr(1, 5, mapper_regs[5]);
	mapper_map_chr(1, 6, mapper_regs[6]);
	mapper_map_chr(1, 7, mapper_regs[7]);

	switch (mapper69_mirror) {
        case 0: set_mirroring(VERTICAL); break;
        case 1: set_mirroring(HORIZONTAL); break;
        case 2: set_mirroring(SINGLE_LOW); break;
        case 3: set_mirroring(SINGLE_HIGH); break;
	}
}

static void mapper69_exp_write(UINT16 address, UINT8 data)
{
	if ((mapper_regs[8 + 3] & 0xc0) == 0xc0) {
		Cart.WorkRAM[address & Cart.WorkRAMMask] = data;
	}
	cart_exp_write_abort = 1;
}

static UINT8 mapper69_exp_read(UINT16 address)
{
	if ((mapper_regs[8 + 3] & 0xc0) == 0x40) { // ram selected, but disabled
		return cpu_open_bus;
	} if ((mapper_regs[8 + 3] & 0xc0) == 0xc0) { // ram selected and enabled
		return Cart.WorkRAM[address & Cart.WorkRAMMask];
	} else { // rom selected
		return Cart.PRGRom[PRGExpMap + (address & 0x1fff)];
	}
}

static void mapper69_cycle()
{
	if (mapper69_irqenable) {
		mapper69_irqcount--;
		if (mapper69_irqcount == 0xffff) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
			mapper69_irqenable = 0;
		}
	}
}

#undef mapper69_mirror
#undef mapper69_regnum
#undef mapper69_irqenable

// --[ mapper 70: (Kamen Rider Club)
static void mapper70_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper70_map()
{
	mapper_map_prg(16, 0, mapper_regs[0] >> 4);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, mapper_regs[0] & 0xf);
}

// --[ mapper 71: Camerica (Bee 52, Big Nose, Fire Hawk, Micro Machines, ...)
static void mapper71_write(UINT16 address, UINT8 data)
{
	if ((address & 0xf000) == 0x9000) {
		mapper_regs[1] = ((data & 0x10) ? SINGLE_HIGH : SINGLE_LOW);
	} else {
		mapper_regs[0] = data;
	}

	mapper_map();
}

static void mapper71_map()
{
	mapper_map_prg(16, 0, mapper_regs[0]);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, 0);

	if (mapper_regs[1]) {
		// only fire hawk sets this, everything else uses hard-mirroring
		set_mirroring(mapper_regs[1]);
	}
}

// --[ mapper 73 - Konami vrc3 (Salamander)
// note: the flickery pixels on the top-right of HUD also happens on FC HW -dink
#define mapper73_irqenable	(mapper_regs[0x1f - 1])
#define mapper73_irqmode	(mapper_regs[0x1f - 2])
#define mapper73_irqreload  (mapper_regs[0x1f - 3])
#define mapper73_prg		(mapper_regs[0])
#define mapper73_irqcount	(mapper_regs16[0x1f - 0])
#define mapper73_irqlatch	(mapper_regs16[0x1f - 1])

static void mapper73_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper73_irqlatch = (mapper73_irqlatch & 0xfff0) | (data & 0xf); break;
		case 0x9000: mapper73_irqlatch = (mapper73_irqlatch & 0xff0f) | ((data & 0xf) << 4); break;
		case 0xa000: mapper73_irqlatch = (mapper73_irqlatch & 0xf0ff) | ((data & 0xf) << 8); break;
		case 0xb000: mapper73_irqlatch = (mapper73_irqlatch & 0x0fff) | ((data & 0xf) << 12); break;
		case 0xc000:
			mapper73_irqreload = data & 1;
			mapper73_irqenable = data & 2;
			mapper73_irqmode = data & 4;

			if (mapper73_irqenable) {
				if (mapper73_irqmode) {
					// when is this mode used? I can't make it happen -dink
					mapper73_irqcount = mapper73_irqlatch & 0xff;
				} else {
					mapper73_irqcount = mapper73_irqlatch;
				}
			}
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xd000:
			mapper73_irqenable = mapper73_irqreload;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf000:
			mapper73_prg = data;
			break;
	}
	mapper_map();
}

static void mapper73_map()
{
	mapper_map_prg(16, 0, mapper73_prg);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, 0);
}

static void mapper73_cycle()
{
	if (mapper73_irqenable) {
		if ( (mapper73_irqmode && (mapper73_irqcount & 0xff) == 0xff) ||
			((mapper73_irqmode == 0) && mapper73_irqcount == 0xffff) ) {
			mapper73_irqcount = mapper73_irqlatch;
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
		} else {
			mapper73_irqcount++;
		}
	}
}
#undef mapper73_irqenable
#undef mapper73_irqmode
#undef mapper73_irqreload
#undef mapper73_prg
#undef mapper73_irqcount
#undef mapper73_irqlatch

// --[ mapper 75: Konami VRC1 (King Kong 2, Tetsuwan Atom, ...)
#define mapper75_prg(x)     (mapper_regs[0 + x]) // 0 - 2
#define mapper75_chr(x)     (mapper_regs[3 + x]) // 3 - 4
#define mapper75_ext        (mapper_regs[8])
static void mapper75_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000: mapper75_prg(0) = data; break;
		case 0x9000: mapper75_ext = data; break;
		case 0xa000: mapper75_prg(1) = data; break;
		case 0xc000: mapper75_prg(2) = data; break;
		case 0xe000: mapper75_chr(0) = data; break;
		case 0xf000: mapper75_chr(1) = data; break;
	}

	mapper_map();
}

static void mapper75_map()
{
	mapper_map_prg( 8, 0, mapper75_prg(0));
	mapper_map_prg( 8, 1, mapper75_prg(1));
	mapper_map_prg( 8, 2, mapper75_prg(2));
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 4, 0, mapper75_chr(0) | ((mapper75_ext & 2) << 3));
	mapper_map_chr( 4, 1, mapper75_chr(1) | ((mapper75_ext & 4) << 2));

	if (Cart.Mapper != 4) {
		set_mirroring((mapper75_ext & 1) ? HORIZONTAL : VERTICAL);
	}
}
#undef mapper75_prg
#undef mapper75_chr
#undef mapper75_ext

// --[ mapper 85 (VRC7) Lagrange Point, Tiny Toon Adventures 2 (Japan)
#define mapper85_prg(x)		(mapper_regs[0x00 + x])
#define mapper85_chr(x)		(mapper_regs[0x03 + x])
#define mapper85_mirror		(mapper_regs[0x10])
#define mapper85_wramen     (mapper_regs[0x11])
#define mapper85_irqenable	(mapper_regs[0x12])
#define mapper85_irqrepeat	(mapper_regs[0x13])
#define mapper85_irqlatch	(mapper_regs[0x14])
#define mapper85_irqmode	(mapper_regs[0x15])
#define mapper85_irqcount	(mapper_regs16[0x1f - 0])
#define mapper85_irqcycle	(mapper_regs16[0x1f - 1])
#define vrc7_mute           (mapper_regs[0x16])

static void vrc7_map()
{
	mapper_map_prg( 8, 0, mapper85_prg(0));
	mapper_map_prg( 8, 1, mapper85_prg(1));
	mapper_map_prg( 8, 2, mapper85_prg(2));
	mapper_map_prg( 8, 3, -1);

	for (INT32 i = 0; i < 8; i++)
		mapper_map_chr( 1, i, mapper85_chr(i));

	switch (mapper85_mirror) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SINGLE_LOW); break;
		case 3: set_mirroring(SINGLE_HIGH); break;
	}

	if (mapper85_wramen) {
		NESMode &= ~NO_WORKRAM;
	} else {
		NESMode |= NO_WORKRAM;
	}
}

static void vrc7_write(UINT16 address, UINT8 data)
{
	if (address & 0x18) {
		// VRC7a uses xx10, VRC7b uses xxx8
		address = (address & ~0x0008) | 0x0010;
	}

	switch (address & 0xf030) {
		case 0x8000: mapper85_prg(0) = data & 0x3f; break;
		case 0x8010: mapper85_prg(1) = data & 0x3f; break;
		case 0x9000: mapper85_prg(2) = data & 0x3f; break;

		case 0x9010: BurnYM2413Write(0, data); break;
		case 0x9030: BurnYM2413Write(1, data); break;

		case 0xa000: mapper85_chr(0) = data; break;
		case 0xa010: mapper85_chr(1) = data; break;
		case 0xb000: mapper85_chr(2) = data; break;
		case 0xb010: mapper85_chr(3) = data; break;
		case 0xc000: mapper85_chr(4) = data; break;
		case 0xc010: mapper85_chr(5) = data; break;
		case 0xd000: mapper85_chr(6) = data; break;
		case 0xd010: mapper85_chr(7) = data; break;

		case 0xe000:
			mapper85_mirror = data & 0x3;
			mapper85_wramen = data & 0x80;
			vrc7_mute = data & 0x40;
			break;

		case 0xe010:
			mapper85_irqlatch = data;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf000:
			mapper85_irqrepeat = data & 1;
			mapper85_irqenable = data & 2;
			mapper85_irqmode = data & 4;
			if (mapper85_irqenable) {
				mapper85_irqcycle = 0;
				mapper85_irqcount = mapper85_irqlatch;
			}
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
		case 0xf010:
			mapper85_irqenable = mapper85_irqrepeat;
			M6502SetIRQLine(0, CPU_IRQSTATUS_NONE);
			break;
	}

	mapper_map();
}

static void vrc7_cycle()
{
	if (mapper85_irqenable) {
		if (mapper85_irqmode) { // cycle mode
			mapper85_irqcount++;
			if (mapper85_irqcount >= 0x100) {
				M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
				mapper_irq(4);
				mapper85_irqcount = mapper85_irqlatch;
			}
		} else {
			mapper85_irqcycle += 3; // scanline mode
			if (mapper85_irqcycle >= 341) {
				mapper85_irqcycle -= 341;
				mapper85_irqcount++;
				if (mapper85_irqcount == 0x100) {
					M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
					mapper_irq(4);
					mapper85_irqcount = mapper85_irqlatch;
				}
			}
		}
	}
}
#undef mapper85_prg
#undef mapper85_chr
#undef mapper85_mirror
#undef mapper85_wramen
#undef mapper85_irqenable
#undef mapper85_irqrepeat
#undef mapper85_irqlatch
#undef mapper85_irqmode
#undef mapper85_irqcount
#undef mapper85_irqcycle

// --[ mapper 116: Somari (AV Girl Fighting)
// it's a frankensteinian mess-mapper -dink
#define mapper116_vrc2_prg(x)		(mapper_regs[0x00 + (x)])
#define mapper116_vrc2_chr(x)		(mapper_regs[0x02 + (x)])
#define mapper116_vrc2_mirror		(mapper_regs[0x0a])
#define mapper116_mode				(mapper_regs[0x0b])

#define mapper116_mmc3_banksel      (mapper_regs[0x0c])
#define mapper116_mmc3_mirror		(mapper_regs[0x0d])
#define mapper116_mmc3_irqlatch 	(mapper_regs[0x0e])
#define mapper116_mmc3_irqcount		(mapper_regs[0x0f])
#define mapper116_mmc3_irqenable	(mapper_regs[0x10])
#define mapper116_mmc3_irqreload	(mapper_regs[0x11])
#define mapper116_mmc3_regs(x)		(mapper_regs16[0x12 + (x)]) // must be regs16!

#define mapper116_mmc1_regs(x)		(mapper_regs[0x1b + (x)])
#define mapper116_mmc1_serialbyte	(mapper_regs16[0])
#define mapper116_mmc1_bitcount		(mapper_regs16[1])

static void mapper116_defaults()
{
	mapper116_vrc2_prg(0) = 0;
	mapper116_vrc2_prg(1) = 1;
	mapper116_vrc2_chr(0) = 0xff;
	mapper116_vrc2_chr(1) = 0xff;
	mapper116_vrc2_chr(2) = 0xff;
	mapper116_vrc2_chr(3) = 0xff;
	mapper116_vrc2_chr(4) = 4;
	mapper116_vrc2_chr(5) = 5;
	mapper116_vrc2_chr(6) = 6;
	mapper116_vrc2_chr(7) = 7;

	mapper116_mmc1_regs(0) = 0x0c;

	mapper116_mmc3_regs(0) = 0;
	mapper116_mmc3_regs(1) = 2;
	mapper116_mmc3_regs(2) = 4;
	mapper116_mmc3_regs(3) = 5;
	mapper116_mmc3_regs(4) = 6;
	mapper116_mmc3_regs(5) = 7;
	mapper116_mmc3_regs(6) = -4;
	mapper116_mmc3_regs(7) = -3;
	mapper116_mmc3_regs(8) = -2;
	mapper116_mmc3_regs(9) = -1;
}

static void mapper116_write(UINT16 address, UINT8 data)
{
	if (address < 0x8000) {
		if ((address & 0x4100) == 0x4100) {
			// Someri
			mapper116_mode = data;
			if (address & 1) {
				mapper116_mmc1_bitcount = 0;
				mapper116_mmc1_serialbyte = 0;
				mapper116_mmc1_regs(0) = 0x0c;
				mapper116_mmc1_regs(3) = 0x00;
			}
			mapper_map();
		}
	} else {
		if (address == 0xa131) {
			// Gouder SL-1632, mode & 2 == mmc3, ~mode & 2 == vrc2
			mapper116_mode = (data & ~3) | ((data & 2) >> 1);
		}
		switch (mapper116_mode & 3) {
			case 0:	{
				if (address >= 0xb000 && address <= 0xe003) {
					INT32 reg = ((((address & 2) | (address >> 10)) >> 1) + 2) & 7;

					if (~address & 1) {
						mapper116_vrc2_chr(reg) = (mapper116_vrc2_chr(reg) & 0xf0) | (data & 0x0f);
					} else {
						mapper116_vrc2_chr(reg) = (mapper116_vrc2_chr(reg) & 0x0f) | ((data & 0x0f) << 4);
					}
				} else {
					switch (address & 0xf000) {
						case 0x8000: mapper116_vrc2_prg(0) = data; break;
						case 0x9000: mapper116_vrc2_mirror = data & 1; break;
						case 0xa000: mapper116_vrc2_prg(1) = data; break;
					}
				}
				mapper_map();
			}
			break;
			case 1: {
				switch (address & 0xe001) {
					case 0x8000: mapper116_mmc3_banksel = data; break;
					case 0x8001: mapper116_mmc3_regs(mapper116_mmc3_banksel & 0x7) = data; break;
					case 0xa000: mapper116_mmc3_mirror = data & 1; break;
					case 0xc000: mapper116_mmc3_irqlatch = data; break;
					case 0xc001: mapper116_mmc3_irqreload = 1; break;
					case 0xe000: mapper116_mmc3_irqenable = 0; M6502SetIRQLine(0, CPU_IRQSTATUS_NONE); break;
					case 0xe001: mapper116_mmc3_irqenable = 1; break;
				}
				mapper_map();
			}
			break;
			case 2:
			case 3: {
				if (address & 0x8000) {
					if (data & 0x80) { // bit 7, mapper reset
						mapper116_mmc1_bitcount = 0;
						mapper116_mmc1_serialbyte = 0;
						mapper116_mmc1_regs(0) |= 0x0c;
						if (mapper_map) mapper_map();
					} else {
						mapper116_mmc1_serialbyte |= (data & 1) << mapper116_mmc1_bitcount;
						mapper116_mmc1_bitcount++;
						if (mapper116_mmc1_bitcount == 5) {
							UINT8 reg = (address >> 13) & 0x3;
							mapper116_mmc1_regs(reg) = mapper116_mmc1_serialbyte;
							mapper116_mmc1_bitcount = 0;
							mapper116_mmc1_serialbyte = 0;
							if (mapper_map) mapper_map();
						}
					}
				}
			}
			break;
		}
	}
}

static void mapper116_map()
{
	INT32 bank = (mapper116_mode & 4) << 6;
	switch (mapper116_mode & 3) {
		case 0: {
			mapper_map_chr( 1, 0, mapper116_vrc2_chr(0) | bank);
			mapper_map_chr( 1, 1, mapper116_vrc2_chr(1) | bank);
			mapper_map_chr( 1, 2, mapper116_vrc2_chr(2) | bank);
			mapper_map_chr( 1, 3, mapper116_vrc2_chr(3) | bank);
			mapper_map_chr( 1, 4, mapper116_vrc2_chr(4) | bank);
			mapper_map_chr( 1, 5, mapper116_vrc2_chr(5) | bank);
			mapper_map_chr( 1, 6, mapper116_vrc2_chr(6) | bank);
			mapper_map_chr( 1, 7, mapper116_vrc2_chr(7) | bank);

			mapper_map_prg( 8, 0, mapper116_vrc2_prg(0));
			mapper_map_prg( 8, 1, mapper116_vrc2_prg(1));
			mapper_map_prg( 8, 2, -2);
			mapper_map_prg( 8, 3, -1);

			set_mirroring((mapper116_vrc2_mirror) ? HORIZONTAL : VERTICAL);
		}
		break;
		case 1: {
			mapper_map_prg(8, ((mapper116_mmc3_banksel & 0x40) >> 5), mapper116_mmc3_regs(6));
			mapper_map_prg(8, 1, mapper116_mmc3_regs(7));
			mapper_map_prg(8, 2 ^ ((mapper116_mmc3_banksel & 0x40) >> 5), mapper116_mmc3_regs(8));
			mapper_map_prg(8, 3, mapper116_mmc3_regs(9));

			INT32 swap = (mapper116_mmc3_banksel & 0x80) >> 5;
			mapper_map_chr( 1, 0 ^ swap, (mapper116_mmc3_regs(0) & 0xfe) | bank);
			mapper_map_chr( 1, 1 ^ swap, (mapper116_mmc3_regs(0) | 0x01) | bank);
			mapper_map_chr( 1, 2 ^ swap, (mapper116_mmc3_regs(1) & 0xfe) | bank);
			mapper_map_chr( 1, 3 ^ swap, (mapper116_mmc3_regs(1) | 0x01) | bank);
			mapper_map_chr( 1, 4 ^ swap, mapper116_mmc3_regs(2) | bank);
			mapper_map_chr( 1, 5 ^ swap, mapper116_mmc3_regs(3) | bank);
			mapper_map_chr( 1, 6 ^ swap, mapper116_mmc3_regs(4) | bank);
			mapper_map_chr( 1, 7 ^ swap, mapper116_mmc3_regs(5) | bank);

			set_mirroring((mapper116_mmc3_mirror) ? HORIZONTAL : VERTICAL);
		}
		break;
		case 2:
		case 3: {
			if (mapper116_mmc1_regs(0) & 0x8) {
				if (mapper116_mmc1_regs(0) & 0x4) {
					mapper_map_prg(16, 0, mapper116_mmc1_regs(3) & 0xf);
					mapper_map_prg(16, 1, 0x0f);
				} else {
					mapper_map_prg(16, 0, 0);
					mapper_map_prg(16, 1, mapper116_mmc1_regs(3) & 0xf);
				}
			} else {
				mapper_map_prg(32, 0, (mapper116_mmc1_regs(3) & 0xf) >> 1);
			}

			if (mapper116_mmc1_regs(0) & 0x10) {
				mapper_map_chr( 4, 0, mapper116_mmc1_regs(1));
				mapper_map_chr( 4, 1, mapper116_mmc1_regs(2));
			} else {
				mapper_map_chr( 8, 0, mapper116_mmc1_regs(1) >> 1);
			}

			switch (mapper116_mmc1_regs(0) & 3) {
				case 0: set_mirroring(SINGLE_LOW); break;
				case 1: set_mirroring(SINGLE_HIGH); break;
				case 2: set_mirroring(VERTICAL); break;
				case 3: set_mirroring(HORIZONTAL); break;
			}
		}
		break;
	}
}

static void mapper14_map()
{
	switch (mapper116_mode & 3) {
		case 0: {
			mapper_map_chr( 1, 0, mapper116_vrc2_chr(0));
			mapper_map_chr( 1, 1, mapper116_vrc2_chr(1));
			mapper_map_chr( 1, 2, mapper116_vrc2_chr(2));
			mapper_map_chr( 1, 3, mapper116_vrc2_chr(3));
			mapper_map_chr( 1, 4, mapper116_vrc2_chr(4));
			mapper_map_chr( 1, 5, mapper116_vrc2_chr(5));
			mapper_map_chr( 1, 6, mapper116_vrc2_chr(6));
			mapper_map_chr( 1, 7, mapper116_vrc2_chr(7));

			mapper_map_prg( 8, 0, mapper116_vrc2_prg(0));
			mapper_map_prg( 8, 1, mapper116_vrc2_prg(1));
			mapper_map_prg( 8, 2, -2);
			mapper_map_prg( 8, 3, -1);

			set_mirroring((mapper116_vrc2_mirror) ? HORIZONTAL : VERTICAL);
		}
		break;
		case 1: {
			mapper_map_prg(8, ((mapper116_mmc3_banksel & 0x40) >> 5), mapper116_mmc3_regs(6));
			mapper_map_prg(8, 1, mapper116_mmc3_regs(7));
			mapper_map_prg(8, 2 ^ ((mapper116_mmc3_banksel & 0x40) >> 5), mapper116_mmc3_regs(8));
			mapper_map_prg(8, 3, mapper116_mmc3_regs(9));

			INT32 swap = (mapper116_mmc3_banksel & 0x80) >> 5;
			INT32 bank0 = (mapper116_mode & 0x08) << 5;
			INT32 bank1 = (mapper116_mode & 0x20) << 3;
			INT32 bank2 = (mapper116_mode & 0x80) << 1;
			mapper_map_chr( 1, 0 ^ swap, (mapper116_mmc3_regs(0) & 0xfe) | bank0);
			mapper_map_chr( 1, 1 ^ swap, (mapper116_mmc3_regs(0) | 0x01) | bank0);
			mapper_map_chr( 1, 2 ^ swap, (mapper116_mmc3_regs(1) & 0xfe) | bank0);
			mapper_map_chr( 1, 3 ^ swap, (mapper116_mmc3_regs(1) | 0x01) | bank0);
			mapper_map_chr( 1, 4 ^ swap, mapper116_mmc3_regs(2) | bank1);
			mapper_map_chr( 1, 5 ^ swap, mapper116_mmc3_regs(3) | bank1);
			mapper_map_chr( 1, 6 ^ swap, mapper116_mmc3_regs(4) | bank2);
			mapper_map_chr( 1, 7 ^ swap, mapper116_mmc3_regs(5) | bank2);

			set_mirroring((mapper116_mmc3_mirror) ? HORIZONTAL : VERTICAL);
		}
		break;
	}
}

static void mapper116_mmc3_scanline()
{
	if ((mapper116_mode & 0x03) != 1) return;

	if (mapper116_mmc3_irqcount == 0 || mapper116_mmc3_irqreload) {
		mapper116_mmc3_irqreload = 0;
		mapper116_mmc3_irqcount = mapper116_mmc3_irqlatch;
	} else {
		mapper116_mmc3_irqcount--;
	}

	if (mapper116_mmc3_irqcount == 0 && mapper116_mmc3_irqenable && (mmc5_mask[0] & 0x18)) {
		mapper_irq(0);
	}
}
#undef mapper116_vrc2_prg
#undef mapper116_vrc2_chr
#undef mapper116_vrc2_mirror
#undef mapper116_mode

#undef mapper116_mmc3_banksel
#undef mapper116_mmc3_mirror
#undef mapper116_mmc3_irqlatch
#undef mapper116_mmc3_irqcount
#undef mapper116_mmc3_irqenable
#undef mapper116_mmc3_irqreload
#undef mapper116_mmc3_regs

#undef mapper116_mmc1_regs
#undef mapper116_mmc1_serialbyte
#undef mapper116_mmc1_bitcount

// --[ mapper 80: Taito X1-005
// --[ mapper 207: with advanced mirroring
#define mapper207_prg(x)     (mapper_regs[0 + x]) // 0 - 2
#define mapper207_chr(x)     (mapper_regs[3 + x]) // 3 - 9
#define mapper207_mirror     (mapper_regs[0x1f - 0])
#define mapper207_ram_en     (mapper_regs[0x1f - 1])
#define mapper207_adv_mirror (mapper_regs[0x1f - 2]) // mapper 207 yes, 80 no

static UINT8 mapper80_ram[0x80];

static void mapper207_scan()
{
	SCAN_VAR(mapper80_ram);
}

static UINT8 mapper207_read(UINT16 address)
{
	if (address >= 0x7f00 && address <= 0x7fff) {
		if (mapper207_ram_en == 0xa3) {
			return mapper80_ram[address & 0x7f];
		}
	}
	return 0xff;
}

static void mapper207_write(UINT16 address, UINT8 data)
{
	if (address >= 0x7f00 && address <= 0x7fff) {
		if (mapper207_ram_en == 0xa3) {
			mapper80_ram[address & 0x7f] = data;
		}
	}

	switch (address) {
		case 0x7ef0: mapper207_chr(0) = data; break;
		case 0x7ef1: mapper207_chr(1) = data; break;
		case 0x7ef2: mapper207_chr(2) = data; break;
		case 0x7ef3: mapper207_chr(3) = data; break;
		case 0x7ef4: mapper207_chr(4) = data; break;
		case 0x7ef5: mapper207_chr(5) = data; break;
		case 0x7ef6:
		case 0x7ef7: mapper207_mirror = data & 1; break; // for mapper80
		case 0x7ef8:
		case 0x7ef9: mapper207_ram_en = data; break; // for mapper80
		case 0x7efa:
		case 0x7efb: mapper207_prg(0) = data; break;
		case 0x7efc:
		case 0x7efd: mapper207_prg(1) = data; break;
		case 0x7efe:
		case 0x7eff: mapper207_prg(2) = data; break;
	}

	mapper_map();
}

static void mapper207_map()
{
	mapper_map_prg( 8, 0, mapper207_prg(0));
	mapper_map_prg( 8, 1, mapper207_prg(1));
	mapper_map_prg( 8, 2, mapper207_prg(2));
	mapper_map_prg( 8, 3, -1);

	mapper_map_chr( 2, 0, (mapper207_chr(0) >> 1) & 0x3f);
	mapper_map_chr( 2, 1, (mapper207_chr(1) >> 1) & 0x3f);
	mapper_map_chr( 1, 4, mapper207_chr(2));
	mapper_map_chr( 1, 5, mapper207_chr(3));
	mapper_map_chr( 1, 6, mapper207_chr(4));
	mapper_map_chr( 1, 7, mapper207_chr(5));

	if (mapper207_adv_mirror == 0) {
		set_mirroring((mapper207_mirror & 1) ? VERTICAL : HORIZONTAL);
	} else {
		nametable_mapall((mapper207_chr(0) >> 7) & 1, (mapper207_chr(0) >> 7) & 1, (mapper207_chr(1) >> 7) & 1, (mapper207_chr(1) >> 7) & 1);
	}
}
#undef mapper207_prg
#undef mapper207_chr
#undef mapper207_mirror
#undef mapper207_ram_en

// ---[ mapper 81 NTDEC Super Gun (Caltron)
static void mapper81_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper81_map()
{
    mapper_map_prg(16, 0, (mapper_regs[0] >> 2) & 0x03);
    mapper_map_prg(16, 1, -1);
    mapper_map_chr( 8, 0, mapper_regs[0] & 0x03);
}

// --[ mapper 232: Camerica (Quattro Adventure, Arcade, Sports)
static void mapper232_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000 && address <= 0xbfff) {
		mapper_regs[0] = data;
	}

	if (address >= 0xc000 && address <= 0xffff) {
		mapper_regs[1] = data;
	}

	mapper_map();
}

static void mapper232_map()
{
	UINT32 block = (mapper_regs[0] & 0x18) >> 1;
	mapper_map_prg(16, 0, block | (mapper_regs[1] & 3));
	mapper_map_prg(16, 1, block | 3);
	mapper_map_chr( 8, 0, 0);
}

// --[ mapper 78: Irem? (Holy Diver, Uchuusen: Cosmo Carrier)
static void mapper78_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper78_map()
{
	mapper_map_prg(16, 0, mapper_regs[0] & 7);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, mapper_regs[0] >> 4);
	if (Cart.Mirroring == 4) // Holy Diver
		set_mirroring(((mapper_regs[0] >> 3) & 1) ? VERTICAL : HORIZONTAL);
	else
		set_mirroring(((mapper_regs[0] >> 3) & 1) ? SINGLE_HIGH : SINGLE_LOW);
}

// --[ mapper 32: Irem G101 (Image Fight, Kaiketsu Yanchamaru 2, Major League, Meikyuu Jima, ...)
#define mapper32_mirror  (mapper_regs[0x1f - 0])
#define mapper32_prg(x)  (mapper_regs[0 + x])
#define mapper32_chr(x)  (mapper_regs[2 + (x)])

static void mapper32_write(UINT16 address, UINT8 data)
{
	switch (address & 0xf000) {
		case 0x8000:
			mapper32_prg(0) = data;
			break;
		case 0x9000:
			mapper32_mirror = data;
			break;
		case 0xa000:
			mapper32_prg(1) = data;
			break;
		case 0xb000:
			mapper32_chr((address & 7)) = data;
			break;
	}
	mapper_map();
}

static void mapper32_map()
{
	mapper_map_prg( 8, 0 ^ (mapper32_mirror & 2), mapper32_prg(0));
	mapper_map_prg( 8, 1, mapper32_prg(1));
	mapper_map_prg( 8, 2 ^ (mapper32_mirror & 2), -2);
	mapper_map_prg( 8, 3, -1);

	for (INT32 i = 0; i < 8; i++)
		mapper_map_chr( 1, i, mapper32_chr(i));

	if (Cart.Crc == 0xd8dfd3d1) {
		// Major League (Japan) hardwired to SINGLE_LOW
		set_mirroring(SINGLE_LOW);
	} else {
		set_mirroring((mapper32_mirror & 1) ? HORIZONTAL : VERTICAL);
	}
}

#undef mapper32_mirror
#undef mapper32_prg
#undef mapper32_chr

// --[ mapper 34: BNROM, NINA-001 (Deadly Towers, Darkseed) (Impossible Mission 2)
static void mapper34_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8000) { // BNROM
		mapper_regs[0] = data;
	} else {
		switch (address) {   // NINA-001
			case 0x7ffd:
			case 0x7ffe:
			case 0x7fff:
				mapper_regs[(address & 0x3) - 1] = data;
				break;
		}
	}
	mapper_map();
}

static void mapper34_map()
{
	mapper_map_prg(32, 0, mapper_regs[0]);

	mapper_map_chr( 4, 0, mapper_regs[1]);
	mapper_map_chr( 4, 1, mapper_regs[2]);
}

// --[ mapper 87: Jaleco (Argus, Ninja JajaMaru Kun, City Connection)
static void mapper87_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = ((data >> 1) & 1) | ((data << 1) & 2);
	mapper_map();
}

static void mapper87_map()
{
	mapper_map_prg(32, 0, 0);
	mapper_map_chr( 8, 0, mapper_regs[0]);
}

// --[ mapper 88,154: Namcot (Quinty, Dragon Spirit..)
#define mapper88_cmd	(mapper_regs[0x1f - 0])
#define mapper88_mirror	(mapper_regs[0x1f - 1])
#define mapper154       (mapper_regs[0x1f - 2])

static void mapper88_write(UINT16 address, UINT8 data)
{
	switch (address & 0x8001) {
		case 0x8000:
			mapper88_cmd = data & 7;
			mapper88_mirror = (data >> 6) & 1;
			break;
		case 0x8001:
			mapper_regs[mapper88_cmd] = data;
			break;
	}

	mapper_map();
}

static void mapper88_map()
{
	mapper_map_chr( 2, 0, (mapper_regs[0] >> 1) & 0x1f);
	mapper_map_chr( 2, 1, (mapper_regs[1] >> 1) & 0x1f);

	mapper_map_chr( 1, 4, mapper_regs[2] | 0x40);
	mapper_map_chr( 1, 5, mapper_regs[3] | 0x40);
	mapper_map_chr( 1, 6, mapper_regs[4] | 0x40);
	mapper_map_chr( 1, 7, mapper_regs[5] | 0x40);

	mapper_map_prg( 8, 0, mapper_regs[6]);
	mapper_map_prg( 8, 1, mapper_regs[7]);
	mapper_map_prg( 8, 2, -2);
	mapper_map_prg( 8, 3, -1);

	if (mapper154) {
		switch (mapper88_mirror) {
			case 0: set_mirroring(SINGLE_LOW); break;
			case 1: set_mirroring(SINGLE_HIGH); break;
		}
	}
}
#undef mapper88_cmd
#undef mapper88_mirror

// --[ mapper 89: (Tenka no Goikenban: Mito Koumon)
static void mapper89_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper89_map()
{
	mapper_map_prg(16, 0, (mapper_regs[0] >> 4) & 7);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, (mapper_regs[0] & 7) | ((mapper_regs[0] >> 4) & 8));
	set_mirroring((mapper_regs[0] & 8) ? SINGLE_HIGH : SINGLE_LOW);
}

// --[ mapper 93: Sunsoft-1/2 PCB: Fantasy Zone, Shanghai
static void mapper93_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper93_map()
{
	mapper_map_prg(16, 0, mapper_regs[0] >> 4);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, 0);
}

// --[ mapper 94: Senjou no Ookami (Commando)
static void mapper94_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper94_map()
{
	mapper_map_prg(16, 0, mapper_regs[0] >> 2);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, 0);
}

// --[ mapper 97: Kaiketsu Yanchamaru
static void mapper97_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper97_map()
{
	mapper_map_prg(16, 0, -1);
	mapper_map_prg(16, 1, mapper_regs[0] & 0xf);
	mapper_map_chr( 8, 0, ((mapper_regs[0] >> 1) & 1) | ((mapper_regs[0] << 1) & 2));
	set_mirroring((mapper_regs[0] & 0x80) ? VERTICAL : HORIZONTAL);
}

// --[ mapper 107: (Magic Dragon)
static void mapper107_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper107_map()
{
	mapper_map_prg(32, 0, (mapper_regs[0] >> 1) & 3);
	mapper_map_chr( 8, 0, mapper_regs[0] & 7);
}

// --[ mapper 101: (Urusei Yatsura - Lum no Wedding Bell (Japan))
static void mapper101_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper101_map()
{
	mapper_map_prg(32, 0, 0);
	mapper_map_chr( 8, 0, mapper_regs[0]);
}

// --[ mapper 112: ASDER (Huang Di, ...)
#define mapper112_mirror		(mapper_regs[0x1f - 0])
#define mapper112_regnum		(mapper_regs[0x1f - 1])
#define mapper112_chrbank		(mapper_regs[0x1f - 2])

static void mapper112_write(UINT16 address, UINT8 data)
{
	switch (address & 0xe001) {
		case 0x8000: mapper112_regnum = data & 7; break;
		case 0xa000: mapper_regs[mapper112_regnum] = data; break;
		case 0xc000: mapper112_chrbank = data; break;
		case 0xe000: mapper112_mirror = data & 1; break;
	}

	mapper_map();
}

static void mapper112_map()
{
	mapper_map_prg( 8, 0, mapper_regs[0]);
	mapper_map_prg( 8, 1, mapper_regs[1]);

	mapper_map_chr( 2, 0, mapper_regs[2] >> 1);
	mapper_map_chr( 2, 1, mapper_regs[3] >> 1);

	mapper_map_chr( 1, 4, ((mapper112_chrbank & 0x10) << 4) | mapper_regs[4]);
	mapper_map_chr( 1, 5, ((mapper112_chrbank & 0x20) << 3) | mapper_regs[5]);
	mapper_map_chr( 1, 6, ((mapper112_chrbank & 0x40) << 2) | mapper_regs[6]);
	mapper_map_chr( 1, 7, ((mapper112_chrbank & 0x80) << 1) | mapper_regs[7]);

	set_mirroring(mapper112_mirror ? HORIZONTAL : VERTICAL);
}

// --[ mapper 113 NINA-03/NINA-06 ext. (Papillon Gals) / Mojon Twins Multicart
static void mapper113_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper113_map()
{
	mapper_map_prg(32, 0, (mapper_regs[0] >> 3) & 7);
	mapper_map_chr( 8, 0, ((mapper_regs[0] >> 3) & 8) | (mapper_regs[0] & 7));
	set_mirroring((mapper_regs[0] & 0x80) ? VERTICAL : HORIZONTAL);
}

// --[ mapper 79, 146: NINA-03/NINA-06 (Death Bots), Sachen (146 / Metal Fighter)
static void mapper79_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper79_map()
{
	mapper_map_prg(32, 0, (mapper_regs[0] >> 3) & 1);
	mapper_map_chr( 8, 0, (mapper_regs[0] & 7));
}

// --[ mapper 137, 138, 139, 141: Sachen 8259x (Super Pang, etc)
static void mapper8259_write(UINT16 address, UINT8 data)
{
	if (address < 0x4100) return;

	if ((address & 0x4101) == 0x4100) {
		mapper_regs[8] = data; // command
	} else {
		mapper_regs[mapper_regs[8] & 7] = data;
		mapper_map();
	}
}

static void mapper8259_map()
{
	mapper_map_prg(32, 0, mapper_regs[5] & 7);

	if (Cart.CHRRomSize != 0) {
		for (INT32 i = 0; i < 4; i++) {
			INT32 bank = 0;

			if (mapper_regs[7] & 1)
				bank = (mapper_regs[0] & 7) | ((mapper_regs[4] & 7) << 3);
			else
				bank = (mapper_regs[i] & 7) | ((mapper_regs[4] & 7) << 3);

			switch (mapper_regs[0x1f]) {
				case 137:                         // 137 8259D
					bank = mapper_regs[i] & 7;
					switch (i & 3) {
						case 1:
							bank |= (mapper_regs[4] & 1) << 4;
							break;
						case 2:
							bank |= (mapper_regs[4] & 2) << 3;
							break;
						case 3:
							bank |= ((mapper_regs[4] & 4) << 2) | ((mapper_regs[6] & 1) << 3);
							break;
					}
					mapper_map_chr( 1, i, bank);
					mapper_map_chr( 4, 1, -1);
					break;
				case 138:                         // 138 8259B
					mapper_map_chr( 2, i, bank);
					break;
				case 139:
					bank = (bank << 2) | (i & 3); // 139 8259C
					mapper_map_chr( 2, i, bank);
					break;
				case 141:
					bank = (bank << 1) | (i & 1); // 141 8259A
					mapper_map_chr( 2, i, bank);
					break;
			}
		}
	}

	if (mapper_regs[7] & 1) {
		set_mirroring(VERTICAL);
	} else {
		switch ((mapper_regs[7] >> 1) & 3) {
			case 0: set_mirroring(VERTICAL); break;
			case 1: set_mirroring(HORIZONTAL); break;
			case 2: set_mirroring(SACHEN); break;
			case 3: set_mirroring(SINGLE_LOW); break;
		}
	}
}

// --[ mapper 150: Sachen 74LS374XX (Tasac)
#define mapper150_cmd       (mapper_regs[0x1f - 0])
#define mapper150_prg       (mapper_regs[0x1f - 1])

static void mapper150_write(UINT16 address, UINT8 data)
{
	switch (address & 0xc101) {
		case 0x4100:
			mapper150_cmd = data & 0x7;
			break;
		case 0x4101:
			switch (mapper150_cmd) {
				case 2:
					mapper150_prg = data & 1;
					break;
				case 5:
					mapper150_prg = data & 7;
					break;
				default:
					mapper_regs[mapper150_cmd] = data;
					break;
			}
			mapper_map();
			break;
	}
}

static UINT8 mapper150_read(UINT16 address)
{
	if ((address & 0xc101) == 0x4000) {
		return (~mapper150_cmd) ^ 1; // '1' (or 0) is switch for Sachen / Sachen Hacker
	}

	return 0;
}

static void mapper150_map()
{
	mapper_map_prg(32, 0, mapper150_prg);

	mapper_map_chr( 8, 0, ((mapper_regs[2] & 0x1) << 3) | ((mapper_regs[4] & 0x1) << 2) | (mapper_regs[6] & 0x03));

	switch ((mapper_regs[7] >> 1) & 3) {
		case 0: set_mirroring(VERTICAL); break;
		case 1: set_mirroring(HORIZONTAL); break;
		case 2: set_mirroring(SACHEN); break;
		case 3: set_mirroring(SINGLE_LOW); break;
	}
}

// ---[ mapper 162 Zelda Triforce of the Gods
#define mapper162_reg(x)			(mapper_regs[0x00 + (x)])

static void mapper162_write(UINT16 address, UINT8 data)
{
	if (address >= 0x5000 && address <= 0x5fff) {
		mapper162_reg((address >> 8) & 0x03) = data;

		mapper_map();
	}
}

static void mapper162_map()
{
	switch (mapper162_reg(3) & 0x05) {
		case 0:
			mapper_map_prg(32, 0, (mapper162_reg(1) & 0x02) | ((mapper162_reg(2) & 0x0f) << 4) | (mapper162_reg(0) & 0x0c));
			break;
		case 1:
			mapper_map_prg(32, 0, ((mapper162_reg(2) & 0x0f) << 4) | (mapper162_reg(0) & 0x0c));
			break;
		case 4:
			mapper_map_prg(32, 0, ((mapper162_reg(1) & 0x02) >> 1) | ((mapper162_reg(2) & 0x0f) << 4) | (mapper162_reg(0) & 0x0e));
			break;
		case 5:
			mapper_map_prg(32, 0, ((mapper162_reg(2) & 0x0f) << 4) | (mapper162_reg(0) & 0x0f));
			break;
	}
}

// ---[ mapper 163 Final Fantasy VII (NJ063) + Advent Children ver. (&etc.)
#define mapper163_reg(x)			(mapper_regs[0x00 + (x)])
#define mapper163_chr(x)			(mapper_regs[0x08 + (x)])
#define mapper163_prg				(mapper_regs[0x1f - 0])
#define mapper163_prot_flip			(mapper_regs[0x1f - 1])
#define mapper163_raster_split		(mapper_regs[0x1f - 2])

static UINT8 mapper163_read(UINT16 address)
{
	switch (address & 0x7700) {
		case 0x5100:
			return mapper163_reg(3) | mapper163_reg(1) | mapper163_reg(0) | (mapper163_reg(2) ^ 0xff);
		case 0x5500:
			return (mapper163_prot_flip == 0) ? (mapper163_reg(3) | mapper163_reg(0)) : 0;
	}

	return 0x4;
}

static void mapper163_write(UINT16 address, UINT8 data)
{
	if ((address & 0xf000) == 0x5000) {
		if (address == 0x5101) {
			if (mapper163_reg(4) != 0 && data == 0) {
				mapper163_prot_flip ^= 1;
			}
			mapper163_reg(4) = data;
		} else if (address == 0x5100 && data == 0x6) {
			mapper163_prg = 0x3;
		} else {
			switch (address & 0x7300) {
				case 0x5000:
					mapper163_reg(0) = data;
					if (!(data & 0x80) && scanline < 128) {
						mapper163_chr(0) = 0;
						mapper163_chr(1) = 1;
					}
					mapper163_raster_split = data & 0x80;
					mapper163_prg = (mapper163_reg(0) & 0xf) | ((mapper163_reg(2) & 0xf) << 4);
					break;
				case 0x5100:
					mapper163_reg(1) = data;
					if (data == 0x6) {
						mapper163_prg = 3;
					}
					break;
				case 0x5200:
					mapper163_reg(2) = data;
					mapper163_prg = (mapper163_reg(0) & 0xf) | ((mapper163_reg(2) & 0xf) << 4);
					break;
				case 0x5300:
					mapper163_reg(3) = data;
					break;
			}
		}
		mapper_map();
	}
}

static void mapper163_map()
{
	mapper_map_prg(32, 0, mapper163_prg);
	mapper_map_chr(4, 0, mapper163_chr(0));
	mapper_map_chr(4, 1, mapper163_chr(1));
}

static void mapper163_cycle(UINT16 address)
{
	static INT32 last_127 = 0;
	static INT32 last_239 = 0;

	if (mapper163_raster_split && pixel > 257) {
		switch (scanline) {
			case 127:
				if (nCurrentFrame != last_127) {
					mapper_map_chr(4, 0, 1);
					mapper_map_chr(4, 1, 1);
					mapper163_chr(0) = 1;
					mapper163_chr(1) = 1;
				}
				last_127 = nCurrentFrame;
				break;
			case 239:
				if (nCurrentFrame != last_239) {
					mapper_map_chr(4, 0, 0);
					mapper_map_chr(4, 1, 0);
					mapper163_chr(0) = 0;
					mapper163_chr(1) = 0;
				}
				last_239 = nCurrentFrame;
				break;
		}
	}
}

#undef mapper163_reg
#undef mapper163_prg
#undef mapper163_chr
#undef mapper163_prot_flip
#undef mapper163_raster_split

// --[ mapper 38: (Crime Busters)
static void mapper38_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper38_map()
{
	mapper_map_prg(32, 0, mapper_regs[0] & 0x3);
	mapper_map_chr( 8, 0, (mapper_regs[0] >> 2) & 0x3);
}

// --[ mapper 66,140: (Bio Senshi Dan: Increaser tono Tatakai, Youkai Club, ..)
static void mapper140_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper140_map()
{
	mapper_map_prg(32, 0, mapper_regs[0] >> 4);
	mapper_map_chr( 8, 0, mapper_regs[0] & 0xf);
}

// --[ mapper 11, 144: ColorDreams, Death Race(144)
static void mapper11_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper144_write(UINT16 address, UINT8 data)
{
	if (address >= 0x8001) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper11_map()
{
	mapper_map_prg(32, 0, mapper_regs[0] & 0xf);
	mapper_map_chr( 8, 0, mapper_regs[0] >> 4);
}

// --[ mapper 152: (Arkanoid II, Pocket Zaurus: Juu Ouken no Nazo, ...)
static void mapper152_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper152_map()
{
	mapper_map_prg(16, 0, (mapper_regs[0] >> 4) & 7);
	mapper_map_prg(16, 1, -1);
	mapper_map_chr( 8, 0, mapper_regs[0] & 0xf);
	set_mirroring((mapper_regs[0] & 0x80) ? SINGLE_HIGH : SINGLE_LOW);
}

// --[ mapper 156: Open (Metal Force, Buzz & Waldog, Koko Adv., Janggun-ui Adeul)
#define mapper156_chr_lo(x)     (mapper_regs[0 + (x)])  // x = 0 - 7
#define mapper156_chr_hi(x)     (mapper_regs[8 + (x)])  // x = 0 - 7
#define mapper156_prg           (mapper_regs[0x1f - 0])
#define mapper156_mirror        (mapper_regs[0x1f - 1])

static void mapper156_write(UINT16 address, UINT8 data)
{
	switch (address) {
		case 0xc000:
		case 0xc001:
		case 0xc002:
		case 0xc003:
		case 0xc008:
		case 0xc009:
		case 0xc00a:
		case 0xc00b:
			mapper156_chr_lo((address & 0x3) | ((address & 0x8) >> 1)) = data;
			break;
		case 0xc004:
		case 0xc005:
		case 0xc006:
		case 0xc007:
		case 0xc00c:
		case 0xc00d:
		case 0xc00e:
		case 0xc00f:
			mapper156_chr_hi((address & 0x3) | ((address & 0x8) >> 1)) = data;
			break;
		case 0xc010:
			mapper156_prg = data;
			break;
		case 0xc014:
			mapper156_mirror = 0x10 | (data & 1);
			break;
	}

	mapper_map();
}

static void mapper156_map()
{
	mapper_map_prg(16, 0, mapper156_prg);
	mapper_map_prg(16, 1, -1);

	for (INT32 i = 0; i < 8; i++) {
		mapper_map_chr( 1, i, (mapper156_chr_hi(i) << 8) | mapper156_chr_lo(i));
	}

	switch (mapper156_mirror) {
		case 0: set_mirroring(SINGLE_LOW); break;
		case 0x10: set_mirroring(VERTICAL); break;
		case 0x11: set_mirroring(HORIZONTAL); break;
	}
}

// --[ mapper 180: Crazy Climber
static void mapper180_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper180_map()
{
	mapper_map_prg(16, 0, 0);
	mapper_map_prg(16, 1, mapper_regs[0] & 0x7);
	mapper_map_chr( 8, 0, 0);
}

// --[ mapper 184: Atlantis no Nazo, Kanshakudama Nage Kantarou no Toukaidou Gojuusan Tsugi, Wing of Madoola
static void mapper184_write(UINT16 address, UINT8 data)
{
	mapper_regs[0] = data;
	mapper_map();
}

static void mapper184_map()
{
	mapper_map_prg(32, 0, 0);
	mapper_map_chr( 4, 0, mapper_regs[0] & 0xf);
	mapper_map_chr( 4, 1, mapper_regs[0] >> 4);
}

// --[ mapper 185 CNROM+prot
static void mapper185_write(UINT16 address, UINT8 data)
{
	if (address & 0x8000) {
		mapper_regs[0] = data;
		mapper_map();
	}
}

static void mapper185_map()
{
	if ( (mapper_regs[0] & 3) && (mapper_regs[0] != 0x13) ) { // map rom
		mapper_map_chr_ramrom( 8, 0, 0, MEM_ROM);
	} else { // map ram
		mapper_map_chr_ramrom( 8, 0, 0, MEM_RAM);
	}
}

// --[ end mappers

static void mapper_reset()
{
	mapper_init(Cart.Mapper);
}

static void mapper_scan(INT32 nAction, INT32 *pnMin)
{
	SCAN_VAR(PRGMap);
	SCAN_VAR(CHRMap);
	SCAN_VAR(CHRType);
	SCAN_VAR(PRGExpMap);

	SCAN_VAR(mapper_regs);
	SCAN_VAR(mapper_regs16);
	SCAN_VAR(mapper_irq_exec);

	if (mapper_scan_cb) {
		mapper_scan_cb();
	}

	if (nAction & ACB_WRITE) {
		if (mapper_map) mapper_map();
	}
}

static INT32 mapper_init(INT32 mappernum)
{
	INT32 retval = -1;

	if (NESMode & MAPPER_NOCLEAR && RESETMode == RESET_BUTTON) {
		// do nothing! certain mappers need regs preserved on RESET_BUTTON
	} else {
		memset(mapper_regs, 0, sizeof(mapper_regs));
		memset(mapper_regs16, 0, sizeof(mapper_regs16));
	}

	mapper_write = NULL; // 8000 - ffff
	mapper_map = NULL;
	mapper_scanline = NULL; // scanline cb
	mapper_cycle = NULL; // cycle cb (1x per cycle)
	mapper_ppu_clock = NULL; // called after busaddress change (see mapper09) (only in visible & prerender!)
	mapper_ppu_clockall = NULL; // called every ppu clock
	mapper_scan_cb = NULL; // savestate cb (see vrc6)
	mapper_scan_cb_nvram = NULL; // savestate cb (nvram, mapper 303)

	mapper_prg_read = mapper_prg_read_int; // 8000-ffff (read)

	PRGExpMap = 0;
	cart_exp_write = NULL; // 6000 - 7fff
	cart_exp_read = NULL;

	psg_area_write = NULL; // 4020 - 5fff
	psg_area_read = NULL;

	bprintf(0, _T("Mapper #%d init/reset!\n"), mappernum);

	switch (mappernum) {
		case 0x808: { // Famicom Disk System
			NESMode |= MAPPER_NOCLEAR; // preserve mapper regs on soft-reset
			mapper_map_prg(32, 0, 0);
			mapper_map_chr( 8, 0, 0);
			psg_area_write = mapperFDS_write; // 4000 - 5fff
			psg_area_read = mapperFDS_read;
			mapper_write = mapperFDS_prg_write; // 8000 - ffff
			mapper_cycle = mapperFDS_cycle;
			mapper_map   = mapperFDS_map;
			nes_ext_sound_cb = mapperFDS_mixer;
			mapper_scan_cb  = mapperFDS_scan;

			BurnLEDInit(1, LED_POSITION_BOTTOM_RIGHT, LED_SIZE_4x4, LED_COLOR_GREEN, 80);

			mapperFDS_reset();
			mapper_map();
			retval = 0;
			break;
		}

		case 0: { // no mapper
			mapper_map_prg(32, 0, 0);
			mapper_map_chr( 8, 0, 0);
			retval = 0;
			break;
		}

		case 1: { // mmc1
			cart_exp_write = mapper01_exp_write; // 6000 - 7fff
			cart_exp_read = mapper01_exp_read;
			mapper_write = mapper01_write;
			mapper_map   = mapper01_map;
			mapper_regs[0] = 0xc;
			mapper_map();
			retval = 0;
			break;
		}

		case 2: { // UxROM
			mapper_write = mapper02_write;
			mapper_map   = mapper02_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 15: { // Contra 168-in-1 Multicart
			mapper_write = mapper15_write;
			mapper_map   = mapper15_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 190: { // Magic Kid Googoo
			mapper_write = mapper190_write;
			mapper_map   = mapper190_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 193: { // NTDEC TC-112 (War in the Gulf, Fighting Hero, ..)
			cart_exp_write = mapper193_write;
			mapper_map   = mapper193_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 41: { // Caltron 6-in-1
			cart_exp_write = mapper41_write;
			mapper_write = mapper41_write;
			mapper_map   = mapper41_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 53: { // SuperVision 16-in-1
			NESMode |= MAPPER_NOCLEAR; // preserve mapper regs on soft-reset
			cart_exp_read = mapper53_exp_read;
			cart_exp_write = mapper53_write;
			mapper_write = mapper53_write;
			mapper_map   = mapper53_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 104: { // Golden Five / Pegasus 5-in-1
			mapper_write = mapper104_write;
			mapper_map   = mapper104_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 389: { // Caltron 9-in-1
			mapper_write = mapper389_write;
			mapper_map   = mapper389_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 216: { // Magic Jewelry 2
			mapper_write = mapper216_write;
			mapper_map   = mapper216_map;
			psg_area_read = mapper216_read;
			mapper_map();
			retval = 0;
			break;
		}

		case 132: { // Qi Wang
			mapper_write = mapper132_write;
			mapper_map   = mapper132_map;
			psg_area_read = mapper132_read;
			psg_area_write = mapper132_write;
			mapper_map();
			retval = 0;
			break;
		}

		case 3: { // CNROM
			mapper_write = mapper03_write;
			mapper_map   = mapper03_map;
			if (Cart.Crc == 0xab29ab28) // dropzone
				mapper_cycle = mapper03_cycle;
			mapper_map_prg(32, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 5: { // MMC5
			cart_exp_write = mapper5_exp_write; // 6000 - 7fff
			cart_exp_read = mapper5_exp_read;
			psg_area_write = mapper5_write; // 4000 - 5fff
			psg_area_read = mapper5_read;
			mapper_write = mapper5_prg_write; // 8000 - ffff
			mapper_map   = mapper5_map;
			mapper_scan_cb  = mapper5_scan;
			mapper_ppu_clockall = mapper5_ppu_clk;
			nes_ext_sound_cb = mapper5_mixer;

			read_nt = &mapper5_ntread;
			write_nt = &mapper5_ntwrite;

			mapper5_reset();

			mapper_map();
			retval = 0;
			break;
		}

		case 7: { // AxROM: Battle Toads, Marble Madness, RC Pro-Am
			mapper_write = mapper07_write;
			mapper_map   = mapper07_map;
		    mapper_map_chr( 8, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 9: { // mmc2: punch out!
			mapper_write = mapper09_write;
			mapper_map   = mapper09_map;
			mapper_ppu_clock = mapper09_ppu_clk;
			mapper_map_prg( 8, 1, -3);
			mapper_map_prg( 8, 2, -2);
			mapper_map_prg( 8, 3, -1);
			mapper_map();
			retval = 0;
			break;
		}

		case 10: { // mmc4: fire emblem (mmc2 + sram + different prg mapping)
			mapper_write = mapper09_write;
			mapper_map   = mapper10_map;
			mapper_ppu_clock = mapper09_ppu_clk;
			mapper_map();
			retval = 0;
			break;
		}

		case 13: {
			mapper_write = mapper13_write;
			mapper_map = mapper13_map;
			mapper_map_prg(32, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 133: {
			psg_area_write = mapper133_write;
			mapper_map = mapper133_map;
			mapper_map_prg(32, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 159: // 128byte eeprom
		case 16: // 256byte eeprom
			{ // Bandai fcg
			mapper_write = mapper16_write;
			cart_exp_write = mapper16_write;
			mapper_map   = mapper16_map;
			mapper_cycle = mapper16_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 153: { // Bandai fcg (Famicom Jump II)
			mapper_write = mapper16_write;
			mapper_map   = mapper153_map;
			mapper_cycle = mapper16_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 18: { // Jaleco (Magical John, Pizza Pop, etc)
			mapper_write = mapper18_write;
			mapper_map   = mapper18_map;
			mapper_cycle = mapper18_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 210:
		case 19: { // Namco n106/163
			psg_area_read  = mapper19_read;
			psg_area_write = mapper19_write; // 4020 - 5fff
			mapper_write = mapper19_write; // 8000 - ffff
			mapper_map   = mapper19_map;
			mapper_cycle = mapper19_cycle;
			nes_ext_sound_cb = mapper19_mixer;
			mapper_scan_cb = mapper19_scan;
			mapper19_mapper210 = (mappernum == 210);

			//waygan 2  0x8aa76b0b   wagyan 3 0x7aff2388  splatterh 0xb7eeb48b
			if (Cart.Crc == 0x8aa76b0b || Cart.Crc == 0x7aff2388 || Cart.Crc == 0xb7eeb48b) {
				mapper19_mapper210 = 2; // namco 340 (210 submapper 2)
				bprintf(0, _T("Namco 210 submapper 2 (Namco 340)\n"));
			}

			mapper19_reset();
			mapper_map();
			retval = 0;
			break;
		}

		case 21: { // vrc2
			mapper_write = mapper21_write;
			mapper_map   = vrc2vrc4_map;
			mapper_cycle = vrc2vrc4_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 22: { // vrc2
			mapper_write = mapper22_write;
			mapper_map   = vrc2vrc4_map;
			vrc2and4_ines22 = 1;
			mapper_map();
			retval = 0;
			break;
		}

		case 23: { // vrc4
			mapper_write = mapper23_write;
			mapper_map   = vrc2vrc4_map;
			mapper_cycle = vrc2vrc4_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 25: { // vrc4
			mapper_write = mapper22_write;
			mapper_map   = vrc2vrc4_map;
			mapper_cycle = vrc2vrc4_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 24: { // vrc6
			mapper_write = vrc6_write;
			mapper_map   = vrc6_map;
			mapper_cycle = vrc6_cycle;
			vrc6_reset();
			mapper_map();
			nes_ext_sound_cb = vrc6_mixer; // vrc6 sound
			mapper_scan_cb = vrc6_scan;
			retval = 0;
			break;
		}

		case 26: { // vrc6
			mapper_write = vrc6_write;
			mapper_map   = vrc6_map;
			mapper_cycle = vrc6_cycle;
			mapper26 = 1;
			vrc6_reset();
			mapper_map();
			nes_ext_sound_cb = vrc6_mixer; // vrc6 sound
			mapper_scan_cb = vrc6_scan;
			retval = 0;
			break;
		}

		case 228: { // Action 52, Cheetah Men II
			mapper_write = mapper228_write;
			psg_area_write = mapper228_psg_write;
			psg_area_read = mapper228_psg_read;
			mapper_map   = mapper228_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 91: { // Older JY Company/Hummer
			cart_exp_write = mapper91_write; // 6000 - 7fff
			mapper_map = mapper91_map;
			mapper_scanline = mapper91_scanline;
			mapper_map();
			retval = 0;
			break;
		}

		case 17: { // Front Far East FFE SMC (Type 17)
			psg_area_write = mapper17_write; // 4020 - 5fff
			mapper_map = mapper17_map;
			mapper_cycle = mapper17_cycle;
			mapper_regs[0] = ~3;
			mapper_regs[1] = ~2;
			mapper_regs[2] = ~1;
			mapper_regs[3] = ~0;
			mapper_map();
			retval = 0;
			break;
		}

		case 211: mapper90_211 = 1;
		case 209: mapper90_209 = 1;
		case 90: { // JY Company
			mapper_write = mapper90_write;       // 8000 - ffff
			psg_area_write = mapper90_psg_write; // 4020 - 5fff
			psg_area_read = mapper90_psg_read;
			cart_exp_write = mapper90_exp_write; // 6000 - 7fff
			cart_exp_read = mapper90_exp_read;

			mapper_scanline = mapper90_scanline;
			mapper_cycle = mapper90_cycle;
			mapper_ppu_clockall = mapper90_ppu_clock; // irq
			if (mapper90_209) {
				read_nt = &mapper90_ntread;
			}

			mapper_map   = mapper90_map;

			// mapper defaults
			mapper90_mul0 = 0xff;
			mapper90_mul1 = 0xff;
			mapper90_accu = 0xff;
			mapper90_testreg = 0xff;
			mapper90_chrlatch(0) = 0;
			mapper90_chrlatch(1) = 4;

			mapper_map();
			retval = 0;
			break;
		}

		case 28: { // Action53 / Home-brew multicart
			mapper_write   = mapper28_write;
			psg_area_write = mapper28_write;
			mapper_map     = mapper28_map;
			//mapper_map()
			NESMode |= MAPPER_NOCLEAR; // preserve mapper regs on soft-reset
			if (RESETMode == RESET_POWER)
				mapper_map_prg(16, 1, -1);
			retval = 0;
			break;
		}

		case 30: { // UNROM-512
			mapper_write = mapper30_write;
			mapper_map   = mapper30_map;
			switch (rom[6] & (1|8)) {
				case 0: set_mirroring(HORIZONTAL); break;
				case 1: set_mirroring(VERTICAL); break;
				case 8: set_mirroring(SINGLE_LOW); mapper30_mirroring_en = 1; break;
				case 9: set_mirroring(FOUR_SCREEN); break;
			}
			mapper_map();
			retval = 0;
			break;
		}

		case 33: { // Taito TC0190
			mapper_write = mapper33_write;
			mapper_map   = mapper33_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 36: { // TXC (Policeman, Strike Wolf)
			psg_area_read  = mapper36_read;
			psg_area_write = mapper36_write;
			mapper_write   = mapper36_write;
			mapper_map     = mapper36_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 42: { // Ai Senshi Nicol FDS Conversion
			mapper_write = mapper42_write;
			mapper_map   = mapper42_map;
			mapper_cycle = mapper42_cycle;
			cart_exp_read = mapper42_exp_read;
			mapper_map();
			retval = 0;
			break;
		}

		case 108: { // Meikyuu Jiin Dababa FDS conversion
			mapper_write = mapper108_write;
			psg_area_write = mapper108_write; // just to silent debug msgs
			mapper_map   = mapper108_map;
			cart_exp_read = mapper108_exp_read;
			mapper_map();
			retval = 0;
			break;
		}

		case 111: { // Cheapocabra (GTROM)
			cart_exp_write = mapper111_write;
			psg_area_write = mapper111_write;
			mapper_map   = mapper111_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 120: { // Tobidase Daisakusen FDS Conversion - garbage! (game crashes after lv1)
			psg_area_write = mapper120_write;
			mapper_map   = mapper120_map;
			cart_exp_read = mapper120_exp_read;
			mapper_map();
			retval = 0;
			break;
		}

		case 48: { // Taito TC0690
			mapper_write = mapper33_write;
			mapper_map   = mapper33_map;
			mapper_scanline = mapper33_scanline;
			mapper48 = 1;
			mapper48_flintstones = (Cart.Crc == 0x12f38740);
			bprintf(0, _T("mapper48 - flintstones? %x\n"), mapper48_flintstones);
			mapper_map();
			retval = 0;
			break;
		}

		case 185: { // CNROM + prot
			mapper_write = mapper185_write;
			mapper_map   = mapper185_map;
			memset(Cart.CHRRam, 0xff, Cart.CHRRamSize);
			mapper_map_prg(32, 0, 0);
			mapper_map();
			retval = 0;
			break;
		}

		case 32: { // Irem G101
			mapper_write = mapper32_write;
			mapper_map = mapper32_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 34: { // BNROM/NINA-001
			mapper_write = mapper34_write;
			cart_exp_write = mapper34_write;
			mapper_map = mapper34_map;
			mapper_regs[1] = 0; // defaults to retain compatibility with BNROM
			mapper_regs[2] = 1;
			mapper_map();
			retval = 0;
			break;
		}

		case 65: { // Irem h3001
			mapper_write = mapper65_write;
			mapper_map = mapper65_map;
			mapper_cycle = mapper65_cycle;
			mapper_regs[2] = -2;
			mapper_map_prg( 8, 3, -1);
			mapper_map();
			retval = 0;
			break;
		}

		case 67: { // Sunsoft-3
			mapper_write = mapper67_write;
			mapper_map = mapper67_map;
			mapper_cycle = mapper67_cycle;
			mapper_map();
			retval = 0;
			break;
		}

		case 68: { // Sunsoft-4
			mapper_write = mapper68_write;
			mapper_map = mapper68_map;
			mapper_cycle = mapper68_cycle;
			cart_exp_read = mapper68_exp_read;
			cart_exp_write = mapper68_exp_write;
			mapper_map();
			retval = 0;
			break;
		}

		case 64: { // Tengen RAMBO-1
			mapper_write = mapper64_write;
			mapper_map   = mapper64_map;
			mapper_scanline = mapper64_scanline;
			mapper_cycle = mapper64_cycle;
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 69: { // sunsoft
			mapper_write = mapper69_write;
			mapper_map   = mapper69_map;
			mapper_cycle = mapper69_cycle;
			cart_exp_read = mapper69_exp_read;
			cart_exp_write = mapper69_exp_write;
			mapper_map();
			AY8910Reset(0);
			retval = 0;
			break;
		}

		case 70: {
			mapper_write = mapper70_write; // 8000 - ffff
			mapper_map   = mapper70_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 71: { // camerica (codemasters) misc
			mapper_write = mapper71_write; // 8000 - ffff
			mapper_map   = mapper71_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 73: { // Konami vrc3 (Salamander)
			mapper_write = mapper73_write;
			mapper_map   = mapper73_map;
			mapper_cycle = mapper73_cycle;
		    mapper_map();
			retval = 0;
			break;
		}

		case 75: { // Konami vrc1
			mapper_write = mapper75_write; // 8000 - ffff
			mapper_map   = mapper75_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 85: { // Konami vrc7 (Lagrange Point, Tiny Toons Adventures 2 (Japan))
			mapper_write = vrc7_write;
			mapper_map   = vrc7_map;
			mapper_cycle = vrc7_cycle;
			BurnYM2413VRC7Reset();
		    mapper_map();
			retval = 0;
			break;
		}

		case 232: { // camerica (codemasters) quattro
			mapper_write = mapper232_write;
			mapper_map   = mapper232_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 78: { // Jaleco JF16
			mapper_write = mapper78_write; // 8000 - ffff
			mapper_map   = mapper78_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 80: { // Taito x1005
			cart_exp_write = mapper207_write;
			cart_exp_read  = mapper207_read;
			mapper_map     = mapper207_map;
			mapper_scan_cb = mapper207_scan;
			memset(mapper80_ram, 0, sizeof(mapper80_ram));
			mapper207_adv_mirror = 0;
			mapper_map();
			retval = 0;
			break;
		}

		case 81: {
			mapper_write = mapper81_write;
			mapper_map   = mapper81_map;
			mapper_map();
			retval = 0;
			break;
		}

		case 207: { // Taito x1005 w/advanced mirroring
			cart_exp_write = mapper207_write;
			cart_exp_read  = mapper207_read;
			mapper_map     = mapper207_map;
			mapper_scan_cb = mapper207_scan;
			memset(mapper80_ram, 0, sizeof(mapper80_ram));
			mapper207_adv_mirror = 1;
			mapper_map();
			retval = 0;
			break;
		}

		case 87: {
			cart_exp_write = mapper87_write; // 6000 - 7fff!
			mapper_map   = mapper87_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 88: {
			mapper_write = mapper88_write;
			mapper_map   = mapper88_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 154: {
			mapper_write = mapper88_write;
			mapper_map   = mapper88_map;
			mapper154 = 1;
		    mapper_map();
			retval = 0;
			break;
		}

		case 101: {
			cart_exp_write = mapper101_write; // 6000 - 7fff!
			mapper_map   = mapper101_map;
			mapper_regs[0] = 0xff;
		    mapper_map();
			retval = 0;
			break;
		}

		case 107: {
			mapper_write = mapper107_write;
			mapper_map   = mapper107_map;
			mapper_regs[0] = 0xff;
		    mapper_map();
			retval = 0;
			break;
		}

		case 38: {
			cart_exp_write = mapper38_write; // 8000 - ffff
			mapper_map   = mapper38_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 66: {
			mapper_write = mapper140_write; // 8000 - ffff
			mapper_map   = mapper140_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 11: {
			mapper_write = mapper11_write; // 8000 - ffff
			mapper_map   = mapper11_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 144: { // similar to 11
			mapper_write = mapper144_write; // 8000 - ffff
			mapper_map   = mapper11_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 112: {
			mapper_write = mapper112_write;
			psg_area_write = mapper112_write;
			mapper_map   = mapper112_map;
			mapper_map_prg(8, 2, -2);
			mapper_map_prg(8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 146: // Sachen Metal Fighter
		case 79: { // NINA-03 / NINA-06
			psg_area_write = mapper79_write; // 4020 - 5fff
			mapper_map     = mapper79_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 113: { // NINA-03 / NINA-06 extended / Mojon Twins Multicart
			psg_area_write = mapper113_write; // 4020 - 5fff
			mapper_map     = mapper113_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 137:
		case 138:
		case 139:
		case 141: { // Sachen 8259A/B/C/D
			psg_area_write = mapper8259_write; // 4020 - 5fff -and-
			cart_exp_write = mapper8259_write; // 6000 - 7fff
			mapper_map     = mapper8259_map;
			mapper_regs[0x1f] = mappernum;
			mapper_map_chr( 8, 0, 0);
		    mapper_map();
			retval = 0;
			break;
		}

		case 150: { // Sachen LS
			psg_area_write = mapper150_write; // 4020 - 5fff -and-
			psg_area_read  = mapper150_read;
			cart_exp_write = mapper150_write; // 6000 - 7fff
			cart_exp_read  = mapper150_read;
			mapper_map     = mapper150_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 162: { // (Waixing) Zelda - Triforce of the Gods
			psg_area_write = mapper162_write; // 4020 - 5fff
			mapper_map     = mapper162_map;
			mapper162_reg(0) = 0x03;
			mapper162_reg(3) = 0x07;
			mapper_map_chr( 8, 0, 0);
		    mapper_map();
			retval = 0;
			break;
		}

		case 163: { // Final Fantasy VII (NJ063)
			psg_area_write = mapper163_write; // 4020 - 5fff
			psg_area_read  = mapper163_read;
			mapper_map     = mapper163_map;
			mapper_ppu_clock = mapper163_cycle;
		    mapper_map();
			retval = 0;
			break;
		}

		case 140: {
			cart_exp_write = mapper140_write; // 6000 - 7fff!
			mapper_map     = mapper140_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 180: { // crazy climber
			mapper_write = mapper180_write;
			mapper_map   = mapper180_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 184: {
			cart_exp_write = mapper184_write; // 6000 - 7fff!
			mapper_map     = mapper184_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 152: {
			mapper_write = mapper152_write;
			mapper_map   = mapper152_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 156: {
			mapper_write = mapper156_write;
			mapper_map   = mapper156_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 89: {
			mapper_write = mapper89_write;
			mapper_map   = mapper89_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 93: { // Sunsoft-1/2: Fantasy Zone, Shanghai
			mapper_write = mapper93_write;
			mapper_map   = mapper93_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 94: { // Senjou no Ookami (Commando)
			mapper_write = mapper94_write;
			mapper_map   = mapper94_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 97: { // Kaiketsu Yanchamaru (Kid Niki)
			mapper_write = mapper97_write;
			mapper_map   = mapper97_map;
		    mapper_map();
			retval = 0;
			break;
		}

		case 206: // mmc3 w/no irq (Tengen mimic-1, Namcot 118)
		case 4: { // mmc3
			mapper_write = mapper04_write;
			mapper_map   = mapper04_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!

			// default mmc3 regs:
			// chr
			mapper_regs[0] = 0;
			mapper_regs[1] = 2;
			mapper_regs[2] = 4;
			mapper_regs[3] = 5;
			mapper_regs[4] = 6;
			mapper_regs[5] = 7;
			// prg
			mapper_regs[6] = 0;
			mapper_regs[7] = 1;

			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 303: { // Haradius Zero (mmc3 + 2xflash)
			mapper_prg_read = mapper303_read;
			mapper_write    = mapper303_write;
			mapper_scan_cb_nvram = mapper303_scan;

			mapper_map      = mapper04_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror  = Cart.Mirroring;

			mapper_map_prg( 8, 3, -1);
			mapper_map();
			retval = 0;
			break;
		}

		case 12: { // mmc3 variant (dbz 5, mk4, umk3)
			psg_area_write = mapper12_write;
			psg_area_read = mapper12_read;
			mapper_write = mapper04_write;
			mapper_map   = mapper04_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 76: { // mmc3 variant (digital devil m.)
			mapper_write = mapper04_write;
			mapper_map   = mapper76_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring;
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 95: { // mmc3 variant (dragon buster)
			mapper_write = mapper95_write;
			mapper_map   = mapper95_map;
			mapper_scanline = mapper04_scanline;
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 115: { // mmc3 variant (thunderbolt 2)
			mapper_write = mapper04_write;

			psg_area_write = mapper115_write;
			psg_area_read = mapper115_read;
			cart_exp_write = mapper115_write;
			cart_exp_read = mapper115_read;

			mapper_map   = mapper115_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!
		    mapper_map();
			retval = 0;
			break;
		}

		case 116: { // Somari (AV Girl Fighter)
			mapper_write = mapper116_write;
			psg_area_write = mapper116_write;
			cart_exp_write = mapper116_write;
			mapper_map   = mapper116_map;
			mapper_scanline = mapper116_mmc3_scanline;
			mapper116_defaults();
		    mapper_map();
			retval = 0;
			break;
		}

		case 14: { // Gouder SL-1632 (Samurai Spirits)
			mapper_write = mapper116_write;
			psg_area_write = mapper116_write;
			cart_exp_write = mapper116_write;
			mapper_map   = mapper14_map; // difference from 116
			mapper_scanline = mapper116_mmc3_scanline;
			mapper116_defaults();
		    mapper_map();
			retval = 0;
			break;
		}

		case 118: { // mmc3-derivative TKSROM/TLSROM (Alien Syndrome, Armadillo)
			mapper_write = mapper04_write;
			mapper_map   = mapper118_map;
			mapper_scanline = mapper04_scanline;
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 119: { // mmc3-derivative w/char ram+rom & ability to map both into diff. banks! (pin-bot, high speed)
			mapper_write = mapper04_write;
			mapper_map   = mapper119_map;
			mapper_scanline = mapper04_scanline;
			mapper_set_chrtype(MEM_RAM);
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}

		case 189: { // mmc3-derivative
			psg_area_write = mapper189_write; // 4020 - 5fff
			cart_exp_write = mapper189_write; // 6000 - 7fff
			mapper_write = mapper04_write;    // 8000 - ffff
			mapper_map   = mapper189_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!
		    mapper_map();
			retval = 0;
			break;
		}

		case 262: { // mmc3-derivative (Street Heroes)
			psg_area_write = mapper262_write;
			psg_area_read = mapper262_read;
			mapper_write = mapper04_write;
			mapper_map   = mapper262_map;
			mapper_scanline = mapper04_scanline;
			mapper4_mirror = Cart.Mirroring; // wagyan land doesn't set the mapper bit!
			mapper_map_prg( 8, 3, -1);
		    mapper_map();
			retval = 0;
			break;
		}
	}
	return retval;
}

static void mapper_irq(INT32 delay_cyc)
{
	if (delay_cyc == 0) { // irq now
		M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
	} else { // irq later (after 'cyc' m2 cycles)
		mapper_irq_exec = delay_cyc;
	}
}

static void mapper_run()
{
	if (mapper_irq_exec) {
		mapper_irq_exec--;
		if (mapper_irq_exec == 0) {
			M6502SetIRQLine(0, CPU_IRQSTATUS_ACK);
		}
	}

	if (mapper_cycle) mapper_cycle();
}

// --------[ end Mappers

static UINT32 nes_palette[2][0x40] = {
{ // dink-fami
	0x5c5c5c, 0x00237e, 0x100e9e, 0x33009b, 0x520074, 0x630037, 0x610500, 0x4c1700, 0x2b2c00, 0x093e00, 0x004700, 0x004505, 0x003744, 0x000000, 0x000000, 0x000000,
	0xa7a7a7, 0x1157d6, 0x3b38ff, 0x6d21fe, 0x9916c8, 0xb11973, 0xae2a1a, 0x904500, 0x626400, 0x307d00, 0x0a8a00, 0x00862a, 0x007385, 0x000000, 0x000000, 0x000000,
	0xfeffff, 0x5ba9ff, 0x8a88ff, 0xc16eff, 0xef61ff, 0xff65c7, 0xff7866, 0xe6961b, 0xb4b700, 0x7fd200, 0x53e027, 0x3cdc79, 0x3fc7da, 0x454545, 0x000000, 0x000000,
	0xfeffff, 0xbadbff, 0xcecdff, 0xe5c2ff, 0xf8bcff, 0xffbee7, 0xffc6be, 0xf4d39c, 0xe0e18a, 0xc9ec8c, 0xb6f2a2, 0xacf0c6, 0xade8ef, 0xb0b0b0, 0x000000, 0x000000,
},
{ // rgb
	0x7c7c7c, 0x0000fc, 0x0000bc, 0x4428bc, 0x940084, 0xa80020, 0xa81000, 0x881400,	0x503000, 0x007800, 0x006800, 0x005800, 0x004058, 0x000000, 0x000000, 0x000000,
	0xbcbcbc, 0x0078f8, 0x0058f8, 0x6844fc, 0xd800cc, 0xe40058, 0xf83800, 0xe45c10,	0xac7c00, 0x00b800, 0x00a800, 0x00a844, 0x008888, 0x000000, 0x000000, 0x000000,
	0xf8f8f8, 0x3cbcfc, 0x6888fc, 0x9878f8, 0xf878f8, 0xf85898, 0xf87858, 0xfca044,	0xf8b800, 0xb8f818, 0x58d854, 0x58f898, 0x00e8d8, 0x787878, 0x000000, 0x000000,
	0xfcfcfc, 0xa4e4fc, 0xb8b8f8, 0xd8b8f8, 0xf8b8f8, 0xf8a4c0, 0xf0d0b0, 0xfce0a8,	0xf8d878, 0xd8f878, 0xb8f8b8, 0xb8f8d8, 0x00fcfc, 0xf8d8f8, 0x000000, 0x000000
} };

#define DIP_PALETTE (DrvDips[2] & 1)
static INT32 last_palette;

static UINT8 GetAvgBrightness(INT32 x, INT32 y)
{
	UINT32 rgbcolor = nes_palette[DIP_PALETTE][pTransDraw[(y) * 256 + x] & 0x3f];
	INT32 t = (rgbcolor & 0xff) + ((rgbcolor >> 8) & 0xff) + ((rgbcolor >> 16) & 0xff);

	return t / 3;
}

static INT32 nes_frame_cycles;
static float nes_ppu_cyc_mult;
static INT32 prerender_line; // ntsc 261, pal 311

static INT32 ppu_frame;
static INT32 ppu_framecycles;
static INT32 mirroring;
static UINT8 nt_ram[0x400 * 4]; // nt (2k std. / 4k four-screen [2k expansion is usually on cart])
static UINT8 pal_ram[0x20];   // palette ram
static UINT8 oam_ram[0x100];  // oamram (sprite)
static OAMBUF oam[0x40];   // "to be drawn" oam
static INT32 oam_cnt;
static OAMBUF oam2[0x40];  // secondary oam (after eval)
static INT32 oam2_cnt;
static UINT8 oamAddr;

static UINT16 vAddr, tAddr; // vram addr (loopy-v), temp vram addr (loopy-t)
static UINT16 vAddr_update;
static INT32 vAddr_update_delay;
static UINT8 fine_x; // fine-x (scroll)

static PPUCTRL ctrl;
static PPUMASK mask;
static PPUSTATUS status;
#define RENDERING (mask.reg & 0x18) // show bg (0x08) + show sprites (0x10)
static INT32 sprite_height; // 8, 16. set on PPUCTRL
static INT32 v_incr; // 1, 32. set on PPUCTRL
static INT32 bgtable_start; // 0, 0x1000. set on PPUCTRL
static INT32 sprtable_start; // 0, 0x1000. set on PPUCTRL
static INT32 ppu_pal_mask; // 0x3f (color) 0x30 (gray). set on PPUMASK
static INT32 ppu_pal_emphasis; // emphasis palette index

static INT32 ppu_no_nmi_this_frame;
static INT32 ppu_odd;
static INT32 ppu_startup; // Start-up delay
static UINT16 ppu_bus_address;
static INT32 ppu_over; // #cycles we've run over/under to be compensated for on next frame
static UINT8 ppu_dbus; // ppu data-bus
static UINT8 ppu_buffer; // VRAM read buffer.
static INT32 ppu_runextranmi;

static UINT8 write_latch;
static UINT8 nt_byte;

static UINT8 bgL, bgH;
static UINT16 bg_shiftL, bg_shiftH;

static UINT8 at_byte;
static UINT8 at_shiftL, at_shiftH;
static UINT8 at_latchL, at_latchH;

#define get_bit(x, n) (((x) >> (n)) & 1)
static UINT8 bitrev_table[0x100];

// Nametable mirroring/mapping
static UINT8 *NTMap[4];
static INT32 NTType[4];

static void nametable_map(INT32 nt, INT32 ntbank)
{
	NTMap[nt & 3] = &nt_ram[0x400 * (ntbank & 3)];
	NTType[nt & 3] = MEM_RAM;
}

static void nametable_mapraw(INT32 nt, UINT8 *ntraw, UINT8 type)
{
	NTMap[nt & 3] = ntraw;
	NTType[nt & 3] = type;
}

static void nametable_mapall(INT32 ntbank0, INT32 ntbank1, INT32 ntbank2, INT32 ntbank3)
{
	nametable_map(0, ntbank0);
	nametable_map(1, ntbank1);
	nametable_map(2, ntbank2);
	nametable_map(3, ntbank3);
}

static void set_mirroring(INT32 mode)
{
	switch (mode)
    {
        case HORIZONTAL:  nametable_mapall(0, 0, 1, 1); break;
        case VERTICAL:	  nametable_mapall(0, 1, 0, 1); break;
		case SINGLE_LOW:  nametable_mapall(0, 0, 0, 0); break;
		case SINGLE_HIGH: nametable_mapall(1, 1, 1, 1); break;
		case FOUR_SCREEN: nametable_mapall(0, 1, 2, 3); break;
		case SACHEN:	  nametable_mapall(0, 1, 1, 1); break;
	}

	mirroring = mode;
}

static UINT8 read_nt_int(UINT16 address)
{
	return NTMap[(address & 0xfff) >> 10][address & 0x3ff];
}

static void write_nt_int(UINT16 address, UINT8 data)
{
	if (NTType[(address & 0xfff) >> 10] == MEM_RAM)
		NTMap[(address & 0xfff) >> 10][address & 0x3ff] = data;
}

// The PPU-Bus
static UINT8 ppu_bus_read(UINT16 address)
{
	if (ppu_startup) return 0; // ignore reads until line 261 on first frame.

	if (address >= 0x0000 && address <= 0x1fff)
		return mapper_chr_read(address);

	if (address >= 0x2000 && address <= 0x3eff)
		return read_nt(address);

	if (address >= 0x3f00 && address <= 0x3fff) {
		if ((address & 0x13) == 0x10) // 0x10, 0x14, 0x18, 0x1c -> 0x00, 0x04, 0x08, 0x0c
			address &= ~0x10;
		return pal_ram[address & 0x1f] & ppu_pal_mask;
	}

	return 0;
}

static void ppu_bus_write(UINT16 address, UINT8 data)
{
	if (address >= 0x0000 && address <= 0x1fff)
		mapper_chr_write(address, data);

	if (address >= 0x2000 && address <= 0x3eff)
		write_nt(address, data);

	if (address >= 0x3f00 && address <= 0x3fff) {
		if ((address & 0x13) == 0x10)
			address &= ~0x10;
		pal_ram[address & 0x1f] = data;
	}
}

static void h_scroll();
static void v_scroll();

static void ppu_inc_v_addr()
{
	if (RENDERING && (scanline < 241 || scanline == prerender_line)) {
		// no updates while rendering!
		//bprintf(0, _T("no updates while rendering! frame: %X  scanline: %d    pixel: %d\n"), nCurrentFrame, scanline, pixel);

		// Burai Fighter (U) statusbar fix
		h_scroll();
		v_scroll();
		return;
	}

	vAddr = (vAddr + v_incr) & 0x7fff;
	ppu_bus_address = vAddr & 0x3fff;
}

static UINT8 ppu_read(UINT16 reg)
{
	reg &= 7;

	switch (reg)
	{
		case 2: // PPUSTATUS
			if (scanline == 241 && pixel < 3) {
				// re: https://wiki.nesdev.com/w/index.php/PPU_frame_timing
				// PPUSTATUS read: 1 cycle before vbl (pixel == 0) skips status & nmi
				// on or 1 pixel after (pixel == 1 or 2) returns vblank status, but skips nmi
				//bprintf(0, _T("PPUSTATUS: CANCEL NMI.  scanline %d  pixel %d\n"), scanline, pixel);
				ppu_no_nmi_this_frame = 1;
				if (pixel == 0) status.bit.VBlank = 0;
			}
			ppu_dbus = (ppu_dbus & 0x1f) | status.reg;
			status.bit.VBlank = 0;
			write_latch = 0;
			//bprintf(0, _T("PPUSTATUS - frame: %d   scanline: %d     pixel: %d    res: %X   PC: %X\n"), nCurrentFrame, scanline, pixel, ppu_dbus, M6502GetPrevPC(-1));
			break;

		case 4: // OAMDATA
			if (RENDERING && scanline < 241) {
				// quirks to keep micromachines happy
				if (pixel >= 1 && pixel <= 64) {
					ppu_dbus = 0xff;
					break;
				}
				if (pixel >= 65 && pixel <= 256) {
					ppu_dbus = 0x00;
					break;
				}

				if (pixel >= 257 && pixel <= 320) {
					ppu_dbus = oam2[((pixel & 0xff) >> 3)].x;
					break;
				}

				if ((pixel >= 321 && pixel <= 340) || pixel == 0) {
					ppu_dbus = oam2[0].y;
					break;
				}
			}
			ppu_dbus = oam_ram[oamAddr];
			break;

		case 7: // PPUDATA
			if ((ppu_bus_address & 0x3fff) < 0x3f00) {
				// vram buffer delay
				ppu_dbus = ppu_buffer;
				ppu_buffer = ppu_bus_read(ppu_bus_address);
			} else {
				// palette has no buffer delay, buffer gets stuffed with vram though (ppu quirk)
				ppu_dbus = ppu_bus_read(ppu_bus_address);
				ppu_buffer = ppu_bus_read(ppu_bus_address - 0x1000);
			}

			ppu_inc_v_addr();
			break;

	}

	return ppu_dbus;
}

static void ppu_write(UINT16 reg, UINT8 data)
{
	reg &= 7;

	ppu_dbus = data;

	switch (reg)
	{
		case 0: // PPUCTRL
			if (ppu_startup) return; // ignore writes until line 261
			if (data & 0x80) {
				if ((~ctrl.reg & 0x80) && status.bit.VBlank) {
					//--Note: If NMI is fired here, it will break:
					//Bram Stokers Dracula, Galaxy 5000, GLUK The Thunder aka Thunder Warrior
					//Solution: Delay NMI by 1 cpu-operation.
					//Note: Dragon Power needs a slightly longer delay? (GetPC instead of GetPPC) Otherwise scrolling goes bad
					bprintf(0, _T("PPUCTRL: toggle-nmi-arm! scanline %d  pixel %d    frame: %d   PPC %X\n"), scanline, pixel, nCurrentFrame, M6502GetPrevPC(-1));
					ppu_runextranmi = M6502GetPC(-1);
				}
			} else {
				//bprintf(0, _T("PPUCTRL: %X  cancel-nmi?  scanline %d  pixel %d   frame %d\n"), data, scanline, pixel, nCurrentFrame);
			}

			ctrl.reg = data;
			//bprintf(0, _T("PPUCTRL reg: %X   scanline %d  pixel %d   frame %d\n"), ctrl.reg, scanline, pixel, nCurrentFrame);
			tAddr = (tAddr & 0x73ff) | ((data & 0x3) << 10);

			sprite_height = ctrl.bit.sprsize ? 16 : 8;
			v_incr = ctrl.bit.incr ? 32 : 1;
			bgtable_start = ctrl.bit.bgtab * 0x1000;
			sprtable_start = ctrl.bit.sprtab * 0x1000;
			break;

		case 1: // PPUMASK
			if (ppu_startup) return; // ignore writes until line 261
			if ((data & 0x18)==0 && RENDERING) { // rendering ON -> OFF assert ppu_bus_address
				ppu_bus_address = vAddr & 0x3fff;
			}
			mask.reg = data;
			ppu_pal_emphasis = (data >> 5) * 0x40;
			if (NESMode & IS_PAL) {
				ppu_pal_emphasis = (((data & 0x80) | ((data & 0x40) >> 1) | ((data & 0x20) << 1)) >> 5) * 0x40;
			}
			//bprintf(0, _T("mask   %x\n"), data);
			ppu_pal_mask = (mask.bit.gray ? 0x30 : 0x3f);
			break;

		case 3: // OAMADDR
			oamAddr = data;
			break;

		case 4: // OAMDATA
			//bprintf(0, _T("Frame %04x:  OAMDATA[%X]  %X      scanline %d  pixel %d\n"), nCurrentFrame, oamAddr, data, scanline, pixel);
			if (RENDERING && (scanline < 241 || scanline == prerender_line)) {
				bprintf(0, _T("write OAM prohibited.  scanline %d\n"), scanline);
				return;
			}
			if ((oamAddr & 3) == 2) data &= 0xe3; // attr mask
			oam_ram[oamAddr] = data;
			oamAddr = (oamAddr + 1) & 0xff;
			break;

		case 5: // PPUSCROLL
			if (ppu_startup) return; // ignore writes until line 261

			if (!write_latch) {      // First write.
				fine_x = data & 7;
				tAddr = (tAddr & 0x7fe0) | ((data & 0xf8) >> 3);
			} else  {                // Second write.
				tAddr = (tAddr & 0x0c1f) | ((data & 0xf8) << 2) | ((data & 0x07) << 12);
			}
			write_latch = !write_latch;
			break;
		case 6: // PPUADDR
			if (ppu_startup) return; // ignore writes until line 261
			if (!write_latch) {		// First write.
				tAddr = (tAddr & 0x00ff) | ((data & 0x3f) << 8);
			} else {                // Second write.
				tAddr = (tAddr & 0x7f00) | data;
				vAddr_update = tAddr;
				vAddr_update_delay = 2;
			}
			write_latch = !write_latch;
			break;
		case 7: // PPUDATA
			ppu_bus_write(ppu_bus_address, data);

			ppu_inc_v_addr();
			break;
	}
}

static void h_scroll()
{
	if (RENDERING) {
		if ((vAddr & 0x1f) == 0x1f) {
			vAddr ^= 0x041f;
		} else {
			vAddr++;
		}
	}
}

static void v_scroll()
{
	if (RENDERING) {
		if ((vAddr & 0x7000) == 0x7000) {
			switch (vAddr & 0x03e0) {
				case 0x03a0: vAddr ^= 0x7800 | 0x03a0; break;
				case 0x03e0: vAddr &= ~0x73e0; break;
				default: vAddr = (vAddr & ~0x7000) + 0x0020;
			}
		} else {
			vAddr += 0x1000;
		}
	}
}

static void reload_shifters()
{
	at_latchL = at_byte & 1;
    at_latchH = (at_byte & 2) >> 1;

	bg_shiftL = (bg_shiftL & 0xff00) | bgL;
    bg_shiftH = (bg_shiftH & 0xff00) | bgH;
}

static void evaluate_sprites()
{
	INT32 cur_line = (scanline == prerender_line) ? -1 : scanline;
	oam2_cnt = 0;

    for (INT32 i = oamAddr; i < 0x40; i++) {
        INT32 line = cur_line - oam_ram[(i << 2) + 0];
        if (line >= 0 && line < sprite_height) {
			if (oam2_cnt == 8) {
				status.bit.spovrf = 1;
				if (~DrvDips[0] & 1) // DIP option: disable sprite limit
					break;
			}
			oam2[oam2_cnt].idx  = i;
            oam2[oam2_cnt].y    = oam_ram[(i << 2) + 0];
            oam2[oam2_cnt].tile = oam_ram[(i << 2) + 1];
            oam2[oam2_cnt].attr = oam_ram[(i << 2) + 2];
			oam2[oam2_cnt++].x  = oam_ram[(i << 2) + 3];
		}
	}
}

static void load_sprites()
{
	oam_cnt = 0;
	for (INT32 i = 0; i < oam2_cnt; i++) {
		oam[oam_cnt++] = oam2[i];

		UINT32 sprY = scanline - oam[i].y;
		sprY = (oam[i].attr & 0x80) ? ~sprY : sprY; // invert y-bits if y-flipped

		if (sprite_height == 16) {
			ppu_bus_address = ((oam[i].tile & 1) * 0x1000) + ((oam[i].tile & 0xfe) * 16)
				+ ((sprY & 8) << 1) + (sprY & 7);
		} else {
			ppu_bus_address = sprtable_start + (oam[i].tile * 16) + (sprY & 7);
		}

		oam[i].tileL = ppu_bus_read(ppu_bus_address);
		ppu_bus_address += 8;
		oam[i].tileH = ppu_bus_read(ppu_bus_address);

		if (mapper_ppu_clock) mapper_ppu_clock(ppu_bus_address); // Punch-Out (mapper 9) latches on high-byte read (+8)

		if (oam[i].attr & 0x40) { // reverse bits if x-flipped
			oam[i].tileL = bitrev_table[oam[i].tileL];
			oam[i].tileH = bitrev_table[oam[i].tileH];
		}
    }
}

static void draw_and_shift()
{
    INT32 x = pixel - 2;

	if (scanline < 240 && x >= 0 && x < 256) {
		UINT8 pix = 0;
		UINT8 sprPal = 0;
		UINT8 sprPri = 0;
		UINT8 spr = 0;
		UINT16 eff_x = 0;

		if (!RENDERING && (vAddr & 0x3f00) == 0x3f00) {
			// https://wiki.nesdev.com/w/index.php/PPU_palettes "The background palette hack"
			pix = vAddr & 0x1f;
		}

		if (mask.bit.bg && (mask.bit.bgLeft == 0 && x < 8) == 0) {
			pix = (get_bit(bg_shiftH, 15 - fine_x) << 1) |
				  (get_bit(bg_shiftL, 15 - fine_x) << 0);

			if (pix) {
				pix |= ((get_bit(at_shiftH, 7 - fine_x) << 1) |
						(get_bit(at_shiftL, 7 - fine_x) << 0)) << 2;
			}
        }

		if (mask.bit.spr && (mask.bit.sprLeft == 0 && x < 8) == 0) {
			for (INT32 i = oam_cnt - 1; i >= 0; i--) {
				if (oam[i].idx == 0xff) // no sprite
					continue;

				eff_x = x - oam[i].x;
				if (eff_x >= 8) // sprite out of view
					continue;

				spr = (get_bit(oam[i].tileH, 7 - eff_x) << 1) |
					  (get_bit(oam[i].tileL, 7 - eff_x) << 0);

				if (spr == 0) // transparent sprite, ignore
					continue;

				if (oam[i].idx == 0 && pix && x != 0xff) {
					status.bit.sp0hit = 1;
				}

                spr |= (oam[i].attr & 3) << 2; // add color (attr), 2bpp shift
                sprPal = spr + 0x10; // add sprite color palette-offset
                sprPri = ~oam[i].attr & 0x20; // sprite over bg?
            }
		}

		if (~nBurnLayer & 1) pix = 0; // if tile layer disabled, clear pixel.
        if (sprPal && (pix == 0 || sprPri) && nSpriteEnable & 1) pix = sprPal;

		if (NESMode & SHOW_OVERSCAN) {
			pTransDraw[scanline * 256 + x] = ppu_bus_read(0x3f00 + pix) | ppu_pal_emphasis;
		} else {
			if (scanline >= 8 && scanline < (nScreenHeight + 8)/*232*/)
				pTransDraw[(scanline - 8) * 256 + x] = ppu_bus_read(0x3f00 + pix) | ppu_pal_emphasis;
		}
    }

	bg_shiftL <<= 1;
	bg_shiftH <<= 1;
    at_shiftL = (at_shiftL << 1) | at_latchL;
    at_shiftH = (at_shiftH << 1) | at_latchH;
}

static void scanlinestate(INT32 state)
{
	if (state == VBLANK) {

		switch (pixel) {
			case 1: // enter VBlank
				ppu_bus_address = vAddr & 0x3fff;
				status.bit.VBlank = 1;
				ppu_startup = 0;

				if (NESMode & ALT_TIMING2) {
					if (ctrl.bit.nmi && ppu_no_nmi_this_frame == 0) {
						//bprintf(0, _T("nmi @ frame %d  scanline %d  pixel %d  PPUCTRL %x\n"), nCurrentFrame, scanline, pixel, ctrl.reg);
						M6502SetIRQLine(CPU_IRQLINE_NMI, CPU_IRQSTATUS_AUTO);
					}
					ppu_no_nmi_this_frame = 0;
				}
				break;

			case (6 * 3):
				// 6 CPU-cycles later, do nmi.  fixes boot w/b-wings, bad dudes, gunnac
				// crap on screen with marble madness. (Also passes blargg timing tests)
				if (~NESMode & ALT_TIMING2) {
					if (ctrl.bit.nmi && ppu_no_nmi_this_frame == 0) {
						//bprintf(0, _T("nmi @ frame %d  scanline %d  pixel %d  PPUCTRL %x\n"), nCurrentFrame, scanline, pixel, ctrl.reg);
						M6502SetIRQLine(CPU_IRQLINE_NMI, CPU_IRQSTATUS_AUTO);
					}
					ppu_no_nmi_this_frame = 0;
				}
				break;
		}
	}
	else if (state == VISIBLE || state == PRERENDER) {

		// Sprites
		switch (pixel) {
			case 1:
				memset(oam2, 0xff, sizeof(oam2)); // clear secondary oam

				if (state == PRERENDER) {
					status.reg &= 0x1f; // clear vbl, spr0 hit & overflow bits

					if (oamAddr > 7) { // 2c02 oamram corruption (Huge Insect, Tatakai no Banka pre-revA)
						for (INT32 i = 0; i < 8; i++) {
							oam_ram[i] = oam_ram[(oamAddr & 0xf8) + i];
						}
					}
				}
				break;
			case 257:
				evaluate_sprites();
				if (RENDERING) {
					oamAddr = 0;
				}
				break;
			case 321:
				if (RENDERING) {
					load_sprites();
				} else {
					oam_cnt = 0;
				}
				break;
		}

		// Tiles
		if ( (pixel >= 1 && pixel <= 257) || (pixel >= 321 && pixel <= 337) ) {
			if (pixel != 1) draw_and_shift();
			if (RENDERING) {
				switch (pixel & 7) {
					case 1:
						ppu_bus_address = 0x2000 | (vAddr & 0x0fff); // nametable address
						reload_shifters();
						if (pixel == 257 && RENDERING) {
							// copy horizontal bits from loopy-T to loopy-V
							vAddr = (vAddr & ~0x041f) | (tAddr & 0x041f);
						}
						break;
					case 2:
						nt_byte = ppu_bus_read(ppu_bus_address);
						break;
					case 3:
						ppu_bus_address = 0x23c0 | (vAddr & 0x0c00) | ((vAddr >> 4) & 0x38) | ((vAddr >> 2) & 7); // attribute address
						break;
					case 4:
						at_byte = ppu_bus_read(ppu_bus_address);
						if (vAddr & (1 << 6) /* coarse Y & 2 */) at_byte >>= 4;
						if (vAddr & (1 << 1) /* coarse X & 2 */) at_byte >>= 2;
						break;
					case 5:
						ppu_bus_address = bgtable_start + (nt_byte * 16) + (vAddr >> 12); // background address
						break;
					case 6:
						bgL = ppu_bus_read(ppu_bus_address);
						break;
					case 7:
						ppu_bus_address = 8 + bgtable_start + (nt_byte * 16) + (vAddr >> 12); // background address
						break;
					case 0:
						bgH = ppu_bus_read(ppu_bus_address);
						if (pixel == 256)
							v_scroll();
						else
							h_scroll();
						break;
				}
			}
		}

		if (state == PRERENDER && pixel >= 280 && pixel <= 304 && RENDERING) {
			// copy vertical bits from loopy-T to loopy-V
			vAddr = (vAddr & ~0x7be0) | (tAddr & 0x7be0);
		}

		if (pixel >= 337 && pixel <= 340 && RENDERING) {
			// dummy nt fetches
			ppu_bus_address = 0x2000 | (vAddr & 0x0fff);
		}

		// Signal scanline to mapper: (+18 gets rid of jitter in Kirby, Yume Penguin Monogatari, Klax, ...)
		// Note: was 20, but caused occasional flickering in the rasterized water in "Kira Kira Star Night GOLD" (maybe the others in the series, too?)
		if (pixel == (260+18) /*&& RENDERING*/ && mapper_scanline) mapper_scanline();
		// Mapper callback w/ppu_bus_address - used by mmc2/mmc4 (mapper9/mapper10)
		if (mapper_ppu_clock) mapper_ppu_clock(ppu_bus_address); // ppu callback (visible lines)
    }
}

void ppu_cycle()
{
	ppu_framecycles++;	// keep track of cycles/frame since reset
	pixel++; 			// cycle (pixel)/line

	if (pixel > 340) {
		pixel = 0;
		scanline++;
		if (scanline > prerender_line) {
			scanline = 0;
			ppu_frame++;
			ppu_odd ^= 1;
			if (RENDERING && ppu_odd) {
				// this real hw hack was added to the ppu back in the day for better
				// rf/composite output.  since it messes with current cpu:ppu
				// sync. and isn't really necessary w/emu, let's disable it. -dink
				//pixel++;
				//ppu_framecycles++;
			}
        }
	}

	if (scanline >= 0 && scanline <= 239)
		scanlinestate(VISIBLE);
	else if (scanline == 241)
		scanlinestate(VBLANK);
	else if (scanline == prerender_line) {
		ppu_runextranmi = 0;
		scanlinestate(PRERENDER);
	}

	if (mapper_ppu_clockall) mapper_ppu_clockall(ppu_bus_address); // ppu callback (every line)

	if (vAddr_update_delay > 0) {
		vAddr_update_delay--;
		if (vAddr_update_delay == 0) {
			vAddr = vAddr_update;
			if ((scanline >= 240 && scanline < prerender_line) || !RENDERING) {
				ppu_bus_address = vAddr & 0x3fff;
			}
		}
	}

	if (ppu_runextranmi && ppu_runextranmi != M6502GetPrevPC(-1)) {
		// Delay by 1 cpu-op when nmi toggled via PPUCTRL during vblank.
		// Bram Stokers Dracula, Galaxy 5000, GLUK The Thunder aka Thunder Warrior
		// will get stuck in the game's nmi handler if the "sta $nmistatus_addr"
		// op gets executed after the nmi:
		// "lda #$80, sta PPUCTRL, sta $nmistatus_addr"
		bprintf(0, _T("toggle-nmi @ scanline %d  pixel %d  PC %X\n"), scanline, pixel, M6502GetPrevPC(-1));
		M6502SetIRQLine(CPU_IRQLINE_NMI, CPU_IRQSTATUS_AUTO);
		//M6502Stall(7); // nmi takes 7 cycles (old m6502 core needs this)
		ppu_runextranmi = 0;
	}
}

static void ppu_run(INT32 cyc)
{
	while (ppu_over < 0) { // we didn't run enough cycles last frame, catch-up!
		cyc++;
		ppu_over++;
	}

	if ((NESMode & IS_PAL) && (mega_cyc_counter % 5) == 0) {
		cyc++;
	}

	while (cyc) {
		if (ppu_over > 0) { // if we're over some cycles on the start of next:
			ppu_over--;     // frame - idle them away
		} else {
			ppu_cycle();
		}
		cyc--;
	}
}

static void ppu_scan(INT32 nAction)
{
	SCAN_VAR(mirroring);
	SCAN_VAR(scanline);
	SCAN_VAR(pixel);
	SCAN_VAR(ppu_frame);

	SCAN_VAR(vAddr);
	SCAN_VAR(tAddr);
	SCAN_VAR(vAddr_update);
	SCAN_VAR(vAddr_update_delay);

	SCAN_VAR(fine_x);
	SCAN_VAR(oamAddr);

	SCAN_VAR(ctrl);
	SCAN_VAR(mask);
	SCAN_VAR(status);

	SCAN_VAR(sprite_height);
	SCAN_VAR(v_incr);
	SCAN_VAR(bgtable_start);
	SCAN_VAR(sprtable_start);
	SCAN_VAR(ppu_pal_mask);

	SCAN_VAR(ppu_no_nmi_this_frame);
	SCAN_VAR(ppu_odd);
	SCAN_VAR(ppu_startup);
	SCAN_VAR(ppu_bus_address);
	SCAN_VAR(ppu_over);
	SCAN_VAR(ppu_dbus);
	SCAN_VAR(ppu_buffer);
	SCAN_VAR(write_latch);

	SCAN_VAR(nt_byte);
	SCAN_VAR(bgL);
	SCAN_VAR(bgH);
	SCAN_VAR(bg_shiftL);
	SCAN_VAR(bg_shiftH);

	SCAN_VAR(at_byte);
	SCAN_VAR(at_shiftL);
	SCAN_VAR(at_shiftH);
	SCAN_VAR(at_latchL);
	SCAN_VAR(at_latchH);

	// ppu-ram
	SCAN_VAR(nt_ram);
	SCAN_VAR(pal_ram);
	SCAN_VAR(oam_ram);

	if (nAction & ACB_WRITE) {
		set_mirroring(mirroring);
	}
}

static void ppu_reset()
{
	mmc5_nt_ram = &nt_ram[0];

	// start at (around) vblank to remove 1 full frame of input lag
	scanline = 239; // line on titlescreen of micromachines if starts on 240
	pixel = 0;
	ppu_frame = 0;

	vAddr = 0;
	tAddr = 0;
	fine_x = 0;
	oamAddr = 0;

	ctrl.reg = 0;
	mask.reg = 0;
	status.reg = 0;

	mmc5_mask = &mask.reg; // point to ppumask register
	mmc5_ctrl = &ctrl.reg; // point to ppuctrl register

	sprite_height = 8;
	v_incr = 1;
	bgtable_start = 0;
	sprtable_start = 0;
	ppu_pal_mask = 0x3f;
	ppu_pal_emphasis = 0;

	ppu_odd = 0;
	ppu_no_nmi_this_frame = 0;
	ppu_startup = 0; // ignore some reads/writes until ppu "warms up" *disabled, see next line*
	// ppu_startup breaks Magic John - but why??  A: games writes ppu registers too early..
	ppu_bus_address = 0;
	ppu_over = 0; // cycles overrun this frame
	ppu_dbus = 0;
	ppu_buffer = 0;

	write_latch = 0;

	ppu_framecycles = 0; // total ran cycles this frame

    memset(nt_ram, 0xff, sizeof(nt_ram));
    memset(pal_ram, 0x00, sizeof(pal_ram));
    memset(oam_ram, 0xff, sizeof(oam_ram));

	memset(oam, 0xff, sizeof(oam));
	memset(oam2, 0xff, sizeof(oam2));

	nt_byte = 0;
	bgL = bgH = 0;
	bg_shiftL = bg_shiftH = 0;

	at_byte = 0;
	at_shiftL = at_shiftH = 0;
	at_latchL = at_latchH = 0;

	oam_cnt = 0;
	oam2_cnt = 0;

	last_palette = DIP_PALETTE;
}

static void ppu_init(INT32 is_pal)
{
	read_nt = &read_nt_int;
	write_nt = &write_nt_int;
	for (INT32 i = 0; i < 0x100; i++)
		bitrev_table[i] = BITSWAP08(i, 0, 1, 2, 3, 4, 5, 6, 7);

	if (is_pal) {
		nes_frame_cycles = 33248; // pal
		prerender_line = 311;
		nes_ppu_cyc_mult = 3.2;
	} else {
		nes_frame_cycles = 29781; // ntsc (default)
		nes_ppu_cyc_mult = 3.0;
		prerender_line = 261;
	}

	ppu_reset();
}

static UINT8 nes_read_zapper()
{
	if (RENDERING == 0 || scanline < 8 || scanline > 240)
		return 0x08;

	INT32 in_y = ((BurnGunReturnY(0) * 224) / 255);
	INT32 in_x = BurnGunReturnX(0);
	INT32 real_sl = scanline - 8;

	for (INT32 yy = in_y - 2; yy < in_y + 2; yy++) {
		if (yy < real_sl-8 || yy > real_sl) continue;

		for (INT32 xx = in_x - 2; xx < in_x + 2; xx++) {
			if (yy == real_sl && xx >= pixel) break;
			if (GetAvgBrightness(xx, yy) > 0x88) { // makes time travel possible
				return 0x00;
			}
		}
	}

	return 0x08;
}

static UINT8 nes_read_joy(INT32 port)
{
	UINT8 ret = 0;

	if ((NESMode & USE_ZAPPER) && port == 1) {
		// Zapper on second port (0x4017)
		ret = nes_read_zapper(); // light sensor
		ret |= (ZapperFire) ? 0x10 : 0x00; // trigger
		if (ZapperReload) {
			ret = 0x18;
		}

	} else {
		if (NESMode & USE_HORI4P) {
			UINT8 a = (JoyShifter[port&1] >> 0) & 0xff; // joy 1,3
			UINT8 b = (JoyShifter[port&1] >> 8) & 0xff; // joy 2,4
			ret  = (a & 1);			// joy 1,2 d0 (FC hardwired controllers)
			ret |= (b & 1) << 1;	// joy 3,4 d1 (controllers connected to HORI4P)
			a >>= 1;                // shift next bit
			b >>= 1;                // "
			a |= 0x80;              // feed empty high bit (active low)
			b |= 0x80;              // "

			JoyShifter[port&1] = (a & 0xff) | ((b << 8) & 0xff00);
		} else {
			ret = JoyShifter[port&1]&1;
			JoyShifter[port&1] >>= 1;   // shift next
			JoyShifter[port&1] |= 0x80000000; // feed high bit so reads past our data return high
		}

		// MIC "blow" hack (read on 1p controller address, yet, mic is on 2p controller!)
		if (NESMode & IS_FDS && port == 0) {
			if (DrvInputs[1] & ((1<<2) | (1<<3))) { // check if 2p select or start pressed. note: famicom 2p controller doesn't have start or select, we use it as a MIC button.
				ret |= 4;
			}
		}

		// note, some games (f.ex: paperboy) relies on the open bus bits in the high nibble
		ret |= cpu_open_bus & 0xf0;
	}

	return ret;
}

static UINT8 psg_io_read(UINT16 address)
{
	if (address == 0x4016 || address == 0x4017)
	{	// Gamepad 1 & 2 / Zapper
		return nes_read_joy(address & 1);
	}

	if (address == 0x4015)
	{
		return nesapuRead(0, address & 0x1f, cpu_open_bus);
	}

	if (address >= 0x4020 && psg_area_read) {
		return psg_area_read(address);
	}

	//bprintf(0, _T("psg_io_read(unmapped) %X\n"), address);

	return cpu_open_bus;
}

static void psg_io_write(UINT16 address, UINT8 data)
{
	if (address == 0x4016)
	{
		if ((JoyStrobe & 1) && (~data & 1)) {
			switch (NESMode & (USE_4SCORE | USE_HORI4P)) {
				case USE_4SCORE:
					// "Four Score" 4-player adapter (NES / World)
					JoyShifter[0] = DrvInputs[0] | (DrvInputs[2] << 8) | (bitrev_table[0x10] << 16);
					JoyShifter[1] = DrvInputs[1] | (DrvInputs[3] << 8) | (bitrev_table[0x20] << 16);
					break;
				case USE_HORI4P: // mode 2
					// "Hori" 4-player adapter Mode 4 (Japanese / Famicom)
					JoyShifter[0] = DrvInputs[0] | (DrvInputs[2] << 8);
					JoyShifter[1] = DrvInputs[1] | (DrvInputs[3] << 8);
					break;
				default:
					// standard nes controllers
					JoyShifter[0] = DrvInputs[0] | 0xffffff00;
					JoyShifter[1] = DrvInputs[1] | 0xffffff00;
					break;
			}
		}
		JoyStrobe = data;
		return;
	}

	if (address >= 0x4000 && address <= 0x4017)
	{
		nesapuWrite(0, address & 0x1f, data);
		return;
	}

	if (address >= 0x4020 && psg_area_write) {
		psg_area_write(address, data);
		return;
	}

	bprintf(0, _T("psg_io_write(unmapped) %X    %x\n"), address, data);
}

static UINT8 cpu_ram_read(UINT16 address)
{
	return NES_CPU_RAM[address & 0x7FF];
}

static void cpu_ram_write(UINT16 address, UINT8 data)
{
	NES_CPU_RAM[address & 0x7FF] = data;
}

static UINT8 prg_ram_read(UINT16 address) // expansion ram / rom on the cartridge.
{
	if (cart_exp_read) {
		return cart_exp_read(address);
	}

	// some games get irritated (low g-man, mapper 7: double dragon, battle toads) if prg ram exists
	return (NESMode & NO_WORKRAM) ? cpu_open_bus : Cart.WorkRAM[address & Cart.WorkRAMMask];
}

static void prg_ram_write(UINT16 address, UINT8 data)
{
	if (cart_exp_write) {
		cart_exp_write_abort = 0;
		cart_exp_write(address, data);
		if (cart_exp_write_abort) return;
		// fallthrough if abort wasn't set! :)
	}

	if (NESMode & NO_WORKRAM) return;
	Cart.WorkRAM[address & Cart.WorkRAMMask] = data;
}

// cheat system
static UINT8 gg_bit(UINT8 g)
{
	const UINT8 gg_str[0x11] = "APZLGITYEOXUKSVN";

	for (UINT8 i = 0; i < 0x10; i++) {
		if (g == gg_str[i]) {
			return i;
		}
	}
	return 0;
}

static INT32 gg_decode(char *gg_code, UINT16 &address, UINT8 &value, INT32 &compare)
{
	INT32 type = strlen(gg_code);

	if (type != 6 && type != 8) {
		// bad code!
		return 1;
	}

	UINT8 str_bits[8];

	for (UINT8 i = 0; i < type; i++) {
		str_bits[i] = gg_bit(gg_code[i]);
	}

	address = 0x8000 | ((str_bits[1] & 8) << 4) | ((str_bits[2] & 7) << 4) | ((str_bits[3] & 7) << 12) | ((str_bits[3] & 8) << 0) | ((str_bits[4] & 7) << 0) | ((str_bits[4] & 8) << 8) | ((str_bits[5] & 7) << 8);
	value = ((str_bits[0] & 7) << 0) | ((str_bits[0] & 8) << 4) | ((str_bits[1] & 7) << 4);
	compare = -1;

	switch (type) {
		case 6: {
			value |= ((str_bits[5] & 8) << 0);
			break;
		}
		case 8: {
			value |= ((str_bits[7] & 8) << 0);
			compare = ((str_bits[5] & 8) << 0) | ((str_bits[6] & 7) << 0) | ((str_bits[6] & 8) << 4) | ((str_bits[7] & 7) << 4);
			break;
		}
	}

	return 0;
}

const INT32 cheat_MAX = 0x10;
static INT32 cheats_active = 0;

struct cheat_struct {
	char code[0x10]; // gamegenie code
	UINT16 address;
	UINT8 value;
	INT32 compare; // -1, compare disabled.
};

static cheat_struct cheats[cheat_MAX];

void nes_add_cheat(char *code) // 6/8 character game genie codes allowed
{
	UINT16 address;
	UINT8 value;
	INT32 compare;

	if (!gg_decode(code, address, value, compare)) {
		strncpy(cheats[cheats_active].code, code, 9);
		cheats[cheats_active].address = address;
		cheats[cheats_active].value = value;
		cheats[cheats_active].compare = compare;
		bprintf(0, _T("cheat #%d (%S) added.  (%x, %x, %d)\n"), cheats_active, cheats[cheats_active].code, address, value, compare);
		cheats_active++;
	} else {
		bprintf(0, _T("nes cheat engine: bad GameGenie code %S\n"), code);
	}
}

void nes_remove_cheat(char *code)
{
	cheat_struct cheat_temp[cheat_MAX];
	INT32 temp_num = 0;

	for (INT32 i = 0; i < cheats_active; i++) {
		if (strcmp(code, cheats[i].code) != 0) {
			memcpy(&cheat_temp[temp_num], &cheats[i], sizeof(cheat_struct));
			temp_num++;
		} else {
			bprintf(0, _T("cheat %S disabled.\n"), cheats[i].code);
		}
	}

	memcpy(cheats, cheat_temp, sizeof(cheats));
	cheats_active = temp_num;
}

static inline UINT8 cheat_check(UINT16 address, UINT8 value)
{
	for (INT32 i = 0; i < cheats_active; i++) {
		if (cheats[i].address == address && (cheats[i].compare == -1 || cheats[i].compare == value)) {
		 //   bprintf(0, _T("."));
			return cheats[i].value;
		}
	}

	return value;
}


static UINT8 cpu_bus_read(UINT16 address)
{
	UINT8 ret = 0;
	switch (address & 0xe000)
	{
		case 0x0000:  //    0 - 1fff
			ret = cpu_ram_read(address); break;
		case 0x2000:  // 2000 - 3fff
			ret = ppu_read(address); break;
		case 0x4000:  // 4000 - 5fff
			ret = psg_io_read(address); break;
		case 0x6000:  // 6000 - 7fff
			ret = prg_ram_read(address); break;
		default:      // 8000 - ffff
			ret = mapper_prg_read(address); break;
	}

	ret = cheat_check(address, ret);

	cpu_open_bus = ret;

	return ret;
}


#define DEBUG_DMA 0

static void cpu_bus_write(UINT16 address, UINT8 data)
{
	cpu_open_bus = data;

	if (address == 0x4014) { // OAM_DMA
#if DEBUG_DMA
		bprintf(0, _T("DMA, tcyc %d   scanline: %d    pixel: %d\n"), M6502TotalCycles(), scanline, pixel);
#endif
		UINT8 byte = 0;
		M6502Stall(1);		// 1 dummy cyc

		if (M6502TotalCycles() & 1) {
			M6502Stall(1);	// 1 if starts on odd cycle
		}

		for (INT32 i = 0; i < 256; i++) {
			M6502Stall(1); // 1 for cpu_read
			byte = cpu_bus_read(data * 0x100 + i);
			M6502Stall(1); // 1 for cpu_write to ppu
			ppu_write(0x2004, byte);
		}
#if DEBUG_DMA
		bprintf(0, _T("..end-DMA, tcyc %d   scanline: %d    pixel: %d\n"), M6502TotalCycles(), scanline, pixel);
#endif
		return;
	}

	switch (address & 0xe000) {
		case 0x0000: //    0 - 1fff
			cpu_ram_write(address, data); break;
		case 0x2000: // 2000 - 3fff
			ppu_write(address, data); break;
		case 0x4000: // 4000 - 5fff
			psg_io_write(address, data); break;
		case 0x6000: // 6000 - 7fff
			prg_ram_write(address, data); break;
		default:     // 8000 - ffff
			if (mapper_write) mapper_write(address, data);
			return;
	}
}

static INT32 DrvDoReset()
{
	if (RESETMode == RESET_POWER) {
		memset(NES_CPU_RAM, 0x00, 0x800);  // only cleared @ power-on
	}
	M6502Open(0);
	M6502Reset();
	M6502Close();
	nesapuReset();
	//	if (RESETMode == RESET_POWER) {
	// Nebs 'n Debs thinks we're in 50hz/PAL mode on reset if we don't reset PPU here..
	// note: This is actually due to a bug in the game
		ppu_reset();
	//	}
	mapper_reset();

	JoyShifter[0] = JoyShifter[1] = 0xffffffff;
	JoyStrobe = 0;

	cyc_counter = 0;
	mega_cyc_counter = 0;

	{
		INT32 nAspectX, nAspectY;
		BurnDrvGetAspect(&nAspectX, &nAspectY);
		if (DrvDips[1] & 1) { // 4:3
			if (nAspectX != 4) {
				bprintf(0, _T("*  NES: Changing to 4:3 aspect\n"));
				BurnDrvSetAspect(4, 3);
				Reinitialise();
			}
		} else { // no aspect ratio handling
			if (nAspectX != SCREEN_WIDTH) {
				bprintf(0, _T("*  NES: Changing to pixel aspect\n"));
				BurnDrvSetAspect(SCREEN_WIDTH, SCREEN_HEIGHT);
				Reinitialise();
			}
		}
	}

	return 0;
}

static INT32 NESInit()
{
	GenericTilesInit();

	NES_CPU_RAM = (UINT8*)BurnMalloc(0x800);

	cheats_active = 0;

	struct BurnRomInfo ri;
	char *romname = NULL;
	BurnDrvGetRomInfo(&ri, 0);
	BurnDrvGetRomName(&romname, 0, 0);
	UINT32 length = ri.nLen;
	UINT32 is_FDS = (strstr(romname, ".fds") != 0);
	bprintf(0, _T("ROM Name: %S\n"), romname);
	rom = BurnMalloc((length < 0x100000) ? 0x100000 : length);

	bprintf(0, _T("ROM Length: %d\n"), ri.nLen);
	RESETMode = RESET_POWER;

	if (is_FDS) {
		if (BurnLoadRom(rom + 0x6000, 0x80, 1)) return 1; // bios @ 0xe000
		if (fds_load(rom, ri.nLen, ri.nCrc)) return 1;
	} else {
		if (BurnLoadRom(rom, 0, 1)) return 1;
		if (cartridge_load(rom, ri.nLen, ri.nCrc)) return 1;
	}

	BurnSetRefreshRate((NESMode & IS_PAL) ? 50.00 : 60.00);

	M6502Init(0, TYPE_N2A03);
	M6502Open(0);
	M6502SetWriteHandler(cpu_bus_write);
	M6502SetReadHandler(cpu_bus_read);
	M6502SetReadOpArgHandler(cpu_bus_read);
	M6502SetReadOpHandler(cpu_bus_read);
	M6502Close();

	ppu_init((NESMode & IS_PAL) ? 1 : 0);

	if (NESMode & IS_PAL) {
		nesapuInitPal(0, 1798773, 0);
	} else {
		nesapuInit(0, 1798773, 0);
	}
	nesapuSetAllRoutes(0, 2.40, BURN_SND_ROUTE_BOTH);

	DrvDoReset();

	return 0;
}

static INT32 NES4ScoreInit()
{
	INT32 rc = NESInit();

	NESMode |= USE_4SCORE;

	bprintf(0, _T("*  FourScore 4 Player device.\n"));

	return rc;
}

static INT32 NESHori4pInit()
{
	INT32 rc = NESInit();

	NESMode |= USE_HORI4P;

	bprintf(0, _T("*  Hori 4 Player device (Mode 2).\n"));

	return rc;
}

static INT32 NESZapperInit()
{
	INT32 rc = NESInit();

	NESMode |= USE_ZAPPER;
	BurnGunInit(1, true);

	bprintf(0, _T("*  Zapper on Port #2.\n"));

	return rc;
}

// stereo effector (delay line+comb filter)
struct ms_ring {
	INT16 *l;
	INT16 *r;
	INT32 ring_size;
	INT32 inpos;
	INT32 outpos_l;
	INT32 outpos_r;

	void exit() {
		if (ring_size || l || r) {
			BurnFree(l);
			BurnFree(r);
			ring_size = 0;
			bprintf(0, _T("ms_ring exited.\n"));
		}
	}

	void init() {
		ring_size = (INT32)((double)14 / 1000 * nBurnSoundRate); // 14ms ring buffer

		l = (INT16*)BurnMalloc(ring_size * sizeof(INT16));
		r = (INT16*)BurnMalloc(ring_size * sizeof(INT16));

		for (INT32 i = 0; i < ring_size; i++) {
			write(0, 0);
		}
		inpos = 0; // position in @ beginning of ring, out @ end
		outpos_l = 1;
		outpos_r = 1;
		bprintf(0, _T("ms_ring initted (%d entry ringbuffer)\n"), ring_size);
	}

	INT32 needs_init() {
		return (l == NULL || r == NULL || ring_size == 0);
	}

	void write(INT16 sam_l, INT16 sam_r) {
		l[inpos] = sam_l;
		r[inpos] = sam_r;
		inpos = (inpos + 1) % ring_size;
	}

	INT16 read_r() {
		INT16 temp = r[outpos_r];
		outpos_r = (outpos_r + 1) % ring_size;
		return temp;
	}

	INT16 read_l() {
		INT16 temp = l[outpos_l];
		outpos_l = (outpos_l + 1) % ring_size;
		return temp;
	}

	void process(INT16 *buffer, INT32 samples) {
		if (needs_init()) {
			init();
		}

		for (INT32 i = 0; i < samples; i++) {
			write(buffer[i * 2 + 0], buffer[i * 2 + 1]);
			INT16 sam_mask = ((read_l(), read_r()) / 2) * 0.75;
			INT16 sam_now = (buffer[i * 2 + 0] + buffer[i * 2 + 1]) / 2;
			buffer[i * 2 + 0] = sam_now + sam_mask;
			buffer[i * 2 + 1] = sam_now - sam_mask;
		}
	}
};

static ms_ring ms_delay;

// end stereoeffect

static INT32 NESExit()
{
	GenericTilesExit();
	M6502Exit();
	nesapuExit();

	// Mapper EXT-hardware exits
	if (Cart.Mapper == 69) { // SunSoft fme-7 (5b) audio expansion - ay8910. mapper69
		AY8910Exit(0);
	}

	if (Cart.Mapper == 85) { // VRC7 audio expansion - ym2413
		BurnYM2413Exit();
	}

	if (Cart.FDSMode) {
		// exit saver:
		TCHAR fds_saves_patch[MAX_PATH];
		_stprintf(fds_saves_patch, _T("%s.ips"), BurnDrvGetText(DRV_NAME));
		INT32 ips = ips_make(Cart.FDSDiskRawOrig, Cart.FDSDiskRaw, Cart.FDSDiskRawSize, szAppEEPROMPath, fds_saves_patch);
		bprintf(0, _T("* FDS DISK patch: "));
		switch (ips) {
			case  0: bprintf(0, _T("Saved.\n")); break;
			case -1: bprintf(0, _T("Can't Save (File I/O Error).\n")); break;
			case -2: bprintf(0, _T("No Change.\n")); break;
		}

		BurnFree(Cart.FDSDiskRaw);
		BurnFree(Cart.FDSDiskRawOrig);

		BurnLEDExit();
	}

	if (NESMode & USE_ZAPPER)
		BurnGunExit();

	BurnFree(rom);
	BurnFree(NES_CPU_RAM);
	BurnFree(Cart.WorkRAM);
	BurnFree(Cart.CHRRam);

	ms_delay.exit();

	return 0;
}

static UINT32 EmphRGB(INT32 pal_idx, UINT8 type)
{
/*
	dink ppu color emphasis notes
	ppumask	register	(reg >> 5) (r/g swapped for PAL)

	red		20		 	1
	green	40			2
	blue	80			4

	// All possible combinations, palette offset, (reg >> 5)
	normal	0-3f		0
	r		40-7f		1
	g		80-bf		2
	rg		c0-ff		3
	b		100 - 13f	4
	rb		140 - 17f	5
	gb		180 - 1bf	6
	rgb		1c0 - 1ff	7
*/

	UINT64 er = 1.0 * (1 << 16);
	UINT64 eg = 1.0 * (1 << 16);
	UINT64 eb = 1.0 * (1 << 16);
	UINT32 EMPH   = 1.2 * (1 << 16);
	UINT32 DEEMPH = 0.8 * (1 << 16);
	if (type & 1) {
		er = (er *   EMPH) >> 16;
		eg = (eg * DEEMPH) >> 16;
		eb = (eb * DEEMPH) >> 16;
	}
	if (type & 2) {
		er = (er * DEEMPH) >> 16;
		eg = (eg *   EMPH) >> 16;
		eb = (eb * DEEMPH) >> 16;
	}
	if (type & 4) {
		er = (er * DEEMPH) >> 16;
		eg = (eg * DEEMPH) >> 16;
		eb = (eb *   EMPH) >> 16;
	}

	UINT32 r = (((nes_palette[DIP_PALETTE][pal_idx & 0x3f] >> 16) & 0xff) * er) >> 16;
	if (r > 0xff) r = 0xff;
	UINT32 g = (((nes_palette[DIP_PALETTE][pal_idx & 0x3f] >>  8) & 0xff) * eg) >> 16;
	if (g > 0xff) g = 0xff;
	UINT32 b = (((nes_palette[DIP_PALETTE][pal_idx & 0x3f] >>  0) & 0xff) * eb) >> 16;
	if (b > 0xff) b = 0xff;

	return BurnHighCol(r, g, b, 0);
}

static void DrvCalcPalette()
{
	// Normal NES Palette
	for (INT32 i = 0; i < 0x40; i++) {
		DrvPalette[i] = BurnHighCol((nes_palette[DIP_PALETTE][i] >> 16) & 0xff, (nes_palette[DIP_PALETTE][i] >> 8) & 0xff, nes_palette[DIP_PALETTE][i] & 0xff, 0);
	}

	// Emphasized Palettes (all combinations, see comments-table in EmphRGB)
	for (INT32 i = 0x40; i < 0x200; i++) {
		DrvPalette[i] = EmphRGB(i, i >> 6);
	}

	// Palette for the FDS Swap Disk icon
	for (INT32 fadelv = 0; fadelv < 0x10; fadelv++) {
		for (INT32 i = 0x200; i < 0x210; i++) {
			DrvPalette[i + (fadelv * 0x10)] = BurnHighCol(disk_ab_pal[(i & 0xf) * 3 + 2] / (fadelv + 1), disk_ab_pal[(i & 0xf) * 3 + 1] / (fadelv + 1), disk_ab_pal[(i & 0xf) * 3 + 0] / (fadelv + 1), 0);
		}
	}
}

static INT32 NESDraw()
{
	if (NESRecalc || last_palette != DIP_PALETTE) {
		DrvCalcPalette();
		NESRecalc = 0;
		last_palette = DIP_PALETTE;
	}

	if ((NESMode & IS_FDS) && (FDSFrameAction == SWAP_DISK || FDSFrameAction == FA_FADEOUT)) {
		static INT32 fader = 0;
		if (FDSFrameAction == SWAP_DISK) fader = 0;
		if (FDSFrameAction == FA_FADEOUT && ((FDSFrameCounter%2)==0)) {
			if (fader < 0x8)
				fader++;
		}

		switch (FDSFrameDiskIcon) {
			case 0:
				RenderCustomTile(pTransDraw, 38, 18, 0, 109/*x*/, 100/*y*/, 0, 8, 0x200 + (fader * 0x10), &disk_ab[0]);
				break;
			case 1:
				RenderCustomTile(pTransDraw, 38, 18, 0, 109/*x*/, 100/*y*/, 0, 8, 0x200 + (fader * 0x10), &disk_ab[18*38]);
				break;
		}
	}

	BurnTransferCopy(DrvPalette);

	if (NESMode & USE_ZAPPER)
		BurnGunDrawTargets();

	if (NESMode & IS_FDS)
		BurnLEDRender();

	return 0;
}

static void clear_opposites(UINT8 &inpt)
{
	// some games will straight-up crash or go berzerk if up+down or left+right
	// is pressed.  This can easily happen when playing via kbd or severly
	// worn gamepad.

	if ((inpt & ( (1 << 4) | (1 << 5) )) == ((1 << 4) | (1 << 5)) )
		inpt &= ~((1 << 4) | (1 << 5)); // up + down pressed, cancel both

	if ((inpt & ( (1 << 6) | (1 << 7) )) == ((1 << 6) | (1 << 7)) )
		inpt &= ~((1 << 6) | (1 << 7)); // left + right pressed, cancel both
}

#define DEBUG_CYC 0

//#define nes_frame_cycles 29781(ntsc) 33248(pal)

INT32 nes_scanline()
{
	return scanline;
}

static INT32 NESFrame()
{
#if DEBUG_CYC
	bprintf(0, _T("enter frame:  scanline %d   pixel %d  ppucyc %d    cyc_counter %d\n"), scanline, pixel, ppu_framecycles, cyc_counter);
#endif
	if (DrvReset)
	{
		RESETMode = RESET_BUTTON;
		DrvDoReset();
	}

	{	// compile inputs!
		DrvInputs[0] = DrvInputs[1] = 0x00;
		DrvInputs[2] = DrvInputs[3] = 0x00;
		for (INT32 i = 0; i < 8; i++) {
			DrvInputs[0] ^= (NESJoy1[i] & 1) << i;
			DrvInputs[1] ^= (NESJoy2[i] & 1) << i;
			DrvInputs[2] ^= (NESJoy3[i] & 1) << i;
			DrvInputs[3] ^= (NESJoy4[i] & 1) << i;
		}

		clear_opposites(DrvInputs[0]);
		clear_opposites(DrvInputs[1]);
		clear_opposites(DrvInputs[2]);
		clear_opposites(DrvInputs[3]);

		if (NESMode & USE_ZAPPER) {
			BurnGunMakeInputs(0, ZapperX, ZapperY);
		}

		if (NESMode & IS_FDS) {
			FDS_Insert(FDSEject);

			if (~DrvDips[0] & 2) {
				FDS_FrameTicker();
				FDS_SwapSidesAuto(FDSSwap);
			} else {
				FDS_SwapSides(FDSSwap);
			}
		}
	}

	M6502Open(0);
	M6502NewFrame();
	M6502Idle(cyc_counter);
	cyc_counter = 0;
#if DEBUG_CYC
	INT32 last_ppu = ppu_framecycles;
#endif
	if (ppu_over > 0) { // idle away extra ppu cycles
		ppu_framecycles = ppu_over;
		ppu_over = 0;
	} else {
		ppu_framecycles = 0;
	}

	for (INT32 i = 0; i < nes_frame_cycles; i++)
	{
		cyc_counter ++; // frame-based
		mega_cyc_counter ++; // "since reset"-based

		nesapu_runclock(cyc_counter - 1);  // clock dmc & external sound (n163, vrc, fds)

		if ((cyc_counter - M6502TotalCycles()) > 0)	{
			M6502Run(cyc_counter - M6502TotalCycles());
			// to debug game hard-lock: uncomment and press n 4 times to get pc.
			// tip: 99.9% its either a mapper bug or game needs ALT_TIMING flag
#if defined (FBNEO_DEBUG)
			extern int counter;
			if (counter == -4) bprintf(0, _T("PC:  %x   tc: %d   cyc_ctr: %d\n"), M6502GetPC(-1), M6502TotalCycles(), cyc_counter);
#endif
		}

		mapper_run();

		INT32 p_cyc = (cyc_counter * nes_ppu_cyc_mult) - ppu_framecycles;
		if (p_cyc > 0) {
			ppu_run(p_cyc);
		}
	}

	if (NESMode & ALT_TIMING) {
		ppu_framecycles--;
	}
	if (NESMode & IS_PAL) {
		ppu_over = ppu_framecycles - 106392;
	} else {
		ppu_over = ppu_framecycles - ((~NESMode & ALT_TIMING) ? 89342 : 89343);
	}

#if DEBUG_CYC
	INT32 cc = cyc_counter;
#endif

	cyc_counter = M6502TotalCycles() - nes_frame_cycles; // the overflow of cycles for next frame to idle away

#if DEBUG_CYC
	bprintf(0, _T("6502 cycles ran: %d   cyc_counter %d   rollover: %d    ppu.over %d   ppu.framecyc %d    last_ppu %d\n"), M6502TotalCycles(), cc, cyc_counter, ppu_over, ppu_framecycles, last_ppu);
#endif

	if (pBurnSoundOut) {
		nesapuUpdate(0, pBurnSoundOut, nBurnSoundLen);
		BurnSoundDCFilter();

		// Mapper EXT-hardware sound renders
		if (Cart.Mapper == 69) { // SunSoft fme-7 (5b) audio expansion - ay8910. mapper69
			AY8910Render(pBurnSoundOut, nBurnSoundLen);
		}

		if (Cart.Mapper == 85) { // VRC7 audio expansion - YM2413
			BurnYM2413Render(pBurnSoundOut, nBurnSoundLen);
		}

		if (DrvDips[1] & 0x02) {
			ms_delay.process(pBurnSoundOut, nBurnSoundLen);
		}
	}
	M6502Close();

	if (pBurnDraw)
	{
		NESDraw();
	}
	return 0;
}

static INT32 NESScan(INT32 nAction, INT32 *pnMin)
{
	if (pnMin) {
		*pnMin = 0x029708;
	}

	if (nAction & ACB_VOLATILE) {
		M6502Scan(nAction);
		nesapuScan(nAction, pnMin);

		SCAN_VAR(cpu_open_bus);
		SCAN_VAR(cyc_counter);
		SCAN_VAR(JoyShifter);
		SCAN_VAR(JoyStrobe);

		ScanVar(NES_CPU_RAM, 0x800, "CPU Ram");
		ScanVar(Cart.WorkRAM, Cart.WorkRAMSize, "Work Ram");
		ScanVar(Cart.CHRRam, Cart.CHRRamSize, "CHR Ram");

		mapper_scan(nAction, pnMin);

		ppu_scan(nAction);

		// Mapper EXT-hardware sound scans
		if (Cart.Mapper == 69) { // SunSoft fme-7 (5b) audio expansion - ay8910. mapper69
			AY8910Scan(nAction, pnMin);
		}
		if (Cart.Mapper == 85) { // VRC7 audio expansion - YM2413
			BurnYM2413Scan(nAction, pnMin);
		}

		if (NESMode & USE_ZAPPER)
			BurnGunScan();

		if (nAction & ACB_WRITE) {
			// save for later?
		}
	}

	if (nAction & ACB_NVRAM) {
		if (Cart.BatteryBackedSRAM) {
			ScanVar(Cart.WorkRAM, Cart.WorkRAMSize, "s-ram");
		}

		if (mapper_scan_cb_nvram) {
			mapper_scan_cb_nvram();
		}
	}

	return 0;
}

static INT32 NESGetZipName(char** pszName, UINT32 i)
{
	static char szFilename[MAX_PATH];
	char* pszGameName = NULL;

	if (pszName == NULL) {
		return 1;
	}

	if (i == 0) {
		pszGameName = BurnDrvGetTextA(DRV_NAME);
	} else {
		if (i == 1 && BurnDrvGetTextA(DRV_BOARDROM)) {
			pszGameName = BurnDrvGetTextA(DRV_BOARDROM);
		} else {
			pszGameName = BurnDrvGetTextA(DRV_PARENT);
		}
	}

	if (pszGameName == NULL || i > 2) {
		*pszName = NULL;
		return 1;
	}

	// remove nes_
	memset(szFilename, 0, MAX_PATH);
	for (UINT32 j = 0; j < (strlen(pszGameName) - 4); j++) {
		szFilename[j] = pszGameName[j+4];
	}

	*pszName = szFilename;

	return 0;
}

// FDS WIP.
static struct BurnRomInfo emptyRomDesc[] = {
	{ "",                    0,          0, 0 },
};

static struct BurnRomInfo fds_fdsbiosRomDesc[] = {
    { "fdsbios.nes",     0x2000, 0x5e607dcf, BRF_BIOS }, // 0x80 - standard bios
};

STD_ROM_PICK(fds_fdsbios)
STD_ROM_FN(fds_fdsbios)

struct BurnDriver BurnDrvfds_fdsbios = {
	"fds_fdsbios", NULL, NULL, NULL, "198x",
	"FDS System BIOS\0", "BIOS only", "NES", "NES",
	NULL, NULL, NULL, NULL,
	BDF_BOARDROM, 0, HARDWARE_FDS, GBF_BIOS, 0,
	NESGetZipName, fds_fdsbiosRomInfo, fds_fdsbiosRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_fdsirqtestRomDesc[] = {
	{ "fdsirqtests.fds",          65516, 0xa5a63b30, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_fdsirqtest, fds_fdsirqtest, fds_fdsbios)
STD_ROM_FN(fds_fdsirqtest)

struct BurnDriver BurnDrvfds_fdsirqtest = {
	"fds_fdsirqtest", NULL, "fds_fdsbios", NULL, "1989?",
	"FDS IRQ Test (Sour, v7)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_fdsirqtestRomInfo, fds_fdsirqtestRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_19neunzehnjRomDesc[] = {
	{ "19 - Neunzehn (Japan).fds",          131016, 0xe6e06610, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_19neunzehnj, fds_19neunzehnj, fds_fdsbios)
STD_ROM_FN(fds_19neunzehnj)

struct BurnDriver BurnDrvfds_19neunzehnj = {
	"fds_19neunzehnj", "fds_19neunzehn", "fds_fdsbios", NULL, "1989?",
	"19 - Neunzehn (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_19neunzehnjRomInfo, fds_19neunzehnjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_19neunzehnRomDesc[] = {
	{ "19 - Neunzehn (T-Eng).fds",          131000, 0xda11432a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_19neunzehn, fds_19neunzehn, fds_fdsbios)
STD_ROM_FN(fds_19neunzehn)

struct BurnDriver BurnDrvfds_19neunzehn = {
	"fds_19neunzehn", NULL, "fds_fdsbios", NULL, "1989?",
	"19 - Neunzehn (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_19neunzehnRomInfo, fds_19neunzehnRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_adiannotsueRomDesc[] = {
	{ "Adian no Tsue (Japan).fds",          131000, 0xdcc7a8d6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_adiannotsue, fds_adiannotsue, fds_fdsbios)
STD_ROM_FN(fds_adiannotsue)

struct BurnDriver BurnDrvfds_adiannotsue = {
	"fds_adiannotsue", NULL, "fds_fdsbios", NULL, "1989?",
	"Adian no Tsue (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_adiannotsueRomInfo, fds_adiannotsueRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_aisenshinicolRomDesc[] = {
	{ "Ai Senshi Nicol (Japan).fds",          131000, 0x21a1ad6e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_aisenshinicol, fds_aisenshinicol, fds_fdsbios)
STD_ROM_FN(fds_aisenshinicol)

struct BurnDriver BurnDrvfds_aisenshinicol = {
	"fds_aisenshinicol", NULL, "fds_fdsbios", NULL, "1989?",
	"Ai Senshi Nicol (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_aisenshinicolRomInfo, fds_aisenshinicolRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_akitsunofusnokaRomDesc[] = {
	{ "Aki to Tsukasa no Fushigi no Kabe (Japan).fds",          131016, 0xd7b4a01e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_akitsunofusnoka, fds_akitsunofusnoka, fds_fdsbios)
STD_ROM_FN(fds_akitsunofusnoka)

struct BurnDriver BurnDrvfds_akitsunofusnoka = {
	"fds_akitsunofusnoka", NULL, "fds_fdsbios", NULL, "1989?",
	"Aki to Tsukasa no Fushigi no Kabe (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_akitsunofusnokaRomInfo, fds_akitsunofusnokaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_akumajoudraculajRomDesc[] = {
	{ "Akumajou Dracula (Japan).fds",          131016, 0xffed3158, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_akumajoudraculaj, fds_akumajoudraculaj, fds_fdsbios)
STD_ROM_FN(fds_akumajoudraculaj)

struct BurnDriver BurnDrvfds_akumajoudraculaj = {
	"fds_akumajoudraculaj", "fds_akumajoudracula", "fds_fdsbios", NULL, "1989?",
	"Akumajou Dracula (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_akumajoudraculajRomInfo, fds_akumajoudraculajRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_akumajoudraculaRomDesc[] = {
	{ "Akumajou Dracula (T-Eng).fds",          131016, 0xf72f22af, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_akumajoudracula, fds_akumajoudracula, fds_fdsbios)
STD_ROM_FN(fds_akumajoudracula)

struct BurnDriver BurnDrvfds_akumajoudracula = {
	"fds_akumajoudracula", NULL, "fds_fdsbios", NULL, "1989?",
	"Akumajou Dracula (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_akumajoudraculaRomInfo, fds_akumajoudraculaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_akuusenraiRomDesc[] = {
	{ "Akuu Senki Raijin (Japan).fds",          131016, 0x5117e64d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_akuusenrai, fds_akuusenrai, fds_fdsbios)
STD_ROM_FN(fds_akuusenrai)

struct BurnDriver BurnDrvfds_akuusenrai = {
	"fds_akuusenrai", NULL, "fds_fdsbios", NULL, "1989?",
	"Akuu Senki Raijin (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_akuusenraiRomInfo, fds_akuusenraiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_allnignipsupmabrRomDesc[] = {
	{ "All Night Nippon Super Mario Brothers (Japan).fds",          65500, 0x43e7fe95, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_allnignipsupmabr, fds_allnignipsupmabr, fds_fdsbios)
STD_ROM_FN(fds_allnignipsupmabr)

struct BurnDriver BurnDrvfds_allnignipsupmabr = {
	"fds_allnignipsupmabr", NULL, "fds_fdsbios", NULL, "1989?",
	"All Night Nippon Super Mario Brothers (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_allnignipsupmabrRomInfo, fds_allnignipsupmabrRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_appletowmonRomDesc[] = {
	{ "Apple Town Monogatari - Little Computer People (Japan).fds",          131016, 0xbf89dd4a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_appletowmon, fds_appletowmon, fds_fdsbios)
STD_ROM_FN(fds_appletowmon)

struct BurnDriver BurnDrvfds_appletowmon = {
	"fds_appletowmon", NULL, "fds_fdsbios", NULL, "1989?",
	"Apple Town Monogatari - Little Computer People (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_appletowmonRomInfo, fds_appletowmonRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_armananokisekijRomDesc[] = {
	{ "Armana no Kiseki (Japan).fds",          131016, 0x0bad64a5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_armananokisekij, fds_armananokisekij, fds_fdsbios)
STD_ROM_FN(fds_armananokisekij)

struct BurnDriver BurnDrvfds_armananokisekij = {
	"fds_armananokisekij", "fds_armananokiseki", "fds_fdsbios", NULL, "1989?",
	"Armana no Kiseki (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_armananokisekijRomInfo, fds_armananokisekijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_armananokisekiRomDesc[] = {
	{ "Armana no Kiseki (T-Eng).fds",          131000, 0x8b61e398, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_armananokiseki, fds_armananokiseki, fds_fdsbios)
STD_ROM_FN(fds_armananokiseki)

struct BurnDriver BurnDrvfds_armananokiseki = {
	"fds_armananokiseki", NULL, "fds_fdsbios", NULL, "1989?",
	"Armana no Kiseki (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_armananokisekiRomInfo, fds_armananokisekiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bakutpatkunRomDesc[] = {
	{ "Bakutoushi Patton-kun (Japan).fds",          131016, 0xbdb5104d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bakutpatkun, fds_bakutpatkun, fds_fdsbios)
STD_ROM_FN(fds_bakutpatkun)

struct BurnDriver BurnDrvfds_bakutpatkun = {
	"fds_bakutpatkun", NULL, "fds_fdsbios", NULL, "1989?",
	"Bakutoushi Patton-kun (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_bakutpatkunRomInfo, fds_bakutpatkunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bigchadogspiRomDesc[] = {
	{ "Big Challenge! Dogfight Spirit (Japan).fds",          131016, 0xfcdb9fc9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bigchadogspi, fds_bigchadogspi, fds_fdsbios)
STD_ROM_FN(fds_bigchadogspi)

struct BurnDriver BurnDrvfds_bigchadogspi = {
	"fds_bigchadogspi", NULL, "fds_fdsbios", NULL, "1989?",
	"Big Challenge! Dogfight Spirit (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_bigchadogspiRomInfo, fds_bigchadogspiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bigchagogoboRomDesc[] = {
	{ "Big Challenge! Go! Go! Bowling (Japan).fds",          131016, 0x11519f4a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bigchagogobo, fds_bigchagogobo, fds_fdsbios)
STD_ROM_FN(fds_bigchagogobo)

struct BurnDriver BurnDrvfds_bigchagogobo = {
	"fds_bigchagogobo", NULL, "fds_fdsbios", NULL, "1989?",
	"Big Challenge! Go! Go! Bowling (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_bigchagogoboRomInfo, fds_bigchagogoboRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bigchagunfigRomDesc[] = {
	{ "Big Challenge! Gun Fighter (Japan).fds",          131016, 0x691cb0c1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bigchagunfig, fds_bigchagunfig, fds_fdsbios)
STD_ROM_FN(fds_bigchagunfig)

struct BurnDriver BurnDrvfds_bigchagunfig = {
	"fds_bigchagunfig", NULL, "fds_fdsbios", NULL, "1989?",
	"Big Challenge! Gun Fighter (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_bigchagunfigRomInfo, fds_bigchagunfigRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bigchajuusenRomDesc[] = {
	{ "Big Challenge! Juudou Senshuken (Japan).fds",          131016, 0xa65712a7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bigchajuusen, fds_bigchajuusen, fds_fdsbios)
STD_ROM_FN(fds_bigchajuusen)

struct BurnDriver BurnDrvfds_bigchajuusen = {
	"fds_bigchajuusen", NULL, "fds_fdsbios", NULL, "1989?",
	"Big Challenge! Juudou Senshuken (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_bigchajuusenRomInfo, fds_bigchajuusenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_biomirbokupaRomDesc[] = {
	{ "Bio Miracle Bokutte Upa (Japan).fds",          131016, 0x75e44b0d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_biomirbokupa, fds_biomirbokupa, fds_fdsbios)
STD_ROM_FN(fds_biomirbokupa)

struct BurnDriver BurnDrvfds_biomirbokupa = {
	"fds_biomirbokupa", NULL, "fds_fdsbios", NULL, "1989?",
	"Bio Miracle Bokutte Upa (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_biomirbokupaRomInfo, fds_biomirbokupaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bombermanRomDesc[] = {
	{ "Bomber Man (Japan).fds",          65516, 0x4f7bee06, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bomberman, fds_bomberman, fds_fdsbios)
STD_ROM_FN(fds_bomberman)

struct BurnDriver BurnDrvfds_bomberman = {
	"fds_bomberman", NULL, "fds_fdsbios", NULL, "1989?",
	"Bomber Man (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_bombermanRomInfo, fds_bombermanRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_bubblebobbleRomDesc[] = {
	{ "Bubble Bobble (Japan).fds",          131000, 0x8c8d153e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_bubblebobble, fds_bubblebobble, fds_fdsbios)
STD_ROM_FN(fds_bubblebobble)

struct BurnDriver BurnDrvfds_bubblebobble = {
	"fds_bubblebobble", NULL, "fds_fdsbios", NULL, "1989?",
	"Bubble Bobble (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_bubblebobbleRomInfo, fds_bubblebobbleRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_burgertimeRomDesc[] = {
	{ "BurgerTime (Japan).fds",          65516, 0x44e41aba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_burgertime, fds_burgertime, fds_fdsbios)
STD_ROM_FN(fds_burgertime)

struct BurnDriver BurnDrvfds_burgertime = {
	"fds_burgertime", NULL, "fds_fdsbios", NULL, "1989?",
	"BurgerTime (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_burgertimeRomInfo, fds_burgertimeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_chitetaiordRomDesc[] = {
	{ "Chitei Tairiku Ordola (Japan).fds",          131000, 0xaacd4f64, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_chitetaiord, fds_chitetaiord, fds_fdsbios)
STD_ROM_FN(fds_chitetaiord)

struct BurnDriver BurnDrvfds_chitetaiord = {
	"fds_chitetaiord", NULL, "fds_fdsbios", NULL, "1989?",
	"Chitei Tairiku Ordola (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_chitetaiordRomInfo, fds_chitetaiordRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_cluclulandRomDesc[] = {
	{ "Clu Clu Land (Japan).fds",          65516, 0x72d0bde6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_clucluland, fds_clucluland, fds_fdsbios)
STD_ROM_FN(fds_clucluland)

struct BurnDriver BurnDrvfds_clucluland = {
	"fds_clucluland", NULL, "fds_fdsbios", NULL, "1989?",
	"Clu Clu Land (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_cluclulandRomInfo, fds_cluclulandRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_coconaworldRomDesc[] = {
	{ "Cocona World (Japan).fds",          131016, 0x0b429e6c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_coconaworld, fds_coconaworld, fds_fdsbios)
STD_ROM_FN(fds_coconaworld)

struct BurnDriver BurnDrvfds_coconaworld = {
	"fds_coconaworld", NULL, "fds_fdsbios", NULL, "1989?",
	"Cocona World (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_coconaworldRomInfo, fds_coconaworldRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dandyjRomDesc[] = {
	{ "Dandy - Zeuon no Fukkatsu (Japan).fds",          131016, 0xa2bfcbcc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dandyj, fds_dandyj, fds_fdsbios)
STD_ROM_FN(fds_dandyj)

struct BurnDriver BurnDrvfds_dandyj = {
	"fds_dandyj", "fds_dandy", "fds_fdsbios", NULL, "1989?",
	"Dandy - Zeuon no Fukkatsu (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_dandyjRomInfo, fds_dandyjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dandyRomDesc[] = {
	{ "Dandy - Zeuon no Fukkatsu (T-Eng).fds",          131000, 0xee9bdc93, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dandy, fds_dandy, fds_fdsbios)
STD_ROM_FN(fds_dandy)

struct BurnDriver BurnDrvfds_dandy = {
	"fds_dandy", NULL, "fds_fdsbios", NULL, "1989?",
	"Dandy - Zeuon no Fukkatsu (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_dandyRomInfo, fds_dandyRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_deadzonejRomDesc[] = {
	{ "Dead Zone (Japan).fds",          131000, 0xde673254, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_deadzonej, fds_deadzonej, fds_fdsbios)
STD_ROM_FN(fds_deadzonej)

struct BurnDriver BurnDrvfds_deadzonej = {
	"fds_deadzonej", "fds_deadzone", "fds_fdsbios", NULL, "1989?",
	"Dead Zone (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_deadzonejRomInfo, fds_deadzonejRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_deadzoneRomDesc[] = {
	{ "Dead Zone (T-Eng).fds",          131000, 0x1bf2409a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_deadzone, fds_deadzone, fds_fdsbios)
STD_ROM_FN(fds_deadzone)

struct BurnDriver BurnDrvfds_deadzone = {
	"fds_deadzone", NULL, "fds_fdsbios", NULL, "1989?",
	"Dead Zone (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_deadzoneRomInfo, fds_deadzoneRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_digdugRomDesc[] = {
	{ "Dig Dug (Japan).fds",          65516, 0xeb126143, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_digdug, fds_digdug, fds_fdsbios)
STD_ROM_FN(fds_digdug)

struct BurnDriver BurnDrvfds_digdug = {
	"fds_digdug", NULL, "fds_fdsbios", NULL, "1989?",
	"Dig Dug (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_digdugRomInfo, fds_digdugRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_digdugiiRomDesc[] = {
	{ "Dig Dug II (Japan).fds",          65516, 0x1b2b1342, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_digdugii, fds_digdugii, fds_fdsbios)
STD_ROM_FN(fds_digdugii)

struct BurnDriver BurnDrvfds_digdugii = {
	"fds_digdugii", NULL, "fds_fdsbios", NULL, "1989?",
	"Dig Dug II (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_digdugiiRomInfo, fds_digdugiiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dirtypaijRomDesc[] = {
	{ "Dirty Pair - Project Eden (Japan).fds",          131016, 0x0bba3177, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dirtypaij, fds_dirtypaij, fds_fdsbios)
STD_ROM_FN(fds_dirtypaij)

struct BurnDriver BurnDrvfds_dirtypaij = {
	"fds_dirtypaij", "fds_dirtypai", "fds_fdsbios", NULL, "1989?",
	"Dirty Pair - Project Eden (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_dirtypaijRomInfo, fds_dirtypaijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dirtypaiRomDesc[] = {
	{ "Dirty Pair - Project Eden (T-Eng).fds",          131000, 0x5707b40a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dirtypai, fds_dirtypai, fds_fdsbios)
STD_ROM_FN(fds_dirtypai)

struct BurnDriver BurnDrvfds_dirtypai = {
	"fds_dirtypai", NULL, "fds_fdsbios", NULL, "1989?",
	"Dirty Pair - Project Eden (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_dirtypaiRomInfo, fds_dirtypaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dokidokipanicRomDesc[] = {
	{ "Doki Doki Panic (Japan).fds",          131016, 0xb36dfcd2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dokidokipanic, fds_dokidokipanic, fds_fdsbios)
STD_ROM_FN(fds_dokidokipanic)

struct BurnDriver BurnDrvfds_dokidokipanic = {
	"fds_dokidokipanic", NULL, "fds_fdsbios", NULL, "1989?",
	"Doki Doki Panic (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_dokidokipanicRomInfo, fds_dokidokipanicRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_donkeykongRomDesc[] = {
	{ "Donkey Kong (Japan).fds",          65516, 0x6c0740e9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_donkeykong, fds_donkeykong, fds_fdsbios)
STD_ROM_FN(fds_donkeykong)

struct BurnDriver BurnDrvfds_donkeykong = {
	"fds_donkeykong", NULL, "fds_fdsbios", NULL, "1989?",
	"Donkey Kong (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_donkeykongRomInfo, fds_donkeykongRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_donkeykongjrRomDesc[] = {
	{ "Donkey Kong Jr. (Japan).fds",          65516, 0x3046966a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_donkeykongjr, fds_donkeykongjr, fds_fdsbios)
STD_ROM_FN(fds_donkeykongjr)

struct BurnDriver BurnDrvfds_donkeykongjr = {
	"fds_donkeykongjr", NULL, "fds_fdsbios", NULL, "1989?",
	"Donkey Kong Jr. (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_donkeykongjrRomInfo, fds_donkeykongjrRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_drchaRomDesc[] = {
	{ "Dr. Chaos - Jigoku no Tobira (Japan).fds",          131000, 0x7954bd8f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_drcha, fds_drcha, fds_fdsbios)
STD_ROM_FN(fds_drcha)

struct BurnDriver BurnDrvfds_drcha = {
	"fds_drcha", NULL, "fds_fdsbios", NULL, "1989?",
	"Dr. Chaos - Jigoku no Tobira (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_drchaRomInfo, fds_drchaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_dracuiiRomDesc[] = {
	{ "Dracula II - Noroi no Fuuin (Japan).fds",          131016, 0x2d1ec77c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_dracuii, fds_dracuii, fds_fdsbios)
STD_ROM_FN(fds_dracuii)

struct BurnDriver BurnDrvfds_dracuii = {
	"fds_dracuii", NULL, "fds_fdsbios", NULL, "1989?",
	"Dracula II - Noroi no Fuuin (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_dracuiiRomInfo, fds_dracuiiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_druidRomDesc[] = {
	{ "Druid - Kyoufu no Tobira (Japan).fds",          131016, 0xb5d4e955, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_druid, fds_druid, fds_fdsbios)
STD_ROM_FN(fds_druid)

struct BurnDriver BurnDrvfds_druid = {
	"fds_druid", NULL, "fds_fdsbios", NULL, "1989?",
	"Druid - Kyoufu no Tobira (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_druidRomInfo, fds_druidRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_eggerlandRomDesc[] = {
	{ "Egger Land (Japan).fds",          131016, 0x84f75912, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_eggerland, fds_eggerland, fds_fdsbios)
STD_ROM_FN(fds_eggerland)

struct BurnDriver BurnDrvfds_eggerland = {
	"fds_eggerland", NULL, "fds_fdsbios", NULL, "1989?",
	"Egger Land (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_eggerlandRomInfo, fds_eggerlandRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_eggerlansouenotaRomDesc[] = {
	{ "Egger Land - Souzouhe no Tabidachi (Japan).fds",          131016, 0x2441b3b9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_eggerlansouenota, fds_eggerlansouenota, fds_fdsbios)
STD_ROM_FN(fds_eggerlansouenota)

struct BurnDriver BurnDrvfds_eggerlansouenota = {
	"fds_eggerlansouenota", NULL, "fds_fdsbios", NULL, "1989?",
	"Egger Land - Souzouhe no Tabidachi (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_eggerlansouenotaRomInfo, fds_eggerlansouenotaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_electrician1RomDesc[] = {
	{ "Electrician (Japan).fds",          131016, 0x9c37e289, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_electrician1, fds_electrician1, fds_fdsbios)
STD_ROM_FN(fds_electrician1)

struct BurnDriver BurnDrvfds_electrician1 = {
	"fds_electrician1", "fds_electrician", "fds_fdsbios", NULL, "1989?",
	"Electrician (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_electrician1RomInfo, fds_electrician1RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_electricianRomDesc[] = {
	{ "Electrician (T-eng).fds",          131016, 0x4aa51f48, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_electrician, fds_electrician, fds_fdsbios)
STD_ROM_FN(fds_electrician)

struct BurnDriver BurnDrvfds_electrician = {
	"fds_electrician", NULL, "fds_fdsbios", NULL, "1989?",
	"Electrician (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_electricianRomInfo, fds_electricianRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_esperdreamjRomDesc[] = {
	{ "Esper Dream (Japan).fds",          131000, 0x1d40f83f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_esperdreamj, fds_esperdreamj, fds_fdsbios)
STD_ROM_FN(fds_esperdreamj)

struct BurnDriver BurnDrvfds_esperdreamj = {
	"fds_esperdreamj", "fds_esperdream", "fds_fdsbios", NULL, "1989?",
	"Esper Dream (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_esperdreamjRomInfo, fds_esperdreamjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_esperdreamRomDesc[] = {
	{ "Esper Dream (T-Eng).fds",          131000, 0x51d8c9f1, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_esperdream, fds_esperdream, fds_fdsbios)
STD_ROM_FN(fds_esperdream)

struct BurnDriver BurnDrvfds_esperdream = {
	"fds_esperdream", NULL, "fds_fdsbios", NULL, "1989?",
	"Esper Dream (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_esperdreamRomInfo, fds_esperdreamRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_excitsocRomDesc[] = {
	{ "Exciting Soccer - Konami Cup (Japan).fds",          131016, 0x43ccd4ba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_excitsoc, fds_excitsoc, fds_fdsbios)
STD_ROM_FN(fds_excitsoc)

struct BurnDriver BurnDrvfds_excitsoc = {
	"fds_excitsoc", NULL, "fds_fdsbios", NULL, "1989?",
	"Exciting Soccer - Konami Cup (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_excitsocRomInfo, fds_excitsocRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_fairypinRomDesc[] = {
	{ "Fairy Pinball - Yousei Tachi no Pinball (Japan).fds",          131016, 0x1ce10f77, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_fairypin, fds_fairypin, fds_fdsbios)
STD_ROM_FN(fds_fairypin)

struct BurnDriver BurnDrvfds_fairypin = {
	"fds_fairypin", NULL, "fds_fdsbios", NULL, "1989?",
	"Fairy Pinball - Yousei Tachi no Pinball (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_fairypinRomInfo, fds_fairypinRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_falsionRomDesc[] = {
	{ "Falsion (Japan).fds",          131016, 0x398bc9dc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_falsion, fds_falsion, fds_fdsbios)
STD_ROM_FN(fds_falsion)

struct BurnDriver BurnDrvfds_falsion = {
	"fds_falsion", NULL, "fds_fdsbios", NULL, "1989?",
	"Falsion (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_falsionRomInfo, fds_falsionRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famicgrapriRomDesc[] = {
	{ "Famicom Grand Prix - F1 Race (Japan).fds",          131016, 0x7040bf35, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famicgrapri, fds_famicgrapri, fds_fdsbios)
STD_ROM_FN(fds_famicgrapri)

struct BurnDriver BurnDrvfds_famicgrapri = {
	"fds_famicgrapri", NULL, "fds_fdsbios", NULL, "1989?",
	"Famicom Grand Prix - F1 Race (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_famicgrapriRomInfo, fds_famicgrapriRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famicgrapriiiRomDesc[] = {
	{ "Famicom Grand Prix II - 3D Hot Rally (Japan).fds",          131016, 0x012267d8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famicgrapriii, fds_famicgrapriii, fds_fdsbios)
STD_ROM_FN(fds_famicgrapriii)

struct BurnDriver BurnDrvfds_famicgrapriii = {
	"fds_famicgrapriii", NULL, "fds_fdsbios", NULL, "1989?",
	"Famicom Grand Prix II - 3D Hot Rally (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_famicgrapriiiRomInfo, fds_famicgrapriiiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol1RomDesc[] = {
	{ "Famimaga Disk Vol. 1 - Hong Kong (Japan).fds",          65516, 0xc6f69987, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol1, fds_famimdisvol1, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol1)

struct BurnDriver BurnDrvfds_famimdisvol1 = {
	"fds_famimdisvol1", NULL, "fds_fdsbios", NULL, "1989?",
	"Famimaga Disk Vol. 1 - Hong Kong (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_famimdisvol1RomInfo, fds_famimdisvol1RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol2RomDesc[] = {
	{ "Famimaga Disk Vol. 2 - Panic Space (Japan).fds",          131016, 0x1274645d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol2, fds_famimdisvol2, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol2)

struct BurnDriver BurnDrvfds_famimdisvol2 = {
	"fds_famimdisvol2", NULL, "fds_fdsbios", NULL, "1989?",
	"Famimaga Disk Vol. 2 - Panic Space (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_famimdisvol2RomInfo, fds_famimdisvol2RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol3RomDesc[] = {
	{ "Famimaga Disk Vol. 3 - All 1 (Japan).fds",          65516, 0x936c8017, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol3, fds_famimdisvol3, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol3)

struct BurnDriver BurnDrvfds_famimdisvol3 = {
	"fds_famimdisvol3", NULL, "fds_fdsbios", NULL, "1989?",
	"Famimaga Disk Vol. 3 - All 1 (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_famimdisvol3RomInfo, fds_famimdisvol3RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol4RomDesc[] = {
	{ "Famimaga Disk Vol. 4 - Clox (Japan).fds",          65516, 0xe46ffd9b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol4, fds_famimdisvol4, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol4)

struct BurnDriver BurnDrvfds_famimdisvol4 = {
	"fds_famimdisvol4", NULL, "fds_fdsbios", NULL, "1989?",
	"Famimaga Disk Vol. 4 - Clox (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_famimdisvol4RomInfo, fds_famimdisvol4RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol5RomDesc[] = {
	{ "Famimaga Disk Vol. 5 - Puyo Puyo (Japan).fds",          131016, 0xe84c7927, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol5, fds_famimdisvol5, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol5)

struct BurnDriver BurnDrvfds_famimdisvol5 = {
	"fds_famimdisvol5", NULL, "fds_fdsbios", NULL, "1989?",
	"Famimaga Disk Vol. 5 - Puyo Puyo (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_famimdisvol5RomInfo, fds_famimdisvol5RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_famimdisvol6RomDesc[] = {
	{ "Famimaga Disk Vol. 6 - Janken Disk Jou (Japan).fds",          131016, 0x251ddeaa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_famimdisvol6, fds_famimdisvol6, fds_fdsbios)
STD_ROM_FN(fds_famimdisvol6)

struct BurnDriver BurnDrvfds_famimdisvol6 = {
	"fds_famimdisvol6", NULL, "fds_fdsbios", NULL, "1989?",
	"Famimaga Disk Vol. 6 - Janken Disk Jou (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_famimdisvol6RomInfo, fds_famimdisvol6RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_finalcomRomDesc[] = {
	{ "Final Command - Akai Yousai (Japan).fds",          131016, 0xb89d5131, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_finalcom, fds_finalcom, fds_fdsbios)
STD_ROM_FN(fds_finalcom)

struct BurnDriver BurnDrvfds_finalcom = {
	"fds_finalcom", NULL, "fds_fdsbios", NULL, "1989?",
	"Final Command - Akai Yousai (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_finalcomRomInfo, fds_finalcomRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_firebamRomDesc[] = {
	{ "Fire Bam (Japan).fds",          131016, 0x9e5c5ede, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_firebam, fds_firebam, fds_fdsbios)
STD_ROM_FN(fds_firebam)

struct BurnDriver BurnDrvfds_firebam = {
	"fds_firebam", NULL, "fds_fdsbios", NULL, "1989?",
	"Fire Bam (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_firebamRomInfo, fds_firebamRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_firerockRomDesc[] = {
	{ "Fire Rock (Japan).fds",          131016, 0xe44c7313, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_firerock, fds_firerock, fds_fdsbios)
STD_ROM_FN(fds_firerock)

struct BurnDriver BurnDrvfds_firerock = {
	"fds_firerock", NULL, "fds_fdsbios", NULL, "1989?",
	"Fire Rock (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_firerockRomInfo, fds_firerockRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_fuuunshokenRomDesc[] = {
	{ "Fuuun Shourin Ken (Japan).fds",          131016, 0xb5b75934, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_fuuunshoken, fds_fuuunshoken, fds_fdsbios)
STD_ROM_FN(fds_fuuunshoken)

struct BurnDriver BurnDrvfds_fuuunshoken = {
	"fds_fuuunshoken", NULL, "fds_fdsbios", NULL, "1989?",
	"Fuuun Shourin Ken (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_fuuunshokenRomInfo, fds_fuuunshokenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_galagaRomDesc[] = {
	{ "Galaga (Japan).fds",          65516, 0xa9ba7801, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_galaga, fds_galaga, fds_fdsbios)
STD_ROM_FN(fds_galaga)

struct BurnDriver BurnDrvfds_galaga = {
	"fds_galaga", NULL, "fds_fdsbios", NULL, "1989?",
	"Galaga (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_galagaRomInfo, fds_galagaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_galaxianRomDesc[] = {
	{ "Galaxian (Japan).fds",          65516, 0x429169f5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_galaxian, fds_galaxian, fds_fdsbios)
STD_ROM_FN(fds_galaxian)

struct BurnDriver BurnDrvfds_galaxian = {
	"fds_galaxian", NULL, "fds_fdsbios", NULL, "1989?",
	"Galaxian (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_galaxianRomInfo, fds_galaxianRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gallforjRomDesc[] = {
	{ "Gall Force - Eternal Story (Japan).fds",          131016, 0x51ef74fb, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gallforj, fds_gallforj, fds_fdsbios)
STD_ROM_FN(fds_gallforj)

struct BurnDriver BurnDrvfds_gallforj = {
	"fds_gallforj", "fds_gallfor", "fds_fdsbios", NULL, "1989?",
	"Gall Force - Eternal Story (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_gallforjRomInfo, fds_gallforjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gallforRomDesc[] = {
	{ "Gall Force - Eternal Story (T-Eng).fds",          131000, 0xd66e007f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gallfor, fds_gallfor, fds_fdsbios)
STD_ROM_FN(fds_gallfor)

struct BurnDriver BurnDrvfds_gallfor = {
	"fds_gallfor", NULL, "fds_fdsbios", NULL, "1989?",
	"Gall Force - Eternal Story (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_gallforRomInfo, fds_gallforRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gingadenRomDesc[] = {
	{ "Ginga Denshou - Galaxy Odyssey (Japan).fds",          131016, 0xd38240da, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gingaden, fds_gingaden, fds_fdsbios)
STD_ROM_FN(fds_gingaden)

struct BurnDriver BurnDrvfds_gingaden = {
	"fds_gingaden", NULL, "fds_fdsbios", NULL, "1989?",
	"Ginga Denshou - Galaxy Odyssey (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_gingadenRomInfo, fds_gingadenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_golfRomDesc[] = {
	{ "Golf (Japan).fds",          65516, 0x4667eafa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_golf, fds_golf, fds_fdsbios)
STD_ROM_FN(fds_golf)

struct BurnDriver BurnDrvfds_golf = {
	"fds_golf", NULL, "fds_fdsbios", NULL, "1989?",
	"Golf (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_golfRomInfo, fds_golfRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_golfjapcouRomDesc[] = {
	{ "Golf - Japan Course (Japan).fds",          131016, 0xdbacfaaa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_golfjapcou, fds_golfjapcou, fds_fdsbios)
STD_ROM_FN(fds_golfjapcou)

struct BurnDriver BurnDrvfds_golfjapcou = {
	"fds_golfjapcou", NULL, "fds_fdsbios", NULL, "1989?",
	"Golf - Japan Course (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_golfjapcouRomInfo, fds_golfjapcouRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_golfuscourseRomDesc[] = {
	{ "Golf - US Course (Japan).fds",          131016, 0x0ef5dae4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_golfuscourse, fds_golfuscourse, fds_fdsbios)
STD_ROM_FN(fds_golfuscourse)

struct BurnDriver BurnDrvfds_golfuscourse = {
	"fds_golfuscourse", NULL, "fds_fdsbios", NULL, "1989?",
	"Golf - US Course (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_golfuscourseRomInfo, fds_golfuscourseRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gooniesRomDesc[] = {
	{ "Goonies (Japan).fds",          65516, 0x81b760a2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_goonies, fds_goonies, fds_fdsbios)
STD_ROM_FN(fds_goonies)

struct BurnDriver BurnDrvfds_goonies = {
	"fds_goonies", NULL, "fds_fdsbios", NULL, "1989?",
	"Goonies (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_gooniesRomInfo, fds_gooniesRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_greenberetRomDesc[] = {
	{ "Green Beret (Japan).fds",          131016, 0xe7f0577b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_greenberet, fds_greenberet, fds_fdsbios)
STD_ROM_FN(fds_greenberet)

struct BurnDriver BurnDrvfds_greenberet = {
	"fds_greenberet", NULL, "fds_fdsbios", NULL, "1989?",
	"Green Beret (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_greenberetRomInfo, fds_greenberetRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gunsmokeRomDesc[] = {
	{ "Gun.Smoke (Japan).fds",          131016, 0xbe761b95, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gunsmoke, fds_gunsmoke, fds_fdsbios)
STD_ROM_FN(fds_gunsmoke)

struct BurnDriver BurnDrvfds_gunsmoke = {
	"fds_gunsmoke", NULL, "fds_fdsbios", NULL, "1989?",
	"Gun.Smoke (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_gunsmokeRomInfo, fds_gunsmokeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gyruss1RomDesc[] = {
	{ "Gyruss (Japan).fds",          131016, 0x4552202c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gyruss1, fds_gyruss1, fds_fdsbios)
STD_ROM_FN(fds_gyruss1)

struct BurnDriver BurnDrvfds_gyruss1 = {
	"fds_gyruss1", "fds_gyruss", "fds_fdsbios", NULL, "1989?",
	"Gyruss (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_gyruss1RomInfo, fds_gyruss1RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_gyrussRomDesc[] = {
	{ "Gyruss (T-eng).fds",          131000, 0xf17f2728, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_gyruss, fds_gyruss, fds_fdsbios)
STD_ROM_FN(fds_gyruss)

struct BurnDriver BurnDrvfds_gyruss = {
	"fds_gyruss", NULL, "fds_fdsbios", NULL, "1989?",
	"Gyruss (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_gyrussRomInfo, fds_gyrussRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_halleywarsRomDesc[] = {
	{ "Halley Wars (Japan).fds",          131016, 0x583b11ca, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_halleywars, fds_halleywars, fds_fdsbios)
STD_ROM_FN(fds_halleywars)

struct BurnDriver BurnDrvfds_halleywars = {
	"fds_halleywars", NULL, "fds_fdsbios", NULL, "1989?",
	"Halley Wars (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_halleywarsRomInfo, fds_halleywarsRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_haokunnofusnataRomDesc[] = {
	{ "Hao-kun no Fushigi na Tabi (Japan).fds",          131016, 0x085f5f88, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_haokunnofusnata, fds_haokunnofusnata, fds_fdsbios)
STD_ROM_FN(fds_haokunnofusnata)

struct BurnDriver BurnDrvfds_haokunnofusnata = {
	"fds_haokunnofusnata", NULL, "fds_fdsbios", NULL, "1989?",
	"Hao-kun no Fushigi na Tabi (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_haokunnofusnataRomInfo, fds_haokunnofusnataRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_hikarshiRomDesc[] = {
	{ "Hikari Shinwa - Palutena no Kagami (Japan).fds",          131000, 0x32516fcf, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_hikarshi, fds_hikarshi, fds_fdsbios)
STD_ROM_FN(fds_hikarshi)

struct BurnDriver BurnDrvfds_hikarshi = {
	"fds_hikarshi", NULL, "fds_fdsbios", NULL, "1989?",
	"Hikari Shinwa - Palutena no Kagami (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_hikarshiRomInfo, fds_hikarshiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_icehockeyRomDesc[] = {
	{ "Ice Hockey (Japan).fds",          65516, 0x7bb390de, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_icehockey, fds_icehockey, fds_fdsbios)
STD_ROM_FN(fds_icehockey)

struct BurnDriver BurnDrvfds_icehockey = {
	"fds_icehockey", NULL, "fds_fdsbios", NULL, "1989?",
	"Ice Hockey (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_icehockeyRomInfo, fds_icehockeyRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ishidoujRomDesc[] = {
	{ "Ishidou (Japan).fds",          131000, 0x0c54a19e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ishidouj, fds_ishidouj, fds_fdsbios)
STD_ROM_FN(fds_ishidouj)

struct BurnDriver BurnDrvfds_ishidouj = {
	"fds_ishidouj", "fds_ishidou", "fds_fdsbios", NULL, "1989?",
	"Ishidou (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_ishidoujRomInfo, fds_ishidoujRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ishidouRomDesc[] = {
	{ "Ishidou (T-Eng).fds",          131000, 0xebfcf648, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ishidou, fds_ishidou, fds_fdsbios)
STD_ROM_FN(fds_ishidou)

struct BurnDriver BurnDrvfds_ishidou = {
	"fds_ishidou", NULL, "fds_fdsbios", NULL, "1989?",
	"Ishidou (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_ishidouRomInfo, fds_ishidouRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_jikaishometmagRomDesc[] = {
	{ "Jikai Shounen Met Mag (Japan).fds",          131016, 0xc8ce0200, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_jikaishometmag, fds_jikaishometmag, fds_fdsbios)
STD_ROM_FN(fds_jikaishometmag)

struct BurnDriver BurnDrvfds_jikaishometmag = {
	"fds_jikaishometmag", NULL, "fds_fdsbios", NULL, "1989?",
	"Jikai Shounen Met Mag (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_jikaishometmagRomInfo, fds_jikaishometmagRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kaettmarbroRomDesc[] = {
	{ "Kaettekita Mario Brothers (Japan).fds",          131016, 0x12358ded, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kaettmarbro, fds_kaettmarbro, fds_fdsbios)
STD_ROM_FN(fds_kaettmarbro)

struct BurnDriver BurnDrvfds_kaettmarbro = {
	"fds_kaettmarbro", NULL, "fds_fdsbios", NULL, "1989?",
	"Kaettekita Mario Brothers (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kaettmarbroRomInfo, fds_kaettmarbroRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kalinnotsurugijRomDesc[] = {
	{ "Kalin no Tsurugi (Japan).fds",          131000, 0xc63a0059, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kalinnotsurugij, fds_kalinnotsurugij, fds_fdsbios)
STD_ROM_FN(fds_kalinnotsurugij)

struct BurnDriver BurnDrvfds_kalinnotsurugij = {
	"fds_kalinnotsurugij", "fds_kalinnotsurugi", "fds_fdsbios", NULL, "1989?",
	"Kalin no Tsurugi (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kalinnotsurugijRomInfo, fds_kalinnotsurugijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kalinnotsurugiRomDesc[] = {
	{ "Kalin no Tsurugi (T-Eng).fds",          131000, 0x7825c974, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kalinnotsurugi, fds_kalinnotsurugi, fds_fdsbios)
STD_ROM_FN(fds_kalinnotsurugi)

struct BurnDriver BurnDrvfds_kalinnotsurugi = {
	"fds_kalinnotsurugi", NULL, "fds_fdsbios", NULL, "1989?",
	"Kalin no Tsurugi (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kalinnotsurugiRomInfo, fds_kalinnotsurugiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kamenridblaRomDesc[] = {
	{ "Kamen Rider Black - Taiketsu Shadow Moon (Japan).fds",          131016, 0x0db848ea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kamenridbla, fds_kamenridbla, fds_fdsbios)
STD_ROM_FN(fds_kamenridbla)

struct BurnDriver BurnDrvfds_kamenridbla = {
	"fds_kamenridbla", NULL, "fds_fdsbios", NULL, "1989?",
	"Kamen Rider Black - Taiketsu Shadow Moon (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kamenridblaRomInfo, fds_kamenridblaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kickandrunRomDesc[] = {
	{ "Kick and Run (Japan).fds",          131016, 0xa37d4c85, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kickandrun, fds_kickandrun, fds_fdsbios)
STD_ROM_FN(fds_kickandrun)

struct BurnDriver BurnDrvfds_kickandrun = {
	"fds_kickandrun", NULL, "fds_fdsbios", NULL, "1989?",
	"Kick and Run (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kickandrunRomInfo, fds_kickandrunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kickchaRomDesc[] = {
	{ "Kick Challenger - Air Foot (Japan).fds",          131016, 0xc83ef278, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kickcha, fds_kickcha, fds_fdsbios)
STD_ROM_FN(fds_kickcha)

struct BurnDriver BurnDrvfds_kickcha = {
	"fds_kickcha", NULL, "fds_fdsbios", NULL, "1989?",
	"Kick Challenger - Air Foot (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kickchaRomInfo, fds_kickchaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kietaprincessRomDesc[] = {
	{ "Kieta Princess (Japan).fds",          131000, 0x60d55cea, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kietaprincess, fds_kietaprincess, fds_fdsbios)
STD_ROM_FN(fds_kietaprincess)

struct BurnDriver BurnDrvfds_kietaprincess = {
	"fds_kietaprincess", NULL, "fds_fdsbios", NULL, "1989?",
	"Kieta Princess (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kietaprincessRomInfo, fds_kietaprincessRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kikikaiRomDesc[] = {
	{ "Kiki Kaikai - Dotou Hen (Japan).fds",          131000, 0x7f446c09, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kikikai, fds_kikikai, fds_fdsbios)
STD_ROM_FN(fds_kikikai)

struct BurnDriver BurnDrvfds_kikikai = {
	"fds_kikikai", NULL, "fds_fdsbios", NULL, "1989?",
	"Kiki Kaikai - Dotou Hen (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kikikaiRomInfo, fds_kikikaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_kinniRomDesc[] = {
	{ "Kinnikuman - Kinnikusei Oui Soudatsusen (Japan).fds",          131016, 0x5149b663, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_kinni, fds_kinni, fds_fdsbios)
STD_ROM_FN(fds_kinni)

struct BurnDriver BurnDrvfds_kinni = {
	"fds_kinni", NULL, "fds_fdsbios", NULL, "1989?",
	"Kinnikuman - Kinnikusei Oui Soudatsusen (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_kinniRomInfo, fds_kinniRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_knighlorRomDesc[] = {
	{ "Knight Lore - Majou no Ookami Otoko (Japan).fds",          131016, 0xc07a8b04, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_knighlor, fds_knighlor, fds_fdsbios)
STD_ROM_FN(fds_knighlor)

struct BurnDriver BurnDrvfds_knighlor = {
	"fds_knighlor", NULL, "fds_fdsbios", NULL, "1989?",
	"Knight Lore - Majou no Ookami Otoko (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_knighlorRomInfo, fds_knighlorRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_knightmovejRomDesc[] = {
	{ "Knight Move (Japan).fds",          65500, 0x6d365e2f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_knightmovej, fds_knightmovej, fds_fdsbios)
STD_ROM_FN(fds_knightmovej)

struct BurnDriver BurnDrvfds_knightmovej = {
	"fds_knightmovej", "fds_knightmove", "fds_fdsbios", NULL, "1989?",
	"Knight Move (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_knightmovejRomInfo, fds_knightmovejRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_knightmoveRomDesc[] = {
	{ "Knight Move (T-Eng).fds",          65500, 0x474b83c4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_knightmove, fds_knightmove, fds_fdsbios)
STD_ROM_FN(fds_knightmove)

struct BurnDriver BurnDrvfds_knightmove = {
	"fds_knightmove", NULL, "fds_fdsbios", NULL, "1989?",
	"Knight Move (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_knightmoveRomInfo, fds_knightmoveRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_konamictennisRomDesc[] = {
	{ "Konamic Tennis (Japan).fds",          131000, 0xedbaee55, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_konamictennis, fds_konamictennis, fds_fdsbios)
STD_ROM_FN(fds_konamictennis)

struct BurnDriver BurnDrvfds_konamictennis = {
	"fds_konamictennis", NULL, "fds_fdsbios", NULL, "1989?",
	"Konamic Tennis (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_konamictennisRomInfo, fds_konamictennisRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_konekmonRomDesc[] = {
	{ "Koneko Monogatari - The Adventures of Chatran (Japan).fds",          131016, 0xe3387bf0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_konekmon, fds_konekmon, fds_fdsbios)
STD_ROM_FN(fds_konekmon)

struct BurnDriver BurnDrvfds_konekmon = {
	"fds_konekmon", NULL, "fds_fdsbios", NULL, "1989?",
	"Koneko Monogatari - The Adventures of Chatran (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_konekmonRomInfo, fds_konekmonRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_linknoboujRomDesc[] = {
	{ "Link no Bouken - The Legend of Zelda 2 (Japan).fds",          131016, 0xa118214f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_linknobouj, fds_linknobouj, fds_fdsbios)
STD_ROM_FN(fds_linknobouj)

struct BurnDriver BurnDrvfds_linknobouj = {
	"fds_linknobouj", "fds_linknobou", "fds_fdsbios", NULL, "1989?",
	"Link no Bouken - The Legend of Zelda 2 (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_linknoboujRomInfo, fds_linknoboujRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_linknobouRomDesc[] = {
	{ "Link no Bouken - The Legend of Zelda 2 (T-Eng).fds",          131000, 0x359d3f2b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_linknobou, fds_linknobou, fds_fdsbios)
STD_ROM_FN(fds_linknobou)

struct BurnDriver BurnDrvfds_linknobou = {
	"fds_linknobou", NULL, "fds_fdsbios", NULL, "1989?",
	"Link no Bouken - The Legend of Zelda 2 (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_linknobouRomInfo, fds_linknobouRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_lutterRomDesc[] = {
	{ "Lutter (Japan).fds",          131016, 0x54335232, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_lutter, fds_lutter, fds_fdsbios)
STD_ROM_FN(fds_lutter)

struct BurnDriver BurnDrvfds_lutter = {
	"fds_lutter", NULL, "fds_fdsbios", NULL, "1989?",
	"Lutter (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_lutterRomInfo, fds_lutterRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_maerchenveilRomDesc[] = {
	{ "Maerchen Veil (Japan).fds",          131016, 0x618d10a2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_maerchenveil, fds_maerchenveil, fds_fdsbios)
STD_ROM_FN(fds_maerchenveil)

struct BurnDriver BurnDrvfds_maerchenveil = {
	"fds_maerchenveil", NULL, "fds_fdsbios", NULL, "1989?",
	"Maerchen Veil (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_maerchenveilRomInfo, fds_maerchenveilRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_magmaproRomDesc[] = {
	{ "Magma Project - Hacker (Japan).fds",          131016, 0x59d51115, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_magmapro, fds_magmapro, fds_fdsbios)
STD_ROM_FN(fds_magmapro)

struct BurnDriver BurnDrvfds_magmapro = {
	"fds_magmapro", NULL, "fds_fdsbios", NULL, "1989?",
	"Magma Project - Hacker (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_magmaproRomInfo, fds_magmaproRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_matounohouRomDesc[] = {
	{ "Matou no Houkai - The Hero of Babel (Japan).fds",          131016, 0x0596b36a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_matounohou, fds_matounohou, fds_fdsbios)
STD_ROM_FN(fds_matounohou)

struct BurnDriver BurnDrvfds_matounohou = {
	"fds_matounohou", NULL, "fds_fdsbios", NULL, "1989?",
	"Matou no Houkai - The Hero of Babel (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_matounohouRomInfo, fds_matounohouRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_meikyjiidabjRomDesc[] = {
	{ "Meikyuu Jiin Dababa (Japan).fds",          131000, 0x1526d32e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_meikyjiidabj, fds_meikyjiidabj, fds_fdsbios)
STD_ROM_FN(fds_meikyjiidabj)

struct BurnDriver BurnDrvfds_meikyjiidabj = {
	"fds_meikyjiidabj", "fds_meikyjiidab", "fds_fdsbios", NULL, "1989?",
	"Meikyuu Jiin Dababa (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_meikyjiidabjRomInfo, fds_meikyjiidabjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_meikyjiidabRomDesc[] = {
	{ "Meikyuu Jiin Dababa (T-Eng).fds",          131000, 0x4d13e6b7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_meikyjiidab, fds_meikyjiidab, fds_fdsbios)
STD_ROM_FN(fds_meikyjiidab)

struct BurnDriver BurnDrvfds_meikyjiidab = {
	"fds_meikyjiidab", NULL, "fds_fdsbios", NULL, "1989?",
	"Meikyuu Jiin Dababa (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_meikyjiidabRomInfo, fds_meikyjiidabRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_metroidjRomDesc[] = {
	{ "Metroid (Japan).fds",          131000, 0x5e4f013d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_metroidj, fds_metroidj, fds_fdsbios)
STD_ROM_FN(fds_metroidj)

struct BurnDriver BurnDrvfds_metroidj = {
	"fds_metroidj", "fds_metroid", "fds_fdsbios", NULL, "1989?",
	"Metroid (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_metroidjRomInfo, fds_metroidjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_metroidRomDesc[] = {
	{ "Metroid (T-Eng).fds",          131000, 0x2711adba, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_metroid, fds_metroid, fds_fdsbios)
STD_ROM_FN(fds_metroid)

struct BurnDriver BurnDrvfds_metroid = {
	"fds_metroid", NULL, "fds_fdsbios", NULL, "1989?",
	"Metroid (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_metroidRomInfo, fds_metroidRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_michaengdaiRomDesc[] = {
	{ "Michael English Daibouken (Japan).fds",          131000, 0xe5ab0e6c, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_michaengdai, fds_michaengdai, fds_fdsbios)
STD_ROM_FN(fds_michaengdai)

struct BurnDriver BurnDrvfds_michaengdai = {
	"fds_michaengdai", NULL, "fds_fdsbios", NULL, "1989?",
	"Michael English Daibouken (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_michaengdaiRomInfo, fds_michaengdaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_moerotwiRomDesc[] = {
	{ "Moero TwinBee - Cinnamon Hakase wo Sukue! (Japan).fds",          131016, 0x0d684aa7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_moerotwi, fds_moerotwi, fds_fdsbios)
STD_ROM_FN(fds_moerotwi)

struct BurnDriver BurnDrvfds_moerotwi = {
	"fds_moerotwi", NULL, "fds_fdsbios", NULL, "1989?",
	"Moero TwinBee - Cinnamon Hakase wo Sukue! (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_moerotwiRomInfo, fds_moerotwiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_montyonrunjRomDesc[] = {
	{ "Monty on the Run - Monty no Doki Doki Dai Dassou (Japan).fds",          131016, 0xbb7ab325, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_montyonrunj, fds_montyonrunj, fds_fdsbios)
STD_ROM_FN(fds_montyonrunj)

struct BurnDriver BurnDrvfds_montyonrunj = {
	"fds_montyonrunj", "fds_montyonrun", "fds_fdsbios", NULL, "1989?",
	"Monty on the Run - Monty no Doki Doki Dai Dassou (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_montyonrunjRomInfo, fds_montyonrunjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_montyonrunRomDesc[] = {
	{ "Monty on the Run - Monty no Doki Doki Dai Dassou (T-Eng).fds",          131001, 0xd7242483, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_montyonrun, fds_montyonrun, fds_fdsbios)
STD_ROM_FN(fds_montyonrun)

struct BurnDriver BurnDrvfds_montyonrun = {
	"fds_montyonrun", NULL, "fds_fdsbios", NULL, "1989?",
	"Monty on the Run - Monty no Doki Doki Dai Dassou (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_montyonrunRomInfo, fds_montyonrunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_moonballmagicRomDesc[] = {
	{ "Moonball Magic (Japan).fds",          131016, 0x99a87df8, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_moonballmagic, fds_moonballmagic, fds_fdsbios)
STD_ROM_FN(fds_moonballmagic)

struct BurnDriver BurnDrvfds_moonballmagic = {
	"fds_moonballmagic", NULL, "fds_fdsbios", NULL, "1989?",
	"Moonball Magic (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_moonballmagicRomInfo, fds_moonballmagicRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_mystemurcasRomDesc[] = {
	{ "Mysterious Murasame Castle (T-Eng).fds",          131000, 0x5da99951, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_mystemurcas, fds_mystemurcas, fds_fdsbios)
STD_ROM_FN(fds_mystemurcas)

struct BurnDriver BurnDrvfds_mystemurcas = {
	"fds_mystemurcas", NULL, "fds_fdsbios", NULL, "1989?",
	"Mysterious Murasame Castle (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_mystemurcasRomInfo, fds_mystemurcasRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_nazonokabRomDesc[] = {
	{ "Nazo no Kabe - Block Kuzushi (Japan).fds",          131000, 0x09292f11, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_nazonokab, fds_nazonokab, fds_fdsbios)
STD_ROM_FN(fds_nazonokab)

struct BurnDriver BurnDrvfds_nazonokab = {
	"fds_nazonokab", NULL, "fds_fdsbios", NULL, "1989?",
	"Nazo no Kabe - Block Kuzushi (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_nazonokabRomInfo, fds_nazonokabRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_nazonomurRomDesc[] = {
	{ "Nazo no Murasamejou (Japan).fds",          131016, 0x2acde825, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_nazonomur, fds_nazonomur, fds_fdsbios)
STD_ROM_FN(fds_nazonomur)

struct BurnDriver BurnDrvfds_nazonomur = {
	"fds_nazonomur", "fds_mystemurcas", "fds_fdsbios", NULL, "1989?",
	"Nazo no Murasamejou (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_nazonomurRomInfo, fds_nazonomurRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_otockyRomDesc[] = {
	{ "Otocky (Japan).fds",          131016, 0xd7a49861, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_otocky, fds_otocky, fds_fdsbios)
STD_ROM_FN(fds_otocky)

struct BurnDriver BurnDrvfds_otocky = {
	"fds_otocky", NULL, "fds_fdsbios", NULL, "1989?",
	"Otocky (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_otockyRomInfo, fds_otockyRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_patlaRomDesc[] = {
	{ "Patlabor - The Mobile Police - Dai 2 Shoutai Shutsudou Seyo! (Japan).fds",          131016, 0xb5639859, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_patla, fds_patla, fds_fdsbios)
STD_ROM_FN(fds_patla)

struct BurnDriver BurnDrvfds_patla = {
	"fds_patla", NULL, "fds_fdsbios", NULL, "1989?",
	"Patlabor - The Mobile Police - Dai 2 Shoutai Shutsudou Seyo! (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_patlaRomInfo, fds_patlaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_progolsarkagnotoRomDesc[] = {
	{ "Pro Golfer Saru - Kage no Tournament (Japan).fds",          131016, 0x8261910e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_progolsarkagnoto, fds_progolsarkagnoto, fds_fdsbios)
STD_ROM_FN(fds_progolsarkagnoto)

struct BurnDriver BurnDrvfds_progolsarkagnoto = {
	"fds_progolsarkagnoto", NULL, "fds_fdsbios", NULL, "1989?",
	"Pro Golfer Saru - Kage no Tournament (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_progolsarkagnotoRomInfo, fds_progolsarkagnotoRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_puttputtgolfRomDesc[] = {
	{ "Putt Putt Golf (Japan).fds",          131016, 0x0ac52ae0, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_puttputtgolf, fds_puttputtgolf, fds_fdsbios)
STD_ROM_FN(fds_puttputtgolf)

struct BurnDriver BurnDrvfds_puttputtgolf = {
	"fds_puttputtgolf", NULL, "fds_fdsbios", NULL, "1989?",
	"Putt Putt Golf (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_puttputtgolfRomInfo, fds_puttputtgolfRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_puzzleboysjRomDesc[] = {
	{ "Puzzle Boys (Japan).fds",          131000, 0xb9f22094, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_puzzleboysj, fds_puzzleboysj, fds_fdsbios)
STD_ROM_FN(fds_puzzleboysj)

struct BurnDriver BurnDrvfds_puzzleboysj = {
	"fds_puzzleboysj", "fds_puzzleboys", "fds_fdsbios", NULL, "1989?",
	"Puzzle Boys (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_puzzleboysjRomInfo, fds_puzzleboysjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_puzzleboysRomDesc[] = {
	{ "Puzzle Boys (T-Eng).fds",          131000, 0xfa03eba9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_puzzleboys, fds_puzzleboys, fds_fdsbios)
STD_ROM_FN(fds_puzzleboys)

struct BurnDriver BurnDrvfds_puzzleboys = {
	"fds_puzzleboys", NULL, "fds_fdsbios", NULL, "1989?",
	"Puzzle Boys (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_puzzleboysRomInfo, fds_puzzleboysRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_radicbomjirkunRomDesc[] = {
	{ "Radical Bomber!! Jirai-kun (Japan).fds",          131016, 0x2b41056b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_radicbomjirkun, fds_radicbomjirkun, fds_fdsbios)
STD_ROM_FN(fds_radicbomjirkun)

struct BurnDriver BurnDrvfds_radicbomjirkun = {
	"fds_radicbomjirkun", NULL, "fds_fdsbios", NULL, "1989?",
	"Radical Bomber!! Jirai-kun (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_radicbomjirkunRomInfo, fds_radicbomjirkunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_reflectworldRomDesc[] = {
	{ "Reflect World (Japan).fds",          131016, 0x2c24cd2d, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_reflectworld, fds_reflectworld, fds_fdsbios)
STD_ROM_FN(fds_reflectworld)

struct BurnDriver BurnDrvfds_reflectworld = {
	"fds_reflectworld", NULL, "fds_fdsbios", NULL, "1989?",
	"Reflect World (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_reflectworldRomInfo, fds_reflectworldRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_relicsRomDesc[] = {
	{ "Relics - Ankoku Yousai (Japan).fds",          131016, 0xfb0ae491, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_relics, fds_relics, fds_fdsbios)
STD_ROM_FN(fds_relics)

struct BurnDriver BurnDrvfds_relics = {
	"fds_relics", NULL, "fds_fdsbios", NULL, "1989?",
	"Relics - Ankoku Yousai (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_relicsRomInfo, fds_relicsRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_replicartRomDesc[] = {
	{ "Replicart (Japan).fds",          131016, 0xc3cbc847, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_replicart, fds_replicart, fds_fdsbios)
STD_ROM_FN(fds_replicart)

struct BurnDriver BurnDrvfds_replicart = {
	"fds_replicart", NULL, "fds_fdsbios", NULL, "1989?",
	"Replicart (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_replicartRomInfo, fds_replicartRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_rogerrabbitRomDesc[] = {
	{ "Roger Rabbit (Japan).fds",          131016, 0xb43f9b16, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_rogerrabbit, fds_rogerrabbit, fds_fdsbios)
STD_ROM_FN(fds_rogerrabbit)

struct BurnDriver BurnDrvfds_rogerrabbit = {
	"fds_rogerrabbit", NULL, "fds_fdsbios", NULL, "1989?",
	"Roger Rabbit (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_rogerrabbitRomInfo, fds_rogerrabbitRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_sdgunworgacsescwaRomDesc[] = {
	{ "SD Gundam World - Gachapon Senshi - Scramble Wars (Japan).fds",          131000, 0x9d287c3f, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_sdgunworgacsescwa, fds_sdgunworgacsescwa, fds_fdsbios)
STD_ROM_FN(fds_sdgunworgacsescwa)

struct BurnDriver BurnDrvfds_sdgunworgacsescwa = {
	"fds_sdgunworgacsescwa", NULL, "fds_fdsbios", NULL, "1989?",
	"SD Gundam World - Gachapon Senshi - Scramble Wars (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_sdgunworgacsescwaRomInfo, fds_sdgunworgacsescwaRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_sectionzRomDesc[] = {
	{ "Section-Z (Japan).fds",          131016, 0x392488e2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_sectionz, fds_sectionz, fds_fdsbios)
STD_ROM_FN(fds_sectionz)

struct BurnDriver BurnDrvfds_sectionz = {
	"fds_sectionz", NULL, "fds_fdsbios", NULL, "1989?",
	"Section-Z (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_sectionzRomInfo, fds_sectionzRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_seikepsyjRomDesc[] = {
	{ "Seiken Psychocalibur - Majuu no Mori Densetsu (Japan).fds",          131000, 0xaea5691a, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_seikepsyj, fds_seikepsyj, fds_fdsbios)
STD_ROM_FN(fds_seikepsyj)

struct BurnDriver BurnDrvfds_seikepsyj = {
	"fds_seikepsyj", "fds_seikepsy", "fds_fdsbios", NULL, "1989?",
	"Seiken Psychocalibur - Majuu no Mori Densetsu (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_seikepsyjRomInfo, fds_seikepsyjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_seikepsyRomDesc[] = {
	{ "Seiken Psychocalibur - Majuu no Mori Densetsu (T-Eng).fds",          131000, 0x1ec1552b, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_seikepsy, fds_seikepsy, fds_fdsbios)
STD_ROM_FN(fds_seikepsy)

struct BurnDriver BurnDrvfds_seikepsy = {
	"fds_seikepsy", NULL, "fds_fdsbios", NULL, "1989?",
	"Seiken Psychocalibur - Majuu no Mori Densetsu (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_seikepsyRomInfo, fds_seikepsyRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_sexyinvadersRomDesc[] = {
	{ "Sexy Invaders (Japan).fds",          131000, 0x413e49da, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_sexyinvaders, fds_sexyinvaders, fds_fdsbios)
STD_ROM_FN(fds_sexyinvaders)

struct BurnDriver BurnDrvfds_sexyinvaders = {
	"fds_sexyinvaders", NULL, "fds_fdsbios", NULL, "1989?",
	"Sexy Invaders (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_sexyinvadersRomInfo, fds_sexyinvadersRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_silvijRomDesc[] = {
	{ "Silviana - Ai Ippai no Little Angel (Japan).fds",          131016, 0x4d721d90, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_silvij, fds_silvij, fds_fdsbios)
STD_ROM_FN(fds_silvij)

struct BurnDriver BurnDrvfds_silvij = {
	"fds_silvij", "fds_silvi", "fds_fdsbios", NULL, "1989?",
	"Silviana - Ai Ippai no Little Angel (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_silvijRomInfo, fds_silvijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_silviRomDesc[] = {
	{ "Silviana - Ai Ippai no Little Angel (T-Eng).fds",          131000, 0xc5b1bff7, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_silvi, fds_silvi, fds_fdsbios)
STD_ROM_FN(fds_silvi)

struct BurnDriver BurnDrvfds_silvi = {
	"fds_silvi", NULL, "fds_fdsbios", NULL, "1989?",
	"Silviana - Ai Ippai no Little Angel (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_silviRomInfo, fds_silviRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_smashpingpongRomDesc[] = {
	{ "Smash Ping Pong (Japan).fds",          131016, 0xcc1eb0dc, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_smashpingpong, fds_smashpingpong, fds_fdsbios)
STD_ROM_FN(fds_smashpingpong)

struct BurnDriver BurnDrvfds_smashpingpong = {
	"fds_smashpingpong", NULL, "fds_fdsbios", NULL, "1989?",
	"Smash Ping Pong (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_smashpingpongRomInfo, fds_smashpingpongRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_solomonnokagiRomDesc[] = {
	{ "Solomon no Kagi (Japan).fds",          65500, 0x5e298400, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_solomonnokagi, fds_solomonnokagi, fds_fdsbios)
STD_ROM_FN(fds_solomonnokagi)

struct BurnDriver BurnDrvfds_solomonnokagi = {
	"fds_solomonnokagi", NULL, "fds_fdsbios", NULL, "1989?",
	"Solomon no Kagi (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_solomonnokagiRomInfo, fds_solomonnokagiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_superboyallanRomDesc[] = {
	{ "Super Boy Allan (Japan).fds",          131016, 0xfdc53496, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_superboyallan, fds_superboyallan, fds_fdsbios)
STD_ROM_FN(fds_superboyallan)

struct BurnDriver BurnDrvfds_superboyallan = {
	"fds_superboyallan", NULL, "fds_fdsbios", NULL, "1989?",
	"Super Boy Allan (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_superboyallanRomInfo, fds_superboyallanRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_superlodrunRomDesc[] = {
	{ "Super Lode Runner (Japan).fds",          131000, 0xe66f4682, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_superlodrun, fds_superlodrun, fds_fdsbios)
STD_ROM_FN(fds_superlodrun)

struct BurnDriver BurnDrvfds_superlodrun = {
	"fds_superlodrun", NULL, "fds_fdsbios", NULL, "1989?",
	"Super Lode Runner (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_superlodrunRomInfo, fds_superlodrunRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_superlodruniiRomDesc[] = {
	{ "Super Lode Runner II (Japan).fds",          131000, 0x758888f4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_superlodrunii, fds_superlodrunii, fds_fdsbios)
STD_ROM_FN(fds_superlodrunii)

struct BurnDriver BurnDrvfds_superlodrunii = {
	"fds_superlodrunii", NULL, "fds_fdsbios", NULL, "1989?",
	"Super Lode Runner II (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_superlodruniiRomInfo, fds_superlodruniiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_supermarbroRomDesc[] = {
	{ "Super Mario Brothers (Japan).fds",          65500, 0xe0890cf2, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_supermarbro, fds_supermarbro, fds_fdsbios)
STD_ROM_FN(fds_supermarbro)

struct BurnDriver BurnDrvfds_supermarbro = {
	"fds_supermarbro", NULL, "fds_fdsbios", NULL, "1989?",
	"Super Mario Brothers (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_supermarbroRomInfo, fds_supermarbroRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_supermarbro2RomDesc[] = {
	{ "Super Mario Brothers 2 (Japan).fds",          131016, 0x443ed7b3, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_supermarbro2, fds_supermarbro2, fds_fdsbios)
STD_ROM_FN(fds_supermarbro2)

struct BurnDriver BurnDrvfds_supermarbro2 = {
	"fds_supermarbro2", NULL, "fds_fdsbios", NULL, "1989?",
	"Super Mario Brothers 2 (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_supermarbro2RomInfo, fds_supermarbro2RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_tamafriRomDesc[] = {
	{ "Tama & Friends - 3 Choume Daibouken (Japan).fds",          131016, 0x8751a949, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_tamafri, fds_tamafri, fds_fdsbios)
STD_ROM_FN(fds_tamafri)

struct BurnDriver BurnDrvfds_tamafri = {
	"fds_tamafri", NULL, "fds_fdsbios", NULL, "1989?",
	"Tama & Friends - 3 Choume Daibouken (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_tamafriRomInfo, fds_tamafriRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_tennisRomDesc[] = {
	{ "Tennis (Japan).fds",          65516, 0x136c64ed, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_tennis, fds_tennis, fds_fdsbios)
STD_ROM_FN(fds_tennis)

struct BurnDriver BurnDrvfds_tennis = {
	"fds_tennis", NULL, "fds_fdsbios", NULL, "1989?",
	"Tennis (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_tennisRomInfo, fds_tennisRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_tobiddaiRomDesc[] = {
	{ "Tobidase Daisakusen (Japan).fds",          131016, 0x95546107, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_tobiddai, fds_tobiddai, fds_fdsbios)
STD_ROM_FN(fds_tobiddai)

struct BurnDriver BurnDrvfds_tobiddai = {
	"fds_tobiddai", NULL, "fds_fdsbios", NULL, "1989?",
	"Tobidase Daisakusen (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_tobiddaiRomInfo, fds_tobiddaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_topplezipRomDesc[] = {
	{ "Topple Zip (Japan).fds",          131016, 0x7bf73c75, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_topplezip, fds_topplezip, fds_fdsbios)
STD_ROM_FN(fds_topplezip)

struct BurnDriver BurnDrvfds_topplezip = {
	"fds_topplezip", NULL, "fds_fdsbios", NULL, "1989?",
	"Topple Zip (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_topplezipRomInfo, fds_topplezipRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_transRomDesc[] = {
	{ "Transformers - The Headmasters (Japan).fds",          131016, 0x8bd904cd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_trans, fds_trans, fds_fdsbios)
STD_ROM_FN(fds_trans)

struct BurnDriver BurnDrvfds_trans = {
	"fds_trans", NULL, "fds_fdsbios", NULL, "1989?",
	"Transformers - The Headmasters (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_transRomInfo, fds_transRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_twinbeeRomDesc[] = {
	{ "TwinBee (Japan).fds",          65500, 0x517be396, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_twinbee, fds_twinbee, fds_fdsbios)
STD_ROM_FN(fds_twinbee)

struct BurnDriver BurnDrvfds_twinbee = {
	"fds_twinbee", NULL, "fds_fdsbios", NULL, "1989?",
	"TwinBee (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_twinbeeRomInfo, fds_twinbeeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ultraRomDesc[] = {
	{ "Ultraman - Kaijuu Teikoku no Gyakushuu (Japan).fds",          131016, 0x2ab0c9bd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ultra, fds_ultra, fds_fdsbios)
STD_ROM_FN(fds_ultra)

struct BurnDriver BurnDrvfds_ultra = {
	"fds_ultra", NULL, "fds_fdsbios", NULL, "1989?",
	"Ultraman - Kaijuu Teikoku no Gyakushuu (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_ultraRomInfo, fds_ultraRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ultra2RomDesc[] = {
	{ "Ultraman 2 - Shutsugeki Katoku Tai (Japan).fds",          131016, 0x4010e726, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ultra2, fds_ultra2, fds_fdsbios)
STD_ROM_FN(fds_ultra2)

struct BurnDriver BurnDrvfds_ultra2 = {
	"fds_ultra2", NULL, "fds_fdsbios", NULL, "1989?",
	"Ultraman 2 - Shutsugeki Katoku Tai (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_ultra2RomInfo, fds_ultra2RomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_ultracluRomDesc[] = {
	{ "Ultraman Club - Chikyuu Dakkan Sakusen (Japan).fds",          131000, 0xfd150e81, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_ultraclu, fds_ultraclu, fds_fdsbios)
STD_ROM_FN(fds_ultraclu)

struct BurnDriver BurnDrvfds_ultraclu = {
	"fds_ultraclu", NULL, "fds_fdsbios", NULL, "1989?",
	"Ultraman Club - Chikyuu Dakkan Sakusen (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_ultracluRomInfo, fds_ultracluRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_volleyballRomDesc[] = {
	{ "Volleyball (Japan).fds",          65516, 0x0533c149, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_volleyball, fds_volleyball, fds_fdsbios)
STD_ROM_FN(fds_volleyball)

struct BurnDriver BurnDrvfds_volleyball = {
	"fds_volleyball", NULL, "fds_fdsbios", NULL, "1989?",
	"Volleyball (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_volleyballRomInfo, fds_volleyballRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_vsexcitebikeRomDesc[] = {
	{ "Vs. Excitebike (Japan).fds",          131016, 0xb8a7c0a9, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_vsexcitebike, fds_vsexcitebike, fds_fdsbios)
STD_ROM_FN(fds_vsexcitebike)

struct BurnDriver BurnDrvfds_vsexcitebike = {
	"fds_vsexcitebike", NULL, "fds_fdsbios", NULL, "1989?",
	"Vs. Excitebike (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_vsexcitebikeRomInfo, fds_vsexcitebikeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_wakusatogaiRomDesc[] = {
	{ "Wakusei Aton Gaiden (Japan).fds",          131016, 0x94a825fa, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_wakusatogai, fds_wakusatogai, fds_fdsbios)
STD_ROM_FN(fds_wakusatogai)

struct BurnDriver BurnDrvfds_wakusatogai = {
	"fds_wakusatogai", NULL, "fds_fdsbios", NULL, "1989?",
	"Wakusei Aton Gaiden (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_wakusatogaiRomInfo, fds_wakusatogaiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_wardnernomoriRomDesc[] = {
	{ "Wardner no Mori (Japan).fds",          131016, 0xb5a04f7e, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_wardnernomori, fds_wardnernomori, fds_fdsbios)
STD_ROM_FN(fds_wardnernomori)

struct BurnDriver BurnDrvfds_wardnernomori = {
	"fds_wardnernomori", NULL, "fds_fdsbios", NULL, "1989?",
	"Wardner no Mori (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_wardnernomoriRomInfo, fds_wardnernomoriRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_wintergamesRomDesc[] = {
	{ "Winter Games (Japan).fds",          131000, 0x6ab5e1d6, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_wintergames, fds_wintergames, fds_fdsbios)
STD_ROM_FN(fds_wintergames)

struct BurnDriver BurnDrvfds_wintergames = {
	"fds_wintergames", NULL, "fds_fdsbios", NULL, "1989?",
	"Winter Games (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_wintergamesRomInfo, fds_wintergamesRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_wreckingcrewRomDesc[] = {
	{ "Wrecking Crew (Japan).fds",          65516, 0xf341f135, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_wreckingcrew, fds_wreckingcrew, fds_fdsbios)
STD_ROM_FN(fds_wreckingcrew)

struct BurnDriver BurnDrvfds_wreckingcrew = {
	"fds_wreckingcrew", NULL, "fds_fdsbios", NULL, "1989?",
	"Wrecking Crew (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_wreckingcrewRomInfo, fds_wreckingcrewRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_youkaiyashikijRomDesc[] = {
	{ "Youkai Yashiki (Japan).fds",          131016, 0xa888c6df, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_youkaiyashikij, fds_youkaiyashikij, fds_fdsbios)
STD_ROM_FN(fds_youkaiyashikij)

struct BurnDriver BurnDrvfds_youkaiyashikij = {
	"fds_youkaiyashikij", "fds_youkaiyashiki", "fds_fdsbios", NULL, "1989?",
	"Youkai Yashiki (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_youkaiyashikijRomInfo, fds_youkaiyashikijRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_youkaiyashikiRomDesc[] = {
	{ "Youkai Yashiki (T-Eng).fds",          131000, 0x546589ab, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_youkaiyashiki, fds_youkaiyashiki, fds_fdsbios)
STD_ROM_FN(fds_youkaiyashiki)

struct BurnDriver BurnDrvfds_youkaiyashiki = {
	"fds_youkaiyashiki", NULL, "fds_fdsbios", NULL, "1989?",
	"Youkai Yashiki (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_youkaiyashikiRomInfo, fds_youkaiyashikiRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_yuumazeRomDesc[] = {
	{ "Yuu Maze (Japan).fds",          131016, 0xde297106, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_yuumaze, fds_yuumaze, fds_fdsbios)
STD_ROM_FN(fds_yuumaze)

struct BurnDriver BurnDrvfds_yuumaze = {
	"fds_yuumaze", NULL, "fds_fdsbios", NULL, "1989?",
	"Yuu Maze (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_yuumazeRomInfo, fds_yuumazeRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_zanacRomDesc[] = {
	{ "Zanac (Japan).fds",          65516, 0xdaa74bc4, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_zanac, fds_zanac, fds_fdsbios)
STD_ROM_FN(fds_zanac)

struct BurnDriver BurnDrvfds_zanac = {
	"fds_zanac", NULL, "fds_fdsbios", NULL, "1989?",
	"Zanac (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_zanacRomInfo, fds_zanacRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_zeldanodenjRomDesc[] = {
	{ "Zelda no Densetsu - The Hyrule Fantasy (Japan).fds",          131016, 0x3fbdddcd, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_zeldanodenj, fds_zeldanodenj, fds_fdsbios)
STD_ROM_FN(fds_zeldanodenj)

struct BurnDriver BurnDrvfds_zeldanodenj = {
	"fds_zeldanodenj", "fds_zeldanoden", "fds_fdsbios", NULL, "1989?",
	"Zelda no Densetsu - The Hyrule Fantasy (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_zeldanodenjRomInfo, fds_zeldanodenjRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo fds_zeldanodenRomDesc[] = {
	{ "Zelda no Densetsu - The Hyrule Fantasy (T-Eng).fds",          131016, 0xee11aa63, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_zeldanoden, fds_zeldanoden, fds_fdsbios)
STD_ROM_FN(fds_zeldanoden)

struct BurnDriver BurnDrvfds_zeldanoden = {
	"fds_zeldanoden", NULL, "fds_fdsbios", NULL, "1989?",
	"Zelda no Densetsu - The Hyrule Fantasy (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_zeldanodenRomInfo, fds_zeldanodenRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


// FDS END ---------------------------------------------------------------


// CPU TESTS
/*
#define WINNERS_DONT_USE_DRUGS 0
#if WINNERS_DONT_USE_DRUGS

static struct BurnRomInfo nestestRomDesc[] = {
	{ "nestest.nes",          24592, 0x9e179d92, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(nestest)
STD_ROM_FN(nestest)

struct BurnDriver BurnDrvnestest = {
	"nes_nestest", NULL, NULL, NULL, "1986",
	"nestest\0", NULL, "Kevtris", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName, nestestRomInfo, nestestRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


// PPU Tests

static struct BurnRomInfo nes_fullRomDesc[] = {
	{ "full_nes_palette2.nes",          16400, 0x85784e11, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_full)
STD_ROM_FN(nes_full)

struct BurnDriver BurnDrvnes_full = {
	"nes_full", NULL, NULL, NULL, "1989?",
	"full nes palette (Blargg)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_fullRomInfo, nes_fullRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bpaltestRomDesc[] = {
	{ "palette_ram.nes",          16400, 0x5d447e3c, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bpaltest)
STD_ROM_FN(bpaltest)

struct BurnDriver BurnDrvbpaltest = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"Blargg pal ram test\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName, bpaltestRomInfo, bpaltestRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bpalpowertestRomDesc[] = {
	{ "power_up_palette.nes",          16400, 0x156f41f0, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bpalpowertest)
STD_ROM_FN(bpalpowertest)

struct BurnDriver BurnDrvbpalpowertest = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"Blargg pal power on test\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName, bpalpowertestRomInfo, bpalpowertestRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bspriteramRomDesc[] = {
	{ "sprite_ram.nes",          16400, 0xd8d42111, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bspriteram)
STD_ROM_FN(bspriteram)

struct BurnDriver BurnDrvbspriteram = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"Blargg sprite ram\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName, bspriteramRomInfo, bspriteramRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bvblRomDesc[] = {
	{ "vbl_clear_time.nes",          16400, 0x1e381801, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bvbl)
STD_ROM_FN(bvbl)

struct BurnDriver BurnDrvbvbl = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"VBL test\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bvblRomInfo, bvblRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bvramRomDesc[] = {
	{ "vram_access.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bvram)
STD_ROM_FN(bvram)

struct BurnDriver BurnDrvbbvram = {
	"nes_blarggppu", NULL, NULL, NULL, "1986",
	"VRAM test\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bvramRomInfo, bvramRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


static struct BurnRomInfo bsprite1RomDesc[] = {
	{ "01.basics.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite1)
STD_ROM_FN(bsprite1)

struct BurnDriver BurnDrvbsprite1 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 1\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite1RomInfo, bsprite1RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bsprite2RomDesc[] = {
	{ "02.alignment.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite2)
STD_ROM_FN(bsprite2)

struct BurnDriver BurnDrvbsprite2 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 2\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite2RomInfo, bsprite2RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bsprite3RomDesc[] = {
	{ "03.corners.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite3)
STD_ROM_FN(bsprite3)

struct BurnDriver BurnDrvbsprite3 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 3\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite3RomInfo, bsprite3RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


static struct BurnRomInfo bsprite4RomDesc[] = {
	{ "04.flip.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite4)
STD_ROM_FN(bsprite4)

struct BurnDriver BurnDrvbsprite4 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 4\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite4RomInfo, bsprite4RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bsprite5RomDesc[] = {
	{ "05.left_clip.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite5)
STD_ROM_FN(bsprite5)

struct BurnDriver BurnDrvbsprite5 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 5\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite5RomInfo, bsprite5RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bsprite6RomDesc[] = {
	{ "06.right_edge.nes",          16400, 0xee115c9a, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bsprite6)
STD_ROM_FN(bsprite6)

struct BurnDriver BurnDrvbsprite6 = {
	"nes_bsprites", NULL, NULL, NULL, "1986",
	"sprite test 6\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bsprite6RomInfo, bsprite6RomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bcputimeRomDesc[] = {
	{ "cpu_timing_test.nes",          16400, 0x161bc1fe, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bcputime)
STD_ROM_FN(bcputime)

struct BurnDriver BurnDrvbcputime = {
	"nes_bcputime", NULL, NULL, NULL, "1986",
	"bcputime\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bcputimeRomInfo, bcputimeRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo bcpuallRomDesc[] = {
	{ "all_instrs.nes",          262160, 0xA4400963, BRF_ESS | BRF_PRG },
};


STD_ROM_PICK(bcpuall)
STD_ROM_FN(bcpuall)

struct BurnDriver BurnDrvbcpuall = {
	"nes_bcputime", NULL, NULL, NULL, "1986",
	"cpu all instructions\0", NULL, "Blargg", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING , 2, HARDWARE_NES, GBF_SHOOT,0 ,
	NESGetZipName,bcpuallRomInfo,bcpuallRomName,NULL,NULL,NULL,NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ruderRomDesc[] = {
	{ "ruder (HB).nes",          24592, 0x4c5c6008, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ruder)
STD_ROM_FN(nes_ruder)

struct BurnDriver BurnDrvnes_ruder = {
	"nes_ruder", NULL, NULL, NULL, "1989?",
	"Zapper ruder (test) (HB)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ruderRomInfo, nes_ruderRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_allpadsRomDesc[] = {
	{ "allpads.nes",          32784, 0x9180a163, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_allpads)
STD_ROM_FN(nes_allpads)

struct BurnDriver BurnDrvnes_allpads = {
	"nes_allpads", NULL, NULL, NULL, "2016",
	"allpads\0", NULL, "tepples", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_allpadsRomInfo, nes_allpadsRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NESHori4pInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_apudinknoiseRomDesc[] = {
	{ "apu_dink_noise.nes",          32784, 0xb19c71f8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_apudinknoise)
STD_ROM_FN(nes_apudinknoise)

struct BurnDriver BurnDrvnes_apudinknoise = {
	"nes_apudinknoise", NULL, NULL, NULL, "1989?",
	"apu_dink_noise\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_apudinknoiseRomInfo, nes_apudinknoiseRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

#endif
*/
// Non Homebrew (hand-added!)

static INT32 topriderInit()
{
	INT32 rc = NESInit();

	if (!rc) {
		// Top Rider size / crc: 163856, 0xca1a395a
		// Patch in standard controller -dink feb.25, 2021
		// Game Genie code: IUEOKOAL
		// lda $494
		// ora $311
		// and #$10
		//*bne 99c5 ; start pressed, next screen
		// jsr 9f68 ; check if code entered on controller #2
		// bcs 99c2 ; yes? start game with normal controller in port #1
		// ...
		//*d0 38  bne 99c5 -> d0 35  bne 99c2

		Cart.PRGRom[0x1998c] = 0x35;
	}

	return rc;
}

static struct BurnRomInfo nes_topriderRomDesc[] = {
	{ "Top Rider (Japan).nes",          163856, 0xca1a395a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_toprider)
STD_ROM_FN(nes_toprider)

struct BurnDriver BurnDrvnes_toprider = {
	"nes_toprider", NULL, NULL, NULL, "1988",
	"Top Rider (Japan)\0", NULL, "Varie", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_topriderRomInfo, nes_topriderRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	topriderInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_samuraispiritsRomDesc[] = {
	{ "Samurai Spirits (Unl).nes",          786448, 0x9b7305f7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_samuraispirits)
STD_ROM_FN(nes_samuraispirits)

struct BurnDriver BurnDrvnes_samuraispirits = {
	"nes_samuraispirits", NULL, NULL, NULL, "199x",
	"Samurai Spirits (Unl)\0", NULL, "Rex Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_samuraispiritsRomInfo, nes_samuraispiritsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_avpregirfigRomDesc[] = {
	{ "AV Pretty Girl Fight (Unl).nes",          655376, 0xc3d2b090, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_avpregirfig)
STD_ROM_FN(nes_avpregirfig)

struct BurnDriver BurnDrvnes_avpregirfig = {
	"nes_avpregirfig", NULL, NULL, NULL, "1994",
	"AV Pretty Girl Fight (Unl)\0", NULL, "Someri Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_avpregirfigRomInfo, nes_avpregirfigRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_janggunuiadeulRomDesc[] = {
	{ "Janggun-ui Adeul (Korea) (Unl).nes",          655376, 0x54171ca4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_janggunuiadeul)
STD_ROM_FN(nes_janggunuiadeul)

struct BurnDriver BurnDrvnes_janggunuiadeul = {
	"nes_janggunuiadeul", NULL, NULL, NULL, "1992",
	"Janggun-ui Adeul (Korea) (Unl)\0", NULL, "Daou", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_janggunuiadeulRomInfo, nes_janggunuiadeulRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Famista '93 (T-Eng)
// https://www.romhacking.net/
static struct BurnRomInfo nes_famista93eRomDesc[] = {
	{ "famista '93 (t-eng).nes",          262160, 0xd9301c12, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famista93e)
STD_ROM_FN(nes_famista93e)

struct BurnDriver BurnDrvnes_famista93e = {
	"nes_famista93e", "nes_famista93", NULL, NULL, "1993",
	"Famista '93 (T-Eng)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famista93eRomInfo, nes_famista93eRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ultimexoremRomDesc[] = {
	{ "Ultima - Exodus Remastered (USA)(Hack).nes",          262160, 0x8afe467a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ultimexorem)
STD_ROM_FN(nes_ultimexorem)

struct BurnDriver BurnDrvnes_ultimexorem = {
	"nes_ultimexorem", "nes_ultimaexodus", NULL, NULL, "2020",
	"Ultima - Exodus Remastered (USA)(Hack)\0", NULL, "Fox Cunning", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ultimexoremRomInfo, nes_ultimexoremRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_supermarallstanesRomDesc[] = {
	{ "Super Mario All Stars NES (Hack).nes",          2097168, 0xbe155d3e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_supermarallstanes)
STD_ROM_FN(nes_supermarallstanes)

struct BurnDriver BurnDrvnes_supermarallstanes = {
	"nes_supermarioallst", NULL, NULL, NULL, "2020",
	"Super Mario All Stars NES (Hack)\0", NULL, "infidelity", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_supermarallstanesRomInfo, nes_supermarallstanesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smbchrediRomDesc[] = {
	{ "Super Mario Bros. Christmas Edition (Hack).nes",          73744, 0xb293a7c4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbchredi)
STD_ROM_FN(nes_smbchredi)

struct BurnDriver BurnDrvnes_smbchredi = {
	"nes_smbchredi", "nes_smb", NULL, NULL, "2020",
	"Super Mario Bros. Christmas Edition (Hack)\0", NULL, "U Retro", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_smbchrediRomInfo, nes_smbchrediRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smbendssmbha1RomDesc[] = {
	{ "Super Mario Bros. Ende's SMB Hack 1 (Hack).nes",          40976, 0x1eeab4b1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbendssmbha1)
STD_ROM_FN(nes_smbendssmbha1)

struct BurnDriver BurnDrvnes_smbendssmbha1 = {
	"nes_smbendssmbha1", "nes_smb", NULL, NULL, "2004",
	"Super Mario Bros. Ende's SMB Hack 1 (Hack)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_smbendssmbha1RomInfo, nes_smbendssmbha1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_maniacmanuncRomDesc[] = {
	{ "Maniac Mansion Uncensored (Hack).nes",          262160, 0x8a13acc3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmanunc)
STD_ROM_FN(nes_maniacmanunc)

struct BurnDriver BurnDrvnes_maniacmanunc = {
	"nes_maniacmanunc", "nes_maniacmansion", NULL, NULL, "2020",
	"Maniac Mansion Uncensored (Hack)\0", NULL, "Gzip", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmanuncRomInfo, nes_maniacmanuncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blockoutRomDesc[] = {
	{ "Block Out (USA).nes",          131088, 0xdeff01f6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blockout)
STD_ROM_FN(nes_blockout)

struct BurnDriver BurnDrvnes_blockout = {
	"nes_blockout", NULL, NULL, NULL, "1989?",
	"Block Out (USA)\0", NULL, "California Dreams / Technos", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blockoutRomInfo, nes_blockoutRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bookymanRomDesc[] = {
	{ "Bookyman (1991)(Mega Soft).nes",          24592, 0xd8f11b78, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bookyman)
STD_ROM_FN(nes_bookyman)

struct BurnDriver BurnDrvnes_bookyman = {
	"nes_bookyman", NULL, NULL, NULL, "1991",
	"Bookyman (Unl)\0", NULL, "Mega Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bookymanRomInfo, nes_bookymanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_destroyerRomDesc[] = {
	{ "Destroyer (1992)(Mega Soft).nes",          65552, 0xd9362123, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_destroyer)
STD_ROM_FN(nes_destroyer)

struct BurnDriver BurnDrvnes_destroyer = {
	"nes_destroyer", NULL, NULL, NULL, "1992",
	"Destroyer (Unl)\0", NULL, "Mega Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_destroyerRomInfo, nes_destroyerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_somariRomDesc[] = {
	{ "Somari (Unl).nes",          524304, 0x51589daf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_somari)
STD_ROM_FN(nes_somari)

struct BurnDriver BurnDrvnes_somari = {
	"nes_somari", NULL, NULL, NULL, "1994",
	"Somari (Unl)\0", NULL, "Hummer Team", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_somariRomInfo, nes_somariRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Sonic Improvement vol.2
// http://www.romhacking.net/hacks/5026/
static struct BurnRomInfo nes_sonicRomDesc[] = {
	{ "Sonic the Hedgehog (Hack).nes",          786448, 0xb9ebd682, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sonic)
STD_ROM_FN(nes_sonic)

struct BurnDriver BurnDrvnes_sonic = {
	"nes_sonic", "nes_somari", NULL, NULL, "2020",
	"Sonic the Hedgehog (Hack)\0", NULL, "Terwilf", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sonicRomInfo, nes_sonicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Bowsette 2 (Hack, Red version)
// https://www.romhacking.net/hacks/4326/

static struct BurnRomInfo nes_bowsette2RomDesc[] = {
	{ "Bowsette 2 (Hack).nes",          262160, 0x94390680, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bowsette2)
STD_ROM_FN(nes_bowsette2)

struct BurnDriver BurnDrvnes_bowsette2 = {
	"nes_bowsette2", "nes_smb2", NULL, NULL, "2019",
	"Bowsette 2 (Hack, Red version)\0", NULL, "hack", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bowsette2RomInfo, nes_bowsette2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Unlimited Deluxe
// http://www.romhacking.net/hacks/5546/
static struct BurnRomInfo nes_smbunldlxRomDesc[] = {
	{ "Super Mario Unlimited Deluxe (Hack).nes",          196624, 0x716c9eb4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbunldlx)
STD_ROM_FN(nes_smbunldlx)

struct BurnDriver BurnDrvnes_smbunldlx = {
	"nes_smbunldlx", "nes_smb", NULL, NULL, "2020",
	"Super Mario Unlimited Deluxe (Hack)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_smbunldlxRomInfo, nes_smbunldlxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Little Samson
// Restoration Hack by SCD @ https://www.romhacking.net/hacks/5413/
// Pleasant Shot Sound hack by Nesrocks @ https://www.romhacking.net/hacks/5189/

static struct BurnRomInfo nes_littlesamsonhRomDesc[] = {
	{ "Little Samson (Hack).nes",          393232, 0x2178190e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_littlesamsonh)
STD_ROM_FN(nes_littlesamsonh)

struct BurnDriver BurnDrvnes_littlesamsonh = {
	"nes_littlesamsonh", "nes_littlesamson", NULL, NULL, "1992",
	"Little Samson (Restoration & Sound Hack)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_littlesamsonhRomInfo, nes_littlesamsonhRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Aliens - Alien 2 (Fix Hack)
// Hack by lancuster
// https://www.romhacking.net/hacks/3790/
static struct BurnRomInfo fds_aliensfixRomDesc[] = {
	{ "Aliens - Alien 2 (Prototype)(Fix Hack).fds",          131016, 0x0d487b19, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_aliensfix, fds_aliensfix, fds_fdsbios)
STD_ROM_FN(fds_aliensfix)

struct BurnDriver BurnDrvfds_aliensfix = {
	"fds_aliensfix", NULL, "fds_fdsbios", NULL, "2017-12-30",
	"Aliens - Alien 2 (Fix Hack)\0", NULL, "lancuster", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_aliensfixRomInfo, fds_aliensfixRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Bubble Bath Babes (USA) (Unl)
static struct BurnRomInfo nes_bbbabesRomDesc[] = {
	{ "Bubble Bath Babes (USA) (Unl).nes",          98320, 0xf36edcc5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bbbabes)
STD_ROM_FN(nes_bbbabes)

struct BurnDriver BurnDrvnes_bbbabes = {
	"nes_bbbabes", NULL, NULL, NULL, "1991",
	"Bubble Bath Babes (USA) (Unl)\0", NULL, "Panesian", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bbbabesRomInfo, nes_bbbabesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Jungle Book, The (Brightness fix)
// Hacked by Ilya Indigo
// https://www.romhacking.net/hacks/5323/

static struct BurnRomInfo nes_junglebookthebRomDesc[] = {
	{ "jungle book, the (2020)(brightness fix).nes",          262160, 0x1c9a2869, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_junglebooktheb)
STD_ROM_FN(nes_junglebooktheb)

struct BurnDriver BurnDrvnes_junglebooktheb = {
	"nes_junglebooktheb", "nes_junglebookthe", NULL, NULL, "2020-08-09",
	"Jungle Book, The (Brightness fix)\0", NULL, "Ilya Indigo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_junglebookthebRomInfo, nes_junglebookthebRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Royal Pals (Hack)
// Hacked by QUXNLADY
// https://www.romhacking.net/hacks/5320/

static struct BurnRomInfo nes_srpRomDesc[] = {
	{ "super royal pals (2020)(QUXNLADY).nes",          40976, 0x62797e6e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_srp)
STD_ROM_FN(nes_srp)

struct BurnDriver BurnDrvnes_srp = {
	"nes_srp", "nes_smb", NULL, NULL, "2020-08-06",
	"Super Royal Pals (Hack)\0", NULL, "QUXNLADY", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_srpRomInfo, nes_srpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_rockm4miRomDesc[] = {
	{ "Rockman 4 - Minus Infinity (Hack).nes",          1048592, 0xe9887ba6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rockm4mi)
STD_ROM_FN(nes_rockm4mi)

struct BurnDriver BurnDrvnes_rockm4mi = {
	"nes_rockm4mi", "nes_megaman4", NULL, NULL, "2015",
	"Rockman 4 - Minus Infinity (Hack)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_rockm4miRomInfo, nes_rockm4miRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


// Pacman CE
static struct BurnRomInfo nes_pacmanceRomDesc[] = {
	{ "pacman-ce (2020).nes",          262160, 0xb86c09af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pacmance)
STD_ROM_FN(nes_pacmance)

struct BurnDriver BurnDrvnes_pacmance = {
	"nes_pacmance", NULL, NULL, NULL, "2020",
	"Pac-Man Championship Edition\0", NULL, "Namco - BNEI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MAZE, 0,
	NESGetZipName, nes_pacmanceRomInfo, nes_pacmanceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gaplus
static struct BurnRomInfo nes_gaplusRomDesc[] = {
	{ "gaplus (bnei).nes",          262160, 0x60811720, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gaplus)
STD_ROM_FN(nes_gaplus)

struct BurnDriver BurnDrvnes_gaplus = {
	"nes_gaplus", NULL, NULL, NULL, "2020",
	"Gaplus\0", NULL, "Namco - BNEI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_gaplusRomInfo, nes_gaplusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saint Seiya - Ougon Densetsu (T-Chi, v2.0)
// Translation by Han Hua Ni Mei Team - Jiang Wei Di Er & Fen Mo & Zeng Ge
static struct BurnRomInfo nes_saintseiyaodcRomDesc[] = {
	{ "Saint Seiya - Ougon Densetsu (T-Chi, v2.0).nes",          1310736, 0xf7df4b56, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saintseiyaodc)
STD_ROM_FN(nes_saintseiyaodc)

struct BurnDriver BurnDrvnes_saintseiyaodc = {
	"nes_saintseiyaodc", "nes_saintseiougden", NULL, NULL, "2018-08-11",
	"Saint Seiya - Ougon Densetsu (T-Chi, v2.0)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_saintseiyaodcRomInfo, nes_saintseiyaodcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saint Seiya - Ougon Densetsu Kanketsu Hen (T-Chi)
// Translation by Han Hua Ni Mei Team - Jiang Wei Di Er & Xiao Ben Ben
static struct BurnRomInfo nes_saintseiyaodkhcRomDesc[] = {
	{ "Saint Seiya - Ougon Densetsu Kanketsu Hen (T-Chi).nes",          524304, 0xb1651b70, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saintseiyaodkhc)
STD_ROM_FN(nes_saintseiyaodkhc)

struct BurnDriver BurnDrvnes_saintseiyaodkhc = {
	"nes_saintseiyaodkhc", "nes_saintseiougdenkahen", NULL, NULL, "2017-01-26",
	"Saint Seiya - Ougon Densetsu Kanketsu Hen (T-Chi)\0", NULL, "Shinsei", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_saintseiyaodkhcRomInfo, nes_saintseiyaodkhcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// AV Mahjong Club (T-Chi) (Unl)
// Translation by Han Hua Ni Mei Team - Shao Nian Bu Zhi Chou
static struct BurnRomInfo nes_avmahjongclubcRomDesc[] = {
	{ "AV Mahjong Club (T-Chi) (Unl).nes",          262160, 0x663870dd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_avmahjongclubc)
STD_ROM_FN(nes_avmahjongclubc)

struct BurnDriver BurnDrvnes_avmahjongclubc = {
	"nes_avmahjongclubc", NULL, NULL, NULL, "2018-08-17",
	"AV Mahjong Club (T-Chi) (Unl)\0", NULL, "Hack International", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_avmahjongclubcRomInfo, nes_avmahjongclubcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Jackal (T-Chi, v1.1)
// Translation & Support 4 Players by Lei Jing Ling
// mapper 19
static struct BurnRomInfo nes_jackalcRomDesc[] = {
	{ "Jackal (T-Chi, v1.1).nes",          262160, 0x0eca3358, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jackalc)
STD_ROM_FN(nes_jackalc)

struct BurnDriver BurnDrvnes_jackalc = {
	"nes_jackalc", "nes_jackal", NULL, NULL, "2020-02-12",
	"Jackal (T-Chi, v1.1)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 4, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_jackalcRomInfo, nes_jackalcRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NESHori4pInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Darkwing Duck (T-Chi)
// Translation by Han Hua Ni Mei Team - Shao Nian Bu Zhi Chou
static struct BurnRomInfo nes_darkwingduckcRomDesc[] = {
	{ "Darkwing Duck (T-Chi).nes",          393232, 0x585f3500, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_darkwingduckc)
STD_ROM_FN(nes_darkwingduckc)

struct BurnDriver BurnDrvnes_darkwingduckc = {
	"nes_darkwingduckc", "nes_darkwingduck", NULL, NULL, "2020-10-01",
	"Darkwing Duck (T-Chi)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_darkwingduckcRomInfo, nes_darkwingduckcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Guardic Gaiden (T-Chi)
// Translation by Han Hua Ni Mei Team - Shao Nian Bu Zhi Chou & Thirteen & Xing Ye Zhi Huan
static struct BurnRomInfo nes_guardicgaidencRomDesc[] = {
	{ "Guardic Gaiden (T-Chi, v1.1).nes",          262160, 0x77df8d83, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_guardicgaidenc)
STD_ROM_FN(nes_guardicgaidenc)

struct BurnDriver BurnDrvnes_guardicgaidenc = {
	"nes_guardicgaidenc", "nes_guardleg", NULL, NULL, "2017-07-07",
	"Guardic Gaiden (T-Chi, v1.1)\0", NULL, "Irem", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_guardicgaidencRomInfo, nes_guardicgaidencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Ryuuken Den III - Yomi no Hakobune (T-Chi)
// Translation by TPU
static struct BurnRomInfo nes_ninjaryudeniiicRomDesc[] = {
	{ "Ninja Ryuuken Den III - Yomi no Hakobune (T-Chi).nes",          344080, 0xba865119, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjaryudeniiic)
STD_ROM_FN(nes_ninjaryudeniiic)

struct BurnDriver BurnDrvnes_ninjaryudeniiic = {
	"nes_ninjaryudeniiic", "nes_ninjagaiiii", NULL, NULL, "2002",
	"Ninja Ryuuken Den III - Yomi no Hakobune (T-Chi)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ninjaryudeniiicRomInfo, nes_ninjaryudeniiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Ryuuken Den II - Ankoku no Jashin Ken (T-Chi)
// Translation by TPU
static struct BurnRomInfo nes_ninjaryudeniicRomDesc[] = {
	{ "Ninja Ryuuken Den II - Ankoku no Jashin Ken (T-Chi).nes",          335888, 0xf7d5d63b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjaryudeniic)
STD_ROM_FN(nes_ninjaryudeniic)

struct BurnDriver BurnDrvnes_ninjaryudeniic = {
	"nes_ninjaryudeniic", "nes_ninjagaiii", NULL, NULL, "200?",
	"Ninja Ryuuken Den II - Ankoku no Jashin Ken (T-Chi)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ninjaryudeniicRomInfo, nes_ninjaryudeniicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Ryuuken Den (T-Chi)
// Translation by FlameCyclone
static struct BurnRomInfo nes_ninjaryudencRomDesc[] = {
	{ "Ninja Ryuuken Den (T-Chi).nes",          524304, 0x7724d3d0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjaryudenc)
STD_ROM_FN(nes_ninjaryudenc)

struct BurnDriver BurnDrvnes_ninjaryudenc = {
	"nes_ninjaryudenc", "nes_ninjagaiden", NULL, NULL, "2017-06-13",
	"Ninja Ryuuken Den (T-Chi)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ninjaryudencRomInfo, nes_ninjaryudencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ningen Heiki - Dead Fox (T-Chi, v2.0)
// Translation by Xing Kong
static struct BurnRomInfo nes_deadfoxcRomDesc[] = {
	{ "Ningen Heiki - Dead Fox (T-Chi, v2.0).nes",          303120, 0x3cf5df56, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deadfoxc)
STD_ROM_FN(nes_deadfoxc)

struct BurnDriver BurnDrvnes_deadfoxc = {
	"nes_deadfoxc", "nes_codenam", NULL, NULL, "2010",
	"Ningen Heiki - Dead Fox (T-Chi, v2.0)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_deadfoxcRomInfo, nes_deadfoxcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Kage (T-Chi, v2.0)
// Translation by Xing Kong
static struct BurnRomInfo nes_kagecRomDesc[] = {
	{ "Kage (T-Chi, v2.0).nes",          270352, 0xa481e2bd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kagec)
STD_ROM_FN(nes_kagec)

struct BurnDriver BurnDrvnes_kagec = {
	"nes_kagec", "nes_shadonin", NULL, NULL, "2010",
	"Kage (T-Chi, v2.0)\0", NULL, "NATSUME", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_kagecRomInfo, nes_kagecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Bros. 3 (T-Chi)
// Translation by NOKOH
static struct BurnRomInfo nes_smb3cRomDesc[] = {
	{ "Super Mario Bros. 3 (T-Chi).nes",          458768, 0xe39b725d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb3c)
STD_ROM_FN(nes_smb3c)

struct BurnDriver BurnDrvnes_smb3c = {
	"nes_smb3c", "nes_smb3", NULL, NULL, "2016-06-03",
	"Super Mario Bros. 3 (T-Chi)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_smb3cRomInfo, nes_smb3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Akumajou Dracula (T-Chi)
// Translation by Fo Yi Team - Xi Xue Nan Jue
static struct BurnRomInfo nes_akumajoudraculacRomDesc[] = {
	{ "Akumajou Dracula (T-Chi).nes",          131088, 0xdc749e67, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumajoudraculac)
STD_ROM_FN(nes_akumajoudraculac)

struct BurnDriver BurnDrvnes_akumajoudraculac = {
	"nes_akumajoudraculac", "nes_castlevania", NULL, NULL, "1993",
	"Akumajou Dracula (T-Chi)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akumajoudraculacRomInfo, nes_akumajoudraculacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Contra (T-Chi)
// Translation by FlameCyclone
static struct BurnRomInfo nes_contracRomDesc[] = {
	{ "Contra (T-Chi).nes",          393232, 0x2a39628d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contrac)
STD_ROM_FN(nes_contrac)

struct BurnDriver BurnDrvnes_contrac = {
	"nes_contrac", "nes_contra", NULL, NULL, "2017-07-29",
	"Contra (T-Chi)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contracRomInfo, nes_contracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Wai Wai World (T-Chi)
// Translation by NOKOH
static struct BurnRomInfo nes_waiwaiworldcRomDesc[] = {
	{ "Wai Wai World (T-Chi).nes",          524304, 0xae26fdde, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_waiwaiworldc)
STD_ROM_FN(nes_waiwaiworldc)

struct BurnDriver BurnDrvnes_waiwaiworldc = {
	"nes_waiwaiworldc", "nes_waiwaiworld", NULL, NULL, "2012-09-04",
	"Wai Wai World (T-Chi)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_waiwaiworldcRomInfo, nes_waiwaiworldcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saiyuuki World (T-Chi, v1.04)
// Translation by Han Hua Ni Mei Team - Thirteen & Fen Mo & Jiang Wei Di Er & Zeng Ge & Xing Ye Zhi Huan & N0PFUUN
static struct BurnRomInfo nes_saiyuukiworldcRomDesc[] = {
	{ "Saiyuuki World (T-Chi, v1.04).nes",          262160, 0x6e66cf6f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saiyuukiworldc)
STD_ROM_FN(nes_saiyuukiworldc)

struct BurnDriver BurnDrvnes_saiyuukiworldc = {
	"nes_saiyuukiworldc", "nes_saiyuukiworld", NULL, NULL, "2016-02-07",
	"Saiyuuki World (T-Chi, v1.04)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_saiyuukiworldcRomInfo, nes_saiyuukiworldcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mad City (T-Chi, v2)
// Translation by Tan Xiao Feng Sheng
static struct BurnRomInfo nes_madcitycRomDesc[] = {
	{ "Mad City (T-Chi, v2).nes",          393232, 0x3b2ad28b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_madcityc)
STD_ROM_FN(nes_madcityc)

struct BurnDriver BurnDrvnes_madcityc = {
	"nes_madcityc", "nes_advenbaybil", NULL, NULL, "2018",
	"Mad City (T-Chi, v2)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_madcitycRomInfo, nes_madcitycRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Contra Force (2nd option, T-Chi)
// Translation by Tan Xiao Feng Sheng
static struct BurnRomInfo nes_contraforcecRomDesc[] = {
	{ "Contra Force (T-Chi).nes",          524304, 0xcf2a15fb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contraforcec)
STD_ROM_FN(nes_contraforcec)

struct BurnDriver BurnDrvnes_contraforcec = {
	"nes_contraforcec", "nes_contraforce", NULL, NULL, "2018",
	"Contra Force (2nd option, T-Chi)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contraforcecRomInfo, nes_contraforcecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gun Nac (T-Chi)
// Translation by Jiu Ban & KASURA·J
static struct BurnRomInfo nes_gunnaccRomDesc[] = {
	{ "Gun Nac (T-Chi).nes",          393232, 0x3c63f85a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gunnacc)
STD_ROM_FN(nes_gunnacc)

struct BurnDriver BurnDrvnes_gunnacc = {
	"nes_gunnacc", "nes_gunnac", NULL, NULL, "2013-10-31",
	"Gun Nac (T-Chi)\0", NULL, "Tonkin House", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_gunnaccRomInfo, nes_gunnaccRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mitsume ga Tooru (T-Chi)
// Translation by Xing Kong
static struct BurnRomInfo nes_mitsumegatoorucRomDesc[] = {
	{ "Mitsume ga Tooru (T-Chi).nes",          278544, 0x296c2307, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mitsumegatooruc)
STD_ROM_FN(nes_mitsumegatooruc)

struct BurnDriver BurnDrvnes_mitsumegatooruc = {
	"nes_mitsumegatooruc", "nes_mitsumegatooru", NULL, NULL, "2010-10-10",
	"Mitsume ga Tooru (T-Chi)\0", NULL, "tomy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_mitsumegatoorucRomInfo, nes_mitsumegatoorucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Akumajou Densetsu (T-Chi, v2.3)
// Translation by Han Hua Ni Mei Team - Zeng Ge & Jiang Wei Di Er & Xing Ye Zhi Huan
static struct BurnRomInfo nes_akumadencRomDesc[] = {
	{ "Akumajou Densetsu (T-Chi, v2.3).nes",          524304, 0xb810b79c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumadenc)
STD_ROM_FN(nes_akumadenc)

struct BurnDriver BurnDrvnes_akumadenc = {
	"nes_akumadenc", "nes_castliii", NULL, NULL, "2018-10-15",
	"Akumajou Densetsu (T-Chi, v2.3)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akumadencRomInfo, nes_akumadencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Batman - The Video Game (T-Chi)
// Translation by Han Hua Ni Mei Team - Shao Nian Bu Zhi Chou & Luo Yun
static struct BurnRomInfo nes_batmavidgamcRomDesc[] = {
	{ "Batman - The Video Game (T-Chi).nes",          393232, 0x4b2de665, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batmavidgamc)
STD_ROM_FN(nes_batmavidgamc)

struct BurnDriver BurnDrvnes_batmavidgamc = {
	"nes_batmavidgamc", "nes_batmavidgam", NULL, NULL, "2020-01-25",
	"Batman - The Video Game (T-Chi)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_batmavidgamcRomInfo, nes_batmavidgamcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Salamander (T-Chi, v1.2)
// Translation by Lei Jing Ling
// mapper 19
static struct BurnRomInfo nes_salamandercRomDesc[] = {
	{ "Salamander (T-Chi, v1.2).nes",          393232, 0xfbe84b97, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_salamanderc)
STD_ROM_FN(nes_salamanderc)

struct BurnDriver BurnDrvnes_salamanderc = {
	"nes_salamanderc", "nes_salamander", NULL, NULL, "2020-04-30",
	"Salamander (T-Chi, v1.2)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_salamandercRomInfo, nes_salamandercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gun-Dec (T-Chi)
// Translation by Han Hua Ni Mei Team - Shao Nian Bu Zhi Chou & Jiang Wei Di Er & Luo Yun
static struct BurnRomInfo nes_gundeccRomDesc[] = {
	{ "Gun-Dec (T-Chi).nes",          393232, 0x647eccfa, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gundecc)
STD_ROM_FN(nes_gundecc)

struct BurnDriver BurnDrvnes_gundecc = {
	"nes_gundecc", "nes_vice", NULL, NULL, "2019-08-01",
	"Gun-Dec (T-Chi)\0", NULL, "Sammy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_gundeccRomInfo, nes_gundeccRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles (T-Chi, v1.0)
// Translation by MEZARANSU & Jiang Wei Di Er & Luo Yun
// mapper 05 (MMC5)
static struct BurnRomInfo nes_tmntcRomDesc[] = {
	{ "Teenage Mutant Ninja Turtles (T-Chi, v1.0).nes",          393232, 0x43ae114f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntc)
STD_ROM_FN(nes_tmntc)

struct BurnDriver BurnDrvnes_tmntc = {
	"nes_tmntc", "nes_tmnt", NULL, NULL, "2020-06-09",
	"Teenage Mutant Ninja Turtles (T-Chi, v1.0)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_tmntcRomInfo, nes_tmntcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Chinese II - Dragon Kid (T-Chi, v1.4)
// Translation by Han Hua Ni Mei Team - Thirteen & Jiang Wei Di Er & Xing Ye Zhi Huan
static struct BurnRomInfo nes_superchiiRomDesc[] = {
	{ "Super Chinese II - Dragon Kid (T-Chi, v1.4).nes",          524304, 0xa9e92ef3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superchii)
STD_ROM_FN(nes_superchii)

struct BurnDriver BurnDrvnes_superchii = {
	"nes_superchii", "nes_littlninbro", NULL, NULL, "2019-01-01",
	"Super Chinese II - Dragon Kid (T-Chi, v1.4)\0", NULL, "Culture Brain", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_superchiiRomInfo, nes_superchiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Majou Densetsu II - Daimashikyou Galious (T-Chi, v2.2)
// Translation by Han Hua Ni Mei Team - Zeng Ge & Jiang Wei Di Er & Luo Yun & Xia Tian & Xing Ye Zhi Huan
static struct BurnRomInfo nes_majoudeniicRomDesc[] = {
	{ "Majou Densetsu II - Daimashikyou Galious (T-Chi, v2.2).nes",          262160, 0xa032170a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_majoudeniic)
STD_ROM_FN(nes_majoudeniic)

struct BurnDriver BurnDrvnes_majoudeniic = {
	"nes_majoudeniic", "nes_majoudenii", NULL, NULL, "2018-06-10",
	"Majou Densetsu II - Daimashikyou Galious (T-Chi, v2.2)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_majoudeniicRomInfo, nes_majoudeniicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Silk Worm (T-Chi, v1.1)
// Translation by Han Hua Ni Mei Team - Shao Nian Bu Zhi Chou
static struct BurnRomInfo nes_silkwormcRomDesc[] = {
	{ "Silk Worm (T-Chi, v1.1).nes",          393232, 0x3d53c8ad, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_silkwormc)
STD_ROM_FN(nes_silkwormc)

struct BurnDriver BurnDrvnes_silkwormc = {
	"nes_silkwormc", "nes_silkworm", NULL, NULL, "2020-01-07",
	"Silk Worm (T-Chi, v1.1)\0", NULL, "Sammy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_silkwormcRomInfo, nes_silkwormcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Argos no Senshi (T-Chi, v1.11)
// Translation by Han Hua Ni Mei Team - Thirteen & Fen Mo & Jiang Wei Di Er & MM Zhi Shen & Xing Ye Zhi Huan & N0PFUUN & Zeng Ge
static struct BurnRomInfo nes_argosnosenshicRomDesc[] = {
	{ "Argos no Senshi (T-Chi, v1.11).nes",          262160, 0xa1a22c10, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_argosnosenshic)
STD_ROM_FN(nes_argosnosenshic)

struct BurnDriver BurnDrvnes_argosnosenshic = {
	"nes_argosnosenshic", "nes_rygar", NULL, NULL, "2016-01-27",
	"Argos no Senshi (T-Chi, v1.11)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_argosnosenshicRomInfo, nes_argosnosenshicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Contra (T-Chi)
// Translation by FlameCyclone
static struct BurnRomInfo nes_supercontracRomDesc[] = {
	{ "Super Contra (T-Chi).nes",          393232, 0x048f7dd8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_supercontrac)
STD_ROM_FN(nes_supercontrac)

struct BurnDriver BurnDrvnes_supercontrac = {
	"nes_supercontrac", "nes_superc", NULL, NULL, "2017-08-01",
	"Super Contra (T-Chi)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_supercontracRomInfo, nes_supercontracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Hebereke (T-Chi, v1.2)
// Translation by Lei Jing Ling
// mapper 69
static struct BurnRomInfo nes_heberekecRomDesc[] = {
	{ "Hebereke (T-Chi, v1.2).nes",          393232, 0x48e8bcee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_heberekec)
STD_ROM_FN(nes_heberekec)

struct BurnDriver BurnDrvnes_heberekecc= {
	"nes_heberekec", "nes_ufouria", NULL, NULL, "2019-07-23",
	"Hebereke (T-Chi, v1.2)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_heberekecRomInfo, nes_heberekecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Gun.Smoke (T-Chi)
// Translation by MEZARANSU & gonerFC
static struct BurnRomInfo nes_gunsmokecRomDesc[] = {
	{ "Gun.Smoke (T-Chi).nes",          163856, 0x4e0b7a7d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gunsmokec)
STD_ROM_FN(nes_gunsmokec)

struct BurnDriver BurnDrvnes_gunsmokec = {
	"nes_gunsmokec", "nes_gunsmoke", NULL, NULL, "2019-11-17",
	"Gun.Smoke (T-Chi)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_gunsmokecRomInfo, nes_gunsmokecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Mappy Kids (T-Chi, v1.1)
// Translation by Lei Jing Ling & Jiang Wei Di Er
// mapper 19
static struct BurnRomInfo nes_mappykidscRomDesc[] = {
	{ "Mappy Kids (T-Chi, v1.1).nes",          524304, 0x159bc653, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mappykidsc)
STD_ROM_FN(nes_mappykidsc)

struct BurnDriver BurnDrvnes_mappykidsc = {
	"nes_mappykidsc", "nes_mappykids", NULL, NULL, "2019-10-25",
	"Mappy Kids (T-Chi, v1.1)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_mappykidscRomInfo, nes_mappykidscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Daiku no Gen-san 2 - Akage no Dan no Gyakushuu (T-Chi)
// Translation by Advance Team & DMG Team - MEZARANSU & Ju Jue Rong Hua
// mapper 065
static struct BurnRomInfo nes_daikunogensan2cRomDesc[] = {
	{ "Daiku no Gen-san 2 - Akage no Dan no Gyakushuu (T-Chi).nes",          524304, 0x66c08cd8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daikunogensan2c)
STD_ROM_FN(nes_daikunogensan2c)

struct BurnDriver BurnDrvnes_daikunogensan2c = {
	"nes_daikunogensan2c", "nes_daikunogensan2", NULL, NULL, "2019-07-28",
	"Daiku no Gen-san 2 - Akage no Dan no Gyakushuu (T-Chi)\0", NULL, "Irem", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_daikunogensan2cRomInfo, nes_daikunogensan2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saiyuuki World 2 - Tenjoukai no Majin (T-Chi)
// Translation by Du Gu Jiu Jian
static struct BurnRomInfo nes_saiyuwor2cRomDesc[] = {
	{ "Saiyuuki World 2 - Tenjoukai no Majin (T-Chi).nes",          262160, 0x784fc411, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saiyuwor2c)
STD_ROM_FN(nes_saiyuwor2c)

struct BurnDriver BurnDrvnes_saiyuwor2c = {
	"nes_saiyuwor2c", "nes_whompem", NULL, NULL, "2019-10-26",
	"Saiyuuki World 2 - Tenjoukai no Majin (T-Chi)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_saiyuwor2cRomInfo, nes_saiyuwor2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// DuckTales 2 (T-Chi, v1.1)
// Translation by DMG Team - MEZARANSU & Nai Niu
// mapper 04 (MMC3)
static struct BurnRomInfo nes_ducktales2cRomDesc[] = {
	{ "DuckTales 2 (T-Chi, v1.1).nes",          262160, 0x794dbb67, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ducktales2c)
STD_ROM_FN(nes_ducktales2c)

struct BurnDriver BurnDrvnes_ducktales2c = {
	"nes_ducktales2c", "nes_ducktales2", NULL, NULL, "2019-06-01",
	"DuckTales 2 (T-Chi, v1.1)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ducktales2cRomInfo, nes_ducktales2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Daiku no Gen-san (T-Chi)
// Translation by MEZARANSU & Nai Niu & Di Tan CEO
// mapper 04 (MMC3)
static struct BurnRomInfo nes_daikunogensancRomDesc[] = {
	{ "Daiku no Gen-san (T-Chi).nes",          393232, 0xf0543c45, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daikunogensanc)
STD_ROM_FN(nes_daikunogensanc)

struct BurnDriver BurnDrvnes_daikunogensanc = {
	"nes_daikunogensanc", "nes_daikunogensan", NULL, NULL, "2019-03-05",
	"Daiku no Gen-san (T-Chi)\0", NULL, "Irem", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_daikunogensancRomInfo, nes_daikunogensancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Jigoku Gokuraku Maru (T-Chi)
// Translation by MEZARANSU & Jin Di
// mapper 04
static struct BurnRomInfo nes_jigokgokmarcRomDesc[] = {
	{ "Jigoku Gokuraku Maru (T-Chi).nes",          393232, 0x375bf357, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jigokgokmarc)
STD_ROM_FN(nes_jigokgokmarc)

struct BurnDriver BurnDrvnes_jigokgokmarc = {
	"nes_jigokgokmarc", "nes_kabuki", NULL, NULL, "2019-11-08",
	"Jigoku Gokuraku Maru (T-Chi)\0", NULL, "Pack-In-Video", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_jigokgokmarcRomInfo, nes_jigokgokmarcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_triforcegcRomDesc[] = {
	{ "Legend of Zelda - Triforce of the Gods (China).nes",          2097168, 0x3a1b4502, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_triforcegc)
STD_ROM_FN(nes_triforcegc)

struct BurnDriver BurnDrvnes_triforcegc = {
	"nes_triforcegc", "nes_triforceg", NULL, NULL, "1989?",
	"Legend of Zelda - Triforce of the Gods (China)\0", NULL, "Waixing", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_triforcegcRomInfo, nes_triforcegcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_triforcegRomDesc[] = {
	{ "Legend of Zelda - Triforce of the Gods (T-Eng).nes",          2097168, 0x0be2d8fd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_triforceg)
STD_ROM_FN(nes_triforceg)

struct BurnDriver BurnDrvnes_triforceg = {
	"nes_triforceg", NULL, NULL, NULL, "1989?",
	"Legend of Zelda - Triforce of the Gods (T-Eng)\0", NULL, "Waixing", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_triforcegRomInfo, nes_triforcegRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// wanpakuduck (T-Chi)
// Translation by Advance Team - Ju Jue Rong Hua & Shao Nian Bu Zhi Chou
static struct BurnRomInfo nes_wanpakuduckRomDesc[] = {
	{ "Wanpaku Duck Yume Bouken (T-Chi).nes",          196624, 0x9fe55e12, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wanpakuduck)
STD_ROM_FN(nes_wanpakuduck)

struct BurnDriver BurnDrvnes_wanpakuduck = {
	"nes_wanpakuduck", "nes_ducktales", NULL, NULL, "2020",
	"Wanpaku Duck Yume Bouken (T-Chi)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_wanpakuduckRomInfo, nes_wanpakuduckRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//  Addams Family, The - Pugsley's Scavenger Hunt (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_addamfampugsschucRomDesc[] = {
	{ "addams family, the - pugsley's scavenger hunt (usa) - castellano v1.1.nes",          262160, 0x54bbec54, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_addamfampugsschuc)
STD_ROM_FN(nes_addamfampugsschuc)

struct BurnDriver BurnDrvnes_addamfampugsschuc = {
	"nes_addamfampugsschuc", "nes_addamfampugsschu", NULL, NULL, "1989 ?",
	"Addams Family, The - Pugsley's Scavenger Hunt (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_addamfampugsschucRomInfo, nes_addamfampugsschucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//  Addams Family, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_addamfamcRomDesc[] = {
	{ "addams family, the (usa) - castellano v1.0.nes",          262160, 0x6bd3e973, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_addamfamc)
STD_ROM_FN(nes_addamfamc)

struct BurnDriver BurnDrvnes_addamfamc = {
	"nes_addamfamc", "nes_addamfam", NULL, NULL, "1989 ?",
	"Addams Family, The (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_addamfamcRomInfo, nes_addamfamcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//  Adventure Island (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_adventureislandcRomDesc[] = {
	{ "adventure island (usa) - castellano v1.0.nes",          65552, 0xabf6745d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_adventureislandc)
STD_ROM_FN(nes_adventureislandc)

struct BurnDriver BurnDrvnes_adventureislandc = {
	"nes_adventureislandc", "nes_adventureisland", NULL, NULL, "1989 ?",
	"Adventure Island (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_adventureislandcRomInfo, nes_adventureislandcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Adventure Island II (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_advenisliicRomDesc[] = {
	{ "adventure island ii (usa) - castellano v1.0.nes",          262160, 0x92f21250, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenisliic)
STD_ROM_FN(nes_advenisliic)

struct BurnDriver BurnDrvnes_advenisliic = {
	"nes_advenisliic", "nes_advenislii", NULL, NULL, "1989 ?",
	"Adventure Island II (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenisliicRomInfo, nes_advenisliicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Adventure Island 3 (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_advenisl3cRomDesc[] = {
	{ "adventure island iii (usa) - castellano v1.1.nes",          262160, 0x31c3056e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenisl3c)
STD_ROM_FN(nes_advenisl3c)

struct BurnDriver BurnDrvnes_advenisl3c = {
	"nes_advenisl3c", "nes_advenisl3", NULL, NULL, "1989 ?",
	"Adventure Island 3 (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenisl3cRomInfo, nes_advenisl3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Alien 3 (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_alien3cRomDesc[] = {
	{ "alien 3 (usa) - castellano v1.1.nes",          262160, 0xd16b91b7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alien3c)
STD_ROM_FN(nes_alien3c)

struct BurnDriver BurnDrvnes_alien3c = {
	"nes_alien3c", "nes_alien3", NULL, NULL, "1989 ?",
	"Alien 3 (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_alien3cRomInfo, nes_alien3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragon Strike (USA)
static struct BurnRomInfo nes_advdddsRomDesc[] = {
	{ "advanced dungeons & dragons - dragon strike (usa).nes",          524304, 0x4c87cdde, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddds)
STD_ROM_FN(nes_advddds)

struct BurnDriver BurnDrvnes_advddds = {
	"nes_advddds", NULL, NULL, NULL, "1990",
	"Advanced Dungeons & Dragons - Dragon Strike (USA)\0", NULL, "Natsume", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advdddsRomInfo, nes_advdddsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragon Strike (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_advdddscRomDesc[] = {
	{ "advanced dungeons & dragons - dragon strike (usa) - castellano v1.0.nes",          524304, 0x22cfb91d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advdddsc)
STD_ROM_FN(nes_advdddsc)

struct BurnDriver BurnDrvnes_advdddsc = {
	"nes_advdddsc", "nes_advddds", NULL, NULL, "1989 ?",
	"Advanced Dungeons & Dragons - Dragon Strike (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advdddscRomInfo, nes_advdddscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragons of Flame (T-eng)
// https://www.romhacking.net/
static struct BurnRomInfo nes_advdddfRomDesc[] = {
	{ "advanced dungeons & dragons - dragons of flame (t-eng).nes",          262160, 0x8196b978, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advdddf)
STD_ROM_FN(nes_advdddf)

struct BurnDriver BurnDrvnes_advdddf = {
	"nes_advdddf", NULL, NULL, NULL, "1989 ?",
	"Advanced Dungeons & Dragons - Dragons of Flame (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advdddfRomInfo, nes_advdddfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Dragons of Flame (Japan)
static struct BurnRomInfo nes_advdddfjRomDesc[] = {
	{ "advanced dungeons & dragons - dragons of flame (japan).nes",          262160, 0x8ba5de24, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advdddfj)
STD_ROM_FN(nes_advdddfj)

struct BurnDriver BurnDrvnes_advdddfj = {
	"nes_advdddfj", "nes_advdddf", NULL, NULL, "1989 ",
	"Advanced Dungeons & Dragons - Dragons of Flame (Japan)\0", NULL, "U.S Gold", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advdddfjRomInfo, nes_advdddfjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Heroes of the Lance (USA)
static struct BurnRomInfo nes_advddhlRomDesc[] = {
	{ "advanced dungeons & dragons - heroes of the lance (usa).nes",          262160, 0xe880d426, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddhl)
STD_ROM_FN(nes_advddhl)

struct BurnDriver BurnDrvnes_advddhl = {
	"nes_advddhl", NULL, NULL, NULL, "1988",
	"Advanced Dungeons & Dragons - Heroes of the Lance (USA)\0", NULL, "U.S Gold", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advddhlRomInfo, nes_advddhlRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Heroes of the Lance (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_advddhlcRomDesc[] = {
	{ "advanced dungeons & dragons - heroes of the lance (usa) - castellano v1.0.nes",          262160, 0x365f9624, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddhlc)
STD_ROM_FN(nes_advddhlc)

struct BurnDriver BurnDrvnes_advddhlc = {
	"nes_advddhlc", "nes_advddhl", NULL, NULL, "1989 ?",
	"Advanced Dungeons & Dragons - Heroes of the Lance (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advddhlcRomInfo, nes_advddhlcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Hillsfar (USA)
static struct BurnRomInfo nes_advddhllfrRomDesc[] = {
	{ "advanced dungeons & dragons - hillsfar (usa).nes",          262160, 0x3849d0ee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddhllfr)
STD_ROM_FN(nes_advddhllfr)

struct BurnDriver BurnDrvnes_advddhllfr = {
	"nes_advddhllfr", NULL, NULL, NULL, "1992",
	"Advanced Dungeons & Dragons - Hillsfar (USA)\0", NULL, "FCI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advddhllfrRomInfo, nes_advddhllfrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Advanced Dungeons & Dragons - Pool of Radiance (USA)
static struct BurnRomInfo nes_advddporRomDesc[] = {
	{ "advanced dungeons & dragons - pool of radiance (usa).nes",          655376, 0xbbea8d23, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advddpor)
STD_ROM_FN(nes_advddpor)

struct BurnDriver BurnDrvnes_advddpor = {
	"nes_advddpor", NULL, NULL, NULL, "1992",
	"Advanced Dungeons & Dragons - Pool of Radiance (USA)\0", NULL, "FCI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advddporRomInfo, nes_advddporRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Battle of Olympus, The (Europe) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_battlolycRomDesc[] = {
	{ "battle of olympus, the (europe) - castellano v1.1.nes",          131088, 0xab9d1a2e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battlolyc)
STD_ROM_FN(nes_battlolyc)

struct BurnDriver BurnDrvnes_battlolyc = {
	"nes_battlolyc", "nes_battloly", NULL, NULL, "1989 ?",
	"Battle of Olympus, The (Europe) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battlolycRomInfo, nes_battlolycRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Battleship (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_battleshipcRomDesc[] = {
	{ "battleship (usa) - castellano v1.0.nes",          65552, 0xcf1ee9d6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battleshipc)
STD_ROM_FN(nes_battleshipc)

struct BurnDriver BurnDrvnes_battleshipc = {
	"nes_battleshipc", "nes_battleship", NULL, NULL, "1989 ?",
	"Battleship (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battleshipcRomInfo, nes_battleshipcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};							

// Bionic Commando (Japan) - Castellano v1.03
// https://www.romhacking.net/
static struct BurnRomInfo nes_bioniccommandocRomDesc[] = {
	{ "bionic commando (japan) - castellano v1.03.nes",          262160, 0xec5f3e79, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bioniccommandoc)
STD_ROM_FN(nes_bioniccommandoc)

struct BurnDriver BurnDrvnes_bioniccommandoc = {
	"nes_bioniccommandoc", "nes_bioniccommando", NULL, NULL, "1989 ?",
	"Bionic Commando (Japan) - Castellano v1.03\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bioniccommandocRomInfo, nes_bioniccommandocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Blaster Master (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_blastermastercRomDesc[] = {
	{ "blaster master (usa) - castellano v1.0.nes",          262160, 0xb63123db, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blastermasterc)
STD_ROM_FN(nes_blastermasterc)

struct BurnDriver BurnDrvnes_blastermasterc = {
	"nes_blastermasterc", "nes_blastermaster", NULL, NULL, "1989 ?",
	"Blaster Master (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blastermastercRomInfo, nes_blastermastercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Bram Stoker's Dracula (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_bramstosdracRomDesc[] = {
	{ "bram stoker's dracula (usa) - castellano v1.0.nes",          262160, 0xd6eb5f83, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bramstosdrac)
STD_ROM_FN(nes_bramstosdrac)

struct BurnDriver BurnDrvnes_bramstosdrac = {
	"nes_bramstosdrac", "nes_bramstosdra", NULL, NULL, "1989 ?",
	"Bram Stoker's Dracula (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bramstosdracRomInfo, nes_bramstosdracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Captain Tsubasa Vol. II - Super Striker (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_captatsuvoliicRomDesc[] = {
	{ "captain tsubasa vol. ii - super striker (japan) - castellano v1.0.nes",          393233, 0x49c692c5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captatsuvoliic)
STD_ROM_FN(nes_captatsuvoliic)

struct BurnDriver BurnDrvnes_captatsuvoliic = {
	"nes_captatsuvoliic", "nes_captatsuvolii", NULL, NULL, "1989 ?",
	"Captain Tsubasa Vol. II - Super Striker (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captatsuvoliicRomInfo, nes_captatsuvoliicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Castlevania (USA) - Castellano 1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_castlevaniacRomDesc[] = {
	{ "castlevania (usa) - castellano 1.0.nes",          131088, 0xde2e6b55, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castlevaniac)
STD_ROM_FN(nes_castlevaniac)

struct BurnDriver BurnDrvnes_castlevaniac = {
	"nes_castlevaniac", "nes_castlevania", NULL, NULL, "1989 ?",
	"Castlevania (USA) - Castellano 1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castlevaniacRomInfo, nes_castlevaniacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Castlevania II - Simon's Quest (USA) - Castellano v2.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_castliicRomDesc[] = {
	{ "castlevania ii - simon's quest (usa) - castellano v2.0.nes",          262160, 0x38d393a9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castliic)
STD_ROM_FN(nes_castliic)

struct BurnDriver BurnDrvnes_castliic = {
	"nes_castliic", "nes_castlii", NULL, NULL, "1989 ?",
	"Castlevania II - Simon's Quest (USA) - Castellano v2.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castliicRomInfo, nes_castliicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Akumajou Densetsu (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_akumadenjcRomDesc[] = {
	{ "akumajou densetsu (japan) - castellano v1.1.nes",          393232, 0x33ead706, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumadenjc)
STD_ROM_FN(nes_akumadenjc)

struct BurnDriver BurnDrvnes_akumadenjc = {
	"nes_akumadenjc", "nes_castliii", NULL, NULL, "1989 ?",
	"Akumajou Densetsu (Japan) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akumadenjcRomInfo, nes_akumadenjcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Saint Seiya - Ougon Densetsu - Castellano v1.0.nes
// https://www.romhacking.net/
static struct BurnRomInfo nes_saintseiougdencRomDesc[] = {
	{ "saint seiya - ougon densetsu - castellano v1.0.nes.nes",          262160, 0x2abdcc86, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saintseiougdenc)
STD_ROM_FN(nes_saintseiougdenc)

struct BurnDriver BurnDrvnes_saintseiougdenc = {
	"nes_saintseiougdenc", "nes_saintseiougden", NULL, NULL, "1989 ?",
	"Saint Seiya - Ougon Densetsu - Castellano v1.0.nes\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_saintseiougdencRomInfo, nes_saintseiougdencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Chip 'n Dale - Rescue Rangers (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_chipndalresracRomDesc[] = {
	{ "chip 'n dale - rescue rangers (usa) - castellano v0.9.nes",          262160, 0x7c6aed46, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chipndalresrac)
STD_ROM_FN(nes_chipndalresrac)

struct BurnDriver BurnDrvnes_chipndalresrac = {
	"nes_chipndalresrac", "nes_chipndalresra", NULL, NULL, "1989 ?",
	"Chip 'n Dale - Rescue Rangers (USA) - Castellano v0.9\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chipndalresracRomInfo, nes_chipndalresracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Chip 'n Dale - Rescue Rangers 2 (USA) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_chipndalresra2cRomDesc[] = {
	{ "chip 'n dale - rescue rangers 2 (usa) - castellano v0.99.nes",          262160, 0xaa6e37e6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chipndalresra2c)
STD_ROM_FN(nes_chipndalresra2c)

struct BurnDriver BurnDrvnes_chipndalresra2c = {
	"nes_chipndalresra2c", "nes_chipndalresra2", NULL, NULL, "1989 ?",
	"Chip 'n Dale - Rescue Rangers 2 (USA) - Castellano v0.99\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chipndalresra2cRomInfo, nes_chipndalresra2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Contra (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_contracastRomDesc[] = {
	{ "contra (usa) - castellano v1.0.nes",          131088, 0x6a56fc08, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contracast)
STD_ROM_FN(nes_contracast)

struct BurnDriver BurnDrvnes_contracast = {
	"nes_contracast", "nes_contra", NULL, NULL, "1989 ?",
	"Contra (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contracastRomInfo, nes_contracastRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Crystalis (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_crystaliscRomDesc[] = {
	{ "crystalis (usa) - castellano v1.0.nes",          393232, 0xb23c737c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crystalisc)
STD_ROM_FN(nes_crystalisc)

struct BurnDriver BurnDrvnes_crystalisc = {
	"nes_crystalisc", "nes_crystalis", NULL, NULL, "1989 ?",
	"Crystalis (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crystaliscRomInfo, nes_crystaliscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Darkman (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_darkmancRomDesc[] = {
	{ "darkman (usa) - castellano v1.0.nes",          262160, 0xe2cb1c36, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_darkmanc)
STD_ROM_FN(nes_darkmanc)

struct BurnDriver BurnDrvnes_darkmanc = {
	"nes_darkmanc", "nes_darkman", NULL, NULL, "1989 ?",
	"Darkman (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_darkmancRomInfo, nes_darkmancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Die Hard (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_diehardcRomDesc[] = {
	{ "die hard (usa) - castellano v1.1.nes",          262160, 0x602b9a29, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_diehardc)
STD_ROM_FN(nes_diehardc)

struct BurnDriver BurnDrvnes_diehardc = {
	"nes_diehardc", "nes_diehard", NULL, NULL, "1989 ?",
	"Die Hard (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_diehardcRomInfo, nes_diehardcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Disney's Aladdin (Europe) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_aladdincRomDesc[] = {
	{ "disney's aladdin (europe) - castellano v1.0.nes",          262160, 0xb857fcf2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_aladdinc)
STD_ROM_FN(nes_aladdinc)

struct BurnDriver BurnDrvnes_aladdinc = {
	"nes_aladdinc", "nes_aladdin", NULL, NULL, "1989 ?",
	"Disney's Aladdin (Europe) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_aladdincRomInfo, nes_aladdincRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Double Dragon (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_doubledragoncRomDesc[] = {
	{ "double dragon (usa) - castellano v1.1.nes",          262160, 0xf5e19c32, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubledragonc)
STD_ROM_FN(nes_doubledragonc)

struct BurnDriver BurnDrvnes_doubledragonc = {
	"nes_doubledragonc", "nes_doubledragon", NULL, NULL, "1989 ?",
	"Double Dragon (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doubledragoncRomInfo, nes_doubledragoncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Double Dragon II - The Revenge (USA) (Rev A) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_doubldraiicRomDesc[] = {
	{ "double dragon ii - the revenge (usa) (rev a) - castellano v1.0.nes",          262160, 0x67a0d781, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubldraiic)
STD_ROM_FN(nes_doubldraiic)

struct BurnDriver BurnDrvnes_doubldraiic = {
	"nes_doubldraiic", "nes_doubldraii", NULL, NULL, "1989 ?",
	"Double Dragon II - The Revenge (USA) (Rev A) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doubldraiicRomInfo, nes_doubldraiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Double Dragon III - The Sacred Stones (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_doubldraiiicRomDesc[] = {
	{ "double dragon iii - the sacred stones (usa) - castellano v1.0.nes",          262160, 0xaa40b251, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubldraiiic)
STD_ROM_FN(nes_doubldraiiic)

struct BurnDriver BurnDrvnes_doubldraiiic = {
	"nes_doubldraiiic", "nes_doubldraiii", NULL, NULL, "1989 ?",
	"Double Dragon III - The Sacred Stones (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doubldraiiicRomInfo, nes_doubldraiiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Ball Z - Super Butouden 2 (Unl) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragobalzsupbu2cRomDesc[] = {
	{ "dragon ball z - super butouden 2 (unl) - castellano v1.1.nes",          393232, 0x8e8ebd23, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalzsupbu2c)
STD_ROM_FN(nes_dragobalzsupbu2c)

struct BurnDriver BurnDrvnes_dragobalzsupbu2c = {
	"nes_dragobalzsupbu2c", "nes_dragobalzsupbu2", NULL, NULL, "1989 ?",
	"Dragon Ball Z - Super Butouden 2 (Unl) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalzsupbu2cRomInfo, nes_dragobalzsupbu2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Ball Z III - Ressen Jinzou Ningen (Japan) - Castellano v1.0a
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragobalziiirejinicRomDesc[] = {
	{ "dragon ball z iii - ressen jinzou ningen (japan) - castellano v1.0a.nes",          524304, 0xff1d3cf3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalziiirejinic)
STD_ROM_FN(nes_dragobalziiirejinic)

struct BurnDriver BurnDrvnes_dragobalziiirejinic = {
	"nes_dragobalziiirejinic", "nes_dragobalziiirejini", NULL, NULL, "1989 ?",
	"Dragon Ball Z III - Ressen Jinzou Ningen (Japan) - Castellano v1.0a\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalziiirejinicRomInfo, nes_dragobalziiirejinicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Buster (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragonbustercRomDesc[] = {
	{ "dragon buster (japan) - castellano v1.0.nes",          163856, 0xe718a85d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonbusterc)
STD_ROM_FN(nes_dragonbusterc)

struct BurnDriver BurnDrvnes_dragonbusterc = {
	"nes_dragonbusterc", "nes_dragonbuster", NULL, NULL, "1989 ?",
	"Dragon Buster (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonbustercRomInfo, nes_dragonbustercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Warrior (USA) (Rev A) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragonwarriorcRomDesc[] = {
	{ "dragon warrior (usa) (rev a) - castellano v1.0.nes",          81936, 0x7442bd9f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonwarriorc)
STD_ROM_FN(nes_dragonwarriorc)

struct BurnDriver BurnDrvnes_dragonwarriorc = {
	"nes_dragonwarriorc", "nes_dragonwarrior", NULL, NULL, "1989 ?",
	"Dragon Warrior (USA) (Rev A) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonwarriorcRomInfo, nes_dragonwarriorcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Warrior III (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragowariiicRomDesc[] = {
	{ "dragon warrior iii (usa) - castellano v0.9.nes",          524304, 0xc3d2f589, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragowariiic)
STD_ROM_FN(nes_dragowariiic)

struct BurnDriver BurnDrvnes_dragowariiic = {
	"nes_dragowariiic", "nes_dragowariii", NULL, NULL, "1989 ?",
	"Dragon Warrior III (USA) - Castellano v0.9\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragowariiicRomInfo, nes_dragowariiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon Warrior IV (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragowarivcRomDesc[] = {
	{ "dragon warrior iv (usa) - castellano v1.0.nes",          524304, 0xf5577473, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragowarivc)
STD_ROM_FN(nes_dragowarivc)

struct BurnDriver BurnDrvnes_dragowarivc = {
	"nes_dragowarivc", "nes_dragowariv", NULL, NULL, "1989 ?",
	"Dragon Warrior IV (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragowarivcRomInfo, nes_dragowarivcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dragon's Lair (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dragonslaircRomDesc[] = {
	{ "dragon's lair (usa) - castellano v1.0.nes",          131088, 0xdb3d2678, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonslairc)
STD_ROM_FN(nes_dragonslairc)

struct BurnDriver BurnDrvnes_dragonslairc = {
	"nes_dragonslairc", "nes_dragonslair", NULL, NULL, "1989 ?",
	"Dragon's Lair (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonslaircRomInfo, nes_dragonslaircRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Duck Tales (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_ducktalescRomDesc[] = {
	{ "duck tales (usa) - castellano v1.1.nes",          131088, 0xfcb58532, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ducktalesc)
STD_ROM_FN(nes_ducktalesc)

struct BurnDriver BurnDrvnes_ducktalesc = {
	"nes_ducktalesc", "nes_ducktales", NULL, NULL, "1989 ?",
	"Duck Tales (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ducktalescRomInfo, nes_ducktalescRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Duck Tales 2 (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_ducktales2castRomDesc[] = {
	{ "duck tales 2 (usa) - castellano v1.1.nes",          131088, 0x908c28ff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ducktales2cast)
STD_ROM_FN(nes_ducktales2cast)

struct BurnDriver BurnDrvnes_ducktales2cast = {
	"nes_ducktales2cast", "nes_ducktales2", NULL, NULL, "1989 ?",
	"Duck Tales 2 (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ducktales2castRomInfo, nes_ducktales2castRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Earth Bound (USA) (Proto) - Castellano v2.2
// https://www.romhacking.net/
static struct BurnRomInfo nes_earthboundcRomDesc[] = {
	{ "earth bound (usa) (proto) - castellano v2.2.nes",          524304, 0xd80e3217, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_earthboundc)
STD_ROM_FN(nes_earthboundc)

struct BurnDriver BurnDrvnes_earthboundc = {
	"nes_earthboundc", "nes_earthbound", NULL, NULL, "1989 ?",
	"Earth Bound (USA) (Proto) - Castellano v2.2\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_earthboundcRomInfo, nes_earthboundcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Elevator Action (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_elevatoractioncRomDesc[] = {
	{ "elevator action (usa) - castellano v1.0.nes",          40976, 0xcb844314, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_elevatoractionc)
STD_ROM_FN(nes_elevatoractionc)

struct BurnDriver BurnDrvnes_elevatoractionc = {
	"nes_elevatoractionc", "nes_elevatoraction", NULL, NULL, "1989 ?",
	"Elevator Action (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_elevatoractioncRomInfo, nes_elevatoractioncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Faxanadu (USA) (Rev A) - Castellano v0.7
// https://www.romhacking.net/
static struct BurnRomInfo nes_faxanaducRomDesc[] = {
	{ "faxanadu (usa) (rev a) - castellano v0.7.nes",          262160, 0x181b0549, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_faxanaduc)
STD_ROM_FN(nes_faxanaduc)

struct BurnDriver BurnDrvnes_faxanaduc = {
	"nes_faxanaduc", "nes_faxanadu", NULL, NULL, "1989 ?",
	"Faxanadu (USA) (Rev A) - Castellano v0.7\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_faxanaducRomInfo, nes_faxanaducRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Final Fantasy (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_finalfantasycRomDesc[] = {
	{ "final fantasy (usa) - castellano v1.0.nes",          262160, 0x83a1074c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_finalfantasyc)
STD_ROM_FN(nes_finalfantasyc)

struct BurnDriver BurnDrvnes_finalfantasyc = {
	"nes_finalfantasyc", "nes_finalfantasy", NULL, NULL, "1989 ?",
	"Final Fantasy (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_finalfantasycRomInfo, nes_finalfantasycRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Final Fantasy II (Japan) (Headered) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_finalfantasyiicRomDesc[] = {
	{ "final fantasy ii (japan) (headered) - castellano v1.0.nes",          262160, 0x3fba77eb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_finalfantasyiic)
STD_ROM_FN(nes_finalfantasyiic)

struct BurnDriver BurnDrvnes_finalfantasyiic = {
	"nes_finalfantasyiic", "nes_finalfantasyii", NULL, NULL, "1989 ?",
	"Final Fantasy II (Japan) (Headered) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_finalfantasyiicRomInfo, nes_finalfantasyiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Final Fantasy III (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_finalfaniiicRomDesc[] = {
	{ "final fantasy iii (japan) - castellano v1.0.nes",          1048592, 0xb328a638, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_finalfaniiic)
STD_ROM_FN(nes_finalfaniiic)

struct BurnDriver BurnDrvnes_finalfaniiic = {
	"nes_finalfaniiic", "nes_finalfaniii", NULL, NULL, "1989 ?",
	"Final Fantasy III (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_finalfaniiicRomInfo, nes_finalfaniiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Friday the 13th (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_fridaythe13thcRomDesc[] = {
	{ "friday the 13th (usa) - castellano v1.1.nes",          65552, 0xabecfd68, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fridaythe13thc)
STD_ROM_FN(nes_fridaythe13thc)

struct BurnDriver BurnDrvnes_fridaythe13thc = {
	"nes_fridaythe13thc", "nes_fridaythe13th", NULL, NULL, "1989 ?",
	"Friday the 13th (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_fridaythe13thcRomInfo, nes_fridaythe13thcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ganbare Goemon 2 (Japan) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_ganbaregoemon2cRomDesc[] = {
	{ "ganbare goemon 2 (japan) - castellano v0.99.nes",          262160, 0xeca0d0e1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ganbaregoemon2c)
STD_ROM_FN(nes_ganbaregoemon2c)

struct BurnDriver BurnDrvnes_ganbaregoemon2c = {
	"nes_ganbaregoemon2c", "nes_ganbaregoemon2", NULL, NULL, "1989 ?",
	"Ganbare Goemon 2 (Japan) - Castellano v0.99\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ganbaregoemon2cRomInfo, nes_ganbaregoemon2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ganbare Goemon Gaiden 2 - Tenka no Zaihou (Japan) - Castellano v0.98
// https://www.romhacking.net/
static struct BurnRomInfo nes_ganbagoegai2cRomDesc[] = {
	{ "ganbare goemon gaiden 2 - tenka no zaihou (japan) - castellano v0.98.nes",          524304, 0xc2e96c08, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ganbagoegai2c)
STD_ROM_FN(nes_ganbagoegai2c)

struct BurnDriverD BurnDrvnes_ganbagoegai2c = {
	"nes_ganbagoegai2c", "nes_ganbagoegai2", NULL, NULL, "1989 ?",
	"Ganbare Goemon Gaiden 2 - Tenka no Zaihou (Japan) - Castellano v0.98\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ganbagoegai2cRomInfo, nes_ganbagoegai2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ganbare Goemon! - Karakuri Douchuu (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_ganbagoecRomDesc[] = {
	{ "ganbare goemon! - karakuri douchuu (japan) - castellano v1.0.nes",          262160, 0xbf99d7e4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ganbagoec)
STD_ROM_FN(nes_ganbagoec)

struct BurnDriver BurnDrvnes_ganbagoec = {
	"nes_ganbagoec", "nes_ganbagoe", NULL, NULL, "1989 ?",
	"Ganbare Goemon! - Karakuri Douchuu (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ganbagoecRomInfo, nes_ganbagoecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Golgo 13 - Top Secret Episode (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_golgo13cRomDesc[] = {
	{ "golgo 13 - top secret episode (usa) - castellano v1.0.nes",          262160, 0x47cd6b0f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_golgo13c)
STD_ROM_FN(nes_golgo13c)

struct BurnDriver BurnDrvnes_golgo13c = {
	"nes_golgo13c", "nes_golgo13", NULL, NULL, "1989 ?",
	"Golgo 13 - Top Secret Episode (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_golgo13cRomInfo, nes_golgo13cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Goonies (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_gooniescRomDesc[] = {
	{ "goonies (japan) - castellano v1.0.nes",          49168, 0xb74b6dc4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gooniesc)
STD_ROM_FN(nes_gooniesc)

struct BurnDriver BurnDrvnes_gooniesc = {
	"nes_gooniesc", "nes_goonies", NULL, NULL, "1989 ?",
	"Goonies (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_gooniescRomInfo, nes_gooniescRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Goonies II, The (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_gooniesiithecRomDesc[] = {
	{ "goonies ii, the (usa) - castellano v1.1.nes",          131088, 0x6bc02b52, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gooniesiithec)
STD_ROM_FN(nes_gooniesiithec)

struct BurnDriver BurnDrvnes_gooniesiithec = {
	"nes_gooniesiithec", "nes_gooniesiithe", NULL, NULL, "1989 ?",
	"Goonies II, The (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_gooniesiithecRomInfo, nes_gooniesiithecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hokuto no Ken 2 (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_hokutonoken2cRomDesc[] = {
	{ "hokuto no ken 2 (japan) - castellano v1.1.nes",          131088, 0x3a3cd6df, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hokutonoken2c)
STD_ROM_FN(nes_hokutonoken2c)

struct BurnDriver BurnDrvnes_hokutonoken2c = {
	"nes_hokutonoken2c", "nes_hokutonoken2", NULL, NULL, "1989 ?",
	"Hokuto no Ken 2 (Japan) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_hokutonoken2cRomInfo, nes_hokutonoken2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hydlide (USA)
static struct BurnRomInfo nes_hydlideRomDesc[] = {
	{ "hydlide (usa).nes",          40976, 0x8962aebf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hydlide)
STD_ROM_FN(nes_hydlide)

struct BurnDriver BurnDrvnes_hydlide = {
	"nes_hydlide", NULL, NULL, NULL, "1989 ?",
	"Hydlide (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_hydlideRomInfo, nes_hydlideRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hydlide (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_hydlidecRomDesc[] = {
	{ "hydlide (usa) - castellano v1.0.nes",          40976, 0x9b4f06c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hydlidec)
STD_ROM_FN(nes_hydlidec)

struct BurnDriver BurnDrvnes_hydlidec = {
	"nes_hydlidec", "nes_hydlide", NULL, NULL, "1989 ?",
	"Hydlide (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_hydlidecRomInfo, nes_hydlidecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hokuto no Ken 3 - Shin Seiki Souzou Seiken Restuden (Japan)
static struct BurnRomInfo nes_hokutonoken3RomDesc[] = {
	{ "hokuto no ken 3 - shin seiki souzou seiken restuden (japan).nes",          262160, 0x12b32580, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hokutonoken3)
STD_ROM_FN(nes_hokutonoken3)

struct BurnDriver BurnDrvnes_hokutonoken3 = {
	"nes_hokutonoken3", NULL, NULL, NULL, "1989 ?",
	"Hokuto no Ken 3 - Shin Seiki Souzou Seiken Restuden (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_hokutonoken3RomInfo, nes_hokutonoken3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Hokuto no Ken 3 - Shin Seiki Souzou Seiken Restuden (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_hokutonoken3cRomDesc[] = {
	{ "hokuto no ken 3 - shin seiki souzou seiken restuden (japan) - castellano v1.0.nes",          262160, 0x9e7cc5dc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_hokutonoken3c)
STD_ROM_FN(nes_hokutonoken3c)

struct BurnDriver BurnDrvnes_hokutonoken3c = {
	"nes_hokutonoken3c", "nes_hokutonoken3", NULL, NULL, "1989 ?",
	"Hokuto no Ken 3 - Shin Seiki Souzou Seiken Restuden (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_hokutonoken3cRomInfo, nes_hokutonoken3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Immortal, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_immortalthecRomDesc[] = {
	{ "immortal, the (usa) - castellano v1.0.nes",          393232, 0xbf87b488, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_immortalthec)
STD_ROM_FN(nes_immortalthec)

struct BurnDriver BurnDrvnes_immortalthec = {
	"nes_immortalthec", "nes_immortalthe", NULL, NULL, "1989 ?",
	"Immortal, The (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_immortalthecRomInfo, nes_immortalthecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Jungle Book, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_junglebookthecRomDesc[] = {
	{ "jungle book, the (usa) - castellano v1.0.nes",          262160, 0x996a8677, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_junglebookthec)
STD_ROM_FN(nes_junglebookthec)

struct BurnDriver BurnDrvnes_junglebookthec = {
	"nes_junglebookthec", "nes_junglebookthe", NULL, NULL, "1989 ?",
	"Jungle Book, The (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_junglebookthecRomInfo, nes_junglebookthecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Just Breed (Japan) - Castellano v2.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_justbreedcRomDesc[] = {
	{ "just breed (japan) - castellano v2.0.nes",          786448, 0xe415a13f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_justbreedc)
STD_ROM_FN(nes_justbreedc)

struct BurnDriver BurnDrvnes_justbreedc = {
	"nes_justbreedc", "nes_justbreed", NULL, NULL, "1989 ?",
	"Just Breed (Japan) - Castellano v2.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_justbreedcRomInfo, nes_justbreedcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Karate Kid, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_karatekidthecRomDesc[] = {
	{ "karate kid, the (usa) - castellano v1.0.nes",          65552, 0x99cc2044, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_karatekidthec)
STD_ROM_FN(nes_karatekidthec)

struct BurnDriver BurnDrvnes_karatekidthec = {
	"nes_karatekidthec", "nes_karatekidthe", NULL, NULL, "1989 ?",
	"Karate Kid, The (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_karatekidthecRomInfo, nes_karatekidthecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Karateka (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_karatekacRomDesc[] = {
	{ "karateka (japan) - castellano v1.0.nes",          24592, 0x3839edb9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_karatekac)
STD_ROM_FN(nes_karatekac)

struct BurnDriver BurnDrvnes_karatekac = {
	"nes_karatekac", "nes_karateka", NULL, NULL, "1989 ?",
	"Karateka (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_karatekacRomInfo, nes_karatekacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Krusty's Fun House (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_krustsfunhoucRomDesc[] = {
	{ "krusty's fun house (usa) - castellano v1.0.nes",          393232, 0xb09308d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_krustsfunhouc)
STD_ROM_FN(nes_krustsfunhouc)

struct BurnDriver BurnDrvnes_krustsfunhouc = {
	"nes_krustsfunhouc", "nes_krustsfunhou", NULL, NULL, "1989 ?",
	"Krusty's Fun House (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_krustsfunhoucRomInfo, nes_krustsfunhoucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Legacy of the Wizard (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_legacwizcRomDesc[] = {
	{ "legacy of the wizard (usa) - castellano v1.0.nes",          196624, 0xe87e3d09, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legacwizc)
STD_ROM_FN(nes_legacwizc)

struct BurnDriver BurnDrvnes_legacwizc = {
	"nes_legacwizc", "nes_legacwiz", NULL, NULL, "1989 ?",
	"Legacy of the Wizard (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_legacwizcRomInfo, nes_legacwizcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Legend of Zelda, The (Europe) - Castellano v1.2
// https://www.romhacking.net/
static struct BurnRomInfo nes_legenzelcRomDesc[] = {
	{ "legend of zelda, the (europe) - castellano v1.2.nes",          131088, 0x9818ccd9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legenzelc)
STD_ROM_FN(nes_legenzelc)

struct BurnDriver BurnDrvnes_legenzelc = {
	"nes_legenzelc", "nes_legenzel", NULL, NULL, "1989 ?",
	"Legend of Zelda, The (Europe) - Castellano v1.2\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_legenzelcRomInfo, nes_legenzelcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Lion King, The (Europe) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_lionkingthecRomDesc[] = {
	{ "lion king, the (europe) - castellano v1.0.nes",          262160, 0x562245ae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lionkingthec)
STD_ROM_FN(nes_lionkingthec)

struct BurnDriver BurnDrvnes_lionkingthec = {
	"nes_lionkingthec", "nes_lionkingthe", NULL, NULL, "1989 ?",
	"Lion King, The (Europe) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_lionkingthecRomInfo, nes_lionkingthecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Lupin Sansei - Pandora no Isan (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_lupinsancRomDesc[] = {
	{ "lupin sansei - pandora no isan (japan) - castellano v1.0.nes",          196624, 0x0bb24755, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lupinsanc)
STD_ROM_FN(nes_lupinsanc)

struct BurnDriver BurnDrvnes_lupinsanc = {
	"nes_lupinsanc", "nes_lupinsan", NULL, NULL, "1989 ?",
	"Lupin Sansei - Pandora no Isan (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_lupinsancRomInfo, nes_lupinsancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Mafat Conspiracy - Golgo 13 (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_mafatconcRomDesc[] = {
	{ "mafat conspiracy - golgo 13 (usa) - castellano v1.1.nes",          262160, 0xb527ad27, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mafatconc)
STD_ROM_FN(nes_mafatconc)

struct BurnDriver BurnDrvnes_mafatconc = {
	"nes_mafatconc", "nes_mafatcon", NULL, NULL, "1989 ?",
	"Mafat Conspiracy - Golgo 13 (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_mafatconcRomInfo, nes_mafatconcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Magic of Scheherazade, The (USA) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_magicschcRomDesc[] = {
	{ "magic of scheherazade, the (usa) - castellano v0.99.nes",          262160, 0x9d080895, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_magicschc)
STD_ROM_FN(nes_magicschc)

struct BurnDriver BurnDrvnes_magicschc = {
	"nes_magicschc", "nes_magicsch", NULL, NULL, "1989 ?",
	"Magic of Scheherazade, The (USA) - Castellano v0.99\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_magicschcRomInfo, nes_magicschcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megamancRomDesc[] = {
	{ "megaman (usa) - castellano v1.0.nes",          131088, 0x53db1d74, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megamanc)
STD_ROM_FN(nes_megamanc)

struct BurnDriver BurnDrvnes_megamanc = {
	"nes_megamanc", "nes_megaman", NULL, NULL, "1989 ?",
	"Megaman (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_megamancRomInfo, nes_megamancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman II (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman2cRomDesc[] = {
	{ "megaman ii (usa) - castellano v1.0.nes",          262160, 0x35babd12, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman2c)
STD_ROM_FN(nes_megaman2c)

struct BurnDriver BurnDrvnes_megaman2c = {
	"nes_megaman2c", "nes_megaman2", NULL, NULL, "1989 ?",
	"Megaman II (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_megaman2cRomInfo, nes_megaman2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman III (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman3cRomDesc[] = {
	{ "megaman iii (usa) - castellano v1.0.nes",          393232, 0x7e1471b2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman3c)
STD_ROM_FN(nes_megaman3c)

struct BurnDriver BurnDrvnes_megaman3c = {
	"nes_megaman3c", "nes_megaman3", NULL, NULL, "1989 ?",
	"Megaman III (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_megaman3cRomInfo, nes_megaman3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman IV (USA) (Rev A) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman4cRomDesc[] = {
	{ "megaman iv (usa) (rev a) - castellano v1.0.nes",          524304, 0xd4141bb2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman4c)
STD_ROM_FN(nes_megaman4c)

struct BurnDriver BurnDrvnes_megaman4c = {
	"nes_megaman4c", "nes_megaman4", NULL, NULL, "1989 ?",
	"Megaman IV (USA) (Rev A) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_megaman4cRomInfo, nes_megaman4cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman V (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman5cRomDesc[] = {
	{ "megaman v (usa) - castellano v1.0.nes",          524304, 0xc46d7cd6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman5c)
STD_ROM_FN(nes_megaman5c)

struct BurnDriver BurnDrvnes_megaman5c = {
	"nes_megaman5c", "nes_megaman5", NULL, NULL, "1989 ?",
	"Megaman V (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_megaman5cRomInfo, nes_megaman5cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Megaman VI (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_megaman6cRomDesc[] = {
	{ "megaman vi (usa) - castellano v1.1.nes",          524304, 0x1b902805, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_megaman6c)
STD_ROM_FN(nes_megaman6c)

struct BurnDriver BurnDrvnes_megaman6c = {
	"nes_megaman6c", "nes_megaman6", NULL, NULL, "1989 ?",
	"Megaman VI (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_megaman6cRomInfo, nes_megaman6cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Metal Gear (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_metalgearcRomDesc[] = {
	{ "metal gear (usa) - castellano v1.0.nes",          131088, 0xbabc19e2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_metalgearc)
STD_ROM_FN(nes_metalgearc)

struct BurnDriver BurnDrvnes_metalgearc = {
	"nes_metalgearc", "nes_metalgear", NULL, NULL, "1989 ?",
	"Metal Gear (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_metalgearcRomInfo, nes_metalgearcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Metroid (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_metroidcRomDesc[] = {
	{ "metroid (usa) - castellano v1.0.nes",          131088, 0x12767f2e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_metroidc)
STD_ROM_FN(nes_metroidc)

struct BurnDriver BurnDrvnes_metroidc = {
	"nes_metroidc", "nes_metroid", NULL, NULL, "1989 ?",
	"Metroid (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_metroidcRomInfo, nes_metroidcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Mighty Final Fight (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_mightfinfigcRomDesc[] = {
	{ "mighty final fight (usa) - castellano v1.0.nes",          262160, 0x79b72603, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mightfinfigc)
STD_ROM_FN(nes_mightfinfigc)

struct BurnDriver BurnDrvnes_mightfinfigc = {
	"nes_mightfinfigc", "nes_mightfinfig", NULL, NULL, "1989 ?",
	"Mighty Final Fight (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_mightfinfigcRomInfo, nes_mightfinfigcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Europe)
static struct BurnRomInfo nes_maniacmansioneRomDesc[] = {
	{ "maniac mansion (europe).nes",          262160, 0x90333aea, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansione)
STD_ROM_FN(nes_maniacmansione)

struct BurnDriver BurnDrvnes_maniacmansione = {
	"nes_maniacmansione", "nes_maniacmansion", NULL, NULL, "1988 ?",
	"Maniac Mansion (Europe)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmansioneRomInfo, nes_maniacmansioneRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (France)
static struct BurnRomInfo nes_maniacmansionfRomDesc[] = {
	{ "maniac mansion (france).nes",          262160, 0x9118cd29, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansionf)
STD_ROM_FN(nes_maniacmansionf)

struct BurnDriver BurnDrvnes_maniacmansionf = {
	"nes_maniacmansionf", "nes_maniacmansion", NULL, NULL, "1988",
	"Maniac Mansion (France)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmansionfRomInfo, nes_maniacmansionfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Germany)
static struct BurnRomInfo nes_maniacmansiongRomDesc[] = {
	{ "maniac mansion (germany).nes",          262160, 0x05455e77, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansiong)
STD_ROM_FN(nes_maniacmansiong)

struct BurnDriver BurnDrvnes_maniacmansiong = {
	"nes_maniacmansiong", "nes_maniacmansion", NULL, NULL, "1988",
	"Maniac Mansion (Germany)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmansiongRomInfo, nes_maniacmansiongRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Italy)
static struct BurnRomInfo nes_maniacmansioniRomDesc[] = {
	{ "maniac mansion (italy).nes",          262160, 0xb9fd5255, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansioni)
STD_ROM_FN(nes_maniacmansioni)

struct BurnDriver BurnDrvnes_maniacmansioni = {
	"nes_maniacmansioni", "nes_maniacmansion", NULL, NULL, "1988",
	"Maniac Mansion (Italy)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmansioniRomInfo, nes_maniacmansioniRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Japan)
static struct BurnRomInfo nes_maniacmansionjRomDesc[] = {
	{ "maniac mansion (japan).nes",          262160, 0x101f9604, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansionj)
STD_ROM_FN(nes_maniacmansionj)

struct BurnDriver BurnDrvnes_maniacmansionj = {
	"nes_maniacmansionj", "nes_maniacmansion", NULL, NULL, "1988",
	"Maniac Mansion (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmansionjRomInfo, nes_maniacmansionjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Spain)
static struct BurnRomInfo nes_maniacmansionsRomDesc[] = {
	{ "maniac mansion (spain).nes",          262160, 0x901d691d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansions)
STD_ROM_FN(nes_maniacmansions)

struct BurnDriver BurnDrvnes_maniacmansions = {
	"nes_maniacmansions", "nes_maniacmansion", NULL, NULL, "1988",
	"Maniac Mansion (Spain)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmansionsRomInfo, nes_maniacmansionsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

//Maniac Mansion (Sweden)
static struct BurnRomInfo nes_maniacmansionswRomDesc[] = {
	{ "maniac mansion (sweden).nes",          262160, 0x5a841cb2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansionsw)
STD_ROM_FN(nes_maniacmansionsw)

struct BurnDriver BurnDrvnes_maniacmansionsw = {
	"nes_maniacmansionsw", "nes_maniacmansion", NULL, NULL, "1988",
	"Maniac Mansion (Sweden)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmansionswRomInfo, nes_maniacmansionswRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Maniac Mansion (USA) (Prototype)
static struct BurnRomInfo nes_maniacmansionuprotoRomDesc[] = {
	{ "maniac mansion (usa) (prototype).nes",          262160, 0xbba3ef7e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_maniacmansionuproto)
STD_ROM_FN(nes_maniacmansionuproto)

struct BurnDriver BurnDrvnes_maniacmansionuproto = {
	"nes_maniacmansionuproto", "nes_maniacmansion", NULL, NULL, "1988 ?",
	"Maniac Mansion (USA) (Prototype)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_maniacmansionuprotoRomInfo, nes_maniacmansionuprotoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Maniac Mansion (USA) (Prototype) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_mancmansuprocRomDesc[] = {
	{ "maniac mansion (usa) (prototype) - castellano v1.0.nes",          262160, 0x3be70f25, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mancmansuproc)
STD_ROM_FN(nes_mancmansuproc)

struct BurnDriver BurnDrvnes_mancmansuproc = {
	"nes_mancmansuproc", "nes_maniacmansion", NULL, NULL, "1989 ?",
	"Maniac Mansion (USA) (Prototype) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_mancmansuprocRomInfo, nes_mancmansuprocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Nekketsu! Street Basket - Ganbare Dunk Heroes (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_nekkestrbascastcRomDesc[] = {
	{ "nekketsu! street basket - ganbare dunk heroes (japan) - castellano v1.0.nes",          262160, 0x3a2487c3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nekkestrbascastc)
STD_ROM_FN(nes_nekkestrbascastc)

struct BurnDriver BurnDrvnes_nekkestrbascastc = {
	"nes_nekkestrbascastc", "nes_nekkestrbas", NULL, NULL, "1989 ?",
	"Nekketsu! Street Basket - Ganbare Dunk Heroes (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_nekkestrbascastcRomInfo, nes_nekkestrbascastcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Gaiden (USA) - Castellano v1.01 + Addendum
// https://www.romhacking.net/
static struct BurnRomInfo nes_ninjagaidencRomDesc[] = {
	{ "ninja gaiden (usa) - castellano v1.01.nes",          262160, 0x7da08225, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjagaidenc)
STD_ROM_FN(nes_ninjagaidenc)

struct BurnDriver BurnDrvnes_ninjagaidenc = {
	"nes_ninjagaidenc", "nes_ninjagaiden", NULL, NULL, "1989 ?",
	"Ninja Gaiden (USA) - Castellano v1.01 + Addendum\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ninjagaidencRomInfo, nes_ninjagaidencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Gaiden II - The Dark Sword of Chaos (USA) - Castellano v1.0 + Addendum
// https://www.romhacking.net/
static struct BurnRomInfo nes_ninjagaiiicRomDesc[] = {
	{ "ninja gaiden ii - the dark sword of chaos (usa) - castellano v1.0.nes",          262160, 0x5742d6c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjagaiiic)
STD_ROM_FN(nes_ninjagaiiic)

struct BurnDriver BurnDrvnes_ninjagaiiic = {
	"nes_ninjagaiiic", "nes_ninjagaiii", NULL, NULL, "1989 ?",
	"Ninja Gaiden II - The Dark Sword of Chaos (USA) - Castellano v1.0 + Addendum\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ninjagaiiicRomInfo, nes_ninjagaiiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ninja Gaiden III - The Ancient Ship of Doom (USA) - Castellano v1.0 + Addendum
// https://www.romhacking.net/
static struct BurnRomInfo nes_ninjagaiiiicRomDesc[] = {
	{ "ninja gaiden iii - the ancient ship of doom (usa) - castellano v1.0.nes",          262160, 0xdfb2692b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninjagaiiiic)
STD_ROM_FN(nes_ninjagaiiiic)

struct BurnDriver BurnDrvnes_ninjagaiiiic = {
	"nes_ninjagaiiiic", "nes_ninjagaiiii", NULL, NULL, "1989 ?",
	"Ninja Gaiden III - The Ancient Ship of Doom (USA) - Castellano v1.0 + Addendum\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ninjagaiiiicRomInfo, nes_ninjagaiiiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// North & South (USA) - Castellano v2.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_northsouthcRomDesc[] = {
	{ "north & south (usa) - castellano v2.0.nes",          262160, 0x98907558, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_northsouthc)
STD_ROM_FN(nes_northsouthc)

struct BurnDriver BurnDrvnes_northsouthc = {
	"nes_northsouthc", "nes_northsouth", NULL, NULL, "1989 ?",
	"North & South (USA) - Castellano v2.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_northsouthcRomInfo, nes_northsouthcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Parodius Da! (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_parodiusdacRomDesc[] = {
	{ "parodius da! (japan) - castellano v1.1.nes",          393232, 0x746bb8a8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_parodiusdac)
STD_ROM_FN(nes_parodiusdac)

struct BurnDriver BurnDrvnes_parodiusdac = {
	"nes_parodiusdac", "nes_parodiusda", NULL, NULL, "1989 ?",
	"Parodius Da! (Japan) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_parodiusdacRomInfo, nes_parodiusdacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// P.O.W. - Prisoners of War (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_powcRomDesc[] = {
	{ "p.o.w. - prisoners of war (usa) - castellano v1.0.nes",          262160, 0xc24c8bcb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_powc)
STD_ROM_FN(nes_powc)

struct BurnDriver BurnDrvnes_powc = {
	"nes_powc", "nes_pow", NULL, NULL, "1989 ?",
	"P.O.W. - Prisoners of War (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_powcRomInfo, nes_powcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Probotector (Europe) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_probotectorcRomDesc[] = {
	{ "probotector (europe) - castellano v1.0.nes",          131088, 0x9e9a6477, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_probotectorc)
STD_ROM_FN(nes_probotectorc)

struct BurnDriver BurnDrvnes_probotectorc = {
	"nes_probotectorc", "nes_contra", NULL, NULL, "1989 ?",
	"Probotector (Europe) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_probotectorcRomInfo, nes_probotectorcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Probotector II - Return of the Evil Forces (Europe) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_probotectorretcRomDesc[] = {
	{ "probotector ii - return of the evil forces (europe) - castellano v1.0.nes",          262160, 0xf8d46d0b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_probotectorretc)
STD_ROM_FN(nes_probotectorretc)

struct BurnDriver BurnDrvnes_probotectorretc = {
	"nes_probotectorretc", "nes_superc", NULL, NULL, "1989 ?",
	"Probotector II - Return of the Evil Forces (Europe) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_probotectorretcRomInfo, nes_probotectorretcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Puyo Puyo (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_puyopuyocRomDesc[] = {
	{ "puyo puyo (japan) - castellano v1.1.nes",          131088, 0xf55b69bd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_puyopuyoc)
STD_ROM_FN(nes_puyopuyoc)

struct BurnDriver BurnDrvnes_puyopuyoc = {
	"nes_puyopuyoc", "nes_puyopuyo", NULL, NULL, "1989 ?",
	"Puyo Puyo (Japan) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_puyopuyocRomInfo, nes_puyopuyocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_princeofpersiacRomDesc[] = {
	{ "prince of persia (usa) - castellano v1.0.nes",          131088, 0x3f11d7a3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiac)
STD_ROM_FN(nes_princeofpersiac)

struct BurnDriver BurnDrvnes_princeofpersiac = {
	"nes_princeofpersiac", "nes_princeofpersia", NULL, NULL, "1989 ?",
	"Prince of Persia (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_princeofpersiacRomInfo, nes_princeofpersiacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (Europe)
static struct BurnRomInfo nes_princeofpersiaeRomDesc[] = {
	{ "prince of persia (europe).nes",          131088, 0x84fa4a07, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiae)
STD_ROM_FN(nes_princeofpersiae)

struct BurnDriver BurnDrvnes_princeofpersiae = {
	"nes_princeofpersiae", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (Europe)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_princeofpersiaeRomInfo, nes_princeofpersiaeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (France)
static struct BurnRomInfo nes_princeofpersiafRomDesc[] = {
	{ "prince of persia (france).nes",          131088, 0xb19a530e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiaf)
STD_ROM_FN(nes_princeofpersiaf)

struct BurnDriver BurnDrvnes_princeofpersiaf = {
	"nes_princeofpersiaf", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (France)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_princeofpersiafRomInfo, nes_princeofpersiafRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (Germany)
static struct BurnRomInfo nes_princeofpersiagRomDesc[] = {
	{ "Prince of persia (germany).nes",          131088, 0x53a349d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiag)
STD_ROM_FN(nes_princeofpersiag)

struct BurnDriver BurnDrvnes_princeofpersiag = {
	"nes_princeofpersiag", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (Germany)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_princeofpersiagRomInfo, nes_princeofpersiagRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (Russia)
static struct BurnRomInfo nes_princeofpersiarRomDesc[] = {
	{ "prince of persia (russia).nes",          131088, 0x094e6961, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersiar)
STD_ROM_FN(nes_princeofpersiar)

struct BurnDriver BurnDrvnes_princeofpersiar = {
	"nes_princeofpersiar", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (Russia)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_princeofpersiarRomInfo, nes_princeofpersiarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Prince of Persia (Spain)
static struct BurnRomInfo nes_princeofpersiasRomDesc[] = {
	{ "prince of persia (spain).nes",          131088, 0x7817051a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_princeofpersias)
STD_ROM_FN(nes_princeofpersias)

struct BurnDriver BurnDrvnes_princeofpersias = {
	"nes_princeofpersias", "nes_princeofpersia", NULL, NULL, "1992",
	"Prince of Persia (Spain)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_princeofpersiasRomInfo, nes_princeofpersiasRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rambo (USA) (Rev 1) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_rambocRomDesc[] = {
	{ "rambo (usa) (rev 1) - castellano v1.0.nes",          131088, 0xf28cc969, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ramboc)
STD_ROM_FN(nes_ramboc)

struct BurnDriver BurnDrvnes_ramboc = {
	"nes_ramboc", "nes_rambo", NULL, NULL, "1989 ?",
	"Rambo (USA) (Rev 1) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_rambocRomInfo, nes_rambocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Renegade (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_renegadecRomDesc[] = {
	{ "renegade (usa) - castellano v1.0.nes",          131088, 0xacac361a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_renegadec)
STD_ROM_FN(nes_renegadec)

struct BurnDriver BurnDrvnes_renegadec = {
	"nes_renegadec", "nes_renegade", NULL, NULL, "1989 ?",
	"Renegade (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_renegadecRomInfo, nes_renegadecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rescue - The Embassy Mission (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_rescuecRomDesc[] = {
	{ "rescue - the embassy mission (usa) - castellano v1.0.nes",          262160, 0xab441e08, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rescuec)
STD_ROM_FN(nes_rescuec)

struct BurnDriver BurnDrvnes_rescuec = {
	"nes_rescuec", "nes_rescue", NULL, NULL, "1989 ?",
	"Rescue - The Embassy Mission (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_rescuecRomInfo, nes_rescuecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// RoboCop (USA) - Castellano v0.91
// https://www.romhacking.net/
static struct BurnRomInfo nes_robocopcRomDesc[] = {
	{ "robocop (usa) - castellano v0.91.nes",          262160, 0x0f97e1e8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_robocopc)
STD_ROM_FN(nes_robocopc)

struct BurnDriver BurnDrvnes_robocopc = {
	"nes_robocopc", "nes_robocop", NULL, NULL, "1989 ?",
	"RoboCop (USA) - Castellano v0.91\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_robocopcRomInfo, nes_robocopcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// RoboCop 2 (USA) (Rev 1) - Castellano v0.93
// https://www.romhacking.net/
static struct BurnRomInfo nes_robocop2cRomDesc[] = {
	{ "robocop 2 (usa) (rev 1) - castellano v0.93.nes",          262160, 0x4b3e87d9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_robocop2c)
STD_ROM_FN(nes_robocop2c)

struct BurnDriver BurnDrvnes_robocop2c = {
	"nes_robocop2c", "nes_robocop2", NULL, NULL, "1989 ?",
	"RoboCop 2 (USA) (Rev 1) - Castellano v0.93\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_robocop2cRomInfo, nes_robocop2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Robocop 3 (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_robocop3cRomDesc[] = {
	{ "robocop 3 (usa) - castellano v0.9.nes",          262160, 0xeda2dcb2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_robocop3c)
STD_ROM_FN(nes_robocop3c)

struct BurnDriver BurnDrvnes_robocop3c = {
	"nes_robocop3c", "nes_robocop3", NULL, NULL, "1989 ?",
	"Robocop 3 (USA) - Castellano v0.9\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_robocop3cRomInfo, nes_robocop3cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Rush'n Attack (USA) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_rushnattackcRomDesc[] = {
	{ "rush'n attack (usa) - castellano v1.1.nes",          131088, 0xb6bdb5e5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rushnattackc)
STD_ROM_FN(nes_rushnattackc)

struct BurnDriver BurnDrvnes_rushnattackc = {
	"nes_rushnattackc", "nes_rushnattack", NULL, NULL, "1989 ?",
	"Rush'n Attack (USA) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_rushnattackcRomInfo, nes_rushnattackcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Smash T.V. (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_smashtvcRomDesc[] = {
	{ "smash t.v. (usa) - castellano v1.0.nes",          262160, 0x658cbecc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smashtvc)
STD_ROM_FN(nes_smashtvc)

struct BurnDriver BurnDrvnes_smashtvc = {
	"nes_smashtvc", "nes_smashtv", NULL, NULL, "1989 ?",
	"Smash T.V. (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_smashtvcRomInfo, nes_smashtvcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Snake's Revenge (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_snakesrevengecRomDesc[] = {
	{ "snake's revenge (usa) - castellano v1.0.nes",          262160, 0x3d4a2507, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snakesrevengec)
STD_ROM_FN(nes_snakesrevengec)

struct BurnDriver BurnDrvnes_snakesrevengec = {
	"nes_snakesrevengec", "nes_snakesrevenge", NULL, NULL, "1989 ?",
	"Snake's Revenge (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_snakesrevengecRomInfo, nes_snakesrevengecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Snow Brothers (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_snowbrotherscRomDesc[] = {
	{ "snow brothers (usa) - castellano v1.0.nes",          262160, 0x29c40f70, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_snowbrothersc)
STD_ROM_FN(nes_snowbrothersc)

struct BurnDriver BurnDrvnes_snowbrothersc = {
	"nes_snowbrothersc", "nes_snowbrothers", NULL, NULL, "1989 ?",
	"Snow Brothers (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_snowbrotherscRomInfo, nes_snowbrotherscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Spy Vs Spy (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_spyvsspycRomDesc[] = {
	{ "spy vs spy (usa) - castellano v1.0.nes",          40976, 0x0c4a3d91, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spyvsspyc)
STD_ROM_FN(nes_spyvsspyc)

struct BurnDriver BurnDrvnes_spyvsspyc = {
	"nes_spyvsspyc", "nes_spyvsspy", NULL, NULL, "1989 ?",
	"Spy Vs Spy (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_spyvsspycRomInfo, nes_spyvsspycRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Star Wars (Japan) (Namco) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_starwarsjcRomDesc[] = {
	{ "star wars (japan) (namco) - castellano v1.0.nes",          262160, 0xeef6974a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starwarsjc)
STD_ROM_FN(nes_starwarsjc)

struct BurnDriver BurnDrvnes_starwarsjc = {
	"nes_starwarsjc", "nes_starwars", NULL, NULL, "1989 ?",
	"Star Wars (Japan) (Namco) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_starwarsjcRomInfo, nes_starwarsjcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Star Wars (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_starwarscRomDesc[] = {
	{ "star wars (usa) - castellano v1.0.nes",          262160, 0x0e6e000c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starwarsc)
STD_ROM_FN(nes_starwarsc)

struct BurnDriver BurnDrvnes_starwarsc = {
	"nes_starwarsc", "nes_starwars", NULL, NULL, "1989 ?",
	"Star Wars (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_starwarscRomInfo, nes_starwarscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Street Fighter 2010 - The Final Fight (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_streefig201cRomDesc[] = {
	{ "street fighter 2010 - the final fight (usa) - castellano v1.0.nes",          262160, 0x0c963bf7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_streefig201c)
STD_ROM_FN(nes_streefig201c)

struct BurnDriver BurnDrvnes_streefig201c = {
	"nes_streefig201c", "nes_streefig201", NULL, NULL, "1989 ?",
	"Street Fighter 2010 - The Final Fight (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_streefig201cRomInfo, nes_streefig201cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Street Fighter II - The World Warrior (Unl) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_streefigiicRomDesc[] = {
	{ "street fighter ii - the world warrior (unl) - castellano v1.0.nes",          262160, 0x3861489, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_streefigiic)
STD_ROM_FN(nes_streefigiic)

struct BurnDriver BurnDrvnes_streefigiic = {
	"nes_streefigiic", "nes_streefigii", NULL, NULL, "1989 ?",
	"Street Fighter II - The World Warrior (Unl) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_streefigiicRomInfo, nes_streefigiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Bros. (World) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_smbcRomDesc[] = {
	{ "super mario bros. (world) - castellano v1.0.nes",          40976, 0x54de5344, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbc)
STD_ROM_FN(nes_smbc)

struct BurnDriver BurnDrvnes_smbc = {
	"nes_smbc", "nes_smb", NULL, NULL, "1989 ?",
	"Super Mario Bros. (World) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_smbcRomInfo, nes_smbcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Bros. 2 (USA) (Rev 1) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_smb2cRomDesc[] = {
	{ "super mario bros. 2 (usa) (rev 1) - castellano v1.0.nes",          262160, 0x36114415, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb2c)
STD_ROM_FN(nes_smb2c)

struct BurnDriver BurnDrvnes_smb2c = {
	"nes_smb2c", "nes_smb2", NULL, NULL, "1989 ?",
	"Super Mario Bros. 2 (USA) (Rev 1) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_smb2cRomInfo, nes_smb2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Bros. 3 (USA) (Rev 1) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_smb3castRomDesc[] = {
	{ "super mario bros. 3 (usa) (rev 1) - castellano v1.0.nes",          393232, 0x2e094f49, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smb3cast)
STD_ROM_FN(nes_smb3cast)

struct BurnDriver BurnDrvnes_smb3cast = {
	"nes_smb3cast", "nes_smb3", NULL, NULL, "1989 ?",
	"Super Mario Bros. 3 (USA) (Rev 1) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_smb3castRomInfo, nes_smb3castRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Sword Master (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_swordmastercRomDesc[] = {
	{ "sword master (usa) - castellano v1.0.nes",          262160, 0x0f7fcbaf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_swordmasterc)
STD_ROM_FN(nes_swordmasterc)

struct BurnDriver BurnDrvnes_swordmasterc = {
	"nes_swordmasterc", "nes_swordmaster", NULL, NULL, "1989 ?",
	"Sword Master (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_swordmastercRomInfo, nes_swordmastercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Target - Renegade (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_targetrenegadecRomDesc[] = {
	{ "target - renegade (usa) - castellano v1.0.nes",          262160, 0x6754acae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_targetrenegadec)
STD_ROM_FN(nes_targetrenegadec)

struct BurnDriver BurnDrvnes_targetrenegadec = {
	"nes_targetrenegadec", "nes_targetrenegade", NULL, NULL, "1989 ?",
	"Target - Renegade (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_targetrenegadecRomInfo, nes_targetrenegadecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles - Tournament Fighters (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_tmnttoufigcRomDesc[] = {
	{ "teenage mutant ninja turtles - tournament fighters (usa) - castellano v1.0.nes",          262160, 0x32eac8c5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmnttoufigc)
STD_ROM_FN(nes_tmnttoufigc)

struct BurnDriver BurnDrvnes_tmnttoufigc = {
	"nes_tmnttoufigc", "nes_tmnttoufig", NULL, NULL, "1989 ?",
	"Teenage Mutant Ninja Turtles - Tournament Fighters (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_tmnttoufigcRomInfo, nes_tmnttoufigcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_tmntcastRomDesc[] = {
	{ "teenage mutant ninja turtles (usa) - castellano v0.9.nes",          262160, 0x30cd0997, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntcast)
STD_ROM_FN(nes_tmntcast)

struct BurnDriver BurnDrvnes_tmntcast = {
	"nes_tmntcast", "nes_tmnt", NULL, NULL, "1989 ?",
	"Teenage Mutant Ninja Turtles (USA) - Castellano v0.9\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_tmntcastRomInfo, nes_tmntcastRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles II - The Arcade Game (USA) - Castellano v0.9
// https://www.romhacking.net/
static struct BurnRomInfo nes_tmntiiarcgamcRomDesc[] = {
	{ "teenage mutant ninja turtles ii - the arcade game (usa) - castellano v0.9.nes",          524304, 0xb83bcec8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntiiarcgamc)
STD_ROM_FN(nes_tmntiiarcgamc)

struct BurnDriver BurnDrvnes_tmntiiarcgamc = {
	"nes_tmntiiarcgamc", "nes_tmntiiarcgam", NULL, NULL, "1989 ?",
	"Teenage Mutant Ninja Turtles II - The Arcade Game (USA) - Castellano v0.9\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_tmntiiarcgamcRomInfo, nes_tmntiiarcgamcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Teenage Mutant Ninja Turtles III - The Manhattan Project (USA) - Castellano v0.91
// https://www.romhacking.net/
static struct BurnRomInfo nes_tmntiiicRomDesc[] = {
	{ "teenage mutant ninja turtles iii - the manhattan project (usa) - castellano v0.91.nes",          524304, 0x3e9ddb5c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tmntiiic)
STD_ROM_FN(nes_tmntiiic)

struct BurnDriver BurnDrvnes_tmntiiic = {
	"nes_tmntiiic", "nes_tmntiii", NULL, NULL, "1989 ?",
	"Teenage Mutant Ninja Turtles III - The Manhattan Project (USA) - Castellano v0.91\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_tmntiiicRomInfo, nes_tmntiiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Terminator 2 - Judgment Day (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_termi2cRomDesc[] = {
	{ "terminator 2 - judgment day (usa) - castellano v1.0.nes",          262160, 0x43d08e31, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_termi2c)
STD_ROM_FN(nes_termi2c)

struct BurnDriver BurnDrvnes_termi2c = {
	"nes_termi2c", "nes_termi2", NULL, NULL, "1989 ?",
	"Terminator 2 - Judgment Day (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_termi2cRomInfo, nes_termi2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Thexder (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_thexdercRomDesc[] = {
	{ "Thexder (japan) - castellano v1.0.nes",          40976, 0x52ac30c1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_thexderc)
STD_ROM_FN(nes_thexderc)

struct BurnDriver BurnDrvnes_thexderc = {
	"nes_thexderc", "nes_thexder", NULL, NULL, "1989 ?",
	"Thexder (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_thexdercRomInfo, nes_thexdercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Tiny Toon Adventures (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_tinytooadvcRomDesc[] = {
	{ "tiny toon adventures (usa) - castellano v1.0.nes",          262160, 0xbcd94fa2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tinytooadvc)
STD_ROM_FN(nes_tinytooadvc)

struct BurnDriver BurnDrvnes_tinytooadvc = {
	"nes_tinytooadvc", "nes_tinytooadv", NULL, NULL, "1989 ?",
	"Tiny Toon Adventures (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_tinytooadvcRomInfo, nes_tinytooadvcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Tiny Toon Adventures 6 (Unl) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_tinytooadv6cRomDesc[] = {
	{ "tiny toon adventures 6 (unl) - castellano v1.0.nes",          524304, 0x93db8d83, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tinytooadv6c)
STD_ROM_FN(nes_tinytooadv6c)

struct BurnDriver BurnDrvnes_tinytooadv6c = {
	"nes_tinytooadv6c", "nes_tinytooadv6", NULL, NULL, "1989 ?",
	"Tiny Toon Adventures 6 (Unl) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_tinytooadv6cRomInfo, nes_tinytooadv6cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Tom & Jerry - The Ultimate Game of Cat and Mouse! (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_tomjerultgamcatandmcRomDesc[] = {
	{ "tom & jerry - the ultimate game of cat and mouse! (usa) - castellano v1.0.nes",          262160, 0x9c6a9bcc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tomjerultgamcatandmc)
STD_ROM_FN(nes_tomjerultgamcatandmc)

struct BurnDriver BurnDrvnes_tomjerultgamcatandmc = {
	"nes_tomjerultgamcatandmc", "nes_tomjerultgamcatandm", NULL, NULL, "1989 ?",
	"Tom & Jerry - The Ultimate Game of Cat and Mouse! (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_tomjerultgamcatandmcRomInfo, nes_tomjerultgamcatandmcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Top Gun - The Second Mission (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_topgunsecmiscRomDesc[] = {
	{ "top gun - the second mission (usa) - castellano v1.0.nes",          262160, 0x350f534b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_topgunsecmisc)
STD_ROM_FN(nes_topgunsecmisc)

struct BurnDriver BurnDrvnes_topgunsecmisc = {
	"nes_topgunsecmisc", "nes_topgunsecmis", NULL, NULL, "1989 ?",
	"Top Gun - The Second Mission (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_topgunsecmiscRomInfo, nes_topgunsecmiscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Top Gun (USA) (Rev 1) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_topguncRomDesc[] = {
	{ "top gun (usa) (rev 1) - castellano v1.0.nes",          131088, 0x879481af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_topgunc)
STD_ROM_FN(nes_topgunc)

struct BurnDriver BurnDrvnes_topgunc = {
	"nes_topgunc", "nes_topgun", NULL, NULL, "1989 ?",
	"Top Gun (USA) (Rev 1) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_topguncRomInfo, nes_topguncRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};							

// Total Recall (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_totalrecallcRomDesc[] = {
	{ "Total Recall (USA) - Castellano v1.0.nes",          131088, 0x93a02a67, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_totalrecallc)
STD_ROM_FN(nes_totalrecallc)

struct BurnDriver BurnDrvnes_totalrecallc = {
	"nes_totalrecallc", "nes_totalrecall", NULL, NULL, "1989 ?",
	"Total Recall (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_totalrecallcRomInfo, nes_totalrecallcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Uncanny X-Men, The (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_uncanxmencRomDesc[] = {
	{ "uncanny x-men, the (usa) - castellano v1.0.nes",          131088, 0x21a9db24, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_uncanxmenc)
STD_ROM_FN(nes_uncanxmenc)

struct BurnDriver BurnDrvnes_uncanxmenc = {
	"nes_uncanxmenc", "nes_uncanxmen", NULL, NULL, "1989 ?",
	"Uncanny X-Men, The (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_uncanxmencRomInfo, nes_uncanxmencRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Valkyrie no Bouken - Toki no Kagi Densetsu (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_valkynoboucRomDesc[] = {
	{ "valkyrie no bouken - toki no kagi densetsu (japan) - castellano v1.0.nes",          65552, 0x9e222fab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_valkynobouc)
STD_ROM_FN(nes_valkynobouc)

struct BurnDriver BurnDrvnes_valkynobouc = {
	"nes_valkynobouc", "nes_valkynobou", NULL, NULL, "1989 ?",
	"Valkyrie no Bouken - Toki no Kagi Densetsu (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_valkynoboucRomInfo, nes_valkynoboucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Wizards & Warriors III - Kuros - Visions of Power (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_wizarwariiicRomDesc[] = {
	{ "wizards & warriors iii - kuros - visions of power (usa) - castellano v1.0.nes",          262160, 0x7135c442, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wizarwariiic)
STD_ROM_FN(nes_wizarwariiic)

struct BurnDriver BurnDrvnes_wizarwariiic = {
	"nes_wizarwariiic", "nes_wizarwariii", NULL, NULL, "1989 ?",
	"Wizards & Warriors III - Kuros - Visions of Power (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_wizarwariiicRomInfo, nes_wizarwariiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Wolverine (USA) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_wolverinecRomDesc[] = {
	{ "wolverine (usa) - castellano v1.0.nes",          262160, 0xdb5ee5cb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wolverinec)
STD_ROM_FN(nes_wolverinec)

struct BurnDriver BurnDrvnes_wolverinec = {
	"nes_wolverinec", "nes_wolverine", NULL, NULL, "1989 ?",
	"Wolverine (USA) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_wolverinecRomInfo, nes_wolverinecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Yie Ar Kung-Fu (Japan) (V1.4) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_yiearkungfucRomDesc[] = {
	{ "yie ar kung-fu (japan) (v1.4) - castellano v1.0.nes",          24592, 0x032019a7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_yiearkungfuc)
STD_ROM_FN(nes_yiearkungfuc)

struct BurnDriver BurnDrvnes_yiearkungfuc = {
	"nes_yiearkungfuc", "nes_yiearkungfu", NULL, NULL, "1989 ?",
	"Yie Ar Kung-Fu (Japan) (V1.4) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_yiearkungfucRomInfo, nes_yiearkungfucRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ys (Japan) - Castellano v1.1
// https://www.romhacking.net/
static struct BurnRomInfo nes_yscRomDesc[] = {
	{ "ys (japan) - castellano v1.1.nes",          262160, 0x024dd1cf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ysc)
STD_ROM_FN(nes_ysc)

struct BurnDriver BurnDrvnes_ysc = {
	"nes_ysc", "nes_ys", NULL, NULL, "1989 ?",
	"Ys (Japan) - Castellano v1.1\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_yscRomInfo, nes_yscRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ys II - Ancient Ys Vanished - The Final Chapter (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_ysiicRomDesc[] = {
	{ "ys ii - ancient ys vanished - the final chapter (japan) - castellano v1.0.nes",          393232, 0x20770bf0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ysiic)
STD_ROM_FN(nes_ysiic)

struct BurnDriver BurnDrvnes_ysiic = {
	"nes_ysiic", "nes_ysii", NULL, NULL, "1989 ?",
	"Ys II - Ancient Ys Vanished - The Final Chapter (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ysiicRomInfo, nes_ysiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};	

// Zelda II - The Adventure of Link (Europe) (Rev 2) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_zeldaiicRomDesc[] = {
	{ "zelda ii - the adventure of link (europe) (rev 2) - castellano v1.0.nes",          262160, 0xd31d09ae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zeldaiic)
STD_ROM_FN(nes_zeldaiic)

struct BurnDriver BurnDrvnes_zeldaiic = {
	"nes_zeldaiic", "nes_zeldaii", NULL, NULL, "1989 ?",
	"Zelda II - The Adventure of Link (Europe) (Rev 2) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_zeldaiicRomInfo, nes_zeldaiicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Adventure Island IV (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_advenislivcRomDesc[] = {
	{ "adventure island iv (japan) - castellano v1.0.nes",          393232, 0x9ba3eb86, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenislivc)
STD_ROM_FN(nes_advenislivc)

struct BurnDriver BurnDrvnes_advenislivc = {
	"nes_advenislivc", "nes_advenisliv", NULL, NULL, "1989 ?",
	"Adventure Island IV (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenislivcRomInfo, nes_advenislivcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Ganbare Goemon Gaiden - Kieta Ougon Kiseru (Japan) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_ganbagoegaicRomDesc[] = {
	{ "ganbare goemon gaiden - kieta ougon kiseru (japan) - castellano v0.99.nes",          524304, 0x90ac0bda, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ganbagoegaic)
STD_ROM_FN(nes_ganbagoegaic)

struct BurnDriver BurnDrvnes_ganbagoegaic = {
	"nes_ganbagoegaic", "nes_ganbagoegai", NULL, NULL, "1989 ?",
	"Ganbare Goemon Gaiden - Kieta Ougon Kiseru (Japan) - Castellano v0.99\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ganbagoegaicRomInfo, nes_ganbagoegaicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Battle Oozumou - Heisei Hero Basho (Japan)
static struct BurnRomInfo nes_sdbttloozumouRomDesc[] = {
	{ "sd battle oozumou - heisei hero basho (japan).nes",          262160, 0xcd1931de, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdbttloozumou)
STD_ROM_FN(nes_sdbttloozumou)

struct BurnDriver BurnDrvnes_sdbttloozumou = {
	"nes_sdbttloozumou", NULL, NULL, NULL, "1989 ?",
	"SD Battle Oozumou - Heisei Hero Basho (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdbttloozumouRomInfo, nes_sdbttloozumouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};								

// SD Gundam - Gachapon Senshi 2 - Capsule Senki (Japan)
static struct BurnRomInfo nes_sdgundgs2RomDesc[] = {
	{ "sd gundam - gachapon senshi 2 - capsule senki (japan).nes",          262160, 0x88641b52, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgundgs2)
STD_ROM_FN(nes_sdgundgs2)

struct BurnDriver BurnDrvnes_sdgundgs2 = {
	"nes_sdgundgs2", NULL, NULL, NULL, "1989 ?",
	"SD Gundam - Gachapon Senshi 2 - Capsule Senki (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgundgs2RomInfo, nes_sdgundgs2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam - Gachapon Senshi 2 - Capsule Senki (Japan) - Castellano v1.0
static struct BurnRomInfo nes_sdgundgs2cRomDesc[] = {
	{ "sd gundam - gachapon senshi 2 - capsule senki (japan) - castellano v1.0.nes",          262160, 0x86274ffd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgundgs2c)
STD_ROM_FN(nes_sdgundgs2c)

struct BurnDriver BurnDrvnes_sdgundgs2c = {
	"nes_sdgundgs2c", "nes_sdgundgs2", NULL, NULL, "1989 ?",
	"SD Gundam - Gachapon Senshi 2 - Capsule Senki (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgundgs2cRomInfo, nes_sdgundgs2cRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam - Gachapon Senshi 3 - Eiyuu Senki (Japan)
static struct BurnRomInfo nes_sdgundgs3RomDesc[] = {
	{ "sd gundam - gachapon senshi 3 - eiyuu senki (japan).nes",          262160, 0x3af5beff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgundgs3)
STD_ROM_FN(nes_sdgundgs3)

struct BurnDriver BurnDrvnes_sdgundgs3 = {
	"nes_sdgundgs3", NULL, NULL, NULL, "1989 ?",
	"SD Gundam - Gachapon Senshi 3 - Eiyuu Senki (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgundgs3RomInfo, nes_sdgundgs3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};						

// SD Gundam - Gachapon Senshi 4 - New Type Story (Japan)
static struct BurnRomInfo nes_sdgundgs4RomDesc[] = {
	{ "sd gundam - gachapon senshi 4 - new type story (japan).nes",          524304, 0x078b916e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgundgs4)
STD_ROM_FN(nes_sdgundgs4)

struct BurnDriver BurnDrvnes_sdgundgs4 = {
	"nes_sdgundgs4", NULL, NULL, NULL, "1989 ?",
	"SD Gundam - Gachapon Senshi 4 - New Type Story (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgundgs4RomInfo, nes_sdgundgs4RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam - Gachapon Senshi 5 - Battle of Universal Century (Japan)
static struct BurnRomInfo nes_sdgachapon5RomDesc[] = {
	{ "sd gundam - gachapon senshi 5 - battle of universal century (japan).nes",          262160, 0x971d0527, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgachapon5)
STD_ROM_FN(nes_sdgachapon5)

struct BurnDriver BurnDrvnes_sdgachapon5 = {
	"nes_sdgachapon5", NULL, NULL, NULL, "1989 ?",
	"SD Gundam - Gachapon Senshi 5 - Battle of Universal Century (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgachapon5RomInfo, nes_sdgachapon5RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam Gaiden - Knight Gundam Monogatari (Japan) (V1.0) )T-Eng)
static struct BurnRomInfo nes_sdgunkgmRomDesc[] = {
	{ "sd gundam gaiden - knight gundam monogatari (japan) (v1.0) (t-eng).nes",          393232, 0x2c91773e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgunkgm)
STD_ROM_FN(nes_sdgunkgm)

struct BurnDriver BurnDrvnes_sdgunkgm = {
	"nes_sdgunkgm", NULL, NULL, NULL, "1989 ?",
	"SD Gundam Gaiden - Knight Gundam Monogatari (Japan) (V1.0) (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgunkgmRomInfo, nes_sdgunkgmRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam Gaiden - Knight Gundam Monogatari (Japan) (V1.1)
static struct BurnRomInfo nes_sdgunkgmjRomDesc[] = {
	{ "sd gundam gaiden - knight gundam monogatari (japan) (v1.1).nes",          393232, 0xb6cafb31, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgunkgmj)
STD_ROM_FN(nes_sdgunkgmj)

struct BurnDriver BurnDrvnes_sdgunkgmj = {
	"nes_sdgunkgmj", "nes_sdgunkgm", NULL, NULL, "1989 ?",
	"SD Gundam Gaiden - Knight Gundam Monogatari (Japan) (V1.1)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgunkgmjRomInfo, nes_sdgunkgmjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam Gaiden - Knight Gundam Monogatari 2 - Hikari no Kishi (Japan)
static struct BurnRomInfo nes_sdgunkgm2RomDesc[] = {
	{ "sd gundam gaiden - knight gundam monogatari 2 - hikari no kishi (japan).nes",          524304, 0x0c1319dd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgunkgm2)
STD_ROM_FN(nes_sdgunkgm2)

struct BurnDriver BurnDrvnes_sdgunkgm2 = {
	"nes_sdgunkgm2", NULL, NULL, NULL, "1989 ?",
	"SD Gundam Gaiden - Knight Gundam Monogatari 2 - Hikari no Kishi (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgunkgm2RomInfo, nes_sdgunkgm2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam Gaiden - Knight Gundam Monogatari 3 - Densetsu no Kishi Dan (Japan)
static struct BurnRomInfo nes_sdgunkgm3RomDesc[] = {
	{ "sd gundam gaiden - knight gundam monogatari 3 - densetsu no kishi dan (japan).nes",          524304, 0xf02a0aab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdgunkgm3)
STD_ROM_FN(nes_sdgunkgm3)

struct BurnDriver BurnDrvnes_sdgunkgm3 = {
	"nes_sdgunkgm3", NULL, NULL, NULL, "1989 ?",
	"SD Gundam Gaiden - Knight Gundam Monogatari 3 - Densetsu no Kishi Dan (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdgunkgm3RomInfo, nes_sdgunkgm3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Hero Soukessen - Taose! Aku no Gundan (Japan)
static struct BurnRomInfo nes_sdhstangRomDesc[] = {
	{ "SD Hero Soukessen - Taose! Aku no Gundan (Japan).nes",          262160, 0xf17fefdd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdhstang)
STD_ROM_FN(nes_sdhstang)

struct BurnDriver BurnDrvnes_sdhstang = {
	"nes_sdhstang", NULL, NULL, NULL, "1989 ?",
	"SD Hero Soukessen - Taose! Aku no Gundan (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdhstangRomInfo, nes_sdhstangRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Hero Soukessen - Taose! Aku no Gundan (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_sdhstangcRomDesc[] = {
	{ "sd hero soukessen - taose! aku no gundan (japan) - castellano v1.0.nes",          262160, 0x7c0a15a4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdhstangc)
STD_ROM_FN(nes_sdhstangc)

struct BurnDriver BurnDrvnes_sdhstangc = {
	"nes_sdhstangc", "nes_sdhstang", NULL, NULL, "1989 ?",
	"SD Hero Soukessen - Taose! Aku no Gundan (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdhstangcRomInfo, nes_sdhstangcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Keiji - Blader (Japan) [T-Eng]
static struct BurnRomInfo nes_sdkbladeRomDesc[] = {
	{ "sd keiji - blader (japan) [t-eng].nes",          393232, 0x1fa8a57d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdkblade)
STD_ROM_FN(nes_sdkblade)

struct BurnDriverD BurnDrvnes_sdkblade = {
	"nes_sdkblade", NULL, NULL, NULL, "1989 ?",
	"SD Keiji - Blader (Japan) [T-Eng]\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdkbladeRomInfo, nes_sdkbladeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Keiji - Blader (Japan)
static struct BurnRomInfo nes_sdkbladejRomDesc[] = {
	{ "sd keiji - blader (japan).nes",          262160, 0x452c373c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sdkbladej)
STD_ROM_FN(nes_sdkbladej)

struct BurnDriverD BurnDrvnes_sdkbladej = {
	"nes_sdkbladej", "nes_sdkblade", NULL, NULL, "1989 ?",
	"SD Keiji - Blader (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_sdkbladejRomInfo, nes_sdkbladejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// SD Gundam World - Gachapon Senshi - Scramble Wars (Japan) [b] - Castellano v1.0
static struct BurnRomInfo fds_sdgunworgacsescwacRomDesc[] = {
	{ "sd gundam world - gachapon senshi - scramble wars (japan) [b] - castellano v1.0.fds",          131016, 0x53f73319, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_sdgunworgacsescwac, fds_sdgunworgacsescwac, fds_fdsbios)
STD_ROM_FN(fds_sdgunworgacsescwac)

struct BurnDriver BurnDrvfds_sdgunworgacsescwac = {
	"fds_sdgunworgacsescwac", "fds_sdgunworgacsescwa", "fds_fdsbios", NULL, "1989?",
	"SD Gundam World - Gachapon Senshi - Scramble Wars (Japan) [b] - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_sdgunworgacsescwacRomInfo, fds_sdgunworgacsescwacRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Super Mario Brothers (Japan) - Castellano v1.0
static struct BurnRomInfo fds_supermarbrocRomDesc[] = {
	{ "super mario brothers (japan) - castellano v1.0.fds",          65516, 0x42d4a7c5, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_supermarbroc, fds_supermarbroc, fds_fdsbios)
STD_ROM_FN(fds_supermarbroc)

struct BurnDriver BurnDrvfds_supermarbroc = {
	"fds_supermarbroc", "fds_supermarbro", "fds_fdsbios", NULL, "1989?",
	"Super Mario Brothers (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_supermarbrocRomInfo, fds_supermarbrocRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Link no Bouken - The Legend of Zelda 2 (Japan) (Rev 1) - Castellano v1.0
static struct BurnRomInfo fds_linknoboucRomDesc[] = {
	{ "link no bouken - the legend of zelda 2 (japan) (rev 1) - castellano v1.0.fds",          131000, 0x60a26198, BRF_ESS | BRF_PRG },
};

STDROMPICKEXT(fds_linknobouc, fds_linknobouc, fds_fdsbios)
STD_ROM_FN(fds_linknobouc)

struct BurnDriver BurnDrvfds_linknobouc = {
	"fds_linknobouc", "fds_linknobou", "fds_fdsbios", NULL, "1989?",
	"Link no Bouken - The Legend of Zelda 2 (Japan) (Rev 1) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_FDS, GBF_MISC, 0,
	NESGetZipName, fds_linknoboucRomInfo, fds_linknoboucRomName, NULL, NULL, NULL, NULL, NESFDSInputInfo, NESFDSDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Earth Bound Zero (USA)
// https://www.romhacking.net/
static struct BurnRomInfo nes_earthboundzeroRomDesc[] = {
	{ "earth bound zero (usa).nes",          524304, 0x6a5e39e2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_earthboundzero)
STD_ROM_FN(nes_earthboundzero)

struct BurnDriver BurnDrvnes_earthboundzero = {
	"nes_earthboundzero", "nes_earthbound", NULL, NULL, "1989 ?",
	"Earth Bound Zero (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_earthboundzeroRomInfo, nes_earthboundzeroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Earth Bound Zero (USA) - Castellano v0.99
// https://www.romhacking.net/
static struct BurnRomInfo nes_earthboundzerocRomDesc[] = {
	{ "earth bound zero (usa) - castellano v0.99.nes",          524304, 0x2e24c315, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_earthboundzeroc)
STD_ROM_FN(nes_earthboundzeroc)

struct BurnDriver BurnDrvnes_earthboundzeroc = {
	"nes_earthboundzeroc", "nes_earthbound", NULL, NULL, "1989 ?",
	"Earth Bound Zero (USA) - Castellano v0.99\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_earthboundzerocRomInfo, nes_earthboundzerocRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Dai-2-ji Super Robot Taisen (Japan) - Castellano v1.0
// https://www.romhacking.net/
static struct BurnRomInfo nes_dai2jisuprotajcRomDesc[] = {
	{ "Dai-2-ji Super Robot Taisen (Japan) - Castellano v1.0.nes",          524304, 0x5b7ab480, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dai2jisuprotajc)
STD_ROM_FN(nes_dai2jisuprotajc)

struct BurnDriver BurnDrvnes_dai2jisuprotajc = {
	"nes_dai2jisuprotajc", "nes_dai2jisuprota", NULL, NULL, "1989 ?",
	"Dai-2-ji Super Robot Taisen (Japan) - Castellano v1.0\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dai2jisuprotajcRomInfo, nes_dai2jisuprotajcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Grand Master (T-Chi, v1.1)
// Translation by Han Hua Ni Mei Team - THIRTEEN, 姜维第二
static struct BurnRomInfo nes_grandmastercRomDesc[] = {
	{ "Grand Master (T-Chi, v1.1).nes",          1048592, 0x505d11c0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_grandmasterc)
STD_ROM_FN(nes_grandmasterc)

struct BurnDriver BurnDrvnes_grandmasterc = {
	"nes_grandmasterc", "nes_grandmaster", NULL, NULL, "2021-01-01",
	"Grand Master (T-Chi, v1.1)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_grandmastercRomInfo, nes_grandmastercRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Nekketsu! Street Basket - Ganbare Dunk Heroes (T-Chi, v2)
// Translation by Han Hua Ni Mei Team - 惊风, 空气, 火光 & goner 
static struct BurnRomInfo nes_nekkestrbascRomDesc[] = {
	{ "nekketsu! street basket - ganbare dunk heroes (T-Chi, v2).nes",          524304, 0x8f359b4f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nekkestrbasc)
STD_ROM_FN(nes_nekkestrbasc)

struct BurnDriver BurnDrvnes_nekkestrbasc = {
	"nes_nekkestrbasc", "nes_nekkestrbas", NULL, NULL, "2020-10-05",
	"Nekketsu! Street Basket - Ganbare Dunk Heroes (T-Chi, v2)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_nekkestrbascRomInfo, nes_nekkestrbascRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Grand Master (Japan) (Castellano)
// https://www.romhacking.net/
static struct BurnRomInfo nes_grandmasterjcRomDesc[] = {
	{ "grand master (japan) - castellano.nes",          393232, 0x3ad512ce, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_grandmasterjc)
STD_ROM_FN(nes_grandmasterjc)

struct BurnDriver BurnDrvnes_grandmasterjc = {
	"nes_grandmasterjc", "nes_grandmaster", NULL, NULL, "1989 ?",
	"Grand Master (Japan) - Castellano\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_grandmasterjcRomInfo, nes_grandmasterjcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// A Ressha de Ikou (T-Eng)
// https://www.romhacking.net/
static struct BurnRomInfo nes_atrainRomDesc[] = {
{ "a ressha de ikou (t-eng).nes",          147472, 0xfd7bc532, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_atrain)
STD_ROM_FN(nes_atrain)

struct BurnDriver BurnDrvnes_atrain = {
"nes_atrain", NULL, NULL, NULL, "1989 ?",
"A Ressha de Ikou (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
NULL, NULL, NULL, NULL,
BDF_GAME_WORKING | BDF_HACK, 2, HARDWARE_NES, GBF_MISC, 0,
NESGetZipName, nes_atrainRomInfo, nes_atrainRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// A Ressha de Ikou (Japan)
static struct BurnRomInfo nes_atrainjRomDesc[] = {
	{ "a ressha de ikou (japan).nes",          147472, 0x58fe5467, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_atrainj)
STD_ROM_FN(nes_atrainj)

	struct BurnDriver BurnDrvnes_atrainj = {
	"nes_atrainj", "nes_atrain", NULL, NULL, "1991",
	"A Ressha de Ikou (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_atrainjRomInfo, nes_atrainjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// END of "Non Homebrew (hand-added!)"

// Homebrew (hand-added)

static struct BurnRomInfo nes_luckypenguinRomDesc[] = {
	{ "Lucky Penguin (2019)(Mega Cat Studios).nes",          65552, 0x2a4c310b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_luckypenguin)
STD_ROM_FN(nes_luckypenguin)

struct BurnDriver BurnDrvnes_luckypenguin = {
	"nes_luckypenguin", NULL, NULL, NULL, "2019",
	"Lucky Penguin (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_luckypenguinRomInfo, nes_luckypenguinRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_batlizardRomDesc[] = {
	{ "Bat Lizard Bonanza (2020)(John Vanderhoef).nes",          524304, 0x4418f0f3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batlizard)
STD_ROM_FN(nes_batlizard)

struct BurnDriver BurnDrvnes_batlizard = {
	"nes_batlizard", NULL, NULL, NULL, "2020",
	"Bat Lizard Bonanza (HB)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_batlizardRomInfo, nes_batlizardRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_eyraRomDesc[] = {
	{ "Eyra - The Crow Maiden (2020)(2nd Dimension).nes",          40976, 0xe03ea9f9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_eyra)
STD_ROM_FN(nes_eyra)

struct BurnDriver BurnDrvnes_eyra = {
	"nes_eyra", NULL, NULL, NULL, "2020",
	"Eyra - The Crow Maiden (HB)\0", NULL, "2nd Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_eyraRomInfo, nes_eyraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ghostbrmRomDesc[] = {
	{ "Ghostbusters Remastered v1.1 (2019)(Nesrocks).nes",          98320, 0x29c2e0d0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ghostbrm)
STD_ROM_FN(nes_ghostbrm)

struct BurnDriver BurnDrvnes_ghostbrm = {
	"nes_ghostbrm", NULL, NULL, NULL, "2019",
	"Ghostbusters Remastered (HB, v1.1)\0", NULL, "NesRocks", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_HACK, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_ghostbrmRomInfo, nes_ghostbrmRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_littlelancRomDesc[] = {
	{ "Little Lancelot (2019)(Piko Int).nes",          262160, 0x65e1bc64, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_littlelanc)
STD_ROM_FN(nes_littlelanc)

struct BurnDriver BurnDrvnes_littlelanc = {
	"nes_littlelanc", NULL, NULL, NULL, "2019",
	"Little Lancelot (HB)\0", "Developed by Ocean", "Piko Interactive", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_littlelancRomInfo, nes_littlelancRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pyramidsraRomDesc[] = {
	{ "Pyramids of Ra (2019)(Piko Int.).nes",          393232, 0x0bcee664, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pyramidsra)
STD_ROM_FN(nes_pyramidsra)

struct BurnDriver BurnDrvnes_pyramidsra = {
	"nes_pyramidsra", NULL, NULL, NULL, "2019",
	"Pyramids of Ra (HB)\0", NULL, "Piko Interactive", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_pyramidsraRomInfo, nes_pyramidsraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_starversusRomDesc[] = {
	{ "Star Versus (2015)(Dustmop).nes",          196624, 0x33e83fa5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starversus)
STD_ROM_FN(nes_starversus)

struct BurnDriver BurnDrvnes_starversus = {
	"nes_starversus", NULL, NULL, NULL, "2015",
	"Star Versus (HB)\0", NULL, "Dustmop", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_starversusRomInfo, nes_starversusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitpicsRomDesc[] = {
	{ "8bit Illustrations (2015)(RIKI).nes",          524304, 0x19ca739b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitpics)
STD_ROM_FN(nes_8bitpics)

struct BurnDriver BurnDrvnes_8bitpics = {
	"nes_8bitpics", NULL, NULL, NULL, "2015",
	"8-bit Illustrations (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8bitpicsRomInfo, nes_8bitpicsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitmusicRomDesc[] = {
	{ "8bit Music Power (2017)(RIKI).nes",          786448, 0x351426ab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitmusic)
STD_ROM_FN(nes_8bitmusic)

struct BurnDriver BurnDrvnes_8bitmusic = {
	"nes_8bitmusic", NULL, NULL, NULL, "2017",
	"8-bit Music Power (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8bitmusicRomInfo, nes_8bitmusicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitmusiceRomDesc[] = {
	{ "8bit Music Power ENCORE (2017)(RIKI).nes",          524304, 0x8b381de2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitmusice)
STD_ROM_FN(nes_8bitmusice)

struct BurnDriver BurnDrvnes_8bitmusice = {
	"nes_8bitmusice", "nes_8bitmusic", NULL, NULL, "2017",
	"8-bit Music Power ENCORE (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_CLONE, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8bitmusiceRomInfo, nes_8bitmusiceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chknightRomDesc[] = {
	{ "Chrono Knight (2020)(Artix Games).nes",          524304, 0x88a47918, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chknight)
STD_ROM_FN(nes_chknight)

struct BurnDriver BurnDrvnes_chknight = {
	"nes_chknight", NULL, NULL, NULL, "2020",
	"Chrono Knight (HB)\0", NULL, "Artix Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_chknightRomInfo, nes_chknightRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chumleeadvRomDesc[] = {
	{ "Chumlee Adventure (2020)(Khan Games).nes",          131088, 0x0f81f876, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chumleeadv)
STD_ROM_FN(nes_chumleeadv)

struct BurnDriver BurnDrvnes_chumleeadv = {
	"nes_chumleeadv", NULL, NULL, NULL, "2020",
	"Chumlee's Adventure - the quest for Pinky (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_chumleeadvRomInfo, nes_chumleeadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doodleworldRomDesc[] = {
	{ "Doodle World (2020)(Nate Peters).nes",          524304, 0x6c71a681, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doodleworld)
STD_ROM_FN(nes_doodleworld)

struct BurnDriver BurnDrvnes_doodleworld = {
	"nes_doodleworld", NULL, NULL, NULL, "2020",
	"Doodle World (HB)\0", NULL, "Nate Peters", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_doodleworldRomInfo, nes_doodleworldRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_frenzyRomDesc[] = {
	{ "Frenzy A7800 (2016)(Bob Decrescenzo).nes",          40976, 0x82b0f8f4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_frenzy)
STD_ROM_FN(nes_frenzy)

struct BurnDriver BurnDrvnes_frenzy = {
	"nes_frenzy", NULL, NULL, NULL, "2016",
	"Frenzy A7800 (HB)\0", "The Atari 7800 Frenzy on NES (conversion)", "Bob Decrescenzo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_frenzyRomInfo, nes_frenzyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_jimpowerRomDesc[] = {
	{ "Jim Power - The Lost Dimension (2020)(Piko).nes",          524304, 0x3f088114, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jimpower)
STD_ROM_FN(nes_jimpower)

struct BurnDriver BurnDrvnes_jimpower = {
	"nes_jimpower", NULL, NULL, NULL, "2020",
	"Jim Power - The Lost Dimension (HB)\0", NULL, "Piko Interactive", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_RUNGUN, 0,
	NESGetZipName, nes_jimpowerRomInfo, nes_jimpowerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_leglutlivRomDesc[] = {
	{ "Leggite Luta Livre (2021)(John Vanderhoef).nes",          524304, 0xbcbc972c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_leglutliv)
STD_ROM_FN(nes_leglutliv)

struct BurnDriver BurnDrvnes_leglutliv = {
	"nes_leglutliv", NULL, NULL, NULL, "2021",
	"Leggite Luta Livre (HB)\0", NULL, "John Vanderhoef", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_leglutlivRomInfo, nes_leglutlivRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ninja4sRomDesc[] = {
	{ "Ninja of the 4 Seasons (2021)(Kool Doob).nes",          524304, 0x3943cf5f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ninja4s)
STD_ROM_FN(nes_ninja4s)

struct BurnDriver BurnDrvnes_ninja4s = {
	"nes_ninja4s", NULL, NULL, NULL, "2021",
	"Ninja of the 4 Seasons (HB)\0", NULL, "Kool Doob", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ninja4sRomInfo, nes_ninja4sRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ploidRomDesc[] = {
	{ "Ploid (2020)(NAPE Games).nes",          524304, 0xc7567823, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ploid)
STD_ROM_FN(nes_ploid)

struct BurnDriver BurnDrvnes_ploid = {
	"nes_ploid", NULL, NULL, NULL, "2020",
	"Ploid (HB)\0", NULL, "NAPE Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_ploidRomInfo, nes_ploidRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_astronmRomDesc[] = {
	{ "Astro Ninja Man (2020)(RiKi).nes",          786448, 0xf709e1e0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_astronm)
STD_ROM_FN(nes_astronm)

struct BurnDriver BurnDrvnes_astronm = {
	"nes_astronm", NULL, NULL, NULL, "2020",
	"Astro Ninja Man (HB)\0", NULL, "RiKi", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_astronmRomInfo, nes_astronmRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_porunchanRomDesc[] = {
	{ "Porun-chan no onigiri daisuki (v1.1-2019)(mook-tv).nes",          262160, 0xe15026cd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_porunchan)
STD_ROM_FN(nes_porunchan)

struct BurnDriver BurnDrvnes_porunchan = {
	"nes_porunchan", NULL, NULL, NULL, "2019",
	"Porun-chan No Onigiri Daisuki (HB, v1.1)\0", NULL, "Mook-TV", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_porunchanRomInfo, nes_porunchanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_trophyRomDesc[] = {
	{ "Trophy (2020)(the 6502 collective).nes",          786448, 0x94c16bfc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_trophy)
STD_ROM_FN(nes_trophy)

struct BurnDriver BurnDrvnes_trophy = {
	"nes_trophy", NULL, NULL, NULL, "2020",
	"Trophy (HB)\0", NULL, "the 6502 collective", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_trophyRomInfo, nes_trophyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_arkaderushRomDesc[] = {
	{ "Arkade Rush (HB)(2021)(Nathan Tolbert).nes",          32784, 0xd40c212e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_arkaderush)
STD_ROM_FN(nes_arkaderush)

struct BurnDriver BurnDrvnes_arkaderush = {
	"nes_arkaderush", NULL, NULL, NULL, "2021",
	"Arkade Rush (HB)\0", NULL, "Nathan Tolbert", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_arkaderushRomInfo, nes_arkaderushRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bmetal2RomDesc[] = {
	{ "bare metal 2.0(2021)(scarduadev).nes",          49168, 0xe37e0c59, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bmetal2)
STD_ROM_FN(nes_bmetal2)

struct BurnDriver BurnDrvnes_bmetal2 = {
	"nes_bmetal2", NULL, NULL, NULL, "2021",
	"Bare Metal 2.0 (HB)\0", "NesDev Competition build", "ScarduaDev", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM | GBF_PUZZLE, 0,
	NESGetZipName, nes_bmetal2RomInfo, nes_bmetal2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blobquestRomDesc[] = {
	{ "Blob Quest (HB)(2021)(Tom Livak).nes",          32784, 0xe3ac2b6d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blobquest)
STD_ROM_FN(nes_blobquest)

struct BurnDriver BurnDrvnes_blobquest = {
	"nes_blobquest", NULL, NULL, NULL, "2021",
	"Blob Quest (HB)\0", NULL, "Tom Livak", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_blobquestRomInfo, nes_blobquestRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_spacegullsRomDesc[] = {
	{ "Spacegulls (HB)(2021)(Morphcat Games).nes",          65552, 0x539f87d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_spacegulls)
STD_ROM_FN(nes_spacegulls)

struct BurnDriver BurnDrvnes_spacegulls = {
	"nes_spacegulls", NULL, NULL, NULL, "2021",
	"Spacegulls (HB, v1.1)\0", NULL, "Morphcat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_spacegullsRomInfo, nes_spacegullsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_witchnwizRomDesc[] = {
	{ "Witch n' Wiz (HB)(2021)(Matt Hughson).nes",          65552, 0xbddd6cba, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_witchnwiz)
STD_ROM_FN(nes_witchnwiz)

struct BurnDriver BurnDrvnes_witchnwiz = {
	"nes_witchnwiz", NULL, NULL, NULL, "2021",
	"Witch n' Wiz (HB)\0", NULL, "Matt Hughson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE | GBF_PLATFORM, 0,
	NESGetZipName, nes_witchnwizRomInfo, nes_witchnwizRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wolfspiritRomDesc[] = {
	{ "Wolf Spirit (HB)(2021)(Valdir Salgueiro).nes",          65552, 0x74574809, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wolfspirit)
STD_ROM_FN(nes_wolfspirit)

struct BurnDriver BurnDrvnes_wolfspirit = {
	"nes_wolfspirit", NULL, NULL, NULL, "2021",
	"Wolf Spirit (HB)\0", "Demo version", "Valdir Salgueiro", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_wolfspiritRomInfo, nes_wolfspiritRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_tapewpuzdisRomDesc[] = {
	{ "Tapeworm Puzzle Disco (HB)(Demo 8).nes",          65552, 0x30e0689f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_tapewpuzdis)
STD_ROM_FN(nes_tapewpuzdis)

struct BurnDriver BurnDrvnes_tapewpuzdis = {
	"nes_tapewpuzdis", NULL, NULL, NULL, "2020",
	"Tapeworm Puzzle Disco (HB)(Demo 8)\0", NULL, "LowtekGames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_tapewpuzdisRomInfo, nes_tapewpuzdisRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_whatremainsRomDesc[] = {
	{ "What Remains (HB).nes",          524304, 0x2f2d03e5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_whatremains)
STD_ROM_FN(nes_whatremains)

struct BurnDriver BurnDrvnes_whatremains = {
	"nes_whatremains", NULL, NULL, NULL, "2019",
	"What Remains (HB)\0", NULL, "Iodine Dynamics", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ADV, 0,
	NESGetZipName, nes_whatremainsRomInfo, nes_whatremainsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_galfRomDesc[] = {
	{ "Galf (2018)(Limited Run Games).nes",          262160, 0x52340070, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_galf)
STD_ROM_FN(nes_galf)

struct BurnDriver BurnDrvnes_galf = {
	"nes_galf", NULL, NULL, NULL, "2018",
	"Galf (HB)\0", NULL, "Limited Run Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SPORTSMISC, 0,
	NESGetZipName, nes_galfRomInfo, nes_galfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sharksharkRomDesc[] = {
	{ "Shark! Shark! (HB).nes",          24592, 0x5f33e5b4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sharkshark)
STD_ROM_FN(nes_sharkshark)

struct BurnDriver BurnDrvnes_sharkshark = {
	"nes_sharkshark", NULL, NULL, NULL, "2014",
	"Shark! Shark! (HB)\0", "Original title developed for Intellivision in 1982", "Mattel Electronics", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_sharksharkRomInfo, nes_sharksharkRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_caveRomDesc[] = {
	{ "cave (2017)(mojon twins).nes",          49168, 0xb86ce86b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cave)
STD_ROM_FN(nes_cave)

struct BurnDriver BurnDrvnes_cave = {
	"nes_cave", NULL, NULL, NULL, "2017",
	"Cave (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_caveRomInfo, nes_caveRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nixrelicRomDesc[] = {
	{ "Nix Paradox Relic v1.9(2020)(Dustin Huddleston).nes",          524304, 0xc85e7bcd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nixrelic)
STD_ROM_FN(nes_nixrelic)

struct BurnDriver BurnDrvnes_nixrelic = {
	"nes_nixrelic", NULL, NULL, NULL, "2020",
	"Nix: the Paradox Relic (HB, v1.9)\0", NULL, "Dustin Huddleston", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN | GBF_PLATFORM, 0,
	NESGetZipName, nes_nixrelicRomInfo, nes_nixrelicRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_rollieRomDesc[] = {
	{ "Rollie (HB).nes",          40976, 0x8165c356, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_rollie)
STD_ROM_FN(nes_rollie)

struct BurnDriver BurnDrvnes_rollie = {
	"nes_rollie", NULL, NULL, NULL, "2019",
	"Rollie (HB)\0", NULL, "Optovania", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_rollieRomInfo, nes_rollieRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_kosatakRomDesc[] = {
	{ "Kosmity Atakujo (HB, v10.07.2020).nes",          131088, 0x0570865c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kosatak)
STD_ROM_FN(nes_kosatak)

struct BurnDriver BurnDrvnes_kosatak = {
	"nes_kosatak", NULL, NULL, NULL, "2020",
	"Kosmity Atakujo (HB, v10.07.2020)\0", NULL, "sdm2", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_kosatakRomInfo, nes_kosatakRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_jaysilentRomDesc[] = {
	{ "Jay and Silent Bob Mall Brawl.nes",          393232, 0xf167590d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jaysilent)
STD_ROM_FN(nes_jaysilent)

struct BurnDriver BurnDrvnes_jaysilent = {
	"nes_jaysilent", NULL, NULL, NULL, "2020",
	"Jay and Silent Bob Mall Brawl (HB)\0", NULL, "Interabang Entertainment", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_jaysilentRomInfo, nes_jaysilentRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_journeyalRomDesc[] = {
	{ "journey to the center of the alien (2019)(mojon twins).nes",          73744, 0xd4b8eaa5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_journeyal)
STD_ROM_FN(nes_journeyal)

struct BurnDriver BurnDrvnes_journeyal = {
	"nes_journeyal", NULL, NULL, NULL, "2019",
	"Journey to the Center of the Alien (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_journeyalRomInfo, nes_journeyalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blockageRomDesc[] = {
	{ "Blockage (HB, v0.3.2).nes",          40976, 0x1c5f7bcf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blockage)
STD_ROM_FN(nes_blockage)

struct BurnDriver BurnDrvnes_blockage = {
	"nes_blockage", NULL, NULL, NULL, "2019",
	"Blockage (HB, v0.3.2)\0", NULL, "Scott Lowe", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_blockageRomInfo, nes_blockageRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_frombelowRomDesc[] = {
	{ "From Below (HB, v1.0).nes",          40976, 0x8ce49adb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_frombelow)
STD_ROM_FN(nes_frombelow)

struct BurnDriver BurnDrvnes_frombelow = {
	"nes_frombelow", NULL, NULL, NULL, "2020",
	"From Below (HB, v1.0)\0", NULL, "Goose2k", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_frombelowRomInfo, nes_frombelowRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_legenzeldxRomDesc[] = {
	{ "legend of zelda dx (2014)(pacnsacdave).nes",          131088, 0x0698579d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legenzeldx)
STD_ROM_FN(nes_legenzeldx)

struct BurnDriver BurnDrvnes_legenzeldx = {
	"nes_legenzeldx", "nes_legenzel", NULL, NULL, "2014",
	"Legend of Zelda DX, The (HB)\0", NULL, "pacnsacdave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_CLONE | BDF_HACK, 1, HARDWARE_NES, GBF_RPG | GBF_ADV, 0,
	NESGetZipName, nes_legenzeldxRomInfo, nes_legenzeldxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_multidudeRomDesc[] = {
	{ "multidude (2014)(retrosouls).nes",          40976, 0x154af940, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_multidude)
STD_ROM_FN(nes_multidude)

struct BurnDriver BurnDrvnes_multidude = {
	"nes_multidude", NULL, NULL, NULL, "2014",
	"Multidude (HB)\0", NULL, "RetroSouls", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_multidudeRomInfo, nes_multidudeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nimnomRomDesc[] = {
	{ "nim and nom (2018)(metakrill) v1.2.nes",          40976, 0xe0096736, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nimnom)
STD_ROM_FN(nes_nimnom)

struct BurnDriver BurnDrvnes_nimnom = {
	"nes_nimnom", NULL, NULL, NULL, "2018",
	"Nim & Nom (HB, v1.2)\0", NULL, "Metakrill", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_nimnomRomInfo, nes_nimnomRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_smbdxRomDesc[] = {
	{ "super mario bros dx (2018)(flamephanter) v3.4.nes",          40978, 0x004768e4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_smbdx)
STD_ROM_FN(nes_smbdx)

struct BurnDriver BurnDrvnes_smbdx = {
	"nes_smbdx", "nes_smb", NULL, NULL, "2018",
	"Super Mario Bros DX (HB, v3.4)\0", NULL, "Flamephanter", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW | BDF_CLONE | BDF_HACK, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_smbdxRomInfo, nes_smbdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_yunr5RomDesc[] = {
	{ "yun r5 (2018)(mojon twins).nes",          65552, 0x04940713, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_yunr5)
STD_ROM_FN(nes_yunr5)

struct BurnDriver BurnDrvnes_yunr5 = {
	"nes_yunr5", NULL, NULL, NULL, "2018",
	"Yun R5 (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_yunr5RomInfo, nes_yunr5RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_demondistrictRomDesc[] = {
	{ "Demon District (HB).nes",          319504, 0x33e22dcb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_demondistrict)
STD_ROM_FN(nes_demondistrict)

struct BurnDriver BurnDrvnes_demondistrict = {
	"nes_demondistrict", NULL, NULL, NULL, "2019",
	"Demon District (HB)\0", NULL, "Klonoa", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_demondistrictRomInfo, nes_demondistrictRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_badappleRomDesc[] = {
	{ "bad_apple (2015 v2.5)(Little Limit).nes",          786448, 0xf96a5a8b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_badapple)
STD_ROM_FN(nes_badapple)

struct BurnDriver BurnDrvnes_badapple = {
	"nes_badapple", NULL, NULL, NULL, "2015",
	"Bad Apple (HB, Tech-Demo)\0", NULL, "Little Limit", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_badappleRomInfo, nes_badappleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cadaverionRomDesc[] = {
	{ "Cadaverion (2018)(Mojon Twins).nes",          40976, 0x6a45c96a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cadaverion)
STD_ROM_FN(nes_cadaverion)

struct BurnDriver BurnDrvnes_cadaverion = {
	"nes_cadaverion", NULL, NULL, NULL, "2018",
	"Cadaverion (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE | GBF_ADV, 0,
	NESGetZipName, nes_cadaverionRomInfo, nes_cadaverionRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chaseRomDesc[] = {
	{ "Chase (2012)(shiru).nes",          24592, 0xb887529d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chase)
STD_ROM_FN(nes_chase)

struct BurnDriver BurnDrvnes_chase = {
	"nes_chase", NULL, NULL, NULL, "2012",
	"Chase (HB)\0", NULL, "Shiru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_MAZE, 0,
	NESGetZipName, nes_chaseRomInfo, nes_chaseRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cherilpcRomDesc[] = {
	{ "Cheril Perils Classic (2018)(Mojon Twins).nes",          40976, 0x8132afd5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cherilpc)
STD_ROM_FN(nes_cherilpc)

struct BurnDriver BurnDrvnes_cherilpc = {
	"nes_cherilpc", NULL, NULL, NULL, "2018",
	"Cheril Perils Classic (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cherilpcRomInfo, nes_cherilpcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_driarRomDesc[] = {
	{ "Driar (2012)(Eriksson & Adolfsson).nes",          131088, 0x2c31ce23, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_driar)
STD_ROM_FN(nes_driar)

struct BurnDriver BurnDrvnes_driar = {
	"nes_driar", NULL, NULL, NULL, "2018",
	"Driar (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_driarRomInfo, nes_driarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dushlanRomDesc[] = {
	{ "Dushlan (2016)(Peter McQuillan).nes",          40976, 0x07614f71, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dushlan)
STD_ROM_FN(nes_dushlan)

struct BurnDriver BurnDrvnes_dushlan = {
	"nes_dushlan", NULL, NULL, NULL, "2016",
	"Dushlan (HB)\0", NULL, "Peter McQuillan", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_dushlanRomInfo, nes_dushlanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sirababolrRomDesc[] = {
	{ "Sir Ababol Remastered Edition (2016)(Mojon Twins).nes",          40976, 0x36d8ca2b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sirababolr)
STD_ROM_FN(nes_sirababolr)

struct BurnDriver BurnDrvnes_sirababolr = {
	"nes_sirababolr", "nes_sirababol", NULL, NULL, "2016",
	"Sir Ababol Remastered Edition (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_sirababolrRomInfo, nes_sirababolrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_underadvRomDesc[] = {
	{ "Underground Adventure (2019)(Dale Coop).nes",          524304, 0xeaf67029, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_underadv)
STD_ROM_FN(nes_underadv)

struct BurnDriver BurnDrvnes_underadv = {
	"nes_underadv", NULL, NULL, NULL, "2019",
	"Underground Adventure (HB)\0", NULL, "Dale Coop", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_underadvRomInfo, nes_underadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wxniaoRomDesc[] = {
	{ "Wo Xiang Niao Niao (2016)(Xin Xin).nes",          40976, 0x7063779c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wxniao)
STD_ROM_FN(nes_wxniao)

struct BurnDriver BurnDrvnes_wxniao = {
	"nes_wxniao", NULL, NULL, NULL, "2016",
	"Wo Xiang Niao Niao (HB)\0", NULL, "Xin Xin", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_wxniaoRomInfo, nes_wxniaoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_aspectstarnRomDesc[] = {
	{ "Aspect Star N (HB).nes",          40976, 0x857dbdca, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_aspectstarn)
STD_ROM_FN(nes_aspectstarn)

struct BurnDriver BurnDrvnes_aspectstarn = {
	"nes_aspectstarn", NULL, NULL, NULL, "2019",
	"Aspect Star N (HB)\0", NULL, "Nicole Express", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_aspectstarnRomInfo, nes_aspectstarnRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bloodfallRomDesc[] = {
	{ "Bloodfall (HB).nes",          40976, 0x3c5548ef, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bloodfall)
STD_ROM_FN(nes_bloodfall)

struct BurnDriver BurnDrvnes_bloodfall = {
	"nes_bloodfall", NULL, NULL, NULL, "2019",
	"Bloodfall (HB)\0", NULL, "Dr. Ludos", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_bloodfallRomInfo, nes_bloodfallRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_superhomwarRomDesc[] = {
	{ "Super Homebrew War (HB).nes",          65552, 0x16b8d50a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superhomwar)
STD_ROM_FN(nes_superhomwar)

struct BurnDriver BurnDrvnes_superhomwar = {
	"nes_superhomwar", NULL, NULL, NULL, "2019",
	"Super Homebrew War (HB)\0", NULL, "gauauu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_superhomwarRomInfo, nes_superhomwarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_veggieinvadersRomDesc[] = {
	{ "Veggie Invaders (HB).nes",          65552, 0x6276c0a0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_veggieinvaders)
STD_ROM_FN(nes_veggieinvaders)

struct BurnDriver BurnDrvnes_veggieinvaders = {
	"nes_veggieinvaders", NULL, NULL, NULL, "2019",
	"Veggie Invaders (HB)\0", NULL, "dustmop", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_veggieinvadersRomInfo, nes_veggieinvadersRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Project Blue ROM @ https://toggleswitch.itch.io/projectblue
static struct BurnRomInfo nes_projectblueRomDesc[] = {
	{ "Project Blue (HB).nes",          524304, 0x43b9e57e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_projectblue)
STD_ROM_FN(nes_projectblue)

struct BurnDriver BurnDrvnes_projectblue = {
	"nes_projectblue", NULL, NULL, NULL, "2019",
	"Project Blue (HB)\0", NULL, "toggle switch", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_projectblueRomInfo, nes_projectblueRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chemanRomDesc[] = {
	{ "Che-Man (HB).nes",          40976, 0x9677c8a1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cheman)
STD_ROM_FN(nes_cheman)

struct BurnDriver BurnDrvnes_cheman = {
	"nes_cheman", NULL, NULL, NULL, "2018",
	"Che-Man (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 1, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_chemanRomInfo, nes_chemanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_eeeaaaoooRomDesc[] = {
	{ "EEEAAAOOO (HB).nes",          589840, 0xea1cb7b6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_eeeaaaooo)
STD_ROM_FN(nes_eeeaaaooo)

struct BurnDriver BurnDrvnes_eeeaaaooo = {
	"nes_eeeaaaooo", NULL, NULL, NULL, "2020",
	"EEEAAAOOO (HB)\0", NULL, "N.K", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_eeeaaaoooRomInfo, nes_eeeaaaoooRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fallingtilesRomDesc[] = {
	{ "Falling Tiles (HB, rev.B).nes",          40976, 0x5b8fcd00, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fallingtiles)
STD_ROM_FN(nes_fallingtiles)

struct BurnDriver BurnDrvnes_fallingtiles = {
	"nes_fallingtiles", NULL, NULL, NULL, "2020",
	"Falling Tiles (HB, rev.B)\0", NULL, "Doug Fraker", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_fallingtilesRomInfo, nes_fallingtilesRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_machinecaveRomDesc[] = {
	{ "Machine Cave (HB, v1.9).nes",          40976, 0x2d904baf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_machinecave)
STD_ROM_FN(nes_machinecave)

struct BurnDriver BurnDrvnes_machinecave = {
	"nes_machinecave", NULL, NULL, NULL, "2018",
	"Machine Cave (HB, v1.9)\0", NULL, "Simo Sievanen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SHOOT | GBF_PUZZLE, 0,
	NESGetZipName, nes_machinecaveRomInfo, nes_machinecaveRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wolflingRomDesc[] = {
	{ "Wolfling (HB, v1.4).nes",          65552, 0xf22ae9b8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wolfling)
STD_ROM_FN(nes_wolfling)

struct BurnDriver BurnDrvnes_wolfling = {
	"nes_wolfling", NULL, NULL, NULL, "2018",
	"Wolfling (HB, v1.4)\0", NULL, "Lazy Cow", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_wolflingRomInfo, nes_wolflingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_scrambleRomDesc[] = {
	{ "Scramble (HB).nes",          131088, 0xca6130a6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_scramble)
STD_ROM_FN(nes_scramble)

struct BurnDriver BurnDrvnes_scramble = {
	"nes_scramble", NULL, NULL, NULL, "2017",
	"Scramble (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_scrambleRomInfo, nes_scrambleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ultimfrochaRomDesc[] = {
	{ "Ultimate Frogger Champion (HB).nes",          49168, 0xab862073, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ultimfrocha)
STD_ROM_FN(nes_ultimfrocha)

struct BurnDriver BurnDrvnes_ultimfrocha = {
	"nes_ultimfrocha", NULL, NULL, NULL, "2019",
	"Ultimate Frogger Champion (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_ultimfrochaRomInfo, nes_ultimfrochaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cowligam2ndadvRomDesc[] = {
	{ "Cowlitz Gamers 2nd Adventure (HB).nes",          131088, 0x0db2dd59, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cowligam2ndadv)
STD_ROM_FN(nes_cowligam2ndadv)

struct BurnDriver BurnDrvnes_cowligam2ndadv = {
	"nes_cowligam2ndadv", NULL, NULL, NULL, "2017",
	"Cowlitz Gamers 2nd Adventure (HB)\0", NULL, "Cowlitz", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cowligam2ndadvRomInfo, nes_cowligam2ndadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cowligamadvRomDesc[] = {
	{ "Cowlitz Gamers Adventure (HB).nes",          32784, 0x0ae2354b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cowligamadv)
STD_ROM_FN(nes_cowligamadv)

struct BurnDriver BurnDrvnes_cowligamadv = {
	"nes_cowligamadv", NULL, NULL, NULL, "2016",
	"Cowlitz Gamers Adventure (HB)\0", NULL, "Cowlitz", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_cowligamadvRomInfo, nes_cowligamadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_madwizRomDesc[] = {
	{ "The Mad Wizard - A Candelabra Chronicle (HB).nes",          131088, 0x7750dac6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_madwiz)
STD_ROM_FN(nes_madwiz)

struct BurnDriver BurnDrvnes_madwiz = {
	"nes_madwiz", NULL, NULL, NULL, "2014",
	"The Mad Wizard - A Candelabra Chronicle (HB)\0", NULL, "Robert L. Bryant", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_madwizRomInfo, nes_madwizRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_riseamoRomDesc[] = {
	{ "The Rise of Amondus (HB).nes",          131088, 0xb15ac0fe, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_riseamo)
STD_ROM_FN(nes_riseamo)

struct BurnDriver BurnDrvnes_riseamo = {
	"nes_riseamo", NULL, NULL, NULL, "2015",
	"The Rise of Amondus (HB)\0", NULL, "Robert L. Bryant", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_riseamoRomInfo, nes_riseamoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_towerofturmoilRomDesc[] = {
	{ "Tower of Turmoil (HB, v1.03).nes",          524304, 0x89cbd840, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_towerofturmoil)
STD_ROM_FN(nes_towerofturmoil)

struct BurnDriver BurnDrvnes_towerofturmoil = {
	"nes_towerofturmoil", NULL, NULL, NULL, "2019",
	"Tower of Turmoil (HB, v1.03)\0", NULL, "CutterCross", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_towerofturmoilRomInfo, nes_towerofturmoilRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_jamgRomDesc[] = {
	{ "JAMG - The Lost Coins (HB).nes",          262160, 0x7a8877aa, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jamg)
STD_ROM_FN(nes_jamg)

struct BurnDriver BurnDrvnes_jamg = {
	"nes_jamg", NULL, NULL, NULL, "2018",
	"JAMG - The Lost Coins (HB)\0", NULL, "OCDreproductions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_jamgRomInfo, nes_jamgRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_pogocatsRomDesc[] = {
	{ "Pogo Cats (HB).nes",          24592, 0x57fc5391, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_pogocats)
STD_ROM_FN(nes_pogocats)

struct BurnDriver BurnDrvnes_pogocats = {
	"nes_pogocats", NULL, NULL, NULL, "2012",
	"Pogo Cats (HB)\0", NULL, "YGGI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_pogocatsRomInfo, nes_pogocatsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_purplecapemanRomDesc[] = {
	{ "Purple Cape Man (HB).nes",          24592, 0xd8a7825b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_purplecapeman)
STD_ROM_FN(nes_purplecapeman)

struct BurnDriver BurnDrvnes_purplecapeman = {
	"nes_purplecapeman", NULL, NULL, NULL, "2015",
	"Purple Cape Man (HB)\0", NULL, "Vigilante Ninja", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_purplecapemanRomInfo, nes_purplecapemanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sgthelRomDesc[] = {
	{ "Sgt. Helmet - Training Day (HB).nes",          40976, 0x44c6ca4d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sgthel)
STD_ROM_FN(nes_sgthel)

struct BurnDriver BurnDrvnes_sgthel = {
	"nes_sgthel", NULL, NULL, NULL, "2013",
	"Sgt. Helmet - Training Day (HB)\0", NULL, "The Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_sgthelRomInfo, nes_sgthelRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blackboxchaRomDesc[] = {
	{ "Black Box Challenge (HB).nes",          524304, 0x6b30c83b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blackboxcha)
STD_ROM_FN(nes_blackboxcha)

struct BurnDriver BurnDrvnes_blackboxcha = {
	"nes_blackboxcha", NULL, NULL, NULL, "2017",
	"Black Box Challenge (HB)\0", NULL, "Sly Dog Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_blackboxchaRomInfo, nes_blackboxchaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_flappybirdRomDesc[] = {
	{ "Flappy Bird (HB).nes",          24592, 0x3653c07d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_flappybird)
STD_ROM_FN(nes_flappybird)

struct BurnDriver BurnDrvnes_flappybird = {
	"nes_flappybird", NULL, NULL, NULL, "2014",
	"Flappy Bird (HB)\0", NULL, "Nioreh", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_flappybirdRomInfo, nes_flappybirdRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// https://lowtekgames.itch.io/fleanes
static struct BurnRomInfo nes_fleaRomDesc[] = {
	{ "flea! (HB).nes",          524304, 0xd208c6da, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_flea)
STD_ROM_FN(nes_flea)

struct BurnDriver BurnDrvnes_flea = {
	"nes_flea", NULL, NULL, NULL, "2020",
	"FLEA! (HB)\0", NULL, "Lowtek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_fleaRomInfo, nes_fleaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fleademoRomDesc[] = {
	{ "flea! demo (HB).nes",          524304, 0x6d18f7c4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fleademo)
STD_ROM_FN(nes_fleademo)

struct BurnDriver BurnDrvnes_fleademo = {
	"nes_fleademo", "nes_flea", NULL, NULL, "2019",
	"FLEA! demo (HB)\0", NULL, "Lowtek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_fleademoRomInfo, nes_fleademoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_supertiltbroRomDesc[] = {
	{ "Super Tilt Bro (HB).nes",          524304, 0x58829008, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_supertiltbro)
STD_ROM_FN(nes_supertiltbro)

struct BurnDriver BurnDrvnes_supertiltbro = {
	"nes_supertiltbro", NULL, NULL, NULL, "1989?",
	"Super Tilt Bro (HB)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_supertiltbroRomInfo, nes_supertiltbroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cmc80sdemoRomDesc[] = {
	{ "CMC 80's Demo (HB).nes",          49168, 0x0135b0e8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cmc80sdemo)
STD_ROM_FN(nes_cmc80sdemo)

struct BurnDriver BurnDrvnes_cmc80sdemo = {
	"nes_cmc80sdemo", NULL, NULL, NULL, "2000",
	"CMC 80's Demo (HB)\0", NULL, "Chris Covell", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cmc80sdemoRomInfo, nes_cmc80sdemoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_etRomDesc[] = {
	{ "E.T. - The Extra Terrestrial (HB).nes",          131088, 0xad871918, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_et)
STD_ROM_FN(nes_et)

struct BurnDriver BurnDrvnes_et = {
	"nes_et", NULL, NULL, NULL, "2014",
	"E.T. - The Extra Terrestrial (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_etRomInfo, nes_etRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_larryaRomDesc[] = {
	{ "Larry (HB, alt).nes",          524304, 0x2c8d188e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_larrya)
STD_ROM_FN(nes_larrya)

struct BurnDriver BurnDrvnes_larrya = {
	"nes_larrya", "nes_larry", NULL, NULL, "2014",
	"Larry (HB, alt)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_larryaRomInfo, nes_larryaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_larryRomDesc[] = {
	{ "Larry (HB).nes",          524304, 0x96f31fb9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_larry)
STD_ROM_FN(nes_larry)

struct BurnDriver BurnDrvnes_larry = {
	"nes_larry", NULL, NULL, NULL, "2014",
	"Larry (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_larryRomInfo, nes_larryRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mrsplashRomDesc[] = {
	{ "Mr. Splash (HB).nes",          750052, 0x8b86978b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mrsplash)
STD_ROM_FN(nes_mrsplash)

struct BurnDriver BurnDrvnes_mrsplash = {
	"nes_mrsplash", NULL, NULL, NULL, "2008",
	"Mr. Splash (HB)\0", NULL, "Project F", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_mrsplashRomInfo, nes_mrsplashRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nescapeRomDesc[] = {
	{ "NEScape (HB).nes",          524304, 0x60b6ad99, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nescape)
STD_ROM_FN(nes_nescape)

struct BurnDriver BurnDrvnes_nescape = {
	"nes_nescape", NULL, NULL, NULL, "2019",
	"NEScape (HB)\0", NULL, "Khan Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_nescapeRomInfo, nes_nescapeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_steinsgateRomDesc[] = {
	{ "Steins Gate (HB).nes",          786448, 0x547c4f1d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_steinsgate)
STD_ROM_FN(nes_steinsgate)

struct BurnDriver BurnDrvnes_steinsgate = {
	"nes_steinsgate", NULL, NULL, NULL, "2018",
	"Steins Gate (HB)\0", NULL, "Chiyomaru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_steinsgateRomInfo, nes_steinsgateRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_boblRomDesc[] = {
	{ "bobl (HB).nes",          65552, 0x831956ea, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bobl)
STD_ROM_FN(nes_bobl)

struct BurnDriver BurnDrvnes_bobl = {
	"nes_bobl", NULL, NULL, NULL, "2020",
	"BoBL (HB, v1.1)\0", NULL, "Morphcat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_boblRomInfo, nes_boblRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_novasquRomDesc[] = {
	{ "Nova the Squirrel (HB).nes",          262160, 0x0b752bee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_novasqu)
STD_ROM_FN(nes_novasqu)

struct BurnDriver BurnDrvnes_novasqu = {
	"nes_novasqu", NULL, NULL, NULL, "2018",
	"Nova the Squirrel (HB)\0", NULL, "NovaSquirrel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_novasquRomInfo, nes_novasquRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nallelandRomDesc[] = {
	{ "Nalle Land (HB, v040).nes",          49168, 0xc1a447b0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nalleland)
STD_ROM_FN(nes_nalleland)

struct BurnDriver BurnDrvnes_nalleland = {
	"nes_nalleland", NULL, NULL, NULL, "2020",
	"Nalle Land (HB, v040)\0", NULL, "Benjamin Larsson", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_nallelandRomInfo, nes_nallelandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mysticoriginsRomDesc[] = {
	{ "Mystic Origins (HB).nes",          524304, 0x249c666e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mysticorigins)
STD_ROM_FN(nes_mysticorigins)

struct BurnDriver BurnDrvnes_mysticorigins = {
	"nes_mysticorigins", NULL, NULL, NULL, "2019",
	"Mystic Origins (HB)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_mysticoriginsRomInfo, nes_mysticoriginsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_amazorundiejRomDesc[] = {
	{ "Amazons Running Diet (Japan).nes",          524304, 0x1eced454, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_amazorundiej)
STD_ROM_FN(nes_amazorundiej)

struct BurnDriver BurnDrvnes_amazorundiej = {
	"nes_amazorundiej", "nes_amazorundie", NULL, NULL, "2017",
	"Amazons Running Diet (HB, Japan)\0", NULL, "Ancient", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_amazorundiejRomInfo, nes_amazorundiejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_amazorundieRomDesc[] = {
	{ "Amazons Running Diet (USA).nes",          524304, 0x4db6acdd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_amazorundie)
STD_ROM_FN(nes_amazorundie)

struct BurnDriver BurnDrvnes_amazorundie = {
	"nes_amazorundie", NULL, NULL, NULL, "2017",
	"Amazons Running Diet (HB, USA)\0", NULL, "Ancient", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_amazorundieRomInfo, nes_amazorundieRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_haradiuszeroRomDesc[] = {
	{ "Haradius Zero (HB).nes",          786448, 0xd2f19ba1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_haradiuszero)
STD_ROM_FN(nes_haradiuszero)

struct BurnDriver BurnDrvnes_haradiuszero = {
	"nes_haradiuszero", NULL, NULL, NULL, "2019",
	"Haradius Zero (HB)\0", NULL, "Impact Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_HORSHOOT, 0,
	NESGetZipName, nes_haradiuszeroRomInfo, nes_haradiuszeroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_meatingRomDesc[] = {
	{ "The Meating - Cafeparty Demo (HB).nes",          262160, 0xe704ef0a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_meating)
STD_ROM_FN(nes_meating)

struct BurnDriver BurnDrvnes_meating = {
	"nes_meating", NULL, NULL, NULL, "2019",
	"The Meating - Cafeparty Demo (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_meatingRomInfo, nes_meatingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wallRomDesc[] = {
	{ "wall (Demo).nes",          24592, 0x5da28b4f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wall)
STD_ROM_FN(nes_wall)

struct BurnDriver BurnDrvnes_wall = {
	"nes_wall", NULL, NULL, NULL, "1999",
	"Wall (HB, Demo)\0", NULL, "Chris Covell", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_wallRomInfo, nes_wallRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x200,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};


static struct BurnRomInfo nes_actio53vol1stbuRomDesc[] = {
	{ "Action 53 vol 1 Streemerz Bundle (HB).nes",          524304, 0x54fc4fae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_actio53vol1stbu)
STD_ROM_FN(nes_actio53vol1stbu)

struct BurnDriver BurnDrvnes_actio53vol1stbu = {
	"nes_actio53vol1stbu", NULL, NULL, NULL, "20XX?",
	"Action 53 vol 1 Streemerz Bundle (HB)\0", NULL, "Nintendo?", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_actio53vol1stbuRomInfo, nes_actio53vol1stbuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_actio53vol2doacRomDesc[] = {
	{ "Action 53 vol 2 Double Action (HB).nes",          524304, 0x4e9c9174, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_actio53vol2doac)
STD_ROM_FN(nes_actio53vol2doac)

struct BurnDriver BurnDrvnes_actio53vol2doac = {
	"nes_actio53vol2doac", NULL, NULL, NULL, "20XX?",
	"Action 53 vol 2 Double Action (HB)\0", NULL, "Nintendo?", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_actio53vol2doacRomInfo, nes_actio53vol2doacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_actio53vol3retwRomDesc[] = {
	{ "Action 53 vol 3 Revenge of the Twins (HB).nes",          1048592, 0x0dc9110b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_actio53vol3retw)
STD_ROM_FN(nes_actio53vol3retw)

struct BurnDriver BurnDrvnes_actio53vol3retw = {
	"nes_actio53vol3retw", NULL, NULL, NULL, "20XX?",
	"Action 53 vol 3 Revenge of the Twins (HB)\0", NULL, "Nintendo?", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_actio53vol3retwRomInfo, nes_actio53vol3retwRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_alteregoRomDesc[] = {
	{ "Alter Ego (HB).nes",          40976, 0x5f322f79, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alterego)
STD_ROM_FN(nes_alterego)

struct BurnDriver BurnDrvnes_alterego = {
	"nes_alterego", NULL, NULL, NULL, "2011",
	"Alter Ego (HB)\0", NULL, "Shiru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_alteregoRomInfo, nes_alteregoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_assimilateRomDesc[] = {
	{ "Assimilate (HB).nes",          262160, 0xa2d504a8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_assimilate)
STD_ROM_FN(nes_assimilate)

struct BurnDriver BurnDrvnes_assimilate = {
	"nes_assimilate", NULL, NULL, NULL, "2012",
	"Assimilate (HB)\0", NULL, "Nessylum", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_assimilateRomInfo, nes_assimilateRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bladebusterRomDesc[] = {
	{ "Blade Buster (HB).nes",          393232, 0x3aba6565, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bladebuster)
STD_ROM_FN(nes_bladebuster)

struct BurnDriver BurnDrvnes_bladebuster = {
	"nes_bladebuster", NULL, NULL, NULL, "2011",
	"Blade Buster (HB)\0", NULL, "HLC! Project (tm)", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_bladebusterRomInfo, nes_bladebusterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bombsweeperRomDesc[] = {
	{ "Bomb Sweeper (HB).nes",          24592, 0xe667858d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bombsweeper)
STD_ROM_FN(nes_bombsweeper)

struct BurnDriver BurnDrvnes_bombsweeper = {
	"nes_bombsweeper", NULL, NULL, NULL, "2002",
	"Bomb Sweeper (HB)\0", NULL, "SNOWBRO", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MAZE, 0,
	NESGetZipName, nes_bombsweeperRomInfo, nes_bombsweeperRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_espitundherpimRomDesc[] = {
	{ "Espitenen und herr pimponen (HB).nes",          65552, 0xea5fa8a8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_espitundherpim)
STD_ROM_FN(nes_espitundherpim)

struct BurnDriver BurnDrvnes_espitundherpim = {
	"nes_espitundherpim", NULL, NULL, NULL, "2019",
	"Espitenen und herr pimponen (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_espitundherpimRomInfo, nes_espitundherpimRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_gruni2RomDesc[] = {
	{ "Gruniozerca 2 - The Great Cavy Rescue! (HB).nes",          65552, 0x911a2917, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gruni2)
STD_ROM_FN(nes_gruni2)

struct BurnDriver BurnDrvnes_gruni2 = {
	"nes_gruni2", NULL, NULL, NULL, "2019",
	"Gruniozerca 2 - The Great Cavy Rescue! (HB)\0", NULL, "Lukasz Kur & M-Tee", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_gruni2RomInfo, nes_gruni2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_gruni3RomDesc[] = {
	{ "Gruniozerca 3 - The Great Cavy Clean-Up! (HB).nes",          65552, 0xa92f5930, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_gruni3)
STD_ROM_FN(nes_gruni3)

struct BurnDriver BurnDrvnes_gruni3 = {
	"nes_gruni3", NULL, NULL, NULL, "2019",
	"Gruniozerca 3 - The Great Cavy Clean-Up! (HB)\0", NULL, "Lukasz Kur & M-Tee", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_gruni3RomInfo, nes_gruni3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_lalathemagicalRomDesc[] = {
	{ "Lala the Magical (HB).nes",          81936, 0x496c2e59, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lalathemagical)
STD_ROM_FN(nes_lalathemagical)

struct BurnDriver BurnDrvnes_lalathemagical = {
	"nes_lalathemagical", NULL, NULL, NULL, "2016",
	"Lala the Magical (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_lalathemagicalRomInfo, nes_lalathemagicalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_lunarlimitRomDesc[] = {
	{ "Lunar Limit (HB).nes",          40976, 0xeb26681e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lunarlimit)
STD_ROM_FN(nes_lunarlimit)

struct BurnDriver BurnDrvnes_lunarlimit = {
	"nes_lunarlimit", NULL, NULL, NULL, "2017",
	"Lunar Limit (HB)\0", NULL, "pubby", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_lunarlimitRomInfo, nes_lunarlimitRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_miedowRomDesc[] = {
	{ "Miedow (HB).nes",          65552, 0x6389b721, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_miedow)
STD_ROM_FN(nes_miedow)

struct BurnDriver BurnDrvnes_miedow = {
	"nes_miedow", NULL, NULL, NULL, "2018",
	"Miedow (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_miedowRomInfo, nes_miedowRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mojontwibooRomDesc[] = {
	{ "Mojon Twins - Bootee (HB).nes",          40976, 0xc71f42bc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mojontwiboo)
STD_ROM_FN(nes_mojontwiboo)

struct BurnDriver BurnDrvnes_mojontwiboo = {
	"nes_mojontwiboo", NULL, NULL, NULL, "2018",
	"Bootee (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_mojontwibooRomInfo, nes_mojontwibooRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mojontwichegodRomDesc[] = {
	{ "Mojon Twins - Cheril The Goddess (HB).nes",          65552, 0xe7c81856, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mojontwichegod)
STD_ROM_FN(nes_mojontwichegod)

struct BurnDriver BurnDrvnes_mojontwichegod = {
	"nes_mojontwichegod", NULL, NULL, NULL, "2017",
	"Cheril The Goddess (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_mojontwichegodRomInfo, nes_mojontwichegodRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mojontwichewriRomDesc[] = {
	{ "Mojon Twins - Cheril The Writer (HB).nes",          131088, 0x2405a68f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mojontwichewri)
STD_ROM_FN(nes_mojontwichewri)

struct BurnDriver BurnDrvnes_mojontwichewri = {
	"nes_mojontwichewri", NULL, NULL, NULL, "2018",
	"Cheril The Writer (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_mojontwichewriRomInfo, nes_mojontwichewriRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_saturnsmashRomDesc[] = {
	{ "Saturn Smash (HB).nes",          65552, 0xabc36edb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_saturnsmash)
STD_ROM_FN(nes_saturnsmash)

struct BurnDriver BurnDrvnes_saturnsmash = {
	"nes_saturnsmash", NULL, NULL, NULL, "2019",
	"Saturn Smash (HB)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_saturnsmashRomInfo, nes_saturnsmashRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sinkingfeelingRomDesc[] = {
	{ "Sinking Feeling (HB).nes",          49168, 0xb2f79ee2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sinkingfeeling)
STD_ROM_FN(nes_sinkingfeeling)

struct BurnDriver BurnDrvnes_sinkingfeeling = {
	"nes_sinkingfeeling", NULL, NULL, NULL, "2017",
	"Sinking Feeling (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SHOOT, 0,
	NESGetZipName, nes_sinkingfeelingRomInfo, nes_sinkingfeelingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_sirababolRomDesc[] = {
	{ "Sir Ababol (HB).nes",          40976, 0x5e7e4c60, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_sirababol)
STD_ROM_FN(nes_sirababol)

struct BurnDriver BurnDrvnes_sirababol = {
	"nes_sirababol", NULL, NULL, NULL, "2013",
	"Sir Ababol (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_sirababolRomInfo, nes_sirababolRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_starevilRomDesc[] = {
	{ "Star Evil (HB).nes",          40976, 0xfbd4ce1c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_starevil)
STD_ROM_FN(nes_starevil)

struct BurnDriver BurnDrvnes_starevil = {
	"nes_starevil", NULL, NULL, NULL, "2018",
	"Star Evil (HB)\0", NULL, "PDRoms", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_starevilRomInfo, nes_starevilRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_superpit30tRomDesc[] = {
	{ "Super Pitfall 30th (HB).nes",          131088, 0xe55ba70d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superpit30t)
STD_ROM_FN(nes_superpit30t)

struct BurnDriver BurnDrvnes_superpit30t = {
	"nes_superpit30t", NULL, NULL, NULL, "2016",
	"Super Pitfall 30th Anniversary Ed.(HB)\0", NULL, "NESRocks", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_superpit30tRomInfo, nes_superpit30tRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_touhourououmuRomDesc[] = {
	{ "TouhouRououmu (HB).nes",          524304, 0x0f240b9c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_touhourououmu)
STD_ROM_FN(nes_touhourououmu)

struct BurnDriver BurnDrvnes_touhourououmu = {
	"nes_touhourououmu", NULL, NULL, NULL, "2019",
	"Touhou Rououmu - Perfect Cherry Blossom (HB, v0.71)\0", NULL, "takahirox", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_VERSHOOT, 0,
	NESGetZipName, nes_touhourououmuRomInfo, nes_touhourououmuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_zombiecalaveraRomDesc[] = {
	{ "Zombie Calavera (HB).nes",          40976, 0x23ec3047, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zombiecalavera)
STD_ROM_FN(nes_zombiecalavera)

struct BurnDriver BurnDrvnes_zombiecalavera = {
	"nes_zombiecalavera", NULL, NULL, NULL, "2016",
	"Zombie Calavera (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RUNGUN, 0,
	NESGetZipName, nes_zombiecalaveraRomInfo, nes_zombiecalaveraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_zoomisecRomDesc[] = {
	{ "Zooming Secretary (HB).nes",          40976, 0x947972d0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_zoomisec)
STD_ROM_FN(nes_zoomisec)

struct BurnDriver BurnDrvnes_zoomisec = {
	"nes_zoomisec", NULL, NULL, NULL, "2011",
	"Zooming Secretary (HB)\0", NULL, "Pinwizz & Shiru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_zoomisecRomInfo, nes_zoomisecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8bitrhylanRomDesc[] = {
	{ "8-Bit Rhythm Land (HB).nes",          786448, 0x48859a73, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8bitrhylan)
STD_ROM_FN(nes_8bitrhylan)

struct BurnDriver BurnDrvnes_8bitrhylan = {
	"nes_8bitrhylan", NULL, NULL, NULL, "2018",
	"8-Bit Rhythm Land (HB)\0", NULL, "Columbus Circle", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8bitrhylanRomInfo, nes_8bitrhylanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_alfonsarcadvRomDesc[] = {
	{ "Alfonzo's Arctic Adventure (HB).nes",          262160, 0xcd2ada76, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alfonsarcadv)
STD_ROM_FN(nes_alfonsarcadv)

struct BurnDriver BurnDrvnes_alfonsarcadv = {
	"nes_alfonsarcadv", NULL, NULL, NULL, "2018",
	"Alfonzo's Arctic Adventure (HB)\0", NULL, "Spoony Bard Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_alfonsarcadvRomInfo, nes_alfonsarcadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_almostheroRomDesc[] = {
	{ "Almost Hero (HB).nes",          262160, 0x2419f370, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_almosthero)
STD_ROM_FN(nes_almosthero)

struct BurnDriver BurnDrvnes_almosthero = {
	"nes_almosthero", NULL, NULL, NULL, "2016",
	"Almost Hero (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_almostheroRomInfo, nes_almostheroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bassedefadvRomDesc[] = {
	{ "Basse Def Adventures (HB).nes",          524304, 0xbb87065d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bassedefadv)
STD_ROM_FN(nes_bassedefadv)

struct BurnDriver BurnDrvnes_bassedefadv = {
	"nes_bassedefadv", NULL, NULL, NULL, "2019",
	"Basse Def Adventures (HB)\0", NULL, "Broke Studio", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_bassedefadvRomInfo, nes_bassedefadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_beerslingerRomDesc[] = {
	{ "BeerSlinger (HB).nes",          40976, 0xae0ebf22, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_beerslinger)
STD_ROM_FN(nes_beerslinger)

struct BurnDriver BurnDrvnes_beerslinger = {
	"nes_beerslinger", NULL, NULL, NULL, "2018",
	"BeerSlinger (HB)\0", NULL, "Second Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_beerslingerRomInfo, nes_beerslingerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blowemoutRomDesc[] = {
	{ "Blow'Em Out! (HB).nes",          40976, 0xdc1468fc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blowemout)
STD_ROM_FN(nes_blowemout)

struct BurnDriver BurnDrvnes_blowemout = {
	"nes_blowemout", NULL, NULL, NULL, "2018",
	"Blow'Em Out! (HB)\0", NULL, "Second Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_blowemoutRomInfo, nes_blowemoutRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_creepybrawlersRomDesc[] = {
	{ "Creepy Brawlers (HB).nes",          524304, 0xc8f05be4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_creepybrawlers)
STD_ROM_FN(nes_creepybrawlers)

struct BurnDriver BurnDrvnes_creepybrawlers = {
	"nes_creepybrawlers", NULL, NULL, NULL, "2017",
	"Creepy Brawlers (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_VSFIGHT, 0,
	NESGetZipName, nes_creepybrawlersRomInfo, nes_creepybrawlersRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cryptoRomDesc[] = {
	{ "Crypto (HB).nes",          49168, 0x45aaa381, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crypto)
STD_ROM_FN(nes_crypto)

struct BurnDriver BurnDrvnes_crypto = {
	"nes_crypto", NULL, NULL, NULL, "2016",
	"Crypto (HB)\0", NULL, "EEMU", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_cryptoRomInfo, nes_cryptoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_eskimbobdelediRomDesc[] = {
	{ "Eskimo Bob Deluxe Edition (HB).nes",          65552, 0xa3b7848f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_eskimbobdeledi)
STD_ROM_FN(nes_eskimbobdeledi)

struct BurnDriver BurnDrvnes_eskimbobdeledi = {
	"nes_eskimbobdeledi", NULL, NULL, NULL, "2019",
	"Eskimo Bob Deluxe Edition (HB)\0", NULL, "Spoony Bard Productions", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_eskimbobdelediRomInfo, nes_eskimbobdelediRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_expeditionRomDesc[] = {
	{ "Expedition (HB).nes",          393232, 0x1b83916d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_expedition)
STD_ROM_FN(nes_expedition)

struct BurnDriver BurnDrvnes_expedition = {
	"nes_expedition", NULL, NULL, NULL, "2017",
	"Expedition (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_expeditionRomInfo, nes_expeditionRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_familypicrossRomDesc[] = {
	{ "Family Picross! (HB).nes",          40976, 0xbd459402, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_familypicross)
STD_ROM_FN(nes_familypicross)

struct BurnDriver BurnDrvnes_familypicross = {
	"nes_familypicross", NULL, NULL, NULL, "2018",
	"Family Picross! (HB)\0", NULL, "FG Software", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_familypicrossRomInfo, nes_familypicrossRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_getemgaryRomDesc[] = {
	{ "Get'Em Gary (HB).nes",          40976, 0xb559ce3e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_getemgary)
STD_ROM_FN(nes_getemgary)

struct BurnDriver BurnDrvnes_getemgary = {
	"nes_getemgary", NULL, NULL, NULL, "2016",
	"Get'Em Gary (HB)\0", NULL, "Second Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_getemgaryRomInfo, nes_getemgaryRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_halloween85RomDesc[] = {
	{ "Halloween'85 (HB).nes",          524304, 0x4bcf7ae5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_halloween85)
STD_ROM_FN(nes_halloween85)

struct BurnDriver BurnDrvnes_halloween85 = {
	"nes_halloween85", NULL, NULL, NULL, "2015",
	"Haunted Halloween'85 (HB)\0", NULL, "Retrotainment Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_halloween85RomInfo, nes_halloween85RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_kirakirstanigacRomDesc[] = {
	{ "Kira Kira Star Night AC (HB).nes",          524304, 0x9e650f69, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kirakirstanigac)
STD_ROM_FN(nes_kirakirstanigac)

struct BurnDriver BurnDrvnes_kirakirstanigac = {
	"nes_kirakirstanigac", NULL, NULL, NULL, "2017",
	"Kira Kira Star Night AC (HB)\0", NULL, "RIKI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_kirakirstanigacRomInfo, nes_kirakirstanigacRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_kirakirstanigdxRomDesc[] = {
	{ "Kira Kira Star Night DX (HB).nes",          524304, 0x66a30110, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kirakirstanigdx)
STD_ROM_FN(nes_kirakirstanigdx)

struct BurnDriver BurnDrvnes_kirakirstanigdx = {
	"nes_kirakirstanigdx", NULL, NULL, NULL, "2016",
	"Kira Kira Star Night DX (HB)\0", NULL, "RIKI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_kirakirstanigdxRomInfo, nes_kirakirstanigdxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_kirakirstaniggoRomDesc[] = {
	{ "Kira Kira Star Night GOLD (HB).nes",          524304, 0xd2526d60, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_kirakirstaniggo)
STD_ROM_FN(nes_kirakirstaniggo)

struct BurnDriver BurnDrvnes_kirakirstaniggo = {
	"nes_kirakirstaniggo", NULL, NULL, NULL, "2016",
	"Kira Kira Star Night GOLD (HB)\0", NULL, "RIKI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_kirakirstaniggoRomInfo, nes_kirakirstaniggoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_littlemedusaRomDesc[] = {
	{ "Little Medusa (HB).nes",          524304, 0x126b5927, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_littlemedusa)
STD_ROM_FN(nes_littlemedusa)

struct BurnDriver BurnDrvnes_littlemedusa = {
	"nes_littlemedusa", NULL, NULL, NULL, "2018",
	"Little Medusa (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_littlemedusaRomInfo, nes_littlemedusaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_lizardRomDesc[] = {
	{ "Lizard (HB).nes",          524304, 0x27a5348f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lizard)
STD_ROM_FN(nes_lizard)

struct BurnDriver BurnDrvnes_lizard = {
	"nes_lizard", NULL, NULL, NULL, "2018",
	"Lizard (HB)\0", NULL, "Brad Smith", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_lizardRomInfo, nes_lizardRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_logjammersRomDesc[] = {
	{ "Log Jammers (HB).nes",          393232, 0x451dd988, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_logjammers)
STD_ROM_FN(nes_logjammers)

struct BurnDriver BurnDrvnes_logjammers = {
	"nes_logjammers", NULL, NULL, NULL, "2017",
	"Log Jammers (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_BALLPADDLE, 0,
	NESGetZipName, nes_logjammersRomInfo, nes_logjammersRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nebsanddebsRomDesc[] = {
	{ "Nebs and Debs (HB).nes",          40976, 0x570383a9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nebsanddebs)
STD_ROM_FN(nes_nebsanddebs)

struct BurnDriver BurnDrvnes_nebsanddebs = {
	"nes_nebsanddebs", NULL, NULL, NULL, "2018",
	"Nebs and Debs (HB)\0", NULL, "Dullahan", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_nebsanddebsRomInfo, nes_nebsanddebsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_neoheialiRomDesc[] = {
	{ "NEO Heiankyo Alien (HB).nes",          655376, 0xf9450fc9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_neoheiali)
STD_ROM_FN(nes_neoheiali)

struct BurnDriver BurnDrvnes_neoheiali = {
	"nes_neoheiali", NULL, NULL, NULL, "2017",
	"NEO Heiankyo Alien (HB)\0", NULL, "Hyperware Co.", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MAZE, 0,
	NESGetZipName, nes_neoheialiRomInfo, nes_neoheialiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nesvircleRomDesc[] = {
	{ "NES Virus Cleaner (HB).nes",          40976, 0x5f17cb98, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nesvircle)
STD_ROM_FN(nes_nesvircle)

struct BurnDriver BurnDrvnes_nesvircle = {
	"nes_nesvircle", NULL, NULL, NULL, "2016",
	"Virus Cleaner (HB)\0", NULL, "Second Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_nesvircleRomInfo, nes_nesvircleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_nomolosRomDesc[] = {
	{ "Nomolos (HB).nes",          524304, 0x917154ac, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_nomolos)
STD_ROM_FN(nes_nomolos)

struct BurnDriver BurnDrvnes_nomolos = {
	"nes_nomolos", NULL, NULL, NULL, "2014",
	"Nomolos (HB)\0", NULL, "Gradual Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_nomolosRomInfo, nes_nomolosRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_panicdizzyRomDesc[] = {
	{ "Panic Dizzy (HB).nes",          65552, 0xc5b23abd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_panicdizzy)
STD_ROM_FN(nes_panicdizzy)

struct BurnDriver BurnDrvnes_panicdizzy = {
	"nes_panicdizzy", NULL, NULL, NULL, "2019",
	"Panic Dizzy (HB)\0", NULL, "Oliver Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_panicdizzyRomInfo, nes_panicdizzyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_wonderlanddizzyRomDesc[] = {
	{ "Wonderland Dizzy (HB).nes",          262160, 0x62c97c6c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_wonderlanddizzy)
STD_ROM_FN(nes_wonderlanddizzy)

struct BurnDriver BurnDrvnes_wonderlanddizzy = {
	"nes_wonderlanddizzy", NULL, NULL, NULL, "2015",
	"Wonderland Dizzy (HB)\0", NULL, "Oliver Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_wonderlanddizzyRomInfo, nes_wonderlanddizzyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mystewordizRomDesc[] = {
	{ "Mystery World Dizzy (HB).nes",          262160, 0x03124bf3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mystewordiz)
STD_ROM_FN(nes_mystewordiz)

struct BurnDriver BurnDrvnes_mystewordiz = {
	"nes_mystewordiz", NULL, NULL, NULL, "2017",
	"Mystery World Dizzy (HB)\0", NULL, "Oliver Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_mystewordizRomInfo, nes_mystewordizRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_perfectpairRomDesc[] = {
	{ "Perfect Pair (HB).nes",          40976, 0xefef8b35, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_perfectpair)
STD_ROM_FN(nes_perfectpair)

struct BurnDriver BurnDrvnes_perfectpair = {
	"nes_perfectpair", NULL, NULL, NULL, "2016",
	"Perfect Pair (HB)\0", NULL, "Second Dimension", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_perfectpairRomInfo, nes_perfectpairRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_questforgeRomDesc[] = {
	{ "Quest Forge (HB).nes",          40976, 0xe81cba21, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_questforge)
STD_ROM_FN(nes_questforge)

struct BurnDriver BurnDrvnes_questforge = {
	"nes_questforge", NULL, NULL, NULL, "2016",
	"Quest Forge (HB)\0", NULL, "Ludosity", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_questforgeRomInfo, nes_questforgeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_twindragonsRomDesc[] = {
	{ "Twin Dragons (HB).nes",          524304, 0x0d29351e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_twindragons)
STD_ROM_FN(nes_twindragons)

struct BurnDriver BurnDrvnes_twindragons = {
	"nes_twindragons", NULL, NULL, NULL, "2018",
	"Twin Dragons (HB)\0", NULL, "Broke Studio", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_twindragonsRomInfo, nes_twindragonsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_twindracomediRomDesc[] = {
	{ "Twin Dragons Compo Edition (HB).nes",          65552, 0x311cb660, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_twindracomedi)
STD_ROM_FN(nes_twindracomedi)

struct BurnDriver BurnDrvnes_twindracomedi = {
	"nes_twindracomedi", "nes_twindragons", NULL, NULL, "2016",
	"Twin Dragons Compo Edition (HB)\0", NULL, "Broke Studio", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_twindracomediRomInfo, nes_twindracomediRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_micromagesRomDesc[] = {
	{ "Micro Mages.nes",          40976, 0xb0b6b5e4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_micromages)
STD_ROM_FN(nes_micromages)

struct BurnDriver BurnDrvnes_micromages = {
	"nes_micromages", NULL, NULL, NULL, "2019",
	"Micro Mages (HB)\0", NULL, "Morphcat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_micromagesRomInfo, nes_micromagesRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battlkidRomDesc[] = {
	{ "Battle Kid - Fortress of Peril (Japan).nes",          524304, 0xc7f3a581, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battlkid)
STD_ROM_FN(nes_battlkid)

struct BurnDriver BurnDrvnes_battlkid = {
	"nes_battlkid", NULL, NULL, NULL, "2010",
	"Battle Kid - Fortress of Peril (HB, Japan)\0", NULL, "Sivak", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_battlkidRomInfo, nes_battlkidRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battlkid2RomDesc[] = {
	{ "Battle Kid 2 - Mountain of Torment (USA).nes",          524304, 0x4916dc6d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battlkid2)
STD_ROM_FN(nes_battlkid2)

struct BurnDriver BurnDrvnes_battlkid2 = {
	"nes_battlkid2", NULL, NULL, NULL, "2012",
	"Battle Kid 2 - Mountain of Torment (HB, USA)\0", NULL, "Sivak", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_battlkid2RomInfo, nes_battlkid2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_superbatpunRomDesc[] = {
	{ "Super Bat Puncher (Demo).nes",          131088, 0xd04302fa, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superbatpun)
STD_ROM_FN(nes_superbatpun)

struct BurnDriver BurnDrvnes_superbatpun = {
	"nes_superbatpun", NULL, NULL, NULL, "2011",
	"Super Bat Puncher (HB, Demo)\0", NULL, "Morphcat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_superbatpunRomInfo, nes_superbatpunRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_indivisibleRomDesc[] = {
	{ "Indivisible (HB).nes",          262160, 0x9f304e88, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_indivisible)
STD_ROM_FN(nes_indivisible)

struct BurnDriver BurnDrvnes_indivisible = {
	"nes_indivisible", NULL, NULL, NULL, "201x",
	"Indivisible (HB)\0", NULL, "Lab Zero Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_indivisibleRomInfo, nes_indivisibleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_citytroubleRomDesc[] = {
	{ "City Trouble (HB).nes",          40976, 0x2481be7e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_citytrouble)
STD_ROM_FN(nes_citytrouble)

struct BurnDriver BurnDrvnes_citytrouble = {
	"nes_citytrouble", NULL, NULL, NULL, "2016",
	"City Trouble (HB)\0", NULL, "Den Kat Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_citytroubleRomInfo, nes_citytroubleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_lawnmowerRomDesc[] = {
	{ "Lawn Mower (HB).nes",          24592, 0x2e03cb7f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_lawnmower)
STD_ROM_FN(nes_lawnmower)

struct BurnDriver BurnDrvnes_lawnmower = {
	"nes_lawnmower", NULL, NULL, NULL, "2011",
	"Lawn Mower (HB)\0", NULL, "Shiru", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_lawnmowerRomInfo, nes_lawnmowerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_halloween86aRomDesc[] = {
	{ "Halloween'86 (HB, alt).nes",          524304, 0xd008091a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_halloween86a)
STD_ROM_FN(nes_halloween86a)

struct BurnDriver BurnDrvnes_halloween86a = {
	"nes_halloween86a", "nes_halloween86", NULL, NULL, "2016",
	"Halloween'86 (HB, alt)\0", NULL, "Retrotainment Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_halloween86aRomInfo, nes_halloween86aRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_halloween86RomDesc[] = {
	{ "Halloween'86 (HB).nes",          524304, 0x454252a4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_halloween86)
STD_ROM_FN(nes_halloween86)

struct BurnDriver BurnDrvnes_halloween86 = {
	"nes_halloween86", NULL, NULL, NULL, "2016",
	"Halloween'86 (HB)\0", NULL, "Retrotainment Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_SCRFIGHT, 0,
	NESGetZipName, nes_halloween86RomInfo, nes_halloween86RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_legendsofowliaRomDesc[] = {
	{ "Legends of Owlia (HB).nes",          524304, 0x26858010, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_legendsofowlia)
STD_ROM_FN(nes_legendsofowlia)

struct BurnDriver BurnDrvnes_legendsofowlia = {
	"nes_legendsofowlia", NULL, NULL, NULL, "2014",
	"Legends of Owlia (HB)\0", NULL, "Gradual Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_RPG, 0,
	NESGetZipName, nes_legendsofowliaRomInfo, nes_legendsofowliaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_mojontwimulRomDesc[] = {
	{ "Mojon Twins Multicart (HB).nes",          393232, 0xeec1e108, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_mojontwimul)
STD_ROM_FN(nes_mojontwimul)

struct BurnDriver BurnDrvnes_mojontwimul = {
	"nes_mojontwimul", NULL, NULL, NULL, "2016",
	"Mojon Twins Multicart (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_mojontwimulRomInfo, nes_mojontwimulRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_superpainterRomDesc[] = {
	{ "Super Painter (HB).nes",          40976, 0x5d483272, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superpainter)
STD_ROM_FN(nes_superpainter)

struct BurnDriver BurnDrvnes_superpainter = {
	"nes_superpainter", NULL, NULL, NULL, "2015",
	"Super Painter (HB)\0", NULL, "Retro Souls", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_superpainterRomInfo, nes_superpainterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_superuwolRomDesc[] = {
	{ "Super UWOL (HB).nes",          40976, 0x9fbb9e1d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_superuwol)
STD_ROM_FN(nes_superuwol)

struct BurnDriver BurnDrvnes_superuwol = {
	"nes_superuwol", NULL, NULL, NULL, "2015",
	"Super UWOL (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_superuwolRomInfo, nes_superuwolRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_jetpacoRomDesc[] = {
	{ "Jet-Paco (HB).nes",          40976, 0xb5f18574, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jetpaco)
STD_ROM_FN(nes_jetpaco)

struct BurnDriver BurnDrvnes_jetpaco = {
	"nes_jetpaco", NULL, NULL, NULL, "2015",
	"Jet-Paco (HB)\0", NULL, "Mojon Twins", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PLATFORM, 0,
	NESGetZipName, nes_jetpacoRomInfo, nes_jetpacoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_draiochtRomDesc[] = {
	{ "Draiocht (HB).nes",          40976, 0x3e7aa82d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_draiocht)
STD_ROM_FN(nes_draiocht)

struct BurnDriver BurnDrvnes_draiocht = {
	"nes_draiocht", NULL, NULL, NULL, "2019",
	"Draiocht (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 2, HARDWARE_NES, GBF_PUZZLE, 0,
	NESGetZipName, nes_draiochtRomInfo, nes_draiochtRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_justiceduelRomDesc[] = {
	{ "Justice Duel (HB).nes",          196624, 0xada8853b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_justiceduel)
STD_ROM_FN(nes_justiceduel)

struct BurnDriver BurnDrvnes_justiceduel = {
	"nes_justiceduel", NULL, NULL, NULL, "2019",
	"Justice Duel (HB)\0", NULL, "Mega Cat Studios", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_HOMEBREW, 4, HARDWARE_NES, GBF_ACTION, 0,
	NESGetZipName, nes_justiceduelRomInfo, nes_justiceduelRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

// Homebrew section end.

// GAMES (robot-generated)
static struct BurnRomInfo nes_10yardfightRomDesc[] = {
	{ "10-Yard Fight (USA).nes",          40976, 0xc38b62cb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_10yardfight)
STD_ROM_FN(nes_10yardfight)

struct BurnDriver BurnDrvnes_10yardfight = {
	"nes_10yardfight", NULL, NULL, NULL, "1985",
	"10-Yard Fight (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_10yardfightRomInfo, nes_10yardfightRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_1942RomDesc[] = {
	{ "1942 (Japan, USA).nes",          40976, 0xe9cf747f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_1942)
STD_ROM_FN(nes_1942)

struct BurnDriver BurnDrvnes_1942 = {
	"nes_1942", NULL, NULL, NULL, "1985",
	"1942 (Japan, USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_1942RomInfo, nes_1942RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_1943RomDesc[] = {
	{ "1943 - The Battle of Midway (USA).nes",          131088, 0xd131bf15, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_1943)
STD_ROM_FN(nes_1943)

struct BurnDriver BurnDrvnes_1943 = {
	"nes_1943", NULL, NULL, NULL, "1988",
	"1943 - The Battle of Midway (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_1943RomInfo, nes_1943RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_2010RomDesc[] = {
	{ "2010 - Street Fighter (Japan).nes",          262160, 0xc5a04eff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_2010)
STD_ROM_FN(nes_2010)

struct BurnDriver BurnDrvnes_2010 = {
	"nes_2010", NULL, NULL, NULL, "1990",
	"2010 - Street Fighter (Japan)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_2010RomInfo, nes_2010RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_3dbatworruRomDesc[] = {
	{ "3-D Battles of World Runner, The (USA).nes",          131088, 0xd5ada486, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_3dbatworru)
STD_ROM_FN(nes_3dbatworru)

struct BurnDriver BurnDrvnes_3dbatworru = {
	"nes_3dbatworru", NULL, NULL, NULL, "1987",
	"3-D Battles of World Runner, The (USA)\0", NULL, "Acclaim", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_3dbatworruRomInfo, nes_3dbatworruRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_3dblockRomDesc[] = {
	{ "3D Block (Unl).nes",          32784, 0xa760eeeb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_3dblock)
STD_ROM_FN(nes_3dblock)

struct BurnDriver BurnDrvnes_3dblock = {
	"nes_3dblock", NULL, NULL, NULL, "1989?",
	"3D Block (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_3dblockRomInfo, nes_3dblockRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_720degreesRomDesc[] = {
	{ "720 Degrees (USA).nes",          131088, 0x6bc968af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_720degrees)
STD_ROM_FN(nes_720degrees)

struct BurnDriver BurnDrvnes_720degrees = {
	"nes_720degrees", NULL, NULL, NULL, "1989",
	"720 Degrees (USA)\0", NULL, "Mindscape", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_720degreesRomInfo, nes_720degreesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_8eyesRomDesc[] = {
	{ "8 Eyes (USA).nes",          262160, 0x18f9bb24, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_8eyes)
STD_ROM_FN(nes_8eyes)

struct BurnDriver BurnDrvnes_8eyes = {
	"nes_8eyes", NULL, NULL, NULL, "1990",
	"8 Eyes (USA)\0", NULL, "Taxan", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_8eyesRomInfo, nes_8eyesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_abadoxRomDesc[] = {
	{ "Abadox - The Deadly Inner War (USA).nes",          262160, 0xdc87f63d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_abadox)
STD_ROM_FN(nes_abadox)

struct BurnDriver BurnDrvnes_abadox = {
	"nes_abadox", NULL, NULL, NULL, "1990",
	"Abadox - The Deadly Inner War (USA)\0", NULL, "Milton Bradley", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_abadoxRomInfo, nes_abadoxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_action52RomDesc[] = {
	{ "Action 52 (USA).nes",          2097168, 0x7fa90614, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_action52)
STD_ROM_FN(nes_action52)

struct BurnDriver BurnDrvnes_action52 = {
	"nes_action52", NULL, NULL, NULL, "1991",
	"Action 52 (USA)\0", NULL, "Active Enterprises", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_action52RomInfo, nes_action52RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_actioinnewyorRomDesc[] = {
	{ "Action in New York (Europe).nes",          262160, 0x5773a224, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_actioinnewyor)
STD_ROM_FN(nes_actioinnewyor)

struct BurnDriver BurnDrvnes_actioinnewyor = {
	"nes_actioinnewyor", "nes_scat", NULL, NULL, "1992",
	"Action in New York (Europe)\0", NULL, "Infogrames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_actioinnewyorRomInfo, nes_actioinnewyorRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL, SCREEN_WIDTH, SCREEN_HEIGHT_PAL
};

static struct BurnRomInfo nes_addamfamRomDesc[] = {
	{ "Addams Family, The (USA).nes",          262160, 0x08e2af80, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_addamfam)
STD_ROM_FN(nes_addamfam)

struct BurnDriver BurnDrvnes_addamfam = {
	"nes_addamfam", NULL, NULL, NULL, "1992",
	"Addams Family, The (USA)\0", NULL, "Ocean", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_addamfamRomInfo, nes_addamfamRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_addamfampugsschuRomDesc[] = {
	{ "Addams Family, The - Pugsley's Scavenger Hunt (USA).nes",          262160, 0x6b8d777d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_addamfampugsschu)
STD_ROM_FN(nes_addamfampugsschu)

struct BurnDriver BurnDrvnes_addamfampugsschu = {
	"nes_addamfampugsschu", NULL, NULL, NULL, "1993",
	"Addams Family, The - Pugsley's Scavenger Hunt (USA)\0", NULL, "Ocean", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_addamfampugsschuRomInfo, nes_addamfampugsschuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_adventureislandRomDesc[] = {
	{ "Adventure Island (USA).nes",          65552, 0xb78c09a2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_adventureisland)
STD_ROM_FN(nes_adventureisland)

struct BurnDriver BurnDrvnes_adventureisland = {
	"nes_adventureisland", NULL, NULL, NULL, "1988",
	"Adventure Island (USA)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_adventureislandRomInfo, nes_adventureislandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenisl3RomDesc[] = {
	{ "Adventure Island 3 (USA).nes",          262160, 0x952cdacf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenisl3)
STD_ROM_FN(nes_advenisl3)

struct BurnDriver BurnDrvnes_advenisl3 = {
	"nes_advenisl3", NULL, NULL, NULL, "1992",
	"Adventure Island 3 (USA)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenisl3RomInfo, nes_advenisl3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenisliiuRomDesc[] = {
	{ "Adventure Island II (USA) (Beta).nes",          262160, 0x51a401ca, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenisliiu)
STD_ROM_FN(nes_advenisliiu)

struct BurnDriver BurnDrvnes_advenisliiu = {
	"nes_advenisliiu", "nes_advenislii", NULL, NULL, "1991",
	"Adventure Island II (USA) (Beta)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenisliiuRomInfo, nes_advenisliiuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenisliiRomDesc[] = {
	{ "Adventure Island II (USA).nes",          262160, 0x5d8f80c7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenislii)
STD_ROM_FN(nes_advenislii)

struct BurnDriver BurnDrvnes_advenislii = {
	"nes_advenislii", NULL, NULL, NULL, "1991",
	"Adventure Island II (USA)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenisliiRomInfo, nes_advenisliiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenislivjRomDesc[] = {
	{ "Adventure Island IV (Japan).nes",          393232, 0xa58fe808, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenislivj)
STD_ROM_FN(nes_advenislivj)

struct BurnDriver BurnDrvnes_advenislivj = {
	"nes_advenislivj", "nes_advenisliv", NULL, NULL, "1989?",
	"Adventure Island IV (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenislivjRomInfo, nes_advenislivjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenislivRomDesc[] = {
	{ "Adventure Island IV (T-Eng).nes",          393232, 0x54c153c6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenisliv)
STD_ROM_FN(nes_advenisliv)

struct BurnDriver BurnDrvnes_advenisliv = {
	"nes_advenisliv", NULL, NULL, NULL, "1989?",
	"Adventure Island IV (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenislivRomInfo, nes_advenislivRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_adveninmagkinRomDesc[] = {
	{ "Adventures in the Magic Kingdom (USA).nes",          262160, 0x300e41b7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_adveninmagkin)
STD_ROM_FN(nes_adveninmagkin)

struct BurnDriver BurnDrvnes_adveninmagkin = {
	"nes_adveninmagkin", NULL, NULL, NULL, "1990",
	"Adventures in the Magic Kingdom (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_adveninmagkinRomInfo, nes_adveninmagkinRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenbaybilRomDesc[] = {
	{ "Adventures of Bayou Billy, The (USA).nes",          262160, 0x0ac631ba, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenbaybil)
STD_ROM_FN(nes_advenbaybil)

struct BurnDriver BurnDrvnes_advenbaybil = {
	"nes_advenbaybil", NULL, NULL, NULL, "1989",
	"Adventures of Bayou Billy, The (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenbaybilRomInfo, nes_advenbaybilRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advendinrikRomDesc[] = {
	{ "Adventures of Dino Riki (USA).nes",          65552, 0xd4f5287b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advendinrik)
STD_ROM_FN(nes_advendinrik)

struct BurnDriver BurnDrvnes_advendinrik = {
	"nes_advendinrik", NULL, NULL, NULL, "1989",
	"Adventures of Dino Riki (USA)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advendinrikRomInfo, nes_advendinrikRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advengilsislRomDesc[] = {
	{ "Adventures of Gilligan's Island, The (USA).nes",          131088, 0x667eebb4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advengilsisl)
STD_ROM_FN(nes_advengilsisl)

struct BurnDriver BurnDrvnes_advengilsisl = {
	"nes_advengilsisl", NULL, NULL, NULL, "1990",
	"Adventures of Gilligan's Island, The (USA)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advengilsislRomInfo, nes_advengilsislRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenlolRomDesc[] = {
	{ "Adventures of Lolo (USA).nes",          65552, 0xd9c4cbf7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenlol)
STD_ROM_FN(nes_advenlol)

struct BurnDriver BurnDrvnes_advenlol = {
	"nes_advenlol", NULL, NULL, NULL, "1989",
	"Adventures of Lolo (USA)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenlolRomInfo, nes_advenlolRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenlol2RomDesc[] = {
	{ "Adventures of Lolo 2 (USA).nes",          65552, 0x862ab1e5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenlol2)
STD_ROM_FN(nes_advenlol2)

struct BurnDriver BurnDrvnes_advenlol2 = {
	"nes_advenlol2", NULL, NULL, NULL, "1990",
	"Adventures of Lolo 2 (USA)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenlol2RomInfo, nes_advenlol2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenlol3RomDesc[] = {
	{ "Adventures of Lolo 3 (USA).nes",          262160, 0xb852cb26, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenlol3)
STD_ROM_FN(nes_advenlol3)

struct BurnDriver BurnDrvnes_advenlol3 = {
	"nes_advenlol3", NULL, NULL, NULL, "1991",
	"Adventures of Lolo 3 (USA)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenlol3RomInfo, nes_advenlol3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenradgraRomDesc[] = {
	{ "Adventures of Rad Gravity, The (USA).nes",          262160, 0xb55d5747, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenradgra)
STD_ROM_FN(nes_advenradgra)

struct BurnDriver BurnDrvnes_advenradgra = {
	"nes_advenradgra", NULL, NULL, NULL, "1990",
	"Adventures of Rad Gravity, The (USA)\0", NULL, "Activision", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenradgraRomInfo, nes_advenradgraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_advenrocandbulandfrRomDesc[] = {
	{ "Adventures of Rocky and Bullwinkle and Friends, The (USA).nes",          262160, 0x08effde5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_advenrocandbulandfr)
STD_ROM_FN(nes_advenrocandbulandfr)

struct BurnDriver BurnDrvnes_advenrocandbulandfr = {
	"nes_advenrocandbulandfr", NULL, NULL, NULL, "1992",
	"Adventures of Rocky and Bullwinkle and Friends, The (USA)\0", NULL, "THQ", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_advenrocandbulandfrRomInfo, nes_advenrocandbulandfrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_adventomsawRomDesc[] = {
	{ "Adventures of Tom Sawyer (USA).nes",          262160, 0x0336f9f3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_adventomsaw)
STD_ROM_FN(nes_adventomsaw)

struct BurnDriver BurnDrvnes_adventomsaw = {
	"nes_adventomsaw", NULL, NULL, NULL, "1989",
	"Adventures of Tom Sawyer (USA)\0", NULL, "Seta", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_adventomsawRomInfo, nes_adventomsawRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_afterburnerRomDesc[] = {
	{ "After Burner (USA).nes",          393232, 0xa0c558a9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_afterburner)
STD_ROM_FN(nes_afterburner)

struct BurnDriver BurnDrvnes_afterburner = {
	"nes_afterburner", NULL, NULL, NULL, "1989",
	"After Burner (USA)\0", NULL, "Tengen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_afterburnerRomInfo, nes_afterburnerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_afterburneriiRomDesc[] = {
	{ "After Burner II (Japan).nes",          393232, 0xa4928096, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_afterburnerii)
STD_ROM_FN(nes_afterburnerii)

struct BurnDriver BurnDrvnes_afterburnerii = {
	"nes_afterburnerii", NULL, NULL, NULL, "1989",
	"After Burner II (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_afterburneriiRomInfo, nes_afterburneriiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_airfortressRomDesc[] = {
	{ "Air Fortress (USA).nes",          163856, 0xb8fd02b7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_airfortress)
STD_ROM_FN(nes_airfortress)

struct BurnDriver BurnDrvnes_airfortress = {
	"nes_airfortress", NULL, NULL, NULL, "1989",
	"Air Fortress (USA)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_airfortressRomInfo, nes_airfortressRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_airwolfjRomDesc[] = {
	{ "Airwolf (Japan).nes",          262160, 0x9639baae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_airwolfj)
STD_ROM_FN(nes_airwolfj)

struct BurnDriver BurnDrvnes_airwolfj = {
	"nes_airwolfj", "nes_airwolf", NULL, NULL, "1988",
	"Airwolf (Japan)\0", NULL, "Kyugo Boueki", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_airwolfjRomInfo, nes_airwolfjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_airwolftRomDesc[] = {
	{ "Airwolf (T-eng).nes",          262160, 0xdeb8ef4a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_airwolft)
STD_ROM_FN(nes_airwolft)

struct BurnDriver BurnDrvnes_airwolft = {
	"nes_airwolft", "nes_airwolf", NULL, NULL, "1988",
	"Airwolf (T-eng)\0", NULL, "Kyugo Boueki", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_airwolftRomInfo, nes_airwolftRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_airwolfRomDesc[] = {
	{ "Airwolf (USA).nes",          163856, 0xf0a5eb24, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_airwolf)
STD_ROM_FN(nes_airwolf)

struct BurnDriver BurnDrvnes_airwolf = {
	"nes_airwolf", NULL, NULL, NULL, "1989",
	"Airwolf (USA)\0", NULL, "Acclaim", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_airwolfRomInfo, nes_airwolfRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_akirajRomDesc[] = {
	{ "Akira (Japan).nes",          393232, 0x9d83974c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akiraj)
STD_ROM_FN(nes_akiraj)

struct BurnDriver BurnDrvnes_akiraj = {
	"nes_akiraj", "nes_akira", NULL, NULL, "1988",
	"Akira (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akirajRomInfo, nes_akirajRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_akiraRomDesc[] = {
	{ "Akira (T-Eng).nes",          524304, 0xab6a9ee1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akira)
STD_ROM_FN(nes_akira)

struct BurnDriver BurnDrvnes_akira = {
	"nes_akira", NULL, NULL, NULL, "1988",
	"Akira (T-Eng)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akiraRomInfo, nes_akiraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_akumadenjRomDesc[] = {
	{ "Akumajou Densetsu (Japan).nes",          393232, 0x2e93ce72, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumadenj)
STD_ROM_FN(nes_akumadenj)

struct BurnDriver BurnDrvnes_akumadenj = {
	"nes_akumadenj", "nes_castliii", NULL, NULL, "1989",
	"Akumajou Densetsu (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akumadenjRomInfo, nes_akumadenjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_akumadenRomDesc[] = {
	{ "Akumajou Densetsu (T-Eng).nes",          393232, 0x1965a3dc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumaden)
STD_ROM_FN(nes_akumaden)

struct BurnDriver BurnDrvnes_akumaden = {
	"nes_akumaden", "nes_castliii", NULL, NULL, "1989",
	"Akumajou Densetsu (T-Eng)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akumadenRomInfo, nes_akumadenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_akumajoudraculaRomDesc[] = {
	{ "Akumajou Dracula (Japan).nes",          131088, 0x93f0488c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumajoudracula)
STD_ROM_FN(nes_akumajoudracula)

struct BurnDriver BurnDrvnes_akumajoudracula = {
	"nes_akumajoudracula", "nes_castlevania", NULL, NULL, "1993",
	"Akumajou Dracula (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akumajoudraculaRomInfo, nes_akumajoudraculaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_akumaspeRomDesc[] = {
	{ "Akumajou Special - Boku Dracula-kun (Japan).nes",          262160, 0xf916d969, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_akumaspe)
STD_ROM_FN(nes_akumaspe)

struct BurnDriver BurnDrvnes_akumaspe = {
	"nes_akumaspe", "nes_kiddracula", NULL, NULL, "1990",
	"Akumajou Special - Boku Dracula-kun (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_akumaspeRomInfo, nes_akumaspeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_alunsjrturraRomDesc[] = {
	{ "Al Unser Jr. Turbo Racing (USA).nes",          262160, 0x023c7774, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alunsjrturra)
STD_ROM_FN(nes_alunsjrturra)

struct BurnDriver BurnDrvnes_alunsjrturra = {
	"nes_alunsjrturra", NULL, NULL, NULL, "1990",
	"Al Unser Jr. Turbo Racing (USA)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_alunsjrturraRomInfo, nes_alunsjrturraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_aladdinRomDesc[] = {
	{ "Aladdin (Europe).nes",          262160, 0xdcd55bec, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_aladdin)
STD_ROM_FN(nes_aladdin)

struct BurnDriver BurnDrvnes_aladdin = {
	"nes_aladdin", NULL, NULL, NULL, "1994",
	"Aladdin (Europe)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_aladdinRomInfo, nes_aladdinRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL, SCREEN_WIDTH, SCREEN_HEIGHT_PAL
};

static struct BurnRomInfo nes_aladdin4RomDesc[] = {
	{ "Aladdin 4 (Unl).nes",          524304, 0xdda79434, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_aladdin4)
STD_ROM_FN(nes_aladdin4)

struct BurnDriver BurnDrvnes_aladdin4 = {
	"nes_aladdin4", NULL, NULL, NULL, "1989?",
	"Aladdin 4 (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_aladdin4RomInfo, nes_aladdin4RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_alfommaglaRomDesc[] = {
	{ "Alfombra Magica, La (Spain) (Rev 1) (Gluk Video) (Unl).nes",          65552, 0x7f165221, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alfommagla)
STD_ROM_FN(nes_alfommagla)

struct BurnDriver BurnDrvnes_alfommagla = {
	"nes_alfommagla", NULL, NULL, NULL, "1989?",
	"Alfombra Magica, La (Spain) (Rev 1) (Gluk Video) (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_alfommaglaRomInfo, nes_alfommaglaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_alfredchickenRomDesc[] = {
	{ "Alfred Chicken (USA).nes",          131088, 0xa01ef87e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alfredchicken)
STD_ROM_FN(nes_alfredchicken)

struct BurnDriver BurnDrvnes_alfredchicken = {
	"nes_alfredchicken", NULL, NULL, NULL, "1994",
	"Alfred Chicken (USA)\0", NULL, "Mindscape", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_alfredchickenRomInfo, nes_alfredchickenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_alien3RomDesc[] = {
	{ "Alien 3 (USA).nes",          262160, 0xefb4ca05, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alien3)
STD_ROM_FN(nes_alien3)

struct BurnDriver BurnDrvnes_alien3 = {
	"nes_alien3", NULL, NULL, NULL, "1993",
	"Alien 3 (USA)\0", NULL, "LJN", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_alien3RomInfo, nes_alien3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_aliensyndromeRomDesc[] = {
	{ "Alien Syndrome (USA).nes",          262160, 0x9e8351af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_aliensyndrome)
STD_ROM_FN(nes_aliensyndrome)

struct BurnDriver BurnDrvnes_aliensyndrome = {
	"nes_aliensyndrome", NULL, NULL, NULL, "1989",
	"Alien Syndrome (USA)\0", NULL, "Tengen", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_aliensyndromeRomInfo, nes_aliensyndromeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_alphamissionRomDesc[] = {
	{ "Alpha Mission (USA).nes",          65552, 0x02e67223, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_alphamission)
STD_ROM_FN(nes_alphamission)

struct BurnDriver BurnDrvnes_alphamission = {
	"nes_alphamission", NULL, NULL, NULL, "1987",
	"Alpha Mission (USA)\0", NULL, "SNK", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_alphamissionRomInfo, nes_alphamissionRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_amagonRomDesc[] = {
	{ "Amagon (USA).nes",          131088, 0xd1bde95c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_amagon)
STD_ROM_FN(nes_amagon)

struct BurnDriver BurnDrvnes_amagon = {
	"nes_amagon", NULL, NULL, NULL, "1989",
	"Amagon (USA)\0", NULL, "Sammy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_amagonRomInfo, nes_amagonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_americandreamjRomDesc[] = {
	{ "American Dream (Japan).nes",          262160, 0x5249764c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_americandreamj)
STD_ROM_FN(nes_americandreamj)

struct BurnDriver BurnDrvnes_americandreamj = {
	"nes_americandreamj", "nes_americandream", NULL, NULL, "1989",
	"American Dream (Japan)\0", NULL, "Coconuts", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_americandreamjRomInfo, nes_americandreamjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_americandreamRomDesc[] = {
	{ "American Dream (T-Eng).nes",          262160, 0xd2275854, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_americandream)
STD_ROM_FN(nes_americandream)

struct BurnDriver BurnDrvnes_americandream = {
	"nes_americandream", NULL, NULL, NULL, "1989",
	"American Dream (T-Eng)\0", NULL, "Coconuts", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_americandreamRomInfo, nes_americandreamRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ameriglaRomDesc[] = {
	{ "American Gladiators (USA).nes",          262160, 0x74c08f86, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_amerigla)
STD_ROM_FN(nes_amerigla)

struct BurnDriver BurnDrvnes_amerigla = {
	"nes_amerigla", NULL, NULL, NULL, "1991",
	"American Gladiators (USA)\0", NULL, "GameTek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ameriglaRomInfo, nes_ameriglaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_antaradvRomDesc[] = {
	{ "Antarctic Adventure (Japan).nes",          24592, 0x9da252b4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_antaradv)
STD_ROM_FN(nes_antaradv)

struct BurnDriver BurnDrvnes_antaradv = {
	"nes_antaradv", NULL, NULL, NULL, "1985",
	"Antarctic Adventure (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_antaradvRomInfo, nes_antaradvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_anticipationRomDesc[] = {
	{ "Anticipation (USA).nes",          65552, 0x31d239d6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_anticipation)
STD_ROM_FN(nes_anticipation)

struct BurnDriver BurnDrvnes_anticipation = {
	"nes_anticipation", NULL, NULL, NULL, "1988",
	"Anticipation (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_anticipationRomInfo, nes_anticipationRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_archrivRomDesc[] = {
	{ "Arch Rivals - A Basketbrawl! (USA).nes",          131088, 0x37ef8319, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_archriv)
STD_ROM_FN(nes_archriv)

struct BurnDriver BurnDrvnes_archriv = {
	"nes_archriv", NULL, NULL, NULL, "1990",
	"Arch Rivals - A Basketbrawl! (USA)\0", NULL, "Acclaim", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_archrivRomInfo, nes_archrivRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_archonRomDesc[] = {
	{ "Archon (USA).nes",          131088, 0xc00d228d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_archon)
STD_ROM_FN(nes_archon)

struct BurnDriver BurnDrvnes_archon = {
	"nes_archon", NULL, NULL, NULL, "1989",
	"Archon (USA)\0", NULL, "Activision", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_archonRomInfo, nes_archonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_argusRomDesc[] = {
	{ "Argus (Japan).nes",          49168, 0xd6396aa7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_argus)
STD_ROM_FN(nes_argus)

struct BurnDriver BurnDrvnes_argus = {
	"nes_argus", NULL, NULL, NULL, "1986",
	"Argus (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_argusRomInfo, nes_argusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_arkanoidRomDesc[] = {
	{ "Arkanoid (USA).nes",          49168, 0xee93a28c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_arkanoid)
STD_ROM_FN(nes_arkanoid)

struct BurnDriver BurnDrvnes_arkanoid = {
	"nes_arkanoid", NULL, NULL, NULL, "1987",
	"Arkanoid (USA)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_arkanoidRomInfo, nes_arkanoidRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_arkanoidiiRomDesc[] = {
	{ "Arkanoid II (Japan).nes",          262160, 0xc7db2710, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_arkanoidii)
STD_ROM_FN(nes_arkanoidii)

struct BurnDriver BurnDrvnes_arkanoidii = {
	"nes_arkanoidii", NULL, NULL, NULL, "1988",
	"Arkanoid II (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_arkanoidiiRomInfo, nes_arkanoidiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_arkistasringRomDesc[] = {
	{ "Arkista's Ring (USA).nes",          65552, 0xcd3aa2a5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_arkistasring)
STD_ROM_FN(nes_arkistasring)

struct BurnDriver BurnDrvnes_arkistasring = {
	"nes_arkistasring", NULL, NULL, NULL, "1990",
	"Arkista's Ring (USA)\0", NULL, "Sammy", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_arkistasringRomInfo, nes_arkistasringRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_armadillojRomDesc[] = {
	{ "Armadillo (Japan).nes",          393232, 0xae73e0c2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_armadilloj)
STD_ROM_FN(nes_armadilloj)

struct BurnDriver BurnDrvnes_armadilloj = {
	"nes_armadilloj", "nes_armadillo", NULL, NULL, "1991",
	"Armadillo (Japan)\0", NULL, "IGS", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_armadillojRomInfo, nes_armadillojRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_armadilloRomDesc[] = {
	{ "Armadillo (T-eng).nes",          393232, 0x651c9916, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_armadillo)
STD_ROM_FN(nes_armadillo)

struct BurnDriver BurnDrvnes_armadillo = {
	"nes_armadillo", NULL, NULL, NULL, "1991",
	"Armadillo (T-eng)\0", NULL, "IGS", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_armadilloRomInfo, nes_armadilloRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_armeddrafanviljRomDesc[] = {
	{ "Armed Dragon Fantasy Villgust (Japan).nes",          393232, 0x5a8bfdf5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_armeddrafanvilj)
STD_ROM_FN(nes_armeddrafanvilj)

struct BurnDriver BurnDrvnes_armeddrafanvilj = {
	"nes_armeddrafanvilj", "nes_armeddrafanvil", NULL, NULL, "1989?",
	"Armed Dragon Fantasy Villgust (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_armeddrafanviljRomInfo, nes_armeddrafanviljRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_armeddrafanvilRomDesc[] = {
	{ "Armed Dragon Fantasy Villgust (T-Eng).nes",          524304, 0xde77f40a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_armeddrafanvil)
STD_ROM_FN(nes_armeddrafanvil)

struct BurnDriver BurnDrvnes_armeddrafanvil = {
	"nes_armeddrafanvil", NULL, NULL, NULL, "1989?",
	"Armed Dragon Fantasy Villgust (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_armeddrafanvilRomInfo, nes_armeddrafanvilRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_asmikkunlandRomDesc[] = {
	{ "Asmik-kun Land (Japan).nes",          262160, 0x3f871c93, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_asmikkunland)
STD_ROM_FN(nes_asmikkunland)

struct BurnDriver BurnDrvnes_asmikkunland = {
	"nes_asmikkunland", NULL, NULL, NULL, "1989?",
	"Asmik-kun Land (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_asmikkunlandRomInfo, nes_asmikkunlandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_asterixRomDesc[] = {
	{ "Asterix (Europe).nes",          131088, 0xde7e6767, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_asterix)
STD_ROM_FN(nes_asterix)

struct BurnDriver BurnDrvnes_asterix = {
	"nes_asterix", NULL, NULL, NULL, "1993",
	"Asterix (Europe)\0", NULL, "Infogrames", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_asterixRomInfo, nes_asterixRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL, SCREEN_WIDTH, SCREEN_HEIGHT_PAL
};

static struct BurnRomInfo nes_astrofanjRomDesc[] = {
	{ "Astro Fang - Super Machine (Japan).nes",          262160, 0xc3dae718, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_astrofanj)
STD_ROM_FN(nes_astrofanj)

struct BurnDriver BurnDrvnes_astrofanj = {
	"nes_astrofanj", "nes_astrofan", NULL, NULL, "1990",
	"Astro Fang - Super Machine (Japan)\0", NULL, "A Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_astrofanjRomInfo, nes_astrofanjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_astrofanRomDesc[] = {
	{ "Astro Fang - Super Machine (T-eng).nes",          262160, 0xcfdb30c9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_astrofan)
STD_ROM_FN(nes_astrofan)

struct BurnDriver BurnDrvnes_astrofan = {
	"nes_astrofan", NULL, NULL, NULL, "1990",
	"Astro Fang - Super Machine (T-eng)\0", NULL, "A Wave", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_astrofanRomInfo, nes_astrofanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_astrorobosasaRomDesc[] = {
	{ "Astro Robo Sasa (Japan).nes",          40976, 0xcab240b1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_astrorobosasa)
STD_ROM_FN(nes_astrorobosasa)

struct BurnDriver BurnDrvnes_astrorobosasa = {
	"nes_astrorobosasa", NULL, NULL, NULL, "1985",
	"Astro Robo Sasa (Japan)\0", NULL, "ASCII", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_astrorobosasaRomInfo, nes_astrorobosasaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_astyanaxRomDesc[] = {
	{ "Astyanax (USA).nes",          262160, 0x2fdfbc79, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_astyanax)
STD_ROM_FN(nes_astyanax)

struct BurnDriver BurnDrvnes_astyanax = {
	"nes_astyanax", NULL, NULL, NULL, "1990",
	"Astyanax (USA)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_astyanaxRomInfo, nes_astyanaxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_athenaRomDesc[] = {
	{ "Athena (USA).nes",          131088, 0x14d42113, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_athena)
STD_ROM_FN(nes_athena)

struct BurnDriver BurnDrvnes_athena = {
	"nes_athena", NULL, NULL, NULL, "1987",
	"Athena (USA)\0", NULL, "SNK", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_athenaRomInfo, nes_athenaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_atlantisnonazojRomDesc[] = {
	{ "Atlantis no Nazo (Japan).nes",          49168, 0xdccbe3ae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_atlantisnonazoj)
STD_ROM_FN(nes_atlantisnonazoj)

struct BurnDriver BurnDrvnes_atlantisnonazoj = {
	"nes_atlantisnonazoj", "nes_atlantisnonazo", NULL, NULL, "1986",
	"Atlantis no Nazo (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_atlantisnonazojRomInfo, nes_atlantisnonazojRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_atlantisnonazoRomDesc[] = {
	{ "Atlantis no Nazo (T-Eng).nes",          49168, 0x6a5ee2fc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_atlantisnonazo)
STD_ROM_FN(nes_atlantisnonazo)

struct BurnDriver BurnDrvnes_atlantisnonazo = {
	"nes_atlantisnonazo", NULL, NULL, NULL, "1986",
	"Atlantis no Nazo (T-Eng)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_atlantisnonazoRomInfo, nes_atlantisnonazoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_attacanigakjRomDesc[] = {
	{ "Attack Animal Gakuen (Japan).nes",          131088, 0x4e5c1c1d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_attacanigakj)
STD_ROM_FN(nes_attacanigakj)

struct BurnDriver BurnDrvnes_attacanigakj = {
	"nes_attacanigakj", "nes_attacanigak", NULL, NULL, "1987",
	"Attack Animal Gakuen (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_attacanigakjRomInfo, nes_attacanigakjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_attacanigakRomDesc[] = {
	{ "Attack Animal Gakuen (T-eng).nes",          131088, 0xefbf63ee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_attacanigak)
STD_ROM_FN(nes_attacanigak)

struct BurnDriver BurnDrvnes_attacanigak = {
	"nes_attacanigak", NULL, NULL, NULL, "1987",
	"Attack Animal Gakuen (T-eng)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_attacanigakRomInfo, nes_attacanigakRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_attackiltomRomDesc[] = {
	{ "Attack of the Killer Tomatoes (USA).nes",          262160, 0x11dc4071, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_attackiltom)
STD_ROM_FN(nes_attackiltom)

struct BurnDriver BurnDrvnes_attackiltom = {
	"nes_attackiltom", NULL, NULL, NULL, "1992",
	"Attack of the Killer Tomatoes (USA)\0", NULL, "THQ", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_attackiltomRomInfo, nes_attackiltomRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bwingsRomDesc[] = {
	{ "B-Wings (Japan).nes",          40976, 0x15a4facb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bwings)
STD_ROM_FN(nes_bwings)

struct BurnDriver BurnDrvnes_bwings = {
	"nes_bwings", NULL, NULL, NULL, "1986",
	"B-Wings (Japan)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bwingsRomInfo, nes_bwingsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_babelnotouRomDesc[] = {
	{ "Babel no Tou (Japan).nes",          65552, 0x154a31b6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_babelnotou)
STD_ROM_FN(nes_babelnotou)

struct BurnDriver BurnDrvnes_babelnotou = {
	"nes_babelnotou", NULL, NULL, NULL, "1986",
	"Babel no Tou (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_babelnotouRomInfo, nes_babelnotouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_babyboomerRomDesc[] = {
	{ "Baby Boomer (USA).nes",          81936, 0xed58dddd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_babyboomer)
STD_ROM_FN(nes_babyboomer)

struct BurnDriver BurnDrvnes_babyboomer = {
	"nes_babyboomer", NULL, NULL, NULL, "1989",
	"Baby Boomer (USA)\0", NULL, "Color Dreams", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_babyboomerRomInfo, nes_babyboomerRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_backfutRomDesc[] = {
	{ "Back to the Future (USA).nes",          65552, 0x7c40d6c6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_backfut)
STD_ROM_FN(nes_backfut)

struct BurnDriver BurnDrvnes_backfut = {
	"nes_backfut", NULL, NULL, NULL, "1989",
	"Back to the Future (USA)\0", NULL, "LJN", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_backfutRomInfo, nes_backfutRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_backfutpariiiiiRomDesc[] = {
	{ "Back to the Future Part II & III (USA).nes",          262160, 0x5a09134f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_backfutpariiiii)
STD_ROM_FN(nes_backfutpariiiii)

struct BurnDriver BurnDrvnes_backfutpariiiii = {
	"nes_backfutpariiiii", NULL, NULL, NULL, "1990",
	"Back to the Future Part II & III (USA)\0", NULL, "LJN", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_backfutpariiiiiRomInfo, nes_backfutpariiiiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_baddudesRomDesc[] = {
	{ "Bad Dudes (USA).nes",          262160, 0x3c8e79e9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_baddudes)
STD_ROM_FN(nes_baddudes)

struct BurnDriver BurnDrvnes_baddudes = {
	"nes_baddudes", NULL, NULL, NULL, "1990",
	"Bad Dudes (USA)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_baddudesRomInfo, nes_baddudesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_badnewbasRomDesc[] = {
	{ "Bad News Baseball (USA).nes",          262160, 0x2d69dd94, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_badnewbas)
STD_ROM_FN(nes_badnewbas)

struct BurnDriver BurnDrvnes_badnewbas = {
	"nes_badnewbas", NULL, NULL, NULL, "1990",
	"Bad News Baseball (USA)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_badnewbasRomInfo, nes_badnewbasRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_badstrbraRomDesc[] = {
	{ "Bad Street Brawler (USA).nes",          131088, 0xc8272d9a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_badstrbra)
STD_ROM_FN(nes_badstrbra)

struct BurnDriver BurnDrvnes_badstrbra = {
	"nes_badstrbra", NULL, NULL, NULL, "1989",
	"Bad Street Brawler (USA)\0", NULL, "Mattel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_badstrbraRomInfo, nes_badstrbraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ballstoRomDesc[] = {
	{ "Ball Story - Jong Yuk Chuen Suet Fa Jong II (China).nes",          131088, 0x1090a2e5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ballsto)
STD_ROM_FN(nes_ballsto)

struct BurnDriver BurnDrvnes_ballsto = {
	"nes_ballsto", NULL, NULL, NULL, "1989?",
	"Ball Story - Jong Yuk Chuen Suet Fa Jong II (China)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ballstoRomInfo, nes_ballstoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ballblazerRomDesc[] = {
	{ "Ballblazer (Japan).nes",          131088, 0x7b43b3ef, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ballblazer)
STD_ROM_FN(nes_ballblazer)

struct BurnDriver BurnDrvnes_ballblazer = {
	"nes_ballblazer", NULL, NULL, NULL, "1988",
	"Ballblazer (Japan)\0", NULL, "Pony Canyon", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ballblazerRomInfo, nes_ballblazerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_balloonfightRomDesc[] = {
	{ "Balloon Fight (USA).nes",          24592, 0xcb8fd7b2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_balloonfight)
STD_ROM_FN(nes_balloonfight)

struct BurnDriver BurnDrvnes_balloonfight = {
	"nes_balloonfight", NULL, NULL, NULL, "1986",
	"Balloon Fight (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_balloonfightRomInfo, nes_balloonfightRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_balloonmonsterRomDesc[] = {
	{ "Balloon Monster (Unl).nes",          40976, 0xb5ae9f8b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_balloonmonster)
STD_ROM_FN(nes_balloonmonster)

struct BurnDriver BurnDrvnes_balloonmonster = {
	"nes_balloonmonster", NULL, NULL, NULL, "1992",
	"Balloon Monster (Unl)\0", NULL, "Gluk Video", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_balloonmonsterRomInfo, nes_balloonmonsterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_baltronRomDesc[] = {
	{ "Baltron (Japan).nes",          40976, 0xf85f8769, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_baltron)
STD_ROM_FN(nes_baltron)

struct BurnDriver BurnDrvnes_baltron = {
	"nes_baltron", NULL, NULL, NULL, "1986",
	"Baltron (Japan)\0", NULL, "Toei Animation", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_baltronRomInfo, nes_baltronRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bananaRomDesc[] = {
	{ "Banana (Japan).nes",          49168, 0x5ac41464, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_banana)
STD_ROM_FN(nes_banana)

struct BurnDriver BurnDrvnes_banana = {
	"nes_banana", NULL, NULL, NULL, "1986",
	"Banana (Japan)\0", NULL, "Victor", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bananaRomInfo, nes_bananaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bananaprincegRomDesc[] = {
	{ "Banana Prince (Germany).nes",          262160, 0xa0f6edee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bananaprinceg)
STD_ROM_FN(nes_bananaprinceg)

struct BurnDriver BurnDrvnes_bananaprinceg = {
	"nes_bananaprinceg", "nes_bananaprince", NULL, NULL, "1992",
	"Banana Prince (Germany)\0", NULL, "Takara", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bananaprincegRomInfo, nes_bananaprincegRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bananaprincejRomDesc[] = {
	{ "Banana Prince (Japan).nes",          262160, 0x4c7e9492, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bananaprincej)
STD_ROM_FN(nes_bananaprincej)

struct BurnDriver BurnDrvnes_bananaprincej = {
	"nes_bananaprincej", "nes_bananaprince", NULL, NULL, "1992",
	"Banana Prince (Japan)\0", NULL, "Takara", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bananaprincejRomInfo, nes_bananaprincejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bananaprinceRomDesc[] = {
	{ "Banana Prince (T-Eng).nes",          262160, 0x3a2e77a7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bananaprince)
STD_ROM_FN(nes_bananaprince)

struct BurnDriver BurnDrvnes_bananaprince = {
	"nes_bananaprince", NULL, NULL, NULL, "1992",
	"Banana Prince (T-Eng)\0", NULL, "Takara", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bananaprinceRomInfo, nes_bananaprinceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bandikinancchiRomDesc[] = {
	{ "Bandit Kings of Ancient China (USA).nes",          393232, 0xa9e2bf31, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bandikinancchi)
STD_ROM_FN(nes_bandikinancchi)

struct BurnDriver BurnDrvnes_bandikinancchi = {
	"nes_bandikinancchi", NULL, NULL, NULL, "1990",
	"Bandit Kings of Ancient China (USA)\0", NULL, "Koei", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bandikinancchiRomInfo, nes_bandikinancchiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_barkebilstrishRomDesc[] = {
	{ "Barker Bill's Trick Shooting (USA).nes",          196624, 0xc898e6c1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_barkebilstrish)
STD_ROM_FN(nes_barkebilstrish)

struct BurnDriver BurnDrvnes_barkebilstrish = {
	"nes_barkebilstrish", NULL, NULL, NULL, "1990",
	"Barker Bill's Trick Shooting (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_barkebilstrishRomInfo, nes_barkebilstrishRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_basewarsRomDesc[] = {
	{ "Base Wars (USA).nes",          262160, 0xeed00ddf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_basewars)
STD_ROM_FN(nes_basewars)

struct BurnDriver BurnDrvnes_basewars = {
	"nes_basewars", NULL, NULL, NULL, "1991",
	"Base Wars (USA)\0", NULL, "Ultra Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_basewarsRomInfo, nes_basewarsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_basebsim1000RomDesc[] = {
	{ "Baseball Simulator 1.000 (USA).nes",          262160, 0x469b04f6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_basebsim1000)
STD_ROM_FN(nes_basebsim1000)

struct BurnDriver BurnDrvnes_basebsim1000 = {
	"nes_basebsim1000", NULL, NULL, NULL, "1990",
	"Baseball Simulator 1.000 (USA)\0", NULL, "Culture Brain", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_basebsim1000RomInfo, nes_basebsim1000RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_baseballstarsRomDesc[] = {
	{ "Baseball Stars (USA).nes",          262160, 0x1924f963, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_baseballstars)
STD_ROM_FN(nes_baseballstars)

struct BurnDriver BurnDrvnes_baseballstars = {
	"nes_baseballstars", NULL, NULL, NULL, "1989",
	"Baseball Stars (USA)\0", NULL, "SNK", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_baseballstarsRomInfo, nes_baseballstarsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_basebstaiiRomDesc[] = {
	{ "Baseball Stars II (USA).nes",          393232, 0x104fabee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_basebstaii)
STD_ROM_FN(nes_basebstaii)

struct BurnDriver BurnDrvnes_basebstaii = {
	"nes_basebstaii", NULL, NULL, NULL, "1992",
	"Baseball Stars II (USA)\0", NULL, "Romstar", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_basebstaiiRomInfo, nes_basebstaiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_basesloadedRomDesc[] = {
	{ "Bases Loaded (USA).nes",          327696, 0x5f1b9397, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_basesloaded)
STD_ROM_FN(nes_basesloaded)

struct BurnDriver BurnDrvnes_basesloaded = {
	"nes_basesloaded", NULL, NULL, NULL, "1988",
	"Bases Loaded (USA)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_basesloadedRomInfo, nes_basesloadedRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_basesloaiiRomDesc[] = {
	{ "Bases Loaded II - Second Season (USA).nes",          393232, 0x965b9a2f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_basesloaii)
STD_ROM_FN(nes_basesloaii)

struct BurnDriver BurnDrvnes_basesloaii = {
	"nes_basesloaii", NULL, NULL, NULL, "1990",
	"Bases Loaded II - Second Season (USA)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_basesloaiiRomInfo, nes_basesloaiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bashibazRomDesc[] = {
	{ "Bashi Bazook - Morphoid Masher (USA) (Proto).nes",          262160, 0x11c92b5d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bashibaz)
STD_ROM_FN(nes_bashibaz)

struct BurnDriver BurnDrvnes_bashibaz = {
	"nes_bashibaz", NULL, NULL, NULL, "0000",
	"Bashi Bazook - Morphoid Masher (USA) (Proto)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bashibazRomInfo, nes_bashibazRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_batmaretjokRomDesc[] = {
	{ "Batman - Return of the Joker (USA).nes",          393232, 0xe14a7971, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batmaretjok)
STD_ROM_FN(nes_batmaretjok)

struct BurnDriver BurnDrvnes_batmaretjok = {
	"nes_batmaretjok", NULL, NULL, NULL, "1991",
	"Batman - Return of the Joker (USA)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_batmaretjokRomInfo, nes_batmaretjokRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_batmavidgamjRomDesc[] = {
	{ "Batman - The Video Game (Japan).nes",          262160, 0x15f9a645, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batmavidgamj)
STD_ROM_FN(nes_batmavidgamj)

struct BurnDriver BurnDrvnes_batmavidgamj = {
	"nes_batmavidgamj", "nes_batmavidgam", NULL, NULL, "1989",
	"Batman - The Video Game (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_batmavidgamjRomInfo, nes_batmavidgamjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_batmavidgamRomDesc[] = {
	{ "Batman - The Video Game (USA).nes",          262160, 0x395569ec, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batmavidgam)
STD_ROM_FN(nes_batmavidgam)

struct BurnDriver BurnDrvnes_batmavidgam = {
	"nes_batmavidgam", NULL, NULL, NULL, "1990",
	"Batman - The Video Game (USA)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_batmavidgamRomInfo, nes_batmavidgamRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_batmanreturnsRomDesc[] = {
	{ "Batman Returns (USA).nes",          393232, 0xe625e398, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batmanreturns)
STD_ROM_FN(nes_batmanreturns)

struct BurnDriver BurnDrvnes_batmanreturns = {
	"nes_batmanreturns", NULL, NULL, NULL, "1993",
	"Batman Returns (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_batmanreturnsRomInfo, nes_batmanreturnsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_batsuteriiRomDesc[] = {
	{ "Batsu & Terii (Japan).nes",          131088, 0xca80fa58, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_batsuterii)
STD_ROM_FN(nes_batsuterii)

struct BurnDriver BurnDrvnes_batsuterii = {
	"nes_batsuterii", NULL, NULL, NULL, "1987",
	"Batsu & Terii (Japan)\0", NULL, "Use", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_batsuteriiRomInfo, nes_batsuteriiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battlebaseballRomDesc[] = {
	{ "Battle Baseball (Japan).nes",          524304, 0x8b65f724, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battlebaseball)
STD_ROM_FN(nes_battlebaseball)

struct BurnDriver BurnDrvnes_battlebaseball = {
	"nes_battlebaseball", NULL, NULL, NULL, "1989?",
	"Battle Baseball (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battlebaseballRomInfo, nes_battlebaseballRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battlechessRomDesc[] = {
	{ "Battle Chess (USA).nes",          262160, 0x7dc49898, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battlechess)
STD_ROM_FN(nes_battlechess)

struct BurnDriver BurnDrvnes_battlechess = {
	"nes_battlechess", NULL, NULL, NULL, "1990",
	"Battle Chess (USA)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battlechessRomInfo, nes_battlechessRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battleformulaRomDesc[] = {
	{ "Battle Formula (Japan).nes",          262160, 0xf6454be1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battleformula)
STD_ROM_FN(nes_battleformula)

struct BurnDriver BurnDrvnes_battleformula = {
	"nes_battleformula", NULL, NULL, NULL, "1991",
	"Battle Formula (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battleformulaRomInfo, nes_battleformulaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battlolyRomDesc[] = {
	{ "Battle of Olympus, The (USA).nes",          131088, 0xb99394c3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battloly)
STD_ROM_FN(nes_battloly)

struct BurnDriver BurnDrvnes_battloly = {
	"nes_battloly", NULL, NULL, NULL, "1989",
	"Battle of Olympus, The (USA)\0", NULL, "Broderbund", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battlolyRomInfo, nes_battlolyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battletankRomDesc[] = {
	{ "Battle Tank (USA).nes",          65552, 0x49c9ff12, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battletank)
STD_ROM_FN(nes_battletank)

struct BurnDriver BurnDrvnes_battletank = {
	"nes_battletank", NULL, NULL, NULL, "1990",
	"Battle Tank (USA)\0", NULL, "Absolute", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battletankRomInfo, nes_battletankRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battlecityRomDesc[] = {
	{ "BattleCity (Japan).nes",          24592, 0xf599a07e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battlecity)
STD_ROM_FN(nes_battlecity)

struct BurnDriver BurnDrvnes_battlecity = {
	"nes_battlecity", NULL, NULL, NULL, "1985",
	"BattleCity (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battlecityRomInfo, nes_battlecityRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battleshipRomDesc[] = {
	{ "Battleship (USA).nes",          65552, 0x89d3bdbc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battleship)
STD_ROM_FN(nes_battleship)

struct BurnDriver BurnDrvnes_battleship = {
	"nes_battleship", NULL, NULL, NULL, "1993",
	"Battleship (USA)\0", NULL, "Mindscape", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battleshipRomInfo, nes_battleshipRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battletoadsRomDesc[] = {
	{ "Battletoads (USA).nes",          262160, 0xba9164e7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battletoads)
STD_ROM_FN(nes_battletoads)

struct BurnDriver BurnDrvnes_battletoads = {
	"nes_battletoads", NULL, NULL, NULL, "1991",
	"Battletoads (USA)\0", NULL, "Tradewest", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battletoadsRomInfo, nes_battletoadsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_battldoudraRomDesc[] = {
	{ "Battletoads-Double Dragon (USA).nes",          262160, 0x53b02f3d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_battldoudra)
STD_ROM_FN(nes_battldoudra)

struct BurnDriver BurnDrvnes_battldoudra = {
	"nes_battldoudra", NULL, NULL, NULL, "1993",
	"Battletoads-Double Dragon (USA)\0", NULL, "Tradewest", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_battldoudraRomInfo, nes_battldoudraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_beautandbeaRomDesc[] = {
	{ "Beauty and the Beast (Europe).nes",          393232, 0xe08a5881, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_beautandbea)
STD_ROM_FN(nes_beautandbea)

struct BurnDriver BurnDrvnes_beautandbea = {
	"nes_beautandbea", NULL, NULL, NULL, "1989?",
	"Beauty and the Beast (Europe)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_beautandbeaRomInfo, nes_beautandbeaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL, SCREEN_WIDTH, SCREEN_HEIGHT_PAL
};

static struct BurnRomInfo nes_bee52RomDesc[] = {
	{ "Bee 52 (USA) (Unl).nes",          65552, 0x77a8486f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bee52)
STD_ROM_FN(nes_bee52)

struct BurnDriver BurnDrvnes_bee52 = {
	"nes_bee52", NULL, NULL, NULL, "1992",
	"Bee 52 (USA) (Unl)\0", NULL, "Camerica", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bee52RomInfo, nes_bee52RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_beetlejuiceRomDesc[] = {
	{ "Beetlejuice (USA).nes",          131088, 0x1b2bad13, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_beetlejuice)
STD_ROM_FN(nes_beetlejuice)

struct BurnDriver BurnDrvnes_beetlejuice = {
	"nes_beetlejuice", NULL, NULL, NULL, "1991",
	"Beetlejuice (USA)\0", NULL, "LJN", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_beetlejuiceRomInfo, nes_beetlejuiceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bibleadventuresRomDesc[] = {
	{ "Bible Adventures (Unl).nes",          131088, 0x5eb290af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bibleadventures)
STD_ROM_FN(nes_bibleadventures)

struct BurnDriver BurnDrvnes_bibleadventures = {
	"nes_bibleadventures", NULL, NULL, NULL, "1991",
	"Bible Adventures (Unl)\0", NULL, "Wisdom Tree", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bibleadventuresRomInfo, nes_bibleadventuresRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_biblebuffetRomDesc[] = {
	{ "Bible Buffet (Unl).nes",          262160, 0xdbbd1de9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_biblebuffet)
STD_ROM_FN(nes_biblebuffet)

struct BurnDriver BurnDrvnes_biblebuffet = {
	"nes_biblebuffet", NULL, NULL, NULL, "1993",
	"Bible Buffet (Unl)\0", NULL, "Wisdom Tree", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_biblebuffetRomInfo, nes_biblebuffetRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bignosfreoutRomDesc[] = {
	{ "Big Nose Freaks Out (USA).nes",          262160, 0xad16f6c7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bignosfreout)
STD_ROM_FN(nes_bignosfreout)

struct BurnDriver BurnDrvnes_bignosfreout = {
	"nes_bignosfreout", NULL, NULL, NULL, "1992",
	"Big Nose Freaks Out (USA)\0", NULL, "Camerica", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bignosfreoutRomInfo, nes_bignosfreoutRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bignoscavRomDesc[] = {
	{ "Big Nose the Caveman (USA).nes",          262160, 0xdcdf053f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bignoscav)
STD_ROM_FN(nes_bignoscav)

struct BurnDriver BurnDrvnes_bignoscav = {
	"nes_bignoscav", NULL, NULL, NULL, "1991",
	"Big Nose the Caveman (USA)\0", NULL, "Camerica", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bignoscavRomInfo, nes_bignoscavRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bigfootRomDesc[] = {
	{ "Bigfoot (USA).nes",          262160, 0xa99d45a4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bigfoot)
STD_ROM_FN(nes_bigfoot)

struct BurnDriver BurnDrvnes_bigfoot = {
	"nes_bigfoot", NULL, NULL, NULL, "1990",
	"Bigfoot (USA)\0", NULL, "Acclaim", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bigfootRomInfo, nes_bigfootRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_billtedsexcvigaadRomDesc[] = {
	{ "Bill & Ted's Excellent Video Game Adventure (USA).nes",          262160, 0xa905cc12, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_billtedsexcvigaad)
STD_ROM_FN(nes_billtedsexcvigaad)

struct BurnDriver BurnDrvnes_billtedsexcvigaad = {
	"nes_billtedsexcvigaad", NULL, NULL, NULL, "1991",
	"Bill & Ted's Excellent Video Game Adventure (USA)\0", NULL, "LJN", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_billtedsexcvigaadRomInfo, nes_billtedsexcvigaadRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_binarylandRomDesc[] = {
	{ "Binary Land (Japan).nes",          24592, 0x7e68abab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_binaryland)
STD_ROM_FN(nes_binaryland)

struct BurnDriver BurnDrvnes_binaryland = {
	"nes_binaryland", NULL, NULL, NULL, "1985",
	"Binary Land (Japan)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_binarylandRomInfo, nes_binarylandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bioforceapeRomDesc[] = {
	{ "Bio Force Ape (Japan) (En) (Proto).nes",          262160, 0x339d5a6d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bioforceape)
STD_ROM_FN(nes_bioforceape)

struct BurnDriver BurnDrvnes_bioforceape = {
	"nes_bioforceape", NULL, NULL, NULL, "1989?",
	"Bio Force Ape (Japan) (En) (Proto)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bioforceapeRomInfo, nes_bioforceapeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_biohazardcRomDesc[] = {
	{ "Bio Hazard (China).nes",          1048592, 0x75914cb8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_biohazardc)
STD_ROM_FN(nes_biohazardc)

struct BurnDriver BurnDrvnes_biohazardc = {
	"nes_biohazardc", "nes_biohazard", NULL, NULL, "1989?",
	"Bio Hazard (China)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_biohazardcRomInfo, nes_biohazardcRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_biohazardRomDesc[] = {
	{ "Bio Hazard (T-Eng).nes",          1048592, 0x8ca694fc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_biohazard)
STD_ROM_FN(nes_biohazard)

struct BurnDriver BurnDrvnes_biohazard = {
	"nes_biohazard", NULL, NULL, NULL, "1989?",
	"Bio Hazard (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_biohazardRomInfo, nes_biohazardRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_biomirbokupaRomDesc[] = {
	{ "Bio Miracle Bokutte Upa (Japan).nes",          262160, 0xe50ad737, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_biomirbokupa)
STD_ROM_FN(nes_biomirbokupa)

struct BurnDriver BurnDrvnes_biomirbokupa = {
	"nes_biomirbokupa", NULL, NULL, NULL, "1993",
	"Bio Miracle Bokutte Upa (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_biomirbokupaRomInfo, nes_biomirbokupaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_biosendanRomDesc[] = {
	{ "Bio Senshi Dan - Increaser Tono Tatakai (Japan).nes",          262160, 0x5b2d91d0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_biosendan)
STD_ROM_FN(nes_biosendan)

struct BurnDriver BurnDrvnes_biosendan = {
	"nes_biosendan", NULL, NULL, NULL, "1987",
	"Bio Senshi Dan - Increaser Tono Tatakai (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_biosendanRomInfo, nes_biosendanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bioniccommandoRomDesc[] = {
	{ "Bionic Commando (USA).nes",          262160, 0x83be000c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bioniccommando)
STD_ROM_FN(nes_bioniccommando)

struct BurnDriver BurnDrvnes_bioniccommando = {
	"nes_bioniccommando", NULL, NULL, NULL, "1988",
	"Bionic Commando (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bioniccommandoRomInfo, nes_bioniccommandoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bionicomRomDesc[] = {
	{ "Bionic Commando - Return of Hitler (Hack).nes",          262160, 0x519a3005, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bionicom)
STD_ROM_FN(nes_bionicom)

struct BurnDriver BurnDrvnes_bionicom = {
	"nes_bionicom", "nes_bioniccommando", NULL, NULL, "1989?",
	"Bionic Commando - Return of Hitler (Hack)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bionicomRomInfo, nes_bionicomRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_birdweekRomDesc[] = {
	{ "Bird Week (Japan).nes",          24592, 0xd9b9b3ab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_birdweek)
STD_ROM_FN(nes_birdweek)

struct BurnDriver BurnDrvnes_birdweek = {
	"nes_birdweek", NULL, NULL, NULL, "1986",
	"Bird Week (Japan)\0", NULL, "Toemiland", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_birdweekRomInfo, nes_birdweekRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blackbas2jRomDesc[] = {
	{ "Black Bass 2, The (Japan).nes",          131088, 0x5a514495, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blackbas2j)
STD_ROM_FN(nes_blackbas2j)

struct BurnDriver BurnDrvnes_blackbas2j = {
	"nes_blackbas2j", "nes_blackbas2", NULL, NULL, "1989?",
	"Black Bass 2, The (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blackbas2jRomInfo, nes_blackbas2jRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blackbas2RomDesc[] = {
	{ "Black Bass 2, The (USA).nes",          131088, 0x2e069e5f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blackbas2)
STD_ROM_FN(nes_blackbas2)

struct BurnDriver BurnDrvnes_blackbas2 = {
	"nes_blackbas2", NULL, NULL, NULL, "1989",
	"Black Bass 2, The (USA)\0", NULL, "HOT・B", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blackbas2RomInfo, nes_blackbas2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blackbasstheRomDesc[] = {
	{ "Black Bass, The (Japan).nes",          131088, 0x73bccc8b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blackbassthe)
STD_ROM_FN(nes_blackbassthe)

struct BurnDriver BurnDrvnes_blackbassthe = {
	"nes_blackbassthe", NULL, NULL, NULL, "1987",
	"Black Bass, The (Japan)\0", NULL, "HOT・B", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blackbasstheRomInfo, nes_blackbasstheRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bladesofsteelRomDesc[] = {
	{ "Blades of Steel (USA).nes",          131088, 0xb9bcf910, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bladesofsteel)
STD_ROM_FN(nes_bladesofsteel)

struct BurnDriver BurnDrvnes_bladesofsteel = {
	"nes_bladesofsteel", NULL, NULL, NULL, "1988",
	"Blades of Steel (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bladesofsteelRomInfo, nes_bladesofsteelRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blastermasterRomDesc[] = {
	{ "Blaster Master (USA).nes",          262160, 0x52bcf64a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blastermaster)
STD_ROM_FN(nes_blastermaster)

struct BurnDriver BurnDrvnes_blastermaster = {
	"nes_blastermaster", NULL, NULL, NULL, "1988",
	"Blaster Master (USA)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blastermasterRomInfo, nes_blastermasterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blobmanRomDesc[] = {
	{ "Blob Man (Unl).nes",          40976, 0xeef9db4b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blobman)
STD_ROM_FN(nes_blobman)

struct BurnDriver BurnDrvnes_blobman = {
	"nes_blobman", NULL, NULL, NULL, "1989?",
	"Blob Man (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blobmanRomInfo, nes_blobmanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blodilanRomDesc[] = {
	{ "Blodia Land - Puzzle Quest (Japan).nes",          262160, 0x6d3d0c1e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blodilan)
STD_ROM_FN(nes_blodilan)

struct BurnDriver BurnDrvnes_blodilan = {
	"nes_blodilan", NULL, NULL, NULL, "1990",
	"Blodia Land - Puzzle Quest (Japan)\0", NULL, "Tonkin House", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blodilanRomInfo, nes_blodilanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bluemarlintheRomDesc[] = {
	{ "Blue Marlin, The (USA).nes",          262160, 0xd9e8e747, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bluemarlinthe)
STD_ROM_FN(nes_bluemarlinthe)

struct BurnDriver BurnDrvnes_bluemarlinthe = {
	"nes_bluemarlinthe", NULL, NULL, NULL, "1992",
	"Blue Marlin, The (USA)\0", NULL, "HOT・B", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bluemarlintheRomInfo, nes_bluemarlintheRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_blueshadowRomDesc[] = {
	{ "Blue Shadow (Europe).nes",          262160, 0xe3b889d9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_blueshadow)
STD_ROM_FN(nes_blueshadow)

struct BurnDriver BurnDrvnes_blueshadow = {
	"nes_blueshadow", "nes_shadonin", NULL, NULL, "1991",
	"Blue Shadow (Europe)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_blueshadowRomInfo, nes_blueshadowRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL, SCREEN_WIDTH, SCREEN_HEIGHT_PAL
};

static struct BurnRomInfo nes_bluesbroRomDesc[] = {
	{ "Blues Brothers, The (USA).nes",          131088, 0xac273c14, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bluesbro)
STD_ROM_FN(nes_bluesbro)

struct BurnDriver BurnDrvnes_bluesbro = {
	"nes_bluesbro", NULL, NULL, NULL, "1992",
	"Blues Brothers, The (USA)\0", NULL, "Titus", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bluesbroRomInfo, nes_bluesbroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bokosukawarsRomDesc[] = {
	{ "Bokosuka Wars (Japan).nes",          40976, 0x196afc68, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bokosukawars)
STD_ROM_FN(nes_bokosukawars)

struct BurnDriver BurnDrvnes_bokosukawars = {
	"nes_bokosukawars", NULL, NULL, NULL, "1985",
	"Bokosuka Wars (Japan)\0", NULL, "ASCII", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bokosukawarsRomInfo, nes_bokosukawarsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bombermanRomDesc[] = {
	{ "Bomberman (USA).nes",          24592, 0xb9804046, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bomberman)
STD_ROM_FN(nes_bomberman)

struct BurnDriver BurnDrvnes_bomberman = {
	"nes_bomberman", NULL, NULL, NULL, "1989",
	"Bomberman (USA)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bombermanRomInfo, nes_bombermanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bombermaniiRomDesc[] = {
	{ "Bomberman II (USA).nes",          131088, 0xcc7bcfeb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bombermanii)
STD_ROM_FN(nes_bombermanii)

struct BurnDriver BurnDrvnes_bombermanii = {
	"nes_bombermanii", NULL, NULL, NULL, "1993",
	"Bomberman II (USA)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bombermaniiRomInfo, nes_bombermaniiRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bonksadventureRomDesc[] = {
	{ "Bonk's Adventure (USA).nes",          393232, 0x6b53d59a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bonksadventure)
STD_ROM_FN(nes_bonksadventure)

struct BurnDriver BurnDrvnes_bonksadventure = {
	"nes_bonksadventure", NULL, NULL, NULL, "1994",
	"Bonk's Adventure (USA)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bonksadventureRomInfo, nes_bonksadventureRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_boobykidsRomDesc[] = {
	{ "Booby Kids (Japan).nes",          131088, 0x828f23a8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_boobykids)
STD_ROM_FN(nes_boobykids)

struct BurnDriver BurnDrvnes_boobykids = {
	"nes_boobykids", NULL, NULL, NULL, "1987",
	"Booby Kids (Japan)\0", NULL, "Nichibutsu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_boobykidsRomInfo, nes_boobykidsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_boogermanRomDesc[] = {
	{ "Boogerman (Unl).nes",          524304, 0xf10d5e5b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_boogerman)
STD_ROM_FN(nes_boogerman)

struct BurnDriver BurnDrvnes_boogerman = {
	"nes_boogerman", NULL, NULL, NULL, "1989?",
	"Boogerman (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_boogermanRomInfo, nes_boogermanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_boulderdashRomDesc[] = {
	{ "Boulder Dash (USA).nes",          65552, 0x008f1536, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_boulderdash)
STD_ROM_FN(nes_boulderdash)

struct BurnDriver BurnDrvnes_boulderdash = {
	"nes_boulderdash", NULL, NULL, NULL, "1990",
	"Boulder Dash (USA)\0", NULL, "JVC", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_boulderdashRomInfo, nes_boulderdashRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bramstosdraRomDesc[] = {
	{ "Bram Stoker's Dracula (USA).nes",          262160, 0xf5321963, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bramstosdra)
STD_ROM_FN(nes_bramstosdra)

struct BurnDriver BurnDrvnes_bramstosdra = {
	"nes_bramstosdra", NULL, NULL, NULL, "1993",
	"Bram Stoker's Dracula (USA)\0", NULL, "Sony (CSG) Imagesoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bramstosdraRomInfo, nes_bramstosdraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_breakthruRomDesc[] = {
	{ "BreakThru (USA).nes",          262160, 0xc85bf3e3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_breakthru)
STD_ROM_FN(nes_breakthru)

struct BurnDriver BurnDrvnes_breakthru = {
	"nes_breakthru", NULL, NULL, NULL, "1987",
	"BreakThru (USA)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_breakthruRomInfo, nes_breakthruRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bubblebobbleRomDesc[] = {
	{ "Bubble Bobble (USA).nes",          163856, 0xd3a91b41, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bubblebobble)
STD_ROM_FN(nes_bubblebobble)

struct BurnDriver BurnDrvnes_bubblebobble = {
	"nes_bubblebobble", NULL, NULL, NULL, "1988",
	"Bubble Bobble (USA)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bubblebobbleRomInfo, nes_bubblebobbleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bubblbobpar2RomDesc[] = {
	{ "Bubble Bobble Part 2 (USA).nes",          262160, 0xca06cb60, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bubblbobpar2)
STD_ROM_FN(nes_bubblbobpar2)

struct BurnDriver BurnDrvnes_bubblbobpar2 = {
	"nes_bubblbobpar2", NULL, NULL, NULL, "1993",
	"Bubble Bobble Part 2 (USA)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bubblbobpar2RomInfo, nes_bubblbobpar2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_buckyohareRomDesc[] = {
	{ "Bucky O'Hare (USA).nes",          262160, 0x3c9622d3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_buckyohare)
STD_ROM_FN(nes_buckyohare)

struct BurnDriver BurnDrvnes_buckyohare = {
	"nes_buckyohare", NULL, NULL, NULL, "1992",
	"Bucky O'Hare (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_buckyohareRomInfo, nes_buckyohareRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_buggypopperRomDesc[] = {
	{ "Buggy Popper (Japan).nes",          65552, 0x46cc2d53, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_buggypopper)
STD_ROM_FN(nes_buggypopper)

struct BurnDriver BurnDrvnes_buggypopper = {
	"nes_buggypopper", "nes_bumpnjump", NULL, NULL, "1986",
	"Buggy Popper (Japan)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_buggypopperRomInfo, nes_buggypopperRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bugsbunbirbloRomDesc[] = {
	{ "Bugs Bunny Birthday Blowout, The (USA).nes",          262160, 0x38fdb7f4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bugsbunbirblo)
STD_ROM_FN(nes_bugsbunbirblo)

struct BurnDriver BurnDrvnes_bugsbunbirblo = {
	"nes_bugsbunbirblo", NULL, NULL, NULL, "1990",
	"Bugs Bunny Birthday Blowout, The (USA)\0", NULL, "Kemco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bugsbunbirbloRomInfo, nes_bugsbunbirbloRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bugsbuncracasRomDesc[] = {
	{ "Bugs Bunny Crazy Castle, The (USA).nes",          98320, 0xdb0c3656, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bugsbuncracas)
STD_ROM_FN(nes_bugsbuncracas)

struct BurnDriver BurnDrvnes_bugsbuncracas = {
	"nes_bugsbuncracas", NULL, NULL, NULL, "1989",
	"Bugs Bunny Crazy Castle, The (USA)\0", NULL, "Kemco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bugsbuncracasRomInfo, nes_bugsbuncracasRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_bumpnjumpRomDesc[] = {
	{ "Bump'n'Jump (USA).nes",          65552, 0x79bfe095, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_bumpnjump)
STD_ROM_FN(nes_bumpnjump)

struct BurnDriver BurnDrvnes_bumpnjump = {
	"nes_bumpnjump", NULL, NULL, NULL, "1988",
	"Bump'n'Jump (USA)\0", NULL, "Vic Tokai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_bumpnjumpRomInfo, nes_bumpnjumpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_buraifighterjRomDesc[] = {
	{ "Burai Fighter (Japan).nes",          65552, 0x76af01af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_buraifighterj)
STD_ROM_FN(nes_buraifighterj)

struct BurnDriver BurnDrvnes_buraifighterj = {
	"nes_buraifighterj", "nes_buraifighter", NULL, NULL, "1990",
	"Burai Fighter (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_buraifighterjRomInfo, nes_buraifighterjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_buraifighterRomDesc[] = {
	{ "Burai Fighter (USA).nes",          65552, 0x519ae8f2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_buraifighter)
STD_ROM_FN(nes_buraifighter)

struct BurnDriver BurnDrvnes_buraifighter = {
	"nes_buraifighter", NULL, NULL, NULL, "1990",
	"Burai Fighter (USA)\0", NULL, "Taxan", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_buraifighterRomInfo, nes_buraifighterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_burgertimeRomDesc[] = {
	{ "BurgerTime (USA).nes",          24592, 0x516549f9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_burgertime)
STD_ROM_FN(nes_burgertime)

struct BurnDriver BurnDrvnes_burgertime = {
	"nes_burgertime", NULL, NULL, NULL, "1987",
	"BurgerTime (USA)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_burgertimeRomInfo, nes_burgertimeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_buzzwaldogRomDesc[] = {
	{ "Buzz & Waldog (USA) (Proto) (Unl).nes",          262160, 0x1587bc5f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_buzzwaldog)
STD_ROM_FN(nes_buzzwaldog)

struct BurnDriver BurnDrvnes_buzzwaldog = {
	"nes_buzzwaldog", NULL, NULL, NULL, "1989?",
	"Buzz & Waldog (USA) (Proto) (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_buzzwaldogRomInfo, nes_buzzwaldogRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cabalRomDesc[] = {
	{ "Cabal (USA).nes",          131088, 0x4d5f2eb0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cabal)
STD_ROM_FN(nes_cabal)

struct BurnDriver BurnDrvnes_cabal = {
	"nes_cabal", NULL, NULL, NULL, "1990",
	"Cabal (USA)\0", NULL, "Milton Bradley", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cabalRomInfo, nes_cabalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_californiagamesRomDesc[] = {
	{ "California Games (USA).nes",          131088, 0x1c6f3036, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_californiagames)
STD_ROM_FN(nes_californiagames)

struct BurnDriver BurnDrvnes_californiagames = {
	"nes_californiagames", NULL, NULL, NULL, "1989",
	"California Games (USA)\0", NULL, "Milton Bradley", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_californiagamesRomInfo, nes_californiagamesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_caltron6in1RomDesc[] = {
	{ "Caltron 6-in-1 (USA).nes",          393232, 0xf78e68db, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_caltron6in1)
STD_ROM_FN(nes_caltron6in1)

struct BurnDriver BurnDrvnes_caltron6in1 = {
	"nes_caltron6in1", NULL, NULL, NULL, "1992",
	"Caltron 6-in-1 (USA)\0", NULL, "Caltron", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_caltron6in1RomInfo, nes_caltron6in1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_caltron9in1RomDesc[] = {
	{ "Caltron 9-in-1 (USA).nes",          786448, 0x9c529e3b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_caltron9in1)
STD_ROM_FN(nes_caltron9in1)

struct BurnDriver BurnDrvnes_caltron9in1 = {
	"nes_caltron9in1", NULL, NULL, NULL, "1989?",
	"Caltron 9-in-1 (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_caltron9in1RomInfo, nes_caltron9in1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_capcosgolmedch92RomDesc[] = {
	{ "Capcom's Gold Medal Challenge '92 (USA).nes",          262160, 0xa0f08ae1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_capcosgolmedch92)
STD_ROM_FN(nes_capcosgolmedch92)

struct BurnDriver BurnDrvnes_capcosgolmedch92 = {
	"nes_capcosgolmedch92", NULL, NULL, NULL, "1992",
	"Capcom's Gold Medal Challenge '92 (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_capcosgolmedch92RomInfo, nes_capcosgolmedch92RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captaameandaveRomDesc[] = {
	{ "Captain America and the Avengers (USA).nes",          262160, 0x7254d53d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captaameandave)
STD_ROM_FN(nes_captaameandave)

struct BurnDriver BurnDrvnes_captaameandave = {
	"nes_captaameandave", NULL, NULL, NULL, "1991",
	"Captain America and the Avengers (USA)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captaameandaveRomInfo, nes_captaameandaveRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captacomRomDesc[] = {
	{ "Captain Comic - The Adventure (USA).nes",          131088, 0x9357a157, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captacom)
STD_ROM_FN(nes_captacom)

struct BurnDriver BurnDrvnes_captacom = {
	"nes_captacom", NULL, NULL, NULL, "1989",
	"Captain Comic - The Adventure (USA)\0", NULL, "Color Dreams", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captacomRomInfo, nes_captacomRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captainedRomDesc[] = {
	{ "Captain ED (Japan).nes",          262160, 0x6ebfe4be, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captained)
STD_ROM_FN(nes_captained)

struct BurnDriver BurnDrvnes_captained = {
	"nes_captained", NULL, NULL, NULL, "1989",
	"Captain ED (Japan)\0", NULL, "CBS / Sony Group", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captainedRomInfo, nes_captainedRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captaplaandplaRomDesc[] = {
	{ "Captain Planet and the Planeteers (USA).nes",          262160, 0x21d34187, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captaplaandpla)
STD_ROM_FN(nes_captaplaandpla)

struct BurnDriver BurnDrvnes_captaplaandpla = {
	"nes_captaplaandpla", NULL, NULL, NULL, "1991",
	"Captain Planet and the Planeteers (USA)\0", NULL, "Mindscape", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captaplaandplaRomInfo, nes_captaplaandplaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captainsaverRomDesc[] = {
	{ "Captain Saver (Japan).nes",          262160, 0xfc97fc14, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captainsaver)
STD_ROM_FN(nes_captainsaver)

struct BurnDriver BurnDrvnes_captainsaver = {
	"nes_captainsaver", "nes_powerblade2", NULL, NULL, "1989?",
	"Captain Saver (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captainsaverRomInfo, nes_captainsaverRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captainsilverRomDesc[] = {
	{ "Captain Silver (Japan).nes",          262160, 0xa6b91bda, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captainsilver)
STD_ROM_FN(nes_captainsilver)

struct BurnDriver BurnDrvnes_captainsilver = {
	"nes_captainsilver", NULL, NULL, NULL, "1988",
	"Captain Silver (Japan)\0", NULL, "Tokuma Shoten", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captainsilverRomInfo, nes_captainsilverRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captainskyhawkRomDesc[] = {
	{ "Captain Skyhawk (USA) (Rev A).nes",          131088, 0x1f7d0668, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captainskyhawk)
STD_ROM_FN(nes_captainskyhawk)

struct BurnDriver BurnDrvnes_captainskyhawk = {
	"nes_captainskyhawk", NULL, NULL, NULL, "1990",
	"Captain Skyhawk (USA) (Rev A)\0", NULL, "Milton Bradley", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captainskyhawkRomInfo, nes_captainskyhawkRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captaintsubasaRomDesc[] = {
	{ "Captain Tsubasa (Japan).nes",          262160, 0x9e4207b3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captaintsubasa)
STD_ROM_FN(nes_captaintsubasa)

struct BurnDriver BurnDrvnes_captaintsubasa = {
	"nes_captaintsubasa", "nes_tecmocup", NULL, NULL, "1988",
	"Captain Tsubasa (Japan)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captaintsubasaRomInfo, nes_captaintsubasaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captatsuvoliijRomDesc[] = {
	{ "Captain Tsubasa Vol. II - Super Striker (Japan).nes",          393232, 0x14a366bb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captatsuvoliij)
STD_ROM_FN(nes_captatsuvoliij)

struct BurnDriver BurnDrvnes_captatsuvoliij = {
	"nes_captatsuvoliij", "nes_captatsuvolii", NULL, NULL, "1990",
	"Captain Tsubasa Vol. II - Super Striker (Japan)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captatsuvoliijRomInfo, nes_captatsuvoliijRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_captatsuvoliiRomDesc[] = {
	{ "Captain Tsubasa Vol. II - Super Striker (T-eng).nes",          393232, 0x465917f7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_captatsuvolii)
STD_ROM_FN(nes_captatsuvolii)

struct BurnDriver BurnDrvnes_captatsuvolii = {
	"nes_captatsuvolii", NULL, NULL, NULL, "1989?",
	"Captain Tsubasa Vol. II - Super Striker (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_captatsuvoliiRomInfo, nes_captatsuvoliiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_casinokidRomDesc[] = {
	{ "Casino Kid (USA).nes",          131088, 0xc651e21a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_casinokid)
STD_ROM_FN(nes_casinokid)

struct BurnDriver BurnDrvnes_casinokid = {
	"nes_casinokid", NULL, NULL, NULL, "1989",
	"Casino Kid (USA)\0", NULL, "Sofel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_casinokidRomInfo, nes_casinokidRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_casinokid2RomDesc[] = {
	{ "Casino Kid 2 (USA).nes",          131088, 0xb0de7b5f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_casinokid2)
STD_ROM_FN(nes_casinokid2)

struct BurnDriver BurnDrvnes_casinokid2 = {
	"nes_casinokid2", NULL, NULL, NULL, "1993",
	"Casino Kid 2 (USA)\0", NULL, "Sofel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_casinokid2RomInfo, nes_casinokid2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castelianRomDesc[] = {
	{ "Castelian (USA).nes",          131088, 0x39ef1ad6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castelian)
STD_ROM_FN(nes_castelian)

struct BurnDriver BurnDrvnes_castelian = {
	"nes_castelian", NULL, NULL, NULL, "1991",
	"Castelian (USA)\0", NULL, "Triffix", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castelianRomInfo, nes_castelianRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castleexcellentRomDesc[] = {
	{ "Castle Excellent (Japan).nes",          65552, 0x3089d649, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castleexcellent)
STD_ROM_FN(nes_castleexcellent)

struct BurnDriver BurnDrvnes_castleexcellent = {
	"nes_castleexcellent", NULL, NULL, NULL, "1986",
	"Castle Excellent (Japan)\0", NULL, "ASCII", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castleexcellentRomInfo, nes_castleexcellentRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castleofdeceitRomDesc[] = {
	{ "Castle of Deceit (USA).nes",          131088, 0x02e20d38, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castleofdeceit)
STD_ROM_FN(nes_castleofdeceit)

struct BurnDriver BurnDrvnes_castleofdeceit = {
	"nes_castleofdeceit", NULL, NULL, NULL, "1990",
	"Castle of Deceit (USA)\0", NULL, "Bunch Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castleofdeceitRomInfo, nes_castleofdeceitRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castleofdragonRomDesc[] = {
	{ "Castle of Dragon (USA).nes",          131088, 0x1c24cc9d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castleofdragon)
STD_ROM_FN(nes_castleofdragon)

struct BurnDriver BurnDrvnes_castleofdragon = {
	"nes_castleofdragon", NULL, NULL, NULL, "1990",
	"Castle of Dragon (USA)\0", NULL, "Seta", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castleofdragonRomInfo, nes_castleofdragonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castlevaniaRomDesc[] = {
	{ "Castlevania (USA).nes",          131088, 0x856114c8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castlevania)
STD_ROM_FN(nes_castlevania)

struct BurnDriver BurnDrvnes_castlevania = {
	"nes_castlevania", NULL, NULL, NULL, "1987",
	"Castlevania (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castlevaniaRomInfo, nes_castlevaniaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castliihRomDesc[] = {
	{ "Castlevania II - Simon's Quest (Hack, ReTrans+Map).nes",          262160, 0x153143e9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castliih)
STD_ROM_FN(nes_castliih)

struct BurnDriver BurnDrvnes_castliih = {
	"nes_castliih", "nes_castlii", NULL, NULL, "1988",
	"Castlevania II - Simon's Quest (Hack, ReTrans+Map)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castliihRomInfo, nes_castliihRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castliiRomDesc[] = {
	{ "Castlevania II - Simon's Quest (USA).nes",          262160, 0xa9c2c503, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castlii)
STD_ROM_FN(nes_castlii)

struct BurnDriver BurnDrvnes_castlii = {
	"nes_castlii", NULL, NULL, NULL, "1988",
	"Castlevania II - Simon's Quest (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castliiRomInfo, nes_castliiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castliiiRomDesc[] = {
	{ "Castlevania III - Dracula's Curse (USA).nes",          393232, 0x7cc9c669, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castliii)
STD_ROM_FN(nes_castliii)

struct BurnDriver BurnDrvnes_castliii = {
	"nes_castliii", NULL, NULL, NULL, "1990",
	"Castlevania III - Dracula's Curse (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castliiiRomInfo, nes_castliiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_castlsimRomDesc[] = {
	{ "Castlevania Simplified (Hack).nes",          131088, 0x451a0c75, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_castlsim)
STD_ROM_FN(nes_castlsim)

struct BurnDriver BurnDrvnes_castlsim = {
	"nes_castlsim", "nes_castlevania", NULL, NULL, "1989?",
	"Castlevania Simplified (Hack)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_castlsimRomInfo, nes_castlsimRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_catninteyjRomDesc[] = {
	{ "Cat Ninden Teyandee (Japan).nes",          262160, 0x6f1485ed, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_catninteyj)
STD_ROM_FN(nes_catninteyj)

struct BurnDriver BurnDrvnes_catninteyj = {
	"nes_catninteyj", "nes_catnintey", NULL, NULL, "1991",
	"Cat Ninden Teyandee (Japan)\0", NULL, "Tecmo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_catninteyjRomInfo, nes_catninteyjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_catninteyRomDesc[] = {
	{ "Cat Ninden Teyandee (T-eng).nes",          393232, 0x60f29773, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_catnintey)
STD_ROM_FN(nes_catnintey)

struct BurnDriver BurnDrvnes_catnintey = {
	"nes_catnintey", NULL, NULL, NULL, "1989?",
	"Cat Ninden Teyandee (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_catninteyRomInfo, nes_catninteyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chacknpopRomDesc[] = {
	{ "Chack'n Pop (Japan).nes",          24592, 0xe9b39b87, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chacknpop)
STD_ROM_FN(nes_chacknpop)

struct BurnDriver BurnDrvnes_chacknpop = {
	"nes_chacknpop", NULL, NULL, NULL, "1985",
	"Chack'n Pop (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chacknpopRomInfo, nes_chacknpopRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_challdraRomDesc[] = {
	{ "Challenge of the Dragon (USA).nes",          131088, 0xe737a11f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_challdra)
STD_ROM_FN(nes_challdra)

struct BurnDriver BurnDrvnes_challdra = {
	"nes_challdra", NULL, NULL, NULL, "1990",
	"Challenge of the Dragon (USA)\0", NULL, "Color Dreams", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_challdraRomInfo, nes_challdraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_challengerRomDesc[] = {
	{ "Challenger (Japan).nes",          40976, 0x94339dff, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_challenger)
STD_ROM_FN(nes_challenger)

struct BurnDriver BurnDrvnes_challenger = {
	"nes_challenger", NULL, NULL, NULL, "1985",
	"Challenger (Japan)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_challengerRomInfo, nes_challengerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_champbowRomDesc[] = {
	{ "Championship Bowling (USA).nes",          65552, 0xa5e89b19, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_champbow)
STD_ROM_FN(nes_champbow)

struct BurnDriver BurnDrvnes_champbow = {
	"nes_champbow", NULL, NULL, NULL, "1989",
	"Championship Bowling (USA)\0", NULL, "Romstar", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_champbowRomInfo, nes_champbowRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_champlodrunRomDesc[] = {
	{ "Championship Lode Runner (Japan).nes",          24592, 0xcfa85f7c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_champlodrun)
STD_ROM_FN(nes_champlodrun)

struct BurnDriver BurnDrvnes_champlodrun = {
	"nes_champlodrun", NULL, NULL, NULL, "1985",
	"Championship Lode Runner (Japan)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_champlodrunRomInfo, nes_champlodrunRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chaosworldjRomDesc[] = {
	{ "Chaos World (Japan).nes",          393232, 0xe2ae3e63, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chaosworldj)
STD_ROM_FN(nes_chaosworldj)

struct BurnDriver BurnDrvnes_chaosworldj = {
	"nes_chaosworldj", "nes_chaosworld", NULL, NULL, "1989?",
	"Chaos World (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chaosworldjRomInfo, nes_chaosworldjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chaosworldRomDesc[] = {
	{ "Chaos World (T-eng).nes",          393232, 0xfbd2a0e0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chaosworld)
STD_ROM_FN(nes_chaosworld)

struct BurnDriver BurnDrvnes_chaosworld = {
	"nes_chaosworld", NULL, NULL, NULL, "1989?",
	"Chaos World (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chaosworldRomInfo, nes_chaosworldRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chestfieRomDesc[] = {
	{ "Chester Field - Ankoku Shin e no Chousen (Japan).nes",          131088, 0xa6f40bd1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chestfie)
STD_ROM_FN(nes_chestfie)

struct BurnDriver BurnDrvnes_chestfie = {
	"nes_chestfie", NULL, NULL, NULL, "1987",
	"Chester Field - Ankoku Shin e no Chousen (Japan)\0", NULL, "Vic Tokai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chestfieRomInfo, nes_chestfieRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chikichimacmouraRomDesc[] = {
	{ "Chiki Chiki Machine Mou Race (Japan).nes",          262160, 0xfda7b555, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chikichimacmoura)
STD_ROM_FN(nes_chikichimacmoura)

struct BurnDriver BurnDrvnes_chikichimacmoura = {
	"nes_chikichimacmoura", "nes_wackyraces", NULL, NULL, "1989?",
	"Chiki Chiki Machine Mou Race (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chikichimacmouraRomInfo, nes_chikichimacmouraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chillerRomDesc[] = {
	{ "Chiller (USA).nes",          65552, 0xe4e7539a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chiller)
STD_ROM_FN(nes_chiller)

struct BurnDriver BurnDrvnes_chiller = {
	"nes_chiller", NULL, NULL, NULL, "1990",
	"Chiller (USA)\0", NULL, "AGCI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chillerRomInfo, nes_chillerRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chinarabbabRomDesc[] = {
	{ "China Rabbit Baby (Unl).nes",          524304, 0x6ab0db42, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chinarabbab)
STD_ROM_FN(nes_chinarabbab)

struct BurnDriver BurnDrvnes_chinarabbab = {
	"nes_chinarabbab", NULL, NULL, NULL, "1989?",
	"China Rabbit Baby (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chinarabbabRomInfo, nes_chinarabbabRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chipndalresraRomDesc[] = {
	{ "Chip 'n Dale - Rescue Rangers (USA).nes",          262160, 0x11da7e45, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chipndalresra)
STD_ROM_FN(nes_chipndalresra)

struct BurnDriver BurnDrvnes_chipndalresra = {
	"nes_chipndalresra", NULL, NULL, NULL, "1990",
	"Chip 'n Dale - Rescue Rangers (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chipndalresraRomInfo, nes_chipndalresraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chipndalresra2RomDesc[] = {
	{ "Chip 'n Dale - Rescue Rangers 2 (USA).nes",          262160, 0x91e4a289, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chipndalresra2)
STD_ROM_FN(nes_chipndalresra2)

struct BurnDriver BurnDrvnes_chipndalresra2 = {
	"nes_chipndalresra2", NULL, NULL, NULL, "1994",
	"Chip 'n Dale - Rescue Rangers 2 (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chipndalresra2RomInfo, nes_chipndalresra2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chitesenvazRomDesc[] = {
	{ "Chitei Senkuu Vazolder (Japan).nes",          262160, 0xa5487ac2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chitesenvaz)
STD_ROM_FN(nes_chitesenvaz)

struct BurnDriver BurnDrvnes_chitesenvaz = {
	"nes_chitesenvaz", NULL, NULL, NULL, "1989?",
	"Chitei Senkuu Vazolder (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chitesenvazRomInfo, nes_chitesenvazRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chiyonoooiRomDesc[] = {
	{ "Chiyonofuji no Ooichou (Japan).nes",          262160, 0x2e2f83ee, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chiyonoooi)
STD_ROM_FN(nes_chiyonoooi)

struct BurnDriver BurnDrvnes_chiyonoooi = {
	"nes_chiyonoooi", NULL, NULL, NULL, "1990",
	"Chiyonofuji no Ooichou (Japan)\0", NULL, "Face", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chiyonoooiRomInfo, nes_chiyonoooiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_choplifterRomDesc[] = {
	{ "Choplifter (Japan).nes",          49168, 0x8bb9afa7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_choplifter)
STD_ROM_FN(nes_choplifter)

struct BurnDriver BurnDrvnes_choplifter = {
	"nes_choplifter", NULL, NULL, NULL, "1986",
	"Choplifter (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_choplifterRomInfo, nes_choplifterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_choujyouRomDesc[] = {
	{ "Choujikuu Yousai - Macross (Japan).nes",          24592, 0xa3ca24d2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_choujyou)
STD_ROM_FN(nes_choujyou)

struct BurnDriver BurnDrvnes_choujyou = {
	"nes_choujyou", NULL, NULL, NULL, "1985",
	"Choujikuu Yousai - Macross (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_choujyouRomInfo, nes_choujyouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_choujsenjetjRomDesc[] = {
	{ "Choujin Sentai Jetman (Japan).nes",          262160, 0xf899bf59, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_choujsenjetj)
STD_ROM_FN(nes_choujsenjetj)

struct BurnDriver BurnDrvnes_choujsenjetj = {
	"nes_choujsenjetj", "nes_choujsenjet", NULL, NULL, "1989?",
	"Choujin Sentai Jetman (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_choujsenjetjRomInfo, nes_choujsenjetjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_choujsenjetRomDesc[] = {
	{ "Choujin Sentai Jetman (T-eng).nes",          262160, 0x7649c688, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_choujsenjet)
STD_ROM_FN(nes_choujsenjet)

struct BurnDriver BurnDrvnes_choujsenjet = {
	"nes_choujsenjet", NULL, NULL, NULL, "1989?",
	"Choujin Sentai Jetman (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_choujsenjetRomInfo, nes_choujsenjetRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_choujsenwarRomDesc[] = {
	{ "Choujinrou Senki Warwolf (Japan).nes",          262160, 0x58e8c244, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_choujsenwar)
STD_ROM_FN(nes_choujsenwar)

struct BurnDriver BurnDrvnes_choujsenwar = {
	"nes_choujsenwar", "nes_werew", NULL, NULL, "1991",
	"Choujinrou Senki Warwolf (Japan)\0", NULL, "Takara", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_choujsenwarRomInfo, nes_choujsenwarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chubbycherubRomDesc[] = {
	{ "Chubby Cherub (USA).nes",          40976, 0x90f12ac8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chubbycherub)
STD_ROM_FN(nes_chubbycherub)

struct BurnDriver BurnDrvnes_chubbycherub = {
	"nes_chubbycherub", NULL, NULL, NULL, "1986",
	"Chubby Cherub (USA)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chubbycherubRomInfo, nes_chubbycherubRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_chuukataisenRomDesc[] = {
	{ "Chuuka Taisen (Japan).nes",          262160, 0xe3beb057, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_chuukataisen)
STD_ROM_FN(nes_chuukataisen)

struct BurnDriver BurnDrvnes_chuukataisen = {
	"nes_chuukataisen", NULL, NULL, NULL, "1989",
	"Chuuka Taisen (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_chuukataisenRomInfo, nes_chuukataisenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_circuscaperRomDesc[] = {
	{ "Circus Caper (USA).nes",          262160, 0xbce1da2c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_circuscaper)
STD_ROM_FN(nes_circuscaper)

struct BurnDriver BurnDrvnes_circuscaper = {
	"nes_circuscaper", NULL, NULL, NULL, "1990",
	"Circus Caper (USA)\0", NULL, "Toho", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_circuscaperRomInfo, nes_circuscaperRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_circuscharlieRomDesc[] = {
	{ "Circus Charlie (Japan).nes",          24592, 0xda673c95, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_circuscharlie)
STD_ROM_FN(nes_circuscharlie)

struct BurnDriver BurnDrvnes_circuscharlie = {
	"nes_circuscharlie", NULL, NULL, NULL, "1986",
	"Circus Charlie (Japan)\0", NULL, "Soft Pro", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_circuscharlieRomInfo, nes_circuscharlieRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cityconnectionRomDesc[] = {
	{ "City Connection (USA).nes",          65552, 0xe1ad7ca8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cityconnection)
STD_ROM_FN(nes_cityconnection)

struct BurnDriver BurnDrvnes_cityconnection = {
	"nes_cityconnection", NULL, NULL, NULL, "1988",
	"City Connection (USA)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cityconnectionRomInfo, nes_cityconnectionRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_clashatdemRomDesc[] = {
	{ "Clash at Demonhead (USA).nes",          262160, 0xef1c8906, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_clashatdem)
STD_ROM_FN(nes_clashatdem)

struct BurnDriver BurnDrvnes_clashatdem = {
	"nes_clashatdem", NULL, NULL, NULL, "1990",
	"Clash at Demonhead (USA)\0", NULL, "Vic Tokai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_clashatdemRomInfo, nes_clashatdemRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cliffhangerRomDesc[] = {
	{ "Cliffhanger (USA).nes",          262160, 0x7d51a6dc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cliffhanger)
STD_ROM_FN(nes_cliffhanger)

struct BurnDriver BurnDrvnes_cliffhanger = {
	"nes_cliffhanger", NULL, NULL, NULL, "1993",
	"Cliffhanger (USA)\0", NULL, "Sony (CSG) Imagesoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cliffhangerRomInfo, nes_cliffhangerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cluclulandRomDesc[] = {
	{ "Clu Clu Land (World).nes",          24592, 0xc36a135a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_clucluland)
STD_ROM_FN(nes_clucluland)

struct BurnDriver BurnDrvnes_clucluland = {
	"nes_clucluland", NULL, NULL, NULL, "1984",
	"Clu Clu Land (World)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cluclulandRomInfo, nes_cluclulandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cobracommandRomDesc[] = {
	{ "Cobra Command (USA).nes",          262160, 0x40c6e687, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cobracommand)
STD_ROM_FN(nes_cobracommand)

struct BurnDriver BurnDrvnes_cobracommand = {
	"nes_cobracommand", NULL, NULL, NULL, "1988",
	"Cobra Command (USA)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cobracommandRomInfo, nes_cobracommandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cobramissionRomDesc[] = {
	{ "Cobra Mission (Asia).nes",          131088, 0x0f6bc06c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cobramission)
STD_ROM_FN(nes_cobramission)

struct BurnDriver BurnDrvnes_cobramission = {
	"nes_cobramission", NULL, NULL, NULL, "1994",
	"Cobra Mission (Asia)\0", NULL, "Asder", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cobramissionRomInfo, nes_cobramissionRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cobratriangleRomDesc[] = {
	{ "Cobra Triangle (USA).nes",          131088, 0x3802276d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cobratriangle)
STD_ROM_FN(nes_cobratriangle)

struct BurnDriver BurnDrvnes_cobratriangle = {
	"nes_cobratriangle", NULL, NULL, NULL, "1989",
	"Cobra Triangle (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cobratriangleRomInfo, nes_cobratriangleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cocoronjRomDesc[] = {
	{ "Cocoron (Japan).nes",          262160, 0x1d6b80da, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cocoronj)
STD_ROM_FN(nes_cocoronj)

struct BurnDriver BurnDrvnes_cocoronj = {
	"nes_cocoronj", "nes_cocoron", NULL, NULL, "1989?",
	"Cocoron (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cocoronjRomInfo, nes_cocoronjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cocoronRomDesc[] = {
	{ "Cocoron (T-eng).nes",          262160, 0xe486f7d3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cocoron)
STD_ROM_FN(nes_cocoron)

struct BurnDriver BurnDrvnes_cocoron = {
	"nes_cocoron", NULL, NULL, NULL, "1989?",
	"Cocoron (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cocoronRomInfo, nes_cocoronRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_codenamRomDesc[] = {
	{ "Code Name - Viper (USA).nes",          262160, 0xc8a23081, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_codenam)
STD_ROM_FN(nes_codenam)

struct BurnDriver BurnDrvnes_codenam = {
	"nes_codenam", NULL, NULL, NULL, "1990",
	"Code Name - Viper (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_codenamRomInfo, nes_codenamRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_coloradinosaurRomDesc[] = {
	{ "Color A Dinosaur (USA).nes",          131088, 0x322a6cca, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_coloradinosaur)
STD_ROM_FN(nes_coloradinosaur)

struct BurnDriver BurnDrvnes_coloradinosaur = {
	"nes_coloradinosaur", NULL, NULL, NULL, "1993",
	"Color A Dinosaur (USA)\0", NULL, "Virgin Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_coloradinosaurRomInfo, nes_coloradinosaurRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_commandoRomDesc[] = {
	{ "Commando (USA).nes",          131088, 0x41492df6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_commando)
STD_ROM_FN(nes_commando)

struct BurnDriver BurnDrvnes_commando = {
	"nes_commando", NULL, NULL, NULL, "1986",
	"Commando (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_commandoRomInfo, nes_commandoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_conanRomDesc[] = {
	{ "Conan (USA).nes",          131088, 0x05f76a57, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_conan)
STD_ROM_FN(nes_conan)

struct BurnDriver BurnDrvnes_conan = {
	"nes_conan", NULL, NULL, NULL, "1991",
	"Conan (USA)\0", NULL, "Mindscape", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_conanRomInfo, nes_conanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_conflictRomDesc[] = {
	{ "Conflict (USA).nes",          262160, 0x5f7c6229, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_conflict)
STD_ROM_FN(nes_conflict)

struct BurnDriver BurnDrvnes_conflict = {
	"nes_conflict", NULL, NULL, NULL, "1990",
	"Conflict (USA)\0", NULL, "Vic Tokai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_conflictRomInfo, nes_conflictRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_conqucrypalRomDesc[] = {
	{ "Conquest of the Crystal Palace (USA).nes",          262160, 0x0a362909, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_conqucrypal)
STD_ROM_FN(nes_conqucrypal)

struct BurnDriver BurnDrvnes_conqucrypal = {
	"nes_conqucrypal", NULL, NULL, NULL, "1990",
	"Conquest of the Crystal Palace (USA)\0", NULL, "Asmik", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_conqucrypalRomInfo, nes_conqucrypalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_contrajRomDesc[] = {
	{ "Contra (Japan).nes",          262160, 0x8a96a3c4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contraj)
STD_ROM_FN(nes_contraj)

struct BurnDriver BurnDrvnes_contraj = {
	"nes_contraj", "nes_contra", NULL, NULL, "1988",
	"Contra (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contrajRomInfo, nes_contrajRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_contratRomDesc[] = {
	{ "Contra (T-Eng).nes",          262160, 0xd93f2c45, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contrat)
STD_ROM_FN(nes_contrat)

struct BurnDriver BurnDrvnes_contrat = {
	"nes_contrat", "nes_contra", NULL, NULL, "1988",
	"Contra (T-Eng)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contratRomInfo, nes_contratRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_contraRomDesc[] = {
	{ "Contra (USA).nes",          131088, 0xc50a8304, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contra)
STD_ROM_FN(nes_contra)

struct BurnDriver BurnDrvnes_contra = {
	"nes_contra", NULL, NULL, NULL, "1988",
	"Contra (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contraRomInfo, nes_contraRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_contrredfalwarRomDesc[] = {
	{ "Contra - Red Falcon War (USA).nes",          262160, 0x59ef411a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contrredfalwar)
STD_ROM_FN(nes_contrredfalwar)

struct BurnDriver BurnDrvnes_contrredfalwar = {
	"nes_contrredfalwar", "nes_contra", NULL, NULL, "1989?",
	"Contra - Red Falcon War (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contrredfalwarRomInfo, nes_contrredfalwarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_contrrevredfalRomDesc[] = {
	{ "Contra - Revenge of the Red Falcon (USA).nes",          262160, 0xfa28193b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contrrevredfal)
STD_ROM_FN(nes_contrrevredfal)

struct BurnDriver BurnDrvnes_contrrevredfal = {
	"nes_contrrevredfal", "nes_contra", NULL, NULL, "1989?",
	"Contra - Revenge of the Red Falcon (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contrrevredfalRomInfo, nes_contrrevredfalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_contra168in1RomDesc[] = {
	{ "Contra 168-in-1 (Unl).nes",          1048592, 0xc6fd114e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contra168in1)
STD_ROM_FN(nes_contra168in1)

struct BurnDriver BurnDrvnes_contra168in1 = {
	"nes_contra168in1", NULL, NULL, NULL, "1989?",
	"Contra 168-in-1 (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contra168in1RomInfo, nes_contra168in1RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_contraforceRomDesc[] = {
	{ "Contra Force (USA).nes",          262160, 0x83d69922, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contraforce)
STD_ROM_FN(nes_contraforce)

struct BurnDriver BurnDrvnes_contraforce = {
	"nes_contraforce", NULL, NULL, NULL, "1992",
	"Contra Force (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contraforceRomInfo, nes_contraforceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_contraspiritsRomDesc[] = {
	{ "Contra Spirits (Unl).nes",          524304, 0xcea432be, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_contraspirits)
STD_ROM_FN(nes_contraspirits)

struct BurnDriver BurnDrvnes_contraspirits = {
	"nes_contraspirits", NULL, NULL, NULL, "1989?",
	"Contra Spirits (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_contraspiritsRomInfo, nes_contraspiritsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_coolworldRomDesc[] = {
	{ "Cool World (USA).nes",          262160, 0xba898162, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_coolworld)
STD_ROM_FN(nes_coolworld)

struct BurnDriver BurnDrvnes_coolworld = {
	"nes_coolworld", NULL, NULL, NULL, "1993",
	"Cool World (USA)\0", NULL, "Ocean", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_coolworldRomInfo, nes_coolworldRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_correbennyRomDesc[] = {
	{ "Corre Benny (Spain).nes",          65552, 0xc92325a0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_correbenny)
STD_ROM_FN(nes_correbenny)

struct BurnDriver BurnDrvnes_correbenny = {
	"nes_correbenny", NULL, NULL, NULL, "1992",
	"Corre Benny (Spain)\0", NULL, "Gluk Video", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_correbennyRomInfo, nes_correbennyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cosmicepsilonRomDesc[] = {
	{ "Cosmic Epsilon (Japan).nes",          393232, 0x0357095e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cosmicepsilon)
STD_ROM_FN(nes_cosmicepsilon)

struct BurnDriver BurnDrvnes_cosmicepsilon = {
	"nes_cosmicepsilon", NULL, NULL, NULL, "1989",
	"Cosmic Epsilon (Japan)\0", NULL, "Asmik", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cosmicepsilonRomInfo, nes_cosmicepsilonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cosmogenesisRomDesc[] = {
	{ "Cosmo Genesis (Japan).nes",          65552, 0xd5c8ee20, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cosmogenesis)
STD_ROM_FN(nes_cosmogenesis)

struct BurnDriver BurnDrvnes_cosmogenesis = {
	"nes_cosmogenesis", "nes_starvoyager", NULL, NULL, "1986",
	"Cosmo Genesis (Japan)\0", NULL, "ASCII", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cosmogenesisRomInfo, nes_cosmogenesisRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cosmopolgaljRomDesc[] = {
	{ "Cosmo Police Galivan (Japan).nes",          262160, 0xbccada80, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cosmopolgalj)
STD_ROM_FN(nes_cosmopolgalj)

struct BurnDriver BurnDrvnes_cosmopolgalj = {
	"nes_cosmopolgalj", "nes_cosmopolgal", NULL, NULL, "1989?",
	"Cosmo Police Galivan (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cosmopolgaljRomInfo, nes_cosmopolgaljRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cosmopolgalRomDesc[] = {
	{ "Cosmo Police Galivan (T-eng).nes",          262160, 0x4f031cb0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cosmopolgal)
STD_ROM_FN(nes_cosmopolgal)

struct BurnDriver BurnDrvnes_cosmopolgal = {
	"nes_cosmopolgal", NULL, NULL, NULL, "1989?",
	"Cosmo Police Galivan (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cosmopolgalRomInfo, nes_cosmopolgalRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cowboykidRomDesc[] = {
	{ "Cowboy Kid (USA).nes",          524304, 0xb150ae9a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cowboykid)
STD_ROM_FN(nes_cowboykid)

struct BurnDriver BurnDrvnes_cowboykid = {
	"nes_cowboykid", NULL, NULL, NULL, "1992",
	"Cowboy Kid (USA)\0", NULL, "Romstar", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cowboykidRomInfo, nes_cowboykidRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crashnboyRomDesc[] = {
	{ "Crash 'n' the Boys - Street Challenge (USA).nes",          262160, 0xed63ccc5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crashnboy)
STD_ROM_FN(nes_crashnboy)

struct BurnDriver BurnDrvnes_crashnboy = {
	"nes_crashnboy", NULL, NULL, NULL, "1992",
	"Crash 'n' the Boys - Street Challenge (USA)\0", NULL, "Technos", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crashnboyRomInfo, nes_crashnboyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crayoshichaRomDesc[] = {
	{ "Crayon Shin-chan - Ora to Poi Poi (Japan).nes",          262160, 0x564205d0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crayoshicha)
STD_ROM_FN(nes_crayoshicha)

struct BurnDriver BurnDrvnes_crayoshicha = {
	"nes_crayoshicha", NULL, NULL, NULL, "1993",
	"Crayon Shin-chan - Ora to Poi Poi (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crayoshichaRomInfo, nes_crayoshichaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crazyclimberRomDesc[] = {
	{ "Crazy Climber (Japan).nes",          131088, 0x83763841, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crazyclimber)
STD_ROM_FN(nes_crazyclimber)

struct BurnDriver BurnDrvnes_crazyclimber = {
	"nes_crazyclimber", NULL, NULL, NULL, "1986",
	"Crazy Climber (Japan)\0", NULL, "Nichibutsu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crazyclimberRomInfo, nes_crazyclimberRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crimebustersRomDesc[] = {
	{ "Crime Busters (USA).nes",          163856, 0x86c7d81e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crimebusters)
STD_ROM_FN(nes_crimebusters)

struct BurnDriver BurnDrvnes_crimebusters = {
	"nes_crimebusters", NULL, NULL, NULL, "1989",
	"Crime Busters (USA)\0", NULL, "Gradiente", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crimebustersRomInfo, nes_crimebustersRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crisisforcejRomDesc[] = {
	{ "Crisis Force (Japan).nes",          262160, 0xc4520781, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crisisforcej)
STD_ROM_FN(nes_crisisforcej)

struct BurnDriver BurnDrvnes_crisisforcej = {
	"nes_crisisforcej", "nes_crisisforce", NULL, NULL, "1991",
	"Crisis Force (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crisisforcejRomInfo, nes_crisisforcejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crisisforceRomDesc[] = {
	{ "Crisis Force (T-Eng).nes",          262160, 0x8a053054, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crisisforce)
STD_ROM_FN(nes_crisisforce)

struct BurnDriver BurnDrvnes_crisisforce = {
	"nes_crisisforce", NULL, NULL, NULL, "1991",
	"Crisis Force (T-Eng)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crisisforceRomInfo, nes_crisisforceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crossfireRomDesc[] = {
	{ "Cross Fire (Japan).nes",          262160, 0xc3c75c6e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crossfire)
STD_ROM_FN(nes_crossfire)

struct BurnDriver BurnDrvnes_crossfire = {
	"nes_crossfire", NULL, NULL, NULL, "1989?",
	"Cross Fire (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crossfireRomInfo, nes_crossfireRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_crystalisRomDesc[] = {
	{ "Crystalis (USA).nes",          393232, 0x1bd39032, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_crystalis)
STD_ROM_FN(nes_crystalis)

struct BurnDriver BurnDrvnes_crystalis = {
	"nes_crystalis", NULL, NULL, NULL, "1990",
	"Crystalis (USA)\0", NULL, "SNK", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_crystalisRomInfo, nes_crystalisRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cyberstaserRomDesc[] = {
	{ "Cyber Stadium Series - Base Wars (USA).nes",          262160, 0xda968c24, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cyberstaser)
STD_ROM_FN(nes_cyberstaser)

struct BurnDriver BurnDrvnes_cyberstaser = {
	"nes_cyberstaser", NULL, NULL, NULL, "1991",
	"Cyber Stadium Series - Base Wars (USA)\0", NULL, "Ultra Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cyberstaserRomInfo, nes_cyberstaserRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cyberRomDesc[] = {
	{ "Cybernoid - The Fighting Machine (USA).nes",          65552, 0xe3a6d7f6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cyber)
STD_ROM_FN(nes_cyber)

struct BurnDriver BurnDrvnes_cyber = {
	"nes_cyber", NULL, NULL, NULL, "1989",
	"Cybernoid - The Fighting Machine (USA)\0", NULL, "Acclaim", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cyberRomInfo, nes_cyberRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_cycleracRomDesc[] = {
	{ "Cycle Race - Road Man (Japan).nes",          262160, 0x899ad1fd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_cyclerac)
STD_ROM_FN(nes_cyclerac)

struct BurnDriver BurnDrvnes_cyclerac = {
	"nes_cyclerac", NULL, NULL, NULL, "1988",
	"Cycle Race - Road Man (Japan)\0", NULL, "Tokyo Shoseki", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_cycleracRomInfo, nes_cycleracRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dai2jisuprotajRomDesc[] = {
	{ "Dai-2-ji Super Robot Taisen (Japan).nes",          524304, 0x62c5d10c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dai2jisuprotaj)
STD_ROM_FN(nes_dai2jisuprotaj)

struct BurnDriver BurnDrvnes_dai2jisuprotaj = {
	"nes_dai2jisuprotaj", "nes_dai2jisuprota", NULL, NULL, "1991",
	"Dai-2-ji Super Robot Taisen (Japan)\0", NULL, "Banpresto", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dai2jisuprotajRomInfo, nes_dai2jisuprotajRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dai2jisuprotaRomDesc[] = {
	{ "Dai-2-ji Super Robot Taisen (T-eng).nes",          524304, 0x648c69fe, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dai2jisuprota)
STD_ROM_FN(nes_dai2jisuprota)

struct BurnDriver BurnDrvnes_dai2jisuprota = {
	"nes_dai2jisuprota", NULL, NULL, NULL, "1991",
	"Dai-2-ji Super Robot Taisen (T-eng)\0", NULL, "Banpresto", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dai2jisuprotaRomInfo, nes_dai2jisuprotaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_daikunogensanRomDesc[] = {
	{ "Daiku no Gen-san (Japan).nes",          262160, 0xa529d5f0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daikunogensan)
STD_ROM_FN(nes_daikunogensan)

struct BurnDriver BurnDrvnes_daikunogensan = {
	"nes_daikunogensan", NULL, NULL, NULL, "1991",
	"Daiku no Gen-san (Japan)\0", NULL, "Irem", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_daikunogensanRomInfo, nes_daikunogensanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_daikunogensan2RomDesc[] = {
	{ "Daiku no Gen-san 2 - Akage no Dan no Gyakushuu (Japan).nes",          524304, 0x3774121e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daikunogensan2)
STD_ROM_FN(nes_daikunogensan2)

struct BurnDriver BurnDrvnes_daikunogensan2 = {
	"nes_daikunogensan2", NULL, NULL, NULL, "1993",
	"Daiku no Gen-san 2 - Akage no Dan no Gyakushuu (Japan)\0", NULL, "Irem", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_daikunogensan2RomInfo, nes_daikunogensan2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dannysulsindheRomDesc[] = {
	{ "Danny Sullivan's Indy Heat (USA).nes",          131088, 0x311b5a58, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dannysulsindhe)
STD_ROM_FN(nes_dannysulsindhe)

struct BurnDriver BurnDrvnes_dannysulsindhe = {
	"nes_dannysulsindhe", NULL, NULL, NULL, "1992",
	"Danny Sullivan's Indy Heat (USA)\0", NULL, "Tradewest", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dannysulsindheRomInfo, nes_dannysulsindheRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_daoshuaiRomDesc[] = {
	{ "Dao Shuai (Asia) (Unl).nes",          65552, 0x00951874, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daoshuai)
STD_ROM_FN(nes_daoshuai)

struct BurnDriver BurnDrvnes_daoshuai = {
	"nes_daoshuai", NULL, NULL, NULL, "1989?",
	"Dao Shuai (Asia) (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_daoshuaiRomInfo, nes_daoshuaiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_darklordRomDesc[] = {
	{ "Dark Lord (Japan).nes",          393232, 0x1be8483a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_darklord)
STD_ROM_FN(nes_darklord)

struct BurnDriver BurnDrvnes_darklord = {
	"nes_darklord", NULL, NULL, NULL, "1989?",
	"Dark Lord (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_darklordRomInfo, nes_darklordRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doublmoodenjRomDesc[] = {
	{ "Double Moon Densetsu (Japan).nes",          524304, 0xe6ecf970, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doublmoodenj)
STD_ROM_FN(nes_doublmoodenj)

struct BurnDriver BurnDrvnes_doublmoodenj = {
	"nes_doublmoodenj", "nes_doublmooden", NULL, NULL, "1992",
	"Double Moon Densetsu (Japan)\0", NULL, "Masaya", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doublmoodenjRomInfo, nes_doublmoodenjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doublmoodenRomDesc[] = {
	{ "Double Moon Densetsu (T-Eng).nes",          524304, 0x2c2c43a4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doublmooden)
STD_ROM_FN(nes_doublmooden)

struct BurnDriver BurnDrvnes_doublmooden = {
	"nes_doublmooden", NULL, NULL, NULL, "1992",
	"Double Moon Densetsu (T-Eng)\0", NULL, "Masaya", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doublmoodenRomInfo, nes_doublmoodenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_jumpjumpRomDesc[] = {
	{ "Jump-Jump (Unl).nes",          40976, 0x3fe43784, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_jumpjump)
STD_ROM_FN(nes_jumpjump)

struct BurnDriver BurnDrvnes_jumpjump = {
	"nes_jumpjump", NULL, NULL, NULL, "1989?",
	"Jump-Jump (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_jumpjumpRomInfo, nes_jumpjumpRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_darkmanRomDesc[] = {
	{ "Darkman (USA).nes",          262160, 0x5438a0ac, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_darkman)
STD_ROM_FN(nes_darkman)

struct BurnDriver BurnDrvnes_darkman = {
	"nes_darkman", NULL, NULL, NULL, "1991",
	"Darkman (USA)\0", NULL, "Ocean", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_darkmanRomInfo, nes_darkmanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_darkwingduckRomDesc[] = {
	{ "Darkwing Duck (USA).nes",          262160, 0xc7e6cc19, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_darkwingduck)
STD_ROM_FN(nes_darkwingduck)

struct BurnDriver BurnDrvnes_darkwingduck = {
	"nes_darkwingduck", NULL, NULL, NULL, "1992",
	"Darkwing Duck (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_darkwingduckRomInfo, nes_darkwingduckRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dashgalinaliasRomDesc[] = {
	{ "Dash Galaxy in the Alien Asylum (USA).nes",          65552, 0x28aa07ba, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dashgalinalias)
STD_ROM_FN(nes_dashgalinalias)

struct BurnDriver BurnDrvnes_dashgalinalias = {
	"nes_dashgalinalias", NULL, NULL, NULL, "1990",
	"Dash Galaxy in the Alien Asylum (USA)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dashgalinaliasRomInfo, nes_dashgalinaliasRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dashyarouRomDesc[] = {
	{ "Dash Yarou (Japan).nes",          131088, 0x2ddd5f68, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dashyarou)
STD_ROM_FN(nes_dashyarou)

struct BurnDriver BurnDrvnes_dashyarou = {
	"nes_dashyarou", NULL, NULL, NULL, "1990",
	"Dash Yarou (Japan)\0", NULL, "Visco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dashyarouRomInfo, nes_dashyarouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_datsugokuRomDesc[] = {
	{ "Datsugoku (Japan).nes",          262160, 0x5bb9899a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_datsugoku)
STD_ROM_FN(nes_datsugoku)

struct BurnDriver BurnDrvnes_datsugoku = {
	"nes_datsugoku", "nes_pow", NULL, NULL, "1989",
	"Datsugoku (Japan)\0", NULL, "KAC", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_datsugokuRomInfo, nes_datsugokuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_davidcrasaboyandhisRomDesc[] = {
	{ "David Crane's A Boy and His Blob - Trouble on Blobolonia (USA).nes",          262160, 0x20a9e4a2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_davidcrasaboyandhis)
STD_ROM_FN(nes_davidcrasaboyandhis)

struct BurnDriver BurnDrvnes_davidcrasaboyandhis = {
	"nes_davidcrasaboyandhis", NULL, NULL, NULL, "1990",
	"David Crane's A Boy and His Blob - Trouble on Blobolonia (USA)\0", NULL, "Absolute", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_davidcrasaboyandhisRomInfo, nes_davidcrasaboyandhisRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_daydredavRomDesc[] = {
	{ "Day Dreamin' Davey (USA).nes",          393232, 0xced13971, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daydredav)
STD_ROM_FN(nes_daydredav)

struct BurnDriver BurnDrvnes_daydredav = {
	"nes_daydredav", NULL, NULL, NULL, "1992",
	"Day Dreamin' Davey (USA)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_daydredavRomInfo, nes_daydredavRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_daysofthunderRomDesc[] = {
	{ "Days of Thunder (USA).nes",          393232, 0x3616c7dd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_daysofthunder)
STD_ROM_FN(nes_daysofthunder)

struct BurnDriver BurnDrvnes_daysofthunder = {
	"nes_daysofthunder", NULL, NULL, NULL, "1990",
	"Days of Thunder (USA)\0", NULL, "Mindscape", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_daysofthunderRomInfo, nes_daysofthunderRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_deadfoxRomDesc[] = {
	{ "Dead Fox (Japan).nes",          262160, 0xf316b497, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deadfox)
STD_ROM_FN(nes_deadfox)

struct BurnDriver BurnDrvnes_deadfox = {
	"nes_deadfox", "nes_codenam", NULL, NULL, "1989?",
	"Dead Fox (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_deadfoxRomInfo, nes_deadfoxRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_deadlytowersRomDesc[] = {
	{ "Deadly Towers (USA).nes",          131088, 0x25225c70, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deadlytowers)
STD_ROM_FN(nes_deadlytowers)

struct BurnDriver BurnDrvnes_deadlytowers = {
	"nes_deadlytowers", NULL, NULL, NULL, "1987",
	"Deadly Towers (USA)\0", NULL, "Broderbund", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_deadlytowersRomInfo, nes_deadlytowersRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_deathraceRomDesc[] = {
	{ "Death Race (USA) (Unl).nes",          131088, 0x837c5171, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deathrace)
STD_ROM_FN(nes_deathrace)

struct BurnDriver BurnDrvnes_deathrace = {
	"nes_deathrace", NULL, NULL, NULL, "1990",
	"Death Race (USA) (Unl)\0", NULL, "AGCI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_deathraceRomInfo, nes_deathraceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_deathbotsRomDesc[] = {
	{ "Deathbots (USA) (Rev 1) (Unl).nes",          131088, 0xcd12d2be, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deathbots)
STD_ROM_FN(nes_deathbots)

struct BurnDriver BurnDrvnes_deathbots = {
	"nes_deathbots", NULL, NULL, NULL, "1990",
	"Deathbots (USA) (Rev 1) (Unl)\0", NULL, "AVE", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_deathbotsRomInfo, nes_deathbotsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_deblockRomDesc[] = {
	{ "Deblock (Japan).nes",          65552, 0x70eac605, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_deblock)
STD_ROM_FN(nes_deblock)

struct BurnDriver BurnDrvnes_deblock = {
	"nes_deblock", NULL, NULL, NULL, "1989?",
	"Deblock (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_deblockRomInfo, nes_deblockRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_defenderiiRomDesc[] = {
	{ "Defender II (USA).nes",          24592, 0xc0b2aa1f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_defenderii)
STD_ROM_FN(nes_defenderii)

struct BurnDriver BurnDrvnes_defenderii = {
	"nes_defenderii", NULL, NULL, NULL, "1988",
	"Defender II (USA)\0", NULL, "HAL Laboratory", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_defenderiiRomInfo, nes_defenderiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_defencroRomDesc[] = {
	{ "Defender of the Crown (USA).nes",          262160, 0x79123682, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_defencro)
STD_ROM_FN(nes_defencro)

struct BurnDriver BurnDrvnes_defencro = {
	"nes_defencro", NULL, NULL, NULL, "1989",
	"Defender of the Crown (USA)\0", NULL, "Ultra Games", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_defencroRomInfo, nes_defencroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_defendyncitRomDesc[] = {
	{ "Defenders of Dynatron City (USA).nes",          262160, 0x8db6ba41, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_defendyncit)
STD_ROM_FN(nes_defendyncit)

struct BurnDriver BurnDrvnes_defendyncit = {
	"nes_defendyncit", NULL, NULL, NULL, "1992",
	"Defenders of Dynatron City (USA)\0", NULL, "JVC", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_defendyncitRomInfo, nes_defendyncitRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dejavuRomDesc[] = {
	{ "Deja Vu (USA).nes",          393232, 0x67b25db5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dejavu)
STD_ROM_FN(nes_dejavu)

struct BurnDriver BurnDrvnes_dejavu = {
	"nes_dejavu", NULL, NULL, NULL, "1990",
	"Deja Vu (USA)\0", NULL, "Kemco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dejavuRomInfo, nes_dejavuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_demonswordRomDesc[] = {
	{ "Demon Sword (USA).nes",          262160, 0x2b324834, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_demonsword)
STD_ROM_FN(nes_demonsword)

struct BurnDriver BurnDrvnes_demonsword = {
	"nes_demonsword", NULL, NULL, NULL, "1990",
	"Demon Sword (USA)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_demonswordRomInfo, nes_demonswordRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dengekiRomDesc[] = {
	{ "Dengeki - Big Bang! (Japan).nes",          262160, 0x4e0c2429, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dengeki)
STD_ROM_FN(nes_dengeki)

struct BurnDriver BurnDrvnes_dengeki = {
	"nes_dengeki", "nes_clashatdem", NULL, NULL, "1989?",
	"Dengeki - Big Bang! (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dengekiRomInfo, nes_dengekiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_destiearRomDesc[] = {
	{ "Destination Earthstar (USA).nes",          65552, 0xa4b27af2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_destiear)
STD_ROM_FN(nes_destiear)

struct BurnDriver BurnDrvnes_destiear = {
	"nes_destiear", NULL, NULL, NULL, "1990",
	"Destination Earthstar (USA)\0", NULL, "Acclaim", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_destiearRomInfo, nes_destiearRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_destianempRomDesc[] = {
	{ "Destiny of an Emperor (USA).nes",          262160, 0xc0f73d85, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_destianemp)
STD_ROM_FN(nes_destianemp)

struct BurnDriver BurnDrvnes_destianemp = {
	"nes_destianemp", NULL, NULL, NULL, "1990",
	"Destiny of an Emperor (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_destianempRomInfo, nes_destianempRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_destructorelRomDesc[] = {
	{ "Destructor, El (Spain) (Gluk Video) (Unl).nes",          65552, 0xeb33ea3b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_destructorel)
STD_ROM_FN(nes_destructorel)

struct BurnDriver BurnDrvnes_destructorel = {
	"nes_destructorel", NULL, NULL, NULL, "1992",
	"Destructor, El (Spain) (Gluk Video) (Unl)\0", NULL, "Gluk Video", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_destructorelRomInfo, nes_destructorelRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_devilmanjRomDesc[] = {
	{ "Devil Man (Japan).nes",          262160, 0xc4c1ab20, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_devilmanj)
STD_ROM_FN(nes_devilmanj)

struct BurnDriver BurnDrvnes_devilmanj = {
	"nes_devilmanj", "nes_devilman", NULL, NULL, "1989",
	"Devil Man (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_devilmanjRomInfo, nes_devilmanjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_devilmanRomDesc[] = {
	{ "Devil Man (T-eng).nes",          262160, 0x0be5f3e3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_devilman)
STD_ROM_FN(nes_devilman)

struct BurnDriver BurnDrvnes_devilman = {
	"nes_devilman", NULL, NULL, NULL, "1989",
	"Devil Man (T-eng)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_devilmanRomInfo, nes_devilmanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_devilworldRomDesc[] = {
	{ "Devil World (Japan) (Rev A).nes",          24592, 0x9fc60476, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_devilworld)
STD_ROM_FN(nes_devilworld)

struct BurnDriver BurnDrvnes_devilworld = {
	"nes_devilworld", NULL, NULL, NULL, "1984",
	"Devil World (Japan) (Rev A)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_devilworldRomInfo, nes_devilworldRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_diabloiiRomDesc[] = {
	{ "Diablo II (Hack).nes",          2097168, 0xb870d5cd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_diabloii)
STD_ROM_FN(nes_diabloii)

struct BurnDriver BurnDrvnes_diabloii = {
	"nes_diabloii", NULL, NULL, NULL, "1989?",
	"Diablo II (Hack)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_diabloiiRomInfo, nes_diabloiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_diehardRomDesc[] = {
	{ "Die Hard (USA).nes",          262160, 0x65eec6e7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_diehard)
STD_ROM_FN(nes_diehard)

struct BurnDriver BurnDrvnes_diehard = {
	"nes_diehard", NULL, NULL, NULL, "1992",
	"Die Hard (USA)\0", NULL, "Activision", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_diehardRomInfo, nes_diehardRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_digdugRomDesc[] = {
	{ "Dig Dug (Japan).nes",          24592, 0x3ae8289a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_digdug)
STD_ROM_FN(nes_digdug)

struct BurnDriver BurnDrvnes_digdug = {
	"nes_digdug", NULL, NULL, NULL, "1985",
	"Dig Dug (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_digdugRomInfo, nes_digdugRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_digdugiiRomDesc[] = {
	{ "Dig Dug II (Japan).nes",          40976, 0xcc904e37, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_digdugii)
STD_ROM_FN(nes_digdugii)

struct BurnDriver BurnDrvnes_digdugii = {
	"nes_digdugii", NULL, NULL, NULL, "1986",
	"Dig Dug II (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_digdugiiRomInfo, nes_digdugiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_digdugiitroinpaRomDesc[] = {
	{ "Dig Dug II - Trouble in Paradise (USA).nes",          40976, 0x3cc270fb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_digdugiitroinpa)
STD_ROM_FN(nes_digdugiitroinpa)

struct BurnDriver BurnDrvnes_digdugiitroinpa = {
	"nes_digdugiitroinpa", NULL, NULL, NULL, "1989",
	"Dig Dug II - Trouble in Paradise (USA)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_digdugiitroinpaRomInfo, nes_digdugiitroinpaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_diggerRomDesc[] = {
	{ "Digger - The Legend of the Lost City (USA).nes",          131088, 0xec41647e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_digger)
STD_ROM_FN(nes_digger)

struct BurnDriver BurnDrvnes_digger = {
	"nes_digger", NULL, NULL, NULL, "1990",
	"Digger - The Legend of the Lost City (USA)\0", NULL, "Milton Bradley", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_diggerRomInfo, nes_diggerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_digitdevmonmegtejRomDesc[] = {
	{ "Digital Devil Monogatari - Megami Tensei (Japan).nes",          262160, 0xf906acab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_digitdevmonmegtej)
STD_ROM_FN(nes_digitdevmonmegtej)

struct BurnDriver BurnDrvnes_digitdevmonmegtej = {
	"nes_digitdevmonmegtej", "nes_digitdevmonmegte", NULL, NULL, "1987",
	"Digital Devil Monogatari - Megami Tensei (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_digitdevmonmegtejRomInfo, nes_digitdevmonmegtejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_digitdevmonmegteRomDesc[] = {
	{ "Digital Devil Monogatari - Megami Tensei (T-eng).nes",          393232, 0xac5acc89, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_digitdevmonmegte)
STD_ROM_FN(nes_digitdevmonmegte)

struct BurnDriver BurnDrvnes_digitdevmonmegte = {
	"nes_digitdevmonmegte", NULL, NULL, NULL, "1989?",
	"Digital Devil Monogatari - Megami Tensei (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_digitdevmonmegteRomInfo, nes_digitdevmonmegteRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_digitdevmonmegteiijRomDesc[] = {
	{ "Digital Devil Monogatari - Megami Tensei II (Japan).nes",          524304, 0xf933d37d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_digitdevmonmegteiij)
STD_ROM_FN(nes_digitdevmonmegteiij)

struct BurnDriver BurnDrvnes_digitdevmonmegteiij = {
	"nes_digitdevmonmegteiij", "nes_digitdevmonmegteii", NULL, NULL, "1990",
	"Digital Devil Monogatari - Megami Tensei II (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_digitdevmonmegteiijRomInfo, nes_digitdevmonmegteiijRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_digitdevmonmegteiiRomDesc[] = {
	{ "Digital Devil Monogatari - Megami Tensei II (T-eng).nes",          524304, 0x31d915cd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_digitdevmonmegteii)
STD_ROM_FN(nes_digitdevmonmegteii)

struct BurnDriver BurnDrvnes_digitdevmonmegteii = {
	"nes_digitdevmonmegteii", NULL, NULL, NULL, "1989?",
	"Digital Devil Monogatari - Megami Tensei II (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_digitdevmonmegteiiRomInfo, nes_digitdevmonmegteiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dirtyharRomDesc[] = {
	{ "Dirty Harry - The War Against Drugs (USA).nes",          393232, 0x293952bd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dirtyhar)
STD_ROM_FN(nes_dirtyhar)

struct BurnDriver BurnDrvnes_dirtyhar = {
	"nes_dirtyhar", NULL, NULL, NULL, "1989?",
	"Dirty Harry - The War Against Drugs (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dirtyharRomInfo, nes_dirtyharRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_divemanRomDesc[] = {
	{ "Dive Man (Unl).nes",          262160, 0xb12bfcc2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_diveman)
STD_ROM_FN(nes_diveman)

struct BurnDriver BurnDrvnes_diveman = {
	"nes_diveman", NULL, NULL, NULL, "1989?",
	"Dive Man (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_divemanRomInfo, nes_divemanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dizzyadvRomDesc[] = {
	{ "Dizzy The Adventurer (USA).nes",          131088, 0x92847456, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dizzyadv)
STD_ROM_FN(nes_dizzyadv)

struct BurnDriver BurnDrvnes_dizzyadv = {
	"nes_dizzyadv", NULL, NULL, NULL, "1992",
	"Dizzy The Adventurer (USA)\0", NULL, "Camerica", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dizzyadvRomInfo, nes_dizzyadvRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dokidokyuujRomDesc[] = {
	{ "Doki!Doki! Yuuenchi - Crazy Land Daisakusen (Japan).nes",          262160, 0x6058397f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dokidokyuuj)
STD_ROM_FN(nes_dokidokyuuj)

struct BurnDriver BurnDrvnes_dokidokyuuj = {
	"nes_dokidokyuuj", "nes_dokidokyuu", NULL, NULL, "1991",
	"Doki!Doki! Yuuenchi - Crazy Land Daisakusen (Japan)\0", NULL, "Vap", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dokidokyuujRomInfo, nes_dokidokyuujRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dokidokyuuRomDesc[] = {
	{ "Doki!Doki! Yuuenchi - Crazy Land Daisakusen (T-Eng).nes",          262160, 0x9aad9b7a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dokidokyuu)
STD_ROM_FN(nes_dokidokyuu)

struct BurnDriver BurnDrvnes_dokidokyuu = {
	"nes_dokidokyuu", NULL, NULL, NULL, "1989?",
	"Doki!Doki! Yuuenchi - Crazy Land Daisakusen (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dokidokyuuRomInfo, nes_dokidokyuuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dondokodonRomDesc[] = {
	{ "Don Doko Don (Japan).nes",          393232, 0x6db7e711, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dondokodon)
STD_ROM_FN(nes_dondokodon)

struct BurnDriver BurnDrvnes_dondokodon = {
	"nes_dondokodon", NULL, NULL, NULL, "1990",
	"Don Doko Don (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dondokodonRomInfo, nes_dondokodonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dondokodon2jRomDesc[] = {
	{ "Don Doko Don 2 (Japan).nes",          393232, 0xf583796b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dondokodon2j)
STD_ROM_FN(nes_dondokodon2j)

struct BurnDriver BurnDrvnes_dondokodon2j = {
	"nes_dondokodon2j", "nes_dondokodon2", NULL, NULL, "1992",
	"Don Doko Don 2 (Japan)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dondokodon2jRomInfo, nes_dondokodon2jRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dondokodon2RomDesc[] = {
	{ "Don Doko Don 2 (T-Eng).nes",          393232, 0x560142bc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dondokodon2)
STD_ROM_FN(nes_dondokodon2)

struct BurnDriver BurnDrvnes_dondokodon2 = {
	"nes_dondokodon2", NULL, NULL, NULL, "1992",
	"Don Doko Don 2 (T-Eng)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dondokodon2RomInfo, nes_dondokodon2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_donaldduckRomDesc[] = {
	{ "Donald Duck (Japan).nes",          262160, 0x8bd8fce1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_donaldduck)
STD_ROM_FN(nes_donaldduck)

struct BurnDriver BurnDrvnes_donaldduck = {
	"nes_donaldduck", NULL, NULL, NULL, "1988",
	"Donald Duck (Japan)\0", NULL, "Kemco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_donaldduckRomInfo, nes_donaldduckRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_donaldlandRomDesc[] = {
	{ "Donald Land (Japan).nes",          262160, 0x4b539429, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_donaldland)
STD_ROM_FN(nes_donaldland)

struct BurnDriver BurnDrvnes_donaldland = {
	"nes_donaldland", NULL, NULL, NULL, "1988",
	"Donald Land (Japan)\0", NULL, "Data East", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_donaldlandRomInfo, nes_donaldlandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_donkeykongRomDesc[] = {
	{ "Donkey Kong (World) (Rev A).nes",          24592, 0xe40b593b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_donkeykong)
STD_ROM_FN(nes_donkeykong)

struct BurnDriver BurnDrvnes_donkeykong = {
	"nes_donkeykong", NULL, NULL, NULL, "1983",
	"Donkey Kong (World) (Rev A)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_donkeykongRomInfo, nes_donkeykongRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_donkeykong3RomDesc[] = {
	{ "Donkey Kong 3 (USA).nes",          24592, 0xd1cac3c2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_donkeykong3)
STD_ROM_FN(nes_donkeykong3)

struct BurnDriver BurnDrvnes_donkeykong3 = {
	"nes_donkeykong3", NULL, NULL, NULL, "1984",
	"Donkey Kong 3 (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_donkeykong3RomInfo, nes_donkeykong3RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_donkekoncou4RomDesc[] = {
	{ "Donkey Kong Country 4 (Unl).nes",          524304, 0x5ae55685, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_donkekoncou4)
STD_ROM_FN(nes_donkekoncou4)

struct BurnDriver BurnDrvnes_donkekoncou4 = {
	"nes_donkekoncou4", NULL, NULL, NULL, "1989?",
	"Donkey Kong Country 4 (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_donkekoncou4RomInfo, nes_donkekoncou4RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_donkeykongjrRomDesc[] = {
	{ "Donkey Kong Jr. (World) (Rev A).nes",          24592, 0x2a794ccb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_donkeykongjr)
STD_ROM_FN(nes_donkeykongjr)

struct BurnDriver BurnDrvnes_donkeykongjr = {
	"nes_donkeykongjr", NULL, NULL, NULL, "1983",
	"Donkey Kong Jr. (World) (Rev A)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_donkeykongjrRomInfo, nes_donkeykongjrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doolybravolandRomDesc[] = {
	{ "Dooly Bravo Land (Korea).nes",          262160, 0xe7740e47, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doolybravoland)
STD_ROM_FN(nes_doolybravoland)

struct BurnDriver BurnDrvnes_doolybravoland = {
	"nes_doolybravoland", NULL, NULL, NULL, "1989?",
	"Dooly Bravo Land (Korea)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doolybravolandRomInfo, nes_doolybravolandRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doordoorRomDesc[] = {
	{ "Door Door (Japan).nes",          24592, 0x70fb44b6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doordoor)
STD_ROM_FN(nes_doordoor)

struct BurnDriver BurnDrvnes_doordoor = {
	"nes_doordoor", NULL, NULL, NULL, "1985",
	"Door Door (Japan)\0", NULL, "Enix", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doordoorRomInfo, nes_doordoorRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doraemonRomDesc[] = {
	{ "Doraemon (Japan) (Rev A).nes",          163856, 0x2768309d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doraemon)
STD_ROM_FN(nes_doraemon)

struct BurnDriver BurnDrvnes_doraemon = {
	"nes_doraemon", NULL, NULL, NULL, "1986",
	"Doraemon (Japan) (Rev A)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doraemonRomInfo, nes_doraemonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doraejRomDesc[] = {
	{ "Doraemon - Giga Zombie no Gyakushuu (Japan).nes",          262160, 0xb28e9ea4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doraej)
STD_ROM_FN(nes_doraej)

struct BurnDriver BurnDrvnes_doraej = {
	"nes_doraej", "nes_dorae", NULL, NULL, "1990",
	"Doraemon - Giga Zombie no Gyakushuu (Japan)\0", NULL, "Epoch", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doraejRomInfo, nes_doraejRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doraeRomDesc[] = {
	{ "Doraemon - Giga Zombie no Gyakushuu (T-eng).nes",          262160, 0xa9c510ac, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dorae)
STD_ROM_FN(nes_dorae)

struct BurnDriver BurnDrvnes_dorae = {
	"nes_dorae", NULL, NULL, NULL, "1989?",
	"Doraemon - Giga Zombie no Gyakushuu (T-eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doraeRomInfo, nes_doraeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doubledareRomDesc[] = {
	{ "Double Dare (USA).nes",          262160, 0xb631f92a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubledare)
STD_ROM_FN(nes_doubledare)

struct BurnDriver BurnDrvnes_doubledare = {
	"nes_doubledare", NULL, NULL, NULL, "1990",
	"Double Dare (USA)\0", NULL, "GameTek", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doubledareRomInfo, nes_doubledareRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doubledragonRomDesc[] = {
	{ "Double Dragon (USA).nes",          262160, 0x62afe166, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubledragon)
STD_ROM_FN(nes_doubledragon)

struct BurnDriver BurnDrvnes_doubledragon = {
	"nes_doubledragon", NULL, NULL, NULL, "1988",
	"Double Dragon (USA)\0", NULL, "Technos", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doubledragonRomInfo, nes_doubledragonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doubldraiiRomDesc[] = {
	{ "Double Dragon II - The Revenge (USA).nes",          262160, 0xcecfbf92, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubldraii)
STD_ROM_FN(nes_doubldraii)

struct BurnDriver BurnDrvnes_doubldraii = {
	"nes_doubldraii", NULL, NULL, NULL, "1990",
	"Double Dragon II - The Revenge (USA)\0", NULL, "Acclaim", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doubldraiiRomInfo, nes_doubldraiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doubldraiiiRomDesc[] = {
	{ "Double Dragon III - The Sacred Stones (USA).nes",          262160, 0x7a6e0454, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubldraiii)
STD_ROM_FN(nes_doubldraiii)

struct BurnDriver BurnDrvnes_doubldraiii = {
	"nes_doubldraiii", NULL, NULL, NULL, "1991",
	"Double Dragon III - The Sacred Stones (USA)\0", NULL, "Acclaim", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doubldraiiiRomInfo, nes_doubldraiiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doubledribbleRomDesc[] = {
	{ "Double Dribble (USA).nes",          131088, 0x7077a85d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doubledribble)
STD_ROM_FN(nes_doubledribble)

struct BurnDriver BurnDrvnes_doubledribble = {
	"nes_doubledribble", NULL, NULL, NULL, "1987",
	"Double Dribble (USA)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doubledribbleRomInfo, nes_doubledribbleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doublestrikeRomDesc[] = {
	{ "Double Strike (Unl).nes",          65552, 0xf4bb370a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doublestrike)
STD_ROM_FN(nes_doublestrike)

struct BurnDriver BurnDrvnes_doublestrike = {
	"nes_doublestrike", NULL, NULL, NULL, "1990",
	"Double Strike (Unl)\0", NULL, "AVE", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doublestrikeRomInfo, nes_doublestrikeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_doughboyRomDesc[] = {
	{ "Dough Boy (Japan).nes",          40976, 0x1012eb15, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_doughboy)
STD_ROM_FN(nes_doughboy)

struct BurnDriver BurnDrvnes_doughboy = {
	"nes_doughboy", NULL, NULL, NULL, "1985",
	"Dough Boy (Japan)\0", NULL, "Kemco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_doughboyRomInfo, nes_doughboyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_downtnekkousordaRomDesc[] = {
	{ "Downtown - Nekketsu Koushinkyoku - Soreyuke Daiundoukai (Japan).nes",          262160, 0xe34fc4af, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_downtnekkousorda)
STD_ROM_FN(nes_downtnekkousorda)

struct BurnDriver BurnDrvnes_downtnekkousorda = {
	"nes_downtnekkousorda", NULL, NULL, NULL, "1990",
	"Downtown - Nekketsu Koushinkyoku - Soreyuke Daiundoukai (Japan)\0", NULL, "Technos", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_downtnekkousordaRomInfo, nes_downtnekkousordaRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NESHori4pInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_downtnekmonRomDesc[] = {
	{ "Downtown - Nekketsu Monogatari (Japan).nes",          262160, 0xf7346e46, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_downtnekmon)
STD_ROM_FN(nes_downtnekmon)

struct BurnDriver BurnDrvnes_downtnekmon = {
	"nes_downtnekmon", NULL, NULL, NULL, "1989?",
	"Downtown - Nekketsu Monogatari (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_downtnekmonRomInfo, nes_downtnekmonRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_downtspekunkunnojidRomDesc[] = {
	{ "Downtown Special - Kunio-kun no Jidaigeki Da yo Zenin Shuugou! (Japan).nes",          262160, 0x014d63c9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_downtspekunkunnojid)
STD_ROM_FN(nes_downtspekunkunnojid)

struct BurnDriver BurnDrvnes_downtspekunkunnojid = {
	"nes_downtspekunkunnojid", NULL, NULL, NULL, "1991",
	"Downtown Special - Kunio-kun no Jidaigeki Da yo Zenin Shuugou! (Japan)\0", NULL, "Technos", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_downtspekunkunnojidRomInfo, nes_downtspekunkunnojidRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_drchaosRomDesc[] = {
	{ "Dr. Chaos (USA).nes",          131088, 0x406bda35, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_drchaos)
STD_ROM_FN(nes_drchaos)

struct BurnDriver BurnDrvnes_drchaos = {
	"nes_drchaos", NULL, NULL, NULL, "1988",
	"Dr. Chaos (USA)\0", NULL, "FCI", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_drchaosRomInfo, nes_drchaosRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_drjekandmrhyRomDesc[] = {
	{ "Dr. Jekyll and Mr. Hyde (USA).nes",          163856, 0xc006a41b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_drjekandmrhy)
STD_ROM_FN(nes_drjekandmrhy)

struct BurnDriver BurnDrvnes_drjekandmrhy = {
	"nes_drjekandmrhy", NULL, NULL, NULL, "1989",
	"Dr. Jekyll and Mr. Hyde (USA)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_drjekandmrhyRomInfo, nes_drjekandmrhyRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_drmarioRomDesc[] = {
	{ "Dr. Mario (Japan, USA) (Rev A).nes",          65552, 0x7623dffd, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_drmario)
STD_ROM_FN(nes_drmario)

struct BurnDriver BurnDrvnes_drmario = {
	"nes_drmario", NULL, NULL, NULL, "1990",
	"Dr. Mario (Japan, USA) (Rev A)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_drmarioRomInfo, nes_drmarioRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dracsnightoutRomDesc[] = {
	{ "Drac's Night Out (USA) (Prototype).nes",          262160, 0x2f81e727, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dracsnightout)
STD_ROM_FN(nes_dracsnightout)

struct BurnDriver BurnDrvnes_dracsnightout = {
	"nes_dracsnightout", NULL, NULL, NULL, "1989?",
	"Drac's Night Out (USA) (Prototype)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dracsnightoutRomInfo, nes_dracsnightoutRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobaldaimaofujRomDesc[] = {
	{ "Dragon Ball - Dai Maou Fukkatsu (Japan).nes",          262160, 0xbeff8c77, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobaldaimaofuj)
STD_ROM_FN(nes_dragobaldaimaofuj)

struct BurnDriver BurnDrvnes_dragobaldaimaofuj = {
	"nes_dragobaldaimaofuj", "nes_dragobaldaimaofu", NULL, NULL, "1988",
	"Dragon Ball - Dai Maou Fukkatsu (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobaldaimaofujRomInfo, nes_dragobaldaimaofujRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobaldaimaofuRomDesc[] = {
	{ "Dragon Ball - Dai Maou Fukkatsu (T-Eng).nes",          393232, 0x69beb7a5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobaldaimaofu)
STD_ROM_FN(nes_dragobaldaimaofu)

struct BurnDriver BurnDrvnes_dragobaldaimaofu = {
	"nes_dragobaldaimaofu", NULL, NULL, NULL, "1988",
	"Dragon Ball - Dai Maou Fukkatsu (T-Eng)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobaldaimaofuRomInfo, nes_dragobaldaimaofuRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalshelonnonajRomDesc[] = {
	{ "Dragon Ball - Shen Long no Nazo (Japan).nes",          163856, 0x815a4bad, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalshelonnonaj)
STD_ROM_FN(nes_dragobalshelonnonaj)

struct BurnDriver BurnDrvnes_dragobalshelonnonaj = {
	"nes_dragobalshelonnonaj", "nes_dragonpower", NULL, NULL, "1986",
	"Dragon Ball - Shen Long no Nazo (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalshelonnonajRomInfo, nes_dragobalshelonnonajRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalshelonnonaRomDesc[] = {
	{ "Dragon Ball - Shen Long no Nazo (T-Eng).nes",          163856, 0xb4e31bf4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalshelonnona)
STD_ROM_FN(nes_dragobalshelonnona)

struct BurnDriver BurnDrvnes_dragobalshelonnona = {
	"nes_dragobalshelonnona", "nes_dragonpower", NULL, NULL, "1986",
	"Dragon Ball - Shen Long no Nazo (T-Eng)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalshelonnonaRomInfo, nes_dragobalshelonnonaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobal3gokdenRomDesc[] = {
	{ "Dragon Ball 3 - Gokuu Den (Japan).nes",          393232, 0x0544b2a4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobal3gokden)
STD_ROM_FN(nes_dragobal3gokden)

struct BurnDriver BurnDrvnes_dragobal3gokden = {
	"nes_dragobal3gokden", NULL, NULL, NULL, "1989",
	"Dragon Ball 3 - Gokuu Den (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobal3gokdenRomInfo, nes_dragobal3gokdenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalzkyosajinjRomDesc[] = {
	{ "Dragon Ball Z - Kyoushuu! Saiya Jin (Japan).nes",          524304, 0xd78484b7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalzkyosajinj)
STD_ROM_FN(nes_dragobalzkyosajinj)

struct BurnDriver BurnDrvnes_dragobalzkyosajinj = {
	"nes_dragobalzkyosajinj", "nes_dragobalzkyosajin", NULL, NULL, "1990",
	"Dragon Ball Z - Kyoushuu! Saiya Jin (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalzkyosajinjRomInfo, nes_dragobalzkyosajinjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalzkyosajinRomDesc[] = {
	{ "Dragon Ball Z - Kyoushuu! Saiya Jin (T-Eng).nes",          524304, 0x31d0d50a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalzkyosajin)
STD_ROM_FN(nes_dragobalzkyosajin)

struct BurnDriver BurnDrvnes_dragobalzkyosajin = {
	"nes_dragobalzkyosajin", NULL, NULL, NULL, "1989?",
	"Dragon Ball Z - Kyoushuu! Saiya Jin (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalzkyosajinRomInfo, nes_dragobalzkyosajinRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalzsupbu2jRomDesc[] = {
	{ "Dragon Ball Z - Super Butouden 2 (Japan).nes",          393232, 0x2490d360, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalzsupbu2j)
STD_ROM_FN(nes_dragobalzsupbu2j)

struct BurnDriver BurnDrvnes_dragobalzsupbu2j = {
	"nes_dragobalzsupbu2j", "nes_dragobalzsupbu2", NULL, NULL, "1989?",
	"Dragon Ball Z - Super Butouden 2 (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalzsupbu2jRomInfo, nes_dragobalzsupbu2jRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalzsupbu2RomDesc[] = {
	{ "Dragon Ball Z - Super Butouden 2 (T-Eng).nes",          393232, 0xdf87586f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalzsupbu2)
STD_ROM_FN(nes_dragobalzsupbu2)

struct BurnDriver BurnDrvnes_dragobalzsupbu2 = {
	"nes_dragobalzsupbu2", NULL, NULL, NULL, "1989?",
	"Dragon Ball Z - Super Butouden 2 (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalzsupbu2RomInfo, nes_dragobalzsupbu2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonballz5RomDesc[] = {
	{ "Dragon Ball Z 5 (China).nes",          786448, 0x1f2b86ea, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonballz5)
STD_ROM_FN(nes_dragonballz5)

struct BurnDriver BurnDrvnes_dragonballz5 = {
	"nes_dragonballz5", NULL, NULL, NULL, "1989?",
	"Dragon Ball Z 5 (China)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonballz5RomInfo, nes_dragonballz5RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalzgaisajinzejeRomDesc[] = {
	{ "Dragon Ball Z Gaiden - Saiya Jin Zetsumetsu Keikaku (Japan).nes",          524304, 0x69ae6291, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalzgaisajinzeje)
STD_ROM_FN(nes_dragobalzgaisajinzeje)

struct BurnDriver BurnDrvnes_dragobalzgaisajinzeje = {
	"nes_dragobalzgaisajinzeje", "nes_dragobalzgaisajinze", NULL, NULL, "1993",
	"Dragon Ball Z Gaiden - Saiya Jin Zetsumetsu Keikaku (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalzgaisajinzejeRomInfo, nes_dragobalzgaisajinzejeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalzgaisajinzeRomDesc[] = {
	{ "Dragon Ball Z Gaiden - Saiya Jin Zetsumetsu Keikaku (T-Eng).nes",          524304, 0xe7a5a03d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalzgaisajinze)
STD_ROM_FN(nes_dragobalzgaisajinze)

struct BurnDriver BurnDrvnes_dragobalzgaisajinze = {
	"nes_dragobalzgaisajinze", NULL, NULL, NULL, "1989?",
	"Dragon Ball Z Gaiden - Saiya Jin Zetsumetsu Keikaku (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalzgaisajinzeRomInfo, nes_dragobalzgaisajinzeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalziigefrjRomDesc[] = {
	{ "Dragon Ball Z II - Gekishin Freeza!! (Japan).nes",          524304, 0xd396d28a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalziigefrj)
STD_ROM_FN(nes_dragobalziigefrj)

struct BurnDriver BurnDrvnes_dragobalziigefrj = {
	"nes_dragobalziigefrj", "nes_dragobalziigefr", NULL, NULL, "1991",
	"Dragon Ball Z II - Gekishin Freeza!! (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalziigefrjRomInfo, nes_dragobalziigefrjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalziigefrRomDesc[] = {
	{ "Dragon Ball Z II - Gekishin Freeza!! (T-Eng).nes",          524304, 0xffa1e20e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalziigefr)
STD_ROM_FN(nes_dragobalziigefr)

struct BurnDriver BurnDrvnes_dragobalziigefr = {
	"nes_dragobalziigefr", NULL, NULL, NULL, "1991",
	"Dragon Ball Z II - Gekishin Freeza!! (T-Eng)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalziigefrRomInfo, nes_dragobalziigefrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalziiirejinijRomDesc[] = {
	{ "Dragon Ball Z III - Ressen Jinzou Ningen (Japan).nes",          524304, 0xa69079d4, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalziiirejinij)
STD_ROM_FN(nes_dragobalziiirejinij)

struct BurnDriver BurnDrvnes_dragobalziiirejinij = {
	"nes_dragobalziiirejinij", "nes_dragobalziiirejini", NULL, NULL, "1992",
	"Dragon Ball Z III - Ressen Jinzou Ningen (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalziiirejinijRomInfo, nes_dragobalziiirejinijRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobalziiirejiniRomDesc[] = {
	{ "Dragon Ball Z III - Ressen Jinzou Ningen (T-Eng).nes",          524304, 0x209955ea, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobalziiirejini)
STD_ROM_FN(nes_dragobalziiirejini)

struct BurnDriver BurnDrvnes_dragobalziiirejini = {
	"nes_dragobalziiirejini", NULL, NULL, NULL, "1992",
	"Dragon Ball Z III - Ressen Jinzou Ningen (T-Eng)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobalziiirejiniRomInfo, nes_dragobalziiirejiniRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonbusterRomDesc[] = {
	{ "Dragon Buster (Japan).nes",          163856, 0xabb83b0f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonbuster)
STD_ROM_FN(nes_dragonbuster)

struct BurnDriver BurnDrvnes_dragonbuster = {
	"nes_dragonbuster", NULL, NULL, NULL, "1987",
	"Dragon Buster (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonbusterRomInfo, nes_dragonbusterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobusiijRomDesc[] = {
	{ "Dragon Buster II - Yami no Fuuin (Japan).nes",          196624, 0xfa3c1312, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobusiij)
STD_ROM_FN(nes_dragobusiij)

struct BurnDriver BurnDrvnes_dragobusiij = {
	"nes_dragobusiij", "nes_dragobusii", NULL, NULL, "1989",
	"Dragon Buster II - Yami no Fuuin (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobusiijRomInfo, nes_dragobusiijRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragobusiiRomDesc[] = {
	{ "Dragon Buster II - Yami no Fuuin (T-eng).nes",          196624, 0xba54967b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragobusii)
STD_ROM_FN(nes_dragobusii)

struct BurnDriver BurnDrvnes_dragobusii = {
	"nes_dragobusii", NULL, NULL, NULL, "1989",
	"Dragon Buster II - Yami no Fuuin (T-eng)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragobusiiRomInfo, nes_dragobusiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonfighterRomDesc[] = {
	{ "Dragon Fighter (USA).nes",          262160, 0xccd575a1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonfighter)
STD_ROM_FN(nes_dragonfighter)

struct BurnDriver BurnDrvnes_dragonfighter = {
	"nes_dragonfighter", NULL, NULL, NULL, "1992",
	"Dragon Fighter (USA)\0", NULL, "Sofel", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonfighterRomInfo, nes_dragonfighterRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonknifecRomDesc[] = {
	{ "Dragon Knife (Chinese).nes",          524304, 0xeb5dbb86, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonknifec)
STD_ROM_FN(nes_dragonknifec)

struct BurnDriver BurnDrvnes_dragonknifec = {
	"nes_dragonknifec", "nes_dragonknife", NULL, NULL, "1989?",
	"Dragon Knife (Chinese)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonknifecRomInfo, nes_dragonknifecRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonknifeRomDesc[] = {
	{ "Dragon Knife (T-Eng).nes",          524304, 0x142f21a6, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonknife)
STD_ROM_FN(nes_dragonknife)

struct BurnDriver BurnDrvnes_dragonknife = {
	"nes_dragonknife", NULL, NULL, NULL, "1989?",
	"Dragon Knife (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonknifeRomInfo, nes_dragonknifeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonpowerRomDesc[] = {
	{ "Dragon Power (USA).nes",          163856, 0x9517a5ab, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonpower)
STD_ROM_FN(nes_dragonpower)

struct BurnDriver BurnDrvnes_dragonpower = {
	"nes_dragonpower", NULL, NULL, NULL, "1988",
	"Dragon Power (USA)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonpowerRomInfo, nes_dragonpowerRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonrunningRomDesc[] = {
	{ "Dragon Running (Unl).nes",          262160, 0xdfb4c50a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonrunning)
STD_ROM_FN(nes_dragonrunning)

struct BurnDriver BurnDrvnes_dragonrunning = {
	"nes_dragonrunning", NULL, NULL, NULL, "1989?",
	"Dragon Running (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonrunningRomInfo, nes_dragonrunningRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragoscrRomDesc[] = {
	{ "Dragon Scroll - Yomigaerishi Maryuu (Japan).nes",          262160, 0x9475bafc, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragoscr)
STD_ROM_FN(nes_dragoscr)

struct BurnDriver BurnDrvnes_dragoscr = {
	"nes_dragoscr", NULL, NULL, NULL, "1987",
	"Dragon Scroll - Yomigaerishi Maryuu (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragoscrRomInfo, nes_dragoscrRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragosla4RomDesc[] = {
	{ "Dragon Slayer 4 - Drasle Family (Japan).nes",          196624, 0x8870a286, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragosla4)
STD_ROM_FN(nes_dragosla4)

struct BurnDriver BurnDrvnes_dragosla4 = {
	"nes_dragosla4", NULL, NULL, NULL, "1987",
	"Dragon Slayer 4 - Drasle Family (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragosla4RomInfo, nes_dragosla4RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragospiaradenRomDesc[] = {
	{ "Dragon Spirit - Aratanaru Densetsu (Japan).nes",          262160, 0x28b33380, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragospiaraden)
STD_ROM_FN(nes_dragospiaraden)

struct BurnDriver BurnDrvnes_dragospiaraden = {
	"nes_dragospiaraden", NULL, NULL, NULL, "1989",
	"Dragon Spirit - Aratanaru Densetsu (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragospiaradenRomInfo, nes_dragospiaradenRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragospinewlegRomDesc[] = {
	{ "Dragon Spirit - The New Legend (USA).nes",          262160, 0xfd719491, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragospinewleg)
STD_ROM_FN(nes_dragospinewleg)

struct BurnDriver BurnDrvnes_dragospinewleg = {
	"nes_dragospinewleg", NULL, NULL, NULL, "1990",
	"Dragon Spirit - The New Legend (USA)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragospinewlegRomInfo, nes_dragospinewlegRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonwarriorRomDesc[] = {
	{ "Dragon Warrior (USA).nes",          81936, 0xd4d5f5d7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonwarrior)
STD_ROM_FN(nes_dragonwarrior)

struct BurnDriver BurnDrvnes_dragonwarrior = {
	"nes_dragonwarrior", NULL, NULL, NULL, "1989",
	"Dragon Warrior (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonwarriorRomInfo, nes_dragonwarriorRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragowariiRomDesc[] = {
	{ "Dragon Warrior II (USA).nes",          262160, 0xe9f5be99, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragowarii)
STD_ROM_FN(nes_dragowarii)

struct BurnDriver BurnDrvnes_dragowarii = {
	"nes_dragowarii", NULL, NULL, NULL, "1990",
	"Dragon Warrior II (USA)\0", NULL, "Enix", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragowariiRomInfo, nes_dragowariiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragowariiiRomDesc[] = {
	{ "Dragon Warrior III (USA).nes",          524304, 0x0eb63e83, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragowariii)
STD_ROM_FN(nes_dragowariii)

struct BurnDriver BurnDrvnes_dragowariii = {
	"nes_dragowariii", NULL, NULL, NULL, "1992",
	"Dragon Warrior III (USA)\0", NULL, "Enix", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragowariiiRomInfo, nes_dragowariiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragowarivRomDesc[] = {
	{ "Dragon Warrior IV (USA).nes",          524304, 0xf6b24806, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragowariv)
STD_ROM_FN(nes_dragowariv)

struct BurnDriver BurnDrvnes_dragowariv = {
	"nes_dragowariv", NULL, NULL, NULL, "1992",
	"Dragon Warrior IV (USA)\0", NULL, "Enix", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragowarivRomInfo, nes_dragowarivRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dragonslairRomDesc[] = {
	{ "Dragon's Lair (USA).nes",          131088, 0xf90ae80e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dragonslair)
STD_ROM_FN(nes_dragonslair)

struct BurnDriver BurnDrvnes_dragonslair = {
	"nes_dragonslair", NULL, NULL, NULL, "1990",
	"Dragon's Lair (USA)\0", NULL, "Sony (CSG) Imagesoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dragonslairRomInfo, nes_dragonslairRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dreamworldpogieRomDesc[] = {
	{ "Dreamworld Pogie (Prototype).nes",          131088, 0x2ace2f1b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dreamworldpogie)
STD_ROM_FN(nes_dreamworldpogie)

struct BurnDriver BurnDrvnes_dreamworldpogie = {
	"nes_dreamworldpogie", NULL, NULL, NULL, "1989?",
	"Dreamworld Pogie (Prototype)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dreamworldpogieRomInfo, nes_dreamworldpogieRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dringleRomDesc[] = {
	{ "Dringle (Unl).nes",          40976, 0x05c38615, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dringle)
STD_ROM_FN(nes_dringle)

struct BurnDriver BurnDrvnes_dringle = {
	"nes_dringle", NULL, NULL, NULL, "1989?",
	"Dringle (Unl)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dringleRomInfo, nes_dringleRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dropzoneRomDesc[] = {
	{ "Dropzone (Europe).nes",          65552, 0xab29ab28, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dropzone)
STD_ROM_FN(nes_dropzone)

struct BurnDriver BurnDrvnes_dropzone = {
	"nes_dropzone", NULL, NULL, NULL, "1992",
	"Dropzone (Europe)\0", NULL, "Mindscape", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dropzoneRomInfo, nes_dropzoneRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL, SCREEN_WIDTH, SCREEN_HEIGHT_PAL
};

static struct BurnRomInfo nes_druaganotouRomDesc[] = {
	{ "Druaga no Tou (Japan).nes",          40976, 0x0c045fb9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_druaganotou)
STD_ROM_FN(nes_druaganotou)

struct BurnDriver BurnDrvnes_druaganotou = {
	"nes_druaganotou", NULL, NULL, NULL, "1985",
	"Druaga no Tou (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_druaganotouRomInfo, nes_druaganotouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_duckhuntRomDesc[] = {
	{ "Duck Hunt (USA).nes",          24592, 0x4644085e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_duckhunt)
STD_ROM_FN(nes_duckhunt)

struct BurnDriver BurnDrvnes_duckhunt = {
	"nes_duckhunt", NULL, NULL, NULL, "1984",
	"Duck Hunt (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_duckhuntRomInfo, nes_duckhuntRomName, NULL, NULL, NULL, NULL, NESZapperInputInfo, NESZapperDIPInfo,
	NESZapperInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_duckmazeRomDesc[] = {
	{ "Duck Maze (USA).nes",          40976, 0x01c1ca36, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_duckmaze)
STD_ROM_FN(nes_duckmaze)

struct BurnDriver BurnDrvnes_duckmaze = {
	"nes_duckmaze", NULL, NULL, NULL, "1989?",
	"Duck Maze (USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_duckmazeRomInfo, nes_duckmazeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ducktalesRomDesc[] = {
	{ "DuckTales (USA).nes",          131088, 0x06b994b2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ducktales)
STD_ROM_FN(nes_ducktales)

struct BurnDriver BurnDrvnes_ducktales = {
	"nes_ducktales", NULL, NULL, NULL, "1989",
	"DuckTales (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ducktalesRomInfo, nes_ducktalesRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_ducktales2RomDesc[] = {
	{ "DuckTales 2 (USA).nes",          131088, 0x40ce2fc0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_ducktales2)
STD_ROM_FN(nes_ducktales2)

struct BurnDriver BurnDrvnes_ducktales2 = {
	"nes_ducktales2", NULL, NULL, NULL, "1993",
	"DuckTales 2 (USA)\0", NULL, "Capcom", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_ducktales2RomInfo, nes_ducktales2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dudeswitattRomDesc[] = {
	{ "Dudes With Attitude (Unl).nes",          65552, 0xe9151645, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dudeswitatt)
STD_ROM_FN(nes_dudeswitatt)

struct BurnDriver BurnDrvnes_dudeswitatt = {
	"nes_dudeswitatt", NULL, NULL, NULL, "1990",
	"Dudes With Attitude (Unl)\0", NULL, "AVE", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dudeswitattRomInfo, nes_dudeswitattRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dungemagRomDesc[] = {
	{ "Dungeon Magic - Sword of the Elements (USA).nes",          262160, 0x7a365bf8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dungemag)
STD_ROM_FN(nes_dungemag)

struct BurnDriver BurnDrvnes_dungemag = {
	"nes_dungemag", NULL, NULL, NULL, "1990",
	"Dungeon Magic - Sword of the Elements (USA)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dungemagRomInfo, nes_dungemagRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dustydiasallstsoRomDesc[] = {
	{ "Dusty Diamond's All-Star Softball (USA).nes",          262160, 0x5506b1ca, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dustydiasallstso)
STD_ROM_FN(nes_dustydiasallstso)

struct BurnDriver BurnDrvnes_dustydiasallstso = {
	"nes_dustydiasallstso", NULL, NULL, NULL, "1990",
	"Dusty Diamond's All-Star Softball (USA)\0", NULL, "Broderbund", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dustydiasallstsoRomInfo, nes_dustydiasallstsoRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dynamitebatmanRomDesc[] = {
	{ "Dynamite Batman (Japan).nes",          393232, 0xed509c6b, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dynamitebatman)
STD_ROM_FN(nes_dynamitebatman)

struct BurnDriver BurnDrvnes_dynamitebatman = {
	"nes_dynamitebatman", "nes_batmaretjok", NULL, NULL, "1991",
	"Dynamite Batman (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dynamitebatmanRomInfo, nes_dynamitebatmanRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dynamitebowlRomDesc[] = {
	{ "Dynamite Bowl (Japan).nes",          65552, 0x4779d33a, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dynamitebowl)
STD_ROM_FN(nes_dynamitebowl)

struct BurnDriver BurnDrvnes_dynamitebowl = {
	"nes_dynamitebowl", NULL, NULL, NULL, "1987",
	"Dynamite Bowl (Japan)\0", NULL, "Toemiland", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dynamitebowlRomInfo, nes_dynamitebowlRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_dynowRomDesc[] = {
	{ "Dynowarz - Destruction of Spondylus (USA).nes",          262160, 0xd05a18b7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_dynow)
STD_ROM_FN(nes_dynow)

struct BurnDriver BurnDrvnes_dynow = {
	"nes_dynow", NULL, NULL, NULL, "1990",
	"Dynowarz - Destruction of Spondylus (USA)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_dynowRomInfo, nes_dynowRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_earthboundRomDesc[] = {
	{ "Earthbound (USA) (Prototype).nes",          524304, 0xf5ef5002, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_earthbound)
STD_ROM_FN(nes_earthbound)

struct BurnDriver BurnDrvnes_earthbound = {
	"nes_earthbound", NULL, NULL, NULL, "1989?",
	"Earthbound (USA) (Prototype)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_earthboundRomInfo, nes_earthboundRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_egyptjRomDesc[] = {
	{ "Egypt (Japan).nes",          65552, 0x38df4be7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_egyptj)
STD_ROM_FN(nes_egyptj)

struct BurnDriver BurnDrvnes_egyptj = {
	"nes_egyptj", "nes_egypt", NULL, NULL, "1991",
	"Egypt (Japan)\0", NULL, "Human Entertainment", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_egyptjRomInfo, nes_egyptjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_egyptRomDesc[] = {
	{ "Egypt (T-Eng).nes",          65552, 0xbb7f1c06, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_egypt)
STD_ROM_FN(nes_egypt)

struct BurnDriver BurnDrvnes_egypt = {
	"nes_egypt", NULL, NULL, NULL, "1991",
	"Egypt (T-Eng)\0", NULL, "Human Entertainment", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_egyptRomInfo, nes_egyptRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_elevatoractionRomDesc[] = {
	{ "Elevator Action (USA).nes",          40976, 0xd415571f, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_elevatoraction)
STD_ROM_FN(nes_elevatoraction)

struct BurnDriver BurnDrvnes_elevatoraction = {
	"nes_elevatoraction", NULL, NULL, NULL, "1987",
	"Elevator Action (USA)\0", NULL, "Taito", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_elevatoractionRomInfo, nes_elevatoractionRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_elimiboadueRomDesc[] = {
	{ "Eliminator Boat Duel (USA).nes",          262160, 0x682d2df1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_elimiboadue)
STD_ROM_FN(nes_elimiboadue)

struct BurnDriver BurnDrvnes_elimiboadue = {
	"nes_elimiboadue", NULL, NULL, NULL, "1991",
	"Eliminator Boat Duel (USA)\0", NULL, "Electro Brain", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_elimiboadueRomInfo, nes_elimiboadueRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_eliteRomDesc[] = {
	{ "Elite (Europe).nes",          131088, 0x4cf12d39, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_elite)
STD_ROM_FN(nes_elite)

struct BurnDriver BurnDrvnes_elite = {
	"nes_elite", NULL, NULL, NULL, "1991",
	"Elite (Europe)\0", NULL, "Imagineer", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_eliteRomInfo, nes_eliteRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT_PAL, SCREEN_WIDTH, SCREEN_HEIGHT_PAL
};

static struct BurnRomInfo nes_elnarknozaihouRomDesc[] = {
	{ "Elnark no Zaihou (Japan).nes",          131088, 0xc759080e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_elnarknozaihou)
STD_ROM_FN(nes_elnarknozaihou)

struct BurnDriver BurnDrvnes_elnarknozaihou = {
	"nes_elnarknozaihou", NULL, NULL, NULL, "1987",
	"Elnark no Zaihou (Japan)\0", NULL, "Towa Chiki", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_elnarknozaihouRomInfo, nes_elnarknozaihouRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_empircitRomDesc[] = {
	{ "Empire City - 1931 (Japan).nes",          131088, 0xbd736ec8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_empircit)
STD_ROM_FN(nes_empircit)

struct BurnDriver BurnDrvnes_empircit = {
	"nes_empircit", NULL, NULL, NULL, "1989?",
	"Empire City - 1931 (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_empircitRomInfo, nes_empircitRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_escapfroatlRomDesc[] = {
	{ "Escape From Atlantis, The (Prototype).nes",          131088, 0x3c550fd5, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_escapfroatl)
STD_ROM_FN(nes_escapfroatl)

struct BurnDriver BurnDrvnes_escapfroatl = {
	"nes_escapfroatl", NULL, NULL, NULL, "0000",
	"Escape From Atlantis, The (Prototype)\0", NULL, "Color Dreams", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_escapfroatlRomInfo, nes_escapfroatlRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_esperboukentaiRomDesc[] = {
	{ "Esper Bouken Tai (Japan).nes",          262160, 0x56b7dfe3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_esperboukentai)
STD_ROM_FN(nes_esperboukentai)

struct BurnDriver BurnDrvnes_esperboukentai = {
	"nes_esperboukentai", NULL, NULL, NULL, "1987",
	"Esper Bouken Tai (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_esperboukentaiRomInfo, nes_esperboukentaiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_esperdre2jRomDesc[] = {
	{ "Esper Dream 2 - Aratanaru Tatakai (Japan).nes",          393232, 0x72344f1d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_esperdre2j)
STD_ROM_FN(nes_esperdre2j)

struct BurnDriver BurnDrvnes_esperdre2j = {
	"nes_esperdre2j", "nes_esperdre2", NULL, NULL, "1992",
	"Esper Dream 2 - Aratanaru Tatakai (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_esperdre2jRomInfo, nes_esperdre2jRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_esperdre2RomDesc[] = {
	{ "Esper Dream 2 - Aratanaru Tatakai (T-Eng).nes",          393232, 0x892b32eb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_esperdre2)
STD_ROM_FN(nes_esperdre2)

struct BurnDriver BurnDrvnes_esperdre2 = {
	"nes_esperdre2", NULL, NULL, NULL, "1992",
	"Esper Dream 2 - Aratanaru Tatakai (T-Eng)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_esperdre2RomInfo, nes_esperdre2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_excitebikeRomDesc[] = {
	{ "Excitebike (Japan, USA).nes",          24592, 0x2bb33c69, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_excitebike)
STD_ROM_FN(nes_excitebike)

struct BurnDriver BurnDrvnes_excitebike = {
	"nes_excitebike", NULL, NULL, NULL, "1984",
	"Excitebike (Japan, USA)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_excitebikeRomInfo, nes_excitebikeRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_excitingboxingRomDesc[] = {
	{ "Exciting Boxing (Japan).nes",          262160, 0xb374b6b0, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_excitingboxing)
STD_ROM_FN(nes_excitingboxing)

struct BurnDriver BurnDrvnes_excitingboxing = {
	"nes_excitingboxing", NULL, NULL, NULL, "1987",
	"Exciting Boxing (Japan)\0", NULL, "Konami", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_excitingboxingRomInfo, nes_excitingboxingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_excitralRomDesc[] = {
	{ "Exciting Rally - World Rally Championship (Japan).nes",          262160, 0x8cefb613, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_excitral)
STD_ROM_FN(nes_excitral)

struct BurnDriver BurnDrvnes_excitral = {
	"nes_excitral", NULL, NULL, NULL, "1989?",
	"Exciting Rally - World Rally Championship (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_excitralRomInfo, nes_excitralRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_exerionRomDesc[] = {
	{ "Exerion (Japan).nes",          24592, 0x352f9a62, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_exerion)
STD_ROM_FN(nes_exerion)

struct BurnDriver BurnDrvnes_exerion = {
	"nes_exerion", NULL, NULL, NULL, "1985",
	"Exerion (Japan)\0", NULL, "Jaleco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_exerionRomInfo, nes_exerionRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_exodusRomDesc[] = {
	{ "Exodus - Journey to the Promised Land (USA) (Unl).nes",          262160, 0x4b334bdf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_exodus)
STD_ROM_FN(nes_exodus)

struct BurnDriver BurnDrvnes_exodus = {
	"nes_exodus", NULL, NULL, NULL, "1991",
	"Exodus - Journey to the Promised Land (USA) (Unl)\0", NULL, "Wisdom Tree", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_exodusRomInfo, nes_exodusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_explodingfistRomDesc[] = {
	{ "Exploding Fist (USA, Prototype).nes",          65552, 0xc52ea831, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_explodingfist)
STD_ROM_FN(nes_explodingfist)

struct BurnDriver BurnDrvnes_explodingfist = {
	"nes_explodingfist", NULL, NULL, NULL, "0000",
	"Exploding Fist (USA, Prototype)\0", NULL, "Tradewest", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 4, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_explodingfistRomInfo, nes_explodingfistRomName, NULL, NULL, NULL, NULL, NES4ScoreInputInfo, NES4ScoreDIPInfo,
	NES4ScoreInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f1heroRomDesc[] = {
	{ "F-1 Hero (Japan).nes",          262160, 0xab82d025, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f1hero)
STD_ROM_FN(nes_f1hero)

struct BurnDriver BurnDrvnes_f1hero = {
	"nes_f1hero", NULL, NULL, NULL, "1988",
	"F-1 Hero (Japan)\0", NULL, "Varie", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f1heroRomInfo, nes_f1heroRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f1hero2RomDesc[] = {
	{ "F-1 Hero 2 (Japan).nes",          262160, 0x36b9506d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f1hero2)
STD_ROM_FN(nes_f1hero2)

struct BurnDriver BurnDrvnes_f1hero2 = {
	"nes_f1hero2", NULL, NULL, NULL, "1991",
	"F-1 Hero 2 (Japan)\0", NULL, "Varie", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f1hero2RomInfo, nes_f1hero2RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f1raceRomDesc[] = {
	{ "F-1 Race (Japan).nes",          24592, 0x827dc0b9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f1race)
STD_ROM_FN(nes_f1race)

struct BurnDriver BurnDrvnes_f1race = {
	"nes_f1race", NULL, NULL, NULL, "1984",
	"F-1 Race (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f1raceRomInfo, nes_f1raceRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f1sensationRomDesc[] = {
	{ "F-1 Sensation (Japan).nes",          262160, 0xabb7157e, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f1sensation)
STD_ROM_FN(nes_f1sensation)

struct BurnDriver BurnDrvnes_f1sensation = {
	"nes_f1sensation", NULL, NULL, NULL, "1989?",
	"F-1 Sensation (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f1sensationRomInfo, nes_f1sensationRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f117stefigRomDesc[] = {
	{ "F-117A Stealth Fighter (USA).nes",          524304, 0x6aa0a7ad, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f117stefig)
STD_ROM_FN(nes_f117stefig)

struct BurnDriver BurnDrvnes_f117stefig = {
	"nes_f117stefig", NULL, NULL, NULL, "1992",
	"F-117A Stealth Fighter (USA)\0", NULL, "Microprose", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f117stefigRomInfo, nes_f117stefigRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f15citywarRomDesc[] = {
	{ "F-15 City War (USA) (Unl).nes",          65552, 0xa7b81c02, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f15citywar)
STD_ROM_FN(nes_f15citywar)

struct BurnDriver BurnDrvnes_f15citywar = {
	"nes_f15citywar", NULL, NULL, NULL, "1990",
	"F-15 City War (USA) (Unl)\0", NULL, "AVE", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f15citywarRomInfo, nes_f15citywarRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f15streagRomDesc[] = {
	{ "F-15 Strike Eagle (USA).nes",          524304, 0x6b1461ae, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f15streag)
STD_ROM_FN(nes_f15streag)

struct BurnDriver BurnDrvnes_f15streag = {
	"nes_f15streag", NULL, NULL, NULL, "1992",
	"F-15 Strike Eagle (USA)\0", NULL, "Microprose", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f15streagRomInfo, nes_f15streagRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f1circusjRomDesc[] = {
	{ "F1 Circus (Japan).nes",          393232, 0x1f19532d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f1circusj)
STD_ROM_FN(nes_f1circusj)

struct BurnDriver BurnDrvnes_f1circusj = {
	"nes_f1circusj", "nes_f1circus", NULL, NULL, "1992",
	"F1 Circus (Japan)\0", NULL, "Nichibutsu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f1circusjRomInfo, nes_f1circusjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_f1circusRomDesc[] = {
	{ "F1 Circus (T-Eng).nes",          393232, 0x67c75e53, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f1circus)
STD_ROM_FN(nes_f1circus)

struct BurnDriver BurnDrvnes_f1circus = {
	"nes_f1circus", NULL, NULL, NULL, "1992",
	"F1 Circus (T-Eng)\0", NULL, "Nichibutsu", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_f1circusRomInfo, nes_f1circusRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famicjumRomDesc[] = {
	{ "Famicom Jump - Eiyuu Retsuden (Japan).nes",          393232, 0xb91afcaf, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famicjum)
STD_ROM_FN(nes_famicjum)

struct BurnDriver BurnDrvnes_famicjum = {
	"nes_famicjum", NULL, NULL, NULL, "1989",
	"Famicom Jump - Eiyuu Retsuden (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famicjumRomInfo, nes_famicjumRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famicjumiiRomDesc[] = {
	{ "Famicom Jump II - Saikyou no 7 Nin (Japan).nes",          524304, 0x6b41421c, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famicjumii)
STD_ROM_FN(nes_famicjumii)

struct BurnDriver BurnDrvnes_famicjumii = {
	"nes_famicjumii", NULL, NULL, NULL, "1991",
	"Famicom Jump II - Saikyou no 7 Nin (Japan)\0", NULL, "Bandai", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famicjumiiRomInfo, nes_famicjumiiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famicomwarsjRomDesc[] = {
	{ "Famicom Wars (Japan).nes",          196624, 0x29f3a0f7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famicomwarsj)
STD_ROM_FN(nes_famicomwarsj)

struct BurnDriver BurnDrvnes_famicomwarsj = {
	"nes_famicomwarsj", "nes_famicomwars", NULL, NULL, "1988",
	"Famicom Wars (Japan)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famicomwarsjRomInfo, nes_famicomwarsjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famicomwarsRomDesc[] = {
	{ "Famicom Wars (T-Eng).nes",          196624, 0x896759d8, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famicomwars)
STD_ROM_FN(nes_famicomwars)

struct BurnDriver BurnDrvnes_famicomwars = {
	"nes_famicomwars", NULL, NULL, NULL, "1988",
	"Famicom Wars (T-Eng)\0", NULL, "Nintendo", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famicomwarsRomInfo, nes_famicomwarsRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_familyboxingRomDesc[] = {
	{ "Family Boxing (Japan).nes",          131088, 0x381b51f3, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_familyboxing)
STD_ROM_FN(nes_familyboxing)

struct BurnDriver BurnDrvnes_familyboxing = {
	"nes_familyboxing", NULL, NULL, NULL, "1987",
	"Family Boxing (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_familyboxingRomInfo, nes_familyboxingRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_familcir91RomDesc[] = {
	{ "Family Circuit '91 (Japan).nes",          655376, 0xcfbb646d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_familcir91)
STD_ROM_FN(nes_familcir91)

struct BurnDriver BurnDrvnes_familcir91 = {
	"nes_familcir91", NULL, NULL, NULL, "1991",
	"Family Circuit '91 (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_familcir91RomInfo, nes_familcir91RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_familycircuitjRomDesc[] = {
	{ "Family Circuit (Japan).nes",          163856, 0x51ce0655, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_familycircuitj)
STD_ROM_FN(nes_familycircuitj)

struct BurnDriver BurnDrvnes_familycircuitj = {
	"nes_familycircuitj", "nes_familycircuit", NULL, NULL, "1988",
	"Family Circuit (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_familycircuitjRomInfo, nes_familycircuitjRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_familycircuitRomDesc[] = {
	{ "Family Circuit (T-Eng).nes",          163856, 0xe44892f7, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_familycircuit)
STD_ROM_FN(nes_familycircuit)

struct BurnDriver BurnDrvnes_familycircuit = {
	"nes_familycircuit", NULL, NULL, NULL, "1988",
	"Family Circuit (T-Eng)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_familycircuitRomInfo, nes_familycircuitRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_familypinballRomDesc[] = {
	{ "Family Pinball (Japan).nes",          196624, 0x0da8a7b2, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_familypinball)
STD_ROM_FN(nes_familypinball)

struct BurnDriver BurnDrvnes_familypinball = {
	"nes_familypinball", NULL, NULL, NULL, "1989",
	"Family Pinball (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_familypinballRomInfo, nes_familypinballRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famista89RomDesc[] = {
	{ "Famista '89 (Japan).nes",          196624, 0x62b90039, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famista89)
STD_ROM_FN(nes_famista89)

struct BurnDriver BurnDrvnes_famista89 = {
	"nes_famista89", NULL, NULL, NULL, "1989",
	"Famista '89 (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famista89RomInfo, nes_famista89RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famista90RomDesc[] = {
	{ "Famista '90 (Japan).nes",          262160, 0xc942dfe9, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famista90)
STD_ROM_FN(nes_famista90)

struct BurnDriver BurnDrvnes_famista90 = {
	"nes_famista90", NULL, NULL, NULL, "1989",
	"Famista '90 (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famista90RomInfo, nes_famista90RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famista91RomDesc[] = {
	{ "Famista '91 (Japan).nes",          262160, 0x7195ca38, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famista91)
STD_ROM_FN(nes_famista91)

struct BurnDriver BurnDrvnes_famista91 = {
	"nes_famista91", NULL, NULL, NULL, "1990",
	"Famista '91 (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famista91RomInfo, nes_famista91RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famista92RomDesc[] = {
	{ "Famista '92 (Japan).nes",          262160, 0xfb6bdb9d, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famista92)
STD_ROM_FN(nes_famista92)

struct BurnDriver BurnDrvnes_famista92 = {
	"nes_famista92", NULL, NULL, NULL, "1991",
	"Famista '92 (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famista92RomInfo, nes_famista92RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famista93RomDesc[] = {
	{ "Famista '93 (Japan).nes",          262160, 0x38511037, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famista93)
STD_ROM_FN(nes_famista93)

struct BurnDriver BurnDrvnes_famista93 = {
	"nes_famista93", NULL, NULL, NULL, "1992",
	"Famista '93 (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famista93RomInfo, nes_famista93RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_famista94RomDesc[] = {
	{ "Famista '94 (Japan).nes",          262160, 0xd73fc5b1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_famista94)
STD_ROM_FN(nes_famista94)

struct BurnDriver BurnDrvnes_famista94 = {
	"nes_famista94", NULL, NULL, NULL, "1993",
	"Famista '94 (Japan)\0", NULL, "Namco", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_famista94RomInfo, nes_famista94RomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fantaadvdizRomDesc[] = {
	{ "Fantastic Adventures of Dizzy, The (USA).nes",          262160, 0x59318584, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fantaadvdiz)
STD_ROM_FN(nes_fantaadvdiz)

struct BurnDriver BurnDrvnes_fantaadvdiz = {
	"nes_fantaadvdiz", NULL, NULL, NULL, "1991",
	"Fantastic Adventures of Dizzy, The (USA)\0", NULL, "Camerica", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_fantaadvdizRomInfo, nes_fantaadvdizRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fantasyzoneRomDesc[] = {
	{ "Fantasy Zone (Japan).nes",          131088, 0xdd119b43, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fantasyzone)
STD_ROM_FN(nes_fantasyzone)

struct BurnDriver BurnDrvnes_fantasyzone = {
	"nes_fantasyzone", NULL, NULL, NULL, "1987",
	"Fantasy Zone (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_fantasyzoneRomInfo, nes_fantasyzoneRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fantazoniiRomDesc[] = {
	{ "Fantasy Zone II - Opa-Opa no Namida (Japan).nes",          262160, 0x6faf6db1, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_fantazonii)
STD_ROM_FN(nes_fantazonii)

struct BurnDriver BurnDrvnes_fantazonii = {
	"nes_fantazonii", NULL, NULL, NULL, "1988",
	"Fantasy Zone II - Opa-Opa no Namida (Japan)\0", NULL, "Sunsoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_fantazoniiRomInfo, nes_fantazoniiRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_fariaRomDesc[] = {
	{ "Faria - A World of Mystery & Danger! (USA).nes",          262160, 0x1c059dfb, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_faria)
STD_ROM_FN(nes_faria)

struct BurnDriver BurnDrvnes_faria = {
	"nes_faria", NULL, NULL, NULL, "1991",
	"Faria - A World of Mystery & Danger! (USA)\0", NULL, "Nexoft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_fariaRomInfo, nes_fariaRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_faxanadujRomDesc[] = {
	{ "Faxanadu (Japan).nes",          262160, 0xf9e6e6ad, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_faxanaduj)
STD_ROM_FN(nes_faxanaduj)

struct BurnDriver BurnDrvnes_faxanaduj = {
	"nes_faxanaduj", "nes_faxanadu", NULL, NULL, "1987",
	"Faxanadu (Japan)\0", NULL, "Hudson Soft", "Miscellaneous",
	NULL, NULL, NULL, NULL,
	BDF_GAME_WORKING | BDF_CLONE, 2, HARDWARE_NES, GBF_MISC, 0,
	NESGetZipName, nes_faxanadujRomInfo, nes_faxanadujRomName, NULL, NULL, NULL, NULL, NESInputInfo, NESDIPInfo,
	NESInit, NESExit, NESFrame, NESDraw, NESScan, &NESRecalc, 0x40,
	SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_WIDTH, SCREEN_HEIGHT
};

static struct BurnRomInfo nes_faxanadutRomDesc[] = {
	{ "Faxanadu (T-Eng).nes",          524304, 0xed0a6836, BRF_ESS | BRF_PRG },
};

STD_ROM_PICK(nes_f