/*	$NetBSD: fstest_lfs.c,v 1.9 2025/11/06 15:45:32 perseant Exp $	*/

/*-
 * Copyright (c) 2010 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Nicolas Joly.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/mount.h>
#include <sys/stat.h>

#include <atf-c.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <ufs/ufs/ufsmount.h>

#include <rump/rump.h>
#include <rump/rump_syscallshotgun.h>
#include <rump/rump_syscalls.h>

#include "h_fsmacros.h"
#include "mount_lfs.h"

/* Use the in-kenrel cleaner rather than a separate lfs_cleanerd process */
#define USE_KERNEL_CLEANER

struct lfstestargs {
	struct ufs_args ta_uargs;
	pthread_t ta_cleanerthread;
	sem_t ta_cleanerloop;
	char ta_devpath[MAXPATHLEN];
	char ta_imgpath[MAXPATHLEN];
	char ta_mntpath[MAXPATHLEN];
	char ta_hostpath[MAXPATHLEN];
};

int
lfs_fstest_newfs(const atf_tc_t *tc, void **buf, const char *image, off_t size,
	void *fspriv)
{
	char cmd[1024];
	int res;
	static unsigned int num = 0;
	struct lfstestargs *args;

	size /= 512;
	snprintf(cmd, 1024, "newfs_lfs -D -F -s %"PRId64" ./%s >/dev/null",
	    size, image);
	res = system(cmd);
	if (res != 0)
		return res;

	res = rump_init();
	if (res != 0)
		return res;

	args = calloc(1, sizeof(*args));
	if (args == NULL)
		return -1;

	strcpy(args->ta_hostpath, image);
	snprintf(args->ta_devpath, MAXPATHLEN, "/dev/device%d.lfs", num);
	snprintf(args->ta_imgpath, MAXPATHLEN, "%s", image);
	args->ta_uargs.fspec = args->ta_devpath;
	sem_init(&args->ta_cleanerloop, 0, 0);

	res = rump_pub_etfs_register(args->ta_devpath, image, RUMP_ETFS_BLK);
	if (res != 0) {
		free(args);
		return res;
	}

	*buf = args;
	num++;

	return 0;
}

int
lfs_fstest_delfs(const atf_tc_t *tc, void *buf)
{
	int res;
	struct lfstestargs *args = buf;

	res = rump_pub_etfs_remove(args->ta_devpath);
	if (res != 0)
		return res;

	res = unlink(args->ta_imgpath);
	if (res != 0)
		return res;

	pthread_join(args->ta_cleanerthread, NULL);
	free(args);

	return 0;
}

#ifndef USE_KERNEL_CLEANER
static void *
cleaner(void *arg)
{
	char thepath[MAXPATHLEN];
	struct lfstestargs *args = arg;
	const char *the_argv[9];
	char buf[64];

	rump_pub_lwproc_newlwp(rump_sys_getpid());

	/* this inspired by the cleaner code.  fixme */
	sprintf(thepath, "/dev/r%s", args->ta_devpath+5);
	rump_pub_etfs_register(thepath, args->ta_hostpath, RUMP_ETFS_CHR);
	sprintf(buf, "%p", &args->ta_cleanerloop);

	the_argv[0] = "megamaid";
	the_argv[1] = "-D"; /* don't fork() & detach */
	the_argv[2] = "-S";
	the_argv[3] = buf;
	the_argv[4] = "-J";
	the_argv[5] = thepath;
	the_argv[6] = args->ta_mntpath;
	the_argv[7] = NULL;

	/* xxxatf */
	optind = 1;
	opterr = 1;

	lfs_cleaner_main(7, __UNCONST(the_argv));

	rump_pub_lwproc_releaselwp();

	return NULL;
}
#endif /* 0 */

int
lfs_fstest_mount(const atf_tc_t *tc, void *buf, const char *path, int flags)
{
	struct lfstestargs *args = buf;
	int res;
#ifdef USE_KERNEL_CLEANER
	struct lfs_autoclean_params autoclean;
	int fd;
#endif /* USE_KERNEL_CLEANER */

	res = rump_sys_mkdir(path, 0777);
	if (res == -1)
		return res;

	res = rump_sys_mount(MOUNT_LFS, path, flags, &args->ta_uargs,
	    sizeof(args->ta_uargs));
	if (res == -1) {
		fprintf(stderr, "sys_mount errno=%d\n", errno);
		return res;
	}

#ifdef USE_KERNEL_CLEANER
	fd = rump_sys_open(path, O_RDONLY);
	autoclean.size = sizeof(autoclean);
	autoclean.mode = LFS_CLEANMODE_GREEDY;
	autoclean.thresh = -1;
	autoclean.target = -1;
	res = rump_sys_fcntl(fd, LFCNAUTOCLEAN, &autoclean);
	if (res) {
		fprintf(stderr, "LFCNAUTOCLEAN errno=%d\n", errno);
		return res;
	}
	res = rump_sys_close(fd);
	if (res)
		return res;
#else /* ! USE_KERNEL_CLEANER */
	strcpy(args->ta_mntpath, path);
	res = pthread_create(&args->ta_cleanerthread, NULL, cleaner, args);
	if (res)
		return res;
	
	/* wait for cleaner to initialize */
	sem_wait(&args->ta_cleanerloop);
#endif /* ! USE_KERNEL_CLEANER */

	return 0;
}

int
lfs_fstest_unmount(const atf_tc_t *tc, const char *path, int flags)
{
	int res;

	res = rump_sys_unmount(path, flags);
	if (res == -1) {
		return res;
	}

	res = rump_sys_rmdir(path);
	return res;
}
