/* Copyright (c) 2005 Michael Schroeder (mls@suse.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 ****************************************************************
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include "common.h"

void
dump16(i)
int i;
{
  putchar(i >> 8 & 255);
  putchar(i & 255);
}

void
dump32(i)
int i;
{
  putchar(i >> 24 & 255);
  putchar(i >> 16 & 255);
  putchar(i >> 8 & 255);
  putchar(i & 255);
}

void
dumpstr(s, l)
char *s;
int l;
{
  while(l--)
    putchar(*s++);
}

void
dumpfn(fn)
char *fn;
{
  static char *oldfn;
  char *op, *fp;
  int l, ol = 0;

  l = strlen(fn);
  fp = fn;
  if (oldfn)
    {
      op = oldfn;
      while (*op && *op == *fp)
	{
	  op++;
	  fp++;
	  l--;
	  ol++;
	  if (ol == 255)
	    break;
	}
      free(oldfn);
    }
  putchar(ol);
  if (l > 254)
    {
      putchar(255);
      dump16(l);
    }
  else
    putchar(l);
  dumpstr(fp, l);
  oldfn = strdup(fn);
}

void
dumpownergroup(own, grp, ss)
char *own;
char *grp;
int ss;
{
  int lo = 0, lg = 0;
  static char *oldown, *oldgrp;

  if (!oldown || strcmp(own, oldown))
    {
      oldown = strdup(own);
      lo = strlen(own);
      if (lo > 1)
	lo--;
      if (lo == 0)
	{
	  fprintf(stderr, "oops, zero owner\n");
	  exit(1);
	}
    }
  if (!oldgrp || strcmp(grp, oldgrp))
    {
      oldgrp = strdup(grp);
      lg = strlen(grp);
      if (lg > 1)
	lg--;
      if (lg == 0)
	{
	  fprintf(stderr, "oops, zero group\n");
	  exit(1);
	}
    }
  if (lo >= 8 || lg >= 8 || (lo << 5 | lg << 2 | ss) == 0xff)
    {
      putchar(0xff);
      putchar(lo);
      putchar(lg << 2 | ss);
    }
  else
    {
      putchar(lo << 5 | lg << 2 | ss);
    }
  if (lo)
    dumpstr(own, lo + 1);
  if (lg)
    dumpstr(grp, lg + 1);
}

void
dumphl(hl)
struct hardlink *hl;
{
  if (hl->next)
    dumphl(hl->next);
  dumpfn(hl->path);
  dump16(0);
}

void
dumpcpio(srpm)
struct srpm *srpm;
{
  int ss;

  if (srpm->cp.mode == 0)
    {
      fprintf(stderr, "oops, zero file mode\n");
      exit(1);
    }
  if (srpm->cp.next)
    dumphl(srpm->cp.next);
  dumpfn(srpm->cp.path);
  dump16(srpm->cp.mode);
  if (!S_ISREG(srpm->cp.mode) && !S_ISLNK(srpm->cp.mode))
    {
      if (srpm->cp.size)
	rpmeat(srpm);
      srpm->cp.size = 0;
    }
  if (srpm->cp.size >= (1 << 24))
    ss = 3;
  else if (srpm->cp.size >= (1 << 16))
    ss = 2;
  else if (srpm->cp.size >= (1 << 8))
    ss = 1;
  else
    ss = 0;
  dumpownergroup(srpm->cp.owner, srpm->cp.group, ss);
  if (S_ISREG(srpm->cp.mode) || S_ISLNK(srpm->cp.mode))
    {
      if (srpm->cp.size >= (1 << 24))
	putchar(srpm->cp.size >> 24 & 255);
      if (srpm->cp.size >= (1 << 16))
	putchar(srpm->cp.size >> 16 & 255);
      if (srpm->cp.size >= (1 << 8))
	putchar(srpm->cp.size >> 8 & 255);
      putchar(srpm->cp.size & 255);
    }
  if (S_ISCHR(srpm->cp.mode) || S_ISBLK(srpm->cp.mode))
    dump32((int)srpm->cp.rdev);
  else if (srpm->cp.size)
    {
      if (srpm->fp == 0)
        rpmfsetmd5(srpm);
      for (ss = 0; ss < 16; ss++)
	putchar(srpm->cp.md5[ss]);
    }
}

void
dumphlascii(hl)
struct hardlink *hl;
{
  if (hl->next)
    dumphlascii(hl->next);
  printf("%s ->\n", hl->path);
}

void
dumpcpioascii(srpm)
struct srpm *srpm;
{
  int i;

  if (srpm->cp.next)
    dumphlascii(srpm->cp.next);
  printf("%s %o %s %s", srpm->cp.path, srpm->cp.mode, srpm->cp.owner, srpm->cp.group);
  if (S_ISCHR(srpm->cp.mode) || S_ISBLK(srpm->cp.mode))
    printf(" %x", (int)srpm->cp.rdev);
  if (S_ISREG(srpm->cp.mode) || S_ISLNK(srpm->cp.mode))
    {
      printf(" %u", (unsigned int)srpm->cp.size);
      if (srpm->cp.size)
	{
	  if (!srpm->fp)
            rpmfsetmd5(srpm);
	  putchar(' ');
	  for (i = 0; i < 16; i++)
	    {
	      putchar("0123456789abcdef"[srpm->cp.md5[i] >> 4 & 15]);
	      putchar("0123456789abcdef"[srpm->cp.md5[i]      & 15]);
	    }
	}
    }
  else if (srpm->cp.size)
    rpmeat(srpm);
  putchar('\n');
}


int
main(argc, argv)
int argc;
char **argv;
{
  int j, k;
  struct srpm *srpm;
  struct srpm srpms;
  int ascii = 0;

  srpm = &srpms;

  if (argc > 1 && strcmp(argv[1], "-a") == 0)
    {
      ascii = 1;
      argv++;
      argc--;
    }
  if (argc > 2)
    {
      fprintf(stderr, "usage: rpmgetlist [-a] rpm|rpmlist\n");
      exit(1);
    }
  memset(srpm, 0, sizeof(*srpm));
  if (argc > 1)
    {
      srpm->name = argv[1];
      srpm->ffd = open(srpm->name, O_RDONLY);
      if (srpm->ffd == -1)
	{
	  perror(srpm->name);
	  exit(1);
	}
    }
  else
    {
      srpm->name = "stdin";
      srpm->ffd = 0;
    }
  rpmopen(srpm);
  if (srpm->fp)
    rpmlgethead(srpm);
  else
    rpmfgethead(srpm);
  if (ascii)
    {
      char *ct;
      time_t bt;

      printf("//rpml//\n%s\n%s\n%s\n", srpm->rpmname, srpm->rpmevr, srpm->buildhost);
      bt = srpm->buildtime;
      ct = ctime(&bt);
      printf("%d (%.*s)\n", srpm->buildtime, (int)strlen(ct) - 1, ct);
      if (srpm->patchesCount)
	{
	  printf("//patches//\n");
	  for (j = 0; j < srpm->patchesCount; j++)
	    printf("%s\n", srpm->patchesNEVR[j]);
	  printf("//unpatched//\n");
	  for (j = 0; j < srpm->fileCount; j++)
	    if (srpm->fileFlags[j] & RPMFILE_UNPATCHED)
	      printf("%s\n", srpm->fileNames[j] + 1);
	}
    }
  else
    {
      putchar('R');
      putchar('P');
      putchar('M');
      putchar('L');
      k = strlen(srpm->rpmname);
      putchar(k);
      dumpstr(srpm->rpmname, k);
      k = strlen(srpm->rpmevr);
      putchar(k);
      dumpstr(srpm->rpmevr, k);
      k = strlen(srpm->buildhost);
      putchar(k);
      dumpstr(srpm->buildhost, k);
      dump32(srpm->buildtime);
      dump16(srpm->patchesCount);
      if (srpm->patchesCount)
	{
	  for (j = 0; j < srpm->patchesCount; j++)
	    {
	      k = strlen(srpm->patchesNEVR[j]);
	      putchar(k);
	      dumpstr(srpm->patchesNEVR[j], k);
	    }
	  k = 0;
	  for (j = 0; j < srpm->fileCount; j++)
	    if (srpm->fileFlags[j] & RPMFILE_UNPATCHED)
	      k++;
	  dump32(k);
	  for (j = 0; j < srpm->fileCount; j++)
	    if (srpm->fileFlags[j] & RPMFILE_UNPATCHED)
	      dumpfn(srpm->fileNames[j] + 1);
	}
    }
  if (ascii)
    printf("//cpiolist//\n");
  for (;;)
    {
      if (rpmgetent(srpm) == 0)
	break;
      if (srpm->fp == 0)
        rpmfmap(srpm);
      if (ascii)
        dumpcpioascii(srpm);
      else
        dumpcpio(srpm);
    }
  if (!ascii)
    dumpfn("");
  fflush(stdout);
  if (ferror(stdout))
    {
      fprintf(stderr, "rpmgetlist: write error\n");
      exit(1);
    }
  exit(0);
}
