/* Copyright (c) 2005 Michael Schroeder (mls@suse.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 ****************************************************************
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include "common.h"
#include <rpm/rpmpgp.h>

extern int noLibio;

extern Header rpmNewSignature(void);
extern int rpmWriteSignature(FD_t fd, Header header);

/* grrr, can't use headerCopy */
static Header headerMakeImmutable(Header h)
{
  Header nh = headerNew();
  HeaderIterator hi;
  int_32 tag, type, count;
  hPTR_t ptr;
  for (hi = headerInitIterator(h);
    headerNextIterator(hi, &tag, &type, &ptr, &count);
    ptr = headerFreeData((void *)ptr, type))
  {
    if (ptr) (void) headerAddEntry(nh, tag, type, ptr, count);
  }
  hi = headerFreeIterator(hi);
  return headerReload(nh, HEADER_IMMUTABLE);
}

static inline off_t saferead(FD_t cfd, void * vbuf, size_t amount)
{
    off_t rc = 0;
    char * buf = vbuf;

    while (amount > 0) {
        size_t nb;

        nb = Fread(buf, sizeof(buf[0]), amount, cfd);
        if (nb <= 0)
                return nb;
        rc += nb;
        if (rc >= amount)
                break;
        buf += nb;
        amount -= nb;
    }
    return rc;
}

static inline void ourwrite(FD_t fd, void * buf, size_t size)
{
  if (Fwrite(buf, 1, size, fd) != size)
    {
      fprintf(stderr, "write error\n");
      exit(1);
    }
}

void
cpiocpfile(in, out, needed, written)
FD_t in, out;
int needed;
int_32 *written;
{
  struct cpioCrcPhysicalHeader physHeader;
  int size, bite;
  char cpbuf[4096];
  char *end;

  if (saferead(in, &physHeader, sizeof physHeader) != sizeof physHeader) 
    {
      fprintf(stderr, "cpio copy: read failed\n");
      exit(1);
    }
  if (strncmp("070702", physHeader.magic, 6) &&
	strncmp("070701", physHeader.magic, 6))
    {
      fprintf(stderr, "cpio copy: bad cpio magic: %6.6s\n", physHeader.magic);
      exit(1);
    }
  if (needed)
    {
      ourwrite(out, &physHeader, sizeof physHeader);
      *written += sizeof physHeader;
    }
  size = strntoul(physHeader.namesize, &end, 16, sizeof(physHeader.namesize));
  size += -(size + sizeof(physHeader)) & 3;	/* pad */
  size += strntoul(physHeader.filesize, &end, 16, sizeof(physHeader.filesize));
  size += -(size + sizeof(physHeader)) & 3;	/* pad */
  while(size > 0)
    {
      bite = size > sizeof(cpbuf) ? sizeof(cpbuf) : size;
      if (saferead(in, cpbuf, bite) != bite)
	{
	  fprintf(stderr, "cpio copy: read error\n");
	  exit(1);
	}
      if (needed)
	{
          ourwrite(out, cpbuf, bite);
	  *written += bite;
	}
      size -= bite;
    }
}

void
cpiowriteeof(out, written)
FD_t out;
int_32 *written;
{
  static char trailer[14] = "TRAILER!!!";
  struct cpioCrcPhysicalHeader physHeader;
  memset(&physHeader, '0', sizeof(physHeader));
  memcpy(physHeader.magic, "070701", 6);
  memcpy(physHeader.nlink, "00000001", 8);
  memcpy(physHeader.namesize, "0000000b", 8);
  ourwrite(out, &physHeader, sizeof physHeader);
  *written += sizeof physHeader;
  /* pad 3 bytes */
  ourwrite(out, trailer, 11 + 3);
  *written += 11 + 3;
}

/*******************************************************************/

void
calcsigs(fd, hdrsize, size, md5, sha1)
FD_t fd;
off_t hdrsize;
int_32 *size;
unsigned char *md5;
unsigned char *sha1;
{
  unsigned char buf[4096];
  DIGEST_CTX ctx;
  DIGEST_CTX ctxsha;
  void *dig = 0;
  int n;

  memset(&ctx, 0, sizeof(ctx));
  ctx = rpmDigestInit(PGPHASHALGO_MD5, RPMDIGEST_NONE);
  memset(&ctxsha, 0, sizeof(ctxsha));
  ctxsha = rpmDigestInit(PGPHASHALGO_SHA1, RPMDIGEST_NONE);
  while ((n = Fread(buf, 1, sizeof(buf), fd)) > 0)
    {
      *size += n;
      rpmDigestUpdate(ctx, buf, n);
      if (n > hdrsize)
	n = hdrsize;
      if (n)
	{
	  rpmDigestUpdate(ctxsha, buf, n);
	  hdrsize -= n;
	}
    }
  rpmDigestFinal(ctx, &dig, 0, 0);
  memcpy(md5, dig, 16);
  free(dig);
  rpmDigestFinal(ctxsha, &dig, 0, 1);
  memcpy(sha1, dig, 20*2+1);
  free(dig);
  if (Ferror(fd))
    {
      fprintf(stderr, "calcsigs read error\n");
      exit(1);
    }
}

char *rpmlibneeded = "rpmlib(PatchRPMs)";
char *rpmlibneededEVR = "3.0.6-1";
int_32 rpmlibneededFlags = RPMSENSE_LESS|RPMSENSE_EQUAL;
int_32 sigsize;
int_32 archivesize;
unsigned char sigmd5[16];
unsigned char sigsha1[20*2+1] = "0123456789012345678901234567890123456789";

int
main(argc, argv)
int argc;
char **argv;
{
  int i, nrpms, rc, np;
  struct srpm *srpms, *srpm, *msrpm;
  int found, needed;
  FD_t ofd;
  FD_t ocfd;
  char *oname;
  unsigned char lead[RPMLEAD_SIZE];
  struct hardlink *hlink;
  char *cpiolist;
  int cpion;
  int cpioa;
  char **patches, **patchesEVR;
  int_32 *patchesFlags;
  Header dummyh;
  Header sig;
  off_t headerstart;
  off_t archivestart;
  int archsize2;
  int oldmode = 0;
  int nosha1 = 0;
  uint_32 *msrpmFileFlags;

  while (argc > 1)
    {
      if (!strcmp(argv[1], "-b"))
	{
	  oldmode = 1;
	  argc--;
	  argv++;
	}
      else if (!strcmp(argv[1], "-s"))
	{
	  nosha1 = 1;
	  argc--;
	  argv++;
	}
      else
	break;
    }
  if (argc < 4)
    {
      fprintf(stderr, "Usage: makepatchrpm [-b] [-s] old1 ... oldn new new.patch.rpm\n");
      exit(1);
    }
  noLibio = 1;
  nrpms = argc - 2;
  oname = argv[argc - 1];
  srpms = xcalloc(nrpms, sizeof(*srpms));
  cpion = 0;
  cpioa = 1000;
  cpiolist = xmalloc(cpioa);
  for (i = 0, srpm = srpms; i < nrpms; i++, srpm++)
    {
      srpm->name = argv[1 + i];
      srpm->ffd = open(srpm->name, O_RDONLY);
      if (srpm->ffd == -1)
	{
	  perror(srpm->name);
	  exit(1);
	}
      rpmopen(srpm);
      if (srpm->fp)
	rpmlgethead(srpm);
      else
	rpmfgethead(srpm);
    }

  /* check if there is a base file for all patches */
  for (i = 0, srpm = srpms; i < nrpms - 1; i++, srpm++)
    {
      int j, k;
      if (srpm->patchesCount == 0)
	continue;
      for (k = 0; k < srpm->patchesCount; k++)
	{
	  for (j = 0; j < nrpms - 1; j++)
	    if (i != j && srpms[j].patchesCount == 0 && !strcmp(srpm->patchesNEVR[k], srpms[j].rpmnevr))
	      break;
	  if (j < nrpms - 1)
	    break;
	}
      if (k == srpm->patchesCount)
	{
	  fprintf(stderr, "%s: need one of the base rpms\n", srpm->name);
	  exit(1);
	}
    }

  /* read files from all but the last */
  for (i = 0, srpm = srpms; i < nrpms - 1; i++, srpm++)
    rpmgetent(srpm);

  /* all files opened, process cpio archive of last rpm */
  msrpm = srpms + nrpms - 1;
  if (msrpm->fp)
    {
      fprintf(stderr, "%s: new rpm must not be a rpmlist\n", msrpm->name);
      exit(1);
    }
  if (msrpm->patchesCount)
    {
      fprintf(stderr, "%s: new rpm must not be a patch\n", msrpm->name);
      exit(1);
    }

  /* dup file flags so we can modify them */
  msrpmFileFlags = malloc(sizeof(*msrpm) * msrpm->fileCount);
  if (!msrpmFileFlags)
    {
      fprintf(stderr, "no mem for file flags\n");
      exit(1);
    }
  memcpy(msrpmFileFlags, msrpm->fileFlags, sizeof(*msrpm) * msrpm->fileCount);

  printf("comparing files...\n");
  for (;;)
    {
      if (rpmgetent(msrpm) == 0)
	break;
      found = 1;
      for (i = 0, srpm = srpms; i < nrpms - 1; i++, srpm++)
	{
	  for (;;)
	    {
	      rc = -1;
	      if (srpm->cp.eof)
		break;
	      rc = strcmp(msrpm->cp.path, srpm->cp.path);
	      if (rc <= 0)
		break;
	      rpmeat(srpm);
	      rpmgetent(srpm);
	    }
	  if (rc && srpm->patchesCount)
	    {
	      int j;
	      /* check filelist, ok if found (as we also check a base) */
	      for (j = 0; j < srpm->fileCount; j++)
		if (!strcmp(srpm->cp.path + (srpm->cp.path[0] == '.' && srpm->cp.path[1] == '/' ? 2 : 0), srpm->fileNames[j] + 1))
		  break;
	      if (j < srpm->fileCount && (srpm->fileFlags[j] & RPMFILE_UNPATCHED) != 0)
		rc = 0;
	    }
	  if (rc)
	    found = 0;
	}
      needed = 1;
      if (found)
	{
	  /* found in all rpms, have to check bytes */
	  needed = rpmcmp(srpms, nrpms) ? 1 : 0;
	  /* get next headers */
	  for (i = 0, srpm = srpms; i < nrpms - 1; i++, srpm++)
	    rpmgetent(srpm);
	}
      else
        rpmeat(msrpm);
      if (cpion + msrpm->cp.num > cpioa)
	{
	  cpioa = cpion + msrpm->cp.num + 1000;
	  cpiolist = realloc(cpiolist, cpioa);
	  if (!cpiolist)
	    {
	      fprintf(stderr, "Out of memory!\n");
	      exit(1);
	    }
	}
      for (i = 0; i < msrpm->cp.num; i++)
	cpiolist[cpion++] = needed;
      if (!needed)
	{
	  for(;;)
	    {
	      for (i = 0; i < msrpm->fileCount; i++)
		if (!strcmp(msrpm->fileNames[i] + 1, msrpm->cp.path + (msrpm->cp.path[0] == '.' && msrpm->cp.path[1] == '/' ? 2 : 0)))
		  break;
	      if (i == msrpm->fileCount)
		{
		  fprintf(stderr, "%s: %s not found in file list\n", msrpm->name, msrpm->cp.path);
		  exit(1);
		}
	      msrpmFileFlags[i] |= oldmode ? RPMFILE_UNPATCHED_OLD : RPMFILE_UNPATCHED;
	      if ((hlink = msrpm->cp.next) == 0)
		break;
	      free(msrpm->cp.path);
	      msrpm->cp.path = hlink->path;
	      msrpm->cp.next = hlink->next;
	      free(hlink);
	    }
	}
      else
	archivesize += msrpm->cp.bytes;
    }
  archivesize += sizeof(struct cpioCrcPhysicalHeader) + 14;

  /* check all rpms till eof */
  for (i = 0, srpm = srpms; i < nrpms - 1; i++, srpm++)
    {
      while (!srpm->cp.eof)
	{
	  rpmgetent(srpm);
	  rpmeat(srpm);
	}
    }

  patches = xcalloc(nrpms - 1, sizeof(char *));
  patchesEVR = xcalloc(nrpms - 1, sizeof(char *));
  patchesFlags = xcalloc(nrpms - 1, sizeof(int_32));
  np = 0;
  for (i = 0; i < nrpms - 1; i++)
    {
      int j;
      for (j = 0; j < np; j++)
	if (strcmp((char *)srpms[i].rpmname, patches[j]) == 0 && strcmp(srpms[i].rpmevr, patchesEVR[j]) == 0)
	  break;
      if (j < np)
	continue;
      patches[np] = (char *)srpms[i].rpmname;
      patchesEVR[np] = srpms[i].rpmevr;
      patchesFlags[np] = RPMSENSE_EQUAL;
      np++;
    }
  headerRemoveEntry(msrpm->h, RPMTAG_FILEFLAGS);
  headerAddEntry(msrpm->h, RPMTAG_FILEFLAGS, RPM_INT32_TYPE, msrpmFileFlags, msrpm->fileCount);
  headerAddEntry(msrpm->h, RPMTAG_ARCHIVESIZE, RPM_INT32_TYPE, &archivesize, 1);
  headerAddEntry(msrpm->h, RPMTAG_PATCHESNAME, RPM_STRING_ARRAY_TYPE, patches, np);
  headerAddEntry(msrpm->h, RPMTAG_PATCHESFLAGS, RPM_INT32_TYPE, patchesFlags, np);
  headerAddEntry(msrpm->h, RPMTAG_PATCHESVERSION, RPM_STRING_ARRAY_TYPE, patchesEVR, np);
  headerAddOrAppendEntry(msrpm->h, RPMTAG_REQUIRENAME, RPM_STRING_ARRAY_TYPE, &rpmlibneeded, 1);
  headerAddOrAppendEntry(msrpm->h, RPMTAG_REQUIREVERSION, RPM_STRING_ARRAY_TYPE, &rpmlibneededEVR, 1);
  headerAddOrAppendEntry(msrpm->h, RPMTAG_REQUIREFLAGS, RPM_INT32_TYPE, &rpmlibneededFlags, 1);
  headerRemoveEntry(msrpm->h, RPMTAG_ARCHIVESIZE);
#if 0
  /* rpm4 uses RPMSIGTAG_PAYLOADSIZE instead */
  headerAddEntry(msrpm->h, RPMTAG_ARCHIVESIZE, RPM_INT32_TYPE, &archivesize, 1);
#endif
  Fclose(msrpm->cfd);
  if (Fseek(msrpm->fd, 0, SEEK_SET) == -1)
    {
      fprintf(stderr, "%s: cannot rewind\n", msrpm->name);
      exit(1);
    }
  (void) Fflush(msrpm->fd);
  if (saferead(msrpm->fd, lead, sizeof(lead)) != sizeof(lead))
    {
      fprintf(stderr, "cannot read lead of %s\n", msrpm->name);
      exit(1);
    }
  Fseek(msrpm->fd, 0, SEEK_SET);
  (void) Fflush(msrpm->fd);
  dummyh = readPackageHeader(msrpm->fd);
  if (!dummyh)
    {
      fprintf(stderr, "cannot re-read header of %s\n", msrpm->name);
      exit(1);
    }
  (void) Fflush(msrpm->fd);
  msrpm->cfd = Fdopen(fdDup(Fileno(msrpm->fd)), msrpm->comp);

  /* pass 1: write rpm, leave room for sig */
  printf("writing patchrpm...\n");
  ofd = Fopen(oname, "w+.ufdio");
  if (ofd == 0 || Ferror(ofd))
    {
      fprintf(stderr, "cannot open %s: %s\n", oname, Fstrerror(ofd));
      exit(1);
    }
  if (Fwrite(lead, 1, sizeof(lead), ofd) != sizeof(lead))
    {
      fprintf(stderr, "%s: %s\n", oname, Fstrerror(ofd));
      exit(1);
    }
  sig = rpmNewSignature();
  headerAddEntry(sig, RPMSIGTAG_SIZE, RPM_INT32_TYPE, &sigsize, 1);
  headerAddEntry(sig, RPMSIGTAG_MD5, RPM_BIN_TYPE, sigmd5, 16);
  if (!nosha1)
    headerAddEntry(sig, RPMSIGTAG_SHA1, RPM_STRING_TYPE, sigsha1, 1);
  headerAddEntry(sig, RPMSIGTAG_PAYLOADSIZE, RPM_INT32_TYPE, &archivesize, 1);
  sig = headerReload(sig, RPMTAG_HEADERSIGNATURES);
  if ((rc = rpmWriteSignature(ofd, sig)) != 0)
    {
      fprintf(stderr, "%s: couldn't write signature\n", oname);
      exit(1);
    }
  sig = headerFree(sig);
  (void) Fflush(ofd);
  headerstart = lseek(Fileno(ofd), (off_t)0, SEEK_CUR);

  /* Uh oh, rpm-4 can't handle modifications in dribbles */
  /* Re-make header */
  
  msrpm->h = headerMakeImmutable(msrpm->h);

  if (headerWrite(ofd, msrpm->h, HEADER_MAGIC_YES))
    {
      fprintf(stderr, "%s: couldn't write header\n", oname);
      exit(1);
    }
  (void) Fflush(ofd);
  archivestart = lseek(Fileno(ofd), (off_t)0, SEEK_CUR);
  ocfd = Fdopen(fdDup(Fileno(ofd)), !strcmp(msrpm->comp, "r.bzdio") ? "w9.bzdio" : "w9.gzdio");
  archsize2 = 0;
  for (i = 0; i < cpion; i++)
    cpiocpfile(msrpm->cfd, ocfd, cpiolist[i], &archsize2);
  cpiowriteeof(ocfd, &archsize2);
  Fclose(msrpm->cfd);
  Fclose(msrpm->fd);
  Fclose(ocfd);
  if (archsize2 != archivesize)
    {
      fprintf(stderr, "internal error: archivesize mismatch %d %d\n", archsize2, archivesize);
      exit(1);
    }

  /* pass2: calculate and update signature */
  printf("fixing signatures...\n");
  if (Fseek(ofd, headerstart, SEEK_SET) == -1)
    {
      fprintf(stderr, "cannot rewind output file to header\n");
      exit(1);
    }
  (void) Fflush(ofd);
  calcsigs(ofd, (archivestart - headerstart), &sigsize, sigmd5, sigsha1);
  if (Fseek(ofd, sizeof(lead), SEEK_SET) == -1)
    {
      fprintf(stderr, "cannot rewind output file to signatures\n");
      exit(1);
    }
  (void) Fflush(ofd);
  sig = rpmNewSignature();
  headerAddEntry(sig, RPMSIGTAG_SIZE, RPM_INT32_TYPE, &sigsize, 1);
  headerAddEntry(sig, RPMSIGTAG_MD5, RPM_BIN_TYPE, sigmd5, 16);
  if (!nosha1)
    headerAddEntry(sig, RPMSIGTAG_SHA1, RPM_STRING_TYPE, sigsha1, 1);
  headerAddEntry(sig, RPMSIGTAG_PAYLOADSIZE, RPM_INT32_TYPE, &archivesize, 1);
  sig = headerReload(sig, RPMTAG_HEADERSIGNATURES);
  if ((rc = rpmWriteSignature(ofd, sig)) != 0)
    {
      fprintf(stderr, "%s: couldn't re-write signature\n", oname);
      exit(1);
    }
  Fclose(ofd);
  printf("done.\n");
  exit(0);
}
