/* Copyright (c) 2005 Michael Schroeder (mls@suse.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file COPYING); if not, write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 ****************************************************************
 */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <zlib.h>
#include "common.h"
#include <rpm/rpmpgp.h>

extern rpmRC readLead(FD_t fd, struct rpmlead *lead);
extern rpmRC rpmReadSignature(FD_t fd, Header *sighp, int sig_type, const char **msg);
extern void rpmfiBuildFNames(Header h, rpmTag tagN, const char ***fnp, int *fcp);


/****************************************************************
 *
 * utility functions
 *
 */

void *
xmalloc(int len)
{
  void *r = malloc(len ? len : 1);
  if (r)
    return r;
  fprintf(stderr, "Out of memory!\n");
  exit(1);
}

void *
xrealloc(void *old, int len)
{
  if (old == 0)
    old = malloc(len);
  else
    old = realloc(old, len);
  if (old)
    return old;
  fprintf(stderr, "Out of memory!\n");
  exit(1);
}


void *
xcalloc(int num, int len)
{
  void *r = calloc(num, len);
  if (r)
    return r;
  fprintf(stderr, "Out of memory!\n");
  exit(1);
}

int
strntoul(const char *str, char **endptr, int base, int num)
{
    char *buf, *end;
    unsigned long ret;

    buf = alloca(num + 1);
    strncpy(buf, str, num);
    buf[num] = '\0';

    ret = strtoul(buf, &end, base);
    if (*end)
        *endptr = ((char *)str) + (end - buf);
    else        
        *endptr = ((char *)str) + strlen(str);
    return strtoul(buf, endptr, base);
}


Header readPackageHeader(FD_t fd)
{
  struct rpmlead lead;
  Header h;

  readLead(fd, &lead);
  rpmReadSignature(fd, NULL, lead.signature_type, NULL);
  h = headerRead(fd, lead.major >= 3 ? HEADER_MAGIC_YES : HEADER_MAGIC_NO);
  return h;
}

/****************************************************************
 *
 * rpml stuff
 *
 */

static int
rpmlget16(srpm)
struct srpm *srpm;
{
  int r;
  r = getc(srpm->fp);
  r = r << 8 | getc(srpm->fp);
  return r;
}

static int
rpmlget32(srpm)
struct srpm *srpm;
{
  int r;
  r = getc(srpm->fp);
  r = r << 8 | getc(srpm->fp);
  r = r << 8 | getc(srpm->fp);
  r = r << 8 | getc(srpm->fp);
  return r;
}

static char *
rpmlgetstr(srpm)
struct srpm *srpm;
{
  int l;
  char *s, *s2;

  l = getc(srpm->fp);
  s = s2 = xmalloc(l + 2);
  while(l-- > 0)
    *s2++ = getc(srpm->fp);
  *s2 = 0;
  return s;
}

static char *
rpmlgetfn(srpm)
struct srpm *srpm;
{
  int ol, l;
  char *n;

  ol = getc(srpm->fp);
  if (ol == EOF)
    {
      fprintf(stderr, "%s: EOF reached!\n", srpm->name);
      exit(1);
    }
  l = getc(srpm->fp);
  if (l == 255)
    l = rpmlget16(srpm->fp);
  if (l + ol + 1 > srpm->lastfnl)
    {
      srpm->lastfn = xrealloc(srpm->lastfn, l + ol + 1);
      srpm->lastfnl = l + ol + 1;
    }
  n = srpm->lastfn + ol;
  while(l-- > 0)
    *n++ = getc(srpm->fp);
  *n = 0;
  return srpm->lastfn;
}

static void
rpmlgetog(srpm, s, l)
struct srpm *srpm;
char *s;
int l;
{
  while(l-- > 0)
    *s++ = getc(srpm->fp);
  *s = 0;
}


int
rpmlgetent(srpm)
struct srpm *srpm;
{
  struct cpioHeader *chPtr;
  int inlink = 0;
  struct hardlink *hlink;
  int s, ogs;

  chPtr = &srpm->cp;
  while ((hlink = chPtr->next))
    {
      chPtr->next = hlink->next;
      free(hlink->path);
      free(hlink);
    }
  chPtr->bytes = 0;
  srpm->bytesread = 0;
  chPtr->num = 0;
  chPtr->md5set = 0;
  if (chPtr->eof)
    return 0;
  chPtr->num = 1;
  for(;;)
    {
      chPtr->path = rpmlgetfn(srpm);
      if (*chPtr->path == 0)
	{
	  if (getc(srpm->fp) != EOF)
	    {
	      fprintf(stderr, "%s: trailing junk\n", srpm->name);
	      exit(1);
	    }
	  chPtr->eof = 1;
	  chPtr->num = 0;
	  if (inlink)
	    {
	      fprintf(stderr, "%s: missing hard link (trailer reached)\n", srpm->name);
	      exit(1);
	    }
	  return 0;
	}
      chPtr->mode = rpmlget16(srpm);
      chPtr->rdev = 0;
      chPtr->size = 0;
      if (chPtr->mode == 0)
	{
	  hlink = xmalloc(sizeof(*hlink));
	  hlink->next = chPtr->next;
	  hlink->path = strdup(chPtr->path);
	  chPtr->path = 0;
	  chPtr->next = hlink;
	  chPtr->num++;
	  inlink = 1;
	  continue;
	}
      ogs = getc(srpm->fp);
      if (ogs == 0xff)
	{
	  int ogs2;
	  ogs2 = getc(srpm->fp);
	  ogs = getc(srpm->fp);
	  if (ogs2)
	    rpmlgetog(srpm, srpm->lastown, ogs2 + 1);
	  if (ogs & 0xfc)
	    rpmlgetog(srpm, srpm->lastown, (ogs >> 2 & 0x3f) + 1);
	}
      else
	{
	  if (ogs & 0xe0)
	    rpmlgetog(srpm, srpm->lastown, (ogs >> 5 & 7) + 1);
	  if (ogs & 0x1c)
	    rpmlgetog(srpm, srpm->lastgrp, (ogs >> 2 & 7) + 1);
	}
      chPtr->owner = srpm->lastown;
      chPtr->group = srpm->lastgrp;
      if (S_ISCHR(chPtr->mode) || S_ISBLK(chPtr->mode))
	chPtr->rdev = rpmlget32(srpm);
      if (S_ISREG(chPtr->mode) || S_ISLNK(chPtr->mode))
	{
	  ogs &= 3;
	  s = 0;
	  if (ogs > 2)
	    s |= getc(srpm->fp) << 24;
	  if (ogs > 1)
	    s |= getc(srpm->fp) << 16;
	  if (ogs > 0)
	    s |= getc(srpm->fp) << 8;
	  s |= getc(srpm->fp);
	  chPtr->size = s;
	  if (s)
	    {
	      for (s = 0; s < 16; s++)
		chPtr->md5[s] = getc(srpm->fp);
	      chPtr->md5set = 1;
	    }
	}
      return 1;
    }
}

void
rpmlgethead(srpm)
struct srpm *srpm;
{
  int i;
  char *n;

  srpm->rpmname = rpmlgetstr(srpm);
  srpm->rpmevr = rpmlgetstr(srpm);
  srpm->rpmnevr = xmalloc(strlen(srpm->rpmname) + strlen(srpm->rpmevr) + 2);
  sprintf(srpm->rpmnevr, "%s-%s", srpm->rpmname, srpm->rpmevr);
  srpm->buildhost = rpmlgetstr(srpm);
  srpm->buildtime = rpmlget32(srpm);
  srpm->patchesCount = rpmlget16(srpm);
  srpm->fileCount = 0;
  srpm->fileNames = 0;
  srpm->fileFlags = 0;
  if (srpm->patchesCount)
    {
      srpm->patchesNEVR = (char **)xcalloc(srpm->patchesCount, sizeof(char *));
      for (i = 0; i < srpm->patchesCount; i++)
	srpm->patchesNEVR[i] = rpmlgetstr(srpm);
      srpm->fileCount = rpmlget32(srpm);
      srpm->fileNames = xmalloc(srpm->fileCount * sizeof(char *));
      srpm->fileFlags = xmalloc(srpm->fileCount * sizeof(uint_32));
      for (i = 0; i < srpm->fileCount; i++)
	{
	  n = rpmlgetfn(srpm);
	  srpm->fileNames[i] = xmalloc(strlen(n) + 2);
	  srpm->fileNames[i][0] = '/';
	  strcpy(srpm->fileNames[i] + 1, n);
	  srpm->fileFlags[i] = RPMFILE_UNPATCHED;
	}
    }
}



/****************************************************************
 *
 * rpmf stuff
 *
 */

#define ustrcmp(a, b) strcmp((char *)(a), (char *)(b))
#define ustrncmp(a, b, c) strncmp((char *)(a), (char *)(b), c)
#define ustrstr(a, b) ((unsigned char *)strstr((char *)(a), (char *)(b)))
#define ustrchr(a, b) ((unsigned char *)strchr((char *)(a), (b)))

#define GET_NUM_FIELD(phys, log) \
        log = strntoul(phys, &end, 16, sizeof(phys)); \
        if (*end) break;

static inline off_t saferead(FD_t cfd, void * vbuf, size_t amount)
{
    off_t rc = 0;
    char * buf = vbuf;

    while (amount > 0) {
        size_t nb;

        nb = Fread(buf, sizeof(buf[0]), amount, cfd);
        if (nb <= 0)
                return nb;
        rc += nb;
        if (rc >= amount)
                break;
        buf += nb;
        amount -= nb;
    }
    return rc;
}

static inline off_t ourread(FD_t cfd, void * buf, size_t size)
{
    off_t i = saferead(cfd, buf, size);
    return i;
}

static int
rpmfread(srpm, buf, len)
struct srpm *srpm;
unsigned char *buf;
int len;
{
  if (len > srpm->cp.size - srpm->bytesread)
    len = srpm->cp.size - srpm->bytesread;
  if (len <= 0)
    return 0;
  srpm->cpiopos += ourread(srpm->cfd, buf, len);
  srpm->bytesread += len;
  return len;
}

int
rpmfgetent(srpm)
struct srpm *srpm;
{
  struct cpioCrcPhysicalHeader physHeader;
  struct cpioHeader *chPtr;
  FD_t cfd = srpm->cfd;
  int nameSize;
  char *end;
  int major, minor;
  char padbuf[4];
  int inlink = 0, inlinkino = 0, inlinkdev = 0;
  struct hardlink *hlink;

  chPtr = &srpm->cp;
  while ((hlink = chPtr->next))
    {
      chPtr->next = hlink->next;
      free(hlink->path);
      free(hlink);
    }
  chPtr->bytes = 0;
  srpm->bytesread = 0;
  chPtr->num = 0;
  if (chPtr->eof)
    return 0;
  chPtr->num = 1;
  chPtr->md5set = 0;
  srpm->cpiopos += ourread(cfd, padbuf, -srpm->cpiopos & 3);
  for(;;)
    {
      if (ourread(cfd, &physHeader, sizeof physHeader) != sizeof physHeader) 
	{
	  fprintf(stderr, "%s: cpio read failed\n", srpm->name);
	  exit(1);
	}
      srpm->cpiopos += sizeof physHeader;
      chPtr->bytes += sizeof physHeader;
      if (strncmp("070702", physHeader.magic, 6) &&
	    strncmp("070701", physHeader.magic, 6))
	{
	  fprintf(stderr, "%s: bad cpio magic: %.6s at 0x%x\n", srpm->name, physHeader.magic, (unsigned int)(srpm->cpiopos - sizeof(physHeader)));
	  exit(1);
	}
      GET_NUM_FIELD(physHeader.inode, chPtr->inode);
      GET_NUM_FIELD(physHeader.mode, chPtr->mode);
      GET_NUM_FIELD(physHeader.uid, chPtr->uid);
      GET_NUM_FIELD(physHeader.gid, chPtr->gid);
      GET_NUM_FIELD(physHeader.nlink, chPtr->nlink);
      GET_NUM_FIELD(physHeader.mtime, chPtr->mtime);
      GET_NUM_FIELD(physHeader.filesize, chPtr->size);
      GET_NUM_FIELD(physHeader.devMajor, major);
      GET_NUM_FIELD(physHeader.devMinor, minor);
      chPtr->dev = makedev(major, minor);
      GET_NUM_FIELD(physHeader.rdevMajor, major);
      GET_NUM_FIELD(physHeader.rdevMinor, minor);
      chPtr->rdev = makedev(major, minor);
      GET_NUM_FIELD(physHeader.namesize, nameSize);
      if (chPtr->path)
	free(chPtr->path);
      chPtr->path = xmalloc(nameSize + 1);
      if (ourread(cfd, chPtr->path, nameSize) != nameSize) {
	free(chPtr->path);
	chPtr->path = NULL;
        fprintf(stderr, "%s: truncated cpio name", srpm->name);
        exit(1);
	break;
      }
      srpm->cpiopos += nameSize;
      chPtr->bytes += nameSize;
      if (!strcmp(chPtr->path, "TRAILER!!!")) {
	free(chPtr->path);
	chPtr->eof = 1;
        chPtr->num = 0;
        chPtr->bytes = 0;
	if (inlink)
	  {
	    fprintf(stderr, "%s: missing hard link (trailer reached)\n", srpm->name);
	    exit(1);
	  }
	return 0;
      }
      srpm->cpiopos += ourread(cfd, padbuf, -chPtr->bytes & 3);
      chPtr->bytes += -chPtr->bytes & 3;
      if (inlink && (chPtr->nlink != inlink || chPtr->inode != inlinkino || chPtr->dev != inlinkdev))
	{
	  fprintf(stderr, "%s: missing hard link\n", srpm->name);
	  exit(1);
	}
      if (S_ISCHR(chPtr->mode) && chPtr->size)
	{
	  fprintf(stderr, "WARNING: %s: %s: device with non-zero size\n", srpm->name, chPtr->path);
	  chPtr->size = 0;
	}
      if (chPtr->nlink < 2 || chPtr->size || !S_ISREG(chPtr->mode))
	{
          chPtr->bytes += chPtr->size;
          chPtr->bytes += -chPtr->bytes & 3;
	  return 1;
	}
      inlink = chPtr->nlink;
      inlinkino = chPtr->inode;
      inlinkdev = chPtr->dev;
      hlink = xmalloc(sizeof(*hlink));
      hlink->next = chPtr->next;
      hlink->path = chPtr->path;
      chPtr->path = 0;
      chPtr->next = hlink;
      chPtr->num++;
    }
  fprintf(stderr, "%s: bad cpio header\n", srpm->name);
  exit(1);
}

void
rpmfmap(srpm)
struct srpm *srpm;
{
  int i;

  for (i = 0; i < srpm->fileCount; i++)
    if (!strcmp(srpm->fileNames[i] + 1, srpm->cp.path + (srpm->cp.path[0] == '.' && srpm->cp.path[1] == '/' ? 2 : 0)))
      break;
  if (i == srpm->fileCount)
    {
      fprintf(stderr, "Warning: %s not found in file list\n", srpm->cp.path);
      srpm->cp.owner = xmalloc(6);
      srpm->cp.group = xmalloc(6);
      sprintf(srpm->cp.owner, "%d", srpm->cp.uid);
      sprintf(srpm->cp.group, "%d", srpm->cp.gid);
      return;
    }
  srpm->cp.owner = srpm->fileOwners[i];
  srpm->cp.group = srpm->fileGroups[i];
  srpm->cp.mode = srpm->fileModes[i];
}

#define SETMD_BUFSIZE 4096

struct gzdata {
  struct srpm *srpm;
  unsigned char *inbuf;
  z_stream stream;
  int err;
};

static void
rpmfgzreadinit(srpm, gz, buf, ctx)
struct srpm *srpm;
struct gzdata *gz;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int bite, method, flags, l;

  memset(gz, 0, sizeof(gz));
  gz->srpm = srpm;
  rpmfread(srpm, buf + srpm->bytesread, SETMD_BUFSIZE - srpm->bytesread);
  bite = srpm->bytesread;

  method = buf[2];
  flags = buf[3];
  rpmDigestUpdate(ctx, buf, 4);
  l = 2;
  if (flags & 0x04)	/* EXTRA_FIELD */
    l += 2 + buf[8 + l] + (buf[8 + l + 1] << 8);
  if (flags & 0x08)	/* ORIG_NAME */
    {
      while (8 + l < bite && buf[8 + l])
	l++;
      l++;
    }
  if (flags & 0x10)	/* COMMENT */
    {
      while (8 + l < bite && buf[8 + l])
	l++;
      l++;
    }
  if (flags & 0x02)	/* HEAD_CRC */
    l += 2;
  if (l > bite - 8)
    l = bite - 8;
  rpmDigestUpdate(ctx, buf + 8, l);
  gz->stream.zalloc = 0;
  gz->stream.zfree = 0;
  gz->stream.opaque = 0;
  gz->stream.next_in = buf + 8 + l;
  gz->stream.avail_in = bite - (8 + l);
  gz->stream.next_out = 0;
  gz->stream.avail_out = 0;
  gz->inbuf = buf;
  gz->err = inflateInit2(&gz->stream, -MAX_WBITS);
}

static void
rpmfgzreadfinit(gz, ctx)
struct gzdata *gz;
DIGEST_CTX ctx;
{
  int bite;
  unsigned char buf[8];

  /* skip crc32 and real len */
  if (gz->stream.avail_in < 8)
    rpmfread(gz->srpm, buf, 8 - gz->stream.avail_in);
  if (gz->stream.avail_in > 8)
    rpmDigestUpdate(ctx, gz->stream.next_in + 8, gz->stream.avail_in - 8);
  while (gz->srpm->bytesread != gz->srpm->cp.size)
    {
      bite = rpmfread(gz->srpm, gz->inbuf, SETMD_BUFSIZE);
      rpmDigestUpdate(ctx, gz->inbuf, bite);
    }
}

static int
rpmfgzread(gz, buf, len)
struct gzdata *gz;
unsigned char *buf;
int len;
{
  gz->stream.next_out = buf;
  gz->stream.avail_out = len;
  while (gz->stream.avail_out)
    {
      if (gz->stream.avail_in == 0)
	{
	  gz->stream.avail_in = rpmfread(gz->srpm, gz->inbuf, SETMD_BUFSIZE);
	  gz->stream.next_in = gz->inbuf;
	}
      gz->err = inflate(&gz->stream, Z_NO_FLUSH);
      if (gz->err != Z_OK)
	break;
    }
  return len - gz->stream.avail_out;
}

static void
rpmfsetmd_plain(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int bite;

  rpmDigestUpdate(ctx, buf, srpm->bytesread);
  while (srpm->bytesread < srpm->cp.size)
    {
      bite = rpmfread(srpm, buf, SETMD_BUFSIZE);
      rpmDigestUpdate(ctx, buf, bite);
    }
}

static void
rpmfsetmd_skip48(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int bite = srpm->bytesread;
  rpmDigestUpdate(ctx, buf, 4);
  rpmDigestUpdate(ctx, buf + 8, bite - 8);
  while (srpm->bytesread < srpm->cp.size)
    {
      bite = rpmfread(srpm, buf, SETMD_BUFSIZE);
      rpmDigestUpdate(ctx, buf, bite);
    }
}

static void
rpmfsetmd_pfb(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int i, bite;
  unsigned char *q, *r;

  rpmfread(srpm, buf + srpm->bytesread, SETMD_BUFSIZE - srpm->bytesread);
  bite = srpm->bytesread;
  buf[bite] = 0;
  if (bite > 24 && !ustrncmp(buf + 6, "%!PS-AdobeFont-1.0", 18) && (q = ustrchr(buf + 6, '\n')) != 0 && !ustrncmp(q + 1, "%%Creation Date:", 16) && (r = ustrchr(q + 17, '\n')) != 0)
    {
      i = q + 1 - buf;
      rpmDigestUpdate(ctx, buf, i);
      i = r + 1 - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else
    rpmDigestUpdate(ctx, buf, bite);
  while (srpm->bytesread < srpm->cp.size)
    {
      bite = rpmfread(srpm, buf, SETMD_BUFSIZE);
      rpmDigestUpdate(ctx, buf, bite);
    }
}

static void
rpmfsetmd_dvi(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int l, bite;

  rpmfread(srpm, buf + srpm->bytesread, SETMD_BUFSIZE - srpm->bytesread);
  if (srpm->bytesread >= 14)
    {
      rpmDigestUpdate(ctx, buf, 14);
      l = buf[14];
      if (srpm->bytesread - 15 - l > 0)
	rpmDigestUpdate(ctx, buf + 15 + l, srpm->bytesread - 15 - l);
    }
  else
    rpmDigestUpdate(ctx, buf, srpm->bytesread);
  while (srpm->bytesread < srpm->cp.size)
    {
      bite = rpmfread(srpm, buf, SETMD_BUFSIZE);
      rpmDigestUpdate(ctx, buf, bite);
    }
}

static void
rpmfsetmd_elc(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int i, j, bite;

  rpmfread(srpm, buf + srpm->bytesread, SETMD_BUFSIZE - srpm->bytesread);
  bite = srpm->bytesread;
  if (bite > 9 + 15 && (!ustrncmp(buf + 9, ";;; compiled by", 15) || !ustrncmp(buf + 9, ";;; Compiled by", 15)))
    {
      rpmDigestUpdate(ctx, buf, 9);
      for (i = 9 + 16; i < bite; i++)
	if (buf[i] == '\n')
	  break;
      if (i == bite)
	i = 9;
      else
	i++;
      if (i < bite)
        memmove(buf, buf + i, bite - i);
      bite -= i;
      if (bite > 14 && !ustrncmp(buf, ";;; from file ", 14))
	{
	  for (i = 14; i < bite; i++)
	    if (buf[i] == '\n')
	      break;
	  if (i < bite)
	    {
	      i++;
	      memmove(buf, buf + i, bite - i);
	      bite -= i;
	    }
	}
    }
  i = 0;
  for(;;)
    {
      if (i == bite)
	{
	  rpmDigestUpdate(ctx, buf, bite);
          bite = rpmfread(srpm, buf, SETMD_BUFSIZE);
	  if (bite == 0)
	    break;
	  i = 0;
	}
      if (buf[i] != '(' && buf[i] != '#')
	{
	  i++;
	  continue;
	}
      if (bite - i < 32 && srpm->bytesread < srpm->cp.size)
	{
	  if (i)
	    {
	      rpmDigestUpdate(ctx, buf, i);
	      memmove(buf, buf + i, bite - i);
	    }
	  i = bite - i;
          bite = rpmfread(srpm, buf + i, SETMD_BUFSIZE - i);
	  if (bite == 0)
	    {
	      rpmDigestUpdate(ctx, buf, i);
	      break;
	    }
	  bite += i;
	  i = 0;
	}
      if (bite - i > 3 && (!ustrncmp(buf + i, "#:", 2)))
	{
	  /* uninterned symbol, probably generated by gensym/gentemp.
	   * different with each compilation, maybe even in size! */
	  for (j = i + 2; j < bite; j++)
	    if (buf[j] != '-' && buf[j] != 'G' && (buf[j] < 'a' || buf[j] > 'z') && (buf[j] < '0' || buf[j] > '9'))
	      break;
	  if (j == bite || (buf[j - 1] < '0' && buf[j - 1] > '9') || (buf[j] != ' ' && buf[j] != ')' && buf[j] != ']'))
	    {
	      i++;		/* false alert */
	      continue;
	    }
	  if (i)
	    rpmDigestUpdate(ctx, buf, i);
	  i = j + 1;
	  if (i < bite)
	    memmove(buf, buf + i, bite - i);
	  bite -= i;
	  i = 0;
	  continue;
	}
      if (bite - i < 7 || ustrncmp(buf + i, "(#$ . ", 6))
	{
	  i++;
	  continue;
	}
      /* ignore docrefs, as they are too hard to handle, because
       * of gensym... */
      j = i + 6;
      if (j < bite && buf[j] == '-')
	j++;
      for (; j < bite; j++)
	if (!(buf[j] >= '0' && buf[j] <= '9'))
	  break;
      if (j + 1 >= bite || buf[j] != ')')
	{
	  i++;
	  continue;
	}
      if (i)
        rpmDigestUpdate(ctx, buf, i);
      i = j + 1;
      if (i < bite)
        memmove(buf, buf + i, bite - i);
      bite -= i;
      i = 0;
    }
}

static void
rpmfsetmd_ar(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int bite = srpm->bytesread;
  int i, armode, arsize;

  if (srpm->bytesread != 24)
    abort();
  rpmDigestUpdate(ctx, buf, bite);
  armode = 1;
  arsize = 12;
  while (srpm->bytesread < srpm->cp.size)
    {
      bite = SETMD_BUFSIZE;
      if (bite > arsize)
	bite = arsize;
      bite = rpmfread(srpm, buf, bite);
      arsize -= bite;
      if (arsize == 0)
	{
	  armode++;
	  if (armode == 2)
	    {
	      arsize = 6 + 6 + 8 + 10;
	      continue;
	    }
	  if (armode == 3)
	    {
	      for (i = 0; i < 10; i++)
		if (buf[6 + 6 + 8 + i] != ' ')
		  arsize = arsize * 10 + (buf[6 + 6 + 8 + i] - '0');
	      arsize += 2 + 1 + 16;
	      arsize &= ~1;
	    }
	  if (armode == 4)
	    {
	      arsize = 12;
	      armode = 1;
	    }
	}
      rpmDigestUpdate(ctx, buf, bite);
    }
}

static void
rpmfsetmd_perllocal(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int bite;
  int i = 0;

  rpmfread(srpm, buf + srpm->bytesread, SETMD_BUFSIZE - srpm->bytesread);
  bite = srpm->bytesread;
  if (bite > 31 && !ustrncmp(buf, "=head2 ", 7) && buf[31] == ':')
    {
      rpmDigestUpdate(ctx, buf, 7);
      memmove(buf, buf + 31, bite - 31);
      bite -= 31;
    }
  for (;;)
    {
      if (i == bite)
	{
	  rpmDigestUpdate(ctx, buf, bite);
	  bite = rpmfread(srpm, buf, SETMD_BUFSIZE);
	  if (bite == 0)
	    break;
	  i = 0;
	}
      if (buf[i] != '\n')
	{
	  i++;
	  continue;
	}
      if (bite - i < 33)
	{
	  if (i)
	    {
	      rpmDigestUpdate(ctx, buf, i);
	      memmove(buf, buf + i, bite - i);
	    }
	  i = bite - i;
          bite = rpmfread(srpm, buf + i, SETMD_BUFSIZE - i);
	  if (bite == 0)
	    {
	      rpmDigestUpdate(ctx, buf, i);
	      break;
	    }
	  bite += i;
	  i = 0;
	}
      if (ustrncmp(buf + i, "\n=head2 ", 8) || buf[i + 32] != ':')
	{
	  i++;
	  continue;
	}
      rpmDigestUpdate(ctx, buf, i + 8);
      i += 32;
      memmove(buf, buf + i, bite - i);
      bite -= i;
      i = 0;
    }
}

static void
rpmfsetmd_zip(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  int bite = srpm->bytesread;
  int i, needed = 0;

  i = 0;
  for (;;)
    {
      if (i == bite)
	{
	  rpmDigestUpdate(ctx, buf, bite);
	  bite = rpmfread(srpm, buf, SETMD_BUFSIZE);
	  if (bite == 0)
	    break;
	  i = 0;
	}
      if (buf[i] != 'P')
	{
	  i++;
	  continue;
	}
      if (needed < 4)
	needed = 4;
      if (bite - i < needed)
	{
	  if (i)
	    {
	      rpmDigestUpdate(ctx, buf, i);
	      memmove(buf, buf + i, bite - i);
	    }
	  i = bite - i;
          bite = rpmfread(srpm, buf + i, SETMD_BUFSIZE - i);
	  if (bite == 0)
	    {
	      rpmDigestUpdate(ctx, buf, i);
	      break;
	    }
	  bite += i;
	  i = 0;
	  continue;
	}
      if (buf[i + 1] != 'K' || (buf[i + 2] != 0x01 && buf[i + 2] != 0x03 && buf[i + 2] != 0x05 && buf[i + 2] != 0x07) || buf[i + 3] != buf[i + 2] + 1)
	{
	  i++;
	  continue;
	}
      if (buf[i + 2] == 0x01)
	{
	  if (needed != 46)
	    {
	      needed = 46;
	      continue;
	    }
	  rpmDigestUpdate(ctx, buf, i + 12);
	  rpmDigestUpdate(ctx, buf + i + 16, 30);
	}
      if (buf[i + 2] == 0x03)
	{
	  if (needed != 30)
	    {
	      needed = 30;
	      continue;
	    }
	  rpmDigestUpdate(ctx, buf, i + 10);
	  rpmDigestUpdate(ctx, buf + i + 14, 16);
	}
      if (buf[i + 2] == 0x05)
	{
	  if (needed != 22)
	    {
	      needed = 22;
	      continue;
	    }
	  rpmDigestUpdate(ctx, buf, i + 22);
	}
      if (buf[i + 2] == 0x07)
	{
	  if (needed != 16)
	    {
	      needed = 16;
	      continue;
	    }
	  rpmDigestUpdate(ctx, buf, i + 16);
	}
      i += needed;
      needed = 0;
      if (bite - i > 0)
        memmove(buf, buf + i, bite - i);
      bite -= i;
      i = 0;
    }
}

static void
rpmfsetmd_html(srpm, buf, ctx)
struct srpm *srpm;
unsigned char *buf;
DIGEST_CTX ctx;
{
  unsigned char *p, *q, *r;
  int bite, i, j;

  rpmfread(srpm, buf + srpm->bytesread, SETMD_BUFSIZE - srpm->bytesread);
  bite = srpm->bytesread;
  buf[bite] = 0;
  if ((p = ustrstr(buf, "\n<!-- Created on ")) != 0 && (q = ustrchr(p + 1, '\n')) != 0 && !ustrncmp(q - 4, " -->", 4))
    {
      i = p - buf + 17;
      rpmDigestUpdate(ctx, buf, i);
      i = q - 4 - buf;
      if (bite - i > 0)
        memmove(buf, buf + i, bite - i);
      bite -= i;
    }
  if ((p = ustrstr(buf, "\n<!-- This HTML file has been created by texi2html")) != 0 && (q = ustrchr(p + 1, '\n')) != 0 && !ustrncmp(q + 1, "     from ", 10))
    {
      i = p + 1 - buf;
      rpmDigestUpdate(ctx, buf, i);
      if ((q = ustrchr(q + 1, '\n')) != 0 && !ustrncmp(q - 4, " -->", 4))
	i = q + 1 - buf;
      if (bite - i > 0)
        memmove(buf, buf + i, bite - i);
      bite -= i;
    }
  while (srpm->bytesread < srpm->cp.size)
    {
      i = SETMD_BUFSIZE - (srpm->cp.size - srpm->bytesread);
      if (i > bite)
	i = bite;
      else if (i < bite)
	{
	  if (i < 0)
	    i = 0;
	  j = bite - i;
	  if (j > bite)
	    j = bite;
          rpmDigestUpdate(ctx, buf, j);
	  if (j < bite)
            memmove(buf, buf + j, bite - j);
	}
      bite = i + rpmfread(srpm, buf + i, SETMD_BUFSIZE - i);
    }
  buf[bite] = 0;
  if ((p = ustrstr(buf, "\nThis document was created by man2html\nusing the manual pages.<BR>\nTime:")) != 0 && (q = ustrchr(p + 73, '\n')) != 0)
    {
      i = p - buf + 73;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, "Documentation Created : ")) != 0 && (q = ustrchr(p + 24, '\n')) != 0)
    {
      i = p - buf + 24;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, "<hr><address><small>Generated on ")) != 0 && (q = ustrchr(p + 33, '\n')) != 0 && !ustrncmp(q - 3, " by", 3))
    {
      r =  ustrstr(p + 33, " for ");
      if (r && r < q)
        q = r;
      else
	q -= 3;
      i = p - buf + 33;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, "\nThis document was generated on ")) != 0 && (q = ustrchr(p + 32, '\n')) != 0 && !ustrncmp(q - 10, " using the", 10))
    {
      i = p - buf + 32;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf - 10;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, "\nThis document was generated\nby ")) != 0 && (q = ustrchr(p + 32, '\n')) != 0)
    {
      i = p - buf + 32;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, "</LI><LI><i>Generated</i>: ")) != 0 && (q = ustrchr(p + 27, '\n')) != 0 && q[1] == 0 && (r = ustrstr(p, ", using kdoc")) != 0 && r < q)
    {
      q = r;
      i = p - buf + 27;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, "\n\t<tr><td><small>Generated by: ")) != 0 && (q = ustrchr(p + 31, '\n')) != 0 && (r = ustrstr(p, ", using kdoc")) != 0 && r < q)
    {
      q = r;
      i = p - buf + 31;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, " - Zuletzt generiert: ")) != 0 && (q = ustrchr(p + 22, '\n')) != 0 && (ustrncmp(q, "\n (sdb_gen", 10) || (q = ustrchr(q + 1, '\n')) != 0) &&  (!ustrcmp(q, "\n</ADDRESS>\n") || !ustrcmp(q, "\n</ADDRESS>\n</TD></TR></TABLE>\n</body>\n</html>\n")))
    {
      i = p - buf + 22;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, " - Last generated: ")) != 0 && (q = ustrchr(p + 19, '\n')) != 0 && (ustrncmp(q, "\n (sdb_gen", 10) || (q = ustrchr(q + 1, '\n')) != 0) && (!ustrcmp(q, "\n</ADDRESS>\n") || !ustrcmp(q, "\n</ADDRESS>\n</TD></TR></TABLE>\n</body>\n</html>\n")))
    {
      i = p - buf + 19;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, " - Dernire modification: ")) != 0 && (q = ustrchr(p + 26, '\n')) != 0 && (ustrncmp(q, "\n (sdb_gen", 10) || (q = ustrchr(q + 1, '\n')) != 0) && (!ustrcmp(q, "\n</ADDRESS>\n") || !ustrcmp(q, "\n</ADDRESS>\n</TD></TR></TABLE>\n</body>\n</html>\n")))
    {
      i = p - buf + 26;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if ((p = ustrstr(buf, " - Utoljra frisstve: ")) != 0 && (q = ustrchr(p + 23, '\n')) != 0 && (ustrncmp(q, "\n (sdb_gen", 10) || (q = ustrchr(q + 1, '\n')) != 0) && (!ustrcmp(q, "\n</ADDRESS>\n") || !ustrcmp(q, "\n</ADDRESS>\n</TD></TR></TABLE>\n</body>\n</html>\n")))
    {
      i = p - buf + 23;
      rpmDigestUpdate(ctx, buf, i);
      i = q - buf;
      rpmDigestUpdate(ctx, buf + i, bite - i);
    }
  else if (bite > 56 && !ustrcmp("\n</ADDRESS>\n</BODY>\n</HTML>\n", buf + bite - 28) && (p = ustrstr(buf + bite - 56, "<ADDRESS>")) != 0)
    {
      i = p - buf + 10;
      rpmDigestUpdate(ctx, buf, i);
      rpmDigestUpdate(ctx, buf + bite - 56, 56);
    }
  else
    {
      rpmDigestUpdate(ctx, buf, bite);
    }
}

static void
rpmfsetmd_ppd_gz(srpm, gzbuf, ctx)
struct srpm *srpm;
unsigned char *gzbuf;
DIGEST_CTX ctx;
{
  unsigned char buf[SETMD_BUFSIZE + 1], *p;
  struct gzdata gz;
  int i, bite;

  rpmfgzreadinit(srpm, &gz, gzbuf, ctx);
  i = bite = 0;
  for (;;)
    {
      if (i == bite)
	{
	  rpmDigestUpdate(ctx, buf, bite);
	  bite = rpmfgzread(&gz, buf, SETMD_BUFSIZE);
	  if (bite == 0)
	    break;
	  i = 0;
	}
      if (buf[i] != '\n')
	{
	  i++;
	  continue;
	}
      if (bite - i < 61)
	{
	  if (i)
	    {
	      rpmDigestUpdate(ctx, buf, i);
	      memmove(buf, buf + i, bite - i);
	    }
	  i = bite - i;
	  bite = rpmfgzread(&gz, buf + i, SETMD_BUFSIZE - i);
	  if (bite == 0)
	    {
	      rpmDigestUpdate(ctx, buf, i);
	      break;
	    }
	  bite += i;
	  i = 0;
	  continue;
	}
      buf[bite] = 0;
      if (!ustrncmp(buf + i, "\n*% COMDATA #  'compiled-at' => '", 33) && buf[i + 59] == '\n' && !ustrncmp(buf + i + 57, "',", 2))
	{
	  rpmDigestUpdate(ctx, buf, i + 33);
	  i += 33 + 24;
	  memmove(buf, buf + i, bite - i);
	  bite -= i;
	  i = 0;
	  continue;
	}
      if (!ustrncmp(buf + i, "\n*% COMDATA #  'timestamp' => ", 30) && (p = ustrchr(buf + i + 1, '\n')) != 0)
	{
	  rpmDigestUpdate(ctx, buf, i + 30);
	  i = p - buf;
	  memmove(buf, buf + i, bite - i);
	  bite -= i;
	  i = 0;
	  continue;
	}
      if (!ustrncmp(buf + i, "\n*% COMDATA #  'compiled-by' => '", 33) && (p = ustrchr(buf + i + 1, '\n')) != 0 && !ustrncmp(p - 2, "',", 2))
	{
	  rpmDigestUpdate(ctx, buf, i + 33);
	  i = p - buf - 2;
	  memmove(buf, buf + i, bite - i);
	  bite -= i;
	  i = 0;
	  continue;
	}
      i++;
    }
  rpmfgzreadfinit(&gz, ctx);
}

void
rpmfsetmd5(srpm)
struct srpm *srpm;
{
  DIGEST_CTX ctx;
  unsigned char buf[SETMD_BUFSIZE + 1];	/* +1 for setmd5_html */
  void *dig = 0;
  int bite, l;

  l = strlen(srpm->cp.path);
  ctx = rpmDigestInit(PGPHASHALGO_MD5, RPMDIGEST_NONE);

  if (srpm->bytesread)
    abort();
  bite = rpmfread(srpm, buf, 24);
  if (bite > 8 && S_ISREG(srpm->cp.mode))
    {
      if (buf[0] == 0x1f && buf[1] == 0x8b && l > 7 && !strcmp(".ppd.gz", srpm->cp.path + l - 7))
	rpmfsetmd_ppd_gz(srpm, buf, ctx);
      else if (buf[0] == 0x1f && buf[1] == 0x8b && l > 2 && !strcmp("gz", srpm->cp.path + l - 2))
	rpmfsetmd_skip48(srpm, buf, ctx);
      else if (buf[0] == 0x2d && buf[1] == 0xed && l > 4 && (!strcmp(".pyo", srpm->cp.path + l - 4) || !strcmp(".pyc", srpm->cp.path + l - 4)))
	rpmfsetmd_skip48(srpm, buf, ctx);
      else if (buf[0] == 0xf7 && buf[1] == 0x02 && l > 4 && !strcmp(".dvi", srpm->cp.path + l - 4))
	rpmfsetmd_dvi(srpm, buf, ctx);
      else if (buf[0] == 0x50 && buf[1] == 0x4b && l > 4 && (!strcmp(".zip", srpm->cp.path + l - 4) || !strcmp(".jar", srpm->cp.path + l - 4) || !strcmp(".htb", srpm->cp.path + l - 4)))
	rpmfsetmd_zip(srpm, buf, ctx);
      else if (bite == 24 && !ustrncmp(buf, "!<arch>\n", 8) && l > 2 && !strcmp(".a", srpm->cp.path + l - 2))
	rpmfsetmd_ar(srpm, buf, ctx);
      else if (buf[0] == 0x3b && buf[1] == 0x45 && l > 4 && !strcmp(".elc", srpm->cp.path + l - 4))
	rpmfsetmd_elc(srpm, buf, ctx);
      else if (buf[0] == 0x80 && buf[1] == 0x01 && l > 4 && !strcmp(".pfb", srpm->cp.path + l - 4))
	rpmfsetmd_pfb(srpm, buf, ctx);
      else if (l > 4 && (!strcmp(".htm", srpm->cp.path + l - 4) || !strcmp(".html", srpm->cp.path + l - 5)))
	rpmfsetmd_html(srpm, buf, ctx);
      else if (!strncmp(srpm->cp.path, "var/adm/perl-modules/", 21))
	rpmfsetmd_perllocal(srpm, buf, ctx);
      else
	rpmfsetmd_plain(srpm, buf, ctx);
    }
  else
    rpmfsetmd_plain(srpm, buf, ctx);
  rpmDigestFinal(ctx, &dig, 0, 0);
  memcpy(srpm->cp.md5, dig, 16);
  free(dig);
  srpm->cp.md5set = 1;
}

void
rpmfeat(srpm)
struct srpm *srpm;
{
  unsigned char buf[4096];
  
  while (srpm->bytesread < srpm->cp.size)
    rpmfread(srpm, buf, sizeof(buf));
}

void
rpmfgethead(srpm)
struct srpm *srpm;
{
  uint_32 *buildTime;
  int_32 *epoch;
  int i;
  char **pName, **pEVR;
  const char *version, *release;

  srpm->h = readPackageHeader(srpm->fd);
  if (!srpm->h)
    {
      fprintf(stderr, "cannot read header of %s\n", srpm->name);
      exit(1);
    }
  (void) Fflush(srpm->fd);
  if (!headerGetEntry(srpm->h, RPMTAG_PAYLOADCOMPRESSOR, 0, (void *)&srpm->comp, 0))
    srpm->comp = "gzip";
  if (!strcmp(srpm->comp, "gzip"))
    srpm->comp = "r.gzdio";
  else if (!strcmp(srpm->comp, "bzip2"))
    srpm->comp = "r.bzdio";
  else
    {
      fprintf(stderr, "Unknown payloadcompressor\n");
      exit(1);
    }
  if (headerIsEntry(srpm->h, RPMTAG_OLDFILENAMES))
    {
      if (!headerGetEntry(srpm->h, RPMTAG_OLDFILENAMES, NULL, (void *)&srpm->fileNames, &srpm->fileCount))
	{
	  fprintf(stderr, "WARNING: %s: no file list\n", srpm->name);
	  srpm->fileCount = 0;
	}
    }
  else
    {
      rpmfiBuildFNames(srpm->h, RPMTAG_BASENAMES, (const char ***)&srpm->fileNames, &srpm->fileCount);
      if (!srpm->fileNames || srpm->fileCount <= 0)
	{
	  fprintf(stderr, "WARNING: %s: no file list\n", srpm->name);
	  srpm->fileCount = 0;
	}
    }
#if 0
  if (!headerGetEntry(srpm->h, RPMTAG_OLDFILENAMES, NULL, (void **)&srpm->fileNames, &srpm->fileCount))
    {
      fprintf(stderr, "WARNING: %s: no file list\n", srpm->name);
      srpm->fileCount = 0;
    }
#endif
  headerGetEntry(srpm->h, RPMTAG_FILEFLAGS, NULL, (void *)&srpm->fileFlags, NULL);
  headerGetEntry(srpm->h, RPMTAG_FILEUSERNAME, NULL, (void *)&srpm->fileOwners, NULL);
  headerGetEntry(srpm->h, RPMTAG_FILEGROUPNAME, NULL, (void *)&srpm->fileGroups, NULL);
  headerGetEntry(srpm->h, RPMTAG_FILEMODES, NULL, (void *)&srpm->fileModes, NULL);
  headerGetEntry(srpm->h, RPMTAG_BUILDHOST, NULL, (void *)&srpm->buildhost, NULL);
  headerGetEntry(srpm->h, RPMTAG_BUILDTIME, NULL, (void *)&buildTime, NULL);
  srpm->buildtime = *buildTime;
  if (!headerGetEntry(srpm->h, RPMTAG_PATCHESNAME, NULL, (void *)&pName, &srpm->patchesCount))
    srpm->patchesCount = 0;
  if (srpm->patchesCount)
    {
      headerGetEntry(srpm->h, RPMTAG_PATCHESVERSION, NULL, (void *)&pEVR, 0);
      srpm->patchesNEVR = (char **)xcalloc(srpm->patchesCount, sizeof(char *));
      for (i = 0; i < srpm->patchesCount; i++)
	{
	  srpm->patchesNEVR[i] = xmalloc(strlen(pName[i]) + strlen(pEVR[i]) + 2);
	  sprintf(srpm->patchesNEVR[i], "%s-%s", pName[i], pEVR[i]);
	}
    }
  headerNVR(srpm->h, &srpm->rpmname, &version, &release);
  srpm->rpmevr = xmalloc(strlen(version) + strlen(release) + 13);
  if (headerGetEntry(srpm->h, RPMTAG_EPOCH, NULL, (void *)&epoch, NULL))
    sprintf(srpm->rpmevr, "%d:%s-%s", *epoch, version, release);
  else
    sprintf(srpm->rpmevr, "%s-%s", version, release);
  srpm->rpmnevr = xmalloc(strlen(srpm->rpmname) + strlen(srpm->rpmevr) + 2);
  sprintf(srpm->rpmnevr, "%s-%s", srpm->rpmname, srpm->rpmevr);
  srpm->cfd = Fdopen(fdDup(Fileno(srpm->fd)), srpm->comp);
  srpm->cpiopos = 0;
  for (i = 0; i < srpm->fileCount; i++)
    if (srpm->fileFlags[i] & RPMFILE_UNPATCHED_OLD)
      {
	srpm->fileFlags[i] ^= RPMFILE_UNPATCHED_OLD;
	srpm->fileFlags[i] |= RPMFILE_UNPATCHED;
      }
}


/****************************************************************
 *
 * other stuff
 *
 */

int
rpmgetent(srpm)
struct srpm *srpm;
{
  if (srpm->fp)
    return rpmlgetent(srpm);
  else
    return rpmfgetent(srpm);
}

void
rpmeat(srpm)
struct srpm *srpm;
{
  if (srpm->fp == 0)
    rpmfeat(srpm);
}

void
rpmopen(srpm)
struct srpm *srpm;
{
  unsigned char le[4];
  int l;

  if ((l = read(srpm->ffd, (char *)le, 4)) != 4)
    {
      if (l == -1)
        perror(srpm->name);
      else
        fprintf(stderr, "%s: not a rpm or rpmlist\n", srpm->name);
      exit(1);
    }
  if (le[0] == 'R' && le[1] == 'P' && le[2] == 'M' && le[3] == 'L') 
    {
      if ((srpm->fp = fdopen(srpm->ffd, "r")) == 0)
	{
	  perror("fdopen");
	  exit(1);
	}
    }
  else if (le[0] == 0xed && le[1] == 0xab && le[2] == 0xee && le[3] == 0xdb)
    {
      if (lseek(srpm->ffd, (off_t)0, SEEK_SET) == -1) 
	{
	  perror("rewind");
	  exit(1);
	}
      srpm->fd = Fdopen(fdDup(srpm->ffd), "r.ufdio");
    }
  else
    {
      fprintf(stderr, "%s: not a rpm or rpmlist\n", srpm->name);
      exit(1);
    }
}

static int
hlinkcmp(h1, h2)
struct hardlink *h1, *h2;
{
  for (;;)
    {
      if (h1 == 0 && h2 == 0)
        return 0;
      if ((h1 == 0 && h2) || (h2 == 0 && h1)) 
        return 1;
      if (strcmp(h1->path, h2->path))
        return 1;
      h1 = h1->next;
      h2 = h2->next;
    }
}

int
rpmcmp(srpms, nrpms)
struct srpm *srpms;
int nrpms;
{
  struct srpm *srpm, *msrpm;
  int i, l;

  msrpm = srpms + nrpms - 1;
  if (msrpm->fp == 0)
    rpmfmap(msrpm);

  for (i = 0, srpm = srpms; i < nrpms - 1; i++, srpm++)
    {
      if (srpm->cp.size != msrpm->cp.size)
	{
	  if (!srpm->cp.size || !msrpm->cp.size)
	    break;
	  l = strlen(srpm->cp.path);
	  if (l < 5 || !S_ISREG(srpm->cp.mode) || (strcmp(srpm->cp.path + l - 5, ".html") && strcmp(srpm->cp.path + l - 4, ".htm") && strcmp(srpm->cp.path + l - 4, ".elc")))
	    if (l < 7 || strcmp(srpm->cp.path + l - 7, ".ppd.gz"))
	      break;
	}
      if (srpm->cp.num != msrpm->cp.num)
	break;
      if (srpm->fp == 0)
	rpmfmap(srpm);
      if (srpm->cp.mode != msrpm->cp.mode)
	break;
      if (S_ISCHR(srpm->cp.mode) || S_ISBLK(srpm->cp.mode))
	if (srpm->cp.rdev != msrpm->cp.rdev)
	  break;
      if (strcmp(srpm->cp.owner, msrpm->cp.owner) || strcmp(srpm->cp.group, msrpm->cp.group))
	break;
      if (srpm->cp.num > 1 && hlinkcmp(srpm->cp.next, msrpm->cp.next))
	break;
    }
  if (i != nrpms - 1)
    {
      for (i = 0, srpm = srpms; i < nrpms; i++, srpm++)
	rpmeat(srpm);
      return 1;
    }
  if (msrpm->cp.size == 0)
    return 0;
  for (i = 0, srpm = srpms; i < nrpms; i++, srpm++)
    if (srpm->fp == 0)
      rpmfsetmd5(srpm);
  for (i = 0, srpm = srpms; i < nrpms - 1; i++, srpm++)
    if (memcmp(srpm->cp.md5, msrpm->cp.md5, 16))
      return 2;
  return 0;
}
