/*
 * BRLTTY - Access software for Unix for a blind person
 *          using a soft Braille terminal
 *
 * Copyright (C) 1995-1998 by The BRLTTY Team, All rights reserved.
 *
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * BRLTTY comes with ABSOLUTELY NO WARRANTY.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * This software is maintained by Nicolas Pitre <nico@cam.org>.
 */

/* misc.c - Miscellaneous all-purpose routines
 */

#define __EXTENSIONS__		/* for time.h */

#include <stdio.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>

#include "config.h"
#include "misc.h"

#ifdef USE_SYSLOG
#include <syslog.h>
#include <stdarg.h>
#endif

unsigned elapsed_msec (struct timeval *t1, struct timeval *t2)
{
  unsigned diff, error = 0xFFFFFFFF;

  if (t1->tv_sec > t2->tv_sec)
    return (error);
  diff = (t2->tv_sec - t1->tv_sec) * 1000L;
  if (diff == 0 && t1->tv_usec > t2->tv_usec)
    return (error);
  diff += (t2->tv_usec - t1->tv_usec) / 1000L;
  return (diff);
}

void shortdelay (unsigned msec)
{
  struct timeval start, now;
  struct timezone tz;

  gettimeofday (&start, &tz);
  do
   {
     gettimeofday (&now, &tz);
   }
  while (elapsed_msec (&start, &now) < msec);
}

void delay (int msec)
{
  struct timeval del;

  del.tv_sec = 0;
  del.tv_usec = msec * 1000;
  select (0, NULL, NULL, NULL, &del);
}

int timeout_yet (int msec)
{
  static struct timeval tstart = { 0, 0 };
  struct timeval tnow;

  if (msec == 0)		/* initialiseation */
   {
     gettimeofday (&tstart, NULL);
     return 0;
   }
  gettimeofday (&tnow, NULL);
  return ((tnow.tv_sec * 1e6 + tnow.tv_usec) -
	  (tstart.tv_sec * 1e6 + tstart.tv_usec) >= msec * 1e3);
}

#ifdef USE_SYSLOG

static int LogPrio, LogOpened = 0;

void LogOpen (int prio)
{
  openlog ("BRLTTY", LOG_CONS, LOG_DAEMON);
  LogPrio = prio;
  LogOpened = 1;
}

void LogClose ()
{
  closelog ();
  LogOpened = 0;
}

void LogPrint (int prio, char *fmt, ...)
{
  va_list argp;

  if (LogOpened && (prio & LOG_PRIMASK) <= LogPrio)
   {
     va_start (argp, fmt);
     vsyslog (prio, fmt, argp);
     va_end (argp);
   }
}

void LogAndStderr (int prio, char *fmt, ...)
{
  va_list argp;

  va_start (argp, fmt);

  vfprintf (stderr, fmt, argp);
  fprintf (stderr, "\n");
  if (LogOpened && (prio & LOG_PRIMASK) <= LogPrio)
    vsyslog (prio, fmt, argp);

  va_end (argp);
}

#else /* don't USE_SYSLOG */

void LogOpen (int prio)
{
}
void LogClose ()
{
}
void LogPrint (int prio, char *fmt, ...)
{
}
void LogAndStderr (int prio, char *fmt, ...)
{
}
#endif

static void noMemory (void)
{
  LogPrint (LOG_CRIT, "Insufficient memory: %s", strerror (errno));
  exit (3);
}

void *mallocWrapper (size_t size)
{
  void *address = malloc (size);

  if (!address)
    noMemory ();
  return address;
}

void *reallocWrapper (void *address, size_t size)
{
  if (!(address = realloc (address, size)))
    noMemory ();
  return address;
}

char *strdupWrapper (const char *string)
{
  char *address = strdup (string);

  if (!address)
    noMemory ();
  return address;
}

char *makePath (const char *directory, const char *file)
{
  const int count = 3;
  const char *components[count];
  int lengths[count];
  const int last = count - 1;
  int first = last;
  int length = 0;
  int index;
  char *path;

  components[last] = file;
  if (file[0] != '/')
   {
     if (directory && *directory)
      {
	if (directory[strlen (directory) - 1] != '/')
	  components[--first] = "/";
	components[--first] = directory;
      }
   }
  for (index = first; index <= last; ++index)
   {
     length += lengths[index] = strlen (components[index]);
   }
  path = mallocWrapper (length + 1);
  {
    char *target = path;

    for (index = first; index <= last; ++index)
     {
       length = lengths[index];
       memcpy (target, components[index], length);
       target += length;
     }
    *target = 0;
  }
  return path;
}

int makeDirectory (const char *path)
{
  struct stat status;

  if (stat (path, &status) != -1)
   {
     if (S_ISDIR (status.st_mode))
       return 1;
     LogPrint (LOG_ERR, "Not a directory: %s", path);
   }
  else if (errno != ENOENT)
   {
     LogPrint (LOG_ERR, "Directory status error: %s: %s", path,
	       strerror (errno));
   }
  else
   {
     LogPrint (LOG_NOTICE, "Creating directory: %s", path);
     if (mkdir (path, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH) != -1)
       return 1;
     LogPrint (LOG_ERR, "Directory creation error: %s: %s", path,
	       strerror (errno));
   }
  return 0;
}

char *getDevicePath (const char *path)
{
  const char *prefix = DEVICE_DIRECTORY;
  const unsigned int prefixLength = strlen (prefix);
  const unsigned int pathLength = strlen (path);

  if (prefixLength)
   {
     if (*path != '/')
      {
	char buffer[prefixLength + 1 + pathLength + 1];
	unsigned int length = 0;

	memcpy (&buffer[length], prefix, prefixLength);
	length += prefixLength;

	if (buffer[length - 1] != '/')
	  buffer[length++] = '/';

	memcpy (&buffer[length], path, pathLength);
	length += pathLength;

	buffer[length] = 0;
	return strdupWrapper (buffer);
      }
   }
  return strdupWrapper (path);
}

int isQualifiedDevice (const char **path, const char *qualifier)
{
  size_t count = strcspn (*path, ":");

  if (count == strlen (*path))
    return 0;
  if (!qualifier)
    return 1;
  if (!count)
    return 0;

  {
    int ok = strncmp (*path, qualifier, count) == 0;

    if (ok)
      *path += count + 1;
    return ok;
  }
}

void unsupportedDevice (const char *path)
{
  LogPrint (LOG_WARNING, "Unsupported device: %s", path);
}
