/*
 * BRLTTY - Access software for Unix for a blind person
 *          using a soft Braille terminal
 *
 * Copyright (C) 1995-1998 by The BRLTTY Team, All rights reserved.
 *
 * Nicolas Pitre <nico@cam.org>
 * Stphane Doyon <s.doyon@videotron.ca>
 * Nikhil Nair <nn201@cus.cam.ac.uk>
 *
 * BRLTTY comes with ABSOLUTELY NO WARRANTY.
 *
 * This is free software, placed under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation.  Please see the file COPYING for details.
 *
 * This software is maintained by Nicolas Pitre <nico@cam.org>.
 */

/* cut-n-paste.c - cut & paste functionality
 */

#include <stdlib.h>
#include <string.h>
#include "scr.h"
#include "inskey.h"
#include "beeps.h"

/* prototypes */
static void cut (void);

/* Global state variables */
static char *cut_buffer = NULL;
static short cut_begx = 0, cut_begy = 0, cut_endx = 0, cut_endy = 0;

void cut_begin (int x, int y)
{
  if (cut_buffer)
   {
     free (cut_buffer);
     cut_buffer = NULL;
   }
  cut_begx = x;
  cut_begy = y;
  play (snd_cut_beg);
}

void cut_end (int x, int y)
{
  cut_endx = x;
  cut_endy = y;
  cut ();
}

void cut_paste ()
{
  if (cut_buffer)
    inskey ((unsigned char *) cut_buffer);
}

static void cut (void)
{
  short cols, rows;

  cols = cut_endx - cut_begx + 1;
  rows = cut_endy - cut_begy + 1;
  if (cols >= 1 && rows >= 1)
   {
     char *srcbuf, *dstbuf, *srcptr, *dstptr;
     short r, c, spaces;

     srcbuf = (char *) malloc (rows * cols);
     dstbuf = (char *) malloc (rows * cols + rows);
     if (srcbuf && dstbuf)
      {
	/* grab it */
	getscr ((winpos)
		{
		cut_begx, cut_begy, cols, rows}
		, (unsigned char *) srcbuf, SCR_TEXT);
	srcptr = srcbuf;
	dstptr = dstbuf;
	/* remove spaces at end of line, add return (except to last line),
	   and possibly remove non-printables... if any */
	for (r = cut_begy; r <= cut_endy; r++)
	 {
	   for (spaces = 0, c = cut_begx; c <= cut_endx; c++, srcptr++)
	     if (*srcptr <= 32)
	       spaces++;
	     else
	      {
		while (spaces)
		 {
		   *(dstptr++) = ' ';
		   spaces--;
		 }
		*(dstptr++) = *srcptr;
	      }
	   if (r != cut_endy)
	     *(dstptr++) = '\r';
	 }
	*dstptr = 0;
	/* throw away old buffer... we should consider a command to append... */
	if (cut_buffer)
	  free (cut_buffer);
/* Should we just keep dstptr? We would save a malloc... */
	/* make a new permanent buffer of just the right size */
	cut_buffer = (char *) malloc (strlen (dstbuf) + 1);
	strcpy (cut_buffer, dstbuf);
	free (srcbuf);
	free (dstbuf);
	play (snd_cut_end);
      }
     else
      {
	if (srcbuf)
	  free (srcbuf);
	if (dstbuf)
	  free (dstbuf);
      }
   }
}
