/*
 * Orca
 *
 * Copyright 2004-2006 Sun Microsystems Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

#include <string.h>
#include <errno.h>
#include <dlfcn.h>
#include <glib.h>
#include <Python.h>
#include <fcntl.h>
#include "brl.h"
#include "../brld/libbrld.h"

/* global variables - config stuff */
static sbl_config sblconf;
static char varvec[200][150];
static keymap_file keymap;

static brld_interface *brld = NULL;	/* filled by dynamic libbrld */

static void *libbrld = NULL;	/* handle to driver */

static int csrvis = 1, csr_form = 0;
static int old_csr = 0;
static char old_text[100] = "";

#define BRL_SYMBOL "brld"

/* A non-zero value indicates this module has been initialized and is
 * ready for communication with brld.
 */
static long brl_initialized = 0;

/* Python callback for Braille command notifications.  Only ONE callback can
 * be registered at a time.
 */
static PyObject *brl_callback = NULL;

/* The g_io_add_watch (see glib docs) that keeps track of input events
 * coming from brld.
 */
static gboolean brl_sbl_io_cb (GIOChannel * ch,
			       GIOCondition condition, void *data)
{

  int key, pressed;
  static int lastkey = 0;
  PyObject *result;
  PyObject *args;
  PyGILState_STATE gstate;

/* do this to avoid compiler warnings */
  if (ch && condition && data)
    usleep (1);
  while ((key = brld->getkey (&pressed)) > 0)
   {
     if (key && !csrvis)
      {
	csrvis = 1;
	brld->write (old_csr - 1, old_text, NULL);
      }

     if (pressed && lastkey >= 1000)
       lastkey = 0;

     if (!lastkey && pressed)
       lastkey = key;
     else
       lastkey = (lastkey * 1000) + key;
     if (!brl_callback)
      {
	break;
      }
     if (lastkey == keymap.lnup.brl)
       key = ORCA_LNUP;
     else if (lastkey == keymap.lndn.brl)
       key = ORCA_LNDN;
     else if (lastkey == keymap.lnlft.brl)
       key = ORCA_LNLFT;
     else if (lastkey == keymap.lnrgt.brl)
       key = ORCA_LNRGT;
     else if (lastkey == keymap.topleft.brl)
       key = ORCA_TOPLEFT;
     else if (lastkey == keymap.botleft.brl)
       key = ORCA_BOTLEFT;
     else if (lastkey == keymap.csrtrk.brl)
       key = ORCA_CSRTRK;
     else if (lastkey == keymap.csrvisoff.brl)
      {
	csrvis = 0;
	brld->write (-1, old_text, NULL);
	key = 0;
      }
     else if (lastkey == keymap.csrblockonoff.brl)
      {
	if (csr_form)
	  csr_form = 0;
	else
	  csr_form = 1;
	brld->cursor (csr_form);
	brld->write (csrvis ? (old_csr - 1) : -1, old_text, NULL);
	key = 0;
      }
     else if (lastkey >= keymap.csrroutbeg && lastkey <= keymap.csrroutend)
      {
	if (pressed)
	  key = (lastkey - keymap.csrroutbeg) + 0x100;

	lastkey = 0;
      }
     else
       key = 0;

     if (!pressed)
      {
	lastkey = 0;
      }

     pressed = 0;

     args = Py_BuildValue ("(i)", key);

     gstate = PyGILState_Ensure ();
     result = PyObject_CallObject (brl_callback, args);
     if (result != NULL)
      {
	Py_DECREF (result);
      }
     else
      {
	PyErr_Print ();
      }
     PyGILState_Release (gstate);
     Py_DECREF (args);
   }
  return TRUE;
}

/* Initializes the brl module, connecting to BrlTTY.  Returns 0 on failure or
 * 1 on success.  The first argument is optional and indicated the tty to
 * open.  The default value for this argument is -1, which means to let brld
 * use its default logic.  A typical value might be 7 which is usually what
 * the CONTROLVT should be set to for a console running the X11 server.  The
 * second argument is also optional and is to be passed to brlapi_getTty and
 * is used to tell BrlTTY whether to return raw keycodes (BRLKEYCODES=1) or
 * BrlTTY commands (BRLCOMMANDS=0).  The default value is to give us
 * BRLCOMMANDS.
 */
static PyObject *brl_module_init (PyObject * self, PyObject * args)
{
  int tty = -1;
  int how = 0;			/* BRLCOMMANDS */

  GIOChannel *brl_channel;

/* just to avoid compiler warnings */
  if (self)
    usleep (1);
  if (brl_initialized)
   {
     PyErr_SetString (PyExc_StandardError, "Already initialized");
     return NULL;
   }

  if (!PyArg_ParseTuple (args, "|ii:init", &tty, &how))
    return NULL;

  const char *error = NULL;
  int sock_fd = 0;

  getsblconf (&sblconf, NULL);

  if (libbrld)
    dlclose (libbrld);

  libbrld = dlopen ("/usr/lib/libbrld.so", RTLD_LAZY);

  if (libbrld == NULL)
    libbrld = dlopen ("/usr/local/lib/libbrld.so", RTLD_LAZY);

  if (libbrld == NULL)
    libbrld = dlopen ("/tmp/suse-blinux/libbrld.so", RTLD_LAZY);

  if (libbrld == NULL)
   {
     fprintf (stderr, "error: can't load libbrld.so\n");
     return 0;
   }

  brld = (brld_interface *) dlsym (libbrld, BRL_SYMBOL);

  error = dlerror ();

  if (error)
   {
     fprintf (stderr, "%s\n", error);
     return 0;
   }

  if ((sock_fd = brld->open ("localhost", 8888)) < 0)
   {
     fprintf (stderr, "error: connecting brld\n");
     return 0;
   }

  if (brld->reg (X11, "default"))
   {
     fprintf (stderr, "error: brld registration failed\n");
     brld->close ();
     return 0;
   }

  brld->getalias (sblconf.brlname);

/* set event mode  to get SIGIO*/
  brld->seteventmode ();

  /* set translation-table to text */
  brld->texttbl ();

  /* set block-cursor */
  brld->cursor (csr_form);

  getkeymap (sblconf, NULL, NULL);
  /* Setup the GIOChannel to receive notifications of Braille
   * key events
   */
  brl_channel = g_io_channel_unix_new (sock_fd);
  g_io_add_watch (brl_channel, G_IO_IN, brl_sbl_io_cb, NULL);
  brl_initialized = 1;
  return PyInt_FromLong (brl_initialized);
}

static PyObject *brl_module_shutdown (PyObject * self)
{
/* just to avoid compiler warnings */
  if (self)
    usleep (1);
  if (brl_initialized)
   {
     brld->close ();
   }
  brl_initialized = 0;
  if (libbrld)
    dlclose (libbrld);
  return PyInt_FromLong (1);
}

static PyObject *brl_module_getDriverId (PyObject * self)
{

/* just to avoid compiler warnings */
  if (self)
    usleep (1);

  if (!brl_initialized)
   {
     Py_INCREF (Py_None);
     return Py_None;
   }

  return PyString_FromString ((const char *) "sbl");
}

static PyObject *brl_module_getDriverName (PyObject * self)
{
  char name[80];

/* just to avoid compiler warnings */
  if (self)
    usleep (1);
  if (!brl_initialized)
   {
     Py_INCREF (Py_None);
     return Py_None;
   }

  brld->getname (name);
  if (strlen (name) > 0)
   {
     return PyString_FromString ((const char *) name);
   }
  else
   {
     Py_INCREF (Py_None);
     return Py_None;
   }

  return Py_None;
}

static PyObject *brl_module_getDisplayWidth (PyObject * self)
{

  int x = 0, y = 0;

/* just to avoid compiler warnings */
  if (self)
    usleep (1);

  if (!brl_initialized)
   {
     return PyInt_FromLong (0);
   }

  brld->getxy (&x, &y);
  return PyInt_FromLong (x);
}

static PyObject *brl_module_getDisplayHeight (PyObject * self)
{

/* just to avoid compiler warnings */
  if (self)
    usleep (1);

  if (!brl_initialized)
   {
     return PyInt_FromLong (0);
   }

/* we only support displays with one line */
  return PyInt_FromLong (1);
}

static PyObject *brl_module_writeText (PyObject * self, PyObject * args)
{
  int cursor;
  char *str;

/* just to avoid compiler warnings */
  if (self)
    usleep (1);

  if (!PyArg_ParseTuple (args, "is:writeText", &cursor, &str))
   {
     return NULL;
   }

  if (brl_initialized)
   {

     if (old_csr != cursor || strcmp (old_text, str))
      {
	brld->write (csrvis ? (cursor - 1) : -1, str, NULL);
	old_csr = cursor;
	strcpy (old_text, str);
      }
   }

  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject *brl_module_writeDots (PyObject * self, PyObject * args)
{
  char *str;

/* just to avoid compiler warnings */
  if (self)
    usleep (1);

  if (!PyArg_ParseTuple (args, "s:writeText", &str))
   {
     return NULL;
   }

  if (brl_initialized)
   {
     brld->write (-1, str, NULL);
   }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject *brl_module_registerCallback (PyObject * self,
					      PyObject * args)
{
/* just to avoid compiler warnings */
  if (self)
    usleep (1);
  if (brl_callback)
   {
     Py_DECREF (brl_callback);
   }
  if (brl_initialized)
   {
     brl_callback = PyTuple_GetItem (args, 0);
     Py_INCREF (brl_callback);
   }
  Py_INCREF (Py_None);
  return Py_None;
}

static PyObject *brl_module_unregisterCallback (PyObject * self)
{
/* just to avoid compiler warnings */
  if (self)
    usleep (1);

  if (brl_callback)
   {
     Py_DECREF (brl_callback);
   }

  brl_callback = NULL;
  Py_INCREF (Py_None);
  return Py_None;
}

static PyMethodDef brl_methods[] = {
  {"init", (PyCFunction) brl_module_init, METH_VARARGS, NULL},
  {"shutdown", (PyCFunction) brl_module_shutdown, METH_NOARGS, NULL},
  {"getDriverId", (PyCFunction) brl_module_getDriverId, METH_NOARGS, NULL},
  {"getDriverName", (PyCFunction) brl_module_getDriverName, METH_NOARGS,
   NULL},
  {"getDisplayWidth", (PyCFunction) brl_module_getDisplayWidth, METH_NOARGS,
   NULL},
  {"getDisplayHeight", (PyCFunction) brl_module_getDisplayHeight, METH_NOARGS,
   NULL},
  {"writeText", (PyCFunction) brl_module_writeText, METH_VARARGS, NULL},
  {"writeDots", (PyCFunction) brl_module_writeDots, METH_VARARGS, NULL},
  {"registerCallback", (PyCFunction) brl_module_registerCallback,
   METH_VARARGS, NULL},
  {"unregisterCallback", (PyCFunction) brl_module_unregisterCallback,
   METH_NOARGS, NULL},
  {NULL, NULL, 0, NULL}
};

void initbrl (void)
{
  (void) Py_InitModule ("brl", brl_methods);
}

/* load sbl configuration files */

void getsblconf (sbl_config * sblconf, char *conf)
{
  char str[100];

  if (!conf)
    confread ("/etc/suse-blinux.conf");
  else
    confread (conf);

  confsearch (strcpy (sblconf->brld_host, "brld_host"));
  confsearch (strcpy (str, "brld_port"));
  if (atoi (str))
    sblconf->brld_port = atoi (str);
  else
    sblconf->brld_port = 8888;

  confsearch (strcpy (str, "brld_auth_key"));
  if (str[0] != 0)
    strcpy (sblconf->brld_auth_key, str);
  else
    strcpy (sblconf->brld_auth_key, "default");

  confsearch (strcpy (sblconf->progpath, "progpath"));
}

void getkeymap (sbl_config sblconf, char *profname, char *homedir)
{
  char str[120];

  sprintf (str, "%s/.suse-blinux/keymap/%s.map.%s", homedir, sblconf.brlname,
	   profname);
  if (!confread (str))
   {
     sprintf (str, "%s/.suse-blinux/keymap/%s.map", homedir, sblconf.brlname);
     if (!confread (str))
      {
	sprintf (str, "%s/keymap/%s.map.%s", sblconf.progpath,
		 sblconf.brlname, profname);
	if (!confread (str))
	 {
	   sprintf (str, "%s/keymap/%s.map", sblconf.progpath,
		    sblconf.brlname);
	   confread (str);
	 }
      }
   }

  keymap.kbdsniffon1 = keytrans (confsearch (strcpy (str, "kbdsniffon1")));
  keymap.kbdsniffon2 = keytrans (confsearch (strcpy (str, "kbdsniffon2")));
  keymap.kbdsniffoff = keytrans (confsearch (strcpy (str, "kbdsniffoff")));
  keymap.resetbrl = keytrans (confsearch (strcpy (str, "resetbrl")));
  keymap.line01 = keytrans (confsearch (strcpy (str, "line01")));
  keymap.topleft = keytrans (confsearch (strcpy (str, "topleft")));
  keymap.botleft = keytrans (confsearch (strcpy (str, "botleft")));
  keymap.winup = keytrans (confsearch (strcpy (str, "winup")));
  keymap.windn = keytrans (confsearch (strcpy (str, "windn")));
  keymap.lnlft = keytrans (confsearch (strcpy (str, "lnlft")));
  keymap.lnrgt = keytrans (confsearch (strcpy (str, "lnrgt")));
  keymap.lnup = keytrans (confsearch (strcpy (str, "lnup")));
  keymap.lndn = keytrans (confsearch (strcpy (str, "lndn")));
  keymap.csrtrk = keytrans (confsearch (strcpy (str, "csrtrk")));
  keymap.syscsr = keytrans (confsearch (strcpy (str, "syscsr")));
  keymap.softcsr = keytrans (confsearch (strcpy (str, "softcsr")));
  keymap.chrlft = keytrans (confsearch (strcpy (str, "chrlft")));
  keymap.chrrgt = keytrans (confsearch (strcpy (str, "chrrgt")));
  keymap.hwinlft = keytrans (confsearch (strcpy (str, "hwinlft")));
  keymap.hwinrgt = keytrans (confsearch (strcpy (str, "hwinrgt")));
  keymap.fwinlft = keytrans (confsearch (strcpy (str, "fwinlft")));
  keymap.fwinrgt = keytrans (confsearch (strcpy (str, "fwinrgt")));
  keymap.csrjump = keytrans (confsearch (strcpy (str, "csrjump")));
  keymap.csrjmpvert = keytrans (confsearch (strcpy (str, "csrjmpvert")));
  keymap.keyup = keytrans (confsearch (strcpy (str, "keyup")));
  keymap.keydn = keytrans (confsearch (strcpy (str, "keydn")));
  keymap.keylft = keytrans (confsearch (strcpy (str, "keylft")));
  keymap.keyrgt = keytrans (confsearch (strcpy (str, "keyrgt")));
  keymap.keyenter = keytrans (confsearch (strcpy (str, "keyenter")));
  keymap.keyesc = keytrans (confsearch (strcpy (str, "keyesc")));
  keymap.csrvisoff = keytrans (confsearch (strcpy (str, "csrvisoff")));
  keymap.jmpmark1 = keytrans (confsearch (strcpy (str, "jmpmark1")));
  keymap.jmpmark2 = keytrans (confsearch (strcpy (str, "jmpmark2")));
  keymap.jmpmark3 = keytrans (confsearch (strcpy (str, "jmpmark3")));
  keymap.jmpmark4 = keytrans (confsearch (strcpy (str, "jmpmark4")));
  keymap.setmark = keytrans (confsearch (strcpy (str, "setmark")));
  keymap.jmptomark = keytrans (confsearch (strcpy (str, "jmptomark")));
  keymap.attr1 = keytrans (confsearch (strcpy (str, "attr1")));
  keymap.attr2 = keytrans (confsearch (strcpy (str, "attr2")));
  keymap.attr3 = keytrans (confsearch (strcpy (str, "attr3")));
  keymap.attr4 = keytrans (confsearch (strcpy (str, "attr4")));
  keymap.csrtrkoff = keytrans (confsearch (strcpy (str, "csrtrkoff")));
  keymap.prof1 = keytrans (confsearch (strcpy (str, "prof1")));
  keymap.prof2 = keytrans (confsearch (strcpy (str, "prof2")));
  keymap.prof3 = keytrans (confsearch (strcpy (str, "prof3")));
  keymap.prof4 = keytrans (confsearch (strcpy (str, "prof4")));
  keymap.attroff = keytrans (confsearch (strcpy (str, "attroff")));
  keymap.attrmodonoff = keytrans (confsearch (strcpy (str, "attrmodonoff")));
  keymap.sixdotsonoff = keytrans (confsearch (strcpy (str, "sixdotsonoff")));
  keymap.csrblinkonoff =
    keytrans (confsearch (strcpy (str, "csrblinkonoff")));
  keymap.csrblockonoff =
    keytrans (confsearch (strcpy (str, "csrblockonoff")));
  keymap.cutbegin = keytrans (confsearch (strcpy (str, "cutbegin")));
  keymap.cutend = keytrans (confsearch (strcpy (str, "cutend")));
  keymap.loadkeymap = keytrans (confsearch (strcpy (str, "loadkeymap")));
  keymap.paste = keytrans (confsearch (strcpy (str, "paste")));
  keymap.soundonoff = keytrans (confsearch (strcpy (str, "soundonoff")));
  keymap.autoprofonoff =
    keytrans (confsearch (strcpy (str, "autoprofonoff")));
  keymap.spktocsr = keytrans (confsearch (strcpy (str, "spktocsr")));
  keymap.spkcurln = keytrans (confsearch (strcpy (str, "spkcurln")));
  keymap.spkscrtocsr = keytrans (confsearch (strcpy (str, "spkscrtocsr")));
  keymap.spkscrfromcsr =
    keytrans (confsearch (strcpy (str, "spkscrfromcsr")));
  keymap.spkfromcsr = keytrans (confsearch (strcpy (str, "spkfromcsr")));
  keymap.spkmod = keytrans (confsearch (strcpy (str, "spkmod")));
  keymap.spkoff = keytrans (confsearch (strcpy (str, "spkoff")));
  keymap.nextlang = keytrans (confsearch (strcpy (str, "nextlang")));
  keymap.nextvol = keytrans (confsearch (strcpy (str, "nextvol")));
  keymap.nextvoice = keytrans (confsearch (strcpy (str, "nextvoice")));
  keymap.nextspec = keytrans (confsearch (strcpy (str, "nextspec")));
  keymap.nextfrq = keytrans (confsearch (strcpy (str, "nextfrq")));
  keymap.nextspd = keytrans (confsearch (strcpy (str, "nextspd")));
  keymap.prevlang = keytrans (confsearch (strcpy (str, "prevlang")));
  keymap.prevvol = keytrans (confsearch (strcpy (str, "prevvol")));
  keymap.prevvoice = keytrans (confsearch (strcpy (str, "prevvoice")));
  keymap.prevspec = keytrans (confsearch (strcpy (str, "prevspec")));
  keymap.prevfrq = keytrans (confsearch (strcpy (str, "prevfrq")));
  keymap.prevspd = keytrans (confsearch (strcpy (str, "prevspd")));
  keymap.spkcharmod = keytrans (confsearch (strcpy (str, "spkcharmod")));

  keymap.csrroutbeg = atoi (confsearch (strcpy (str, "csrroutbeg")) + 1);
  keymap.csrroutend = atoi (confsearch (strcpy (str, "csrroutend")) + 1);
  keymap.lineroutbeg = atoi (confsearch (strcpy (str, "lineroutbeg")) + 1);
  keymap.lineroutend = atoi (confsearch (strcpy (str, "lineroutend")) + 1);
}

#define CKEYS 37
brlkbd keytrans (char *str)
{
  brlkbd keys;
  unsigned int i = 0;
  int code = 0, kbdcode = 0;
  char key1[10] = "", key2[10] = "", kbdstr[30] = "";

  keys.kbd = 0;
  keys.brl = 0;

  for (i = 0; i < strlen (str); i++)
    if (str[i] == ',')
      kbdcode = i;
  if (strlen (str) < 2)
   {
     keys.brl = 0;
     keys.kbd = 0;
     return keys;
   }
  if (str[kbdcode] == ',')
   {
     str[kbdcode] = 0;
   }
  else
    kbdstr[0] = 0;
  if (strlen (str) > 3)
   {
     strncpy (key1, str, 4);
     key1[4] = 0;
     if (strlen (str) > 4)
       strcpy (key2, str + 5);
     key2[4] = 0;
     code = 0;
     code = (atoi (key1 + 1));
     if (strlen (key2))
       code = (code * 1000) + atoi (key2 + 1);

     keys.brl = code;
   }				/* if */
  else
    keys.brl = 0;

  memset (key1, 0, sizeof (key1));
  code = 0;
  return keys;

}

/* helper function to parse the config files */
int confread (char *realname)
{
  FILE *pname;
  int i = 0;
  char row[100] = "";

  pname = fopen (realname, "r");
  if (pname == NULL)
    return 0;
  while (!feof (pname))
   {
     fgets (row, 80, pname);
     if (row[0] != '#' && strlen (row) > 5)
      {
	strcpy (varvec[i], row);
	i++;
      }
     if (feof (pname))
       break;
   }
  fclose (pname);
  varvec[i][0] = 0;
  return 1;
}

char *confsearch (char *searchstring)
{
  int i = 0, end = 0;

  strcat (searchstring, "=\0");
  do
   {
     if (strncmp (varvec[i], searchstring, strlen (searchstring)) == 0)
      {
	strcpy (searchstring, varvec[i] + strlen (searchstring));
	end = 1;
      }
     i++;
   }
  while (varvec[i][0] != 0 && !end);

  if (!end)
   {
     searchstring[0] = 0;
     return searchstring;
   }

  if (strlen (searchstring))
    searchstring[strlen (searchstring) - 1] = 0;
  return searchstring;
}
