
/*****************************************************************************
 *                                    G P M                                  *
 *                    Generic kernel module for PCI hardware                 *
 *                           (C) SuSE GmbH 1997, 1998                        *
 *****************************************************************************/
/* Author: simon pogarcic, sim@suse.de */
/*
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef _COMMON_GPM_H_
#define _COMMON_GPM_H_

/*****************************************************************************
 * The common data types and defines used in kernel module and in user space *
 *****************************************************************************/

#include <linux/types.h>
#include <linux/version.h>



#define TRUE	1
#define FALSE	0

#define S1K	0x400
#define S2K	0x800
#define S4K	0x1000
#define S8K	0x2000
#define S16K	0x4000
#define S32K	0x8000
#define S64K	0x10000
#define S128K	0x20000
#define S256K	0x40000
#define S512K	0x80000
#define S1M	0x100000
#define S2M	0x200000
#define S4M	0x400000
#define S8M	0x800000



/********************************************************
 * expected number of supported PCI devices in computer *
 ********************************************************/
#define MAX_DEV_ALOWED				16

/************************************************************
 * expected number of supported PCI cards in computer slots *
 ************************************************************/
#define MAX_CARDS_ALOWED			8

/****************************************************
 * max number of interrupts to be handled in driver *
 ****************************************************/
#define MAX_IRQS_ALOWED				16

/************************************************
 * the number of supported interrupts in system *
 ************************************************/
#define MAX_SYSTEM_IRQS				32



/* 8 bit: */
typedef char			Tbyte;
typedef unsigned char		Tubyte;
    
/* 16 bit: */
typedef short			Tshort;
typedef unsigned short		Tushort;

/* 32 bit: */    
typedef int			Tint;
typedef unsigned int		Tuint;

/* 64 or 32 bit */
typedef long			Tlong;
typedef unsigned long		Tulong;

/* single precission: */    
typedef float			Tfloat;

/* double precission: */
typedef double			Tdouble;



typedef unsigned char		Tbool;

typedef unsigned int*		Tpointer_32;
typedef unsigned short*		Tpointer_16;
typedef unsigned char*		Tpointer_8;



/*
 * RAM allocating flags, default (0x00): scather-gather, kernel mem 
 *******************************************************************/
#define MEM_CONT				0x01
#define MEM_HIGH				0x02

/*
 * DMA buffer status flags
 *************************/
#define DB_NOTUSED				0x0
#define DB_ACTIVE				0x1
#define DB_IDLE					0x2



/*
 * memory chunks/fragments info struct
 *************************************/
typedef struct GPMDmaChunk_t {
	Tulong			addr;
	Tubyte			order;
	struct GPMDmaChunk_t *	next;
} GPMDmaChunk_t;

typedef struct GPMDmaArea_t {
	Tulong			addr;
	Tulong			size;
	struct GPMDmaArea_t *	next;
} GPMDmaArea_t;



/*
 * allocated dma buffers info struct
 ***********************************/
typedef struct GPMDmaBuff_t {
/* chunks and areas table: */
	GPMDmaChunk_t *		chunks;
	GPMDmaArea_t *		areas;
	Tint			cnt_areas;
/* whole size of buffer (chunks sum) */
	Tulong			size;
/* DB_ flags */
	Tulong			flags;
/* buffer descriptor: */
	Tint			nr;
	Tint			cnt_mmaps;
/* double linked list */
	struct GPMDmaBuff_t * 	next;
	struct GPMDmaBuff_t * 	prev;
} GPMDmaBuff_t;



/*
 * Scan informations
 *******************/
typedef struct GPMDev_t {
	const Tubyte 		name[32];
	const Tuint 		chip_type;
	const Tushort		ven_id;
	const Tushort 		dev_id;
	const Tuint		irqflags;
	const Tuint		kmmap_base;
} GPMDev_t;

typedef struct GPMCard_t {
	const Tubyte		name[32];
	const Tubyte 		maxdev;		/* devs. on card expected */
	const Tuint		chip_type[8];	/* which dev to look for? */
} GPMCard_t;



/*
 * keeping track of resources usage
 **********************************/
typedef struct GPMAddr_t {
	Tulong			addr;
	struct GPMAddr_t	*next;
} GPMAddr_t;



/*****************************************************************************
 *                              IOCTL operations:                            *
 *****************************************************************************/

/* minor 0 (pci)
 ***************/

#define IOCTL_PCI_SETCONFIG		10
#define IOCTL_PCI_GETCONFIG		11
#define IOCTL_PCI_GETCARDDATA		12
#define IOCTL_PCI_WRITEBASE		13
#define IOCTL_PCI_READBASE		14
#define IOCTL_PCI_ISRINSTALL		15
#define IOCTL_PCI_ISRUNINSTALL		16
#define IOCTL_PCI_ISRCONFIG		17
#define IOCTL_PCI_GETAREAS		18
#define IOCTL_PCI_SETSIGTASK		19

/*
 * PCI read/write config space (_SET/_GETCONFIG)
 ***********************************************/
typedef struct GPMPciConfig_t {
	Tubyte		reg;
	Tubyte		val_size;
	Tuint		value;
	Tuint		real_value;
	Tbool		ignore;
} GPMPciConfig_t;

typedef struct GPMPciParam_t {
	Tint		Icardnr;
	GPMPciConfig_t	cfg[8];
} GPMPciParam_t;

#define PCICONF_CARD(p)			p.Icardnr
#define PCICONF_OFFSET(p,d)		p.cfg[d].reg
#define PCICONF_SIZE(p,d)		p.cfg[d].val_size
#define PCICONF_WVALUE(p,d)		p.cfg[d].value
#define PCICONF_RVALUE(p,d)		p.cfg[d].real_value
#define PCICONF_SKIP(p,d)		p.cfg[d].ignore

/*
 * Card/grouped devices informations (_GETCARDDATA)
 **************************************************/
typedef struct GPMBase_t {
	Tulong	base;
	Tulong	aoffset;
	Tulong	size;
	Tuint	flags;
} GPMBase_t;

typedef struct GPMCardDev_t {
	Tuint		devidx;
	Tuint		basefound;
	GPMBase_t	PCIBase[6];
} GPMCardDev_t;

typedef struct GPMCardInfo_t {
	GPMCardDev_t 	cd[8];		/* devices data */
	Tuint		maxdev;		/* number of devices on card */
	Tubyte		slot;		/* slot */
	Tubyte		bus;		/* bus */
	Tuint		idx;		/* which one from SupportedCards[] */
	Tbool		error;		/* scan error ? */
} GPMCardInfo_t;

/*
 * Read/Write data on PCI base (_WRITE/READBASE)
 ***********************************************/
typedef struct GPMIoctlRw_t {
	Tbyte		Icardnr;
	Tbyte		Idevnr;
	Tbyte		Ibasenr;
	Tulong		offset_bytes;
	Tulong		value;
	Tubyte		value_size;
	Tubyte		write_logop;
} GPMIoctlRw_t;

#define LOGOP_WRITE_OVER	0	/* overwrite old val with new value */
#define LOGOP_WRITE_OR		1	/* old value | new value */
#define LOGOP_WRITE_AND		2	/* old value & new value */
#define LOGOP_WRITE_NAND	3	/* old value & ~(new value) */

/*
 * Config ISR (_ISRCONFIG)
 *************************/
typedef struct GPMIsrCfg_t {
	Tubyte		cardnr;
	Tubyte		devnr;
	Tuint		irqtype;
	Tulong		mask;
} GPMIsrCfg_t;

/*
 * For retrieving of physical RAM address needed for DMA (_GETAREAS)
 *******************************************************************/
typedef struct GPMIoctlArea_t {
	Tulong		addr;
	Tulong		size;
} GPMIoctlArea_t;
	
typedef struct GPMIoctlGetArea_t {
	Tint		dbnr;
	GPMIoctlArea_t	*areas;
} GPMIoctlGetArea_t;



/*****************************************************************************
 *                              MMAP operations:                             *
 *****************************************************************************/

/* minor 0 (pci)
 ***************/

/*
 * minor 1 (mem)
 ***************/
#define MMAP_MEM_ALLOC			0
#define MMAP_MEM_SHARE			1

typedef struct GPMMmapMem_t {
	Tuint		action;
	Tuint		flags;
	Tint		buffer_nr;
	Tuint		ret_areas_cnt;
} GPMMmapMem_t;

/*
 * minor 2 (ctx)
 ***************/
typedef struct GPMMmapCtx_t {
	Tuint		flags;
	Tuint		ctxsize;
	Tushort		buffcnt;		/* 1-3 (A-C) */
	Tushort		ret_buffer_nr_A;
	Tushort		ret_areas_cnt_A;
	Tushort		ret_buffer_nr_B;
	Tushort		ret_areas_cnt_B;
	Tushort		ret_buffer_nr_C;
	Tushort		ret_areas_cnt_C;
} GPMMmapCtx_t;


#define BASE_NONE			0
#define BASE(n)				(1 << n) /* n = 0-5 */

#define BASE_0				0
#define BASE_1				1
#define BASE_2				2
#define BASE_3				3
#define BASE_4				4
#define BASE_5				5
#define BASE_PCI			6

#define BIT(n)				(1 << n)

#define SIZE_BYTE			1 /* 8 bits */
#define SIZE_WORD			2 /* 16 bits */
#define SIZE_DWORD			4 /* 32 bits */
#define SIZE_QWORD			8 /* 64 bits */

#define FAST_IRQ			0x01

#endif
